<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

/**
 * Meta Data class (element location)
 * @package ilias-core
 * @version $Id$
 */
class ilMDLocation extends ilMDBase
{
    private string $location = '';
    private string $location_type = '';

    // SET/GET
    public function setLocation(string $a_location): void
    {
        $this->location = $a_location;
    }

    public function getLocation(): string
    {
        return $this->location;
    }

    public function setLocationType(string $a_location_type): void
    {
        $this->location_type = $a_location_type;
    }

    public function getLocationType(): string
    {
        return $this->location_type;
    }

    public function save(): int
    {
        $fields = $this->__getFields();
        $fields['meta_location_id'] = array('integer', $next_id = $this->db->nextId('il_meta_location'));

        if ($this->db->insert('il_meta_location', $fields)) {
            $this->setMetaId($next_id);
            return $this->getMetaId();
        }
        return 0;
    }

    public function update(): bool
    {
        return $this->getMetaId() && $this->db->update(
            'il_meta_location',
            $this->__getFields(),
            array("meta_location_id" => array('integer', $this->getMetaId()))
        );
    }

    public function delete(): bool
    {
        if ($this->getMetaId()) {
            $query = "DELETE FROM il_meta_location " .
                "WHERE meta_location_id = " . $this->db->quote($this->getMetaId(), 'integer');
            $res = $this->db->manipulate($query);

            return true;
        }
        return false;
    }

    /**
     * @return array<string, array<string, mixed>>
     */
    public function __getFields(): array
    {
        return array(
            'rbac_id' => array('integer', $this->getRBACId()),
            'obj_id' => array('integer', $this->getObjId()),
            'obj_type' => array('text', $this->getObjType()),
            'parent_type' => array('text', $this->getParentType()),
            'parent_id' => array('integer', $this->getParentId()),
            'location' => array('text', $this->getLocation()),
            'location_type' => array('text', $this->getLocationType())
        );
    }

    public function read(): bool
    {
        if ($this->getMetaId()) {
            $query = "SELECT * FROM il_meta_location " .
                "WHERE meta_location_id = " . $this->db->quote($this->getMetaId(), 'integer');

            $res = $this->db->query($query);
            while ($row = $res->fetchRow(ilDBConstants::FETCHMODE_OBJECT)) {
                $this->setRBACId((int) $row->rbac_id);
                $this->setObjId((int) $row->obj_id);
                $this->setObjType($row->obj_type ?? '');
                $this->setParentId((int) $row->parent_id);
                $this->setParentType($row->parent_type);
                $this->setLocation($row->location ?? '');
                $this->setLocationType($row->location_type ?? '');
            }
        }
        return true;
    }

    public function toXML(ilXmlWriter $writer): void
    {
        $writer->xmlElement(
            'Location',
            array(
                'Type' => $this->getLocationType() ?: 'LocalFile'
            ),
            $this->getLocation()
        );
    }

    // STATIC

    /**
     * @return int[]
     */
    public static function _getIds(int $a_rbac_id, int $a_obj_id, int $a_parent_id, string $a_parent_type): array
    {
        global $DIC;

        $ilDB = $DIC->database();

        $query = "SELECT meta_location_id FROM il_meta_location " .
            "WHERE rbac_id = " . $ilDB->quote($a_rbac_id, 'integer') . " " .
            "AND obj_id = " . $ilDB->quote($a_obj_id, 'integer') . " " .
            "AND parent_id = " . $ilDB->quote($a_parent_id, 'integer') . " " .
            "AND parent_type = " . $ilDB->quote($a_parent_type, 'text');

        $res = $ilDB->query($query);
        $ids = [];
        while ($row = $res->fetchRow(ilDBConstants::FETCHMODE_OBJECT)) {
            $ids[] = (int) $row->meta_location_id;
        }
        return $ids;
    }
}
