<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

require_once(__DIR__ . "/../../../../../libs/composer/vendor/autoload.php");
require_once(__DIR__ . "/../../../Base.php");
require_once(__DIR__ . "/InputTest.php");

use ILIAS\UI\Implementation\Component as I;
use ILIAS\Refinery\Factory as Refinery;
use ILIAS\Data;

class HiddenInputTest extends ILIAS_UI_TestBase
{
    protected DefNamesource $name_source;
    protected I\Input\Field\Hidden $input;

    public function setUp(): void
    {
        $this->name_source = new DefNamesource();
        $this->input = new I\Input\Field\Hidden(
            new Data\Factory(),
            new Refinery(
                new Data\Factory(),
                $this->createMock(ilLanguage::class)
            )
        );
    }

    public function testRender(): void
    {
        $input = $this->input->withNameFrom($this->name_source);

        $r = $this->getDefaultRenderer();
        $html = $this->brutallyTrimHTML($r->render($input));

        $expected = $this->brutallyTrimHTML('
            <input id="id_1" type="hidden" name="name_0" value="" />
        ');
        $this->assertEquals($expected, $html);
    }

    public function testRenderDisabled(): void
    {
        $input = $this->input->withNameFrom($this->name_source);
        $input = $input->withDisabled(true);

        $r = $this->getDefaultRenderer();
        $html = $this->brutallyTrimHTML($r->render($input));

        $expected = $this->brutallyTrimHTML('
            <input id="id_1" type="hidden" name="name_0" value="" disabled="disabled"/>
        ');
        $this->assertEquals($expected, $html);
    }

    public function testRenderValue(): void
    {
        $input = $this->input->withNameFrom($this->name_source);
        $input = $input->withValue('some_value');

        $r = $this->getDefaultRenderer();
        $html = $this->brutallyTrimHTML($r->render($input));

        $expected = $this->brutallyTrimHTML('
            <input id="id_1" type="hidden" name="name_0" value="some_value" />
        ');
        $this->assertEquals($expected, $html);
    }

    public function testRenderEscaped(): void
    {
        $expected = $this->brutallyTrimHTML('
            <input id="id_1" type="hidden" name="name_0" value="&lt;script&gt;alert(&quot;XSS&quot;);&lt;/script&gt;" />
        ');
        $actual = $this->brutallyTrimHTML(
            $this->getDefaultRenderer()->render(
                $this->input->withNameFrom($this->name_source)->withValue('<script>alert("XSS");</script>')
            )
        );
        $this->assertEquals($expected, $actual);
    }
}
