<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

/**
* Image map image preview creator
*
* Takes an image and imagemap areas and creates a preview image containing
* the imagemap areas.
*
* @author		Helmut Schottmüller <helmut.schottmueller@mac.com>
* @version	$Id$
* @ingroup components\ILIASTestQuestionPool
*/
class ilImagemapPreview
{
    public $imagemap_filename;
    public $preview_filename;
    public $areas;
    public $points;
    public $linewidth_outer;
    public $linewidth_inner;
    public $lng;
    private \ilGlobalTemplateInterface $main_tpl;

    /**
    * ilImagemapPreview constructor
    *
    * Creates an instance of the ilImagemapPreview class
    *
    * @param integer $id The database id of a image map question object
    * @access public
    */
    public function __construct($imagemap_filename = "")
    {
        global $DIC;
        $this->main_tpl = $DIC->ui()->mainTemplate();
        $lng = $DIC['lng'];
        $this->lng = &$lng;
        $this->imagemap_filename = $imagemap_filename;

        if (!@is_file($this->preview_filename)) {
            $extension = ".jpg";
            if (preg_match("/.*\.(png|jpg|gif|jpeg)$/", $this->imagemap_filename, $matches)) {
                $extension = "." . $matches[1];
            }
            $this->preview_filename = ilFileUtils::ilTempnam() . $extension;
        }
        $this->areas = [];
        $this->points = [];
        $this->linewidth_outer = 4;
        $this->linewidth_inner = 2;
    }

    public function getAreaCount(): int
    {
        return count($this->areas);
    }

    public function getPointCount(): int
    {
        return count($this->points);
    }

    public function addArea(
        $index,
        $shape,
        $coords,
        $title = "",
        $href = "",
        $target = "",
        $visible = true,
        $linecolor = "red",
        $bordercolor = "white",
        $fillcolor = "#FFFFFFA0"
    ): void {
        if (ini_get("safe_mode")) {
            if ((strpos($fillcolor, "#") !== false) || (strpos($fillcolor, "rgb") !== false)) {
                $fillcolor = str_replace("\"", "", $fillcolor);
            }
        }
        $this->areas[$index] = [
            "shape" => "$shape",
            "coords" => "$coords",
            "title" => htmlspecialchars($title),
            "href" => "$href",
            "target" => "$target",
            "linecolor" => '"' . $linecolor . '"',
            "fillcolor" => '"' . $fillcolor . '"',
            "bordercolor" => '"' . $bordercolor . '"',
            "visible" => (int) $visible
        ];
    }

    public function addPoint(
        $index,
        $coords,
        $visible = true,
        $linecolor = "red",
        $bordercolor = "white",
        $fillcolor = "#FFFFFFA0"
    ): void {
        $this->points[$index] = [
            "coords" => "$coords",
            "linecolor" => '"' . $linecolor . '"',
            "fillcolor" => '"' . $fillcolor . '"',
            "bordercolor" => '"' . $bordercolor . '"',
            "visible" => (int) $visible
        ];
    }

    public function getAreaIdent(): string
    {
        if (count($this->areas) + count($this->points) > 0) {
            $arr = array_merge(array_keys($this->areas), array_keys($this->points));
            sort($arr, SORT_NUMERIC);

            $inner = join("_", $arr);
            if (strlen($inner) > 32) {
                $inner = md5($inner);
            }
            return "preview_" . $inner . "_";
        } else {
            return "";
        }
    }

    public function createPreview(): void
    {
        if (count($this->areas) + count($this->points) == 0) {
            return;
        }
        $convert_cmd = "-quality 100 ";
        foreach ($this->points as $point) {
            if ($point["visible"]) {
                preg_match("/(\d+)\s*,\s*(\d+)/", $point["coords"], $matches);
                $x = $matches[1];
                $y = $matches[2];
                $r = 6;
                // draw a circle at the point
                $convert_cmd .= "-stroke " . $point["bordercolor"] . " -fill " . $point["fillcolor"] . " -strokewidth $this->linewidth_outer -draw \"line " .
                ($x - $r) . "," . ($y - $r) . " " . ($x + $r) . "," . ($y + $r) . "\" " .
                "-stroke " . $point["bordercolor"] . " -fill " . $point["fillcolor"] . " -strokewidth $this->linewidth_outer -draw \"line " .
                ($x + $r) . "," . ($y - $r) . " " . ($x - $r) . "," . ($y + $r) . "\" " .
                "-stroke " . $point["linecolor"] . " -fill " . $point["fillcolor"] . " -strokewidth $this->linewidth_inner -draw \"line " .
                ($x - $r) . "," . ($y - $r) . " " . ($x + $r) . "," . ($y + $r) . "\" " .
                "-stroke " . $point["linecolor"] . " -fill " . $point["fillcolor"] . " -strokewidth $this->linewidth_inner -draw \"line " .
                ($x + $r) . "," . ($y - $r) . " " . ($x - $r) . "," . ($y + $r) . "\" ";
            }
        }
        foreach ($this->areas as $area) {
            if ($area["visible"] and strcmp(strtolower($area["shape"]), "rect") == 0) {
                preg_match("/(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)/", $area["coords"], $matches);
                $x0 = $matches[1];
                $y0 = $matches[2];
                $x1 = $matches[3];
                $y1 = $matches[4];
                // draw a rect around the selection
                $convert_cmd .= "-stroke " . $area["bordercolor"] . " -fill " . $area["fillcolor"] . " -strokewidth $this->linewidth_outer -draw \"rectangle " .
                $x0 . "," . $y0 . " " . ($x1) . "," . $y1 . "\" " .
                "-stroke " . $area["linecolor"] . " -fill " . $area["fillcolor"] . " -strokewidth $this->linewidth_inner -draw \"rectangle " .
                $x0 . "," . $y0 . " " . ($x1) . "," . $y1 . "\" ";
            } elseif ($area["visible"] and strcmp(strtolower($area["shape"]), "circle") == 0) {
                preg_match("/(\d+)\s*,\s*(\d+)\s*,\s*(\d+)/", $area["coords"], $matches);
                $x = $matches[1];
                $y = $matches[2];
                $r = $matches[3];
                // draw a circle around the selection
                $convert_cmd .= "-stroke " . $area["bordercolor"] . " -fill " . $area["fillcolor"] . " -strokewidth $this->linewidth_outer -draw \"circle " .
                $x . "," . $y . " " . ($x + $r) . "," . $y . "\" " .
                "-stroke " . $area["linecolor"] . " -fill " . $area["fillcolor"] . " -strokewidth $this->linewidth_inner -draw \"circle " .
                $x . "," . $y . " " . ($x + $r) . "," . $y . "\" ";
            } elseif ($area["visible"] and strcmp(strtolower($area["shape"]), "poly") == 0) {
                $obj = "polygon";
                // draw a polygon around the selection
                preg_match_all("/(\d+)\s*,\s*(\d+)/", $area["coords"], $matches, PREG_PATTERN_ORDER);
                if (count($matches[0]) == 2) {
                    $obj = "line";
                }
                $convert_cmd .= "-stroke " . $area["bordercolor"] . " -fill " . $area["fillcolor"] . " -strokewidth $this->linewidth_outer -draw \"$obj ";
                for ($i = 0; $i < count($matches[0]); $i++) {
                    $convert_cmd .= $matches[1][$i] . "," . $matches[2][$i] . " ";
                }
                $convert_cmd .= "\" ";
                $convert_cmd .= "-stroke " . $area["linecolor"] . " -fill " . $area["fillcolor"] . " -strokewidth $this->linewidth_inner -draw \"$obj ";
                preg_match_all("/(\d+)\s*,\s*(\d+)/", $area["coords"], $matches, PREG_PATTERN_ORDER);
                for ($i = 0; $i < count($matches[0]); $i++) {
                    $convert_cmd .= $matches[1][$i] . "," . $matches[2][$i] . " ";
                }
                $convert_cmd .= "\" ";
            }
        }

        $source = $this->escapeShellCmd($this->imagemap_filename);
        $target = $this->escapeShellCmd($this->preview_filename);
        $convert_cmd = $this->escapeShellCmd($convert_cmd);
        $convert_cmd = preg_replace('/\\\\(#([a-fA-F0-9]{3}|[a-fA-F0-9]{6}|[a-fA-F0-9]{8}))/', '${1}', $convert_cmd);
        $convert_cmd = $source . "[0] " . $convert_cmd . " " . $target;
        $this->execQuoted(PATH_TO_CONVERT, $convert_cmd);
    }

    public static function escapeShellCmd($a_arg)
    {
        if (ini_get('safe_mode') == 1) {
            return $a_arg;
        }
        setlocale(LC_CTYPE, "UTF8", "en_US.UTF-8"); // fix for PHP escapeshellcmd bug. See: http://bugs.php.net/bug.php?id=45132
        return escapeshellcmd($a_arg);
    }

    public static function execQuoted($cmd, $args = null)
    {
        global $DIC;

        if (ilUtil::isWindows() && strpos($cmd, " ") !== false && substr($cmd, 0, 1) !== '"') {
            $cmd = '"' . $cmd . '"';
            if ($args) {
                $cmd .= " " . $args;
            }
        } elseif ($args) {
            $cmd .= " " . $args;
        }
        exec($cmd, $arr);

        $DIC->logger()->root()->debug("ilUtil::execQuoted: " . $cmd . ".");

        return $arr;
    }

    public function getPreviewFilename(
        string $image_path,
        string $base_file_name
    ): string {
        if (count($this->areas) + count($this->points) < 1) {
            return $base_file_name;
        }

        $preview_file = $this->preview_filename;
        if (!is_file($preview_file)) {
            $this->main_tpl->setOnScreenMessage('info', $this->lng->txt("qpl_imagemap_preview_missing"));
            return $base_file_name;
        }

        $ident = $this->getAreaIdent();
        $requested_preview_file = $image_path . $ident . $base_file_name;
        if ((!is_file($requested_preview_file)
            || md5_file($requested_preview_file) !== md5_file($preview_file))
            && $ident !== '') {
            copy($preview_file, $requested_preview_file);
        }
        unlink($preview_file);
        return basename($requested_preview_file);
    }

    /**
    * get imagemap html code
    * note: html code should be placed in template files
    */
    public function getImagemap($title): string
    {
        $map = "<map name=\"$title\"> ";
        foreach ($this->areas as $area) {
            $map .= "<area alt=\"" . $area["title"] . "\"  title=\"" . $area["title"] . "\" ";
            $map .= "shape=\"" . $area["shape"] . "\" ";
            $map .= "coords=\"" . $area["coords"] . "\" ";
            if ($area["href"]) {
                $map .= "href=\"" . $area["href"] . "\" ";
                if ($area["target"]) {
                    $map .= "target=\"" . $area["target"] . "\" ";
                }
                $map .= "/>\n";
            } else {
                $map .= "nohref />\n";
            }
        }
        $map .= "</map>";
        return $map;
    }
}
