<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

use ILIAS\ResourceStorage\Identification\ResourceIdentification;
use ILIAS\ResourceStorage\Services as IRSS;

/**
 * @author  Niels Theen <ntheen@databay.de>
 */
class ilCertificateTemplatePreviewAction
{
    private readonly ilObjUser $user;
    private readonly ilCertificateUtilHelper $utilHelper;
    private readonly ilCertificateUserDefinedFieldsHelper $userDefinedFieldsHelper;
    private readonly ilCertificateRpcClientFactoryHelper $rpcClientFactoryHelper;
    private readonly ilCertificatePdfFileNameFactory $pdfFileNameFactory;

    public function __construct(
        private readonly ilCertificateTemplateRepository $templateRepository,
        private readonly ilCertificatePlaceholderValues $placeholderValuesObject,
        private readonly IRSS $irss,
        private readonly string $rootDirectory = CLIENT_WEB_DIR,
        ?ilObjUser $user = null,
        ?ilCertificateUtilHelper $utilHelper = null,
        ?ilCertificateUserDefinedFieldsHelper $userDefinedFieldsHelper = null,
        ?ilCertificateRpcClientFactoryHelper $rpcClientFactoryHelper = null,
        ?ilCertificatePdfFileNameFactory $pdfFileNameFactory = null
    ) {
        global $DIC;
        if (null === $user) {
            $user = $DIC->user();
        }
        $this->user = $user;

        if (null === $utilHelper) {
            $utilHelper = new ilCertificateUtilHelper();
        }
        $this->utilHelper = $utilHelper;

        if (null === $userDefinedFieldsHelper) {
            $userDefinedFieldsHelper = new ilCertificateUserDefinedFieldsHelper();
        }
        $this->userDefinedFieldsHelper = $userDefinedFieldsHelper;

        if (null === $rpcClientFactoryHelper) {
            $rpcClientFactoryHelper = new ilCertificateRpcClientFactoryHelper();
        }
        $this->rpcClientFactoryHelper = $rpcClientFactoryHelper;

        if (null === $pdfFileNameFactory) {
            $pdfFileNameFactory = new ilCertificatePdfFileNameFactory($DIC->language());
        }
        $this->pdfFileNameFactory = $pdfFileNameFactory;
    }

    /**
     * @throws Exception
     */
    public function createPreviewPdf(int $objectId): void
    {
        $template = $this->templateRepository->fetchCurrentlyUsedCertificate($objectId);

        $xslfo = $template->getCertificateContent();

        $xslfo = $this->exchangeCertificateVariables($xslfo, $template, $objectId);

        $pdf_base64 = $this->rpcClientFactoryHelper
            ->ilFO2PDF('RPCTransformationHandler', $xslfo)
        ;

        $pdfPresentation = new ilUserCertificatePresentation(
            $template->getObjId(),
            $template->getObjType(),
            null,
            '',
            ''
        );

        $this->utilHelper->deliverData(
            $pdf_base64->scalar,
            $this->pdfFileNameFactory->create($pdfPresentation),
            'application/pdf'
        );
    }

    /**
     * Exchanges the variables in the certificate text with given values
     * @param  string $certificate_text The XSL-FO certificate text
     * @return string XSL-FO code
     */
    private function exchangeCertificateVariables(
        string $certificate_text,
        ilCertificateTemplate $template,
        int $objectId
    ): string {
        $insert_tags = $this->placeholderValuesObject->getPlaceholderValuesForPreview($this->user->getId(), $objectId);

        foreach ($this->getCustomCertificateFields() as $value) {
            $insert_tags[$value['ph']] = $this->utilHelper->prepareFormOutput($value['name']);
        }

        foreach ($insert_tags as $placeholderVariable => $value) {
            $certificate_text = str_replace('[' . $placeholderVariable . ']', $value, $certificate_text);
        }

        $certificate_text = str_replace(
            '[CLIENT_WEB_DIR]',
            'file://' . $this->rootDirectory,
            $certificate_text
        );

        $identification = $this->irss->manage()->find($template->getBackgroundImageIdentification());
        if ($identification instanceof ResourceIdentification) {
            $backgroundImagePath = $this->irss->consume()->src($identification)->getSrc(true);
        }

        return str_replace(
            '[BACKGROUND_IMAGE]',
            $backgroundImagePath ?? '',
            $certificate_text
        );
    }

    /**
     * @return array<int, array{name: string, ph: string}>
     */
    private function getCustomCertificateFields(): array
    {
        $user_field_definitions = $this->userDefinedFieldsHelper->createInstance();
        $fds = $user_field_definitions->getDefinitions();

        $fields = [];
        foreach ($fds as $f) {
            if ($f['certificate']) {
                $fields[$f['field_id']] = [
                    'name' => $f['field_name'],
                    'ph' => '[#' . str_replace(' ', '_', strtoupper($f['field_name'])) . ']'
                ];
            }
        }

        return $fields;
    }
}
