<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

/**
 * Class ilChatroomXMLWriter
 */
class ilChatroomXMLWriter extends ilXmlWriter
{
    protected ilChatroom $room;

    public function __construct(protected ilObjChatroom $chat)
    {
        $this->room = ilChatroom::byObjectId($this->chat->getId());
        if (!$this->room) {
            $this->room = new ilChatroom();
            $this->room->setSetting('object_id', $this->chat->getId());
        }

        parent::__construct();
    }

    public function start(): void
    {
        $this->xmlStartTag('Chatroom');

        $this->xmlElement('ObjId', null, (string) $this->chat->getId());
        $this->xmlElement('RoomId', null, (string) $this->room->getRoomId());
        $this->xmlElement('Title', null, $this->chat->getTitle());
        $this->xmlElement('Description', null, $this->chat->getDescription());
        $this->xmlElement('OnlineStatus', null, (string) ((int) (!$this->chat->getOfflineStatus())));
        $this->xmlElement('AllowAnonymousAccess', null, (string) ((int) $this->room->getSetting('allow_anonymous')));
        $this->xmlElement('AllowCustomUsernames', null, (string) ((int) $this->room->getSetting('allow_custom_usernames')));
        $this->xmlElement('EnableHistory', null, (string) ((int) $this->room->getSetting('enable_history')));
        $this->xmlElement('DisplayPastMessages', null, (string) ((int) $this->room->getSetting('display_past_msgs')));
        $this->xmlElement('AutoGeneratedUsernameSchema', null, (string) $this->room->getSetting('autogen_usernames'));

        if ($this->room->getRoomId() > 0) {
            $this->exportData();
        }

        $this->xmlEndTag('Chatroom');
    }

    private function exportData(): void
    {
        $history = $this->room->getHistory(null, null, null, false);
        if (count($history) > 0) {
            $this->xmlStartTag('Messages');
            foreach ($history as $entry) {
                $this->xmlStartTag('Message');
                $this->xmlElement('Body', null, json_encode($entry['message'], JSON_THROW_ON_ERROR));
                $this->xmlElement('CreatedTimestamp', null, (string) ((int) $entry['timestamp']));
                $this->xmlEndTag('Message');
            }
            $this->xmlEndTag('Messages');
        }
    }

    public function getXML(): string
    {
        // Replace ascii code 11 characters because of problems with xml sax parser
        return str_replace('&#11;', '', $this->xmlDumpMem(false));
    }
}
