<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

namespace ILIAS\Test\Participants;

use ILIAS\Test\Logging\AdditionalInformationGenerator;
use ILIAS\Test\Logging\TestAdministrationInteractionTypes;
use ILIAS\Language\Language;
use ILIAS\Test\Results\Data\StatusOfAttempt;
use ILIAS\Test\Results\Data\Repository as TestResultRepository;
use ILIAS\UI\Factory as UIFactory;
use ILIAS\UI\Component\Modal\Modal;
use ILIAS\UI\Component\Table\Action\Standard as StandardAction;
use ILIAS\UI\URLBuilder;
use ILIAS\UI\URLBuilderToken;
use Psr\Http\Message\ServerRequestInterface;

class ParticipantTableFinishTestAction implements TableAction
{
    public const ACTION_ID = 'finish_test';

    public function __construct(
        private readonly Language $lng,
        private readonly \ilGlobalTemplateInterface $tpl,
        private readonly UIFactory $ui_factory,
        private readonly \ilDBInterface $db,
        private readonly \ilTestProcessLockerFactory $process_locker_factory,
        private readonly \ilObjUser $user,
        private readonly \ilTestAccess $test_access,
        private readonly \ilObjTest $test_obj,
        private readonly TestResultRepository $test_pass_result_repository
    ) {
    }

    public function getActionId(): string
    {
        return self::ACTION_ID;
    }

    public function isAvailable(): bool
    {
        return $this->test_access->checkManageParticipantsAccess();
    }

    public function getTableAction(
        URLBuilder $url_builder,
        URLBuilderToken $row_id_token,
        URLBuilderToken $action_token,
        URLBuilderToken $action_type_token
    ): StandardAction {
        return $this->ui_factory->table()->action()->standard(
            $this->lng->txt(self::ACTION_ID),
            $url_builder
                ->withParameter($action_token, self::ACTION_ID)
                ->withParameter($action_type_token, ParticipantTableActions::SHOW_ACTION),
            $row_id_token
        )->withAsync();
    }

    public function getModal(
        URLBuilder $url_builder,
        array $selected_participants,
        bool $all_participants_selected
    ): ?Modal {
        $modal = $this->ui_factory->modal()->interruptive(
            $this->lng->txt('finish_test'),
            $this->resolveMessage($selected_participants, $all_participants_selected),
            $url_builder->buildURI()->__toString()
        )->withActionButtonLabel($this->lng->txt('finish_test'));

        if (count($selected_participants) > 1) {
            $modal = $modal->withAffectedItems(
                array_map(
                    fn(Participant $participant) => $this->ui_factory->modal()->interruptiveItem()->standard(
                        (string) $participant->getUserId(),
                        (new \ilObjUser($participant->getUserId()))->getPublicName()
                    ),
                    $selected_participants
                )
            );
        }

        return $modal;
    }

    public function onSubmit(
        URLBuilder $url_builder,
        ServerRequestInterface $request,
        array $selected_participants,
        bool $all_participants_selected
    ): ?Modal {
        if (!$this->test_access->checkManageParticipantsAccess()) {
            $this->tpl->setOnScreenMessage(
                \ilGlobalTemplateInterface::MESSAGE_TYPE_FAILURE,
                $this->lng->txt('no_permission'),
                true
            );
            return null;
        }

        if (count($selected_participants) > 1
            && $this->test_obj->getNrOfTries() === 1
            && $this->test_obj->getEnableProcessingTime()
            && !$this->test_obj->getResetProcessingTime()
            && !$this->haveAllSelectedParticipantsReachedMaxProcessingTime($selected_participants)) {
            $this->tpl->setOnScreenMessage(
                \ilGlobalTemplateInterface::MESSAGE_TYPE_FAILURE,
                $this->lng->txt('finish_pass_for_multiple_users_in_processing_time'),
                true
            );
            return null;
        }

        // This is required here because of late test object binding
        $test_session_factory = new \ilTestSessionFactory(
            $this->test_obj,
            $this->db,
            $this->user
        );

        foreach ($selected_participants as $participant) {
            $process_locker = $this->process_locker_factory->withContextId($participant->getActiveId())->getLocker();
            (new \ilTestPassFinishTasks(
                $test_session_factory->getSession($participant->getActiveId()),
                $this->test_obj,
                $this->test_pass_result_repository
            ))->performFinishTasks($process_locker, StatusOfAttempt::FINISHED_BY_ADMINISTRATOR);
        }

        $logger = $this->test_obj->getTestLogger();
        if ($logger->isLoggingEnabled()) {
            $logger->logTestAdministrationInteraction(
                $logger->getInteractionFactory()->buildTestAdministrationInteraction(
                    $this->test_obj->getRefId(),
                    $this->user->getId(),
                    TestAdministrationInteractionTypes::TEST_RUN_OF_PARTICIPANT_CLOSED,
                    [
                        AdditionalInformationGenerator::KEY_USERS => array_map(
                            fn(Participant $participant) => $participant->getUserId(),
                            $selected_participants
                        )
                    ]
                )
            );
        }

        $this->tpl->setOnScreenMessage(
            \ilGlobalTemplateInterface::MESSAGE_TYPE_SUCCESS,
            $this->lng->txt('test_attempts_finished'),
            true
        );
        return null;
    }

    public function allowActionForRecord(Participant $record): bool
    {
        return $record->hasUnfinishedAttempts();
    }

    private function resolveMessage(
        array $selected_participants,
        bool $all_participants_selected
    ): string {
        if ($all_participants_selected) {
            return $this->lng->txt('finish_test_all');
        }

        if (count($selected_participants) === 1) {
            return sprintf(
                $this->lng->txt('finish_test_single'),
                (new \ilObjUser($selected_participants[0]->getUserId()))->getPublicName()
            );
        }

        return $this->lng->txt('finish_test_multiple');
    }

    public function getSelectionErrorMessage(): ?string
    {
        return $this->lng->txt('finish_test_no_valid_participants_selected');
    }

    private function haveAllSelectedParticipantsReachedMaxProcessingTime(array $selected_participants): bool
    {
        foreach ($selected_participants as $participant) {
            if (!$participant->hasUnfinishedAttempts()
                || !$this->test_obj->isMaxProcessingTimeReached(
                    $this->test_obj->getStartingTimeOfUser($participant->getActiveId()),
                    $participant->getActiveId()
                )) {
                return false;
            }
        }
        return true;
    }
}
