<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

use ILIAS\TestQuestionPool\Questions\QuestionLMExportable;
use ILIAS\TestQuestionPool\Questions\QuestionAutosaveable;
use ILIAS\TestQuestionPool\ManipulateImagesInChoiceQuestionsTrait;
use ILIAS\Test\Logging\AdditionalInformationGenerator;
use ILIAS\TestQuestionPool\RequestDataCollector;

/**
 * Class for multiple choice tests.
 *
 * assMultipleChoice is a class for multiple choice questions.
 *
 * @extends assQuestion
 *
 * @author		Helmut Schottmüller <helmut.schottmueller@mac.com>
 * @author		Björn Heyser <bheyser@databay.de>
 * @author		Maximilian Becker <bheyser@databay.de>
 *
 * @version		$Id$
 *
 * @ingroup		ModulesTestQuestionPool
 */
class assMultipleChoice extends assQuestion implements ilObjQuestionScoringAdjustable, ilObjAnswerScoringAdjustable, iQuestionCondition, ilAssSpecificFeedbackOptionLabelProvider, QuestionLMExportable, QuestionAutosaveable
{
    use ManipulateImagesInChoiceQuestionsTrait;

    public const OUTPUT_ORDER = 0;
    public const OUTPUT_RANDOM = 1;

    public array $answers = [];
    public bool $is_singleline = true;
    public int $feedback_setting = 0;
    protected ?int $selection_limit = null;

    public function setIsSingleline(bool $is_singleline): void
    {
        $this->is_singleline = $is_singleline;
    }

    /**
     * assMultipleChoice constructor
     *
     * The constructor takes possible arguments an creates an instance of the assMultipleChoice object.
     *
     * @param string     $title       A title string to describe the question
     * @param string     $comment     A comment string to describe the question
     * @param string     $author      A string containing the name of the questions author
     * @param integer    $owner       A numerical ID to identify the owner/creator
     * @param string     $question    The question string of the multiple choice question
     * @param int|string $output_type The output order of the multiple choice answers
     *
     * @see assQuestion:assQuestion()
     */
    public function __construct(
        string $title = "",
        string $comment = "",
        string $author = "",
        int $owner = -1,
        string $question = "",
        private int $output_type = self::OUTPUT_ORDER
    ) {
        parent::__construct($title, $comment, $author, $owner, $question);
        $this->answers = [];
        $this->shuffle = true;
    }

    public function getSelectionLimit(): ?int
    {
        return $this->selection_limit;
    }

    public function setSelectionLimit(?int $selection_limit): void
    {
        $this->selection_limit = $selection_limit;
    }

    public function isComplete(): bool
    {
        return $this->title !== ''
            && $this->author !== ''
            && $this->question !== ''
            && $this->getAnswerCount() > 0
            && $this->getMaximumPoints() >= 0;
    }

    public function saveToDb(?int $original_id = null): void
    {
        $this->saveQuestionDataToDb($original_id);
        $this->saveAdditionalQuestionDataToDb();
        $this->saveAnswerSpecificDataToDb();
        parent::saveToDb($original_id);
    }

    public function loadFromDb(int $question_id): void
    {
        $result = $this->db->queryF(
            "SELECT qpl_questions.*, " . $this->getAdditionalTableName() . ".* FROM qpl_questions LEFT JOIN " . $this->getAdditionalTableName() . " ON " . $this->getAdditionalTableName() . ".question_fi = qpl_questions.question_id WHERE qpl_questions.question_id = %s",
            ["integer"],
            [$question_id]
        );
        if ($result->numRows() == 1) {
            $data = $this->db->fetchAssoc($result);
            $this->setId($question_id);
            $this->setObjId($data["obj_fi"]);
            $this->setTitle($data["title"] ?? '');
            $this->setNrOfTries($data['nr_of_tries']);
            $this->setComment($data["description"] ?? '');
            $this->setOriginalId($data["original_id"]);
            $this->setAuthor($data["author"]);
            $this->setPoints($data["points"]);
            $this->setOwner($data["owner"]);
            $this->setQuestion(ilRTE::_replaceMediaObjectImageSrc($data["question_text"] ?? '', 1));
            $shuffle = (is_null($data['shuffle'])) ? true : $data['shuffle'];
            $this->setShuffle((bool) $shuffle);
            if ($data['thumb_size'] !== null && $data['thumb_size'] >= $this->getMinimumThumbSize()) {
                $this->setThumbSize($data['thumb_size']);
            }
            $this->is_singleline = $data['allow_images'] === null || $data['allow_images'] === '0';
            $this->lastChange = $data['tstamp'];
            $this->setSelectionLimit((int) $data['selection_limit'] > 0 ? (int) $data['selection_limit'] : null);
            if (isset($data['feedback_setting'])) {
                $this->feedback_setting = $data['feedback_setting'];
            }

            try {
                $this->setLifecycle(ilAssQuestionLifecycle::getInstance($data['lifecycle']));
            } catch (ilTestQuestionPoolInvalidArgumentException $e) {
                $this->setLifecycle(ilAssQuestionLifecycle::getDraftInstance());
            }

            try {
                $this->setAdditionalContentEditingMode($data['add_cont_edit_mode']);
            } catch (ilTestQuestionPoolException $e) {
            }
        }

        $result = $this->db->queryF(
            "SELECT * FROM qpl_a_mc WHERE question_fi = %s ORDER BY aorder ASC",
            ['integer'],
            [$question_id]
        );
        if ($result->numRows() > 0) {
            while ($data = $this->db->fetchAssoc($result)) {
                $imagefilename = $this->getImagePath() . $data["imagefile"];
                if (!file_exists($imagefilename)) {
                    $data["imagefile"] = null;
                }
                $data["answertext"] = ilRTE::_replaceMediaObjectImageSrc($data["answertext"] ?? '', 1);

                $answer = new ASS_AnswerMultipleResponseImage(
                    $data["answertext"],
                    $data["points"],
                    $data["aorder"],
                    $data["answer_id"]
                );
                $answer->setPointsUnchecked($data["points_unchecked"]);
                $answer->setImage($data["imagefile"] ? $data["imagefile"] : null);
                array_push($this->answers, $answer);
            }
        }

        parent::loadFromDb($question_id);
    }

    protected function cloneQuestionTypeSpecificProperties(
        \assQuestion $target
    ): \assQuestion {
        $this->cloneImages(
            $this->getId(),
            $this->getObjId(),
            $target->getId(),
            $target->getObjId(),
            $this->getAnswers()
        );
        return $target;
    }

    /**
     * Adds a possible answer for a multiple choice question. A ASS_AnswerBinaryStateImage object will be
     * created and assigned to the array $this->answers.
     *
     * @param string  $answertext 		The answer text
     * @param double  $points     		The points for selecting the answer (even negative points can be used)
     * @param float   $points_unchecked The points for not selecting the answer (even positive points can be used)
     * @param integer $order      		A possible display order of the answer
     * @param string  $answerimage
     * @param int     $answer_id        The Answer id used in the database
     *
     * @see      $answers
     * @see      ASS_AnswerBinaryStateImage
     */
    public function addAnswer(
        string $answertext = '',
        float $points = 0.0,
        float $points_unchecked = 0.0,
        int $order = 0,
        ?string $answerimage = null,
        int $answer_id = -1
    ): void {
        if (array_key_exists($order, $this->answers)) {
            // insert answer
            $answer = new ASS_AnswerMultipleResponseImage(
                $this->getHtmlQuestionContentPurifier()->purify($answertext),
                $points,
                $order,
                -1,
                0
            );
            $answer->setPointsUnchecked($points_unchecked);
            $answer->setImage($answerimage);
            $newchoices = [];
            for ($i = 0; $i < $order; $i++) {
                $newchoices[] = $this->answers[$i];
            }
            $newchoices[] = $answer;
            for ($i = $order, $iMax = count($this->answers); $i < $iMax; $i++) {
                $changed = $this->answers[$i];
                $changed->setOrder($i + 1);
                $newchoices[] = $changed;
            }
            $this->answers = $newchoices;
            return;
        } else {
            $answer = new ASS_AnswerMultipleResponseImage(
                $this->getHtmlQuestionContentPurifier()->purify($answertext),
                $points,
                count($this->answers),
                $answer_id,
                0
            );
            $answer->setPointsUnchecked($points_unchecked);
            $answer->setImage($answerimage);
            $this->answers[] = $answer;
        }
    }

    /**
     * Returns the number of answers
     *
     * @return integer The number of answers of the multiple choice question
     * @see $answers
     */
    public function getAnswerCount(): int
    {
        return count($this->answers);
    }

    /**
     * Returns an answer with a given index. The index of the first
     * answer is 0, the index of the second answer is 1 and so on.
     *
     * @param integer $index A nonnegative index of the n-th answer
     * @return object ASS_AnswerBinaryStateImage-Object containing the answer
     * @see $answers
    */
    public function getAnswer($index = 0): ?object
    {
        if ($index < 0) {
            return null;
        }
        if (count($this->answers) < 1) {
            return null;
        }
        if ($index >= count($this->answers)) {
            return null;
        }

        return $this->answers[$index];
    }

    /**
     * Deletes an answer with a given index. The index of the first
     * answer is 0, the index of the second answer is 1 and so on.
     *
     * @param integer $index A nonnegative index of the n-th answer
     * @see $answers
     */
    public function deleteAnswer($index = 0): void
    {
        if ($index < 0) {
            return;
        }
        if (count($this->answers) < 1) {
            return;
        }
        if ($index >= count($this->answers)) {
            return;
        }
        $answer = $this->answers[$index];
        if ($answer->hasImage()) {
            $this->deleteImage($answer->getImage());
        }
        unset($this->answers[$index]);
        $this->answers = array_values($this->answers);
        for ($i = 0, $iMax = count($this->answers); $i < $iMax; $i++) {
            if ($this->answers[$i]->getOrder() > $index) {
                $this->answers[$i]->setOrder($i);
            }
        }
    }

    /**
     * Deletes all answers
     *
     * @see $answers
     */
    public function flushAnswers(): void
    {
        $this->answers = [];
    }

    /**
     * Returns the maximum points, a learner can reach answering the question
     *
     * @see $points
     */
    public function getMaximumPoints(): float
    {
        $total_max_points = 0.0;
        foreach ($this->getAnswers() as $answer) {
            $total_max_points += max($answer->getPointsChecked(), $answer->getPointsUnchecked());
        }
        return $total_max_points;
    }

    public function calculateReachedPoints(
        int $active_id,
        ?int $pass = null,
        bool $authorized_solution = true
    ): float {
        $found_values = [];
        if ($pass === null) {
            $pass = $this->getSolutionMaxPass($active_id);
        }
        $result = $this->getCurrentSolutionResultSet($active_id, $pass, $authorized_solution);
        while ($data = $this->db->fetchAssoc($result)) {
            if ($data['value1'] !== '') {
                array_push($found_values, $data['value1']);
            }
        }

        return $this->calculateReachedPointsForSolution($found_values, $active_id);
    }

    public function validateSolutionSubmit(): bool
    {
        $submit = $this->getSolutionSubmit();

        if ($this->getSelectionLimit()) {
            if (count($submit) > $this->getSelectionLimit()) {
                $failureMsg = sprintf(
                    $this->lng->txt('ass_mc_sel_lim_exhausted_hint'),
                    $this->getSelectionLimit(),
                    $this->getAnswerCount()
                );

                $this->tpl->setOnScreenMessage('failure', $failureMsg, true);
                return false;
            }
        }

        return true;
    }

    protected function isForcedEmptySolution(array $solutionSubmit): bool
    {
        $tst_force_form_diff_input = $this->questionpool_request->strArray('tst_force_form_diff_input');
        return !count($solutionSubmit) && !empty($tst_force_form_diff_input);
    }

    public function saveWorkingData(
        int $active_id,
        ?int $pass = null,
        bool $authorized = true
    ): bool {
        $pass = $pass ?? ilObjTest::_getPass($active_id);

        $answer = $this->getSolutionSubmit();
        $this->getProcessLocker()->executeUserSolutionUpdateLockOperation(
            function () use ($answer, $active_id, $pass, $authorized) {
                $this->removeCurrentSolution($active_id, $pass, $authorized);

                foreach ($answer as $value) {
                    if ($value !== '') {
                        $this->saveCurrentSolution($active_id, $pass, $value, null, $authorized);
                    }
                }

                // fau: testNav - write a dummy entry for the evil mc questions with "None of the above" checked
                if ($this->isForcedEmptySolution($answer)) {
                    $this->saveCurrentSolution($active_id, $pass, 'mc_none_above', null, $authorized);
                }
                // fau.
            }
        );

        return true;
    }

    public function saveAdditionalQuestionDataToDb()
    {
        if (!$this->is_singleline) {
            ilFileUtils::delDir($this->getImagePath());
        }

        // save additional data
        $this->db->replace(
            $this->getAdditionalTableName(),
            [
                'shuffle' => ['text', $this->getShuffle()],
                'allow_images' => ['text', $this->is_singleline ? 0 : 1],
                'thumb_size' => ['integer', $this->getThumbSize()],
                'selection_limit' => ['integer', $this->getSelectionLimit()],
                'feedback_setting' => ['integer', $this->getSpecificFeedbackSetting()]
            ],
            ['question_fi' => ['integer', $this->getId()]]
        );
    }

    public function saveAnswerSpecificDataToDb(): void
    {
        // Get all feedback entries
        $result = $this->db->queryF(
            "SELECT * FROM qpl_fb_specific WHERE question_fi = %s",
            ['integer'],
            [$this->getId()]
        );
        $db_feedback = $this->db->fetchAll($result);

        // Check if feedback exists and the regular editor is used and not the page editor
        if (sizeof($db_feedback) >= 1 && $this->getAdditionalContentEditingMode() == 'default') {
            // Get all existing answer data for question
            $result = $this->db->queryF(
                "SELECT answer_id, aorder  FROM qpl_a_mc WHERE question_fi = %s",
                ['integer'],
                [$this->getId()]
            );
            $db_answers = $this->db->fetchAll($result);

            // Collect old and new order entries by ids and order to calculate a diff/intersection and remove/update feedback
            $post_answer_order_for_id = [];
            foreach ($this->answers as $answer) {
                // Only the first appearance of an id is used
                if ($answer->getId() !== null && !in_array($answer->getId(), array_keys($post_answer_order_for_id))) {
                    // -1 is happening while import and also if a new multi line answer is generated
                    if ($answer->getId() == -1) {
                        continue;
                    }
                    $post_answer_order_for_id[$answer->getId()] = $answer->getOrder();
                }
            }

            // If there is no usable ids from post, it's better to not touch the feedback
            // This is useful since the import is also using this function or the first creation of a new question in general
            if (sizeof($post_answer_order_for_id) >= 1) {
                $db_answer_order_for_id = [];
                $db_answer_id_for_order = [];
                foreach ($db_answers as $db_answer) {
                    $db_answer_order_for_id[intval($db_answer['answer_id'])] = intval($db_answer['aorder']);
                    $db_answer_id_for_order[intval($db_answer['aorder'])] = intval($db_answer['answer_id']);
                }

                // Handle feedback
                // the diff between the already existing answer ids from the Database and the answer ids from post
                // feedback related to the answer ids should be deleted or in our case not recreated.
                $db_answer_ids = array_keys($db_answer_order_for_id);
                $post_answer_ids = array_keys($post_answer_order_for_id);
                $diff_db_post_answer_ids = array_diff($db_answer_ids, $post_answer_ids);
                $unused_answer_ids = array_keys($diff_db_post_answer_ids);

                // Delete all feedback in the database
                $this->feedbackOBJ->deleteSpecificAnswerFeedbacks($this->getId(), false);
                // Recreate feedback
                foreach ($db_feedback as $feedback_option) {
                    // skip feedback which answer is deleted
                    if (in_array(intval($feedback_option['answer']), $unused_answer_ids)) {
                        continue;
                    }

                    // Reorder feedback
                    $feedback_order_db = intval($feedback_option['answer']);
                    $db_answer_id = $db_answer_id_for_order[$feedback_order_db] ?? null;
                    // This cuts feedback that currently would have no corresponding answer
                    // This case can happen while copying "broken" questions
                    // Or when saving a question with less answers than feedback
                    if (is_null($db_answer_id) || $db_answer_id < 0) {
                        continue;
                    }
                    $feedback_order_post = $post_answer_order_for_id[$db_answer_id];
                    $feedback_option['answer'] = $feedback_order_post;

                    // Recreate remaining feedback in database
                    $next_id = $this->db->nextId('qpl_fb_specific');
                    $this->db->manipulateF(
                        "INSERT INTO qpl_fb_specific (feedback_id, question_fi, answer, tstamp, feedback, question)
                            VALUES (%s, %s, %s, %s, %s, %s)",
                        ['integer', 'integer', 'integer', 'integer', 'text', 'integer'],
                        [
                            $next_id,
                            $feedback_option['question_fi'],
                            $feedback_option['answer'],
                            time(),
                            $feedback_option['feedback'],
                            $feedback_option['question']
                        ]
                    );
                }
            }
        }

        // Delete all entries in qpl_a_mc for question
        $this->db->manipulateF(
            "DELETE FROM qpl_a_mc WHERE question_fi = %s",
            ['integer'],
            [$this->getId()]
        );

        // Recreate answers one by one
        foreach ($this->answers as $key => $value) {
            $answer_obj = $this->answers[$key];
            $next_id = $this->db->nextId('qpl_a_mc');
            $this->db->manipulateF(
                "INSERT INTO qpl_a_mc (answer_id, question_fi, answertext, points, points_unchecked, aorder, imagefile, tstamp)
                        VALUES (%s, %s, %s, %s, %s, %s, %s, %s)",
                ['integer', 'integer', 'text', 'float', 'float', 'integer', 'text', 'integer'],
                [
                    $next_id,
                    $this->getId(),
                    ilRTE::_replaceMediaObjectImageSrc($answer_obj->getAnswertext(), 0),
                    $answer_obj->getPoints(),
                    $answer_obj->getPointsUnchecked(),
                    $answer_obj->getOrder(),
                    $answer_obj->getImage(),
                    time()
                ]
            );
        }
    }

    /**
     * Returns the question type of the question
     *
     * @return integer The question type of the question
     */
    public function getQuestionType(): string
    {
        return "assMultipleChoice";
    }

    /**
     * Returns the name of the additional question data table in the database
     *
     * @return string The additional table name
     */
    public function getAdditionalTableName(): string
    {
        return "qpl_qst_mc";
    }

    /**
     * Returns the name of the answer table in the database
     *
     * @return string The answer table name
     */
    public function getAnswerTableName(): string
    {
        return "qpl_a_mc";
    }

    /**
     * Sets the image file and uploads the image to the object's image directory.
     *
     * @param string $image_filename Name of the original image file
     * @param string $image_tempfilename Name of the temporary uploaded image file
     * @return integer An errorcode if the image upload fails, 0 otherwise
     */
    public function setImageFile($image_filename, $image_tempfilename = ""): int
    {
        $result = 0;
        if (!empty($image_tempfilename)) {
            $image_filename = str_replace(" ", "_", $image_filename);
            $imagepath = $this->getImagePath();
            if (!file_exists($imagepath)) {
                ilFileUtils::makeDirParents($imagepath);
            }
            if (!ilFileUtils::moveUploadedFile($image_tempfilename, $image_filename, $imagepath . $image_filename)) {
                $result = 2;
            } else {
                $mimetype = ilObjMediaObject::getMimeType($imagepath . $image_filename);
                if (!preg_match("/^image/", $mimetype)) {
                    unlink($imagepath . $image_filename);
                    $result = 1;
                } else {
                    // create thumbnail file
                    if ($this->is_singleline && ($this->getThumbSize())) {
                        $this->generateThumbForFile(
                            $image_filename,
                            $this->getImagePath(),
                            $this->getThumbSize()
                        );
                    }
                }
            }
        }
        return $result;
    }

    public function getRTETextWithMediaObjects(): string
    {
        $text = parent::getRTETextWithMediaObjects();
        foreach ($this->answers as $index => $answer) {
            $text .= $this->feedbackOBJ->getSpecificAnswerFeedbackContent($this->getId(), 0, $index);
            $answer_obj = $this->answers[$index];
            $text .= $answer_obj->getAnswertext();
        }
        return $text;
    }

    /**
    * Returns a reference to the answers array
    */
    public function &getAnswers(): array
    {
        return $this->answers;
    }

    public function setAnswers(array $answers): void
    {
        $this->answers = $answers;
    }

    /**
     * @param ilAssSelfAssessmentMigrator $migrator
     */
    protected function lmMigrateQuestionTypeSpecificContent(ilAssSelfAssessmentMigrator $migrator): void
    {
        foreach ($this->getAnswers() as $answer) {
            /* @var ASS_AnswerBinaryStateImage $answer */
            $answer->setAnswertext($migrator->migrateToLmContent($answer->getAnswertext()));
        }
    }

    /**
     * Returns a JSON representation of the question
     */
    public function toJSON(): string
    {
        $result = [];
        $result['id'] = $this->getId();
        $result['type'] = (string) $this->getQuestionType();
        $result['title'] = $this->getTitleForHTMLOutput();
        $result['question'] = $this->formatSAQuestion($this->getQuestion());
        $result['nr_of_tries'] = $this->getNrOfTries();
        $result['shuffle'] = $this->getShuffle();
        $result['selection_limit'] = (int) $this->getSelectionLimit();
        $result['feedback'] = [
            'onenotcorrect' => $this->formatSAQuestion($this->feedbackOBJ->getGenericFeedbackTestPresentation($this->getId(), false)),
            'allcorrect' => $this->formatSAQuestion($this->feedbackOBJ->getGenericFeedbackTestPresentation($this->getId(), true))
        ];

        $answers = [];
        $has_image = false;
        foreach ($this->getAnswers() as $key => $answer_obj) {
            if ((string) $answer_obj->getImage()) {
                $has_image = true;
            }
            array_push($answers, [
                "answertext" => $this->formatSAQuestion($answer_obj->getAnswertext()),
                "points_checked" => (float) $answer_obj->getPointsChecked(),
                "points_unchecked" => (float) $answer_obj->getPointsUnchecked(),
                "order" => (int) $answer_obj->getOrder(),
                "image" => (string) $answer_obj->getImage(),
                "feedback" => $this->formatSAQuestion(
                    $this->feedbackOBJ->getSpecificAnswerFeedbackExportPresentation($this->getId(), 0, $key)
                )
            ]);
        }
        $result['answers'] = $answers;

        if ($has_image) {
            $result['path'] = $this->getImagePathWeb();
            $result['thumb'] = $this->getThumbSize();
        }

        $mobs = ilObjMediaObject::_getMobsOfObject("qpl:html", $this->getId());
        $result['mobs'] = $mobs;

        return json_encode($result);
    }

    public function removeAnswerImage($index): void
    {
        $answer = $this->answers[$index];
        if (is_object($answer)) {
            $this->deleteImage($answer->getImage());
            $answer->setImage(null);
        }
    }

    public function getMultilineAnswerSetting(): int
    {
        return $this->current_user->getPref('tst_multiline_answers') === '1' ? 1 : 0;
    }

    public function setMultilineAnswerSetting($setting = 0): void
    {
        $this->current_user->writePref('tst_multiline_answers', (string) $setting);
    }

    /**
     * Sets the feedback settings in effect for the question.
     * Options are:
     * 1 - Feedback is shown for all answer options.
     * 2 - Feedback is shown for all checked/selected options.
     * 3 - Feedback is shown for all correct options.
     *
     * @param integer $a_feedback_setting
     */
    public function setSpecificFeedbackSetting(int $feedback_setting): void
    {
        $this->feedback_setting = $feedback_setting;
    }

    /**
     * Gets the current feedback settings in effect for the question.
     * Values are:
     * 1 - Feedback is shown for all answer options.
     * 2 - Feedback is shown for all checked/selected options.
     * 3 - Feedback is shown for all correct options.
     *
     * @return integer
     */
    public function getSpecificFeedbackSetting(): int
    {
        if ($this->feedback_setting) {
            return $this->feedback_setting;
        } else {
            return 1;
        }
    }

    public function getSpecificFeedbackAllCorrectOptionLabel(): string
    {
        return 'feedback_correct_sc_mc';
    }

    protected function getSolutionSubmit(): array
    {
        $solutionSubmit = [];
        $post = $this->dic->http()->wrapper()->post();

        foreach ($this->getAnswers() as $index => $a) {
            if ($post->has("multiple_choice_result_$index")) {
                $value = $post->retrieve("multiple_choice_result_$index", $this->dic->refinery()->kindlyTo()->string());
                if (is_numeric($value)) {
                    $solutionSubmit[] = $value;
                }
            }
        }
        return $solutionSubmit;
    }

    protected function calculateReachedPointsForSolution(
        ?array $found_values,
        int $active_id = 0
    ): float {
        if ($found_values === []
            && $active_id !== 0) {
            return 0.0;
        }

        $found_values ??= [];
        $points = 0.0;
        foreach ($this->answers as $key => $answer) {
            if (in_array($key, $found_values)) {
                $points += $answer->getPoints();
                continue;
            }
            $points += $answer->getPointsUnchecked();
        }

        return $points;
    }

    public function getOperators(string $expression): array
    {
        return ilOperatorsExpressionMapping::getOperatorsByExpression($expression);
    }

    public function getExpressionTypes(): array
    {
        return [
            iQuestionCondition::PercentageResultExpression,
            iQuestionCondition::NumberOfResultExpression,
            iQuestionCondition::ExclusiveResultExpression,
            iQuestionCondition::EmptyAnswerExpression,
        ];
    }

    public function getUserQuestionResult(
        int $active_id,
        int $pass
    ): ilUserQuestionResult {
        $result = new ilUserQuestionResult($this, $active_id, $pass);

        $maxStep = $this->lookupMaxStep($active_id, $pass);
        if ($maxStep > 0) {
            $data = $this->db->queryF(
                "SELECT value1+1 as value1 FROM tst_solutions WHERE active_fi = %s AND pass = %s AND question_fi = %s AND step = %s",
                ["integer", "integer", "integer","integer"],
                [$active_id, $pass, $this->getId(), $maxStep]
            );
        } else {
            $data = $this->db->queryF(
                "SELECT value1+1 as value1 FROM tst_solutions WHERE active_fi = %s AND pass = %s AND question_fi = %s",
                ["integer", "integer", "integer"],
                [$active_id, $pass, $this->getId()]
            );
        }

        while ($row = $this->db->fetchAssoc($data)) {
            $result->addKeyValue($row["value1"], $row["value1"]);
        }

        $points = $this->calculateReachedPoints($active_id, $pass);
        $max_points = $this->getMaximumPoints();

        $result->setReachedPercentage(($points / $max_points) * 100);

        return $result;
    }

    /**
     * If index is null, the function returns an array with all anwser options
     * Else it returns the specific answer option
     *
     * @param null|int $index
     */
    public function getAvailableAnswerOptions($index = null)
    {
        if ($index !== null) {
            return $this->getAnswer($index);
        } else {
            return $this->getAnswers();
        }
    }

    protected function buildTestPresentationConfig(): ilTestQuestionConfig
    {
        $config = parent::buildTestPresentationConfig();
        $config->setUseUnchangedAnswerLabel($this->lng->txt('tst_mc_label_none_above'));
        return $config;
    }

    public function isSingleline(): bool
    {
        return $this->is_singleline;
    }

    public function toLog(AdditionalInformationGenerator $additional_info): array
    {
        $result = [
            AdditionalInformationGenerator::KEY_QUESTION_TYPE => (string) $this->getQuestionType(),
            AdditionalInformationGenerator::KEY_QUESTION_TITLE => $this->getTitleForHTMLOutput(),
            AdditionalInformationGenerator::KEY_QUESTION_TEXT => $this->formatSAQuestion($this->getQuestion()),
            AdditionalInformationGenerator::KEY_QUESTION_SHUFFLE_ANSWER_OPTIONS => $additional_info
                ->getTrueFalseTagForBool($this->getShuffle()),
            'ass_mc_sel_lim_setting' => (int) $this->getSelectionLimit(),
            AdditionalInformationGenerator::KEY_FEEDBACK => [
                AdditionalInformationGenerator::KEY_QUESTION_FEEDBACK_ON_INCOMPLETE => $this->formatSAQuestion($this->feedbackOBJ->getGenericFeedbackTestPresentation($this->getId(), false)),
                AdditionalInformationGenerator::KEY_QUESTION_FEEDBACK_ON_COMPLETE => $this->formatSAQuestion($this->feedbackOBJ->getGenericFeedbackTestPresentation($this->getId(), true))
            ]
        ];

        foreach ($this->getAnswers() as $key => $answer_obj) {
            $result[AdditionalInformationGenerator::KEY_QUESTION_ANSWER_OPTIONS][$key + 1] = [
                AdditionalInformationGenerator::KEY_QUESTION_ANSWER_OPTION => $this->formatSAQuestion($answer_obj->getAnswertext()),
                AdditionalInformationGenerator::KEY_QUESTION_POINTS_CHECKED => (float) $answer_obj->getPointsChecked(),
                AdditionalInformationGenerator::KEY_QUESTION_POINTS_UNCHECKED => (float) $answer_obj->getPointsUnchecked(),
                AdditionalInformationGenerator::KEY_QUESTION_ANSWER_OPTION_ORDER => (int) $answer_obj->getOrder(),
                AdditionalInformationGenerator::KEY_QUESTION_ANSWER_OPTION_IMAGE => (string) $answer_obj->getImage(),
                AdditionalInformationGenerator::KEY_FEEDBACK => $this->formatSAQuestion(
                    $this->feedbackOBJ->getSpecificAnswerFeedbackExportPresentation($this->getId(), 0, $key)
                )
            ];
        }

        return $result;
    }

    protected function solutionValuesToLog(
        AdditionalInformationGenerator $additional_info,
        array $solution_values
    ): array {
        $solution_ids = array_map(
            static fn(array $v): string => $v['value1'],
            $solution_values
        );
        $parsed_solutions = [];
        foreach ($this->getAnswers() as $id => $answer) {
            $checked = false;
            if (in_array($id, $solution_ids)) {
                $checked = true;
            }
            $parsed_solutions[$answer->getAnswertext()] = $additional_info
                ->getCheckedUncheckedTagForBool($checked);
        }
        return $parsed_solutions;
    }

    public function solutionValuesToText(array $solution_values): array
    {
        $solution_ids = array_map(
            static fn(array $v): string => $v['value1'],
            $solution_values
        );

        return array_map(
            function (ASS_AnswerMultipleResponseImage $v) use ($solution_ids): string {
                $checked = 'unchecked';
                if (in_array($v->getId(), $solution_ids)) {
                    $checked = 'checked';
                }
                return "{$v->getAnswertext()} ({$this->lng->txt($checked)})";
            },
            $this->getAnswers()
        );
    }

    public function getCorrectSolutionForTextOutput(int $active_id, int $pass): array
    {
        return array_map(
            fn(ASS_AnswerMultipleResponseImage $v): string => $v->getAnswertext()
                . "({$this->lng->txt('points')} "
                . "{$this->lng->txt('checked')}: {$v->getPointsChecked()}, "
                . "{$this->lng->txt('unchecked')}: {$v->getPointsUnchecked()})",
            $this->getAnswers()
        );
    }
}
