<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package MySQL
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Get total number of Rows in a Table
 * @param string $module Module to return count of
 * @param string $table Table to return count of
 * @param bool $skip_cached set to TRUE to force fresh count
 * @param object $con MySQLi database connection object
 * @return int Returns the number of rows in the table
 */
function jrCore_db_number_rows($module, $table, $skip_cached = false, $con = null)
{
    $key = "db_number_rows_{$module}_{$table}";
    $cnt = jrCore_get_flag($key);
    if ($skip_cached || !$cnt) {
        $cnt = 0;
        $tbl = jrCore_db_table_name($module, $table);
        $req = "SELECT COUNT(*) AS r_cnt FROM {$tbl}";
        $_rt = jrCore_db_query($req, 'SINGLE', false, null, false, $con, false);
        if ($_rt && isset($_rt['r_cnt']) && is_numeric($_rt['r_cnt'])) {
            $cnt = intval($_rt['r_cnt']);
        }
        jrCore_set_flag($key, $cnt);
    }
    return $cnt;
}

/**
 * Returns TRUE if a database table is EMPTY (exists but has no rows)
 * @param string $module Module
 * @param string $table Table Name
 * @param object $con MySQLi database connection object
 * @return bool
 */
function jrCore_db_table_is_empty($module, $table, $con = null)
{
    $tbl = jrCore_db_table_name($module, $table);
    $req = "SELECT 1 FROM {$tbl} LIMIT 1";
    $_rt = jrCore_db_query($req, 'SINGLE', false, null, false, $con, false);
    if (is_array($_rt)) {
        return false;
    }
    return true;
}

/**
 * Check if a MySQL table exists
 * @param string $module Module to check table for
 * @param string $table Table to check
 * @param object $con MySQLi database connection
 * @return bool
 */
function jrCore_db_table_exists($module, $table, $con = null)
{
    $key = "jrcore_db_table_exists";
    if (!$_tm = jrCore_get_flag($key)) {
        $_tm = array();
    }
    $tbl = jrCore_db_table_name($module, $table);
    if (isset($_tm[$tbl])) {
        return $_tm[$tbl] == 1;
    }
    $req = "SELECT 1 FROM {$tbl} LIMIT 1";
    $res = jrCore_db_query($req, null, false, null, false, $con, false);
    if ($res && is_object($res)) {
        $_tm[$tbl] = 1;
        jrCore_set_flag($key, $_tm);
        return true;
    }
    $_tm[$tbl] = 0;
    jrCore_set_flag($key, $_tm);
    return false;
}

/**
 * Return array of column definitions in a table
 * @param string $module Module to return column names for
 * @param string $table Table to return column names for
 * @param object $con MySQLi database connection object
 * @return mixed Returns an array of column names/data
 */
function jrCore_db_table_columns($module, $table, $con = null)
{
    $table = jrCore_db_table_name($module, $table);
    // See if we have already done this table
    if ($_uniq = jrCore_get_flag("jrcore_db_table_columns_{$table}")) {
        return $_uniq;
    }
    $req = "DESCRIBE {$table}";
    $_rt = jrCore_db_query($req, 'NUMERIC', false, null, false, $con, false);
    if ($_rt && is_array($_rt)) {
        $_tmp = array();
        foreach ($_rt as $_col) {
            $_tmp["{$_col['Field']}"] = $_col;
        }
        jrCore_set_flag("jrcore_db_table_columns_{$table}", $_tmp);
        return $_tmp;
    }
    return false;
}

/**
 * Returns constructed table name for module/table
 * @param string $module Module Name
 * @param string $table Table Name
 * @return string Returns constructed table name
 */
function jrCore_db_table_name($module, $table)
{
    $pfx = jrCore_get_config_value('jrCore', 'db_prefix', 'jr_');
    return strtolower("{$pfx}{$module}_{$table}");
}

/**
 * Delete an item from a non-DS database table with Recycle Bin support
 * @param int $profile_id Profile ID row belongs to
 * @param string $module Module
 * @param string $table Table
 * @param string $id_column Primary Key column name
 * @param int|array $id Primary Key value
 * @param string $title_column Name of column to use for title OR title string
 * @param bool $recycle_bin set to FALSE to skip recycle bin
 * @param object $con MySQLi database connection object
 * @return int|false Returns number items deleted on success, false on failure
 */
function jrCore_db_delete_row_by_primary_key($profile_id, $module, $table, $id_column, $id, $title_column, $recycle_bin = true, $con = null)
{
    if (is_array($id)) {
        foreach ($id as $k => $i) {
            $id[$k] = intval($i);
        }
    }
    else {
        $id = array(intval($id));
    }
    $ttl = 0;
    $pid = (int) $profile_id;
    $mod = jrCore_db_escape($module);
    $_tb = array();
    foreach ($id as $i) {
        $tbl = jrCore_db_get_archive_table_for_unique_id($module, $table, $i);
        if (!isset($_tb[$tbl])) {
            $_tb[$tbl] = array();
        }
        $_tb[$tbl][$i] = $i;
    }
    foreach ($_tb as $archive => $_ids) {
        $iid = implode(',', $_ids);
        if ($recycle_bin) {
            // Recycle bin is ON - get item data and move to Recycle Bin
            $req = "SELECT * FROM {$archive} WHERE `{$id_column}` IN({$iid})";
            $_ex = jrCore_db_query($req, 'NUMERIC', false, null, false, $con);
            if ($_ex && is_array($_ex)) {
                // Save to Recycle bin
                $_in = array();
                foreach ($_ex as $i) {
                    $in_id = intval($i[$id_column]);
                    $index = "{$mod}/{$table}/{$in_id}";
                    if (!isset($_in[$index])) {
                        $title = jrCore_db_escape($title_column);
                        if (!empty($i[$title_column])) {
                            $title = jrCore_db_escape($i[$title_column]);
                        }
                        $_in[$index] = "(1,UNIX_TIMESTAMP(),'{$mod}','{$table}',{$pid},{$in_id},'{$title}','" . jrCore_db_escape(json_encode($i)) . "')";
                    }
                }
                if (count($_in) > 0) {
                    $tbl = jrCore_db_table_name('jrCore', 'recycle');
                    $req = "INSERT INTO {$tbl} (r_group_id, r_time, r_module, r_table, r_profile_id, r_item_id, r_title, r_data) VALUES " . implode(',', $_in);
                    $cnt = jrCore_db_query($req, 'COUNT', false, null, false);
                    if (!$cnt || $cnt == 0) {
                        // Fail to insert
                        return false;
                    }
                }
            }
        }
        // Delete items
        $req = "DELETE FROM {$archive} WHERE `{$id_column}` IN({$iid})";
        $cnt = jrCore_db_query($req, 'COUNT', false, null, false, $con);
        if ($cnt && $cnt > 0) {
            $ttl += $cnt;
        }
    }
    return $ttl;
}

/**
 * Set the MySQL slow query length
 * @param int $seconds Number of seconds for a query to be considered a "slow" query
 * @return bool
 */
function jrCore_db_set_slow_query_length($seconds)
{
    if (!jrCore_checktype($seconds, 'number_nz')) {
        return false;
    }
    // Save old value
    $_rt = jrCore_db_query("SHOW VARIABLES LIKE '%long_query_time%'", 'SINGLE');
    $val = 10;
    if ($_rt && isset($_rt['Value']) && strlen($_rt['Value']) > 0) {
        $val = round($_rt['Value']);
    }
    // @note: we check for the flag here first since we only ever want
    // to store the ORIGINAL value even if function is called multiple times
    if (!jrCore_get_flag('core_db_long_query_time')) {
        jrCore_set_flag('core_db_long_query_time', $val);
    }
    jrCore_db_query("SET SESSION long_query_time = " . intval($seconds));
    return true;
}

/**
 * Restore the long query time to the original value
 * @return bool
 */
function jrCore_db_restore_slow_query_length()
{
    if ($seconds = jrCore_get_flag('core_db_long_query_time')) {
        if ($seconds > 0) {
            jrCore_db_query("SET SESSION long_query_time = " . intval($seconds));
        }
    }
    return true;
}

/**
 * Escape a string for DB insertion
 * @param mixed $data String or Array to have data escaped for MySQL insertion
 * @return mixed Returns String or Array ready for db insertion
 */
function jrCore_db_escape($data)
{
    if (empty($data)) {
        return $data;
    }
    if (is_array($data)) {
        foreach ($data as $key => $val) {
            $data[$key] = jrCore_db_escape($val);
        }
    }
    else {
        $temp = jrCore_db_connect();
        $data = mysqli_real_escape_string($temp, $data);
    }
    return $data;
}

/**
 * Connect to the MySQL database
 *
 * This function does not need to be called directly in Jamroom.
 * The jrCore_db_query() function will call it as needed if it sees that the
 * Jamroom Database has not been connected to.
 *
 * @param bool $force set to TRUE to force a new connection
 * @param bool $found_rows set to FALSE to disable MYSQLI_CLIENT_FOUND_ROWS flag on mysqli_real_connect()
 * @param string $query_type used with multi node setups
 * @return mysqli Returns database resource on success, exits on failure
 */
function jrCore_db_connect($force = false, $found_rows = true, $query_type = 'write')
{
    $key = "jrcore_dbconnect_mysqli_object_{$query_type}";
    if (!$force) {
        // If we do not have a cached version of this type yet (read or write) - IF this is a
        // READ request and we have a WRITE cache, use it - that way we are always consistent
        // @note If the write flag is set it means we already encountered a DDL change query
        if ($query_type == 'read') {
            if ($tmp = jrCore_get_flag('jrcore_dbconnect_mysqli_object_write')) {
                return $tmp;
            }
        }
        if ($tmp = jrCore_get_flag($key)) {
            return $tmp;
        }
    }
    else {
        // Forcing new connection - close any previous connection
        jrCore_db_close();
    }
    $myi = mysqli_init();
    if (!$myi || !is_object($myi)) {
        jrCore_notice('Error', "unable to initialize database connection - check database support", false);
    }
    // See if we are using persistent connections
    $pfx = '';
    if (jrCore_get_config_value('jrCore', 'db_persistent', 'off') == 'on') {
        $pfx = 'p:';
    }
    $host = jrCore_get_config_value('jrCore', 'db_host', '');
    $_con = array(
        'type' => $query_type,
        'host' => "{$pfx}{$host}",
        'port' => jrCore_get_config_value('jrCore', 'db_port', 3306),
        'base' => jrCore_get_config_value('jrCore', 'db_name', ''),
        'user' => jrCore_get_config_value('jrCore', 'db_user', ''),
        'pass' => jrCore_get_config_value('jrCore', 'db_pass', '')
    );
    $_con = jrCore_trigger_event('jrCore', 'db_connect', $_con, array('myi' => $myi));

    $flag = 0;
    if ($found_rows) {
        $flag = MYSQLI_CLIENT_FOUND_ROWS;
    }
    $timeout = jrCore_get_config_value('jrCore', 'db_connect_timeout', 0);
    mysqli_options($myi, MYSQLI_OPT_CONNECT_TIMEOUT, $timeout);
    try {
        jrCore_start_timer('db_connect');
        $tmp = @mysqli_real_connect($myi, $_con['host'], $_con['user'], $_con['pass'], $_con['base'], $_con['port'], null, $flag);
        jrCore_stop_timer('db_connect');
    }
    catch (mysqli_sql_exception $e) {
        $tmp = false;
    }
    if (!$tmp) {
        $err = mysqli_connect_error();
        $ern = mysqli_connect_errno();
        $_tm = jrCore_trigger_event('jrCore', 'db_connect_warning', array('myi' => $myi, 'persist' => $pfx, 'flag' => $flag, '_con' => $_con, 'error' => "{$ern}:{$err}"));
        if (!isset($_tm['connected'])) {
            // sleep for a bit and try again
            jrCore_start_timer('sleep');
            usleep(100000);
            jrCore_stop_timer('sleep');
            try {
                jrCore_start_timer('db_connect');
                @mysqli_real_connect($myi, $_con['host'], $_con['user'], $_con['pass'], $_con['base'], $_con['port'], null, $flag);
                jrCore_stop_timer('db_connect');
            }
            catch (mysqli_sql_exception $e) {
                $ern = $e->getCode();
                $err = $e->getMessage();
                $_tm = jrCore_trigger_event('jrCore', 'db_connect_error', array('myi' => $myi, 'persist' => $pfx, 'flag' => $flag, '_con' => $_con, 'error' => "{$ern}:{$err}"));
                if (!isset($_tm['connected'])) {
                    jrCore_db_error($ern, $err);
                }
            }
        }
    }
    mysqli_set_charset($myi, 'utf8');
    jrCore_db_init_database_connection($myi);
    jrCore_set_flag($key, $myi);
    return $myi;
}

/**
 * Initialize the DB connect with SESSION params
 * @param object $con Database connection
 * @return mysqli_result|false
 */
function jrCore_db_init_database_connection($con)
{
    if ($con) {
        if ($cfg = jrCore_get_config_value('jrCore', 'sql_session', false)) {
            if (strlen($cfg) > 1) {
                $sql = "SET SESSION {$cfg}";
            }
        }
        else {
            if ($cfg = jrCore_get_config_value('jrCore', 'sql_mode', false)) {
                if (strlen($cfg) > 1) {
                    $sql = "SET SESSION sql_mode = '{$cfg}'";
                }
                else {
                    $sql = "SET SESSION sql_mode = ''";
                }
            }
            else {
                $sql = "SET SESSION sql_mode = 'NO_ENGINE_SUBSTITUTION,STRICT_TRANS_TABLES'";
            }
        }
        if (!empty($sql)) {
            return mysqli_query($con, $sql);
        }
    }
    return false;
}

/**
 * Show a DB Error and Exit
 * @param int $error_number Error number
 * @param string $error_text Error Text
 * @return void|null
 */
function jrCore_db_error($error_number, $error_text)
{
    // @note we do not have access to the DB at this point, and $_conf
    // will not contain information about active modules or skins.
    if (!jrCore_is_ajax_request()) {
        if ($tpl = file_get_contents(APP_DIR . "/modules/jrCore/templates/db_error.tpl")) {
            $_rp = array(
                'trace_id' => md5(microtime() . mt_rand(0, 999999)),
                'db_error' => "Error #{$error_number}: {$error_text}",
                '_server'  => $_SERVER
            );
            $_rp = jrCore_trigger_event('jrCore', 'fatal_error', $_rp);
            $out = jrCore_parse_template($tpl, $_rp, 'jrCore');
            return jrCore_send_response_and_detach($out);
        }
    }
    jrCore_notice('Error', "Error #{$error_number}: {$error_text}", false);
}

/**
 * Close the connection to the MySQL database
 *
 * <b>Note:</b> There is no need to call this function unless your script enters
 * a long processing segment where the database connection is no longer
 * needed, then it is a good idea to free the database resource.
 *
 * @return bool Returns true on success
 */
function jrCore_db_close()
{
    if ($tmp = jrCore_get_flag('jrcore_dbconnect_mysqli_object')) {
        if (mysqli_close($tmp)) {
            jrCore_delete_flag('jrcore_dbconnect_mysqli_object');
            return true;
        }
        return false;
    }
    return true;
}

/**
 * Send a Query to the MySQL database
 * The jrCore_db_query function is the main function used to send MySQL queries to the database.
 * Valid <b>$return</b> values are:
 * * (null)            - if empty or null, the Database connection resource is returned for raw processing<br>
 * * <b>NUMERIC</b>    - returns a multi-dimensional array, numerically indexed beginning at 0<br>
 * * <b>SINGLE</b>     - a single-dimension array is returned<br>
 * * <b>COUNT</b>      - the number of rows affected by an INSERT, DELETE or UPDATE query<br>
 * * <b>INSERT_ID</b>  - the id from the auto_increment column of the table from the last INSERT query<br>
 * * <b>NUM_ROWS</b>   - the number of rows returned from a SELECT query<br>
 * * <b>$column</b>    - if name of valid column given, will be used as key in associative array. The $column used <b>must</b> be a column that is returned as part of the SELECT statement.
 *<br>
 * @param string $query The MySQL query to send to the database
 * @param string $return format of the results you want returned from the query.
 * @param bool $multi Deprecated - use jrCore_db_multi_query()
 * @param string $only_val Set to a valid database table COLUMN name, then ONLY the value for that column will be returned (instead of an array of values)
 * @param bool $exit_on_error By default we exit when a database error is encountered
 * @param mysqli $con MySQL DB Connection object - default will create one to the JR DB
 * @param bool $log_error set to FALSE to prevent error logging
 * @param bool $skip_triggers set to TRUE to skip query triggers
 * @param int $max_query_ms Maximum number of milliseconds SQL query will be allowed to execute for (requires MySQL 5.7+)
 * @param bool $as_transaction if $multi is TRUE, set to FALSE to not run queries as transaction
 * @return array|bool|int|mysqli_result|string[]|null Returns multiple formats - see list of $return values above for details
 * @see https://dev.mysql.com/doc/refman/5.7/en/optimizer-hints.html#optimizer-hints-execution-time
 */
function jrCore_db_query($query, $return = null, $multi = false, $only_val = null, $exit_on_error = true, $con = null, $log_error = true, $skip_triggers = false, $max_query_ms = null, $as_transaction = true)
{
    global $_post;
    // make sure we get a query
    if (!is_array($query)) {
        $multi = false;
        $query = trim($query);
        if (empty($query)) {
            return false;
        }
    }
    else {
        if (empty($query)) {
            return false;
        }
        $multi = $query;  // Backup of incoming query array
        $query = implode(";\n", $query);
    }
    if (!$con) {
        $qtp = (stripos($query, 'SELECT') === 0 && !$multi) ? 'read' : 'write';
        $con = jrCore_db_connect(false, true, $qtp);
    }

    // Trigger Init Event
    $_args = null;
    if (!$skip_triggers) {
        $_args                    = func_get_args();
        $_args[0]                 = $query;
        $_args['initial_trigger'] = 1;
        $_args['loc']             = 1;
        $_init                    = jrCore_trigger_event('jrCore', 'db_query_init', $query, $_args);
        if (is_array($_init)) {
            // We had a listener that needs to change the DB connection
            // We will get back an ARRAY that contains the query and
            // the updated database connect we need to use
            if (!empty($_init['query'])) {
                $query = $_init['query'];
            }
            if (!empty($_init['new_db_connection'])) {
                $con      = $_init['new_db_connection'];
                $_args[5] = $con;
            }
        }
    }

    // Max query MS
    if (strpos($query, 'SELECT') === 0) {
        if (is_null($max_query_ms)) {
            $max_query_ms = (int) jrCore_get_config_value('jrCore', 'max_sql_run_time', 0);
        }
        elseif (!jrCore_checktype($max_query_ms, 'number_nz')) {
            $max_query_ms = 0;
        }
        // @note: 0 = disabled
        if ($max_query_ms > 0) {
            $query = "SELECT /*+ MAX_EXECUTION_TIME({$max_query_ms}) */ " . substr($query, 7);
        }
    }

    // If our $multi flag is true we use our multi_query function so multiple SQL
    // queries can be run in one shot - InnoDB tables only!
    $err = false;
    $ecd = false;
    $now = microtime(true);
    if ($multi) {
        $query = ($as_transaction) ? "START TRANSACTION;\n{$query};\nCOMMIT;" : $query;
        if ($return == 'COUNT') {
            $query = str_replace('COMMIT;', 'SELECT ROW_COUNT() AS afrc; COMMIT;', $query);
        }
        try {
            mysqli_multi_query($con, $query);
        }
        catch (Exception $e) {
            $err = 'Query Error: ' . $e->getMessage();
            $ecd = $e->getCode();
        }
    }
    else {
        try {
            $res = mysqli_query($con, $query);
        }
        catch (Exception $e) {
            $err = 'Query Error: ' . $e->getMessage();
            $ecd = $e->getCode();
        }
    }
    $end = round((microtime(true) - $now) * 1000);

    // Used in db_query_exit listener
    $_data = array(
        'connection' => $con,
        'result'     => null,
        'error'      => null,
        'query_ms'   => $end
    );

    if ($err) {

        $exit = 1;
        if (!strpos(' ' . $query, 'DESCRIBE')) {

            $retry = false;
            // See if this is a "MySQL Server has gone away error" - if it is, we try
            // to force a reconnect here and see if we can continue
            if (stripos($err, 'server has gone away')) {
                jrCore_db_close();

                jrCore_start_timer('sleep');
                usleep(500000);
                jrCore_stop_timer('sleep');

                $_init = false;
                if (!$skip_triggers) {
                    $_args['err'] = $err;
                    $_args['loc'] = 2;
                    $_init        = jrCore_trigger_event('jrCore', 'db_query_init', $_args[0], $_args);
                }
                if (is_array($_init)) {
                    // We had a listener that needs to change the DB connection
                    // We will get back an ARRAY that contains the query and
                    // the updated database connect we need to use
                    if (!empty($_init['query'])) {
                        $query = $_init['query'];
                    }
                    if (!empty($_init['new_db_connection'])) {
                        $con = $_init['new_db_connection'];
                    }
                    else {
                        $con = jrCore_db_connect(true);
                    }
                }
                else {
                    $con = jrCore_db_connect(true);
                }
                $retry = true;
            }
            elseif (stripos($err, 'maximum statement execution')) {
                jrCore_db_close();

                jrCore_record_event('db_timeout');
                if ($log_error) {
                    $_er = array(
                        'error' => $err,
                        'query' => substr($query, 0, 900000),
                        '_post' => $_post
                    );
                    jrCore_logger('CRI', "core: {$err}", $_er, true, null, 0, true);
                }
                if (jrCore_is_view_request()) {
                    jrCore_check_for_fatal_error('Maximum Execution time exceeded');
                }
                jrCore_notice('Error', $err, false);
            }
            elseif (stripos($err, 'deadlock') || stripos($err, 'try restarting')) {

                // With an InnoDB deadlock we can retry
                jrCore_record_event('db_deadlock');

                jrCore_start_timer('sleep');
                usleep(250000);
                jrCore_stop_timer('sleep');

                $_init = false;
                if (!$skip_triggers) {
                    $_args['err'] = $err;
                    $_args['loc'] = 3;
                    $_init        = jrCore_trigger_event('jrCore', 'db_query_init', $_args[0], $_args);
                }
                if (is_array($_init)) {
                    // We had a listener that needs to change the DB connection
                    // We will get back an ARRAY that contains the query and
                    // the updated database connect we need to use
                    if (!empty($_init['query'])) {
                        $query = $_init['query'];
                    }
                    if (!empty($_init['new_db_connection'])) {
                        $con = $_init['new_db_connection'];
                    }
                    else {
                        $con = jrCore_db_connect(true);
                    }
                }
                else {
                    $con = jrCore_db_connect(true);
                }
                $retry = true;
            }
            else {
                if ($log_error) {
                    $_er = array(
                        'error' => $err,
                        'query' => substr($query, 0, 900000),
                        '_post' => $_post
                    );
                    if (strpos($err, 'Query Error:') === 0) {
                        // Do not retry on a query error - we'll just fail the second time
                        jrCore_logger('CRI', "core: {$err}", $_er, true, null, 0, true);
                    }
                    else {
                        jrCore_logger('CRI', "core: {$err} (will retry)", $_er, true, null, 0, true);
                    }
                }
            }

            // Are we retrying this query?
            if ($retry) {
                $err = false;
                if ($multi) {
                    try {
                        mysqli_multi_query($con, $query);
                    }
                    catch (Exception $e) {
                        $err = 'Query Error: ' . $e->getMessage();
                        $ecd = $e->getCode();
                    }
                }
                else {
                    try {
                        $res = mysqli_query($con, $query);
                    }
                    catch (Exception $e) {
                        $err = 'Query Error: ' . $e->getMessage();
                        $ecd = $e->getCode();
                    }
                }
                if ($err) {
                    if ($log_error) {
                        $_er = array(
                            'error' => $err,
                            'query' => substr($query, 0, 900000),
                            '_post' => $_post
                        );
                        if (strpos($err, 'Query Error:') === 0) {
                            // Do not retry on a query error - we'll just fail the second time
                            jrCore_logger('CRI', "core: {$err}", $_er, true, null, 0, true);
                        }
                        else {
                            jrCore_logger('CRI', "core: {$err} (will retry)", $_er, true, null, 0, true);
                        }
                    }
                }
                else {
                    // We were successful on our second try
                    $exit = 0;
                }
            }
        }
        if ($exit === 1) {
            if ($exit_on_error) {
                fdebug("{$err}:\n{$query}");  // OK
                jrCore_db_error($ecd, 'The system has encountered a database query error - check activity log');
            }
            // Trigger Exit Event
            if (!$skip_triggers) {
                $_data['error'] = $err;
                jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
            }
            return false;
        }
    }

    // If this is a MULTI query we always return a multi dimensional array
    if ($multi) {
        $_out = array();
        $i    = -1;
        do {
            if ($result = mysqli_store_result($con)) {
                $i++;
                while ($row = mysqli_fetch_assoc($result)) {
                    $_out[$i][] = $row;
                }
            }
        }
        while (_jrCore_db_multi_query_result_handler($con, $multi, $exit_on_error, $log_error));
        if (!$skip_triggers) {
            $_data['result'] = $_out;
            jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
        }
        return $_out;
    }

    if (is_null($return) || $return === false) {
        if (!$skip_triggers) {
            $_data['result'] = $res;
            jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
        }
        return $res;
    }
    switch ($return) {

        case 'SINGLE':
            $_tmp = (is_object($res)) ? mysqli_fetch_assoc($res) : false;
            if (!$skip_triggers) {
                $_data['result'] = $_tmp;
                jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
            }
            return $_tmp;

        case 'COUNT':
            $num = (int) mysqli_affected_rows($con);
            if (!$skip_triggers) {
                $_data['result'] = $num;
                jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
            }
            return $num;

        case 'NUM_ROWS':
            $num = (int) mysqli_num_rows($res);
            if (!$skip_triggers) {
                $_data['result'] = $num;
                jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
            }
            return $num;

        case 'INSERT_ID':
            $num = (int) mysqli_insert_id($con);
            if (!$skip_triggers) {
                $_data['result'] = $num;
                jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
            }
            return $num;

        default:
            if ($res) {
                $num = mysqli_num_rows($res);
            }
            else {
                if (!$skip_triggers) {
                    $_data['result'] = 0;
                    jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
                }
                return false;
            }
            // more than 1 row - return multidimensional array that is
            // either numeric based (base 0) or associative based if $akey given
            if ($num >= 1) {
                $_rt = array();
                $i   = 0;
                while ($row = mysqli_fetch_assoc($res)) {
                    if ($return == 'NUMERIC') {
                        $_rt[$i] = $row;
                        $i++;
                    }
                    elseif (!empty($only_val)) {
                        if (isset($row[$only_val]) && isset($row[$return])) {
                            $_rt["{$row[$return]}"] = $row[$only_val];
                        }
                    }
                    else {
                        $_rt["{$row[$return]}"] = $row;
                    }
                }
                mysqli_free_result($res);
                if (isset($_rt) && is_array($_rt) && count($_rt) > 0) {
                    if (!$skip_triggers) {
                        $_data['result'] = $_rt;
                        jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
                    }
                    return $_rt;
                }
                // 0 rows
                if (!$skip_triggers) {
                    jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
                }
                return false;
            }
            break;
    }
    if (!$skip_triggers) {
        jrCore_trigger_event('jrCore', 'db_query_exit', $_data, $_args);
    }
    return false;
}

/**
 * Alias for jrCore_db_multi_query
 * @param array $_queries Array of SQL Queries
 * @param bool $exit_on_error
 * @param bool $log_error
 * @param resource $con
 * @return array|false
 * @deprecated use jrCore_db_multi_query()
 */
function jrCore_db_multi_select($_queries, $exit_on_error = true, $log_error = true, $con = null)
{
    return jrCore_db_query($_queries, null, true, null, $exit_on_error, $con, $log_error, false, null, false);
}

/**
 * Send multiple SQL statements to the DB at once
 * each result set will be returned as a key in the result array.
 * @note ONLY SELECT queries will have a key in the result - i.e. the FIRST select query will be index 0, second 1, etc.
 * @note This is a wrapper around jrCore_db_query
 * @param array $_queries Array of SQL Queries
 * @param bool $exit_on_error Set to FALSE not not exit if an error is encountered
 * @param bool $log_error Set to FALSE to prevent logging an SQL error if encountered
 * @param resource $con alternate DB connection
 * @param bool $as_transaction set to FALSE to not wrap the queries in a TRANSACTION
 * @return array|false
 */
function jrCore_db_multi_query($_queries, $exit_on_error = true, $log_error = true, $con = null, $as_transaction = true)
{
    return jrCore_db_query($_queries, null, true, null, $exit_on_error, $con, $log_error, false, null, $as_transaction);
}

/**
 * Iterative result handler for mysqli_multi_query
 * @param object $con mysqli DB connection object
 * @param array $_queries Array of SQL Queries
 * @param bool $exit_on_error Set to FALSE not not exit if an error is encountered
 * @param bool $log_error Set to FALSE to prevent logging an SQL error if encountered
 * @return bool
 */
function _jrCore_db_multi_query_result_handler($con, $_queries, $exit_on_error = true, $log_error = true)
{
    global $_post;
    try {
        return (mysqli_more_results($con) && mysqli_next_result($con));
    }
    catch (Exception $e) {
        if ($log_error || jrCore_is_developer_mode()) {
            $err = $e->getMessage();
            if (stripos(' ' . $err, 'deadlock')) {
                jrCore_record_event('db_deadlock');
            }
            $_dt = array(
                'error'    => $err,
                '_queries' => $_queries,
                '_post'    => $_post
            );
            jrCore_logger('CRI', "core: multi_query error: " . $err, $_dt, true, null, 0, true);
        }
        if ($exit_on_error) {
            jrCore_notice('Error', $err);
        }
        return false;
    }
}

/**
 * Paginate result sets from the MySQL database
 *
 * The jrCore_db_paged_query function is a "paginator" for db results.  It does
 * NOT create any of the output, but will return an array consisting
 * of the actual data, and the "prev" and "next" pages, if they exist.
 * Use in place of jrCore_db_query if you need paginated results.
 *
 * @param string $query SQL query to run
 * @param int $page_num Page number (set to 0 to auto-detect)
 * @param int $rows_per_page Number of rows to return
 * @param string $return_type Return type for data.  This can be any of the valid return
 *        types as used by jrCore_db_query().
 * @param string $c_query optional Counting Query that will be used in place of primary
 *        query to retrieve row count.  If the main query is very complex, this
 *        can seriously speed up the results.  Note that an int can be provided
 *        as well, and that will be used for the total.  You can also use the string
 *        "simplepagebreak" for the simple pager
 * @param string $only_val Only return this value from result set
 * @param bool $cookie_rows set to FALSE to always use $rows_per_page value
 * @param object $con MySQLi database object
 * @return array returns array of data
 */
function jrCore_db_paged_query($query, $page_num, $rows_per_page, $return_type = 'NUMERIC', $c_query = null, $only_val = null, $cookie_rows = true, $con = null)
{
    // LIMIT should not be included in query
    global $_post;
    if (isset($query) && strstr($query, 'LIMIT')) {
        jrCore_notice('Error', "jrCore_db_paged_query - do not include a LIMIT clause in your SQL query - it is added automatically");
    }
    if (is_null($c_query) || $c_query === false || strlen($c_query) === 0) {
        $c_query = $query;
    }
    // Get total number of rows
    if ($c_query == 'simplepagebreak') {
        // When doing a SIMPLE pagebreak we do not show the page jumper - just previous/next page links
        $total = 0;
    }
    elseif (jrCore_checktype($c_query, 'number_nz')) {
        $total = intval($c_query);
    }
    else {
        $total = jrCore_db_query($c_query, 'NUM_ROWS', false, null, false, $con);
    }
    if (!jrCore_checktype($page_num, 'number_nz')) {
        if (isset($_post['p']) && jrCore_checktype($_post['p'], 'number_nz')) {
            $page_num = (int) $_post['p'];
        }
        else {
            $page_num = 1;
        }
    }

    $pagebreak = 12;
    if (jrCore_checktype($rows_per_page, 'number_nz')) {
        $pagebreak = (int) $rows_per_page;
    }
    if ($cookie_rows) {
        if ($num = jrCore_get_pager_rows()) {
            $pagebreak = $num;
        }
    }

    // For our query, we can't use the page number - we have to figure out
    // the offset based on the number of rows per page * page number - 1
    $start = intval(($page_num - 1) * $pagebreak);
    $query .= " LIMIT {$start},{$pagebreak}";

    $_out = array(
        'info' => array()
    );

    // and now get our data
    $_rt = jrCore_db_query($query, $return_type, false, $only_val, false, $con);
    if ($_rt && is_array($_rt)) {
        $_out['_items'] = $_rt;
    }

    // now figure out if we have "prev" or "next" links.
    if ($c_query == 'simplepagebreak') {
        $total = (is_array($_rt)) ? count($_rt) : 0;
        if ($start === 0 && $total < $pagebreak) {
            $_out['info']['prev_page'] = false;
            $_out['info']['next_page'] = false;
            $_out['info']['this_page'] = 1;
        }
        else {
            $_out['info']['prev_page'] = $page_num - 1;
            if ($total < $pagebreak) {
                $_out['info']['next_page'] = false;
            }
            else {
                $_out['info']['next_page'] = $page_num + 1;
            }
            $_out['info']['this_page'] = intval($page_num);
        }
        $_out['info']['total_items'] = $total;
        $_out['info']['total_pages'] = 0;
    }
    else {
        if ($start === 0 && $total <= $pagebreak) {
            $_out['info']['prev_page'] = false;
            $_out['info']['next_page'] = false;
            $_out['info']['this_page'] = 1;
        }
        elseif ($start === 0) {
            $_out['info']['prev_page'] = false;
            $_out['info']['next_page'] = 2;
            $_out['info']['this_page'] = 1;
        }
        elseif (($start + $pagebreak) >= $total) {
            $_out['info']['prev_page'] = $page_num - 1;
            $_out['info']['next_page'] = false;
            $_out['info']['this_page'] = (int) ceil($total / $pagebreak);
        }
        else {
            $_out['info']['prev_page'] = $page_num - 1;
            $_out['info']['next_page'] = $page_num + 1;
            $_out['info']['this_page'] = intval($page_num);
        }
        $_out['info']['total_items'] = intval($total);
        $_out['info']['total_pages'] = 0;
        if ($total > 0) {
            $_out['info']['total_pages'] = (int) ceil($total / $pagebreak);
        }
    }
    return $_out;
}

/**
 * Get indexes in a MySQL table
 * @param string $module Module table belongs to
 * @param string $table Table name
 * @param null $con
 * @return array
 */
function jrCore_db_get_table_indexes($module, $table, $con = null)
{
    $tbl = jrCore_db_table_name($module, $table);
    $_in = array();
    $req = "SHOW INDEX FROM {$tbl}";
    $_tm = jrCore_db_query($req, 'NUMERIC', false, null, false, $con);
    if ($_tm && is_array($_tm)) {
        foreach ($_tm as $_idx) {
            if (isset($_in["{$_idx['Key_name']}"]) && $_idx['Seq_in_index'] > 1) {
                $_in["{$_idx['Key_name']}"] .= ",{$_idx['Column_name']}";
            }
            else {
                $_in["{$_idx['Key_name']}"] = $_idx['Column_name'];
            }
        }
    }
    return $_in;
}

/**
 * Get the PRIMARY KEY for a table if it exists
 * @param string $module Module
 * @param string $table Table
 * @param resource $con DB connection
 * @return string|false
 */
function jrCore_db_get_table_primary_key($module, $table, $con = null)
{
    // Find our auto_increment field
    if (!$_cl = jrCore_db_table_columns($module, $table)) {
        return false;
    }
    foreach ($_cl as $f => $c) {
        // [Extra] => auto_increment
        if (!empty($c['Extra']) && strpos(' ' . $c['Extra'], 'auto_increment')) {
            return $f;
        }
    }
    return false;
}

/**
 * Verify a table schema in the MySQL database
 *
 * The jrCore_db_verify_table function is used to create a MySQL database
 * table (if it does not exist), or validate the columns if it does exist.
 *
 * @param string $module Module table belongs to
 * @param string $table Table to validate
 * @param array $_schema MySQL Database creation SQL query - this will be executed if the table does not exist.
 * @param string $engine MySQL engine to use for table
 * @param mysqli $con MySQL DB Connection to use
 * @param int $partition_count define number of partitions when creating a PARTITION table
 * @param string $partition_column what column will be used to get the partition.  @note: MUST BE AN INTEGER COLUMN
 * @return bool Returns true/false on success/fail
 */
function jrCore_db_verify_table($module, $table, $_schema, $engine = 'InnoDB', $con = null, $partition_count = 0, $partition_column = '')
{
    global $_post;
    if (empty($table)) {
        jrCore_notice('error', "empty or missing table name in jrCore_db_verify_table");
    }
    if (!$_schema || !is_array($_schema)) {
        jrCore_notice('error', "invalid table schema array for {$module}/{$table} - ensure table columns and indexes are defined in an array");
    }
    // Cleanup
    foreach ($_schema as $k => $line) {
        $_schema[$k] = trim(trim($line), ',');
    }

    // Schema event
    $_args   = array(
        'module'  => $module,
        'table'   => $table,
        '_schema' => $_schema,
        'engine'  => $engine,
        'con'     => $con
    );
    $_schema = jrCore_trigger_event('jrCore', 'db_verify_table', $_schema, $_args);
    if (!$_schema || !is_array($_schema) || count($_schema) === 0) {
        // Handled by a listener
        return true;
    }

    // get our info about this table (if it exists)
    $tbl = jrCore_db_table_name($module, $table);
    $req = "DESCRIBE {$tbl}";
    $_rt = jrCore_db_query($req, 'Field', false, null, false, $con, false);
    if (!$_rt || !is_array($_rt)) {

        // Create
        $crq = "CREATE TABLE IF NOT EXISTS {$tbl} (" . implode(', ', $_schema) . ') ENGINE=' . $engine . ' DEFAULT CHARSET=utf8 COLLATE utf8_unicode_ci';
        if (jrCore_checktype($partition_count, 'number_nz') && !empty($partition_column)) {
            if ($partition_count > 1024) {
                jrCore_notice('error', "invalid partition_count in table definition for {$module}/{$table} - max partitions is 1024");
            }
            // Make sure partition column exists and is an integer type
            $good = true;
            foreach ($_schema as $line) {
                if (strpos($line, "{$partition_column} ") === 0) {
                    if (!stripos($line, 'int(')) {
                        $good = false;
                        break;
                    }
                }
            }
            if (!$good) {
                jrCore_notice('error', "invalid partition_column in table definition for {$module}/{$table} - must be a valid column in the table");
            }
            $crq .= " PARTITION BY HASH({$partition_column}) PARTITIONS " . intval($partition_count);
        }
        jrCore_db_query($crq, null, false, null, false, $con);

        // Was it created?
        $req = "DESCRIBE {$tbl}";
        $_rt = jrCore_db_query($req, 'Field', false, null, false, $con, false);
        if ($_rt && is_array($_rt)) {
            jrCore_logger('INF', "core: created missing table: {$tbl}");
            return true;
        }
        jrCore_logger('CRI', "core: error creating missing table: {$tbl}", array('query' => $crq));
        return false;
    }

    // It appears the table already exists.  Now we want to scan the incoming creation
    // schema and verify that each of our columns exist.  First load our indexes
    $_in = jrCore_db_get_table_indexes($module, $table, $con);

    foreach ($_schema as $line) {

        // TABLE start/end and PRIMARY KEY entries - skip
        if (strstr($line, 'PRIMARY KEY') || strlen($line) === 0) {

            // Are we adding a NEW primary key to an existing table?
            // module_id MEDIUMINT(7) UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY
            if (strlen($line) > 0 && !isset($_in['PRIMARY'])) {
                // We do not have a primary key in this table - get existing
                // columns to ensure we don't add a new key that exists
                if ($_xcols = jrCore_db_table_columns($module, $table)) {
                    if (stripos($line, 'PRIMARY') !== 0) {
                        $pk_name = jrCore_string_field($line, 1);
                        if (!isset($_xcols[$pk_name])) {
                            $req = "ALTER TABLE {$tbl} ADD {$line} FIRST";
                            jrCore_db_query($req, false, false, null, false, $con);
                            $_nw = jrCore_db_get_table_indexes($module, $table, $con);
                            if (isset($_nw['PRIMARY'])) {
                                jrCore_logger('INF', "core: created missing primary key in table: {$tbl}");
                            }
                            else {
                                jrCore_logger('CRI', "core: unable to create missing primary key in table: {$tbl}");
                            }
                        }
                    }
                }
            }
            continue;

        }

        // INDEX
        // INDEX band_id (band_id)
        // INDEX stat_index (stat_index) )
        // INDEX band_name (band_name(15))
        if (stripos($line, 'INDEX') === 0) {
            $idx_ikey = jrCore_string_field($line, 2);
            $idx_name = trim(str_replace('`', '', $idx_ikey));
            // Make sure it is built
            if (!isset($_in[$idx_name])) {
                $idx_args = trim(str_ireplace('INDEX ' . $idx_ikey, '', $line));
                $req      = "ALTER TABLE {$tbl} ADD INDEX `{$idx_name}` {$idx_args}";
                jrCore_db_query($req, null, false, null, false, $con);
                // now let's see if we were successful in adding our INDEX or NOT
                $req = "SHOW INDEX FROM {$tbl}";
                $_tp = jrCore_db_query($req, 'Key_name', false, null, false, $con);
                if (is_array($_tp[$idx_name])) {
                    jrCore_logger('INF', "core: created missing table index: {$idx_name} in table: {$tbl}");
                }
                else {
                    jrCore_logger('CRI', "core: unable to create missing table index: {$idx_name} in table: {$tbl}");
                }
            }
        }

        // UNIQUE
        // UNIQUE band_id (band_id)
        // UNIQUE stat_index (stat_index) )
        // UNIQUE template_unique (template_module, template_name)
        // UNIQUE band_name (band_name(15))
        elseif (stripos($line, 'UNIQUE') === 0) {
            $idx_ikey = jrCore_string_field($line, 2);
            $idx_name = trim(str_replace('`', '', $idx_ikey));
            $idx_flds = str_replace(array(' ', ')', '('), '', substr($line, strpos($line, '(')));
            if (!isset($_in[$idx_name])) {
                // We are creating a NEW index that did not exist before
                $idx_args = trim(str_ireplace('UNIQUE ' . $idx_ikey, '', $line));
                $req      = "ALTER TABLE {$tbl} ADD UNIQUE INDEX `{$idx_name}` {$idx_args}";
                jrCore_db_query($req, null, false, null, false, $con);
                // now let's see if we were successful in adding our INDEX or NOT
                $req = "SHOW INDEX FROM {$tbl}";
                $_tp = jrCore_db_query($req, 'Key_name', false, null, false, $con);
                if ($_tp && is_array($_tp) && isset($_tp[$idx_name]) && is_array($_tp[$idx_name])) {
                    jrCore_logger('INF', "core: created missing table unique index: {$idx_name} in table: {$tbl}");
                }
                else {
                    jrCore_logger('CRI', "core: unable to create missing table unique index: {$idx_name} in table: {$tbl}");
                }
            }
            elseif ($_in[$idx_name] != $idx_flds && '`' . str_replace(',', '`,`', $_in[$idx_name]) . '`' != $idx_flds && strpos($idx_flds, ',')) {
                // Our index fields in a compound UNIQUE index have changed
                $idx_args = trim(str_ireplace('UNIQUE ' . $idx_ikey, '', $line));

                // Drop old index (< MySQL 5.7 there is no ALTER TABLE RENAME|MODIFY INDEX)
                $req = "ALTER TABLE {$tbl} DROP INDEX `{$idx_name}`";
                jrCore_db_query($req, null, false, null, false, $con);

                // Create new UNIQUE Index
                $req = "ALTER TABLE {$tbl} ADD UNIQUE INDEX `{$idx_name}` {$idx_args}";
                jrCore_db_query($req, null, false, null, false, $con);

                // now let's see if we were successful in adding our INDEX or NOT
                $req = "SHOW INDEX FROM {$tbl}";
                $_tp = jrCore_db_query($req, 'Key_name', false, null, false, $con);
                if (is_array($_tp[$idx_name])) {
                    jrCore_logger('INF', "core: updated table unique index: {$idx_name} in table: {$tbl}");
                }
                else {
                    jrCore_logger('CRI', "core: unable to update table unique index: {$idx_name} in table: {$tbl}");
                }
            }
        }

        // FULLTEXT
        // FULLTEXT band_id (band_id)
        // FULLTEXT stat_index (stat_index) )
        // FULLTEXT template_unique (template_module, template_name)
        // FULLTEXT band_name (band_name(15))
        elseif (stripos($line, 'FULLTEXT') === 0) {
            $idx_ikey = jrCore_string_field($line, 2);
            $idx_name = trim(str_replace('`', '', $idx_ikey));
            $idx_flds = str_replace(array(' ', ')', '('), '', substr($line, strpos($line, '(')));
            if (!isset($_in[$idx_name])) {
                // We are creating a NEW index that did not exist before
                $idx_args = trim(str_ireplace('FULLTEXT ' . $idx_ikey, '', $line));
                // @note: FULLTEXT index cannot be done lock free
                // @see https://dev.mysql.com/doc/refman/5.7/en/alter-table.html
                $req = "ALTER TABLE {$tbl} ADD FULLTEXT INDEX `{$idx_name}` {$idx_args}";
                jrCore_db_query($req, null, false, null, false, $con);
                // now let's see if we were successful in adding our INDEX or NOT
                $req = "SHOW INDEX FROM {$tbl}";
                $_tp = jrCore_db_query($req, 'Key_name', false, null, false, $con);
                if (is_array($_tp[$idx_name])) {
                    jrCore_logger('INF', "core: created missing table fulltext index: {$idx_name} in table: {$tbl}");
                }
                else {
                    jrCore_logger('CRI', "core: unable to create missing table fulltext index: {$idx_name} in table: {$tbl}");
                }
            }
            elseif ($_in[$idx_name] != $idx_flds && '`' . str_replace(',', '`,`', $_in[$idx_name]) . '`' != $idx_flds && strpos($idx_flds, ',')) {
                // Our index fields in a compound UNIQUE index have changed
                $idx_args = trim(str_ireplace('FULLTEXT ' . $idx_ikey, '', $line));

                // Drop old index (< MySQL 5.7 there is no ALTER TABLE RENAME|MODIFY INDEX)
                $req = "ALTER TABLE {$tbl} DROP INDEX `{$idx_name}`";
                jrCore_db_query($req, null, false, null, false, $con);

                // Create new UNIQUE Index
                $req = "ALTER TABLE {$tbl} ADD FULLTEXT INDEX `{$idx_name}` {$idx_args}";
                jrCore_db_query($req, null, false, null, false, $con);

                // now let's see if we were successful in adding our INDEX or NOT
                $req = "SHOW INDEX FROM {$tbl}";
                $_tp = jrCore_db_query($req, 'Key_name', false, null, false, $con);
                if (is_array($_tp[$idx_name])) {
                    jrCore_logger('INF', "core: updated table fulltext index: {$idx_name} in table: {$tbl}");
                }
                else {
                    jrCore_logger('CRI', "core: unable to update table fulltext index: {$idx_name} in table: {$tbl}");
                }
            }
        }

        // COLUMN
        // [Field] => modal_value
        // [Type] => varchar(128)
        // [Null] => NO
        // [Key] =>
        // [Default] =>
        // [Extra] =>
        else {
            $col_ikey = jrCore_string_field($line, 1);
            $col_name = trim(str_replace('`', '', $col_ikey));
            $col_args = trim(str_replace($col_name . ' ', '', str_replace('`', '', $line)));

            // Are we a varchar that is going to be too long for a UTF8 InnoDB index?
            if (stristr($col_args, 'varchar') && strpos($col_args, '256')) {
                $col_args = str_replace('256', '255', $col_args);
            }

            // Make sure it is built
            if (!isset($_rt[$col_name])) {
                $req = "ALTER TABLE {$tbl} ADD `{$col_name}` {$col_args}";
                jrCore_db_query($req, null, false, null, false, $con);
                // now let's see if we were successful in adding our column or NOT
                $req = "DESCRIBE {$tbl}";
                $_tp = jrCore_db_query($req, 'Field', false, null, false, $con, false);
                if ($_tp && is_array($_tp) && is_array($_tp[$col_name])) {
                    jrCore_logger('INF', "core: created missing table column: {$col_name} in table: {$tbl}");
                }
                else {
                    jrCore_logger('CRI', "core: unable to create missing table column: {$col_name} in table: {$tbl}");
                }
            }
            else {
                // See if we are changing...
                // MEDIUMINT(11) UNSIGNED NOT NULL DEFAULT '0'
                $col_type = strtolower(jrCore_string_field($col_args, 1));
                $cur_type = strtolower(jrCore_string_field($_rt[$col_name]['Type'], 1));
                // @see https://stackoverflow.com/questions/60892749/mysql-8-ignoring-integer-lengths
                if (strpos($col_type, '(')) {
                    $col_type = substr($col_type, 0, strpos($col_type, '('));
                }
                if (strpos($cur_type, '(')) {
                    $cur_type = substr($cur_type, 0, strpos($cur_type, '('));
                }
                $new_args = strtolower(jrCore_string_field($col_args, 1));
                $cur_args = strtolower(jrCore_string_field($_rt[$col_name]['Type'], 1));
                if (strpos(' ' . $new_args, 'int') || strpos(' ' . $cur_args, 'int')) {
                    // for INTEGER comparisons we only compare the actual type - not the value in parens
                    if (strpos($new_args, '(')) {
                        $new_args = substr($new_args, 0, strpos($new_args, '('));
                    }
                    if (strpos($cur_args, '(')) {
                        $cur_args = substr($cur_args, 0, strpos($cur_args, '('));
                    }
                }
                if (($col_type && $cur_type && $col_type !== $cur_type) || $cur_args != $new_args) {
                    $req = "ALTER TABLE {$tbl} MODIFY `{$col_name}` {$col_args}";
                    $res = jrCore_db_query($req, 'COUNT', false, null, false, $con);
                    if (is_numeric($res)) {
                        jrCore_logger('INF', "core: altered table column: {$col_name} in table: {$tbl}", $req);
                    }
                }
            }
        }
    }

    // Are we changing Engine types?
    if (isset($_post['repair_modules']) && $_post['repair_modules'] == 'on') {
        // We are in an integrity check with REPAIR MODULES checked
        jrCore_db_change_table_engine($module, $table, $engine);
    }

    return true;
}

/**
 * Change the MySQL Engine for a Table
 * @param string $module Module Name
 * @param string $table Table Name
 * @param string $engine Engine = MyISAM|InnoDB
 * @param bool $truncate set to TRUE to truncate the table first (faster) but be careful!
 * @return bool
 */
function jrCore_db_change_table_engine($module, $table, $engine, $truncate = false)
{
    switch ($engine) {
        case 'Aria':
        case 'MyISAM':
        case 'InnoDB':
            break;
        default:
            return false;
    }
    $tbl = jrCore_db_table_name($module, $table);
    $req = "SHOW TABLE STATUS WHERE `Name` = '{$tbl}'";
    $_tt = jrCore_db_query($req, 'SINGLE');
    if ($_tt && is_array($_tt) && isset($_tt['Engine']) && $_tt['Engine'] != $engine) {

        // We are NOT running the right engine type - change
        if ($truncate) {
            $req = "TRUNCATE TABLE {$tbl}";
            jrCore_db_query($req);
        }

        // Alter
        $req = "ALTER TABLE {$tbl} ENGINE = {$engine}";
        jrCore_db_query($req);
        jrCore_logger('INF', "core: changed engine type to {$engine} for table: {$tbl}");

    }
    return true;
}

/**
 * Delete a column from a MySQL table
 * @param string $module Module Name
 * @param string $table Table Name
 * @param string $column Column Name
 * @return bool
 */
function jrCore_db_delete_table_column($module, $table, $column)
{
    if (jrCore_db_table_exists($module, $table)) {
        if ($_ex = jrCore_db_table_columns($module, $table)) {
            if (isset($_ex[$column])) {
                $tbl = jrCore_db_table_name($module, $table);
                $req = "ALTER TABLE {$tbl} DROP COLUMN `{$column}`";
                return jrCore_db_query($req);
            }
            return true;
        }
    }
    return false;
}

/**
 * Get the partition that is holding a specific value of the partition column
 * @see https://dev.mysql.com/doc/refman/5.6/en/partitioning-hash.html
 * @param int $partition_count Number of table partitions
 * @param int $value The value to get the partition for
 * @return int
 */
function jrCore_db_get_partition_for_integer($partition_count, $integer)
{
    // When PARTITION BY HASH is used, MySQL determines which partition of num partitions
    // to use based on the modulus of the result of the expression. In other words, for a
    // given expression expr, the partition in which the record is stored is partition
    // number N, where N = MOD(expr, num).
    return (intval($integer) % intval($partition_count));
}

/**
 * Get the number of table partitions on a given table
 * @param string $module Module
 * @param string $table Table
 * @return int Returns number of partitions or 0 (zero) if table is not partitioned
 */
function jrCore_db_get_table_partition_count($module, $table)
{
    $tbl = jrCore_db_table_name($module, $table);
    $req = "SHOW CREATE TABLE {$tbl}";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if (!empty($_rt['Create Table'])) {
        if (stripos($_rt['Create Table'], 'PARTITIONS')) {
            foreach (explode("\n", $_rt['Create Table']) as $line) {
                if (stripos($line, 'partitions') === 0) {
                    return intval(jrCore_string_field($line, 'END'));
                }
            }
        }
    }
    return 0;
}

/**
 * Alter the partition count of a table
 * @param string $module Module
 * @param string $table Table
 * @param int $partition_count NEW partition count
 * @param string $partition_column what column will be HASHED to get the partition.  @note: MUST BE AN INTEGER COLUMN
 */
function jrCore_db_alter_table_partition_count($module, $table, $partition_count, $partition_column)
{
    ini_set('max_execution_time', 28800);  // process could take a long time
    if (jrCore_checktype($partition_count, 'number_nn') && $partition_count <= 1024) {
        $current_count = jrCore_db_get_table_partition_count($module, $table);
        if ($current_count != $partition_count) {
            // We've been asked to either partition an existing table that
            // is NOT partitioned or repartition an existing partitioned table
            $req = false;
            $tbl = jrCore_db_table_name($module, $table);
            if ($partition_count == 0) {
                if ($current_count > 0) {
                    $req = "ALTER TABLE {$tbl} REMOVE PARTITIONING";
                }
            }
            elseif ($current_count == 0) {
                // We are ADDING partitioning to an existing table
                $req = "ALTER TABLE {$tbl} PARTITION BY HASH({$partition_column}) PARTITIONS " . intval($partition_count);
            }
            elseif ($partition_count < $current_count) {
                // We are SHRINKING the partition
                $req = "ALTER TABLE {$tbl} COALESCE PARTITION {$partition_count}";
            }
            else {
                // We are expanding our partitions
                $req = "ALTER TABLE {$tbl} ADD PARTITION PARTITIONS {$partition_count}";
            }
            if ($req) {
                jrCore_db_query($req);
                return true;
            }
        }
        else {
            return true;
        }
    }
    return false;
}

//--------------------------
// Archive Table Functions
//--------------------------

/**
 * Archive table entries that have registered for archiving
 * @note: Should run in queue during daily maintenance!
 */
function jrCore_db_archive_rows_in_registered_tables()
{
    if ($_tmp = jrCore_get_registered_module_features('jrCore', 'archive_table')) {
        $this_month = date('Ym');
        foreach ($_tmp as $module => $_entries) {
            foreach ($_entries as $table => $_info) {
                if (empty($_info['time_field'])) {
                    jrCore_logger('MAJ', "core: missing required time_field key for {$module} in archive_table registered module feature");
                    continue;
                }
                $tfield = trim($_info['time_field']);
                // How many months of data are stored in the active table?
                $months = (!empty($_info['active_months']) && jrCore_checktype($_info['active_months'], 'number_nz')) ? intval($_info['active_months']) : 1;
                if ($_dates = jrCore_db_get_all_archive_dates_in_table($module, $table, $tfield)) {
                    foreach ($_dates as $date) {
                        // Never archive the CURRENT month
                        if ($date == $this_month) {
                            continue;
                        }
                        // Check for extra active months
                        if ($months > 1 && $date > date('Ym', strtotime("- {$months} months"))) {
                            // We are asking to keep this month in the active table
                            continue;
                        }
                        // Archive what we can
                        if ($cnt = jrCore_db_archive_table_rows($module, $table, $tfield, $date)) {
                            // We archived some rows
                            $tmp = jrCore_number_format($cnt);
                            if ($del = jrCore_db_delete_archived_rows($module, $table, $date)) {
                                if ($cnt == $del) {
                                    $tbl = jrCore_db_table_name($module, $table);
                                    jrCore_logger('DBG', "core: successfully archived {$tmp} rows for {$date} in table: {$tbl}");
                                }
                                else {
                                    $del = jrCore_number_format($del);
                                    jrCore_logger('MAJ', "core: count mismatch for {$date} in archive table: {$table} - archived: {$tmp}, deleted: {$del}");
                                }
                            }
                            else {
                                jrCore_logger('MAJ', "core: count mismatch for {$date} in archive table: {$table} - archived: {$tmp}, deleted: 0");
                            }
                        }
                    }
                }
            }
        }
    }
    return true;
}

/**
 * Get all archive dates in a table
 * @param string $module Module
 * @param string $table Table
 * @param string $time_field Time Field
 * @return array|false
 */
function jrCore_db_get_all_archive_dates_in_table($module, $table, $time_field)
{
    $tbl = jrCore_db_table_name($module, $table);
    $req = "SELECT FROM_UNIXTIME({$time_field}, '%Y%m') AS atime FROM {$tbl} GROUP BY atime ORDER BY atime ASC";
    return jrCore_db_query($req, 'atime', false, 'atime');
}

/**
 * Get the table to get data for a primary key
 * @note Always returns a table - will return the ACTIVE table if id is not in an archive table!
 * @param string $module Module
 * @param string $table Table
 * @param int $unique_id Primary Key ID
 * @return string
 */
function jrCore_db_get_archive_table_for_unique_id($module, $table, $unique_id)
{
    if ($_rt = jrCore_db_get_all_archive_table_min_max()) {
        if (isset($_rt["{$module}_{$table}"])) {
            $uid = (int) $unique_id;
            foreach ($_rt["{$module}_{$table}"] as $date => $d) {
                if ($d[0] <= $uid && $d[1] >= $uid) {
                    return jrCore_db_table_name($module, "{$table}_{$date}");
                }
            }
        }
    }
    return jrCore_db_table_name($module, $table);
}

/**
 * Get archive tables for an array of IDs
 * @param string $module Module
 * @param string $table Table
 * @param array $_ids Array of IDs
 * @return array
 */
function jrCore_get_archive_tables_for_array_of_ids($module, $table, $_ids)
{
    $_tb = array();
    foreach ($_ids as $id) {
        $id = (int) $id;
        if (!empty($id)) {
            if ($tbl = jrCore_db_get_archive_table_for_unique_id($module, $table, $id)) {
                if (!isset($_tb[$tbl])) {
                    $_tb[$tbl] = array();
                }
                $_tb[$tbl][$id] = $id;
            }
        }
    }
    return $_tb;
}

/**
 * Create a dated Archive table based on an existing table
 * @param string $module Module
 * @param string $table Table
 * @param int $date Date in YYYYMM format
 * @return false
 */
function jrCore_db_create_archive_table($module, $table, $date)
{
    // Must be in YYYYMM format
    if (strlen($date) !== 6) {
        return false;
    }
    $tbl = jrCore_db_table_name($module, $table);
    $sql = "SHOW CREATE TABLE {$tbl}";
    $_rt = jrCore_db_query($sql, 'SINGLE', false, null, false);
    if ($_rt && is_array($_rt) && !empty($_rt['Create Table'])) {
        $sql = preg_replace(', AUTO_INCREMENT=[0-9]* ,', ' ', $_rt['Create Table']);
        $sql = str_replace(' AUTO_INCREMENT', '', $sql);
        $sql = str_replace($tbl, "{$tbl}_{$date}", $sql);
        jrCore_db_query($sql, null, false, null, false, null, false);
        return jrCore_db_table_exists($module, "{$table}_{$date}");
    }
    return false;
}

/**
 * Move table rows from an "active" table to an "archive" table
 * @param string $module Module
 * @param string $table Table
 * @param string $time_field Column in table with epoch time to use for archive - must be EPOCH time
 * @param int $date Date in YYYYMM format
 * @return int|false
 */
function jrCore_db_archive_table_rows($module, $table, $time_field, $date)
{
    // Must be in YYYYMM format
    if (strlen($date) !== 6) {
        return false;
    }
    // Make sure we are not archiving THIS month
    if ($date == date('Ym')) {
        return false;
    }
    // Make sure table exists
    if (!jrCore_db_create_archive_table($module, $table, $date)) {
        return false;
    }
    if (!$_cl = jrCore_db_table_columns($module, $table)) {
        return false;
    }
    $tb1 = jrCore_db_table_name($module, $table);
    $tb2 = jrCore_db_table_name($module, "{$table}_{$date}");
    $col = implode(',', array_keys($_cl));
    $sql = "INSERT IGNORE INTO {$tb2} (SELECT {$col} FROM {$tb1} WHERE FROM_UNIXTIME({$time_field}, '%Y%m') = '{$date}')";
    $cnt = jrCore_db_query($sql, 'COUNT');
    if ($cnt > 0) {
        jrCore_db_update_archive_table_min_max($module, $table, $date);
        return $cnt;
    }
    return false;
}

/**
 * Delete rows from a table that have already been archived
 * @note Run from a Queue!
 * @param string $module Module
 * @param string $table Table
 * @param int $date Date that has ALREADY been archived!
 * @param int $limit Number of rows to delete at once
 * @param int $usleep Microseconds to sleep between batches
 * @return int|false
 */
function jrCore_db_delete_archived_rows($module, $table, $date, $limit = 500, $usleep = 100000)
{
    // We have to know our MIN and MAX
    if (!$_mm = jrCore_db_get_archive_table_min_max($module, $table, $date)) {
        return false;
    }
    if (!$key = jrCore_db_get_table_primary_key($module, $table)) {
        return false;
    }
    $ttl = 0;
    $lim = intval($limit);
    $min = intval($_mm[0]);
    if ($min === 0) {
        jrCore_logger('CRI', "core: min primary key value for {$module}/{$table} archive date {$date} is 0");
        return false;
    }
    $max = intval($_mm[1]);
    if ($max === 0) {
        jrCore_logger('CRI', "core: max primary key value for {$module}/{$table} archive date {$date} is 0");
        return false;
    }
    $tbl = jrCore_db_table_name($module, $table);
    while (true) {
        $sql = "DELETE FROM {$tbl} WHERE {$key} >= {$min} AND {$key} <= {$max} LIMIT {$lim}";
        $cnt = jrCore_db_query($sql, 'COUNT');
        if ($cnt && $cnt === $lim) {
            $ttl += $cnt;
            jrCore_start_timer('sleep');
            usleep($usleep);
            jrCore_stop_timer('sleep');
        }
        else {
            // We have deleted all matching
            $ttl += $cnt;
            break;
        }
    }
    return $ttl;
}

/**
 * Get all min/max data for modules
 * @param bool $rebuild set to TRUE to force cache rebuild
 * @return array|false
 */
function jrCore_db_get_all_archive_table_min_max($rebuild = false)
{
    $key = 'archive_table_min_max';
    $_rt = false;
    if (!$rebuild) {
        $_rt = jrCore_is_cached('jrCore', $key, false, false);
    }
    if (!$_rt || $rebuild) {
        $tbl = jrCore_db_table_name('jrCore', 'archive_date');
        $sql = "SELECT * FROM {$tbl}";
        $_tm = jrCore_db_query($sql, 'NUMERIC');
        if ($_tm && is_array($_tm)) {
            $_rt = array();
            foreach ($_tm as $t) {
                if (!isset($_rt["{$t['archive_table']}"])) {
                    $_rt["{$t['archive_table']}"] = array();
                }
                $_rt["{$t['archive_table']}"]["{$t['archive_date']}"] = array(intval($t['archive_min']), intval($t['archive_max']));
            }
        }
        else {
            $_rt = 'no_items';
        }
        jrCore_add_to_cache('jrCore', $key, $_rt, 0, 0, false, false);
    }
    return (is_array($_rt)) ? $_rt : false;
}

/**
 * Get all archive dates for an archive table
 * @param string $module Module
 * @param string $table Table
 * @return array|false
 */
function jrCore_db_get_all_archive_dates($module, $table)
{
    if ($_rt = jrCore_db_get_all_archive_table_min_max()) {
        foreach ($_rt as $tbl => $_dates) {
            if ($tbl == "{$module}_{$table}") {
                return array_keys($_dates);
            }
        }
    }
    return false;
}

/**
 * Get the MIN and MAX primary key for an archive table
 * @param string $module Module
 * @param string $table Table
 * @param int $date Date in YYYYMM format
 * @return array|false
 */
function jrCore_db_get_archive_table_min_max($module, $table, $date)
{
    if ($_rt = jrCore_db_get_all_archive_table_min_max()) {
        return (!empty($_rt["{$module}_{$table}"][$date])) ? $_rt["{$module}_{$table}"][$date] : false;
    }
    return false;
}

/**
 * Update the MIN and MAX in an archive table
 * @param string $module Module
 * @param string $table Table
 * @param int $date Date in YYYYMM format
 * @return bool
 */
function jrCore_db_update_archive_table_min_max($module, $table, $date)
{
    // Get auto_increment field
    if (!$key = jrCore_db_get_table_primary_key($module, $table)) {
        return false;
    }
    $dat = intval($date);
    $tb1 = jrCore_db_table_name('jrCore', 'archive_date');
    $tb2 = jrCore_db_table_name($module, "{$table}_{$date}");
    $_rq = array(
        "SELECT MIN({$key}), MAX({$key}) INTO @min, @max FROM {$tb2}",
        "INSERT INTO {$tb1} (archive_table, archive_date, archive_min, archive_max)
         VALUES ('{$module}_{$table}', {$dat}, @min, @max)
         ON DUPLICATE KEY UPDATE archive_min = @min, archive_max = @max"
    );
    jrCore_db_multi_query($_rq, false);
    jrCore_delete_cache('jrCore', 'archive_table_min_max', false, false);
    return true;
}

/**
 * Get an array of Archive tables that are needed for the given date range
 * @param string $module Module
 * @param string $table Table
 * @param int $new_date Date in YYYYMM format
 * @param int $old_date Date in YYYYMM format
 * @return array|false
 */
function jrCore_db_get_union_tables_array_for_archive_dates($module, $table, $new_date, $old_date)
{
    // Must be in YYYYMM format
    if (strlen($new_date) !== 6 || strlen($old_date) !== 6) {
        return false;
    }
    if (!$_tmp = jrCore_get_registered_module_features('jrCore', 'archive_table')) {
        return false;
    }
    if (!isset($_tmp[$module])) {
        return false;
    }
    if (!isset($_tmp[$module][$table])) {
        return false;
    }
    if ($old_date > $new_date) {
        $tmp_date = $old_date;
        $old_date = $new_date;
        $new_date = $tmp_date;
        unset($tmp_date);
    }
    $months = (!empty($_tmp[$module][$table]['active_months'])) ? (intval($_tmp[$module][$table]['active_months']) - 1) : 0;
    $active = date('Ym', strtotime("-{$months} months"));
    if ($old_date >= $active) {
        // No UNION needed
        return array(jrCore_db_table_name($module, $table));
    }

    // Make sure each table in the range exists
    $_rt = array();
    $tmp = $old_date;
    while ($tmp <= $new_date) {
        if (jrCore_db_get_archive_table_min_max($module, $table, $tmp)) {
            $_rt[$tmp] = 1;
        }
        $date = substr($tmp, 0, 4) . '-' . substr($tmp, 4, 2) . '-15 12:00 +1 month';
        $tmp  = date('Ym', strtotime($date));
    }

    $_rq = array();
    if (!isset($_rt[$new_date])) {
        // Must include ACTIVE table as well
        $_rq[] = jrCore_db_table_name($module, $table);
    }
    if (count($_rt) > 0) {
        foreach ($_rt as $date => $ignore) {
            $_rq[] = jrCore_db_table_name($module, "{$table}_{$date}");
        }
    }
    return $_rq;
}

/**
 * Get the UNION ALL string for a range of archive table dates (inclusive)
 * @param string $module Module
 * @param string $table Table
 * @param int $new_date Date in YYYYMM format
 * @param int $old_date Date in YYYYMM format
 * @param string $alias Table alias
 * @param string $columns Columns to select
 * @param string $where additional WHERE clause in UNION selects
 * @return string|false
 */
function jrCore_db_get_union_tables_for_archive_dates($module, $table, $new_date, $old_date, $alias, $columns = '*', $where = null)
{
    if (!$_rq = jrCore_db_get_union_tables_array_for_archive_dates($module, $table, $new_date, $old_date)) {
        return false;
    }
    $cnt = count($_rq);
    if ($cnt === 1) {
        // No union needed
        return reset($_rq) . " {$alias} {$where}";
    }
    return "(SELECT {$columns} FROM " . implode(" {$where} UNION ALL SELECT {$columns} FROM ", $_rq) . " {$where}) AS {$alias}";
}
