<?php
 /**
 * Jamroom 2 Factor Authentication module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrTwoFactor_meta()
{
    return array(
        'name'        => '2 Factor Authentication',
        'url'         => 'twofactor',
        'version'     => '2.0.0',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Enable 2 factor email authentication in user accounts',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/2963/2-factor-authentication',
        'priority'    => 50, // HIGH - run before other listeners
        'requires'    => 'jrCore:6.5.12',
        'license'     => 'mpl',
        'category'    => 'users'
    );
}

/**
 * init
 */
function jrTwoFactor_init()
{
    // Quota Support
    $_tmp = array(
        'label' => 'Enable in User Account',
        'help'  => 'If checked, User Accounts associated with Profiles in this quota will be allowed to use Two Factor Authentication.'
    );
    jrCore_register_module_feature('jrCore', 'quota_support', 'jrTwoFactor', 'on', $_tmp);

    // Recycle Bin support
    jrCore_register_module_feature('jrCore', 'recycle_bin_user_id_table', 'jrTwoFactor', 'session', 'factor_user_id');

    // CSS
    jrCore_register_module_feature('jrCore', 'css', 'jrTwoFactor', 'jrTwoFactor.css');

    // We listen for user log ins
    jrCore_register_event_listener('jrUser', 'login_success', 'jrTwoFactor_login_success_listener');

    // Add a 2 Factor Authentication enabled checkbox in User Account
    jrCore_register_event_listener('jrCore', 'form_display', 'jrTwoFactor_user_account_form_listener');
    jrCore_register_event_listener('jrCore', 'ten_minute_maintenance', 'jrTwoFactor_ten_minute_maintenance_listener');

    return true;
}

//-------------------
// Event Listeners
//-------------------

/**
 * Redirect to Authentication screen on successful login
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrTwoFactor_login_success_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (!empty($_user['_user_id']) && !isset($_SESSION['jrtwofactor_authenticated'])) {
        if (jrTwoFactor_is_enabled_for_user($_SESSION)) {
            // When we get to here, the session will have already been established.
            // We need to destroy as we will set it back up after passing 2 factor.
            // Setup our "remember me" cookie if requested
            if (isset($_post['user_remember']) && $_post['user_remember'] === 'on') {
                $_SESSION['user_remember'] = 'on';
            }
            $uid = (int) $_user['_user_id'];
            $key = jrCore_create_unique_string(6);
            $tbl = jrCore_db_table_name('jrTwoFactor', 'session');
            $req = "INSERT INTO {$tbl} (factor_code, factor_created, factor_user_id, factor_session)
                    VALUES ('{$key}', UNIX_TIMESTAMP(), {$uid}, '" . jrCore_db_escape(json_encode($_SESSION)) . "')";
            $cnt = jrCore_db_query($req, 'COUNT');
            if ($cnt === 1) {
                // Notify User
                $url = jrCore_get_module_url('jrTwoFactor');
                $_rp = array(
                    'factor_code'         => $key,
                    'factor_login_url'    => jrCore_get_base_url() . "/{$url}/login/{$uid}",
                    'factor_continue_url' => jrCore_get_base_url() . "/{$url}/login/{$uid}/{$key}"
                );
                list($sub, $msg) = jrCore_parse_email_templates('jrTwoFactor', 'code', $_rp);
                jrCore_send_email($_user['user_email'], $sub, $msg);
                // Destroy existing session and redirect
                jrUser_session_destroy();
                jrCore_location($_rp['factor_login_url']);
            }
            jrCore_logger('CRI', 'Unable to save two factor session to database - check debug_log');
            jrCore_notice_page('error', 'A system level error was encountered trying to log in - please try again shortly.');
        }
    }
    return $_data;
}

/**
 * Add Two Factor enable checkbox on User Account page
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrTwoFactor_user_account_form_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_data['form_view']) && $_data['form_view'] == 'jrUser/account') {
        if (!empty($_data['form_params']['values']['_user_id']) && $_data['form_params']['values']['_user_id'] == $_user['_user_id']) {
            // Are we enabled?
            $enabled = false;
            if (jrTwoFactor_user_is_in_required_group($_user)) {
                $enabled = true;
            }
            elseif (jrProfile_get_quota_value($_user, 'jrTwoFactor', 'allowed', 'off') == 'on') {
                $enabled = true;
            }
            if ($enabled) {
                // Add in activate checkbox
                $_lng = jrUser_load_lang_strings();
                $_tmp = array(
                    'name'     => 'user_twofactor_enabled',
                    'label'    => $_lng['jrTwoFactor'][8],
                    'help'     => $_lng['jrTwoFactor'][9],
                    'type'     => 'checkbox',
                    'validate' => 'onoff',
                    'required' => false
                );
                if (jrTwoFactor_user_is_in_required_group($_user)) {
                    $_tmp['value']    = 'on';
                    $_tmp['disabled'] = 'disabled';
                    $_tmp['sublabel'] = $_lng['jrTwoFactor'][12];
                }
                jrCore_form_field_create($_tmp);
            }
        }
    }
    return $_data;
}

/**
 * Keep 2 factor session table clean
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrTwoFactor_ten_minute_maintenance_listener($_data, $_user, $_conf, $_args, $event)
{
    $num = 0;
    while (true) {
        $tbl = jrCore_db_table_name('jrTwoFactor', 'session');
        $req = "DELETE FROM {$tbl} WHERE factor_created < (UNIX_TIMESTAMP() - 3600) LIMIT 500";
        $cnt = jrCore_db_query($req, 'COUNT');
        if ($cnt < 500 || $num >= 20) {
            break;
        }
        $num++;
        usleep(250000);
    }
    return $_data;
}

//-------------------
// Functions
//-------------------

/**
 * Return TRUE if 2 factor auth is enabled for a user
 * @param array $_user
 * @return bool
 */
function jrTwoFactor_is_enabled_for_user($_user)
{
    if (jrTwoFactor_user_is_in_required_group($_user)) {
        return true;
    }
    elseif (jrProfile_get_quota_value($_user, 'jrTwoFactor', 'allowed', 'off') != 'off') {
        // This user CAN set by quota
        if (!empty($_user['user_twofactor_enabled']) && $_user['user_twofactor_enabled'] == 'on') {
            return true;
        }
    }
    return false;
}

/**
 * Return TRUE if viewing user is in a required Two Factor group
 * @param array $_user
 * @return bool
 */
function jrTwoFactor_user_is_in_required_group($_user)
{
    if (jrUser_is_logged_in()) {
        if ($groups = jrCore_get_config_value('jrTwoFactor', 'required', false)) {
            if ($groups != 'none') {
                if (jrCore_user_is_part_of_group($groups, $_user)) {
                    return true;
                }
            }
        }
    }
    return false;
}

/**
 * Get required user groups
 * @return array
 */
function jrTwoFactor_get_required_user_groups()
{
    $_opt = array(
        'none'   => 'Disabled',
        'user'   => '(group) All Users',
        'master' => '(group) Master Admins',
        'admin'  => '(group) Profile Admins',
        'power'  => '(group) Power Users (multiple profiles)'
    );
    $_qta = jrProfile_get_quotas();
    if (isset($_qta) && is_array($_qta)) {
        foreach ($_qta as $qid => $qname) {
            $_opt[$qid] = "(quota) {$qname}";
        }
    }
    return $_opt;
}
