<?php
 /**
 * Jamroom Users module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2021 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * view: login_save
 * @param array $_post Posted Data
 * @param array $_user Viewing User data
 * @param array $_conf Global Config
 */
function view_jrUser_login_save($_post, $_user, $_conf)
{
    global $_user;
    // @note: DO NOT use jrCore_form_validate here if we are using TEMPLATES
    // we have to handle validation on our own due to caching
    if (jrCore_get_config_value('jrUser', 'use_templates', 'off') == 'off') {
        jrCore_form_validate($_post);
    }
    else {
        if (empty($_post['user_email_or_name']) || empty($_post['user_password'])) {
            jrCore_set_form_notice('error', 26);
            jrCore_form_result();
        }
        // We have to account for FORM DESIGNER custom fields
        if ($_fields = jrCore_get_designer_form_fields('jrUser', 'login')) {
            foreach ($_fields as $f) {
                $_post = jrCore_form_validate_posted_field('jrUser', $f, $_post);
            }
        }
    }

    $uip = jrCore_get_ip();
    if (jrUser_get_password_attempts($uip) > 10) {
        // We've had more than 10 login attempts from this IP
        jrCore_set_form_notice('error', 26);
        jrCore_form_result();
    }

    // Make sure user is valid
    $key1 = 'user_name';
    $key2 = 'user_email';
    if (strpos($_post['user_email_or_name'], '@')) {
        $key1 = 'user_email';
        $key2 = 'user_name';
    }
    $_rt = jrCore_db_get_item_by_key('jrUser', $key1, $_post['user_email_or_name'], false, true);
    if (!$_rt || !is_array($_rt)) {
        $_rt = jrCore_db_get_item_by_key('jrUser', $key2, $_post['user_email_or_name'], false, true);
        if (!$_rt || !is_array($_rt)) {
            jrCore_set_form_notice('error', 26);
            jrCore_form_result();
        }
    }

    // Validate password
    if (!jrUser_verify_password_hash($_post['user_password'], $_rt['user_password'])) {
        if (isset($_rt['user_group']) && $_rt['user_group'] != 'user') {
            // This is a MASTER or ADMIN user account - log it
            jrCore_logger('MAJ', "user: invalid login attempt for {$_rt['user_group']} user account {$_rt['user_name']} - bad password");
        }
        jrCore_set_form_notice('error', 26);
        jrCore_form_result();
    }

    // Make sure account is validated
    if (!isset($_rt['user_validated']) || $_rt['user_validated'] != '1') {
        $_ln = jrUser_load_lang_strings();
        if (isset($_rt['quota_jrUser_signup_method']) && $_rt['quota_jrUser_signup_method'] == 'email') {
            // Give the user the ability to resend the activation email
            $_SESSION['allow_activation_resend'] = 1;
            $tmp                                 = jrCore_page_button('resend', $_ln['jrUser'][28], "jrCore_window_location('" . $_conf['jrCore_base_url'] . '/' . $_post['module_url'] . '/activation_resend/' . $_rt['_user_id'] . "')");
            jrCore_set_form_notice('error', $_ln['jrUser'][27] . '<br><br>' . $tmp, false);
        }
        else {
            jrCore_set_form_notice('error', $_ln['jrUser'][27]);
        }
        jrCore_form_result();
    }

    // Make sure account is not blocked
    if (isset($_rt['user_blocked']) && $_rt['user_blocked'] == 1) {
        jrCore_set_form_notice('error', 10);
        jrCore_form_result();
    }

    // Make sure account is active
    if (!isset($_rt['user_active']) || $_rt['user_active'] != 1) {
        jrCore_set_form_notice('error', 29);
        jrCore_form_result();
    }

    // Make sure we have a valid profile id
    if (!isset($_rt['_profile_id']) || !jrCore_checktype($_rt['_profile_id'], 'number_nz')) {
        jrCore_set_form_notice('error', 26);
        jrCore_form_result();
    }

    // Make sure the profile we are attached to is still valid
    if (!isset($_rt['profile_url'])) {
        // The profile we are linked to did not get added in - check if it exists
        $_pr = jrCore_db_get_item('jrProfile', $_rt['_profile_id']);
        if (!$_pr || !is_array($_pr)) {
            // The profile this user account was linked to no longer exists
            // See if this user is linked to another profile in the system
            $_lp = jrProfile_get_user_linked_profiles($_rt['_user_id']);
            if ($_lp && is_array($_lp)) {
                if (isset($_lp["{$_rt['_profile_id']}"])) {
                    // Cleanup
                    unset($_lp["{$_rt['_profile_id']}"]);
                    $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
                    $req = "DELETE FROM {$tbl} WHERE user_id = '{$_rt['_user_id']}' AND profile_id = '{$_rt['_profile_id']}'";
                    jrCore_db_query($req);
                }
                if (count($_lp) > 0) {
                    // The first one that is
                    foreach ($_lp as $pid => $uid) {
                        $_pr = jrCore_db_get_item('jrProfile', $pid);
                        if ($_pr && is_array($_pr)) {
                            jrCore_db_update_item('jrUser', $_rt['_user_id'], array(), array('_profile_id' => $pid));
                            break;
                        }
                    }
                }
                else {
                    // Bad user account
                    jrCore_logger('CRI', "user: {$_rt['user_name']} attempted to log in to an account with an invalid or non-linked profile", $_rt);
                    jrCore_notice_page('error', 'Your Account is missing important information and will not function properly.<br>Please contact the site administrator for assistance on accessing your account.', $_conf['jrCore_base_url'], null, false);
                }
            }
        }
    }

    // See if this user is logging in for the first time on a new device
    if (isset($_rt['quota_jrUser_device_notice']) && $_rt['quota_jrUser_device_notice'] == 'on') {
        jrUser_notify_if_new_device($_rt['_user_id']);
    }

    // Startup Session and login
    if (!$_SESSION = jrUser_get_user_session_data($_rt['_user_id'], $_rt['_profile_id'], false)) {
        jrCore_logger('CRI', "user: {$_rt['user_name']} attempted to log in to an account with an invalid or non-linked profile", $_rt);
        jrCore_set_form_notice('error', 'Your Account is missing important information and will not function properly.<br>Please contact the site administrator for assistance on accessing your account.', false);
        jrCore_form_result();
    }
    if (!is_array($_SESSION)) {
        // Could not start a session
        jrCore_set_form_notice('error', 10);
        jrCore_form_result();
    }

    // Save home profile keys
    jrUser_save_profile_home_keys();

    // Maintenance login check
    if (jrCore_is_maintenance_mode($_conf, $_post)) {
        jrCore_set_form_notice('error', 30);
        jrCore_form_result();
    }

    // Remove any forgot password entries
    jrUser_delete_forgot_password_entries($_SESSION['_user_id']);

    // User has logged in - reset any failed password attempts on this IP
    jrUser_reset_password_attempts($uip);

    // Update last login time
    $_data = array(
        'user_last_login' => 'UNIX_TIMESTAMP()'
    );
    jrCore_db_update_item('jrUser', $_SESSION['_user_id'], $_data);
    $_SESSION['user_last_login'] = time();
    $_SESSION['is_logged_in']    = 'yes';

    // Bring in all profile and Quota info
    $_SESSION = jrUser_session_start();

    // Setup our "remember me" cookie if requested
    if (isset($_post['user_remember']) && $_post['user_remember'] === 'on') {
        jrUser_session_set_login_cookie($_SESSION['_user_id']);
    }
    else {
        jrUser_session_delete_login_cookie();
    }

    // Send out trigger on successful account activation
    $_user = $_SESSION;
    $_user = jrCore_trigger_event('jrUser', 'login_success', $_user);
    if (jrCore_get_config_value('jrUser', 'login_activity_log', 'on') == 'on') {
        jrCore_logger('INF', "user: successful login by {$_post['user_email_or_name']}");
    }
    jrCore_form_delete_session();

    // If login_page set in quota, redirect as appropriate
    if (isset($_user['quota_jrUser_login_page'])) {
        $login_page = strtolower(trim($_user['quota_jrUser_login_page']));
        if ($login_page == 'index') {
            jrCore_form_result($_conf['jrCore_base_url']);
        }
        elseif (jrCore_checktype($login_page, 'url')) {
            jrCore_form_result($login_page);
        }
    }

    // Redirect to Profile or Saved Location
    $url = jrUser_get_saved_cookie_location();
    if ($url && $url != $_conf['jrCore_base_url'] && $url != $_conf['jrCore_base_url'] . '/' && !strpos($url, '/signup')) {
        jrCore_form_result($url);
    }
    $url = "{$_conf['jrCore_base_url']}/{$_user['profile_url']}";
    jrCore_form_result($url);
}
