<?php

/**
 * Invoice Ninja (https://invoiceninja.com).
 *
 * @link https://github.com/invoiceninja/invoiceninja source repository
 *
 * @copyright Copyright (c) 2025. Invoice Ninja LLC (https://invoiceninja.com)
 *
 * @license https://www.elastic.co/licensing/elastic-license
 */

namespace App\Livewire;

use App\DataMapper\ClientSettings;
use App\Factory\ClientFactory;
use App\Jobs\Mail\NinjaMailerJob;
use App\Jobs\Mail\NinjaMailerObject;
use App\Libraries\MultiDB;
use App\Mail\ContactPasswordlessLogin;
use App\Models\Client;
use App\Models\ClientContact;
use App\Models\Invoice;
use App\Models\Subscription;
use App\Repositories\ClientContactRepository;
use App\Repositories\ClientRepository;
use App\Utils\Ninja;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Livewire\Component;

class BillingPortalPurchase extends Component
{
    /**
     * Random hash generated by backend to handle the tracking of state.
     *
     * @var string
     */
    public $hash;

    /**
     * Top level text on the left side of billing page.
     *
     * @var string
     */
    public $heading_text;


    /**
     * E-mail address model for user input.
     *
     * @var string
     */
    public $email;

    /**
     * Password model for user input.
     *
     * @var string
     */
    public $password;

    /**
     * This arrives as an int and we resolve in the mount method
     *
     * @var int|Subscription
     */
    public $subscription;

    /**
     * Instance of client contact.
     *
     * @var null|ClientContact
     */
    public $contact;

    /**
     * Rules for validating the form.
     *
     */
    protected $rules = [
        'email' => ['required', 'email'],
    ];

    /**
     * Id for CompanyGateway record.
     *
     * @var string|integer
     */
    public $company_gateway_id;

    /**
     * Id for GatewayType.
     *
     * @var string|integer
     */
    public $payment_method_id;

    private $user_coupon;

    /**
     * List of steps that frontend form follows.
     *
     * @var array
     */
    public $steps = [
        'passed_email' => false,
        'existing_user' => false,
        'check_rff' => false,
        'fetched_payment_methods' => false,
        'fetched_client' => false,
        'show_start_trial' => false,
        'passwordless_login_sent' => false,
        'started_payment' => false,
        'discount_applied' => false,
        'show_loading_bar' => false,
        'not_eligible' => null,
        'not_eligible_message' => null,
        'payment_required' => true,
    ];

    /**
     * List of payment methods fetched from client.
     *
     * @var array
     */
    public $methods = [];

    /**
     * Instance of \App\Models\Invoice
     *
     * @var Invoice
     */
    public $invoice;

    /**
     * Coupon model for user input
     *
     * @var string
     */
    public $coupon;

    /**
     * Quantity for seats
     *
     * @var int
     */
    public $quantity;

    /**
     * First-hit request data (queries, locales...).
     *
     * @var array
     */
    public $request_data;

    /**
     * @var float
     */
    public $price;

    /**
     * Disabled state of passwordless login button.
     *
     * @var bool
     */
    public $passwordless_login_btn = false;

    /**
     * Instance of company.
     *
     * @var \App\Models\Company
     */
    public $company;

    public $db;

    /**
     * Campaign reference.
     *
     * @var string|null
     */
    public $campaign;

    public ?string $contact_first_name;

    public ?string $contact_last_name;

    public ?string $contact_email;

    public ?string $client_city;

    public ?string $client_postal_code;

    public function mount()
    {
        MultiDB::setDb($this->db);

        $this->subscription = Subscription::query()->with('company')->find($this->subscription);

        $this->company = $this->subscription->company;

        $this->quantity = 1;

        $this->price = $this->subscription->price;

        if (request()->query('coupon')) {
            $this->coupon = request()->query('coupon');
            $this->handleCoupon();
        } elseif (strlen($this->subscription->promo_code ?? '') == 0 && $this->subscription->promo_discount > 0) {
            $this->price = $this->subscription->promo_price;
        }

        /* Leave this here, otherwise a logged in user will need to reauth... painfully */
        if (Auth::guard('contact')->check()) {
            return $this->getPaymentMethods(auth()->guard('contact')->user());
        }
    }

    /**
     * Handle user authentication
     *
     * @return $this|bool|void
     */
    public function authenticate()
    {
        $this->validate();

        $contact = ClientContact::where('email', $this->email)
            ->where('company_id', $this->subscription->company_id)
            ->first();

        if ($contact && $this->steps['existing_user'] === false) {
            return $this->steps['existing_user'] = true;
        }

        if ($contact && $this->steps['existing_user']) {
            $attempt = Auth::guard('contact')->attempt(['email' => $this->email, 'password' => $this->password, 'company_id' => $this->subscription->company_id]);

            return $attempt
                ? $this->getPaymentMethods($contact)
                : session()->flash('message', 'These credentials do not match our records.');
        }

        $this->steps['existing_user'] = false;

        $this->contact = $this->createBlankClient();

        if ($this->contact && $this->contact instanceof ClientContact) {
            $this->getPaymentMethods($this->contact);
        }
    }

    /**
     * Create a blank client. Used for new customers purchasing.
     *
     * @return mixed
     * @throws \Laracasts\Presenter\Exceptions\PresenterException
     */
    protected function createBlankClient()
    {
        $company = $this->subscription->company;
        $user = $this->subscription->user;
        $user->setCompany($company);

        $client_repo = new ClientRepository(new ClientContactRepository());

        $data = [
            'name' => '',
            'contacts' => [
                ['email' => $this->email],
            ],
            'client_hash' => Str::random(40),
            'settings' => ClientSettings::defaults(),
        ];

        foreach ($this->request_data as $field => $value) {
            if (in_array($field, Client::$subscriptions_fillable)) {
                $data[$field] = $value;
            }

            if (in_array($field, ClientContact::$subscription_fillable)) {
                $data['contacts'][0][$field] = $value;
            }
        }

        if (array_key_exists('currency_id', $this->request_data)) {

            /** @var \Illuminate\Support\Collection<\App\Models\Currency> */
            $currencies = app('currencies');

            $currency = $currencies->first(function ($item) {
                return $item->id == $this->request_data['currency_id'];
            });

            if ($currency) {
                $data['settings']->currency_id = $currency->id;
            }
        } elseif ($this->subscription->group_settings && property_exists($this->subscription->group_settings->settings, 'currency_id')) {

            /** @var \Illuminate\Support\Collection<\App\Models\Currency> */
            $currencies = app('currencies');

            $currency = $currencies->first(function ($item) {
                return $item->id == $this->subscription->group_settings->settings->currency_id;
            });

            if ($currency) {
                $data['settings']->currency_id = $currency->id;
            }
        }

        if (array_key_exists('locale', $this->request_data)) {
            $request = $this->request_data;

            /** @var \Illuminate\Support\Collection<\App\Models\Language> */
            $languages = app('languages');
            $record = $languages->first(function ($item) use ($request) {
                return $item->locale == $request['locale'];
            });

            if ($record) {
                $data['settings']['language_id'] = (string)$record->id;
            }
        }

        $client = $client_repo->save($data, ClientFactory::create($company->id, $user->id));

        return $client->fresh()->contacts->first();
    }

    /**
     * Fetching payment methods from the client.
     *
     * @param ClientContact $contact
     * @return $this
     */
    protected function getPaymentMethods(ClientContact $contact): self
    {
        $this->contact = $contact;

        Auth::guard('contact')->loginUsingId($contact->id, true);

        if ($this->subscription->trial_enabled) {
            $this->heading_text = ctrans('texts.plan_trial');
            $this->steps['show_start_trial'] = true;

            return $this;
        }

        if ((float)$this->price <= 0) {

            $this->steps['payment_required'] = false;
            $this->steps['fetched_payment_methods'] = false;
            $this->heading_text = ctrans('texts.payment_methods');
            return $this;
        } else {
            // $this->steps['fetched_payment_methods'] = true;
        }

        $this->methods = $contact->client->service()->getPaymentMethods($this->price);

        $method_values = array_column($this->methods, 'is_paypal');
        $is_paypal = in_array('1', $method_values);

        if ($is_paypal && !$this->steps['check_rff']) {
            $this->rff();
        } elseif (!$this->steps['check_rff']) {
            $this->steps['fetched_payment_methods'] = true;
        }

        $this->heading_text = ctrans('texts.payment_methods');

        return $this;
    }

    protected function rff()
    {
        $this->contact_first_name = $this->contact->first_name;
        $this->contact_last_name = $this->contact->last_name;
        $this->contact_email = $this->contact->email;
        $this->client_city = $this->contact->client->city;
        $this->client_postal_code = $this->contact->client->postal_code;

        $this->steps['check_rff'] = true;

        return $this;
    }

    public function handleRff()
    {
        $validated = $this->validate([
            'contact_first_name' => ['required'],
            'contact_last_name' => ['required'],
            'client_city' => ['required'],
            'client_postal_code' => ['required'],
            'contact_email' => ['required', 'email'],
        ]);

        $this->contact->first_name = $validated['contact_first_name'];
        $this->contact->last_name = $validated['contact_last_name'];
        $this->contact->email = $validated['contact_email'];
        $this->contact->client->postal_code = $validated['client_postal_code'];
        $this->contact->client->city = $validated['client_city'];

        $this->contact->pushQuietly();

        $this->steps['fetched_payment_methods'] = true;

        return $this->getPaymentMethods($this->contact);
    }

    /**
     * Middle method between selecting payment method &
     * submitting the from to the backend.
     *
     * @param $company_gateway_id
     * @param $gateway_type_id
     */
    public function handleMethodSelectingEvent($company_gateway_id, $gateway_type_id, $is_paypal = false)
    {
        $this->company_gateway_id = $company_gateway_id;
        $this->payment_method_id = $gateway_type_id;

        $this->handleBeforePaymentEvents();

    }

    /**
     * Method to handle events before payments.
     *
     * @return void
     */
    public function handleBeforePaymentEvents()
    {
        $this->steps['started_payment'] = true;
        $this->steps['show_loading_bar'] = true;

        $data = [
            'client_id' => $this->contact->client->id,
            'date' => now()->format('Y-m-d'),
            'invitations' => [[
                'key' => '',
                'client_contact_id' => $this->contact->hashed_id,
            ]],
            'user_input_promo_code' => $this->coupon,
            'coupon' => empty($this->subscription->promo_code) ? '' : $this->coupon,
            'quantity' => $this->quantity,
        ];

        $is_eligible = $this->subscription->service()->isEligible($this->contact);

        if (is_array($is_eligible) && $is_eligible['message'] != 'Success') {
            $this->steps['not_eligible'] = true;
            $this->steps['not_eligible_message'] = $is_eligible['message'];
            $this->steps['show_loading_bar'] = false;

            return;
        }

        $this->invoice = $this->subscription
            ->service()
            ->createInvoice($data, $this->quantity)
            ->service()
            ->markSent()
            ->fillDefaults()
            ->adjustInventory()
            ->save();

        $context = 'purchase';

        if (config('ninja.ninja_default_company_id') == $this->subscription->company_id && $this->subscription->service()->recurring_products()->first()?->product_key == 'whitelabel') {
            $context = 'whitelabel';
        }

        if (config('ninja.ninja_default_company_id') == $this->subscription->company_id && in_array($this->subscription->service()->products()->first()?->product_key, ['peppol_500','peppol_1000','selfhost_peppol_500','selfhost_peppol_1000'])) {
            $context = $this->subscription->service()->products()->first()?->product_key;
        }

        Cache::put($this->hash, [
            'subscription_id' => $this->subscription->hashed_id,
            'email' => $this->email ?? $this->contact->email,
            'client_id' => $this->contact->client->hashed_id,
            'invoice_id' => $this->invoice->hashed_id,
            'context' => $context,
            'campaign' => $this->campaign,
            'request_data' => $this->request_data,
        ], now()->addMinutes(60));

        $this->dispatch('beforePaymentEventsCompleted');
    }

    /**
     * Proxy method for starting the trial.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function handleTrial()
    {
        return $this->subscription->service()->startTrial([
            'email' => $this->email ?? $this->contact->email,
            'quantity' => $this->quantity,
            'contact_id' => $this->contact->hashed_id,
            'client_id' => $this->contact->client->hashed_id,
        ]);
    }

    public function handlePaymentNotRequired()
    {
        $is_eligible = $this->subscription->service()->isEligible($this->contact);

        if ($is_eligible['status_code'] != 200) {
            $this->steps['not_eligible'] = true;
            $this->steps['not_eligible_message'] = $is_eligible['message'];
            $this->steps['show_loading_bar'] = false;

            return;
        }

        return $this->subscription->service()->handleNoPaymentRequired([
            'email' => $this->email ?? $this->contact->email,
            'quantity' => $this->quantity,
            'contact_id' => $this->contact->id,
            'client_id' => $this->contact->client->id,
            'coupon' => $this->coupon,
            'campaign' => $this->campaign,
        ]);
    }

    /**
     * Update quantity property.
     *
     * @param string $option
     * @return int
     */
    public function updateQuantity(string $option): int
    {
        $this->handleCoupon();

        if ($this->quantity == 1 && $option == 'decrement') {
            $this->price = $this->price * 1;
            return $this->quantity;
        }

        if ($this->quantity > $this->subscription->max_seats_limit && $option == 'increment') {
            $this->price = $this->price * $this->subscription->max_seats_limit;
            return $this->quantity;
        }

        if ($option == 'increment') {
            $this->quantity++;
            $this->price = $this->price * $this->quantity;
            return $this->quantity;
        }

        $this->quantity--;
        $this->price = $this->price * $this->quantity;

        return $this->quantity;
    }

    public function handleCoupon()
    {
        if ($this->steps['discount_applied']) {
            $this->price = $this->subscription->promo_price;
            return;
        }

        if ($this->coupon == $this->subscription->promo_code) {
            $this->price = $this->subscription->promo_price;
            $this->quantity = 1;
            $this->steps['discount_applied'] = true;
        } else {
            $this->price = $this->subscription->price;
        }
    }

    public function passwordlessLogin()
    {
        $this->passwordless_login_btn = true;

        $contact = ClientContact::query()
            ->where('email', $this->email)
            ->where('company_id', $this->subscription->company_id)
            ->first();

        $mailer = new NinjaMailerObject();
        $mailer->mailable = new ContactPasswordlessLogin($this->email, $this->subscription->company, (string)route('client.subscription.purchase', $this->subscription->hashed_id) . '?coupon=' . $this->coupon);
        $mailer->company = $this->subscription->company;
        $mailer->settings = $this->subscription->company->settings;
        $mailer->to_user = $contact;

        NinjaMailerJob::dispatch($mailer);

        $this->steps['passwordless_login_sent'] = true;
        $this->passwordless_login_btn = false;
    }

    public function render()
    {
        if (array_key_exists('email', $this->request_data)) {
            $this->email = $this->request_data['email'];
        }

        if ($this->contact instanceof ClientContact) {
            $this->getPaymentMethods($this->contact);
        }

        return render('components.livewire.billing-portal-purchase');
    }
}
