<?php /*

 Composr
 Copyright (c) ocProducts, 2004-2016

 See text/EN/licence.txt for full licencing information.


 NOTE TO PROGRAMMERS:
   Do not edit this file. If you need to make changes, save your changed file to the appropriate *_custom folder
   **** If you ignore this advice, then your website upgrades (e.g. for bug fixes) will likely kill your changes ****

*/

/*EXTRA FUNCTIONS: PDO*/

/**
 * @license    http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright  ocProducts Ltd
 * @package    core_database_drivers
 */

require_code('database/shared/mysql');

/**
 * Database Driver.
 *
 * @package    core_database_drivers
 */
class Database_Static_mysql_pdo extends Database_super_mysql
{
    public $cache_db = array();
    public $last_select_db = null;
    public $reconnected_once = false;

    /**
     * Get a database connection. This function shouldn't be used by you, as a connection to the database is established automatically.
     *
     * @param  boolean $persistent Whether to create a persistent connection
     * @param  string $db_name The database name
     * @param  string $db_host The database host (the server)
     * @param  string $db_user The database connection username
     * @param  string $db_password The database connection password
     * @param  boolean $fail_ok Whether to on error echo an error and return with a null, rather than giving a critical error
     * @return ?object A database connection (null: error)
     */
    public function db_get_connection($persistent, $db_name, $db_host, $db_user, $db_password, $fail_ok = false)
    {
        if ((!class_exists('PDO')) || (!defined('PDO::MYSQL_ATTR_USE_BUFFERED_QUERY'))) {
            $error = 'The \'pdo_mysql\' PHP extension is not installed (anymore?). You need to contact the system administrator of this server, or use a different MySQL database driver (drivers can be chosen by editing _config.php).';
            if ($fail_ok) {
                echo ((running_script('install')) && (get_param_string('type', '') == 'ajax_db_details')) ? strip_html($error) : $error;
                return null;
            }
            critical_error('PASSON', $error);
        }

        global $SITE_INFO;
        if (empty($SITE_INFO['database_charset'])) {
            $SITE_INFO['database_charset'] = (get_charset() == 'utf-8') ? 'utf8mb4' : 'latin1';
        }

        // Potential caching
        $x = serialize(array($db_user, $db_host, $db_name));
        if (array_key_exists($x, $this->cache_db)) {
            return $this->cache_db[$x];
        }
        $db_port = 3306;
        if (strpos($db_host, ':') !== false) {
            list($db_host, $_db_port) = explode(':', $db_host);
            $db_port = intval($_db_port);
        }
        try {
            $dsn = 'mysql:host=' . $db_host . ';port=' . strval($db_port) . ';dbname=' . $db_name;
            $db = new PDO($dsn, $db_user, $db_password, array(
                PDO::ATTR_PERSISTENT => $persistent,
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::MYSQL_ATTR_INIT_COMMAND => 'SET NAMES ' . $SITE_INFO['database_charset'],
            ));
        }
        catch (PDOException $e) {
            $error = 'Could not connect to database (' . $e->getMessage() . ')';
            if ($fail_ok) {
                echo ((running_script('install')) && (get_param_string('type', '') == 'ajax_db_details')) ? strip_html($error) : $error;
                return null;
            }
            critical_error('PASSON', $error); //warn_exit(do_lang_tempcode('CONNECT_DB_ERROR'));
        }
        if (!empty($SITE_INFO['database_collation'])) {
            $db->query('SET collation_connection=' . $SITE_INFO['database_collation']);
        }

        $this->last_select_db = $db;
        $this->cache_db[$x] = $db;

        try {
            $db->query('SET wait_timeout=28800');
        }
        catch (PDOException $e) {
        }
        try {
            $db->query('SET sql_big_selects=1');
        }
        catch (PDOException $e) {
        }
        if ((get_forum_type() == 'cns') && (!$GLOBALS['IN_MINIKERNEL_VERSION'])) {
            try {
                $db->query('SET sql_mode=\'STRICT_ALL_TABLES\'');
            }
            catch (PDOException $e) {
            }
        } else {
            try {
                $db->query('SET sql_mode=\'MYSQL40\''); // We may be in some legacy context, such as backup restoration, upgrader, or another forum driver
            }
            catch (PDOException $e) {
                try { // Won't work on MySQL 8 for example
                    $db->query('SET sql_mode=\'STRICT_ALL_TABLES\'');
                }
                catch (PDOException $e) {
                }
            }
        }
        // NB: Can add ,ONLY_FULL_GROUP_BY for testing on what other DBs will do, but can_arbitrary_groupby() would need to be made to return false

        return $db;
    }

    /**
     * Find whether full-text-search is present
     *
     * @param  object $db A DB connection
     * @return boolean Whether it is
     */
    public function db_has_full_text($db)
    {
        return true;
    }

    /**
     * Find whether subquery support is present
     *
     * @param  object $db A DB connection
     * @return boolean Whether it is
     */
    public function db_has_subqueries($db)
    {
        return true;
    }

    /**
     * Find whether collate support is present
     *
     * @param  object $db A DB connection
     * @return boolean Whether it is
     */
    public function db_has_collate_settings($db)
    {
        return true;
    }

    /**
     * Find whether full-text-boolean-search is present
     *
     * @return boolean Whether it is
     */
    public function db_has_full_text_boolean()
    {
        return true;
    }

    /**
     * Escape a string so it may be inserted into a query. If SQL statements are being built up and passed using db_query then it is essential that this is used for security reasons. Otherwise, the abstraction layer deals with the situation.
     *
     * @param  string $string The string
     * @return string The escaped string
     */
    public function db_escape_string($string)
    {
        if (function_exists('ctype_alnum')) {
            if (ctype_alnum($string)) {
                return $string; // No non-trivial characters
            }
        }
        if (preg_match('#[^a-zA-Z0-9\.]#', $string) === 0) {
            return $string; // No non-trivial characters
        }

        $string = fix_bad_unicode($string);

        if ($this->last_select_db === null) {
            return addslashes($string);
        }
        return preg_replace("#^'(.*)'$#", '$1', $this->last_select_db->quote($string));
    }

    /**
     * This function is a very basic query executor. It shouldn't usually be used by you, as there are abstracted versions available.
     *
     * @param  string $query The complete SQL query
     * @param  object $db A DB connection
     * @param  ?integer $max The maximum number of rows to affect (null: no limit)
     * @param  ?integer $start The start row to affect (null: no specification)
     * @param  boolean $fail_ok Whether to output an error on failure
     * @param  boolean $get_insert_id Whether to get the autoincrement ID created for an insert query
     * @return ?mixed The results (null: no results), or the insert ID
     */
    public function db_query($query, $db, $max = null, $start = null, $fail_ok = false, $get_insert_id = false)
    {
        if (isset($query[500000])) { // Let's hope we can fail on this, because it's a huge query. We can only allow it if MySQL can.
            $test_result = $this->db_query('SHOW VARIABLES LIKE \'max_allowed_packet\'', $db, null, null, true);

            if (!is_array($test_result)) {
                return null;
            }
            if (intval($test_result[0]['Value']) < intval(strlen($query) * 1.2)) {
                if ($get_insert_id) {
                    fatal_exit(do_lang_tempcode('QUERY_FAILED_TOO_BIG', escape_html($query), escape_html(integer_format(strlen($query))), escape_html(integer_format(intval($test_result[0]['Value'])))));
                } else {
                    attach_message(do_lang_tempcode('QUERY_FAILED_TOO_BIG', escape_html(substr($query, 0, 300)) . '...', escape_html(integer_format(strlen($query))), escape_html(integer_format(intval($test_result[0]['Value'])))), 'warn');
                }
                return null;
            }
        }

        static $version = null;
        if ($version === null) {
            $version = ''; // Temporary, to avoid infinite recursion
            $_version = $this->db_query('SELECT version() AS version', $db, null, null, true);
            if (array_key_exists(0, $_version)) {
                $version = $_version[0]['version'];
            } else {
                $version = '';
            }
        }
        if ($version != '') {
            if (version_compare($version, '8', '>=')) {
                $query = $this->fix_mysql8_query($query);
            }
        }

        $this->apply_sql_limit_clause($query, $max, $start);

        try {
            $results = $db->query($query);
        }
        catch (PDOException $e) {
            if ((!$fail_ok) || (strpos($e->getMessage(), 'is marked as crashed and should be repaired') !== false)) {
                $err = $e->getMessage();

                if (function_exists('ocp_mark_as_escaped')) {
                    ocp_mark_as_escaped($err);
                }
                if ((!running_script('upgrader')) && ((!get_mass_import_mode()) || (get_param_integer('keep_fatalistic', 0) == 1)) && (strpos($err, 'Duplicate entry') === false)) {
                    $matches = array();
                    if (preg_match('#/(\w+)\' is marked as crashed and should be repaired#U', $err, $matches) !== 0) {
                        $this->db_query('REPAIR TABLE ' . $matches[1], $db);
                    }

                    if (!function_exists('do_lang') || is_null(do_lang('QUERY_FAILED', null, null, null, null, false))) {
                        fatal_exit(htmlentities('Query failed: ' . $query . ' : ' . $err));
                    }
                    fatal_exit(do_lang_tempcode('QUERY_FAILED', escape_html($query), ($err)));
                } else {
                    echo htmlentities('Database query failed: ' . $query . ' [') . ($err) . htmlentities(']') . "<br />\n";
                    return null;
                }
            }
            $results = false;
        }

        $sub = substr(ltrim($query), 0, 4);
        if (($results !== true) && (($sub === '(SEL') || ($sub === 'SELE') || ($sub === 'sele') || ($sub === 'CHEC') || ($sub === 'EXPL') || ($sub === 'REPA') || ($sub === 'DESC') || ($sub === 'SHOW')) && ($results !== false)) {
            return $this->db_get_query_rows($results, $query, $start);
        }

        if ($get_insert_id) {
            if (strtoupper(substr($query, 0, 7)) === 'UPDATE ') {
                return $results->rowCount();
            }
            $ins = $db->lastInsertId();
            if ($ins === 0) {
                $table = substr($query, 12, strpos($query, ' ', 12) - 12);
                $rows = $this->db_query('SELECT MAX(id) AS x FROM ' . $table, $db, 1, 0, false, false);
                return $rows[0]['x'];
            }
            return $ins;
        }

        return null;
    }

    /**
     * Get the rows returned from a SELECT query.
     *
     * @param  object $results The query result pointer
     * @param  string $query The complete SQL query (useful for debugging)
     * @param  ?integer $start Whether to start reading from (null: irrelevant)
     * @return array A list of row maps
     */
    public function db_get_query_rows($results, $query, $start = null)
    {
        $names = array();
        $types = array();
        $column_count = $results->columnCount();
        for ($i = 0; $i < $column_count; $i++) {
            $field = $results->getColumnMeta($i);
            $names[$i] = $field['name'];
            $types[$i] = $field['native_type'];
        }

        $out = array();
        $newrow = array();
        while (($row = $results->fetch(PDO::FETCH_NUM)) !== false) {
            foreach ($row as $j => $v) {
                $name = $names[$j];
                $type = $types[$j];

                if (!is_string($v)) {
                    $newrow[$name] = $v;
                } else {
                    switch ($type) {
                        case 'TINY':
                        case 'SHORT':
                        case 'LONG':
                        case 'INT24':
                        case 'LONGLONG':
                            $newrow[$name] = intval($v);
                            break;

                        case 'BIT':
                            $newrow[$name] = ($v === 1 || $v === true || $v === '1') ? 1 : 0;

                        case 'DECIMAL':
                        case 'NEWDECIMAL':
                        case 'FLOAT':
                        case 'DOUBLE':
                            $newrow[$name] = floatval($v);
                            break;

                        case 'NULL':
                            $newrow[$name] = null;

                        default:
                            $newrow[$name] = $v;
                    }
                }
            }

            $out[] = $newrow;
        }
        $results->closeCursor();

        return $out;
    }
}
