PK     Zc}  }    XML/Util.phpnu [        <?php
/**
 * XML_Util
 *
 * XML Utilities package
 *
 * PHP versions 4 and 5
 *
 * LICENSE:
 *
 * Copyright (c) 2003-2008 Stephan Schmidt <schst@php.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    * Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    * The name of the author may not be used to endorse or promote products
 *      derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @category  XML
 * @package   XML_Util
 * @author    Stephan Schmidt <schst@php.net>
 * @copyright 2003-2008 Stephan Schmidt <schst@php.net>
 * @license   http://opensource.org/licenses/bsd-license New BSD License
 * @version   CVS: $Id$
 * @link      http://pear.php.net/package/XML_Util
 */

/**
 * Error code for invalid chars in XML name
 */
define('XML_UTIL_ERROR_INVALID_CHARS', 51);

/**
 * Error code for invalid chars in XML name
 */
define('XML_UTIL_ERROR_INVALID_START', 52);

/**
 * Error code for non-scalar tag content
 */
define('XML_UTIL_ERROR_NON_SCALAR_CONTENT', 60);

/**
 * Error code for missing tag name
 */
define('XML_UTIL_ERROR_NO_TAG_NAME', 61);

/**
 * Replace XML entities
 */
define('XML_UTIL_REPLACE_ENTITIES', 1);

/**
 * Embedd content in a CData Section
 */
define('XML_UTIL_CDATA_SECTION', 5);

/**
 * Do not replace entitites
 */
define('XML_UTIL_ENTITIES_NONE', 0);

/**
 * Replace all XML entitites
 * This setting will replace <, >, ", ' and &
 */
define('XML_UTIL_ENTITIES_XML', 1);

/**
 * Replace only required XML entitites
 * This setting will replace <, " and &
 */
define('XML_UTIL_ENTITIES_XML_REQUIRED', 2);

/**
 * Replace HTML entitites
 * @link http://www.php.net/htmlentities
 */
define('XML_UTIL_ENTITIES_HTML', 3);

/**
 * Do not collapse any empty tags.
 */
define('XML_UTIL_COLLAPSE_NONE', 0);

/**
 * Collapse all empty tags.
 */
define('XML_UTIL_COLLAPSE_ALL', 1);

/**
 * Collapse only empty XHTML tags that have no end tag.
 */
define('XML_UTIL_COLLAPSE_XHTML_ONLY', 2);

/**
 * Utility class for working with XML documents
 *
 * @category  XML
 * @package   XML_Util
 * @author    Stephan Schmidt <schst@php.net>
 * @copyright 2003-2008 Stephan Schmidt <schst@php.net>
 * @license   http://opensource.org/licenses/bsd-license New BSD License
 * @version   Release: 1.4.5
 * @link      http://pear.php.net/package/XML_Util
 */
class XML_Util
{
    /**
     * Return API version
     *
     * @return string $version API version
     */
    public static function apiVersion()
    {
        return '1.4';
    }

    /**
     * Replace XML entities
     *
     * With the optional second parameter, you may select, which
     * entities should be replaced.
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // replace XML entites:
     * $string = XML_Util::replaceEntities('This string contains < & >.');
     * </code>
     *
     * With the optional third parameter, you may pass the character encoding
     * <code>
     * require_once 'XML/Util.php';
     *
     * // replace XML entites in UTF-8:
     * $string = XML_Util::replaceEntities(
     *     'This string contains < & > as well as ä, ö, ß, à and ê',
     *     XML_UTIL_ENTITIES_HTML,
     *     'UTF-8'
     * );
     * </code>
     *
     * @param string $string          string where XML special chars
     *                                should be replaced
     * @param int    $replaceEntities setting for entities in attribute values
     *                                (one of XML_UTIL_ENTITIES_XML,
     *                                XML_UTIL_ENTITIES_XML_REQUIRED,
     *                                XML_UTIL_ENTITIES_HTML)
     * @param string $encoding        encoding value (if any)...
     *                                must be a valid encoding as determined
     *                                by the htmlentities() function
     *
     * @return string string with replaced chars
     * @see    reverseEntities()
     */
    public static function replaceEntities(
        $string, $replaceEntities = XML_UTIL_ENTITIES_XML, $encoding = 'ISO-8859-1'
    ) {
        switch ($replaceEntities) {
        case XML_UTIL_ENTITIES_XML:
            return strtr(
                $string,
                array(
                    '&'  => '&amp;',
                    '>'  => '&gt;',
                    '<'  => '&lt;',
                    '"'  => '&quot;',
                    '\'' => '&apos;'
                )
            );
            break;
        case XML_UTIL_ENTITIES_XML_REQUIRED:
            return strtr(
                $string,
                array(
                    '&' => '&amp;',
                    '<' => '&lt;',
                    '"' => '&quot;'
                )
            );
            break;
        case XML_UTIL_ENTITIES_HTML:
            return htmlentities($string, ENT_COMPAT, $encoding);
            break;
        }
        return $string;
    }

    /**
     * Reverse XML entities
     *
     * With the optional second parameter, you may select, which
     * entities should be reversed.
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // reverse XML entites:
     * $string = XML_Util::reverseEntities('This string contains &lt; &amp; &gt;.');
     * </code>
     *
     * With the optional third parameter, you may pass the character encoding
     * <code>
     * require_once 'XML/Util.php';
     *
     * // reverse XML entites in UTF-8:
     * $string = XML_Util::reverseEntities(
     *     'This string contains &lt; &amp; &gt; as well as'
     *     . ' &auml;, &ouml;, &szlig;, &agrave; and &ecirc;',
     *     XML_UTIL_ENTITIES_HTML,
     *     'UTF-8'
     * );
     * </code>
     *
     * @param string $string          string where XML special chars
     *                                should be replaced
     * @param int    $replaceEntities setting for entities in attribute values
     *                                (one of XML_UTIL_ENTITIES_XML,
     *                                XML_UTIL_ENTITIES_XML_REQUIRED,
     *                                XML_UTIL_ENTITIES_HTML)
     * @param string $encoding        encoding value (if any)...
     *                                must be a valid encoding as determined
     *                                by the html_entity_decode() function
     *
     * @return string string with replaced chars
     * @see    replaceEntities()
     */
    public static function reverseEntities(
        $string, $replaceEntities = XML_UTIL_ENTITIES_XML, $encoding = 'ISO-8859-1'
    ) {
        switch ($replaceEntities) {
        case XML_UTIL_ENTITIES_XML:
            return strtr(
                $string,
                array(
                    '&amp;'  => '&',
                    '&gt;'   => '>',
                    '&lt;'   => '<',
                    '&quot;' => '"',
                    '&apos;' => '\''
                )
            );
            break;
        case XML_UTIL_ENTITIES_XML_REQUIRED:
            return strtr(
                $string,
                array(
                    '&amp;'  => '&',
                    '&lt;'   => '<',
                    '&quot;' => '"'
                )
            );
            break;
        case XML_UTIL_ENTITIES_HTML:
            return html_entity_decode($string, ENT_COMPAT, $encoding);
            break;
        }
        return $string;
    }

    /**
     * Build an xml declaration
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // get an XML declaration:
     * $xmlDecl = XML_Util::getXMLDeclaration('1.0', 'UTF-8', true);
     * </code>
     *
     * @param string $version    xml version
     * @param string $encoding   character encoding
     * @param bool   $standalone document is standalone (or not)
     *
     * @return string xml declaration
     * @uses   attributesToString() to serialize the attributes of the
     *         XML declaration
     */
    public static function getXMLDeclaration(
        $version = '1.0', $encoding = null, $standalone = null
    ) {
        $attributes = array(
            'version' => $version,
        );
        // add encoding
        if ($encoding !== null) {
            $attributes['encoding'] = $encoding;
        }
        // add standalone, if specified
        if ($standalone !== null) {
            $attributes['standalone'] = $standalone ? 'yes' : 'no';
        }

        return sprintf(
            '<?xml%s?>',
            XML_Util::attributesToString($attributes, false)
        );
    }

    /**
     * Build a document type declaration
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // get a doctype declaration:
     * $xmlDecl = XML_Util::getDocTypeDeclaration('rootTag','myDocType.dtd');
     * </code>
     *
     * @param string $root        name of the root tag
     * @param string $uri         uri of the doctype definition
     *                            (or array with uri and public id)
     * @param string $internalDtd internal dtd entries
     *
     * @return string doctype declaration
     * @since  0.2
     */
    public static function getDocTypeDeclaration(
        $root, $uri = null, $internalDtd = null
    ) {
        if (is_array($uri)) {
            $ref = sprintf(' PUBLIC "%s" "%s"', $uri['id'], $uri['uri']);
        } elseif (!empty($uri)) {
            $ref = sprintf(' SYSTEM "%s"', $uri);
        } else {
            $ref = '';
        }

        if (empty($internalDtd)) {
            return sprintf('<!DOCTYPE %s%s>', $root, $ref);
        } else {
            return sprintf("<!DOCTYPE %s%s [\n%s\n]>", $root, $ref, $internalDtd);
        }
    }

    /**
     * Create string representation of an attribute list
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // build an attribute string
     * $att = array(
     *              'foo'   =>  'bar',
     *              'argh'  =>  'tomato'
     *            );
     *
     * $attList = XML_Util::attributesToString($att);
     * </code>
     *
     * @param array      $attributes attribute array
     * @param bool|array $sort       sort attribute list alphabetically,
     *                               may also be an assoc array containing
     *                               the keys 'sort', 'multiline', 'indent',
     *                               'linebreak' and 'entities'
     * @param bool       $multiline  use linebreaks, if more than
     *                               one attribute is given
     * @param string     $indent     string used for indentation of
     *                               multiline attributes
     * @param string     $linebreak  string used for linebreaks of
     *                               multiline attributes
     * @param int        $entities   setting for entities in attribute values
     *                               (one of XML_UTIL_ENTITIES_NONE,
     *                               XML_UTIL_ENTITIES_XML,
     *                               XML_UTIL_ENTITIES_XML_REQUIRED,
     *                               XML_UTIL_ENTITIES_HTML)
     *
     * @return string string representation of the attributes
     * @uses   replaceEntities() to replace XML entities in attribute values
     * @todo   allow sort also to be an options array
     */
    public static function attributesToString(
        $attributes, $sort = true, $multiline = false,
        $indent = '    ', $linebreak = "\n", $entities = XML_UTIL_ENTITIES_XML
    ) {
        /*
         * second parameter may be an array
         */
        if (is_array($sort)) {
            if (isset($sort['multiline'])) {
                $multiline = $sort['multiline'];
            }
            if (isset($sort['indent'])) {
                $indent = $sort['indent'];
            }
            if (isset($sort['linebreak'])) {
                $multiline = $sort['linebreak'];
            }
            if (isset($sort['entities'])) {
                $entities = $sort['entities'];
            }
            if (isset($sort['sort'])) {
                $sort = $sort['sort'];
            } else {
                $sort = true;
            }
        }
        $string = '';
        if (is_array($attributes) && !empty($attributes)) {
            if ($sort) {
                ksort($attributes);
            }
            if (!$multiline || count($attributes) == 1) {
                foreach ($attributes as $key => $value) {
                    if ($entities != XML_UTIL_ENTITIES_NONE) {
                        if ($entities === XML_UTIL_CDATA_SECTION) {
                            $entities = XML_UTIL_ENTITIES_XML;
                        }
                        $value = XML_Util::replaceEntities($value, $entities);
                    }
                    $string .= ' ' . $key . '="' . $value . '"';
                }
            } else {
                $first = true;
                foreach ($attributes as $key => $value) {
                    if ($entities != XML_UTIL_ENTITIES_NONE) {
                        $value = XML_Util::replaceEntities($value, $entities);
                    }
                    if ($first) {
                        $string .= ' ' . $key . '="' . $value . '"';
                        $first   = false;
                    } else {
                        $string .= $linebreak . $indent . $key . '="' . $value . '"';
                    }
                }
            }
        }
        return $string;
    }

    /**
     * Collapses empty tags.
     *
     * @param string $xml  XML
     * @param int    $mode Whether to collapse all empty tags (XML_UTIL_COLLAPSE_ALL)
     *                      or only XHTML (XML_UTIL_COLLAPSE_XHTML_ONLY) ones.
     *
     * @return string XML
     */
    public static function collapseEmptyTags($xml, $mode = XML_UTIL_COLLAPSE_ALL)
    {
        if (preg_match('~<([^>])+/>~s', $xml, $matches)) {
            // it's already an empty tag
            return $xml;
        }
        switch ($mode) {
            case XML_UTIL_COLLAPSE_ALL:
                $preg1 =
                    '~<' .
                        '(?:' .
                            '(https?://[^:\s]+:\w+)' .  // <http://foo.com:bar  ($1)
                            '|(\w+:\w+)' .              // <foo:bar             ($2)
                            '|(\w+)' .                  // <foo                 ($3)
                        ')+' .
                        '([^>]*)' .                     // attributes           ($4)
                    '>' .
                    '<\/(\1|\2|\3)>' .                  // 1, 2, or 3 again     ($5)
                    '~s'
                ;
                $preg2 =
                    '<' .
                        '${1}${2}${3}' .    // tag (only one should have been populated)
                        '${4}' .            // attributes
                    ' />'
                ;
                return (preg_replace($preg1, $preg2, $xml)?:$xml);
                break;
            case XML_UTIL_COLLAPSE_XHTML_ONLY:
                return (
                    preg_replace(
                        '/<(area|base(?:font)?|br|col|frame|hr|img|input|isindex|link|meta|'
                        . 'param)([^>]*)><\/\\1>/s',
                        '<\\1\\2 />',
                        $xml
                    ) ?: $xml
                );
                break;
            case XML_UTIL_COLLAPSE_NONE:
                // fall thru
            default:
                return $xml;
        }
    }

    /**
     * Create a tag
     *
     * This method will call XML_Util::createTagFromArray(), which
     * is more flexible.
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // create an XML tag:
     * $tag = XML_Util::createTag('myNs:myTag',
     *     array('foo' => 'bar'),
     *     'This is inside the tag',
     *     'http://www.w3c.org/myNs#');
     * </code>
     *
     * @param string $qname           qualified tagname (including namespace)
     * @param array  $attributes      array containg attributes
     * @param mixed  $content         the content
     * @param string $namespaceUri    URI of the namespace
     * @param int    $replaceEntities whether to replace XML special chars in
     *                                content, embedd it in a CData section
     *                                or none of both
     * @param bool   $multiline       whether to create a multiline tag where
     *                                each attribute gets written to a single line
     * @param string $indent          string used to indent attributes
     *                                (_auto indents attributes so they start
     *                                at the same column)
     * @param string $linebreak       string used for linebreaks
     * @param bool   $sortAttributes  Whether to sort the attributes or not
     * @param int    $collapseTagMode How to handle a content-less, and thus collapseable, tag
     *
     * @return string XML tag
     * @see    createTagFromArray()
     * @uses   createTagFromArray() to create the tag
     */
    public static function createTag(
        $qname, $attributes = array(), $content = null,
        $namespaceUri = null, $replaceEntities = XML_UTIL_REPLACE_ENTITIES,
        $multiline = false, $indent = '_auto', $linebreak = "\n",
        $sortAttributes = true, $collapseTagMode = XML_UTIL_COLLAPSE_ALL
    ) {
        $tag = array(
            'qname'      => $qname,
            'attributes' => $attributes
        );

        // add tag content
        if ($content !== null) {
            $tag['content'] = $content;
        }

        // add namespace Uri
        if ($namespaceUri !== null) {
            $tag['namespaceUri'] = $namespaceUri;
        }

        return XML_Util::createTagFromArray(
            $tag, $replaceEntities, $multiline,
            $indent, $linebreak, $sortAttributes,
            $collapseTagMode
        );
    }

    /**
     * Create a tag from an array.
     * This method awaits an array in the following format
     * <pre>
     * array(
     *     // qualified name of the tag
     *     'qname' => $qname
     *
     *     // namespace prefix (optional, if qname is specified or no namespace)
     *     'namespace' => $namespace
     *
     *     // local part of the tagname (optional, if qname is specified)
     *     'localpart' => $localpart,
     *
     *     // array containing all attributes (optional)
     *     'attributes' => array(),
     *
     *     // tag content (optional)
     *     'content' => $content,
     *
     *     // namespaceUri for the given namespace (optional)
     *     'namespaceUri' => $namespaceUri
     * )
     * </pre>
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * $tag = array(
     *     'qname'        => 'foo:bar',
     *     'namespaceUri' => 'http://foo.com',
     *     'attributes'   => array('key' => 'value', 'argh' => 'fruit&vegetable'),
     *     'content'      => 'I\'m inside the tag',
     * );
     * // creating a tag with qualified name and namespaceUri
     * $string = XML_Util::createTagFromArray($tag);
     * </code>
     *
     * @param array  $tag             tag definition
     * @param int    $replaceEntities whether to replace XML special chars in
     *                                content, embedd it in a CData section
     *                                or none of both
     * @param bool   $multiline       whether to create a multiline tag where each
     *                                attribute gets written to a single line
     * @param string $indent          string used to indent attributes
     *                                (_auto indents attributes so they start
     *                                at the same column)
     * @param string $linebreak       string used for linebreaks
     * @param bool   $sortAttributes  Whether to sort the attributes or not
     * @param int    $collapseTagMode How to handle a content-less, and thus collapseable, tag
     *
     * @return string XML tag
     *
     * @see  createTag()
     * @uses attributesToString() to serialize the attributes of the tag
     * @uses splitQualifiedName() to get local part and namespace of a qualified name
     * @uses createCDataSection()
     * @uses collapseEmptyTags()
     * @uses raiseError()
     */
    public static function createTagFromArray(
        $tag, $replaceEntities = XML_UTIL_REPLACE_ENTITIES,
        $multiline = false, $indent = '_auto', $linebreak = "\n",
        $sortAttributes = true, $collapseTagMode = XML_UTIL_COLLAPSE_ALL
    ) {
        if (isset($tag['content']) && !is_scalar($tag['content'])) {
            return XML_Util::raiseError(
                'Supplied non-scalar value as tag content',
                XML_UTIL_ERROR_NON_SCALAR_CONTENT
            );
        }

        if (!isset($tag['qname']) && !isset($tag['localPart'])) {
            return XML_Util::raiseError(
                'You must either supply a qualified name '
                . '(qname) or local tag name (localPart).',
                XML_UTIL_ERROR_NO_TAG_NAME
            );
        }

        // if no attributes hav been set, use empty attributes
        if (!isset($tag['attributes']) || !is_array($tag['attributes'])) {
            $tag['attributes'] = array();
        }

        if (isset($tag['namespaces'])) {
            foreach ($tag['namespaces'] as $ns => $uri) {
                $tag['attributes']['xmlns:' . $ns] = $uri;
            }
        }

        if (!isset($tag['qname'])) {
            // qualified name is not given

            // check for namespace
            if (isset($tag['namespace']) && !empty($tag['namespace'])) {
                $tag['qname'] = $tag['namespace'] . ':' . $tag['localPart'];
            } else {
                $tag['qname'] = $tag['localPart'];
            }
        } elseif (isset($tag['namespaceUri']) && !isset($tag['namespace'])) {
            // namespace URI is set, but no namespace

            $parts = XML_Util::splitQualifiedName($tag['qname']);

            $tag['localPart'] = $parts['localPart'];
            if (isset($parts['namespace'])) {
                $tag['namespace'] = $parts['namespace'];
            }
        }

        if (isset($tag['namespaceUri']) && !empty($tag['namespaceUri'])) {
            // is a namespace given
            if (isset($tag['namespace']) && !empty($tag['namespace'])) {
                $tag['attributes']['xmlns:' . $tag['namespace']]
                    = $tag['namespaceUri'];
            } else {
                // define this Uri as the default namespace
                $tag['attributes']['xmlns'] = $tag['namespaceUri'];
            }
        }

        if (!array_key_exists('content', $tag)) {
            $tag['content'] = '';
        }

        // check for multiline attributes
        if ($multiline === true) {
            if ($indent === '_auto') {
                $indent = str_repeat(' ', (strlen($tag['qname'])+2));
            }
        }

        // create attribute list
        $attList = XML_Util::attributesToString(
            $tag['attributes'],
            $sortAttributes, $multiline, $indent, $linebreak
        );

        switch ($replaceEntities) {
        case XML_UTIL_ENTITIES_NONE:
            break;
        case XML_UTIL_CDATA_SECTION:
            $tag['content'] = XML_Util::createCDataSection($tag['content']);
            break;
        default:
            $tag['content'] = XML_Util::replaceEntities(
                $tag['content'], $replaceEntities
            );
            break;
        }
        $tag = sprintf(
            '<%s%s>%s</%s>', $tag['qname'], $attList, $tag['content'],
            $tag['qname']
        );

        return self::collapseEmptyTags($tag, $collapseTagMode);
    }

    /**
     * Create a start element
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // create an XML start element:
     * $tag = XML_Util::createStartElement('myNs:myTag',
     *     array('foo' => 'bar') ,'http://www.w3c.org/myNs#');
     * </code>
     *
     * @param string $qname          qualified tagname (including namespace)
     * @param array  $attributes     array containg attributes
     * @param string $namespaceUri   URI of the namespace
     * @param bool   $multiline      whether to create a multiline tag where each
     *                               attribute gets written to a single line
     * @param string $indent         string used to indent attributes (_auto indents
     *                               attributes so they start at the same column)
     * @param string $linebreak      string used for linebreaks
     * @param bool   $sortAttributes Whether to sort the attributes or not
     *
     * @return string XML start element
     * @see    createEndElement(), createTag()
     */
    public static function createStartElement(
        $qname, $attributes = array(), $namespaceUri = null,
        $multiline = false, $indent = '_auto', $linebreak = "\n",
        $sortAttributes = true
    ) {
        // if no attributes hav been set, use empty attributes
        if (!isset($attributes) || !is_array($attributes)) {
            $attributes = array();
        }

        if ($namespaceUri != null) {
            $parts = XML_Util::splitQualifiedName($qname);
        }

        // check for multiline attributes
        if ($multiline === true) {
            if ($indent === '_auto') {
                $indent = str_repeat(' ', (strlen($qname)+2));
            }
        }

        if ($namespaceUri != null) {
            // is a namespace given
            if (isset($parts['namespace']) && !empty($parts['namespace'])) {
                $attributes['xmlns:' . $parts['namespace']] = $namespaceUri;
            } else {
                // define this Uri as the default namespace
                $attributes['xmlns'] = $namespaceUri;
            }
        }

        // create attribute list
        $attList = XML_Util::attributesToString(
            $attributes, $sortAttributes,
            $multiline, $indent, $linebreak
        );
        $element = sprintf('<%s%s>', $qname, $attList);
        return  $element;
    }

    /**
     * Create an end element
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // create an XML start element:
     * $tag = XML_Util::createEndElement('myNs:myTag');
     * </code>
     *
     * @param string $qname qualified tagname (including namespace)
     *
     * @return string XML end element
     * @see    createStartElement(), createTag()
     */
    public static function createEndElement($qname)
    {
        $element = sprintf('</%s>', $qname);
        return $element;
    }

    /**
     * Create an XML comment
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // create an XML start element:
     * $tag = XML_Util::createComment('I am a comment');
     * </code>
     *
     * @param string $content content of the comment
     *
     * @return string XML comment
     */
    public static function createComment($content)
    {
        $comment = sprintf('<!-- %s -->', $content);
        return $comment;
    }

    /**
     * Create a CData section
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // create a CData section
     * $tag = XML_Util::createCDataSection('I am content.');
     * </code>
     *
     * @param string $data data of the CData section
     *
     * @return string CData section with content
     */
    public static function createCDataSection($data)
    {
        return sprintf(
            '<![CDATA[%s]]>',
            preg_replace('/\]\]>/', ']]]]><![CDATA[>', strval($data))
        );
    }

    /**
     * Split qualified name and return namespace and local part
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // split qualified tag
     * $parts = XML_Util::splitQualifiedName('xslt:stylesheet');
     * </code>
     * the returned array will contain two elements:
     * <pre>
     * array(
     *     'namespace' => 'xslt',
     *     'localPart' => 'stylesheet'
     * );
     * </pre>
     *
     * @param string $qname     qualified tag name
     * @param string $defaultNs default namespace (optional)
     *
     * @return array array containing namespace and local part
     */
    public static function splitQualifiedName($qname, $defaultNs = null)
    {
        if (strstr($qname, ':')) {
            $tmp = explode(':', $qname);
            return array(
                'namespace' => $tmp[0],
                'localPart' => $tmp[1]
            );
        }
        return array(
            'namespace' => $defaultNs,
            'localPart' => $qname
        );
    }

    /**
     * Check, whether string is valid XML name
     *
     * <p>XML names are used for tagname, attribute names and various
     * other, lesser known entities.</p>
     * <p>An XML name may only consist of alphanumeric characters,
     * dashes, undescores and periods, and has to start with a letter
     * or an underscore.</p>
     *
     * <code>
     * require_once 'XML/Util.php';
     *
     * // verify tag name
     * $result = XML_Util::isValidName('invalidTag?');
     * if (is_a($result, 'PEAR_Error')) {
     *    print 'Invalid XML name: ' . $result->getMessage();
     * }
     * </code>
     *
     * @param string $string string that should be checked
     *
     * @return mixed true, if string is a valid XML name, PEAR error otherwise
     *
     * @todo support for other charsets
     * @todo PEAR CS - unable to avoid 85-char limit on second preg_match
     */
    public static function isValidName($string)
    {
        // check for invalid chars
        if (!is_string($string) || !strlen($string) || !preg_match('/^[[:alpha:]_]\\z/', $string[0])) {
            return XML_Util::raiseError(
                'XML names may only start with letter or underscore',
                XML_UTIL_ERROR_INVALID_START
            );
        }

        // check for invalid chars
        $match = preg_match(
            '/^([[:alpha:]_]([[:alnum:]\-\.]*)?:)?'
            . '[[:alpha:]_]([[:alnum:]\_\-\.]+)?\\z/',
            $string
        );
        if (!$match) {
            return XML_Util::raiseError(
                'XML names may only contain alphanumeric '
                . 'chars, period, hyphen, colon and underscores',
                XML_UTIL_ERROR_INVALID_CHARS
            );
        }
        // XML name is valid
        return true;
    }

    /**
     * Replacement for XML_Util::raiseError
     *
     * Avoids the necessity to always require
     * PEAR.php
     *
     * @param string $msg  error message
     * @param int    $code error code
     *
     * @return PEAR_Error
     * @todo   PEAR CS - should this use include_once instead?
     */
    public static function raiseError($msg, $code)
    {
        include_once 'PEAR.php';
        return PEAR::raiseError($msg, $code);
    }
}
?>
PK     Z#F:  :    pearcmd.phpnu [        <?php
/**
 * PEAR, the PHP Extension and Application Repository
 *
 * Command line interface
 *
 * PHP versions 4 and 5
 *
 * @category  pear
 * @package   PEAR
 * @author    Stig Bakken <ssb@php.net>
 * @author    Tomas V.V.Cox <cox@idecnet.com>
 * @copyright 1997-2009 The Authors
 * @license   http://opensource.org/licenses/bsd-license.php New BSD License
 * @link      http://pear.php.net/package/PEAR
 */

@ob_end_clean();
if (!defined('PEAR_RUNTYPE')) {
    // this is defined in peclcmd.php as 'pecl'
    define('PEAR_RUNTYPE', 'pear');
}
define('PEAR_IGNORE_BACKTRACE', 1);
/**
 * @nodep Gtk
 */
//the space is needed for windows include paths with trailing backslash
// http://pear.php.net/bugs/bug.php?id=19482
if ('/opt/alt/php74/usr/share/pear ' != '@'.'include_path'.'@ ') {
    ini_set('include_path', trim('/opt/alt/php74/usr/share/pear '). PATH_SEPARATOR .  get_include_path());
    $raw = false;
} else {
    // this is a raw, uninstalled pear, either a cvs checkout, or php distro
    ini_set('include_path', dirname(__DIR__) . PATH_SEPARATOR . get_include_path());
    $raw = true;
}
@ini_set('allow_url_fopen', true);
@set_time_limit(0);
ob_implicit_flush(true);
@ini_set('track_errors', true);
@ini_set('html_errors', false);
$_PEAR_PHPDIR = '#$%^&*';
set_error_handler('error_handler');

$pear_package_version = "1.10.16";

require_once 'PEAR.php';
require_once 'PEAR/Frontend.php';
require_once 'PEAR/Config.php';
require_once 'PEAR/Command.php';
require_once 'Console/Getopt.php';


PEAR_Command::setFrontendType('CLI');
$all_commands = PEAR_Command::getCommands();

$argv = Console_Getopt::readPHPArgv();
// fix CGI sapi oddity - the -- in pear.bat/pear is not removed
if (php_sapi_name() != 'cli' && isset($argv[1]) && $argv[1] == '--') {
    unset($argv[1]);
    $argv = array_values($argv);
}
$progname = PEAR_RUNTYPE;
array_shift($argv);
$options = Console_Getopt::getopt2($argv, "c:C:d:D:Gh?sSqu:vV");
if (PEAR::isError($options)) {
    usage($options);
}

$opts = $options[0];

$fetype = 'CLI';
if ($progname == 'gpear' || $progname == 'pear-gtk') {
    $fetype = 'Gtk2';
} else {
    foreach ($opts as $opt) {
        if ($opt[0] == 'G') {
            $fetype = 'Gtk2';
        }
    }
}

$pear_user_config = '';
$pear_system_config = '';
$store_user_config = false;
$store_system_config = false;
$verbose = 1;

foreach ($opts as $opt) {
    switch ($opt[0]) {
    case 'c':
        $pear_user_config = $opt[1];
        break;
    case 'C':
        $pear_system_config = $opt[1];
        break;
    }
}

PEAR_Command::setFrontendType($fetype);
$ui = &PEAR_Command::getFrontendObject();
$config = &PEAR_Config::singleton($pear_user_config, $pear_system_config);

if (PEAR::isError($config)) {
    $_file = '';
    if ($pear_user_config !== false) {
        $_file .= $pear_user_config;
    }
    if ($pear_system_config !== false) {
        $_file .= '/' . $pear_system_config;
    }
    if ($_file == '/') {
        $_file = 'The default config file';
    }
    $config->getMessage();
    $ui->outputData("ERROR: $_file is not a valid config file or is corrupted.");
    // We stop, we have no idea where we are :)
    exit(1);
}

// this is used in the error handler to retrieve a relative path
$_PEAR_PHPDIR = $config->get('php_dir');
$ui->setConfig($config);
PEAR::setErrorHandling(PEAR_ERROR_CALLBACK, array($ui, "displayFatalError"));

$verbose = $config->get("verbose");
$cmdopts = array();

if ($raw) {
    if (!$config->isDefinedLayer('user') && !$config->isDefinedLayer('system')) {
        $found = false;
        foreach ($opts as $opt) {
            if ($opt[0] == 'd' || $opt[0] == 'D') {
                // the user knows what they are doing, and are setting config values
                $found = true;
            }
        }
        if (!$found) {
            // no prior runs, try to install PEAR
            $parent = dirname(__FILE__);
            if (strpos($parent, 'scripts')) {
                $grandparent = dirname($parent);
                $packagexml = $grandparent . DIRECTORY_SEPARATOR . 'package2.xml';
                $pearbase = $grandparent;
            } else {
                $packagexml = $parent . DIRECTORY_SEPARATOR . 'package2.xml';
                $pearbase = $parent;
            }
            if (file_exists($packagexml)) {
                $options[1] = array(
                    'install',
                    $packagexml
                );
                $config->set('php_dir', $pearbase . DIRECTORY_SEPARATOR . 'php');
                $config->set('data_dir', $pearbase . DIRECTORY_SEPARATOR . 'data');
                $config->set('doc_dir', $pearbase . DIRECTORY_SEPARATOR . 'docs');
                $config->set('test_dir', $pearbase . DIRECTORY_SEPARATOR . 'tests');
                $config->set(
                    'ext_dir',
                    $pearbase . DIRECTORY_SEPARATOR . 'extensions'
                );
                $config->set('bin_dir', $pearbase);
                $config->mergeConfigFile($pearbase . 'pear.ini', false);
                $config->store();
                $config->set('auto_discover', 1);
            }
        }
    }
}
foreach ($opts as $opt) {
    $param = !empty($opt[1]) ? $opt[1] : true;
    switch ($opt[0]) {
    case 'd':
        if ($param === true) {
            die(
                'Invalid usage of "-d" option, expected -d config_value=value, ' .
                'received "-d"' . "\n"
            );
        }
        $possible = explode('=', $param);
        if (count($possible) != 2) {
            die(
                'Invalid usage of "-d" option, expected ' .
                '-d config_value=value, received "' . $param . '"' . "\n"
            );
        }
        list($key, $value) = explode('=', $param);
        $config->set($key, $value, 'user');
        break;
    case 'D':
        if ($param === true) {
            die(
                'Invalid usage of "-d" option, expected ' .
                '-d config_value=value, received "-d"' . "\n"
            );
        }
        $possible = explode('=', $param);
        if (count($possible) != 2) {
            die(
                'Invalid usage of "-d" option, expected ' .
                '-d config_value=value, received "' . $param . '"' . "\n"
            );
        }
        list($key, $value) = explode('=', $param);
        $config->set($key, $value, 'system');
        break;
    case 's':
        $store_user_config = true;
        break;
    case 'S':
        $store_system_config = true;
        break;
    case 'u':
        $config->remove($param, 'user');
        break;
    case 'v':
        $config->set('verbose', $config->get('verbose') + 1);
        break;
    case 'q':
        $config->set('verbose', $config->get('verbose') - 1);
        break;
    case 'V':
        usage(null, 'version');
    case 'c':
    case 'C':
        break;
    default:
        // all non pear params goes to the command
        $cmdopts[$opt[0]] = $param;
        break;
    }
}

if ($store_system_config) {
    $config->store('system');
}

if ($store_user_config) {
    $config->store('user');
}

$command = (isset($options[1][0])) ? $options[1][0] : null;
if (empty($command) && ($store_user_config || $store_system_config)) {
    exit;
}

if ($fetype == 'Gtk2') {
    if (!$config->validConfiguration()) {
        PEAR::raiseError(
            "CRITICAL ERROR: no existing valid configuration files found in " .
            "files '$pear_user_config' or '$pear_system_config', " .
            "please copy an existing configuration file to one of these " .
            "locations, or use the -c and -s options to create one"
        );
    }
    Gtk::main();
} else {
    do {
        if ($command == 'help') {
            usage(null, isset($options[1][1]) ? $options[1][1] : null);
        }

        if (!$config->validConfiguration()) {
            PEAR::raiseError(
                "CRITICAL ERROR: no existing valid configuration files found " .
                "in files '$pear_user_config' or '$pear_system_config', " .
                "please copy an existing configuration file to one of " .
                "these locations, or use the -c and -s options to create one"
            );
        }

        PEAR::pushErrorHandling(PEAR_ERROR_RETURN);
        $cmd = PEAR_Command::factory($command, $config);
        PEAR::popErrorHandling();
        if (PEAR::isError($cmd)) {
            usage(null, isset($options[1][0]) ? $options[1][0] : null);
        }

        $short_args = $long_args = null;
        PEAR_Command::getGetoptArgs($command, $short_args, $long_args);
        array_shift($options[1]);
        $tmp = Console_Getopt::getopt2($options[1], $short_args, $long_args);

        if (PEAR::isError($tmp)) {
            break;
        }

        list($tmpopt, $params) = $tmp;
        $opts = array();
        foreach ($tmpopt as $foo => $tmp2) {
            list($opt, $value) = $tmp2;
            if ($value === null) {
                $value = true; // options without args
            }

            if (strlen($opt) == 1) {
                $cmdoptions = $cmd->getOptions($command);
                foreach ($cmdoptions as $o => $d) {
                    if (isset($d['shortopt']) && $d['shortopt'] == $opt) {
                        $opts[$o] = $value;
                    }
                }
            } else {
                if (substr($opt, 0, 2) == '--') {
                    $opts[substr($opt, 2)] = $value;
                }
            }
        }

        $ok = $cmd->run($command, $opts, $params);
        if ($ok === false) {
            PEAR::raiseError("unknown command `$command'");
        }

        if (PEAR::isError($ok)) {
            PEAR::setErrorHandling(
                PEAR_ERROR_CALLBACK, array($ui, "displayFatalError")
            );
            PEAR::raiseError($ok);
        }
    } while (false);
}

// {{{ usage()

/**
 * Display usage information
 *
 * @param mixed $error       Optional error message
 * @param mixed $helpsubject Optional subject/command to display help for
 *
 * @return void
 */
function usage($error = null, $helpsubject = null)
{
    global $progname, $all_commands;
    $stdout = fopen('php://stdout', 'w');
    if (PEAR::isError($error)) {
        fputs($stdout, $error->getMessage() . "\n");
    } elseif ($error !== null) {
        fputs($stdout, "$error\n");
    }

    if ($helpsubject != null) {
        $put = cmdHelp($helpsubject);
    } else {
        $put = "Commands:\n";
        $maxlen = max(array_map("strlen", $all_commands));
        $formatstr = "%-{$maxlen}s  %s\n";
        ksort($all_commands);
        foreach ($all_commands as $cmd => $class) {
            $put .= sprintf($formatstr, $cmd, PEAR_Command::getDescription($cmd));
        }
        $put .=
            "Usage: $progname [options] command [command-options] <parameters>\n".
            "Type \"$progname help options\" to list all options.\n".
            "Type \"$progname help shortcuts\" to list all command shortcuts.\n".
            "Type \"$progname help version\" or ".
            "\"$progname version\" to list version information.\n".
            "Type \"$progname help <command>\" to get the help ".
            "for the specified command.";
    }
    fputs($stdout, "$put\n");
    fclose($stdout);

    if ($error === null) {
        exit(0);
    }
    exit(1);
}

/**
 * Return help string for specified command
 *
 * @param string $command Command to return help for
 *
 * @return void
 */
function cmdHelp($command)
{
    global $progname, $all_commands, $config;
    if ($command == "options") {
        return
        "Options:\n".
        "     -v         increase verbosity level (default 1)\n".
        "     -q         be quiet, decrease verbosity level\n".
        "     -c file    find user configuration in `file'\n".
        "     -C file    find system configuration in `file'\n".
        "     -d foo=bar set user config variable `foo' to `bar'\n".
        "     -D foo=bar set system config variable `foo' to `bar'\n".
        "     -G         start in graphical (Gtk) mode\n".
        "     -s         store user configuration\n".
        "     -S         store system configuration\n".
        "     -u foo     unset `foo' in the user configuration\n".
        "     -h, -?     display help/usage (this message)\n".
        "     -V         version information\n";
    } elseif ($command == "shortcuts") {
        $sc = PEAR_Command::getShortcuts();
        $ret = "Shortcuts:\n";
        foreach ($sc as $s => $c) {
            $ret .= sprintf("     %-8s %s\n", $s, $c);
        }
        return $ret;

    } elseif ($command == "version") {
        return "PEAR Version: ".$GLOBALS['pear_package_version'].
               "\nPHP Version: ".phpversion().
               "\nZend Engine Version: ".zend_version().
               "\nRunning on: ".php_uname();

    } elseif ($help = PEAR_Command::getHelp($command)) {
        if (is_string($help)) {
            return "$progname $command [options] $help\n";
        }

        if ($help[1] === null) {
            return "$progname $command $help[0]";
        }

        return "$progname $command [options] $help[0]\n$help[1]";
    }

    return "Command '$command' is not valid, try '$progname help'";
}

// }}}

/**
 * error_handler
 *
 * @param mixed $errno  Error number
 * @param mixed $errmsg Message
 * @param mixed $file   Filename
 * @param mixed $line   Line number
 *
 * @access public
 * @return boolean
 */
function error_handler($errno, $errmsg, $file, $line)
{
    if ((!defined('PHP_VERSION_ID') || PHP_VERSION_ID < 80400) && ($errno & E_STRICT)) {
        return; // E_STRICT
    }
    if ($errno & E_DEPRECATED) {
        return; // E_DEPRECATED
    }
    if (!(error_reporting() & $errno) &&
        isset($GLOBALS['config']) &&
        $GLOBALS['config']->get('verbose') < 4
    ) {
        return false; // @silenced error, show all if debug is high enough
    }
    $errortype = array (
        E_DEPRECATED  => 'Deprecated Warning',
        E_ERROR   =>  "Error",
        E_WARNING   =>  "Warning",
        E_PARSE   =>  "Parsing Error",
        E_NOTICE   =>  "Notice",
        E_CORE_ERROR  =>  "Core Error",
        E_CORE_WARNING  =>  "Core Warning",
        E_COMPILE_ERROR  =>  "Compile Error",
        E_COMPILE_WARNING =>  "Compile Warning",
        E_USER_ERROR =>  "User Error",
        E_USER_WARNING =>  "User Warning",
        E_USER_NOTICE =>  "User Notice"
    );
    if (!defined('PHP_VERSION_ID') || PHP_VERSION_ID < 80400) {
        $errortype[E_STRICT] = 'Strict Warning';
    }
    $prefix = $errortype[$errno];
    global $_PEAR_PHPDIR;
    if (stristr($file, $_PEAR_PHPDIR)) {
        $file = substr($file, strlen($_PEAR_PHPDIR) + 1);
    } else {
        $file = basename($file);
    }
    print "\n$prefix: $errmsg in $file on line $line\n";
    return false;
}


/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * indent-tabs-mode: nil
 * mode: php
 * End:
 */
// vim600:syn=php
PK     Z0$-  -     .pkgxml/Pear_File_MARC-1.4.1.xmlnu [        <?xml version="1.0" encoding="UTF-8"?>
<package packagerversion="1.10.9" version="2.0" xmlns="http://pear.php.net/dtd/package-2.0" xmlns:tasks="http://pear.php.net/dtd/tasks-1.0" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://pear.php.net/dtd/tasks-1.0 http://pear.php.net/dtd/tasks-1.0.xsd http://pear.php.net/dtd/package-2.0 http://pear.php.net/dtd/package-2.0.xsd">
 <name>File_MARC</name>
 <channel>pear.php.net</channel>
 <summary>Parse, modify, and create MARC records</summary>
 <description>The standard for machine-readable cataloging (MARC) records is documented at http://loc.gov/marc/. This package enables you to read existing MARC records from a file, string, or (using the YAZ extension), from a Z39.50 source. You can also use this package to create new MARC records.

This package is based on the PHP MARC package, originally called &quot;php-marc&quot;, that is part of the Emilda Project (http://www.emilda.org).  Christoffer Landtman generously agreed to make the &quot;php-marc&quot; code available under the GNU LGPL so it could be used as the basis of this PEAR package.</description>
 <lead>
  <name>Dan Scott</name>
  <user>dbs</user>
  <email>dbs@php.net</email>
  <active>yes</active>
 </lead>
 <date>2019-11-13</date>
 <time>17:33:33</time>
 <version>
  <release>1.4.1</release>
  <api>1.4.0</api>
 </version>
 <stability>
  <release>stable</release>
  <api>stable</api>
 </stability>
 <license uri="http://www.gnu.org/copyleft/lesser.html">GNU Lesser General Public License</license>
 <notes>
1.4.1
  * Reintroduce include_path to composer.json
 </notes>
 <contents>
  <dir baseinstalldir="File" name="/">
   <file baseinstalldir="/" md5sum="ffe2590635f404af055fe731da105939" name="File/MARC/Lint/CodeData.php" role="php" />
   <file baseinstalldir="/" md5sum="a3b9e1c817e67f59add05593500d463e" name="File/MARC/Control_Field.php" role="php" />
   <file baseinstalldir="/" md5sum="acba4463f8e40d6ee4e7ba1f2dc53123" name="File/MARC/Data_Field.php" role="php" />
   <file baseinstalldir="/" md5sum="378f483ec1fd161cd9dfb9b05db58d66" name="File/MARC/Exception.php" role="php" />
   <file baseinstalldir="/" md5sum="a9ba69030cbf4847292b9975e9d922c6" name="File/MARC/Field.php" role="php" />
   <file baseinstalldir="/" md5sum="c37351c7c80927ce6bb1e662510f9183" name="File/MARC/Lint.php" role="php" />
   <file baseinstalldir="/" md5sum="c5ac39e3d25eb0108b9dd5ad8466c821" name="File/MARC/List.php" role="php" />
   <file baseinstalldir="/" md5sum="7af6ce78cde4bb246e4e96709e90d2c0" name="File/MARC/Record.php" role="php" />
   <file baseinstalldir="/" md5sum="df5f97139685f0ad04e47fdbb67e3968" name="File/MARC/Subfield.php" role="php" />
   <file baseinstalldir="/" md5sum="6e102262dfd5d65d8d5a6221dfc74edd" name="File/MARC.php" role="php" />
   <file baseinstalldir="/" md5sum="8b64040d2d18ac4b994595a148266a66" name="File/MARCBASE.php" role="php" />
   <file baseinstalldir="/" md5sum="f5a17a60d34e84e2ee7ca1e5c5c6c620" name="File/MARCXML.php" role="php" />
   <file baseinstalldir="File" md5sum="6af971fd9ff5ef3e5fcba8a64a0e7a49" name="examples/example.mrc" role="doc" />
   <file baseinstalldir="File" md5sum="1aea40df9f1c2e32f8e065fe8da0a3b5" name="examples/marc_yaz.php" role="doc" />
   <file baseinstalldir="File" md5sum="70ae317e0890d5896e3ffccdd293369a" name="examples/read.php" role="doc" />
   <file baseinstalldir="File" md5sum="54efb9d920ac775bfc847a83a2dec73a" name="examples/subfields.php" role="doc" />
   <file baseinstalldir="File" md5sum="14297da35b510c496cd1f328cf3bb301" name="tests/bad_example.mrc" role="test" />
   <file baseinstalldir="File" md5sum="0dd4c6dbad6e8608d0ebf9d24425955d" name="tests/bad_example.xml" role="test" />
   <file baseinstalldir="File" md5sum="216a6f9d53f6d3d8e22d81fdc10f4bc1" name="tests/camel.mrc" role="test" />
   <file baseinstalldir="File" md5sum="6af971fd9ff5ef3e5fcba8a64a0e7a49" name="tests/example.mrc" role="test" />
   <file baseinstalldir="File" md5sum="9455aec3371f79cf7d31281fd03f5171" name="tests/marc_001.phpt" role="test" />
   <file baseinstalldir="File" md5sum="577e2f5f89fd48cf1a3576a29d2958f8" name="tests/marc_002.phpt" role="test" />
   <file baseinstalldir="File" md5sum="a6a586354b9a06c880c2a7470dd828d7" name="tests/marc_003.phpt" role="test" />
   <file baseinstalldir="File" md5sum="b1c7e2fc5572a0f9362a771c260a1fe3" name="tests/marc_004.phpt" role="test" />
   <file baseinstalldir="File" md5sum="84f85ac7f8a2532e4e90ccde7764556a" name="tests/marc_005.phpt" role="test" />
   <file baseinstalldir="File" md5sum="889e5e906a97de7a0e94b0c5f30e70e7" name="tests/marc_006.phpt" role="test" />
   <file baseinstalldir="File" md5sum="be823dc8a8efb408de06371ab802efcc" name="tests/marc_007.phpt" role="test" />
   <file baseinstalldir="File" md5sum="af44490aafea44e5120f881cb81cdad9" name="tests/marc_008.phpt" role="test" />
   <file baseinstalldir="File" md5sum="ac9f2a7aeb24900a11f5102744179ae3" name="tests/marc_009.phpt" role="test" />
   <file baseinstalldir="File" md5sum="5e43eb5cddbb4f2bd0d6405cd95b80aa" name="tests/marc_010.phpt" role="test" />
   <file baseinstalldir="File" md5sum="2e1cd8509c0bf04ec77fc0f51375f611" name="tests/marc_011.phpt" role="test" />
   <file baseinstalldir="File" md5sum="bc49ca48623102c8ac07679fa77a3dd9" name="tests/marc_012.phpt" role="test" />
   <file baseinstalldir="File" md5sum="245a3b20e0da1973535503e28efd1d3a" name="tests/marc_013.phpt" role="test" />
   <file baseinstalldir="File" md5sum="ce69aa734856dd40d60d30c77bdef53a" name="tests/marc_014.phpt" role="test" />
   <file baseinstalldir="File" md5sum="7b61cdd31035e60682fa92c5d519d997" name="tests/marc_015.phpt" role="test" />
   <file baseinstalldir="File" md5sum="7f4f597fe1adaa3507eb87fc4875dbd2" name="tests/marc_016.phpt" role="test" />
   <file baseinstalldir="File" md5sum="4262efd5fcacbd7503323681f4552430" name="tests/marc_017.phpt" role="test" />
   <file baseinstalldir="File" md5sum="ee1623d7bda54ba4f741b2319b158f82" name="tests/marc_018.phpt" role="test" />
   <file baseinstalldir="File" md5sum="8006c03fbab0f04377342086fafd3982" name="tests/marc_019.phpt" role="test" />
   <file baseinstalldir="File" md5sum="f5ae5b25e0ac1104314433d9e5b1f7fe" name="tests/marc_020.phpt" role="test" />
   <file baseinstalldir="File" md5sum="94d8f99e86585671a1b2d3043aeb6360" name="tests/marc_021.phpt" role="test" />
   <file baseinstalldir="File" md5sum="bb891a890effe0e77e1759043f4846d8" name="tests/marc_022.phpt" role="test" />
   <file baseinstalldir="File" md5sum="faf77dcb5cdd002c09169a14bfded65e" name="tests/marc_023.phpt" role="test" />
   <file baseinstalldir="File" md5sum="0e6c4a23f5da7df64fdd4b3b1149bbee" name="tests/marc_16783.phpt" role="test" />
   <file baseinstalldir="File" md5sum="4ea8b3520beda378db15960f1c50493a" name="tests/marc_field_001.phpt" role="test" />
   <file baseinstalldir="File" md5sum="743850ef61df69c4340be44a03bccb5a" name="tests/marc_field_002.phpt" role="test" />
   <file baseinstalldir="File" md5sum="8c765f1398b49df257d86d22ac6e4f8d" name="tests/marc_field_003.phpt" role="test" />
   <file baseinstalldir="File" md5sum="d4216cf930a308e0d02553bd3914f294" name="tests/marc_field_004.phpt" role="test" />
   <file baseinstalldir="File" md5sum="dd59514b474b78371ad522a3e13b76b7" name="tests/marc_field_005.phpt" role="test" />
   <file baseinstalldir="File" md5sum="5295bfd02a8424acac1c91de26b552c7" name="tests/marc_field_21246.phpt" role="test" />
   <file baseinstalldir="File" md5sum="6f95f09442ccf6b93df92986f58b270d" name="tests/marc_lint_001.phpt" role="test" />
   <file baseinstalldir="File" md5sum="18d721f8871edd925624ccbdd15444fb" name="tests/marc_lint_002.phpt" role="test" />
   <file baseinstalldir="File" md5sum="4eb86a2f4e0a74dedcb0bf2511da6ca1" name="tests/marc_lint_003.phpt" role="test" />
   <file baseinstalldir="File" md5sum="10430fee0a1352d0528f1507b34c57e1" name="tests/marc_lint_004.phpt" role="test" />
   <file baseinstalldir="File" md5sum="0dcd07a18577d7b22392a2214cd6436c" name="tests/marc_lint_005.phpt" role="test" />
   <file baseinstalldir="File" md5sum="fd17798fc535562e5e64461f449516b2" name="tests/marc_record_001.phpt" role="test" />
   <file baseinstalldir="File" md5sum="141b455ccf67a60c4eac44c77960f2e2" name="tests/marc_subfield_001.phpt" role="test" />
   <file baseinstalldir="File" md5sum="b1e23464ee0eec47ddca2af9a9922926" name="tests/marc_subfield_002.phpt" role="test" />
   <file baseinstalldir="File" md5sum="1d7903e8c244d647220b318b4bc758d1" name="tests/marc_xml_001.phpt" role="test" />
   <file baseinstalldir="File" md5sum="22cae3c752d843b78f583dcd92801534" name="tests/marc_xml_002.phpt" role="test" />
   <file baseinstalldir="File" md5sum="c0716d23e0f960abf8d53ad6014a413a" name="tests/marc_xml_003.phpt" role="test" />
   <file baseinstalldir="File" md5sum="7645fa19e28aca041a124fc5e2c12bd3" name="tests/marc_xml_004.phpt" role="test" />
   <file baseinstalldir="File" md5sum="b08b28c3251625b8b6e4d7ab376b0185" name="tests/marc_xml_005.phpt" role="test" />
   <file baseinstalldir="File" md5sum="6cd0934ddbf65d2d7e9ea5d108977ff7" name="tests/marc_xml_006.phpt" role="test" />
   <file baseinstalldir="File" md5sum="ffdce4aa190268fa7cbbf3bb45383abf" name="tests/marc_xml_007.phpt" role="test" />
   <file baseinstalldir="File" md5sum="998a4c1ce00ff100f39ffa9fc49a212f" name="tests/marc_xml_008.phpt" role="test" />
   <file baseinstalldir="File" md5sum="a38a67c590869f2b0a4859a910fd9e89" name="tests/marc_xml_009.phpt" role="test" />
   <file baseinstalldir="File" md5sum="db39c5055a570fcda12bb7c5fa1e3b4d" name="tests/marc_xml_16642.phpt" role="test" />
   <file baseinstalldir="File" md5sum="dc075f4a271bd7182378a3874bd5b39e" name="tests/marc_xml_namespace.phpt" role="test" />
   <file baseinstalldir="File" md5sum="208200d9a979ac360c4d326e9789b6b7" name="tests/marc_xml_namespace_prefix.phpt" role="test" />
   <file baseinstalldir="File" md5sum="d87774ede82222cfc12b2c2a06f57a3a" name="tests/marc_xml_rsinger.phpt" role="test" />
   <file baseinstalldir="File" md5sum="fb44e3eb69f1a37115235fc829c39542" name="tests/namespace.xml" role="test" />
   <file baseinstalldir="File" md5sum="e6de4d809b0af5dd99fd352eb60616a1" name="tests/skipif.inc" role="test" />
   <file baseinstalldir="File" md5sum="a0166a4ff0bc0bb283e854f11beaa793" name="tests/music.mrc" role="test" />
   <file baseinstalldir="File" md5sum="993ce494befbc25dc1326527b30b4782" name="tests/music.xml" role="test" />
   <file baseinstalldir="File" md5sum="6b0c87f0cd459ed7322c25d403b164cc" name="tests/bigarchive.xml" role="test" />
   <file baseinstalldir="File" md5sum="be850aa6ab4ac52b8c6c214c30b574af" name="tests/onerecord.xml" role="test" />
   <file baseinstalldir="File" md5sum="4f43fec7b276267eec5258044853ccc8" name="tests/sandburg.mrc" role="test" />
   <file baseinstalldir="File" md5sum="e216b95f905dd240c9bde372950e4073" name="tests/sandburg.xml" role="test" />
   <file baseinstalldir="File" md5sum="e81a92188ea0dcd9c108a63fb32b3f4f" name="tests/xmlescape.mrc" role="test" />
   <file baseinstalldir="File" md5sum="8f9f63ed6a80191d89a2296fa4c1a30b" name="CHANGELOG" role="doc" />
   <file baseinstalldir="File" md5sum="fbc093901857fcd118f065f900982c24" name="LICENSE" role="doc" />
  </dir>
 </contents>
 <dependencies>
  <required>
   <php>
    <min>5.6</min>
   </php>
   <pearinstaller>
    <min>1.4.0</min>
   </pearinstaller>
  </required>
  <optional>
   <package>
    <name>Validate_ISPN</name>
    <channel>pear.php.net</channel>
   </package>
  </optional>
 </dependencies>
 <phprelease />
</package>
PK       mZPi                    .depdbnu [        PK       mZ{    
              Mail/null.phpnu [        PK       mZE@%>  >              
  Mail/sendmail.phpnu [        PK       mZA~{  {              ;  Mail/RFC822.phpnu [        PK       nZ+V    
            [  Mail/mime.phpnu [        PK       nZxع                D Mail/mimeDecode.phpnu [        PK       nZB_Ǭ=  =  
            @( Mail/smtp.phpnu [        PK       nZwԐE;  E;              )f Mail/smtpmx.phpnu [        PK       nZ؛J  J               Mail/mimePart.phpnu [        PK       nZBsh  h  
            8Y Mail/mock.phpnu [        PK       nZ@iE\  \  
            l Mail/mail.phpnu [        PK       nZ                        v .locknu [        PK       nZ*z%-  %-       