<?php

/**
 * Encoding tests for SimplePie_Misc::change_encoding() and SimplePie_Misc::encoding()
 *
 * SimplePie
 *
 * A PHP-Based RSS and Atom Feed Framework.
 * Takes the hard work out of managing a complete RSS/Atom solution.
 *
 * Copyright (c) 2004-2022, Ryan Parman, Sam Sneddon, Ryan McCue, and contributors
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 	* Redistributions of source code must retain the above copyright notice, this list of
 * 	  conditions and the following disclaimer.
 *
 * 	* Redistributions in binary form must reproduce the above copyright notice, this list
 * 	  of conditions and the following disclaimer in the documentation and/or other materials
 * 	  provided with the distribution.
 *
 * 	* Neither the name of the SimplePie Team nor the names of its contributors may be used
 * 	  to endorse or promote products derived from this software without specific prior
 * 	  written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS
 * AND CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @package SimplePie
 * @copyright 2004-2022 Ryan Parman, Sam Sneddon, Ryan McCue
 * @author Ryan Parman
 * @author Sam Sneddon
 * @author Ryan McCue
 * @link http://simplepie.org/ SimplePie
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

class EncodingTest extends PHPUnit\Framework\TestCase
{
    /**#@+
     * UTF-8 methods
     */
    /**
     * Provider for the convert toUTF8* tests
     */
    public static function toUTF8()
    {
        return [
            ['A', 'A', 'ASCII'],
            ["\xa1\xdb", "\xe2\x88\x9e", 'Big5'],
            ["\xa1\xe7", "\xe2\x88\x9e", 'EUC-JP'],
            ["\xa1\xde", "\xe2\x88\x9e", 'GBK'],
            ["\x81\x87", "\xe2\x88\x9e", 'Shift_JIS'],
            ["\x2b\x49\x68\x34\x2d", "\xe2\x88\x9e", 'UTF-7'],
            ["\xfe\xff\x22\x1e", "\xe2\x88\x9e", 'UTF-16'],
            ["\xff\xfe\x1e\x22", "\xe2\x88\x9e", 'UTF-16'],
            ["\x22\x1e", "\xe2\x88\x9e", 'UTF-16BE'],
            ["\x1e\x22", "\xe2\x88\x9e", 'UTF-16LE'],
        ];
    }

    /**
     * Special cases with mbstring handling
     */
    public static function toUTF8_mbstring()
    {
        return [
            ["\xa1\xc4", "\xe2\x88\x9e", 'EUC-KR'],
        ];
    }

    /**
     * Special cases with iconv handling
     */
    public static function toUTF8_iconv()
    {
        return [
            ["\xfe\xff\x22\x1e", "\xe2\x88\x9e", 'UTF-16'],
        ];
    }

    /**
     * Special cases with uconverter handling
     */
    public static function toUTF8_uconverter()
    {
        return [
            ["\xfe\xff\x22\x1e", "\xe2\x88\x9e", 'UTF-16'],
        ];
    }

    /**
     * Convert * to UTF-8
     *
     * @dataProvider toUTF8
     */
    public function test_convert_UTF8($input, $expected, $encoding)
    {
        $encoding = SimplePie_Misc::encoding($encoding);
        $this->assertSameBin2Hex($expected, SimplePie_Misc::change_encoding($input, $encoding, 'UTF-8'));
    }

    /**
     * Convert * to UTF-8 using mbstring
     *
     * Special cases only
     * @dataProvider toUTF8_mbstring
     */
    public function test_convert_UTF8_mbstring($input, $expected, $encoding)
    {
        $encoding = SimplePie_Misc::encoding($encoding);
        if (extension_loaded('mbstring')) {
            $this->assertSameBin2Hex($expected, Mock_Misc::change_encoding_mbstring($input, $encoding, 'UTF-8'));
        }
    }

    /**
     * Convert * to UTF-8 using iconv
     *
     * Special cases only
     * @dataProvider toUTF8_iconv
     */
    public function test_convert_UTF8_iconv($input, $expected, $encoding)
    {
        $encoding = SimplePie_Misc::encoding($encoding);
        if (extension_loaded('iconv')) {
            $this->assertSameBin2Hex($expected, Mock_Misc::change_encoding_iconv($input, $encoding, 'UTF-8'));
        }
    }

    /**
     * Convert * to UTF-8 using UConverter
     *
     * Special cases only
     * @dataProvider toUTF8_uconverter
     */
    public function test_convert_UTF8_uconverter($input, $expected, $encoding)
    {
        $encoding = SimplePie_Misc::encoding($encoding);
        if (extension_loaded('intl')) {
            $this->assertSameBin2Hex($expected, Mock_Misc::change_encoding_uconverter($input, $encoding, 'UTF-8'));
        }
    }
    /**#@-*/

    /**#@+
     * UTF-16 methods
     */
    public static function toUTF16()
    {
        return [
            ["\x22\x1e", "\x22\x1e", 'UTF-16BE'],
            ["\x1e\x22", "\x22\x1e", 'UTF-16LE'],
        ];
    }

    /**
     * Convert * to UTF-16
     * @dataProvider toUTF16
     */
    public function test_convert_UTF16($input, $expected, $encoding)
    {
        $encoding = SimplePie_Misc::encoding($encoding);
        $this->assertSameBin2Hex($expected, SimplePie_Misc::change_encoding($input, $encoding, 'UTF-16'));
    }
    /**#@-*/

    public function test_nonexistant()
    {
        $this->assertFalse(SimplePie_Misc::change_encoding('', 'TESTENC', 'UTF-8'));
    }

    public static function assertSameBin2Hex($expected, $actual, $message = '')
    {
        if (is_string($expected)) {
            $expected = bin2hex($expected);
        }
        if (is_string($actual)) {
            $actual = bin2hex($actual);
        }
        static::assertSame($expected, $actual, $message);
    }
}

class Mock_Misc extends SimplePie_Misc
{
    public static function __callStatic($name, $args)
    {
        return call_user_func_array(['SimplePie_Misc', $name], $args);
    }
}
