<?php

/**
 * TSignalsDispatcher class file.
 *
 * @author Brad Anderson <belisoful@icloud.com>
 * @link https://github.com/pradosoft/prado
 * @license https://github.com/pradosoft/prado/blob/master/LICENSE
 */

namespace Prado\Util;

use Prado\Collections\TPriorityPropertyTrait;
use Prado\Collections\TWeakCallableCollection;
use Prado\Exceptions\{TExitException};
use Prado\Exceptions\{TInvalidOperationException, TInvalidDataValueException};
use Prado\TComponent;
use Prado\Util\Helpers\TProcessHelper;

/**
 * TSignalsDispatcher class.
 *
 * This class handles linux process signals.  It translates the signals into global
 * events in PRADO.  There are special handlers for SIGALRM to handle time based
 * alarm callbacks and for SIGCHLD to handle specific Process ID.
 *
 * The signals translate to global events as such:
 *	SIGALRM	=> `fxSignalAlarm` ~
 *	SIGHUP	=> `fxSignalHangUp` ^
 *	SIGINT	=> `fxSignalInterrupt` ^
 *	SIGQUIT	=> `fxSignalQuit` ^
 *	SIGTRAP	=> `fxSignalTrap`
 *	SIGABRT => `fxSignalAbort` ^
 *	SIGUSR1 => `fxSignalUser1`
 *	SIGUSR2 => `fxSignalUser2`
 *	SIGTERM => `fxSignalTerminate` ^
 *	SIGCHLD => `fxSignalChild` @
 *	SIGCONT	=> `fxSignalContinue`
 *	SIGTSTP	=> `fxSignalTerminalStop`
 *	SIGTTIN => `fxSignalBackgroundRead`
 *	SIGTTOU	=> `fxSignalBackgroundWrite`
 *	SIGURG	=> `fxSignalUrgent`
 * ^ Designates an exiting signal-event unless changed in the TSignalParameter.
 * ~ SIGALRM has a special handler {@see self::ring()} to process and manage the time
 *   queue of callbacks and reset the alarm.
 * @ SIGCHLD has a special handler {@see self::delegateChild()} to delegate to process
 *   specific handlers.
 *
 * The signal handlers are stored and restored on {@see self::attach()}ing and {@see
 * self::detach()}ing, respectively.  This installs and uninstalls the class as
 * the signals' handler through the {@see self::__invoke()} method.
 *
 * Alarms may be added with (@see self::alarm()) and removed with {@see self::disarm()}.
 * Alarms can be added without callbacks and will raise the `fxSignalAlarm` event
 * without a time callback.  By calling alarm() without parameters, the next alarm
 * time is returned.
 *
 * The methods {@see self::hasEvent()}, {@see self::hasEventHandler}, and {@see self::getEventHandlers}
 * will accept process signals (int) as event names and translate it into the associated
 * PRADO signal global event.  These methods will also return the proper values
 * for PID handlers, as well, by providing the event name in the format "pid:####"
 * (where #### is the PID).  hasEvent and getEventHandlers checks if the PID is running.
 * To get the PID handler without validation use {@see self::getPidHandlers()}.
 *
 * Child PID handlers can be checked for with {@see self::hasPidHandler()}.  The PID
 * handlers can be retrieved with {@see self::getPidHandlers()} (besides using getEventHandlers
 * with a special formatted event name [above]).  Handlers can be attached to a specific
 * PID with {@see self::attachPidHandler()} and detached with {@see self::detachPidHandler()}.
 * Child PID handlers can be cleared with {@see self::clearPidHandlers()}
 *
 * @author Brad Anderson <belisoful@icloud.com>
 * @since 4.3.0
 */
class TSignalsDispatcher extends TComponent implements \Prado\ISingleton
{
	use TPriorityPropertyTrait;

	public const FX_SIGNAL_ALARM = 'fxSignalAlarm';
	public const FX_SIGNAL_HANG_UP = 'fxSignalHangUp';
	public const FX_SIGNAL_INTERRUPT = 'fxSignalInterrupt';
	public const FX_SIGNAL_QUIT = 'fxSignalQuit';
	public const FX_SIGNAL_TRAP = 'fxSignalTrap';
	public const FX_SIGNAL_ABORT = 'fxSignalAbort';
	public const FX_SIGNAL_USER1 = 'fxSignalUser1';
	public const FX_SIGNAL_USER2 = 'fxSignalUser2';
	public const FX_SIGNAL_TERMINATE = 'fxSignalTerminate';
	public const FX_SIGNAL_CHILD = 'fxSignalChild';
	public const FX_SIGNAL_CONTINUE = 'fxSignalContinue';
	public const FX_SIGNAL_TERMINAL_STOP = 'fxSignalTerminalStop';
	public const FX_SIGNAL_BACKGROUND_READ = 'fxSignalBackgroundRead';
	public const FX_SIGNAL_BACKGROUND_WRITE = 'fxSignalBackgroundWrite';
	public const FX_SIGNAL_URGENT = 'fxSignalUrgent';

	public const SIGNAL_MAP = [
		SIGALRM => self::FX_SIGNAL_ALARM, // Alarm signal.  Sent by pcntl_alarm when the time is over.
		SIGHUP => self::FX_SIGNAL_HANG_UP, // Hangup signal. Sent to a process when its controlling terminal is closed.
		SIGINT => self::FX_SIGNAL_INTERRUPT, // Interrupt signal. Typically generated by pressing Ctrl+C.
		SIGQUIT => self::FX_SIGNAL_QUIT,  // Quit signal. Similar to SIGINT but produces a core dump when received by the process.
		SIGTRAP => self::FX_SIGNAL_TRAP,  // Trace/breakpoint trap signal. Used by debuggers to catch trace and breakpoint conditions.
		SIGABRT => self::FX_SIGNAL_ABORT, // Abort signal. Sent by the process itself to terminate due to a critical error condition.
		SIGUSR1 => self::FX_SIGNAL_USER1, // User-defined signal 1.
		SIGUSR2 => self::FX_SIGNAL_USER2, // User-defined signal 2.
		SIGTERM => self::FX_SIGNAL_TERMINATE, // Termination signal. Typically used to request graceful termination of a process.
		SIGCHLD => self::FX_SIGNAL_CHILD, // Child signal. Sent to a parent process when a child process terminates.
		SIGCONT => self::FX_SIGNAL_CONTINUE,  // Continue signal. Sent to resume a process that has been stopped.
		SIGTSTP => self::FX_SIGNAL_TERMINAL_STOP, // Terminal stop signal. Sent by pressing Ctrl+Z to suspend the process.
		SIGTTIN => self::FX_SIGNAL_BACKGROUND_READ,    // Background read signal. Sent to a process when it attempts to read from the terminal while in the background.
		SIGTTOU => self::FX_SIGNAL_BACKGROUND_WRITE,   // Background write signal. Sent to a process when it attempts to write to the terminal while in the background.
		SIGURG => self::FX_SIGNAL_URGENT, // Urgent condition signal. Indicates the arrival of out-of-band data on a socket.
	];

	/** The signals that exit by default. */
	public const EXIT_SIGNALS = [
			SIGABRT => true,
			SIGBUS => true,
			SIGFPE => true,
			SIGHUP => true,
			SIGILL => true,
			SIGINT => true,
			SIGPIPE => true,
			SIGQUIT => true,
			SIGSEGV => true,
			SIGSYS => true,
			SIGTERM => true,
		];

	/** @var callable The alarm when no callback is provided. */
	public const NULL_ALARM = [self::class, 'nullAlarm'];

	/** @var ?TSignalsDispatcher The Singleton instance of the class. This is the class
	 *    that gets installed as the signals handler.
	 */
	private static ?TSignalsDispatcher $_singleton = null;

	/** @var ?bool Are the signals async. */
	private static ?bool $_asyncSignals = null;

	/** @var array The handlers of the signals prior to attaching.
	 *   Format [0 => original value, 1 => closure for the event handler to call the original callable].
	 */
	private static array $_priorHandlers = [];

	/** @var ?float Any signal handlers are installed into PRADO at this priority. */
	private static ?float $_priorHandlerPriority = null;

	/** @var ?bool Were the signals async before TSignalsDispatcher. */
	private static ?bool $_priorAsync = null;

	/** @var array The integer alarm times and handlers */
	protected static array $_alarms = [];

	/** @var bool Is the $_alarms array ordered by time. */
	protected static bool $_alarmsOrdered = true;

	/** @var ?int The next alarm time in the _alarms array. */
	protected static ?int $_nextAlarmTime = null;

	/** @var array The pid handlers. */
	private static array $_pidHandlers = [];

	/**
	 * Returns the singleton of the class.  The singleton is created if/when $create
	 * is true.
	 * @param bool $create Should the singleton be created if not existing, default true.
	 * @return ?object The singleton of the class, null where none is available.
	 */
	public static function singleton(bool $create = true): ?object
	{
		if ($create && static::hasSignals() && !self::$_singleton) {
			$instance = new (static::class)();
		}

		return self::$_singleton;
	}

	/**
	 * Constructs the TSignalsDispatcher.
	 * The first instance is attached and set as the singleton.
	 */
	public function __construct()
	{
		parent::__construct();
		$this->attach();
	}

	/**
	 * This translates the global event into the signal that raises the event.
	 * @param string $event The event name to look up its signal.
	 * @return ?int The signal for an event name.
	 */
	public static function getSignalFromEvent(string $event): ?int
	{
		static $eventMap = null;

		if ($eventMap === null) {
			$eventMap = array_flip(static::SIGNAL_MAP);
		}
		return $eventMap[$event] ?? null;
	}

	/**
	 * @return bool Does the system support Process Signals (pcntl_signal)
	 */
	public static function hasSignals(): bool
	{
		return function_exists('pcntl_signal');
	}

	/**
	 * This sets the signals' handlers to this object, attaches the original handlers
	 * to the PRADO global events at the {@see self::getPriorHandlerPriority()}.
	 * The alarm handler and Child handler is installed for routing.
	 * @return bool Is the instance attached as the singleton.
	 */
	public function attach(): bool
	{
		if (!static::hasSignals() || self::$_singleton !== null) {
			return false;
		}

		self::$_singleton = $this;

		if (self::$_asyncSignals === null) {
			static::setAsyncSignals(true);
		}

		foreach (static::SIGNAL_MAP as $signal => $event) {
			$handler = pcntl_signal_get_handler($signal);

			if ($handler instanceof self) {
				continue;
			}
			self::$_priorHandlers[$signal] = [$handler];

			$callable = is_callable($handler);
			if ($callable) {
				$handler = function ($sender, $param) use ($handler) {
					return $handler($param->getSignal(), $param->getParameter());
				};
				self::$_priorHandlers[$signal][1] = $handler;
			}

			$installHandler = true;
			switch ($signal) {
				case SIGALRM:
					parent::attachEventHandler($event, [$this, 'ring'], $this->getPriority());
					if ($nextAlarm = pcntl_alarm(0)) {
						self::$_nextAlarmTime = $nextAlarm + time();
						if ($callable) {
							static::$_alarms[self::$_nextAlarmTime][] = $handler;
						}
						pcntl_alarm($nextAlarm);
					}
					$installHandler = false;
					break;
				case SIGCHLD:
					parent::attachEventHandler($event, [$this, 'delegateChild'], $this->getPriority());
					break;
			}

			if ($installHandler && $callable) {
				parent::attachEventHandler($event, $handler, static::getPriorHandlerPriority());
			}

			pcntl_signal($signal, $this);
		}
		return true;
	}

	/**
	 * Detaches the singleton when it is the singleton.  Prior signal handlers are
	 * restored.
	 * @return bool Is the instance detached from singleton.
	 */
	public function detach(): bool
	{
		if (self::$_singleton !== $this) {
			return false;
		}

		foreach (self::$_priorHandlers as $signal => $originalCallback) {
			pcntl_signal($signal, $originalCallback[0]);
			$uninstallHandler = true;
			switch ($signal) {
				case SIGALRM:
					parent::detachEventHandler(static::SIGNAL_MAP[$signal], [$this, 'ring']);
					pcntl_alarm(0);
					$uninstallHandler = false;
					break;
				case SIGCHLD:
					parent::detachEventHandler(static::SIGNAL_MAP[$signal], [$this, 'delegateChild']);
					break;
			}
			if ($uninstallHandler && isset($originalCallback[1])) {
				parent::detachEventHandler(static::SIGNAL_MAP[$signal], $originalCallback[1]);
			}
		}

		self::$_priorHandlers = [];
		self::$_pidHandlers = [];
		static::$_alarms = [];
		self::$_singleton = null;

		static::setAsyncSignals(self::$_priorAsync);
		self::$_priorAsync = null;
		self::$_asyncSignals = null;

		return true;
	}

	/**
	 * Determines whether an event is defined.
	 * The event name can be an integer Signal or a pid.  Pid handlers have a prefix
	 * of 'pid:'.
	 * An event is defined if the class has a method whose name is the event name
	 * prefixed with 'on', 'fx', or 'dy'.
	 * Every object responds to every 'fx' and 'dy' event as they are in a universally
	 * accepted event space.  'on' event must be declared by the object.
	 * When enabled, this will loop through all active behaviors for 'on' events
	 * defined by the behavior.
	 * Note, event name is case-insensitive.
	 * @param mixed $name the event name
	 * @return bool Is the event, signal event, or PID event available.
	 */
	public function hasEvent($name)
	{
		if (isset(static::SIGNAL_MAP[$name])) {
			$name = static::SIGNAL_MAP[$name];
		} elseif (strncasecmp('pid:', $name, 4) === 0) {
			if (is_numeric($pid = trim(substr($name, 4)))) {
				return TProcessHelper::isRunning((int) $pid);
			}
			return false;
		}
		return parent::hasEvent($name);
	}

	/**
	 * Checks if an event has any handlers.  This function also checks through all
	 * the behaviors for 'on' events when behaviors are enabled.
	 * The event name can be an integer Signal or a pid.  Pid handlers have a prefix
	 * of 'pid:'.
	 * 'dy' dynamic events are not handled by this function.
	 * @param string $name the event name
	 * @return bool whether an event has been attached one or several handlers
	 */
	public function hasEventHandler($name)
	{
		if (isset(static::SIGNAL_MAP[$name])) {
			$name = static::SIGNAL_MAP[$name];
		} elseif (strncasecmp('pid:', $name, 4) === 0) {
			if (is_numeric($pid = trim(substr($name, 4)))) {
				$pid = (int) $pid;
				return isset(self::$_pidHandlers[$pid]) && self::$_pidHandlers[$pid]->getCount() > 0;
			}
			return false;
		}
		return parent::hasEventHandler($name);
	}

	/**
	 * Returns the list of attached event handlers for an 'on' or 'fx' event.   This function also
	 * checks through all the behaviors for 'on' event lists when behaviors are enabled.
	 * The event name can be an integer Signal or a pid.  Pid handlers have a prefix
	 * of 'pid:'.
	 * @param mixed $name
	 * @throws TInvalidOperationException if the event is not defined or PID not a valid numeric.
	 * @return TWeakCallableCollection list of attached event handlers for an event
	 */
	public function getEventHandlers($name)
	{
		if (isset(static::SIGNAL_MAP[$name])) {
			$name = static::SIGNAL_MAP[$name];
		} elseif (strncasecmp('pid:', $name, 4) === 0) {
			if (!is_numeric($pid = trim(substr($name, 4)))) {
				throw new TInvalidOperationException('signalsdispatcher_bad_pid', $pid);
			}
			$pid = (int) $pid;
			if (!isset(self::$_pidHandlers[$pid]) && TProcessHelper::isRunning($pid)) {
				self::$_pidHandlers[$pid] = new TWeakCallableCollection();
			} elseif (isset(self::$_pidHandlers[$pid]) && !TProcessHelper::isRunning($pid)) {
				unset(self::$_pidHandlers[$pid]);
			}
			return self::$_pidHandlers[$pid] ?? null;
		}
		return parent::getEventHandlers($name);
	}

	/**
	 * @param int $pid The PID to check for handlers.
	 * @return bool Does the PID have handlers.
	 */
	public function hasPidHandler(int $pid): bool
	{
		return isset(self::$_pidHandlers[$pid]);
	}

	/**
	 * Returns the Handlers for a specific PID.
	 * @param int $pid The PID to get the handlers of.
	 * @param bool $validate Ensure that the PID is running before providing its handlers.
	 *   Default false.
	 * @return ?TWeakCallableCollection The handlers for a pid or null if validating
	 *   and the PID is not running.
	 */
	public function getPidHandlers(int $pid, bool $validate = false)
	{
		if ($validate && !TProcessHelper::isRunning($pid)) {
			return null;
		}
		if (!isset(self::$_pidHandlers[$pid])) {
			self::$_pidHandlers[$pid] = new TWeakCallableCollection();
		}
		return self::$_pidHandlers[$pid];
	}

	/**
	 * Attaches a handler to a child PID at a priority.  Optionally validates the process
	 * before attaching.
	 * @param int $pid The PID to install the handler.
	 * @param mixed $handler The handler to attach to the process.
	 * @param null|numeric $priority The priority of the handler, default null for the
	 *   default
	 * @param bool $validate Should the PID be validated before attaching.
	 * @return bool Is the handler attached?  this can only be false if $validate = true
	 *   and the PID is not running any more.
	 */
	public function attachPidHandler(int $pid, mixed $handler, mixed $priority = null, bool $validate = false)
	{
		if ($validate && !TProcessHelper::isRunning($pid)) {
			return false;
		}
		if (!isset(self::$_pidHandlers[$pid])) {
			self::$_pidHandlers[$pid] = new TWeakCallableCollection();
		}
		self::$_pidHandlers[$pid]->add($handler, $priority);
		return true;
	}

	/**
	 * Detaches a handler from a child PID at a priority.
	 * @param int $pid The PID to detach the handler from.
	 * @param mixed $handler The handler to remove.
	 * @param mixed $priority The priority of the handler to remove. default false for
	 *   any priority.
	 */
	public function detachPidHandler(int $pid, mixed $handler, mixed $priority = false)
	{
		if (isset(self::$_pidHandlers[$pid])) {
			try {
				self::$_pidHandlers[$pid]->remove($handler, $priority);
			} catch (\Exception $e) {
			}
			if (self::$_pidHandlers[$pid]->getCount() === 0) {
				$this->clearPidHandlers($pid);
			}
			return true;
		}
		return false;
	}

	/**
	 * Clears the Handlers for a specific PID.
	 * @param int $pid The pid to clear the handlers.
	 * @return bool Were there any handlers for the PID that were cleared.
	 */
	public function clearPidHandlers(int $pid): bool
	{
		$return = isset(self::$_pidHandlers[$pid]);
		unset(self::$_pidHandlers[$pid]);
		return $return;
	}

	/**
	 * The common SIGCHLD callback to delegate per PID.  If there are specific PID
	 * handlers for a child, those PID callbacks are called.  On an exit event, the PID
	 * handlers are cleared.
	 * @param TSignalsDispatcher $sender The object raising the event.
	 * @param TSignalParameter $param The signal parameters.
	 */
	public function delegateChild($sender, $param)
	{
		if (!static::hasSignals()) {
			return;
		}

		if (!$param || ($pid = $param->getParameterPid()) === null) {
			if (($pid = pcntl_waitpid(-1, $status, WNOHANG)) < 1) {
				return;
			}
			if (!$param) {
				$param = new TSignalParameter(SIGCHLD);
			}
			$sigInfo = $param->getParameter() ?? [];
			$sigInfo['pid'] = $pid;
			$sigInfo['status'] = $status;
			$param->setParameter($sigInfo);
		}
		if (!isset(self::$_pidHandlers[$pid])) {
			return;
		}

		array_map(fn ($child) => $child($sender, $param), self::$_pidHandlers[$pid]->toArray());

		if (in_array($param->getParameterCode(), [1, 2])) { // 1 = normal exit, 2 = kill
			unset(self::$_pidHandlers[$pid]);
		}
	}

	/**
	 * {@inheritDoc}
	 *
	 * Raises signal events by converting the $name as a Signal to the PRADO event
	 * for the signal.
	 * @param mixed $name The event name or linux signal to raise.
	 * @param mixed $sender The sender raising the event.
	 * @param mixed $param The parameter for the event.
	 * @param null|mixed $responsetype The response type.
	 * @param null|mixed $postfunction The results post filter.
	 */
	public function raiseEvent($name, $sender, $param, $responsetype = null, $postfunction = null)
	{
		if (isset(static::SIGNAL_MAP[$name])) {
			$name = static::SIGNAL_MAP[$name];
		}
		return parent::raiseEvent($name, $sender, $param, $responsetype, $postfunction);
	}

	/**
	 * This is called when receiving a system process signal.  The global event
	 * for the signal is raised when the signal is received.
	 * @param int $signal The signal being sent.
	 * @param null|mixed $signalInfo The signal information.
	 * @throws TExitException When the signal needs to exit the application.
	 */
	public function __invoke(int $signal, mixed $signalInfo = null)
	{
		if (!isset(static::SIGNAL_MAP[$signal])) {
			return;
		}

		$parameter = new TSignalParameter($signal, isset(self::EXIT_SIGNALS[$signal]), 128 + $signal, $signalInfo);

		parent::raiseEvent(static::SIGNAL_MAP[$signal], $this, $parameter);

		if ($parameter->getIsExiting()) {
			throw new TExitException($parameter->getExitCode());
		}
	}

	/**
	 * Creates a new alarm callback at a specific time from now.  If no callback is
	 * provided, then the alarm will raise `fxSignalAlarm` without a time-based callback.
	 * When calling alarm() without parameters, it will return the next alarm time.
	 * @param int $seconds Seconds from now to trigger the alarm. Default 0 for returning
	 *   the next alarm time and not adding any callback.
	 * @param mixed $callback The alarm callback. Default null.
	 * @return ?int The time of the alarm for the callback or the next alarm time when
	 *   $seconds = 0.
	 */
	public static function alarm(int $seconds = 0, mixed $callback = null): ?int
	{
		if (!static::hasSignals()) {
			return null;
		}

		if ($seconds > 0) {
			static::singleton();
			$alarmTime = time() + $seconds;
			if ($callback === null) {
				$callback = static::NULL_ALARM;
			}
			if (!isset(static::$_alarms[$alarmTime])) {
				static::$_alarmsOrdered = false;
			}
			static::$_alarms[$alarmTime][] = $callback;

			if (self::$_nextAlarmTime === null || $alarmTime < self::$_nextAlarmTime) {
				self::$_nextAlarmTime = $alarmTime;
				pcntl_alarm($seconds);
			}
			return $alarmTime;
		} elseif ($callback === null) {
			return self::$_nextAlarmTime;
		}

		return null;
	}

	/**
	 * Disarms an alarm time-callback, at the optional time.
	 * @param ?int $alarmTime
	 * @param callable $callback
	 */
	public static function disarm(mixed $alarmTime = null, mixed $callback = null): ?int
	{
		if (!static::hasSignals()) {
			return null;
		}

		if ($alarmTime !== null && !is_int($alarmTime)) {
			$tmp = $callback;
			$callback = $alarmTime;
			$alarmTime = $tmp;
		}

		// If alarmTime but has no handlers for the time.
		if ($alarmTime !== null && !isset(static::$_alarms[$alarmTime])) {
			return null;
		}

		if ($callback === null) {
			$callback = static::NULL_ALARM;
		}

		if (!static::$_alarmsOrdered) {
			ksort(static::$_alarms, SORT_NUMERIC);
			static::$_alarmsOrdered = true;
		}
		if ($callback === self::$_priorHandlers[SIGALRM][0]) {
			$callback = self::$_priorHandlers[SIGALRM][1];
		}

		foreach ($alarmTime !== null ? [$alarmTime] : array_keys(static::$_alarms) as $time) {
			if (($key = array_search($callback, static::$_alarms[$time] ?? [], true)) !== false) {
				unset(static::$_alarms[$time][$key]);
				if (is_array(static::$_alarms[$time] ?? false)) {
					unset(static::$_alarms[$time]);
					if ($time === self::$_nextAlarmTime) {
						self::$_nextAlarmTime = array_key_first(static::$_alarms);
						if (self::$_nextAlarmTime !== null) {
							$seconds = min(1, self::$_nextAlarmTime - time());
						} else {
							$seconds = 0;
						}
						pcntl_alarm($seconds);
					}
				}
				return $time;
			}
		}
		return null;
	}

	/**
	 * The common SIGALRM callback time-processing handler raised by `fxSignalAlarm`.
	 * All alarm callbacks before or at `time()` are called.  The next alarm time is
	 * found and the next signal alarm is set.
	 * @param TSignalsDispatcher $sender The object raising the event.
	 * @param TSignalParameter $signalParam The signal parameters.
	 */
	public function ring($sender, $signalParam)
	{
		if (!static::hasSignals()) {
			return null;
		}
		if (!static::$_alarmsOrdered) {
			ksort(static::$_alarms, SORT_NUMERIC);
			static::$_alarmsOrdered = true;
		}
		do {
			$nextTime = null;
			$startTime = time();
			$signalParam->setAlarmTime($startTime);
			foreach (static::$_alarms as $alarmTime => $alarms) {
				if ($alarmTime <= $startTime) {
					array_map(fn ($alarm) => $alarm($this, $signalParam), static::$_alarms[$alarmTime]);
					unset(static::$_alarms[$alarmTime]);
				} elseif ($nextTime === null) {
					$nextTime = $alarmTime;
					break;
				}
			}
			$now = time();
		} while ($startTime !== $now);

		if ($nextTime !== null) {
			pcntl_alarm($nextTime - $now);
		}
		self::$_nextAlarmTime = $nextTime;
	}

	/**
	 * The null alarm to simply trigger an alarm without callback
	 * @param TSignalsDispatcher $sender The object raising the event.
	 * @param TSignalParameter $param The signal parameters.
	 */
	public static function nullAlarm($sender, $param)
	{
	}

	/**
	 * When PHP Signals are not in asynchronous mode, this must be called to dispatch
	 * the pending events.  To change the async mode, use {@see self::setAsyncSignals()}.
	 * @return ?bool Returns true on success or false on failure.
	 */
	public static function syncDispatch(): ?bool
	{
		if (!static::hasSignals()) {
			return null;
		}
		return pcntl_signal_dispatch();
	}

	/**
	 * Gets whether the system is in async signals mode.
	 * @return ?bool Is the system set to handle async signals.  null when there are
	 *  no Process Signals in the PHP instance.
	 */
	public static function getAsyncSignals(): ?bool
	{
		if (!static::hasSignals()) {
			return null;
		}
		return pcntl_async_signals();
	}

	/**
	 * Sets whether the system is in async signals mode.  This is set to true on instancing.
	 * If this is set to false, then {@see self::syncDispatch()} must be called for
	 * signals to be processed.  Any pending signals are dispatched when setting async to true.
	 * @param bool $value Should signals be processed asynchronously.
	 * @return ?bool The prior value AsyncSignals before setting.
	 * @link https://www.php.net/manual/en/function.pcntl-signal-dispatch.php
	 */
	public static function setAsyncSignals(bool $value): ?bool
	{
		if (!static::hasSignals()) {
			return null;
		}
		self::$_asyncSignals = $value;

		$return = pcntl_async_signals($value);

		if (self::$_priorAsync === null) {
			self::$_priorAsync = $return;
		}

		if ($value === true && $return === false) {
			pcntl_signal_dispatch();
		}

		return $return;
	}

	/**
	 * This returns the priority of the signal handlers when they are installed as
	 * event handlers.
	 * @return ?float The priority for prior signal handlers when TSignalsDispatcher
	 *   is attached.
	 */
	public static function getPriorHandlerPriority(): ?float
	{
		return self::$_priorHandlerPriority;
	}

	/**
	 * Sets the priority of the signal handlers when they are installed as
	 * event handlers.
	 * @param ?float $value The priority for prior signal handlers when TSignalsDispatcher
	 *   is attached.
	 * @throws TInvalidOperationException When TSignalsDispatcher is already installed.
	 */
	public static function setPriorHandlerPriority(?float $value): bool
	{
		if (static::singleton(false)) {
			throw new TInvalidOperationException('signalsdispatcher_no_change', 'PriorHandlerPriority');
		}

		self::$_priorHandlerPriority = $value;
		return true;
	}

	/**
	 * This gets the signal handlers that were installed prior to the TSignalsDispatcher
	 * being attached.
	 * @param int $signal The signal to get the prior handler value.
	 * @param bool $original Return the original handler, default false for the signal
	 *   closure handler that wraps the original handler with a PRADO signal event handler.
	 */
	public static function getPriorHandler(int $signal, bool $original = false): mixed
	{
		return self::$_priorHandlers[$signal][$original ? 0 : 1] ?? null;
	}
}
