<?php

/**
 * TActiveRecordManager class file.
 *
 * @author Wei Zhuo <weizhuo[at]gmail[dot]com>
 * @link https://github.com/pradosoft/prado
 * @license https://github.com/pradosoft/prado/blob/master/LICENSE
 */

namespace Prado\Data\ActiveRecord;

use Prado\Prado;
use Prado\Caching\ICache;
use Prado\TPropertyValue;

/**
 * TActiveRecordManager provides the default DB connection,
 * default active record gateway, and table meta data inspector.
 *
 * The default connection can be set as follows:
 * ```php
 * TActiveRecordManager::getInstance()->setDbConnection($conn);
 * ```
 * All new active record created after setting the
 * {@see DbConnection setDbConnection()} will use that connection unless
 * the custom ActiveRecord class overrides the ActiveRecord::getDbConnection().
 *
 * Set the {@see setCache Cache} property to an ICache object to allow
 * the active record gateway to cache the table meta data information.
 *
 * @author Wei Zhuo <weizho[at]gmail[dot]com>
 * @since 3.1
 */
class TActiveRecordManager extends \Prado\TComponent
{
	public const DEFAULT_GATEWAY_CLASS = \Prado\Data\ActiveRecord\TActiveRecordGateway::class;

	/**
	 * Defaults to {@see \Prado\Data\ActiveRecord\TActiveRecordManager::DEFAULT_GATEWAY_CLASS DEFAULT_GATEWAY_CLASS}
	 * @var string
	 */
	private $_gatewayClass = self::DEFAULT_GATEWAY_CLASS;

	private $_gateway;
	private $_meta = [];
	private $_connection;

	private $_cache;

	/**
	 * Defaults to '{@see \Prado\Data\ActiveRecord\TActiveRecordInvalidFinderResult::Null Null}'
	 *
	 * @var TActiveRecordInvalidFinderResult
	 * @since 3.1.5
	 */
	private $_invalidFinderResult = TActiveRecordInvalidFinderResult::Null;

	/**
	 * @return ICache application cache.
	 */
	public function getCache()
	{
		return $this->_cache;
	}

	/**
	 * @param ICache $value application cache
	 */
	public function setCache($value)
	{
		$this->_cache = $value;
	}

	/**
	 * @param \Prado\Data\TDbConnection $conn default database connection
	 */
	public function setDbConnection($conn)
	{
		$this->_connection = $conn;
	}

	/**
	 * @return \Prado\Data\TDbConnection default database connection
	 */
	public function getDbConnection()
	{
		return $this->_connection;
	}

	/**
	 * @param null|mixed $self
	 * @return TActiveRecordManager static instance of record manager.
	 */
	public static function getInstance($self = null)
	{
		static $instance;
		if ($self !== null) {
			$instance = $self;
		} elseif ($instance === null) {
			$instance = new self();
		}
		return $instance;
	}

	/**
	 * @return TActiveRecordGateway record gateway.
	 */
	public function getRecordGateway()
	{
		if ($this->_gateway === null) {
			$this->_gateway = $this->createRecordGateway();
		}
		return $this->_gateway;
	}

	/**
	 * @return TActiveRecordGateway default record gateway.
	 */
	protected function createRecordGateway()
	{
		return Prado::createComponent($this->getGatewayClass(), $this);
	}

	/**
	 * Set implementation class of ActiveRecordGateway
	 * @param string $value
	 */
	public function setGatewayClass($value)
	{
		$this->_gateway = null;
		$this->_gatewayClass = (string) $value;
	}

	/**
	 * @return string the implementation class of ActiveRecordGateway. Defaults to {@see \Prado\Data\ActiveRecord\TActiveRecordManager::DEFAULT_GATEWAY_CLASS DEFAULT_GATEWAY_CLASS}
	 */
	public function getGatewayClass()
	{
		return $this->_gatewayClass;
	}

	/**
	 * @return TActiveRecordInvalidFinderResult Defaults to '{@see \Prado\Data\ActiveRecord\TActiveRecordInvalidFinderResult::Null Null}'.
	 * @since 3.1.5
	 * @see setInvalidFinderResult
	 */
	public function getInvalidFinderResult()
	{
		return $this->_invalidFinderResult;
	}

	/**
	 * Define the way an active record finder react if an invalid magic-finder invoked
	 * @param TActiveRecordInvalidFinderResult $value * @since 3.1.5
	 * @see getInvalidFinderResult
	 */
	public function setInvalidFinderResult($value)
	{
		$this->_invalidFinderResult = TPropertyValue::ensureEnum($value, \Prado\Data\ActiveRecord\TActiveRecordInvalidFinderResult::class);
	}
}
