<?php

/**
 * TImageButton class file
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @link https://github.com/pradosoft/prado
 * @license https://github.com/pradosoft/prado/blob/master/LICENSE
 */

namespace Prado\Web\UI\WebControls;

use Prado\TPropertyValue;
use Prado\Web\UI\IPostBackEventHandler;

/**
 * TImageButton class
 *
 * TImageButton creates an image button on the page. It is used to submit data to a page.
 * You can create either a <b>submit</b> button or a <b>command</b> button.
 *
 * A <b>command</b> button has a command name (specified by
 * the {@see setCommandName CommandName} property) and and a command parameter
 * (specified by {@see setCommandParameter CommandParameter} property)
 * associated with the button. This allows you to create multiple TLinkButton
 * components on a Web page and programmatically determine which one is clicked
 * with what parameter. You can provide an event handler for
 * {@see onCommand OnCommand} event to programmatically control the actions performed
 * when the command button is clicked. In the event handler, you can determine
 * the {@see setCommandName CommandName} property value and
 * the {@see setCommandParameter CommandParameter} property value
 * through the {@see TCommandParameter::getName Name} and
 * {@see TCommandParameter::getParameter Parameter} properties of the event
 * parameter which is of type {@see \Prado\Web\UI\TCommandEventParameter}.
 *
 * A <b>submit</b> button does not have a command name associated with the button
 * and clicking on it simply posts the Web page back to the server.
 * By default, a TImageButton control is a submit button.
 * You can provide an event handler for the {@see onClick OnClick} event
 * to programmatically control the actions performed when the submit button is clicked.
 * The coordinates of the clicking point can be obtained from the {@see onClick OnClick}
 * event parameter, which is of type {@see \Prado\Web\UI\WebControls\TImageClickEventParameter}.
 *
 * Clicking on button can trigger form validation, if
 * {@see setCausesValidation CausesValidation} is true.
 * And the validation may be restricted within a certain group of validator
 * controls by setting {@see setValidationGroup ValidationGroup} property.
 * If validation is successful, the data will be post back to the same page.
 *
 * TImageButton displays the {@see setText Text} property as the hint text to the displayed image.
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @since 3.0
 */
class TImageButton extends TImage implements \Prado\Web\UI\IPostBackDataHandler, IPostBackEventHandler, \Prado\Web\UI\IButtonControl
{
	/**
	 * @var int x coordinate that the image is being clicked at
	 */
	private $_x = 0;
	/**
	 * @var int y coordinate that the image is being clicked at
	 */
	private $_y = 0;
	private $_dataChanged = false;

	/**
	 * @return string tag name of the button
	 */
	protected function getTagName()
	{
		return 'input';
	}

	/**
	 * @return bool whether to render javascript.
	 */
	public function getEnableClientScript()
	{
		return $this->getViewState('EnableClientScript', true);
	}

	/**
	 * @param bool $value whether to render javascript.
	 */
	public function setEnableClientScript($value)
	{
		$this->setViewState('EnableClientScript', TPropertyValue::ensureBoolean($value), true);
	}

	/**
	 * Adds attribute name-value pairs to renderer.
	 * This overrides the parent implementation with additional button specific attributes.
	 * @param \Prado\Web\UI\THtmlWriter $writer the writer used for the rendering purpose
	 */
	protected function addAttributesToRender($writer)
	{
		$page = $this->getPage();
		$page->ensureRenderInForm($this);
		$writer->addAttribute('type', 'image');
		if (($uniqueID = $this->getUniqueID()) !== '') {
			$writer->addAttribute('name', $uniqueID);
		}
		if ($this->getEnabled(true)) {
			if ($this->getEnableClientScript() && $this->needPostBackScript()) {
				$this->renderClientControlScript($writer);
			}
		} elseif ($this->getEnabled()) { // in this case, parent will not render 'disabled'
			$writer->addAttribute('disabled', 'disabled');
		}
		parent::addAttributesToRender($writer);
	}

	/**
	 * Renders the client-script code.
	 * @param mixed $writer
	 */
	protected function renderClientControlScript($writer)
	{
		$writer->addAttribute('id', $this->getClientID());
		$cs = $this->getPage()->getClientScript();
		$cs->registerPostBackControl($this->getClientClassName(), $this->getPostBackOptions());
	}

	/**
	 * Gets the name of the javascript class responsible for performing postback for this control.
	 * This method overrides the parent implementation.
	 * @return string the javascript class name
	 */
	protected function getClientClassName()
	{
		return 'Prado.WebUI.TImageButton';
	}

	/**
	 * @return bool whether to perform validation if the button is clicked
	 */
	protected function canCauseValidation()
	{
		if ($this->getCausesValidation()) {
			$group = $this->getValidationGroup();
			return $this->getPage()->getValidators($group)->getCount() > 0;
		} else {
			return false;
		}
	}

	/**
	 * @param bool $value set by a panel to register this button as the default button for the panel.
	 */
	public function setIsDefaultButton($value)
	{
		$this->setViewState('IsDefaultButton', TPropertyValue::ensureBoolean($value), false);
	}

	/**
	 * @return bool true if this button is registered as a default button for a panel.
	 */
	public function getIsDefaultButton()
	{
		return $this->getViewState('IsDefaultButton', false);
	}

	/**
	 * @return bool whether the button needs javascript to do postback
	 */
	protected function needPostBackScript()
	{
		return $this->canCauseValidation() || $this->getIsDefaultButton();
	}

	/**
	 * Returns postback specifications for the button.
	 * This method is used by framework and control developers.
	 * @return array parameters about how the button defines its postback behavior.
	 */
	protected function getPostBackOptions()
	{
		$options['ID'] = $this->getClientID();
		$options['CausesValidation'] = $this->getCausesValidation();
		$options['ValidationGroup'] = $this->getValidationGroup();
		$options['EventTarget'] = $this->getUniqueID();

		return $options;
	}

	/**
	 * This method checks if the TImageButton is clicked and loads the coordinates of the clicking position.
	 * This method is primarly used by framework developers.
	 * @param string $key the key that can be used to retrieve data from the input data collection
	 * @param array $values the input data collection
	 * @return bool whether the data of the component has been changed
	 */
	public function loadPostData($key, $values)
	{
		$uid = $this->getUniqueID();
		if (isset($values["{$uid}_x"]) && isset($values["{$uid}_y"])) {
			$this->_x = (int) ($values["{$uid}_x"]);
			$this->_y = (int) ($values["{$uid}_y"]);
			if ($this->getPage()->getPostBackEventTarget() === null) {
				$this->getPage()->setPostBackEventTarget($this);
			}
			$this->_dataChanged = true;
		}
		return false;
	}

	/**
	 * A dummy implementation for the \Prado\Web\UI\IPostBackDataHandler interface.
	 */
	public function raisePostDataChangedEvent()
	{
		// no post data to handle
	}

	/**
	 * This method is invoked when the button is clicked.
	 * The method raises 'OnClick' event to fire up the event handlers.
	 * If you override this method, be sure to call the parent implementation
	 * so that the event handler can be invoked.
	 * @param TImageClickEventParameter $param event parameter to be passed to the event handlers
	 */
	public function onClick($param)
	{
		$this->raiseEvent('OnClick', $this, $param);
	}

	/**
	 * This method is invoked when the button is clicked.
	 * The method raises 'OnCommand' event to fire up the event handlers.
	 * If you override this method, be sure to call the parent implementation
	 * so that the event handlers can be invoked.
	 * @param \Prado\Web\UI\TCommandEventParameter $param event parameter to be passed to the event handlers
	 */
	public function onCommand($param)
	{
		$this->raiseEvent('OnCommand', $this, $param);
		$this->raiseBubbleEvent($this, $param);
	}

	/**
	 * Raises the postback event.
	 * This method is required by {@see \Prado\Web\UI\IPostBackEventHandler} interface.
	 * If {@see getCausesValidation CausesValidation} is true, it will
	 * invoke the page's {@see \Prado\Web\UI\TPage::validate validate} method first.
	 * It will raise {@see onClick OnClick} and {@see onCommand OnCommand} events.
	 * This method is mainly used by framework and control developers.
	 * @param \Prado\TEventParameter $param the event parameter
	 */
	public function raisePostBackEvent($param)
	{
		if ($this->getCausesValidation()) {
			$this->getPage()->validate($this->getValidationGroup());
		}
		$this->onClick(new TImageClickEventParameter($this->_x, $this->_y));
		$this->onCommand(new \Prado\Web\UI\TCommandEventParameter($this->getCommandName(), $this->getCommandParameter()));
	}

	/**
	 * Returns a value indicating whether postback has caused the control data change.
	 * This method is required by the \Prado\Web\UI\IPostBackDataHandler interface.
	 * @return bool whether postback has caused the control data change. False if the page is not in postback mode.
	 */
	public function getDataChanged()
	{
		return $this->_dataChanged;
	}

	/**
	 * @return bool whether postback event trigger by this button will cause input validation, default is true
	 */
	public function getCausesValidation()
	{
		return $this->getViewState('CausesValidation', true);
	}

	/**
	 * @param bool $value whether postback event trigger by this button will cause input validation
	 */
	public function setCausesValidation($value)
	{
		$this->setViewState('CausesValidation', TPropertyValue::ensureBoolean($value), true);
	}

	/**
	 * @return string the command name associated with the {@see onCommand OnCommand} event.
	 */
	public function getCommandName()
	{
		return $this->getViewState('CommandName', '');
	}

	/**
	 * @param string $value the command name associated with the {@see onCommand OnCommand} event.
	 */
	public function setCommandName($value)
	{
		$this->setViewState('CommandName', $value, '');
	}

	/**
	 * @return string the parameter associated with the {@see onCommand OnCommand} event
	 */
	public function getCommandParameter()
	{
		return $this->getViewState('CommandParameter', '');
	}

	/**
	 * @param string $value the parameter associated with the {@see onCommand OnCommand} event.
	 */
	public function setCommandParameter($value)
	{
		$this->setViewState('CommandParameter', $value, '');
	}

	/**
	 * @return string the group of validators which the button causes validation upon postback
	 */
	public function getValidationGroup()
	{
		return $this->getViewState('ValidationGroup', '');
	}

	/**
	 * @param string $value the group of validators which the button causes validation upon postback
	 */
	public function setValidationGroup($value)
	{
		$this->setViewState('ValidationGroup', $value, '');
	}

	/**
	 * @return string caption of the button
	 */
	public function getText()
	{
		return $this->getAlternateText();
	}

	/**
	 * @param string $value caption of the button
	 */
	public function setText($value)
	{
		$this->setAlternateText($value);
	}

	/**
	 * Registers the image button to receive postback data during postback.
	 * This is necessary because an image button, when postback, does not have
	 * direct mapping between post data and the image button name.
	 * This method overrides the parent implementation and is invoked before render.
	 * @param mixed $param event parameter
	 */
	public function onPreRender($param)
	{
		parent::onPreRender($param);
		$this->getPage()->registerRequiresPostData($this);
	}

	/**
	 * Renders the body content enclosed between the control tag.
	 * This overrides the parent implementation with nothing to be rendered.
	 * @param \Prado\Web\UI\THtmlWriter $writer the writer used for the rendering purpose
	 */
	public function renderContents($writer)
	{
	}
}
