<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace PrestaShopBundle\Controller\Admin\Configure\ShopParameters;

use Exception;
use PrestaShop\PrestaShop\Core\Domain\SearchEngine\Command\BulkDeleteSearchEngineCommand;
use PrestaShop\PrestaShop\Core\Domain\SearchEngine\Command\DeleteSearchEngineCommand;
use PrestaShop\PrestaShop\Core\Domain\SearchEngine\Exception\DeleteSearchEngineException;
use PrestaShop\PrestaShop\Core\Domain\SearchEngine\Exception\SearchEngineException;
use PrestaShop\PrestaShop\Core\Domain\SearchEngine\Exception\SearchEngineNotFoundException;
use PrestaShop\PrestaShop\Core\Domain\SearchEngine\Query\GetSearchEngineForEditing;
use PrestaShop\PrestaShop\Core\Domain\SearchEngine\QueryResult\SearchEngineForEditing;
use PrestaShop\PrestaShop\Core\Form\IdentifiableObject\Builder\FormBuilderInterface;
use PrestaShop\PrestaShop\Core\Form\IdentifiableObject\Handler\FormHandlerInterface;
use PrestaShop\PrestaShop\Core\Grid\GridFactoryInterface;
use PrestaShop\PrestaShop\Core\Search\Filters\SearchEngineFilters;
use PrestaShopBundle\Controller\Admin\PrestaShopAdminController;
use PrestaShopBundle\Security\Attribute\AdminSecurity;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Responsible for handling "Configure > Shop Parameters > Traffic & SEO > Search Engines" page.
 */
class SearchEnginesController extends PrestaShopAdminController
{
    #[AdminSecurity("is_granted('read', request.get('_legacy_controller'))")]
    public function indexAction(
        Request $request,
        SearchEngineFilters $filters,
        #[Autowire(service: 'prestashop.core.grid.factory.search_engines')]
        GridFactoryInterface $searchEngineGridFactory,
    ): Response {
        $searchEnginesGrid = $searchEngineGridFactory->getGrid($filters);

        return $this->render('@PrestaShop/Admin/Configure/ShopParameters/TrafficSeo/SearchEngines/index.html.twig', [
            'searchEnginesGrid' => $this->presentGrid($searchEnginesGrid),
            'enableSidebar' => true,
            'help_link' => $this->generateSidebarLink($request->attributes->get('_legacy_controller')),
        ]);
    }

    #[AdminSecurity("is_granted('create', request.get('_legacy_controller'))")]
    public function createAction(
        Request $request,
        #[Autowire(service: 'prestashop.core.form.identifiable_object.builder.search_engine_form_builder')]
        FormBuilderInterface $searchEngineFormBuilder,
        #[Autowire(service: 'prestashop.core.form.identifiable_object.handler.search_engine_form_handler')]
        FormHandlerInterface $searchEngineFormHandler,
    ): Response {
        $searchEngineForm = $searchEngineFormBuilder->getForm();
        $searchEngineForm->handleRequest($request);

        try {
            $result = $searchEngineFormHandler->handle($searchEngineForm);

            if (null !== $result->getIdentifiableObjectId()) {
                $this->addFlash('success', $this->trans('Successful creation', [], 'Admin.Notifications.Success'));

                return $this->redirectToRoute('admin_search_engines_index');
            }
        } catch (Exception $e) {
            $this->addFlash('error', $this->getErrorMessageForException($e, $this->getErrorMessages()));
        }

        return $this->render('@PrestaShop/Admin/Configure/ShopParameters/TrafficSeo/SearchEngines/create.html.twig', [
            'searchEngineForm' => $searchEngineForm->createView(),
            'help_link' => $this->generateSidebarLink($request->attributes->get('_legacy_controller')),
            'enableSidebar' => true,
            'multistoreInfoTip' => $this->trans(
                'Note that this feature is only available in the "all stores" context. It will be added to all your stores.',
                [],
                'Admin.Notifications.Info'
            ),
            'multistoreIsUsed' => $this->getShopContext()->isMultiShopUsed(),
        ]);
    }

    #[AdminSecurity("is_granted('update', request.get('_legacy_controller'))")]
    public function editAction(
        int $searchEngineId,
        Request $request,
        #[Autowire(service: 'prestashop.core.form.identifiable_object.builder.search_engine_form_builder')]
        FormBuilderInterface $searchEngineFormBuilder,
        #[Autowire(service: 'prestashop.core.form.identifiable_object.handler.search_engine_form_handler')]
        FormHandlerInterface $searchEngineFormHandler,
    ): Response {
        try {
            $searchEngineForm = $searchEngineFormBuilder->getFormFor($searchEngineId);
        } catch (Exception $e) {
            $this->addFlash('error', $this->getErrorMessageForException($e, $this->getErrorMessages()));

            return $this->redirectToRoute('admin_search_engines_index');
        }

        try {
            $searchEngineForm->handleRequest($request);
            $result = $searchEngineFormHandler->handleFor($searchEngineId, $searchEngineForm);

            if ($result->isSubmitted() && $result->isValid()) {
                $this->addFlash('success', $this->trans('Successful update', [], 'Admin.Notifications.Success'));

                return $this->redirectToRoute('admin_search_engines_index');
            }
        } catch (Exception $e) {
            $this->addFlash('error', $this->getErrorMessageForException($e, $this->getErrorMessages()));

            if ($e instanceof SearchEngineNotFoundException) {
                return $this->redirectToRoute('admin_search_engines_index');
            }
        }

        /** @var SearchEngineForEditing $editableSearchEngine */
        $editableSearchEngine = $this->dispatchQuery(new GetSearchEngineForEditing($searchEngineId));

        return $this->render('@PrestaShop/Admin/Configure/ShopParameters/TrafficSeo/SearchEngines/edit.html.twig', [
            'searchEngineForm' => $searchEngineForm->createView(),
            'searchEngineServer' => $editableSearchEngine->getServer(),
            'help_link' => $this->generateSidebarLink($request->attributes->get('_legacy_controller')),
            'enableSidebar' => true,
            'layoutTitle' => $this->trans(
                'Editing search engine %name%',
                [
                    '%name%' => $editableSearchEngine->getServer(),
                ],
                'Admin.Navigation.Menu',
            ),
        ]);
    }

    #[AdminSecurity("is_granted('delete', request.get('_legacy_controller'))", redirectRoute: 'admin_search_engines_index')]
    public function deleteAction(int $searchEngineId): RedirectResponse
    {
        try {
            $this->dispatchCommand(new DeleteSearchEngineCommand($searchEngineId));

            $this->addFlash('success', $this->trans('Successful deletion', [], 'Admin.Notifications.Success'));
        } catch (SearchEngineException $e) {
            $this->addFlash('error', $this->getErrorMessageForException($e, $this->getErrorMessages()));
        }

        return $this->redirectToRoute('admin_search_engines_index');
    }

    #[AdminSecurity("is_granted('delete', request.get('_legacy_controller'))", redirectRoute: 'admin_search_engines_index')]
    public function bulkDeleteAction(Request $request): RedirectResponse
    {
        $searchEngineIds = $this->getBulkSearchEnginesFromRequest($request);

        try {
            $this->dispatchCommand(new BulkDeleteSearchEngineCommand($searchEngineIds));

            $this->addFlash(
                'success',
                $this->trans('The selection has been successfully deleted.', [], 'Admin.Notifications.Success')
            );
        } catch (SearchEngineException $e) {
            $this->addFlash('error', $this->getErrorMessageForException($e, $this->getErrorMessages()));
        }

        return $this->redirectToRoute('admin_search_engines_index');
    }

    /**
     * Gets error messages for exceptions.
     *
     * @return array
     */
    private function getErrorMessages(): array
    {
        return [
            SearchEngineNotFoundException::class => $this->trans(
                'The object cannot be loaded (or found).',
                [],
                'Admin.Notifications.Error'
            ),
            DeleteSearchEngineException::class => [
                DeleteSearchEngineException::FAILED_DELETE => $this->trans(
                    'An error occurred while deleting the object.',
                    [],
                    'Admin.Notifications.Error'
                ),
                DeleteSearchEngineException::FAILED_BULK_DELETE => $this->trans(
                    'An error occurred while deleting this selection.',
                    [],
                    'Admin.Notifications.Error'
                ),
            ],
        ];
    }

    /**
     * Get search engine IDs from request for bulk actions.
     *
     * @param Request $request
     *
     * @return int[]
     */
    private function getBulkSearchEnginesFromRequest(Request $request): array
    {
        $searchEngineIds = $request->request->all('search_engine_bulk');

        return array_map('intval', $searchEngineIds);
    }
}
