<?php
namespace App;
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
// 
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
/**
 * Blog class, extended from HTML
 */

define ('APP_BLOG_URL_MODE_WITH_DATES', 3);
    
class Blog extends \SCHLIX\cmsApplication_ManyToMany {

    protected $data_directories = array(
      'image_original' => '/media/articles/original', // original      
      'image_thumbnail' => '/media/articles/thumbnail', // for thumbnail
      'image_small' => '/media/articles/small', // for thumbnail
      'image_medium' => '/media/articles/medium', // for article view
      'image_large' => '/media/articles/large', // for full screen view
    );

    protected $default_sizes = [
        'thumbnail' => ['width' => 256, 'height' => 256, 'quality' => 90],
        'small' => ['width' => 512, 'height' => 512, 'quality' => 80],
        'medium' => ['width' => 1024, 'height' => 1024, 'quality' => 70],
        'large' => ['width' => 1920, 'height' => 1920, 'quality' => 60],
    ];
    
    public function __construct() {

        parent::__construct(___('Blog'), 'gk_blog_items','gk_blog_categories','gk_blog_categories_items','is_primary');

        $this->schema_org_type_item = 'BlogPosting';
        $this->url_mode = CMS_APPLICATION_MANY_TO_MANY_URL_WITH_CATEGORY;
        $this->has_versioning = true;  
        $this->has_custom_media_header = true;
    }
    
    /**
     * Returns the URL of an image
     * @param string $image_size
     * @param string $path
     * @param bool $with_hash
     */
    public function getBlogImage($image_size, $path, $with_hash = false)
    {
         if (str_starts_with($path, 'https://') || str_starts_with($path, 'http://')  )
         {
             if (str_contains($path, '{$width}') || str_contains($path, '{$height}'))
             {
                $sz = remove_prefix_from_string($image_size, 'image_');
                $ds = $this->default_sizes[$sz];
                $img_width = $this->getConfig("int_{$sz}_width", $ds['width']);
                $img_height = $this->getConfig("int_{$sz}_height", $ds['height']);
                $path = str_replace(['{$width}', '{$height}'], [strval($img_width), strval($img_height)], $path);
             }              
            return $path;
             
         } elseif (!empty($path) && file_exists($this->getDataFileFullPath($image_size, $path)))
         {
             return $with_hash ? $this->getDataFileURLPathWithHash($image_size, $path) :  $this->getDataFileURLPath($image_size, $path) ;
         }
         return null;
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Loads the item and display it with view.item.template.php
     * @param int $id
     * @param bool $from_cache
     */
    public function viewItemByID($id = 1, $from_cache = false)
    {
        global $HTMLHeader;
        
        $item = $this->getItemByIDWithExtraData($id);
        if ($item['status'] <= 0)
        {
            echo ___('Item is not published');
        } else
        {
            $item_meta_options = $this->translateMetaOptions($item['options']);        
            $this->setPageTitle($item['title']);
            if ($this->itemColumnExists('meta_description'))
                $this->setPageMetaDescription($item['meta_description']);
            if ($this->itemColumnExists('meta_key'))
                $this->setPageMetaKeywords ($item['meta_key']);
            $this->getBreadCrumbsByItemID($item['id']);
            if ($this->itemColumnExists('date_modified') && $item['date_modified'] != NULL_DATE && !empty($item['date_modified']))
                $this->declarePageLastModified($item['date_modified']);

            $intro_image = '';
            $intro_image_caption = $item['description_intro_image_caption'] ? $item['description_intro_image_caption'] : $item['summary_intro_image_caption'];
            $intro_image_field =  $item['description_intro_image'] ? 'description_intro_image' : 'summary_intro_image';
            $intro_image  = $this->getBlogImage('image_large', $item[$intro_image_field]); 
            $intro_image_caption = $item['description_intro_image_caption'] ? $item['description_intro_image_caption'] : $item['summary_intro_image_caption'];
            $secondary_headline = $item['description_secondary_headline'] ? $item['description_secondary_headline'] : $item['summary_secondary_headline'];

            $url_host = SCHLIX_SITE_SSL_ENABLED ? SCHLIX_SITE_HTTPS_URL : SCHLIX_SITE_HTTP_URL;
            $full_canonical_url =  $url_host. remove_multiple_slashes('/' . $this->createFriendlyURL("action=viewitem&id={$item[$this->field_id]}"));

            //$HTMLHeader->add('<link rel="canonical" href="'.$full_canonical_url.'" />');
            $HTMLHeader->setCanonicalURL($full_canonical_url);
            
            $this->processDataOutputWithMacro($item, __FUNCTION__, ['item_meta_options'=>$item_meta_options]);        
            $this->loadTemplateFile('view.item', 
                    ['item' => $item, 
                    'secondary_headline' => $secondary_headline, 
                    'intro_image' => $intro_image, 
                    'intro_image_caption' => $intro_image_caption, 
                    'item_meta_options' => $item_meta_options]);
            $this->increaseItemPageView($item[$this->field_id]);
        }
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Given item ID, return its full path.
     * URL modes: 
     * - CMS_APPLICATION_MANY_TO_MANY_URL_SINGLE, 
     * - CMS_APPLICATION_MANY_TO_MANY_URL_WITH_CATEGORY
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $item_id
     * @param int $force_cid
     * @return boolean|string
     */
    public function getFullPathByItemID($item_id, $force_cid = 0)
    {
        global $SystemDB;

        if ($this->url_mode == APP_BLOG_URL_MODE_WITH_DATES)
        {
            // use date_available, not date_created
            $sql = "SELECT date_available, virtual_filename,{$this->table_categories_items}.{$this->field_category_id} FROM {$this->table_items}
                    INNER JOIN {$this->table_categories_items} ON {$this->table_categories_items}.{$this->field_id} = {$this->table_items}.{$this->field_id}
                    WHERE {$this->table_items}.{$this->field_id} = '{$item_id}'";
            $item = $SystemDB->getQueryResultSingleRow($sql, true);
            if ($item)
            {
                if ($item['date_available'] && $item['date_available'] != NULL_DATE)
                {
                    $int_time = strtotime($item['date_available']);
                    $year = date('Y',$int_time);
                    $month = date('m',$int_time);
                    $day = date('d',$int_time);
                    
                    $category_path = $this->getFullPathByCategoryID($item[$this->field_category_id]);
                    $result = remove_multiple_slashes("/{$year}/{$month}/{$day}/{$category_path}/{$item['virtual_filename']}.html");
                    return $result;
                    
                } else return parent::getFullPathByItemID ($item_id);
            } else return false;
        }
        else
        {
           return parent::getFullPathByItemID($item_id);
        }
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of item options.
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */

    public function getItemMetaOptionKeys() {

        $parent_opt = parent::getItemMetaOptionKeys();
        $arr = [
          ['key' => 'display_description_secondary_headline', 'label' => ___('Display secondary headline')],
          ['key' => 'display_description_intro_image', 'label' => ___('Display intro image in detail view')], 
          ['key' => 'display_description_intro_image_caption', 'label' => ___('Display intro image caption in detail view')], ];
        $parent_opt[0]['options'] = array_merge($arr, $parent_opt[0]['options']);
        return $parent_opt;
        
    }
    

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of category options.
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */
    public function getCategoryMetaOptionKeys() {
        
        $parent_opt = parent::getCategoryMetaOptionKeys();
        
        $arr = ['key' => 'display_summary_secondary_headline', 'label' => ___('Display secondary headline')];
        array_unshift($parent_opt[2]['options'], $arr);
        
        $arr = ['key' => 'display_summary_intro_image', 'label' => ___('Display intro image')];
        for($i = 0;$i < 3;$i++) array_unshift($parent_opt[$i]['options'], $arr);
        
        $parent_opt[] =
          
            
            ['key' => 'image_display_size', 'type' => 'radiogroup', 'label' => ___('Intro Image Size'), 
                      'options' => 
                        [
                        ['key' => 'image_small', 'label' => ___('Small')], 
                        ['key' => 'image_medium', 'label' => ___('Medium') ],
                        ['key' => 'image_large', 'label' => ___('Large') ]
                          ]

              ];            
            
        return $parent_opt;
    }    

    /**
     * Returns an array containing on array of main page options. In this base
     * class, the key is almost similar to getCategoryMetaOptionKeys
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */    
    public function getMainpageMetaOptionKeys()
    {
        
        return
        [
            // items
            [   'label' => ___('Main page listing options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_child_items',
                'options' =>
                [
                    ['key' => 'display_item_summary_intro_image', 'label' => ___('Display intro image')],
                    ['key' => 'display_item_summary_intro_image_caption', 'label' => ___('Display intro image caption')],
                    ['key' => 'display_item_summary_secondary_headline', 'label' => ___('Display secondary headline')],
                    ['key' => 'display_item_summary', 'label' => ___('Display summary')],
                    ['key' => 'display_item_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_item_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_item_date_modified', 'label' => ___('Display date modified')],
                    ['key' => 'display_item_read_more_link', 'label' => ___('Display "Read More" link')],
                    ['key' => 'display_item_view_count', 'label' => ___('Display view count')]
                ]
              ],
            // items
            [   'label' => ___('Sort Options'),
                'options' =>
                [
                  // Items Sort Options
                    ['key' => 'items_sortby', 'type' => 'dropdownlist', 'label' => ___('Sort child items by'), 
                      'options' => $this->getSortableItemColumns()],
                    ['key' => 'items_sortdirection', 'type' => 'radiogroup', 'label' => ___('Child items sort direction'), 
                      'options' => 
                        [['key' => 'desc', 'label' => ___('Descending')], 
                        ['key' => 'asc', 'label' => ___('Ascending') ]]],

                ]
              ],
            ['key' => 'image_display_size', 'type' => 'radiogroup', 'label' => ___('Intro Image Size'), 
                      'options' => 
                        [
                        ['key' => 'image_small', 'label' => ___('Small')], 
                        ['key' => 'image_medium', 'label' => ___('Medium') ],
                        ['key' => 'image_large', 'label' => ___('Large') ]
                          ]],          
          ];         
    }
    //_______________________________________________________________________________________________________________//

    public function RSS() {
        global $SystemConfig, $SiteTheme;

        //header('Content-Type: application/rss+xml; charset=UTF-8');

        $rss_blog_title = $this->getConfig('str_rss_title');
        $rss_author_name = $this->getConfig('str_author_name');
        $rss_author_email = $this->getConfig('str_author_email');
        $rss_desc = $this->getConfig('str_description');
        $rss_max_entrychar = $this->getConfig('int_max_entry_char'); // Number of items to be displayed in the blog (recommended: 200 to 250)',true);
        $rss_maxposts = $this->getConfig('int_max_entries_in_rss'); // Maximum number of entries to be displayed in the RSS file',true);
        
        $total_item_count = $this->getTotalItemCount(true);
        // The limit is introduced to prevent memory exhaustion
        // hardcode all item limit if there's no limit specified
        if ($rss_maxposts == 0 || $rss_maxposts > $total_item_count)
            $rss_maxposts = $total_item_count;
        if ($rss_maxposts > 500)
            $rss_maxposts = 500;
        $latestposts = $this->getAllItems('*', 'status > 0', 0, $rss_maxposts, 'date_created', 'DESC');
        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile('rss', $local_variables);
    }
    

    public function JSON_FEED() {
        global $SystemConfig, $SiteTheme;

        header('Content-Type: application/json; charset=UTF-8');
        $cfg_author_name = $this->getConfig('str_author_name');
        $author = $cfg_author_name ? $cfg_author_name : SCHLIX_MAIL_DEFAULT_SENDER;
        $jsonfeed = [
            'version' => 'https://jsonfeed.org/version/1',
            'user_comment' => $this->getConfig('str_description'),
            'title' => $this->getConfig('str_rss_title'),
            'home_page_url' => SCHLIX_SITE_URL.SCHLIX_SITE_HTTPBASE,
            'feed_url' => SCHLIX_SITE_URL.$this->createFriendlyURL('action=jsonfeed'),
            'author' => ['name' => $author],
            'items' => []
        ];
        $rss_blog_title = $this->getConfig('str_rss_title');
        $rss_author_name = $this->getConfig('str_author_name');
        $rss_author_email = $this->getConfig('str_author_email');
        $rss_desc = $this->getConfig('str_description');
        $rss_max_entrychar = $this->getConfig('int_max_entry_char'); // Number of items to be displayed in the blog (recommended: 200 to 250)',true);
        $rss_maxposts = $this->getConfig('int_max_entries_in_rss'); // Maximum number of entries to be displayed in the RSS file',true);

        $total_item_count = $this->getTotalItemCount(true);
        // The limit is introduced to prevent memory exhaustion
        // hardcode all item limit if there's no limit specified
        if ($rss_maxposts == 0 || $rss_maxposts > $total_item_count)
            $rssmax_posts = $total_item_count;
        if ($rss_max_posts > 500)
            $rss_maxposts = 500;
        $latestposts = $this->getAllItems('*', 'status > 0', 0, $rss_maxposts, 'date_created', 'DESC');
        
        foreach ($latestposts as $item)
        {
            $json_item = [
                'id' => $item['guid'],
                'url' =>$this->createFriendlyURL("action=viewitem&id={$item['id']}"),
                'title' => $item['title'],
                'date_published' => date("c", strtotime($item['date_created'])),
                'content_html' => $item['summary'],
                'tags' => explode(',', $item['tags'])
                    ];
                if ($item['summary_intro_image'] && is_file($this->getDataFileFullPath('image_small', $item['summary_intro_image']))) { 
                    $json_item['image'] = SCHLIX_SITE_URL . $this->getDataFileURLPath('image_small', $item['summary_intro_image']);
                }
                
            $jsonfeed['items'][] = $json_item;
        }
        echo json_encode($jsonfeed,JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }
    
    
    /**
     * Create thumbnail for uploaded images
     * @param array $datavalues
     * @param string $file_upload_var_name
     * @param string $datavalues_key
     * @return string
     */        
    protected function createThumbnailForImageUpload($datavalues, $file_upload_var_name, $datavalues_key = NULL)
    {
        if ($datavalues_key == NULL)
            $datavalues_key = $file_upload_var_name;
        $file_summary_intro_image =$_FILES[$file_upload_var_name];
        $dk = $datavalues_key.'_selection';
        $media_type = isset($datavalues[$dk]) ? $datavalues[$dk] : null;
        
        if (is_uploaded_file($file_summary_intro_image['tmp_name']) && $media_type === 'upload') {            
            if ($file_summary_intro_image['error'] == UPLOAD_ERR_OK) {
                ///////////////////
                $pi = pathinfo($file_summary_intro_image['name']);
                $image_filename = strtolower(convert_to_safe_basename($pi['filename'],180)).'.'.strtolower(convert_to_safe_basename($pi['extension'],5));
                ////////////////////
                $current_year_and_month = date('Y-m');
                $new_image_filename =  $current_year_and_month.'/'.pathinfo($image_filename, PATHINFO_FILENAME) . '.' . pathinfo($image_filename, PATHINFO_EXTENSION);
                // name the image the same as the vfname
                $datavalues[$datavalues_key] = $new_image_filename;
                $img_original_filename = $this->getDataFileFullPath('image_original', $new_image_filename);
                create_directory_if_not_exists(pathinfo($img_original_filename, PATHINFO_DIRNAME), true);
                
                $valid_image = is_valid_uploaded_image_file($file_summary_intro_image['tmp_name'], $img_original_filename);
                if ($valid_image)
                {
                    $move_result = move_uploaded_file($file_summary_intro_image['tmp_name'], $img_original_filename);
                    if ($move_result)
                    {
                        $sizes = $this->getImageSizeNames();
                        $img_width = [];
                        $img_height = [];
                        $img_quality = [];
                        
                        foreach ($sizes as $sz)
                        {
                            $ds = $this->default_sizes[$sz];
                            $img_width[$sz] = $this->getConfig("int_{$sz}_width", $ds['width']);
                            $img_height[$sz] = $this->getConfig("int_{$sz}_height", $ds['height']);
                            $img_quality[$sz] =$this->getConfig("int_{$sz}_quality", $ds['quality']);
                            $dest_filename = $this->getDataFileFullPath("image_{$sz}", $new_image_filename);
                            create_directory_if_not_exists(pathinfo($dest_filename, PATHINFO_DIRNAME), true);
                            $result = create_image_thumbnail($img_original_filename, $dest_filename, $img_width[$sz], $img_height[$sz], $img_quality[$sz] );
                            if (!$result)
                                $this->logError("Error while trying to create {$sz} image for: {$new_image_filename}");
                        }                        
                    } else
                    {
                        $this->logError("File upload move error: {$file_summary_intro_image['name']} to {$img_original_filename}");
                    }
                }
            } else
            {
                $this->logError("File upload error before processing: {$file_summary_intro_image['name']}");
            }
        } elseif ($media_type === 'url')
        {
            //if ($type === 'upload'
            
            $url = $datavalues[$datavalues_key.'_url'];
            $valid_url = (str_starts_with($url, 'https://') || str_starts_with($url, 'http://')) && filter_var($url, FILTER_VALIDATE_URL);
            if ($valid_url)
            {
                $url = str_replace('../', '', $url); // for safety
                $datavalues[$datavalues_key] = $url;
            } else $datavalues[$datavalues_key] = null;
            
        }
        return $datavalues;
    }
    
    
    public function getValidationErrorListBeforeSaveItem($datavalues) {
        $error_list = parent::getValidationErrorListBeforeSaveItem($datavalues);
        $__category_ids_array = \SCHLIX\cmsHttpInputFilter::array_int ($datavalues, '__category_ids');
        // Verify category arrays first
        if (empty($__category_ids_array))
        {
            $error_list[] = ___('A blog post must have at least one category');
        } else if (!$this->verifyAllCategoriesExist($__category_ids_array))
        {
            $error_list[] = ___('One or more of the specified categories are invalid');
        }
        return $error_list;
    }


    /**
     * Do something after item has been saved
     * @param array $datavalues
     * @param array $original_datavalues
     * @param array $previous_item
     * @param array $retval
     */
    public function onAfterSaveItem($datavalues, $original_datavalues, $previous_item, $retval) 
    {
        
        parent::onAfterSaveItem($datavalues, $original_datavalues, $previous_item, $retval);
        if ($retval['status'] == SAVE_OK)
        {
            // set the group
            
            $this->setItemCategories($retval['id'], array_values($original_datavalues['__category_ids']));
            $this->setItemPrimaryCategory($retval['id'], $original_datavalues['primary_cid']);
        }
    }
    
    public function getImageSizeNames()
    {
        $image_sizes = [];  
        $ddir = $this->getDataDirectories();
        foreach ($ddir as $d => $v)
        {
            if (str_starts_with($d, 'image_'))
            {
                $name = str_replace('image_', '', $d);
                if ($name != 'original')
                    $image_sizes[] = $name;
            }
        }
        return $image_sizes;
    }
    
    
    public function getDefaultSizes()
    {
        return $this->default_sizes;
    }
    
    /**
     * Override default modify item data before save
     * @param array $datavalues
     */

    public function modifyDataValuesBeforeSaveItem($datavalues) {
        $datavalues = parent::modifyDataValuesBeforeSaveItem($datavalues);

        $vfn_source = !empty($datavalues['virtual_filename'] ) ? $datavalues['virtual_filename'] : $datavalues['title'];
        $short_name = str_limit(convert_into_sef_friendly_title(strtolower($vfn_source)), 255);
        
        $datavalues['virtual_filename'] = $this->preventDuplicateValueInItemTable('virtual_filename', $short_name, $datavalues[$this->field_id]) ;
        
        $datavalues = $this->createThumbnailForImageUpload($datavalues, 'summary_intro_image_upload', 'summary_intro_image');
        $datavalues = $this->createThumbnailForImageUpload($datavalues, 'description_intro_image_upload', 'description_intro_image');
        
        return $datavalues;
    }
    
    /**
     * Override default modify item data before save
     * @param array $datavalues
     */

    public function modifyDataValuesBeforeSaveCategory($datavalues) {
        $datavalues = parent::modifyDataValuesBeforeSaveCategory($datavalues);
        if (empty($datavalues['virtual_filename']))
            $datavalues['virtual_filename'] = convert_into_sef_friendly_title ($datavalues['title']);
        $datavalues = $this->createThumbnailForImageUpload($datavalues, 'summary_intro_image_upload', 'summary_intro_image');
        $datavalues = $this->createThumbnailForImageUpload($datavalues, 'description_intro_image_upload', 'description_intro_image');
        
        
        return $datavalues;
    }

    //_______________________________________________________________________________________________________________//

    public function getDefaultCategoryID() {
        return \SCHLIX\cmsApplication_CategorizedList::getDefaultCategoryID();
    }

    //_______________________________________________________________________________________________________________//

    public function fullTextSearch($keyword, $sortby = '', $sortdirection = 'ASC') {
        global $SystemDB;

        $cleankeyword = sanitize_string("%{$keyword}%");
        $sql = "SELECT * FROM {$this->table_items} WHERE title LIKE {$cleankeyword} OR summary LIKE {$cleankeyword}";
        if (!empty($sortby) && $this->itemColumnExists($sortby))
            $sql.= " ORDER BY {$sortby} {$sortdirection}";
        $search_result = $SystemDB->getQueryResultArray($sql);
        return $search_result;
    }

    //_______________________________________________________________________________________________________________//
    public function createFriendlyURL($str) {

        if (SCHLIX_SEF_ENABLED) {
            $command_array = [];
            parse_str($str, $command_array);
            $final_url = '';
            if (array_key_exists('action', $command_array))
            {
                
                switch($command_array['action'])
                {
                    case 'rss':
                        $final_url.='/rss';
                        
                        break;
                    case 'main':
                        $final_url.= (isset($command_array['pg']) && $command_array['pg'] > 1) ? 
                            "/pg{$command_array['pg']}.html" : '/';
                            
                        break;
                    case 'viewarchives':
                        unset($command_array['action']);
                         $int_month = (int) $command_array['month'];
                         $int_year = (int) $command_array['year'];
                         $int_day = isset($command_array['day']) ? (int) $command_array['day'] : 0;
                         $str_month = str_pad ($int_month, 2,'0', STR_PAD_LEFT);
                         $final_url = "/{$int_year}/{$str_month}/";
                         if ($int_day > 0)
                         {
                            $str_day = str_pad ($str_day, 2,'0', STR_PAD_LEFT);
                            $final_url.= "{$str_day}/";
                         }
                         $final_url.= (isset($command_array['pg'])) ? 
                            "/pg{$command_array['pg']}.html" : '/';
                        break;
                    default:
                        return parent::createFriendlyURL($str);
                }
            } else return parent::createFriendlyURL ($str);
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . $this->getFullApplicationAlias(). $final_url;
        } else
            return parent::createFriendlyURL($str);
        return remove_multiple_slashes($final_url);
    }

    //_______________________________________________________________________________________________________________//
    public function viewMainPage($pg = 1) {
                
        $current_time = sanitize_string(get_current_datetime());
        $invalid_date = sanitize_string(NULL_DATE);
        $sql_criteria = "(status > 0)  AND (date_available is null OR date_available < {$current_time}) AND ((date_expiry IS NULL OR date_expiry = {$invalid_date}) OR date_expiry >= {$current_time}) ";
        // Set Page Title
        $str_page_title = $this->getConfig('str_mainpage_title', $this->getApplicationDescription());
        $pg = (int) $pg;
        if ($pg > 1)
            $str_page_title.=' - '.___('Page').' '.$pg;
        $this->setPageTitle($str_page_title);
        // Set Max Posts per page
        $max_posts_perpage = $this->getConfig('int_mainpage_items_per_page', 10);
        $main_meta_options =  $this->translateMetaOptions($this->getConfig('array_mainpage_meta_options'));
        $main_meta_options['display_child_item_read_more_link'] = isset($main_meta_options['display_child_item_read_more_link'] ) ? $main_meta_options['display_child_item_read_more_link'] : false;
        if ($max_posts_perpage == 0)
            $max_posts_perpage = 10;
        // Set total item count
        $total_item_count = $this->getTotalItemCount($sql_criteria);
        if ($max_posts_perpage * $pg > $total_item_count + $max_posts_perpage) {
            display_http_error(404);
            return false;
        }
        // Set Pagination
        $pagination = $this->getStartAndEndForItemPagination($pg, $max_posts_perpage, $total_item_count);
        $sort_by = isset($main_meta_options['items_sortby']) ?  $main_meta_options['items_sortby'] : 'date_created';
        $sort_dir = isset($main_meta_options['items_sortdirection']) ?  $main_meta_options['items_sortdirection'] : 'DESC';
        // Get Items to display
        $items = $this->getAllItems('*', $sql_criteria, $pagination['start'], $pagination['end'], "featured, {$sort_by}", $sort_dir);
        $this->declarePageLastModified($this->getMaxDateFromArray($items));
        // Get main page meta options
        
        $pagination_url_action = "action=main";
        $blog_view_mode = 'main';
        $this->loadTemplateFile('view.main', array('pagination_url_action' => $pagination_url_action, 'blog_view_mode' => $blog_view_mode, 
          'pagination' => $pagination, 'pg' => $pg,'max_posts_perpage'=> $max_posts_perpage, 'total_item_count' => $total_item_count, 
          'items' => $items,'main_meta_options' => $main_meta_options));
    }

    /**
     * View blog archives by year, month, and optionally day.
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $year
     * @param int $month
     * @param int $day
     * @param int $page
     * @param string $sortby
     * @param string $sortdir
     */
    public function viewArchives($year, $month, $day = 0, $page = 0, $sortby = 'date_created', $sortdir = 'DESC')
    {
        global $SystemDB;
        
        $year = (int) $year;
        $month = (int) $month;
        $day = (int) $day;
        if ($year + $month + $day == 0)
            return;
        $testdate = $year.'-'.str_pad ($month, 2,'0', STR_PAD_LEFT);
        $testformat = 'Y-m';
        $pg = (int) $page;
        $str_page_title_prefix = ___('Archives for ');
        
        $current_time = sanitize_string(get_current_datetime());
        $invalid_date = sanitize_string(NULL_DATE);
        $extra_sql_criteria = " (date_available < {$current_time} OR date_available IS NULL) AND ((date_expiry IS NULL OR date_expiry = {$invalid_date}) OR date_expiry >= {$current_time}) ";
        
        if ($day > 0)
        {
            // daily archives
            $testdate.='-'.str_pad ($day, 2,'0', STR_PAD_LEFT);
            $testformat = 'Y-m-d';
            $str_page_title ="{$str_page_title_prefix} {$testdate}";
            
            $sql_where_condition = "status > 0 AND YEAR(date_created) = {$year} AND MONTH(date_created) = {$month} AND DAY(date_created) = {$day} AND {$extra_sql_criteria}";
            $str_pagination_url = "action=viewarchives&year={$year}&month={$month}&day={$day}";
            //$sqlcount = "SELECT COUNT(`id`) as total_item_count FROM {$this->table_items} WHERE status > 0 AND YEAR(date_created) = {$year} && MONTH(date_created) = {$month} AND DAY(date_created) = {$day}";            
        } else
        {
            // monthly archives
            $str_page_title ="{$str_page_title_prefix} {$testdate}";            
            $sql_where_condition = "status > 0 AND YEAR(date_created) = {$year} and MONTH(date_created) = {$month} AND {$extra_sql_criteria}";
            $str_pagination_url = "action=viewarchives&year={$year}&month={$month}";
            
        }
        if (!is_date( $testdate, $testformat))
        {
            echo ___('Invalid date specified: '.$testdate);
            return;
        }
        $pg = (int) $page;
        
        if ($pg > 1)
            $str_page_title.=' - '.___('Page').' '.$pg;
        $this->setPageTitle($str_page_title);

        // Get count
        $result = $SystemDB->getQueryResultSingleRow("SELECT COUNT(`id`) as total_item_count FROM {$this->table_items} WHERE {$sql_where_condition}");
        $total_item_count = $result['total_item_count'];        
        
        ///// options ///
        $max_posts_perpage = $this->getConfig('int_mainpage_items_per_page', 10);
        $main_meta_options =  $this->translateMetaOptions($this->getConfig('array_mainpage_meta_options'));
        $pagination = $this->getStartAndEndForItemPagination($pg, $max_posts_perpage, $total_item_count);        
        // Now begin
        $items = $this->getAllItems('*', $sql_where_condition, $pagination['start'], $pagination['end'], "featured, {$sortby}", $sortdir);
        ///// end options ///

        $pagination_url_action = "action=viewarchives";
        $blog_view_mode = 'main';
        $this->loadTemplateFile('view.archives', array('pagination_url_action' => $pagination_url_action, 'blog_view_mode' => $blog_view_mode, 
          'pagination' => $pagination, 'pg' => $pg,'max_posts_perpage'=> $max_posts_perpage, 'total_item_count' => $total_item_count, 
          'items' => $items,'main_meta_options' => $main_meta_options,'str_pagination_url' => $str_pagination_url, 'str_page_title' => $str_page_title, 'year' => $year, 'month' => $month));
        
        
    }
    //_______________________________________________________________________________________________________________//
    public function interpretFriendlyURL($urlpath) {
        $command = [];
        if (SCHLIX_SEF_ENABLED && !fget_string('app')) {
            $parsedurl = $this->probeFriendlyURLDestination($urlpath);
            $url = $parsedurl['url'];
            $url_array = $parsedurl['url_array'];
            if ($url_array[0] == 'rss')
                $command['action'] = 'rss';
            elseif ($url_array[0] == 'jsonfeed')
                $command['action'] = 'jsonfeed';
            
            else
            // main
            if ($c = preg_match_all("/(pg)(\d+).*?(html)/is", $url_array[0], $x)) {
                $command = [];
                $folder_requestpage = $x[2][0];
                if ($folder_requestpage == 1)
                {
                    //$command['pg'] = $folder_requestpage;
                    $this->redirectToOtherAction("");
                    exit();
                }
                $command['pg'] = $folder_requestpage;                
                $command['action'] = 'main';
            }  else
            // daily archive or viewitem
            if ((int)$url_array[0] > 0 && (int) $url_array[1] > 0 && (int) $url_array[2] > 0) {
                $command['action'] = 'viewarchives';
                $command['year'] = $url_array[0];
                $command['month'] = $url_array[1];
                $command['day'] = $url_array[2];
                $command['pg'] = 0;
                if ($c = preg_match_all("/(pg)(\d+).*?(html)/is", $url_array[3], $x))
                {
                    $command['pg'] =  $x[2][0];
                }
            } else                
            //monthly archive
            if ((int)$url_array[0] > 0 && (int) $url_array[1] > 0) {
                $command['action'] = 'viewarchives';
                $command['year'] = $url_array[0];
                $command['month'] = $url_array[1];
                $command['day'] = '0';
                $command['pg'] = 0;
                if ($c = preg_match_all("/(pg)(\d+).*?(html)/is", $url_array[2], $x))
                {
                    $command['pg'] =  $x[2][0];
                }
            }            
            else
                return parent::interpretFriendlyURL($urlpath);
        } else {
            return parent::interpretFriendlyURL($urlpath);
        }
        return $command;
    }
    
    public function getMonthAndYearArchives()
    {
        global $SystemDB;
        
        return $SystemDB->getQueryResultArray("SELECT distinct year(date_created) AS year_created, month(date_created) as month_created FROM `gk_blog_items` WHERE `status` > 0 ORDER BY date_created DESC");
    }

    //_______________________________________________________________________________________________________________//
    public function Run($command) {
        //$this->insertTest();
        
        switch ($command['action']) {
            case 'viewarchives':
                
                $this->viewArchives($command['year'],$command['month'],$command['day'], $command['pg'], 'date_created', 'DESC');
                return true;
                break;
            case 'jsonfeed': $this->JSON_FEED();
                return false;
            case 'rss': $this->RSS();
                return false;
                break; // yayaya
            case 'viewcategory':$this->viewCategoryByID($command['cid'], $command['pg'], 'date_created', 'DESC');
                return true;
                break;
            case 'main': 
                $this->viewMainPage($command['pg']);
                return true;
                break;
            default: return parent::Run($command);
        }
    }

}
            