<?php
namespace App;
/**
 * Core: CronScheduler - Main Class
 * 
 * Core - CronScheduler
 * 
 * @copyright 2019 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @package core
 * @version 1.0
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */

define('IDX_MINUTE', 0);
define('IDX_HOUR', 1);
define('IDX_DAY', 2);
define('IDX_MONTH', 3);
define('IDX_WEEKDAY', 4);
define('IDX_YEAR', 5);

define('CRON_STATUS_INACTIVE', 0);
define('CRON_STATUS_ACTIVE', 1);
define('CRON_STATUS_RUNNING', 2);
define('MAX_CRON_LOG_ITEM_COUNT', 10);

class Core_CronScheduler extends \SCHLIX\cmsApplication_List {

    const CRON_COMMAND_INTERNAL = 0;
    const CRON_COMMAND_EXTERNAL_URL = 1;

    private $cf = [];
    private $ct = [];
    private $cm = [];

    public function __construct() {
        parent::__construct(___('System Scheduler'), 'gk_cronscheduler_items');
    }

    //_______________________________________________________________________________________________________________//

    public function viewItemByID($id = 1, $from_cache = false) {
        return false;
    }

    //______________________________________________________________________________________________________________//
    /**
     * Find out the nearest time from now when this cron string going to be run
     * @param string $string
     * @return boolean
     */
    public function getNearestUNIXTimeFromCronString($string) { // min hour day_of_month month day_of_week

        $string = trim(preg_replace('/[\s]{2,}/', ' ', $string));
        $this->cm[IDX_HOUR] = $this->cm[IDX_MINUTE] = $this->cm[IDX_MONTH] = [];
        $this->ct = array('minute' => null, 'hour' => null, 'day' => null, 'month' => null, 'year' => null);
        $this->cf = @explode(' ', $string);
        if ((preg_match('/[^-\/,* \\d]/', $string) !== 0) || (sizeof($this->cf) != 5)) {
            return false;
        }                 
        // Jan 28, 2023 - fix for PHP8.1 - strftime has been deprecated
        $str_now_date = date('i,H,d,m,Y'); // was strftime("%M,%H,%d,%m,%Y", time())
        list ($current_minute, $current_hour, $current_day, $current_month, $current_year) = explode(',', $str_now_date);
        $this->ct[IDX_YEAR] = $current_year;
        $array_months = $this->getFieldArray($this->cm[IDX_MONTH], $this->cf[IDX_MONTH], 1, 12);
        $this->ct[IDX_MONTH] = $this->getMaxRangeFromArray($array_months, $current_month);
        if ($this->ct[IDX_MONTH] === NULL) {
            $this->ct[IDX_YEAR] --;
            $this->gotoPreviousMonth($this->getFieldArray($this->cm[IDX_MONTH], $this->cf[IDX_MONTH], 1, 12));
        } elseif ($this->ct[IDX_MONTH] == $current_month) {
            $array_days = $this->getDaysArray($this->ct[IDX_MONTH], $this->ct[IDX_YEAR]);
            $this->ct[IDX_DAY] = $this->getMaxRangeFromArray($array_days, $current_day);
            if ($this->ct[IDX_DAY] === NULL)
                $this->gotoPreviousMonth($array_months);
            elseif ($this->ct[IDX_DAY] == $current_day) {
                $array_hours = $this->getFieldArray($this->cm[IDX_HOUR], $this->cf[IDX_HOUR], 0, 23);
                $this->ct[IDX_HOUR] = $this->getMaxRangeFromArray($array_hours, $current_hour);
                if ($this->ct[IDX_HOUR] === NULL)
                    $this->gotoPreviousDay($array_days, $array_months);
                elseif ($this->ct[IDX_HOUR] < $current_hour)
                {
                    $p_fr = $this->getFieldArray($this->cm[IDX_MINUTE], $this->cf[IDX_MINUTE], 0, 60);
                    $this->ct[IDX_MINUTE] = array_pop($p_fr);
                }
                else {
                    $mr_arr = $this->getFieldArray($this->cm[IDX_MINUTE], $this->cf[IDX_MINUTE], 0, 60);
                    $this->ct[IDX_MINUTE] = $this->getMaxRangeFromArray($mr_arr, $current_minute);
                    if ($this->ct[IDX_MINUTE] === NULL)
                        $this->gotoPreviousHour($array_hours, $array_days, $array_months);
                }
            }
            else {
                $p_hr = $this->getFieldArray($this->cm[IDX_HOUR], $this->cf[IDX_HOUR], 0, 23);
                $p_mn = $this->getFieldArray($this->cm[IDX_MINUTE], $this->cf[IDX_MINUTE], 0, 60);
                $this->ct[IDX_HOUR] =   array_pop($p_hr);                
                $this->ct[IDX_MINUTE] = array_pop($p_mn);
            }
        } else {
            $this->ct[IDX_DAY] = array_pop($this->getDaysArray($this->ct[IDX_MONTH], $this->ct[IDX_YEAR]));
            if ($this->ct[IDX_DAY] === NULL)
                $this->gotoPreviousMonth($array_months);
            else {
                $x_hr = $this->getFieldArray($this->cm[IDX_HOUR], $this->cf[IDX_HOUR], 0, 23);
                $x_mn = $this->getFieldArray($this->cm[IDX_MINUTE], $this->cf[IDX_MINUTE], 0, 60);
                $this->ct[IDX_HOUR] = array_pop($x_hr);
                $this->ct[IDX_MINUTE] = array_pop($x_mn);
            }
        }
        return ($this->ct[IDX_MINUTE] === NULL) ? false : mktime($this->ct[IDX_HOUR], $this->ct[IDX_MINUTE], 0, $this->ct[IDX_MONTH], $this->ct[IDX_DAY], $this->ct[IDX_YEAR]);
    }

    //______________________________________________________________________________________________________________//
    private function getRangeArray($str, $low, $high) {
        $str = trim($str);
        $ret = [];
        if (str_contains($str, ',')) {
            $parts_array = explode(',', $str);
            foreach ($parts_array as $part) {
                if (str_contains($part, '-')) {
                    list ($rmin, $rmax) = explode('-', $part);
                    $ret = range($rmin, $rmax);
                } else
                    $ret[] = $part;
            }
        }
        elseif (str_contains($str, '/')) {
            list($t1, $step) = explode('/', $str);
            if ($step == 0)
                $step = 1; // cannot divide by zero
            $ret = range($low, $high, $step);
        }
        elseif (str_contains($str, '-')) {
            list ($rmin, $rmax) = explode('-', $str);
            $ret = range($rmin, $rmax);
        } else
            $ret[] = $str;
        $ret = array_unique($ret);
        sort($ret);

        $x = sizeof($ret);
        for ($i = 0; $i < $x; $i++)
            if ($ret[$i] < $low)
                unset($ret[$i]);
            else
                break;
        $x--;
        for ($i = $x; $i >= 0; $i--)
            if ($ret[$i] > $high)
                unset($ret[$i]);
            else
                break;
        sort($ret);

        return $ret;
    }

    //______________________________________________________________________________________________________________//
    private function getMaxRangeFromArray(array &$the_array, $max) {
        $x = array_pop($the_array);
        while ($x > $max)
            $x = array_pop($the_array);
        return $x;
    }

    //______________________________________________________________________________________________________________//
    function gotoPreviousMonth(array $array_months) {
        $this->ct[IDX_MONTH] = array_pop($array_months);
        if ($this->ct[IDX_MONTH] === NULL) {
            $this->ct[IDX_YEAR] --;
            $this->gotoPreviousMonth($this->getFieldArray($this->cm[IDX_MONTH], $this->cf[IDX_MONTH], 1, 12));
        } else {
            $this->ct[IDX_DAY] = array_pop($this->getDaysArray($this->ct[IDX_MONTH], $this->ct[IDX_YEAR]));
            if ($this->ct[IDX_DAY] === NULL) {
                $this->gotoPreviousMonth($array_months);
            } else {
                $this->ct[IDX_HOUR] = array_pop($this->getFieldArray($this->cm[IDX_HOUR], $this->cf[IDX_HOUR], 0, 23));
                ;
                $this->ct[IDX_MINUTE] = array_pop($this->getFieldArray($this->cm[IDX_MINUTE], $this->cf[IDX_MINUTE], 0, 60));
            }
        }
    }

    //______________________________________________________________________________________________________________//
    private function gotoPreviousDay(array $array_days, array $array_months) {
        $this->ct[IDX_DAY] = array_pop($array_days);
        if ($this->ct[IDX_DAY] === NULL) {
            $this->gotoPreviousMonth($array_months);
        } else {
            $this->ct[IDX_HOUR] = array_pop($this->getFieldArray($this->cm[IDX_HOUR], $this->cf[IDX_HOUR], 0, 23));
            $this->ct[IDX_MINUTE] = array_pop($this->getFieldArray($this->cm[IDX_MINUTE], $this->cf[IDX_MINUTE], 0, 60));
        }
    }

    //______________________________________________________________________________________________________________//
    private function gotoPreviousHour(array $array_hours, array $array_days, array $array_months) {
        $this->ct[IDX_HOUR] = array_pop($array_hours);
        if ($this->ct[IDX_HOUR] === NULL)
            $this->gotoPreviousDay($array_days, $array_months);
        else
            $this->ct[IDX_MINUTE] = array_pop($this->getFieldArray($this->cm[IDX_MINUTE], $this->cf[IDX_MINUTE], 0, 60));
    }

    //______________________________________________________________________________________________________________//
    private function getDaysArray($month, $year = 0) {
        $days = [];
        if ($year == 0)
            $year = $this->ct[IDX_YEAR];
        if ($this->cf[IDX_DAY] == '*' AND $this->cf[IDX_WEEKDAY] == '*')
            $days = range(1, date('t', mktime(0, 0, 0, $month, 1, $year)));
        else {
            if ($this->cf[IDX_WEEKDAY] == '*')
                $array_weekdays = range(0, 6);
            else {
                $array_weekdays = $this->getRangeArray($this->cf[IDX_WEEKDAY], 0, 7);
                if (in_array(7, $array_weekdays)) {
                    if (in_array(0, $array_weekdays))
                        array_pop($array_weekdays);
                    else {
                        array_pop($array_weekdays);
                        $array_weekdays = array_merge(array(0), $array_weekdays);
                    }
                }
            }
            foreach ((($this->cf[IDX_DAY] == '*') ? range(1, date('t', mktime(0, 0, 0, $month, 1, $year))) : $this->getRangeArray($this->cf[IDX_DAY], 1, date('t', mktime(0, 0, 0, $month, 1, $year)))) as $day)
                if (in_array(date('w', mktime(0, 0, 0, $month, $day, $year)), $array_weekdays))
                    $days[] = $day;
        }
        return $days;
    }
    //______________________________________________________________________________________________________________//
    private function getFieldArray(array $the_array, $bitval, $minrange, $maxrange) {
        return (empty($the_array)) ? (($bitval == '*') ? range($minrange, $maxrange) : $this->getRangeArray($bitval, $minrange, $maxrange)) : $the_array;
    }

    //______________________________________________________________________________________________________________//
    protected function updateExistingCronItem($id, $datavalues) {
        global $SystemDB;

        $id = (int) $id;
        $SystemDB->simpleUpdate($this->table_items, $datavalues, $this->field_id, $id);
    }

    //_______________________________________________________________________________________________________________//
    protected function executeCronByID(array $item) {
        $command = '';
        $id = $item['id'];
        $process_name = "cron_item_{$id}";
        if (\SCHLIX\cmsProcessMutex::getLockStatus($process_name) == false) {
            $output = '';
            \SCHLIX\cmsProcessMutex::Lock($process_name);
            $date_start = date('Y-m-d H:i:s', time());
            // Validate process name and run it
            if ($item['command_type'] == self::CRON_COMMAND_EXTERNAL_URL) {
                $filtered_name = filter_var($item['command_external_url'], FILTER_VALIDATE_URL);
                if ($filtered_name !== false) {
                    $command = $item['command_external_url'];
                    $output = @file_get_contents($command);
                    if ($output === false)
                        $this->recordLog("Error executing {$process_name}");
                    else
                        $output = htmlspecialchars($output);
                } else {
                    $output = "Invalid URL!";
                }
            } else {
                $command = $item['command_internal'];
                if (is_callable($command)) {
                    ob_start();
                    call_user_func($command);
                    $output = ob_get_clean();
                } else {
                    $output = "Invalid function!";
                }
            }
            $date_end = date('Y-m-d H:i:s', time());
            $this->recordItemActivity($item[$this->field_id], $item['cron'], $date_start, $date_end, $item['date_expected_next_run'], $command, $output);
            \SCHLIX\cmsProcessMutex::Unlock($process_name);
        }
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Record log
     * @global \App\SCHLIX\cmsDatabase $SystemDB
     * @param int $cron_id
     * @param string $cron
     * @param string $date_start
     * @param string $date_end
     * @param string $date_expected_next_run
     * @param string $command
     * @param string $log
     */
    protected function recordItemActivity($cron_id, $cron, $date_start, $date_end, $date_expected_next_run, $command, $log) {
        global $SystemDB;

        $datavalues = array('cron_id' => $cron_id, 'cron' => $cron, 'date_start' => $date_start, 'date_end' => $date_end, 'date_expected_next_run' => $date_expected_next_run, 'command' => $command, 'log' => $log);
        $SystemDB->simpleInsertInto('gk_cronscheduler_log', $datavalues);
        
    }

    //_________________________________________________________________________//
    /**
     * Create a default scheduled task for your own app
     * @global \App\SCHLIX\cmsDatabase $SystemDB
     * @param string $description
     * @param string $app_static_method
     * @param string $default_cron_string
     * @return boolean
     */
    public function createDefaultSchedulerForMyApplicationStaticMethod($description, $app_static_method, $default_cron_string) {
        global $SystemDB;

        $existing_static_methods = $this->getApplicationStaticMethods();
        if (in_array($app_static_method, $existing_static_methods)) {
            $sql = "SELECT * FROM gk_cronscheduler_items WHERE command_internal = " . sanitize_string($app_static_method);
            $existing_entry = $SystemDB->getQueryResultSingleRow($sql);
            if (!$existing_entry) {
                $datavalues = array('title' => $description, 'cron' => $default_cron_string, 'command_internal' => $app_static_method, 'status' => 1);
                $SystemDB->simpleInsertInto('gk_cronscheduler_items', $datavalues);
                
            } else
                return true;
        } else
            return false;
    }

    private function getValidAppClassName($item, $parent_class = '')
    {
        try 
        {
            $class_name = $item->getFileName();
            
            $class_file = $item->getPathName(). "/{$class_name}.class.php";
            if ($parent_class)
            {
                $class_file = $item->getPathName(). "/{$parent_class}.{$class_name}.class.php";
            }
            if (!str_contains($class_name, 'uninstalled_') && file_exists($class_file))
            {
                if ($parent_class)
                {
                    $class_name = $parent_class.'_'.$class_name;
                }
                if (class_exists('\\App\\'.$class_name, true))
                {
                        return '\\App\\'.$class_name;
                }
            }
        } catch (\Exception $exc)
        {
            
        }
        return null;
    }
    //_________________________________________________________________________//

    private function getClassesFromDirectory($dir, $include_sub_apps = false)
    {
        $all_classes = null;
        $items = \SCHLIX\cmsDirectoryFilter::getDirectoryIterator($dir, \SCHLIX\cmsDirectoryFilter::FILTER_DIR_ONLY);        
        if ($items)
        {            
            foreach ($items as $item) {
                $class_name = $this->getValidAppClassName($item);
                if ($class_name)
                {
                    //echo $item->getPathName() ;die;
                    $all_classes[] = $class_name;
                    $sub_app_dir = $item->getPathName();//.'/'.$item->getFileName();
                    //echo $sub_app_dir;die;
                    if ($class_name && $include_sub_apps)
                    {
                        $parent_class_name =  $item->getFileName();
                        $sub_apps = \SCHLIX\cmsDirectoryFilter::getDirectoryIterator($sub_app_dir, \SCHLIX\cmsDirectoryFilter::FILTER_DIR_ONLY);        
                        if ($sub_apps)
                        {
                            foreach ($sub_apps as $sub_app)
                            {
                                $sub_app_class_name = $this->getValidAppClassName($sub_app, $parent_class_name);
                                if ($sub_app_class_name)
                                {
                                    $all_classes[] = $sub_app_class_name;
                                }

                            }
                        }
                    }
                    
                }
            }
        }
        return $all_classes;
    }
    
    private function getBuiltinClasses()
    {
        $all_classes = get_declared_classes();
        $schlix_classes = [];
        foreach ($all_classes as $cl)
        {
            $cllo = strtolower($cl);
            if (str_starts_with($cllo, 'schlix\\') || str_starts_with($cllo, 'app\\'))
            {
                
                $schlix_classes[] = '\\'.$cl;
            }
        }
        return $schlix_classes;
    }
    
    private function getAllSCHLIXClasses()
    {
        $items = $this->getClassesFromDirectory(SCHLIX_SYSTEM_PATH. '/apps/', true);
        $site_app_items = $this->getClassesFromDirectory(SCHLIX_SITE_PATH. '/apps/'); 
        if (is_array($site_app_items))
            $items = array_merge($items, $site_app_items);
        $classes = $this->getBuiltinClasses();
        $items = array_merge($items, $classes);
        $items = array_unique($items);
        return $items;
    }
    //_________________________________________________________________________//
    public function getApplicationStaticMethods() {
        
        $items = $this->getAllSCHLIXClasses();
        $method_list = [];
        //print_r($classes);die;
        //print_r($items);die;
        foreach ($items as $item) 
        {
            $app_reflect_class = new \ReflectionClass($item);
            if ($app_reflect_class->isUserDefined()) {
                $app_static_methods = $app_reflect_class->getMethods(\ReflectionMethod::IS_STATIC);
                if (is_array($app_static_methods))
                    foreach ($app_static_methods as $method) {
                        $reflected_method = new \ReflectionMethod($method->class, $method->name);
                        $reflected_method_parameters = $reflected_method->getParameters();
                        // Only STATIC methods without parameters qualify as a cron item
                        if ((___c($reflected_method_parameters) == 0) && stripos($method->name, "run") !== false)
                            $method_list[] = "\\{$method->class}::{$method->name}";
                    }
            }
        }
        $method_list = array_unique($method_list);
        return $method_list;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * CRON command - clean up log
     * @global SCHLIX\cmsDatabase $SystemDB
     * @global SCHLIX\cmsLogger $SystemLog
     */
    public static function processRunCleanupLog() {
        global $SystemDB, $SystemLog;

        $SystemLog->record('Cleaning up old CRON log begins....', 'core.cronscheduler');
        $sql = "SELECT DISTINCT(cron_id) FROM gk_cronscheduler_log";
        $all_cron_logs = $SystemDB->getQueryResultArray($sql);
        foreach ($all_cron_logs as $cron_logs) {
            $cron_id = intval($cron_logs['cron_id']);
            // 1. Get total count
            $sql = "SELECT COUNT(id) as total_count FROM gk_cronscheduler_log WHERE cron_id = {$cron_id} ORDER BY id ASC";
            $cron_id_count = $SystemDB->getQueryResultSingleRow($sql);
            // 2. If it's over the hardcoded limit of 20 then delete to prevent performance problem
            if ($cron_id_count['total_count'] > MAX_CRON_LOG_ITEM_COUNT) {
                $number_of_items_to_delete = $cron_id_count['total_count'] - MAX_CRON_LOG_ITEM_COUNT;
                $sql = "DELETE FROM gk_cronscheduler_log WHERE id < (SELECT MAX(id) FROM (SELECT id FROM gk_cronscheduler_log WHERE cron_id = {$cron_id} ORDER BY id ASC LIMIT 0, {$number_of_items_to_delete}) table_cron)";
                $SystemDB->query($sql);
                $SystemLog->record ("Deleting {$number_of_items_to_delete} log entries for cron ID #{$cron_id}", 'core.cronscheduler');
            }
        }
        echo "Cleanup";
        $SystemLog->record('Done cleaning up old CRON log....', 'core.cronscheduler');
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Deprecated as of v2.0.3. Checks CRON log if this task has been run.
     * @deprecated since version 2.0.3
     * @global \App\SCHLIX\cmsDatabase $SystemDB
     * @param int $cron_id
     * @param string $date_expected_next_run
     * @return bool
     */
    protected function isThereTaskWithCronIDandNextExpectedDateRun($cron_id, $date_expected_next_run) {
        global $SystemDB;

        $cron_id = (int) $cron_id;
        $date_expected_next_run = sanitize_string($date_expected_next_run);
        $sql = "SELECT cron_id FROM gk_cronscheduler_log WHERE cron_id = {$cron_id} and date_expected_next_run = {$date_expected_next_run}";
        return $SystemDB->getQueryResultSingleRow($sql);
    }

    //_______________________________________________________________________________________________________________//
    public function runAllScheduledItems() {

        $process_name = __CLASS__;
        if (\SCHLIX\cmsProcessMutex::getLockStatus($process_name) == false) {

            \SCHLIX\cmsProcessMutex::Lock($process_name);
            //ignore_user_abort(true);
            $all_items = $this->getAllItems('*', "((date_available IS NULL) OR (date_available < NOW()) OR (date_available = '0000-00-00 00:00:00')) AND ((date_expiry IS NULL) OR (date_expiry > NOW()) OR (date_expiry = '0000-00-00 00:00:00')) AND status = " . CRON_STATUS_ACTIVE);
            foreach ($all_items as $item) {

                $next_date_expected_next_run = $this->getNearestUNIXTimeFromCronString($item['cron']);
                $str_next_date_expected_next_run = date('Y-m-d H:i:s', $next_date_expected_next_run);
                // sehr wichtig!! this line ensures there's no duplicate since date_expected_next_run is in the past! prana
                // v2.0.3 - update it since this is inefficient
                //if ($next_date_expected_next_run !== false && !$this->isThereTaskWithCronIDandNextExpectedDateRun($item[$this->field_id], $str_next_date_expected_next_run)) {
                if ($next_date_expected_next_run !== false && $item['date_expected_next_run'] != $str_next_date_expected_next_run) {
                /////////////
                    //1. Run
                    $data = [];
                    
                    if ($next_date_expected_next_run < time()) {
                        $data['date_lastrun_start'] = get_current_datetime();
                        $this->executeCronByID($item);
                        $data['date_lastrun_end'] = get_current_datetime();
                    }
                    //2. UPDATE THE ENTRY
                    $data['date_modified'] = get_current_datetime();
                    $data['date_expected_next_run'] = $str_next_date_expected_next_run;
                    $this->updateExistingCronItem($item['id'], $data);
                }
            }
            \SCHLIX\cmsProcessMutex::Unlock($process_name);
        }
    }
    
    public function runScheduledItemsManually()
    {
        if ($this->getConfig('bool_use_manual_scheduler'))
        {
            $restrict_to_ip_addr = trim($this->getConfig('str_manual_scheduler_restrict_ip_address'));
            if (!empty($restrict_to_ip_addr) && get_user_real_ip_address() !=  $restrict_to_ip_addr)
            {
                echo ___('Your IP address has been denied for access to this system scheduler');
            } else
            {
                echo ___('Running scheduler');
                echo '...';
                $this->runAllScheduledItems();
                echo ___('Done');
            }
        } else 
        {
           echo ___('Scheduler is set to run automatically. Ignoring this request');
        }
    }

    //_______________________________________________________________________________________________________________//
    public function Run($command) {
        switch ($command['action'])
        {
            case 'main': 
                $this->runScheduledItemsManually();
                return FALSE;
                break;
            default:
                echo ___('Nothing to see');
                return FALSE;break;
        }
        
        return false;
    }

//_______________________________________________________________________________________________________________//
}
