<?php
namespace App;


/**
 * Aliases for League Provider Classes
 * Make sure you have added these to your composer.json and run `composer install`
 * Plenty to choose from here:
 * @see http://oauth2-client.thephpleague.com/providers/thirdparty/
 */
// @see https://github.com/thephpleague/oauth2-google
use League\OAuth2\Client\Provider\Google;
// @see https://packagist.org/packages/hayageek/oauth2-yahoo
use Hayageek\OAuth2\Client\Provider\Yahoo;
// @see https://github.com/stevenmaguire/oauth2-microsoft
use Stevenmaguire\OAuth2\Client\Provider\Microsoft;

/**
 * Core: EmailQueue - Main Class
 * 
 * Core - EmailQueue
 * 
 * @copyright 2019 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @package core
 * @version 1.0
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */
class Core_EmailQueue extends \SCHLIX\cmsApplication_List {

    // can't use Trait yet to connect to basicApplication, wait for php 5.4
    protected $is_composed = false;
    
    /**
     * Message ID that you can customize if you derive this class
     * @var string 
     */
    protected static $xmessage_id = 'X-Schlix-CMS-MsgID';
    protected static $max_retry_count = 5;
    protected static $queue_table_items = 'gk_email_queue_items';
    protected static $dir_storage_folder = NULL;
    protected static $dir_relative_storage_folder = '/data/private/mail';
    protected static $cfg_max_email_per_hour = 180;
    protected static $cfg_max_email_per_batch = 10;
    
    protected static $is_debug = false;
    protected static $last_debug_output = null;
    /**
     * Default name of the sender, e.g. Webmaster
     * @var string
     */
    protected static $default_sender = SCHLIX_MAIL_DEFAULT_SENDER;
    /**
     * Default email address
     * @var string 
     */
    protected static $default_email = SCHLIX_MAIL_DEFAULT_EMAIL;    
    /**
     * Either use the local mailer or SMTP server
     * @var bool 
     */
    protected static $use_smtp_server;
    /**
     * Before October 2020: TRUE/FALSE
     * After October 2020: 0 = no auth, 1 = regular password, 2 = oauth2
     * @var int
     */
    protected static $smtp_use_auth;
    /**
     * SMTP - Server host name
     * @var string 
     */
    protected static $smtp_host;
    /**
     * SMTP - Server port number (usually 25, 465, or 587)
     * @var string
     */
    protected static $smtp_port;
    /**
     * SMTP - Server username
     * @var string 
     */
    protected static $smtp_username;
    /**
     * SMTP - Server password
     * @var string 
     */
    protected static $smtp_password;
    /**
     * SMTP - Server security, either empty, TLS or SSL
     * @var string 
     */
    protected static $smtp_security;
    /**
     * SMTP - Allow self signed SSL certificate
     * @var bool 
     */
    protected static $smtp_security_allow_self_signed_cert;    
    /**
     * SMTP - OAUTH2 - Provider (google, microsoft, yahoo) - so far in Oct 2020
     * @var string 
     */
    protected static $smtp_oauth2_provider;        
    /**
     * SMTP - OAUTH2 - Application/Client ID
     * @var string 
     */
    protected static $smtp_oauth2_client_id;
    /**
     * SMTP - OAUTH2 - Secret
     * @var string 
     */
    protected static $smtp_oauth2_secret;
    /**
     * SMTP - OAUTH2 - Offline refresh token
     * @var string 
     */
    protected static $smtp_oauth2_refresh_token;
    /**
     * SMTP - OAUTH2 - Indicates that OAUTH2 config values are complete
     * @var bool 
     */
    protected static $smtp_oauth2_config_complete;
    
    /**
     * X-Mailer
     * @var string 
     */
    protected static $xmailer = 'Website Emailer';
    /**
     * Constructor
     */
    public function __construct() {
        
        parent::__construct('Email Queue', static::$queue_table_items);
        
        // PHP5.5 fix        
        if (static::$dir_storage_folder == NULL)
            static::$dir_storage_folder = SCHLIX_SITE_PATH . '/data/private/mail';
        // end PHP5.5 fix as per Punit Varia 
        if (!is_dir(static::$dir_storage_folder)) {
            if (!create_directory_if_not_exists(static::$dir_storage_folder))
                die('Fatal error: Mail storage folder cannot be created');
        }
        $max_hourly_email = $this->getConfig('int_max_email_per_hour');
        $max_perbatch_email = $this->getConfig('int_max_email_per_batch');
        static::$cfg_max_email_per_hour = $max_hourly_email ? $max_hourly_email : 180;
        static::$cfg_max_email_per_batch = $max_perbatch_email ? $max_perbatch_email : 10;

        $this->disable_frontend_runtime = true;  
        $this->setEmailServerConfiguration();
    }
    
    /**
     * Set SCHLIX default email server config. You can override this in your own derived class
     */
    protected function setEmailServerConfiguration()
    {
        $valid_providers = ['google','yahoo','microsoft'];
        static::$use_smtp_server = defined('SCHLIX_SMTP_HOST') && (strlen(SCHLIX_SMTP_HOST) > 0);
        static::$smtp_use_auth = defined('SCHLIX_SMTP_USE_AUTH') ? (int) SCHLIX_SMTP_USE_AUTH : 0;        
        static::$smtp_host = defined('SCHLIX_SMTP_HOST') ? SCHLIX_SMTP_HOST : 'localhost';
        static::$smtp_port = (defined('SCHLIX_SMTP_PORT') && intval(SCHLIX_SMTP_PORT) > 0) ? (int) SCHLIX_SMTP_PORT : NULL;
        static::$smtp_username = (defined('SCHLIX_SMTP_USERNAME') && strlen(SCHLIX_SMTP_USERNAME) > 0) ? SCHLIX_SMTP_USERNAME : NULL;
        static::$smtp_password = (defined('SCHLIX_SMTP_PASSWORD') && strlen(SCHLIX_SMTP_PASSWORD) > 0) ? SCHLIX_SMTP_PASSWORD : NULL;
        static::$smtp_security = (defined('SCHLIX_SMTP_SECURITY') && ((strtolower(SCHLIX_SMTP_SECURITY) == 'ssl') || (strtolower(SCHLIX_SMTP_SECURITY) == 'tls'))) ? strtolower(SCHLIX_SMTP_SECURITY) : NULL;
        static::$smtp_security_allow_self_signed_cert = defined('SCHLIX_SMTP_SECURITY_SELF_SIGNED') ? (int) SCHLIX_SMTP_SECURITY_SELF_SIGNED : 0;
        
        static::$smtp_oauth2_provider = (defined('SCHLIX_SMTP_OAUTH2_PROVIDER') && strlen(SCHLIX_SMTP_OAUTH2_PROVIDER) > 0) ? SCHLIX_SMTP_OAUTH2_PROVIDER : NULL;
        static::$smtp_oauth2_client_id = (defined('SCHLIX_SMTP_OAUTH2_CLIENT_ID') && strlen(SCHLIX_SMTP_OAUTH2_CLIENT_ID) > 0) ? SCHLIX_SMTP_OAUTH2_CLIENT_ID : NULL;
        static::$smtp_oauth2_secret = (defined('SCHLIX_SMTP_OAUTH2_SECRET') && strlen(SCHLIX_SMTP_OAUTH2_SECRET) > 0) ? SCHLIX_SMTP_OAUTH2_SECRET : NULL;
        static::$smtp_oauth2_refresh_token = (defined('SCHLIX_SMTP_OAUTH2_REFRESH_TOKEN') && strlen(SCHLIX_SMTP_OAUTH2_REFRESH_TOKEN) > 0) ? SCHLIX_SMTP_OAUTH2_REFRESH_TOKEN : NULL;
        
        static::$smtp_oauth2_config_complete = in_array(static::$smtp_oauth2_provider, $valid_providers) &&
                strlen(static::$smtp_oauth2_client_id) > 3 &&
                strlen(static::$smtp_oauth2_secret) > 3 &&
                strlen(static::$smtp_oauth2_refresh_token) > 3;
        if (static::$smtp_use_auth >= 2)
        {
            if (!static::$smtp_oauth2_config_complete)
                static::$smtp_use_auth =  (!empty(static::$smtp_username)  && !empty(static::$smtp_password)) ? 1 : 0;
        }
    }
    
    //_______________________________________________________________________________________________________________//
    public function html2text($html, $advanced = false) {
        $temp_php_mailer = new \PHPMailer\PHPMailer\PHPMailer (true);
        $html = str_replace(array('<br />', '<hr />', '<br>', '<hr>'), "\r\n", $html);
        $temp_php_mailer->MsgHTML($html, SCHLIX_SITE_PATH);
        return $temp_php_mailer->AltBody;
    }

    //_______________________________________________________________________________________________________________//
    public function sendEmailToUserByUserObject($user, $subject, $body_html, $body_text) {
        if ($user) {
            if ($user['email_address']) {
                return $this->sendSimpleEmailToQueue($subject, static::$default_sender, static::$default_email, $user['firstname'] . ' ' . $user['lastname'], $user['email_address'], $body_html, $body_text);
            } else
                trigger_error('User does not have an email address', E_USER_NOTICE);
        } else
            trigger_error('User is NULL', E_USER_NOTICE);
        return false;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Send an email to user by the user ID
     * @global \App\Users $CurrentUser
     * @param int $userid
     * @param string $subject
     * @param string $body_html
     * @param string $body_text
     * @return boolean
     */
    public function sendEmailToUserByUserID($userid, $subject, $body_html, $body_text) {
        global $CurrentUser;
        if ($userid > 0) {
            $user_obj = ($CurrentUser != NULL) ? $CurrentUser : $CurrentUser;
            if ($user_obj) {
                $user = $user_obj->getItemByID($userid);

                if ($user) {
                    return $this->sendEmailToUserByUserObject($user, $subject, $body_html, $body_text);
                } else
                    trigger_error('No user by that ID was found', E_USER_NOTICE);
            } else
                trigger_error('No user object defined', E_USER_WARNING);
        }
        return false;
    }

    //_______________________________________________________________________________________________________________//
    public function getEmailTemplate($virtual_filename) {
        $mailtemplate = new \App\Core_Emailtemplate();
        return $mailtemplate->getPreProcessedEmailTemplateByVirtualFileName($virtual_filename);
    }

    //_______________________________________________________________________________________________________________//
    public function getEmailSubjectAndBodyFromTemplate($virtual_filename, $vars) {
        $email_template = $this->getEmailTemplate($virtual_filename);
        
        if ($email_template) {
            $subject = $this->processMessageTemplate($email_template['title'], $vars);
            $body_html = $this->processMessageTemplate($email_template['body_html'], $vars);
            $body_text = $this->processMessageTemplate($email_template['body_text'], $vars);
            return array('subject' => $subject, 'html' => $body_html, 'text' => $body_text);
            
        } else {
            trigger_error('Email template cannot be found', E_USER_WARNING);
            return false;
        }
    }

    /**
     * Replaces variables (in format {$variablename} in text with variable values from $vars
     * The regex has changed for Schlix v2
     * @param string $text
     * @param array $vars
     * @param bool $htmlencode
     * @return type
     */
    public function processMessageTemplate($text, $vars, $htmlencode = false) {
        $count = preg_match_all('/\\{\\$((?:[a-z][a-z0-9_]*))}/is', $text, $matches);

        if ($count > 0) {
            for ($i = 0; $i < $count; $i++) {
                $text_tobe_replaced = $matches[0][$i];
                $variable_name_only = $matches[1][$i];
                if (array_key_exists($variable_name_only, $vars)) {
                    $v = $htmlencode ? ___h($vars[$variable_name_only]) : $vars[$variable_name_only];
                    $text = str_replace($text_tobe_replaced, $v, $text);
                } else
                {
                    // quick fix - 2020-Sept-1 - behaviour change - {$varname} will be nullified if it does not exist
                    $text = str_replace($text_tobe_replaced, '', $text);
                }
            }
        }
            return $text;
        
    }

    //_______________________________________________________________________________________________________________//
    /**
     * v2.2.4-1 and above uses master template (responsive email template)
     * @deprecated since version 2.2.4-1
     * @return boolean
     */
    public function getDefaultSignature() {
        return null;
    }
    
    
    //_______________________________________________________________________________________________________________//
    public function sendEmailFromWebMasterToUserByUserIDWithTemplateName($userid, $virtual_filename,  $vars) {
        global $CurrentUser, $CurrentUser;
        if ($userid > 0) {
            $user_obj = ($CurrentUser != NULL) ? $CurrentUser : $CurrentUser;
            if ($user_obj) {
                $user = $user_obj->getItemByID($userid);

                if ($user) {
                    $processed_email_template = $this->getEmailSubjectAndBodyFromTemplate($virtual_filename, $vars);
                    if(!$processed_email_template) return FALSE;
                    $subject = $processed_email_template['subject'];
                    $body_html = $processed_email_template['html'];
                    $body_text = $processed_email_template['text'];
                    
                    return $this->sendEmailToUserByUserObject($user, $subject, $body_html, $body_text);
                } else
                    trigger_error('No user by that ID was found', E_USER_NOTICE);
            } else
                trigger_error('No user object defined', E_USER_WARNING);
        }
        return false;
    }
    
    //_______________________________________________________________________________________________________________//
    public function sendEmailFromWebMasterWithTemplateName($field_to_name, $field_to_email, $virtual_filename,  $vars) {
        $processed_email_template = $this->getEmailSubjectAndBodyFromTemplate($virtual_filename, $vars);
        if(!$processed_email_template) return FALSE;
        $subject = $processed_email_template['subject'];
        $body_html = $processed_email_template['html'];
        $body_text = $processed_email_template['text'];
        
        if ($subject) {
           return $this->sendEmailToQueue('', $subject, static::$default_sender, static::$default_email, $field_to_name, $field_to_email, '', '', '', '', '', '', '', '', $body_html, $body_text, null, null);
        } else return FALSE;
    }
    
    //_______________________________________________________________________________________________________________//
    public function sendEmailWithTemplateName($field_from_name, $field_from_email, $field_to_name, $field_to_email, $field_cc_name, $field_cc_email, $field_replyto_name, $field_replyto_email, $virtual_filename,  $vars) {
        $processed_email_template = $this->getEmailSubjectAndBodyFromTemplate($virtual_filename, $vars);
        if(!$processed_email_template) return FALSE;
        $subject = $processed_email_template['subject'];
        $body_html = $processed_email_template['html'];
        $body_text = $processed_email_template['text'];
        
        if ($subject) { 
           return $this->sendEmailToQueue('', $subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, $field_cc_name, $field_cc_email, $field_replyto_name, $field_replyto_email, '', '', '', '', $body_html, $body_text, null, null);
        } else return FALSE;
    }
    
    /**
     * Send email with default sender
     * @param string $field_subject
     * @param string $field_to_name
     * @param string $field_to_email
     * @param string $body_html
     * @param string $body_text
     * @param array $attachments
     */
    public function sendSimpleSignedEmailFromWebmaster($field_subject, $field_to_name, $field_to_email, $body_html, $body_text, $attachments = NULL)
    {
        return $this->sendSimpleSignedEmailToQueue($field_subject, static::$default_sender, static::$default_email, $field_to_name, $field_to_email, $body_html, $body_text,$attachments);
    }
    
    /**
     * Send signed email
     * @deprecated since version 2.2.4-1
     * @param string $field_subject
     * @param string $field_from_name
     * @param string $field_from_email
     * @param string $field_to_name
     * @param string $field_to_email
     * @param string $body_html
     * @param string $body_text
     * @param array $attachments
     */
    public function sendSimpleSignedEmailToQueue($field_subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, $body_html, $body_text, $attachments = NULL) {
        
        return $this->sendEmailToQueue('', $field_subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, '', '', '', '', '', 0, '', '', $body_html, $body_text, $attachments,  null);
    }
    
    /**
     * 
     * @param string $field_subject
     * @param string $field_from_name
     * @param string $field_from_email
     * @param string $field_to_name
     * @param string $field_to_email
     * @param string $body_html
     * @param string $body_text
     */
    public function sendSimpleEmailToQueue($field_subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, $body_html, $body_text, $attachments = NULL) {
        
        return $this->sendEmailToQueue('', $field_subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, '', '', '', '', '', 0, '', '', $body_html, $body_text, $attachments, '');
    }
    
    /**
     * 
     * @param string $field_subject
     * @param string $field_from_name
     * @param string $field_from_email
     * @param string $field_to_name
     * @param string $field_to_email
     * @param string $field_cc_name
     * @param string $field_cc_email
     * @param string $body_html
     * @param string $body_text
     * @param array $attachments
     */
    public function sendSimpleEmailToQueueWithCC($field_subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, $field_cc_name, $field_cc_email, $body_html, $body_text, $attachments = NULL) {
        $this->sendEmailToQueue('', $field_subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, $field_cc_name, $field_cc_email, '', '', '', 0, '', '', $body_html, $body_text, $attachments, '');
    }
    
    /**
     * Get email storage folder
     * @param int $id
     * @param bool $is_relative
     * @return string
     */
    public static function getEmailStorageFolderForID($id, $is_relative = FALSE) {
        $id = (int) $id;
        $id = floor($id / 10000.0);
        // PHP5.5 fix
        if (static::$dir_storage_folder == NULL)
            static::$dir_storage_folder = SCHLIX_SITE_PATH . '/data/private/mail';
        // end PHP5.5 fix
        $parent_folder = $is_relative ? static::$dir_relative_storage_folder : static::$dir_storage_folder;
        $foldername = $parent_folder . '/' .date('Y').'/'.date('m').'/'.$id;
        return $foldername;
    }

    /**
     * create email storage folder if not exists
     * @param int $id
     * @return boolean
     */
    protected function setupMailStorageFolderForEmailID($id) {
        $foldername = static::getEmailStorageFolderForID($id);
        if (!is_dir($foldername)) {
            if (!create_directory_if_not_exists($foldername))
                die('Fatal error: Mail storage folder cannot be created. Email request ' . $id . ' denied');
            return TRUE;
        }
        return TRUE;
    }


    /**
     * 
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $queue_marker
     * @param string $field_subject
     * @param string $field_from_name
     * @param string $field_from_email
     * @param string $field_to_name
     * @param string $field_to_email
     * @param string $field_cc_name
     * @param string $field_cc_email
     * @param string $field_replyto_name
     * @param string $field_replyto_email
     * @param string $field_bcc
     * @param string $field_priority
     * @param string $field_xmailer
     * @param string $field_return_receipt
     * @param string $body_html
     * @param string $body_text
     * @param string $attachments
     * @param string $extra_headers
     * @return boolean
     */
    public function sendEmailToQueue($queue_marker, $field_subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, $field_cc_name, $field_cc_email, $field_replyto_name, $field_replyto_email, $field_bcc, $field_priority, $field_xmailer, $field_return_receipt, $body_html, $body_text, $attachments, $extra_headers) {
        global $SystemDB;

        $body_html = trim ($body_html);
        $body_text = trim($body_text);
        $field_xmailer = trim($field_xmailer);
        $field_subject = trim($field_subject);
        if (empty($field_subject) || empty($field_to_email) || empty($field_from_email) || (empty($body_html) && empty($body_text)))
            return FALSE;

        $guid = new_uuid_v4();
        $the_attachment = NULL;
        $datavalues = array(
          'queue_marker' => $queue_marker,
          'field_subject' => $field_subject,
          'field_from_name' => $field_from_name,
          'field_from_email' => $field_from_email,
          'field_to_name' => $field_to_name,
          'field_to_email' => $field_to_email,
          'field_cc_name' => $field_cc_name,
          'field_cc_email' => $field_cc_email,
          'field_replyto_name' => $field_replyto_name,
          'field_replyto_email' => $field_replyto_email,
          'field_bcc' => $field_bcc,
          'field_priority' => $field_priority ? $field_priority : '0',
          'field_xmailer' => $field_xmailer,
          'field_return_receipt' => $field_return_receipt,
          'guid' => $guid,
          'date_queued' => get_current_datetime()
        );
        
        if (is_array($attachments) && ___c($attachments) > 0)
        {
            $attachments = array_unique($attachments);
            foreach ($attachments as $file)
            {
                if (!file_exists(SCHLIX_SITE_PATH.'/data/private/attachments/'.$file))
                    return false;
            }
            $the_attachment = implode('|', $attachments);
        }        
        
        if ($the_attachment)
            $datavalues['attachments'] = $the_attachment;
        if ($extra_headers)
        {
            if (is_array($extra_headers))
            {
                if (___c($extra_headers > 0))
                {
                    $datavalues['extra_headers'] = implode("|", $extra_headers);
                }
            } else
            {
                $datavalues['extra_headers'] = $extra_headers;
            }
        }
        $serialized_data = @serialize($datavalues).$body_html.$body_text;
        $datavalues['field_sha1'] = sha1($serialized_data); // to check authenticity
        
        $SystemDB->simpleInsertInto($this->table_items, $datavalues);
       
        $new_email_id = $SystemDB->getLastInsertID();

        $mailstore_folder = static::getEmailStorageFolderForID($new_email_id);
        $relative_mailstore_folder = static::getEmailStorageFolderForID($new_email_id, TRUE);
        
        $upd_store_folder = array('store_folder' => $relative_mailstore_folder);
        $SystemDB->simpleUpdate($this->table_items, $upd_store_folder,'id', $new_email_id);
        
        if ($this->setupMailStorageFolderForEmailID($new_email_id)) 
        {
            $body_html_filename = $body_html ? "{$mailstore_folder}/{$new_email_id}.html" : '';
            $body_text_filename = $body_text ? "{$mailstore_folder}/{$new_email_id}.txt" : '';
            $store_html_ok = $store_txt_ok = TRUE;
            if ($body_html_filename)
                $store_html_ok = file_put_contents($body_html_filename, $body_html);
            if ($body_text_filename)
                $store_txt_ok = file_put_contents($body_text_filename, $body_text);
            if ($store_html_ok === FALSE || $store_txt_ok === FALSE)
                die('Fatal error: Email storage system malfunction');
            
            return TRUE;
        }
        
        return FALSE;
    }
    
    /**
     * Set debug to true/false
     * @param bool $is_debug
     */
     public static function setDebug($is_debug)
     {
         static::$is_debug = $is_debug;
     }

     public static function getDebugString($str, $level) 
     {
        $debug_cmd = [
            'SMTP INBOUND' => 'cyan',
            'SERVER -&gt; CLIENT' => 'yellow',
            'CLIENT -&gt; SERVER' => 'lime',
            'Auth method selected' => 'salmon',
            'Auth method requested' => 'pink',
            'Auth methods available on the server' => 'gold',
            'Requested auth method not available' => 'red',
            'SMTP ERROR' => 'red',
            'Auth methods available on the server' => 'lavender',
            'Connection' => 'deepskyblue'
        ];
        $text = ___h( preg_replace('/[\r\n]+/', '', $str) );
        $l = strpos($text,':');
        if ($l > 1)
        {
            $start = substr($text, 0, $l);
            $color = array_key_exists($start, $debug_cmd) ? $debug_cmd[$start] : null;
            if ($color)
            {
                $text = "<span class=\"txt-{$color}\">{$text}</span>";
            }
            
        }
        static::$last_debug_output.=   $text. "<br />\n";
    }
    /**
     * Send email directly using PHPMailer. Don't set it to protected, it's a public method but users shouldn't access this method
     * unless it's really critical so all emails are logged in the database
     * @param string $unique_id
     * @param string $field_subject
     * @param string $field_from_name
     * @param string $field_from_email
     * @param string $field_to_name
     * @param string $field_to_email
     * @param string $field_cc_name
     * @param string $field_cc_email
     * @param string $field_replyto_name
     * @param string $field_replyto_email
     * @param string $field_bcc
     * @param string $field_priority
     * @param string $extra_headers
     * @param string $field_xmailer
     * @param string $field_return_receipt
     * @param string $body_html
     * @param string $body_text
     * @param array $attachments
     */

    public static function __sendEmailDirectly($timeout, $unique_id, $field_subject, $field_from_name, $field_from_email, $field_to_name, $field_to_email, $field_cc_name, $field_cc_email, $field_replyto_name, $field_replyto_email, $field_bcc, $field_priority, $extra_headers, $field_xmailer, $field_return_receipt, $body_html, $body_text, $attachments)
    {
        global $SystemLog;
        $mymailer = new \PHPMailer\PHPMailer\PHPMailer(true);
        $error_msg= null;
        try
        {
            if (static::$is_debug)
            {
                static::$last_debug_output = '';
                //$mymailer->Debugoutput = 'html';
                $mymailer->Debugoutput = '\\App\\Core_EmailQueue::getDebugString';  
            }
            /// Load Settings
            if (static::$use_smtp_server) {
                $mymailer->IsSMTP();                
                if (static::$is_debug)
                    $mymailer->SMTPDebug = 4;
                $mymailer->Port = (int) static::$smtp_port;
                $mymailer->Host = static::$smtp_host;
                if (static::$smtp_use_auth == 1 && !empty(static::$smtp_username) && !empty(static::$smtp_password)) {
                    $mymailer->SMTPAuth = true;                    
                    $mymailer->Username = static::$smtp_username;
                    $mymailer->Password = static::$smtp_password;
                } elseif (static::$smtp_use_auth == 2) {
                    $oauth2_provider = null;
                    switch (static::$smtp_oauth2_provider)
                    {
                        case 'google':
                            $oauth2_provider = new Google(['clientId' => static::$smtp_oauth2_client_id,'clientSecret' => static::$smtp_oauth2_secret, 'scope' => [ 'https://mail.google.com/' ] ]);
                            break;
                        case 'microsoft':
                            $oauth2_provider = new Microsoft(['clientId' => static::$smtp_oauth2_client_id,'clientSecret' => static::$smtp_oauth2_secret]);
                            break;
                        
                    }
                    if ($oauth2_provider != null)
                    {
                        $mymailer->SMTPAuth = true;
                        $mymailer->AuthType = 'XOAUTH2';
                        $mymailer->setOAuth( new \PHPMailer\PHPMailer\OAuth( [ 'provider' => $oauth2_provider, 'clientId' => static::$smtp_oauth2_client_id, 
                            'clientSecret' => static::$smtp_oauth2_secret, 'refreshToken' => static::$smtp_oauth2_refresh_token, 'userName' => static::$smtp_username ]));                        
                    }
                }
                
                if (static::$smtp_security)
                {
                    $mymailer->SMTPSecure = strtolower(static::$smtp_security);
                    if (static::$smtp_security_allow_self_signed_cert)
                    {
                        $mymailer->SMTPOptions = [ 'ssl' => [ 'verify_peer' => false, 'verify_peer_name' => false, 'allow_self_signed' => true ] ];
                    }
                    /*$mymailer->AuthType = 'LOGIN/AUTH'; */                                 
                }
            }
            $mymailer->Timeout = ($timeout > 0) ? $timeout : 180;
            /// End load settings
            // Basic header


            if ($field_from_email)
                $mymailer->SetFrom($field_from_email,$field_from_name);
            if ($field_to_email)
                $mymailer->AddAddress($field_to_email, $field_to_name);        
            if ($field_replyto_email)
                $mymailer->AddReplyTo($field_replyto_email, $field_replyto_name);
            if ($field_cc_email)
                $mymailer->AddCC($field_cc_email, $field_cc_name);
            if ($field_bcc)
                $mymailer->AddBCC($field_bcc);
            $mymailer->Subject = $field_subject;        
            // end basic header
            /////////////////////////////////////////////////////////////////////
            // Advanced header
            $mymailer->XMailer = $field_xmailer ?  $field_xmailer : static::$xmailer;
            $mymailer->CharSet = 'UTF-8';        

            //$mymailer->addCustomHeader('MIME-Version', '1.0');
            if ($field_priority != 0)
            {
                switch ($field_priority)
                {
                    case EMAIL_PRIORITY_LOW:
                        $mymailer->addCustomHeader('X-Priority', '5 (Lowest)');
                        $mymailer->addCustomHeader('X-MSMail-Priority', 'Low');
                        $mymailer->addCustomHeader('Importance', 'Low');    

                        break;
                    case EMAIL_PRIORITY_HIGH:
                        $mymailer->addCustomHeader('X-Priority', '1 (Highest)');
                        $mymailer->addCustomHeader('X-MSMail-Priority', 'High');
                        $mymailer->addCustomHeader('Importance', 'High');    
                        break;
                }
                //X-MSMail-Priority: High, Normal, or Low
                //Importance: High, Normal, or Low
            }
            if ($field_return_receipt) {
                $mymailer->addCustomHeader('Disposition-Notification-To', $field_return_receipt);
                $mymailer->addCustomHeader('X-Confirm-Reading-To', $field_return_receipt);
            }        
            if ($extra_headers)
            {
                $header_lines = explode('|', $extra_headers);
                if (is_array($header_lines) && count ($header_lines) > 0)
                {
                    foreach ($header_lines as $header)
                    {
                        $mymailer->addCustomHeader($header);
                    }
                }
            }
            if ($unique_id)
                $mymailer->addCustomHeader(static::$xmessage_id,$unique_id);
            // end Advanced header
            /////////////////////////////////////////////////////////////////////        
            // Body & Text
            $mymailer->WordWrap = 80;
            if (!empty($body_html)) {
                $mymailer->IsHTML(true); // send as HTML
                $converted_text = $mymailer->MsgHTML($body_html, SCHLIX_SITE_PATH);
                $mymailer->AltBody = (!empty($body_text)) ? $body_text : $converted_text;
            } else {
                $mymailer->IsHTML(false); // send as plaintext
                $mymailer->Body = $body_text;
            }
            if ($attachments) 
            {
                $attachment_array = explode('|', $attachments);
                foreach ($attachment_array as $file)
                {
                    $attachment_filename = SCHLIX_SITE_PATH.'/data/private/attachments/'.$file;
                    if (file_exists($attachment_filename))
                    {
                        $mymailer->AddAttachment($attachment_filename);
                    }
                }
            }

            $mymailer->Send();
            if (static::$use_smtp_server)
            {
                $mymailer->smtpClose();
            }
            if (static::$is_debug)
            {
                static::$last_debug_output.= "\n ------- Completed -------\n";
            }
            return 'SENDMAIL_OK';
        }
        catch (\PHPMailer\PHPMailer\Exception $e) {
            $error_msg = $e->errorMessage();
            $SystemLog->error($error_msg);
            /*if (static::$is_debug)
            {
                static::$last_debug_output = ob_get_clean();
            }*/
            static::$last_debug_output.= $error_msg;
            return $error_msg;
            
        }
        catch (\Exception $e) {
            $error_msg = $e->errorMessage();
            $SystemLog->error($error_msg);
            static::$last_debug_output.= $error_msg;
            /*
            if (static::$is_debug)
            {
                static::$last_debug_output =  ob_get_clean();
            }*/
            return $error_msg;
        }
        return null;
    }
    //_______________________________________________________________________________________________________________//
    public function processQueueByID($id) { // just for testing single script. leave it.
        global $SystemDB;

        $id = (int) $id;
        $sql = "SELECT * FROM  " . static::$queue_table_items . "  WHERE id = {$id}";
        $data = $SystemDB->getQueryResultSingleRow($sql);
        return ($data != false) ? static::processSingleQueue($data) : false;
    }

    //_______________________________________________________________________________________________________________//
    protected static function markEmailQueueAsSentByID($id) {
        global $SystemDB;

        $id = (int) $id;
        $sql = "UPDATE  " . static::$queue_table_items . "  SET status = 1, date_sent = :date_sent, hourly_marker = :hourly_marker WHERE id = :id";
        $SystemDB->query($sql, ['date_sent' => get_current_datetime(),  'hourly_marker' => date('Y-m-d-H'),  'id' => $id ]);
    }

    //_______________________________________________________________________________________________________________//
    protected static function markEmailQueueAsFailedByID($id, $error_msg = '') {
        global $SystemDB;

        $id = (int) $id;
        
        $SystemDB->query("UPDATE  " . static::$queue_table_items . "  SET date_last_retry = :last_retry_date, retry_count = retry_count + 1 WHERE id = :id",
                ['last_retry_date' => get_current_datetime(), 'id' => $id ]);        
        $SystemDB->query("UPDATE  " . static::$queue_table_items . "  SET status = -1 WHERE retry_count > " . static::$max_retry_count);
        if ($error_msg)
        {
            $mailstore_folder = static::getEmailStorageFolderForID($id);
            file_put_contents("{$mailstore_folder}/{$id}.error.log", $error_msg,  FILE_APPEND);
        }

    }

    public static function getLastDebugOutput()
    {
        return static::$last_debug_output;
    }
    
    //_______________________________________________________________________________________________________________//
    protected static function processSingleQueue(array $queue) {
        global $SystemLog, $SystemConfig;

        //$mailstore_folder = static::getEmailStorageFolderForID($queue['id']);
        $mailstore_folder = SCHLIX_SITE_PATH.$queue['store_folder'];
        $f_html = "{$mailstore_folder}/{$queue['id']}.html";
        $f_txt = "{$mailstore_folder}/{$queue['id']}.txt";
        $body_html = is_readable($f_html) ? file_get_contents($f_html) : "Stored email queue ID# {$queue['id']}HTML file is unreadable";
        $body_text = is_readable($f_txt)  ? file_get_contents($f_txt)  : "Stored email queue ID# {$queue['id']}text file is unreadable";
        
        if (empty($body_html) && empty($body_text))
        {
            $SystemLog->error("Cannot find either HTML or text content for email ID# {$queue['id']}");
            return false;
        }
        $unq = $queue['field_sha1'].'-'.$queue['id'];
        $dbconfig_timeout = $SystemConfig->get('emailqueue','int_smtp_timeout');
        $result = static::__sendEmailDirectly($dbconfig_timeout, $unq, $queue['field_subject'], $queue['field_from_name'], $queue['field_from_email'], 
            $queue['field_to_name'], $queue['field_to_email'], $queue['field_cc_name'], $queue['field_cc_email'], 
            $queue['field_replyto_name'], $queue['field_replyto_email'], $queue['field_bcc'], $queue['field_priority'],
            $queue['extra_headers'], $queue['field_xmailer'], $queue['field_return_receipt'],
            $body_html, $body_text, $queue['attachments']);


        if ($result == 'SENDMAIL_OK')
        {
            static::markEmailQueueAsSentByID($queue['id']);
            return true;
        } else
        {
            static::markEmailQueueAsFailedByID($queue['id'], $result);
            return false;
        }
        return false;
    }

    //_______________________________________________________________________________________________________________//
    public static function getCountOfEmailsSentWithinLastHour() {
        global $SystemDB;

        $hourly_marker = sanitize_string(date('Y-m-d-H'));
        $sql = "SELECT COUNT(id) AS total FROM  " . static::$queue_table_items . "  WHERE status > 0 AND hourly_marker = {$hourly_marker}";
        $result = $SystemDB->getQueryResultSingleRow($sql);
        return $result['total'];
    }

    //_______________________________________________________________________________________________________________//
    public static function getListOfUnsentEMails($max_count) {
        global $SystemDB;

        $max_count = intval($max_count);
        $sql = "SELECT * FROM  " . static::$queue_table_items . "  WHERE status = 0 AND retry_count < 5 LIMIT 0, {$max_count}";
        
        $result = $SystemDB->getQueryResultArray($sql);
        return $result;
    }
    
    /**
     * Return the max email sent per hour
     * @return int
     */
    public static function getMaxEmailsPerHour()
    {
        $max_rate_db_value = static::$cfg_max_email_per_hour;  
        $max_rate_per_hour = (int) $max_rate_db_value > 0 ? (int) $max_rate_db_value : HARDCODE_MAX_EMAILS_PER_HOUR; // mail per hour
        return $max_rate_per_hour;
    }
    /**
     * Return the max email sent per batch
     * @return int
     */
    public static function getMaxEmailsPerBatch()
    {
        $max_rate_db_value = static::$cfg_max_email_per_batch;  
        $max_rate_per_batch = (int) $max_rate_db_value > 0 ? (int) $max_rate_db_value : HARDCODE_MAX_EMAILS_PER_BATCH; // mail per hour
        $max_per_hour = static::getMaxEmailsPerHour();
        if ($max_rate_per_batch > $max_per_hour)
            $max_rate_per_batch = $max_per_hour;
        return $max_rate_per_batch;
    }

    //_______________________________________________________________________________________________________________//
    public static function processRunEmailQueue() {
        global $SystemLog;
        $max_rate_per_hour = static::getMaxEmailsPerHour();

        $email_sent_last_hour = static::getCountOfEmailsSentWithinLastHour();
        $count_tobe_sent = $max_rate_per_hour - $email_sent_last_hour;
        if ($count_tobe_sent <= 0) {
            $output = "Would *NOT* process email queue because max emails/hour = {$max_rate_per_hour} and within the last hour the system sent {$email_sent_last_hour} emails";
            echo $output;
            $SystemLog->record($output, 'core.emailqueue', 'info');
            return false;
        } else {
            $max_perbatch = static::getMaxEmailsPerBatch();
            $count_tobe_sent = $max_rate_per_hour - $email_sent_last_hour;
            if ($count_tobe_sent > $max_perbatch)
                $count_tobe_sent = $max_perbatch;
            // Now process the queue
            $unsent_emails = static::getListOfUnsentEMails($count_tobe_sent);
            $total_unsent_emails = ___c($unsent_emails);
            $count_success = 0;
            $count_failure = 0;
            foreach ($unsent_emails as $unsent_email)
            {
                if (static::processSingleQueue($unsent_email))
                {
                    $count_success++;
                } else
                {
                    $count_failure++;
                }
            }
            if ($total_unsent_emails > 0)
            {
                $output = "Processed {$total_unsent_emails} queue items. Success: {$count_success}. Failure: {$count_failure}. Max emails/hour = {$max_rate_per_hour}. Max email/batch = {$max_perbatch}. Previous batch sent {$email_sent_last_hour} emails";
                echo $output; // output to cron log
                // output to system log
                if ($total_unsent_emails > 0)
                    $SystemLog->record($output, 'core.emailqueue', 'info');
            }
            return true;
        }
    }

    //_______________________________________________________________________________________________________________//
    protected static function error(array $mail, $errormessage) {
        global $SystemLog;

        $dest = is_array($mail['to']) ? $mail['to']['email'] : $mail['to'];
        $error_msg = "Failed - send {$mail['subject']} to {$dest}:  " . $errormessage;
        $SystemLog->record($error_msg, 'core.emailqueue', 'error');
    }

    
    //_______________________________________________________________________________________________________________//
    public function viewMainPage()
    {
            return false;
    }
    //_______________________________________________________________________________________________________________//
    public function Run($command)
    {
            return false;
    }
    

} 