<?php
namespace App;

define('WM_POSITION_TOP_LEFT', 'tl');
define('WM_POSITION_TOP', 't');
define('WM_POSITION_TOP_RIGHT', 'tr');
define('WM_POSITION_LEFT', 'l');
define('WM_POSITION_CENTER', 'c');
define('WM_POSITION_RIGHT', 'r');
define('WM_POSITION_BOTTOM_LEFT', 'bl');
define('WM_POSITION_BOTTOM', 'b');
define('WM_POSITION_BOTTOM_RIGHT', 'br');

/**
 * Gallery - Main Class
 * 
 * Gallery
 * 
 * @copyright 2019 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @package gallery
 * @version 1.0
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */
class Gallery extends \SCHLIX\cmsApplication_HierarchicalTree_List {
    
    protected $data_directories = [
      'watermark' =>  '/media/gallery/watermark',
      'image_original' => '/data/private/gallery/original',
      'image_thumbnail' => '/media/gallery/thumbnail',
      'image_small' => '/media/gallery/small',
      'image_medium' => '/media/gallery/medium',
      'image_large' => '/media/gallery/large',      
    ];
    
    protected $default_sizes = [
        'thumbnail' => ['width' => 256, 'height' => 256, 'quality' => 90],
        'small' => ['width' => 512, 'height' => 512, 'quality' => 80],
        'medium' => ['width' => 1024, 'height' => 1024, 'quality' => 70],
        'large' => ['width' => 1920, 'height' => 1920, 'quality' => 60],
    ];
    
    public function __construct() {
        parent::__construct('Gallery', 'gk_gallery_items', 'gk_gallery_categories');
        $this->schema_org_type_item = '';
        $this->has_versioning = true;
        $this->has_custom_media_header = true;
        
    }



    /**
     * View Main Page
     * @param int $pg
     */
    public function viewMainPage($pg = 1)
    {
        $str_mainpage_title = $this->getConfig('str_mainpage_title', $this->getApplicationDescription());
        $standard_criteria = 'status > 0';
        $perpage = $this->getConfig('int_mainpage_items_per_page');
        $main_meta_options =  $this->translateMetaOptions($this->getConfig('array_mainpage_meta_options'));                
        
        if ($main_meta_options == NULL)
            $main_meta_options = array();
        if ($perpage == 0)
            $perpage = $this->getNumberOfListingsPerPage();
        $total_item_count = $this->getTotalItemCountByCategoryID(0, $standard_criteria);
        $total_category_count = $this->getTotalChildCategoryCountByCategoryID(0, $standard_criteria);
        $pagination = $this->getStartAndEndForItemPagination($pg, $perpage, $total_category_count + $total_item_count);
        $this->setPageTitle(($pg > 1) ? $str_mainpage_title . " - Page {$pg}" : $str_mainpage_title);
        $this->setPageMetaDescription($this->getConfig('meta_description'));
        $this->setPageMetaKeywords($this->getConfig('meta_key'));

        $category_sortby = isset($main_meta_options['child_categories_sortby']) ?  $main_meta_options['child_categories_sortby'] : $this->getFieldCategoryID();
        $category_sortdir = isset($main_meta_options['child_categories_sortdirection']) ?  $main_meta_options['child_categories_sortdirection'] : 'DESC';
        
        $item_sortby = isset($main_meta_options['items_sortby']) ?  $main_meta_options['items_sortby'] : $this->getFieldID();
        $item_sortdir = isset($main_meta_options['items_sortdirection']) ?  $main_meta_options['items_sortdirection'] : 'DESC';

        $total_start = ($pg - 1) * $perpage;
        if ($total_start <= $total_item_count + $total_category_count) {
            if ($total_category_count > 0 && $total_category_count > $total_start) {
                //$category_end = $pagination['start'] + min($total_child_category_count - $pagination['start'], $perpage);
                $categories = $this->getChildCategoriesByParentID(0, '*', $standard_criteria, $pagination['start'], $pagination['end'], $category_sortby, $category_sortdir, false);
            }

            if ($total_item_count > 0 && $pagination['end'] - $total_category_count > 0) {
                $item_start = max(0, $pagination['start'] - $total_category_count);
                $item_end = $item_start + min($pagination['end'] - $total_category_count, $perpage);

                $items = $this->getItemsByCategoryID(0, '*', $standard_criteria, $item_start, $item_end, $item_sortby, $item_sortdir, $from_cache);
            }
        }
        else
            $error_message = ___("Page is outside of valid range");

        $main_meta_options = convert_array_values_to_keys($this->getConfig('array_mainpage_meta_options'));  
        $main_meta_options['display_item_summary'] = isset($main_meta_options['display_item_summary'] ) ? $main_meta_options['display_item_summary'] : false;
        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile('view.main', $local_variables);
        
    } 
    /**
     * Creates a friendly URL
     * @param string $str
     * @return string
     */
    public function createFriendlyURL($str) {

        if (SCHLIX_SEF_ENABLED) {
            $command_array = array();
            parse_str($str, $command_array);
            if (array_key_exists('action', $command_array) && $command_array['action'] == 'main') {
                if ($command_array['pg'])
                    $final_url.="/pg{$command_array['pg']}.html";
                else
                    $final_url.='/';
            } else
                return parent::createFriendlyURL($str);
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . $this->app_name . $final_url;
        } else
            return parent::createFriendlyURL($str);
        return remove_multiple_slashes($final_url);
    }

    /**
     * Interprets command from SEO-friendly URL
     * @param string $urlpath
     * @return array
     */
    public function interpretFriendlyURL($urlpath) {
        if (SCHLIX_SEF_ENABLED && !fget_string('app')) {
            $parsedurl = $this->probeFriendlyURLDestination($urlpath);
            $url = $parsedurl['url'];
            $url_array = $parsedurl['url_array'];
            if ($url_array[0] == 'rss')
                $command['action'] = 'rss';
            else
            if ($c = preg_match_all("/(pg)(\d+).*?(html)/is", $url_array[0], $x)) {
                $command = array();
                $folder_requestpage = $x[2][0];
                $command['pg'] = $folder_requestpage;
                $command['action'] = 'main';
            } else
                return parent::interpretFriendlyURL($urlpath);
        } else {
            return parent::interpretFriendlyURL($urlpath);
        }
        return $command;
    }
            

    //_______________________________________________________________________________________________________________//
    /**
     * Delete objects. Parameter $mixed_items_to_delete is a  category/item comma separated values
     * e.g.  c5,c6,c9,i4,i14
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $mixed_items_to_delete
     */
    function delete($mixed_items_to_delete)
    { // mixed item = categories + items
        //test case:
        //http://schlixcms/admin/index.php?page=html&ajax=1&action=ajax_delete&items=c3
        //http://schlixcms/admin/index.php?page=html&ajax=1&action=ajax_delete&items=c5,c6,c9,i4,i14
        global $SystemDB;
        $id = 1;
        $mixed_items_array = explode('|', $mixed_items_to_delete); // e.g: c5,c6,c9,i4,i14

        $cats_to_delete = NULL;
        $cat_id_filler_for_sql = [];
        $cats_to_delete_str = '';        
        $all_cats_to_delete = [];
        $items_to_delete = [];
        $items_to_delete_str = '';
        // Process sub-folders first

        foreach ($mixed_items_array as $mixed_item)
        {
            $current_id = substr($mixed_item, 1); // 11 is the next string after
            if (strpos($mixed_item, 'c') > -1) {
                // Get the sub-child for each cats
                $cats_to_delete = $this->traverseCategories($current_id);
                // Now Process
                if (___c($all_cats_to_delete) > 0)
                    $all_cats_to_delete = array_merge($all_cats_to_delete, $cats_to_delete);
                else
                    $all_cats_to_delete = $cats_to_delete;
            } else { // else if it's an item instead
                $items_to_delete[] = (int) $current_id;
            }
        }
        $result_array_items_in_these_categories = null;
        // Process files
        
        if (___c($all_cats_to_delete) > 0) {
            foreach ($all_cats_to_delete as $a_cat_to_delete)
                $cat_id_filler_for_sql[] = $a_cat_to_delete[$this->field_category_id];
            $result_array_items_in_these_categories = $this->getAllChildItemsInMultipleCategories($cat_id_filler_for_sql);
        }
        
        // Did the user select any items to delete?
        if (___c($items_to_delete) > 0) {
            if ($result_array_items_in_these_categories)
                $items_to_delete = array_merge($items_to_delete, $result_array_items_in_these_categories);
        }
        else
            $items_to_delete = $result_array_items_in_these_categories;

        // now delete all of them - $all_cats_to_delete + $items_to_delete
        if (___c($items_to_delete) > 0)
            $items_to_delete_str = implode(",", $items_to_delete);
        if (___c($cat_id_filler_for_sql) > 0)
            $cats_to_delete_str = implode(",", $cat_id_filler_for_sql);

        /////////////////////////////////////
        
        if ($items_to_delete_str) {
            
            // DELETE THE FILES FIRST
            $sql = "SELECT url_media_file FROM {$this->table_items} WHERE {$this->field_id} in ({$items_to_delete_str})";
            $item_objs_to_delete = $SystemDB->getQueryResultArray($sql);
            foreach ($item_objs_to_delete as $item)            
            {
                $this->deleteImageFilesByRelativePath($item['url_media_file']);
            }
            
            
            $sql1 = "DELETE FROM {$this->table_items} WHERE {$this->field_id} in ({$items_to_delete_str})";
            $SystemDB->query($sql1);
        }

        if ($cats_to_delete_str) {
            // 2024-02-18 - fix, no full_dir column ... leftover from older version
           /* $sql = "SELECT full_dir FROM {$this->table_categories} WHERE {$this->field_category_id} in ({$cats_to_delete_str})";
            $cat_objs_to_delete = $SystemDB->getQueryResultArray($sql);
            foreach ($cat_objs_to_delete as $cat)            
            {
                foreach ($this->data_directories as $key => $value)
                __del_tree($this->getDataFileFullPath($key, $cat['full_dir']));

            }*/
            
            $cats_to_delete_str = implode(",", $cat_id_filler_for_sql);
            $sql2 = "DELETE FROM {$this->table_categories} WHERE {$this->field_category_id} in ({$cats_to_delete_str})";
            $SystemDB->query($sql2);
        }
    }    

    
    protected function deleteImageFilesByRelativePath($url_media_file)
    {
        if (!str_starts_with($url_media_file, 'https://') && !str_starts_with($url_media_file, 'http://'))
        {
            $result = @unlink($this->getDataFileFullPath('image_original', $url_media_file)) &&
                @unlink($this->getDataFileFullPath('image_large', $url_media_file)) &&
                @unlink($this->getDataFileFullPath('image_medium', $url_media_file)) &&
                @unlink($this->getDataFileFullPath('image_small', $url_media_file)) &&
                @unlink($this->getDataFileFullPath('image_thumbnail', $url_media_file));
            return $result;
        } 
        return true;
    }    
    
    
    public function onAfterSaveCategory($datavalues, $original_datavalues, $previous_category, $retval) {
        $result = parent::onAfterSaveCategory($datavalues, $original_datavalues, $previous_category, $retval);
        $id = $retval['id'];
        $this->ensureDataDirectoryExists();
        
        return $result;
    }
                
    /**
     * Returns the URL of an image
     * @param string $image_size
     * @param string $path
     * @param int $category_id
     * @param bool $with_hash
     */
    public function getGalleryImage($image_size, $path, $category_id,  $with_hash = false)
    {
         if (str_starts_with($path, 'https://') || str_starts_with($path, 'http://')  )
         {
             if (str_contains($path, '{$width}') || str_contains($path, '{$height}'))
             {
                $category = $this->getCategoryByID($category_id);
                $sz = remove_prefix_from_string($image_size, 'image_');                
                $ds = $this->default_sizes[$sz];
                if (is_array($category))
                {
                    $img_width = $category["{$sz}_width"] > 0 ? $category["{$sz}_width"] :  $this->getConfig("int_{$sz}_width", $ds['width']);
                    $img_height = $category["{$sz}_height"] > 0 ? $category["{$sz}_height"] :  $this->getConfig("int_{$sz}_height", $ds['height']);  
                    
                } else 
                {
                    $img_width = $this->getConfig("int_{$sz}_width", $ds['width']);
                    $img_height = $this->getConfig("int_{$sz}_height", $ds['height']);  
                }
                $path = str_replace(['{$width}', '{$height}'], [strval($img_width), strval($img_height)], $path);
             }              
            return $path;
             
         } elseif (!empty($path) && file_exists($this->getDataFileFullPath($image_size, $path)))
         {
             return $with_hash ? $this->getDataFileURLPathWithHash($image_size, $path) :  $this->getDataFileURLPath($image_size, $path) ;
         }
         return null;
    }
    
    /**
     * Override default modify item data before save
     * @param array $datavalues
     */
    public function modifyDataValuesBeforeSaveItem($datavalues) {
        
        $this->ensureDataDirectoryExists();      
        
        $datavalues = parent::modifyDataValuesBeforeSaveItem($datavalues);
        $category =  $this->getCategoryByID($datavalues['category_id']);
        $cid_prefix = strval( $category ? $category[$this->field_category_id] : '0' );

        //$new_icon_filename_only = $datavalues['name'];
        $file_image = $_FILES['image_file_upload'];
        $move_upload_result = false;
        if (is_uploaded_file($file_image['tmp_name'])) {
            
            if ($file_image['error'] == UPLOAD_ERR_OK) {
                $base_filename = convert_to_safe_filename($file_image['name']);

                $valid_image = is_valid_uploaded_image_file($file_image['tmp_name'], $base_filename);
                if ($valid_image)
                {
                    // save
                    $img_rel_path = "/{$cid_prefix}/{$base_filename}";
                    $img_abs_path = remove_multiple_slashes( $this->getDataFileFullPath('image_original', $img_rel_path ) );
                    create_directory_if_not_exists(dirname($img_abs_path));
                    $move_upload_result = move_uploaded_file($file_image['tmp_name'], $img_abs_path);
                    
                    if ($move_upload_result)
                    {
                        //$datavalues['parent_dir'] = "/{$cid_prefix}";
                        //$datavalues['url_media_file'] = $img_rel_path;
                        $datavalues['url_media_file'] = $img_rel_path;
                        $datavalues['filename'] = $base_filename;
                        $datavalues['size'] = filesize($img_abs_path);
                        if (function_exists('exif_read_data'))
                        {
                            $ext = pathinfo($base_filename, PATHINFO_EXTENSION);
                            $datavalues['exif_data'] = NULL;
                            if ($ext == 'jpg' || $ext == 'jpeg')
                            {
                                $exif_arr_data = @exif_read_data($img_abs_path);
                                //suppress error, e.g.  Process tag(x2001=PreviewImag): Illegal components(0)
                                if ($exif_arr_data) 
                                    $datavalues['exif_data'] = @serialize($exif_arr_data);

                            }
                            if (in_array($ext,['png','jpg','jpeg']))
                                $result = $this->createMultipleImageSizesForOriginalImage($img_rel_path, $datavalues['category_id']);
                            
                        }

                    }
                }
              
            }
        } elseif($datavalues['image_file_url'])
        {
            $datavalues['url_media_file'] = $datavalues['image_file_url'];
            $datavalues['filename'] = '';
            $datavalues['size'] = 0;
        }
            
        
        return $datavalues;
    }    
    
    public function getImageSizeNames()
    {
        $image_sizes = [];  
        $ddir = $this->getDataDirectories();
        foreach ($ddir as $d => $v)
        {
            if (str_starts_with($d, 'image_'))
            {
            $name = str_replace('image_', '', $d);
            if ($name != 'original')
                $image_sizes[] = $name;
            }
        }
        return $image_sizes;
    }
    
    public function getDefaultSizes()
    {
        return $this->default_sizes;
    }
    public function getImageDataDirectoryNames()
    {
        $img_ddir = [];
        $ddir = $this->getDataDirectories();
        foreach ($ddir as $d => $v)
        {
            if (str_starts_with($d, 'image_'))
            {
                $img_ddir[] = $d;
            }
        }
        return $img_ddir;
    }
    /**
     * Ensure directory exists
     */
    public function ensureDataDirectoryExists()
    {
        $this->initializeDataDirectories();
        $this->initializeImageGalleryDirectories();
    }     
    
    /**
     * Creates all the folder as defined in $this->data_directory if not exists.
     * Returns a string array of errorlist, or an empty/null array if there's no error
     * @global \SCHLIX\cmsLogger $SystemLog
     * @return array
     */
    protected function initializeImageGalleryDirectories() {

        $error_list = array();
        $categories = $this->getAllCategories('*');
        $img_data_dirs = $this->getImageDataDirectoryNames();
        
        if (___c($categories) > 0)
        {
            foreach ($categories as $category) {
                
                foreach ($img_data_dirs as $dir_name)
                {
                    $error = null;
                    $cid_value = strval($category[$this->field_category_id]);
                    $dirpath = $this->getDataFileFullPath($dir_name, $cid_value);
                    if (empty($dirpath))
                        $error = sprintf( ___('Fatal error: empty directory path to create for category ID %s'), $cid_value);
                    else if (!is_dir($dirpath)) {
                        if (!create_directory_if_not_exists($dirpath)) 
                            $error = sprintf( ___('Fatal error: unable to create folder [%s] for category ID %s'), $dirpath, $cid_value);
                        else
                            $this->logInfo(sprintf( ___('New folder [%s] for category ID %s has been created'), $dirpath, $cid_value));
                        if (!is_writable($dirpath))
                            $error = sprintf( ___('Fatal error: Folder [%s] for category ID %s is not writable'), $dirpath, $cid_value);
                            
                    }
                    if ($error)
                    {
                        $this->logError($error);
                        $error_list[] = $error;
                    }
                    
                }
            }
        }
        return $error_list;
    }
    
        
    public function viewIndividualCategoryByID($id)
    {
        global $HTMLHeader;
        
        $category = $this->getCategoryByID($id);
        $items =  $this->getItemsByCategoryID($id, '*', 'status > 0 AND featured = 1', 0, 5, 'title', 'ASC');
        if (empty($items)) // no featured images
            $items=  $this->getItemsByCategoryID($id, '*', 'status > 0', 0, 5, 'title', 'ASC');
            
        $local_variables = compact(array_keys(get_defined_vars()));
        
        $this->loadTemplateFile('view.individual.category', $local_variables, false);      
    }
    
    
    public function getCacheableImages()
    {
        $all_imgs = [];
        $all_cats = $this->getAllCategories('cid', 'status > 0', 0, 100);
        foreach ($all_cats as $cat)
        {
            $cid = $cat['cid'];
            $cat_itms = $this->getAllItems('url_media_file', "status > 0 AND category_id = {$cid}", 0,20, 'ID', 'desc'); //$this->getAllItems();
            if ($cat_itms)
            {
                $cat_imgs = array_column($cat_itms, 'url_media_file');
                foreach ($cat_imgs as $img)
                {
                    $all_imgs[] =  encode_url($this->getDataFileURLPath('image_medium', $img));
                    //$all_imgs[] =  encode_url($this->getDataFileURLPath('image_large', $img));
                }
            }
        }
        $all_imgs = array_unique($all_imgs);
        return $all_imgs;
    }
    /**
     * Returns the default category ID 
     * @return int
     */
    public function getDefaultCategoryID() {
        return 0;
    }  

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. 
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */    
    
    public function getMainpageMetaOptionKeys() {
        return parent::getMainpageMetaOptionKeys();
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. In this base
     * class, the key is almost similar to getCategoryMetaOptionKeys
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */        
    public function getCategoryMetaOptionKeys() {
        return parent::getCategoryMetaOptionKeys();
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of item options.
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */
    public function getItemMetaOptionKeys() {
        return parent::getItemMetaOptionKeys();
    }
                
    /**
     * Create a small, medium, and large image from original file in the data directory
     * @param string $url_media_file
     * @param int $category_id
     * @return bool
     */
    public function createMultipleImageSizesForOriginalImage($url_media_file, $category_id) {

        $full_filename = $this->getDataFileFullPath('image_original', $url_media_file);
        $category = $this->getCategoryByID($category_id);
        $is_file = (!str_starts_with($url_media_file, 'https://') && !str_starts_with($url_media_file, 'http://'));
        if ($is_file && file_exists($full_filename) && ($category != null)) {
            
            $sizes = $this->getImageSizeNames(); // ['thumbnail','small','medium','large'];
            //$sizes = ['thumbnail','small','medium'];
            //$sizes = ['small'];
            $img_width = [];
            $img_height = [];
            $img_quality = [];
            $use_watermark = [];
            $watermark_img_path = [];
            
            $watermark_position = [];
            $watermark_margin_x = [];
            $watermark_margin_y = [];
            $watermark_opacity = [];
            $result = true;
            foreach ($sizes as $sz)
            {
                $ds = $this->default_sizes[$sz];
                $img_width[$sz] = $category["{$sz}_width"] ? $category["{$sz}_width"] :  $this->getConfig("int_{$sz}_width", $ds['width']);
                $img_height[$sz] = $category["{$sz}_height"] ? $category["{$sz}_height"] : $this->getConfig("int_{$sz}_height", $ds['height']);
                $img_quality[$sz] = $category["{$sz}_quality"] ? $category["{$sz}_quality"] : $this->getConfig("int_{$sz}_quality", $ds['quality']);
                
                $use_watermark[$sz] = $this->getConfig("bool_use_watermark_{$sz}");
                $watermark_position[$sz] = $this->getConfig("str_watermark_pos_{$sz}");
                $watermark_margin_x[$sz] = $this->getConfig("int_{$sz}_wm_margin_x");
                $watermark_margin_y[$sz] = $this->getConfig("int_{$sz}_wm_margin_y");
                $watermark_opacity[$sz] = $this->getConfig("int_wm_opacity_{$sz}");
                $watermark_img_path[$sz] = null;
                if ($use_watermark[$sz])
                {
                    $wm_img =  $this->getDataFileFullPath('watermark', $this->getConfig("str_watermark_image_{$sz}"));
                    if (file_exists($wm_img))
                        $watermark_img_path[$sz] = $this->getDataFileFullPath('watermark', $this->getConfig("str_watermark_image_{$sz}")); 
                        
                }
                $dst = $this->getDataFileFullPath("image_{$sz}", $url_media_file);
                
                $result_i = $this->createImageThumbnail($full_filename, $dst, 
                        $img_width[$sz], 
                        $img_height[$sz], false, 
                        $img_quality[$sz], 
                        $watermark_img_path[$sz], 
                        $watermark_opacity[$sz], 
                        $watermark_position[$sz], 
                        $watermark_margin_x[$sz], 
                        $watermark_margin_y[$sz]);
                $result &= $result_i;
                
                if (gc_enabled())
                    gc_collect_cycles();

                
            }
            return $result;
        } else {
            return false;
        }
    }    //_________________________________________________________________________//    

    //_________________________________________________________________________//
    /**
     * Create an image thumbnail. If dont_resize_if_smaller flag is specified and the source image is smaller than the specified
     * thumbnail width and height then it will just be copied. Quality is a number between 1 - 100
     * @param string $src
     * @param string $dest
     * @param int $thumb_width
     * @param int $thumb_height
     * @param bool $dont_resize_if_smaller
     * @param int $quality
     * @param string $watermark_img_path
     * @return boolean
     */
    function createThumbnailWithWatermark($src, $dest, $thumb_width = 120, $thumb_height = 120, $dont_resize_if_smaller = false, $quality = 100, $watermark_img_path = '', $waternark_opacity = 100, $watermark_pos = '', $watermark_margin_x = 0, $watermark_margin_y = 0) {
        $waternark_opacity = (int) $waternark_opacity;
        
            if ($waternark_opacity < 1)
                $waternark_opacity = 1;
            if ($waternark_opacity > 100)
                $waternark_opacity = 100;        
        
            
            $temporarily_raised_memory_limit = false;
            $original_memory_limit = get_estimated_php_memory_limit();

            $image_info = getimagesize($src);
            $img_width = $image_info[0];
            $img_height = $image_info[1];
            $img_mimetype = $image_info['mime'];
            $required_memory = round( ($img_height * $img_width * 3 * 2));

            if ($required_memory > $original_memory_limit)
            {

                $temporarily_raised_memory_limit = true;
                $raised_result = @ini_set('memory_limit', $required_memory);
                if (!$raised_result)
                    return false;
            }
            // check if it's animated gif and then return
            // hard code limit 3Mb for animated GIF
            if (is_animated_gif($src) && filesize($src) < 3000000) 
            {
                $anim_gif_resizer = new \SCHLIX\AnimatedGIFResizer($src, true);
                $result = $anim_gif_resizer->resize($dest, $thumb_width, $thumb_height, true, false);
                if ($temporarily_raised_memory_limit)
                    @ini_set('memory_limit', $original_memory_limit);
                return $result;

            }
            $use_watermark = false;
            //$watermark_img_path = '';
            if (!empty($watermark_img_path))
            {
                //die ($src.' to '.$dest);
                if (file_exists($watermark_img_path))
                {
                    $use_watermark = true;

                    $watermark = ImageCreateFromPNG($watermark_img_path);
                    $watermark_width = imagesx($watermark);
                    $watermark_height = imagesy($watermark);
                    $watermark_image = imagecreatetruecolor($watermark_width, $watermark_height);
                    
                    imageAlphaBlending($watermark_image, false);
                    imageSaveAlpha($watermark_image, true);
                    //$transparent = imagecolorallocatealpha($watermark_image, 255, 255, 255, 127);
                    //@imageFilledRectangle($watermark_image, 0, 0, $watermark_width, $watermark_height, $transparent);


                    
                } else throw new \Exception ("Specified watermark image  does not exist: {$watermark_img_path}");
            }
            switch ($img_mimetype) {
                case 'image/jpeg':$base_img = \ImageCreateFromJPEG($src);
                    break;
                case 'image/gif':$base_img = \ImageCreateFromGIF($src);
                    break;
                case 'image/png':$base_img = \ImageCreateFromPNG($src);
                    break;
                default:$base_img = null;
                    break;
            }
            
            if ($dont_resize_if_smaller && ($thumb_width >= $img_width) && ($thumb_height >= $img_height))
            {
                //echo "Copying {$src} to {$dest}";
                return copy($src, $dest);
            }
            

            $ext = pathinfo($src, PATHINFO_EXTENSION);
            $new_ext = pathinfo($dest, PATHINFO_EXTENSION);

            if ($base_img == null)
            {
                if ($temporarily_raised_memory_limit)
                    @ini_set('memory_limit', $original_memory_limit);
                return false; 
            }

            // Work out which way it needs to be resized
            $img_width_per = $thumb_width / $img_width;
            $img_height_per = $thumb_height / $img_height;

            if ($img_width_per <= $img_height_per || $img_width > $thumb_width) {
                $thumb_width = $thumb_width;
                $thumb_height = (int) ($img_height * $img_width_per);
            } else {
                $thumb_width = (int) ($img_width * $img_height_per);
                $thumb_height = $thumb_height;
            }
            if ($img_width <= $thumb_width && $img_height <= $thumb_height) {

                $thumb_height = $img_height;
                $thumb_width = $img_width;
            }
            if ($thumb_width == 0)
                $thumb_width++;
            if ($thumb_height == 0)
                $thumb_height++;
            $thumb_img = ImageCreateTrueColor($thumb_width, $thumb_height);
            if ($image_info[2] == IMAGETYPE_GIF || $image_info[2] == IMAGETYPE_PNG) {
                imageAlphaBlending($thumb_img, false);
                imageSaveAlpha($thumb_img, true);
                $transparent = imagecolorallocatealpha($thumb_img, 255, 255, 255, 127);
                @imageFilledRectangle($thumb_img, 0, 0, $thumb_width, $thumb_height, $transparent);
            }
            ImageCopyResampled($thumb_img, $base_img, 0, 0, 0, 0, $thumb_width, $thumb_height, $img_width, $img_height);
            if ($use_watermark)
            {
                $diff_width =  $thumb_width - $watermark_width;
                $diff_height =  $thumb_height - $watermark_height;
                if (($diff_width < 1) || ($diff_height < 1))
                    throw new \Exception("Watermark is too large");
                if (($diff_width - $watermark_margin_x < 1) || ($diff_height - $watermark_margin_y < 1))
                    throw new \Exception("Watermark margin is invalid");

                switch ($watermark_pos)
                {
                    case WM_POSITION_TOP_RIGHT:
                        // top right
                        $dest_x = $diff_width ;
                        $dest_y = 0;
                        break;
                    case WM_POSITION_RIGHT:
                        // right
                        $dest_x = $diff_width ;
                        $dest_y = ($diff_height / 2) ;
                        break;
                    case WM_POSITION_BOTTOM_RIGHT:
                        // bottom right
                        $dest_x = $diff_width ;
                        $dest_y = $diff_height;
                        break;
                    
                    case WM_POSITION_BOTTOM_LEFT:
                        $dest_x = 0 ;
                        $dest_y = $diff_height;
                        break;                    
                    case WM_POSITION_TOP_LEFT:
                        $dest_x = 0 ;
                        $dest_y = 0;
                        break;                    
                    case WM_POSITION_BOTTOM:
                        $dest_x = ($diff_width / 2);
                        $dest_y = $diff_height;
                        
                        break;                    
                    case WM_POSITION_LEFT:
                        $dest_x = 0 ;
                        $dest_y = ($diff_height / 2) ;                        
                        break;
                    
                    case WM_POSITION_TOP:
                        $dest_x = ($diff_width / 2);
                        $dest_y = 0;                        
                        break;      
                    case WM_POSITION_CENTER:
                    default:
                        // center
                        $dest_x = ($diff_width) / 2;
                        $dest_y = ($diff_height) / 2;
                        break;
                }
                $dest_x += $watermark_margin_x;
                $dest_y += $watermark_margin_y;
                imagecopy($thumb_img, $watermark, $dest_x, $dest_y, 0, 0, $watermark_width, $watermark_height);
                //imagecopymerge($thumb_img, $watermark, $dest_x, $dest_y, 0, 0, $watermark_width, $watermark_height, 30);

            }

            switch (strtolower($new_ext))
            {
                case 'png':
                    imageAlphaBlending($thumb_img, false);
                    imageSaveAlpha($thumb_img, true);
                    $result = ImagePNG($thumb_img, $dest, 8);
                    break;

                case 'jpeg':
                case 'jpg':$result = ImageJPEG($thumb_img, $dest, $quality);break;
                case 'gif':$result = ImageGIF($thumb_img, $dest);break;
            }
            // Clean up our images
            ImageDestroy($base_img);
            ImageDestroy($thumb_img);
            if ($use_watermark)
                imagedestroy($watermark_image);
            if ($temporarily_raised_memory_limit)
                @ini_set('memory_limit', $original_memory_limit);
            /*if (!$result)
                die ($src.' to '.$dest);*/
            return $result;
    }
    

    //_________________________________________________________________________//
    /**
     * Create an image thumbnail and raise memory limit if possible. If dont_resize_if_smaller flag is specified and the source image is smaller than the specified
     * thumbnail width and height then it will just be copied. Quality is a number between 1 - 100
     * @param string $src
     * @param string $dest
     * @param int $thumb_width
     * @param int $thumb_height
     * @param bool $dont_resize_if_smaller
     * @param int $quality
     * @param string $watermark_img_path
     * @param int $watermark_opacity
     * @param string $watermark_pos
     * @param int $watermark_margin_x
     * @param int $watermark_margin_y
     * @return bool
     * @throws \Exception
     */
     protected function createImageThumbnail($src, $dest, $thumb_width = 120, $thumb_height = 120, $dont_resize_if_smaller = false, $quality = 100, 
            $watermark_img_path = '', $watermark_opacity = 100, $watermark_pos = '', $watermark_margin_x = 0, $watermark_margin_y = 0) {
         
         
        //$dont_resize_if_smaller = false;
        //$real_img_watermark = !empty($watermark_img_path) ? $this->getDataFileFullPath('watermark', $watermark_img_path) : '';
//        $this->create_image_thumbnail($src, $dest, $thumb_width, $thumb_height, $dont_resize_if_smaller, $quality, $real_img_watermark, $watermark_pos, $watermark_margin_x, $watermark_margin_y);
        //echo $real_img_watermark;        
        $result = $this->createThumbnailWithWatermark($src, $dest, $thumb_width, $thumb_height, $dont_resize_if_smaller, $quality, $watermark_img_path, $watermark_opacity, $watermark_pos, $watermark_margin_x, $watermark_margin_y);
        
        if (file_exists($dest)) {
            $image_info = getimagesize($dest);
            $img_width = $image_info[0];
            $img_height = $image_info[1];
            $img_mimetype = $image_info['mime'];
            if ($img_mimetype == 'image/jpeg') {
                if (class_exists('Imagick'))
                {
                    $img = new \Imagick(realpath($dest));
                    $profiles = $img->getImageProfiles("icc", true);
                    $img->setImageResolution(72, 72);
                    $img->resampleImage(72, 72, \Imagick::FILTER_UNDEFINED, 1);
                    $img->stripImage();

                    if (!empty($profiles)) {
                        $img->profileImage("icc", $profiles['icc']);
                    }
                    $img->writeImage($dest);
                    $img->clear();
                    $img->destroy();
                }
            }
        } else throw new \Exception("File {$src} does not exist");
        return $result;
    }

    /**
     * Run command passed by the main router
     * @param array $command
     * @return bool
     */
    public function Run($command) {
        switch ($command['action']) {
            case 'viewitem':
                $this->viewItemByID($command['id'], $this->cache);
                break;
            case 'viewcategory': $this->viewCategoryByID($command['cid'], $command['pg'], 'date_created', 'DESC', $this->cache);
                break;
            case 'main': $this->viewMainPage($command['pg']);
                break;

            default: return parent::Run($command);
        }

        return RETURN_FULLPAGE;
    }
}
            
