<?php

namespace App;

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
// 
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//

class SiteManager_Admin extends \SCHLIX\cmsAdmin_List {

    protected $restricted_view_files = ['edit.item.hosts.template.php', 'edit.item.site.template.php', 'edit.item.template.php'];
    private $config_file;
    protected $current_error_reporting;
    protected  $error_levels = array(
            E_ERROR => 'E_ERROR',
            E_WARNING => 'E_WARNING',
            E_PARSE => 'E_PARSE',
            E_NOTICE => 'E_NOTICE',
            E_CORE_ERROR => 'E_CORE_ERROR',
            E_CORE_WARNING => 'E_CORE_WARNING',
            E_COMPILE_ERROR => 'E_COMPILE_ERROR',
            E_COMPILE_WARNING => 'E_COMPILE_WARNING',
            E_USER_ERROR => 'E_USER_ERROR',
            E_USER_WARNING => 'E_USER_WARNING',
            E_USER_NOTICE => 'E_USER_NOTICE',
            //E_STRICT => 'E_STRICT', // deprecated in PHP 8.4
            E_RECOVERABLE_ERROR => 'E_RECOVERABLE_ERROR',
            E_DEPRECATED => 'E_DEPRECATED',
            E_USER_DEPRECATED => 'E_USER_DEPRECATED'
        );
    

    protected $error_choices = array
    (
        array( 'value' => 'E_ERROR', 'label' => 'Error'), // E_ERROR
        array( 'value' => 'E_WARNING', 'label' => 'Warning'), // E_WARNING
        array( 'value' => 'E_PARSE', 'label' => 'Parse error'), // E_PARSE
        array( 'value' => 'E_NOTICE', 'label' => 'Notice'), // E_NOTICE
        array( 'value' => 'E_CORE_ERROR', 'label' => 'Core error'), // E_CORE_ERROR
        array( 'value' => 'E_CORE_WARNING', 'label' => 'Core warning'), // E_CORE_WARNING
        array( 'value' => 'E_COMPILE_ERROR', 'label' => 'Compile error'), // E_COMPILE_ERROR
        array( 'value' => 'E_COMPILE_WARNING', 'label' => 'Compiler warning'), // E_COMPILE_WARNING
        array( 'value' => 'E_USER_ERROR', 'label' => 'User-defined error'), // E_USER_ERROR
        array( 'value' => 'E_USER_WARNING', 'label' => 'User-defined warning'), // E_USER_WARNING
        array( 'value' => 'E_USER_NOTICE', 'label' => 'User-defined notice'), // E_USER_NOTICE
        // array( 'value' => 'E_STRICT', 'label' => 'Strict PHP'), // E_STRICT  // deprecated in PHP 8.4
        array( 'value' => 'E_RECOVERABLE_ERROR', 'label' => 'Recoverable Error'), // E_RECOVERABLE_ERROR
        array( 'value' => 'E_DEPRECATED', 'label' => 'Deprecated'), // E_DEPRECATED
        array( 'value' => 'E_USER_DEPRECATED', 'label' => 'User-defined deprecation') // E_USER_DEPRECATED
  );

//_________________________________________________________________________//
    public function __construct() {
        // Data: Item
        $this->config_file = SCHLIX_ROOT_PATH . '/multisite-config.inc.php';
        parent::__construct(true, []);
        
        $this->current_error_reporting = $this->getCurrentErrorReportingInArray();
        $this->setItemFieldNamesForAjaxListing('id', 'title', 'status', 'db_host','db_database','mail_default_sender','mail_default_email','system_timezone');
    }
    public function viewMainPage() {

        $this->checkMultisiteConfigFile();
        parent::viewMainPage();
    }
    
    private function getValidSitesFromWebFolder()
    {
        global $SchlixWebsites;
        
        clearstatcache(true);
        $trans_sites = [];
        foreach ($SchlixWebsites as $hostname => $sitename) {
            $trans_sites[$sitename][] = $hostname;
        }
        
        $site_folders = \SCHLIX\cmsDirectoryFilter::getDirectoryIterator(SCHLIX_ROOT_PATH . "/web/", \SCHLIX\cmsDirectoryFilter::FILTER_DIR_ONLY);
        foreach ($site_folders as $item)
        {
            $site_name = $item->getFileName();
            $config_exists =  file_exists($item->getPathName() . "/config.inc.php");
            $exists_in_multi_config = array_key_exists($site_name, $trans_sites);
            if ($config_exists && !$exists_in_multi_config)
            {
                $trans_sites[$site_name][] = $site_name;
            }
        }
        return $trans_sites;
    }

    private function syncMultisiteConfigFileToDatabase() {
        global $SchlixWebsites, $SystemDB;

        $trans_sites = $this->getValidSitesFromWebFolder();
        $SystemDB->query("TRUNCATE TABLE gk_sites_hosts");
        $SystemDB->query("TRUNCATE TABLE gk_sites_websites");

        $trans_sites_names = array_keys($trans_sites);
        $pos = 1;
        
        foreach ($trans_sites_names as $sitename) {
            
            $configvar = $this->parseSiteConfigFile($sitename);
            $datavalues['title'] = $sitename;
            $datavalues['db_host'] = $configvar['SCHLIX_DB_HOST'];
            $datavalues['db_database'] = $configvar['SCHLIX_DB_DATABASE'];            
            $datavalues['mail_default_sender'] =  $configvar['SCHLIX_MAIL_DEFAULT_SENDER'] ;
            $datavalues['mail_default_email'] =  $configvar['SCHLIX_MAIL_DEFAULT_EMAIL'];
            $datavalues['system_timezone'] =  $configvar['SCHLIX_SYSTEM_TIMEZONE'];
            $datavalues['status'] = ($configvar['SCHLIX_SITE_OFFLINE'] === 'false' ||  $configvar['SCHLIX_SITE_OFFLINE'] === '0');
            //$sanitized_sitename = sanitize_string($sitename);
            //$SystemDB->query("INSERT INTO gk_sites_websites (`title`,`db_host`) VALUES ({$sanitized_sitename})");
            $this->table_items->quickInsert($datavalues);
            $trans_site_hosts = $trans_sites[$sitename];
            foreach ($trans_site_hosts as $www) {
                $sanitized_www = sanitize_string($www);
                $SystemDB->query("INSERT INTO gk_sites_hosts (`hostname`,`category_id`) VALUES ({$sanitized_www}, $pos)");
            }
            $pos++;
        }
        $this->app->setConfig('md5config', md5_file($this->config_file));
    }
    
    private function syncMultisiteConfigDatabaseToFile() {
        global $SchlixWebsites, $SystemDB;

        
        $site_names = $SystemDB->getQueryResultArray("SELECT * FROM gk_sites_websites", FALSE);
        $hostnames = [];
        $trans_www = [];
        foreach ($site_names as $site_name)
        {
            $site_id = $site_name['id'];
            $site_www_addresses = $SystemDB->getQueryResultArray("SELECT * FROM gk_sites_hosts WHERE category_id = {$site_id}");
            foreach ($site_www_addresses as $www)
            {
                $trans_www[$www['hostname']] = $site_name['title'];
                $hostnames[] = $www['hostname'];
            }
        }
       
        $str_hostnames = ___c($hostnames) ? implode(', ', $hostnames) : 'None';
        $this->recordLog("Synced {$str_hostnames} to multisite-config.inc.php");
        $data = '<?php'."\n";
        $data.='// Automatically generated on '.  get_current_datetime()."\n\n";
        $data.= '/* 
 * Define multisites here. You **MUST EXPLICITLY** define all domains (including with or without www).
 * If you want to enable all sites pointing to a certain subsite, use the wildcard. Only one wildcard is allowed.*/'."\n\n";
        $data.='global $SchlixWebsites;'."\n\n";
        $data.='$SchlixWebsites ='.var_export($trans_www, TRUE).';';
        $result = file_put_contents(SCHLIX_ROOT_PATH.'/multisite-config.inc.php', $data);
        if (!$result)
        {
            display_schlix_alert(___('Please ensure that multisite-config.inc.php is writeable. This should not be an issue for PHP websites with SuExec enabled'));
        }
        clearstatcache(true, $this->config_file);
        //$SystemDB->query("DELETE FROM gk_sites_websites WHERE status = -3");
        
        //$site_hosts = $SystemDB->get_query_result("SELECT * FROM gk_sites_hosts", FALSE);
        
    }
    
    //_________________________________________________________//
    function execsql($sql, $newsqldb)
    {
        $newsqldb->query("SET SQL_MODE=''"); // fix most errors reported by microsoft tester

        foreach ($sql as $line)
        {
    ///		$line = str_replace('"/images','"'.$_SESSION['site_httpbase'].'/images',$line);
            if (strpos($line,'-- Table structure for table')!== false) echo \__HTML::P($line);
            // Skip it if it's a comment
            if (substr($line, 0, 2) == '--' || $line == '')
                continue;

            // Add this line to the current segment
            $templine .= $line;
            // If it has a semicolon at the end, it's the end of the query
            if (substr(trim($line), -1, 1) == ';')
            {
                // Perform the query

                $newsqldb->query($templine);
                // Reset temp variable to empty
                $templine = '';
            }
        }
    }
    
    /**
     * Create website skeleton
     * @param string $sitename
     * @return boolean
     */
    public function createSiteSkeleton($sitename)
    {
        return create_site_skeleton(SCHLIX_ROOT_PATH, $sitename);
    }

    function parseSiteConfigFile($sitename) {
        $defines = [];
        $state = 0;
        $key = '';
        $value = '';

        $config_filename = SCHLIX_ROOT_PATH . '/web/' . $sitename . '/config.inc.php';
        if (!file_exists($config_filename))
            return [];
        $file = file_get_contents($config_filename);
        $tokens = token_get_all($file);
        $token = reset($tokens);
        while ($token) {
            if (is_array($token)) {
                if ($token[0] == T_WHITESPACE || $token[0] == T_COMMENT || $token[0] == T_DOC_COMMENT) {
                } else if ($token[0] == T_STRING && strtolower($token[1]) == 'define') {
                    $state = 1;
                } else if ($state == 2 && $this->isConstant($token[0])) {
                    $key = $token[1];
                    $state = 3;
                } else if ($state == 4 && $this->isConstant($token[0])) {
                    $value = $token[1];
                    $state = 5;
                }
            } else {
                $symbol = trim($token);
                if ($symbol == '(' && $state == 1) {
                    $state = 2;
                } else if ($symbol == ',' && $state == 3) {
                    $state = 4;
                } else if ($symbol == ')' && $state == 5) {
                    $defines[$this->stripStuff($key)] = stripcslashes($this->stripStuff($value)); // $this->replaceSlash($this->stripStuff($value));
                    $state = 0;
                }
            }
            $token = next($tokens);
        }
        return $defines;
    }

    private function isConstant($token) {
        return ($token == T_CONSTANT_ENCAPSED_STRING) || ($token == T_STRING) || ($token == T_LNUMBER) || ($token == T_DNUMBER);
    }

    private function replaceSlash($str) 
    {
        return str_replace("\'","'", $str);
    }
            
    private function stripStuff($value) {
        return preg_replace('!^([\'"])(.*)\1$!', '$2', $value);
    }

    /**
     * Check if multisite config has been changed
     */
    private function checkMultisiteConfigFile() {
        $md5_of_currentconfig = md5_file($this->config_file);

        if ($md5_of_currentconfig !== FALSE) {
            $md5_of_prevconfig = $this->app->getConfig('md5config');
            if ($md5_of_currentconfig != $md5_of_prevconfig || $_SESSION['refresh_multi_config'] == TRUE) {
                $this->recordLog('multisite-config.inc.php must be refreshed');
                $this->syncMultisiteConfigFileToDatabase();
                $_SESSION['refresh_multi_config'] = FALSE;
                
            }
        } else
            die(___('Cannot read multisite-config.inc.php'));
    }
    
    protected function ajxg_forceRefresh()
    {
        $this->syncMultisiteConfigFileToDatabase();
        return ajax_reply(200, 'OK');
    }
    
    
    protected function ajxp_TestDatabase()
    {
        check_csrf_halt_on_error();
        $success_message = '';
        $error_rep = error_reporting();
        error_reporting(0);
        //ini_set("display_errors",0);
        $db_host = fpost_string('host',255);
        $db_username = fpost_string('username',255);
        $db_password = fpost_string('password',255);  
        $db_database =fpost_string('database',255);  
        $db_site_id =fpost_int('siteid');          
        
        // new - sept 2019
        $db_port = fpost_int('db_port');
        $db_socket = fpost_string('db_socket');
        $db_use_ssl = fpost_int('db_use_ssl');
        $db_ssl_ca = fpost_string('db_ssl_ca');        
        try 
        {
            if (empty($db_host))
                throw new \Exception (___('Host name cannot be empty'));
            if (empty($db_username))
                throw new \Exception (___('Username cannot be empty'));
            if (empty($db_password))
                throw new \Exception (___('Password cannot be empty'));
            if (empty($db_database))
                throw new \Exception (___('Database name cannot be empty'));
            if ($db_port < 0 || $db_port > 65535)
                throw new \Exception (___('Invalid database port number'));           
            if (!empty($db_ssl_ca) && !file_exists($db_ssl_ca))
                throw new \Exception (___('SSL Certificate Authority file cannot be found'));            
            if ($this->siteDatabaseHasAlreadyBeenUsed($db_host, $db_database, $db_site_id))
                throw new \Exception (___('You cannot reuse a database from another subsite!'));
            if (($db_host === 'localhost' || $db_host === '127.0.0.1') && $db_use_ssl > 0)
                throw new \Exception (___('SSL is not required for database server in localhost/127.0.0.1'));
            if ($db_use_ssl == 1)
            {
                $link = mysqli_init();
                if ($db_ssl_ca)
                    mysqli_ssl_set($link,NULL,NULL, $db_ssl_ca, NULL, NULL); 
                $opt = MYSQLI_CLIENT_SSL | MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT;
                @mysqli_real_connect($link, $db_host, $db_username, $db_password, NULL, $db_port, $db_socket, $opt);                    
            } else
                $link = @mysqli_connect($db_host, $db_username, $db_password, NULL, $db_port, $db_socket);
            
            
            if ($link)
            {
                $sql = 'SHOW DATABASES LIKE '.  sanitize_string($db_database);
                $result = [];
                $last_query_result = mysqli_query($link, $sql);
                while ($row = mysqli_fetch_assoc($last_query_result))
                    $result[] = $row;
                mysqli_free_result($last_query_result);
                mysqli_close($link);    
                if (___c($result) > 0)
                {
                    $db = array_values($result[0])[0];
                    $success_message = sprintf(___('Connected successfully to database [%s]'), $db);
                } else
                {
                    $success_message = ___('Connected successfully');
                }
            } else
            {
                throw new \Exception (___('Cannot connect to the database'));
            }
        } catch (\Exception $exc)
        {
             return ajax_reply(300,$exc->getMessage());
        }
        
        return ajax_reply(200, $success_message);
    }
    

    protected function ajxp_UpdateHost() {
        $hostm = new sitemanager_hosts_Admin();

        return $hostm->ajaxUpdateField();
    }

    protected function ajxp_deleteHost() {
        $hostm = new sitemanager_hosts_Admin();

        return $hostm->ajaxDeleteObjects($_POST['items']);
    }
    
    private function createNewDatabase( $db_database, $db_host, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca)
    {
        $db_result = false;
        $filter_host = filter_var($db_host, FILTER_VALIDATE_DOMAIN, FILTER_FLAG_HOSTNAME);
        if (!$filter_host)
            $db_host = 'localhost';
        $db_database = convert_to_safe_filename($db_database);
        $db_username = convert_to_safe_filename($db_username);
        if ($db_use_ssl == 1)
        {
            $link = mysqli_init();
            if ($db_ssl_ca)
                mysqli_ssl_set($link,NULL,NULL, $db_ssl_ca, NULL, NULL); 
            $opt = MYSQLI_CLIENT_SSL | MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT;
            @mysqli_real_connect($link, $db_host, $db_username, $db_password, NULL, $db_port, $db_socket, $opt);
        } else
            $link = @mysqli_connect($db_host, $db_username, $db_password, NULL, $db_port, $db_socket);                       
        if ($link)
        {
            // check if database already exists
            $sql = 'SHOW DATABASES LIKE '.  sanitize_string($db_database);
            $result = [];
            $last_query_result = mysqli_query($link, $sql);
            while ($row = mysqli_fetch_assoc($last_query_result))
                $result[] = $row;
            mysqli_free_result($last_query_result);
            if ($result)
            {
                $db = array_values($result[0])[0];
                if (strlen($db) > 0)
                $db_result = true;
            }
            else
            {
                $sql = 'CREATE DATABASE `' . $db_database. '` DEFAULT CHARACTER SET = utf8mb4 DEFAULT COLLATE utf8mb4_unicode_ci;';
                $last_query_result = mysqli_query($link, $sql);
                $db_result = (!$last_query_result) ? false: true;                    
            }
        }
        
        if ($link)
            mysqli_close($link);    
    
        return $db_result;
    }

    protected  function runNewSiteSQLInstallScript($short_site_name, $db_database, $db_host, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca)
    {
        $NewSQLDB = new \SCHLIX\cmsDatabase($db_host, $db_database, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca);
        global $SystemLog, $CurrentUser;
        
        $result = true;
        
        $file_install_sql_step1 = __DIR__.'/sql/install_step1.sql';
        $file_install_sql_step2 = __DIR__.'/sql/install_step2.sql';
        $file_install_sql_empty = __DIR__.'/sql/install_sample_empty.sql';
        if (!file_exists($file_install_sql_step1) || !file_exists($file_install_sql_step2)) 
            return false;
        else {
            $install_sql = file($file_install_sql_step1);
            $install_sql_step2 = file($file_install_sql_step2);
            $install_sql_empty = file($file_install_sql_empty);
        }

        $this->execsql($install_sql, $NewSQLDB);
        $this->execsql($install_sql_step2, $NewSQLDB); // separate it. Install.SQL is run on upgrade
        
        $current_user_info = $CurrentUser->getCurrentUserInfo();
        $salt = ''; //$CurrentUser->generateRandomPasswordSalt('blowfish');
        $hashed_password = $current_user_info['password']; //password_hash($current_user_info['password'], PASSWORD_DEFAULT);
        //crypt($current_user_info['password'], $salt);
        $sanitized_hashed_password = sanitize_string($hashed_password);
        $sanitized_email = sanitize_string($current_user_info['email_address']);
        $NewSQLDB->query("UPDATE gk_user_items set `password` = {$sanitized_hashed_password}, salt='{$salt}', email_address = {$sanitized_email} WHERE id = 1"); // change password, add salt
        

        // Replace images path
        $new_images_path = SCHLIX_SITE_HTTPBASE.'/web/'.$short_site_name.'/media/images/';
        $new_images_replacement = 'src="'.$new_images_path;
        $install_sql_empty = str_replace('src="system/images/',$new_images_replacement, $install_sql_empty);
        $this->execsql($install_sql_empty, $NewSQLDB); // separate it. Install.SQL is run on upgrade

            
        $NewSQLDB->query("update `gk_menu_items` set sefurl = concat('" . SCHLIX_SITE_HTTPBASE . "',sefurl) where id <> 11");
        $hbase = fsession_string('site_httpbase');
        if ($hbase != '') // Dec 12, 2011
            $NewSQLDB->query("update `gk_menu_items` set sefurl = '" . SCHLIX_SITE_HTTPBASE . "' where id = 1");
        
        $NewSQLDB->query("update `gk_contact_items` set email = {$sanitized_email}");
        
        return true;
        
 
    }
    /**
     * Modify data before save item
     * @param array $datavalues
     * @return array
     */
    public function onModifyDataBeforeSaveItem($datavalues) {
        
        
        $datavalues = parent::onModifyDataBeforeSaveItem($datavalues);
        $datavalues['title'] = strtolower(convert_into_sef_friendly_title($datavalues['title']));
        // what happens if a user tries to overwrite the current site
        if ($datavalues['id'] != $this->getCurrentSiteID())
        {
            if ($datavalues['title'] == CURRENT_SUBSITE_NAME)
                $datavalues['title'].= '1';
        }
        
        // TODO: real fix, this is just a temporary fix
        if ($this->siteNameExists($datavalues['title'], $datavalues['id']))
            $datavalues['title'].=mt_rand(10, 100);
        
        return $datavalues;
    }    
   
    /**
     * Validation
     * @param array $datavalues
     * @return array
     */
    public function onGetAdminValidationErrorListBeforeSaveItem($datavalues) 
    {
        $error_list = [];
        
        
        if (empty($datavalues['title']))
            $error_list[] = ___('Subsite name must be specified');
        if (empty($datavalues['mail_default_sender']))
            $error_list[] = ___('Mail sender cannot be empty');
        if (!is_valid_email_address($datavalues['mail_default_email'], true ))
            $error_list[] = ___('Invalid email address');      
        $smtp_host = $datavalues['smtp_host'];
        $smtp_port = (int) $datavalues['smtp_port'];
        $smtp_secure_with = $datavalues['smtp_secure_with'];
        $smtp_use_auth = $datavalues['smtp_use_auth'];
        $smtp_username = $datavalues['smtp_username'];
        $smtp_password = $datavalues['smtp_password']; 
        
        if (!empty($smtp_host))
        {
            $valid_host = is_valid_hostname($smtp_host);
            if ($valid_host)
            {
                if ( ($smtp_port < 1) || ($smtp_port > 65535) )
                    $error_list[] = ___('Port must be between 1 and 65535');
                if (!empty($smtp_secure_with) && ($smtp_secure_with != 'tls') && ($smtp_secure_with != 'ssl'))
                    $error_list[] = ___('SMTP security can only be SSL or TLS');
                if ($smtp_use_auth)
                {
                    if (empty($smtp_username))
                        $error_list[] = ___('SMTP username must be specified if authentication is required');
                    /*if (empty($smtp_password))
                        $error_list[] = ___('SMTP password must be specified if authentication is required'); */
                } else 
                {
                    // no need to blank out username/password
                }
                
            } else $error_list[] = ___('Invalid SMTP host name');
        }        
        $current_site_id = $this->getCurrentSiteID();
        if ($datavalues['id'] != $current_site_id)
        {
            if (strtolower(CURRENT_SUBSITE_NAME) == strtolower($datavalues['title']))
                $error_list [] = ___('You cannot create a site with the same name as the current one').': '.CURRENT_SUBSITE_NAME;
            if ($this->siteDatabaseHasAlreadyBeenUsed($datavalues['db_host'], $datavalues['db_database'], $datavalues['id']))
                $error_list [] = ___('This database has already been used by another subsite on the same database host').': '.$datavalues['db_database'];
        }
        
        return $error_list;
    }
    
    /**
     * Returns true if there's a site registered with the same but not with this id
     * @param string $site_name
     * @param int $id
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @return int
     */
    public function siteNameExists($site_name, $id)
    {
        global $SystemDB;
        
        $sanitized_site_name= sanitize_string($site_name);
        $sanitized_id = sanitize_string((int) $id );
        $sql = "SELECT 1 FROM {$this->table_items} WHERE `title` = {$sanitized_site_name} AND id <> {$sanitized_id}";
        $row = $SystemDB->getQueryResultSingleRow($sql);
        return $row != NULL;
    }
    
    /**
     * Returns true if there's a site registered with the same but not with this id
     * @param string $db_host
     * @param string $db_database
     * @param int $id
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @return int
     */
    public function siteDatabaseHasAlreadyBeenUsed($db_host, $db_database, $id)
    {
        global $SystemDB;
        
        $sanitized_db_host = sanitize_string($db_host);
        $sanitized_db_database = sanitize_string($db_database);
        $sanitized_id = sanitize_string((int) $id );
        $sql = "SELECT 1 FROM {$this->table_items} WHERE `db_host` = {$sanitized_db_host } AND  `db_database` = {$sanitized_db_database} AND  id <> {$sanitized_id}";
        $row = $SystemDB->getQueryResultSingleRow($sql);
        return $row != NULL;
    }
    
    
    /**
     * Return the current site ID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @return int
     */
    public function getCurrentSiteID()
    {
        global $SystemDB;
        
        $sanitized_site_name= sanitize_string(CURRENT_SUBSITE_NAME);
        $sql = "SELECT * FROM {$this->table_items} WHERE `title` = {$sanitized_site_name}";
        $row = $SystemDB->getQueryResultSingleRow($sql);
        return $row['id'];
    }

    /**
     * Write config once item has been saved
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param array $datavalues
     * @param array $original_datavalues
     * @param array $previous_item
     * @param array $retval
     * @return array
     */
    public function onAfterSaveItem($datavalues, $original_datavalues, $previous_item, $retval) {
        
        global $SystemLog;
        
        parent::onAfterSaveItem($datavalues, $original_datavalues, $previous_item, $retval);
        if ($retval['status'] == SAVE_OK)
        {
            //////////////////////////////
            $current_site_id = $this->getCurrentSiteID();
            $is_editing_current_site = ((int) $retval['id'] === (int) $current_site_id);
            
            
            $site_folder = convert_into_sef_friendly_title($datavalues['title'], false);
            $config_file_location = SCHLIX_ROOT_PATH . '/web/' . $site_folder . '/config.inc.php';
            $old_config_vars = $this->parseSiteConfigFile($site_folder);

            $is_brand_new_site = !file_exists($config_file_location);

            $site_http_url = fpost_string('site_http_url');
            $site_https_url = fpost_string('site_https_url');
            $site_name = add_single_quote_slashes(strip_tags(fpost_string('site_name')));
            // Clean variables first
            $default_user_class = convert_into_sef_friendly_title(fpost_string('default_user_class'));
            //$default_xmlrpc_class = convert_into_sef_friendly_title(fpost_string('default_xmlrpc_class'));
            //$default_comment_class = convert_into_sef_friendly_title(fpost_string('default_comment_class'));
            $db_host = null;
            $db_username = null;
            $db_password = null;
            $db_database = null;
            // new - sept 2019
            $db_port = 0;
            $db_socket = null;
            $db_use_ssl = false;
            $db_ssl_ca = '';
            if ($is_brand_new_site || !$is_editing_current_site)
            {
                $db_host = add_single_quote_slashes(fpost_string('db_host'));
                $db_username = add_single_quote_slashes(fpost_string('db_username'));
                $db_password = add_single_quote_slashes(fpost_string('db_password'));
                $db_database = add_single_quote_slashes(fpost_string('db_database'));
                $db_port = fpost_int('db_port');
                $db_socket =  add_single_quote_slashes(fpost_string('db_socket'));
                $db_use_ssl = fpost_int('db_use_ssl');
                $db_ssl_ca = add_single_quote_slashes(fpost_string('db_socket'));
            } else 
            {
                // else don't change it
                $db_host = add_single_quote_slashes(SCHLIX_DB_HOST);
                $db_username = add_single_quote_slashes(SCHLIX_DB_USERNAME);
                $db_password = add_single_quote_slashes(SCHLIX_DB_PASSWORD);
                $db_database = add_single_quote_slashes(SCHLIX_DB_DATABASE);                    
                $db_port = defined('SCHLIX_DB_PORT') ? (int)  SCHLIX_DB_PORT : '0';
                $db_socket = defined('SCHLIX_DB_SOCKET') ? add_single_quote_slashes(SCHLIX_DB_SOCKET) : '';
                $db_use_ssl = defined('SCHLIX_DB_USE_SSL') ? (int)  SCHLIX_DB_USE_SSL : 'false';
                $db_ssl_ca =  defined('SCHLIX_DB_SSL_CA') ? add_single_quote_slashes(SCHLIX_DB_SSL_CA) : '';
            }

            $smtp_host = add_single_quote_slashes(fpost_string('smtp_host'));
            $smtp_username = add_single_quote_slashes(fpost_string('smtp_username'));
            $smtp_password = add_single_quote_slashes(fpost_string('smtp_password'));
            $smtp_port = fpost_int('smtp_port');
            if ($smtp_port == 0)
                $smtp_port = '25';
            
            $smtp_oauth2_provider = add_single_quote_slashes(fpost_string('mail_oauth2_provider'));
            $smtp_oauth2_client_id = add_single_quote_slashes(fpost_string('mail_oauth2_client_id'));
            $smtp_oauth2_secret = add_single_quote_slashes(fpost_string('mail_oauth2_secret'));
            $smtp_oauth2_refresh_token = add_single_quote_slashes(fpost_string('mail_oauth2_refresh_token'));

            //$site_meta_key = add_single_quote_slashes(___h(fpost_string('site_meta_key')));
            //$site_meta_description = add_single_quote_slashes(___h(fpost_string('site_meta_description')));
            $mail_default_sender = add_single_quote_slashes(fpost_string('mail_default_sender'));
            $mail_default_email = add_single_quote_slashes(fpost_string('mail_default_email'));
            $site_httpbase = add_single_quote_slashes(fpost_string('site_httpbase'));
            $system_timezone = add_single_quote_slashes(strip_tags(fpost_string('system_timezone')));
            $smtp_secure_with = (in_array(fpost_string('smtp_secure_with'), array('ssl','tls'))) ? fpost_string('smtp_secure_with') : '';
            $smtp_security_allow_self_signed_cert = fpost_bool('smtp_security_self_signed') ? 'true' : 'false' ;
            
            $sql_cache_time = strval(fpost_int('sql_cache_time'));
            $default_admin_group = add_single_quote_slashes(fpost_string('default_admin_group'));
            $admin_login_time = fpost_int('admin_login_time');
            $remember_me_cookie_time = fpost_int('remember_me_cookie_time');
            $site_offline_status = (fpost_string('site_offline')) ? 'true' : 'false';
            $sef_enabled_status = (fpost_string('sef_enabled')) ? 'true' : 'false';
            $site_ssl_enabled = (fpost_string('ssl_enabled') || isCurrentRequestSSL()) ? 'true' : 'false';
            $site_force_ssl_admin_enabled = (fpost_string('force_ssl_admin_login')) ? 'true' : 'false';
            $site_enable_page_cache = (fpost_string('enable_page_cache')) ? 'true' : 'false'; // TODO
            $site_enable_sql_cache = (fpost_string('enable_sql_cache')) ? 'true' : 'false'; // TODO
            $site_enforce_sql_row_limit = (fpost_string('enforce_sql_row_limit')) ? 'true' : 'false';
            $schlix_js_mode = fpost_int('schlix_js_mode');
            $record_404_errors = fpost_int('record_404_errors');
            $display_errors = fpost_int('display_errors') == 1 ? '1' :'0';
            $samesite_cookie_mode = fpost_alphanumeric('samesite_cookie_mode', 7);
            $error_report = $this->getErrorReportingAsString(fpost_array_any('error_report'));
            
            if ($admin_login_time <= 0)
                $admin_login_time = 7200; // 2 hours
            if ($remember_me_cookie_time <= 0)
                $remember_me_cookie_time = 172800; // 2 days
            
            //  All settings have been validated
            $data = '<?php' . "\n";
            $data.= "/* generated by SCHLIX CMS Site Manager ~ " . get_current_datetime() . " */\n";
            $data.="define('SCHLIX_SITE_NAME','{$site_name}');\n";
            
            if (empty ($site_http_url)  || !filter_var($_POST['site_http_url'], FILTER_VALIDATE_URL ))
            {
                $proto =  (isCurrentRequestSSL()) ? 'https' : 'http';
                $data.="define('SCHLIX_SITE_HTTP_URL','{$proto}://'.\$_SERVER['HTTP_HOST']); /* Automatic URL. Beware of DNS spoofing if you have wildcard host set in multisite-config.inc.php - must define each site manually. */ \n";
            } else 
            {
                $data.="define('SCHLIX_SITE_HTTP_URL','{$site_http_url}'); /* Format: http://sitename.com .. if port number = 80 then ignore the number */ \n";
            }            
               //''://' . $_SERVER['HTTP_HOST'] . (($_SERVER['SERVER_PORT'] != 80 && $_SERVER['SERVER_PORT'] != 443) ? ':' . $_SERVER['SERVER_PORT'] : '''
               // $data.="define('SCHLIX_SITE_HTTP_URL','http://'.\$_SERVER['HTTP_HOST'] . ((\$_SERVER['SERVER_PORT'] != 80 && \$_SERVER['SERVER_PORT'] != 443) ? ':' . \$_SERVER['SERVER_PORT'] : '')); /* Automatic URL */ \n";
               
            // https
            if (empty ($site_https_url)  || !filter_var($_POST['site_https_url'], FILTER_VALIDATE_URL ))
            {
                // logic is actually incorrect, need another variable - always force SSL (dec 2019)
                if (isCurrentRequestSSL())
                    $data.="define('SCHLIX_SITE_HTTPS_URL','https://'.\$_SERVER['HTTP_HOST']); /* Format: https://sitename.com .. if port number = 443 then ignore the number */\n";
                else 
                    $data.="define('SCHLIX_SITE_HTTPS_URL','');";
            }
            else 
                $data.="define('SCHLIX_SITE_HTTPS_URL','{$site_https_url}'); /* Format: https://sitename.com .. if port number = 443 then ignore the number */\n";


            // if (!empty($_SERVER['WINDIR']))
            // removed because the user may move from win/linux and vice versa.
            $data.="define('SCHLIX_SITE_PATH',str_replace('\\\','/',dirname(__FILE__)));\n";
            //else
    //            $data.="define('SCHLIX_SITE_PATH',dirname(__FILE__));\n";
            //	$data.="define('SITE_TEMPLATE','{$_POST['site_template']}');\n";
            //$data.="define('SCHLIX_SITE_HTTPBASE','{$site_httpbase}');\n";
            $data.="define('SCHLIX_SITE_OFFLINE',{$site_offline_status});\n";
            $data.="define('SCHLIX_SEF_ENABLED', true);  /* Search Engine Friendly URL  */\n";

            $data.="define('SCHLIX_SITE_SSL_ENABLED', {$site_ssl_enabled}); /* Please ensure that you have a valid SSL certificate if you want to enable SSL */ \n";            
            $data.="define('SCHLIX_FORCE_SSL_ADMIN_LOGIN', {$site_force_ssl_admin_enabled}); /* For backend login. Please ensure that you have a valid SSL certificate or this will result in an error */ \n";

            $data.="define('SCHLIX_DEFAULT_USER_CLASS','{$default_user_class}'); /* Default user class - e.g: LDAP, smf forum, opencart, etc */\n";
    //            $data.="define('DEFAULT_XMLRPC_CLASS','{$default_xmlrpc_class}');\n";   //TODO
    //            $data.="define('DEFAULT_COMMENT_CLASS','{$default_comment_class}');\n";  //TODO

            $data.="define('SCHLIX_DEFAULT_LANGUAGE','en_us');\n";
            //$data.="define('SCHLIX_ADMIN_LANGUAGE','en_us');\n";
            $data.="define('SCHLIX_DB_HOST','{$db_host}'); /* database host, e.g. localhost */\n";
            $data.="define('SCHLIX_DB_DATABASE','{$db_database}'); /* database name */\n";
            $data.="define('SCHLIX_DB_USERNAME','{$db_username}'); /* database username */\n";
            $data.="define('SCHLIX_DB_PASSWORD','{$db_password}'); /* database password */\n";
            
            $data.="///////// advanced db param - leave blank for default /////////\n";
            $data.="define('SCHLIX_DB_PORT',{$db_port}); /* if the value is zero then it will use the default port number, usually 3306 for mysql */ \n";
            $data.="define('SCHLIX_DB_SOCKET','{$db_socket}'); /* if the value is blank then it will the default value */\n";
            $data.="define('SCHLIX_DB_USE_SSL',{$db_use_ssl}); /* set this to true if MySQL uses SSL connection */\n";
            $data.="define('SCHLIX_DB_SSL_CA','{$db_ssl_ca}'); /* specify the SSL CA certificate file if SCHLIX_DB_USE_SSL is set to true */\n";
            $data.="///////// end advanced db param  /////////";
            
            $data.="define('SCHLIX_PAGE_CACHE_ENABLED', {$site_enable_page_cache});\n"; //TODO
            $data.="define('SCHLIX_SQL_CACHE_ENABLED', {$site_enable_sql_cache});\n";
            $data.="define('SCHLIX_SQL_ENFORCE_ROW_LIMIT', {$site_enforce_sql_row_limit});\n";

            $data.="define('SCHLIX_SQL_CACHE_TIME', {$sql_cache_time});\n";
            
            $data.="define('SCHLIX_DEFAULT_ADMIN_GROUP','{$default_admin_group}');\n";
            $data.="define('SCHLIX_SESSION_LOGIN_TIME',{$admin_login_time});\n";
            $data.="define('SCHLIX_SESSION_REMEMBER_COOKIE_TIME', {$remember_me_cookie_time});\n";
            $data.="define('SCHLIX_COOKIE_SAMESITE', '{$samesite_cookie_mode}'); /* Samesite=none cookie for HTTPS by default as per new standard in 2020 */ \n";            
            
            $data.="define('SCHLIX_ADMIN_THEME','admindefault');\n";
            $data.="define('SCHLIX_USER_IMAGE_DIRECTORY',SCHLIX_SITE_PATH.'/media/images');\n";
            /*if (empty($_POST['site_meta_description']))
                $site_meta_description = $site_name;*/

            $data.="define('SCHLIX_MAIL_DEFAULT_SENDER','{$mail_default_sender}'); /* Default sender name, e.g. Webmaster */\n";
            $data.="define('SCHLIX_MAIL_DEFAULT_EMAIL','{$mail_default_email}'); /* Default email address for sending out an email if none specified */\n";
            
            $data.="define('SCHLIX_SMTP_HOST', '{$smtp_host}'); /* If you use an external SMTP server, fill in the server host here */\n";
            $data.="define('SCHLIX_SMTP_PORT', {$smtp_port}); /* Some common values: 25, 587 (TLS), 465 (SSL) */\n";
            $data.= ($_POST['smtp_use_auth']) ? "define('SCHLIX_SMTP_USE_AUTH', 1);" : "define('SCHLIX_SMTP_USE_AUTH', false);";
            $data.= " /* Specify whether an SMTP authentication is required */\n";
            $data.="define('SCHLIX_SMTP_USERNAME', '{$smtp_username}'); /* Fill in the SMTP username if SMTP authentication is required */ \n";
            $data.="define('SCHLIX_SMTP_PASSWORD', '{$smtp_password}'); /* Fill in the password for the SMTP user if SMTP authentication is required */\n";
            $data.="define('SCHLIX_SMTP_SECURITY', '{$smtp_secure_with}'); /* possible values: tls (port 587), ssl (port 465) or simply blank (port 25) */\n";
            $data.="define('SCHLIX_SMTP_SECURITY_SELF_SIGNED', {$smtp_security_allow_self_signed_cert}); /* true/false value - allow self signed certificate */\n";
            
            $data.=" /* Oauth2 */\n";
            $data.="define('SCHLIX_SMTP_OAUTH2_PROVIDER', '{$smtp_oauth2_provider}'); /* OAUTH2 providers - google, microsoft, yahoo */ \n";
            $data.="define('SCHLIX_SMTP_OAUTH2_CLIENT_ID', '{$smtp_oauth2_client_id}'); /* OAUTH2 Client/Application ID */ \n";
            $data.="define('SCHLIX_SMTP_OAUTH2_SECRET', '{$smtp_oauth2_secret}'); /* OAUTH2 secret */ \n";
            $data.="define('SCHLIX_SMTP_OAUTH2_REFRESH_TOKEN', '{$smtp_oauth2_refresh_token}'); /* OAUTH2 offline refresh token */ \n";
            $data.=" /* timezone */\n";
            $data.="define('SCHLIX_SYSTEM_TIMEZONE', '{$system_timezone}');\n";
            //Deprecated as of September 1, 2017
            //$data.="define('SCHLIX_SITE_META_KEYWORDS','{$site_meta_key}'); /* Default meta keywords if there is none defined in the app/item/category */\n";
            //$data.="define('SCHLIX_SITE_META_DESCRIPTION','{$site_meta_description}'); /* Default meta keywords if there is none defined in the app/item/category */\n";
            //$data.="define('SCHLIX_JS_MODE','{$schlix_js_mode}'); /* Javascript mode. 1 = Debug, 2 = Minified, 3 = Minified and Gzipped */ \n";
            $data.="define('SCHLIX_JS_MODE','3'); /* Javascript mode. 1 = Debug, 2 = Minified, 3 = Minified and Gzipped */ \n";
            $data.="define('SCHLIX_RECORD_404_ERRORS','{$record_404_errors}'); /* Record 404 errors in the database */ \n";

                $data.="/**********************************************************************/\n";
                $data.="/* Do not modify the SCHLIX_SITE_URL - this is for auto http/https switch */\n";
                $data.='define(\'SCHLIX_SITE_URL\', (defined(\'SCHLIX_SITE_HTTPS_URL\') && SCHLIX_SITE_SSL_ENABLED && SCHLIX_SITE_HTTPS_URL !=\'\' && (isset($_SERVER[\'HTTPS\']) && (($_SERVER[\'HTTPS\']===\'on\') ||  ($_SERVER[\'HTTPS\']===1))  || $_SERVER[\'SERVER_PORT\']===443)) ? SCHLIX_SITE_HTTPS_URL : SCHLIX_SITE_HTTP_URL);';
                $data.="\n/**********************************************************************/\n";

            $data.="error_reporting({$error_report});\n";
            $data.="@ini_set('display_errors',$display_errors);\n";

            if ($is_brand_new_site)
            {
                $this->recordLog("Created a brand new site under /web/{$site_folder}");
                
                $this->createSiteSkeleton($site_folder);
                $config_save_result = file_put_contents($config_file_location, $data);
                if (!$config_save_result)
                    die(___('Cannot save to configuration file').' '.$config_file_location);
                else
                {
                    clearstatcache(true);
                    sleep(1);
                    
                    $this->syncMultisiteConfigDatabaseToFile();
                }

                if ($this->createNewDatabase($db_database, $db_host, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca))
                {
                    if (!$this->runNewSiteSQLInstallScript($site_folder, $db_database, $db_host, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca))
                    {
                        $retval['status'] = SAVE_INVALID_DATA;
                        $retval['errors'] = [];
                        $retval['errors'][] = ___('Cannot run SQL install script for the new site');                        
                    }
                } else
                {
                    $retval['status'] = SAVE_INVALID_DATA;
                    $retval['errors'] = [];
                    $retval['errors'][] = ___('Cannot establish database connection');
                }
                copy_recursive(SCHLIX_ROOT_PATH.'/system/themes/companyprofile', SCHLIX_ROOT_PATH.'/web/'.$site_folder.'/themes/companyprofile');
                copy_recursive(SCHLIX_ROOT_PATH.'/system/images/demo', SCHLIX_ROOT_PATH.'/web/'.$site_folder.'/media/images/demo');
                copy_recursive(SCHLIX_ROOT_PATH.'/system/images/gallery', SCHLIX_ROOT_PATH.'/web/'.$site_folder.'/media/gallery');

                // The problem is the content of multisite-config.inc.php is still being used. Once the main page is reloaded then it should re-read it again.
                $_SESSION['refresh_multi_config'] = TRUE;
                clearstatcache(true);
            } else
            {
                $config_save_result = file_put_contents($config_file_location, $data);
                if (!$config_save_result)
                    die(___('Cannot save to configuration file').' '.$config_file_location);
                else
                {
                    $this->syncMultisiteConfigDatabaseToFile();
                }

            }
            // Force Refresh Menu Links
            $menuadm = new \App\Core_Menu_Admin;
            $menuadm->refreshMenuLinks(); // prana - may 24, 2010
            
            return $retval;
            //////////////////////////////
        }
    }
    
    /**
     * Create a blank site
     * @return array
     */
    protected function createBlankSite() {
        global $SystemDB;
        
        $datavalues = array('status' => -3);
        $this->table_items->quickInsert($datavalues);
        $ret_id = $SystemDB->getLastInsertID();
        return $ret_id;
    }

    //_________________________________________________________________________//

    public function ajaxGetAllItems($start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC') {
        global $SystemDB;

        
        if ((int) $start > (int) $end)
            return ajax_reply(400, 'Invalid Request');
        $sortby = $SystemDB->escapeString($sortby);
        $items_per_page = min(HARDCODE_MAX_ROWLIMIT, $end - $start);
        $fields = implode(',', quote_array_of_field_names_for_query($this->getItemFieldNamesForAjaxListing()));
        $total_item_count = $this->app->getTotalItemCount();
        $result_array_files = $this->app->getAllItems($fields, 'status > -3', $start, $end, $sortby, $sortdirection, false);
        return ajax_datasource_reply(200, $result_array_files, $start, $end, $items_per_page, $total_item_count, $sortby, $sortdirection);
    }

    public function getHostDataResponseSchema() {
        $hostm = new sitemanager_hosts_Admin();
        return $hostm->ajaxGetAllDataResponseSchema();
    }

    //_________________________________________________________________________//
    protected function getTotalHostCountforWebsiteByID($host_id) {
        global $SystemDB;

        $host_id = intval($host_id);
        $sql = "SELECT COUNT(id) AS total_count FROM gk_sites_hosts WHERE category_id = {$host_id}";
        $result = $SystemDB->getQueryResultSingleRow($sql);
        return $result['total_count'];
    }

    //_________________________________________________________________________//
    public function getHostNames($id, $start, $end, $sortby, $sortdirection) {
        global $SystemDB;

        $id = (int) $id;
        $sortby = $SystemDB->escapeString($sortby);
        $items_per_page = min(HARDCODE_MAX_ROWLIMIT, $end - $start);

        $total_item_count = $this->getTotalHostCountforWebsiteByID($id);

        $sql = $SystemDB->generateSelectSQLStatement('gk_sites_hosts', '*', "category_id = {$id}", $start, $end, $sortby, $sortdirection, true, SCHLIX_SQL_ENFORCE_ROW_LIMIT);
        $result = $SystemDB->getQueryResultArray($sql);

        return ajax_datasource_reply(200, $result, $start, $end, $items_per_page, $total_item_count, $sortby, $sortdirection);
    }

    protected function setAdditionalVariablesForEditItem($vars)
    {
        
        $item = $vars['item'];
        if ($vars['id'] > 0)
        {
            $configvar = $this->parseSiteConfigFile($item['title']);
            if (___c($configvar) > 0)
                $vars = array_merge($vars, $configvar);
        }
        if ($vars['SCHLIX_SITE_HTTP_URL'] == 'http://')
$vars['SCHLIX_SITE_HTTP_URL'] = '';

        return $vars;
    }

//_________________________________________________________________________//
    public function saveSettings($site_name) {
        check_csrf_halt_on_error();

        return "";
    }

//_________________________________________________________________________//
    public function deleteCache($dir) {
        $forbidden_listing = array('.', '..');

        if ($dir && $_POST['confirm'] == 1) {
            $cache_path = SCHLIX_SITE_PATH . "/cache/{$dir}/";
            $cache_array = [];

            $dir_handle = @opendir($cache_path);

            while ($file = readdir($dir_handle)) {
                if (!in_array($file, $forbidden_listing) && strpos($file, '.array') !== false) {
                    unlink($cache_path . $file);
                }
            }
            closedir($dir_handle);
        }
        return ajax_reply(200, 'OK');
    }

//_________________________________________________________________________//
    private function deleteTinyMCECache() {
        $forbidden_listing = array('.', '..');
        $cache_path = SCHLIX_SITE_PATH . "/cache/";
        $cache_array = [];

        $dir_handle = @opendir($cache_path);

        while ($file = readdir($dir_handle)) {
            if (!in_array($file, $forbidden_listing) && strpos($file, 'tinymce') !== false) {

                unlink($cache_path . $file);
            }
        }
        closedir($dir_handle);
    }

//_________________________________________________________________________//
    public function getUserGroupsArray() {
        global $CurrentUser;

        $all_groups = $CurrentUser->getAllCategories();
        $result = [];
        foreach ($all_groups as $group) {
            $result[] = array('label' => $group['groupname'], 'value' => $group['groupname']);
        }
        return $result;
    }

    //_________________________________________________________________________//
    public function getFirstLevelSubdirectoryListing($directory) {
        $result = [];
        $iterator = new DirectoryIterator(remove_multiple_slashes(SCHLIX_SITE_PATH . $directory));
        foreach ($iterator as $fileinfo) {
            if ($fileinfo->isDir() && !$fileinfo->isDot()) {
                $result[] = $fileinfo->getBasename();
                //$filenames[$fileinfo->getMTime()] = $fileinfo->getFilename();
            }
        }
        return $result;
    }

    public function getUsersClassesInDirectory($dir) {
        $result = [];
        $real_dir = remove_multiple_slashes(SCHLIX_ROOT_PATH . $dir);
        $iterator = new \DirectoryIterator(remove_multiple_slashes($real_dir));
        foreach ($iterator as $fileinfo) {
            if ($fileinfo->isDir() && !$fileinfo->isDot()) {
                $class_name = $fileinfo->getBasename();
                if (file_exists($real_dir . $class_name . "/{$class_name}.class.php") && strpos($class_name, 'users') !== false)
                    $result[] = $class_name;
            }
        }
        return $result;
    }

//_________________________________________________________________________//
    public function getUsersClassArray() {
        $system_dir_classes = $this->getUsersClassesInDirectory("/system/apps/");
        return $system_dir_classes;
    }
    
    protected function getCurrentErrorReportingInArray()
    {
        $current_err = error_reporting();
        $result = [];
        
        foreach ($this->error_levels as $k => $v)
        {
            if ($current_err & $k)
                $result[] = $v;
        }
        return $result;
    }
    
    protected function getErrorReportingAsString($opt)
    {
        $result = [];
        foreach ($this->error_levels as $k => $v)
        {
            if (in_array($v, $opt))
                $result[] = $v;
        }
        
        return ___c($result) > 0 ? implode('|', $result) : '0';
    }
            
    //_________________________________________________________________________//
    public function Run() {
        switch (fget_alphanumeric('action')) {
            case 'gethostnames': 
                return ajax_echo($this->getHostNames(fget_int('id'), fget_int('start'), fget_int('end'), fget_string_noquotes_notags('sortby'), fget_string_noquotes_notags('sortdirection')));
                break;
            case 'gethostdataresponseschemas':
                return ajax_echo($this->getHostDataResponseSchema());
                break;

            default:return parent::Run();
        }
    }

}
