<?php
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
//
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
namespace SCHLIX;
            

trait cmsTemplateViewer {

    /**
     * SCHLIX Namespace: App, Block, Macro, WYSIWYGEditor
     * @var string
     */
    protected $schlix_namespace;

    /**
     * Class name only without namespace
     * @var string
     */
    protected $schlix_class;

    /**
     * Class type: app, block, macro, wysiwygeditor
     * @var string 
     */
    protected $schlix_class_type;

    /**
     * The parent directory: apps, blocks, macros, wysiwygeditors
     * @var string 
     */
    protected $schlix_parent_directory;

    /**
     * Returns the relative directory, e.g. /apps/appname
     * @var string
     */
    protected $schlix_class_relative_directory;

    /**
     * Absolute URL path
     * @var string 
     */
    protected $schlix_class_absolute_url_path;

    /**
     * Current subclass
     * @var string 
     */
    protected $schlix_subclass;
    
    
    /**
     * Relative directory
     * @var string  
     */
    protected $app_relative_directory;

    // temporary fixes for PHP 8.2 - 2023-Nov-16
    protected $schlix_master_class;
    protected $schlix_master_directory;
    protected $schlix_class_js_controller;
    
    //PHP 8.2 compatibility
    protected $app_name;    
    
    //PHP 8.2 compatibility
    protected $full_app_name;
    
    //_______________________________________________________________________________________________________________//
    /**
     * Loads a generic PHP template file, with the following lookup order:
     * Current theme folder to Current Site Folder to System Folder
     * @param string $script_path
     * @param array $vars
     * @param bool $include_once
     * @return bool
     */
    public function loadGenericFile($script_path, $vars, $include_once = false, $is_master = false) {
        $filename = $this->findTemplateScriptFile($script_path, self::class, $is_master);
        if ($filename) {
            if (is_array($vars))
                extract($vars, EXTR_REFS);            
            if ($include_once)
                @include_once($filename);
            else
                @include($filename);
            return true;
        } else
            return FALSE;
    }

    /**
     * loads a template file for viewing from the master
     * @param string $script_path
     * @param array $vars
     * @return bool
     */
    public function loadMasterTemplateFile($script_path, $vars, $include_once = false) {
        return $this->loadTemplateFile($script_path , $vars, $include_once, true);
    }

    //_________________________________________________________________________//
    /**
     * loads a template file for viewing and return the string, not parsed with x-ui
     * @param string $script_path
     * @param array $vars
     * @return string
     */
    public function loadNativeTemplateFile($script_path, $vars, $include_once = false) {
        /*ob_start();
        $this->loadGenericFile($script_path . '.template.php', $vars, $include_once, false);
        $html = ob_get_contents();
        ob_end_clean();
        return $html;                */
        return $this->loadGenericFile($script_path . '.template.php', $vars, $include_once, false);
    }

    //_________________________________________________________________________//
    /**
     * @internal Clean up template variables
     * @param array $local_variables
     * @return bool
     */
    protected function _cleanTemplateVars($local_variables)
    {
        $vars_to_remove = ['vars', 'CurrentUser','SystemUI','SystemConfig'];
        foreach ($vars_to_remove as $v)                
            unset($local_variables[$v]);   
        return $local_variables;
        
    }

    
    public function loadTemplateString($html_string, $vars) {
        if (is_array($vars))
        {
            extract($vars, EXTR_REFS);
        }
        $local_variables = (compact(array_keys((get_defined_vars()))));
        if (___c($local_variables) > 0)
        {
            $vtc = ['vars','CurrentUser'];
            foreach ($vtc as $c)
                unset($local_variables[$c]);
            $vars = is_array($vars) ? array_merge($vars, $local_variables) : $local_variables;
        }

        echo skin_html ($html_string, $vars);
                    
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Loads a generic PHP template file, with the following lookup order:
     * Current theme folder to Current Site Folder to System Folder
     * @param string $script_path
     * @param array $vars
     * @param bool $include_once
     * @return bool
     */
    public function loadTemplateFile($script_path, $vars, $include_once = false, $is_master = false) {

        $filename = $this->findTemplateScriptFile($script_path .'.template.php', self::class, $is_master);        
        if ($filename) {
            //$globals = array_keys($GLOBALS);
            if (is_array($vars))
            {
                extract($vars, EXTR_REFS);
            }
            ob_start();                
            if ($include_once)
                @include_once($filename);
            else
                @include($filename);
            $buffered_html = ob_get_contents();
            ob_end_clean();
            $__app__ = $this;            
            $local_variables = (compact(array_keys((get_defined_vars()))));            
            if (___c($local_variables) > 0)
            {                
                $vtc = ['vars','CurrentUser'];
                foreach ($vtc as $c)
                    unset($local_variables[$c]);
                $vars = is_array($vars) ? array_merge($vars, $local_variables) : $local_variables;                
            }
            
            echo skin_html ($buffered_html, $vars);
                
            return true;
        } 
        return false;
    }
    
    
    //_________________________________________________________________________//
    /**
     * Load the language file
     * @return boolean
     */
    public function loadLanguageFile() {
        if (SCHLIX_DEFAULT_LANGUAGE != 'en_us')
            return $this->loadGenericFile('languages/' . SCHLIX_DEFAULT_LANGUAGE . '.lang.php', NULL);
        else
            return true;
    }

    public function getFullApplicationName()
    {
        return $this->full_app_name;
    }
    /**
     * Determines the class information (namespace, class, class type, parent directory, subclass, JS controller, etc)
     */
    protected function setSchlixClassInformation() {
        $schlix_class = ltrim(static::class, '\\'); 
        /*$schlix_class = ltrim(static::class, '\\'); // $app_name;
        $last_ns_pos = strrpos($schlix_class, '\\');
        $namespace = substr($schlix_class, 0, $last_ns_pos);
        $this->schlix_namespace = $namespace; // App / Macro / Block, etc
        $this->schlix_class = substr($schlix_class, $last_ns_pos + 1);*/
        $class_info = get_class_namespace_breakdown(static::class);
        $namespace = $this->schlix_namespace = $class_info['namespace'];
        $this->schlix_class = $class_info['class'];
        
        $this->schlix_class_type = strtolower($namespace); // app/block/macro/wysiwygeditor
        $this->schlix_parent_directory = strtolower($namespace) . 's';
        $first_classname_pos = strpos($this->schlix_class, '_');
        if ($first_classname_pos === FALSE) {
            $first_classname_pos = strlen($this->schlix_class);
        }
        $desired_comp_dir = strtolower(substr($this->schlix_class, 0, $first_classname_pos));
        $this->schlix_master_directory = $desired_comp_dir;
        // PHP8 workaround
        if (!property_exists($this, 'app_name'))
                $this->app_name = '';
        // end PHP8 workaround
        
        $this->schlix_master_class = $first_classname_pos ? $desired_comp_dir : $this->app_name;
        $this->schlix_class_js_controller = strtolower(str_replace('_', '.', $this->schlix_class)) . '.js';
        $this->full_app_name = strtolower(str_replace('_', '.', $this->schlix_class)) ;
        $this->schlix_class_relative_directory = $this->schlix_parent_directory . '/' . $desired_comp_dir;
        $this->schlix_class_absolute_url_path = SCHLIX_SITE_HTTPBASE . '/' . $this->app_relative_directory;
        if ($this->schlix_class_type == 'app') {
            // Is this the admin class for app?
            $this->app_name = $this->schlix_master_class; // backward compatibility
            $this->app_js_controller = $this->schlix_class_js_controller;
            $this->app_relative_directory = $this->schlix_class_relative_directory;
            $this->app_real_class_alias = $this->schlix_class_relative_directory;
            $php8_tmp_fix_class_name = $this->schlix_class;
            $admin_str_pos = strrpos($php8_tmp_fix_class_name, '_Admin');
            if ($admin_str_pos !== FALSE) {
                
                $this->app_admin_url_path = SCHLIX_SITE_HTTPBASE . '/' . get_application_alias('admin') . '/app/' . $desired_comp_dir;
                $this->app_admin_url_path_no_base = '/' . get_application_alias('admin') . '/app/' . $desired_comp_dir;                
                $this->app_frontend_classname = substr($php8_tmp_fix_class_name, 0, $admin_str_pos);
                                
                $this->app_subclass_suffix = $this->schlix_subclass = strtolower(substr($this->app_frontend_classname, $first_classname_pos + 1, $admin_str_pos));
                //echo "HAS ADMIN {$admin_str_pos} {$this->app_frontend_classname}<br />";
            } else {
                $this->app_subclass_suffix = $this->schlix_subclass = strtolower(substr($this->schlix_class, $first_classname_pos + 1, strlen($this->schlix_class) - $first_classname_pos - 1));
                //$c = $this->schlix_class;
                //$xpos = strpos(trim($c), '_Admin');
                //echo "HAS NO ADMIN!!! this-schlix_class = '{$this->schlix_class}' admin_str_pos={$admin_str_pos} vs {$xpos}<br />";
            }
        }
        //echo "setSchlixClassInformation for |{$this->schlix_class_type}| {$schlix_class}: {$this->schlix_class} with frontend: {$this->app_frontend_classname}<br />";
    }

    /**
     * Load Javascript file
     * @global \SCHLIX\cmsHTMLPageHeader $HTMLHeader
     * @param string $script_path
     * @param bool $is_master
     */
    public function JAVASCRIPT($script_path, $is_master = false, $get_hash = false) {
        global $HTMLHeader;

        $script = $this->getURLofScript($script_path, $is_master, $get_hash);
        
        if ($script)
            $HTMLHeader->JAVASCRIPT($script);
        else {
            $script_path = addslashes($script_path);
            $HTMLHeader->JAVASCRIPT_TEXT("console.error('Cannot find Javascript file {$script_path} ')");
        }
    }

    /**
     * Load CSS file
     * @global \SCHLIX\cmsHTMLPageHeader $HTMLHeader
     * @param string $script_path
     * @param bool $is_master
     */
    public function CSS($script_path, $is_master = false, $get_hash = false) {
        global $HTMLHeader;

        $script = $this->getURLofScript($script_path, $is_master, $get_hash);

        if ($script)
            $HTMLHeader->CSS($script);
        else {
            $script_path = addslashes($script_path);
            $HTMLHeader->JAVASCRIPT_TEXT("console.error('Cannot find CSS file: {$script_path} ')");
        }
    }

    /**
     * Returns the relative path of a Javascript/CSS file. Returns false if file is not found
     * @param string $script_path
     * @return string|bool
     */
    public function getURLofScript($script_path, $is_master = false, $get_hash = false) {
        
        $possibilities = array(
            array('url' => CURRENT_THEME_URL_PATH_NO_BASE, 'dir' => CURRENT_THEME_PATH),
            array('url' => CURRENT_SUBSITE_URL_PATH_NO_BASE, 'dir' => CURRENT_SUBSITE_PATH),
            array('url' => SCHLIX_SYSTEM_URL_PATH_NO_BASE, 'dir' => SCHLIX_SYSTEM_PATH));
        foreach ($possibilities as $possible_path) {
            $possible_files = [];
            if ($this->schlix_subclass && !$is_master) 
                $possible_files[] = '/' . $this->schlix_class_relative_directory . '/' . $this->schlix_subclass . '/' . $script_path;
            $possible_files[] = '/' . $this->schlix_class_relative_directory . '/' . $script_path;
            foreach ($possible_files as $possible_file) {
                $filename = remove_multiple_slashes($possible_path['dir'] . $possible_file);
                if (is_file($filename))
                {
                    $h = $get_hash ? '?h='.get_simplified_file_hash($filename) : '';
                    return remove_multiple_slashes($possible_path['url'] . $possible_file).$h;
                }
            }
        }
        return FALSE;
    }

    /**
     * Returns a truncated article excerpt
     * @param array $child_item
     * @param string $primary_field
     * @param string $secondary_field
     * @param int $max_length
     * @return string
     */
    public function getArticleExcerptFromObjectArray($child_item, $primary_field = 'summary_secondary_headline', $secondary_field = 'summary', $max_length = 255)
    {
        $secondary_headline = '';
        //isset($child_item[$primary_field]) && !empty($child_item[$primary_field]) ? $child_item[$primary_field] : substr(strip_tags($child_item[$secondary_field]),0, $max_length).'...';
        if (isset($child_item[$primary_field]) && !empty($child_item[$primary_field])  )
        {
            $secondary_headline = $child_item[$primary_field];
        } else
        {
            $secondary_headline = truncate_text($child_item[$secondary_field], $max_length, true, true);
        }

        return $secondary_headline;
        
    }
    /**
     * Internal helper - find template script in the following order: theme path, app path, system path
     * @ignore
     * @param string $script_path
     * @param string $class_to_skip_str_pos
     * @return boolean|string
     */
    public function findTemplateScriptFile($script_path, $class_to_skip_str_pos, $is_master) {
        $classes = get_class_ancestors($this);
        $script_path = trim($script_path);
        $schlix_lib_path = SCHLIX_SYSTEM_PATH.'/libs/schlix';
        if ($script_path)
        foreach ($classes as $class) {
            if (stripos($class, $class_to_skip_str_pos) === FALSE) {
                $tp = defined('CURRENT_THEME_PATH') ? CURRENT_THEME_PATH : SCHLIX_SITE_PATH;
                $possibilities = [$tp, SCHLIX_SITE_PATH, SCHLIX_SYSTEM_PATH ];                
                foreach ($possibilities as $possible_path) {
                    $possible_files = [];
                    if ($this->schlix_subclass && !$is_master) {
                        
                        $possible_files[] = $possible_path . '/' . $this->schlix_class_relative_directory . '/' . $this->schlix_subclass . '/' . $script_path;
                        
                        // July 2017 - This is for backward compatibility pre v2.1.5-x
                        $first_dot_pos = strpos($script_path, '.');
                        $fn_1 = substr($script_path, 0, $first_dot_pos);
                        $fn_2 = substr($script_path, $first_dot_pos + 1, strlen($script_path) - $first_dot_pos - 1);
                        $possible_files[] = $possible_path . '/' . $this->schlix_class_relative_directory . '/' . $fn_1 . '.' . $this->schlix_subclass . '.' . $fn_2;
                        // end backward compatibility - TODO: remove by July 2018
                        
                    } else
                    $possible_files[] = $possible_path . '/' . $this->schlix_class_relative_directory . '/' . $script_path;
                    //print_r($possible_files);die;
                    foreach ($possible_files as $filename) {
                        if (is_file($filename)) {                            
                            return $filename;
                        }
                    }
                }
                $last_attempt = $schlix_lib_path.'/'.$script_path;
                if (is_file($last_attempt))
                    return $last_attempt;
            }
        }
        return false;
    }

}


abstract class cmsWysiwygEditor implements interface_cmsEditor
{
    use cmsTemplateViewer;
    
    protected $forbidden_names = [];
    protected $editor_name = null;
    protected $profile_id;
    protected $profile_name;

    /**
     * Config registry
     * @var \SCHLIX\cmsConfigRegistry
     */
    protected $wysiwyg_config;
    /**
     * Configuration values in key/value array
     * @var array 
     */
    protected $config;
    /**
     * Array of default profiles. For example:
     * ['tinymce4-full' => ['title' => 'TinyMCE4 Full Featured', 'profile'=>'full'],
     * 'tinymce4-simple' => ['title' => 'TinyMCE4 Simple', 'profile'=>'simple'],
     * 'tinymce4-limited' => ['title' => 'TinyMCE4 Limited (No Media Manager)', 'profile'=>'simple-nomediamanager']];
     * @var array 
     */
    protected $default_profiles = [];
               
    /**
     * Constructor
     * @param string $profile_id
     */
    public function __construct($profile_id = 0)
    {
        $this->setSchlixClassInformation();
        $this->editor_name = strtolower($this->schlix_class);
        if ($profile_id > 0)
        {
            $this->loadProfileByID($profile_id);
        }
    } 
    /**
     * Verify folder name
     * @param \SCHLIX\cmsDirectoryFilter $items
     * @return array
     */
    protected function verifyDirBasedExtensions(\SCHLIX\cmsDirectoryFilter $items, $file_to_check = '') {
        $forbidden_listing = ['.','..','.svn','.cvs', '.git', '__MACOSX'];

        $item_array = [];
        foreach ($items as $item) {
            $file = $item->getFileName();
            if ((!in_array($file, $forbidden_listing)) && (strpos($file, 'uninstalled_') === false ))
            {
                if ($file_to_check)
                {   
                    $plugin_js_file = $item->getPath().'/'.$item->getFileName().'/'.$file_to_check;
                    if (file_exists($plugin_js_file))
                        $item_array[] = $file;
                } else
                {
                    $item_array[] = $file;
                }
            }
        }

        return $item_array;
    }
     
    /**
     * Returns a list of directory based extensions, where $from is either 'system' or 'user'
     * and $type can be any subdirectory within it, e.g. plugins, themes, languages
     * If $file_to_check is specified, it will try to verify if that file exist in that subfolder
     * @param string $from
     * @param string $type
     * @param string $file_to_check
     * @return type
     */
    public function getListOfDirBasedExtensions($from, $type, $file_to_check = '')
    {
        $ext_dir = $this->getExtensionFullPath($from, $type);
        if (is_dir($ext_dir))
        {
                $items = \SCHLIX\cmsDirectoryFilter::getDirectoryIterator($ext_dir, \SCHLIX\cmsDirectoryFilter::FILTER_DIR_ONLY);
            $items = $this->verifyDirBasedExtensions($items, $file_to_check);
            if ($items)
            sort($items);
            return $items;
        }
        return NULL;
    }     
     
    /**
     * Returns a list of file based extensions, where $from is either 'system' or 'user'
     * and $type can be any subdirectory within it, e.g. plugins, themes, languages
     * @param string $from
     * @param string $type
     * @param string $ext
     * @return type
     */
    public function getListOfFileBasedExtensions($from, $type, $ext)
    {
        $ext_dir = $this->getExtensionFullPath($from, $type);
        if (is_dir($ext_dir))
        {
            $items = [];
            $files = \SCHLIX\cmsDirectoryFilter::getDirectoryIterator($ext_dir, \SCHLIX\cmsDirectoryFilter::FILTER_FILE_ONLY);
            foreach ($files as $file)
            {
                $filename = pathinfo($file->getFileName(), PATHINFO_FILENAME);
                $fileext = pathinfo($file->getFileName(), PATHINFO_EXTENSION);
                if ($fileext == $ext)
                    $items[] = $filename;
            }
            if ($items)
                sort($items);
            return $items;
        }
        return NULL;
    }     
    
    
    /**
     * Ensure default profiles exists
     * @global \App\wysiwygEditors $WYSIWYGEditor
     */
    public function ensureDefaultProfilesExist()
    {
        global $WYSIWYGEditor;
        if ($this->default_profiles)
        {
            $profiles = $WYSIWYGEditor->getProfilesByEditorName($this->editor_name);
            $profile_names = array_column($profiles,'virtual_filename');
            foreach ($this->default_profiles as $profile_name => $opt)
            {
                if (!in_array($profile_name, $profile_names))
                {
                    $config = $this->getDefaultProfileResetSettings($opt['profile']);
                    $WYSIWYGEditor->createEmptyProfile($this->editor_name, $profile_name, $opt['title'], true, $config);
                }
            }
        }
    }
    /**
     * Returns a list of extensions, where $from is either 'system' or 'user'
     * and $type can be any subdirectory within it, e.g. plugins, themes, languages
     * @param string $from
     * @param string $type
     * @return type
     */
    public function getListOfExtensions($from, $type)
    {

        $ext_dir = $this->getExtensionFullPath($from, $type);
        if (is_dir($ext_dir))
        {
            $items = \SCHLIX\cmsDirectoryFilter::getDirectoryIterator($ext_dir, \SCHLIX\cmsDirectoryFilter::FILTER_DIR_ONLY);
            $items = $this->verifyExtensionFolders($items);
            if ($items)
                sort($items);
            return $items;
        }
        return NULL;
    }     
    /**
     * Returns extension URL path, where $from is either system or user,
     * $type is the subdirectory, and $item is the specific extension
     * @param string $from
     * @param string $type
     * @param string $item
     * @return string
     */
    public function getExtensionURLPath($from, $type, $item= '')
    {
        $type = alpha_numeric_with_dash_underscore($type); // filter directory name
        $parent_dir = ($from == 'system') ? SCHLIX_SYSTEM_URL_PATH : CURRENT_SUBSITE_URL_PATH; 
        $str = "{$parent_dir}/wysiwygeditors/{$this->editor_name}/{$type}/";
        if ($item)
            $str.= $item;
        $ext_dir = remove_multiple_slashes($str);
        return $ext_dir;
    }

    /**
     * Returns extension path, where $from is either system or user,
     * $type is the subdirectory, and $item is the specific extension
     * @param string $from
     * @param string $type
     * @param string $item
     * @return string
     */    
    public function getExtensionFullPath($from, $type, $item= '')
    {
        $type = alpha_numeric_with_dash_underscore($type); // filter directory name
        $parent_dir = ($from == 'system') ? SCHLIX_SYSTEM_PATH : CURRENT_SUBSITE_PATH; 
        $str = "{$parent_dir}/wysiwygeditors/{$this->editor_name}/{$type}/";
        if ($item)
            $str.= $item;
        $ext_dir = remove_multiple_slashes($str);
        return $ext_dir;
    }
    
    
    /**
     * Modify data before save config
     * @global \App\Users $CurrentUser
     * @param array $datavalues
     * @return array
     */
    public function onModifyDataBeforeSaveConfig($datavalues) {
        
        //$datavalues['options'] = serialize($datavalues['options']);
        return $datavalues;
    }
    /**
     * After config has been saved
     * @param int $profile_id
     */
    public function onAfterSaveConfig($profile_id) 
    {
    }    
    
    /**
     * Returns a key/value array from the config. If the config key contains
     * a prefix of str_config_ or bool_config_ or int_config_, then it will
     * be moved into 'processed' key in the result array, otherwise it will be
     * moved to 'unprocessed' key
     * @return array
     */
    public function generateConfigurationArrayFromConfig()
    {
        $config = [];
        if ($this->config)
        {
            foreach ($this->config as $key => $value)
            {
                if (strpos ($key,'str_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'str_config_');
                    $trimmed_value = trim($value);
                    if ($trimmed_value)
                    {
                        $config['processed'][$new_key] = '"'.str_replace('"','\"', $trimmed_value).'"' ;
                    }
                } elseif (strpos ($key,'bool_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'bool_config_');
                    $config['processed'][$new_key] = $value ? 'true' : 'false';
                } 
                elseif (strpos ($key,'int_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'int_config_');
                    $config['processed'][$new_key] = (int) $value;
                }                
                else
                {
                    $config['unprocessed'][$key] = $value;
                }
            }
            return $config;
        } else return NULL;
    }    
    
    /**
     * Generate configuration script
     * @param array $config
     * @return string
     */
    public function generateConfigurationScriptFromConfigArray($config)
    {
        if (is_array($config) && !empty($config))
        {
            $arr_config = [];

            foreach ($config as $key => $value)
            {
                $arr_config[]= "\t{$key}: {$value}";
            }

            $str_config = "\n".implode(",\n",$arr_config)."";
            return $str_config;        
        } else return '""';
    }
    /**
     * Given $array_config containing 2 keys: unprocessed and processed,
     * process them and move it to $array_config['processed']
     * @param string $array_config
     * @return string
     */
    public function modifyUnprocessedConfig($array_config)
    {
        return $array_config['processed'];
    }
    /**
     * The options specified here cannot be modified by user configuration
     * @param array $options
     * @return string
     */
    protected function forceDefaultConfigOptions($options)
    {
        return $options;
    }   
    
    public function reloadConfigFromDatabase()
    {
        if (($this->profile_name) && ($this->profile_id > 0))            
        {
            $this->config = $this->wysiwyg_config->get($this->profile_name);
        }
    }
    /**
     * Initialize current editor and load it by profile ID
     * @param int $profile_id
     */
    public function loadProfileByID($profile_id)
    {
        $this->profile_id = $profile_id;
        $this->profile_name =  \App\Core_EditorManager::getProfileNameByID($profile_id);
        $wysiwyg_config = new \SCHLIX\cmsConfigRegistry('gk_wysiwyg_config');
        $this->config = $wysiwyg_config->get($this->profile_name);
        $this->wysiwyg_config = $wysiwyg_config;
    }

    public function getConfig($key, $default = null)
    {
        if (is_array($this->config) && array_key_exists($key, $this->config)) 
            return $this->config[$key] ? $this->config[$key] : $default;
        else
            return $default;
    }
    
    /**
     * Returns extra header script, given $url where it's the load config URL
     * @param string $url
     * @return string
     */
    public function getHeaderVariableInitScript($url)
    {
      /*  start_output_buffer(false);
        $this->loadTemplateFile('view.vars.js', ['wysiwyg_init' => $url]);
        $result = end_output_buffer();
        return $result;*/
        return FALSE;
    }

    /**
     * View init script
     * @return boolean
     */
    public function viewInitScript()
    {
        $config = $this->getUncachedConfigurationScript();
        $local_variables = compact(array_keys(get_defined_vars())); 
        $result = $this->loadTemplateFile('view.wysiwyg.init.js', $local_variables);
        if (!$result)
        {
            echo " alert('Cannot load view.wysiwyg.init.js')";
        }
        return false;
    }
    
    /**
     * Run a command
     * @param array $command
     */
    public function Run($command)
    {
        
    }
        
}

abstract class cmsMacro implements interface_cmsMacro
{
    use cmsTemplateViewer;
    /**
     * Configuration value
     * @var array 
     */
    protected $config;
    
    /**
     * Macro Name
     * @var string 
     */
    protected $macro_name;
    
    
    /**
     * Constructor
     * @param array $config
     */
    public function __construct($config)
    {
        $this->config = $config;
        $this->setSchlixClassInformation();
        $this->macro_name = $this->schlix_class;
        
    }
    
    /**
     * Record System Log. Type can be info, error, warning, or debug
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param string $message
     * @param string $type
     */
    public function recordLog($message, $type = 'info')
    {
        global $SystemLog;

        if (method_exists($SystemLog, 'Record'))
            $SystemLog->record($message, $this->app_name, $type);
        else
            echo $message;
    }

    /**
     * Append before output
     * @param array $data
     * @param string $text
     */
    public function appendBeforeArticleOuput(&$data, $text)
    {
        if (array_key_exists('macro_processed_text_before_article', $data))
            $data['macro_processed_text_before_article'].= $text;
        else
            $data['macro_processed_text_before_article'] = $text;
    }
    
    /**
     * Append before output
     * @param array $data
     * @param string $text
     */
    public function appendAfterArticleOuput(&$data, $text)
    {
        if (array_key_exists('macro_processed_text_after_article', $data))
            $data['macro_processed_text_after_article'].= $text;
        else
            $data['macro_processed_text_after_article'] = $text;
    }
    /**
     * Append before output
     * @param array $data
     * @param string $text
     */
    public function appendAtTheTopOfArticleOuput(&$data, $text)
    {
        if (array_key_exists('macro_processed_text_outside_article_top', $data))
            $data['macro_processed_text_outside_article_top'].= $text;
        else
            $data['macro_processed_text_outside_article_top'] = $text;
    }
    /**
     * Append before output
     * @param array $data
     * @param string $text
     */
    public function appendAtTheBottomOfArticleOuput(&$data, $text)
    {
        if (array_key_exists('macro_processed_text_outside_article_bottom', $data))
            $data['macro_processed_text_outside_article_bottom'].= $text;
        else
            $data['macro_processed_text_outside_article_bottom'] = $text;
    }
    
    /**
     * Given HTML string, returns all attributes of macro text
     * e.g. {contact id="1" header="Test"} where "{" is $open_tag and "}" is 
     * $close_tag. There can be more than 1 macro in the $html_string.
     * 
     * It will return an array where for each array, it will contain 2 main
     * keys: 'attributes' and 'macro', where 'macro' is the original text
     * and 'attributes' contain another array of all the parsed attributes
     * @param string $html_string
     * @param string $prefix
     * @param string $open_tag
     * @param string $close_tag
     * @return array
     */
    protected function getGenericAttributesFromText($html_string, $prefix, $open_tag = '{', $close_tag = '}')
    {
        $attrs = [];
        
        $regex = '#'.$open_tag.$prefix.'([^}]+)'.$close_tag.'#iU';
        $match_count = preg_match_all($regex, $html_string, $attrmatches);
        if ($match_count > 0) {

            for ($i = 0; $i < $match_count; $i++) { 
                $str_attr = $attrmatches[1][$i];
                preg_match_all('/\s+?(.+)="([^"]*)"/U', $str_attr, $extra_attrs, PREG_SET_ORDER);
                if ($extra_attrs) {
                    foreach ($extra_attrs as $attr) {
                        $key = alpha_numeric_with_dash_underscore($attr[1]);
                        $attrs[$i]['attributes'][$key] = strip_tags($attr[2]);
                    }
                    $attrs[$i]['macro'] = $attrmatches[0][$i];
                }
            }
            return $attrs;
        }
        return NULL;
    }
    
    /**
     * Get macro config
     * @return array
     */
    public function getConfiguration()
    {
        return $this->config;
    }

    /**
     * Run the macro
     * @param string $data
     * @param object $caller_object
     * @param string $caller_function
     * @param array $extra_info
     * @return bool
     */
    public function Run(&$data, $caller_object, $caller_function, $extra_info = false)
    {
        echo 'Please implement this function';
        return false;
    }

//_______________________________________________________________________________________________________________//
}

abstract class cmsBlock implements interface_cmsBlock
{
    use cmsTemplateViewer;
    /**
     * Block Name
     * @var string 
     */
    protected $block_name;
    
    /**
     * Original block class name
     * @var string 
     */
    protected $original_block_name;
    /**
     * Configuration value
     * @var array 
     */
    protected $config;

    /**
     * Relative directory
     * @var string 
     */
    
    protected $block_relative_directory;
    /**
     * Constructor
     * @param string $block_name
     * @param array $config
     */
    
    /**
     * Filesystem cache
     * @var \SCHLIX\cmsFSCache
     */
    protected $cacheSystem;
    
    public function __construct($block_name, $config)
    {

        $this->block_name = $block_name;
        $this->config = $config;
        $this->setSchlixClassInformation();
        $this->original_block_name = $this->schlix_class;
        $this->loadLanguageFile();
        $this->cacheSystem = new \SCHLIX\cmsFSCache('generic');        
    }
    
    /**
     * Return config
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function getConfig($key, $default = null)
    {
        if (is_array($this->config) && array_key_exists($key, $this->config)) 
            return $this->config[$key] ? $this->config[$key] : $default;
        else
            return $default;
    }

    /**
     * Returns the block instance name
     * @return string
     */
    public function getBlockInstanceName()
    {
        return $this->block_name;
    }
    /**
     * Record System Log. Type can be info, error, warning, or debug
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param string $message
     * @param string $type
     */
    public function recordLog($message, $type = 'info')
    {
        global $SystemLog;

        if (method_exists($SystemLog, 'Record'))
            $SystemLog->record($message, $this->app_name, $type);
        else
            echo $message;
    }

    /**
     * Get configuration from the database
     * @return array
     */
    public function getConfiguration()
    {
        return $this->config;
    }

    /**
     * Returns true if it has int_cache_result_seconds config key and if it is 
     * set to greater than 0
     * @return bool
     */
    public function isCachingEnabled()
    {
        return $this->config['int_cache_result_seconds'] > 0;
    }
    /**
     * View cached result
     */
    public function viewCachedRun()
    {
            $cache_seconds = $this->getConfig('int_cache_result_seconds',0);// array_key_exists('int_cache_result_seconds', $this->config) ? $this->config['int_cache_result_seconds'] : 0;
            if ($cache_seconds === 0)
            {
                
                $cache_key = 'block_'.sha1($this->block_name);
                $cached_output = $this->cacheSystem->get($cache_key);
                if ($cached_output != NULL)
                {
                    echo $cached_output;
                } else {
                
                    start_output_buffer();
                    $this->Run();
                    $fresh_output = end_output_buffer();
                    $this->cacheSystem->set($cache_key, $fresh_output,0,0,$cache_seconds);            
                    echo $fresh_output;
                }
            } else
            {                
                $this->Run();
            }
    }
    
    /**
     * Run this block
     * @return bool
     */
    public function Run()
    {
        echo 'Please implement this function';
        return false;
    }

//_______________________________________________________________________________________________________________//
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
//
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//


abstract class cmsApplication_Basic implements interface_cmsApplication_Basic
{
    use cmsTemplateViewer;
    
    protected $hook_priority = 10;
    protected $disable_frontend_runtime = false;
    protected $has_versioning = false;
    protected $has_custom_media_header = false;
    protected $has_custom_fields = true;
    //protected $app_name; PHP5.6 fix already defined in cmsTemplateViewer
    protected $app_description;
    protected $disable_app_description_in_page_title = false;
    protected $cache;
    protected $has_logged_in_user_main_page = false;
    private $_tables = [];
    protected $data_directories = NULL;
    protected $config = null;
    
    protected $disable_app;
    
    
    public $full_application_alias;
    

    protected $app_js_controller;
    // protected $full_app_name; ; PHP5.6 fix already defined in cmsTemplateViewer
    protected $app_real_class_alias;
    protected $app_subclass_suffix;
    
    protected $app_admin_url_path;
    protected $app_admin_url_path_no_base;
    
    
    /**
     * Breadcrumb array
     * @var array 
     */
    protected $bread_crumbs;
    /**
     * Array containing a list of entities. Format: key/value
     * @var array 
     */
    protected $data_entities = [];
    
    protected $page_title;
    /**
     *
     * @var bool 
     */
    private $config_retrieved = false;
    
    protected $page_meta_keywords;
    protected $page_meta_description;
    
    protected $paginationStringFormat;
    
    protected $master_config;

//_______________________________________________________________________________________________________________//

    public function __construct($app_description)
    {
        $this->app_name = strtolower((new \ReflectionClass($this))->getShortName()); // $app_name;
        
        $this->app_description = $app_description;
        
        $this->cache = true; // enable by default, disable it in your custom class
        $this->paginationStringFormat = TXT_PAGINATION_STRING;
        $this->setSchlixClassInformation();
        $this->loadLanguageFile();
        $this->refreshApplicationAlias();
        $this->resetBreadCrumbs();
        $this->disable_app = (int) $this->getConfig('bool_disable_app');
    }
    
    /**
     * Gets the application alias from the database. The purpose of this function is to fix an issue where 
     * the regenerated links still refer to the old alias and needed to be saved twice
     * @param bool $use_cache
     */
    public function refreshApplicationAlias($use_cache = true)
    {
        $this->full_application_alias = get_application_alias($this->app_name, $this->schlix_subclass, $use_cache);
    }
    /**
     * Returns the hook priority
     * @return int
     */
    public function getHookPriority()
    {
        return $this->hook_priority;
    }
    
    /**
     * Set the entity (can be a table, or join table, etc)
     * @param string $entity_name
     * @param \SCHLIX\cmsSQLQueryBuilder $sqltable
     */
    protected function setEntity($entity_name, \SCHLIX\cmsSQLQueryBuilder $sqltable)
    {
        $this->data_entities[$entity_name] = $sqltable;
    }
    
    /**
     * Return the entity (can be a table, or join table, etc)
     * @param string $entity_name
     * @return \SCHLIX\cmsSQLQueryBuilder $sqltable
     */
    public function getEntity ($entity_name)
    {
        return clone $this->data_entities[$entity_name];
    }
    
    /**
     * GDPR - returns an array of personal data by email
     * @param int $user_id
     * @return array
     */
    public function getPersonalDataByUserID($user_id)
    {
        return [];
    }
    
    /**
     * GDPR - returns an array of personal data by email
     * @return array
     */
    public function getPersonalDataByEmail($email_address)
    {
        return [];
    }    
    
    /**
     * GDPR - remove personal data by email
     * @param int $user_id
     * @param string $request_guid
     */
    public function removePersonalDataByUserID($user_id, $request_guid)
    {
        
    }
    
    /**
     * GDPR - remove personal data by email
     * @param string $email_address
     * @param string $request_guid
     */
    public function removePersonalDataByEmail($email_address, $request_guid)
    {
        
    }    
    
    /**
     * Returns the original full application alias
     * @return string
     */
    
    public function getOriginalFullApplicationAlias()
    {
        
        $s = $this->app_name;
        if (!empty($this->schlix_subclass))
            $s.= '.'.$this->schlix_subclass;
        return $s;
    }
    /**
     * Returns the full application alias, e.g. users.history, for the the purpose of SEO URL
     * @return string
     */
    public function getFullApplicationAlias()
    {
        return $this->full_application_alias;
    }
    /**
     * Create a sub class instance. e.g. Newsletters_Admin --> Newsletters_Queues
     * @param string $sibling
     * @return object
     */
    public function createSubClassInstance($sibling)
    {
        $numargs = func_num_args();
        $args = [];
        if ($numargs > 1)
        {
            $original_args = func_get_args();            
            for ($i = 1, $count = $numargs;$i < $count; $i++)
                $args[] = $original_args[$i];
        }
        $full_class_name = '\\'.$this->schlix_namespace.'\\'.$this->schlix_master_class.'_'.$sibling;
        $class = new \ReflectionClass($full_class_name);
        return $class->newInstanceArgs($args);
    }

    
    public function createMasterClassInstance()
    {
        if (strtolower($this->schlix_class) != strtolower($this->schlix_master_class))
        {
            $numargs = func_num_args();
            $args = [];
            if ($numargs > 1)
            {
                $original_args = func_get_args();            
                for ($i = 1, $count = $numargs;$i < $count; $i++)
                    $args[] = $original_args[$i];
            }
            $full_class_name = '\\'.$this->schlix_namespace.'\\'.$this->schlix_master_class;
            
            $class = new \ReflectionClass($full_class_name);
            return $class->newInstanceArgs($args);
        }
        else return $this;
    }

    /**
     * Create master class friendly URL
     * @param string $str
     * @return string
     */
    public function createMasterClassFriendlyURL($str)
    {
        if ($this->app_name != $this->schlix_master_class)
        {
            $full_class_name = '\\'.$this->schlix_namespace.'\\'.$this->schlix_master_class;
            $obj = new $full_class_name();
            return $obj->createFriendlyURL($str);
            
        } else return $this->createFriendlyURL ($str);
    }
    
    /**
     * Create sibling class friendly URL
     * @param string $str
     * @return string
     */
    
    public function createSiblingClassFriendlyURL($sibling, $str)
    {
        if ($this->app_name != $this->schlix_master_class)
        {
            $full_class_name = '\\'.$this->schlix_namespace.'\\'.$this->schlix_master_class.'_'.$sibling;
            $obj = new $full_class_name();
            return $obj->createFriendlyURL($str);
            
        } else return $this->createFriendlyURL ($str);
    }
    /**
     * Return the relative path to CURRENT_SUBSITE_PATH of a specific data directory as defined in the $this->data_directory protected variable
     * @param string $key
     * @return string
     */
    public function getDataDirectoryRelativePath($key) {
        return (is_array($this->data_directories) && array_key_exists($key, $this->data_directories)) ? $this->data_directories[$key] : '';
    }

    /**
     * Return the directory path to CURRENT_SUBSITE_PATH of a specific data directory as defined in the $this->data_directory protected variable
     * @param string $key
     */
    public function getDataDirectoryFullPath($key) {
        $relpath = $this->getDataDirectoryRelativePath($key);
        return $relpath ? remove_multiple_slashes(CURRENT_SUBSITE_PATH . '/' . $relpath) : '';
    }

    /**
     * Return the absolute URL path  of a specific data directory as defined in the $this->data_directory protected variable
     * @param string $key
     * @return string
     */
    public function getDataDirectoryURLPath($key) {
        $relpath = $this->getDataDirectoryRelativePath($key);
        return $relpath ? remove_multiple_slashes(CURRENT_SUBSITE_URL_PATH . '/' . $relpath) : '';
    }

    /**
     * Return the relative path to CURRENT_SUBSITE_PATH of a file as defined in the $this->data_directory protected variable
     * @param string $key
     * @param string $filename
     * @return string
     */
    public function getDataFileRelativePath($key, $filename) {
        $relpath = $this->getDataDirectoryRelativePath($key);
        return $relpath ? remove_multiple_slashes($relpath . '/' . $filename) : '';
    }

    /**
     * Return the full file path of a file inside $this->data_directory[$key]
     * @param string $key
     * @param string $filename
     * @return string
     */
    public function getDataFileFullPath($key, $filename) {
        $fullpath = $this->getDataDirectoryFullPath($key);
        return $fullpath ? remove_multiple_slashes($fullpath . '/' . $filename) : '';
    }

    /**
     * Return encoded absolute URL path of a file inside $this->data_directory[$key]
     * @param string $key
     * @param string $filename
     * @return string
     */
    public function getDataFileURLPath($key, $filename) {
        $urlpath = $this->getDataDirectoryURLPath($key);
        return $urlpath ? ___u(remove_multiple_slashes($urlpath . '/' . $filename)) : '';
    }
    
    /**
     * Return encoded absolute URL path of a file inside $this->data_directory[$key]
     * @param string $key
     * @param string $filename
     * @return string
     */
    public function getDataFileURLPathWithHash($key, $filename) {
        $pf = $this->getDataFileFullPath($key, $filename);
        if (file_exists($pf))
        {
            $urlpath = $this->getDataDirectoryURLPath($key);
            $hash = filesize($pf).'_'.filemtime($pf);
            return $urlpath ? ___u(remove_multiple_slashes($urlpath . '/' . $filename)).'?c='.$hash : '';
        }
        return '';
    }
    
    /**
     * Creates all the folder as defined in $this->data_directory if not exists.
     * Returns a string array of errorlist, or an empty/null array if there's no error
     * @global \SCHLIX\cmsLogger $SystemLog
     * @return array
     */
    public function initializeDataDirectories() {
        global $SystemLog;

        $error_list = [];
        if (is_array($this->data_directories))
        {
            foreach ($this->data_directories as $key => $dir) {
                $dirpath = $this->getDataDirectoryFullPath($key);
                if (!is_dir($dirpath)) {
                    if (!create_directory_if_not_exists($dirpath, 0755, true)) {
                        $error = 'Fatal error: cannot create ' . $dirpath;                        
                        $SystemLog->error($error, $this->app_name);
                        $error_list[] = $error;
                        //die($error);
                    }
                } else if (!is_writable($dirpath))
                {
                    $error = 'Fatal error: Directory is not writable - ' . $dirpath;
                    $SystemLog->error($error, $this->app_name);
                    $error_list[] = $error;
                }
            }
        }
        return $error_list;
    }

    /**
     * Returns a key-value array of data directories as defined in $this->data_directories
     * @return array
     */
    public function getDataDirectories()
    {
        return $this->data_directories;
    }

    
    /**
     * Returns true if this application is a backend only app
     * The value is set from $this->disable_frontend_runtime
     * @return bool
     */
    public function isFrontendRuntimeDisabled()
    {
        return $this->disable_frontend_runtime;
    }    
    /**
     * Get the ajax controller default script file name.
     * @return string
     */
    public function getAjaxControllerScript()
    {
        $expected_name = "/apps/{$this->app_name}/{$this->app_name}.js";
        //$desired_file = str_replace('\\', '/', strtolower($this->app_name));
        if (file_exists(SCHLIX_SITE_PATH . $expected_name)) {
            return SCHLIX_SITE_HTTPBASE . $expected_name;
        }
        elseif (file_exists(SCHLIX_SYSTEM_PATH. $expected_name)) {
            return SCHLIX_SYSTEM_URL_PATH. $expected_name;
        }
        else
            return FALSE;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns true if versioning is enabled
     * @return bool
     */
    public function hasVersioning()
    {
        return $this->has_versioning;
    }
    //_________________________________________________________________________//
    public function hasAuthenticatedUserMainPage()
    {
        return $this->has_logged_in_user_main_page;
    }

    /**
     * Returns true if custom media header allowed
     * @return bool
     */
    public function hasCustomMediaHeader()
    {
        return $this->has_custom_media_header;
    }
    
    //_________________________________________________________________________//
    public function recordLog($message, $type = 'info')
    {
        global $SystemLog;

        if (method_exists($SystemLog, 'Record'))
            $SystemLog->record($message, $this->app_name, $type);
        else
            echo $message;
    }

    
    //_________________________________________________________________________//
    public function logError($message)
    {
        $this->recordLog($message, 'error');
    }
    public function logInfo($message)
    {
        $this->recordLog($message, 'info');
    }
    
    public function logWarning($message)
    {
        $this->recordLog($message, 'warn');
    }
    
    //_________________________________________________________________________//
    /**
     * Redirects to internal action, e.g. /action/myaction. Protocol can be 
     * OPT_REDIRECT_DEFAULT or OPT_REDIRECT_HTTPS or OPT_REDIRECT_HTTP 
     * @param string $action
     * @param string $protocol
     */
    public function redirectToOtherAction($action, $protocol = OPT_REDIRECT_DEFAULT)
    {

        $qry = $this->createFriendlyURL($action);
        switch ($protocol)
        {
            case OPT_REDIRECT_HTTPS: $qry = force_https_url() . $qry;
                break;
            case OPT_REDIRECT_HTTP: $qry = (defined('SCHLIX_SITE_HTTP_URL') ? SCHLIX_SITE_HTTP_URL : SCHLIX_SITE_URL) . $qry;
                break;
        }
        $_SESSION['schlix_internal_redirect'] = 1;
        ob_end_clean();
        ob_start();
        header("Location: {$qry}");
        exit();
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Reset the breadcrumb for this app
     */
    protected function resetBreadCrumbs()
    {
        global $CurrentFrontendApplication;
        
        if ( ($this->app_name != $CurrentFrontendApplication) && ($this->app_name != 'html' || $this->app_name != 'landing')) 
            $this->bread_crumbs[] = array('title' => $this->getApplicationDescription(), 'link' => $this->createFriendlyURL(''));
    }

//_______________________________________________________________________________________________________________//
    public function declarePageLastModified($date)
    {
        if (!is_numeric($date))
            $date = strtotime($date);
        $last_modified = gmdate("D, d M Y H:i:s", $date) . " GMT";
//		header("Expires: {$last_modified}");
        header("Last-Modified: {$last_modified}");
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Adds a name and link to breadcrumb
     * @param string $name
     * @param string $link
     */
    protected function addToBreadCrumbs($name, $link)
    {
        $this->bread_crumbs[] = array('title' => ___h($name), 'link' => $link);
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Displays the breadcrumb
     * @param string $separator
     * @return string
     */
    public function displayBreadCrumbs($separator = ' &raquo; ')
    { 
        $skin_mod = get_current_skin_module();
        return $this->disable_frontend_runtime ? '' :  $skin_mod->standard_breadcrumb($this->bread_crumbs); 
    }
    
    /**
     * Get raw breadcrumbs array
     * @return array
     */
    public function getRawBreadCrumbs()
    { 
        return $this->bread_crumbs;
    }    
    
    /**
     * Returns an array of HTML link of breadcrumbs
     * @deprecated since version 2.2.0
     * @return array
     */
    public function getBreadCrumbsArray()
    { 
        $breadcrumbs = [];
        $site_httpbase = (SCHLIX_SITE_HTTPBASE === '') ? '/' : SCHLIX_SITE_HTTPBASE;
        $breadcrumbs[] = \__HTML::A('Home', $site_httpbase, array ('class'=>'breadcrumb-home')) ;
        $total = ___c($this->bread_crumbs);
        for ($i = 0; $i < $total; $i++)
        {
            $crumb = $this->bread_crumbs[$i];
            $breadcrumbs[] = \__HTML::A($crumb['title'], $crumb['link']);
        }
        return $breadcrumbs;
    }
    /**
     * Returns a config value from key for this app
     * Note: this function has been changed as of July 2019. To get the master config, use getMasterConfig method instead.
     * @global \SCHLIX\cmsRegistryConfig $SystemConfig
     * @global \SCHLIX\cmsUser $CurrentUser
     * @param string $key
     * @param string $default_value
     * @return mixed
     */
    public function getConfig($key, $default_value = null)
    {
        global $SystemConfig;
         
        $app_name = $this->getFullApplicationName();
        if ($this->config == null)
        {
            $existing_cache_retr = \SCHLIX\cmsContextCache::get('app_config_retr', $app_name);
            if ((int) $existing_cache_retr === 1)
                $this->config = \SCHLIX\cmsContextCache::get('app_config', $app_name);
            else 
            {
                $this->config = $SystemConfig->get($app_name);
                \SCHLIX\cmsContextCache::set('app_config', $app_name, $this->config);
                \SCHLIX\cmsContextCache::set('app_config_retr', $app_name, 1);
            }            
        }
        return $this->config ? (array_key_exists($key, $this->config) ? $this->config[$key] : $default_value) : $default_value;
        /* too many queriesglobal $SystemConfig;

        return $SystemConfig->get($this->app_name, $key);*/
    }

    /**
     * Set a config value for this app. 
     * Note: this function has been changed as of July 2019. To set the master config, use setMasterConfig method instead.
     * @global \SCHLIX\cmsRegistryConfig $SystemConfig
     * @param string $key
     * @param string $value
     * @return mixed
     */
    public function setConfig($key, $value)
    {
        global $SystemConfig;

        return $SystemConfig->set($this->getFullApplicationName(), $key, $value);
    }
    
    /**
     * Returns a config value from key for the master app     
     * @global \SCHLIX\cmsRegistryConfig $SystemConfig
     * @global \SCHLIX\cmsUser $CurrentUser
     * @param string $key
     * @param string $default_value
     * @return mixed
     */
    public function getMasterConfig($key, $default_value = FALSE)
    {
        global $SystemConfig;
                
        if (empty($this->master_config))
            $this->master_config = $SystemConfig->get($this->schlix_master_class);
        return $this->master_config ? (array_key_exists($key, $this->master_config) ? $this->master_config[$key] : $default_value) : $default_value;
        /* too many queriesglobal $SystemConfig;

        return $SystemConfig->get($this->app_name, $key);*/
    }

    /**
     * Set a config value for this app
     * @global \SCHLIX\cmsRegistryConfig $SystemConfig
     * @param string $key
     * @param string $value
     * @return mixed
     */
    public function setMasterConfig($key, $value)
    {
        global $SystemConfig;

        return $SystemConfig->set($this->schlix_master_class, $key, $value);
    }
    

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the application name
     * @return type
     */
    public function getApplicationName()
    {
        return $this->app_name;
    }


    //_______________________________________________________________________________________________________________//
    public function getApplicationNameOnly()
    {
        
        return strtolower($this->schlix_master_class);
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Returns the application description not from the database
     * @return type
     */
    public function getOriginalApplicationDescription()
    {
        return $this->app_description;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the application description from database. If it's empty, then the
     * default application description set in the class will be used instead
     * @return string
     */
    public function getApplicationDescription()
    {
        $desc_cache = \SCHLIX\cmsContextCache::get('app_description',$this->app_name);
        if ($desc_cache) 
            return ___h($desc_cache);
        else
        {
            $result = $this->getConfig('str_app_description');
            $app_description = ($result != null) ? $result : $this->app_description;
            return ___h($app_description);
        }
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Change output data
     * @global \App\Core_MacroManager $Macros
     * @param array $data
     * @param string $function_name
     * @param array $extra_info
     * @return bool
     */
    public function processDataOutputWithMacro(&$data, $function_name, $extra_info = NULL)
    {
        global $Macros;

        $data['macro_processed_text_outside_article_bottom'] = null;
        $data['macro_processed_text_outside_article_top'] = null;
        $data['macro_processed_text_outside_article_top'] = null;
        return $Macros->modifyData($data, $this, $function_name, $extra_info); // yes, will process even if text = ''
    }    
    //_______________________________________________________________________________________________________________//
    /**
     * Outputs the page title
     */
    public function displayPageTitle()
    {
        echo $this->getPageTitle();
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Outputs the page meta description
     */
    public function displayPageMetaDescription()
    {
        echo ___h(trim($this->getPageMetaDescription()));
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Outputs the page metta keywords
     */
    public function displayPageMetaKeywords()
    {
        echo ___h(trim($this->getPageMetaKeywords()));
    }

    /*
     *  Returns true if there is a content, otherwise returns false.
     */

//_______________________________________________________________________________________________________________//
    public function viewAuthenticatedUserAuthenticationPage()
    {

        global $CurrentUser;

        $userinfo = $CurrentUser->getCurrentUserInfo();
        if ($userinfo) {
            // display user content;
            return $this->viewAuthenticatedUserMainPage();
        }
        else {
            echo \__HTML::H1(___('Authentication Required'));
            $CurrentUser->viewLoginPage();
            return false;
        }
    }

//_______________________________________________________________________________________________________________//

    public function viewAuthenticatedUserMainPage()
    {
        return false;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Frontend: view the main page.
     */
    public function viewMainPage()
    {
        echo 'Please implement this function';
    }

//_______________________________________________________________________________________________________________//
    protected function probeFriendlyURLDestination($urlpath)
    {
        $site_httpbase_length = strlen(SCHLIX_SITE_HTTPBASE);
        $urlpath = remove_multiple_slashes($urlpath);
        if ($site_httpbase_length > 0) {
            $site_httpbase_pos = strpos($urlpath, SCHLIX_SITE_HTTPBASE);
            if ($site_httpbase_pos !== false && $site_httpbase_pos == 0)
                $urlpath = substr($urlpath, $site_httpbase_length, strlen($urlpath) - $site_httpbase_length);
        }
        $app_alias = $this->full_application_alias;
        if (SCHLIX_SEF_ENABLED && $urlpath == '/' . $app_alias) {
            if (empty($app_alias)) {
                $error_message = sprintf(___('Error: Application alias for %s is empty. Please go to the backend and click the Configuration button for this app'), $this->app_name);
                die($error_message);
            }            
            redirect_url(SCHLIX_SITE_URL . SCHLIX_SITE_HTTPBASE . "/{$app_alias}/");
            exit;
        }
        $url_info = parse_url($urlpath);
        $url = $url_info['path'];
        $url_array = explode('/', $url);
        array_splice($url_array, 0, 2);
        $url = implode('/', $url_array);
        return array('url' => trim($url), 'url_array' => $url_array);
    }

    /**
     * Returns an array of command, given $urlpath.
     * @param string $urlpath
     * @return array
     */
    public function interpretFriendlyURL($urlpath)
    {

        //	$url = str_replace(SCHLIX_SITE_HTTPBASE,'',$url); // must
        // TODO: Optimize with  http_build_str
        $command = null;
        $gt_app = fget_string('app');
        if (SCHLIX_SEF_ENABLED && empty($gt_app)) {
            $command = [];
            $parsedurl = $this->probeFriendlyURLDestination($urlpath);
            $url = $parsedurl['url'];
            $url_array = $parsedurl['url_array'];
            if (empty($url))
            {
                $command['action'] = 'main';
                $command['pg'] = 0;
            }
            else
            if ($url_array[0] == 'action' && $url_array[1]) {
                $chunked_url_array = array_chunk($url_array, 2);
                foreach ($chunked_url_array as $key => $command_chunk)
                    $command[$command_chunk[0]] = $command_chunk[1];
            }
        }
        else {
            $command = $_GET;
        }
        return $command;
    }

    /**
     * create SEO friendly URL. Format is action={...}&param1={....}&param2={...}
     * @param string $str
     * @return string
     */
    public function createFriendlyURL($str)
    {
        $final_url = '';
        if (SCHLIX_SEF_ENABLED) {
            $command_array = [];
            parse_str($str, $command_array); // output to $command_array
            if ((is_array($command_array)) && (array_key_exists('action', $command_array)) && ($command_array['action'] !== '')) {
                foreach ($command_array as $key => $value)
                {
                    $final_url.= "/{$key}/{$value}";
                }
            }
            elseif (strlen($str) > 1)
                $final_url = "/{$str}/";
            else
                $final_url = '/';  // Feb 4, 2012 fix
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . $this->full_application_alias . $final_url;
        }
        else
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . "index.php?app={$this->app_name}&{$str}";
        return remove_multiple_slashes($final_url);
    }
    

    //_______________________________________________________________________________________________________________//
    protected function getStartAndEndForItemPagination($pg = 1, $perpage = DEFAULT_FRONTEND_ITEMS_PERPAGE, $itemscount = 0) { //


        $pg = (int) $pg;
        if (!$pg || $pg < 1)
            $pg = 1;
        if ($perpage == 0)
            $perpage = 10; // division by zero fix - Prana - Nov 11, 2011
        $startat = ($pg - 1) * $perpage;
        $totalpages = ($itemscount + $perpage - 1) / $perpage;
        $endat = $startat + $perpage;

        if ($itemscount < ($startat + $perpage))
            $endat = $itemscount;


        return array('start' => $startat, 'end' => $endat, 'total' => (int) $totalpages);
    }
    
    /**
     * Returns an array of paging items
     * @deprecated since version 2.2.0
     * @param int $pg
     * @param int $pages
     * @param string $new_url_query_string
     * @param bool $friendly
     * @return array
     */
    public function getItemPagination($pg, $pages, $new_url_query_string, $friendly = true)
    {
        $pagination_array = [];

        $pg = (int) $pg;
        $pages = (int)$pages;
        $str = '';
        if ($pg <= 0)
            $pg = 1;
        $prev = $pg - 1;        
        $next = $pg + 1;        
        if ($pg > $pages)
            return false; // page outside of range
        $itemsperpage = $this->getNumberOfListingsPerPage();
        if ($pages > 1) {
            if (strpos($new_url_query_string, '?') !== false) {
                $urlcomp = parse_url($new_url_query_string);
                $new_url_query_array = [];
                parse_str($urlcomp['query'], $new_url_query_array);
                if (array_key_exists('pg', $new_url_query_array))
                    unset($new_url_query_array['pg']);
                $new_url_query_string = $urlcomp['path'] . '?' . http_build_query($new_url_query_array);
            }
          //  $str = sprintf($this->paginationStringFormat . ' ', $pg, $pages);
            //FIRST
            $urlprev = "{$new_url_query_string}"; //&pg=1";
            if ($friendly)
                $urlprev = $this->createFriendlyURL($urlprev);
            //$str.= 
            $pagination_array[] = \__HTML::A(TXT_PAGINATION_FIRST, $urlprev,  ['class'=> 'pagination_first']);
            //PREV
            $urlprev = "{$new_url_query_string}&pg=$prev";
            if ($friendly)
                $urlprev = $this->createFriendlyURL($urlprev);
            if ($prev > 0)
                $pagination_array[] = \__HTML::A(TXT_PAGINATION_PREV, $urlprev,  ['class'=> 'pagination_prev'])  ;
            $beginning = ($itemsperpage * intval($pg / $itemsperpage)) + 1;

            if ($beginning <= 0)
                $beginning = 1;
            if (($pg % $itemsperpage) == 0)
                $beginning -= $itemsperpage;
            $end = $beginning + $itemsperpage;
            for ($i = $beginning; $i < $end && $i <= $pages; $i++)
            {
                $urlpg = "{$new_url_query_string}";
                $urlpg.= ($i > 1) ? "&pg={$i}" : '';
                if ($friendly)
                    $urlpg = $this->createFriendlyURL($urlpg);
                if ($i != $pg)
                     $pagination_array[] = \__HTML::A($i, $urlpg,  ['class'=> 'pagination_number']) ;
                else
                     $pagination_array[] = \__HTML::SPAN(" {$i} ", ['class'=>'pagination_current']);
            }
            //NEXT
            $urlnext = "{$new_url_query_string}&pg={$next}";
            if ($friendly)
                $urlnext = $this->createFriendlyURL($urlnext);
            if ($next < $pages)
                 $pagination_array[] = \__HTML::A(TXT_PAGINATION_NEXT, $urlnext,  ['class'=> 'pagination_next']);
            //LAST
            $urllast = "{$new_url_query_string}&pg={$pages}";
            if ($friendly)
                $urllast = $this->createFriendlyURL($urllast);
             $pagination_array[] = \__HTML::A(TXT_PAGINATION_LAST, $urllast,  ['class'=> 'pagination_last']);
        }
             return $pagination_array;        
    }
    
    
    /**
     * Returns an array of paging items
     * @param int $pg
     * @param int $pages
     * @param string $new_url_query_string
     * @param bool $friendly
     * @return array
     */
    public function getItemPaginationArray($pg, $pages, $new_url_query_string, $friendly = true)
    {
        $pagination_array = [];

        $pg = (int) $pg;
        $pages = (int)$pages;
        $str = '';
        if ($pg <= 0)
            $pg = 1;
        $prev = $pg - 1;        
        $next = $pg + 1;        
        if ($pg > $pages)
            return false; // page outside of range
        $itemsperpage = $this->getNumberOfListingsPerPage();
        if ($pages > 1) {
            if (strpos($new_url_query_string, '?') !== false) {
                $urlcomp = parse_url($new_url_query_string);
                $new_url_query_array = [];
                parse_str($urlcomp['query'], $new_url_query_array);
                if (array_key_exists('pg', $new_url_query_array))
                    unset($new_url_query_array['pg']);
                $new_url_query_string = $urlcomp['path'] . '?' . http_build_query($new_url_query_array);
            }
          //  $str = sprintf($this->paginationStringFormat . ' ', $pg, $pages);
            //FIRST
            $urlprev = "{$new_url_query_string}"; //&pg=1";
            if ($friendly)
                $urlprev = $this->createFriendlyURL($urlprev);
            //$str.= 
            $pagination_array[] = ['type' => 'first', 'url' => $urlprev]; // \__HTML::A(TXT_PAGINATION_FIRST, $urlprev,  ['class'=> 'pagination_first']);
            //PREV
            $urlprev = "{$new_url_query_string}&pg=$prev";
            if ($friendly)
                $urlprev = $this->createFriendlyURL($urlprev);
            if ($prev > 0)
                //$pagination_array[] = \__HTML::A(TXT_PAGINATION_PREV, $urlprev,  ['class'=> 'pagination_prev'])  ;
                $pagination_array[] = ['type' => 'prev', 'url' => $urlprev];
            $beginning = ($itemsperpage * intval($pg / $itemsperpage)) + 1;

            if ($beginning <= 0)
                $beginning = 1;
            if (($pg % $itemsperpage) == 0)
                $beginning -= $itemsperpage;
            $end = $beginning + $itemsperpage;
            for ($i = $beginning; $i < $end && $i <= $pages; $i++)
            {
                $urlpg = "{$new_url_query_string}";
                $urlpg.= ($i > 1) ? "&pg={$i}" : '';
                if ($friendly)
                    $urlpg = $this->createFriendlyURL($urlpg);
                if ($i != $pg)
                     //$pagination_array[] = \__HTML::A($i, $urlpg,  ['class'=> 'pagination_number']) ;
                    $pagination_array[] = ['type' => 'page', 'url' => $urlpg, 'number' => $i];
                else
                    $pagination_array[] = ['type' => 'current', 'url' => $urlpg, 'number' => $i];
                    //$pagination_array[] = \__HTML::SPAN(" {$i} ", ['class'=>'pagination_current']);
            }
            //NEXT
            $urlnext = "{$new_url_query_string}&pg={$next}";
            if ($friendly)
                $urlnext = $this->createFriendlyURL($urlnext);
            if ($next < $pages)
                 //$pagination_array[] = \__HTML::A(TXT_PAGINATION_NEXT, $urlnext,  ['class'=> 'pagination_next']);
                 $pagination_array[] = ['type' => 'next', 'url' => $urlnext];
            //LAST
            $urllast = "{$new_url_query_string}&pg={$pages}";
            if ($friendly)
                $urllast = $this->createFriendlyURL($urllast);
             //$pagination_array[] = \__HTML::A(TXT_PAGINATION_LAST, $urllast,  ['class'=> 'pagination_last']);
            $pagination_array[] = ['type' => 'last', 'url' => $urllast];
        }
             return $pagination_array;        
    }    
//_______________________________________________________________________________________________________________//
    public function displayItemPagination($pg, $pages, $new_url_query_string, $friendly = true)
    {
        $page_array = $this->getItemPaginationArray($pg, $pages, $new_url_query_string, $friendly);
        $skin = get_current_skin_module();
        return $skin->standard_pagination($page_array);
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Set the current page title
     * @param string $title
     */
    public function setPageTitle($title)
    {
        $this->page_title = $title;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the current page title, HTML-encoded UTF-8
     * @return string
     */
    public function getPageTitle()
    {
        global $CurrentFrontendApplication;
        
        $pg_title = $this->page_title ? ___h(trim($this->page_title)) : '';
        // application description is already safe
        $app_description = $this->disable_app_description_in_page_title ? '' : $this->getApplicationDescription();
        $display = [];
        if ($pg_title)
            $display[] = $pg_title;
        if ($app_description && ($app_description != $pg_title))
            $display[] = $app_description;        
        $page_title = implode(' | ', $display);
        return empty($page_title) ? $app_description : $page_title;
    }

    //_______________________________________________________________________________________________________________//    
    /**
     * Returns the page meta description
     * @return string
     */
    public function getPageMetaDescription()
    {
        return $this->page_meta_description;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the page meta keywords
     * @return string
     */
    public function getPageMetaKeywords()
    {
        return $this->page_meta_keywords;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Set the page meta description
     * @param string $meta_description
     */
    public function setPageMetaDescription($meta_description)
    {
        $this->page_meta_description = $meta_description;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Set the page meta keywords
     * @param string $meta_keywords
     */
    public function setPageMetaKeywords($meta_keywords)
    {
        $this->page_meta_keywords = $meta_keywords;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Given a serialized string coded from an array of key/value pair, returns
     * a translated array containing the option
     * @param string|array $serialized_options
     * @return array
     */
    public function translateMetaOptions($serialized_options)
    {
        if ($serialized_options) {
            $options = [];
            $options_array = (is_serialized($serialized_options)) ? @unserialize($serialized_options) : $serialized_options;

            if (is_array($options_array)) {
                //$array_keys = array_keys($options_array);
                foreach ($options_array as $opt => $value)
                    if (is_numeric($opt))
                        $options[$options_array[$opt]] = 1;
                    else
                        $options[$opt] = $options_array[$opt];
                return $options;
            } else 
                return false;
        }
        else
            return false;
    }

    /**
     * Set the property of $this->table_{$table_alias} to an instantiated 
     * class of cmsSQLTable
     * @param string $table_alias
     * @param string $real_table_name
     * @return boolean
     */
    protected function setTable($table_alias, $real_table_name)
    {
        if (!empty($real_table_name)) {

            $table_property_name = "table_{$table_alias}";
            $data_property_name = "_fieldname_{$table_alias}";
            if ($this->$table_property_name != NULL)
                $this->$table_property_name = NULL;
            $this->$table_property_name = new cmsSQLTable($real_table_name);
            //$this->$data_property_name = $this->$table_property_name->getFieldNamesAsArrayKeys(NO_CACHE);
            $this->$data_property_name = $this->$table_property_name->getFieldNamesAsArray(NO_CACHE);
            return true;
        }
        return false;
    }

    public function RunAjax($command)
    {
        if (!is_ajax_request())
            return RETURN_FULLPAGE;
        else
        {            
            $prefix = 'ajx';
            $h = is_postback() ? 'p' : 'g';
            $s_command=  sanitize_action_command($command['action']);
            $func = $prefix.$h.'_'. $s_command;
            if (method_exists($this, $func))
            {
                $result = call_user_func([$this, $func], $command);
                return ajax_echo($result);
            } else 
            {
                return ajax_echo (ajax_reply_invalid_method($s_command));
            }
            
        }        
        return RETURN_FULLPAGE;
    }
    
    /**
     * Runs command. If return value is true, then it will be displayed as a full page
     * If return value is false, AJAX method is assumed
     * @param string $command
     * @return boolean
     */
    public function Run($command)
    {
        if (is_ajax_request())
            return $this->RunAjax ($command);
        else 
        {
            if ($this->disable_app)
                redirect_url(SCHLIX_SITE_URL . SCHLIX_SITE_HTTPBASE );
            
            switch ($command['action'])
            {
                case 'main':$this->viewMainPage();
                    break;
                case '404error':
                default:                
                    display_http_error(404);
                    break;
            }
        }
        return true;
    }

//_______________________________________________________________________________________________________________//
}

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
abstract class cmsApplication_List extends cmsApplication_Basic implements interface_cmsApplication_List
{
    /**
     * Item Field names
     * @var array
     */
    protected $_fieldname_items;
    /**
     * Item SQL table
     * @var cmsSQLTable
     */
    protected $table_items;
    /**
     * Array of functions to be called before save item is performed
     * @var array 
     */
    protected $_before_save_item_functions = [];
    /**
     * Array of functions to be called after save item is performed
     * @var array 
     */
    protected $_after_save_item_functions = [];    
    /**
     * Array of functions to be called for more item validation
     * @var array 
     */    
    protected $_more_save_item_validations = [];
    /**
     * Check item permission before saving
     * @var bool
     */
    protected $_checkSaveItemWritePermission = true;
    /**
     * Item Primary key field. Does not handle multicolumn primary key
     * @var string 
     */
    protected $field_id;
    /**
     * Item title field
     * @var string 
     */
    protected $field_item_title = 'title';
    /**
     * How many items to be listed in the view function (frontend
     * @var int 
     */
    protected $number_of_listings_per_page;
    
    /**
     * Array of string containing fields to be ignored during versioning restore
     * @var string 
     */
    protected $item_restore_keys_to_ignore;    
    /**
     * TODO: FIX THIS - should be in CategorizedList, not List
     * @var string 
     */
    protected $field_item_category_id = 'category_id'; // should be in cmsApplication_CategorizedList    

    /**
     * An array of sortable item fields. This will be translated to other language
     * through fixSortableItemFields. The fieldnames will also be verified
     * @var array 
     */

    protected $sortable_item_fields = 
        [
          ['key' => 'date_created', 'label' => 'Date Created'],
          ['key' => 'date_modified', 'label' => 'Date Modified'],          
          ['key' => 'id', 'label' => 'Item ID'],
          ['key' => 'title', 'label' => 'Title'],
          ['key' => 'sort_order', 'label' => 'Sort Order'],
        ];

    
    /*
     * Array of default options for newly created item
     * @var array 
     */
    protected $default_item_options = array ('display_pagetitle','display_item_summary_noread','display_item_created_by', 'display_item_date_created', 'display_item_date_modified');
    /**
     *Array of default options for mainpage view
     * @var array
     */
    protected $default_mainpage_options = array ('display_items','display_item_summary_noread','display_item_created_by', 'display_item_date_created');    
    
    /**
     * An array of label/value pair for item display options
     * @var array 
     */
    
    protected $item_meta_options;
    
    /**
     * An array of label/value pair for main page display options
     * @var array 
     */
    protected $mainpage_meta_options;
    /**
     * Construct the class with the description and set the main tiem table name 
     * @param string $app_description
     * @param string $table_items
     */
    
    protected $page_meta_keywords;
    
    protected $page_meta_description;
    
    protected $schema_org_type_item;
    
    public function __construct($app_description, $table_items)
    {
        global $SystemDB;
        
        parent::__construct($app_description);
        if ($table_items != NULL) {
            $this->setTable('items', $table_items);
            $this->setEntity('items',  $this->table_items->q());
            $this->field_id = $this->table_items->getPrimaryKeyFieldNames();
        }
        $this->item_restore_keys_to_ignore = array($this->field_id);
        $this->number_of_listings_per_page = DEFAULT_FRONTEND_ITEMS_PERPAGE;
        $this->schema_org_type_item = 'Thing';
        $this->fixSortableItemFields();
    }

    //_______________________________________________________________________________________________________________//
    public function getItemsToViewFromFullVirtualFilename($url, $enable_redirect_no_trailingslash_folder = false)
    {

        define('SLASH_INDEX_DOT_HTML_LENGTH', 10); // '/index.html'
        $url = rawurldecode($url);
        $url_array = explode('/', $url);
//		DEBUG_ARRAY($url_array);
        $command = [];
        $folder_requestpage = 0;
        $found = false;
        $filetype = ''; // category = c, item = i
        $depth = ___c($url_array);
        $url_length = strlen($url);
        $request_filename = basename($url);
        $file_path_info = null;
        if ($request_filename)
            $file_path_info = pathinfo($request_filename);
        if (empty($url) || $url == 'index.html') {
            $command['action'] = 'main';
            $command['pg'] = 0;
            return $command;
        }
        // Step 0 - Reset - clean up possible directory view mode with "/index.html"
        // possible /html/folder1/folder2 (but no trailing slash)

        $is_extension_excluded = isset($file_path_info['extension']) && (( $file_path_info['extension'] == 'html' ) || ( $file_path_info['extension'] == 'do' ));
                
                // array_key_exists('extension', $file_path_info) && in_array($file_path_info['extension'], ['html', 'do']);
        if ($enable_redirect_no_trailingslash_folder && !$is_extension_excluded && ( strrpos($url, '/') != $url_length - 1 )) {
            $url.= '/';
            $url_array = explode('/', $url);
            array_splice($url_array, 0, 2);
            $depth = ___c($url_array);
            $url_length = strlen($url);
            if ((method_exists($this, 'getCategoriesByVirtualFilename')) && $this->getCategoriesByVirtualFilename($request_filename)) {

                $app_alias = $this->full_application_alias;
                redirect_url(SCHLIX_SITE_URL . SCHLIX_SITE_HTTPBASE . "/{$app_alias}/{$url}");
                exit;
            }
            else
                return array('action' => '404error');
        }

        $items = $this->getItemsByVirtualFilename($file_path_info['filename']);
        if (___c($items) > 0) {
            $item = $items[0];
            $compare1 = '/' . $file_path_info['filename'] . '.' . $file_path_info['extension'];
            $compare1 = remove_multiple_slashes($compare1);
            $compare2 = '/' . $url;
            if ($compare1 == $compare2) {
                $command['action'] = 'viewitem';
                $command['id'] = $item['id'];
                return $command;
            }
        }
        $command['action'] = '404error';
        return $command;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array of command, given $urlpath.
     * @param string $urlpath
     * @return array
     */   
    public function interpretFriendlyURL($urlpath)
    {

        if (SCHLIX_SEF_ENABLED && !isset($_GET['app'])) {
            $parsedurl = $this->probeFriendlyURLDestination($urlpath);
            $url = $parsedurl['url'];
            $url_array = $parsedurl['url_array'];

            $command = $this->getItemsToViewFromFullVirtualFilename($url, true);

            if ($command['action'] == '404error') {
                // let's try passing this to child
                if ($url_array[0] == 'action' && $url_array[1]) {
                    $chunked_url_array = array_chunk($url_array, 2);
                    foreach ($chunked_url_array as $key => $command_chunk)
                        $command[$command_chunk[0]] = $command_chunk[1];
                }
            }
        }
        else {
            if (___c($_GET) == 0 || empty($_GET['action']))
            {
                $command['action'] = 'main';
                $command['pg'] = 0;
            }
            else
                $command = $_GET;
        }
        return $command;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * create SEO friendly URL. Format is action={...}&param1={....}&param2={...}
     * @param string $str
     * @return string
     */    
    public function createFriendlyURL($str)
    {
        /*
          input: index.php?app=html&action=view&id=c1
          input: index.php?app=html&action=view&id=i10
          input: index.php?app=users&action=register

          action=view&id=c5&page=10
          action=checkout
          action=login
         */
        $final_url = '';
        if (SCHLIX_SEF_ENABLED) {
            $command_array = [];
            parse_str($str, $command_array); // replaced with this- May 24, 2010
            $command_action = (array_key_exists('action', $command_array)) ? $command_array['action'] : '';
            switch ($command_action)
            {
                case 'viewitem': $final_url = encode_url($this->getFullPathByItemID($command_array[$this->field_id]));
                    break;
                default:if (array_key_exists('action', $command_array) && $command_array['action'] != '') {
                        if (array_key_exists('app', $command_array))
                            unset($command_array['app']);
                        foreach ($command_array as $key => $value)
                        {
                            $final_url.= "/{$key}/{$value}";
                        }
                    }
                    else
                        $final_url = '/'; // For the main app - Feb 4, 2012
            }
            $app_alias = $this->full_application_alias; // May 24, 2010
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . $app_alias . $final_url; // delete - Oct 1, 2010
        } else {
            parse_str("app={$this->app_name}&{$str}", $command_array);
            $array_keys = array_keys($command_array); // separate it
            foreach ($array_keys as $key)
                if (empty($command_array[$key]))
                    unset($command_array[$key]);
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . "index.php?" . http_build_query($command_array);
        }
        $final_url = remove_multiple_slashes($final_url);
//		echo $final_url;
        return $final_url;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the path given item ID. e.g. /folder1/folder2/item.html
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $item_id
     * @return string
     */    
    public function getFullPathByItemID($item_id)
    {
        global $SystemDB;

        $item_id = (int) $item_id;
        $sql = "SELECT virtual_filename FROM {$this->table_items} where {$this->field_id} = '{$item_id}'";
        $item = $SystemDB->getQueryResultSingleRow($sql, true);
        if (!$item)
            return false;
        return '/' . $item['virtual_filename'] . '.html';
    }

    //_______________________________________________________________________________________________________________//
    protected function getMaxDateFromArray($items)
    {
        $last_mod_date = 0;
        if ($items)
        {
            foreach ($items as $item)
            {
                if (strtotime($item['date_modified']) > $last_mod_date)
                {
                    $last_mod_date = $item['date_modified'];
                }
            }
        }
        return $last_mod_date;
    }

    //_______________________________________________________________________________________________________________//
    public function getNumberOfListingsPerPage()
    {
        return $this->number_of_listings_per_page;
    }

//_______________________________________________________________________________________________________________//
    public function setNumberOfListingsPerPage($number)
    {
        $number = (int) $number;
        if ($number <= 0)
            $number = DEFAULT_FRONTEND_ITEMS_PERPAGE;
        $this->number_of_listings_per_page = $number;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the item's table name
     * @return string
     */
    public function getItemTableName()
    {
        return $this->table_items->__toString();
    }    
    
    /**
     * Returns a list of custom fields for the item table
     * @return array
     */
    public function getItemCustomFields()
    {
        $app_cf = new \App\Core_CustomField();
        
        return $app_cf->getCustomFields($this->getItemTableName());
        
    }    
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns the item's table
     * @return \SCHLIX\cmsSQLTable
     */
    public function getItemTable()
    {
        return $this->table_items;
    }

    //_________________________________________________________________________//

    protected function pageCounterIncreaseImpression($id, $app_name, $data_fields, $table, $fieldname, $field_id, $reset = false) 
    {

        global $SystemConfig, $SystemDB;

        if ($SystemConfig->get($app_name, 'bool_enable_pageview_stats') == 1) {
            if (in_array($fieldname, $data_fields)) {
                $id = (int) $id;
                if ($id > 0) {
                    if ($reset)
                        $sql = "UPDATE LOW_PRIORITY {$table} SET {$fieldname}=0 WHERE {$field_id} = :id";
                    else
                        $sql = "UPDATE LOW_PRIORITY {$table} SET {$fieldname}=({$fieldname} + 1) WHERE {$field_id} = :id";
                    $SystemDB->query($sql, ['id' => $id]);
                }
            }
        }
    }

    
    //_______________________________________________________________________________________________________________//
    public function increaseItemPageView($id)
    {
        $this->pageCounterIncreaseImpression($id, $this->app_name, $this->_fieldname_items, $this->table_items, 'pageview', $this->field_id);
    }

    //_______________________________________________________________________________________________________________//
    public function resetItemPageView($id)
    {
        $this->pageCounterIncreaseImpression($id, $this->app_name, $this->_fieldname_items, $this->table_items, 'pageview', $this->field_id, true);
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing field names from $this->table_items
     * @return array
     */
    public function getItemFieldNames()
    {
        return $this->_fieldname_items;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * * Returns the ID field name from $this->table_items (usually just id)
     * @return string
     */
    public function getFieldID()
    {
        return $this->field_id;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the title field name from $this->table_items (usually just title)
     * @return string
     */
    public function getFieldItemTitle()
    {
        return $this->field_item_title;
    }

    /**
     * Delete Item By ID
     * @global \App\Users $CurrentUser
     * @param int $id
     */
    public function deleteItemByID($id)
    {
        global $CurrentUser;
        
        $id = (int) $id;
        $existing_item = $this->getItemByID($id);
        if ($existing_item)
        {
            $this->table_items->q()->delete()->where("{$this->field_id} = :id")->execute(['id' => $id]);
            $CurrentUser->recordCurrentUserActivity("Deleted {$this->app_name} {$this->field_id}# {$id}");
        } else 
        {
            $CurrentUser->recordCurrentUserActivity("FAILED: Attempt to delete {$this->app_name} {$this->field_id}# {$id}");
        }
        
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Deletes items from table. The parameter $mixed_items_to_delete is a pipe-separated
     * value of items, e.g. i4|i5|i1
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $mixed_items_to_delete
     */
    function delete($mixed_items_to_delete)
    {        
        $mixed_items_array = explode('|', $mixed_items_to_delete); // e.g: i4|i14
        // Process sub-folders first
        $items_to_delete = [];
        foreach ($mixed_items_array as $mixed_item)
        {
            $item_id = substr($mixed_item, 1);
            $items_to_delete[] = (int) $item_id;
        }

        if (___c($items_to_delete) > 0) {
            foreach ($items_to_delete as $item_id)
            {
                $this->deleteItemByID($item_id);
            }
        }
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns total item count with specified critiera (e.g. status > 0)
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $criteria
     * @param bool $cache
     * @return int
     */
    public function getTotalItemCount($criteria = '', $cache = false)
    {
        global $SystemDB;

        $criteria_txt = '';
        if (!empty($criteria))
            $criteria_txt = " WHERE {$criteria}";

        $sql = "SELECT COUNT({$this->field_id}) as total_item_count FROM {$this->table_items} {$criteria_txt}";
        $total = $SystemDB->getQueryResultSingleRow($sql, $cache);
        return $total['total_item_count'];
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Queries $this->table_items
     * 
     * @param string $fields
     * @param string $extra_criteria
     * @param int $start
     * @param int $end
     * @param string $sortby
     * @param string $sortdirection
     * @param bool $from_cache
     * @return array
     */
    public function getAllItems($fields = '*', $extra_criteria = '', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC', $from_cache = false)
    {
        return $this->table_items->selectAll($fields, $extra_criteria, $sortby, $sortdirection, $start, $end, false, SCHLIX_SQL_ENFORCE_ROW_LIMIT, $from_cache);
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Returns any object by ID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     * @param bool $from_cache
     * @return array
     */
    public function getItemByID($id, $from_cache = false)
    {
        global $SystemDB;
        
        $id = (int) $id;
        
        if ($id > 0)
        {
            $item = $from_cache ? 
                 $SystemDB->getCachedQueryResultSingleRow("SELECT * FROM {$this->table_items} WHERE {$this->field_id} = '{$id}'") :
                $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_items} WHERE {$this->field_id} = '{$id}'");
        }
        else
            $item = null;
        return $item;
    }
    
    /**
     * Returns an item with specified GUID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $guid
     * @return array
     */
    public function getAnyObjectByGUID($guid)
    {
        global $SystemDB;

        if ($this->itemColumnExists('guid') )
        {
            $item_row = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_items} WHERE `guid` = :guid", ['guid' => $guid]);
            if ($item_row)
            {
                $item_row['__type__'] = 'item';
                return $item_row;
            }
        }
        return NULL;
    }
    
    /**
     * Returns an item with additional data gathered by calling hook function
     * @param int $id
     * @return array
     */
    public function getItemByIDWithExtraData($id)
    {
        $item = $this->getItemByID($id);
        if ($item)
        {
            $hook_result = cmsHooks::executeModifyReturn(2, __FUNCTION__, $this, $item) ;
            if ($hook_result)
                return $hook_result;
        }        
        return $item;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns one or more item with the same virtual filename. Used in later classes.
     * There's a parameter $category_id that's not supposed to be in this class but it's there 
     * for compatibility with inherited classes
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $input_filename
     * @param int $category_id
     * @return array
     */
    public function getItemsByVirtualFilename($input_filename, $category_id = -1)
    {
        global $SystemDB;

        if (!empty($input_filename)) {
            $str = "";
            $category_id = (int) $category_id;
            $filename = sanitize_string(rawurldecode($input_filename));
            if ($category_id >= 0)
                $str = " AND {$this->field_item_category_id} = {$category_id}";
            $sql = "SELECT * FROM {$this->table_items} WHERE virtual_filename = {$filename}{$str}";
            $items = $SystemDB->getQueryResultArray($sql, true);
            return $items;
        }
        else
            return false;
    }

    //_______________________________________________________________________________________________________________//
    public function getBreadCrumbsByItemID($item_id)
    {
        $item_id = (int) $item_id;
        $item = $this->getItemByID($item_id);
        if (!$item)
            return false;

        if (!($this->app_name == 'html' && $item['virtual_filename'] == 'home'))
            $this->addToBreadCrumbs($item['title'], $this->createFriendlyURL("action=viewitem&id={$item['id']}"));
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Displays a HTML-encoded page meta description
     */
    public function displayPageMetaDescription()
    {
        $default_app_meta_desc = $this->getConfig('meta_description');
        echo trim(___h((empty($this->page_meta_description) ? ___h($default_app_meta_desc) : $this->page_meta_description)));
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Displays a HTML-encoded page meta keywords
     */
    public function displayPageMetaKeywords()
    {
        $default_app_meta_keywords = $this->getConfig('meta_keywords');
        echo trim(___h((empty($this->page_meta_key) ? ___h($default_app_meta_keywords) : $this->page_meta_key)));
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns a non-HTML encoded string of meta description
     * @return string
     */
    public function getPageMetaDescription()
    {
        return $this->page_meta_description;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns a non-HTML encoded string of meta keywords
     * @return string
     */
    public function getPageMetaKeywords()
    {
        return $this->page_meta_keywords;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Sets the page meta description
     * @param string $meta
     */
    public function setPageMetaDescription($meta)
    {
        $this->page_meta_description = $meta;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Sets the page meta keywords
     * @param string $meta
     */
    public function setPageMetaKeywords($meta)
    {
        $this->page_meta_keywords = $meta;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Loads the item and display it with view.item.template.php
     * @param int $id
     * @param bool $from_cache
     */
    public function viewItemByID($id = 1, $from_cache = false)
    {
        $item = $this->getItemByIDWithExtraData($id, $from_cache);
        if ($item['status'] <= 0)
        {
            echo ___('Item is not published');
        } else
        {
            $opt = isset($item['options']) ? $item['options'] : null;
            $item_meta_options = $this->translateMetaOptions($opt);        
            $this->setPageTitle($item['title']);
            if ($this->itemColumnExists('meta_description'))
                $this->setPageMetaDescription($item['meta_description']);
            if ($this->itemColumnExists('meta_key'))
                $this->setPageMetaKeywords ($item['meta_key']);
            $this->getBreadCrumbsByItemID($item['id']);
            if ($this->itemColumnExists('date_modified') && $item['date_modified'] != NULL_DATE && !empty($item['date_modified']))
                $this->declarePageLastModified($item['date_modified']);

            $item['macro_processed_text_outside_article_top'] = null;
            $item['macro_processed_text_after_article'] = null;
            $this->processDataOutputWithMacro($item, __FUNCTION__, array('item_meta_options'=>$item_meta_options));        
            $this->loadTemplateFile('view.item', array('item' => $item, 'item_meta_options' => $item_meta_options));
            $this->increaseItemPageView($item[$this->field_id]);
        }
    }

    // internal functions
//_______________________________________________________________________________________________________________//
    public function displayPageTitle()
    {
        echo htmlspecialchars($this->page_title);
    }
    

//_______________________________________________________________________________________________________________//
    public function genericSearch($fieldname, $keyword, $fields_tobe_selected = '*', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC')
    {
        global $SystemDB;

        $cleankeyword = sanitize_string("%{$keyword}%");

        $search_result = $this->getAllItems($fields_tobe_selected, "{$fieldname} LIKE {$cleankeyword}", $start, $end, $sortby, $sortdirection, true, true);

        return $search_result;
    }

    /**
     * Suggest a non duplicate value from an associative array
     * Internally used to prevent duplicate value in field name (usually virtual_filename)
     * 
     * @internal
     * @param string $key
     * @param array $associative_array
     * @param string $possible_duplicate
     * @return string
     */
    protected function __suggestNewNonDuplicateValueFromArray($key, $associative_array, $possible_duplicate)
    {
        if (is_array($associative_array) && $key)
        {
            $list_of_duplicates = array_column($associative_array,$key);
            $i = 1;
            $suggested = $possible_duplicate . '_' . $i;
            while (in_array($suggested, $list_of_duplicates))
            {
                $suggested = $possible_duplicate . '_' . $i;
                $i++;
            }
            return $suggested;        
        } else return $possible_duplicate;        
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns a new name if there's an item with the same name in the specified
     * $fieldname
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $fieldname
     * @param int $id
     * @param string $possible_duplicate
     * @return string
     */
    public function preventDuplicateValueInItemTable($fieldname, $possible_duplicate, $id)
    {
        global $SystemDB;

        $id = (int) $id;
        if (!$this->itemColumnExists($fieldname))
            return $possible_duplicate;
        //echo "ID = {$id}";die;
        //$possible_duplicate = str_replace('%','', $possible_duplicate);
        //$sanitized_dup_value = sanitize_string($possible_duplicate.'%');
        $sql = "SELECT {$fieldname} FROM {$this->table_items} WHERE LOWER(`{$fieldname}`) = LOWER (:possible_duplicate) AND ({$this->field_id} <> $id) ";
        $results = $SystemDB->getQueryResultArray($sql, ['possible_duplicate' => $possible_duplicate]);        
        $suggestion = null;
        if (___c($results) > 0)
        {
            $suggestion = $this->__suggestNewNonDuplicateValueFromArray($fieldname, $results, $possible_duplicate) ;
            return $this->preventDuplicateValueInItemTable($fieldname, $suggestion, $id);
        } else return $possible_duplicate;
        //return ($results) ?  $this->__suggestNewNonDuplicateValueFromArray($fieldname, $results, $possible_duplicate) : $possible_duplicate;
        return $suggestion;
    }


//_______________________________________________________________________________________________________________//
    public function itemColumnExists($fieldname)
    {
        return in_array($fieldname, $this->_fieldname_items);
    }

    /**
     * Restores an item from global versioning
     * @global \App\Core_Versioning $Versioning
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param int $id
     * @param int $version
     * @return null
     */
    public function restoreItem($id, $version)
    {
        $current_item = $this->getItemByID($id);
        return $this->restoreGeneric($id, $this->field_id, $version, $current_item, $this->table_items, $this->item_restore_keys_to_ignore);
    }

    //_______________________________________________________________________________________________________________//
    /**
     *
     * @global \App\Core_Versioning $Versioning
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param int $id
     * @param string $id_field_name
     * @param double $version
     * @param array $current_item
     * @param string $table_name
     * @param array $keys_to_ignore
     * @return boolean|null
     */
    protected function restoreGeneric($id, $id_field_name, $version, $current_item, $table_name, $keys_to_ignore)
    {
        global $Versioning, $SystemDB, $SystemLog;
        // keys that will not be updated

        $id = (int) $id;
        $version = doubleval($version);

        $guid = $current_item['guid'];
        $item_previous_version = $Versioning->getItemArchiveByVersion($this->app_name, $guid, $version);
        if ($item_previous_version) {
            $previous_column_names = array_keys($item_previous_version);
            $current_column_names = array_keys($current_item);
            // set NULL for all columns not exists in previous version
            foreach (array_diff($current_column_names, $previous_column_names) as $col)
            {
                $item_previous_version[$col] = NULL;
            }
            // unset all columns not exists in current version
            foreach (array_diff($previous_column_names, $current_column_names) as $col)
            {
                if (!$SystemDB->tableColumnExists($table_name, $col))
                    unset($item_previous_version[$col]);
            }

            // set the current version to the new one
            $item_previous_version['version'] = $current_item['version'] + 0.2;
            // saves current version
            $current_item['version']+=0.1;
            $archiving_result = $Versioning->save($this, $current_item);

            // delete all the keys to ignore
            foreach ($keys_to_ignore as $key)
            {
                unset($item_previous_version[$key]);
            }
            $SystemDB->query("LOCK TABLES `{$table_name}` WRITE");
            $SystemDB->simpleUpdate($table_name, $item_previous_version, $id_field_name, $id);                
            $SystemDB->query("UNLOCK TABLES");
            if ($item_previous_version['title'] != '')
                $extra_info = "and title {$item_previous_version['title']}";
            $SystemLog->record("RESTORE: App {$this->app_name} with ID #{$id} {$extra_info} and GUID {$guid} was restored to version {$version}");
            return TRUE;
        } else {
            $SystemLog->record("RESTORE: FAILED - App {$this->app_name} - could not find item ID #{$id} with version {$version} to be restored");

            return NULL;
        }
    }
   
    /**
     * Internal helper to set the return value of save to OK. 
     * @internal
     * @param array $retval
     * @param int $id
     */
    protected function setSaveStatusOK(&$retval,$id, $is_new = false) {
        $retval['id'] = $id;
        $retval['status'] = SAVE_OK;
        $retval['is_new'] = $is_new;
    }
    

    /**
     * Internal helper to set the return value of save to OK. 
     * @internal
     * @param array $retval
     * @param int|string $id
     * @param string|array $error_string_or_array
     */
    protected function setSaveStatusError(&$retval,$id,$error_string_or_array) {
        $retval['id'] = $id;
        $retval['status'] = SAVE_INVALID_DATA;
        if ($error_string_or_array)
            $this->addErrorToSaveStatus($retval, $error_string_or_array);
    }
    
    /**
     * Adds a string or an array to error list in the retval
     * @param array $retval
     * @param int $id
     * @param string|array $error_string_or_array
     */
    protected function addErrorToSaveStatus(&$retval,$error_string_or_array) 
    {
        if (!array_key_exists('errors', $retval))
        {
            $retval['errors'] = [];
        }
        if (is_array($error_string_or_array))
        {
            $retval['errors'] = array_merge($retval['errors'], $error_string_or_array);
        } else
        {
            $retval['errors'][] = $error_string_or_array;        
        }
    }
    
    /**
     * Given an associative array $data, find possible duplicate from item's table
     * @param array $data
     * @return boolean
     */
    public function findDuplicateItems($data)
    {
        return false;
    }
    /**
     * Validates save item. If there's an error, it will return an array
     * with one or more error string, otherwise it will return a boolean true
     * @global \App\Users $CurrentUser
     * @param array $datavalues
     * @return bool|array String array
     */
    public function getValidationErrorListBeforeSaveItem($datavalues)
    {
        global $CurrentUser;
        
        $id = $datavalues[$this->field_id];
        $int_id = (int)$id;        
        $error_list = [];
        if ($this->findDuplicateItems($datavalues)) 
        {
            $error_list[] =  ___('Duplicate item found');
        }

        if ($int_id == 0 && $id != 'new')
        {
            $error_list[] = ___('Invalid item ID ');
        }       
        
        if ($this->_checkSaveItemWritePermission && $int_id > 0 && $this->itemColumnExists('permission_write')) {
            $existing_item = $this->getItemByID($id);
            $hasWritePermission = $CurrentUser->hasWritePermission($existing_item['permission_write']);
            if (!$hasWritePermission) {
                $error_list[] = ___('You do not have a write access to this item');
            }
        }
        
        $dir_errors = $this->initializeDataDirectories();
        $hook_errors = cmsHooks::execute(__FUNCTION__, $this, $datavalues) ;
        if (is_array($hook_errors))
            return @array_merge($error_list, $dir_errors, $hook_errors);
        else
            return @array_merge($error_list, $dir_errors);
        //return @array_merge($error_list, $dir_errors, $hook_errors);
    }
    
    /**
     * Do something after save item
     * @global \App\Core_Versioning $Versioning
     * @global \App\Tag $TagManager
     * @param array $datavalues
     * @param array $original_datavalues
     * @param array $previous_item
     * @param array $retval
     */
    protected function onAfterSaveItem($datavalues, $original_datavalues, $previous_item, $retval)
    {
        
        if ($retval['status'] == SAVE_OK)
        {
            cmsHooks::execute(__FUNCTION__, $this, $datavalues, $original_datavalues, $previous_item, $retval) ;
        }
    }
    /**
     * Set default save item variables
     * @param array $datavalues
     * @return array
     */
    protected function modifyDataValuesBeforeSaveItem($datavalues)
    {
        
        $current_date_time = get_current_datetime();
        $id = $datavalues[$this->field_id];
        $int_id = (int) $id;        
        
        // If it's a new item
        if ($id == 'new')
        {          
            if ($this->itemColumnExists('date_created')) 
                $datavalues['date_created'] = $current_date_time;
            if ($this->itemColumnExists('date_modified')) 
                $datavalues['date_modified'] = $current_date_time;
            
            /*if (array_key_exists('date_modified', $datavalues))
                if (strtotime($datavalues['date_modified']) == 0)
                    $datavalues['date_modified'] = $current_date_time;*/
            if (array_key_exists('date_available', $datavalues))
                if (strtotime($datavalues['date_available']) == 0)
                    $datavalues['date_available'] = $current_date_time;
            if ($this->itemColumnExists('guid'))
                $datavalues['guid'] = new_uuid_v4();
        } 
        // If it's an existing item
        else if ($int_id > 0)
        {
            $existing_item_id = (int) $datavalues[$this->field_id];
            $previous_item = $this->getItemByID($existing_item_id);
            
            if ($this->itemColumnExists('date_created')) {
                if ((strtotime($previous_item['date_created']) == 0))
                    $datavalues['date_created'] = $current_date_time;
            }

            $dv_date_modified = isset($datavalues['date_modified']) ? $datavalues['date_modified'] : null;
            if ($this->itemColumnExists('date_modified') &&
                (($dv_date_modified == $previous_item['date_modified']) || ($dv_date_modified == NULL) ||
                (strtotime($dv_date_modified) == 0))
            ) {
                $datavalues['date_modified'] = $current_date_time;
            }
            if ($this->itemColumnExists('guid') && $previous_item['guid'] == '')
                $datavalues['guid'] = new_uuid_v4();
            $id = $existing_item_id;
            
        }
        $datavalues = cmsHooks::executeModifyReturn(2, __FUNCTION__, $this, $datavalues) ;
        return $datavalues;
    }
    
    /**
     * Internal helper to add hook to function array
     * @param array $array
     * @param object $object
     * @param string $function_name
     * @return bool
     */
    protected function __addHookToFunctionArray(&$array, $object, $function_name)
    {
        if (method_exists($object, $function_name))
        {
            $array[] = array('object' => $object, 'function' => $function_name);
            return true;
        }
        return false;
    }
    /**
     * Add hook to before save Item Event function
     * @param object $object
     * @param string $function_name
     * @return bool
     */
    public function addBeforeSaveItemValidationHook($object, $function_name)
    {
        return $this->__addHookToFunctionArray($this->_more_save_item_validations, $object, $function_name);
    }    

    /**
     * Add hook to after save Item Event function
     * @param object $object
     * @param string $function_name
     * @return bool      
     */
    public function addAfterSaveItemFunctionHook($object, $function_name)
    {
        return $this->__addHookToFunctionArray($this->_after_save_item_functions, $object, $function_name);
    }        
    /**
     * Add hook to before save Item Event function
     * @param object $object
     * @param string $function_name
     * @return bool* 
     */
    public function addBeforeSaveItemFunctionHook($object, $function_name)
    {
        return $this->__addHookToFunctionArray($this->_before_save_item_functions, $object, $function_name);
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Saves item to database. If the item field has the following field:
     * guid, version, date_created, date_modified, date_available, it will be filled
     * automatically
     * @global cmsDatabase $SystemDB
     * @global basicVersioning $Versioning
     * @param int $id
     * @return array Array containing ('status','id','errors')
     */
    public function saveItem($id, $alternative_datavalues = NULL) {
        global $SystemDB;

        $retval = [];

        if ($alternative_datavalues !== NULL && !is_array($alternative_datavalues))
        {
            $this->setSaveStatusError($retval, $id, ___('Invalid save data from array - type is not array'));
            return $retval;
        }
        // first, accept any POST keys regardless. It will be filtered again
        $datavalues = $original_datavalues = ($alternative_datavalues == NULL) ? select_http_post_variables() : $alternative_datavalues;
        if (empty($datavalues))
        {
            $this->setSaveStatusError($retval, $id, ___('Invalid save data from array - no data values matches field names'));
            return $retval;
        }
        // modify it first ... dev can override it
        $datavalues = $this->modifyDataValuesBeforeSaveItem($datavalues);
       // Save Item Hook
        if ($this->_before_save_item_functions)
            foreach ($this->_before_save_item_functions as $objfunc)
                if (method_exists($objfunc['object'], $objfunc['function']))
                    $datavalues = $objfunc['object']->{$objfunc['function']}($datavalues);
        /////////////////////////// BACKWARD COMPATIBILITY UNTIL END OF 2021 ////////////////////
        ///////// TODO: REMOVE
        if (array_key_exists('permission_read', $datavalues) && (is_serialized($datavalues['permission_read']) && (str_starts_with($datavalues['permission_read'], 's:') && $datavalues['permission_read'] != 's:8:"everyone";')))
                $datavalues['permission_read'] = unserialize ($datavalues['permission_read']);
        if (array_key_exists('permission_write', $datavalues) && is_serialized($datavalues['permission_write']) && str_starts_with($datavalues['permission_write'], 's:'))
                $datavalues['permission_write'] = @unserialize ($datavalues['permission_write']);
        /////////////////////////// END WORKAROUND ////////////////////////////
                    
        // Validation
        $error_list = $this->getValidationErrorListBeforeSaveItem($datavalues);
        if (!empty($error_list)) {
            $this->setSaveStatusError($retval, $id, $error_list);
            return $retval;
        }
        
        // Perform additional validation hook
        if ($this->_more_save_item_validations)
            foreach ($this->_more_save_item_validations as $objfunc)
                if (method_exists($objfunc['object'], $objfunc['function']))
                {
                    $more_error_list = $objfunc['object']->{$objfunc['function']}($datavalues);
                    if (!empty($more_error_list)) {
                        $this->setSaveStatusError($retval, $id, $more_error_list);
                        return $retval;
                    }
                }                
        // end validation
        // just in case there's more extra stuff after modification of data values
                
        $datavalues = array_intersect_key($datavalues,  array_fill_keys($this->_fieldname_items, NULL));                
        $previous_item = NULL;
        if ($id == 'new') {
            unset($datavalues[$this->field_id]);
            $SystemDB->query("LOCK TABLES `{$this->table_items}` WRITE");
            $this->table_items->quickInsert($datavalues);
            $this->setSaveStatusOK($retval, $SystemDB->getLastInsertID(), true);
            $SystemDB->query("UNLOCK TABLES");
        }
        else if ((int) $datavalues[$this->field_id] > 0) {
            $existing_item_id = (int) $datavalues[$this->field_id];
            $previous_item = $this->getItemByID($existing_item_id);
            if ($previous_item)
            {
                $this->table_items->quickUpdate($datavalues, "{$this->field_id} = {$existing_item_id}");                
                $this->setSaveStatusOK($retval, $existing_item_id, false);
            } else
            {
                 $this->setSaveStatusError($retval, $id, sprintf(___('Could not save an existing item with ID: %s'), $existing_item_id));
            }
        } else 
        {
            $this->setSaveStatusError($retval, $id, ___('Invalid item ID - validation failed'));
        }
        $this->onAfterSaveItem($datavalues, $original_datavalues, $previous_item, $retval);
        // AFTER Save Item Hook
        if ($this->_after_save_item_functions)
            foreach ($this->_after_save_item_functions as $objfunc)
                if (method_exists($objfunc['object'], $objfunc['function']))
                    $objfunc['object']->{$objfunc['function']}($datavalues, $original_datavalues, $previous_item, $retval);
        return $retval;
    }

//_______________________________________________________________________________________________________________//
    protected function refineFullTextSearchCriteria($keyword, $fields)
    {
        if (empty($fields) || empty($keyword))
            return false;
        $keyword = str_replace('%','', $keyword);
        $cleankeyword = sanitize_string("%{$keyword}%");
        $search_criteria_array = [];
        if (strpos($fields, ',')) {
            $fields_tobe_selected = explode(',', $fields);
            $fields_count = ___c($fields_tobe_selected);
            $total_available_fields = 0;
            $i = 0;
            foreach ($fields_tobe_selected as $field)
            {
                if ($this->itemColumnExists($field))
                    $total_available_fields++;
            }
            if ($total_available_fields == 0)
                return false;

            for ($i = 0; $i < $fields_count; $i++)
            {
                $field = $fields_tobe_selected[$i];
                if ($this->itemColumnExists($field)) {
                    if ($field != 'id' && $field != 'status')
                        $search_criteria_array[] = "{$field} LIKE {$cleankeyword}";
                    //if ($i!= $total_available_fields-1 && $total_available_fields != 1) $search_criteria.= ' OR ';
                }
            }
            $search_criteria = implode(' OR ', $search_criteria_array);
        } else {
            $fields_tobe_selected = $fields;
            if ($fields_tobe_selected != '')
                if (!$this->itemColumnExists ($fields_tobe_selected))
                    return false;
            $search_criteria = "{$fields} LIKE {$cleankeyword}";
        }
        return $search_criteria;
    }


    //_______________________________________________________________________________________________________________//
    /**
     * Returns the number of search result with a sepcific keyword. Default criteria is status > 0
     * 
     * @param string $keyword
     * @param string $fields
     * @param string $criteria
     * @return int
     */
    public function getItemTotalFullTextSearchResultCount($keyword, $fields = 'title,summary,description', $criteria = 'status > 0')
    {   // April 2012 - moved here to app_basic
        $unvalidated_field_names = str_replace('`','', $fields);
        $unvalidated_columns = explode(',', $unvalidated_field_names);
        $validated_columns = [];
        foreach ($unvalidated_columns as $column)
        {
            $column = trim($column);
            if ($this->itemColumnExists($column))
                $validated_columns[] = $column;// '`'.$column.'`';
        }
        if ($validated_columns)
            $fields = implode(',', $validated_columns);
        else
            return false;
        
        $criteria_txt = '';
        if (!empty($criteria))
            $criteria_txt = " AND {$criteria}";
        $search_criteria = $this->refineFullTextSearchCriteria($keyword, $fields);
        if ($search_criteria)
            return $this->getTotalItemCount('('.$search_criteria .') '. $criteria_txt);
        else
            return false;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array of search result
     * 
     * @param string $keyword
     * @param string $criteria
     * @param string $fields
     * @param string $start
     * @param string $end
     * @param string $sortby
     * @param string $sortdirection
     * @return array
     */
    public function getItemFullTextSearchResult($keyword, $criteria = '', $fields = 'title,summary,description', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC')
    {
        $unvalidated_field_names = str_replace('`','', $fields);
        $unvalidated_columns = explode(',', $unvalidated_field_names);
        $validated_columns = [];
        foreach ($unvalidated_columns as $column)
        {
            $column = trim($column);
            if ($this->itemColumnExists($column))
                $validated_columns[] = $column;// '`'.$column.'`';
        }
        
        if ($validated_columns)
            $fields = implode(',', $validated_columns);
        else
            return false;
        $criteria_txt = '';
        if (!empty($criteria))
            $criteria_txt = " AND ({$criteria})";
        $search_criteria = $this->refineFullTextSearchCriteria($keyword, $fields);
        if ($search_criteria) {
            $full_criteria = $search_criteria;
            $full_criteria = '(' . $search_criteria . ')' . $criteria_txt;
            $all_items = $this->getAllItems('*', $full_criteria, $start, $end, $sortby, $sortdirection, true, true);
            $final_result = [];
            if ($this->itemColumnExists('options'))
            {
                foreach ($all_items as $item)
                {
                    $option = $this->translateMetaOptions($item['options']);
                    if (!isset($option['exclude_from_search']))
                        $final_result[] = $item;
                }
                return $final_result;
            } else return $all_items;
        }
        else
            return false;
    }
    
    /**
     * Translate an array to the current language and match it with the other fields
     * @param array $fields
     * @param array $match_with_fields
     * @return array
     */
    protected function fixSortableFields($fields, $match_with_fields)
    {
        $newfields = [];
        $count = ___c($fields);
        $match_with_fields = empty($match_with_fields) ? [] : $match_with_fields;
        for ($i = 0; $i < $count; $i++)
        {
            if (in_array($fields[$i]['key'], $match_with_fields))
            {
                // PHP 7.2
                $fields[$i]['label'] = ___($fields[$i]['label']);                
                $newfields[] = $this->sortable_item_fields[$i];
            }
        }
        return $newfields;
    }
    
    /**
     * Translate label of sortable item fields and match it with item fields
     */
    protected function fixSortableItemFields()
    {
        
        $this->sortable_item_fields = $this->fixSortableFields($this->sortable_item_fields, $this->_fieldname_items);
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Returns a label/key pair array containing a list of sortable item fields
     * This method replaces the obsoleted getSortableItemColumns
     * @return array
     */
    public function getSortableItemColumns() {
        
        return $this->sortable_item_fields;
    }

    /**
     * Returns an array of default item options. If it's not defined, the 
     * hardcoded property of default_item_options is returned instead
     * @return array
     */
    public function getDefaultItemMetaOptionKeys()
    {
        $db_config_Value = $this->getConfig('array_default_item_meta_options');
        return (empty($db_config_Value) || !is_array($db_config_Value)) ? $this->default_item_options : $db_config_Value;
            
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of item options.
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */

    public function getItemMetaOptionKeys() {
        return [
            ['label' => ___('Item Options'),
            'type' => 'checkboxgroup',
            //'key' => 'opt_items',
            'options' =>
              [
                ['key' => 'display_pagetitle', 'label' => ___('Display page title')],
                ['key' => 'display_error_no_access', 'label' => ___('Display errors for inaccesible item')],
                ['key' => 'display_item_summary_noread', 'label' => ___('Display summary for users with no read access')],
                ['key' => 'display_item_created_by', 'label' => ___('Display created by')],
                ['key' => 'display_item_modified_by', 'label' => ___('Display last modified by')],
                ['key' => 'display_item_date_created', 'label' => ___('Display date created')],
                ['key' => 'display_item_date_modified', 'label' => ___('Display date modified')],
                ['key' => 'display_item_view_count', 'label' => ___('Display view count')],
                // new as of v2.0.9
                ['key' => 'hide_item_summary_in_detail_view', 'label' => ___('Display only description in detail view without summary')],
                // new as of v2.0.3
                ['key' => 'exclude_from_search', 'label' => ___('Exclude from search')],
                ['key' => 'exclude_from_sitemap', 'label' => ___('Exclude from XML sitemap')],
                ['key' => 'disable_commenting', 'label' => ___('Disable further commenting')],
                ['key' => 'hide_all_comments', 'label' => ___('Hide all existing comments')]
            ]
          ]
        ];
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. 
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */    
    public function getMainpageMetaOptionKeys()
    {
        
        return [
            // items
            [   'label' => ___('Child Items Listing Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_child_items',
                'options' =>
                [
                    ['key' => 'display_items', 'label' => ___('Display list of items')],
                    ['key' => 'display_item_summary', 'label' => ___('Display summary')],
                    ['key' => 'display_item_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_item_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_item_date_modified', 'label' => ___('Display date modified')],
                    ['key' => 'display_item_read_more_link', 'label' => ___('Display "Read More" link')],
                    ['key' => 'display_item_view_count', 'label' => ___('Display view count')]
                ]
              ],
            // items
            [   'label' => ___('Sort Options'),
                'options' =>
                [
                    ['key' => 'items_sortby', 'type' => 'dropdownlist', 'label' => ___('Sort child items by'), 
                      'options' => $this->getSortableItemColumns()],
                    ['key' => 'items_sortdirection', 'type' => 'radiogroup', 'label' => ___('Child items sort direction'), 
                      'options' => 
                        [['key' => 'desc', 'label' => ___('Descending')], 
                        ['key' => 'asc', 'label' => ___('Ascending') ]]]

                ]
              ]
          ];         
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array of default mainpage options. If it's not defined, the 
     * hardcoded property of default_mainpage_options
     * @return array
     */
    public function getDefaultMainpageMetaOptionKeys()
    {
        $db_config_Value = $this->getConfig('array_mainpage_meta_options');
        return (empty($db_config_Value) || !is_array($db_config_Value)) ? $this->default_mainpage_options : $db_config_Value;
    }    
    //_______________________________________________________________________________________________________________//
    /**
     * Runs command. The main key 'action' in the command array is the router
     * @param array $command
     * @return boolean
     */
    public function Run($command)
    {

        switch ($command['action'])
        {
            case 'viewitem': $this->viewItemByID(intval($command[$this->field_id]), $this->cache);
                break;
            default: return parent::Run($command);
                break;
        }
        return true;
    }

//_______________________________________________________________________________________________________________//
}

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
abstract class cmsApplication_CategorizedList extends cmsApplication_List implements interface_cmsApplication_CategorizedList
{

    /* Array of functions to be called before save item is performed
     * @var array 
     */
    protected $_before_save_category_functions = [];
    /**
     * Array of functions to be called after save item is performed
     * @var array 
     */
    protected $_after_save_category_functions = [];    
    /**
     * Array of functions to be called for more item validation
     * @var array 
     */    
    protected $_more_save_category_validations = [];
    /**
     * Check item permission before saving
     * @var bool
     */
    protected $_checkSaveCategoryWritePermission = true;

    /**
     * An array of sortable item fields
     * @var array 
     */

    protected $sortable_category_fields =
        [
          ['key' => 'cid', 'label' => 'Category ID'],
          ['key' => 'title', 'label' => 'Title'],
          ['key' => 'date_created', 'label' => 'Date Created'],
          ['key' => 'date_modified', 'label' => 'Date Modified'],
          ['key' => 'sort_order', 'label' => 'Sort Order']
        ];
    
    /**
     * Array of default options for newly created category
     * @var array 
     */

    protected $default_category_options = array ('display_pagetitle','display_child_categories','display_items');
    
    /**
     * An array of label/value pair for item display options
     * @var array 
     */
    protected $category_meta_options;
    
    /**
     *Array of default options for mainpage view
     * @var array
     */
    protected $default_mainpage_options = array ('display_child_categories', 'display_items','display_item_summary_noread','display_item_created_by', 'display_item_date_created');    
    
    /**
     * An array containing category field names
     * @var array 
     */
    protected $_fieldname_categories;

    /**
     * $table_categories
     * @var cmsSQLTable
     */
    protected $table_categories;
    protected $field_category_id;
    protected $field_category_title = 'title';
    protected $view_template_item_file = 'view.item';
    protected $view_template_category_file = 'view.category.simple';

    /**
     * $category_restore_keys_to_ignore
     * @var array
     */
    protected $category_restore_keys_to_ignore;

    public function __construct($app_description, $table_items, $table_categories)
    {
        if ($table_categories != NULL) {
            $this->setTable('categories', $table_categories);
            $this->setEntity('categories',  $this->table_categories->q());
            $this->field_category_id = $this->table_categories->getPrimaryKeyFieldNames();
            $this->fixSortableCategoryFields();            
        }
        $this->item_restore_keys_to_ignore = array($this->field_id, 'category_id');
        $this->category_restore_keys_to_ignore = array($this->field_category_id);
        parent::__construct($app_description, $table_items);
    }

    /**
     * Returns an item or category with specified GUID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $guid
     * @return array
     */
    public function getAnyObjectByGUID($guid)
    {
        global $SystemDB;

        if ($this->itemColumnExists('guid')) 
        {
            $item_row = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_items} WHERE `guid` = :guid", ['guid' => $guid]);
            if (!$item_row)
            {
                if ($this->categoryColumnExists('guid')) 
                {
                    $category_row = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE `guid` = :guid", ['guid' => $guid]);
                    if ($category_row)
                    {
                        $category_row['__type__'] = 'category';
                        return $category_row;                        
                    }
                }
            } else
            {
                $item_row['__type__'] = 'item';
                return $item_row;
            }
        }
        return NULL;
    }
    
//_______________________________________________________________________________________________________________//
    public function increaseCategoryPageView($cid)
    {
        $this->pageCounterIncreaseImpression($cid, $this->app_name, $this->_fieldname_categories, $this->table_categories, 'pageview', $this->field_category_id);
    }

//_______________________________________________________________________________________________________________//
    public function resetCategoryPageView($cid)
    {
        $this->pageCounterIncreaseImpression($cid, $this->app_name, $this->_fieldname_categories, $this->table_categories, 'pageview', $this->field_category_id, true);
    }

//_______________________________________________________________________________________________________________//
    public function getTotalItemCountByCategoryID($category_id, $criteria = '', $cache = false)
    {
        global $SystemDB;

        $cid = (int) $category_id;
        $criteria_txt = '';
        if (!empty($criteria))
            $criteria_txt = " AND {$criteria}";

        $sql = "SELECT COUNT({$this->field_id}) as total_item_count FROM {$this->table_items} WHERE {$this->field_item_category_id} = {$cid} {$criteria_txt}";
        $total = $SystemDB->getQueryResultSingleRow($sql, $cache);
        return $total['total_item_count'];
    }

//_______________________________________________________________________________________________________________//
    public function getTotalCategoryCount($criteria = '', $cache = false)
    {
        global $SystemDB;

        $criteria_txt = '';
        if (!empty($criteria))
            $criteria_txt = " WHERE {$criteria}";

        $sql = "SELECT COUNT({$this->field_category_id}) as total_category_count FROM {$this->table_categories} {$criteria_txt}";
        $total = $SystemDB->getQueryResultSingleRow($sql, $cache);
        return $total['total_category_count'];
    }

//_______________________________________________________________________________________________________________//
    public function getBreadCrumbsByCategoryID($cat_id)
    {
        $current_id = intval($cat_id);
        $last_id = -1;
        $path = '';
        if ($current_id) {
            $category = $this->getCategoryByID($current_id);
            $cat_id = $category[$this->field_category_id];
            $title = $category['title'];
            $this->addToBreadCrumbs($title, $this->createFriendlyURL("action=viewcategory&cid={$cat_id}"));
        }
    }

//_______________________________________________________________________________________________________________//
    public function getBreadCrumbsByItemID($item_id)
    {
        $item = $this->getItemByID(intval($item_id));
        if (!$item)
            return false;

        $item_id = $item[$this->field_id];
        $current_cat_id = $item[$this->field_category_id];
        $last_id = -1;
        $path = '';

        if ($current_cat_id) {
            $category = $this->getCategoryByID(intval($current_cat_id));
            $category_cid = $category[$this->field_category_id];
            $category_title = $category['title']; // Prana - 2012-Nov-25
            $this->addToBreadCrumbs($category_title, $this->createFriendlyURL("action=viewcategory&cid={$category_cid}"));
            $current_id = $last_id;
        }
        if (!($this->app_name == 'html' && $item['virtual_filename'] == 'home'))
            $this->addToBreadCrumbs($item['title'], $this->createFriendlyURL("action=viewitem&id={$item_id}"));
    }

    protected function probeMainPageURLRedirect()
    {
        global $CurrentFrontendApplication;
        
        if (SCHLIX_SEF_ENABLED)
        {
            $my_alias = $this->getOriginalFullApplicationAlias();
            if ($my_alias ==  $CurrentFrontendApplication)
            {
                $url = get_relative_url_request_path();
                if ($url === "/{$my_alias}/" )
                {
                    redirect_url(SCHLIX_SITE_URL . SCHLIX_SITE_HTTPBASE );
                    exit();
                }
            }
        }
    }
    
//_______________________________________________________________________________________________________________//
    /**
     * Returns an array of command, given $urlpath.
     * @param string $urlpath
     * @return array
     */    
    public function interpretFriendlyURL($urlpath)
    {/*

      input: http://schlixcms/directory/category/nested/listing.page5.html
      input: http://schlixcms/html/testcategory/welcome.html
      input: http://schlixcms/html/category/nested/
      input: http://schlixcms/contacts/balioffice.html
      input: http://schlixcms/users/action/register.html
      input: http://schlixcms/users/action/login.html
      input: http://schlixcms/users/action/logout
      input: http://schlixcms/store/action/checkout
     */

        $this->probeMainPageURLRedirect();
        $the_app = fget_string('app');
        if (SCHLIX_SEF_ENABLED && empty($the_app)) {
            $parsedurl = $this->probeFriendlyURLDestination($urlpath);
            $url = $parsedurl['url'];
            $url_array = $parsedurl['url_array'];
            $command = $this->getItemOrCategoryToViewFromFullVirtualFilename($url, true);

            if ($command['action'] == '404error') {
                // let's try passing this to child
                if ($url_array[0] == 'action' && $url_array[1]) {
                    $chunked_url_array = array_chunk($url_array, 2);
                    foreach ($chunked_url_array as $key => $command_chunk)
                        $command[$command_chunk[0]] = $command_chunk[1];
                }
            }
        }
        else {
            if (___c($_GET) == 0 || empty($_GET['action']))
            {
                $command['action'] = 'main';
                $command['pg'] = 0;
            }
            else
                $command = $_GET;
        }
        return $command;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * create SEO friendly URL. Format is action={...}&param1={....}&param2={...}
     * @param string $str
     * @return string
     */    
    public function createFriendlyURL($str)
    {
        /*
          input: index.php?app=html&action=view&id=c1
          input: index.php?app=html&action=view&id=i10
          input: index.php?app=users&action=register

          action=view&id=c5&page=10
          action=checkout
          action=login
         */
        $final_url = '';
        if (SCHLIX_SEF_ENABLED) {
            $command_array = [];
            parse_str($str, $command_array); // replaced with this- May 24, 2010
            $command_action = (array_key_exists('action', $command_array)) ? $command_array['action'] : '';
            switch ($command_action)
            {
                case 'viewcategory':$final_url = encode_url($this->getFullPathByCategoryID($command_array[$this->field_category_id]));
                    if (array_key_exists('pg', $command_array) && $command_array['pg'])
                        $final_url.="pg{$command_array['pg']}.html";

                    break;
                case 'viewitem': $final_url = encode_url($this->getFullPathByItemID($command_array[$this->field_id]));
                    break;
                default:if (!empty($command_array) && array_key_exists('action', $command_array) && $command_array['action'] != '') {
                        if (array_key_exists('app', $command_array))
                            unset($command_array['app']);
                        foreach ($command_array as $key => $value)
                        {
                            $final_url.= "/{$key}/{$value}";
                        }
                        //$final_url.='/';
                        //$final_url = '/'.$command_array['action'].'.ghtml';
                    }
                    else
                        $final_url = '/'; // For the main app - Feb 4, 2012
            }
            $app_alias = $this->full_application_alias; // May 24, 2010
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . $app_alias . $final_url; // delete - Oct 1, 2010
//			$final_url = '/'.$app_alias.$final_url;
        } else {
            parse_str("app={$this->app_name}&{$str}", $command_array);
            $array_keys = array_keys($command_array); // separate it
            foreach ($array_keys as $key)
                if (empty($command_array[$key]))
                    unset($command_array[$key]);
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . "index.php?" . http_build_query($command_array);
        }
        $final_url = remove_multiple_slashes($final_url);
//		echo $final_url;
        return $final_url;
    }

//_______________________________________________________________________________________________________________//
    public function getItemOrCategoryToViewFromFullVirtualFilename($url, $enable_redirect_no_trailingslash_folder = false)
    {

        define('SLASH_INDEX_DOT_HTML_LENGTH', 10); // '/index.html'
        $url_array = explode('/', $url);
        $command = [];
        $folder_requestpage = 0;
        $found = false;
        $filetype = ''; // category = c, item = i
        $depth = ___c($url_array);
        $url_length = strlen($url);
        $request_filename = basename($url);
        $file_path_info = null;
        if ($request_filename)
            $file_path_info = pathinfo($request_filename);
        if (empty($url) || $url == 'index.html') {
            $command['action'] = 'main';
            $command['pg'] = 0;
            return $command;
        }
        // Step 0 - Reset - clean up possible directory view mode with "/index.html"
        // possible /html/folder1/folder2 (but no trailing slash)
        $fext = isset($file_path_info['extension'] ) ? $file_path_info['extension'] : null;
        if ($enable_redirect_no_trailingslash_folder && ( $fext != 'html' ) && ( $fext != 'do' ) && ( strrpos($url, '/') != $url_length - 1 )) {
            $url.= '/';
            $url_array = explode('/', $url);
            array_splice($url_array, 0, 2);
            $depth = ___c($url_array);
            $url_length = strlen($url);
            if ($this->getCategoriesByVirtualFilename($request_filename)) {
                $app_alias = $this->full_application_alias;                
                redirect_url(SCHLIX_SITE_URL . SCHLIX_SITE_HTTPBASE . "/{$app_alias}/{$url}");
                exit;
            }
            else
                return array('action' => '404error');
        }
        if ($url_array[$depth - 1] == 'index.html') {
            // $filetype = 'c';
            $url_array[$depth - 1] = '';
            $url = substr($url, 0, $url_length - SLASH_INDEX_DOT_HTML_LENGTH);
            $depth = ___c($url_array); // reset again
            $url_length = strlen($url); // reset again
        }
        if ($c = preg_match_all("/(pg)(\d+).*?(html)/is", $url_array[$depth - 1], $x)) {
            $folder_requestpage = $x[2][0];
            $pg_length = strlen($url_array[$depth - 1]);
            $url_array[$depth - 1] = '';
            $url = substr($url, 0, $url_length - $pg_length);
            $depth = ___c($url_array); // reset again
            $url_length = strlen($url); // reset again
        }
        // 1 - check for '/' in the end of url, or '/index.html'
        $pos1a = strrpos($url, '/');
        if ($pos1a == $url_length - 1) {
            $last_item_name = $url_array[$depth - 2];
            $cats = $this->getCategoriesByVirtualFilename($last_item_name);
            $user_request_url = '/' . implode('/', $url_array);
            if ($cats)
                foreach ($cats as $cat)
                {
                    $possible_cat_id = $cat['cid'];
                    $possible_cat_path = encode_path($this->getFullPathByCategoryID($possible_cat_id));
                    if ($possible_cat_path == $user_request_url) {
                        $command['action'] = 'viewcategory';
                        $command['cid'] = $possible_cat_id;
                        // fix for PHP8
                        $command['pg'] = ($folder_requestpage > 0) ? $folder_requestpage : 0;
                        
                        return $command;
                    }
                }
            // End of find folder section
        } else {
        // //  FIND FILE
            $items = $this->getItemsByVirtualFilename($file_path_info['filename']);

            if ($items)
            {
                foreach ($items as $item)
                {
                    $catpath = $this->getFullPathByCategoryID($item[$this->field_item_category_id]);
                    
                    $compare1 = encode_path($catpath) . $file_path_info['filename'] . '.' . $file_path_info['extension'];
                    $compare1 = remove_multiple_slashes($compare1);
                    $compare2 = '/' . $url;
                    
                    if ($compare1 == $compare2) {
                        $command['action'] = 'viewitem';
                        $command['id'] = $item['id'];
                        return $command;
                    }
                }
            }
        }
        $command['action'] = '404error';
        return $command;
    }

//_______________________________________________________________________________________________________________//
    public function getFullPathByCategoryID($cat_id)
    {
        global $SystemDB;

        if ($this->categoryColumnExists('virtual_filename'))
        {        
            $current_id = intval($cat_id);
            $path = '';
            $sql = "SELECT virtual_filename FROM {$this->table_categories} where {$this->field_category_id} = '{$current_id}'";
            $id_r = $SystemDB->getQueryResultSingleRow($sql);
            $dirname = $id_r ? $id_r['virtual_filename'] : null;
            $path = $dirname . '/' . $path;

            return '/' . $path;
        } else return null;
    }

    //_______________________________________________________________________________________________________________//
    public function getFullPathByItemID($item_id)
    {
        global $SystemDB;

        if ($this->itemColumnExists('virtual_filename'))
        {        
            $item_id = (int) $item_id;
            $sql = "SELECT virtual_filename, {$this->field_item_category_id} FROM {$this->table_items} where {$this->field_id} = '{$item_id}'";
            $item = $SystemDB->getQueryResultSingleRow($sql, true);
            if (!$item)
                return false;
            $category_path = $this->getFullPathByCategoryID($item[$this->field_item_category_id]);
            $item_path = $item['virtual_filename'].'.html';
            return remove_multiple_slashes($category_path.$item_path);
        } else return null;
        
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Return the default category ID for save item operation
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @return int
     */
    public function getDefaultCategoryID()
    {
        global $SystemDB;

        $sql = "SELECT {$this->field_category_id} FROM {$this->table_categories} ORDER BY {$this->field_category_id} ASC LIMIT 0,1";
        $category = $SystemDB->getQueryResultSingleRow($sql);
        return ($category) ?  $category[$this->field_category_id] : 0;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the category's table name
     * @return string
     */
    public function getCategoryTableName()
    {
        return $this->table_categories->__toString();
    }

    
    /**
     * Returns a list of custom fields for the item table
     * @return array
     */
    public function getCategoryCustomFields()
    {
        $app_cf = new \App\Core_CustomField();
        
        return $app_cf->getCustomFields($this->getCategoryTableName());
        
    }    
    

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the category's table
     * @return \SCHLIX\cmsSQLTable
     */
    public function getCategoryTable()
    {
        return $this->table_categories;
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns the category field names
     * @return array
     */
    public function getCategoryFieldNames()
    {

        return $this->_fieldname_categories;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the primary key of category table, usually 'cid'
     * @return string
     */
    public function getFieldCategoryID()
    {
        return $this->field_category_id;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the foreign key in item's table that connects to the primary key of category table
     * Usually 'category_id'
     * @return string
     */
    public function getFieldItemCategoryID()
    {
        return $this->field_item_category_id;
    }

//_________________________________________________________________________//
    public function getAllChildItemsInMultipleCategories($multiple_category_ids)
    {
        global $SystemDB;

        $refined_result = [];
        $str_sql = implode(",", $multiple_category_ids);
        if ($str_sql) {
            $sql = "SELECT {$this->field_id} FROM {$this->table_items} WHERE {$this->field_item_category_id} in ({$str_sql})";
            $result_array_items_in_these_categories = $SystemDB->getQueryResultArray($sql);
            foreach ($result_array_items_in_these_categories as $individual_item)
                $refined_result[] = $individual_item[$this->field_id];
        }
        return $refined_result;
    }

//_______________________________________________________________________________________________________________//
    public function getAllCategories($fields = '*', $extra_criteria = '', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC')
    {
        global $SystemDB;

        $start = intval($start);
        $end = intval($end);

        if (!empty($sortby) && strpos($sortby, ',') === false)
            if (!in_array($sortby, $this->_fieldname_categories)) {
                $sortby = $this->getFieldCategoryID();
            }
        $sql = $SystemDB->generateSelectSQLStatement($this->table_categories, $fields, $extra_criteria, $start, $end, $sortby, $sortdirection, true, SCHLIX_SQL_ENFORCE_ROW_LIMIT);
//		echo $sql;die;
        $categories = $SystemDB->getQueryResultArray($sql);
        return $categories;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns a category by I
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     * @param bool $from_cache
     * @return array
     */
    public function getCategoryByID($id, $from_cache = false)
    {
        global $SystemDB;

        $category = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE {$this->field_category_id} = :cid", ['cid' => $id]);
        return $category;
    }

    public function getCategoryByIDWithExtraData($cid)
    {
        $category = $this->getCategoryByID($cid);
        if ($category)
        {
            $hook_result = cmsHooks::executeModifyReturn(2, __FUNCTION__, $this, $category) ;
            if ($hook_result)
                return $hook_result;
        }
        return $category;
    }
    
    /**
     * Typo but still needed for backward compatibility until 2.1.8 TODO: remove
     * @ignore
     * @deprecated since version 2.1.6-1
     * @param string $input_filename
     * @param int $parent_id
     * @return array
     */
    public function getCategoryByVirtualFilename($input_filename, $parent_id = -1)
    {
        return $this->getCategoriesByVirtualFilename($input_filename, $parent_id = -1);
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Return multiple categories given a possible virtual filename
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $input_filename
     * @param int $parent_id
     * @return array
     */
    public function getCategoriesByVirtualFilename($input_filename, $parent_id = -1)
    {
        global $SystemDB;

        if (!empty($input_filename) && $this->categoryColumnExists('virtual_filename')) {
            $input_filename = rawurldecode($input_filename);
            $filename = sanitize_string($input_filename);
            $str = "";
            $parent_id = intval($parent_id);
            if ($parent_id >= 0)
                $str = " AND parent_id = {$parent_id}";
            $sql = "SELECT * FROM {$this->table_categories} WHERE virtual_filename = {$filename}{$str}";
            $categories = $SystemDB->getQueryResultArray($sql, true);
            return $categories;
        }
        else
            return NULL;
    }

    //_______________________________________________________________________________________________________________//
    public function getItemsByCategoryID($id, $fields = '*', $extra_criteria = '', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC', $from_cache = false)
    {

        if (!empty($extra_criteria))
            $criteria_txt = " AND {$extra_criteria}";
        else
            $criteria_txt = '';

        return $this->getAllItems($fields, "{$this->field_item_category_id} = '" . intval($id) . "' {$criteria_txt}", intval($start), intval($end), $sortby, $sortdirection, $from_cache);
    }

//_______________________________________________________________________________________________________________//
    public function viewCategoryByID($id = 1, $pg = 1, $sortby = '', $sortdirection = 'ASC', $standard_criteria = 'status > 0', $from_cache = false)
    {
        $category = $this->getCategoryByIDWithExtraData($id);
        
        $perpage = $category['items_per_page'];
        if ($perpage == 0)
            $perpage = $this->getNumberOfListingsPerPage();
        $total_item_count = $this->getTotalItemCountByCategoryID($id, $standard_criteria, $from_cache);
        $pagination = $this->getStartAndEndForItemPagination($pg, $perpage, $total_item_count);
        $category_meta_options = $this->translateMetaOptions($category['options']);
        
        // backward compatibility v2.0.3
        if (isset($category_meta_options['display_childcategories']))
            $category_meta_options['display_child_categories'] = 1;
        
        if ($pg <= 0) $pg = 1;
        $item_start = abs($pg - 1) * $perpage;
        $item_end = $item_start + $perpage;
        $items  = $this->getItemsByCategoryID($id, '*', $standard_criteria, $item_start, $item_end, $sortby, $sortdirection, $from_cache);
        
        
        $str_title = $category['title'];
        if ($pg > 1)
            $str_title.=' - '.___('Page').' '.$pg;
        $this->setPagetitle($str_title);
            
        if (array_key_exists('meta_description', $category))
            $this->setPageMetaDescription ($category['meta_description']);
        if (array_key_exists('meta_key', $category))
            $this->setPageMetaKeywords ($category['meta_key']);

        $child_items_sortby = (is_array($category_meta_options) && array_key_exists('items_sortby', $category_meta_options)) ? $category_meta_options['items_sortby'] : $sortby;
        $child_items_sortdirection = (is_array($category_meta_options) && array_key_exists('items_sortdirection', $category_meta_options)) ? $category_meta_options['items_sortdirection'] : $sortdirection;

        $this->getBreadCrumbsByCategoryID($category[$this->field_category_id]);
        $this->processDataOutputWithMacro($category, __FUNCTION__,array('category_meta_options'=>$category_meta_options));        

        $this->increaseCategoryPageView($category[$this->field_category_id]);
        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile(($this->view_template_category_file), $local_variables);
    }

    /**
     * Delete category by ID
     * @global \App\Users $CurrentUser
     * @param int $cid
     */
    public function deleteCategoryByID($cid)
    {
        global $CurrentUser;
        
        $cid = (int) $cid;
        $existing_category = $this->getCategoryByID($cid);
        if ($existing_category)
        {
            $CurrentUser->recordCurrentUserActivity("Requesting to delete category in {$this->app_name} {$this->field_category_id}# {$cid} and its child items");
            
            $item_ids_in_category = $this->getAllChildItemsInMultipleCategories([$cid]);
            if (___c($item_ids_in_category) > 0)
                foreach ($item_ids_in_category as $id)
                    $this->deleteItemByID($id);
            
            $this->table_categories->q()->delete()->where("{$this->field_category_id} = :cid")->execute(['cid' => $cid]);
            
            $CurrentUser->recordCurrentUserActivity("Deleted {$this->app_name} {$this->field_category_id}# {$cid}");
        } else 
        {
            $CurrentUser->recordCurrentUserActivity("FAILED: Attempt to delete {$this->app_name} {$this->field_category_id}# {$cid}");
        }
        
    }       
//_______________________________________________________________________________________________________________//
    function delete($mixed_items_to_delete)
    { // mixed item = categories + items
        //test case:
        //http://schlixcms/admin/index.php?page=html&ajax=1&action=ajax_delete&items=c3
        //http://schlixcms/admin/index.php?page=html&ajax=1&action=ajax_delete&items=c5,c6,c9,i4,i14
        global $SystemDB;
        $id = 1;
        $mixed_items_array = explode('|', $mixed_items_to_delete); // e.g: c5,c6,c9,i4,i14
        $cats_to_delete = NULL;
        $all_cats_to_delete = [];
        $items_to_delete = [];

        // Process sub-folders first

        foreach ($mixed_items_array as $mixed_item)
        {
            $current_id = substr($mixed_item, 1);
            if (strpos($mixed_item, 'c') > -1) {
                $this->deleteCategoryByID($current_id);
            }
            else { // else if it's an item instead
                $this->deleteItemByID($current_id);
                
            }
        }
        
        /*
        // Process files
        if ($all_cats_to_delete) {
            foreach ($all_cats_to_delete as $a_cats_to_delete)
                $cat_id_filler_for_sql[] = $a_cats_to_delete;//[$this->field_category_id];
            $result_array_items_in_these_categories = $this->getAllChildItemsInMultipleCategories($cat_id_filler_for_sql);
        }
        // Did the user select any items to delete?
        if ($items_to_delete) {
            if ($result_array_items_in_these_categories)
                $items_to_delete = array_merge($items_to_delete, $result_array_items_in_these_categories);
        }
        else
            $items_to_delete = $result_array_items_in_these_categories;

        // now delete all of them - $all_cats_to_delete + $items_to_delete
        if ($items_to_delete)
            $items_to_delete_str = implode(",", $items_to_delete);
        if ($cat_id_filler_for_sql)
            $cats_to_delete_str = implode(",", $cat_id_filler_for_sql);
        if ($items_to_delete_str) {
            $items_to_delete_str = implode(",", $items_to_delete);
            $sql1 = "DELETE FROM {$this->table_items} WHERE {$this->field_id} in ({$items_to_delete_str})";
            $SystemDB->query($sql1);
        }
        if ($cats_to_delete_str) {
            $cats_to_delete_str = implode(",", $cat_id_filler_for_sql);
            $sql2 = "DELETE FROM {$this->table_categories} WHERE {$this->field_category_id} in ({$cats_to_delete_str})";

            $SystemDB->query($sql2);
        }*/
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns true if category table contains this column name
     * @param string $fieldname
     * @return bool
     */
    public function categoryColumnExists($fieldname)
    {
        return $this->_fieldname_categories ? in_array($fieldname, $this->_fieldname_categories) : false;
    }


    /**
     * Restores an item from global versioning
     * @param int $cid
     * @param double $version
     * @return null
     */
    public function restoreCategory($cid, $version)
    {
        $current_category = $this->getCategoryByID($cid);
        return $this->restoreGeneric($cid, $this->field_category_id, $version, $current_category, $this->table_categories, $this->category_restore_keys_to_ignore);
    }

//_______________________________________________________________________________________________________________//
   /**
     * Validates save item. If there's an error, it will return an array
     * with one or more error string, otherwise it will return a boolean true
     * @global \App\Users $CurrentUser
     * @param array $datavalues
     * @return bool|array String array
     */
    public function getValidationErrorListBeforeSaveCategory($datavalues)
    {
        global $CurrentUser;
        
        $cid = $datavalues[$this->field_category_id];
        $int_cid = (int) $cid;        
        $error_list = [];
        if ($this->findDuplicateCategories($datavalues)) 
        {
            $error_list[] =  ___('Duplicate item found');
        }

        if ($int_cid == 0 && $cid != 'new')
        {
            $error_list[] = ___('Invalid category ID');
        }        

        if ($this->_checkSaveCategoryWritePermission && $int_cid > 0 && $this->categoryColumnExists('permission_write')) {
            $existing_category = $this->getCategoryByID($int_cid);
            $hasWritePermission = $CurrentUser->hasWritePermission($existing_category['permission_write']);
            if (!$hasWritePermission) {
                $error_list[] = ___('You do not have a write access to this category');
            }
        }        
        $dir_errors = $this->initializeDataDirectories();
        $hook_errors = cmsHooks::executeModifyReturn(2, __FUNCTION__, $this, $datavalues) ;
        return @array_merge($error_list, $dir_errors, $hook_errors);
        
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Returns true if any element in $data has a duplicate with an existing category
     * User override.
     * @param array $datavalues
     * @return boolean
     */
    public function findDuplicateCategories($datavalues)
    {
        return false;
    }    

    /**
     * Set default save category variables
     * @param array $datavalues
     * @return type
     */
    protected function modifyDataValuesBeforeSaveCategory($datavalues)
    {
        
        $current_date_time = get_current_datetime();
        $cid = $datavalues[$this->field_category_id];
        $int_cid = (int)$cid;        
        
        // If it's a new category
        if ($cid == 'new')
        {
            if ($this->categoryColumnExists('date_created')) 
                $datavalues['date_created'] = $current_date_time;
            if ($this->categoryColumnExists('date_modified')) 
                $datavalues['date_modified'] = $current_date_time;
            
            if (array_key_exists('date_modified', $datavalues))
                if (strtotime($datavalues['date_modified']) == 0)
                    $datavalues['date_modified'] = $current_date_time;
            if (array_key_exists('date_available', $datavalues))
                if (strtotime($datavalues['date_available']) == 0)
                    $datavalues['date_available'] = $current_date_time;
            if ($this->categoryColumnExists('guid'))
                $datavalues['guid'] = new_uuid_v4();
           
        } 
        // If it's an existing category
        else if ($int_cid > 0)
        {
            $existing_category_id = (int) $datavalues[$this->field_category_id];
            $previous_category = $this->getCategoryByID($existing_category_id);
            if ($this->categoryColumnExists('date_created')) {
                if ((strtotime($previous_category['date_created']) == 0))
                    $datavalues['date_created'] = $current_date_time;
            }
            $dv_date_modified = isset($datavalues['date_modified']) ? $datavalues['date_modified'] : null;
            if ($this->categoryColumnExists('date_modified') &&
                (($dv_date_modified == $previous_category['date_modified']) ||
                (strtotime($dv_date_modified) == 0))
            ) {
                $datavalues['date_modified'] = $current_date_time;
            }
            if ($this->categoryColumnExists('guid') && $previous_category['guid'] == '')
                $datavalues['guid'] = new_uuid_v4();
        }
        $datavalues = cmsHooks::executeModifyReturn(2, __FUNCTION__, $this, $datavalues) ;

        return $datavalues;
    }                
    /**
     * Add hook to before save category
     * @param object $object
     * @param string $function_name
     * @return bool
     */
    public function addBeforeSaveCategoryValidationHook($object, $function_name)
    {
        return $this->__addHookToFunctionArray($this->_more_save_category_validations, $object, $function_name);
    }    
    /**
     * Add hook to before save category
     * @param object $object
     * @param string $function_name
     */
    public function addAfterSaveCategoryFunctionHook($object, $function_name)
    {
        return $this->__addHookToFunctionArray($this->_after_save_category_functions, $object, $function_name);
    }        
    /**
     * Add hook to before save category. The function must receive be like this:
     * $obj->function_name ($datavalues, $return_value_of_save)
     * @param object $object
     * @param string $function_name
     * @return bool
     */
    public function addBeforeSaveCategoryFunctionHook($object, $function_name)
    {
        return $this->__addHookToFunctionArray($this->_before_save_category_functions, $object, $function_name);
    }
    /**
     * Do something after save item.
     * @param array $datavalues
     * @param array $original_datavalues
     * @param array $retval
     * @param array $previous_category
     */
    protected function onAfterSaveCategory($datavalues, $original_datavalues, $previous_category, $retval)        
    {
        global $Versioning;
        
        if ($retval['status'] == SAVE_OK)
        {
            cmsHooks::execute(__FUNCTION__, $this, $datavalues, $original_datavalues, $previous_category, $retval) ;
        }
        
        return $retval;
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Saves the category to database. If id's type is string, it must only be 'new'
     * @global cmsDatabase $SystemDB
     * @param array $alternative_datavalues
     * @param int|string $id
     * @return string
     */
    public function saveCategory($cid, $alternative_datavalues = NULL)
    {
        global $SystemDB, $Versioning;

        $retval = [];
        if ($alternative_datavalues !== NULL && !is_array($alternative_datavalues))
        {
            $this->setSaveStatusError($retval, $cid, ___('Invalid save data from array - type is not array'));
            return $retval;
        }
        // Accept every $_POST or data variables first, unfiltered
        $datavalues = $original_datavalues = ($alternative_datavalues == NULL) ? select_http_post_variables() : $alternative_datavalues;
        
        if (empty($datavalues))
        {
            $this->setSaveStatusError($retval, $cid, ___('Invalid save data from array - no data values matches field names'));
            return $retval;
        }
        // modify it
        $datavalues = $this->modifyDataValuesBeforeSaveCategory($datavalues);
        
        // Save Category Hook
        if ($this->_before_save_category_functions)
        {
            foreach ($this->_before_save_category_functions as $objfunc)
                if (method_exists($objfunc['object'], $objfunc['function']))
                    $datavalues = $objfunc['object']->{$objfunc['function']}($datavalues);
        }
        // Validation
        $error_list = $this->getValidationErrorListBeforeSaveCategory($datavalues);
        if (!empty($error_list)) {
            $this->setSaveStatusError($retval, $cid, $error_list);
            return $retval;
        }
        // Perform additional validation hook
        if ($this->_more_save_category_validations)
        {
            foreach ($this->_more_save_category_validations as $objfunc)
            {
                if (method_exists($objfunc['object'], $objfunc['function']))
                {
                    $more_error_list = $objfunc['object']->{$objfunc['function']}($datavalues);
                    if (!empty($more_error_list)) {
                        $this->setSaveStatusError($retval, $cid, $more_error_list);
                        return $retval;
                    }
                }                
            }            
        }
        
        // Filter further data, if any
        $datavalues = array_intersect_key($datavalues,  array_fill_keys($this->_fieldname_categories, NULL));
        $previous_category = 0;
        // The save function
        if ($cid == 'new') {
            unset($datavalues[$this->field_category_id]);
            $SystemDB->query("LOCK TABLES `{$this->table_categories}` WRITE");            
            $this->table_categories->quickInsert($datavalues);
            $this->setSaveStatusOK($retval, $SystemDB->getLastInsertID());            
            $SystemDB->query("UNLOCK TABLES");            
        }
        else if (intval($datavalues[$this->field_category_id]) > 0) {
            $cid = (int) $datavalues[$this->field_category_id];       
            $previous_category = $this->getCategoryByID($cid);
            if ($previous_category)
            {
                $this->table_categories->quickUpdate($datavalues, "{$this->field_category_id} = {$cid}");                
                
                $this->setSaveStatusOK($retval, $cid);                
            }
            else  $this->setSaveStatusError($retval, $cid, sprintf(___('Could not save an existing category with ID: %s'), $cid));
        }
        else  $this->setSaveStatusError($retval, $cid, ___('Invalid category ID - validation failed '));
        $this->onAfterSaveCategory($datavalues, $original_datavalues, $previous_category, $retval);
        // AFTER Save Category Hook
        if ($this->_after_save_category_functions)
        {
            foreach ($this->_after_save_category_functions as $objfunc)
                if (method_exists($objfunc['object'], $objfunc['function']))
                    $datavalues = $objfunc['object']->{$objfunc['function']}($datavalues, $original_datavalues, $previous_category, $retval);
        }        
        return $retval;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Override - modify data values before save item
     * @param array $datavalues
     * @return array
     */
    protected function modifyDataValuesBeforeSaveItem($datavalues)
    {
        $datavalues = parent::modifyDataValuesBeforeSaveItem($datavalues);
        if (array_key_exists($this->field_item_category_id, $datavalues) && isset($datavalues[$this->field_item_category_id]) && $datavalues[$this->field_item_category_id] == 0) {
            $datavalues[$this->field_item_category_id] = $this->getDefaultCategoryID();
        }
        return $datavalues;
    }


    //_______________________________________________________________________________________________________________//
    /**
     * Returns a new name if there's an item with the same name in the specified
     * $fieldname
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $fieldname
     * @param int $id
     * @param int $category_id
     * @param string $possible_duplicate
     * @return string
     */
    public function preventDuplicateValueInItemTableUnderCategory($fieldname, $possible_duplicate, $id, $category_id)
    {
        global $SystemDB;

        $id = (int) $id;
        $category_id = (int) $category_id;
        if (!$this->itemColumnExists($fieldname))
            return $possible_duplicate;
        $possible_duplicate = str_replace('%','', $possible_duplicate);
        $sanitized_dup_value = sanitize_string($possible_duplicate.'%');
        //$sql = "SELECT {$fieldname} FROM {$this->table_items} WHERE LOWER(`{$fieldname}`) = LOWER (:possible_duplicate) AND ({$this->field_id} <> $id) ";
        
        $sql = "SELECT {$fieldname} FROM {$this->table_items} WHERE LOWER(`{$fieldname}`) = LOWER (:possible_duplicate) AND ( ({$this->field_id} <> :id) AND ({$this->field_item_category_id} = :category_id)) ";
        
        $results = $SystemDB->getQueryResultArray($sql, ['possible_duplicate' => $possible_duplicate, 'id' => $id, 'category_id' => $category_id]);
        return ($results) ?  $this->__suggestNewNonDuplicateValueFromArray($fieldname, $results, $possible_duplicate) : $possible_duplicate;
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns a new name if there's an item with the same name in the specified
     * $fieldname
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $fieldname
     * @param int $cid
     * @param string $possible_duplicate
     * @return string
     */
    public function preventDuplicateValueInCategory($fieldname, $possible_duplicate, $cid)
    {
        global $SystemDB;

        $cid = (int) $cid;
        
        if (!$this->categoryColumnExists($fieldname))
            return $possible_duplicate;
        $possible_duplicate = str_replace('%','', $possible_duplicate);
        $sanitized_dup_value = sanitize_string($possible_duplicate.'%');
        $sql = "SELECT {$fieldname} FROM {$this->table_categories} WHERE `{$fieldname}` LIKE {$sanitized_dup_value} AND  ({$this->field_category_id} <> $cid)";
        $results = $SystemDB->getQueryResultArray($sql);
        return ($results) ?  $this->__suggestNewNonDuplicateValueFromArray($fieldname, $results, $possible_duplicate) : $possible_duplicate;
    } 

    /**
     * Translate label of sortable item fields and match it with item fields
     */
    protected function fixSortableCategoryFields()
    {
        
        $this->sortable_category_fields = $this->fixSortableFields($this->sortable_category_fields, $this->_fieldname_categories);
    }    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns a label/key pair array containing a list of sortable category fields
     * @return array
     */
    public function getSortableCategoryColumns() {
        
        return  $this->sortable_category_fields; 
                
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array of default category options. If it's not defined, the 
     * hardcoded property of default_category_options
     * @return array
     */
    public function getDefaultCategoryMetaOptionKeys()
    {
        $db_config_Value = $this->getConfig('array_default_category_meta_options');
        return (empty($db_config_Value) || !is_array($db_config_Value)) ? $this->default_category_options : $db_config_Value;
    }
    
    /**
     * Returns the default number of items per page
     * @return int
     */
    public function getDefaultCategoryItemsPerPage()
    {
        $db_config_Value = $this->getConfig('int_category_items_per_page');
        return ($db_config_Value == 0) ? 10 :  $db_config_Value;
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. In this base
     * class, the key is almost similar to getCategoryMetaOptionKeys
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */    
    public function getCategoryMetaOptionKeys() {
        
        return [
            // main
            [   'label' => ___('Category Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_main',
                'options' => 
                [
                    ['key' => 'display_pagetitle', 'label' => ___('Display category title')],
                    ['key' => 'display_error_no_access', 'label' => ___('Display errors for inaccesible item')],
                    ['key' => 'display_link_title', 'label' => ___('Display links in the title')],
                    ['key' => 'display_category_summary_noread', 'label' => ___('Display summary for users with no read access')],
                    ['key' => 'display_category_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_category_modified_by', 'label' => ___('Display modified by')],
                    ['key' => 'display_category_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_category_date_modified', 'label' => ___('Display date modified')]
          //array('value' => 'display_user_can_change_sort_options', 'label' => ___('Visitors can change sort options')), // todo
          //array('value' => 'display_user_can_change_items_perpage', 'label' => ___('Visitors can change items per page')), // todo
                  
                ]
              ],
            // items
            [   'label' => ___('Child Items Listing Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_child_items',
                'options' =>
                [
                    ['key' => 'display_items', 'label' => ___('Display list of items')],
                    ['key' => 'display_item_summary', 'label' => ___('Display summary')],
                    ['key' => 'display_item_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_item_modified_by', 'label' => ___('Display modified by')],
                    ['key' => 'display_item_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_item_date_modified', 'label' => ___('Display date modified')],
                    ['key' => 'display_item_read_more_link', 'label' => ___('Display "Read More" link')],
                    ['key' => 'display_item_view_count', 'label' => ___('Display view count')]
                ]
              ],
            // Sort
            [   'label' => ___('Sort Options'),
                'options' =>
                [
                  // Items
                    ['key' => 'items_sortby', 'type' => 'dropdownlist', 'label' => ___('Sort child items by'), 
                      'options' => $this->getSortableItemColumns()],
                    ['key' => 'items_sortdirection', 'type' => 'radiogroup', 'label' => ___('Child items sort direction'), 
                      'options' => 
                        [['key' => 'desc', 'label' => ___('Descending')], 
                        ['key' => 'asc', 'label' => ___('Ascending') ]]],

                ]
              ]
          ]; 
    }    

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. 
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */    
    public function getMainpageMetaOptionKeys()
    {
        
        return [
            // child category
            [   'label' => ___('Child Category Listing Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_child_categories',
                'options' =>   
                [              
                    ['key' => 'display_child_categories', 'label' => ___('Display list of child categories')],
                    ['key' => 'display_child_category_summary', 'label' => ___('Display summary')],
                    ['key' => 'display_child_category_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_child_category_modified_by', 'label' => ___('Display modified by')],
                    ['key' => 'display_child_category_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_child_category_date_modified', 'label' => ___('Display date modified')],
                    ['key' => 'display_child_category_view_count', 'label' => ___('Display view count')],                    
                    ['key' => 'display_child_category_read_more_link', 'label' => ___('Display "Read More" link')]
                ]
              
              ],
            // sort options
            [   'label' => ___('Sort Options'),
                'options' =>
                [
                  // Child Categories
                    ['key' => 'child_categories_sortby', 'type' => 'dropdownlist', 'label' => ___('Sort child categories by'), 
                      'options' => $this->getSortableCategoryColumns()],

                    ['key' => 'child_categories_sortdirection', 'type' => 'radiogroup', 'label' => ___('Child categories sort direction'), 
                      'options' => 
                        [['key' => 'desc', 'label' => ___('Descending')], 
                        ['key' => 'asc', 'label' => ___('Ascending')]]]
                ]
              ]
          ];         
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Run Command for routing. Main command is in $command['action']
     * @param array $command
     * @return boolean
     */
    public function Run($command)
    {

        switch ($command['action'])
        {
            case 'viewcategory':
                $pg = isset($command['pg']) ? $command['pg'] : 0;
                $this->viewCategoryByID($command['cid'], $pg, '', 'ASC', $this->cache);
                break;
            default: return parent::Run($command);
                break;
        }

        return true;
    }

//_______________________________________________________________________________________________________________//
}

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
abstract class cmsApplication_HierarchicalTree_List extends cmsApplication_CategorizedList implements interface_cmsApplication_HierarchicalTree_List
{

    protected $mode;
    /**
     * Category's Parent ID field name
     * @var string 
     */
    protected $field_category_parent_id;
    protected $view_template_category_file = 'view.category.nested';

//_______________________________________________________________________________________________________________//
    public function __construct($app_description, $table_items, $table_categories)
    {
        parent::__construct($app_description, $table_items, $table_categories);        
        $this->field_category_parent_id = 'parent_id';
        $this->category_restore_keys_to_ignore = array($this->field_category_id, $this->field_category_parent_id);
        $this->mode = 'nested';
    }

    /**
     * Return the default category ID for save item operation. Returns 0 in this class 
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @return int
     */
    public function getDefaultCategoryID()
    {
        return 0;
    }
    /**
     * return Category's Parent ID field name
     * @return string
     */
    public function getFieldCategoryParentID()
    {
        return $this->field_category_parent_id;
    }
//_______________________________________________________________________________________________________________//
    public function getTotalChildCategoryCountByCategoryID($parent_cid, $criteria = '', $cache = false)
    {
        global $SystemDB;

        $pid = (int) $parent_cid;
        $criteria_txt = '';
        if (!empty($criteria))
            $criteria_txt = " AND {$criteria}";
        $sql = "SELECT COUNT({$this->field_category_id}) as total_category_count FROM {$this->table_categories} WHERE {$this->field_category_parent_id} = {$pid} {$criteria_txt} ";
        $total = $SystemDB->getQueryResultSingleRow($sql, $cache);
        $total_category_count = $total['total_category_count'];
        return $total_category_count;
    }

//_______________________________________________________________________________________________________________//
    public function getBreadCrumbsByCategoryID($cat_id)
    {
        global $SystemDB;
        $tmp_bread_crumbs = [];

        $current_id = intval($cat_id);
        $last_id = -1;
        $path = '';
        while ($last_id != 0 && $current_id != 0)
        {
            $category = $this->getCategoryByID($current_id);
            $cat_id = $category[$this->field_category_id];
            $last_id = $category['parent_id'];
            $title = $category['title'];
            $tmp_bread_crumbs[] = array('title' => $title, 'link' => $this->createFriendlyURL("action=viewcategory&cid={$cat_id}"));
            $current_id = $last_id;
        }
        $tmp_bread_crumbs = array_reverse($tmp_bread_crumbs);
        foreach ($tmp_bread_crumbs as $tmp_bread_crumb)
            $this->addToBreadCrumbs($tmp_bread_crumb['title'], $tmp_bread_crumb['link']);
    }

//_______________________________________________________________________________________________________________//
    public function getBreadCrumbsByItemID($item_id)
    {
        global $SystemDB;

        $tmp_bread_crumbs = [];
        $item = $this->getItemByID(intval($item_id));
        if (!$item)
            return false;

        $current_id = $item[$this->field_item_category_id];
        $last_id = -1;
        $path = '';

        while ($last_id != 0 && $current_id != 0)
        {
            $category = $this->getCategoryByID($current_id);
            $cat_id = $category[$this->field_category_id];
            $last_id = $category['parent_id'];
            $title = $category['title'];
            $tmp_bread_crumbs[] = array('title' => $title, 'link' => $this->createFriendlyURL("action=viewcategory&cid={$cat_id}"));
            $current_id = $last_id;
        }
        $tmp_bread_crumbs = array_reverse($tmp_bread_crumbs);
        foreach ($tmp_bread_crumbs as $tmp_bread_crumb)
            $this->addToBreadCrumbs($tmp_bread_crumb['title'], $tmp_bread_crumb['link']);

        if (!($this->app_name == 'html' && $item['virtual_filename'] == 'home'))
            $this->addToBreadCrumbs($item['title'], $this->createFriendlyURL("action=viewitem&id={$item['id']}"));
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns the path given item ID. e.g. /folder1/folder2/item.html
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $item_id
     * @return string
     */
    public function getFullPathByItemID($item_id) 
    {
        global $SystemDB;
        if ($this->itemColumnExists('virtual_filename'))
        {        

            $item_id = (int) $item_id; 
            if ($item_id == 0) 
                return null;
            $context_cache_key = $this->full_application_alias . '_item_full_path';
            $cache_full_path = cmsContextCache::get($context_cache_key, $item_id);
            if ($cache_full_path) {
                return $cache_full_path;
            } else {
                $sql = "SELECT virtual_filename,  {$this->field_item_category_id}  FROM {$this->table_items} where {$this->field_id} = '{$item_id}'";
                $item = $SystemDB->getQueryResultSingleRow($sql, true);
                if (!$item)
                    return false;
                $xpath = $this->getFullPathByCategoryID($item[$this->field_item_category_id]);
                $result = remove_multiple_slashes($xpath.'/'.$item['virtual_filename']).'.html';
                cmsContextCache::set($context_cache_key, $item_id, $result);
                return $result;
            }
        } else return null;
        
    }
    /**
     * Returns the full path with trailing slash of current category. e.g. /folder1/folder2/
     * @param int $cat_id
     * @return string
     */
    public function getFullPathByCategoryID($cat_id)
    {
        $cat_id = (int) $cat_id; 
        if ($cat_id == 0)
            return '/';
        $context_cache_key = $this->full_application_alias. '_category_full_path';
        $cache_full_path = cmsContextCache::get($context_cache_key, $cat_id);
        if ($cache_full_path) {
            return $cache_full_path;
        } else {
            $field_cid = $this->field_category_id;
            $field_cpid = $this->field_category_parent_id;                
            $field_cvfn = 'virtual_filename';            
            if ($this->categoryColumnExists($field_cvfn) && $this->categoryColumnExists($this->field_category_parent_id))
            {        
                $data = $this->table_categories->q()->select("{$field_cpid}, {$field_cvfn}")->where(["{$field_cid} = :cid"])->getQueryResultSingleRow(['cid' => $cat_id]);
                $result = [];
                if ($data != null) $result[] = $data;
                $l = 0;
                while (is_array($data) && $data[$field_cpid] > 0 && $l < 20)
                {
                    $data = $this->table_categories->q()->select("{$field_cpid}, {$field_cvfn}")->where(["{$field_cid} = :cid"])->getQueryResultSingleRow(['cid' => $data[$field_cpid]]);
                    if ($data != null) $result[] = $data;                
                    $l++;
                }
                if ($result)
                {
                    $the_path = '/'.implode('/', array_column( array_reverse($result), $field_cvfn)).'/';
                    cmsContextCache::set($context_cache_key, $cat_id, $the_path);                    
                    return $the_path;
                }
            }
        }
        return '/';
    }



//_______________________________________________________________________________________________________________//
    public function getChildCategoriesByParentID($id, $fields = '*', $extra_criteria = '', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC', $from_cache = false)
    {
        if (!empty($extra_criteria))
            $criteria_txt = " AND {$extra_criteria}";
        else
            $criteria_txt = '';
        return $this->getAllCategories($fields, "parent_id='" . intval($id) . "' {$criteria_txt}", $start, $end, $sortby, $sortdirection, $from_cache);
    }
    
    /**
     * Returns an array of all parent categories for this category
     * @param int $category_id
     * @return array
     */
    public function getAllParentsByCategoryID($category_id)
    {
        $i = 0;
        $result = []; 
        $category_id = (int) $category_id;
        $category = $this->getCategoryByID($category_id);
        $parent_id = isset($category['parent_id']) ? $category['parent_id'] : 0;
        $result[] = $parent_id;        
        while ($category != null && $parent_id > 0 && $i < 1000)
        {
            $category = $this->getCategoryByID($parent_id);
            $parent_id = $category['parent_id'];
            $result[] = $parent_id;
            $i++; // put a max iteration of 1000 just in case there's a bad node
        }
        return $result;
    }
//_______________________________________________________________________________________________________________//
    public function getChildCategoriesByParentIDWithChildCount($id, $fields = '*', $extra_criteria = '', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC', $from_cache = false)
    {
        global $SystemDB;
        
        $limit_criteria = ''; // TODO - FIX - PHP8
        
        $default_criteria = "_parent.parent_id =  ".intval($id);
        if (!empty($extra_criteria))
            $criteria_txt = "{$default_criteria} AND {$extra_criteria}";
        else
            $criteria_txt = $default_criteria;
        //return $this->getAllCategories($fields, "parent_id='" . intval($id) . "' {$criteria_txt}", $start, $end, $sortby, $sortdirection, $from_cache);
        
        $start = (int) $start;
        $end = (int) $end;
        if (!empty($sortby) && strpos($sortby, ',') === false)
            if (!$this->categoryColumnExists($sortby)) {
                $sortby = $this->getFieldCategoryID();
            }
            
        $params = $SystemDB->generateSQLSelectParameters('_parent', $fields, $criteria_txt, $start, $end, $sortby, $sortdirection, TRUE);
        $cid = $this->field_category_id; 
         $sql = "SELECT {$params['fields']} , coalesce(COUNT(_child.{$cid}),0) AS __child_count
FROM {$this->table_categories} _parent LEFT OUTER JOIN  {$this->table_categories} _child ON _child.parent_id = _parent.{$cid} WHERE  
{$params['criteria']} GROUP BY _parent.{$cid} {$params['sort_criteria']} {$limit_criteria}";
            
                
        $categories = $SystemDB->getQueryResultArray($sql);        
        return $categories;        
    }    
            
    
    //_______________________________________________________________________________________________________________//
    /**
     * View Category By ID
     * @param int $id
     * @param int $pg
     * @param string $sortby
     * @param string $sortdirection
     * @param bool $from_cache
     * @param string $standard_criteria
     * @return boolean
     */
    public function viewCategoryByID($id = 1, $pg = 1, $sortby = '', $sortdirection = 'ASC', $from_cache = false, $standard_criteria = 'status > 0')
    {

        $category = $this->getCategoryByID($id, $from_cache);
        if (!$category) {
            display_http_error(404);
            return false;
        }
        $perpage = isset($category['items_per_page']) ? $category['items_per_page'] : 0;
        $category_meta_options = $this->translateMetaOptions($category['options']);
        $category_meta_options['display_child_item_read_more_link'] = isset($category_meta_options['display_child_item_read_more_link'] ) ? $category_meta_options['display_child_item_read_more_link'] : false;
        $category_meta_options['display_item_summary'] = isset($category_meta_options['display_item_summary'] ) ? $category_meta_options['display_item_summary'] : false;
        // backward compatibility v2.0.3
        if (isset($category_meta_options['display_childcategories']))
            $category_meta_options['display_child_categories'] = 1;        
        // end ackward compatibility v2.0.3
        if ($category_meta_options == NULL)
            $category_meta_options = [];
        if ($perpage == 0)
            $perpage = $this->getNumberOfListingsPerPage();
        $total_item_count = $this->getTotalItemCountByCategoryID($id, $standard_criteria, $from_cache);
        $total_child_category_count = $this->getTotalChildCategoryCountByCategoryID($id, $standard_criteria, $from_cache);
        $pagination = $this->getStartAndEndForItemPagination($pg, $perpage, $total_child_category_count + $total_item_count);
        $this->setPageTitle(($pg > 1) ? $category['title'] . " - Page {$pg}" : $category['title']);
        if (array_key_exists('meta_description', $category))
            $this->setPageMetaDescription($category['meta_description']);
        if (array_key_exists('meta_key', $category))
            $this->setPageMetaKeywords($category['meta_key']);
        $this->getBreadCrumbsByCategoryID($category[$this->field_category_id]);
        $this->increaseCategoryPageView($category[$this->field_category_id]);
        $child_categories = [];
        $items = [];
        // Dec 12, 2016
        // Backward compatibility with versions below v.2.0.3
        // The key was changed from categories_sortby to child_categories_sortby (misnomer)
        if (array_key_exists('child_categories_sortby', $category_meta_options))
        {
            $child_categories_sortby = (array_key_exists('child_categories_sortby', $category_meta_options)) ? $category_meta_options['child_categories_sortby'] : $sortby;
            $child_categories_sortdirection = (array_key_exists('child_categories_sortdirection', $category_meta_options)) ? $category_meta_options['child_categories_sortdirection'] : $sortdirection;            
        } else
        {
            $child_categories_sortby = (array_key_exists('categories_sortby', $category_meta_options)) ? $category_meta_options['categories_sortby'] : $sortby;
            $child_categories_sortdirection = (array_key_exists('categories_sortdirection', $category_meta_options)) ? $category_meta_options['categories_sortdirection'] : $sortdirection;            
        }

        $items_sortby = (array_key_exists('items_sortby', $category_meta_options)) ? $category_meta_options['items_sortby'] : $sortby;
        $items_sortdirection = (array_key_exists('items_sortdirection', $category_meta_options)) ? $category_meta_options['items_sortdirection'] : $sortdirection;
        $this->processDataOutputWithMacro($category, __FUNCTION__,array('category_meta_options'=>$category_meta_options));        

        /*
          Variables to be passed to the viewer:
          $categories, $childcategories, $items, $pg, $perpage, $pagination, $category_meta_options, $total_item_count, $total_child_category_count, $error_message
         */

        /* Don't enable it just yet (Feb 25, 2012) .. pending testing
          if ($this->itemColumnExists('date_modified'))
          $this->declarePageLastModified($this->getMaxDateFromArray($categories));
         */
        $total_start = ($pg - 1) * $perpage;
        if ($total_start <= $total_item_count + $total_child_category_count) {
            if ($total_child_category_count > 0 && $total_child_category_count > $total_start) {
                //$category_end = $pagination['start'] + min($total_child_category_count - $pagination['start'], $perpage);
                $child_categories = $this->getChildCategoriesByParentID($id, '*', $standard_criteria, $pagination['start'], $pagination['end'], $child_categories_sortby, $child_categories_sortdirection, $from_cache);
            }

            if ($total_item_count > 0 && $pagination['end'] - $total_child_category_count > 0) {
                $item_start = max(0, $pagination['start'] - $total_child_category_count);
                $item_end = $item_start + min($pagination['end'] - $total_child_category_count, $perpage);
                $items = $this->getItemsByCategoryID($id, '*', $standard_criteria, $item_start, $item_end, $items_sortby, $items_sortdirection, $from_cache);
                //query style
                //$items = $this->table_items->q()->select('*')->where(['status > 0','AND','category_id = :category_id'])->orderBy([$sortby => $sortdirection])->startEnd($item_start, $item_end)->getQueryResultArray(['category_id' => $id]);
            }
        }
        else
            $error_message = "Page is outside of valid range";

        $current_method = __FUNCTION__;
        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile(($this->view_template_category_file), $local_variables);
    }

//_________________________________________________________________________//
     public function traverseCategories($catid)
    {

        if ($catid > 0) {
            $my_own_cat = $this->table_categories->q()->select($this->field_category_id)->where(["{$this->field_category_id} = :parent_id "])->getQueryResultArray(['parent_id' => $catid]);
            if (___c($my_own_cat) > 0)
            {

                $categories = $this->table_categories->q()->select($this->field_category_id)->where(["{$this->field_category_parent_id} = :parent_id "])->getQueryResultArray(['parent_id' => $catid]);
                $i = 0;
                while ($i < ___c($categories))
                {
                    $current_id = $categories[$i][$this->field_category_id];
                    $i++;
                    $categories_tmp = $this->table_categories->q()->select($this->field_category_id)->where(["{$this->field_category_parent_id} = :parent_id "])->getQueryResultArray(['parent_id' => $current_id]);
                    $categories = array_merge($categories, $categories_tmp);
                } // end while
                $categories = array_merge($my_own_cat, $categories);
                
            }
        }
        return $categories;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of category options.
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */
    public function getCategoryMetaOptionKeys() {
        
        return [
          
            // main
            [   'label' => ___('Category Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_main',
                'options' => 
                [
                    ['key' => 'display_pagetitle', 'label' => ___('Display category title')],
                    ['key' => 'display_error_no_access', 'label' => ___('Display errors for inaccesible item')],
                    ['key' => 'display_link_title', 'label' => ___('Display links in the title')],
                    ['key' => 'display_category_summary_noread', 'label' => ___('Display summary for users with no read access')],
                    ['key' => 'display_category_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_category_modified_by', 'label' => ___('Display modified by')],
                    ['key' => 'display_category_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_category_date_modified', 'label' => ___('Display date modified')]
          //array('value' => 'display_user_can_change_sort_options', 'label' => ___('Visitors can change sort options')), // todo
          //array('value' => 'display_user_can_change_items_perpage', 'label' => ___('Visitors can change items per page')), // todo
                  
                ]
              ],
            // child category
            [   'label' => ___('Child Category Listing Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_child_categories',
                'options' =>   
                [              
                    ['key' => 'display_child_categories', 'label' => ___('Display list of child categories')],
                    ['key' => 'display_child_category_summary', 'label' => ___('Display summary')],
                    ['key' => 'display_child_category_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_child_category_modified_by', 'label' => ___('Display modified by')],
                    ['key' => 'display_child_category_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_child_category_date_modified', 'label' => ___('Display date modified')],
                    ['key' => 'display_child_category_view_count', 'label' => ___('Display view count')],                    
                    ['key' => 'display_child_category_read_more_link', 'label' => ___('Display "Read More" link')]
                ]
              
              ],
            // items
            [   'label' => ___('Child Items Listing Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_child_items',
                'options' =>
                [
                    ['key' => 'display_items', 'label' => ___('Display list of items')],
                    ['key' => 'display_item_summary', 'label' => ___('Display summary')],
                    ['key' => 'display_item_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_item_modified_by', 'label' => ___('Display modified by')],
                    ['key' => 'display_item_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_item_date_modified', 'label' => ___('Display date modified')],
                    ['key' => 'display_item_read_more_link', 'label' => ___('Display "Read More" link')],
                    ['key' => 'display_item_view_count', 'label' => ___('Display view count')]
                ]
              ],
            // Sort
            [   'label' => ___('Sort Options'),
                'options' =>
                [
                  // Child Categories
                    ['key' => 'child_categories_sortby', 'type' => 'dropdownlist', 'label' => ___('Sort child categories by'), 
                      'options' => $this->getSortableCategoryColumns()],

                    ['key' => 'child_categories_sortdirection', 'type' => 'radiogroup', 'label' => ___('Child categories sort direction'), 
                      'options' => 
                        [['key' => 'desc', 'label' => ___('Descending')], 
                        ['key' => 'asc', 'label' => ___('Ascending')]]],
                  // Items
                    ['key' => 'items_sortby', 'type' => 'dropdownlist', 'label' => ___('Sort child items by'), 
                      'options' => $this->getSortableItemColumns()],
                    ['key' => 'items_sortdirection', 'type' => 'radiogroup', 'label' => ___('Child items sort direction'), 
                      'options' => 
                        [['key' => 'desc', 'label' => ___('Descending')], 
                        ['key' => 'asc', 'label' => ___('Ascending') ]]],

                ]
              ]
          ]; 
    }    
    

    //_______________________________________________________________________________________________________________//
    /**
     * Returns a new name if there's an item with the same name in the specified
     * $fieldname
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $fieldname
     * @param int $cid
     * @param int $parent_id
     * @param string $possible_duplicate
     * @return string
     */
    public function preventDuplicateValueInCategoryTableUnderParentCategory($fieldname, $possible_duplicate, $cid, $parent_id)
    {
        global $SystemDB;

        $cid = (int) $cid;
        $parent_id = (int) $parent_id;
        if (!$this->categoryColumnExists($fieldname) || !$this->categoryColumnExists($this->field_category_parent_id))
            return $possible_duplicate;
        $possible_duplicate = str_replace('%','', $possible_duplicate);
        $sanitized_dup_value = sanitize_string($possible_duplicate.'%');
        //$sql = "SELECT {$fieldname} FROM {$this->table_categories} WHERE `{$fieldname}` LIKE {$sanitized_dup_value} AND ( ({$this->field_category_id} <> $cid) AND ({$this->field_category_parent_id} = {$parent_id})) ";
        $sql = "SELECT {$fieldname} FROM {$this->table_categories} WHERE LOWER(`{$fieldname}`) = LOWER (:possible_duplicate) AND ( ({$this->field_category_id} <> :cid) AND ({$this->field_category_parent_id} = :parent_id)) ";
        //echo $sql;
        
        $results = $SystemDB->getQueryResultArray($sql, ['possible_duplicate' => $possible_duplicate, 'id', 'cid' => $cid, 'parent_id' => $parent_id]);
        
        return ($results) ?  $this->__suggestNewNonDuplicateValueFromArray($fieldname, $results, $possible_duplicate) : $possible_duplicate;
    }    
    
    /**
     * Delete category by ID, recursive
     * @global \App\Users $CurrentUser
     * @param int $cid
     */
    public function deleteCategoryByID($cid)
    {
        global $CurrentUser;
        
        $cid = (int) $cid;
        $existing_category = $this->getCategoryByID($cid);
        if ($existing_category)
        {
            $CurrentUser->recordCurrentUserActivity("Requesting to delete category in {$this->app_name} {$this->field_category_id}# {$cid} and its child items");
            $cats_to_delete = $this->traverseCategories($cid);
            foreach ($cats_to_delete as $cat)
            {
                $cid_to_delete = $cat[$this->field_category_id];
                $CurrentUser->recordCurrentUserActivity("Deleting category in {$this->app_name} {$this->field_category_id}# {$cid_to_delete} and its child items...");
                $item_ids_in_category = $this->getAllChildItemsInMultipleCategories([$cid_to_delete]);
                if (___c($item_ids_in_category) > 0)
                    foreach ($item_ids_in_category as $id)
                        $this->deleteItemByID($id);
                $this->table_categories->q()->delete()->where("{$this->field_category_id} = :cid")->execute(['cid' => $cid_to_delete]);
                $CurrentUser->recordCurrentUserActivity("Deleted {$this->app_name} {$this->field_category_id}# {$cid_to_delete}");
            } 
        } else 
        {
            $CurrentUser->recordCurrentUserActivity("FAILED: Attempt to delete {$this->app_name} {$this->field_category_id}# {$cid}");
        }
        
    }    
    
    //_______________________________________________________________________________________________________________//
    /**
     * Delete objects. Parameter $mixed_items_to_delete is a  category/item comma separated values
     * e.g.  c5,c6,c9,i4,i14
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $mixed_items_to_delete
     
    function delete($mixed_items_to_delete)
    { // mixed item = categories + items
        //test case:
        //http://schlixcms/admin/index.php?page=html&ajax=1&action=ajax_delete&items=c3
        //http://schlixcms/admin/index.php?page=html&ajax=1&action=ajax_delete&items=c5,c6,c9,i4,i14
        global $SystemDB;
        $id = 1;
        $mixed_items_array = explode('|', $mixed_items_to_delete); // e.g: c5,c6,c9,i4,i14

        $cats_to_delete = NULL;
        $all_cats_to_delete = NULL;

        // Process sub-folders first

        foreach ($mixed_items_array as $mixed_item)
        {
            $current_id = substr($mixed_item, 1); // 11 is the next string after
            if (strpos($mixed_item, 'c') > -1) {
                // Get the sub-child for each cats
                $cats_to_delete = $this->traverseCategories($current_id);
                // Now Process
                if ($all_cats_to_delete)
                    $all_cats_to_delete = array_merge($all_cats_to_delete, $cats_to_delete);
                else
                    $all_cats_to_delete = $cats_to_delete;
            } else { // else if it's an item instead
                $items_to_delete[] = (int) $current_id;
            }
        }
        // Process files
        if ($all_cats_to_delete) {
            foreach ($all_cats_to_delete as $a_cat_to_delete)
                $cat_id_filler_for_sql[] = $a_cat_to_delete[$this->field_category_id];
            $result_array_items_in_these_categories = $this->getAllChildItemsInMultipleCategories($cat_id_filler_for_sql);
        }
        // Did the user select any items to delete?
        if ($items_to_delete) {
            if ($result_array_items_in_these_categories)
                $items_to_delete = array_merge($items_to_delete, $result_array_items_in_these_categories);
        }
        else
            $items_to_delete = $result_array_items_in_these_categories;

        // now delete all of them - $all_cats_to_delete + $items_to_delete
        if ($items_to_delete)
            $items_to_delete_str = implode(",", $items_to_delete);
        if ($cat_id_filler_for_sql)
            $cats_to_delete_str = implode(",", $cat_id_filler_for_sql);

        if ($items_to_delete_str) {
            $items_to_delete_str = implode(",", $items_to_delete);
            $sql1 = "DELETE FROM {$this->table_items} WHERE {$this->field_id} in ({$items_to_delete_str})";
            $SystemDB->query($sql1);
        }
        if ($cats_to_delete_str) {
            $cats_to_delete_str = implode(",", $cat_id_filler_for_sql);
            $sql2 = "DELETE FROM {$this->table_categories} WHERE {$this->field_category_id} in ({$cats_to_delete_str})";
            $SystemDB->query($sql2);
        }
    }*/

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. 
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */    
    public function getMainpageMetaOptionKeys()
    {
        
        return [
            // child category
            [   'label' => ___('Child Category Listing Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_child_categories',
                'options' =>   
                [              
                    ['key' => 'display_child_categories', 'label' => ___('Display list of child categories')],
                    ['key' => 'display_child_category_summary', 'label' => ___('Display summary')],
                    ['key' => 'display_child_category_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_child_category_modified_by', 'label' => ___('Display modified by')],
                    ['key' => 'display_child_category_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_child_category_date_modified', 'label' => ___('Display date modified')],
                    ['key' => 'display_child_category_view_count', 'label' => ___('Display view count')],                    
                    ['key' => 'display_child_category_read_more_link', 'label' => ___('Display "Read More" link')]
                ]
              
              ],
        
            // items
            [   'label' => ___('Child Items Listing Options'),
                'type' => 'checkboxgroup',
                'key' => 'opt_child_items',
                'options' =>
                [
                    ['key' => 'display_items', 'label' => ___('Display list of items')],
                    ['key' => 'display_item_summary', 'label' => ___('Display summary')],
                    ['key' => 'display_item_created_by', 'label' => ___('Display created by')],
                    ['key' => 'display_item_date_created', 'label' => ___('Display date created')],
                    ['key' => 'display_item_date_modified', 'label' => ___('Display date modified')],
                    ['key' => 'display_item_read_more_link', 'label' => ___('Display "Read More" link')],
                    ['key' => 'display_item_view_count', 'label' => ___('Display view count')]
                ]
              ],
            // items
            [   'label' => ___('Sort Options'),
                'options' =>
                [
                  // Child Categories
                    ['key' => 'child_categories_sortby', 'type' => 'dropdownlist', 'label' => ___('Sort child categories by'), 
                      'options' => $this->getSortableCategoryColumns()],

                    ['key' => 'child_categories_sortdirection', 'type' => 'radiogroup', 'label' => ___('Child categories sort direction'), 
                      'options' => 
                        [['key' => 'desc', 'label' => ___('Descending')], 
                        ['key' => 'asc', 'label' => ___('Ascending')]]],
                  // Items
                
                    ['key' => 'items_sortby', 'type' => 'dropdownlist', 'label' => ___('Sort child items by'), 
                      'options' => $this->getSortableItemColumns()],
                    ['key' => 'items_sortdirection', 'type' => 'radiogroup', 'label' => ___('Child items sort direction'), 
                      'options' => 
                        [['key' => 'desc', 'label' => ___('Descending')], 
                        ['key' => 'asc', 'label' => ___('Ascending') ]]]

                ]
              ]
          ];         
    }
}

define('CMS_APPLICATION_MANY_TO_MANY_URL_SINGLE', 1);
define('CMS_APPLICATION_MANY_TO_MANY_URL_WITH_CATEGORY', 2);
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
abstract class cmsApplication_ManyToMany extends cmsApplication_HierarchicalTree_List implements interface_cmsApplication_HierarchicalTree_List
{

    /**
     * Table field names of $table_categories_items table
     * @var array 
     */
    protected $_fieldname_categories_items;
    /**
     * The table name that holds primary key for both items & categories table,
     * Typical structure is [id, cid] as the primary key
     * @var string
     */
    protected $table_categories_items;

    /**
     * A tinyint(1) or boolean column in the categories_items table that indicates
     * if a category is the primary category of an item
     * @var string 
     */
    protected $field_categories_items_is_primary = NULL;
    
    /**
     * URL mode - with or without category
     * @var int 
     */
    protected $url_mode = CMS_APPLICATION_MANY_TO_MANY_URL_SINGLE;
    /**
     * Constructor
     * @param string $app_description
     * @param string $table_items
     * @param string $table_categories
     * @param string $table_categories_items
     * @param string $field_categories_items_is_primary
     */   
    public function __construct($app_description, $table_items,  $table_categories,  $table_categories_items, $field_categories_items_is_primary = 'is_primary')
    {
                
        parent::__construct($app_description, $table_items, $table_categories);
        if ($table_categories_items != NULL) {
            $this->setTable('categories_items', $table_categories_items);            
            if ($field_categories_items_is_primary != NULL && ($this->categoriesToitemJointColumnExists($field_categories_items_is_primary)))
            {
                $this->field_categories_items_is_primary = $field_categories_items_is_primary;
            }
        }
        $this->mode = 'multiplecategories';
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns true if categories_items table contains fieldname
     * @param string $fieldname
     * @return bool
     */
    public function categoriesToitemJointColumnExists($fieldname)
    {
        return $this->_fieldname_categories_items ? in_array($fieldname, $this->_fieldname_categories_items) : NULL;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns a new name if there's an item with the same name in the specified
     * $fieldname
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $fieldname
     * @param int $id
     * @param int $category_id
     * @param string $possible_duplicate
     * @return string
     */
    public function preventDuplicateValueInItemTableUnderCategory($fieldname, $possible_duplicate, $id, $category_id)
    {
        global $SystemDB;

        $id = (int) $id;
        $category_id = (int) $category_id;        
        if (!$this->itemColumnExists($fieldname))
            return $possible_duplicate;
        //$possible_duplicate = str_replace('%','', $possible_duplicate);
        //$sanitized_dup_value = sanitize_string($possible_duplicate.'%');
        
        $sql = "SELECT {$this->table_items}.{$fieldname} FROM {$this->table_items} INNER JOIN {$this->table_categories_items} ON  {$this->table_items}.{$this->field_id} = {$this->table_categories_items}.{$this->field_id} WHERE LOWER(`{$fieldname}`) = LOWER (:possible_duplicate)  AND ( ({$this->table_items}.{$this->field_id} <> :id) AND ({$this->table_categories_items}.{$this->field_category_id} = :category_id)) ";

        $results = $SystemDB->getQueryResultArray($sql, ['possible_duplicate' => $possible_duplicate, 'id' => $id, 'category_id' => $category_id]);
        
        return ($results) ? $this->__suggestNewNonDuplicateValueFromArray($fieldname, $results, $possible_duplicate) : $possible_duplicate; 
    }
    //_______________________________________________________________________________________________________________//
    public function getTotalItemCountByCategoryID($category_id, $criteria = '', $cache = false)
    {
        global $SystemDB;

        $cid = (int) $category_id;
        $criteria_txt = '';
        if (!empty($criteria))
            $criteria_txt = " AND {$criteria}";
        if ($cid > 0)
            $sql = "SELECT COUNT({$this->table_items}.{$this->field_id}) as total_item_count FROM {$this->table_items} INNER JOIN {$this->table_categories_items} ON {$this->table_items}.{$this->field_id}= {$this->table_categories_items}.{$this->field_id} WHERE {$this->table_categories_items}.{$this->field_category_id} = {$cid} {$criteria_txt}";
        else
            $sql = "SELECT COUNT({$this->table_items}.{$this->field_id}) as total_item_count FROM {$this->table_items} WHERE {$this->table_items}.{$this->field_id} NOT IN (SELECT {$this->field_id} FROM {$this->table_categories_items})";
        $total = $SystemDB->getQueryResultSingleRow($sql, $cache);
        return $total['total_item_count'];
    }

    /**
     * Get breadcrumbs
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $item_id
     * @return boolean
     */
    public function getBreadCrumbsByItemID($item_id)
    {
        global $SystemDB;

        $tmp_bread_crumbs = [];
        $item = $this->getItemByID(intval($item_id));
        if (!$item)
            return false;

        $current_id = $this->getItemPrimaryCategoryID($item[$this->field_id]); // $item[$this->field_item_category_id];
        $last_id = -1;
        $path = '';

        while ($last_id != 0 && $current_id != 0)
        {
            $category = $this->getCategoryByID($current_id);
            $cat_id = $category[$this->field_category_id];
            $last_id = $category['parent_id'];
            $title = $category['title'];
            $tmp_bread_crumbs[] = array('title' => $title, 'link' => $this->createFriendlyURL("action=viewcategory&cid={$cat_id}"));
            $current_id = $last_id;
        }
        $tmp_bread_crumbs = array_reverse($tmp_bread_crumbs);
        foreach ($tmp_bread_crumbs as $tmp_bread_crumb)
            $this->addToBreadCrumbs($tmp_bread_crumb['title'], $tmp_bread_crumb['link']);

        if (!($this->app_name == 'html' && $item['virtual_filename'] == 'home'))
            $this->addToBreadCrumbs($item['title'], $this->createFriendlyURL("action=viewitem&id={$item['id']}"));
    }

    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns the item-category field names
     * @return array
     */
    public function getItemsToCategoryFieldNames()
    {
        return $this->_fieldname_categories_items;
    }

    /**
     * Returns the item-category table
     * @return \SCHLIX\cmsSQLTable
     */
    public function getItemsToCategoryTable()
    {
        return $this->table_categories_items;
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns the item-category table name
     * @return string
     */
    public function getItemsToCategoryTableName()
    {
        return $this->table_categories_items->__toString();
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Given item ID, return its full path. Set the force_cid to the category ID
     * of the category
     * URL modes: 
     * - CMS_APPLICATION_MANY_TO_MANY_URL_SINGLE, 
     * - CMS_APPLICATION_MANY_TO_MANY_URL_WITH_CATEGORY
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $item_id
     * @param int $foce_cid
     * @return boolean|string
     */
    public function getFullPathByItemID($item_id, $force_cid = 0)
    {

        global $SystemDB; 

        if ($this->itemColumnExists('virtual_filename'))
        {        
            $item_id = (int) $item_id;

            if (empty($this->table_categories_items))
                return parent::getFullPathByItemID($item_id);

            if ($this->url_mode == CMS_APPLICATION_MANY_TO_MANY_URL_SINGLE)
            {

                return \SCHLIX\cmsApplication_List::getFullPathByItemID($item_id);
            }
            else
            {
                $sql = "SELECT virtual_filename,{$this->table_categories_items}.{$this->field_category_id} FROM {$this->table_items}
                        INNER JOIN {$this->table_categories_items} ON {$this->table_categories_items}.{$this->field_id} = {$this->table_items}.{$this->field_id}
                        WHERE {$this->table_items}.{$this->field_id} = '{$item_id}'";
                $force_cid = (int) $force_cid;
                if ($force_cid > 0)
                {
                    if ($this->isItemInCategoryID($item_id, $force_cid))
                    {
                        $sql.= " AND `{$this->field_category_id}` = {$force_cid}";
                    } else
                    {
                        return \SCHLIX\cmsApplication_List::getFullPathByItemID($item_id);
                    }
                } else
                if ($this->field_categories_items_is_primary)
                {
                    $sql.= " AND `{$this->field_categories_items_is_primary}` = 1";
                }                    
                $item = $SystemDB->getQueryResultSingleRow($sql, true);
                if ($item)
                {
                    $category_path = $this->getFullPathByCategoryID($item[$this->field_category_id]);
                    $result = $category_path.$item['virtual_filename'].'.html';
                    return $result;
                } else
                {
                    return false;
                }
            }
        } else return null;
        
    }

//_______________________________________________________________________________________________________________//
    public function getItemsByCategoryID($id, $fields = '*', $extra_criteria = '', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC', $from_cache = false) {
        global $SystemDB;
        if ($fields == '*')
            $fieldnames = $this->getItemFieldNames();
        else
            $fieldnames = explode(',', $fields);
        $fields_tobe_selected = implode(',', $fieldnames);

        if (!empty($extra_criteria))
            $criteria_txt = " AND {$extra_criteria}";
        else
            $criteria_txt = '';

        if ($id == 0) /* $the_criteria = "RIGHT OUTER JOIN {$this->table_categories_items} ON {$this->table_categories_items}.{$this->field_id} != {$this->table_items}.{$this->field_id} {$criteria_txt}"; */
            $the_criteria = " WHERE {$this->table_items}.{$this->field_id} NOT IN (SELECT {$this->field_id} FROM {$this->table_categories_items})";
            //$the_criteria = "RIGHT OUTER JOIN {$this->table_categories_items} ON {$this->table_categories_items}.{$this->field_id} != {$this->table_items}.{$this->field_id} {$criteria_txt}";
        else
            $the_criteria = "LEFT JOIN {$this->table_categories_items} ON {$this->table_categories_items}.{$this->field_id} = {$this->table_items}.{$this->field_id} WHERE {$this->table_categories_items}.{$this->field_category_id} = '" . intval($id) . "' {$criteria_txt}";
        $sql = $SystemDB->generateSelectSQLStatement($this->table_items, $fields_tobe_selected, $the_criteria, $start, $end, $sortby, $sortdirection, false, SCHLIX_SQL_ENFORCE_ROW_LIMIT);
       
        $items = $SystemDB->getQueryResultArray($sql, $from_cache);

        return $items;
    }

    

    /**
     * Returns true if all category IDs in $category_array are valid categories
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param array $category_array
     * @return boolean
     */
    public function verifyAllCategoriesExist($category_array)
    {
        global $SystemDB;
        
        if (!is_array($category_array) || empty($category_array))
        {
            return false;
        } else
        {
            $cat_array_values = array_unique(array_values($category_array));
            $cat_ids = implode(',', $cat_array_values);
            $sql = "SELECT COUNT(*) as total_count FROM {$this->table_categories} WHERE {$this->field_category_id} IN ({$cat_ids})";
            
            $result = $SystemDB->getQueryResultSingleRow($sql);
            $int_total = (int) $result['total_count'];
            return ($int_total === ___c($cat_array_values));
        }
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Set the item to the specified category if the state is true, otherwise the 
     * item will be removed from the category. The content of cid_array must be
     * an array of a positive number greater than 1
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param int $id
     * @param array $cid_array
     * @param bool $state
     * @return boolean
     */
    public function setItemCategories($id, $cid_array)
    {
        global $SystemDB, $SystemLog;
        $sqls = [];

        $id = (int) $id;
        if (!is_array($cid_array) || $id == 0)
            return false;
        $item = $this->getItemByID($id);                
        $existing_categories = $this->getItemCategoriesByItemID($id);
        if (is_array($existing_categories))
        {
            $existing_cid_array = array_column($existing_categories,$this->field_category_id);
            sort($existing_cid_array);
            sort($cid_array);
            // If it's already the same then no need to modify
            if ($cid_array == $existing_cid_array)
            {
                return true;
            }
        }
        if ($item) {
            $cid_array_to_keep = []; // separate it in case there's bad data with cid = 0 or less
            foreach ($cid_array as $cid) {
                $cid = (int) $cid;
                if ($cid > 0)
                {
                    $category = $this->getCategoryByID($cid);
                    if ($category)
                        $cid_array_to_keep[] = $cid;
                    if (!in_array($cid, $existing_cid_array)) {
                        if ($category)
                        {
                            $sqls[] = "INSERT IGNORE INTO {$this->table_categories_items} ({$this->field_id},{$this->field_category_id}) VALUES ($id, $cid)";
                        } else {
                            $SystemLog->error("Trying to set an item to a non-existing category ID# {$cid}", $this->app_name);
                        }
                    }
                }
            }
            if (___c($sqls) > 0 || ___c($cid_array_to_keep) > 0) {
                // Only do this if the categories & the item have been verified
                $cid_str_to_keep = implode(',', $cid_array_to_keep);
                $sql = "DELETE FROM {$this->table_categories_items} WHERE {$this->field_id} = {$id}";
                if ($cid_str_to_keep) // just in case there's still bad data ... 
                {
                    $sql.= " AND {$this->field_category_id} NOT IN ({$cid_str_to_keep})";
                }
                $SystemDB->query($sql);                    
                if ($sqls)
                {
                    foreach ($sqls as $sql) {
                        $SystemDB->query($sql);
                    }
                }
                return true;
            } else
            {
                $SystemLog->error('Failed to perform setItemCategories', $this->app_name);
            }
        } else {
            $SystemLog->error('Trying to set category to a non-existing item', $this->app_name);
        }
        return false;
    }
    
//_______________________________________________________________________________________________________________//
    /**
     * Set the item to the specified category if the state is true, otherwise the 
     * item will be removed from the category
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     * @param int $cid
     * @param bool $state
     * @return boolean
     */
    public function setItemCategory($id, $cid, $state)
    {
        global $SystemDB;

        $id = (int) $id;
        $cid = (int) $cid;
        if ($id > 0 and $cid >= 0) {
            if ($state) {
                if ($this->getCategoryByID($cid) && $this->getItemByID($id))
                    $sql = "INSERT IGNORE INTO {$this->table_categories_items} ({$this->field_id},{$this->field_category_id}) VALUES ($id, $cid)";
                else
                    return false;
            }
            else
                $sql = "DELETE FROM {$this->table_categories_items} WHERE {$this->field_id} = {$id} AND {$this->field_category_id} = {$cid}";
            $SystemDB->query($sql);
            return true;
        }
        return false;
    }

//_______________________________________________________________________________________________________________//
    public function getItemCategoryIDsByItemID($id, $sortby = '', $sortdirection = 'ASC', $from_cache = false)
    {
        global $SystemDB;

        //$sql = "SELECT {$this->table_items}.{$this->field_id}, {$this->table_categories_items}.{$this->field_category_id} FROM {$this->table_items} INNER JOIN {$this->table_categories_items} ON {$this->table_categories_items}.{$this->field_id} = {$this->table_items}.{$this->field_id} WHERE {$this->field_category_id} = '".intval($id)."'";
        $id = (int) $id;
        $sanitized_id = sanitize_string($id);
        $sql = "SELECT * FROM {$this->table_categories_items} WHERE {$this->field_id} = {$sanitized_id}";
        if (!empty($sortby) && $this->itemColumnExists($sortby))
            $sql.= " ORDER BY {$sortby} {$sortdirection}";
        $category = $SystemDB->getQueryResultArray($sql, $from_cache);

        return $category;
    }

    /**
     * This method is incorrect (detected Jan 2020)
     * @deprecated since version 2.2.2 
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     */
    public function getItemPrimaryCategory($id)
    {
        global $SystemDB;        
        
        if ($this->field_categories_items_is_primary)
        {
            $id = (int) $id;        
            $sql = "SELECT {$this->field_id} FROM {$this->table_categories_items} WHERE `{$this->field_categories_items_is_primary}` = 1";

            $row = $SystemDB->getQueryResultSingleRow($sql, ['id' => $id]);
            if ($row)
            {
                return $row[$this->field_id];
            }
        }
        return FALSE;        
    }
    
    /**
     * 
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     * @return boolean
     */
    public function getItemPrimaryCategoryID($id)
    {
        global $SystemDB;
        
        if ($this->field_categories_items_is_primary)
        {
            $id = (int) $id;        
            $sql = "SELECT {$this->field_category_id} FROM {$this->table_categories_items} WHERE {$this->field_id} = :id AND `{$this->field_categories_items_is_primary}` = 1";

            $row = $SystemDB->getQueryResultSingleRow($sql, ['id' => $id]);
            if ($row)
            {
                return $row[$this->field_category_id];
            }
        }
        return FALSE;        
    }
    
    /**
     * Set the primary category of an item
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     * @param int $primary_cid
     * @return bool
     */
    public function setItemPrimaryCategory($id, $primary_cid)
    {
        global $SystemDB;
        
        if ($this->field_categories_items_is_primary)
        {
            $field_name = $this->field_categories_items_is_primary;
            $id = (int) $id;
            $primary_cid = (int) $primary_cid;
            if ($this->categoriesToitemJointColumnExists($field_name))
            {
                if ($this->isItemInCategoryID($id, $primary_cid))
                {
                    $sanitized_id = sanitize_string($id);
                    $sanitized_primary_cid = sanitize_string($primary_cid);
                    $SystemDB->query("UPDATE {$this->table_categories_items} SET `{$field_name}` = 0 WHERE {$this->field_id} = {$sanitized_id} AND {$this->field_category_id} <> {$sanitized_primary_cid}");
                    $SystemDB->query("UPDATE {$this->table_categories_items} SET `{$field_name}` = 1 WHERE {$this->field_id} = {$sanitized_id} AND {$this->field_category_id} = {$sanitized_primary_cid}");
                } else
                {
                    $this->recordLog("Error while trying to set item id {$id} primary category to {$primary_cid}. Specified category ID does not exist");
                    return false;
                }
            }
            else
            {
                $this->recordLog("Item to categories column does not have this column: '{$field_name}'");
            }   
        }
        return false;
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns true if item is in this this category
     * @global SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     * @param int $cid
     * @return type
     */
    public function isItemInCategoryID($id, $cid)
    {
        global $SystemDB;

        //$sql = "SELECT {$this->table_items}.{$this->field_id}, {$this->table_categories_items}.{$this->field_category_id} FROM {$this->table_items} INNER JOIN {$this->table_categories_items} ON {$this->table_categories_items}.{$this->field_id} = {$this->table_items}.{$this->field_id} WHERE {$this->field_category_id} = '".intval($id)."'";
        $data = [
            'id' => (int) $id,
            'cid' => (int) $cid
            ];        
        $sql = "SELECT 1 AS result FROM {$this->table_categories_items} WHERE `{$this->field_id}` = :id AND `{$this->field_category_id}` = :cid";        
        $result = $SystemDB->getQueryResultSingleRow($sql, $data);
        return $result['result'] == 1;
    }
    
    //_______________________________________________________________________________________________________________//
    
    protected function getNonAmbiguousFieldNames($tablename)
    {
        switch ($tablename)
        {
            case $this->table_items: $datavalues = $this->_fieldname_items;
                break;
            case $this->table_categories: $datavalues = $this->_fieldname_categories;
                break;
            case $this->table_categories_items: $datavalues = $this->_fieldname_categories_items;
                break;
            default: return false;
        }
        $array_fields = [];
        foreach ($datavalues as $field)
            $array_fields[] = $tablename . '.' . $field;
        $selected_fields = implode(', ', $array_fields);
        return $selected_fields;
    }

//_________________________________________________________________________//
    /**
     * Given a comma-separated $multiple_category_ids, returns all item IDs
     * of items with that category I
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $multiple_category_ids
     * @return array
     */
    public function getAllChildItemsInMultipleCategories($multiple_category_ids)
    {
        global $SystemDB;

        $refined_result = [];
        $str_sql = implode(",", $multiple_category_ids);
        if ($str_sql) {
            $sql = "SELECT {$this->field_id} FROM {$this->table_categories_items} WHERE {$this->field_category_id} in ({$str_sql})";
            $result_array_items_in_these_categories = $SystemDB->getQueryResultArray($sql);
            foreach ($result_array_items_in_these_categories as $individual_item)
                $refined_result[] = $individual_item[$this->field_id];
        }
        return $refined_result;
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns item categoryies by item ID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     * @param string $sortby
     * @param string $sortdirection
     * @param bool $from_cache
     * @return array
     */
    public function getItemCategoriesByItemID($id, $sortby = '', $sortdirection = 'ASC', $from_cache = false)
    {
        global $SystemDB;

        $id = (int)$id;
        $fields = $this->getNonAmbiguousFieldNames($this->table_categories);
        $sql = "SELECT {$fields} FROM {$this->table_categories} INNER JOIN {$this->table_categories_items} ON {$this->table_categories}.{$this->field_category_id} = {$this->table_categories_items}.{$this->field_category_id}  WHERE {$this->table_categories_items}.{$this->field_id} = '" . intval($id) . "'";
        if (!empty($sortby) && $this->itemColumnExists($sortby))
            $sql.= " ORDER BY {$sortby} {$sortdirection}";
        $category = $SystemDB->getQueryResultArray($sql, $from_cache);

        return $category;
    }
    
    
    /**
     * Returns one or more item with the same virtual filename. Used in later classes.
     * There's a parameter $category_id that's not supposed to be in this class but it's there 
     * for compatibility with inherited classes
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $input_filename
     * @param int $category_id
     * @return array
     */
    public function getItemsByVirtualFilename($input_filename, $category_id = -1)
    {
        global $SystemDB;

        if (!empty($input_filename) && $this->itemColumnExists('virtual_filename')) {

            $str = "";
            $category_id = (int) $category_id;
            
            $input_filename = rawurldecode($input_filename);

            $filename = sanitize_string($input_filename);
            if ($category_id >= 0)
                $str = " AND {$this->field_category_id} = {$category_id}";
            
            
            $sql = "SELECT {$this->table_items}.*,{$this->table_categories_items}.{$this->field_category_id} as {$this->field_item_category_id} FROM {$this->table_items}
                    INNER JOIN {$this->table_categories_items} ON {$this->table_categories_items}.{$this->field_id} = {$this->table_items}.{$this->field_id}
                    WHERE virtual_filename = {$filename}{$str}";
            $items = $SystemDB->getQueryResultArray($sql, true);
            
            return $items;
        }
        else
            return false;
    }
    

    //_______________________________________________________________________________________________________________//
    /**
     * create SEO friendly URL. Format is action={...}&param1={....}&param2={...}
     * @param string $str
     * @return string
     */    
    public function createFriendlyURL($str)
    {
        /*
          input: index.php?app=html&action=view&id=c1
          input: index.php?app=html&action=view&id=i10
          input: index.php?app=users&action=register

          action=view&id=c5&page=10
          action=checkout
          action=login
         */
        $final_url = '';
        if (SCHLIX_SEF_ENABLED) {
            $command_array = [];
            parse_str($str, $command_array); // replaced with this- May 24, 2010
            $command_action = (array_key_exists('action', $command_array)) ? $command_array['action'] : '';
            if ($command_action === 'viewitem' && (array_key_exists($this->field_category_id, $command_array)))
            {
               $final_url = encode_url( $this->getFullPathByItemID($command_array[$this->field_id], $command_array[$this->field_category_id]) );
            } else return parent::createFriendlyURL ($str);

            $app_alias = $this->full_application_alias; 
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . $app_alias . $final_url; 
            return remove_multiple_slashes($final_url);

        } else return parent::createFriendlyURL ($str);
    }
    
    /**
     * Get all categories that are associated to an item
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $id
     * @return array
     */
    public function getAllCategoriesWithAssociationToItem($id)
    {
        $id = (int) $id;
        if ($id > 0)
        {
            global $SystemDB;
            
            return $SystemDB->q()->select("ci.*, c.*")
                    ->from(strval($this->table_categories), 'c')->
                            leftJoin('c', $this->table_categories_items, 'ci', "c.{$this->field_category_id} = ci.{$this->field_category_id} AND ci.{$this->field_id} = {$id}")
                            ->getQueryResultArray();
        } 
        return $this->getAllCategories ();
        
    }
            
}
