<?php

namespace SCHLIX;

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
//
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//

/**
 * class SCHLIX\cmsDatabase is about to be replaced in v2.1 
 * it will use PDO instead of MySQLi
 */
class cmsDatabase extends \MySQLi {

    const _debug_sql_ = 0;
    protected $_cache_table_columns = null;
//________________________________________________________________________//
    /**
     * Is it initialized?
     * @var bool
     */
    protected $is_initalized = false;
    
    /**
     * Server hostname, e.g. localhost, 127.0.0.1
     * @var string
     */
    protected $server;
    /**
     * Database login - username
     * @var string
     */
    protected $username;
    /**
     * Database login - password
     * @var string
     */
    protected $password;
    /**
     * Database name
     * @var string 
     */
    protected $db;
    /**
     * Last Query string
     * @var string
     */
    protected $last_query;
    /**
     * Last query result
     * @var mixed
     */
    protected $last_query_result;
    
    private $has_warning = false;
    
    private $cacheSystem;
    
    private $__dbg_ts__;
    private $__dbg_qc__;
    
    private $db_link;
    
//________________________________________________________________________//
    public function __construct($server, $db, $username, $password, $optional_port = 0, $optional_socket = '', $use_ssl = false, $ssl_ca = '') {

        if (defined('SCHLIX_SQL_CACHE_ENABLED') && (SCHLIX_SQL_CACHE_ENABLED === true))
            $this->cacheSystem = new \SCHLIX\cmsFSCache('sql');
        $this->server = $server;
        $this->username = $username;
        $this->password = $password;
        $this->db = $db;
        $_tmp = error_reporting();
        error_reporting(0);
        $port = ini_get('mysqli.default_port');
        $socket = ini_get("mysqli.default_socket") ;
        if ($optional_port > 0 && $optional_port < 65536 && $optional_port != ini_get('mysqli.default_port') && strpos($this->server, ':') === FALSE)
            $port = $optional_port;
        if ($optional_socket)
            $socket = $optional_socket;
        $connect_error = false;
        $err_msg = null;
        try
        {
            if (!$use_ssl)
            {
                $result = parent::__construct($this->server, $this->username, $this->password, $this->db, $port, $socket);
                $connect_error = $this->connect_error;

            } else 
            {
                $result = parent::__construct();
                // beta ...
                $opt = MYSQLI_CLIENT_SSL | MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT;
                if (!empty($ssl_ca))
                {
                     $this->ssl_set(NULL,NULL, $ssl_ca, NULL, NULL) ; 
                }                     
                else
                {                    
                    $this->options(MYSQLI_OPT_SSL_VERIFY_SERVER_CERT, false);
                }

                $connect_result = $this->real_connect($this->server, $this->username, $this->password, $this->db, $port, $socket, $opt);
                $connect_error = !$connect_result;
            }
        } catch (\mysqli_sql_exception $exc)
        {
            $err_msg = $exc->getMessage();
            $connect_error = true;
        }

        if ($connect_error) {
            
            $err_no = mysqli_connect_errno();
            $err_msg = mysqli_connect_error();
            if ($this->isAjaxRequest())
                return ajax_echo(ajax_reply('141', "Cannot connect to the database {$err_no}: {$err_msg}"));
            else
                $error_message = '###[SCHLIX Database offline]###<br />SQL Error '.$err_no.' - '.___h($err_msg). '<br />We will be back shortly to fix this.';
            include('view.offline.template.php');
            exit;
        }
        error_reporting($_tmp);
        $this->set_charset('utf8mb4'); // ALWAYS SET to utf8  - changed to utf8mb4 - June 2019
        $this->is_initalized = true;
        if (self::_debug_sql_)
        {
            $this->__dbg_ts__ = microtime(true);
            $this->__dbg("==================BEGIN @{$this->__dbg_ts__} ================\n");
        }
        $this->db_link = $result;
        return $result;
    }
    
    public static function genericInit()
    {
        $db_port = 0;
        $db_socket = '';
        $db_use_ssl = false;
        $db_ssl_ca = '';
        // the following lines are workaround since prior to v2.2.0-8 the variables weren't specified in the config.inc.php    
        if (defined('SCHLIX_DB_PORT') && (strpos(SCHLIX_DB_HOST,':') === FALSE))
        {
            
            $port = (int) defined('SCHLIX_DB_PORT') ? SCHLIX_DB_PORT : 0;
            if ($port > 0 && $port < 65536)
                $db_port = $port;
        }
        // socket
        if (defined('SCHLIX_DB_SOCKET') && !empty(SCHLIX_DB_SOCKET) )
        {
            $db_socket = SCHLIX_DB_SOCKET;
        }
        // use SSL
        if (defined('SCHLIX_DB_USE_SSL')  )
        {
            $db_use_ssl = (bool) SCHLIX_DB_USE_SSL;
            if (defined('SCHLIX_DB_SSL_CA')  )
            {
                $db_ssl_ca = SCHLIX_DB_SSL_CA;
            }
        }    
        return new cmsDatabase(SCHLIX_DB_HOST,SCHLIX_DB_DATABASE, SCHLIX_DB_USERNAME,SCHLIX_DB_PASSWORD, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca);
        
    }
    /**
     * Returns the query builder
     * @return \SCHLIX\cmsSQLQueryBuilder
     */
    public function q()
    {
        return new cmsSQLQueryBuilder($this);
    }
    //________________________________________________________________________//

    private function isAjaxRequest() {
        $ajax = fget_int('ajax');
        return ((!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') && ($ajax == 1));
    }

    //________________________________________________________________________//
    public function reselectDatabase() {
        if ($this->db && !$this->connect_error)
            $this->select_db($this->db);
    }

    //________________________________________________________________________//
    /**
     * Escape string, without the quote
     * @param string $s
     * @return string
     */
    public function escapeString($s) {
        return $this->real_escape_string($s);
    }

    //________________________________________________________________________//
    /**
     * Escape string and add a quote by default. Set to FALSE so it won't add a quote
     * @param string $s
     * @param bool $add_quote
     * @return string
     */
    public function sanitizeString($s, $add_quote = true) {
        $s = (string) $s;
        $val = $this->real_escape_string($s);
        return $add_quote ? "'{$val}'" : $val;
    }

    //________________________________________________________________________//
    public function getTableColumns($tablename) {
        $sql = "SHOW COLUMNS FROM `{$tablename}`";
        $columns = $this->getQueryResultArray($sql);
        if ($columns)
        {
            $this->_cache_table_columns[$tablename] = $columns;
            return $columns;
        }
        else
            return false;

    }

    //________________________________________________________________________//
    /**
     * Returns true if a table has the specified column
     * @param string $tablename Table Name
     * @param string $tablecolumn Column Name
     * @return boolean
     */
    public function tableColumnExists($tablename, $tablecolumn) {
        $tablecolumn = $this->sanitizeString($tablecolumn);
        $sql = "SHOW COLUMNS FROM `{$tablename}` LIKE {$tablecolumn}";
        $columns = $this->getQueryResultArray($sql);
        return (___c($columns) > 0);
    }

    //________________________________________________________________________//
    /**
     * Returns true if table exists. Use runtime (this script only
     * @param string $tablename
     * @param bool $current_runtime_cache
     * @return boolean
     */
    public function tableExists($tablename) {

        if ($tablename)
        {
            $tablename = $this->sanitizeString($tablename);
            $sql = "SHOW TABLES LIKE {$tablename}";
            $columns = $this->getQueryResultArray($sql);
            return (___c($columns) > 0); 
        } else return false;
    } 
    
    public function getFullSQLStatement($sql, $data = NULL) 
    {
        if (is_array($data)) {
            $unmatched = [];
            preg_match_all('/:([a-zA-Z_][a-zA-Z0-9_]+)/', $sql, $prms);
            foreach ($data as $k => $v)
            {
                $data[':'.$k] = $this->sanitizeString($v);
                unset ($data[$k]);
            }
            foreach ($prms[0] as $k)
                if (!array_key_exists($k, $data))
                        $unmatched[] = $k;
            $sql = strtr($sql, $data);
            if (!empty($unmatched))
                die (sprintf("Error: SQL Parameter [%s] was unspecified", implode(',', $unmatched)));
        }
        return $sql;
    }
    /**
     * Executes an SQL Query. If $data is specified, the variables will be bound
     * to the specified query and escaped properly with mysqli_real_escape_string.
     * If you are using a paremterized query, the default syntax is named statements
     * like the PDO one, not the MySQLi one. 
     * 
     * e.g.   INTO tbl_example (id,value) VALUES (:id,:value)
     * 
     * Although it's doable, DO NOT mix sql with the value, e.g. if you want
     * to do a non-paramterized statement, then don't mix it with the one
     * that uses paramaterized statment
     * 
     * This is wrong: INTO tbl_example (id,value) VALUES (:id,'Sanitized string')
     * 
     * Although it will work it can have an unexpected outcome if the string contains
     * variable name. e.g. ' .... String string string :example ....text'
     * 
     * @param string $sql SQL Query String
     * @param array $data parameters in an associative array that needs to be bound
     * @return mixed
     */
    #[\ReturnTypeWillChange]
    public function query($sql, $data = NULL) 
    {              
        $sql = $this->getFullSQLStatement($sql, $data);
        if (self::_debug_sql_)
        {
            $this->__dbg_qc__++;
            $caller = get_calling_class_function();
            $this->__dbg($this->__dbg_qc__."\t".$caller['class'].'->'.$caller['function']."\t".$sql."\n");
        }
        $this->last_query_result = parent::query($sql);
        $this->last_query = $sql;
        if ($this->error) { {
                $error_txt = 'Could not run query: ' . $sql . "\n Error:" . $this->error;
                if ($this->isAjaxRequest()) {
                    $result = ajax_echo(ajax_reply('151', "SQL Query Error: {$sql}.\n{$error_txt}"));
                    exit(1);
                    return $result;
                } else {
                    echo '<div style="background:pink; border:1px solid orange">DB Error - ' . $error_txt . '</div>';
                }
            }
            exit;
        }
        return $this->last_query_result;
    }
    /**
     * @internal
     * @param string $text
     */
    private function __dbg($text)
    {
        file_put_contents(CURRENT_SUBSITE_PATH.'/data/private/logs/sql-'.date('Y-m-d').'.log', $text, FILE_APPEND);
    }
    /**
     * Fetch the result of the last query and return it as an associative array
     * Do not call this method directly
     * @return array
     */
    protected function fetch_result_as_array() {
        $array = null;
        if ($this->last_query_result)
        {
            $array = $this->last_query_result->fetch_all(MYSQLI_ASSOC);
        }

        if ($this->last_query_result)
            $this->last_query_result->close();

        return $array;
    }

    /**
     * Returns a cached query result as an associative array (data row collection)
     * @param int $hours
     * @param int $minutes
     * @param int $seconds
     * @param string $sql
     * @param array $data
     * @return array
     */
    public function getTimedCachedQueryResultArray($hours, $minutes, $seconds, $sql, $data = NULL) {

        if (SCHLIX_SQL_CACHE_ENABLED && !empty($sql)) {
            $cache_key = sha1($sql); // sha1(sql);            
            $cached_result = $this->cacheSystem->get($cache_key);
            if ($cached_result !== false) {
                return $cached_result;
            } else {
                $this->query($sql, $data);
                $array = $this->fetch_result_as_array();
                $this->cacheSystem->set($cache_key, $array, $hours, $minutes, $seconds);
                return $array;
            }
        } else {
            $this->query($sql, $data);
            return $this->fetch_result_as_array();
        }
    }

    /**
     * Returns a cached query result as an associative array (data row collection)
     * @param string $sql
     * @param array $data
     * @return array Associative Array
     */
    public function getCachedQueryResultArray($sql, $data = NULL) {

        if (SCHLIX_SQL_CACHE_ENABLED && ((int) SCHLIX_SQL_CACHE_TIME > 0) && isset($sql)) {
            return $this->getTimedCachedQueryResultArray(0, 0, SCHLIX_SQL_CACHE_TIME, $sql);
        } else {
            $this->query($sql, $data);
            return $this->fetch_result_as_array();
        }
    }

    //________________________________________________________________________//
    /**
     * Returns a single row from cache
     * @param string $sql SQL Statement
     * @param array $data Parameters that needs to be bound
     * @return array Associative Array
     */
    public function getCachedQueryResultSingleRow($sql, $data = NULL) {
        $result = $this->getCachedQueryResultArray($sql, $data);
        if ($result)
            return $result[0];
    }

    //________________________________________________________________________//
    /**
     * Get the first result of a query in (a single data row)
     * @param string $sql
     * @param array $data
     * @return array
     */
    public function getQueryResultSingleRow($sql, $data = NULL) {
        $result = $this->getQueryResultArray($sql, $data);
        if ($result)
            return $result[0];
    }

    //________________________________________________________________________//
    /**
     * Get SQL query result as an associative array (data row collections)
     * @param string $sql
     * @param array $data
     * @return array
     */
    public function getQueryResultArray($sql, $data = NULL) {

        $this->query($sql, $data);
        return $this->fetch_result_as_array();
    }    
    
    //________________________________________________________________________//    
    /**
     * Returns the last insert ID
     * @return int
     */
    public function getLastInsertID() {
        return $this->insert_id;
    }

    public function getLastQueryResult()
    {
        return $this->last_query_result;
    }
    //________________________________________________________________________//
    public function simpleInsertInto($table_name, $array) {
        $temp_keys = [];
        $temp_values = [];
        foreach ($array as $key => $value) {
            $temp_keys[] = '`' . $key . '`';
            $temp_values[] = $this->sanitizeString($value);
        }
        $str_keys = implode(', ', $temp_keys);
        $str_values = implode(', ', $temp_values);

        $sql = "INSERT INTO `{$table_name}` ($str_keys) VALUES ($str_values)";
        return $this->query($sql);
    }

    //________________________________________________________________________//
    public function simpleUpdate($table_name, $array, $where_key, $where_value) {
        $string = '';
        $keys = array_keys($array);
        foreach ($keys as $key) {
            if ($key != $where_key)
                $string.= $key . '=' . $this->sanitizeString($array[$key]) . ', ';
        }
        $sql_set_cmd = substr($string, 0, strlen($string) - 2); // take out the last comma
        $where_value = $this->sanitizeString($where_value);
        $sql = "UPDATE {$table_name} SET " . $sql_set_cmd . " WHERE `{$where_key}` = $where_value;";
        return $this->query($sql);
    }

    /**
     * Performs search and replace
     * @param string $table_name
     * @param string $field_name
     * @param string $search
     * @param string $replace
     * @param string $where_condition
     */
    public function searchAndReplace($table_name, $field_name, $search, $replace, $where_condition = NULL)
    {
        $str_search = $this->sanitizeString($search);
        $str_replace = $this->sanitizeString($replace);
        $str_where = empty($where_condition) ? '' : " WHERE {$where_condition}";
        return $this->query("UPDATE `{$table_name}` SET `{$field_name}` = REPLACE(`{$field_name}`, '{$str_search}', '{$str_replace}') {$str_where}");
    }
    
    //_______________________________________________________________________________________________________________//
    public function getAllRecordsFromTable($table_name, $data_fields, $extra_criteria, $fields = '*', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC', $from_cache = false) {
        global $SystemDB;

        if ($sortby != '')
            if (strpos($sortby, ',') === false)
                if (!array_key_exists($sortby, $data_fields))
                    $sortby = '';
        $sql = $SystemDB->generateSelectSQLStatement($table_name, $fields, $extra_criteria, $start, $end, $sortby, $sortdirection, true, SCHLIX_SQL_ENFORCE_ROW_LIMIT);

        $search_result = $SystemDB->getQueryResultArray($sql);
        return $search_result;
    }
    
    //_______________________________________________________________________________________________________________//
    function generateSQLSelectParameters($tablename, $fields = '*', $extra_criteria = '', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC', $force_limit = false) {
        global $SystemDB;

        $start = intval($start);
        $end = intval($end);
        $sort_criteria = '';
        $additional_criteria = '';
        $sortby = $SystemDB->escapeString(preg_replace("/[^a-z.,_\d]/i", '', $sortby)); // further sanitize to preventy sql injection
        if (empty($fields))
            $fields = '*'; // double check
        if ($start > $end)
            $start = $end = 0; // autocorrect

        $fields = trim($fields);

        if (!is_array($fields) && $fields !== '*' && strpos($fields, ',') !== FALSE) {
            $fields_array = explode(',', $fields);
            $fields_count = ___c($fields_array);
            for ($i = 0; $i < $fields_count; $i++) {
                $tmp_field = str_replace('`', '', $fields_array[$i]);
                $fields_array[$i] = "`{$tablename}`.`{$tmp_field}`";
            }
            $fields = implode(',', $fields_array);
        } else if ($fields === '*') {
            $fields = "{$tablename}.*";
        }
        if (!empty($sortby) && !empty($sortdirection)) {
            //$sortby = quote_field_name_for_query($sortby); if there's more than one sort, this is invalid - dec 8, 2011
            $sortdirection = (strtoupper($sortdirection) == 'DESC') ? 'DESC' : 'ASC';
            $sort_criteria = " ORDER BY {$sortby} {$sortdirection}";
        }

        if ($force_limit)
            $limiter = HARDCODE_MAX_ROWLIMIT;
        else
            $limiter = 0;
        $item_start = max(0, $start);
        $item_limit = min($end - $start, $limiter);
        if ($force_limit == true && $item_limit == 0 && $item_start == 0)
            $item_limit = HARDCODE_MAX_ROWLIMIT;
        if (!empty($extra_criteria))
            $additional_criteria = " {$extra_criteria} ";

        return array('fields' => $fields, 'tablename' => $tablename, 'criteria' => $additional_criteria, 'sort_criteria' => $sort_criteria, 'start' => $item_start, 'limit' => $item_limit);
    }

//_______________________________________________________________________________________________________________//
    function generateSelectSQLStatement($tablename, $fields = '*', $extra_criteria = '', $start = 0, $end = 0, $sortby = '', $sortdirection = 'ASC', $add_where_clause = true, $force_limit = false) {
        
        $limit_criteria = '';

        $start = intval($start);
        $end = intval($end);

        if ($add_where_clause && !empty($extra_criteria))
            $criteria = " WHERE {$extra_criteria} ";
        else
            $criteria = $extra_criteria;
        $params = $this->generateSQLSelectParameters($tablename, $fields, $criteria, $start, $end, $sortby, $sortdirection, $force_limit);
        if ($params) {
            if ($params['start'] >= 0 && $params['limit'] > 0)
                $limit_criteria = "	LIMIT {$params['start']}, {$params['limit']}";
            $sql = "SELECT {$params['fields']} FROM {$params['tablename']} {$params['criteria']} {$params['sort_criteria']} {$limit_criteria}";
            return $sql;
        } else
            return false;
    }

//________________________________________________________________________//
    function __destruct() {
        if ($this->is_initalized && isset($this->thread_id))
        {                        
            if ($this->db_link)            
                @$this->close(); //@$this->close($this->db_link); PHP8.2 fix
            if (self::_debug_sql_)
            {
                $date_end = get_current_datetime();
                $diff = round(microtime(true) - $this->__dbg_ts__, 3);
                $this->__dbg("\n========= END @ {$date_end} ~ {$diff} seconds ================\n");
            }        
            
        }
    }

}

//________________________________________________________________________//

class cmsSessionHandler implements \SessionHandlerInterface{

    private $session_table = 'gk_session_handler';
    private $db;
    private $debug_filename;
    /**
     *
     * @var string 
     */
    private static $samesite_mode = null;
    /**
     *
     * @var boolean
     */
    private static $samesite_set = null;

    public function __construct() {
        $this->debug_filename = SCHLIX_SITE_PATH . '/debug_session.txt';
        $this->db = cmsDatabase::genericInit();//
        /*$save_handler_result = session_set_save_handler(
            [$this, "open"], [$this, "close"], [$this, "read"], [$this, "write"], [$this, "destroy"], [$this, "gc"]
        );

        if (!$save_handler_result)
            die('Failure when calling session_set_save_handler() ');*/
        // the following prevents unexpected effects when using objects as save handlers
        register_shutdown_function('session_write_close');
    }

    /**
     * Start session
     */
    public function start()
    {
        $secure = isCurrentRequestSSL();
        $ua_same_site_ok = user_current_browser_samesite_none_compatible();
        $httponly = true; // prevent JavaScript access to session cookie
        $samesite = strtolower(defined('SCHLIX_COOKIE_SAMESITE') ? SCHLIX_COOKIE_SAMESITE : ($secure ? 'none' : '')); // default to none

        $site_base_path = SCHLIX_SITE_HTTPBASE.'/';
        $maxlifetime = get_schlix_max_user_session_time(); // 30 days
        $host = schlix_get_current_host_no_port();
        
        schlix_session_set_cookie_params($maxlifetime, $site_base_path, $host, $secure, $httponly, $samesite);
        
        session_name(SCHLIX_SESSION_NAME);
        session_start();
        // Reset the expiration time upon page load
        if (isset($_COOKIE[SCHLIX_SESSION_NAME]))
        {
            schlix_set_cookie(SCHLIX_SESSION_NAME, $_COOKIE[SCHLIX_SESSION_NAME], time() + $maxlifetime, $site_base_path, $host, $secure, $httponly, $samesite);
        }
        
    }
    /**
     * Open the session
     * @return bool
     */
    #[\ReturnTypeWillChange]
    public function open($save_path, $session_name){
        // $limit = time() - (3600 * 24);
        //  $this->db->query("DELETE FROM {$this->session_table} WHERE timestamp < {$limit}");
        return true;
    }

    /**
     * Close the session
     * @return bool
     */
    #[\ReturnTypeWillChange]
    public function close(){
        return true;
    }

    /**
     * Read the session
     * @param int session id
     * @return string string of the sessoin
     */
    #[\ReturnTypeWillChange]
    public function read($id) {

        $sanitized_id = $this->db->sanitizeString($id);
        $sql = "SELECT data FROM {$this->session_table} WHERE id =  {$sanitized_id}";
        $result = $this->db->getQueryResultSingleRow($sql);

        return $result ? $result['data'] : ''; // PHP 7.1.x fix - OCt 2, 2017
    }

    /**
     * Get count of unique IP addresses within x days
     * @param int $days
     * @return int
     */
    public function getUniqueIPAddressWithinDay($days) {
        $days = (int) $days;
        $sql = "SELECT COUNT(DISTINCT(`ip_address`)) AS session_count FROM `gk_session_handler` WHERE DATEDIFF(NOW(),`timestamp`) <= {$days}";
        $result = $this->db->getQueryResultSingleRow($sql);
        return $result['session_count'];
    }

    /**
     * Return the total number of active sessions within x seconds
     * @param int $seconds
     * @return int
     */
    public function getCountWithinSeconds($seconds) {
        $seconds = (int) $seconds;
        $sql = "SELECT COUNT(*) as session_count FROM `gk_session_handler` WHERE TIMESTAMPDIFF(SECOND, `timestamp`,NOW()) < {$seconds}";
        $result = $this->db->getQueryResultSingleRow($sql);
        return $result['session_count'];
    }

    /**
     * Write the session
     * @param int session id
     * @param string data of the session
     * @return bool
     */
    #[\ReturnTypeWillChange]
    public function write($id, $data) {
        global $SystemLog;

        if ($id != '' /* && $data != '' */ && $this->db != NULL) {
            $sanitized_id = "'" . $this->db->real_escape_string($id) . "'";
            $sanitized_data = "'" . $this->db->real_escape_string($data) . "'";
            $ip_addr = "'" . $this->db->real_escape_string(get_user_real_ip_address()) . "'";
            $sanitized_user_agent =  "'" . $this->db->real_escape_string(fserver_string('HTTP_USER_AGENT', 255)) . "'";
            $sql = "REPLACE INTO {$this->session_table} (`id`,`data`,`ip_address`,`user_agent`) VALUES({$sanitized_id}, {$sanitized_data}, {$ip_addr}, {$sanitized_user_agent})";
            $this->db->query($sql);
            return true; // fix for PHP7 - Oct 5, 2016
        } else {
            debug_backtrace();
            $error_message = "Error when trying to write session";
            $SystemLog->record($error_message);
            die($error_message);
        }
    }

    /**
     * Destroy the session
     * @param int session id
     * @return bool
     */
    #[\ReturnTypeWillChange]
    public function destroy($id) {
        $sanitized_id = $this->db->sanitizeString($id);
        $this->db->query("DELETE FROM {$this->session_table} WHERE `id` = {$sanitized_id}");
        schlix_simple_set_cookie( session_name(), "", time() - 3600);
        return true;
    }

    /**
     * Garbage Collector
     * @param int life time (sec.)
     */   
    #[\ReturnTypeWillChange]
    public function gc($maxlifetime) {
        $maxlifetime = (int) $maxlifetime;
        if ($maxlifetime < 1)
            $maxlifetime = 1;
        $sql = "DELETE FROM {$this->session_table} WHERE timestamp < DATE_SUB(NOW(), INTERVAL {$maxlifetime} SECOND)";
        $this->db->query($sql);
        return 0;
    }

}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
class cmsLogger {

    protected $table_name;

    public function __construct($table_name) {
        $this->table_name = $table_name;
    }

    //________________________________________________________________________//
    public function record($error, $module = 'system', $type = 'info') {
        global $SystemDB;

        // create new data - will be sanitized with InsertSQL
        $datavalues = [];
        $datavalues['ip_address'] = get_user_real_ip_address();
        $datavalues['module'] = $module;
        $datavalues['type'] = $type;
        $datavalues['referrer'] = fserver_string('HTTP_REFERER', 255); 
        $datavalues['user_agent'] =  fserver_string('HTTP_USER_AGENT', 255); 
        $datavalues['request_uri'] =  fserver_string('REQUEST_URI', 255);  
        $datavalues['description'] = $error;
        $SystemDB->simpleInsertInto($this->table_name, $datavalues);
    }

    //________________________________________________________________________//
    public function error($error, $module = 'system') {

        $this->record($error, $module, 'error');
    }

    public function warn($error, $module = 'system') {

        $this->record($error, $module, 'warn');
    }

    public function info($error, $module = 'system') {

        $this->record($error, $module, 'info');
    }

    public function debug($error, $module = 'system') {

        $this->record($error, $module, 'debug');
    }

    //________________________________________________________________________//
    /**
     * Returns the textual data of log by month and year
     * @param int $month
     * @param int $year
     */
    private function saveLogDataByMonthAndYear($year, $month) {
        global $SystemDB;

        $month = (int) $month;
        $year = (int) $year;
        $sql = "SELECT * FROM {$this->table_name}  WHERE MONTH(`date_created`) = {$month} AND YEAR(`date_created`) = {$year} ORDER BY `date_created`";
        //$data = $SystemDB->getQueryResultArray($sql);
        $data = $SystemDB->query($sql);
        $result = '';

        $logfilename = SCHLIX_SITE_PATH . "/data/private/logs/log-{$year}-{$month}.txt";
        $file = @fopen($logfilename, 'w');
        if ($file) {
            while ($log = $SystemDB->getLastQueryResult()->fetch_assoc())
            {
                $logline = "#{$log['id']}|{$log['date_created']}|Module:{$log['module']}|Type:{$log['type']}|{$log['description']}|Info: {$log['ip_address']} - {$log['user_agent']} - {$log['referrer']} - {$log['request_uri']}\n";
                fwrite($file, $logline);
            }
            $SystemDB->getLastQueryResult()->close();
            $data = NULL;
            fclose($file);
        } else {
            $this->record("Unable to save {$logfilename}");
        }
    }

    //________________________________________________________________________//
    /**
     * Archive all the logs separated by month and year, e.g. 2010-10.txt, etc
     */
    public function Archive() {
        global $SystemDB;
        $sql = "SELECT DISTINCT YEAR(`date_created`) AS log_year, MONTH(`date_created`) AS log_month FROM {$this->table_name}  WHERE MONTH(`date_created`) <> MONTH(NOW());";
        $yearmonths = $SystemDB->getQueryResultArray($sql);
        foreach ($yearmonths as $yearmonth) {
            $log_year = $yearmonth['log_year'];
            $log_month = $yearmonth['log_month'];
            $this->saveLogDataByMonthAndYear($log_year, $log_month);
        }
        $sql = "DELETE FROM {$this->table_name}  WHERE MONTH(`date_created`) <> MONTH(NOW());;";
        $SystemDB->query($sql);
    }

    /**
     * Resolve hostnames from ip_address field
     * @global SCHLIX\cmsDatabase $SystemDB
     */
    public function updateLogHostnames() {
        global $SystemDB;
        $count = 0;

        $sql = "SELECT id, ip_address FROM {$this->table_name} WHERE host_name IS NULL";
        $unprocessed_logs = $SystemDB->getQueryResultArray($sql);
        if ($unprocessed_logs) {
            foreach ($unprocessed_logs as $log) {
                $log_id = $log['id'];
                $host_name = @gethostbyaddr($log['ip_address']);
                if ($host_name == $log_id || empty($host_name))
                    $host_name = '-';
                else
                    $count++;
                $data_update = array('host_name' => $host_name);
                $SystemDB->simpleUpdate($this->table_name, $data_update, 'id', $log_id);
                $data_update = null;
            }
            //if ($count > 0)
              //  $this->record("{$count} IP addresses resolved");
        }
    }

    //_______________________________________________________________________________________________________________//
    /**
     * CRON Scheduler Method - resolve hostnames in log
     * @global cmsLogger $SystemLog
     */
    public static function processRunResolveHostnames() {
        global $SystemLog;

        $SystemLog->updateLogHostnames();
    }

    //_______________________________________________________________________________________________________________//
    /**
     * CRON Scheduler Method
     * @global cmsLogger $SystemLog
     */
    public static function processRunCleanupLog() {
        global $SystemLog;

        $SystemLog->Archive();
    }

}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
/**
 * A helper class that represents an SQL table
 */
class cmsSQLTable {

    private $table_name;
    private $table_exists = false;
    private $columns = NULL;
    private $primary_key_names;
    private $enable_cache;
    private $db; // database - so users can reuse this class for another db

    //________________________________________________________________________//

    /**
     * Construct
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $table_name
     * @param \SCHLIX\cmsDatabase $from_database
     * @param bool $populate_field_names
     */
    public function __construct($table_name, $from_database = null, $populate_field_names = true) {
        global $SystemDB;

        if (!empty($table_name)) {
            if ( isset($from_database) && !($from_database instanceof cmsDatabase))
                die ('Please initialize with instance of cmsDatabase');
            $this->table_name = $table_name;
            $this->db = ($from_database == null) ? $SystemDB : $from_database;
            $this->table_exists = $this->tableExists($this->table_name); // $this->tableExists();
        } else {
            ob_start();
            debug_print_backtrace();
            $debug_content = ob_get_contents();
            ob_end_clean();
            $debug_content = nl2br($debug_content);
            die("Empty Table Name<br />" . $debug_content);
        }
        /* if ($populate_field_names)
          return $this->tableExists(true); */
    }

    //________________________________________________________________________//
    public function __toString() {
        return $this->table_name;
    }

    //________________________________________________________________________//
    /**
     * Verify if this table exists
     * @param bool $from_cache
     * @return bool
     */
    public function tableExists() 
    {
        return \SCHLIX\cmsContextCache::get('db_table_exists',$this->table_name);
        //return $this->table_exists ? true : $this->db->tableExists($this->table_name);
    }

    public function q()
    {
        global $SystemDB;
        
        return $SystemDB->q()->from($this->table_name);
    }
    //________________________________________________________________________//
    /**
     * Get Table columns as array - this is the raw output from MySQL
     * @param bool $from_cache
     * @return array
     */
    public function getFields() {
        if ($this->table_exists)
        {
            $fields = \SCHLIX\cmsContextCache::get('tablecolumns',$this->table_name);
            if ($fields)
                return $fields;
            else 
            {
                $fields = $this->db->getTableColumns($this->table_name);
                \SCHLIX\cmsContextCache::set('tablecolumns',$this->table_name, $fields);
                return $fields;
            }
        } 
        return NULL;        
    }

    /**
     * The purpose of this function is type safety check. It's important and this 
     * is introduced in v2.2.0. Given a key-value array data array, conform the 
     * types to table definition for safety check (length).
     * If the field does not exist, then it will be removed from the variable. 
     * If it violates the type constrain then it will be modified according to the database type.
     * @param array $datavalues
     * @return array
     */
    public function conformDataValuesToFieldTypes($datavalues)
    {
        $result = [];
        if ($datavalues)
        {
            $fields = $this->getParsedFields();
            foreach ($datavalues as $k => $v)
            {                
                if (array_key_exists($k, $fields))
                {                    
                    $pt = $fields[$k]['php_type'];
                    switch ($pt)
                    {
                        case 'datetime':                            
                            if (!is_date($v) || $v === NULL_DATE || $v === NULL)
                                $result[$k] = NULL;     
                            else 
                                $result[$k] = $v;
                            break;
                        // TODO: add 'time' 
                        case 'int':
                            if ($v === NULL)
                            {
                                if  ($fields[$k]['allow_null']) 
                                    $result[$k] = NULL;
                                else 
                                {
                                    $result[$k] = isset($fields[$k]['default_value']) ? (int) $fields[$k]['default_value'] : 0;                                    
                                }
                            } else 
                            {                                
                                $v = (int) $v;
                                if (isset($fields[$k]['unsigned'])) $v = abs($v);
                                $result[$k] = $v;
                            }
                            break;
                        case 'string':                            
                            $result[$k] = str_limit($v, $fields[$k]['max_length']);                            
                            break;
                        default: 
                            $result[$k] = $v;
                            break;
                    }
                    
                } 
            }
        }
        return $result;
    }
    //________________________________________________________________________//
    /**
     * Get table fields with field types parsed
     * @param bool $from_cache
     * @return array|null
     */
    public function getParsedFields() {

        $m_to_p = [
            'varchar' => 'string', 'char' => 'string', 'text' => 'string', 'tinytext' => 'string', 'mediumtext' => 'string', 'longtext' => 'string',
            
            'int' => 'int','tinyint' => 'int', 'smallint' => 'int', 'mediumint' => 'int', 'bigint' => 'int',
                        
            'decimal' => 'float', 'double' => 'float', 'real' => 'float', 'float' => 'float',
            
            'datetime' => 'datetime', 'timestamp' => 'datetime', 'date' => 'date', 'time' => 'time',
            
            ];
        $signed_limits = [
            'tinyint' => ['min' => -128, 'max' => 127],
            'smallint' => ['min' => -32768, 'max' => 32767],
            'mediumint' => ['min' => -8388608, 'max' => 8388607],
            'int' => ['min' => -2147483648, 'max' => 2147483647],
            'bigint' => ['min' => PHP_INT_MIN, 'max' => PHP_INT_MAX],
        ];
        
        $unsigned_limits = [
            'tinyint' => ['min' => 0, 'max' => 255],
            'smallint' => ['min' => 0, 'max' => 65535],
            'mediumint' => ['min' => 0, 'max' => 16777215],
            'int' => ['min' => 0, 'max' => 4294967295],
            'bigint' => ['min' => 0, 'max' => PHP_INT_MAX],
        ];
        
        $text_max_length = [
            'tinytext' => 255, 'text' => 65535, 'mediumtext' =>   16777215 , 'longtext' => 4294967295
        ];
        
        $fields = $this->getFields();
        $def = [];
        if ($fields)
        {
            foreach ($fields as $f)
            {
                $id = $f['Field'];
                $full_type = strtolower($f['Type']);
                $val = ['full_type' => $full_type, 'allow_null' => strtolower($f['Null']) === 'yes'];
                switch ($f['Key'])
                {
                    case 'PRI':
                        $val['primary_key'] = true;
                        break;
                    case 'UNI':
                        $val['unique_key'] = true;
                        break;           
                    case 'MUL':
                        $val['indexed'] = true;
                        break;                                   
                }
                if ($f['Default'])
                    $val['default_value'] = $f['Default'];
                $pattern = '/^((?:[a-z][a-z]+))?(?:\((\d+|(\d+),(\d+))\))?(\\s+?)?((?:[a-z][a-z]+))?/';
                preg_match_all($pattern, $full_type, $m);
                if ($m)
                {
                    $val['type'] = strtolower($m[1][0]);
                    if ((int) $m[3][0] > 0 && (int) $m[4][0] > 0)
                    {
                        $val['size'] = $m[3][0];
                        $val['precision'] = $m[4][0];
                    } else if ((int) $m[2][0] > 0)
                    {
                        $val['size'] = (int) $m[2][0];
                    }
                    $php_type =  $val['php_type'] = $m_to_p[$val['type']];
                    switch ($php_type)
                    {
                        case 'int':
                            if (strtolower($m[6][0]) === 'unsigned')
                            {
                                $val['unsigned'] = true;
                                $val['min'] = $unsigned_limits[$val['type']]['min'];
                                $val['max'] = $unsigned_limits[$val['type']]['max'];
                            } else 
                            {
                                $val['min'] = $signed_limits[$val['type']]['min'];
                                $val['max'] = $signed_limits[$val['type']]['max'];                            
                            }
                            
                            break;
                        case 'float':
                            $val['min'] = PHP_FLOAT_MIN;
                            $val['max'] = PHP_FLOAT_MAX;
                            break;
                        case 'string':
                            if (isset($val['size']) && ($val['size'] > 0))
                                $val['max_length'] = $val['size'];
                            else 
                            {
                                $val['max_length'] = $text_max_length[$val['type']];
                            }
                            if (isset($val['max_length']) && ($val['max_length'] == 0))
                                $val['max_length'] = 255; // just in case
                            unset($val['size']);
                    }                    
                 }
                
                $def[$id] = $val;
                
            }
            return $def;
        }
        return NULL;        
    }
    
    /**
     * Returns true if fieldname exists
     * @param string $fieldname
     * @return bool
     */
    public function fieldExists($fieldname)
    {
        $all_field_names = $this->getFieldNamesAsArray();
        return is_array($all_field_names) && in_array($fieldname, $all_field_names);
    }
    
    //________________________________________________________________________//
    /**
     * Returns this table's column name as array from cache
     * @return array
     */
    public function getFieldNamesAsArray() {

        global $__schlix_table_columns;
        
        // TODO - INEFFICIENT - PLEASE FIX - QUICK FIX FOR PHP8 
        $tbl = isset($__schlix_table_columns[$this->table_name]) ? $__schlix_table_columns[$this->table_name] : null;
        if ($tbl != null)
        {
            return $tbl;
        } else
        {
            $columns_array = $this->getFields();
            $column_names = [];
            if ($columns_array) {
                foreach ($columns_array as $column) {
                    $column_names[] = $column['Field'];
                }
                $__schlix_table_columns[$this->table_name] = $column_names;
                return $column_names;
            }
            return NULL;
        }
    }

    /**
     * Returns this table's column name as array - set to key 
     * @param bool $from_cache
     * @return array
     */
    public function getFieldNamesAsArrayKeys() {
        $columns_array = $this->getFieldNamesAsArray();
        return ($columns_array !== NULL) ? array_fill_keys($columns_array, NULL) : NULL;
    }

    //________________________________________________________________________//
    /**
     * Returns the list of field/column names
     * @return string
     */
    public function getFieldNamesAsString() {
        $column_names = $this->getFieldNamesAsArray();
        return implode(',', $column_names);
    }

    //________________________________________________________________________//
    /**
     * Returns the primary key in array form
     * @return array
     */
    public function getPrimaryKeysAsArray() {
        if ($this->table_exists) {
                $sql = "SHOW KEYS FROM {$this->table_name} WHERE key_name = 'PRIMARY'";
                return $this->db->getQueryResultArray($sql);
        } else
            return NULL;
    }

    //________________________________________________________________________//
    /**
     * Returns a comma separated field names
     * @param bool $from_cache
     * @return string
     */
    public function getPrimaryKeyFieldNames() {
        $result = $this->getPrimaryKeyFieldNamesAsArray();
        return (___c($result) > 0) ? implode(',', $result) : null;
    }

    //________________________________________________________________________//
    /**
     * Returns an array containing field names
     * @param bool $from_cache
     * @return array
     */
    public function getPrimaryKeyFieldNamesAsArray() {
        global $__schlix_table_pks;
        if (is_array($__schlix_table_pks) &&  array_key_exists($this->table_name, $__schlix_table_pks) && $__schlix_table_pks[$this->table_name] != NULL)
        {
            return $__schlix_table_pks[$this->table_name];
        } else
        {
            if ($this->table_exists)
            {
                $primary_keys_array = $this->getPrimaryKeysAsArray();
                if ($primary_keys_array)
                {
                    $__schlix_table_pks[$this->table_name] = array_column($primary_keys_array,'Column_name');
                    return $__schlix_table_pks[$this->table_name];
                }
            }
        }  
        return NULL;
    }

    //________________________________________________________________________//
    /**
     * Executes SELECT * FROM table_name WHERE ...
     *
     * @param string $fields
     * @param string $extra_criteria
     * @param string $sortby
     * @param string $sortdirection
     * @param int $start
     * @param int $end
     * @param bool $add_where_clause
     * @param bool $force_limit
     * @param bool $from_cache
     * @return array
     */
    public function selectAll($fields = '*', $extra_criteria = '', $sortby = '', $sortdirection = 'ASC', $start = 0, $end = 0, $add_where_clause = true, $force_limit = false, $from_cache = false) {
        $start = (int) $start;
        $end = (int) $end;

        $sql = $this->generateSQLSelectStatement($this->table_name, $fields, $extra_criteria, $sortby, $sortdirection, $start, $end, true, SCHLIX_SQL_ENFORCE_ROW_LIMIT);
        $search_result = $this->db->getQueryResultArray($sql, $from_cache);
        return $search_result;
    }

    //________________________________________________________________________//
    /**
     * Returns a single row from SELECT * FROM table_name WHERE ...
     *
     * @param string $fields
     * @param string $extra_criteria
     * @param string $sortby
     * @param string $sortdirection 
     * @param bool $add_where_clause
     * @param bool $force_limit
     * @param bool $from_cache
     * @return array
     */
    public function selectOne($fields = '*', $extra_criteria = '', $sortby = '', $sortdirection = 'ASC', $add_where_clause = true, $force_limit = false, $from_cache = false) {
        $result = $this->selectAll($fields, $extra_criteria, 0, 1, $sortby, $sortdirection, $add_where_clause, $force_limit, $from_cache);
        return (($result != NULL) && (___c($result) > 0)) ? $result[0] : NULL;
    }

    //________________________________________________________________________//
    /**
     * Given a comma separated fieldnames, return them escaped with backtick
     * @param string $fieldnames
     * @param bool $check_if_field_exists
     * @return string
     */
    public function escapeFieldNames($fieldnames, $check_if_field_exists = false) {
        if ($check_if_field_exists)
            $table_fields = $this->getFieldNamesAsArray();
        if (strpos($fieldnames, ',') !== FALSE) {
            $fieldnames_array = explode(',', $fieldnames);
            $fieldnames_result = [];
            $count = ___c($fieldnames_array);
            // reduce if in loop
            if ($check_if_field_exists) {
                for ($i = 0; $i < $count; $i++) {
                    $field_name = trim($fieldnames_array[$i]);
                    if (in_array($field_name, $table_fields))
                        $fieldnames_result[] = "`{$field_name}`";
                }
            } else {
                for ($i = 0; $i < $count; $i++) {
                    $field_name = trim($fieldnames_array[$i]);
                    $fieldnames_result[] = "`{$field_name}`";
                }
            }
            return implode(',', $fieldnames_result);
        } else {
            $fieldnames = trim($fieldnames);
            if ($check_if_field_exists)
                return (in_array($field_name, $fieldnames_result)) ? "`{$fieldnames}`" : EMPTY_STRING;
            else
                return "`{$fieldnames}`";
        }
    }

    //________________________________________________________________________//
    /**
     * An alias of SELECT COUNT(fieldnames) FROM (this->table) WHERE (criteria)
     * @param string $field_names
     * @param string $criteria
     * @param bool $from_cache
     * @return int
     */
    public function count($field_names = '*', $criteria = '', $from_cache = false) {
        $criteria_txt = '';
        if (!empty($criteria))
            $criteria_txt = " WHERE {$criteria}";
        $field_names = $this->escapeFieldNames($field_names);
        $sql = "SELECT COUNT({$field_names}) as total_item_count FROM {$this->table_name} {$criteria_txt}";
        $result = $this->db->getQueryResultSingleRow($sql, $from_cache);
        return $result['total_item_count'];
    }

    //________________________________________________________________________//
    /**
     * Executes INSERT INTO table_name VALUES ( ... ) ON DUPLICATE KEY UPDATE ...
     * @param array $datavalues
     * @param string $insert_options
     * @param bool $update_if_duplicate
     * @param array $update_datavalues
     * @return query result
     */
    public function quickInsert(array $datavalues, $insert_options = EMPTY_STRING, $update_if_duplicate = false, array $update_datavalues = []) {
        $valid_options = array('', 'LOW_PRIORITY', 'HIGH_PRIORITY', 'DELAYED', 'IGNORE');
        
        $datavalues = $this->conformDataValuesToFieldTypes($datavalues);

        $sql = "INSERT {$insert_options} INTO {$this->table_name} " . $this->convertDataValuesArrayIntoSQLForInsertOperation($datavalues);
        if ($update_if_duplicate && is_array($update_datavalues) && ___c($update_datavalues) > 0) {
            $sql.= " ON DUPLICATE KEY UPDATE " . $this->convertDataValuesArrayIntoSQLForUpdateOperation($update_datavalues);
        }
        return $this->db->query($sql);
    }

    //________________________________________________________________________//
    /**
     * Executes UPDATE table_name SET field=value WHERE ...
     * @param array $datavalues
     * @param string $unsanitized_where_condition
     * @return query result
     */
    public function quickUpdate(array $datavalues, $unsanitized_where_condition) {
        $datavalues = $this->conformDataValuesToFieldTypes($datavalues);        
        $sql = "UPDATE {$this->table_name} SET " . $this->convertDataValuesArrayIntoSQLForUpdateOperation($datavalues);
        if ($unsanitized_where_condition !== EMPTY_STRING) {
            $sql.=" WHERE " . $unsanitized_where_condition;
        }
        return $this->db->query($sql);
    }

    /**
     * Performs search and replace
     * @param string $field_name
     * @param string $search
     * @param string $replace
     * @param string $where_condition
     */
    public function searchAndReplace($field_name, $search, $replace, $where_condition = NULL)
    {
        if ($this->fieldExists($field_name))
        {
            $str_search = sanitize_string($search);
            $str_replace = sanitize_string($replace);
            $str_where = empty($where_condition) ? '' : " WHERE {$where_condition}";
            return $this->db->query("UPDATE `{$this->table_name}` SET `{$field_name}` = REPLACE(`{$field_name}`, '{$str_search}', '{$str_replace}') {$str_where}");
        } else return false;
    }
    
    //________________________________________________________________________//
    /**
     * Executes DELETE FROM table_name WHERE ....
     * @param string $unsanitized_where_condition
     * @return query result
     */
    public function quickDelete($unsanitized_where_condition) {
        $sql = "DELETE FROM {$this->table_name}";
        if ($unsanitized_where_condition !== EMPTY_STRING) {
            $sql.=" WHERE " . $unsanitized_where_condition;
        }
        return $this->db->query($sql);
    }

    //________________________________________________________________________//
    /**
     * Executes TRUNCATE TABLE table_name
     * @return array
     */
    public function truncate() {
        $sql = "TRUNCATE TABLE {$this->table_name}";
        return $this->db->query($sql);
    }

    //________________________________________________________________________//
    /**
     * Optimize MySQL table
     * @param string $unsanitized_options
     * @return array
     */
    public function optimize($unsanitized_options = EMPTY_STRING) {
        $sql = "OPTIMIZE {$unsanitized_options} TABLE {$this->table_name}";
        return $this->db->getQueryResultSingleRow($sql);
    }

    //________________________________________________________________________//
    /**
     * Executes ANALYZE TABLE table_name
     * @param string $unsanitized_options
     * @return array
     */
    public function analyze($unsanitized_options = EMPTY_STRING) {
        $sql = "ANALYZE {$unsanitized_options} TABLE {$this->table_name}";
        return $this->db->getQueryResultSingleRow($sql);
    }

    //________________________________________________________________________//
    /**
     * Returns the CREATE TABLE syntax of this table
     * @return string
     */
    public function showCreateTable() {
        $sql = "SHOW CREATE TABLE {$this->table_name}";
        $result = $this->db->getQueryResultSingleRow($sql);
        return is_array($result) ? $result['Create Table'] : NULL;
    }

    /**
     * Escape field name
     * @param string $field_name
     * @return string
     */
    public function escapeFieldName($field_name)
    { 
        return str_replace( '`', '``', $field_name );
        
    }
    //________________________________________________________________________//
    /**
     * Given a key/value pair array, returns the SQL for INSERT statement
     * @param array $array
     * @param bool $dont_sanitize
     * @return string
     */
    public function convertDataValuesArrayIntoSQLForInsertOperation(array $array, $dont_sanitize = false) {

        $temp_keys = [];
        $temp_values = [];        
        foreach ($array as $key => $value) {
            if ($this->fieldExists($key))
            {
                $key = $this->escapeFieldName($key);
                $temp_keys[] = "`{$key}`";
                if ($value !== NULL)
                    $temp_values[] = ($dont_sanitize ? $value : sanitize_string($value));
                else 
                    $temp_values[] = 'NULL';            
            }
        }
        $str_keys = implode(', ', $temp_keys);
        $str_values = implode(', ', $temp_values);
        $string = "($str_keys) VALUES ($str_values)";
        return $string;
    }

    //________________________________________________________________________//
    /**
     * Given a key/value pair array, returns the SQL for UPDATE statement
     * @param array $array
     * @param bool $dont_sanitize
     * @return string
     */
    public function convertDataValuesArrayIntoSQLForUpdateOperation(array $array, $dont_sanitize = false) {
        $string = EMPTY_STRING;
        // remove primary key from update statement
        $primary_keys = $this->getPrimaryKeyFieldNamesAsArray();
        foreach ($primary_keys as $primary_key) {
            if (array_key_exists($primary_key, $array))
                unset($array[$primary_key]);
        }
        // merge
        foreach ($array as $key => $value) {
            $key = $this->escapeFieldName($key);            
            if ($value !== NULL)            
                $update_val = ($dont_sanitize ? $value : sanitize_string($value));
            else
                $update_val = 'NULL';
            $string.= "`{$key}`={$update_val}, ";
        }
        $string = substr($string, 0, strlen($string) - 2); // take out the last comma

        return $string;
    }

    //_______________________________________________________________________________________________________________//
    public function generateSQLSelectStatement($tablename, $fields = '*', $extra_criteria = '', $sortby = '', $sortdirection = 'ASC', $start = 0, $end = 0, $add_where_clause = true, $force_limit = false) {
        // 1) fields cannot be empty
        if (empty($fields))
            $fields = '*'; // double check

            
// 2) sanitize integer value
        $start = max(0, (int) $start);
        $end = (int) $end;
        if ($start >= 0 && $end > 0) {
            if ($start > $end)
                $start = $end = 0; // autocorrect, now guaranteed that $end >= $start
            $item_start = $start;
            $item_limit = $end - $start;
            if ($force_limit === true) {
                if (($item_limit === 0 && $item_start === 0) || ($item_limit > HARDCODE_MAX_ROWLIMIT))
                    $item_limit = HARDCODE_MAX_ROWLIMIT;
            }
            $statement_limit_criteria = " LIMIT {$item_start}, {$item_limit}";
        } else {
            $statement_limit_criteria = EMPTY_STRING;
        }
        // 3) Add where clause?
        $statement_criteria = ($add_where_clause && !empty($extra_criteria)) ? " WHERE {$extra_criteria} " : $extra_criteria;
        // 4) Sort By
        if (!empty($sortby) && !empty($sortdirection)) {
            //$sortby = quote_field_name_for_query($sortby); if there's more than one sort, this is invalid - dec 8, 2011
            $fixed_sort_by = $this->escapeFieldNames($sortby);
            $fixed_sort_direction = (strtoupper($sortdirection) == 'DESC') ? 'DESC' : 'ASC';
            $statement_sort_criteria = " ORDER BY {$fixed_sort_by} {$fixed_sort_direction}";
        } else {
            $statement_sort_criteria = EMPTY_STRING;
        }
        // DO NOT sanitize table field names by default, e.g. COUNT(*) as table_name can be correct
        // $statement_table_name = $this->escapeFieldNames($tablename);
        $sql = "SELECT {$fields} FROM {$tablename} {$statement_criteria} {$statement_sort_criteria} {$statement_limit_criteria}";
        return $sql;
    }

}

/**
 * cmsConfigRegistry - stores config in the database
 */
class cmsConfigRegistry {

    protected $table_name;

    /**
     *
     * @var \DOMDocument 
     */
    protected $config_document;
    /**
     * Please specify an existing table name where the config is stored
     * @param string $table_name
     */
    public function __construct($table_name) {

        $this->table_name = $table_name;
    }

    //________________________________________________________________________//
    /**
     * Check if a config key exists or not
     * @global SCHLIX\cmsDatabase $SystemDB
     * @param string $section
     * @param string $key
     * @param boolean $from_cache
     * @return boolean
     */
    public function exists($section, $key) {
        global $SystemDB;

        
        $results_array = $SystemDB->getQueryResultArray("SELECT * FROM {$this->table_name} WHERE section= :section AND `key`= :key", ['section' => $section, 'key' => $key]);
        return (___c($results_array) > 0);
    }
    
    private function sanitizeResult($c)
    {
        $config_key = $c['key'];
        $config_prefix = '';
        $prefix_pos = strpos($config_key, '_');
        if ($prefix_pos !== FALSE) {
            $config_prefix = substr($config_key, 0, $prefix_pos);
        }
        switch ($config_prefix) {
            case 'array':
                $result = unserialize($c['value']);
                break;
            case 'float':
                $result = (double) $c['value'];
                break;
            case 'double':
                $result = (double) $c['value'];
                break;
            case 'int':
                $result = (int) $c['value'];
                break;
            case 'bool':
                if ($c['value'] === '' || $c['value'] === NULL)
                    $result = NULL;
                else
                {
                    $result = ((bool) $c['value']) ? 1: 0;
                }
                
                break;
            case 'str':
            case 'string':
            default:
                $result = strval($c['value']);
                break;
        }
        return $result;
    }
    
    /**
     * Clear context cache
     * @param string $section
     * @param string $key
     */
    public function clearCache($section, $key = '')
    {
        $cx_key = "_config_{$section}_{$key}";      
        cmsContextCache::purge("__config__", $cx_key);
    }
     //________________________________________________________________________//
    /**
     * Get configuration content by key
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $section
     * @param string $key
     * @param boolean $use_cached_result
     * @return boolean|int|double|float|string|array
     */
    public function get($section, $key = '', $use_cached_result = false) {
        global $SystemDB;

        $cx_key = "_config_{$section}_{$key}";
        $context_cache = cmsContextCache::get("__config__", $cx_key);
        if ($context_cache)
            return $context_cache;
        else
        {
            $sanitized_section = sanitize_string($section);        
            $sql = "SELECT * FROM {$this->table_name} WHERE section={$sanitized_section} ";
            if (!empty($key))
            {
                $sanitized_key = sanitize_string($key);
                $sql.= " AND `key`={$sanitized_key}";
            }
            $results_array = $use_cached_result ? $SystemDB->getCachedQueryResultArray($sql) : $SystemDB->getQueryResultArray($sql);
            $config = [];
            foreach ($results_array as $result) {
                $config_key = $result['key'];
                $config[$config_key] = $this->sanitizeResult($result);
            }
            if (!empty($key) && array_key_exists($key, $config))
            {
                cmsContextCache::set("__config__", $cx_key, $config[$key]);
                return $config[$key];
            }
            else {
                cmsContextCache::set("__config__", $cx_key, $config);
                return (empty($results_array)) ? null : $config;
            }
        }
    }

    /**
     * Delete all keys within this section
     * @param string $section
     */
    public function deleteSection($section)
    {
        global $SystemDB;
        
        $SystemDB->query("DELETE FROM {$this->table_name} WHERE `section` = :section", ['section'=> $section]);
    }
    //________________________________________________________________________//
    /**
     *  Returns multiple sections
     * @global \SCHLIX\SCHLIX\cmsDatabase $SystemDB
     * @param array $sections
     * @return array
     */
    public function getMultipleSections(array $sections) {
        global $SystemDB;

        if (!is_array($sections) || ___c($sections) == 0)
            return NULL;
        
        $sanitized_sections_array = [];
        foreach ($sections as $section)
            $sanitized_sections_array[] = sanitize_string($section);
        
        $sanitized_sections_string = implode(',', $sanitized_sections_array);
        $sql = "SELECT * FROM {$this->table_name} WHERE section IN ($sanitized_sections_string) ";
        $results_array = $SystemDB->getQueryResultArray($sql);
        
        $config = [];
        for ($i = 0, $count = ___c($results_array); $i < $count; $i++) {
            
            $section = $results_array[$i]['section'];
            $key = $results_array[$i]['key'];
            $config[$section][$key] =  $this->sanitizeResult($results_array[$i]);
        }
        return $config;

    }

    //________________________________________________________________________//
    /**
     * Prevents a key name collision in another section
     * @global \SCHLIX\SCHLIX\cmsDatabase $SystemDB
     * @param string $section
     * @param string $key
     * @param string $value
     * @return string
     */
    public function preventFaultyAliasName($section, $key, $value) {
        global $SystemDB;


        
        $key = sanitize_string($key);
        $section = sanitize_string($section);
        $str_value = sanitize_string($value);
        $sql = "SELECT * FROM {$this->table_name} WHERE section <> {$section} AND `key`={$key} AND value = {$str_value}";
        $results = $SystemDB->getQueryResultArray($sql);
        $existing_items_with_the_same_name = [];
        if (___c($results) > 0)
            foreach ($results as $result)
                $existing_items_with_the_same_name[] = $result['value'];
        if (___c($existing_items_with_the_same_name) > 0) {
            foreach ($existing_items_with_the_same_name as $name) {
                foreach ($results as $result)
                    $existing_items_with_the_same_name[] = $result['value'];
                $name = unserialize(stripslashes($value));
                $suggested = unserialize(stripslashes($value));
                $i = 1;
                while (in_array($suggested, $existing_items_with_the_same_name)) {
                    $suggested = $name . '-' . $i;
                    $i++;
                }
            }
            return convert_into_sef_friendly_title($suggested);
        } else {
            return $value;
        }
    }

    //________________________________________________________________________//
    /**
     * Set a config key. The key must have prefix 'array_','int_','string_',
     * 'str_','bool_','double_','float_'. Otherwise it won't be set
     * @global \SCHLIX\SCHLIX\cmsDatabase $SystemDB
     * @param string $section
     * @param string $key
     * @param string $value
     * @return boolean
     */
    public function set($section, $key, $value) {
        global $SystemDB;

        $datavalues = [];
        if (is_string($value))
            $value = trim($value);
        $original_value = $value; // before serialized



        $config_prefix = '';
        $prefix_pos = strpos($key, '_');
        if ($prefix_pos !== FALSE) {
            $config_prefix = substr($key, 0, $prefix_pos);
        }
        switch ($config_prefix) {
            case 'array':
                if (!is_array($value))
                    $value = [];
                $value = serialize($value);
                break;
            case 'int':
                $value = (int) $value;
                break;
            case 'dbl':
            case 'flt':                
            case 'real':
            case 'float':
            case 'double':
                $value = (double) $value;
                break;
            
            case 'bool':
                $value = ((bool) $value) ? 1: 0;
                break;
            case 'str':
            case 'string':
                $value = strval($value);
                break;
            default:
                //$value = strval($value);
                // REJECT all without prefix
                return false;
                break;
        }

        if ($key === 'str_alias' || $key === 'str_app_description') {
            if (empty($original_value)) {
                // name it as the original app name or uppercase first letter if it's a description
                $value = ($key === 'str_alias') ? $section : ucwords($section);
            }
            if ($key === 'str_alias')
                $value = $this->preventFaultyAliasName($section, $key, $value);
        }
        if (!$this->exists($section, $key)) {

            // create new data
            $datavalues['section'] = $section;
            $datavalues['key'] = $key;
            $datavalues['value'] = $value;
            $SystemDB->simpleInsertInto($this->table_name, $datavalues);
            return true;
        } else {
            // update existing data
            //$newValue = sanitize_string($value);
            
            $SystemDB->query("UPDATE {$this->table_name} SET `value` = :new_value WHERE `section`= :section AND `key`= :key", ['new_value' => $value, 'section' => $section, 'key' => $key]);

            return true;
        }
    }
    


    //________________________________________________________________________//
    /**
     * Load config file for components
     * @param string $component_name
     * @param string $item
     * @param string $item_name
     * @param string $instance_name
     * @return string
     */
    public static function loadConfigFile($component_name, $item, $item_name, $instance_name)
    {
        return \SCHLIX\cmsSkin::loadConfigFile($component_name, $item, $item_name, $instance_name);
    }
    

    /**
     * Encodes an array and return it as a string
     * @deprecated since version 2.1.9-2
     * @param array $array
     * @return string
     */
    public static function encodeArray($array)
    {
        return \SCHLIX\cmsSkin::encodeArray($array);        
    }
    
    /**
     * Decodes a base-64, JSON-encoded array
     * @deprecated since version 2.1.9-2
     * @param string $str
     * @return array
     */
    public static function decodeArray($str)
    {
        return \SCHLIX\cmsSkin::decodeArray($str);
    }

}
/**
 * Filter $_POST
 */
class cmsHttpInputFilter {

    /**
     * Return string value
     * @param array $_arr
     * @param string $varname
     * @param int $limit
     * @return string
     */
    public static function string($_arr, $varname, $trim = true, $limit = 0) {
        if (!isset($_arr) || !array_key_exists($varname, $_arr)) return '';
        $s = $_arr[$varname];
        if ($trim)
        {
            if (empty($s)) $s = '';
            $s = trim($s);
        }
        return str_limit($s, $limit);
    }

    /**
     * Return string value
     * @param array $_arr
     * @param string $varname
     * @param int $limit
     * @return string
     */
    public static function alphanumeric($_arr, $varname, $limit = 0) {
        $s = self::string($_arr, $varname, true, $limit);        
        return alpha_numeric_only($s,'');
    }    
    
    /**
     * Returns boolean value
     * @param array $_arr
     * @param string $varname
     * @return boolean
     */
    public static function bool($_arr, $varname) {
        if (!array_key_exists($varname, $_arr)) return FALSE;
        $val = $_arr[$varname];
        
        $v_int = (int) $_arr[$varname];
        if ($v_int > 0 )
            return true;
        else 
        {            
            return strtolower(strval($val)) === 'true';
        }        
    }

    /**
     * Return integer value
     * @param array $_arr
     * @param string $varname
     * @return int
     */
    public static function int($_arr, $varname) {
        if (!array_key_exists($varname, $_arr)) return 0;
        return (int) $_arr[$varname];
    }

    /**
     * Returns unsigned int
     * @param array $_arr
     * @param string $varname
     * @return int
     */
    public static function uint($_arr, $varname) {
        if (!array_key_exists($varname, $_arr)) return 0;
        $x = (int) $_arr[$varname];
        if ($x < 0)
            $x = 0;
        return $x;
    }

    /**
     * Returns float value
     * @param array $_arr
     * @param string $varname
     * @return float
     */
    public static function float($_arr, $varname) {
        if (!array_key_exists($varname, $_arr)) return 0;
        return (float) $_arr[$varname];
    }

    /**
     * Returns double value
     * @param array $_arr
     * @param string $varname
     * @return double
     */
    public static function double($_arr, $varname) {
        if (!array_key_exists($varname, $_arr)) return 0;
        return (double) $_arr[$varname];
    }

    /**
     * Returns a sanitized filename
     * @param array $_arr
     * @param string $varname
     * @return string
     */
    public static function filename_only($_arr, $varname) {
        if (!array_key_exists($varname, $_arr)) return '';
        $invalid_chars = array(">", "<", "~", "../", '"', "'", "&", "/", "\\", "?", "#");
        return str_replace($invalid_chars, '_', trim($_arr[$varname]));
    }

    /**
     * Returns a sanitized full path
     * @param array $_arr
     * @param string $varname
     * @return string
     */
    public static function filename_fullpath($_arr, $varname) {
        if (!array_key_exists($varname, $_arr)) return '';
        $invalid_chars = array(">", "<", "~", "../", '"', "'", "&", "\\", "?", "#");

        $string = str_replace($invalid_chars, '_', trim($_arr[$varname]));
        $string = preg_replace('/[\x00-\x1F\x7F-\x9F]/u', '', $string);                
        return $string;
    }

    /**
     * returns a sanitized string with no tags
     * @param array $_arr
     * @param string $varname
     * @param type $limit
     * @return string
     */
    public static function string_notags($_arr, $varname, $limit = 0) {
        if (!array_key_exists($varname, $_arr)) return '';
        return str_limit(strip_tags($_arr[$varname]),$limit);
    }
    
    /**
     * returns a sanitized string with no tags and no quotes
     * @param array $_arr
     * @param string $varname
     * @param type $limit
     * @return string
     */
    public static function string_noquotes_notags($_arr, $varname, $limit = 0) {
        if (array_key_exists($varname, $_arr))
        {
            $value = preg_replace("/[\'\")(;|`,<>]/", "", trim($_arr[$varname])); //FIX - potential bug
            return str_limit($value, $limit);            
        }
        return '';
    }
    
    /**
     * Returns array var
     * @param array $_arr
     * @param string $varname
     * @return array
     */
    public static function array_any($_arr, $varname)
    {
        return (array_key_exists($varname, $_arr) && is_array($_arr[$varname])) ?  $_arr[$varname] : NULL;
    }

    /**
     * Returns a sanitized array of integer
     * @param array $_arr
     * @param string $varname
     * @return boolean
     */
    public static function array_int($_arr, $varname)
    {
        $v_ar = isset($_arr[$varname]) ? $_arr[$varname] : null;
        if(is_array($v_ar))
        {
            foreach ($v_ar as $key => $value)
            {
                $v_ar[$key] = (int) $value;
            }         
            return $v_ar;
        }
        return NULL;
    }    
    

    /**
     * Returns a sanitized combobox date
     * @param array $_arr
     * @param string $varname
     * @return boolean
     */
    public static function date_combobox($_arr, $varname)
    {
        $v_year = (int) $_arr[$varname.'_year'];
        $v_month = (int) $_arr[$varname.'_month'];
        $v_day = (int) $_arr[$varname.'_day'];
        if ($v_year > 0 && ($v_month >= 1 && $v_month <= 12) && ($v_day >= 1 && $v_day <= 31))
        {
            $dt = $v_year.'-'.str_pad($v_month,2,'0',STR_PAD_LEFT).'-'.str_pad($v_day,2,'0',STR_PAD_LEFT);
            return is_date($dt,'Y-m-d') ? $dt : NULL;            
        }
        return NULL; 
    }    
    
}
                

/* 
 * Adapted from Doctrine DBAL QueryBuilder
 * licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 * 
 */
class cmsSQLQueryBuilder
{
    /*
     * The query types.
     */
    const SELECT = 0;
    const DELETE = 1;
    const UPDATE = 2;
    const INSERT = 3;

    /*
     * The builder states.
     */
    const STATE_DIRTY = 0;
    const STATE_CLEAN = 1;

    /**
     * The DBAL Connection.
     *
     * @var \SCHLIX\cmsDatabase
     */
    private $connection;

    /**
     * @var array The array of SQL parts collected.
     */
    private $sqlParts = array(
        'select'  => [],
        'from'    => [],
        'join'    => [],
        'set'     => [],
        'where'   => null,
        'groupBy' => [],
        'having'  => null,
        'orderBy' => [],
        'values'  => [],
    );

    /**
     * The complete SQL string for this query.
     *
     * @var string
     */
    private $sql;

    /**
     * The query parameters.
     *
     * @var array
     */
    private $params = [];

    /**
     * The parameter type map of this query.
     *
     * @var array
     */
    private $paramTypes = [];

    /**
     * The type of query this is. Can be select, update or delete.
     *
     * @var integer
     */
    private $type = self::SELECT;

    /**
     * The state of the query object. Can be dirty or clean.
     *
     * @var integer
     */
    private $state = self::STATE_CLEAN;

    /**
     * The index of the first result to retrieve.
     *
     * @var integer
     */
    private $firstResult = null;

    /**
     * The maximum number of results to retrieve.
     *
     * @var integer
     */
    private $maxResults = null;

    /**
     * The counter of bound parameters used with {@see bindValue).
     *
     * @var integer
     */
    private $boundCounter = 0;

    /**
     * Initializes a new <tt>QueryBuilder</tt>.
     *
     * @param \SCHLIX\cmsDatabase
     */
    public function __construct($connection = null)
    {
        global $SystemDB;
        
        if ($connection == NULL && $SystemDB instanceof \SCHLIX\cmsDatabase)
        {
            $this->connection = $SystemDB;
        } else $this->connection = $connection;
    } 

    /**
     * Gets the type of the currently built query.
     *
     * @return integer
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * Gets the associated DBAL Connection for this query builder.
     *
     * @return \SCHLIX\cmsDatabase
     */
    public function getConnection()
    {
        return $this->connection;
    }

    /**
     * Gets the state of this query builder instance.
     *
     * @return integer Either QueryBuilder::STATE_DIRTY or QueryBuilder::STATE_CLEAN.
     */
    public function getState()
    {
        return $this->state;
    }

    /**
     * Executes this query using the bound parameters and their types.
     *
     * @param array $data parameters in an associative array that needs to be bound
     * @return mixed
     */
    public function execute($data = NULL)
    {
        if ($this->type !== self::SELECT) {
            return $this->connection->query($this->getSQL(), $data);
        }
        return null;
    }
    
    /**
     * Returns the full SQL statement before executed
     * @param array $data
     * @return string
     */
    public function getFullSQLStatement($data = NULL)
    {        
        return $this->connection->getFullSQLStatement($this->getSQL(), $data);
    }
    /**
     * Executes this query using the bound parameters and their types.
     *
     * @param array $data parameters in an associative array that needs to be bound
     * @return mixed
     */
    public function getQueryResultArray($data = NULL)
    {
        if ($this->type === self::SELECT) {
            return $this->connection->getQueryResultArray($this->getSQL(), $data);
        }   
        return null;
    }
    
    /**
     * Returns the number of rows
     * @return int
     */
    public function getQueryResultCount($data = NULL)
    {
        if ($this->type === self::SELECT) {
            $this->select('COUNT(*) AS total_item_count');
            $result = $this->connection->getQueryResultSingleRow($this->getSQL(), $data);
            return $result['total_item_count'];
        }        
        return null;
    }

    /**
     * Executes this query using the bound parameters and their types.
     *
     * @param array $data parameters in an associative array that needs to be bound
     * @return mixed
     */
    public function getQueryResultSingleRow($data = NULL)
    {
        if ($this->type === self::SELECT) {
            return $this->connection->getQueryResultSingleRow($this->getSQL(), $data);
        } 
        return null;
        
    }

    /**
     * Gets the complete SQL string formed by the current specifications of this QueryBuilder.
     *
     * <code>
     *     $qb = $em->createQueryBuilder()
     *         ->select('u')
     *         ->from('User', 'u')
     *     echo $qb->getSQL(); // SELECT u FROM User u
     * </code>
     *
     * @return string The SQL query string.
     */
    public function getSQL()
    {
        if ($this->sql !== null && $this->state === self::STATE_CLEAN) {
            return $this->sql;
        }

        switch ($this->type) {
            case self::INSERT:
                $sql = 'INSERT INTO ' . $this->sqlParts['from']['table'] .
                ' (' . implode(', ', array_keys($this->sqlParts['values'])) . ')' .
                ' VALUES(' . implode(', ', $this->sqlParts['values']) . ')';
                if (array_key_exists('onDuplicateKeyUpdate', $this->sqlParts))
                {
                    $sql.= ' ON DUPLICATE KEY UPDATE '.implode(", ", $this->sqlParts['onDuplicateKeyUpdate']);
                }
                break;
            case self::DELETE:
                
                $table = $this->sqlParts['from']['table'] . ($this->sqlParts['from']['alias'] ? ' ' . $this->sqlParts['from']['alias'] : '');
                $sql = 'DELETE FROM ' . $table . ($this->sqlParts['where'] !== null ? ' WHERE ' . ((string) $this->sqlParts['where']) : '');

                break;

            case self::UPDATE:
                
                $table = $this->sqlParts['from']['table'] . ($this->sqlParts['from']['alias'] ? ' ' . $this->sqlParts['from']['alias'] : '');
                $sql = 'UPDATE ' . $table
                    . ' SET ' . implode(", ", $this->sqlParts['set'])
                    . ($this->sqlParts['where'] !== null ? ' WHERE ' . ((string) $this->sqlParts['where']) : '');


                
                break;

            case self::SELECT:
            default:
                $sql = 'SELECT ' . implode(', ', $this->sqlParts['select']);

                $sql .= ($this->sqlParts['from'] ? ' FROM ' . implode(', ', $this->getFromClauses()) : '')
                    . ($this->sqlParts['where'] !== null ? ' WHERE ' . ((string) $this->sqlParts['where']) : '')
                    . ($this->sqlParts['groupBy'] ? ' GROUP BY ' . implode(', ', $this->sqlParts['groupBy']) : '')
                    . ($this->sqlParts['having'] !== null ? ' HAVING ' . ((string) $this->sqlParts['having']) : '')
                    . ($this->sqlParts['orderBy'] ? ' ORDER BY ' . implode(', ', $this->sqlParts['orderBy']) : '')
                    . $this->getLimitOffset();
                
                if ($this->isLimitQuery()) {
                    /*return $this->connection->getDatabasePlatform()->modifyLimitQuery(
                        $sql,
                        $this->maxResults,
                        $this->firstResult
                    );*/
                }

                
                break;
        }

        $this->state = self::STATE_CLEAN;
        $this->sql = $sql;

        return $sql;
    }
    
    private function getLimitOffset()
    {        
        
        if (array_key_exists('limitOffset',$this->sqlParts))
        {
            $offset = (int) abs($this->sqlParts['limitOffset']['offset']);
            $limit = isset($this->sqlParts['limitOffset']['limit']) ? (int) abs($this->sqlParts['limitOffset']['limit']) : 0;
            if ($limit > 0)
            {
                return " LIMIT {$offset}, {$limit}";
            }
        }
        return "";
        
    }
    /**
     * Set the start & end
     * @param int $start
     * @param int $end
     * @param bool $force_limit
     */
    public function startEnd($start, $end, $force_limit = true)
    {
        $start = max(0, (int) $start);
        $end = (int) $end;
        $params = ['offset' => $start];
        if ($start >= 0 && $end > 0) {
            if ($start > $end)
                $start = $end = 0; // autocorrect, now guaranteed that $end >= $start
            $item_limit = $end - $start;
            if ($force_limit === true) {
                if (($item_limit === 0 && $start === 0) || ($item_limit > HARDCODE_MAX_ROWLIMIT))
                    $item_limit = HARDCODE_MAX_ROWLIMIT;
            }
            $params['limit'] = $item_limit;
        } 
        $this->add('limitOffset', $params, false);
        return $this;
    }
    
    /**
     * Set the limit & offset
     * @param int $limit
     * @param int $offset
     * @param bool $force_limit
     */
    public function limitOffset($limit, $offset = 0, $force_limit = true)
    {
        $limit = max (0, (int) $limit);
        $offset = max (0, (int) $offset);
        if ($force_limit === true && (($limit === 0 && $offset === 0) || ($limit > HARDCODE_MAX_ROWLIMIT))) 
            $limit = HARDCODE_MAX_ROWLIMIT;
        $this->add('limitOffset',['limit' => $limit, 'offset' => $offset], false);
        return $this;
    }

    /**
     * Sets the maximum number of results to retrieve (the "limit").
     *
     * @param integer $maxResults The maximum number of results to retrieve.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function setMaxResults($maxResults)
    {
        $this->state = self::STATE_DIRTY;
        $this->maxResults = $maxResults;

        return $this;
    }

    /**
     * Gets the maximum number of results the query object was set to retrieve (the "limit").
     * Returns NULL if {@link setMaxResults} was not applied to this query builder.
     *
     * @return integer The maximum number of results.
     */
    public function getMaxResults()
    {
        return $this->maxResults;
    } 
    /**
     * Either appends to or replaces a single, generic query part.
     *
     * The available parts are: 'select', 'from', 'set', 'where',
     * 'groupBy', 'having' and 'orderBy'.
     *
     * @param string  $sqlPartName
     * @param string  $sqlPart
     * @param boolean $append
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function add($sqlPartName, $sqlPart, $append = false)
    {
        $isArray = is_array($sqlPart);
        $isMultiple = array_key_exists($sqlPartName, $this->sqlParts) && is_array($this->sqlParts[$sqlPartName]);

        if ($isMultiple && !$isArray) {
            $sqlPart = array($sqlPart);
        }

        $this->state = self::STATE_DIRTY;

        if ($append) {
            if ($sqlPartName == "orderBy" || $sqlPartName == "groupBy" || $sqlPartName == "select" || $sqlPartName == "set") {
                foreach ($sqlPart as $part) {
                    $this->sqlParts[$sqlPartName][] = $part;
                }
            } elseif ($isArray && is_array($sqlPart[key($sqlPart)])) {
                $key = key($sqlPart);
                $this->sqlParts[$sqlPartName][$key][] = $sqlPart[$key];
            } elseif ($isMultiple) {
                $this->sqlParts[$sqlPartName][] = $sqlPart;
            } else {
                $this->sqlParts[$sqlPartName] = $sqlPart;
            }

            return $this;
        }

        $this->sqlParts[$sqlPartName] = $sqlPart;

        return $this;
    }

    /**
     * Specifies an item that is to be returned in the query result.
     * Replaces any previously specified selections, if any.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->select('u.id', 'p.id')
     *         ->from('users', 'u')
     *         ->leftJoin('u', 'phonenumbers', 'p', 'u.id = p.user_id');
     * </code>
     *
     * @param mixed $select The selection expressions.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function select($select = null)
    {
        $this->type = self::SELECT;
        if ($select)
        {
            $selects = is_array($select) ? $select : func_get_args();
            $this->add('select', $selects, false);
        }
        return $this;
    }


    /**
     * Turns the query being built into a bulk delete query that ranges over
     * a certain table.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->delete('users', 'u')
     *         ->where('u.id = :user_id');
     *         ->setParameter(':user_id', 1);
     * </code>
     *
     * @param string $delete The table whose rows are subject to the deletion.
     * @param string $alias  The table alias used in the constructed query.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function delete($delete = null, $alias = null)
    {
        
        $prev_type = $this->type;
        $this->type = self::DELETE;
        if ($delete)
        {
                $this->add('from', ['table' => $delete,'alias' => $alias]);

        } else
        {
            if ($prev_type === self::SELECT)
            {                
                if (is_array($this->sqlParts['from']))
                {                    
                    $this->add('from',['table' => $this->sqlParts['from'][0]['table'],'alias' =>  $this->sqlParts['from'][0]['alias']  ]);
                }
            } else throw new \Exception ('Table name must be specified for DELETE');
        }
        
        return $this;        
    }
    
    /**
     * Turns the query being built into a bulk update query that ranges over
     * a certain table
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->update('users', 'u')
     *         ->set('u.password', md5('password'))
     *         ->where('u.id = ?');
     * </code>
     *
     * @param string $update The table whose rows are subject to the update.
     * @param string $alias  The table alias used in the constructed query.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function update($update = null, $alias = null)
    {
        
        $prev_type = $this->type;
        $this->type = self::UPDATE;
        if ($update)
        {
                $this->add('from', ['table' => $update,'alias' => $alias]);

        } else
        {
            if ($prev_type === self::SELECT)
            {                
                if (is_array($this->sqlParts['from']))
                {                    
                    $this->add('from',['table' => $this->sqlParts['from'][0]['table'],'alias' =>  $this->sqlParts['from'][0]['alias']  ]);
                }
            } else throw new \Exception ('Table name must be specified for UPDATE');
        }
        
        return $this;        
    }
    /**
     * Escape array to prevent SQL injection
     * @param array $key_values
     * @return string
     */
    protected function escapeArrayForInsert(array $key_values)
    {
        $kv = [];
        foreach ($key_values as $key => $value)
           $kv[$this->field($key)] = $this->escapeValue ($value); 
        unset($key_values);
        return $kv;

    }
    /**
     * Escape array to prevent SQL injection
     * @param array $key_values
     * @return string
     */
    protected function escapeArrayForUpdate(array $key_values, $escape_value = true)
    {
        $kv = [];
        foreach ($key_values as $key => $value)
           $kv[] = $this->field($key).' = '.($escape_value ? $this->escapeValue ($value) : $value); 
        unset($key_values);
        return $kv;

    }
    /**
     * Escape string if it's a value, don't escape it if it's a variable starting
     * with ':'.
     * @param string $s
     * @return string
     */
        protected function escapeValue($s)
    {
        if (isset($s))
        {
            $s = strval($s);
            if (strlen($s) > 0)
            {
                return $s[0] === ':' ? $s : "'" . $this->connection->escapeString($s) . "'";
            } else return "''";
        } 
        return 'null';
    }

    /**
     * insertinto('table', ['id' => 'value'], true/false)
     * if automatic escape is specified, then it will automatically escape it
     * @param string $table
     * @param array $values
     * @param bool $automatic_escape
     * @return $this
     */
    public function insertInto($table = null)
    {
        $prev_type = $this->type;
        $this->type = self::INSERT;
        if ($table)
        {
            $this->add('from', ['table' => $table]);
        } else
        {
            if ($prev_type === self::SELECT)
            {                
                if (is_array($this->sqlParts['from']))
                {                    
                    $this->add('from',['table' => $this->sqlParts['from'][0]['table']]);
                }
            } else throw new \Exception ('Table name must be specified for INSERT');
        }
        return  $this;
    }
    /**
     * if automatic escape is specified, then it will automatically escape it
     * @param array $values
     * @param bool $automatic_escape
     * @return $this
     */    
    public function values(array $values, $automatic_escape = true)
    {
        if ($this->type == self::INSERT && is_array($values))
        {
            $this->add('values', $automatic_escape ? $this->escapeArrayForInsert($values) : $values);
        } else throw new \Exception('Invalid insert syntax');
        return  $this;        
    }
    
    public function onDuplicateKeyUpdate(array $values, $automatic_escape = true)
    {
        if ($this->type == self::INSERT)
        {
            $this->add('onDuplicateKeyUpdate', $this->escapeArrayForUpdate($values, $automatic_escape));
        } else throw new \Exception('Invalid insert syntax');
        return  $this;        
    }

    /**
     * Creates and adds a query root corresponding to the table identified by the
     * given alias, forming a cartesian product with any existing query roots.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->select('u.id')
     *         ->from('users', 'u')
     * </code>
     *
     * @param string      $from  The table.
     * @param string|null $alias The alias of the table.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function from($from, $alias = null)
    {
        
        return $this->add('from', ['table' => $from,'alias' =>  $alias], true);
    } 

    /**
     * Creates and adds a join to the query.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->select('u.name')
     *         ->from('users', 'u')
     *         ->innerJoin('u', 'phonenumbers', 'p', 'p.is_primary = 1');
     * </code>
     *
     * @param string $fromAlias The alias that points to a from clause.
     * @param string $join      The table name to join.
     * @param string $alias     The alias of the join table.
     * @param string $condition The condition for the join.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function innerJoin($fromAlias, $join, $alias, $condition = null)
    {
        return $this->add('join', array(
            $fromAlias => array(
                'joinType'      => 'inner',
                'joinTable'     => $join,
                'joinAlias'     => $alias,
                'joinCondition' => $this->composite($condition)
            )
        ), true);
    }

    /**
     * Creates and adds a left join to the query.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->select('u.name')
     *         ->from('users', 'u')
     *         ->leftJoin('u', 'phonenumbers', 'p', 'p.is_primary = 1');
     * </code>
     *
     * @param string $fromAlias The alias that points to a from clause.
     * @param string $join      The table name to join.
     * @param string $alias     The alias of the join table.
     * @param string $condition The condition for the join.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function leftJoin($fromAlias, $join, $alias, $condition = null)
    {
        return $this->add('join', array(
            $fromAlias => array(
                'joinType'      => 'left',
                'joinTable'     => $join,
                'joinAlias'     => $alias,
                'joinCondition' => $this->composite($condition)
            )
        ), true);
    }

    /**
     * Creates and adds a right join to the query.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->select('u.name')
     *         ->from('users', 'u')
     *         ->rightJoin('u', 'phonenumbers', 'p', 'p.is_primary = 1');
     * </code>
     *
     * @param string $fromAlias The alias that points to a from clause.
     * @param string $join      The table name to join.
     * @param string $alias     The alias of the join table.
     * @param string $condition The condition for the join.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function rightJoin($fromAlias, $join, $alias, $condition = null)
    {
        return $this->add('join', array(
            $fromAlias => array(
                'joinType'      => 'right',
                'joinTable'     => $join,
                'joinAlias'     => $alias,
                'joinCondition' => $this->composite($condition)
            )
        ), true);
    }

    /**
     * Sets a new value for a column in a bulk update query.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->update('users', 'u')
     *         ->set(['u.password', md5('password')])
     *         ->where('u.id = ?');
     * </code>
     *
     * @param array $key_values   The column to set.
     * @param bool $automatic_escape automatically escape (default = true).
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    function set(array $key_values, $automatic_escape = true)
    {
        if (is_array($key_values))
        {
            
            $kv = $this->escapeArrayForUpdate($key_values, $automatic_escape);
            return $this->add('set', implode(', ', $kv), true);
        } else throw  new \Exception ('Invalid key/value array');
    }    


    /**
     * Evaluate where condition
     * @param string|array $stmt
     * @return string
     * @throws \Exception
     */
    private function composite($stmt)
    {
        $cond = '';
        if ($stmt === null)
            return '';
        if (is_string($stmt))
            $cond = '('.$stmt.')';
        else if (is_array($stmt))
        {
            $ar_k = array_keys ($stmt);
            $count = ___c($ar_k);
            if ($count % 2 != 1)
            {
                //print_r($stmt);
                throw new \Exception('Invalid composite SQL condition specified');
            }
            $all_where = [];
            for ($i =0; $i < $count; $i ++ )
            {
                $cur_stmt = $stmt[$i];
                // if it's a statement (must be even or 0)
                if (($i+1) % 2 == 1)
                {
                    if (is_string($cur_stmt))
                        $all_where[] = $cur_stmt;
                    elseif (is_array($cur_stmt))
                        $all_where[] = $this->composite($cur_stmt);
                    else
                        throw new \Exception ('Condition must be either array or string');
                } else // if it's OR/AND (must be odd)
                $all_where[] = $cur_stmt;

            }
            $cond = '('.implode(' ', $all_where).')';
        } else if (is_numeric ($stmt))
            return '('.$stmt.')';
        return $cond;
    }
    
    /**
     * Specifies one or more restrictions to the query result.
     * Replaces any previously specified restrictions, if any.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->select('u.name')
     *         ->from('users', 'u')
     *         ->where('u.id = ?');
      *
     *     $qb->update('users', 'u')
     *         ->set('u.password', md5('password'))
     *         ->where(['x = 1', 'AND', 
            ['id = :id','AND','x = 1', 'AND','x = z'],
            'OR', 'x = 1', 'AND',['x = 1','AND','z = 5']
        ]);
     * </code>
     *
     * @param mixed $predicates The restriction predicates.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function where($predicates)
    {
        $to_add = $this->composite($predicates);
        if ($this->sqlParts['where'])
            $to_add = "{$this->sqlParts['where']} AND {$to_add}";
        return $this->add('where', $to_add, true);
    }
            
    /**
     * Specifies a grouping over the results of the query.
     * Replaces any previously specified groupings, if any.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->select('u.name')
     *         ->from('users', 'u')
     *         ->groupBy('u.id');
     * </code>
     *
     * @param mixed $groupBy The grouping expression.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function groupBy($groupBy)
    {
        if ($groupBy)
        {
            $g = is_array($groupBy) ? $groupBy : func_get_args();
            $this->add('groupBy', $g, false);
        }
        return $this;
    } 
    /**
     * Sets a value for a column in an insert query.
     *
     * <code>
     *     $qb = $conn->createQueryBuilder()
     *         ->insert('users')
     *         ->values(
     *             array(
     *                 'name' => '?'
     *             )
     *         )
     *         ->setValue('password', '?');
     * </code>
     *
     * @param string $column The column into which the value should be inserted.
     * @param string $value  The value that should be inserted into the column.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function setValue($column, $value)
    {
        $this->sqlParts['values'][$column] = $value;

        return $this;
    }

    /**
     * Specifies a restriction over the groups of the query.
     * Replaces any previous having restrictions, if any.
     *
     * @param mixed $having The restriction over the groups.
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function having($having)
    {
        return $this->add('having', $this->composite($having));
    } 
    
    public function field($fieldname)
    {
        return "`{$fieldname}`";
    }

    /**
     * Specifies an ordering for the query results.
     * Parameter must be in the form of key => sort direction.
     * For example: orderBy('id' => 'DESC', 'name' => 'ASC')
     * 
     * @param array $opt
     * @return $this
     */
    public function orderBy(array $opt)
    {
        foreach ($opt as $key => $sort_dir)
        {
            $sort_dir_fixed = strtoupper($sort_dir) == 'DESC' ? 'DESC' : 'ASC';              
            $this->add('orderBy', $this->field($key).' '.$sort_dir_fixed, true);
        } 
        return $this;
        //return $this->add('orderBy', $sort . ' ' . (! $order ? 'ASC' : $order), false);
    } 
    
    /**
     * Gets a query part by its name.
     *
     * @param string $queryPartName
     *
     * @return mixed
     */
    public function getPart($queryPartName)
    {
        return $this->sqlParts[$queryPartName];
    }

    /**
     * Gets all query parts.
     *
     * @return array
     */
    public function getParts()
    {
        return $this->sqlParts;
    }

    /**
     * Resets SQL parts.
     *
     * @param array|null $queryPartNames
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function reset($queryPartNames = null)
    {
        if (is_null($queryPartNames))
            $queryPartNames = array_keys($this->sqlParts);
        foreach ($queryPartNames as $queryPartName)
            $this->resetQueryPart($queryPartName);
        return $this;
    }

    /**
     * Resets a single SQL part.
     *
     * @param string $queryPartName
     *
     * @return $this \SCHLIX\cmsSQLQueryBuilder instance.
     */
    public function resetQueryPart($queryPartName)
    {
        $this->sqlParts[$queryPartName] = is_array($this->sqlParts[$queryPartName])
            ? [] : null;

        $this->state = self::STATE_DIRTY;

        return $this;
    }
            

    /**
     * @return string[]
     */
    private function getFromClauses()
    {
        $fromClauses = array();
        $knownAliases = array();

        // Loop through all FROM clauses
        foreach ($this->sqlParts['from'] as $from) {
            if ($from['alias'] === null) {
                $tableSql = $from['table'];
                $tableReference = $from['table'];
            } else {
                $tableSql = $from['table'] . ' ' . $from['alias'];
                $tableReference = $from['alias'];
            }

            $knownAliases[$tableReference] = true;

            $fromClauses[$tableReference] = $tableSql . $this->getSQLForJoins($tableReference, $knownAliases);
        }

        $this->verifyAllAliasesAreKnown($knownAliases);

        return $fromClauses;
    }

    /**
     * @param array $knownAliases
     *
     * @throws QueryException
     */
    private function verifyAllAliasesAreKnown(array $knownAliases)
    {
        foreach ($this->sqlParts['join'] as $fromAlias => $joins) {
            if ( ! isset($knownAliases[$fromAlias])) {
                throw new \Exception('Unknown alias '.$fromAlias); //QueryException::unknownAlias($fromAlias, array_keys($knownAliases));
            }
        }
    }


    /**
     * Returns true if array is associative, false if it's sequential
     * @param array $arr
     * @return boolean
     */
    private function isAssociativeArray(array $arr)
    {
        if ([] === $arr) return false;
        return array_keys($arr) !== range(0, ___c($arr) - 1);
    }
    /**
     * @return bool
     */
    private function isLimitQuery()
    {
        return $this->maxResults !== null || $this->firstResult !== null;
    }
     /**
     * Gets a string representation of this QueryBuilder which corresponds to
     * the final SQL query being constructed.
     *
     * @return string The string representation of this QueryBuilder.
     */
    public function __toString()
    {
        return $this->getSQL();
    }
            

    /**
     * @param string $fromAlias
     * @param array  $knownAliases
     *
     * @return string
     */
    private function getSQLForJoins($fromAlias, array &$knownAliases)
    {
        $sql = '';

        if (isset($this->sqlParts['join'][$fromAlias])) {
            foreach ($this->sqlParts['join'][$fromAlias] as $join) {
                if (array_key_exists($join['joinAlias'], $knownAliases)) {
                    //throw QueryException::nonUniqueAlias($join['joinAlias'], array_keys($knownAliases));
                    throw new \Exception("Non-unique alias ".$join['joinAlias']. implode(', ', array_keys($knownAliases)));// QueryException::nonUniqueAlias($join['joinAlias'], array_keys($knownAliases));
                }
                $sql .= ' ' . strtoupper($join['joinType'])
                    . ' JOIN ' . $join['joinTable'] . ' ' . $join['joinAlias']
                    . ' ON ' . ((string) $join['joinCondition']);
                $knownAliases[$join['joinAlias']] = true;
            }

            foreach ($this->sqlParts['join'][$fromAlias] as $join) {
                $sql .= $this->getSQLForJoins($join['joinAlias'], $knownAliases);
            }
        }

        return $sql;
    }

}
            
