<?php
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
//
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//


/**
 * Add slashes for single quote and back slash. Used for writing variables in a single quote to config files
 * @param string $str
 * @return string
 */
function add_single_quote_slashes($str)
{
    return addcslashes($str, "'\\");
}

/**
 * Get maximum login time
 * @return int
 */
function get_schlix_max_user_session_time()
{
    return defined('SCHLIX_SESSION_LOGIN_TIME') && is_integer(SCHLIX_SESSION_LOGIN_TIME) ? SCHLIX_SESSION_LOGIN_TIME : (60 * 60 * 24);    
}

/**
 * Get maximum remember me cookie time in seconds
 * @return int
 */
function get_schlix_max_user_remember_cookie_time()
{
    return defined('SCHLIX_SESSION_REMEMBER_COOKIE_TIME') && is_integer(SCHLIX_SESSION_REMEMBER_COOKIE_TIME) ? SCHLIX_SESSION_REMEMBER_COOKIE_TIME : (60 * 60 * 24);    
}

/**
 * Returns true if current browser is samesite compatible
 */
function user_current_browser_samesite_none_compatible ()
{
    global $__schlix_same_site_cookie_set, $__schlix_same_site_cookie_mode;
    
    if ($__schlix_same_site_cookie_set === null)
    {
        $__schlix_same_site_cookie_mode  = user_agent_samesite_none_explicit(get_user_browser_string());

        $__schlix_same_site_cookie_set = true;
    } 
    return $__schlix_same_site_cookie_mode;


}

/**
 * Returns current HTTP_HOST variable without port number
 * @return string
 */
function schlix_get_current_host_no_port()
{
    $domain = $_SERVER['HTTP_HOST'];
    $host = str_contains($domain, ':')  ? substr($domain, 0, strpos($domain.":", ":")) : $domain;
    return $host;
}
/**
 * 
 * @param int $lifetime
 * @param string $path
 * @param string $domain
 * @param bool $secure
 * @param bool $httponly
 * @param string $samesite
 */
function schlix_session_set_cookie_params ($lifetime, $path, $domain, $secure = FALSE, $httponly = FALSE , $samesite = 'none') 
{    
    $samesite = $samesite ? strtolower($samesite) : '';    
    $ua_same_site_ok = user_current_browser_samesite_none_compatible();
    if (!in_array($samesite, ['lax','none','strict']))
    {
        $samesite = $secure ? 'none' : '';
    }
    if ($samesite === 'none' && !$secure)
    {
        $samesite = $ua_same_site_ok ? 'lax' : '';
    }    
    if(PHP_VERSION_ID < 70300) {
        $path = $ua_same_site_ok ? "{$path}; samesite={$samesite}" : $path ;
        session_set_cookie_params($lifetime, $path, $domain, $secure, $httponly);
    } else {
        $opt = ['lifetime' => $lifetime, 'path' => $path, 'domain' => $domain, 'secure' => $secure, 'httponly' => $httponly];            
        if ($ua_same_site_ok && !empty($samesite))
            $opt['samesite'] =  $samesite;

        session_set_cookie_params($opt);
    }        
}

function schlix_simple_set_cookie($name, $value, $expiry = 0)
{
    $path = SCHLIX_SITE_HTTPBASE.'/';
    $domain = schlix_get_current_host_no_port();
    $secure = isCurrentRequestSSL();
    $httponly = true;
    $samesite = strtolower(defined('SCHLIX_COOKIE_SAMESITE') ? SCHLIX_COOKIE_SAMESITE : ($secure ? 'none' : '')); // default to none
    schlix_set_cookie($name, $value, $expiry, $path, $domain, $secure, $httponly, $samesite);
}


function schlix_set_cookie ($name, $value, $expires = 0, $path = "", $domain = "", $secure = FALSE, $httponly = FALSE, $samesite = 'none' ) 
{
    $samesite = $samesite ? strtolower($samesite) : '';    
    $ua_same_site_ok = user_current_browser_samesite_none_compatible();
    if (!in_array($samesite, ['lax','none','strict']))
    {
        $samesite = $secure ? 'none' : '';
    }
    if ($samesite === 'none' && !$secure)
    {
        $samesite = $ua_same_site_ok ? 'lax' : '';
    }    
    if(PHP_VERSION_ID < 70300) {
        $path = $ua_same_site_ok ? "{$path}; samesite={$samesite}" : $path ;
        setcookie($name, $value, $expires, $path, $domain, $secure, $httponly);
    } else {
        $opt = ['expires' => $expires, 'path' => $path, 'domain' => $domain, 'secure' => $secure, 'httponly' => $httponly];            
        if ($ua_same_site_ok && !empty($samesite))
            $opt['samesite'] =  $samesite;

        setcookie($name, $value, $opt);
    }        
    
}

/**
 * Returns true if browser is compatible with the SameSite=None as per IETF 2019 not 2016
 * @param string $ua
 * @return boolean
 */
function user_agent_samesite_none_explicit($ua)
{

    /*$browsers = 
    [
            // note - https://getupdraft.com/blog/ipados-breaking-changes-developers 
            // https://forums.developer.apple.com/thread/119186 
            'Mozilla/5.0 (iPad; CPU OS 13_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0 Mobile/15E148 Safari/604.1',
            'Mozilla/5.0 (iPad; CPU OS 13_2_3 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0.3 Mobile/15E148 Safari/604.1', 
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0.1 Safari/605.1.15',
            'Mozilla/5.0 (iPod; CPU iPhone OS 12_0 like macOS) AppleWebKit/602.1.50 (KHTML, like Gecko) Version/12.0 Mobile/14A5335d Safari/602.1.50',
            'Mozilla/5.0 (iPhone; CPU iPhone OS 13_0 like Mac OS X) AppleWebKit/602.1.38 (KHTML, like Gecko) Version/10.0 Mobile/14A41DS1497c Safari/602.1',
            'Mozilla/5.0 (iPhone; CPU iPhone OS 13_3_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0.5 Mobile/15E148 Safari/604.1',
            'Mozilla/5.0 (iPad; CPU OS 10_3_3 like Mac OS X) AppleWebKit/603.3.8 (KHTML, like Gecko) Version/10.0 Mobile/14G60 Safari/602.1',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0.1 Safari/605.1.15',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0 Safari/605.1.15',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_14_5) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/12.1.1 Safari/605.1.15',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0 Safari/605.1.15',

    ];    
*/        
    // Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/56.14.2924.87 Safari/537.36
    $result = false;
    $chrome_ver = null;
    $matches = null;
   // $chrome_based = preg_match_all('/(Chrome|Chromium)\/(\d+)/', $ua, $cm);
    preg_match_all('/(Chrome|Chromium)\/(\d+)/', $ua, $chrome_ver);
    if ($chrome_ver[2])
    {   
        $chrome_major_version = (int) $chrome_ver[2][0];
        preg_match_all('/UCBrowser\/(\d+\.\d+\.\d+)/', $ua, $matches);
        if ($matches[1])
            $result = version_compare($matches[1][0], '12.13.2', '>=');
        else
        {
            $result = $chrome_major_version > 66 || $chrome_major_version < 51;
        }
    } else
    {   
        // OS X Safari
        //if (preg_match('/Version\/(.*) Safari\//', $ua))
        if (str_contains('Safari/', $ua))
        {
            preg_match_all('/Mozilla\/5.0 \((Macintosh|iPad|iPod|iPhone); ((Intel Mac OS X 10_(\d+)[_\d+]*.*)|(CPU (iPhone OS|OS) (\d+)_(\d).*))\) AppleWebKit\/(.*) \(KHTML, like Gecko\) Version\/(.*) Safari\//', $ua, $matches);
            if ($matches[1])
            {
                $platform = $matches[1][0];
                if ($platform == 'Macintosh')
                {
                    //$osx_minor = $matches[4][0];                    
                    $result = version_compare($matches[10][0],  '13.1', '>=');
                } else 
                {
                    $ios_version = (double) "{$matches[7][0]}.{$matches[8][0]}";
                    $result = $ios_version >= 13.5; //  version_compare($ios_version, '13.5', '>=');

                }
            }
        } else 
        {

            $result = true;
        }
    }

    return $result; 
}

/**
 * Generate random token
 * @param int $length
 * @return string
 */
function get_random_token($length = 16){
    if (function_exists('random_bytes'))  $s = random_bytes($length); 
    else
    if (function_exists('mcrypt_create_iv')) $s = mcrypt_create_iv($length, MCRYPT_DEV_URANDOM);
    else
    if (function_exists('openssl_random_pseudo_bytes'))  $s = openssl_random_pseudo_bytes($length);
    
    return substr(strtr(base64_encode($s), '+', '.'), 0, $length);
    
}

function generate_salt($hash_type, $work_factor = 9) {
    $work_factor = (int) $work_factor;
    if ($work_factor < 4 || $work_factor > 31)
        $work_factor = 9;
    $rand = function_exists('openssl_random_pseudo_bytes') ? openssl_random_pseudo_bytes(16) : random_bytes($work_factor);
    $silly_string = substr(strtr(base64_encode($rand), '+', '.'), 0, 22);
    
    $wfstr6 = (int) $work_factor * 10000;
    $wfstr2 = (string) $work_factor;
    switch ($hash_type) {
        case 'sha256': $salt = '$5$rounds=' . str_pad($wfstr6, 6, '0', STR_PAD_LEFT) . '$' . substr($silly_string, 0, 16);
            break;
        case 'sha512': $salt = '$6$rounds=' . str_pad($wfstr6, 6, '0', STR_PAD_LEFT) . '$' . substr($silly_string, 0, 16);
            break;
        case 'blowfish':
        default: $salt = '$2y$' . str_pad($wfstr2, 2, '0', STR_PAD_LEFT) . '$' . $silly_string;
            break;
    }
    return $salt;
}

function get_skin_module($skin_name)
{
    $skin_module = "\\Skin\\{$skin_name}";
    $skin = null;
    if (class_exists($skin_module))
        $skin = new $skin_module(null, null);
    return $skin;
}

/**
 * Get current skin module
 * @return \SCHLIX\cmsSkin
 */
function get_current_skin_module()
{
    $skin_name = defined('SCHLIX_THEME_SKIN') ? SCHLIX_THEME_SKIN : 'bootstrap3';
    
    return get_skin_module($skin_name);
}

/**
 * Skin HTML
 * @param string $html
 * @param array $vars
 * @param array $allowed_namespaces
 * @return string
 */
function skin_html($html, $vars, $allowed_namespaces = ['x-ui'])
{    

    $skin_name = defined('SCHLIX_THEME_SKIN') ? SCHLIX_THEME_SKIN : 'bootstrap3';
    $skin_module = "\\Skin\\{$skin_name}";
    $skin = null;
    if (class_exists($skin_module))
    {        
        $skin = new $skin_module($allowed_namespaces, $vars);
        if (SCHLIX_DEFAULT_LANGUAGE !== 'en_us')
        {
            $lang_file = SCHLIX_ROOT_PATH.'/system/skins/'.$skin_name.'/languages/'. SCHLIX_DEFAULT_LANGUAGE . '.lang.php';
            if (file_exists($lang_file)) require $lang_file;
        }
        
    }
    return $skin->Run($html);    
}


/**
 * Returns a list of available skin
 * @return array
 */
function get_all_skin_modules()
{

    $result = [];
    $parent = SCHLIX_SYSTEM_PATH.'/skins';
    $handle = opendir($parent);
    if ($handle) 
    {        
        while (false !== ($entry = readdir($handle))) {
            if (!str_starts_with($entry, '.'))
            {
                $class = $parent.'/'.$entry.'/'.$entry.'.class.php';
                $full_fn =  realpath($class);
                if (!empty($full_fn) && file_exists($full_fn))
                    $result[] = $entry;
            }
        }

        closedir($handle);
    }    
    return $result;
}
/**
 * Returns true if a variable == true or > 0
 * @param string $string
 * @return boolean
 */
function is_value_true ($string)
{
    if (!empty($string))
    {
        return (int) ($string) > 0 || (strtolower($string) === "true");
    }
    return false;
}
    
/**
 * Fix self closing XHTML tags
 * @param string $html
 * @return string
 */
function fix_xhtml($html)
{
    preg_match_all( '#<(area|base|br|col|command|embed|hr|img|input|keygen|link|menuitem|meta|param|source|track|wbr)(\b[^>]*)>#', $html, $matches);

    $ttr = $matches[1];

    if (___c($ttr) > 0)
    {
        $ttr = array_unique($ttr);
        foreach ($ttr as $t)
            $html = str_replace("></{$t}>",' />', $html);
    }        
    return $html;
}

/**
 * Get readable file permissions
 * @param string $path
 */
function get_file_permissions($path)
{
    $info = null;
    if (file_exists($path))
    {
        $perms = fileperms($path);
        switch ($perms & 61440) {
            case 49152: $info = 's'; break; // socket
            case 40960: $info = 'l'; break; // symbolic link                
            case 32768: $info = 'r'; break; // regular                
            case 24576: $info = 'b'; break; // block special                
            case 16384: $info = 'd'; break; // directory                
            case 8192: $info = 'c'; break; // character special                
            case 4096: $info = 'p'; break; // FIFO pipe                
            default: $info = 'u'; // unknown                
        }
        // Owner
        $info .= (($perms & 256) ? 'r' : '-');
        $info .= (($perms & 128) ? 'w' : '-');
        $info .= (($perms & 64) ?
                    (($perms & 2048) ? 's' : 'x' ) :
                    (($perms & 2048) ? 'S' : '-'));

        // Group
        $info .= (($perms & 32) ? 'r' : '-');
        $info .= (($perms & 16) ? 'w' : '-');
        $info .= (($perms & 8) ?
                    (($perms & 1024) ? 's' : 'x' ) :
                    (($perms & 1024) ? 'S' : '-'));

        // World
        $info .= (($perms & 4) ? 'r' : '-');
        $info .= (($perms & 2) ? 'w' : '-');
        $info .= (($perms & 1) ?
                    (($perms & 512) ? 't' : 'x' ) :
                    (($perms & 512) ? 'T' : '-'));
        
    }
    return $info;
}

/**
 * Return URL encode path
 * @param string $path
 */
function ___u($path)
{
    return empty($path) ? '' : implode("/", array_map("rawurlencode", explode("/", $path)));
}

/**
 * Return the count of an array without emitting a PHP warning (PHP 7.2 compat)
 * @param array|\Countable $object
 * @return int
 */
function ___c($object)
{
    return (is_array($object) || ($object instanceof \Countable)) ? count ($object) : ($object instanceof \Traversable ? iterator_count($object)  : NULL );
}

/**
 * Raw URL encode a path
 * @param string $url
 * @return string
 */
function encode_path($url)
{
    $part = parse_url($url);
    //explode(',',$part['path']);
    return join('/', array_map('rawurlencode', explode('/', $part['path'])));
    
}


/**
 * Raw URL decode a path
 * @param string $url
 * @return string
 */
function decode_path($url)
{
    $part = parse_url($url);
    //explode(',',$part['path']);
    return join('/', array_map('rawurldecode', explode('/', $part['path'])));
    
}

/**
 * Get remote file content with CURL
 * @param string $url
 * @param bool $use_default_user_agent 
 * @param bool $ignore_ssl_ca
 * @return string
 */
function get_remote_file_content($url, $use_default_user_agent = true, $ignore_ssl_ca = false){
    if (function_exists('curl_init'))
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_AUTOREFERER, TRUE);
        curl_setopt($ch, CURLOPT_HEADER, 0);
        if ($use_default_user_agent)
            curl_setopt($ch, CURLOPT_USERAGENT, 'SCHLIX/1.0 '.$_SERVER['SERVER_NAME'] );
        // For Windows only - dev server 
        // note: do not use SCHLIX_SYSTEM_MSFTWIN here in case it's used during install
        if (str_starts_with($url, 'https'))
        {
            if ($ignore_ssl_ca)
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            else
                curl_setopt($ch, CURLOPT_CAINFO, SCHLIX_DEFAULT_CA_BUNDLE);
        }
        
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_BUFFERSIZE, 64000);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, TRUE);       
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT , 10); 
        $data = curl_exec($ch);
        $err = curl_error($ch);
        $retcode = curl_getinfo($ch, CURLINFO_HTTP_CODE); 
        curl_close($ch);
        if ($err)
            throw new Exception(sprintf(___('Error while downloading file: %s'), $err));        
        if ($retcode != 200)
            throw new Exception(sprintf(___('Unable to download the requested file. HTTP Error Code: %d'), $retcode));
    } else
    {
        $context_opts =[];
        $ssl_self_signed = [
                'allow_self_signed' => true,
                'verify_peer' => false,
                'verify_peer_name' => false,
            ];
        $ssl_strict = [
                'allow_self_signed' => true,
                'verify_peer'=>  true,
                'verify_peer_name' => true,
                'cafile' => SCHLIX_DEFAULT_CA_BUNDLE,
                'verify_depth' => 6
            ];
        if (!ini_get('allow_url_fopen'))
            @ini_set('allow_url_fopen', 1);
        if (str_starts_with($url, 'https'))
        {        
            $context_opts['ssl'] = $ignore_ssl_ca ? $ssl_self_signed : $ssl_strict;
            $data = @file_get_contents($url, false, stream_context_create($context_opts));
        } else $data = @file_get_contents($url);
    }
    return $data;
}

/**
 * Start output buffering
 */
function start_output_buffer($clean_first = false)
{
    if ($clean_first && ob_get_length() > 0)
        ob_end_clean();
    ob_start();   
}

/**
 * End output buffering and return string
 * @return string
 */
function end_output_buffer()
{
    $text = ob_get_contents();
    ob_end_clean();
    return $text;
}

/**
 * Returns a relative request path without preceding directory
 * @return string
 */
function get_relative_url_request_path()
{    
    
    $request_uri = parse_url( $_SERVER['REQUEST_URI'], PHP_URL_PATH);
    
    if (SCHLIX_SITE_HTTPBASE) {
        $x = strpos($request_uri, SCHLIX_SITE_HTTPBASE);
        if ($x !== false && $x == 0) {
            $z = strlen(SCHLIX_SITE_HTTPBASE);
            $s = substr($request_uri, $z, strlen($request_uri) - $z);
        }
    } else
        $s = $request_uri;
    return $s;
}
/**
 * String output sanitizing function to ensure there's no XSS in the text.
 * If maxlen is specified then it will be limited to maxlen
 * @param string $s
 * @param int $maxlen
 * @return string
 */
function ___h($s, $maxlen = 0) {
    if ($maxlen > 0 && strlen($s) > $maxlen) {
        $s = substr($s, 0, $maxlen);
    }
    // htmlspecialchars($s, ENT_QUOTES);    
    return htmlentities($s, ENT_QUOTES, "UTF-8"); 
}
/**
 * Alias of ___h/ Deprecated for an easier to remember function name. Use ___h. 
 * String output sanitizing function to ensure there's no XSS in the text.
 * If maxlen is specified then it will be limited to maxlen.
 * To be removed in v2.3.x-x
 * @deprecated since version 2.1.6-2
 * @param string $s
 * @param int $maxlen
 * @return string
 */
function SAFE_HTML($s, $maxlen = 0) {
    return ___h($s, $maxlen);
}

/**
 * Format date.
 * @param string $date the date, e.g. 2020-05-01 01:02:03
 * @param string $date_format SCHLIX_DATE_FORMAT_READABLE, SCHLIX_DATE_FORMAT_YMDHI, SCHLIX_FORMAT_MDY, SCHLIX_FORMAT_MDYD, SCHLIX_FORMAT_TIME
 * @return string
 */
function date_custom_format($date, $date_format = 'Y-M-d @ H:i' )
{
    if (empty($date))
        return null;
    $d = new \DateTime($date);
    return $d->format($date_format);
}      

/**
 * http://stackoverflow.com/questions/18685/how-to-display-12-minutes-ago-etc-in-a-php-webpage
 * @param int $time_since
 * @return string
 */
function time_since($time_since) {
    $time = time() - $time_since;
    $tokens = array(
      31536000 => ___('year'),
      2592000 => ___('month'),
      604800 => ___('week'),
      86400 => ___('day'),
      3600 => ___('hour'),
      60 => ___('minute'),
      1 => ___('second')
    );
    $extra_s = (SCHLIX_DEFAULT_LANGUAGE !== 'en_us' && SCHLIX_DEFAULT_LANGUAGE !== 'en-CA' && SCHLIX_DEFAULT_LANGUAGE !== 'en-GB') ? '' : 's';
    foreach ($tokens as $unit => $text) {
        if ($time < $unit)
            continue;
        $numberOfUnits = floor($time / $unit);
        return $numberOfUnits . ' ' . $text . (($numberOfUnits > 1) ? $extra_s : '');
    }
}

/**
 * Returns true if haystack starts with needle. Added if for future PHP 8.0 compatibility.
 * @param string $haystack
 * @param string $needle
 * @return boolean
 */
if (!function_exists('str_starts_with'))
{

function str_starts_with($haystack, $needle) {
    return (0 === strpos($haystack, $needle));
}

}
/**
 * Returns true if haystack ends with needle
 * Added if for future PHP 8.0 compatibility
 * @param string $haystack
 * @param string $needle
 * @return boolean
 */
if (!function_exists('str_ends_with'))
{

function str_ends_with($haystack, $needle)
{
    $length = strlen($needle);
    return ($length == 0) ? true :  (strrpos($haystack, $needle) == strlen($haystack) - $length);
}

}
/**
 * Returns true if haystack contains needle
 * Added if for future PHP 8.0 compatibility
 * @param string $haystack
 * @param string $needle
 * @return boolean
 */
if (!function_exists('str_contains'))
{
function str_contains($haystack, $needle) {    
    return (empty($haystack) || empty($needle)) ? FALSE : strpos($haystack, $needle) !== FALSE;
}
}

/**
 * Returns string value of $str and if the length is greater than maxlen
 * then it will be trimmed
 * @param string $str
 * @param int $maxlen
 * @return string
 */
function str_limit($str, $maxlen)
{
    $s = strval($str);
    if ($maxlen > 0 && strlen($s) > $maxlen) {
        $s = substr($s, 0, $maxlen);
    }
    return $s;
}

//_______________________________________________________________________________________________________________//
function get_class_ancestors($class) {
    if (is_object($class)) {
        $classes = [];
        $classes[] = get_class($class);
        while ($class = get_parent_class($class)) {
            $classes[] = $class;
        }
        return $classes;
    } else {
        // echo 'Invalid '.$class;
        return [];
    }
}

//_______________________________________________________________________________________________________________//
/**
 * Returns a user's real IP address. Probes for proxy if the CMS is behind
 * a load balancer.
 * @return string
 */
function get_user_real_ip_address() {
    if (isset($_SERVER["HTTP_CF_CONNECTING_IP"]))
        return $_SERVER['HTTP_CF_CONNECTING_IP'];
    if (isset($_SERVER['HTTP_CLIENT_IP']))
        $ip = $_SERVER['HTTP_CLIENT_IP'];
    elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR']))
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
    else
        $ip = $_SERVER['REMOTE_ADDR'];
    

    if (strstr($ip, ',')) {
        $ips = explode(',', $ip);
        $ip = trim($ips[0]);
    }    
    return $ip;
}

/**
 * Returns the user's browser string limited to 255 chars
 * @return string
 */
function get_user_browser_string()
{
    $ua = isset($_SERVER['HTTP_USER_AGENT']) ? (string) $_SERVER['HTTP_USER_AGENT'] : '';
    return str_limit(strip_tags($_SERVER['HTTP_USER_AGENT']), 255);
}


/**
 * Returns an array containing platform, browser, and version.
 * The array will have 3 keys: platform, browser, version
 * Adapted from https://donatstudios.com/PHP-Parser-HTTP_USER_AGENT
 * @param string $ua
 * @return null|array
 */
function get_user_browser_info( $ua = null ) {
        $windows_nt_version = [
            '5.0' => '2000',
            '5.1' => 'XP (32-bit)',
            '5.2' => 'XP',
            '6.0' => 'Vista',
            '6.1' => '7',
            '6.2' => '8',
            '6.3' => '8.1',
            '10.0' => '10 or 11'
        ];


        $bots = [ 'Googlebot', 'Baiduspider', 'ia_archiver', 'R6_FeedFetcher', 'NetcraftSurveyAgent', 'Sogou web spider', 'bingbot', 'Yahoo! Slurp', 'facebookexternalhit', 'PrintfulBot', 'msnbot', 'Twitterbot', 'UnwindFetchor', 'urlresolver', 'Butterfly', 'TweetmemeBot', 'PaperLiBot', 'MJ12bot', 'AhrefsBot', 'Exabot', 'Ezooms', 'YandexBot', 'SearchmetricsBot', 'picsearch', 'TweetedTimes Bot', 'QuerySeekerSpider', 'ShowyouBot', 'woriobot', 'merlinkbot', 'BazQuxBot', 'Kraken', 'SISTRIX Crawler', 'R6_CommentReader', 'magpie-crawler', 'GrapeshotCrawler', 'PercolateCrawler', 'MaxPointCrawler', 'R6_FeedFetcher', 'NetSeer crawler', 'grokkit-crawler', 'SMXCrawler', 'PulseCrawler', 'Y!J-BRW', '80legs.com/webcrawler', 'Mediapartners-Google', 'Spinn3r', 'InAGist', 'Python-urllib', 'NING', 'TencentTraveler', 'Feedfetcher-Google', 'mon.itor.us', 'spbot', 'Feedly', 'bitlybot', 'ADmantX Platform', 'Niki-Bot', 'Pinterest', 'python-requests', 'DotBot', 'HTTP_Request2', 'linkdexbot', 'A6-Indexer', 'Baiduspider', 'TwitterFeed', 'Microsoft Office', 'Pingdom', 'BTWebClient', 'KatBot', 'SiteCheck', 'proximic', 'Sleuth', 'Abonti', '(BOT for JCE)', 'Baidu', 'Tiny Tiny RSS', 'newsblur', 'updown_tester', 'linkdex', 'baidu', 'searchmetrics', 'genieo', 'majestic12', 'spinn3r', 'profound', 'domainappender', 'VegeBot', 'terrykyleseoagency.com', 'CommonCrawler Node', 'AdlesseBot', 'metauri.com', 'libwww-perl', 'rogerbot-crawler', 'MegaIndex.ru' , 'ltx71' , 'Qwantify' , 'Traackr.com', 'Re-Animator Bot', 'Pcore-HTTP', 'BoardReader', 'omgili', 'okhttp', 'CCBot', 'Java/1.8', 'semrush.com', 'feedbot', 'CommonCrawler', 'AdlesseBot', 'MetaURI', 'ibwww-perl', 'rogerbot', 'MegaIndex', 'BLEXBot', 'FlipboardProxy', 'techinfo@ubermetrics-technologies.com', 'trendictionbot', 'Mediatoolkitbot', 'trendiction', 'ubermetrics', 'ScooperBot', 'TrendsmapResolver', 'Nuzzel', 'Go-http-client', 'Applebot', 'LivelapBot', 'GroupHigh', 'SemrushBot', 'ltx71', 'commoncrawl', 'istellabot', 'DomainCrawler', 'cs.daum.net', 'StormCrawler', 'GarlikCrawler', 'The Knowledge AI', 'getstream.io/winds', 'YisouSpider', 'archive.org_bot', 'semantic-visions.com', 'FemtosearchBot', '360Spider', 'linkfluence.com', 'glutenfreepleasure.com', 'Gluten Free Crawler', 'YaK/1.0', 'Cliqzbot', 'app.hypefactors.com', 'axios', 'semantic-visions.com', 'webdatastats.com', 'schmorp.de', 'SEOkicks', 'DuckDuckBot', 'Barkrowler', 'ZoominfoBot', 'Linguee Bot', 'Mail.RU_Bot', 'OnalyticaBot', 'Linguee Bot', 'admantx-adform', 'Buck/2.2', 'Barkrowler', 'Zombiebot', 'Nutch', 'SemanticScholarBot', 'Jetslide', 'scalaj-http', 'XoviBot', 'sysomos.com', 'PocketParser', 'newspaper', 'serpstatbot', 'MetaJobBot', 'SeznamBot/3.2', 'VelenPublicWebCrawler/1.0', 'WordPress.com mShots', 'adscanner', 'BacklinkCrawler', 'netEstate NE Crawler', 'Astute SRM', 'GigablastOpenSource/1.0', 'DomainStatsBot', 'Winds: Open Source RSS & Podcast', 'dlvr.it', 'BehloolBot', '7Siters', 'AwarioSmartBot', 'Apache-HttpClient/5', 'Seekport Crawler', 'AHC/2.1', 'eCairn-Grabber', 'mediawords bot', 'PHP-Curl-Class', 'Scrapy', 'curl/7', 'Blackboard', 'NetNewsWire', 'node-fetch', 'admantx', 'metadataparser', 'Domains Project', 'SerendeputyBot', 'Moreover', 'DuckDuckGo' , 'monitoring-plugins', 'Selfoss', 'Adsbot', 'acebookexternalhit', 'SpiderLing'];
        $mobile_platforms = ['iPhone', 'iPad', 'iPod', 'Android', 'Tizen', 'Blackberry'];


        $platform = null;
        $browser  = null;
        $version  = null;

        $os_name = null;
        $os_version = null;
        $is_mobile = false;
        $is_bot = false;
        
        $empty = ['platform' => $platform, 'browser' => $browser, 'version' => $version, 'browser_version' => null, 'os' => $os_name, 'os_version' => $os_version];
        
        if( empty($ua) ) {            
                return $empty;
        }        

                foreach($bots as $b)
                {
                    if( stripos( $ua, $b ) !== false ) 
                    {
                        
                            $is_bot = true;
                            $browser = $b;
                            $platform = 'bot';
                            $result = ['browser' => 'Spiderbot', 'platform' => $b, 'browser_version' => null,  'is_bot' => true, 'os_version' => null, 'os' => null];
                            return $result;
                    }
                }                                        


        if( preg_match('/\((.*?)\)/m', $ua, $parent_matches) ) {
                preg_match_all('/(?P<platform>BB\d+;|Android|CrOS|Tizen|iPhone|iPad|iPod|Linux|(Open|Net|Free)BSD|Macintosh|Windows(\ Phone)?|Silk|linux-gnu|BlackBerry|PlayBook|X11|(New\ )?Nintendo\ (WiiU?|3?DS|Switch)|Xbox(\ One)?)(?:\ [^;]*)?(?:;|$)/imx'
                        , $parent_matches[1], $result);

                $priority = array( 'Xbox One', 'Xbox', 'Windows Phone', 'Tizen', 'Android', 'FreeBSD', 'NetBSD', 'OpenBSD', 'CrOS', 'X11' );

                $result['platform'] = array_unique($result['platform']);
                if( count($result['platform']) > 1 ) {

                        if( $keys = array_intersect($priority, $result['platform']) ) {
                                $platform = reset($keys);
                        } else {
                                $platform = $result['platform'][0];

                        }
                } elseif( isset($result['platform'][0]) ) {
                        $platform = $result['platform'][0];                        
                        switch ($platform)
                        {
                            case 'Macintosh':
                                //preg_match('/Mac OS X (?P<version>[0-9A-Z.]+)/i', $ua, $osv_result);
                                $mac_pattern = '/Mac\s?OS X (?P<version>[0-9A-Z._]+)/i';
                                $os_name = 'Mac OS X';
                                preg_match($mac_pattern, $ua, $arr_mac_os_version);
                                if (isset($arr_mac_os_version['version']))
                                {
                                    
                                    $int_macosx_version = -1;
                                    $str_mac_os_version = $arr_mac_os_version['version'];
                                    if (str_contains($str_mac_os_version,'_'))
                                    {
                                        $int_macosx_version = (int) str_replace('_','', $str_mac_os_version);
                                        $os_version =  ($int_macosx_version > 0) ? str_replace('_','.', $str_mac_os_version) : '';
                                    } else  if (str_contains($str_mac_os_version,'.'))
                                    {
                                        $int_macosx_version = (int) str_replace('.','', $str_mac_os_version);
                                        $os_version =  ($int_macosx_version > 0) ? $str_mac_os_version : '';
                                    }

                                }
                                
                                break;
                            case 'Windows':
                                preg_match('/Windows NT (?P<version>[0-9A-Z.]+);/i', $ua, $osv_result);
                                $os_name = 'Windows';
                                $parsed_os_version = isset($osv_result['version']) ? $osv_result['version'] : '';
                                if (array_key_exists($parsed_os_version, $windows_nt_version))
                                    $os_version = $windows_nt_version[ $parsed_os_version ];
                                else
                                    $os_version = 'FAKE';
                                break;
                            case 'Android':
                                preg_match('/Android (?P<version>[0-9A-Z.]+);/i', $ua, $osv_result);
                                $os_name = 'Android';
                                $os_version = $osv_result['version'];
                                $is_mobile = true;
                                break;

                        }
                }
        }

        if( $platform == 'linux-gnu' || $platform == 'X11' ) {
                $platform = 'Linux';
                $os_name = 'Linux';
                $os_version = '';
        } elseif( $platform == 'CrOS' ) {
                $platform = 'Chrome OS';
                $os_name = 'Chrome OS';

        }

        /*preg_match_all('%(?P<browser>Camino|Kindle(\ Fire)?|Firefox|Iceweasel|IceCat|Safari|MSIE|Trident|AppleWebKit|TizenBrowser|(?:Headless)?Chrome|YaBrowser|Vivaldi|IEMobile|Opera|OPR|Silk|Midori|Edge|Edg|CriOS|UCBrowser|Puffin|OculusBrowser|SamsungBrowser|Baiduspider|Applebot|Googlebot|YandexBot|bingbot|Lynx|Version|Wget|curl|Valve\ Steam\ Tenfoot|NintendoBrowser|PLAYSTATION\ (\d|Vita)+)(?:\)?;?)(?:(?:[:/ ])(?P<version>[0-9A-Z.]+)|/(?:[A-Z]*))%ix'
                , $ua, $result);
*/
    preg_match_all(<<<'REGEX'
%(?P<browser>Camino|Kindle(\ Fire)?|Firefox|Iceweasel|IceCat|Safari|MSIE|Trident|AppleWebKit|
TizenBrowser|(?:Headless)?Chrome|YaBrowser|Vivaldi|IEMobile|Opera|OPR|Silk|Midori|(?-i:Edge)|EdgA?|CriOS|UCBrowser|Puffin|
OculusBrowser|SamsungBrowser|SailfishBrowser|XiaoMi/MiuiBrowser|YaApp_Android|Whale|
Baiduspider|Applebot|Facebot|Googlebot|YandexBot|bingbot|Lynx|Version|Wget|curl|ChatGPT-User|GPTBot|OAI-SearchBot|
Valve\ Steam\ Tenfoot|Mastodon|
NintendoBrowser|PLAYSTATION\ (?:\d|Vita)+)
\)?;?
(?:[:/ ](?P<version>[0-9A-Z.]+)|/[A-Z]*)
%ix
REGEX
            , $ua, $result);        
    
        // If nothing matched, return null (to avoid undefined index errors)
        if( !isset($result['browser'][0]) || !isset($result['version'][0]) ) {
                if( preg_match('%^(?!Mozilla)(?P<browser>[A-Z0-9\-]+)(/(?P<version>[0-9A-Z.]+))?%ix', $ua, $result) ) {
                        return [
                            'platform' => $platform ?: null, 
                            'browser' => $result['browser'], 
                            'browser_version' => null,
                            'os_version' => null,
                            'os' => null,
                            'version' => empty($result['version']) ? null : $result['version'] ];
                }
                return $empty;
        }

        if( preg_match('/rv:(?P<version>[0-9A-Z.]+)/i', $ua, $rv_result) ) {
                $rv_result = $rv_result['version'];
        }

        $browser = $result['browser'][0];
        $version = $result['version'][0];

        $lowerBrowser = array_map('strtolower', $result['browser']);

        $find = function ( $search, &$key = null, &$value = null ) use ( $lowerBrowser ) {
                $search = (array)$search;

                foreach( $search as $val ) {
                        $xkey = array_search(strtolower($val), $lowerBrowser);
                        if( $xkey !== false ) {
                                $value = $val;
                                $key   = $xkey;

                                return true;
                        }
                }

                return false;
        };

        $findT = function ( array $search, &$key = null, &$value = null ) use ( $find ) {
                $value2 = null;
                if( $find(array_keys($search), $key, $value2) ) {
                        $value = $search[$value2];

                        return true;
                }

                return false;
        };

        $key = 0;
        $val = '';
        /*if( $findT(array( 'OPR' => 'Opera', 'UCBrowser' => 'UC Browser', 'YaBrowser' => 'Yandex', 'Iceweasel' => 'Firefox', 'Icecat' => 'Firefox', 'CriOS' => 'Chrome', 'Edg' => 'Edge' ), $key, $browser) ) {*/
        if ($findT(['OPR' => 'Opera', 'Facebot' => 'iMessageBot', 'UCBrowser' => 'UC Browser', 'YaBrowser' => 'Yandex', 'YaApp_Android' => 'Yandex', 'Iceweasel' => 'Firefox', 'Icecat' => 'Firefox', 'CriOS' => 'Chrome', 'Edg' => 'Edge', 'EdgA' => 'Edge', 'XiaoMi/MiuiBrowser' => 'MiuiBrowser'], $key, $browser)) {        
                $version = $result['version'][$key];
                //$version = is_numeric(substr($result['browser_version'][$key], 0, 1)) ? $result['browser_version'][$key] : null;
        } elseif( $find('Playstation Vita', $key, $platform) ) {
                $platform = 'PlayStation Vita';
                $browser  = 'Browser';
        } elseif( $find(array( 'Kindle Fire', 'Silk' ), $key, $val) ) {
                $browser  = $val == 'Silk' ? 'Silk' : 'Kindle';
                $platform = 'Kindle Fire';
                if( !($version = $result['version'][$key]) || !is_numeric($version[0]) ) {
                        $version = $result['version'][array_search('Version', $result['browser'])];
                }
        } elseif( $find('NintendoBrowser', $key) || $platform == 'Nintendo 3DS' ) {
                $browser = 'NintendoBrowser';
                $version = $result['version'][$key];
        } elseif( $find('Kindle', $key, $platform) ) {
                $browser = $result['browser'][$key];
                $version = $result['version'][$key];
        } elseif( $find('Opera', $key, $browser) ) {
                $find('Version', $key);
                $version = $result['version'][$key];
        } elseif( $find('Puffin', $key, $browser) ) {
                $version = $result['version'][$key];
                if( strlen($version) > 3 ) {
                        $part = substr($version, -2);
                        if( ctype_upper($part) ) {
                                $version = substr($version, 0, -2);

                                $flags = array( 'IP' => 'iPhone', 'IT' => 'iPad', 'AP' => 'Android', 'AT' => 'Android', 'WP' => 'Windows Phone', 'WT' => 'Windows' );
                                if( isset($flags[$part]) ) {
                                        $platform = $flags[$part];
                                }
                        }
                }
        } 
        elseif ($find(['Googlebot', 'Applebot', 'IEMobile', 'Edge', 'Midori', 'Whale', 'Vivaldi', 'OculusBrowser', 'SamsungBrowser', 'Valve Steam Tenfoot', 'Chrome', 'HeadlessChrome', 'SailfishBrowser'], $key, $browser)) {
        //elseif( $find(array( 'Applebot', 'IEMobile', 'Edge', 'Midori', 'Vivaldi', 'OculusBrowser', 'SamsungBrowser', 'Valve Steam Tenfoot', 'Chrome', 'HeadlessChrome' ), $key, $browser) ) {
                $version = $result['version'][$key];

                
        } elseif( $rv_result && $find('Trident') ) {
                $browser = 'MSIE';
                $version = $rv_result;
        } elseif( $browser == 'AppleWebKit' ) {
            
                if( $platform == 'Android' ) {
                        $browser = 'Android Browser';
                } elseif( !empty($platform) && strpos($platform, 'BB') === 0 ) {
                        $browser  = 'BlackBerry Browser';
                        $platform = 'BlackBerry';
                } elseif( $platform == 'BlackBerry' || $platform == 'PlayBook' ) {
                        $browser = 'BlackBerry Browser';
                } else {
                        $find('Safari', $key, $browser) || $find('TizenBrowser', $key, $browser);
                }

                $find('Version', $key);
                $version = $result['version'][$key];
        } elseif( $pKey = preg_grep('/playstation \d/i', $result['browser']) ) {
                $pKey = reset($pKey);

                $platform = 'PlayStation ' . preg_replace('/\D/', '', $pKey);
                $browser  = 'NetFront';
        }
        ////// Extra Android detection
                
                if (preg_match('/Mozilla\/\d+\.\d+ \([^;\)]+; Android ([\d\.]+)(?:; ([^;\)]+))?/', $ua, $matches)) {
                    $os_name = 'Android';
                    
                    $os_version = $matches[1];
                    $device_name = isset($matches[2]) ? trim($matches[2]) : '';


                    if (preg_match('/(Chrome|Firefox)\/([\d\.]+)/', $ua, $matches)) {
                        $result['browser'] = $matches[1].' (mobile)';
                        $result['browser_version'] = $matches[2];
                    } elseif (preg_match('/([a-zA-Z]+)\/([\d\.]+)/', $ua, $matches)) {
                        $result['browser'] = $matches[1];
                        $result['browser_version'] = $matches[2];
                    }
            }                
                        
        
        foreach($bots as $b)
        {
            if( stripos( $ua, $b ) !== false ) 
            {
                    $is_bot = true;
                    $browser = $b;
                    $platform = 'bot';
            }
        }                
        $result = [];
        if ($platform)
        {
            $result['platform'] = $platform;            
            $result['is_mobile'] = in_array($platform, $mobile_platforms);
        }
        if ($browser)
            $result['browser'] = $browser;
        if ($platform)
            $result['browser_version'] = $version;
        else
            $result['browser_version'] = null;
        if ($os_name)
            $result['os'] = $os_name;
        else 
            $result['os'] = null;
        if ($os_version)
            $result['os_version'] = $os_version;     
        else 
            $result['os_version'] = null;
        $result['is_bot'] = $is_bot;
        return $result;
}


function get_system_or_user_file($filename, $die = true) {
    $sys_file = remove_multiple_slashes(SCHLIX_SYSTEM_PATH . '/' . $filename);
    $user_file = remove_multiple_slashes(SCHLIX_SITE_PATH . '/' . $filename);

    if (file_exists($user_file))
        return $user_file;
    elseif (file_exists($sys_file))
        return $sys_file;
    elseif ($die)
        die("File {$filename} cannot be found anywhere");
}

function get_system_or_user_url_path($filename, $die = true) {
    $user_file = remove_multiple_slashes(SCHLIX_SITE_PATH . '/' . $filename);
    $sys_file = remove_multiple_slashes(SCHLIX_SYSTEM_PATH . $filename);
    if (file_exists($user_file))
        return remove_multiple_slashes('/' . $filename);
    elseif (file_exists($sys_file))
        return remove_multiple_slashes('/system/' . $filename);
    elseif ($die != false)
        die("File {$filename} cannot be found anywhere, URL not generated");
}

//_________________________________________________________________________//
/**
 * Returns the application real name, given the alias as defined
 * in the application configuration page
 * @global \SCHLIX\cmsDatabase $SystemDB
 * @param string $alias
 * @return string
 */
function get_application_real_name_by_alias($alias) {
    global $SystemDB;

    if (empty($alias)) return NULL;
    $sql = 'SELECT title FROM gk_app_items WHERE app_alias = ' . sanitize_string($alias);
    $result = $SystemDB->getQueryResultSingleRow($sql);
    return ($result != null) ? $result['title'] : $alias;
}
/**
 * Returns a new, non-duplicate filename 
 * @param string $folder
 * @param string $original_filename
 * @return string
 */
function prevent_duplicate_filename_in_folder($folder, $original_filename) {
    $filename = pathinfo($original_filename, PATHINFO_FILENAME);
    $extension = pathinfo($original_filename, PATHINFO_EXTENSION);
    $new_filename = $original_filename;
    $count = 2;
    while (file_exists(remove_multiple_slashes($folder . '/' . $new_filename))) {
        $new_filename = $filename . '_' . $count . '.' . $extension;
        $count++;
    }
    return $new_filename;
}

            
//_________________________________________________________________________//
/**
 * Force URL to be HTTPS if the site's SSL variable is enabled and HTTPS URL is defined
 * inthe subsite configuration
 * @return string
 */
function force_https_url() {
    return (defined('SCHLIX_SITE_HTTPS_URL') && SCHLIX_SITE_SSL_ENABLED && SCHLIX_SITE_HTTPS_URL != '') ? SCHLIX_SITE_HTTPS_URL : SCHLIX_SITE_HTTP_URL;
}

//_________________________________________________________________________//
/**
 * Setthe CSRF token if none
 * @return string
 */
function set_csrf_token() {
    $_SESSION['csrf'] = (empty($_SESSION['csrf'])) ? sha1(SCHLIX_SITE_NAME . uniqid(mt_rand(), TRUE)) : $_SESSION['csrf'];
    return $_SESSION['csrf'];
}

//_________________________________________________________________________//
/**
 * Returns the CSRF token
 * @return string
 */
function get_csrf_token() {

    return fsession_string('csrf');
}

//_________________________________________________________________________//
/**
 * Check if the variable _csrftoken matches with the currently set token.
 * If the array is not specified, then $_POST will be used
 * @param array $datavalues
 * @return bool
 */
function is_valid_csrf($datavalues = NULL) {
    $csrf_token = (!is_array($datavalues) || empty($datavalues)) ? $_POST['_csrftoken'] :$datavalues['_csrftoken'];
    return get_csrf_token() === $csrf_token;
}

function check_csrf_halt_on_error($is_ajax = false) {
    if (!is_valid_csrf()) {
        if (is_ajax_request() || $is_ajax) {
            ajax_echo(ajax_reply('400', 'Invalid Ajax Request - CSRF cannot be verified. If this error keeps happening, please clear all your browser cache and reload the page before trying again.'));
            die;
        } else {
            die('Invalid CSRF Verification Token');
        }
    }
}

/**
 * Simple file hash
 * @param string $filename
 * @return string
 */

function get_simplified_file_hash($filename)
{
    return (is_readable_file($filename)) ? crc32(dechex(filesize($filename)).'_'.filemtime($filename)) : '-1s';
}

/**
 * Check if file is readable and is a file
 * @param string $filename
 * @return bool
 */
function is_readable_file($filename)
{
     $rp = realpath($filename);
     return is_file($rp) && is_readable($rp);
}

/**
 * Returns true if image file is valid
 * @param string $filename
 * @param string $check_filename
 * @return boolean
 */
function is_valid_image_file($filename, $check_filename = true)
{
    
    if (!file_exists($filename))
        return false;

    $allowed_exts = ['png','jpg','jpeg','gif'];
    $valid = true;
    $ext = strtolower( pathinfo($filename, PATHINFO_EXTENSION));
    if ($check_filename)
    {
        
        if (in_array($ext, $allowed_exts) === FALSE)
        {
            return false;
        }
    }    
    $img_info = getimagesize($filename);
    $imgtype = $img_info[2];
    if ($imgtype > 0 && $imgtype < 4) // gif, png, jpeg
    {
        
        $sx = (int) $img_info [0];
        $sy = (int) $img_info [1];
        $valid = ($sx > 0 && $sx < 16384) && ($sy > 0 && $sy < 16384);
    }
    
    if ($valid && ($ext == 'gif' || $imgtype == 1))
    {
        $fstr = file_get_contents($filename);
        $valid = !str_contains($fstr,'<?php');
    }
    return $valid;
    
}
//_________________________________________________________________________//
/**
 * Returns true if image is valid (beta)
 * @param string $tmp_filename
 * @param string $uploaded_filename
 * @return boolean
 */
function is_valid_uploaded_image_file($tmp_filename, $uploaded_filename)
{
    if (!file_exists($tmp_filename))
        return false;
    
    $allowed_exts = ['png','jpg','jpeg','gif'];
    $valid = true;
    
    $ext = strtolower( pathinfo($uploaded_filename, PATHINFO_EXTENSION));
    if (in_array($ext, $allowed_exts) === FALSE)
        return false;
    else 
    {        
        $valid = is_valid_image_file($tmp_filename, false);
    }
    return $valid;
}


// This is a quicky , will be fixed later
//_________________________________________________________________________//
/**
 * Record 404 error
 * @global \SCHLIX\cmsDatabase $SystemDB
 * @return type
 */
function record_404_error() {
    global $SystemDB;

    if (SCHLIX_RECORD_404_ERRORS != 1 || !$SystemDB->tableExists('gk_404_error_items'))
        return;
    
    $url_proto = 'http' . (empty($_SERVER['HTTPS']) ? '' : 's') . '://';
    $url_parts = parse_url($url_proto . $_SERVER['SERVER_NAME'] . $_SERVER['REQUEST_URI']);
    //$real_url = $url_parts['scheme'] . '://' . $url_parts['host'] . $url_parts['path'];

    $request_path = strtolower(trim($url_parts['path']));

    if (!empty(SCHLIX_SITE_HTTPBASE) && str_starts_with($request_path, SCHLIX_SITE_HTTPBASE))
    {
        $sl = strlen(SCHLIX_SITE_HTTPBASE);
        $request_path = substr($request_path, $sl + 1, strlen($request_path) - $sl -1);        
    }
    $path_params = ['path' => $request_path];
    $existing_url_data = $SystemDB->getQueryResultSingleRow("SELECT id FROM gk_404_error_items WHERE path = :path", $path_params);
    $SystemDB->query("INSERT INTO gk_404_error_items (path, date_modified, hits) VALUES (:path, NOW(), 1) ON DUPLICATE KEY UPDATE hits = hits + 1", $path_params);
}

//_________________________________________________________________________//
/**
 * Display HTTP Error (e.g. 404, 400, etc). Internal use only
 * @internal
 * @param int $error_number
 */
function display_http_error($error_number) {

    $defaultHTTPErrorMessages = array(
      400 => 'HTTP/1.0 400 Bad Request',
      401 => 'HTTP/1.0 401 Unauthorized',
      402 => 'HTTP/1.0 402 Payment Required',
      403 => 'HTTP/1.0 403 Forbidden',
      404 => 'HTTP/1.0 404 Not Found',
      405 => 'HTTP/1.0 404 Method Not Allowed',
      406 => 'HTTP/1.0 404 Not Acceptable',
      407 => 'HTTP/1.0 404 Proxy Authentication Required',
      408 => 'HTTP/1.0 404 Request Time-out',
      409 => 'HTTP/1.0 404 Conflict',
      410 => 'HTTP/1.0 404 Gone',
      411 => 'HTTP/1.0 404 Length Required',
      412 => 'HTTP/1.0 404 Precondition Failed',
      413 => 'HTTP/1.0 404 Request Entity Too Large',
      414 => 'HTTP/1.0 404 Request-URI Too Large',
      415 => 'HTTP/1.0 404 Unsupported Media Type',
      500 => 'HTTP/1.0 500 Internal Server Error'
    );

    
    header($defaultHTTPErrorMessages[$error_number]);
    if (($error_number >= 400 && $error_number <= 415) || $error_number == 50) {
        if ($error_number == 404)
            record_404_error();
        if (!is_ajax_request()) {
            $http_error = new \App\Core_HttpError();
            $http_error->viewItemByID($error_number);
        } else {
            ajax_echo(ajax_reply($error_number, $defaultHTTPErrorMessages[$error_number]));
        }
    }
}

//_________________________________________________________________________//
/**
 * Returns a Javascript variable that contains csrftoken string
 * e.g. var variable_name = '1231351232asdasd' 
 * @param string $varname
 * @return string
 */
function get_js_csrf_var($varname = '_csrftoken') {
    $s = 'var '.$varname.'="' . get_csrf_token() . '";';
    return $s;
}

//_________________________________________________________________________//
/**
 * Adds an error string that will be displayed when the page loads again
 * next time
 * @internal
 * @param string $error
 */
function display_schlix_alert($error) {
    if (!array_key_exists('schlix_internal_messages', $_SESSION) || !is_array($_SESSION['schlix_internal_messages']))
        $_SESSION['schlix_internal_messages'] = [];
    $add = true;
    $count = ___c($_SESSION['schlix_internal_messages']);
    if ($count > 0)
    {
        if ($_SESSION['schlix_internal_messages'][$count-1] == $error)
            $add = false;
    }
    if ($add)
        $_SESSION['schlix_internal_messages'][] = $error;
}

//_________________________________________________________________________//
/**
 * Clear error string
 */
function clear_schlix_alert() {
    unset($_SESSION['schlix_internal_messages']);
}

//_________________________________________________________________________//
/**
 * Returns true if the input string is serialized
 * @param string $string
 * @return bool
 */
function is_serialized($string) {
    if (is_string($string))
    {
        return ((@unserialize($string) !== false && strpos($string, '{') !== false) || $string === 'b:0;');
    }
    return false;
}

//_______________________________________________________________________________________________________________//
/**
 * Generates a new UUIDv4
 * Source Credit:  copied and pasted this from http://php.net/manual/en/function.uniqid.php
 * @return string
 */
function new_uuid_v4() { 
    // The field names refer to RFC 4122 section 4.1.2
    return sprintf('%04x%04x-%04x-%03x4-%04x-%04x%04x%04x', mt_rand(0, 65535), mt_rand(0, 65535), // 32 bits for "time_low"
        mt_rand(0, 65535), // 16 bits for "time_mid"
        mt_rand(0, 4095), // 12 bits before the 0100 of (version) 4 for "time_hi_and_version"
        bindec(substr_replace(sprintf('%016b', mt_rand(0, 65535)), '01', 6, 2)),
        // 8 bits, the last two of which (positions 6 and 7) are 01, for "clk_seq_hi_res"
        // (hence, the 2nd hex digit after the 3rd hyphen can only be 1, 5, 9 or d)
        // 8 bits for "clk_seq_low"
        mt_rand(0, 65535), mt_rand(0, 65535), mt_rand(0, 65535)); // 48 bits for "node"
}

/**
 * Returns true if format is valid GUID
 * @param string $guid
 * @return bool
 */
function is_valid_guid($guid)
{
    return preg_match('/^[a-f\d]{8}-(?:[a-f\d]{4}-){3}[a-f\d]{12}$/i', $guid) > 0 ;
}


//_________________________________________________________________________//
function get_application_alias($app, $subclass = NULL, $use_cache = true) {
    global $SystemDB;
    
    $subclass_suffix = $subclass ? '.'.$subclass : '';
    $app_alias_cache = \SCHLIX\cmsContextCache::get('app_alias',$app);
    if ($use_cache && $app_alias_cache) 
        return $app_alias_cache.$subclass_suffix;
    else
    {
        $sql = 'SELECT app_alias FROM gk_app_items WHERE title = ' . sanitize_string($app);
        $result = $SystemDB->getQueryResultSingleRow($sql);
        $alias = null;
        if ($result != null) {
            $alias = $result['app_alias'];
            if (!$alias)
                $alias = $app;
        } else
            $alias = $app;
        \SCHLIX\cmsContextCache::set('app_alias',$app, $alias);        
        return $alias.$subclass_suffix;
    }
}

//_________________________________________________________________________//
/**
 * Redirect URL
 * @param string $url
 */
function redirect_url($url) {
    ob_end_clean();
    ob_start();
    header("Location: {$url}");
    exit();
}

/**
 * Return URL encode
 * @param string $path
 */
function encode_url($path)
{
    return ___u($path); // implode("/", array_map("rawurlencode", explode("/", $path)));
}

//_________________________________________________________________________//
/**
 * Returns a list of all applications
 * @global type $SystemConfig
 * @param bool $use_cache
 * @param int $cache_how_many_minutes
 * @return array
 */
function get_list_of_all_apps($use_cache = false, $cache_how_many_minutes = 5) {
    global $SystemConfig;

    $refresh_period_in_second = intval($cache_how_many_minutes) * 60;
    $last_time_when_it_was_cached = intval($SystemConfig->get('system', 'frontend_apps_cache_time'));
    $cached_app_list = $SystemConfig->get('system', 'frontend_apps_cache_content');
    if ($use_cache && ((time() - $last_time_when_it_was_cached) <= $refresh_period_in_second) && (___c($cached_app_list) > 0)) {
        return $cached_app_list;
    } else {
        $main_dir = SCHLIX_SYSTEM_PATH . '/apps/';
        $app_list = [];

        $apps_dir = new DirectoryIterator($main_dir);
        if ($apps_dir)
            foreach ($apps_dir as $app_dir)
                if (!$app_dir->isDot() && $app_dir->isDir() && (strpos($app_dir->getFileName(), 'uninstalled_') === false) && file_exists($main_dir . $app_dir->getFileName() . '/' . $app_dir->getFileName() . '.class.php'))
                    $app_list[] = $app_dir->getFileName();

        $main_dir = SCHLIX_SITE_PATH . '/apps/';
        $apps_dir = new DirectoryIterator($main_dir);
        if ($apps_dir)
            foreach ($apps_dir as $app_dir)
                if (!$app_dir->isDot() && $app_dir->isDir() && (strpos($app_dir->getFileName(), 'uninstalled_') === false) && file_exists($main_dir . $app_dir->getFileName() . '/' . $app_dir->getFileName() . '.class.php'))
                    $app_list[] = $app_dir->getFileName();
        if ($use_cache) {
            $SystemConfig->set('system', 'frontend_apps_cache_time', time());
            $SystemConfig->set('system', 'frontend_apps_cache_content', $app_list);
        }
        return $app_list;
    }
}

//______________________________________________________________________________________//
/**
 * Returns true if Captcha is valid
 * @return bool
 */
function is_captcha_verification_valid() {
    return \App\Core_Captcha::isCaptchaCodeValid();
    //$schlix_captch = new \App\Core_Captcha();
    //return ($securimage->check($_POST['verification_code']) != false);
    
}

//______________________________________________________________________________________//
/**
 * Converts data:image/png;base64 from text to image
 * @global \SCHLIX\cmsLogger $SystemLog
 * @param string $txt
 * @param string $upload_path
 * @param string $url_path
 * @param string $filename_prefix
 * @return string
 */
function convert_pasted_png_images_from_html_text($txt, $upload_path, $url_path, $filename_prefix = 'img') {
    global $SystemLog;

    $filename_prefix = alpha_numeric_only($filename_prefix);
    $save_result = false;
    //preg_match_all('/src="data:image\\/png;base64[^>]+>/i',$txt, $pasted_images_array);
    preg_match_all('/src=("data:image\\/png;base64[^"]*")/i', $txt, $pasted_images_array);
    $i = 1;
    $start_tag = 'src="data:image/png;base64,';
    $end_tag = '"';
    $new_start_tag = 'src="';

    foreach ($pasted_images_array[0] as $img_tag) {
        $outputfile = $filename_prefix.'-'.date('Y-m-d') . "-{$i}.png";
        while (file_exists($upload_path . $outputfile)) {
            $outputfile = $filename_prefix.'-'.date('Y-m-d') . "-{$i}.png";
            $i++;
        }
        $imageData = str_replace($start_tag, '', $img_tag);
        $imageData = str_replace($end_tag, '', $imageData);
        $ifp = fopen($upload_path . $outputfile, "wb");
        if ($ifp) {
            $bin64_decoded_data = base64_decode($imageData);
            $save_result = fwrite($ifp, $bin64_decoded_data);
            fclose($ifp);
            $txt = str_replace($start_tag . $imageData, $new_start_tag . $url_path . $outputfile, $txt);
        }
        $log = ($save_result) ? "Fetched pasted image and saved {$save_result} bytes to {$upload_path}{$outputfile}" : "Cannot convert pasted image to {$upload_path}{$outputfile}";
        $SystemLog->record($log);
    }
    return $txt;
}

//______________________________________________________________________________________//
/**
 * Search for all external images hosted outside of this URL in this HTML text
 * to a local one. If the file already exists with the same name then it will be ignored
 * @param string $txt
 * @param string $upload_path
 * @param string $url_path
 * @return string
 */
function move_static_external_images_from_html_text($txt, $upload_path, $url_path) {
    preg_match_all('/<img[^>]+>/i', $txt, $images_array);
    $imgs = [];
    foreach ($images_array[0] as $img_tag) {
        //echo $img_tag;
        preg_match_all('/(alt|title|src)=("[^"]*")/i', $img_tag, $imgs[$img_tag]);
    }
    foreach (array_keys($imgs) as $x) {

        $src = $imgs[$x][2][0];
        $cleanurl = str_replace('"', '', $src);
        $cleanurl = str_replace("'", '', $cleanurl);
        $prs = parse_url($cleanurl);
        if (($prs['host'] != '') && strpos(SCHLIX_SITE_URL, $prs['host']) === false && ($prs['host'] != '')) {
            $basename = basename($cleanurl);
            $outputfile = convert_to_safe_filename($basename);
            if (!file_exists($upload_path . $outputfile)) {
                $filecontent = get_remote_file_content($cleanurl);
                file_put_contents($upload_path . $outputfile, $filecontent);
            }
            $txt = str_replace($cleanurl, $url_path . $outputfile, $txt);
        }
    }
    return $txt;
}

//______________________________________________________________________________________//
function quote_field_name_for_query($item_field) {
    if ($item_field) {
        $item_field = "`{$item_field}`";
    } else
        $item_field = '';
    return $item_field;
}

//______________________________________________________________________________________//
function quote_array_of_field_names_for_query($item_fields) {
    $total_field_count = ___c($item_fields);
    for ($i = 0; $i < $total_field_count; $i++)
        $item_fields[$i] = quote_field_name_for_query($item_fields[$i]);
    return $item_fields;
}

/**
 * Returns alphanumeric of string with  _ (underscore) allowed
 * @param string $str
 * @param string $replacement
 * @return string
 */
function alpha_numeric_with_underscore($str, $replacement = '-') {
    $str = preg_replace('/[^a-zA-Z0-9_]/', $replacement, $str);
    return trim($str, ' -');
}

/**
 * Returns alphanumeric of string with - (dash) and _ (underscore) allowed
 * @param string $str
 * @param string $replacement
 * @return string
 */
function alpha_numeric_with_dash_underscore($str, $replacement = '-') {
    $str = preg_replace('/[^a-zA-Z0-9-_]/', $replacement, $str);
    return trim($str, ' -');
}

/**
 * Returns alphanumeric of string with - (dash) and _ (underscore) allowed
 * @param string $str
 * @param string $replacement
 * @return string
 */
function sanitize_action_command($str) {
    $str = preg_replace('/[^a-zA-Z0-9]/', '_', $str);
    return trim($str, ' -');
}


/**
 * Like alpha_numeric_with_dash_underscore but without - (dash) and (underscore) allowed
 * @param string $str
 * @param string $replacement
 * @return string
 */
function alpha_numeric_only($str, $replacement = '') {
    $str = preg_replace('/[^a-zA-Z0-9]/', $replacement, $str);
    return trim($str);
}

function convert_to_safe_basename($str, $max_len, $default_empty = 'untitled', $allow_utf8 = true)
{
    $str = str_limit( trim($str), $max_len);
    $str = preg_replace('/[\x00-\x2C\x2E-\x2F\x3A-\x40\x5b-\x5e\x60\x7b-\x7f]+/', '_', $str);
    $str = htmlentities($str, ENT_QUOTES, "utf-8");
    $str = preg_replace("/(&)([a-z])([a-z]+;)/i", '$2', $str);
    $str = rawurlencode($str);
    $str = str_replace('%', '-', $str);
    $str = preg_replace('/^(con|prn|aux|nul|com[0-9]|lpt[0-9])(\..*)?$/i', 'wrsrvd_', $str);
    //$str = strtolower($str);
    if (empty($str))
        $str = $default_empty;
    return $str;
}    

/**
 * Alternative pathinfo
 * @param string $filepath
 * @return array
 */
function mb_pathinfo($filepath) {
    $ret = [];
    preg_match('%^(.*?)[\\\\/]*(([^/\\\\]*?)(\.([^\.\\\\/]+?)|))[\\\\/\.]*$%im',$filepath,$m);
    $ret['dirname'] = !empty($m[1]) ? $m[1] : '';
    $ret['basename'] = !empty($m[2]) ? $m[2] : '';
    $ret['extension']= !empty($m[5]) ? $m[5] : '';
    $ret['filename']= !empty($m[3]) ? $m[3] : '';
    return $ret;
}

/**
 * Converts a string to safe filename
 * @param string $s
 * @return string
 */

function convert_to_safe_filename($str, $default_empty_ext = '')
{
    
    $path_info = mb_pathinfo($str);
    
    $fn = convert_to_safe_basename($path_info['filename'],171);
    $ext = convert_to_safe_basename($path_info['extension'],20,$default_empty_ext);
    $result = $fn;
    if ($ext)
        $result.= '.'.$ext;
    return $result;
}


/**
 * Converts a string to an alphanumeric one (dash and underscores allowed).
 * This is mainly used to convert a title page to virtual filename
 * @param string $str
 * @param bool $allow_utf8
 * @return string
 */
function convert_into_sef_friendly_title($str, $allow_utf8 = true) {
    /*  $str = htmlentities($str, ENT_QUOTES, 'UTF-8');
      $str = preg_replace('~&([a-z]{1,2})(acute|cedil|circ|grave|lig|orn|ring|slash|th|tilde|uml);~i', '$1', $str);
      $str = html_entity_decode($string, ENT_QUOTES, 'UTF-8');
     */
    $str = strtolower(trim($str));
    //   $str = preg_replace(array('~[^0-9a-z]~i', '~[ -]+~'), ' ', $str);

    $pattern = ($allow_utf8) ? '~[^\p{L}\p{N}\p{Mn}\p{Pd}\'\x{2019}_]~u' : '/[^a-z0-9_\-]/';
    
    $str = preg_replace($pattern, '-', $str);
    $str = preg_replace('/-+/', "-", $str);

    return trim($str, ' -');
}

//______________________________________________________________________________________//
/**
 * Returns the number of seconds between start_date and end_date
 * @param string|date $start_date
 * @param string|date $end_date
 * @return int
 */
function time_difference($start_date, $end_date) {
    return  strtotime($end_date) - strtotime($start_date);
}

//______________________________________________________________________________________//
/**
 * Returns the day numbers between start_date and end_date
 * @param string|date $start_date
 * @param string|date $end_date
 * @return int
 */
function days_difference($start_date, $end_date) {
    $x = strtotime((string) $end_date) - strtotime((string)  $start_date);
    $number_of_days = floor($x / (60 * 60 * 24));
    return $number_of_days;
}

//______________________________________________________________________________________//
/**
 * Returns the number of day difference from today
 * @param string|date $the_date
 * @return int
 */
function days_difference_from_today($the_date) {
    return days_difference(get_current_datetime(), $the_date);
}

//_________________________________________________________________________//
/**
 * Return string of current date in ISO-8601 format
 * @return string
 */
function get_current_datetime() {
    return date('Y-m-d H:i:s');
}

/**
 * Returns true if email address is valid
 * @param string $email
 * @param bool $check_for_valid_domain_format
 * @return bool
 */
function is_valid_email_address($email, $check_for_valid_domain_format = false) {
    $result = filter_var($email, FILTER_VALIDATE_EMAIL) !== FALSE;    
    if ($result && $check_for_valid_domain_format)
    {
        list($name, $domain) = explode('@', $email);
        $result = str_contains($domain,'.');
    }
    return $result;
}
/**
 * Returns true if host name is valid
 * @param string $hostname
 * @param bool $check_for_valid_domain_format
 * @return bool
 */
function is_valid_hostname($hostname, $check_for_valid_domain_format = false) {
    $result = filter_var('http://'.$hostname, FILTER_VALIDATE_URL) !== FALSE;    
    if ($result && $check_for_valid_domain_format)
        $result = str_contains($hostname,'.');
    return $result;
}

//_________________________________________________________________________//
/**
 * Returns true if email address is DNS verified with the MX record
 * @param string $email
 * @return bool
 */
function verify_email_dns($email) {
    if (empty($email) || !is_valid_email_address($email))
        return false;
    
    list($name, $domain) = explode('@', $email);
    return (empty($domain) || !checkdnsrr($domain, 'MX')) ? false : $email;
}

//_________________________________________________________________________//
/**
 * Select HTTP POST Variables. Useful to filter variables based on table names
 * @param string|array $str
 * @return array
 */
function select_http_post_variables(/* ... */) {
    if (!$_POST)
        return false;
    if (func_num_args() == 0)
        return $_POST;
    foreach ($_POST as $key => $value)
    {
        if (is_string($_POST[$key]))
            $_POST[$key] = trim($value);
    }
    if (is_array(func_get_arg(0)))
        $array = array_fill_keys(func_get_arg(0), NULL);
    else
        $array = array_fill_keys(func_get_args(), NULL);
    return array_intersect_key($_POST, $array);
}

/**
 * Convert array values to keys and fill it with number
 * @param array $array
 * @param mixed $fill
 * @return array
 */
function convert_array_values_to_keys($array, $fill = 1)
{
    if (!is_array($array))
        return [];
    $result = array_flip($array);
    foreach ($result as $key => $value)
        $result[$key] = $fill;
    return $result;
}
//_________________________________________________________________________//
/**
 * Remove multiple slashes from string. 
 * @param string $str
 * @return string
 */
function remove_multiple_slashes($str) {
    return !empty($str) ? preg_replace('#/{2,}#', '/', $str) : '';
}

//_________________________________________________________________________//
/**
 * Sanitize string - SQL
 * @global \SCHLIX\cmsDatabase $SystemDB
 * @param string $value
 * @param bool $dont_add_quotes
 * @return string
 */
function sanitize_string($value, $dont_add_quotes = false) {
    global $SystemDB;
    if (!is_array($value))
        $value = $SystemDB->escapeString($value);
    $value = ($dont_add_quotes ? $value : "'" . $value . "'");
    //if (!is_numeric($value)) $value = "'" . str_replace("'","''",$value). "'";
    //if (!is_numeric($value)) $value = "'" . $value. "'"; // Nov 9, 2011 - Prana fix ' bug
    //if (!is_numeric($value)) $value = "'" . addslashes($value) . "'";
    return $value;
}


/**
 * Returns true if input is in date format
 * @param object|string|DateTime $date
 * @param string $format
 * @return bool
 */
function is_date($date, $format = 'Y-m-d H:i:s') {
    if (is_null ($date))
        return false;
    $d = DateTime::createFromFormat($format, $date);
    return $d && $d->format($format) == $date;
}


//______________________________________________________________________________________//
/**
 * Returns true if this request is AJAX
 * @return bool
 */
function is_ajax_request() {
    return ((!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest'));
}

//______________________________________________________________________________________//
/**
 * Returns true if it's a postback ... similar to ASP.NET's isPostBack()
 * @return bool
 */
function is_postback() {
    return ($_SERVER['REQUEST_METHOD'] === 'POST');
}

//______________________________________________________________________________________//
/**
 * Perform a real strip tag. Function copied from php.net
 * @param string $i_html
 * @param array $i_allowedtags
 * @param bool $i_trimtext
 * @return string
 */
function real_strip_tags($i_html, $i_allowedtags = [], $i_trimtext = FALSE) {
    if (!is_array($i_allowedtags))
        $i_allowedtags = !empty($i_allowedtags) ? array($i_allowedtags) : [];
    $tags = implode('|', $i_allowedtags);

    if (empty($tags))
        $tags = '[a-z]+';

    preg_match_all('@</?\s*(' . $tags . ')(\s+[a-z_]+=(\'[^\']+\'|"[^"]+"))*\s*/?>@i', $i_html, $matches);

    $full_tags = $matches[0];
    $tag_names = $matches[1];

    foreach ($full_tags as $i => $full_tag) {
        if (!in_array($tag_names[$i], $i_allowedtags))
            if ($i_trimtext)
                unset($full_tags[$i]);
            else
                $i_html = str_replace($full_tag, '', $i_html);
    }

    return $i_trimtext ? implode('', $full_tags) : $i_html;
}

/**
 * Create directory with permission 0755 if not exist
 * @param string $dir
 * @return bool
 */
function create_generic_directory($dir, $permission = 0755, $recursive = true)
{
    if (!is_dir($dir)) {
        if (!mkdir($dir, $permission, $recursive)) {
            return false;
        }
    }
    return true;
}

/**
 * Replaces Windows \ with /
 * @param string $f
 * @return string
 */
function unix_slash($f)
{
    return $f ? str_replace('\\', '/', $f) : '';
}
/**
 * Copy recursive from source to destination
 * @param string $source
 * @param string $dest
 * @return bool
 */
function copy_recursive($source, $dest, $blacklist_file_names = [], $blacklist_extensions = []) {

    if (!is_dir($source)) 
        return false;
    create_generic_directory($dest, 0755, false);
    $has_blacklist_file_names = ___c($blacklist_file_names) > 0;
    $has_blacklist_extensions = ___c($blacklist_extensions) > 0;
    foreach ($iterator = new \RecursiveIteratorIterator(
    new \RecursiveDirectoryIterator($source, \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST) as $item) 
    {
        
        $sub_path = unix_slash($iterator->getSubPathName());
        $filename = $iterator->getFilename();
        if ($item->isDir()) 
        {
            $target_dir = $dest . '/' . $sub_path;
            create_generic_directory($target_dir, 0755, false);
        } else 
        {
            $ext = $iterator->getExtension();
            $dont_copy = ($has_blacklist_extensions && in_array($ext, $blacklist_extensions)) ||
                        ($has_blacklist_file_names && in_array($filename, $blacklist_file_names)) ;
            if (!$dont_copy)
                copy($item, $dest . '/' . $sub_path);
        }
    }
}

/**
 * Sort an associative array by key
 * @param array $array
 * @param string $subkey
 * @param int $sortType
 */
function sort_associative_array_by_key(&$array, $subkey, $sortType = SORT_ASC) {
    if (is_array($array) && (count($array) > 0) && array_key_exists($subkey, $array[0])) {
        $keys = [];
        foreach ($array as $subarray) {
            $keys[] = $subarray[$subkey];
        }
        array_multisort($keys, $sortType, $array);
    }
}
//_________________________________________________________________________//
/**
 * Returns a JSON-encoded string of reply. Set status to 200 for ok
 * @param int $status
 * @param array $data
 * @return string
 */
function ajax_reply($status, $data) {
    $reply = [];
    header('Content-Type: application/json; charset=UTF-8');
    header("Cache-Control: no-cache, must-revalidate"); // HTTP/1.1
    header("Expires: Mon, 26 Jul 1997 05:00:00 GMT"); // Date in the past
    if ( (int) $status < 10)
        $reply['status'] = $status ? AJAX_REPLY_OK : AJAX_REPLY_ERROR;
    else
        $reply ['status'] = $status; // backward compatibility
    $reply ['data'] = $data;
    return json_encode($reply, JSON_NUMERIC_CHECK | JSON_PRESERVE_ZERO_FRACTION);
}

function ajax_reply_invalid_method($method_name)
{
    return ajax_reply_error(['messages' => ['Invalid method '.$method_name]]);
}


function ajax_reply_ok_redirect($success_message =  null, $url_redirect = '')
{
    header('Content-Type: application/json; charset=UTF-8');
    header("Cache-Control: no-cache, must-revalidate"); // HTTP/1.1
    header("Expires: Mon, 26 Jul 1997 05:00:00 GMT"); // Date in the past
    $reply ['status'] = 302;
    /*if ($data)
        $reply ['data'] = $data;*/
    if (!empty ($url_redirect))
        $reply ['url_redirect'] = $url_redirect;
    if (is_string($success_message))
    {
        $reply['messages'] = [$success_message];
    } else if (___c($success_message) > 0)
        $reply['messages'] = $success_message;
    
    return json_encode($reply, JSON_NUMERIC_CHECK | JSON_PRESERVE_ZERO_FRACTION);

}

function ajax_reply_ok_message( $success_message =  null)
{
    return ajax_reply_ok(null, $success_message);
}


function ajax_reply_ok($data = null)
{
    header('Content-Type: application/json; charset=UTF-8');
    header("Cache-Control: no-cache, must-revalidate"); // HTTP/1.1
    header("Expires: Mon, 26 Jul 1997 05:00:00 GMT"); // Date in the past
    $reply ['status'] = 200;
    if ($data)
        $reply ['data'] = $data;
    /*if (is_string($messages))
    {
        $reply['messages'] = [$messages];
    } else if (___c($messages) > 0)
        $reply['messages'] = $messages;
        
      */  
    return json_encode($reply, JSON_NUMERIC_CHECK | JSON_PRESERVE_ZERO_FRACTION);

}

function ajax_reply_error($data = null)
{
    header('Content-Type: application/json; charset=UTF-8');
    header("Cache-Control: no-cache, must-revalidate"); // HTTP/1.1
    header("Expires: Mon, 26 Jul 1997 05:00:00 GMT"); // Date in the past
    $reply ['status'] = 400;
    if ($data)
        $reply ['data'] = $data;
    /*if (is_string($messages))
    {
        $reply['messages'] = [$messages];
    } else if (___c($messages) > 0)
        $reply['messages'] = $messages;
        
      */  
    return json_encode($reply, JSON_NUMERIC_CHECK | JSON_PRESERVE_ZERO_FRACTION);

}



/**
 * AJAX - reply with error
 * @param string|array $messages
 * @param array $element_error_list
 * @param string $url_redirect
 * @return string
 */
/*
function ajax_reply_error($messages, $element_error_list = null, $url_redirect = '')
{
    header('Content-Type: application/json; charset=UTF-8');
    header("Cache-Control: no-cache, must-revalidate"); // HTTP/1.1
    header("Expires: Mon, 26 Jul 1997 05:00:00 GMT"); // Date in the past
    $reply ['status'] = 400;
    if (is_string($messages))
    {
        $reply['messages'] = [$messages];
    } else if (___c($messages) > 0)
        $reply['messages'] = $messages;
    if (___c($element_error_list) > 0)
        $reply ['element_error_messages'] = $element_error_list;
    if (!empty ($url_redirect))
        $reply ['url_redirect'] = $url_redirect;
    return json_encode($reply, JSON_NUMERIC_CHECK | JSON_PRESERVE_ZERO_FRACTION);
}*/


//_________________________________________________________________________//
/**
 * echo ajax reply and return false (RETURN AS AJAX)
 * @param string $s
 * @return string
 */
function ajax_echo($s) {

    echo $s;
    return FALSE;
}

//_________________________________________________________________________//
/**
 * Returns a JSON-encoded array of message for datasource in SCHLIX.UI.DataTable
 * @param int $status
 * @param int $allRecords
 * @param int $start
 * @param int $end
 * @param int $item_per_page
 * @param int $total_item_count
 * @param string $sort
 * @param string $dir
 * @return string
 */
function ajax_datasource_reply($status, $allRecords, $start = 0, $end = 0, $item_per_page = 0, $total_item_count = 0, $sort = null, $dir = 'asc') {
    header('Content-Type: application/json; charset=UTF-8');
    header("Cache-Control: no-cache, must-revalidate"); // HTTP/1.1
    header("Expires: Mon, 26 Jul 1997 05:00:00 GMT"); // Date in the past

    if ($dir != 'desc')
        $dir = 'asc';
    $record_count = ___c($allRecords);
    if ($end == 0)
        $end = $record_count;
    if ($total_item_count == 0)
        $total_item_count = $record_count;

    // Create return value
    $returnValue = array(
      'status' => intval($status),
      'recordsReturned' => intval($record_count),
      'totalRecords' => intval($total_item_count),
      'start' => intval($start),
      'end' => intval($end),
      'itemsperpage' => intval($end - $start),
      'sortby' => $sort,
      'sortdirection' => $dir,
      'data' => $allRecords
    );

    return json_encode($returnValue, JSON_NUMERIC_CHECK | JSON_PRESERVE_ZERO_FRACTION);
}

//_________________________________________________________________________//

function restore_mysql_backup_from_file($filename) {
    global $SystemDB;

    $templine = '';
    $lines = file($filename);
    foreach ($lines as $line) {
        //echo $line;
        if (substr($line, 0, 2) == '--' || empty($line))
            continue;
        $templine .= $line;
        // If it has a semicolon at the end, it's the end of the query
        if (substr(trim($line), -1, 1) == ';') {
            $SystemDB->query($templine);
            $templine = '';
        }
    }
}

/**
 * Returns true if the file is an animated gif
 * From Frank's comment in https://www.php.net/manual/en/function.imagecreatefromgif.php
 * @param string $filename
 * @return boolean
 */
function is_animated_gif($filename) {
    if (filesize($filename) < 10 || !is_readable($filename))
        return false;
    if(!($fh = @fopen($filename, 'rb')))
        return false;
    $count = 0;
    //an animated gif contains multiple "frames", with each frame having a
    //header made up of:
    // * a static 4-byte sequence (\x00\x21\xF9\x04)
    // * 4 variable bytes
    // * a static 2-byte sequence (\x00\x2C)

    // We read through the file til we reach the end of the file, or we've found
    // at least 2 frame headers
    while(!feof($fh) && $count < 2) {
        $chunk = fread($fh, 1024 * 100); //read 100kb at a time
        $count += preg_match_all('#\x00\x21\xF9\x04.{4}\x00[\x2C\x21]#s', $chunk, $matches);
    }

    fclose($fh);
    return $count > 1;
} 
//_________________________________________________________________________//
/**
 * Create an image thumbnail. If dont_resize_if_smaller flag is specified and the source image is smaller than the specified
 * thumbnail width and height then it will just be copied. Quality is a number between 1 - 100
 * @param string $src
 * @param string $dest
 * @param int $thumb_width
 * @param int $thumb_height
 * @param bool $dont_resize_if_smaller
 * @param int $quality
 * @return boolean
 */
function create_image_thumbnail($src, $dest, $thumb_width = 120, $thumb_height = 120, $dont_resize_if_smaller = false, $quality = 100) {

        $temporarily_raised_memory_limit = false;
        $original_memory_limit = ini_get('memory_limit');
        $real_memory_limit = __convert_unit_prefix_to_bytes($original_memory_limit);
        
        $image_info = getimagesize($src);
        $img_width = $image_info[0];
        $img_height = $image_info[1];
        $img_mimetype = $image_info['mime'];
        $required_memory = round( ($img_height * $img_width * 3 * 2));
        
        if ($required_memory > $real_memory_limit)
        {
            
            $temporarily_raised_memory_limit = true;
            $raised_result = @ini_set('memory_limit', $required_memory);
            if (!$raised_result)
                return false;
        }
        // check if it's animated gif and then return
        // hard code limit 3Mb for animated GIF
        if (is_animated_gif($src) && filesize($src) < 3000000) 
        {
            require_once 'gifanim.inc.php';
            $anim_gif_resizer = new \SCHLIX\AnimatedGIFResizer($src, true);
            $result = $anim_gif_resizer->resize($dest, $thumb_width, $thumb_height, true, false);
            if ($temporarily_raised_memory_limit)
                @ini_set('memory_limit', $original_memory_limit);
            return $result;
            
        }
        switch ($img_mimetype) {
            case 'image/jpeg':$base_img = @ImageCreateFromJPEG($src);
                break;
            case 'image/gif':$base_img = @ImageCreateFromGIF($src);
                break;
            case 'image/png':$base_img = @ImageCreateFromPNG($src);
                break;
            default:$base_img = null;
                break;
        }
        
        if ($dont_resize_if_smaller && ($thumb_width >= $img_width) && ($thumb_height >= $img_height))
        {
            return copy($src, $dest);
        }

        $ext = pathinfo($src, PATHINFO_EXTENSION);
        $new_ext = pathinfo($dest, PATHINFO_EXTENSION);

        if ($base_img == null)
        {
            if ($temporarily_raised_memory_limit)
                @ini_set('memory_limit', $original_memory_limit);
            return false; 
        }
        
        // Work out which way it needs to be resized
        $img_width_per = $thumb_width / $img_width;
        $img_height_per = $thumb_height / $img_height;

        if ($img_width_per <= $img_height_per || $img_width > $thumb_width) {
            $thumb_width = $thumb_width;
            $thumb_height = intval($img_height * $img_width_per);
        } else {
            $thumb_width = intval($img_width * $img_height_per);
            $thumb_height = $thumb_height;
        }
        if ($img_width <= $thumb_width && $img_height <= $thumb_height) {

            $thumb_height = $img_height;
            $thumb_width = $img_width;
        }
        if ($thumb_width == 0)
            $thumb_width++;
        if ($thumb_height == 0)
            $thumb_height++;
        $thumb_img = ImageCreateTrueColor($thumb_width, $thumb_height);
        if ($image_info[2] == IMAGETYPE_GIF || $image_info[2] == IMAGETYPE_PNG) {
            imageAlphaBlending($thumb_img, false);
            imageSaveAlpha($thumb_img, true);
            $transparent = imagecolorallocatealpha($thumb_img, 255, 255, 255, 127);
            @imageFilledRectangle($thumb_img, 0, 0, $thumb_width, $thumb_height, $transparent);
        }

        ImageCopyResampled($thumb_img, $base_img, 0, 0, 0, 0, $thumb_width, $thumb_height, $img_width, $img_height);

        
        switch (strtolower($new_ext))
        {
            case 'png':
                imageAlphaBlending($thumb_img, false);
                imageSaveAlpha($thumb_img, true);
                $result = ImagePNG($thumb_img, $dest, 8);
                break;
                
            case 'jpeg':
            case 'jpg':$result = ImageJPEG($thumb_img, $dest, $quality);break;
            case 'gif':$result = ImageGIF($thumb_img, $dest);break;
        }
        // Clean up our images
        ImageDestroy($base_img);
        ImageDestroy($thumb_img);
        if ($temporarily_raised_memory_limit)
            @ini_set('memory_limit', $original_memory_limit);
        
        return $result;
}


/**
 * Create .htaccess and web.config to restrict folder viewing
 * @param string $folder
 * @return boolean
 */
function create_htaccess_restrict_view($folder) {
    $htaccess = "AuthType Basic\n" .
        "AuthName \"Restricted Area\"\n" .
        "require valid-user\n";


    $webconfig = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" .
        "<configuration>\n" .
        "<system.web>\n" .
        "<authorization>\n" .
        "<deny users =\"*\" />\n" .
        "</authorization>\n" .
        "</system.web>\n" .
        "</configuration>\n";

    if (is_dir($folder)) {
        $filename1 = remove_multiple_slashes($folder . '/.htaccess');
        $filename2 = remove_multiple_slashes($folder . '/web.config');
        return (file_put_contents($filename1, $htaccess) && file_put_contents($filename2, $htaccess));
    } else
        return FALSE;
}

function create_directory_if_not_exists($main_folder, $log_if_error = true)
{
    $result = true;
    if (!is_dir($main_folder)) {
        if (!@mkdir($main_folder, 0755, true)) {
            $result = false;
            if ($log_if_error)
            {
                global $SystemLog;
                if ($SystemLog) // in case it has not been instantiated
                    $SystemLog->error("Unable to create folder {$main_folder}");
            }
        }
    }
    return $result;
}

/**
 * Create sub-site folder skeleton. Prefix is usually SCHLIX_SITE_HTTPBASE
 * @param string $prefix
 * @param string $sitename
 * @return boolean
 */
function create_site_skeleton($prefix, $sitename) {

    $subfolders = array
      ('/apps',
      '/blocks',
      '/macros',
      '/themes',
      '/wysiwygeditors',
      '/wysiwygeditors/tinymce4',
      '/wysiwygeditors/tinymce4/plugins',
      '/wysiwygeditors/ckeditor4',
      '/wysiwygeditors/ckeditor4/plugins',
      '/data', // for private stuff
      '/data/user',
      '/data/user/avatars',
      '/data/user/avatars/original',
      '/data/user/avatars/large',
      '/data/user/avatars/medium',
      '/data/user/avatars/small',
      '/data/user/private',
      '/data/user/public',
      '/data/private',
      '/data/private/mail',
      '/data/private/logs',
      '/data/private/downloads',
      '/data/private/attachments',
      '/data/private/tmp',
      '/data/public',
      '/cache',
      '/cache/generic',
      '/cache/page',
      '/cache/sql',
      '/cache/thumbnails',
      '/media',
      '/media/articles',      
      '/media/images',
      '/media/images/clippings',      
      '/media/downloads',
      '/media/pdf',
      '/media/docs',
      '/media/imagegallery',
      '/media/audios',
      '/media/videos',
      '/media/slideshow');


    if (empty($sitename))
        return FALSE;
    
    $main_folder = $prefix . '/web/' . $sitename;
    $result = create_directory_if_not_exists($main_folder);
    if ($result)
    {
        foreach ($subfolders as $subfolder) {
            $target = $main_folder . $subfolder;
            $result = create_directory_if_not_exists($target);
        }
    }
    create_htaccess_restrict_view($main_folder . '/cache/sql');
    create_htaccess_restrict_view($main_folder . '/cache/page');
    create_htaccess_restrict_view($main_folder . '/cache/generic');
    create_htaccess_restrict_view($main_folder . '/data/private');
    return $result;
}
//_________________________________________________________________________//
/**
 * Returns a UNIX-style file path (with forward slash /) and remove
 * multiple slashes.
 * @param string $path
 * @return string
 */
function normalize_file_path($path) {
    return remove_multiple_slashes(str_replace('\\', '/', $path));
}

//_________________________________________________________________________//
/**
 * This function is an alias of remove_prefix_from_string. This function is used to determine
 * the base path of a URL. e.g. $filename = /var/www/html, $prefix = /var
 * then it would return /www/html
 * @param string $filename
 * @param string $prefix
 * @return string
 */
function remove_prefix_from_file_name($filename, $prefix) {
    return remove_prefix_from_string($filename, $prefix);
}
/**
 * Remove leading prefix from a string. 
 * @param string $str
 * @param string $prefix
 * @return string
 */

function remove_prefix_from_string($str, $prefix) {
    $s = strpos($str, $prefix);
    if (($s == 0) && ($s !== false)) {
        $prefix_length = strlen($prefix);
        return substr($str, $prefix_length, strlen($str) - $prefix_length);
    }
}

/**
 * This is a function designed to replace the base path. Not intended to be 
 * called during page exection. However $SystemDB must be initialized.
 * Please call setupSchlixRuntime() before executing this function
 * @internal
 * @ignore
 * @global \SCHLIX\cmsDatabase $SystemDB
 * @param string $str_old_httpbase
 * @param string $str_new_httpbase
 */
function __force_change_site_httpbase($str_old_httpbase, $str_new_httpbase)
{    
    global $SystemDB;

    $str_old_httpbase = remove_multiple_slashes(trim($str_old_httpbase));
    $str_new_httpbase = remove_multiple_slashes(trim($str_new_httpbase));
    if ($str_old_httpbase == '/' || $str_old_httpbase == '')
    {
        // because the one replaced is "/" only, then add trailing slash
        // for the replacement
        $str_old_httpbase = '/';
        $str_new_httpbase = remove_multiple_slashes($str_new_httpbase.'/');
    } else
    {
        if ($str_new_httpbase != '' && $str_new_httpbase != '/')
        {
            // make sure there's no trailing slash for non "/"
            $str_new_httpbase = rtrim($str_new_httpbase,'/');
        }
    }
    $arr_search = [
        'href="'.$str_old_httpbase => 'href="'.$str_new_httpbase,
        'src="'.$str_old_httpbase => 'src="'.$str_new_httpbase,
    ];
    
    $result = $SystemDB->getQueryResultArray("SHOW TABLES");
    $all_tables = array_map('array_values', $result);
    $__schlix_table_columns = [];
    foreach ($all_tables as $tbl) {
        $ticked_table_name = "`{$tbl[0]}`";
        $sql = "SHOW COLUMNS FROM {$ticked_table_name}";
        $result = $SystemDB->getQueryResultArray($sql);
        $__schlix_table_columns[$tbl[0]] = array_column($result, 'Field');

        if (in_array('summary', $__schlix_table_columns[$tbl[0]])) {
            foreach ($arr_search as $str_search => $str_replace)
                    $SystemDB->searchAndReplace($tbl[0], 'summary', $str_search, $str_replace);
        }
        if (in_array('description', $__schlix_table_columns[$tbl[0]])) {
            foreach ($arr_search as $str_search => $str_replace)
                $SystemDB->searchAndReplace($tbl[0], 'description', $str_search, $str_replace);
        }
    }
    // now change the menu
    $menu = new \App\Core_Menu();
    $menu->refreshAllMenuLinks(true); // force refresh

}
//_________________________________________________________________________//
/**
 * Recursive delete a directory
 * @internal
 * @ignore
 * @param string $dir
 * @return bool
 */
function __del_tree($dir) {
    if ($dir == '/')
        return false;
    $dir = realpath($dir);
    if (!file_exists($dir))
        return true;
    if (!is_dir($dir))
        return @unlink($dir);

    $iterator = new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS);
    foreach (new RecursiveIteratorIterator($iterator, RecursiveIteratorIterator::CHILD_FIRST) as $file) {
        if ($file->isDir())
            @rmdir($file->getPathname());
        else
            @unlink($file->getPathname());
    }
    return @rmdir($dir);
}
            
/**
 * Converts unit in Giga, mega, or kilo to bytes e.g. 1M, 2G, 30K etc
 * @internal
 * @param string $val
 * @return int
 */
function __convert_unit_prefix_to_bytes($val) {
    $val = trim($val);        
    $last = strtoupper($val[strlen($val) - 1]);
    
    $real_val = intval($val);
    
    switch ($last) {
        case 'G':$real_val *= 1024;
        case 'M':$real_val *= 1024;
        case 'K':$real_val *= 1024;
    }
    return $real_val;
}

/**
 * Returns max upload size in bytes
 * @return int
 */
function get_real_max_upload_size() {
    

    $max_upload = __convert_unit_prefix_to_bytes(ini_get('upload_max_filesize'));
    $max_post = __convert_unit_prefix_to_bytes(ini_get('post_max_size'));
    $memory_limit = get_estimated_php_memory_limit();
    return min($max_upload, $max_post, $memory_limit);
}

/**
 * Get the class and function name that called the current method
 * @return string
 */
function get_calling_class_function() {

    //get the trace
    $trace = debug_backtrace();

    // Get the class that is asking for who awoke it
    $class = $trace[1]['class'];

    // +1 to i cos we have to account for calling this function
    for ( $i=1; $i<___c( $trace ); $i++ ) {
        if ( isset( $trace[$i] ) ) // is it set?
        {
            
            $trace_i_class = isset($trace[$i]['class']) ? $trace[$i]['class']  : '';
            $trace_i_function = isset($trace[$i]['function']) ? $trace[$i]['function']  : '';

             if ( $class != $trace_i_class) // is it a different class\
             {
                 return array('class'=> $trace_i_class,'function' =>  $trace_i_function);
             }
        }
    }
}


/**
 * Returns $_GET variable as a boolean
 * @param string $varname
 * @return bool
 */
function fget_bool($varname) {
    return \SCHLIX\cmsHttpInputFilter::bool($_GET, $varname);
}

/**
 * Returns $_POST variable as a boolean
 * @param string $varname
 * @return bool
 */
function fpost_bool($varname) {
    return \SCHLIX\cmsHttpInputFilter::bool($_POST, $varname);
}
        
/**
 * Returns $_GET variable as  alphanumeric
 * @param string $varname
 * @param int $limit
 * @return string|int
 */
function fget_alphanumeric($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::alphanumeric($_GET, $varname, $limit);
}

/**
 * Returns $_POST variable as  alphanumeric
 * @param string $varname
 * @return int
 */
function fpost_alphanumeric($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::alphanumeric($_POST, $varname, $limit);
}


/**
 * Returns $_GET variable as an integer
 * @param string $varname
 * @return int
 */
function fget_int($varname) {
    return \SCHLIX\cmsHttpInputFilter::int($_GET, $varname);
}

/**
 * Returns $_GET variable as an unsigned integer
 * @param string $varname
 * @return int
 */
function fget_uint($varname) {
    return \SCHLIX\cmsHttpInputFilter::uint($_GET, $varname);
}

/**
 * Returns $_POST variable as an integer
 * @param string $varname
 * @return int
 */
function fpost_int($varname) {
    return \SCHLIX\cmsHttpInputFilter::int($_POST, $varname);
}


/**
 * Returns $_POST variable as an unsigned integer
 * @param string $varname
 * @return int
 */
function fpost_uint($varname) {
    return \SCHLIX\cmsHttpInputFilter::uint($_POST, $varname);
}

/**
 * Returns $_GET variable as a float number
 * @param string $varname
 * @return int
 */
function fget_float($varname) {
    return \SCHLIX\cmsHttpInputFilter::float($_GET, $varname);
}

/**
 * Returns $_POST variable as a float number
 * @param string $varname
 * @return int
 */
function fpost_float($varname) {
    return \SCHLIX\cmsHttpInputFilter::float($_POST, $varname);
}
/**
 * Returns $_GET variable as adouble
 * @param string $varname
 * @return int
 */
function fget_double($varname) {
    return \SCHLIX\cmsHttpInputFilter::double($_GET, $varname);
}

/**
 * Returns $_POST variable as double
 * @param string $varname
 * @return int
 */
function fpost_double($varname) {
    return \SCHLIX\cmsHttpInputFilter::double($_POST, $varname);
}

/**
 * Returns the value of $_GET[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fget_string($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string($_GET, $varname, true, $limit);
}

/**
 * Returns the value of $_POST[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fpost_string($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string($_POST, $varname, true, $limit);
}

/**
 * Returns the value of $_GET[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fget_string_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_notags($_GET, $varname, $limit);
}

/**
 * Returns the value of $_POST[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fpost_string_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_notags($_POST, $varname, $limit);
}


/**
 * Returns the value of $_GET[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fget_string_noquotes_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_noquotes_notags($_GET, $varname, $limit);
}

/**
 * Returns the value of $_POST[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fpost_string_noquotes_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_noquotes_notags($_POST, $varname, $limit);
}

/**
 * Returns a sanitized filename from $_GET[$varname]
 * @param string $varname
 * @return string
 */
function fget_filename_only($varname) {
    return \SCHLIX\cmsHttpInputFilter::filename_only($_GET, $varname);
}

/**
 * Returns a sanitized filename from $_POST[$varname]
 * @param string $varname
 * @return string
 */
function fpost_filename_only($varname) {
    return \SCHLIX\cmsHttpInputFilter::filename_only($_POST, $varname);
}

/**
 * Returns a sanitized full path from $_GET[$varname]
 * @param string $varname
 * @return string
 */
function fget_filename_fullpath($varname) {
    return SCHLIX\cmsHttpInputFilter::filename_fullpath($_GET, $varname);
}

/**
 * Returns a sanitized full path from $_POST[$varname]
 * @param string $varname
 * @return string
 */
function fpost_filename_fullpath($varname) {
    return SCHLIX\cmsHttpInputFilter::filename_fullpath($_POST, $varname);
}

/**
 * Returns array from $_POST[$varname]
 * @param string $varname
 * @return array
 */
function fpost_array_any($varname)
{
    return SCHLIX\cmsHttpInputFilter::array_any($_POST, $varname);
}

/**
 * Returns array of int from $_POST[$varname]
 * @param string $varname
 * @return array
 */
function fpost_array_int($varname)
{
        return SCHLIX\cmsHttpInputFilter::array_int($_POST, $varname);
}


/**
 * Returns $_POST variable as an integer
 * @param string $varname
 * @return int
 */
function fpost_date_combobox($varname) {
    return \SCHLIX\cmsHttpInputFilter::date_combobox($_POST, $varname);
}

/////////////////////////////////// PHP8 addition /////////////////////////////
//
/**
 * Returns $_COOKIE variable as a boolean
 * @param string $varname
 * @return bool
 */
function fcookie_bool($varname) {
    return \SCHLIX\cmsHttpInputFilter::bool($_COOKIE, $varname);
}

/**
 * Returns $_SESSION variable as a boolean
 * @param string $varname
 * @return bool
 */
function fsession_bool($varname) {
    return \SCHLIX\cmsHttpInputFilter::bool($_SESSION, $varname);
}
        
/**
 * Returns $_COOKIE variable as  alphanumeric
 * @param string $varname
 * @param int $limit
 * @return string|int
 */
function fcookie_alphanumeric($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::alphanumeric($_COOKIE, $varname, $limit);
}

/**
 * Returns $_SESSION variable as  alphanumeric
 * @param string $varname
 * @return int
 */
function fsession_alphanumeric($varname, $limit = 0) {
    return isset($_SESSION) ? \SCHLIX\cmsHttpInputFilter::alphanumeric($_SESSION, $varname, $limit) : '';
}


/**
 * Returns $_COOKIE variable as an integer
 * @param string $varname
 * @return int
 */
function fcookie_int($varname) {
    return \SCHLIX\cmsHttpInputFilter::int($_COOKIE, $varname);
}

/**
 * Returns $_COOKIE variable as an unsigned integer
 * @param string $varname
 * @return int
 */
function fcookie_uint($varname) {
    return \SCHLIX\cmsHttpInputFilter::int($_COOKIE, $varname);
}

/**
 * Returns $_SESSION variable as an integer
 * @param string $varname
 * @return int
 */
function fsession_int($varname) {
    return isset($_SESSION) ? \SCHLIX\cmsHttpInputFilter::int($_SESSION, $varname) : 0;
}


/**
 * Returns $_SESSION variable as an unsigned integer
 * @param string $varname
 * @return int
 */
function fsession_uint($varname) {
    return isset($_SESSION) ? \SCHLIX\cmsHttpInputFilter::uint($_SESSION, $varname) : 0;
}

/**
 * Returns $_COOKIE variable as a float number
 * @param string $varname
 * @return int
 */
function fcookie_float($varname) {
    return \SCHLIX\cmsHttpInputFilter::float($_COOKIE, $varname);
}

/**
 * Returns $_SESSION variable as a float number
 * @param string $varname
 * @return int
 */
function fsession_float($varname) {
    return isset($_SESSION) ? \SCHLIX\cmsHttpInputFilter::float($_SESSION, $varname) : 0;
}
/**
 * Returns $_COOKIE variable as adouble
 * @param string $varname
 * @return int
 */
function fcookie_double($varname) {
    return \SCHLIX\cmsHttpInputFilter::double($_COOKIE, $varname);
}

/**
 * Returns $_SESSION variable as double
 * @param string $varname
 * @return int
 */
function fsession_double($varname) {
    return isset($_SESSION) ? \SCHLIX\cmsHttpInputFilter::double($_SESSION, $varname) : 0;
}

/**
 * Returns the value of $_COOKIE[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fcookie_string($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string($_COOKIE, $varname, true, $limit);
}

/**
 * Returns the value of $_SESSION[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fsession_string($varname, $limit = 0) {
    
    return isset($_SESSION) ? \SCHLIX\cmsHttpInputFilter::string($_SESSION, $varname, true, $limit) : '';
}

/**
 * Returns the value of $_COOKIE[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fcookie_string_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_notags($_COOKIE, $varname, $limit);
}

/**
 * Returns the value of $_SESSION[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fsession_string_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_notags($_SESSION, $varname, $limit);
}


/**
 * Returns the value of $_COOKIE[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fcookie_string_noquotes_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_noquotes_notags($_COOKIE, $varname, $limit);
}

/**
 * Returns the value of $_SESSION[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fsession_string_noquotes_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_noquotes_notags($_SESSION, $varname, $limit);
}

/**
 * Returns array from $_SESSION[$varname]
 * @param string $varname
 * @return array
 */
function fsession_array_any($varname)
{
    return SCHLIX\cmsHttpInputFilter::array_any($_SESSION, $varname);
}

/**
 * Returns array of int from $_SESSION[$varname]
 * @param string $varname
 * @return array
 */
function fsession_array_int($varname)
{
        return SCHLIX\cmsHttpInputFilter::array_int($_SESSION, $varname);
}


/**
 * Returns $_SERVER variable as a boolean
 * @param string $varname
 * @return bool
 */
function fserver_bool($varname) {
    return \SCHLIX\cmsHttpInputFilter::bool($_SERVER, $varname);
} 
/**
 * Returns $_SERVER variable as  alphanumeric
 * @param string $varname
 * @param int $limit
 * @return string|int
 */
function fserver_alphanumeric($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::alphanumeric($_SERVER, $varname, $limit);
} 

/**
 * Returns $_SERVER variable as an integer
 * @param string $varname
 * @return int
 */
function fserver_int($varname) {
    return \SCHLIX\cmsHttpInputFilter::int($_SERVER, $varname);
}

/**
 * Returns $_SERVER variable as an unsigned integer
 * @param string $varname
 * @return int
 */
function fserver_uint($varname) {
    return \SCHLIX\cmsHttpInputFilter::uint($_SERVER, $varname);
}
        
/**
 * Returns $_SERVER variable as a float number
 * @param string $varname
 * @return int
 */
function fserver_float($varname) {
    return \SCHLIX\cmsHttpInputFilter::float($_SERVER, $varname);
} 
/**
 * Returns $_SERVER variable as adouble
 * @param string $varname
 * @return int
 */
function fserver_double($varname) {
    return \SCHLIX\cmsHttpInputFilter::double($_SERVER, $varname);
} 
/**
 * Returns the value of $_SERVER[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fserver_string($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string($_SERVER, $varname, true, $limit);
}
        
/**
 * Returns the value of $_SERVER[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fserver_string_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_notags($_SERVER, $varname, $limit);
} 

/**
 * Returns the value of $_SERVER[$varname], limited to $limit
 * @param string $varname
 * @param int $limit
 * @return string
 */
function fserver_string_noquotes_notags($varname, $limit = 0) {
    return \SCHLIX\cmsHttpInputFilter::string_noquotes_notags($_SERVER, $varname, $limit);
} 

/**
 * Returns a sanitized filename from $_SERVER[$varname]
 * @param string $varname
 * @return string
 */
function fserver_filename_only($varname) {
    return \SCHLIX\cmsHttpInputFilter::filename_only($_SERVER, $varname);
} 
/**
 * Returns a sanitized full path from $_SERVER[$varname]
 * @param string $varname
 * @return string
 */
function fserver_filename_fullpath($varname) {
    return SCHLIX\cmsHttpInputFilter::filename_fullpath($_SERVER, $varname);
}
////////////////////////////////////////////////////////////////////////////////

/**
 * Record a rate limit action
 * @global \SCHLIX\cmsDatabase $SystemDB
 * @param string $action
 * @param string $detail_string
 * @param int $period
 */
function rate_limit_record($action, $detail_string, $period = 3600)
{
    global $SystemDB;


    $row = $SystemDB->q()->select('id, retry_count')->from('gk_rate_limit')->
            where(['status = 1','AND','ip_address = :ip_address','AND','action = :action','AND','TIMESTAMPDIFF(second, date_created, :current_date_time ) < :period_seconds'])->getQueryResultSingleRow(['ip_address' => get_user_real_ip_address(), 'action' =>str_limit($action, 63), 'current_date_time' => get_current_datetime(), 'period_seconds' => $period ]);
    if ($row)
    {
        
        $retry_count = (int) $row['retry_count'] + 1;
        //echo $retry_count;
        $SystemDB->simpleUpdate('gk_rate_limit', ['retry_count' => $retry_count ], 'id', $row['id']);
    } else 
    {
        $data = [
            'user_id' => fsession_uint('userid'),
            'ip_address' => get_user_real_ip_address(),
            'action' =>str_limit($action, 63),
            'date_created' => get_current_datetime(),
            'action_details' => str_limit ($detail_string, 255),            
            'retry_count' => 1,
            'status' => 1
        ];        
        $SystemDB->simpleInsertInto('gk_rate_limit', $data);
    }
}
/**
 * Returns action count for a certain action
 * @global \SCHLIX\cmsDatabase $SystemDB
 * @param string $action
 * @param int $max_retry
 * @param int $period
 * @return bool
 */
function rate_limit_count($action, $period = 3600)
{
    global $SystemDB;
    
    if(!$SystemDB->tableExists('gk_rate_limit'))
        return 0;
    $result = $SystemDB->q()->select('retry_count')->from('gk_rate_limit')->
            where(['status = 1','AND','ip_address = :ip_address','AND','action = :action','AND','TIMESTAMPDIFF(second, date_created, :current_date_time ) < :period_seconds'])->getQueryResultSingleRow(['ip_address' => get_user_real_ip_address(), 'action' =>str_limit($action, 63), 'current_date_time' => get_current_datetime(), 'period_seconds' => $period ]);    
    
    return isset($result['retry_count']) ? $result['retry_count'] : 0;
    

}      
/**
 * Returns true if rate limit for a certain action has been exceeded
 * @param string $action
 * @param int $max_retry
 * @param int $period
 * @return bool
 */
function rate_limit_exceeded($action, $max_retry, $period = 3600)
{    
    $count = rate_limit_count($action, $period);
    return $count > $max_retry;
}    

/**
 * Check if rate limit for a certain action has been exceeded. Upon the first
 * hit of soft retry number, captcha should be displayed. When hard retry count
 * has been exceeded, the system will automatically halt it.
 * 
 * Max retry is set at a very generous number = 50
 * @param string $action
 * @param int $max_soft_retry
 * @param int $max_hard_retry
 * @param int $period
 * @return bool
 */
function rate_limit_and_halt($action, $max_soft_retry, $max_hard_retry = 50, $period = 3600)
{
    $count = rate_limit_count($action, $period);
    
    $soft_exceeded = $count > $max_soft_retry;
    $hard_exceeded = $soft_exceeded &&  ($count > $max_hard_retry); // hard retry number must always be higher
    
    if ($hard_exceeded) {
        if (is_ajax_request()) {
            ajax_echo(ajax_reply('400', ___('Rate limit exceeded')));
            die;
        } else {
            die('Rate limit exceeded');
        }
    } 
    return $soft_exceeded;
    
}

/**
 * Strip array column
 * @param array $data
 * @param array $fields_to_strip
 * @return array
 */
function strip_array_column($data, $fields_to_strip)
{
    if (___c($data) > 0 &&  ___c($fields_to_strip) > 0)
    {
        foreach ($fields_to_strip as $f)
        {
            if (array_key_exists($f, $data))
                unset($data[$f]);
        }
    }
    return $data;
}

function get_sorted_timezone_data()
{
       function formatTimeZoneOffset($offset, $dst) {
            $hours = $offset / 3600;
            $hour = (int) abs($hours);
            $remainder = $offset % 3600;
            $minutes = (int) abs($remainder / 60);
            $sign = $hours > 0 ? '+' : '-';
            $result = ($hour == 0 AND $minutes == 0) ? 'GMT' : 'GMT' . $sign . str_pad($hour, 2, '0', STR_PAD_LEFT) . ':' . str_pad($minutes, 2, '0');
            //if ($dst==1) $result.=' DST';
            return $result;
        }

        function getTimeZoneOffsetByTimeZoneID($timezone_id) {
            try {
                $time_zone = new DateTimeZone($timezone_id);
                $current_time = new DateTime("now", $time_zone);
                return $time_zone->getOffset($current_time);
            } catch (\Exception $e) {
                return -99;
            }
        }

        $list = DateTimeZone::listAbbreviations();
        $idents = DateTimeZone::listIdentifiers();

        $data = $offset = $added = [];
        foreach ($list as $abbr => $zone_collection) {
            foreach ($zone_collection as $zone) {
                if (!empty($zone['timezone_id'])) {
                    $rawgmt = getTimeZoneOffsetByTimeZoneID($zone['timezone_id']);
                    if ($rawgmt != -99) {
                        $gmt = formatTimeZoneOffset($rawgmt, $zone['dst']);
                        $data[$gmt]['places'][] = $zone['timezone_id'];
                    }
                }
            }
        }

        // cleanup
        foreach (array_keys($data) as $gmt) {
            $data[$gmt]['places'] = array_unique($data[$gmt]['places']);
            sort($data[$gmt]['places']);
        }
        ksort($data);
        return $data;
}

function force_download_file($filename, $override_filename = '')
{
    $pi = mb_pathinfo($filename);
    $filename_only = $override_filename ? $override_filename : $pi['filename'];        
    if (file_exists($filename))
    {
        ob_clean();
        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename='.$filename_only);
        header('Content-Transfer-Encoding: binary');
        header('Expires: 0');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Pragma: public');
        header('Content-Length: ' . filesize($filename));
        ob_clean();
        flush();
        readfile($filename);
        exit;               
    } else
    {
        die ("Cannot download {$filename_only} - file not found ");
    }

}

/**
 * Returns truncated words
 * @param string $article_summary
 * @param string $max_string_length
 * @param bool $add_ellipsis
 * @return string
 */
function truncate_text($article_summary, $max_string_length = 100, $add_ellipsis = true, $strip_tags = true)
{
    if ($strip_tags)
        $article_summary = strip_tags($article_summary);
    $len = mb_strlen( $article_summary);
    $topmax = $max_string_length-1;
    $trunc = min ($len, $topmax);
    //$article_summary_space_position = mb_strpos($article_summary, ' ', $max_string_length-5);
    $article_summary_space_position = mb_strpos($article_summary, ' ', $trunc);
    if ($article_summary_space_position == 0 || $article_summary_space_position > $trunc)
        $article_summary_space_position = $trunc;
    $secondary_headline = mb_substr($article_summary, 0, $article_summary_space_position,'UTF-8');
    if ($add_ellipsis)
        $secondary_headline.= '...';
    return $secondary_headline;
    
}
/**
 * Returns truncated words
 * @deprecated since version 2.2.6-2
 * @param string $article_summary
 * @param string $max_string_length
 * @param bool $add_ellipsis
 * @return string
 */
function get_truncated_word($article_summary, $max_string_length = 100, $add_ellipsis = true)
{
    return split_sentence($article_summary, $max_string_length, $add_ellipsis);
}

/**
 * 
 * @param string $haystack
 * @param string $needle
 * @param int $offset
 * @return int|boolean
 */
function schlix_strpos($haystack, $needle, $offset = 0)
{
    $needle = (string) $needle;
    $haystack = (string) $haystack;
    $len = mb_strlen($haystack);
    if ($offset > $len)
        return FALSE;
    else return mb_strpos ($haystack, $needle, $offset, 'UTF-8');
}
 
/**
 * Returns an array of 'class' and 'namespace' from a full class name
 * @param string $fcn
 * @return array
 */
function get_class_namespace_breakdown($fcn)
{
    $final_class_name = $final_namespace = '';
    if (is_string($fcn))
    {
        $fcn = trim( trim($fcn), '\\');
        if (str_contains( $fcn, '\\'))
        {
            $arr_cn = explode('\\', $fcn);
            $final_class_name = array_pop($arr_cn);
            $final_namespace = implode('\\', $arr_cn);
        } else $final_class_name = $fcn;     
    }
    return ['class' => $final_class_name, 'namespace' => $final_namespace];
}


/**
 * Returns estimated PHP memory limit without reading from /proc or executing any command
 * Please note that if it's -1, then we make an assumption of 1Gb memory limit
 * @return int
 */
function get_estimated_php_memory_limit()
{
    $original_memory_limit = ini_get('memory_limit');
    if ($original_memory_limit == -1)
        $original_memory_limit = 1024*1024*1024;
    $real_memory_limit = __convert_unit_prefix_to_bytes($original_memory_limit);    
    return (int) $real_memory_limit;
}

/**
 * Get readable bytes
 * Source: copied from https://www.php.net/manual/en/function.disk-free-space.php 
 * @param int $bytes
 * @return string
 */
function get_readable_bytes($bytes)
{
    $si_prefix = array( 'bytes', 'KB', 'MB', 'GB', 'TB', 'EB', 'ZB', 'YB' );
    $base = 1024;
    $class = min((int)log($bytes , $base) , count($si_prefix) - 1);
    return sprintf('%1.2f' , $bytes / pow($base,$class)) . ' ' . $si_prefix[$class];
}

function gzip_file_write($filename, $data)
{
    $gz = gzopen($filename.'.gz','w9');
    gzwrite($gz, $data);
    gzclose($gz);
}

function gzip_file_read($filename)
{
    $content = null;
    $ext = pathinfo($filename, PATHINFO_EXTENSION);    
    if (file_exists($filename) && ($ext == 'gz'))
    {
        $file = gzopen($filename, 'rb' ); 
        if ($file) { 
            while (!gzeof($file)) { 
                $content .= gzread($file, 4096); 
            } 
            gzclose($file); 
        } 
        return $content; 

        
    }
    return $content;
    
}

                

function get_list_of_view_files($prefix_dir, $sub_dir, $forbidden_listing)
{
    $dir = $prefix_dir.'/'.$sub_dir;

    if (___c($forbidden_listing) == 0)
        $forbidden_listing = ['config.template.php'];
    $allowed_exts = ['php','js','css'];
    $item_array = null;
    if (is_dir($dir))
    {
        $files = \SCHLIX\cmsDirectoryFilter::getDirectoryIterator($dir, \SCHLIX\cmsDirectoryFilter::FILTER_FILE_ONLY);
        foreach ($files as $item) {

            $file = $item->getFileName();
            $ext = $item->getExtension();
            if ( (in_array($ext, $allowed_exts) && (!in_array($file, $forbidden_listing)) ) && 
                  !str_contains($file, 'admin')  )
            {
                $add = false;
                if ($ext === 'php')
                {
                    if (str_ends_with($file, 'template.php'))
                        $add = true;

                } else $add = true;
                if ($add)
                {
                    $is_in_system = file_exists(SCHLIX_SYSTEM_PATH.'/'.$sub_dir.'/'.$file);
                    $is_in_web = file_exists(CURRENT_SUBSITE_PATH.'/'.$sub_dir.'/'.$file);
                    $is_in_theme = file_exists(CURRENT_THEME_PATH.'/'.$sub_dir.'/'.$file);
                    $file_info = ['filename' => $file, 'ext' => $ext, 'is_in_system' => $is_in_system, 'is_in_web' => $is_in_web, 'is_in_theme' => $is_in_theme];
                    $item_array[] = $file_info;
                }
            }
        }

    }
    if (isset($item_array) && ___c($item_array) > 0)
        sort_associative_array_by_key ($item_array, 'ext');
    return $item_array;
}    


function schlix_check_install_errors()
{
    $error_messages = [];
    if (!extension_loaded('mysqli'))
    {
        $error_messages[] = '###[MySQLi extension cannot be found. Please correct your PHP installation]###';
    }
    if (!extension_loaded('xml'))
    {
        $error_messages[] = '###[SimpleXMLElement (php-xml) cannot be found in your installation]###';
    }
    if (!extension_loaded('mbstring'))
    {
        $error_messages[] = '###[Multi-Byte string (php-mbstring) cannot be found in your installation]###';
    }
    if (!extension_loaded('gd'))
    {
        $error_messages[] = '###[GD Library (php-gd) cannot be found in your installation]###';
    }
    if (!extension_loaded('curl'))
    {
        $error_messages[] = '###[Curl (php-curl) cannot be found in your installation. This is required for SCHLIX Updater]###';
    }
    
    if (___c($error_messages) > 0)
    {
    
        $error_message = implode('<br />', $error_messages);
        include('view.offline.template.php');
    }
}

