<?php

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
//
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//

namespace WysiwygEditor;

class CKEditor4 extends \SCHLIX\cmsWysiwygEditor {

    /**
     * Array of default profiles
     * @var array 
     */
    
    
    protected $default_profiles =
            ['ckeditor4-full' => ['title' => 'CKEditor4 Full Featured', 'profile'=>'full'],
             'ckeditor4-simple' => ['title' => 'CKEditor4 Simple', 'profile'=>'simple'],
             'ckeditor4-limited' => ['title' => 'CKEditor4 Limited (No Media Manager)', 'profile'=>'simple-nomediamanager']];
    
    /**
     * Returns a key/value array from the config. If the config key contains
     * a prefix of str_config_ or bool_config_ or int_config_, then it will
     * be moved into 'processed' key in the result array, otherwise it will be
     * moved to 'unprocessed' key
     * @return array
     */
    public function generateConfigurationArrayFromConfig()
    {
        $config = [];
        if ($this->config)
        {                        
            foreach ($this->config as $key => $value)
            {
                if (strpos ($key,'str_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'str_config_');
                    $trimmed_value = trim($value);
                    if ($trimmed_value)
                    {
                        $add_quote = ((int) $trimmed_value ==  0) && ($key != 'str_config_stylesheetParser_skipSelectors') &&
                            ($key != 'str_config_stylesheetParser_validSelectors') &&
                            (!str_starts_with($trimmed_value, 'function ') && 
                             !str_starts_with($trimmed_value, '['));
                        
                        $config['processed'][$new_key] = $add_quote ? '"'.str_replace('"','\"', $trimmed_value).'"' : $trimmed_value;
                    }
                } elseif (strpos ($key,'bool_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'bool_config_');
                    $config['processed'][$new_key] = $value ? 'true' : 'false';
                } 
                elseif (strpos ($key,'int_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'int_config_');
                    $config['processed'][$new_key] = (int) $value;
                }                
                else
                {
                    $config['unprocessed'][$key] = $value;
                }

            }
            return $config;
        } else return NULL;
    }    

    /**
     * Returns an array of CKEditor default build plugins
     * @return array
     */
    public function getCKEditorSystemPlugins()
    {
        $build_config = __DIR__.'/build-config.js';
        if (file_exists($build_config))
        {
            $content = file_get_contents($build_config);
            $content = str_replace(["\r\n","\t", "\n"," "],'', $content);
            
            $regex = 'plugins\s?:\s?{';
           //$regex = "@(?:(plugins : \{)\s*=\s*(\"[^\"]*\"|'[^']*'|[^\"'<>\s]+)\s+)+@isU";        
            $regex = "@(\S+)=(\"|'|)(.*)(\"|'|>)@isU";
            $regex = "@(plugins:\{)(.*)(\})@isU";
            preg_match_all($regex, $content, $matches);
            if ($matches[2][0])
            {
                $arr_plugins = explode(',',str_replace([':1','',"'"],'', $matches[2][0]));
                return $arr_plugins;
            
            }            
        }
        return NULL;
    }
    /**
     * Return extension folders
     * @return array
     */
    public function getExtraExtensionButtons()
    {
        $extra_buttons = [];
        
        $user_plugins = $this->getListOfDirBasedExtensions('user', 'plugins', 'plugin.js'); 
        if ($user_plugins)
        {
            foreach ($user_plugins as $plugin_name)
            {
                $folder = $this->getExtensionFullPath('user', 'plugins', $plugin_name);
                $ext_file = $folder.'/plugin.js';
                if (file_exists($ext_file))
                {
                    //'/\.addButton\s?\(\s?\'((?:[a-z][a-z]+))\',\s{1,}\{(.*?)\s{0,}\}\s{0,}\)\s{0,};/is'
                        $f_content = file_get_contents($ext_file);
                        $f_content = str_replace(array("\r\n","\n","\t", '  '),'', $f_content);

                        preg_match_all('/\.addButton\s?\(\s?\'((?:[a-z][a-z]+))\'/is', $f_content, $matches);
                        $match_count = ___c($matches[0]);        
                        if ($match_count > 0)
                        {
                            $extra_buttons[] = trim($matches[1][0]);
                        }      
                }

            }
        }
        return $extra_buttons; 
    }
    
    /**
     * Given $array_config containing 2 keys: unprocessed and processed,
     * process them and move it to $array_config['processed']
     * @param string $array_config
     * @return string
     */
    public function modifyUnprocessedConfig($array_config)
    {
        $array_toolbar = [];
        foreach ($array_config['unprocessed'] as $key => $value)
        {
            switch ($key)
            {
                case 'str_protectedSource':
                    $protected_source = explode("\r\n", $value);
                    if (empty($protected_source))
                        $protected_source = explode("\n", $value); // just in case
                    if (!empty($protected_source))
                    {
                        $ps = [];
                        foreach ($protected_source as $regexp)
                        {
                            $v = trim($regexp);
                            if ($v) $ps[] = $v;
                        }
                        if ($ps)
                        {
                            $str_ps = implode (",", $ps);
                            $array_config['processed']['protectedSource'] = "[". $str_ps."]";
                        }
                    }
                    break;
                case 'str_behaviour_enter_key':
                case 'str_behaviour_shiftenter_key':                    
                    if ( (int) $value >= 1 && (int) $value <= 3)
                    {
                        $the_key = $key == 'str_behaviour_enter_key' ? 'enterMode' : 'shiftEnterMode';
                        $array_config['processed'][$the_key] = $value; // don't use named const like CKEDITOR.ENTER_P 
                    }
                    break;
                case 'str_allowedContent':
                    $value = trim($value);
                    $array_config['processed']['allowedContent'] = ($value) ? $value : "true";
                        
                case 'str_donot_remove_these_elements':
                    if ($value)
                    {
                        $fixed_values = str_replace(' ','', $value);
                        $array_config['processed']['dtd_do_not_remove_these'] = "'".$fixed_values."'";
                    }
                    break;
                case 'str_protect_regex':
                    $value = trim($value);
                    if ($value)
                    {
                        // config.protectedSource.push( "{$value}")
                    }
                    break;
                case 'str_ProcessNumericalEntities':
                    if ($value)
                    {
                        $array_config['processed']['entities_processNumerical'] = "{$value}";
                    }
                    break;
                case 'bool_emailProtection':
                    if ($value == '1')
                    $array_config['processed']['emailProtection'] = "'encode'";
                    
                    break;
                case 'array_plugins':
                    if ($array_config['unprocessed']['array_plugins'])
                    $array_config['processed']['plugins'] = "'".implode (',', $array_config['unprocessed']['array_plugins'])."'";
                    break;
                case 'array_extra_plugins':
                    $system_plugins = [];
                    $user_plugins = [];
                    foreach ($value as $plugin_name)
                    {
                        if (str_starts_with($plugin_name,'||'))
                        {
                            $real_plugin_name = str_replace('||', '', $plugin_name);
                            $plugin_file_prefix = $this->getExtensionFullPath('user', 'plugins',$real_plugin_name);
                            $plugin_url_prefix = $this->getExtensionURLPath('user', 'plugins',$real_plugin_name);
                            $plugin_js_exists = file_exists($plugin_file_prefix.'/plugin.js');
                            if ($plugin_js_exists)
                            {
                                $user_plugins[] = "'{$real_plugin_name}': '{$plugin_url_prefix}'";
                            }
                            $system_plugins[] = $real_plugin_name;
                            
                        } else {
                            $system_plugins[] = $plugin_name;
                        }
                    }
                    if ($user_plugins)
                    {
                        $str_user_plugins = implode(",\n", $user_plugins);
                        $array_config['processed']['schlix_external_plugins'] = "{{$str_user_plugins}}";
                    }
                    $array_config['processed']['extraPlugins'] = "'".implode (',', $system_plugins)."'";
                    break;
                case 'bool_forced_root_block':
                    if ($value == '1')
                    $array_config['processed']['forced_root_block'] = "'p'";
                    break;
            }
        }
        $str_toolbar = '';
        $toolbars = [];
        for ($i = 1; $i <= 4; $i ++)
        {
            $config_key = "array_toolbar{$i}";
            $unprocessed_toolbar = array_key_exists($config_key, $array_config['unprocessed']) ? $array_config['unprocessed'][$config_key] : null;
            if ($unprocessed_toolbar)
            {
                for ($j = 0, $count = ___c($unprocessed_toolbar); $j < $count; $j++)
                {
                    if ($unprocessed_toolbar[$j] == '|') 
                        $unprocessed_toolbar[$j]  = '-';
                     $unprocessed_toolbar[$j] = "'".$unprocessed_toolbar[$j]."'";
                }
                $toolbar_btns = implode(',', $unprocessed_toolbar);
                $fmt = " { name: 'toolbar{$i}', items: [{$toolbar_btns}]}";
                $array_toolbar[] = $fmt;
                
            }
        }        
        if ($array_toolbar)            
            $array_config['processed']['toolbar'] = '['.implode (",\n'/',\n\t", $array_toolbar).']';
        return $this->forceDefaultConfigOptions($array_config['processed']);
    }

    /**
     * Returns an array of possible CSS files that can be used for content_css
     * @return array
     */
    public function getPossibleContentCSSFiles()
    {
        $possible_css_files = [ '/css/editor.css', 'editor.css', '/css/custom.css',  'custom.css', '/css/style.css', '/style.css'];
        foreach ($possible_css_files as $css_file)
        {
            if (file_exists(CURRENT_THEME_PATH.$css_file))
                $extra_css[] =CURRENT_THEME_URL_PATH.$css_file;
        } 
        return $extra_css;
        
    }
    /**
     * The options specified here cannot be modified by user configuration
     * @param array $options
     * @return string
     */
    protected function forceDefaultConfigOptions($options)
    {
        $extra_css = [];
        $app_mediamanager = new \App\Core_MediaManager();
        $app_admin_mediamanager = new \App\Core_MediaManager();
        $url_full_image_list =   "'".SCHLIX_SITE_HTTPBASE."/admin/app/core.mediamanager?frame=1"."'"; // "'".$app_mediamanager->createFriendlyURL("action=main")."'";
        $url_user_image_list = "''";// not implemented yet"'".$app_mediamanager->createFriendlyURL("action=browseimage")."'";
        $url_full_file_list =  "'".SCHLIX_SITE_HTTPBASE."/admin/app/core.mediamanager?frame=1"."'"; //"'".$app_mediamanager->createFriendlyURL("action=browsefiles")."'";
        $url_user_file_list = "''"; // not implemented yet "'".$app_mediamanager->createFriendlyURL("action=browsefiles")."'";     
        
        $options['autoUpdateElement'] = "true";
        $options['customConfig'] = "false";
        $options['disableNativeSpellChecker'] = 'false';
        $options['toolbarGroupCycling'] = "false";
        $options['baseHref'] = "site_httpbase + '/'";
        $possible_css_files = $this->getPossibleContentCSSFiles();
        foreach ($possible_css_files as $css_file)
        {
                $extra_css[] = "'".$css_file."'";
        } 
        $extra_css_only = '';
        if ($extra_css)
        {
            $extra_css_only = implode(',', $extra_css);
            $extra_theme_css_urls = ','.implode(',', $extra_css);
        }
        $skin_name =   get_class(get_current_skin_module());
        $n = (str_ends_with ($skin_name, 'bootstrap4')) ? '4' : '3';
                
        $options['contentsCss'] = "[ site_httpbase + '/system/skins/bootstrap{$n}/bootstrap.min.css', site_httpbase + '/system/fonts/font-awesome/font-awesome.min.css' {$extra_theme_css_urls}]";

        if ($this->config['str_media_permission'])
        {
            $is_privileged = ($this->config['str_media_permission'] == 'full');
            //if ($this->config['str_media_manager'])
            //SCHLIX.CMS.WysiwygHelper.fileBrowser
            if (!empty($this->config['str_media_permission']))
            {
                if ($this->config['str_file_browser'] == 'schlix')
                    $options['filebrowserBrowseUrl'] = $is_privileged ? $url_full_file_list : $url_user_file_list;
                if ($this->config['str_image_browser'] == 'schlix')
                    $options['filebrowserImageBrowseUrl']  = $is_privileged ? $url_full_image_list : $url_user_image_list;
            }
        }
        $code_mirror_path = SCHLIX_SITE_HTTPBASE.get_system_or_user_url_path('/wysiwygeditors/codemirror');
        $options['codemirror'] = "{ code_mirror_path: '{$code_mirror_path}', theme: 'default',lineNumbers: true,lineWrapping: true,matchBrackets: true,autoCloseTags: true,autoCloseBrackets: true,enableSearchTools: true,enableCodeFolding: false,enableCodeFormatting: true,autoFormatOnStart: true,autoFormatOnModeChange: true,autoFormatOnUncomment: true,mode: 'htmlmixed',showSearchButton: true,showTrailingSpace: true,highlightMatches: true,showFormatButton: true,showCommentButton: true,showUncommentButton: true,showAutoCompleteButton: true}";
                
        return $options;
    }
    /**
     * Modify data before save item
     * @global \App\Users $CurrentUser
     * @param array $datavalues
     * @return array
     */
    public function onModifyDataBeforeSaveConfig($datavalues) {
        $datavalues = parent::onModifyDataBeforeSaveConfig($datavalues);
        
        //$datavalues['options'] = serialize($datavalues['options']);
        return $datavalues;
    }
    
    /**
     * After config has been saved
     * @param int $profile_id
     */
    public function onAfterSaveConfig($profile_id) 
    {
        parent::onAfterSaveConfig($profile_id);
        return;
    }    
    
    /**
     * Returns an array of default config key/value array. Profile name can be simple, full, etc...
     * @param string $reset_profile_name
     * @return array
     */
    public function getDefaultProfileResetSettings($reset_profile_name)
    {
        $default_full_profile = [
     'array_plugins' => [
         'a11ychecker', 'basicstyles','blockquote','clipboard','codesnippet', 'colorbutton','colordialog','contextmenu','copyformatting', 
         'dialogadvtab',  'div','elementspath', 'embed', 'embedbase', /* buggy 'emoji', */ 'enterkey','entities', 'filebrowser','find','floatingspace','font','format',
         'forms', 'horizontalrule','htmlwriter','iframe','iframedialog', 'image','indentblock','indentlist','justify','language','link','list',
         'liststyle','magicline', 'maximize','onchange','pagebreak','pastefromword','pastefromlibreoffice', 'pastetext',
         'preview','removeformat','resize','save','selectall','showblocks','showborders','sourcearea','specialchar','stylescombo',
         'tab','table','tabletools','toolbar','undo','widget','wysiwygarea'],
    'array_extra_plugins' => ['codemirror', 'fontawesome','schlixcmsbrowser'],
'array_toolbar1' => ['Cut', 'Copy', 'Paste', 'PasteText', 'PasteFromWord', '|',  
    'Find', 'Replace', 'Undo', 'Redo' , '|','Format', 'Styles',  'TextColor', 'BGColor' , '|', 
    'CreateDiv', 'Table', '|', 'SpecialChar',  'CodeSnippet' ,'FontAwesome',/* buggy 'EmojiPanel', */ '|',  'ShowBlocks','|',
     ],
'array_toolbar2' => ['Bold', 'Italic', 'Underline', 'Strike', 'Subscript', 'Superscript', '|', 'RemoveFormat' , '|',
    'JustifyLeft', 'JustifyCenter', 'JustifyRight', 'JustifyBlock', '|','NumberedList', 'BulletedList', '|', 
    'Outdent', 'Indent',  'Blockquote', 'HorizontalRule', '|',  'Link', 'Unlink', 'Anchor' , 'Image', 'EmbedSemantic','|', 
    'Source','A11ychecker'],
'array_toolbar3' => [], // for later ['Form', 'Checkbox', 'Radio', 'TextField', 'Textarea', 'Select', 'Button', 'HiddenField'],
'array_toolbar4' => [],          
    'str_media_permission' => 'full',
    'str_file_browser' => 'schlix',
    'str_image_browser' => 'schlix', 
    'str_donot_remove_these_elements' => 'strike,span,b,i,div,script',
    'str_config_filebrowserWindowHeight' => '500',
    'str_config_filebrowserWindowWidth' => '800',
    'str_config_autoGrow_bottomSpace' => '',
    'str_config_autoGrow_maxHeight' => '800',
    'str_config_autoGrow_minHeight' => '400',
    'bool_config_resizeEnabled' => true,          
  'str_config_height' => '200',
  'str_config_width' => 'auto',
  'str_config_removeDialogTabs' => 'link:upload;image:upload',
  'bool_config_disableNativeTableHandles' => true,
  'bool_config_autoGrow_onStartup' => true,
  'bool_config_toolbarStartupExpanded' => true,
  'bool_config_toolbarCanCollapse' => false,          
  'str_config_schlix_selector' => 'wysiwyg',          
    'str_schlix_function_init' => '// please keep this function so you can save with Apply button
for (var instance in CKEDITOR.instances )
{  
    CKEDITOR.instances[instance].on( \'change\', SCHLIX.CMS.WysiwygHelper.fixAutoUpdateForm);
}'
          
];
        
        $default_simple_profile = [
     
     'array_plugins' => ['basicstyles','blockquote','clipboard', 'colorbutton','colordialog','contextmenu','copyformatting','dialogadvtab','div','elementspath','enterkey','entities','filebrowser','find','floatingspace','font','format','horizontalrule','htmlwriter','iframe','iframedialog','image','indentblock','indentlist','justify','language','link','list','liststyle','magicline','maximize','onchange','pagebreak','pastefromword','pastetext','preview','removeformat','resize','save','selectall','showblocks','showborders','sourcearea','specialchar','stylescombo','tab','table','tabletools','toolbar','undo','widget','wysiwygarea'],
    'array_extra_plugins' => ['codemirror', 'schlixcmsbrowser'],
'array_toolbar1' => ['Bold', 'Italic', 'Underline', 'Strike', 'Subscript', 'Superscript', '|', 'RemoveFormat' , '|','JustifyLeft', 'JustifyCenter', 'JustifyRight', 'JustifyBlock', '|','NumberedList', 'BulletedList', '|', 'Outdent', 'Indent',  'Blockquote', 'HorizontalRule', '|',  'Link', 'Unlink', 'Anchor' , 'Image' ],
'array_toolbar2' => [],          
'array_toolbar3' => [],
'array_toolbar4' => [],          
    'str_media_permission' => 'limited',
    'str_file_browser' => 'schlix',
    'str_image_browser' => 'schlix', 
    'str_donot_remove_these_elements' => 'strike,span,b,i,div',
    'str_config_filebrowserWindowHeight' => '500',
    'str_config_filebrowserWindowWidth' => '800',
    'str_config_autoGrow_bottomSpace' => '',
    'str_config_autoGrow_maxHeight' => '800',
    'str_config_autoGrow_minHeight' => '400',
    'bool_config_resizeEnabled' => true,          
  'str_config_height' => '200',
  'str_config_width' => 'auto',
  'str_config_removeDialogTabs' => 'link:upload;image:upload',
  'bool_config_disableNativeTableHandles' => true,
  'bool_config_autoGrow_onStartup' => true,
  'bool_config_toolbarStartupExpanded' => true,
  'bool_config_toolbarCanCollapse' => false,          
  'str_config_schlix_selector' => 'wysiwyg',          
    'str_schlix_function_init' => '// please keep this function so you can save with Apply button
for (var instance in CKEDITOR.instances )
{  
    CKEDITOR.instances[instance].on( \'change\', SCHLIX.CMS.WysiwygHelper.fixAutoUpdateForm);
}'
];
        
        $default_simple_nomedia_profile = [
     'array_plugins' => ['basicstyles','blockquote','clipboard', 'colorbutton','colordialog','contextmenu','copyformatting','dialogadvtab','div','elementspath','enterkey','entities','filebrowser','find','floatingspace','font','format','horizontalrule','htmlwriter','iframe','iframedialog' ,'indentblock','indentlist','justify','language', 'list','liststyle','magicline','maximize','onchange','pagebreak','pastefromword','pastetext','preview','removeformat','resize','save','selectall','showblocks','showborders','sourcearea','specialchar','stylescombo','tab','table','tabletools','toolbar','undo','widget','wysiwygarea'],
    'array_extra_plugins' => ['fontawesome','schlixcmsbrowser'],
'array_toolbar1' => ['Bold', 'Italic', 'Underline', 'Strike', 'Subscript', 'Superscript', '|', 'RemoveFormat' , '|','JustifyLeft', 'JustifyCenter', 'JustifyRight', 'JustifyBlock', '|','NumberedList', 'BulletedList', '|', 'Outdent', 'Indent', '|','Anchor' ,'|', 'Source'],
'array_toolbar2' => [],          
'array_toolbar3' => [],
'array_toolbar4' => [],                    
    'str_media_permission' => '',
    'str_file_browser' => 'none',
    'str_image_browser' => 'none', 
    'str_config_filebrowserWindowHeight' => '500',
    'str_config_filebrowserWindowWidth' => '800',
    'str_config_autoGrow_bottomSpace' => '',
    'str_config_autoGrow_maxHeight' => '800',
    'str_config_autoGrow_minHeight' => '400',
    'bool_config_resizeEnabled' => true,          
  'str_config_height' => '200',
  'str_config_width' => 'auto',
  'str_config_removeDialogTabs' => 'link:upload;image:upload',
  'bool_config_disableNativeTableHandles' => true,
  'bool_config_autoGrow_onStartup' => true,
  'bool_config_toolbarStartupExpanded' => true,
  'bool_config_toolbarCanCollapse' => false,          
  'str_config_schlix_selector' => 'wysiwyg',          
    'str_schlix_function_init' => '// please keep this function so you can save with Apply button
for (var instance in CKEDITOR.instances )
{  
    CKEDITOR.instances[instance].on( \'change\', SCHLIX.CMS.WysiwygHelper.fixAutoUpdateForm);
}'
];
        
        switch ($reset_profile_name)
        {
            case 'simple':
                return $default_simple_profile;            
            case 'simple-nomediamanager':
                return $default_simple_nomedia_profile;
            case 'full':
            default:
                return $default_full_profile;
        }
        
    }
    
    /**
     * Generate configuration script
     * @param array $config
     * @return string
     */
    public function generateConfigurationScriptFromConfigArray($config)
    {
        if (is_array($config) && !empty($config))
        {
            $arr_config = [];

            foreach ($config as $key => $value)
            {
                if (!str_starts_with($key, 'schlix_function'))
                $arr_config[]= "\t{$key}: {$value}";
            }

            $str_config = "\n".implode(",\n",$arr_config)."";
            $config_var = "var __ckeditor4_config = {{$str_config}";
            return $config_var;        
        } else return '""';
    }
    
    /**
     * Returns the configuration script, uncached
     * @return string
     */
    public function getUncachedConfigurationScript()
    {
        if ($this->config)
        {
            $array_config = $this->generateConfigurationArrayFromConfig();
            
            $processed_config = $this->modifyUnprocessedConfig($array_config);
            $config = $this->generateConfigurationScriptFromConfigArray($processed_config);
            return $config;
        } else return NULL;
    }
    /**
     * View init script
     * @return boolean
     */
    public function viewInitScript()
    {
        $this->config['str_replacement_config'] = array_key_exists('str_replacement_config', $this->config) ? trim ($this->config['str_replacement_config']) : '';
        if (empty($this->config['str_replacement_config']))
        {
            $config = array_key_exists('str_generated_config', $this->config) ?  $this->config['str_generated_config'] : null;
            if (empty($config))
            {
                $config = $this->getUncachedConfigurationScript();
            }
            if (array_key_exists('str_additional_config', $this->config))
                $config.= ",\n".trim($this->config['str_additional_config']);
            $config.="\n};";
        } else
        {
            end_output_buffer();
            $config = $this->config['str_replacement_config'];
            header('Content-type: text/javascript; charset=UTF-8');

            echo $config;return;
        }        
        $extra_function_init = array_key_exists('str_schlix_function_init', $this->config) ? $this->config['str_schlix_function_init'] : '';
        $extra_function_setup = array_key_exists('str_schlix_function_setup', $this->config) ? $this->config['str_schlix_function_setup'] : '';
        
        $local_variables = compact(array_keys(get_defined_vars())); 
        $result = $this->loadTemplateFile('view.wysiwyg.init.js', $local_variables);
        if (!$result)
        {
            echo ___('Cannot load view.wysiwyg.init.js');
        }
        return false;
    }
    

}
