<?php

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
//
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//

namespace WysiwygEditor;

if (class_exists('\SCHLIX\cmsWysiwygEditor'))
{    
class tinyMCE4 extends \SCHLIX\cmsWysiwygEditor {

    protected $forbidden_listing = ['.','..','.svn','.cvs', '.git', '__MACOSX'];
    /**
     * Array of default profiles
     * @var array 
     */
    protected $default_profiles =
            ['tinymce4-full' => ['title' => 'TinyMCE4 Full Featured', 'profile'=>'full'],
             'tinymce4-simple' => ['title' => 'TinyMCE4 Simple', 'profile'=>'simple'],
             'tinymce4-limited' => ['title' => 'TinyMCE4 Limited (No Media Manager)', 'profile'=>'simple-nomediamanager']];
    
    /**
     * Returns a key/value array from the config. If the config key contains
     * a prefix of str_config_ or bool_config_ or int_config_, then it will
     * be moved into 'processed' key in the result array, otherwise it will be
     * moved to 'unprocessed' key
     * @return array
     */
    public function generateConfigurationArrayFromConfig()
    {
        $config = [];
        if ($this->config)
        {
            foreach ($this->config as $key => $value)
            {
                if (strpos ($key,'str_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'str_config_');
                    $trimmed_value = trim($value);
                    if ($trimmed_value)
                    {
                        $add_quote = !str_starts_with($trimmed_value, 'function ') && !str_starts_with($trimmed_value, '[');
                        $config['processed'][$new_key] = $add_quote ? '"'.str_replace('"','\"', $trimmed_value).'"' : $trimmed_value;
                    }
                } elseif (strpos ($key,'bool_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'bool_config_');
                    $config['processed'][$new_key] = $value ? 'true' : 'false';
                } 
                elseif (strpos ($key,'int_config_') !== FALSE)
                {
                    $new_key = remove_prefix_from_string($key, 'int_config_');
                    $config['processed'][$new_key] = (int) $value;
                }                
                else
                {
                    $config['unprocessed'][$key] = $value;
                }

            }
            return $config;
        } else return NULL;
    }    
    /**
     * Verify folder name
     * @param \SCHLIX\cmsDirectoryFilter $items
     * @return array
     */
    protected function verifyDirBasedExtensions(\SCHLIX\cmsDirectoryFilter $items, $file_to_check = '') {
        $forbidden_listing = (is_array($this->forbidden_names)) ? $this->forbidden_names : array();

        foreach ($items as $item) {
            $file = $item->getFileName();
            if ((!in_array($file, $forbidden_listing)) && (strpos($file, 'uninstalled_') === false ))
                $item_array[] = $file;
        }

        return $item_array;
    }

    /**
     * Return extension folders
     * @return array
     */
    public function getExtraExtensionButtons()
    {
        $extra_buttons = [];
        
        $user_plugins = $this->getListOfDirBasedExtensions('user', 'plugins', 'plugin.min.js'); 
        if ($user_plugins)
        {
            foreach ($user_plugins as $plugin_name)
            {
                $folder = $this->getExtensionFullPath('user', 'plugins', $plugin_name);
                $tinymce4_ext_file = $folder.'/plugin.min.js';
                if (file_exists($tinymce4_ext_file))
                {
                    //'/\.addButton\s?\(\s?\'((?:[a-z][a-z]+))\',\s{1,}\{(.*?)\s{0,}\}\s{0,}\)\s{0,};/is'
                        $f_content = file_get_contents($tinymce4_ext_file);
                        $f_content = str_replace(array("\r\n","\n","\t", '  '),'', $f_content);

                        preg_match_all('/\.addButton\s?\(\s?\'((?:[a-z][a-z]+))\'/is', $f_content, $matches);
                        $match_count = ___c($matches[0]);        
                        if ($match_count > 0)
                        {
                            $extra_buttons[] = trim($matches[1][0]);
                        }      
                }

            }
        }
        return $extra_buttons; 
    }
    
    /**
     * Given $array_config containing 2 keys: unprocessed and processed,
     * process them and move it to $array_config['processed']
     * @param string $array_config
     * @return string
     */
    public function modifyUnprocessedConfig($array_config)
    {
        
        foreach ($array_config['unprocessed'] as $key => $value)
        {
            switch ($key)
            {
                case 'array_plugins':
                    $system_plugins = [];
                    $user_plugins = [];
                    foreach ($value as $plugin_name)
                    {
                        if (str_starts_with($plugin_name,'||'))
                        {
                            $real_plugin_name = str_replace('||', '', $plugin_name);
                            $plugin_file_prefix = $this->getExtensionFullPath('user', 'plugins',$real_plugin_name);
                            $plugin_url_prefix = $this->getExtensionURLPath('user', 'plugins',$real_plugin_name);
                            $use_v4 = file_exists($plugin_file_prefix.'/plugin.min.js');
                            $plugin_url = $use_v4 ? $plugin_url_prefix.'/plugin.min.js' : $plugin_url_prefix.'/editor_plugin.js';
                            $user_plugins[] = "'{$real_plugin_name}': '{$plugin_url}'";
                        } else {
                            $system_plugins[] = $plugin_name;
                        }
                    }
                    if ($user_plugins)
                    {
                        $str_user_plugins = implode(",\n", $user_plugins);
                        $array_config['processed']['external_plugins'] = "{{$str_user_plugins}}";
                    }
                    $array_config['processed']['plugins'] = "'".implode (' ', $system_plugins)."'";
                case 'array_toolbar1':
                    if ($value) $array_config['processed']['toolbar1'] = "'".implode (' ', $value)."'"; 
                    break;
                case 'array_toolbar2':
                    if ($value) $array_config['processed']['toolbar2'] = "'".implode (' ', $value)."'"; 
                    break;                    
                case 'array_toolbar3':
                    if ($value) $array_config['processed']['toolbar3'] = "'".implode (' ', $value)."'"; 
                    break;                    
                case 'array_toolbar4':                    
                    if ($value) $array_config['processed']['toolbar4'] = "'".implode (' ', $value)."'"; 
                    break;                                        
                case 'bool_forced_root_block':
                    //if ($value == '1')
                    // Force it this way (2017 - July 7)
                    //$array_config['processed']['forced_root_block'] = "''"; // "'p'";
                    //$array_config['processed']['force_p_newlines'] = "true";
                    break;
            }
        }
            
        return $this->forceDefaultConfigOptions($array_config['processed']);
    }

    /**
     * Returns an array of possible CSS files that can be used for content_css
     * @return array
     */
    public function getPossibleContentCSSFiles()
    {
        $possible_css_files = [ '/css/editor.css', 'editor.css', '/css/custom.css',  'custom.css', '/css/style.css', '/style.css'];
        foreach ($possible_css_files as $css_file)
        {
            if (file_exists(CURRENT_THEME_PATH.$css_file))
                $extra_css[] =CURRENT_THEME_URL_PATH.$css_file;
        } 
        return $extra_css;
        
    }
    
    private function getWords($text) {
            preg_match_all('(\w{3,})u', $text, $matches);
            $words = $matches[0];

            for ($i = count($words) - 1;  $i >= 0; $i--) {
                    // Exclude words with numbers in them
                    if (preg_match('/[0-9]+/', $words[$i])) {
                            array_splice($words, $i, 1);
                    }
            }

            return $words;
    }
    
    public function spellCheck()
    {
        global $CurrentUser;
        
        if (function_exists("pspell_new"))
        {
            $lang = fpost_string('lang', 10);
            $method = fpost_string('method');
            $text = fpost_string('text');

            header('Content-Type: application/json');
            header('Content-Encoding: UTF-8');
            header("Expires: Mon, 26 Jul 1997 05:00:00 GMT");
            header("Last-Modified: " . gmdate("D, d M Y H:i:s") . " GMT");
            header("Cache-Control: no-store, no-cache, must-revalidate");
            header("Cache-Control: post-check=0, pre-check=0", false);
            header("Pragma: no-cache");

            if ($method == "spellcheck")
            {
                try
                {
                    if ($CurrentUser->getCurrentUserID() == 0)
                        throw new \Exception ('Not loged in');
                    if (!$text)
                    {
                        throw new \Exception("Missing input parameter 'text'.");
                    }
                    $words = $this->getWords($text);
                    $plink = pspell_new("en_ca", "", "", "",
                            (PSPELL_FAST | PSPELL_RUN_TOGETHER));
                    $corrections = [];
                    foreach ($words as $word)
                    {
                        $w = trim($word);
                        if (!pspell_check($plink, $w))
                        {
                            $suggestions = pspell_suggest($plink, $w);
                            if (___c($suggestions) > 0)
                            {
                                foreach ($suggestions as $suggest)
                                {
                                    $corrections[$w][] = utf8_encode($suggest);
                                }
                            }
                             
                        }
                    }

                    $result =  ["words" => (object) $corrections];
                } catch (\Exception $e)
                {
                    $result = ["error" =>  $e->getMessage()];
                }
            } else
            {
                $result = ["error" => "Invalid JSON input: $method"];
            }
            echo json_encode((object) $result);
        }
    }

        /**
     * The options specified here cannot be modified by user configuration
     * @param array $options
     * @return string
     */
    protected function forceDefaultConfigOptions($options)
    {
        global $CurrentUser, $SystemConfig;
        
        $current_user_id = $CurrentUser->getCurrentUserID();
        $extra_css = [];
        $app_mediamanager = new \App\Core_MediaManager();
        $app_editormanager = new \App\Core_EditorManager();
        $url_full_image_list = "'".$app_mediamanager->createFriendlyURL("action=full_image_list")."'";
        $url_user_image_list = "'".$app_mediamanager->createFriendlyURL("action=user_image_list")."'";
        $url_full_file_list = "'".$app_mediamanager->createFriendlyURL("action=full_file_list")."'";
        $url_user_file_list = "'".$app_mediamanager->createFriendlyURL("action=user_file_list")."'";     
        
        $options['forced_root_block'] = "''"; // "'p'";
        $options['force_p_newlines'] = "true";
        //$options['auto_focus'] = "''";
        //
        if ($current_user_id > 0)
            $options['spellchecker_rpc_url'] = "'".$app_editormanager->createFriendlyURL('action=spellcheck&editor=tinymce4')."/'"; 
        else 
            $options['browser_spellcheck'] = "true";
        $options['document_base_url'] = "site_httpbase + '/'";
        $options['paste_word_valid_elements'] = '"b,strong,i,em,h1,h2,u,p,ol,ul,li,a[href],span,color,font-size,font-color,font-family,mark"';
        $options['paste_retain_style_properties'] = '"color font-size"';
        $options['relative_urls'] = 'false';
        $options['branding'] = 'false';
        $possible_css_files = $this->getPossibleContentCSSFiles();
        if ($possible_css_files)
        {
            foreach ($possible_css_files as $css_file)
            {
                    $extra_css[] = "'".$css_file."'";
            }
        }
        $extra_css_only = '';
        if ($extra_css)
        {
            $extra_css_only = implode(',', $extra_css);
            $extra_theme_css_urls = ','.implode(',', $extra_css);
        }
        $frontend_skin = $SystemConfig->get('system','str_frontend_skin');        
        $skin_module = get_skin_module($frontend_skin);
        if ($skin_module)
        {
            $skin_css = $skin_module->getDefaultCSS();
        }
        
        $options['content_css'] = "[ site_httpbase + '{$skin_css}', site_httpbase + '/system/fonts/font-awesome/font-awesome.css' {$extra_theme_css_urls}]";

        if (array_key_exists('str_media_permission', $this->config ))
        {
            $is_privileged = ($this->config['str_media_permission'] == 'full');
            //if ($this->config['str_media_manager'])
            //SCHLIX.CMS.WysiwygHelper.fileBrowser
            if ($this->config['str_media_manager']!='none')
                $options['file_browser_callback'] = 'SCHLIX.CMS.WysiwygHelper.browse';       
            if (!empty($this->config['str_media_permission']))
            {
                if ($this->config['str_media_manager_link_list'] == 'schlix')
                    $options['link_list'] = $is_privileged ? $url_full_file_list : $url_user_file_list;
                if ($this->config['str_media_manager_image_list'] == 'schlix')
                    $options['image_list']  = $is_privileged ? $url_full_image_list : $url_user_image_list;
                if ($this->config['str_media_manager_media_list'] == 'schlix')
                    $options['media_list']  = $is_privileged ? $url_full_file_list : $url_user_file_list;
            }
        }
        $code_mirror_path = SCHLIX_SITE_HTTPBASE.get_system_or_user_url_path('/wysiwygeditors/codemirror');
        $options['codemirror'] = "{
                    path: '{$code_mirror_path}',            
                    indentOnInit: true, // Whether or not to indent code on init.
                    config: {           // CodeMirror config object
                       mode: 'application/x-httpd-php',
                       lineNumbers: true,
                    autoCloseBrackets: false,
                    autoCloseTags: false,
                    autoFormatOnStart: false,
                    autoFormatOnUncomment: true,
                    continueComments: true,
                    enableCodeFolding: true,
                    enableCodeFormatting: true,
                    enableSearchTools: true,
                    highlightMatches: true,
                    indentWithTabs: false,
                    lineNumbers: true,
                    lineWrapping: true,
                    matchBrackets: true,
                    matchTags: true,
                    showAutoCompleteButton: true,
                    showCommentButton: true,
                    showFormatButton: true,
                    showSearchButton: true,
                    showTrailingSpace: true,
                    showUncommentButton: true,
                    },
                    jsFiles: [
                      //'codemirror.addons.min.js',  'codemirror.mode.htmlmixed.min.js'
                    ]    
                  }";

        return $options;
    }
    /**
     * Modify data before save item
     * @global \App\Users $CurrentUser
     * @param array $datavalues
     * @return array
     */
    public function onModifyDataBeforeSaveConfig($datavalues) {
        $datavalues = parent::onModifyDataBeforeSaveConfig($datavalues);
        
        //$datavalues['options'] = serialize($datavalues['options']);
        return $datavalues;
    }
    
    /**
     * After config has been saved
     * @param int $profile_id
     */
    public function onAfterSaveConfig($profile_id) 
    {
        parent::onAfterSaveConfig($profile_id);
        return;
    }    
    
    /**
     * Returns an array of default config key/value array. Profile name can be simple, full, etc...
     * @param string $reset_profile_name
     * @return array
     */
    public function getDefaultProfileResetSettings($reset_profile_name)
    {
        $default_full_profile = [
     'array_menubar_top' => ['file','edit','insert','view','format','table','tools'],
     'array_plugins' => ['advlist','anchor','autolink','autoresize','charmap','code','codemirror','codesample','colorpicker','contextmenu', 'compat3x', 'directionality','emoticons','fontawesome','fullscreen','hr','image','imagetools','insertdatetime','link','lists','media','nonbreaking','noneditable','pagebreak','paste','preview','print','save','searchreplace','spellchecker', 'table','template','textcolor','textpattern','visualblocks','visualchars','wordcount'],

    'array_toolbar1' => ['undo', 'redo', '|',  'styleselect', 'forecolor', 'backcolor', '|','insert', '|', 'link', 'unlink', 'anchor', 'image', 'media', 'table','|', 'charmap', 'fontawesome', '|',  'visualblocks',  '|',  'codesample'],

    'array_toolbar2' => ['bold', 'italic', 'underline', 'strikethrough', 'subscript', 'superscript', '|', 'removeformat', '|', 'alignleft', 'aligncenter', 'alignright', 'alignjustify', '|', 'bullist', 'numlist', 'outdent', 'indent', 'blockquote', 'hr','|', 'spellchecker', 'code'],
          
     'bool_config_allow_conditional_comments' => true,'bool_config_autoresize_on_init' => true,'bool_config_elementpath' => true,'bool_config_end_container_on_empty_block' => true,'bool_config_fix_list_elements' => true,'bool_config_force_hex_style_colors' => true,'bool_config_image_advtab' => true,'bool_config_image_dimensions' => false,'bool_config_importcss_append' => true,'bool_config_statusbar' => true, 'bool_config_typeahead_urls' => true,'bool_forced_root_block' => false ,'str_config_autoresize_max_height' => '800','str_config_autoresize_min_height' => '400',                    
     'str_config_extended_valid_elements' => 'strike,span[class|style|id],i[class],div[class],div,div[id|class|style],script[language|type|src]',
     'str_config_importcss_groups' => '[ {title: \'Image styles\', filter: /^(img)\\./},
       {title: \'Heading styles\', filter: /^(h1|h2|h3|h4|h5|h6)\\./},
       {title: \'Table styles\', filter: /^(table|td|tr)\\./},
       {title: \'List styles\', filter: /^(ul|li)\\./},
       {title: \'Block styles\', filter: /^(div|p)\\./},
       {title: \'Link styles\', filter: /^(a)\\./},
       {title: \'Generic styles\'} ]',
     'str_config_init_instance_callback' => 'function (inst) { 
           inst.execCommand(\'mceAutoResize\'); // please enable the autoresize plugin
         }',
     'str_config_skin' => 'lightgray',
     'str_config_schema' => 'html5',
     'str_config_selector' => 'textarea.wysiwyg',
     'str_config_setup' => 'function (editor) {
           editor.on(\'change\', function () {
               __schlix_document_changed = true;
               tinymce.triggerSave(); // this function is necessary for SCHLIX CMS - DO NOT REMOVE THIS LINE!
           });
       }',
       'str_config_width' => '100%','str_media_manager' => 'schlix','str_media_manager_image_list' => 'schlix','str_media_manager_link_list' => 'schlix','str_media_manager_media_list' => 'schlix','str_media_permission' => 'full'];
        
        $default_simple_profile = [
     'bool_config_menubar' => false,
     'array_plugins' => ['advlist','autolink','lists','link','image','charmap','print','preview','anchor','searchreplace','visualblocks','insertdatetime','media','table','contextmenu','paste','codemirror'],
     'array_toolbar1' => ['undo','redo','|','insert','|','styleselect','|','bold','italic','|','alignleft','aligncenter','alignright','alignjustify','|','bullist','numlist','outdent','indent','|','link','image'],
     'bool_config_allow_conditional_comments' => true,'bool_config_autoresize_on_init' => true,'bool_config_elementpath' => true,'bool_config_end_container_on_empty_block' => true,'bool_config_fix_list_elements' => true,'bool_config_force_hex_style_colors' => true,'bool_config_image_advtab' => true,'bool_config_image_dimensions' => false,'bool_config_importcss_append' => true,'bool_config_statusbar' => true,'bool_config_typeahead_urls' => true,'bool_forced_root_block' => true,'str_config_autoresize_max_height' => '800','str_config_autoresize_min_height' => '400',                    
     'str_config_extended_valid_elements' => 'strike,span[class|style|id],i[class],div[class],div,div[id|class|style],script[language|type|src]',
     'str_config_importcss_groups' => '[ {title: \'Image styles\', filter: /^(img)\\./},
       {title: \'Heading styles\', filter: /^(h1|h2|h3|h4|h5|h6)\\./},
       {title: \'Table styles\', filter: /^(table|td|tr)\\./},
       {title: \'List styles\', filter: /^(ul|li)\\./},
       {title: \'Block styles\', filter: /^(div|p)\\./},
       {title: \'Link styles\', filter: /^(a)\\./},
       {title: \'Generic styles\'} ]',
     'str_config_init_instance_callback' => 'function (inst) { 
           inst.execCommand(\'mceAutoResize\'); // please enable the autoresize plugin
         }',
     'str_config_schema' => 'html5',
     'str_config_selector' => 'textarea.wysiwyg',
     'str_config_setup' => 'function (editor) {
           editor.on(\'change\', function () {
               tinymce.triggerSave(); // this function is necessary for SCHLIX CMS - DO NOT REMOVE THIS LINE!
               $(editor.getElement()).trigger(\'change\');
           });
       }',
       'str_config_width' => '100%','str_media_manager' => 'schlix','str_media_manager_image_list' => 'schlix','str_media_manager_link_list' => 'schlix','str_media_manager_media_list' => 'schlix','str_media_permission' => 'limited'];
        
        $default_simple_nomedia_profile = [
     'bool_config_menubar' => false,
     'array_plugins' => ['advlist','autolink','lists','link','image','charmap','print','preview','anchor','searchreplace','visualblocks','insertdatetime','table','contextmenu','paste','codemirror'],
     'array_toolbar1' => ['undo','redo','|','insert','|','styleselect','|','bold','italic','|','alignleft','aligncenter','alignright','alignjustify','|','bullist','numlist','outdent','indent','|','link','image'],
     'bool_config_allow_conditional_comments' => true,'bool_config_autoresize_on_init' => true,'bool_config_elementpath' => true,'bool_config_end_container_on_empty_block' => true,'bool_config_fix_list_elements' => true,'bool_config_force_hex_style_colors' => true,'bool_config_image_advtab' => true,'bool_config_image_dimensions' => false,'bool_config_importcss_append' => true,'bool_config_statusbar' => true,'bool_config_typeahead_urls' => true,'bool_forced_root_block' => true,'str_config_autoresize_max_height' => '800','str_config_autoresize_min_height' => '400',                    
     'str_config_extended_valid_elements' => 'strike,span[class|style|id],i[class],div[class],div,div[id|class|style],script[language|type|src]',
     'str_config_init_instance_callback' => 'function (inst) { 
           inst.execCommand(\'mceAutoResize\'); // please enable the autoresize plugin
         }',
     'str_config_schema' => 'html5',
     'str_config_selector' => 'textarea.wysiwyg',
     'str_config_setup' => 'function (editor) {
           editor.on(\'change\', function () {
               tinymce.triggerSave(); // this function is necessary for SCHLIX CMS - DO NOT REMOVE THIS LINE!
           });
       }',
       'str_config_width' => '100%'];
        
        switch ($reset_profile_name)
        {
            case 'simple':
                return $default_simple_profile;            
            case 'simple-nomediamanager':
                return $default_simple_nomedia_profile;
            case 'full':
            default:
                return $default_full_profile;
        }
        
    }
    /**
     * Returns the configuration script, uncached
     * @return string
     */
    public function getUncachedConfigurationScript()
    {
        if ($this->config)
        {
            $array_config = $this->generateConfigurationArrayFromConfig();
            $processed_config = $this->modifyUnprocessedConfig($array_config);
            $config = $this->generateConfigurationScriptFromConfigArray($processed_config);
            return $config;
        } else return NULL;
    }
    /**
     * View init script
     * @return boolean
     */
    public function viewInitScript()
    {
        $str = $this->getConfig('str_replacement_config', '');
        $this->config['str_replacement_config'] = trim ($str);
        if (empty($this->config['str_replacement_config']))
        {
            $config =  $this->getConfig('str_generated_config');
            if (empty($config))
                $config = $this->getUncachedConfigurationScript();
            $config = $this->getUncachedConfigurationScript();
            if (isset($this->config['str_additional_config']))
                $config.= ",\n".trim($this->config['str_additional_config']);
        } else
        {
            $config = $this->config['str_replacement_config'];
        }
        $local_variables = compact(array_keys(get_defined_vars())); 
        $result = $this->loadTemplateFile('view.wysiwyg.init.js', $local_variables);
        if (!$result)
        {
            echo "alert('Cannot load view.wysiwyg.init.js')";
        }
        return false;
    }
    
    private function removeQuote($s)
    {
        if ($s && (strlen($s) > 1))
        {
            return substr($s,1, strlen($s)-2);
        } else return null;
    }
    
    /**
     * Delete cache file
     * @param int $profile_id
     */
    public function deleteCacheFile($profile_id)
    {
        $cache_file = CURRENT_SUBSITE_PATH . "/cache/generic/tinymce_gzip_prf_{$profile_id}.gz" ;
        if (file_exists($cache_file.'.gz'))
            @unlink($cache_file.'.gz');
        if (file_exists($cache_file.'.js'))
            @unlink($cache_file.'.js');        
    }
    /**
     * Serve compressed GZIP of tinyMCE
     * @param int $profile_id
     */
    public function serveCompressedGZIP($profile_id) {
        $config = null;
        if ($profile_id > 0) {
            $this->loadProfileByID($profile_id);
            if ($this->config) {
                $array_config = $this->generateConfigurationArrayFromConfig();
                $config = $this->modifyUnprocessedConfig($array_config);
            }
        } else {
            return;
        }
        $src_files = array();
        $gzip_support = false;
        $expiresOffset = 12 * 3600; // expires in 12 hours
        $my_dir = dirname(__FILE__);


        $theme = $config['theme'] ? $config['theme'] : 'modern';
        $language = $config['language'] ? $config['language'] : 'en';
        $src_files[] = "tinymce";
        $plugins = explode(' ', $this->removeQuote($config['plugins']));
        $src_files[] = "langs/" . $language;

        // Add plugins
        foreach ($plugins as $plugin) {
            $src_files[] = "plugins/" . $plugin . "/plugin";
            $src_files[] = "plugins/" . $plugin . "/langs/" . $language;
        }

        $src_files[] = "themes/" . $theme . "/theme";
        $src_files[] = "themes/" . $theme . "/langs/" . $language;

        // Process source files
        $valid_files = [];
        foreach ($src_files as $src) {
            //echo $my_dir.'/'.$src . ".min.js\n";
            $fn = "{$my_dir}/{$src}.min.js";
            if (file_exists($fn))
                $valid_files[] = $fn;
        }

        $zlibOn = ini_get('zlib.output_compression') || (ini_set('zlib.output_compression', 0) === false);
        $encodings = (isset($_SERVER['HTTP_ACCEPT_ENCODING'])) ? strtolower($_SERVER['HTTP_ACCEPT_ENCODING']) : "";
        $encoding = preg_match('/\b(x-gzip|gzip)\b/', $encodings, $match) ? $match[1] : "";
        $gzip_support = !empty($encoding) && !$zlibOn && function_exists('gzencode');
        $cache_file = CURRENT_SUBSITE_PATH . "/cache/generic/tinymce_gzip_prf_{$profile_id}". ($gzip_support ? ".gz" : ".js");
        $parent_dir = SCHLIX_SYSTEM_URL_PATH . "/wysiwygeditors/{$this->editor_name}";        
        end_output_buffer();
        header("Content-type: text/javascript");
        header("Vary: Accept-Encoding");  // Handle proxies
        header("Expires: " . gmdate("D, d M Y H:i:s", time() + $expiresOffset) . " GMT");
        header("Cache-Control: public, max-age=" . $expiresOffset);
        if ($gzip_support) header("Content-Encoding: " . $encoding);
        if (file_exists($cache_file)) {
            readfile($cache_file);
            return;
        } else
        {

        $buffer = "// SCHLIX CMS compressed TinyMCE for profile ID {$profile_id} \n";
        $buffer .= "var tinyMCEPreInit={base:'{$parent_dir}',suffix:'.min'};";
        foreach ($valid_files as $file) {
            $fileContents = $this->getJSContent($file);
            $buffer .= $fileContents;
        }
        $buffer .= 'tinymce.each("' . implode(',', $src_files) . '".split(","),function(f){tinymce.ScriptLoader.markDone(tinyMCE.baseURL+"/"+f+".js");});';
if ($gzip_support) $buffer = gzencode($buffer, 9, FORCE_GZIP);
        @file_put_contents($cache_file, $buffer);
        echo $buffer;
        }
    }

    /**
     * Returns the contents of the script file if it exists and removes the UTF-8 BOM header if it exists.
     *
     * @param string
     * @return string 
     */
    private function getJSContent($file) {
            $content = file_get_contents($file);

            // Remove UTF-8 BOM
            if (substr($content, 0, 3) === pack("CCC", 0xef, 0xbb, 0xbf)) {
                    $content = substr($content, 3);
            }

            return $content;
    }

    /**
     * Run Command
     * @param array $command
     */
    public function Run($command)
    {        
        
        switch ($command['do'])
        {
            case 'spellcheck':
                
                $this->spellCheck($command['word']);
            case 'compress':
                $this->serveCompressedGZIP($command['profile']);
                break;
            default: break;
        }
    }

}
}