<?php

namespace App;

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
// 
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
include_once(SCHLIX_ROOT_PATH . '/system/libs/schlix/admin_basic.class.php');
include_once(SCHLIX_ROOT_PATH . '/system/libs/schlix/admin_basic_manager.class.php');


class_alias('\\App\\Core_SCHLIXUpdater_Admin', '\\App\\SCHLIXUpdater_Admin');
class_alias('\\App\\Core_Versioning_Admin', '\\App\\Versioning_Admin');
class_alias('\\App\\Core_Menu_Admin', '\\App\\Menus_Admin');
class_alias('\\App\\Core_ApplicationManager_Admin', '\\App\\Applications_Admin');
class_alias('\\App\\Core_BlockManager_Admin', '\\App\\Blocks_Admin');


class Admin extends \SCHLIX\cmsApplication_Basic {

    protected $errormsg = null;
    protected $report_folder = '';
    protected $malicious_files = [];
    protected $fixed_malicious_files = [];
    protected $data_directories = array(
        'syslangpack' => '/data/private/syslangpack',
        'quarantine' => '/data/private/quarantine',
        'tmp' => '/data/private/syslangpack/tmp',
    );

    /**
     * FS Cache
     * @var \SCHLIX\cmsFSCache 
     */
    protected $fs_cache;
    
    /**
     * Application
     * @var \SCHLIX\cmsApplication_Basic
     */
    protected $app;

    public function __construct()
    {
        parent::__construct('Admin');
        $this->fs_cache = new \SCHLIX\cmsFSCache('generic');
    }

    //_______________________________________________________________________________________________________________//
    private function __setTitlePage($title)
    {
        global $HTMLHeader;
        $str = ($this->app) ? $this->app->getApplicationDescription() : '';
        $HTMLHeader->ADD(\__HTML::TITLE($title . ' ' . $str . ' ' . ___('Administration') . ' | SCHLIX'));
    }

    /**
     * View current active sessions
     * @global \App\Users $CurrentUser
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @return boolean
     */
    protected function viewActiveSessions()
    {

        global $CurrentUser, $SystemDB;
        $admadm = new Admin_Admin;
        if (!$admadm->CheckIfCurrentUserAllowedAccess())
        {
            $CurrentUser->logout();
            die(___('Access to the administration is not allowed for your user group'));
        }

        $seconds = 180;
        $sql = "SELECT id, ip_address, user_agent  FROM `gk_session_handler` WHERE TIMESTAMPDIFF(SECOND, `timestamp`,NOW()) < {$seconds}";
        $active_sessions = $SystemDB->getQueryResultArray($sql);

        $this->loadTemplateFile('view.sessions', compact(array_keys(get_defined_vars())));
        $this->__setTitlePage(___('Active Sessions'));
        return true;
    }

    /**
     * Ensure directory exists
     */
    public function ensureDataDirectoryExists()
    {
        $this->initializeDataDirectories();
    }

    public function viewMainPage()
    {
        global $CurrentUser;
        $admadm = new Admin_Admin;
        if (!$admadm->CheckIfCurrentUserAllowedAccess())
        {
            $CurrentUser->logout();
            die(___('Access to the administration is not allowed for your user group'));
        }
        $_SESSION['backend_login'] = true;
        $latest_version = $this->checkLatestVersion();
        $this->loadTemplateFile('view.main.admin', compact(array_keys(get_defined_vars())));
        $this->__setTitlePage('');
    }

    //_________________________________________________________________________//


    public static function renderBackendMenu()
    {
        global $CurrentAdminAppName, $SystemConfig;

        $main_app_replacement = $SystemConfig->get('system', 'str_main_app');


        $menu_items = self::getAdminBackendMenuArray();
        $menu_str = '<ul class="sidebar-menu" id="top_admin_menu">';
        foreach ($menu_items as $m)
        {
            $submenus = '';
            $sub_has_active = false;
            $url = $m['url'] ? $m['url'] : self::getAdminApp($m['app']);
            if ($m['submenus'])
            {
                $url = 'javascript:void(0)';
                $submenus = "<ul class=\"treeview-menu\" id=\"sub_{$m['app']}\">";

                foreach ($m['submenus'] as $sub)
                {
                    $subli_class = [];
                    if ($sub['type'] == 'divider')
                    {
                        $submenus .= '<li class="menu-divider"><span></span></li>';
                    } else
                    {
                        $suburl = self::getAdminApp($sub['app']);
                        $subicon = $sub['icon'] ? "<i class=\"fa {$sub['icon']}\"></i>" : '';
                        $subimg = $sub['image'] ? "<img src=\"{$sub['image']}\" alt=\"{$sub['app']}\" />" : '';
                        if ($sub['image'])
                            $subli_class[] = 'app_list';
                        // also include sub-applications - May 28, 2017
                        if (($sub['app'] == $CurrentAdminAppName || str_starts_with($CurrentAdminAppName . '.', $sub['app']) ) && $main_app_replacement != $sub['app'])
                        {
                            $subli_class[] = 'active';
                            $sub_has_active = true;
                        }
                        if ($sub['extracss'])
                            $subli_class[] = $sub['extracss'];
                        $subli_class_all = implode(' ', $subli_class);
                        $subli_class_str = $subli_class_all ? " class=\"{$subli_class_all}\"" : '';
                        $submenus .= "<li id=\"menu_{$sub['app']}\"{$subli_class_str}><a href=\"{$suburl}\">{$subicon}{$subimg}&nbsp;{$sub['label']}</a></li>\n";
                    }
                }
                $submenus .= '</ul>';
            }

            $root_li_class = [];
            if ($submenus)
                $root_li_class[] = 'treeview';
            if ($m['app'] && (($CurrentAdminAppName == $m['app']) || $sub_has_active))
                $root_li_class[] = 'active';


            if ($m['icon'])
            {
                $image_or_icon = "<i class=\"fa {$m['icon']}\"></i>";
            } else
            {
                $root_li_class[] = 'app_list';
                $image_or_icon = "<img src=\"{$m['image']}\" alt=\"{$m['app']}\" />";
            }

            if ($m['extracss'])
                $root_li_class[] = $m['extracss'];
            $rootli_class_all = implode(' ', $root_li_class);
            $rootli_class_str = $rootli_class_all ? " class=\"{$rootli_class_all}\"" : '';


            $li_id_main = $m['id'] ? $m['id'] : "menu_{$m['app']}";
            $menu_str .= "<li id=\"{$li_id_main}\"{$rootli_class_str}><a href=\"{$url}\">{$image_or_icon}&nbsp;{$m['label']}</a>{$submenus}</li>\n";
        }
        $menu_str .= '</ul>';
        return $menu_str;
    }

    //_________________________________________________________________________//

    /**
     * Returns an array containing the name of 3rd party applications list
     * @return array
     */
    public static function get3rdPartyApplicationList()
    {
        $app_manager = new \App\Core_ApplicationManager;
        return $app_manager->get3rdPartyApplicationList();
    }

    public function getDataModelURL()
    {
        return '/' . get_application_alias('admin') . '/app/admin';
        // return SCHLIX_SITE_HTTPBASE . "/admin/app/{$this->app_name}";
    }

    public static function getAdminApp($str)
    {
        global $Application;
        return $Application->createFriendlyURL('app=' . $str);
    }

//_______________________________________________________________________________________________________________//
    public function createFriendlyURL($str)
    {
        $final_url = '';
        if (SCHLIX_SEF_ENABLED)
        {
            $command_array = [];
            parse_str($str, $command_array); // output to $command_array
           
            if (is_array($command_array) && array_key_exists('app', $command_array) ) //&& $c_act !== '' )
            {
                foreach ($command_array as $key => $value)
                {
                    $final_url .= "/{$key}/{$value}";
                }
                $final_url = SCHLIX_SITE_HTTPBASE . '/' . get_application_alias($this->app_name) . $final_url;
            } else
                return parent::createFriendlyURL($str);
        }
        return remove_multiple_slashes($final_url);
    }

//_______________________________________________________________________________________________________________//
    public function interpretFriendlyURL($urlpath)
    {

        $get_app = fget_string('app');
        if (SCHLIX_SEF_ENABLED && empty($get_app))// !$_GET['app'])
        {
            $command = [];
            $parsedurl = $this->probeFriendlyURLDestination($urlpath);
            /* $url_array = explode('/',$urlpath);
              $appname = $url_array[1];
              array_splice($url_array,0,2); */
            $url = $parsedurl['url'];
            $url_array = $parsedurl['url_array'];

            switch ($url_array[0])
            {
                case 'active-sessions':
                case 'readnews':
                case 'userlogin':
                case 'login':
                case 'logout': $command['action'] = $url_array[0];
                    break;
                case 'app':
                    $command['action'] = 'load_admin_app';
                    $command['appname'] = $url_array[1];
                    break;
                case 'config':$command['action'] = 'config';
                    break;
                default:
                    if (empty($url_array[0]))
                    {
                        $command['action'] = 'main';
                    } else if ($url_array[0] === 'action')
                    {
                        if ($url_array[1] == 'deleteinstalldir')
                        {
                            $command['action'] = 'deleteinstalldir';
                        }
                    } else
                    {
                        return parent::interpretFriendlyURL($url);
                    }
            }
            return $command;
        } else
        {
            return parent::interpretFriendlyURL($url);
        }
    }

    //_______________________________________________________________________________________________________________//
    protected function translateOptions($the_options)
    {
        //echo HTML_WARNING_OLD_VERSION;
        return parent::translateMetaOptions($the_options);
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Load admin app
     * @global \SCHLIX\cmsAdmin_Basic $Administration
     * @global \App\Core_EditorManager $WYSIWYGEditor
     * @global \App\Users $CurrentUser
     * @global string $CurrentAdminAppName
     * @param string $appname
     * @return bool
     */

    /**
     * Load admin app
     * @global \SCHLIX\cmsAdmin_Basic $Administration
     * @global \App\Core_EditorManager $WYSIWYGEditor
     * @global \App\Users $CurrentUser
     * @global string $CurrentAdminAppName
     * @param string $appname
     * @return bool
     */
    public function loadAdminApp($appname)
    {

        global $Administration;
        global $WYSIWYGEditor;
        global $CurrentUser;
        global $CurrentAdminAppName;
        $errormsg = null;
        $pos = strpos($appname, '.');

        $appname = convert_to_safe_filename($appname);

        $possible_dirs = array(SCHLIX_SITE_PATH, SCHLIX_SYSTEM_PATH);

        if ($pos > 0) // It means there's a submodule
        {
            $classname = str_replace('.', '_', $appname) . '_Admin';
            $file_name = $appname;
            $folder = substr($appname, 0, $pos);
        } else
        {
            $classname = str_replace('.', '_', $appname) . '_Admin';
            $file_name = $folder = $appname;
        }

        $classname = 'App\\' . $classname;

        $possible_files = [];
        if (strpos($file_name, '.') > 0)
        {

            list ($main_app, $sub_app) = explode('.', $file_name);
            $possible_files[] = "/apps/{$folder}/{$sub_app}/{$file_name}.admin.class.php";
        }
        $possible_files[] = "/apps/{$folder}/{$file_name}.admin.class.php";
        $possible_file_found = false;
        foreach ($possible_dirs as $possible_dir)
        {
            foreach ($possible_files as $possible_file)
            {
                if (file_exists($possible_dir . $possible_file))
                {
                    $possible_file_found = true;
                    break;
                }
            }
        }
        if (!$possible_file_found)
        {
            $errormsg = 'Administration class file cannot be found - ' . convert_to_safe_filename($file_name);
            return $this->viewErrorPage($errormsg);
            //return true;
            //die($errormsg);
        }
        $Administration = new $classname($errormsg);
        $Administration->loadApplicationJavascript();
        if ($Administration->CheckIfCurrentUserAllowedAccess())
            $running_app = $Administration->Run();
        else
        {
            $CurrentUser->logout();
            die(___('Access not allowed for your user group'));
        }
        $WYSIWYGEditor->runDefaultFullPrivilegeWysiwygEditor();
        $CurrentAdminAppName = $appname;
        return $running_app;
    }
    
    /**
     * View error message
     * @param string $error_message
     * @return boolean
     */
    private function viewErrorPage($error_message)
    {
        if (!is_ajax_request())
        {            
            //$this->setPageTitle("Error");
            $local_variables = compact(array_keys(get_defined_vars()));
            $this->loadTemplateFile('view.errors', $local_variables);
            return true;
            
        } else 
        {
            echo $error_message;
            return false;
        }
    }

    private function setIntendedAdminAppBeforeLogin()
    {
        $_SESSION['admin_intention'] = filter_input(INPUT_SERVER, 'REQUEST_URI', FILTER_SANITIZE_URL);
        return $this->viewLogin();
    }

    //_________________________________________________________________________//
    function getIntendedAdminApplication()
    {
        $sess_admin_intention = fsession_string('admin_intention');
        
        if (!empty($sess_admin_intention) && ($sess_admin_intention != SCHLIX_SITE_HTTPBASE . '/admin/') && ($sess_admin_intention != SCHLIX_SITE_HTTPBASE . '/admin/login/'))
        {

            $intended_url = $sess_admin_intention;
            unset($_SESSION['admin_intention']);
            $url_redirect = $intended_url;
        } else
        {
            $url_redirect = $this->createFriendlyURL('');
        }
        return $url_redirect;
    }

    /**
     * Checks the latest version
     * @global \SCHLIX\cmsConfigRegistry $SystemConfig
     * @return array|NULL
     */
    public function checkLatestVersion()
    {
        global $SystemConfig;

        $UpdaterAdmin = new \App\Core_SCHLIXUpdater_Admin();
        $latest_version = NULL;
        $last_update_check = $this->getConfig('str_last_update_check');
        $current_time = get_current_datetime();
        $this->setConfig('str_last_update_check', $current_time);
        $update_frequency_days = $SystemConfig->get('system', 'int_system_update_frequency');
        if ($update_frequency_days == 0)
            $update_frequency_days = 3;
        if (is_date($last_update_check))
        {
            $how_many_days_since_last_check = days_difference($last_update_check, $current_time);
            if ($how_many_days_since_last_check > $update_frequency_days)
            {
                $latest_version = $UpdaterAdmin->getLatestCMSUniversalVersion();
            }
        } else
        {
            $latest_version = $UpdaterAdmin->getLatestCMSUniversalVersion();
        }
        if (($latest_version == null) || version_compare(strtoupper($latest_version['version']), strtoupper(SCHLIX_VERSION)) <= 0)
        {
            return NULL;
        }

        return $latest_version;
    }

    /**
     * Returns RSS cache
     * @param string $source
     * @param int $cachetime
     * @return string
     */
    private function getCachedSCHLIXNews($source, $cachetime = 24)
    {

        $cache_key = 'rss' . md5($source);
        $cached_content = $this->fs_cache->get($cache_key);
        if (!$cached_content)
        {
            $fcontent = get_remote_file_content($source);
            $this->fs_cache->set($cache_key, $fcontent, $cachetime);
        } else
        {
            $fcontent = $cached_content;
        }
        return $fcontent;
    }

    /**
     * Load Iframe ext url
     * @param string $source
     */
    public function loadExternalRSS($source)
    {
        global $__forceOverrideSubTemplate;


        $__forceOverrideSubTemplate = "blank.template";
        switch ($source)
        {
            case 'news-ext': $rss_url = 'https://www.schlix.com/cmsservices/action/rss-extensions';
                break;
            case 'news-themes': $rss_url = 'https://www.schlix.com/cmsservices/action/rss-themes';
                break;
            case 'news-main':
            default: $rss_url = 'https://www.schlix.com/cmsservices/action/rss-news';
                break;
        }
        try 
        {
            $rss_content = $this->getCachedSCHLIXNews($rss_url);
        }
        catch (\Exception $exc)
        {
            echo sprintf(___('Network connection error - unable to connect to %s'), 'www.schlix.com');
            RETURN_FULLPAGE;
        }
        if (empty($rss_content))
            return RETURN_FULLPAGE;

        $simplerss = simplexml_load_string($rss_content);
        $this->loadTemplateFile('view.externalrss', compact(array_keys(get_defined_vars())));
        return RETURN_FULLPAGE;
    }

    public function ajxp_UserLogin($command)
    {
        global $CurrentUser;

        $url_admin = $this->getIntendedAdminApplication();

        return $CurrentUser->ajxp_UserLogin(null, $url_admin);

    }

    /**
     * View the login box
     * @global string $__forceOverrideSubTemplate
     * @global Users $CurrentUser
     * @return bool
     */
    public function viewLogin()
    {

        global $__forceOverrideSubTemplate;
        global $CurrentUser;
        global $SystemConfig;

        $__forceOverrideSubTemplate = "login.template";
        $this->__setTitlePage('Login');
        $error_list = [];
        if (is_postback())
            $error_list[] = ___('Direct postback is not allowed');

        if (is_ajax_request())
        {
            return ajax_echo(ajax_reply(456, ___('Unable to authenticate your request. Either your session has expired or you have not logged in. Please refresh this page to login.')));
            
        } else 
        {
            $this->loadTemplateFile('view.adminlogin', compact(array_keys(get_defined_vars())));
            return RETURN_FULLPAGE;
        }
    }

    /**
     * Display a box with info
     * @param int $number
     * @param string $title
     * @param url $link
     */
    public function viewBox($number, $title, $link, $bg_color_class, $icon_class)
    {
        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile('view.box', $local_variables, false);
    }

    /**
     * View incoming Messages
     */
    public function viewIncomingMessages()
    {
        global $SystemDB;

        $sql = "SELECT * FROM gk_contact_messages ORDER BY date_created DESC LIMIT 0,5";
        $incoming_messages = $SystemDB->getQueryResultArray($sql);

        $this->loadTemplateFile('view.inbox', array('incoming_messages' => $incoming_messages), false);
    }

    /**
     * View incoming Messages
     */
    public function getCountOfIncomingMessagesWithinDays($days)
    {
        global $SystemDB;

        $days = (int) $days;
        $sql = "SELECT COUNT(*) AS total_count FROM gk_contact_messages WHERE DATEDIFF(NOW(),`date_created`) <= {$days} ORDER BY date_created DESC";
        $incoming_messages = $SystemDB->getQueryResultSingleRow($sql);
        return $incoming_messages['total_count'];
    }

    /**
     * Display a box with info
     * @param int $number
     * @param string $title
     * @param url $link
     */
    public function viewSystemInfo()
    {

        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile('view.systeminfo', $local_variables, false);
    }

    //_________________________________________________________________________//

    public function getApplicationDescription()
    {
        return 'Main';
    }

    public function checkIfInstallationDirectoryStillExists()
    {
        $local_variables = compact(array_keys(get_defined_vars()));

        if (is_dir(SCHLIX_ROOT_PATH . '/install'))
        {

            $this->loadTemplateFile('view.installdir.warning', $local_variables, false);
            return true;
        }
        return false;
    }

    private function disableMaliciousUser($id)
    {
        global $SystemDB, $SystemLog;

        $id = (int) $id;
        $SystemDB->query("UPDATE gk_user_items SET status = -10 WHERE id = :id", ['id' => $id]);
        $SystemLog->record("Disabling user ID # {$id} because of a malicious action");
    }

    private function getMaliciousUsers()
    {
        global $SystemDB, $CurrentUser;

        $mal_users_final = [];
        $sql = "SELECT id, guid, avatar, username, email_address, last_ip_address FROM gk_user_items WHERE `avatar` IS NOT NULL";
        $all_users = $SystemDB->getQueryResultArray($sql);
        $c = ___c($all_users);
        for ($i = 0; $i < $c; $i++)
        {
            $mal_fn = $all_users[$i]['guid'] . '.php';
            $all_users[$i]['malicious'] = false;
            $possible_mals = [
                        $CurrentUser->getDataFileFullPath('avatar_original', $mal_fn),
                        $CurrentUser->getDataFileFullPath('avatar_large', $mal_fn),
                        $CurrentUser->getDataFileFullPath('avatar_medium', $mal_fn),
                        $CurrentUser->getDataFileFullPath('avatar_small', $mal_fn),
            ];
            foreach ($possible_mals as $mal)
            {
                if (file_exists($mal))
                {

                    $all_users[$i]['malicious'] = true;
                    //$this->malicious_files[] = $mal;
                    $this->addMFBeta($mal);
                }
            }

            $mal_avatar_fn = $all_users[$i]['avatar'];
            $possible_mal_avatars = [
                        $CurrentUser->getDataFileFullPath('avatar_original', $mal_avatar_fn),
                        $CurrentUser->getDataFileFullPath('avatar_large', $mal_avatar_fn),
                        $CurrentUser->getDataFileFullPath('avatar_medium', $mal_avatar_fn),
                        $CurrentUser->getDataFileFullPath('avatar_small', $mal_avatar_fn),
            ];
            foreach ($possible_mal_avatars as $avt)
            {
                if (file_exists($avt))
                {
                    $str_content = file_get_contents($avt);
                    if (str_contains($str_content, 'php'))
                    {
                        $all_users[$i]['malicious'] = true;
                        //$this->malicious_files[] = $avt;
                        $this->addMFBeta($avt);
                    }
                }
            }
            if ($all_users[$i]['malicious'] === true)
            {
                $mal_users_final[] = $all_users[$i];
                $this->disableMaliciousUser($all_users[$i]['id']);
            }
        }
        return $mal_users_final;
    }

    public function getWorldWritableFiles()
    {
        global $CurrentUser, $SystemLog;
        $result = [];
        $unfixable = [];
        $phpscript_owner_id = getmyuid();
        $items = \SCHLIX\cmsDirectoryFilter::getRecursiveDirectoryIterator(SCHLIX_ROOT_PATH, \SCHLIX\cmsDirectoryFilter::FILTER_NONE, ['..', '.']);

        foreach ($items as $item)
        {
            $is_dir = $item->isDir();
            $is_file = $item->isFile();
            if ($is_file || $is_dir)
            {
                $ext = strtolower($item->getExtension());
                $path = $item->getPath();
                $full_fn = remove_multiple_slashes($path . '/' . $item->getFileName());

                $perm = fileperms($full_fn);
                $world_wrtb = ($perm & 0x0002);

                if ($world_wrtb && is_writable($full_fn))
                {
                    $owner = fileowner($full_fn);
                    if ($owner == $phpscript_owner_id)
                    {
                        $new_perm = $perm & ~0x0002 & ~0x0010; // go-w
                        //
                        $fix_result = @chmod($full_fn, $new_perm);
                        // quick fix - Dec 4 - scenario for non-suphp
                        if ($fix_result)
                            $result[] = ['filename' => $full_fn, 'fixed' => $fix_result];
                    }
                }
            }
        }
        return $result;
    }

    /**
     * Quick check - BETA
     * @global \App\Users $CurrentUser
     * @global \SCHLIX\cmsLogger $SystemLog
     */
    public function getImproperPHPFiles()
    {
        global $CurrentUser, $SystemLog;
        $result = [];
        $folders = [
            CURRENT_SUBSITE_PATH . '/media',
            CURRENT_SUBSITE_PATH . '/data',
        ];

        foreach ($folders as $folder)
        {
            $items = \SCHLIX\cmsDirectoryFilter::getRecursiveDirectoryIterator($folder, \SCHLIX\cmsDirectoryFilter::FILTER_FILE_ONLY, ['web.config', '.htaccess', 'README.txt']);

            foreach ($items as $item)
            {
                if ($item->isFile())
                {
                    $ext = strtolower($item->getExtension());
                    $path = $item->getPath();
                    $full_fn = remove_multiple_slashes($path . '/' . $item->getFileName());
                    $is_malicious = false;
                    if ($ext == 'php' || $ext == 'phtml')
                    {
                        if ($folder !== CURRENT_SUBSITE_PATH . '/data')
                        {
                            $is_malicious = true;
                        } else
                        if (str_contains($folder, '/languages'))
                            $is_malicious = (!str_contains($full_fn, 'lang.php'));
                        else
                        {
                            $fc = file_get_contents($full_fn);
                            if (str_contains($fc, 'bin2hex') || str_contains($fc, 'gzinflate'))
                                $is_malicious = true;
                        }
                    } else if ($ext == 'gif' || $ext == 'png' || $ext == 'jpg' || $ext == 'jpeg')
                    {
                        $fc = file_get_contents($full_fn);
                        if (str_contains($fc, '<?php'))
                            $is_malicious = true;
                    }

                    if ($is_malicious)
                    {
                        $this->addMFBeta($full_fn);
                    }
                }
            }
        }
    }

    private function addMFBeta($full_fn)
    {
        //$sha1 = sha1_file($full_fn);
        //$qfn = remove_multiple_slashes($this->report_folder.'/'.$sha1);
        //$fixed_result = @rename($full_fn, $qfn);
        $this->malicious_files[] = ['filename' => $full_fn, 'fixed' => false]; //,'quarantine_filename' => $qfn];
    }

    private function fixMF()
    {
        $c = ___c($this->malicious_files);
        for ($i = 0; $i < $c; $i++)
        {
            $full_fn = $this->malicious_files[$i]['filename'];
            $sha1 = sha1_file($full_fn);
            $qfn = remove_multiple_slashes($this->report_folder . '/' . $sha1);
            $fixed_result = @rename($full_fn, $qfn);

            $this->malicious_files[$i]['fixed'] = $fixed_result;
            $this->malicious_files[$i]['quarantine_filename'] = $qfn;
        }
    }

    public function performSecurityCheckBETA()
    {
        global $CurrentUser, $SystemDB;

        $this->initializeDataDirectories();
        $is_windows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';
        // 1. malicious users
        $mal_users = $this->getMaliciousUsers();
        // 2. world writable
        $world_writables = $is_windows ? NULL : $this->getWorldWritableFiles();
        //PHP Script Owner
        // 3. check php files in /media
        $this->getImproperPHPFiles();

        $ist_mu = ___c($mal_users) > 0;
        $ist_ww = ___c($world_writables) > 0;
        $ist_mf = ___c($this->malicious_files) > 0;
        $maybe_vulnerable = $ist_mu || $ist_ww || $ist_mf;
        if ($maybe_vulnerable)
        {

            $this->report_folder = $this->getDataFileFullPath('quarantine', date('Y-m-d_H_i_s'));
            $main_report_filename = remove_multiple_slashes($this->report_folder . '/report.txt');
            $error = '';
            if (!is_dir($this->report_folder))
                if (!create_directory_if_not_exists($this->report_folder ))
                    $error = "Fatal error: Quarantine storage folder {$this->report_folder} cannot be created";

            if ($ist_mu)
            {
                $report_content .= ___('Malicious users') . "\n";
                $report_content .= var_export($mal_users, true);
            }
            if ($ist_ww)
            {
                $report_content .= ___('World writable files') . "\n";
                $report_content .= var_export($world_writables, true);
            }
            if ($ist_mf)
            {
                $this->fixMF();
                $report_content .= ___('Potentially malicious files') . "\n";
                $report_content .= var_export($this->malicious_files, true);
            }

            file_put_contents($main_report_filename, $report_content);

            $local_variables = compact(array_keys(get_defined_vars()));
            $this->loadTemplateFile('view.security.warning', $local_variables, false);
            return true;
        }
        return false;
    }

    public function checkBasicSecurityIssueBETA()
    {
        global $SystemConfig;


        $last_security_check = $this->getConfig('str_last_basic_security_check');
        $current_time = get_current_datetime();
        $this->setConfig('str_last_basic_security_check', $current_time);
        $security_check = $SystemConfig->get('system', 'int_system_securitycheck_frequency');
        if ($security_check == 0)
            $security_check = 7;
        //$last_security_check = NULL;
        if (is_date($last_security_check))
        {
            $how_many_days_since_last_check = days_difference($last_security_check, $current_time);
            if ($how_many_days_since_last_check > $security_check)
            {
                return $this->performSecurityCheckBETA();
            }
        } else
        {
            return $this->performSecurityCheckBETA();
        }
        return false;
    }

    //_________________________________________________________________________//

    function deleteInstallDirectory($ok)
    {
        if ($ok == 1 && is_dir(SCHLIX_ROOT_PATH . '/install'))
        {
            $newname = SCHLIX_ROOT_PATH . '/install' . mt_rand();
            $result = __del_tree(SCHLIX_ROOT_PATH . '/install'); // @rename (SCHLIX_SITE_PATH.DIRECTORY_SEPARATOR.'install',$newname);

            if ($result)
                $this->errormsg = ___('Installation directory has been removed completely');
            else
                $this->errormsg = 'Cannot rename installation directory. Please check the permission on /install and try again or delete it manually - ' . SCHLIX_ROOT_PATH . '/install';
        }


        $local_variables = compact(array_keys(get_defined_vars()));
        $this->setPageTitle(___('Security Warning'));
        $this->loadTemplateFile('view.deleteinstalldir', $local_variables);
    }

    //_________________________________________________________________________//

    public function checkIfWindowsAndNoSMTP()
    {
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN' && SCHLIX_SMTP_HOST == '')
            echo \__HTML::DIV_start(array('class' => 'alert alert-warning'))
            . \__HTML::P(___('Notice - your SMTP server is not configured. This is okay, but system message/contact mail/registration will not be sent to users'))
            . \__HTML::DIV_end();
    }

    //This function transforms the php.ini notation for numbers (like '2M') to an integer (2*1024*1024 in this case)   

    //_________________________________________________________________________//

    public static function getApplicationIcon($app)
    {
        $arr_default_icons = ['_folder_tools', '_folder_settings', '_folder_apps'];
        $i = array_search($app, $arr_default_icons);
        if ($i !== false)
            return SCHLIX_SYSTEM_URL_PATH . '/images/icons/icon' . $arr_default_icons[$i] . '.png';

        $icon = \App\Core_ApplicationManager::getApplicationIcon($app);
//        if (!$icon)
//            $icon = SCHLIX_SYSTEM_URL_PATH . '/images/icon_plugins.png';
        return $icon;
    }


    public static function getAdminBackendMenuArray()
    {
        global $SystemConfig;

        $third_party_app_list = self::get3rdPartyApplicationList();
        $main_app_replacement = $SystemConfig->get('system', 'str_main_app');
        $frontend_app = [];
        $submenu_third_parth_apps = [
            ['app' => 'core.applicationmanager', 'icon' => 'fa-arrow-right', 'label' => ___('View All')],
            ['app' => 'core.extgallery', 'icon' => 'fa-arrow-right', 'label' => ___('Extension Gallery')],
            ['app' => '', 'type' => 'divider'],
        ];
        foreach ($third_party_app_list as $app)
        {
            $submenu_third_parth_apps[] = ['extracss' => 'color-apps', 'app' => $app['title'], 'label' => $app['app_description'], 'image' => \App\Core_ApplicationManager::getApplicationIcon($app['title'])];
            if ($app['title'] == $main_app_replacement)
            {
                $frontend_app[] = ['extracss' => 'color-frontend-app', 'id' => 'frontend-app', 'app' => $app['title'], 'label' => $app['app_description'], 'image' => \App\Core_ApplicationManager::getApplicationIcon($app['title'])];
            }
        }
        $menu = [
            ['app' => 'html', 'icon' => 'fa-file', 'label' => ___('Web Pages'), 'extracss' => 'color-apps'],
            ['app' => 'blog', 'icon' => 'fa-newspaper-o', 'label' => ___('Blog'), 'extracss' => 'color-apps'],
            ['app' => '_folder_apps', 'icon' => 'fa-puzzle-piece', 'label' => ___('Applications'), 'extracss' => 'color-apps',
                // app submenus
                'submenus' => $submenu_third_parth_apps
            ],
            // end submenu
            ['app' => 'core.blockmanager', 'icon' => 'fa-th-large', 'label' => ___('Blocks'), 'extracss' => 'color-blocks'],
            ['app' => 'core.macromanager', 'icon' => 'fa-indent', 'label' => ___('Macros'), 'extracss' => 'color-macros'],
            ['app' => 'core.menu', 'icon' => 'fa-th-list', 'label' => ___('Menus'), 'extracss' => 'color-menus'],
            ['app' => 'users', 'icon' => 'fa-users', 'label' => ___('Users'), 'extracss' => 'color-users'],
            ['app' => '_folder_tools', 'icon' => 'fa-wrench', 'label' => ___('Tools'), 'extracss' => 'color-tools',
                // tools submenus
                'submenus' =>
                [
                    ['app' => 'mediamanager', 'icon' => 'fa-picture-o', 'label' => ___('Media Manager')],
                    ['app' => 'tag', 'icon' => 'fa-tags', 'label' => ___('Tags')],
                    ['app' => 'core.redirection', 'icon' => 'fa-undo', 'label' => ___('URL Redirection')],
                    ['app' => 'simplesitemap', 'icon' => 'fa-map-signs', 'label' => ___('Site Map')],
                    ['app' => 'core.logviewer', 'icon' => 'fa-info-circle', 'label' => ___('Log Viewer')],
                    ['app' => 'core.error404', 'icon' => 'fa-undo', 'label' => ___('404 Error Log')],                    
                    ['app' => 'core.emailqueue', 'icon' => 'fa-envelope', 'label' => ___('Email Queue')],
                ]
            ],
            // end submenu
            ['app' => '_folder_settings', 'icon' => 'fa-gears', 'label' => ___('Settings'), 'extracss' => 'color-settings',
                // settings submenus
                'submenus' =>
                [
                    ['app' => 'core.schlixupdater', 'icon' => 'fa-shield', 'label' => ___('System Update')],
                    ['app' => 'sitemanager', 'icon' => 'fa-sitemap', 'label' => ___('Site Manager')],
                    ['app' => 'core.emailsetup', 'icon' => 'fa-envelope', 'label' => ___('Email setup')],
                    ['app' => 'core.thememanager', 'icon' => 'fa-television', 'label' => ___('Themes')],
                    ['app' => 'core.editormanager', 'icon' => 'fa-pencil-square', 'label' => ___('WYSIWYG Editors')],
                    ['app' => 'core.emailtemplate', 'icon' => 'fa-envelope-square', 'label' => ___('Email Templates')],
                    ['app' => 'core.cronscheduler', 'icon' => 'fa-clock-o', 'label' => ___('System Scheduler')],
                    ['app' => 'core.httperror', 'icon' => 'fa-exclamation-circle', 'label' => ___('HTTP Error')],
                    ['app' => 'core.bannedip', 'icon' => 'fa-hand-stop-o', 'label' => ___('Banned IP')],
                    ['app' => 'core.mediaheader', 'icon' => 'fa-image', 'label' => ___('Custom Media Header')]
                ]
            ],
            // end submenu
            ['app' => 'help', 'icon' => 'fa-question-circle', 'label' => ___('Help'), 'extracss' => 'color-help'],
            ['app' => '', 'icon' => 'fa-sign-out', 'label' => ___('Logout'), 'url' => SCHLIX_SITE_HTTPBASE . '/admin/logout', 'extracss' => 'color-logout']
        ];


        return ($main_app_replacement != 'html' && $main_app_replacement != 'blog') ? array_merge($frontend_app, $menu) : $menu;
    }
    
    /**
     * Get admin menu - TODO - refactor by Jan 2021 to editable
     * @global \SCHLIX\cmsConfigRegistry $SystemConfig
     * @global string $CurrentAdminAppName
     * @global \SCHLIX\cmsLogger $SystemLog
     * @return array
     */
    public static function getAdminBackendMenuArrayV2()
    {
        global $SystemConfig, $CurrentAdminAppName, $SystemLog;

        $app_manager = new \App\Core_ApplicationManager();
        $third_party_app_list = $app_manager->get3rdPartyApplicationList();
        $main_app_replacement = $SystemConfig->get('system', 'str_main_app');
        $frontend_app = null;
        $current_app = null;
        $submenu_third_parth_apps = [
            ['app' => 'core.applicationmanager', 'icon' => 'fa-arrow-right', 'label' => ___('View All')],
            ['app' => 'core.extgallery', 'icon' => 'fa-arrow-right', 'label' => ___('Extension Gallery')],
                //['app'=>'','type' => 'divider'],
        ];
        $front_docked_apps = ['applications',  'shoperatus', 'html', 'blog', 'blocks', 'macros', 'menus', 'users', 'core.help'];

        foreach ($third_party_app_list as $app)
        {
            $submenu_third_parth_apps[] = ['extracss' => 'color-apps', 'app' => $app['title'], 'label' => $app['app_description'], 'image' => \App\Core_ApplicationManager::getApplicationIcon($app['title'])];
            if ($app['title'] == $main_app_replacement)
            {
                $app_info = $app_manager->getItemByAppName($app['title']);
                $frontend_app = ['extracss' => 'color-frontend-app', 'id' => 'frontend-app', 'app' => $app['title'], 'label' => $app_info['app_description'], 'image' => \App\Core_ApplicationManager::getApplicationIcon($app['title'])];
            }
        }
        $the_frontend_app = null;
        $the_current_app = null;
        if ($main_app_replacement != 'html' && $main_app_replacement != 'blog')
            $the_frontend_app = $frontend_app;
        $master_app_name = $CurrentAdminAppName;
        if (!empty($CurrentAdminAppName) && str_contains($CurrentAdminAppName, "."))
                $master_app_name = substr($CurrentAdminAppName, 0, strpos($CurrentAdminAppName, "."));
        if (!empty($CurrentAdminAppName) && !in_array($master_app_name, $front_docked_apps) && $CurrentAdminAppName != $main_app_replacement)
        {
            $master_app = (str_contains($CurrentAdminAppName, ".")) ? substr($CurrentAdminAppName, 0, strpos($CurrentAdminAppName, ".")) : $CurrentAdminAppName;
            $app_info = $app_manager->getItemByAppName($master_app);
            $the_current_app = ['extracss' => 'color-frontend-app', 'id' => 'frontend-app', 'app' => $CurrentAdminAppName,
                'label' => $app_info['description'], 'image' => \App\Core_ApplicationManager::getApplicationIcon($CurrentAdminAppName)];
        }
        // THIS IS A TEMPORARY CODE - TODO - REPLACE WITH FULLY EDITABLE MENU!!!! - TODO - Sept 22, 2020
        //$shoperatus_icon =  \App\Core_ApplicationManager::getApplicationIcon('shoperatus');        
        //if ($shoperatus_icon && !str_contains($CurrentAdminAppName, 'shoperatus') )
        //{
            $temp_ecommerce_app = ['app' => 'shoperatus', 'icon' => 'fa-shopping-cart', 'label' => ___('Shoperatus'), 'extracss' => 'color-apps'];
            
        //}
        /////////////// 
        $menu = [
            ['app' => '_folder_apps', 'icon' => 'fa-puzzle-piece', 'label' => ___('Applications'), 'extracss' => 'color-apps',
                // app submenus
                'submenus' => $submenu_third_parth_apps
            ],
            $the_current_app,
            $the_frontend_app,
            
            // end submenu
            
            ['app' => 'html', 'icon' => 'fa-file', 'label' => ___('Web Pages'), 'extracss' => 'color-apps'],
            ['app' => 'blog', 'icon' => 'fa-newspaper-o', 'label' => ___('Blog'), 'extracss' => 'color-apps'],
            $temp_ecommerce_app,
            ['app' => '', 'type' => 'divider'],
            // 
            /////////
            ['app' => 'core.blockmanager', 'icon' => 'fa-th-large', 'label' => ___('Block Manager'), 'extracss' => 'color-blocks'],
            ['app' => 'core.macromanager', 'icon' => 'fa-indent', 'label' => ___('Macro Manager'), 'extracss' => 'color-macros'],
            ['app' => 'core.menu', 'icon' => 'fa-th-list', 'label' => ___('Menu'), 'extracss' => 'color-menu'],
            ['app' => 'users', 'icon' => 'fa-users', 'label' => ___('Users'), 'extracss' => 'color-users'],
            ['app' => '_folder_tools', 'icon' => 'fa-wrench', 'label' => ___('Tools'), 'extracss' => 'color-tools',
                // tools submenus
                'submenus' =>
                [
                    ['app' => 'core.mediamanager', 'icon' => 'fa-picture-o', 'label' => ___('Media Manager')],                    
                    ['app' => 'tag', 'icon' => 'fa-tags', 'label' => ___('Tags')],
                    ['app' => 'core.redirection', 'icon' => 'fa-undo', 'label' => ___('URL Redirection')],                    
                    ['app' => 'simplesitemap', 'icon' => 'fa-map-signs', 'label' => ___('Site Map')],
                    ['app' => 'core.logviewer', 'icon' => 'fa-info-circle', 'label' => ___('Log Viewer')],
                    ['app' => 'core.error404', 'icon' => 'fa-undo', 'label' => ___('404 Error Log')],
                    ['app' => 'core.emailqueue', 'icon' => 'fa-envelope', 'label' => ___('Email Queue')],
                    
                ]
            ],
            // end submenu
            ['app' => '_folder_settings', 'icon' => 'fa-gears', 'label' => ___('Settings'), 'extracss' => 'color-settings',
                // settings submenus
                'submenus' =>
                [
                    ['app' => 'core.schlixupdater', 'icon' => 'fa-shield', 'label' => ___('System Update')],
                    ['app' => 'sitemanager', 'icon' => 'fa-sitemap', 'label' => ___('Site Manager')],
                    ['app' => 'core.emailsetup', 'icon' => 'fa-envelope', 'label' => ___('Email setup')],
                    ['app' => 'core.thememanager', 'icon' => 'fa-television', 'label' => ___('Themes')],
                    ['app' => 'core.editormanager', 'icon' => 'fa-pencil-square', 'label' => ___('Editor Manager')],
                    ['app' => 'core.emailtemplate', 'icon' => 'fa-envelope-square', 'label' => ___('Email Templates')],
                    ['app' => 'core.cronscheduler', 'icon' => 'fa-clock-o', 'label' => ___('System Scheduler')],
                    ['app' => 'core.httperror', 'icon' => 'fa-exclamation-circle', 'label' => ___('HTTP Error')],
                    ['app' => 'core.bannedip', 'icon' => 'fa-hand-stop-o', 'label' => ___('Banned IP')],
                    ['app' => 'core.mediaheader', 'icon' => 'fa-image', 'label' => ___('Custom Media Header')]
                ]
            ],
            // end submenu
            ['app' => 'core.help', 'icon' => 'fa-question-circle', 'label' => ___('Help'), 'extracss' => 'color-help']
        ];

        return $menu;
        //return ($main_app_replacement != 'html' && $main_app_replacement != 'blog') ? array_merge($frontend_app, $menu) : $menu;
    }

    public static function renderBackendMenuV2()
    {

        global $CurrentAdminAppName, $SystemConfig;

        $main_app_replacement = $SystemConfig->get('system', 'str_main_app');


        $menu_items = self::getAdminBackendMenuArrayV2();
        $menu_str = '<ul class="schlix-launchbar" id="schlix-launcher">';
        foreach ($menu_items as $m)
        {
            if (!$m)
                continue;
            $m_type = isset($m['type']) ? $m['type'] : null;
            $m_url  = isset($m['url']) ? $m['url'] : null;
            $m_app  = isset($m['app']) ? $m['app'] : null;
            $m_submenus =  array_key_exists('submenus', $m) ? $m['submenus'] : null;
            $m_id =  isset($m['id']) ?  $m['id'] : null;
            //$m_app =  array_key_exists('app', $m) ? $m['app'] : null;
            if ($m_type == 'divider')
            {
                $menu_str .= '<li class="menu-divider"><span></span></li>';
            } else
            {
                $submenus = '';
                $sub_has_active = false;
                $url = $m_url ? $m_url : self::getAdminApp($m['app']);
                if ($m_submenus)
                {
                    $url = 'javascript:void(0)';
                    $submenus = '';
                    $submenus .= '<span class="schlix-launcher-subheader"><span class="schlix-subheader-title">' . ___h($m['label']) . '</span></span>';
                    $submenus .= "<ul class=\"ssm\" id=\"sub_{$m['app']}\">";
                    //$submenus.= '<li class="schlix-launcher-subheader"><span>'.___h($m['label']).'</span></li>';
                    foreach ($m['submenus'] as $sub)
                    {
                        $subli_class = [];
                        $sub_type = isset($sub['type']) ? $sub['type'] : null;
                        $sub_url =  isset($sub['url']) ? $sub['url'] : null;
                        $sub_app =  isset($sub['app']) ? $sub['app'] : null;
                        $sub_extracss =  isset($sub['extracss']) ? $sub['extracss'] : null;
                        
                        if ($sub_type == 'divider')
                        {
                            $submenus .= '<li class="menu-divider"><span></span></li>';
                        } else
                        {
                            $subicon = '';
                            $suburl = self::getAdminApp($sub_app);
                            $app_icon = self::getApplicationIcon($sub_app);
                            $subimg = "<img src=\"{$app_icon}\" alt=\"{$sub_app}\" />";
                            // also include sub-applications - May 28, 2017
                            if (($sub_app == $CurrentAdminAppName || str_starts_with($CurrentAdminAppName . '.', $sub_app) ) && $main_app_replacement != $sub_app)
                            {
                                $subli_class[] = 'active';
                                $sub_has_active = true;
                            }
                            if ($sub_extracss)
                                $subli_class[] = $sub_extracss;
                            $subli_class_all = implode(' ', $subli_class) . ' sub-menu';
                            $subli_class_str = $subli_class_all ? " class=\"{$subli_class_all}\"" : '';
                            if (empty($sub['label']))
                                $sub['label'] = $sub_app;

                            $sub['label'] = ___h($sub['label']);
                            $submenus .= "<li id=\"menu_{$sub['app']}\"{$subli_class_str}><a href=\"{$suburl}\">{$subicon}{$subimg}<span>{$sub['label']}</span></a></li>\n";
                        }
                    }
                    $submenus .= '</ul>';
                }

                $root_li_class = [];
                if ($submenus)
                    $root_li_class[] = 'ssm-folder';
                if ($m['app'] && (($CurrentAdminAppName == $m['app'] || str_starts_with($CurrentAdminAppName . '.', $m['app'])) ))
                    $root_li_class[] = 'active';

                $app_icon = self::getApplicationIcon($m['app']);


                $image_or_icon = "<img src=\"{$app_icon}\" alt=\"{$m['app']}\" />";


                if ($m['extracss'])
                    $root_li_class[] = $m['extracss'];
                $rootli_class_all = implode(' ', $root_li_class);
                $rootli_class_str = $rootli_class_all ? " class=\"{$rootli_class_all}\"" : '';


                $li_id_main = $m_id ? $m_id : "menu_{$m['app']}";
                $a_class = $submenus ? ' class="schlix-launcher-dropdown"' : '';
                if ($app_icon)
                    $menu_str .= "<li id=\"{$li_id_main}\"{$rootli_class_str}><a href=\"{$url}\"{$a_class}>{$image_or_icon}<span>{$m['label']}</span></a>{$submenus}</li>\n";
            }
        }
        $menu_str .= '</ul>';
        return $menu_str;
    }

    /**
     * Logout the current user
     * @global type $CurrentUser
     */
    public function logOut()
    {
        global $CurrentUser;

        $CurrentUser->logout();
        redirect_url($this->createFriendlyURL('login'));
    }

    //_______________________________________________________________________________________________________________//

    /**
     * Main controller
     * @param array $command
     * @return boolean
     */
    public function Run($command)
    {
        global $CurrentUser;

        if ($CurrentUser->adminAuthenticated() === true && $CurrentUser->getCurrentUserID() > 0)
        {

            if ($command['action'] != 'deleteinstalldir')
            {
                if ($this->checkIfInstallationDirectoryStillExists())
                    return true;
                // quick fix - TODO REMOVE - Nov 20 2018
                /* if ($command['action']  == 'main')
                  if ($this->checkBasicSecurityIssueBETA()) return true; */
            }

            switch ($command['action'])
            {
                case 'deleteinstalldir': $this->deleteInstallDirectory(fpost_int('deleteinstalldir'));
                    break;

                case 'login':
                    redirect_url($this->createFriendlyURL(''));
                    break;
                case 'logout':
                    return $this->logout();
                case 'readnews':
                    return $this->loadExternalRSS(fget_string('source'));
                    break;
                case 'active-sessions':
                    return $this->viewActiveSessions();
                case 'load_admin_app':
                    return $this->loadAdminApp($command['appname']);
                    break;

                default: return parent::Run($command);
            }
        } else
        // if not authenticated
        {

            switch ($command['action'])
            {
                case '':
                case 'main':
                case 'login':
                    return $this->viewLogin();
                    break;

                case 'load_admin_app':
                    return $this->setIntendedAdminAppBeforeLogin();
                    break;
                case 'logout':
                    redirect_url($this->createFriendlyURL(''));
                    break;

                default: return parent::Run($command);
            }
        }

        return true;
    }

}
