<?php
namespace App;
/**
 * Core: Captcha - Main Class
 * 
 * Core - Captcha
 * 
 * @copyright 2019 SCHLIX Web Inc
 * @copyright 2011 Drew Phillips - SecurImage 3.6.4  (BSD License)
 * @license GPLv3
 *
 * @package core
 * @version 1.0
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */

class Core_Captcha extends \SCHLIX\cmsApplication_Basic {


    // All of the public variables below are securimage options
    // They can be passed as an array to the Securimage constructor, set below,
    // or set from securimage_show.php and securimage_play.php

    /**
     * Constant for generating a captcha consisting of a simple math problem
     *
     * @var int
     */
    const SI_CAPTCHA_MATHEMATIC = 1;


    protected $iscale = 1;
    protected $image_height = 200;    
    protected $image_width = 80;
    protected $perturbation = 0;
    protected $charset = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    protected $image_bg_color ;
    protected $multi_text_color = [];
    protected $ttf_file ;
            //$this->text_color = $this->hex2rgb(array_rand ($array_colors));
    protected $text_color;						   
    protected $use_multi_text = true;

    protected $use_transparent_text = true;
    protected $text_transparency_percentage = 10; // 100 = completely transparent       
    protected $num_lines = 2;
    protected $line_color;
    protected $noise_level  = 2;
    
    protected $signature_color = [];

    protected $bgimg;
    protected $background_directory;
    protected $captcha_type;
    protected $case_sensitive = false;
    protected $font_ratio;
    protected $image_signature;
    
    protected $securimage_path = null;
    protected $distortion_level = null;
    protected $gdtextcolor;
    protected $gdnoisecolor;    
    protected $gdbgcolor = null;
    protected $noise_color = null;    
    protected $signature_font = null;
    protected $code_length = null;
    protected $code_display = null;
    protected $code_entered = null;
    protected $code = null;
    protected $correct_code = null;
    protected $_timeToSolve = null;
        
    protected $tmpimg = null;
    protected $im = null;
            
    protected $gdlinecolor;
    protected $gdsignaturecolor;
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct("Captcha");
        
        $this->securimage_path = dirname(__FILE__);
            

        $array_colors = array($this->rgb(27,78,181), // blue
                $this->rgb(22,163,35), // green
                $this->rgb(214,36,7),
                        $this->hex2rgb('#FF4500'),
                        $this->hex2rgb('#8A2BE2')
                        );  // red
        $array_fonts = [
            'kranky/kranky-v10-latin-regular.ttf', 
            'shadows-into-light-two/shadows-into-light-two-v7-latin-regular.ttf',
            'reenie-beanie/reenie-beanie-v10-latin-regular.ttf'
            
        ];
        $this->image_width = 200;
        $this->image_height = 80;
        $this->distortion_level = 0;
        $this->charset = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
        $this->image_bg_color = $this->hex2rgb("#FFFFFF");
        $this->multi_text_color = array($this->hex2rgb("#3399ff"),
                                       $this->hex2rgb("#3300cc"),
                                       $this->hex2rgb("#3333cc"),
                                       $this->hex2rgb("#6666ff"),
                                       $this->hex2rgb("#99cccc")
                                       );
        //$this->ttf_file =  $this->securimage_path.'/'.$array_fonts[mt_rand (0,sizeof($array_fonts)-1)];							   
        $this->ttf_file = SCHLIX_SYSTEM_PATH.'/fonts/'.$array_fonts[mt_rand (0,sizeof($array_fonts)-1)];							   
        //$this->text_color = $this->hex2rgb(array_rand ($array_colors));
        $this->text_color  = $array_colors[mt_rand (0,sizeof($array_colors)-1)]	;						   
        $this->use_multi_text = true;
        
        $this->use_transparent_text = true;
        $this->text_transparency_percentage = 10; // 100 = completely transparent       
        $this->num_lines = 2;
        $this->line_color = $this->rgb (rand (0xA0, 0xEA), rand( 0xA0, 0xEA), rand (0xA0, 0xEA) ); // $this->hex2rgb("#eaeaea");
        //$this->noise_level  = $this->config['int_noise_level'];
            
        //$this->text_color      = $this->initColor($this->text_color,      '#616161');
        
        $this->noise_color     =  $this->hex2rgb(  '#eaeaea' );
        $this->signature_color     =  $this->hex2rgb(  '#eaeaea' );
        
        // 
        $this->signature_font = $this->ttf_file;
        $this->code_length = 6;
        $this->distortion_level = 1;
        $this->noise_level = 0;
            
    }


    /**
     * Copies the captcha image to the final image with distortion applied
     */
    protected function distortedCopy()
    {
        $numpoles = 3; // distortion factor
        // make array of poles AKA attractor points
        for ($i = 0; $i < $numpoles; ++ $i) {
            $px[$i]  = mt_rand($this->image_width  * 0.2, $this->image_width  * 0.8);
            $py[$i]  = mt_rand($this->image_height * 0.2, $this->image_height * 0.8);
            $rad[$i] = mt_rand($this->image_height * 0.2, $this->image_height * 0.8);
            $tmp     = ((- $this->frand()) * 0.15) - .15;
            $amp[$i] = $this->distortion_level * $tmp;
        }

        $bgCol =  imagecolorat($this->tmpimg, 0, 0);
        $width2 = $this->iscale * $this->image_width;
        $height2 = $this->iscale * $this->image_height;
        imagepalettecopy($this->im, $this->tmpimg); // copy palette to final image so text colors come across
        // loop over $img pixels, take pixels from $tmpimg with distortion field
        for ($ix = 0; $ix < $this->image_width; ++ $ix) {
            for ($iy = 0; $iy < $this->image_height; ++ $iy) {
                $x = $ix;
                $y = $iy;
                for ($i = 0; $i < $numpoles; ++ $i) {
                    $dx = $ix - $px[$i];
                    $dy = $iy - $py[$i];
                    if ($dx == 0 && $dy == 0) {
                        continue;
                    }
                    $r = sqrt($dx * $dx + $dy * $dy);
                    if ($r > $rad[$i]) {
                        continue;
                    }
                    $rscale = $amp[$i] * sin(3.14 * $r / $rad[$i]);
                    $x += $dx * $rscale;
                    $y += $dy * $rscale;
                }
                $c = $bgCol;
                $x *= $this->iscale;
                $y *= $this->iscale;
                if ($x >= 0 && $x < $width2 && $y >= 0 && $y < $height2) {
                    $c = imagecolorat($this->tmpimg, (int) $x, (int) $y);
                }
                if ($c != $bgCol) { // only copy pixels of letters to preserve any background image
                    imagesetpixel($this->im, $ix, $iy, $c);
                }
            }
        }
    }
    
    /**
     * Generates a random captcha code from the set character set
     *
     * @see Securimage::$charset  Charset option
     * @return string A randomly generated CAPTCHA code
     */
    protected function generateCode()
    {
        $code = '';

        for($i = 1, $cslen = strlen($this->charset); $i <= $this->code_length; ++$i) {
            $code .= substr($this->charset, mt_rand(0, $cslen - 1), 1);
        }

        return $code;
    }

    /**
     * Save CAPTCHA data to session and database (if configured)
     */
    protected function saveData()
    {
        $val = fsession_string('_scaptcha_value');
            if (!empty($val) && is_scalar($val)) {
                // fix for migration from v2 - v3
                unset($_SESSION['_scaptcha_value']);
                unset($_SESSION['_scaptcha_ctime']);
            }

            $_SESSION['_scaptcha_disp'] = $this->code_display;
            $_SESSION['_scaptcha_value'] = $this->code;
            $_SESSION['_scaptcha_ctime'] = time(); 
    }

    /**
     * Checks a given code against the correct value from the session and/or database.
     *
     * @param string $code  The captcha code to check
     *
     *     $code = $_POST['code'];
     *     $img  = new Securimage();
     *     if ($img->check($code) == true) {
     *         $captcha_valid = true;
     *     } else {
     *         $captcha_valid = false;
     *     }
     *
     * @return bool true if the given code was correct, false if not.
     */
    public function check($code)
    {
        $this->code_entered = $code;
        $this->validate();
        return $this->correct_code;
    }


    public static function isCaptchaCodeValid()
    {
        $captcha = new self();                
        return $captcha->check(fpost_string('verification_code', 30));
    }
    
    function getCode()
    {        
        $result = null;
        $val = $_SESSION['_scaptcha_value'];
        if (isset($val) && !empty($val)) {            
            $result  = [
                    'display' => $_SESSION['_scaptcha_disp'],
                    'code' => strtolower($_SESSION['_scaptcha_value']),
                    'time' => $_SESSION['_scaptcha_ctime']];
            
        }
        return $result;
    }    
    /**
     * Validate a code supplied by the user
     *
     * Checks the entered code against the value stored in the session and/or database (if configured).  Handles case sensitivity.
     * Also removes the code from session/database if the code was entered correctly to prevent re-use attack.
     *
     * This function does not return a value.
     *
     * @see Securimage::$correct_code 'correct_code' property
     */
    protected function validate()
    {        
        if (is_null($this->code) || strlen($this->code) == 0) {
            $code = $this->getCode(true);            
            
            // returns stored code, or an empty string if no stored code was found
            // checks the session and database if enabled
        } else {
            $code = $this->code;
        }
        
        if (is_array($code)) {
            if (!empty($code)) {
                $ctime = $code['time'];
                $code  = $code['code'];

                $this->_timeToSolve = time() - $ctime;
            } else {
                $code = '';
            }
        }
    $this->case_sensitive = false; /*
        if ($this->case_sensitive == false && ctype_upper( $code)) {
            // case sensitive was set from securimage_show.php but not in class
            // the code saved in the session has capitals so set case sensitive to true
            $this->case_sensitive = true;
        }*/

        $code_entered = trim( (($this->case_sensitive) ? $this->code_entered
                                                       : strtolower($this->code_entered))
                        );
        $this->correct_code = false;


        if ($code != '') {            
            if (strpos($code, ' ') !== false) {
                // for multi word captchas, remove more than once space from input
                $code_entered = preg_replace('/\s+/', ' ', $code_entered);
                $code_entered = strtolower($code_entered);
            }

            if ((string)$code === (string)$code_entered) {
                    $this->correct_code = true;
                    $_SESSION['_scaptcha_disp']  = '';
                    $_SESSION['_scaptcha_value'] = '';
                    $_SESSION['_scaptcha_ctime'] = '';
            }
        }
        

    }
    
    /**
     * Convert an html color code to a ColorSpecs
     * @param string $color
     * @param ColorSpecs|string $default The defalt color to use if $color is invalid
     */
    protected function initColor($color, $default)
    {
        if ($color == null) {
            return $this->hex2rgb($default);
        } else if (is_string($color)) {
            try {
                return $this->hex2rgb($color);
            } catch(Exception $e) {
                return $this->hex2rgb($default);
            }
        } else if (is_array($color) && sizeof($color) == 3) {
            return $this->rgb($color[0], $color[1], $color[2]);
        } else {
            return $this->hex2rgb($default);
        }
    }
    


    /**
     * Allocate the colors to be used for the image
     */
    protected function allocateColors()
    {
        // allocate bg color first for imagecreate
        $this->gdbgcolor = imagecolorallocate($this->im,
                                              $this->image_bg_color['r'],
                                              $this->image_bg_color['g'],
                                              $this->image_bg_color['b'] );

        $alpha = intval($this->text_transparency_percentage / 100 * 127);

        if ($this->use_transparent_text == true) {
            $this->gdtextcolor = imagecolorallocatealpha($this->im,
                                                         $this->text_color['r'],
                                                         $this->text_color['g'],
                                                         $this->text_color['b'] ,
                                                         $alpha);
            $this->gdlinecolor = imagecolorallocatealpha($this->im,
                                                         $this->line_color['r'],
                                                         $this->line_color['g'],
                                                         $this->line_color['b'] ,
                                                         $alpha);
            $this->gdnoisecolor = imagecolorallocatealpha($this->im,
                                                          $this->noise_color['r'],
                                                          $this->noise_color['g'],
                                                          $this->noise_color['b'] ,
                                                          $alpha);
        } else {
            $this->gdtextcolor = imagecolorallocate($this->im,
                                                    $this->text_color['r'],
                                                    $this->text_color['g'],
                                                    $this->text_color['b'] );
            $this->gdlinecolor = imagecolorallocate($this->im,
                                                    $this->line_color['r'],
                                                    $this->line_color['g'],
                                                    $this->line_color['b'] );
            $this->gdnoisecolor = imagecolorallocate($this->im,
                                                          $this->noise_color['r'],
                                                          $this->noise_color['g'],
                                                          $this->noise_color['b'] );
        }

        $this->gdsignaturecolor = imagecolorallocate($this->im,
                                                     $this->signature_color['r'],
                                                     $this->signature_color['g'],
                                                     $this->signature_color['b'] );

    }

    /**
     * The the background color, or background image to be used
     */
    protected function setBackground()
    {
        // set background color of image by drawing a rectangle since imagecreatetruecolor doesn't set a bg color
        imagefilledrectangle($this->im, 0, 0,
                             $this->image_width, $this->image_height,
                             $this->gdbgcolor);
        imagefilledrectangle($this->tmpimg, 0, 0,
                             $this->image_width * $this->iscale, $this->image_height * $this->iscale,
                             $this->gdbgcolor);

        if ($this->bgimg == '') {
            if ($this->background_directory != null &&
                is_dir($this->background_directory) &&
                is_readable($this->background_directory))
            {
                $img = $this->getBackgroundFromDirectory();
                if ($img != false) {
                    $this->bgimg = $img;
                }
            }
        }

        if ($this->bgimg == '') {
            return;
        }

        $dat = @getimagesize($this->bgimg);
        if($dat == false) {
            return;
        }

        switch($dat[2]) {
            case 1:  $newim = @imagecreatefromgif($this->bgimg); break;
            case 2:  $newim = @imagecreatefromjpeg($this->bgimg); break;
            case 3:  $newim = @imagecreatefrompng($this->bgimg); break;
            default: return;
        }

        if(!$newim) return;

        imagecopyresized($this->im, $newim, 0, 0, 0, 0,
                         $this->image_width, $this->image_height,
                         imagesx($newim), imagesy($newim));
    }

    /**
     * This method generates a new captcha code.
     *
     * Generates a random captcha code based on *charset*, math problem, or captcha from the wordlist and saves the value to the session and/or database.
     */
    public function createCode()
    {
        $this->code = false;

        switch($this->captcha_type) {
            case self::SI_CAPTCHA_MATHEMATIC:
            {
                do {
                    $signs = array('+', '-', 'x');
                    $left  = mt_rand(1, 10);
                    $right = mt_rand(1, 5);
                    $sign  = $signs[mt_rand(0, 2)];

                    switch($sign) {
                        case 'x': $c = $left * $right; break;
                        case '-': $c = $left - $right; break;
                        default:  $c = $left + $right; break;
                    }
                } while ($c <= 0); // no negative #'s or 0

                $this->code         = "$c";
                $this->code_display = "$left $sign $right";
                break;
            }
            default:
            {
                if ($this->code == false) {
                    $this->code = $this->generateCode($this->code_length);
                }

                $this->code_display = $this->code;
                $this->code         = ($this->case_sensitive) ? $this->code : strtolower($this->code);
            } // default
        }

        $this->saveData();
    }
    


    /**
     * Draws the captcha code on the image
     */
    protected function drawWord()
    {
        $width2  = $this->image_width * $this->iscale;
        $height2 = $this->image_height * $this->iscale;
        $ratio   = ($this->font_ratio) ? $this->font_ratio : 0.4;

        if ((float)$ratio < 0.1 || (float)$ratio >= 1) {
            $ratio = 0.4;
        }

        if (!is_readable($this->ttf_file)) {
            imagestring($this->im, 4, 10, ($this->image_height / 2) - 5, 'Failed to load TTF font file!', $this->gdtextcolor);
        } else {
            $angle = rand(-7, 7);
            if ($this->distortion_level > 0) {
                $font_size = $height2 * $ratio;
                $bb = imageftbbox($font_size, $angle, $this->ttf_file, $this->code_display);
                $tx = $bb[4] - $bb[0];
                $ty = $bb[5] - $bb[1];
                $x  = floor($width2 / 2 - $tx / 2 - $bb[0]);
                $y  = round($height2 / 2 - $ty / 2 - $bb[1]);

                imagettftext($this->tmpimg, $font_size, 0, (int)$x, (int)$y, $this->gdtextcolor, $this->ttf_file, $this->code_display);
            } else {
                $font_size = $this->image_height * $ratio;
                $bb = imageftbbox($font_size, $angle, $this->ttf_file, $this->code_display);
                $tx = $bb[4] - $bb[0];
                $ty = $bb[5] - $bb[1];
                $x  = floor($this->image_width / 2 - $tx / 2 - $bb[0]);
                $y  = round($this->image_height / 2 - $ty / 2 - $bb[1]);

                imagettftext($this->im, $font_size, 0, (int)$x, (int)$y, $this->gdtextcolor, $this->ttf_file, $this->code_display);
            }
        }

        // DEBUG
        //$this->im = $this->tmpimg;
        //$this->output();

    }

  
    /**
     * Return a random float between 0 and 0.9999
     *
     * @return float Random float between 0 and 0.9999
     */
    function frand()
    {
        return 0.0001 * mt_rand(0,9999);
    }
    
                        
                        
    /**
     * Draws distorted lines on the image
     */
    protected function drawLines()
    {
        for ($line = 0; $line < $this->num_lines; ++ $line) {
            $x = $this->image_width * (1 + $line) / ($this->num_lines + 1);
            $x += (0.5 - $this->frand()) * $this->image_width / $this->num_lines;
            $y = mt_rand($this->image_height * 0.1, $this->image_height * 0.9);

            $theta = ($this->frand() - 0.5) * M_PI * 0.7;
            $w = $this->image_width;
            $len = mt_rand($w * 0.4, $w * 0.7);
            $lwid = mt_rand(0, 2);

            $k = $this->frand() * 0.6 + 0.2;
            $k = $k * $k * 0.5;
            $phi = $this->frand() * 6.28;
            $step = 0.5;
            $dx = $step * cos($theta);
            $dy = $step * sin($theta);
            $n = $len / $step;
            $amp = 1.5 * $this->frand() / ($k + 5.0 / $len);
            $x0 = $x - 0.5 * $len * cos($theta);
            $y0 = $y - 0.5 * $len * sin($theta);

            $ldx = round(- $dy * $lwid);
            $ldy = round($dx * $lwid);

            for ($i = 0; $i < $n; ++ $i) {
                $x = $x0 + $i * $dx + $amp * $dy * sin($k * $i * $step + $phi);
                $y = $y0 + $i * $dy - $amp * $dx * sin($k * $i * $step + $phi);
                imagefilledrectangle($this->im, (int) $x, (int) $y, (int) ($x + $lwid), (int) ($y + $lwid), $this->gdlinecolor);
            }
        }
    }

    /**
     * Draws random noise on the image
     */
    protected function drawNoise()
    {
        if ($this->noise_level > 10) {
            $noise_level = 10;
        } else {
            $noise_level = $this->noise_level;
        }

        $t0 = microtime(true);

        $noise_level *= 125; // an arbitrary number that works well on a 1-10 scale

        $points = $this->image_width * $this->image_height * $this->iscale;
        $height = $this->image_height * $this->iscale;
        $width  = $this->image_width * $this->iscale;
        for ($i = 0; $i < $noise_level; ++$i) {
            $x = mt_rand(10, $width);
            $y = mt_rand(10, $height);
            $size = mt_rand(7, 10);
            if ($x - $size <= 0 && $y - $size <= 0) continue; // dont cover 0,0 since it is used by imagedistortedcopy
            imagefilledarc($this->tmpimg, $x, $y, $size, $size, 0, 360, $this->gdnoisecolor, IMG_ARC_PIE);
        }

        $t1 = microtime(true);

        $t = $t1 - $t0; 
    }
    /**
     * The main image drawing routing, responsible for constructing the entire image and serving it
     */
    protected function showCaptcha()
    {
        header_remove();
        ob_end_clean();

        
        if( ($this->use_transparent_text == true || $this->bgimg != '') && function_exists('imagecreatetruecolor')) {
            $imagecreate = 'imagecreatetruecolor';
        } else {
            $imagecreate = 'imagecreate';
        }

        $this->im     = $imagecreate($this->image_width, $this->image_height);
        $this->tmpimg = $imagecreate($this->image_width * $this->iscale, $this->image_height * $this->iscale);

        $this->allocateColors();
        imagepalettecopy($this->tmpimg, $this->im);

        $this->setBackground();

        $code = '';

        $this->createCode();

        if ($this->noise_level > 0) {
            $this->drawNoise();
        }

        $this->drawWord();

        if ($this->distortion_level > 0 && is_readable($this->ttf_file)) {
            $this->distortedCopy();
        }

        if ($this->num_lines > 0) {
            $this->drawLines();
        }

        if ($this->image_signature !=- null && trim($this->image_signature) != '') {
            $this->addSignature();
        }

        header("Expires: Mon, 26 Jul 1997 05:00:00 GMT");
        header("Last-Modified: " . gmdate("D, d M Y H:i:s") . "GMT");
        header("Cache-Control: no-store, no-cache, must-revalidate");
        header("Cache-Control: post-check=0, pre-check=0", false);
        header("Pragma: no-cache");
        header("Content-Type: image/png");
        imagepng($this->im);

        imagedestroy($this->im);
        restore_error_handler();

        exit;

    }    
    
    protected function rgb($red, $green, $blue)
    {
        if ($red < 0)     $red   = 0;
        if ($red > 255)   $red   = 255;
        if ($green < 0)   $green = 0;
        if ($green > 255) $green = 255;
        if ($blue < 0)    $blue  = 0;
        if ($blue > 255)  $blue  = 255;

        $result = ['r' => $red, 'g' => $green, 'b' => $blue];
        return $result;
        
    }
    
    public static function getURL()
    {
        $captcha = new \App\Core_Captcha();
        return $captcha->createFriendlyURL('');
    }
    
    /**
     * Hex to RGB. Returns an array with key ['r' => $r, 'g' => $g, 'b' => $b]
     * @param string $hex
     * @return array
     */
    protected function hex2rgb($hex) {
       $hex = str_replace('#', '', $hex);

       if(strlen($hex) == 3) {
          $r = hexdec(substr($hex,0,1).substr($hex,0,1));
          $g = hexdec(substr($hex,1,1).substr($hex,1,1));
          $b = hexdec(substr($hex,2,1).substr($hex,2,1));
       } else {
          $r = hexdec(substr($hex,0,2));
          $g = hexdec(substr($hex,2,2));
          $b = hexdec(substr($hex,4,2));
       }
       $rgb = ['r' => $r, 'g' => $g, 'b' => $b];
       return $rgb;
    }    
    //_______________________________________________________________________________________________________________//
    public function Run($command) {
        
        $this->showCaptcha();
        return false;
    }

}
                        