<?php
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
//
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//

namespace SCHLIX;

/**
 * XML Tool. Always use this when converting to and from XML as there are some basic security measures to prevent XML vulnerabilities
 */
class cmsXMLTool
{
    private static $opts = LIBXML_ERR_WARNING |  LIBXML_PARSEHUGE |  LIBXML_NONET | LIBXML_HTML_NODEFDTD | LIBXML_HTML_NOIMPLIED;
    
    private static function processXMLToArray($array, &$xml) {        
        foreach($array as $key => $value) {
            if(is_array($value)) {            
                if(!is_numeric($key)){
                    $subnode = $xml->addChild($key);
                    static::processXMLToArray($value, $subnode);
                } else {
                    static::processXMLToArray($value, $subnode);
                }
            } else {
                $xml->addChild($key, $value);
            }
        }        
    }

    /**
     * Convert array to XML
     * @param array $array
     * @param string $root
     * @param string $xmlns
     * @return string
     */
    public static function arrayToXML($array, $root, $xmlns)
    {
        
        $xml = new \SimpleXMLElement('<?xml version="1.0" encoding="utf-8" ?>'."<{$root} xmlns=\"{$xmlns}\"/>", static::$opts, false); 
        static::processXMLToArray($array, $xml);

        // TO PRETTY PRINT OUTPUT
        $domxml = new \DOMDocument('1.0','utf-8');
        $domxml->preserveWhiteSpace = false;
        $domxml->standalone = false;
        $domxml->formatOutput = true;
        
        $domxml->loadXML($xml->asXML());
        return $domxml->saveXML();
    }
    
    /**
     * 
     * @param string $xml
     * @return array
     */
    public static function xmlToArray($xml)
    {
        if (\PHP_VERSION_ID < 80000) {
            libxml_disable_entity_loader(true);
        }
        libxml_use_internal_errors(true);
        $opt = LIBXML_ERR_WARNING |  LIBXML_PARSEHUGE |  LIBXML_NONET | LIBXML_HTML_NODEFDTD | LIBXML_HTML_NOIMPLIED;
        $xmldoc = new \SimpleXMLElement($xml, $opt, false); // simplexml_load_string($config_doc);
        $xml_errors = libxml_get_errors();
        if (empty($xml_errors))
        {
            $json  = json_encode($xmldoc, JSON_OBJECT_AS_ARRAY);
            return json_decode($json, JSON_OBJECT_AS_ARRAY);
        }
        return null;
    }
 
}
/**
 * cURL response representation class.
 */
class cmsCurlResponse {

    /**
     * The response headers.
     * @var array
     */
    private $headers = [];

    /**
     * The response body.
     * @var string
     */
    private $body;

    /**
     * The results of curl_getinfo on the response request.
     * @var array|false
     */
    private $info;

    /**
     * The response code including text, e.g. '200 OK'.
     * @var string
     */
    private $statusText;

    /**
     * The response code.
     * @var int
     */
    private $statusCode;
    /**
     * Curl Resource
     * @var resource 
     */
    private $curl;
    /**
     * Response
     * @var string 
     */
    private $response;
    
    /**
     * Request header
     * @var array 
     */
    private $request_headers;
    
    /**
     * Response header
     * @var array 
     */
    private $response_headers;
 

    /**
     * Error number
     * @var int
     */
    private $error_no;
    
    /**
     * Error message
     * @var string 
     */
    private $error_msg;
    
    /**
     *
     * @var array 
     */
    private $response_cookies = [];
    /**
     * Constructor
     * @param resource $curl
     * @param string $response
     */
    public function __construct($curl, $response) {
        $this->curl = $curl;
        $this->response = $response;
        $info = curl_getinfo($this->curl);
        $headerSize = curl_getinfo($this->curl, CURLINFO_HEADER_SIZE);
        $this->error_msg = curl_error($this->curl);
        $this->error_no = curl_errno($this->curl);

        $headers = substr($response, 0, $headerSize);
        $body = substr($response, $headerSize);


        $this->body = $body;
        $this->info = $info;
        $this->response_headers = $this->parseHeaders($headers);
        $this->request_headers = $this->parseHeaders($this->info['request_header']);
    }

    /**
     * Parse header
     * @param string $str_header
     * @return array
     */
    private function parseHeaders($str_header) {
        $raw_headers = explode("\r\n", trim($str_header));

        $http_headers = [];

        $raw_headers_count = count($raw_headers);
            
        for ($i = 0; $i < $raw_headers_count; $i++) {
            if (strpos($raw_headers[$i], ':') !== false) {
                list($key, $value) = explode(':', $raw_headers[$i], 2);
                $key = trim($key);
                $value = trim($value);
                // Use isset() as array_key_exists() and ArrayAccess are not compatible.
                if (isset($http_headers[$key])) {
                    $http_headers[$key] .= ',' . $value;
                } else {
                    $http_headers[$key] = $value;
                }
                if (preg_match('/^Set-Cookie:\s*([^=]+)=([^;]+)/mi', $raw_headers[$i], $cookie) === 1)
                {
                    $this->response_cookies[$cookie[1]] = trim($cookie[2], " \n\r\t\0\x0B");            
                }
            } else {
                if (preg_match('/^HTTP\/\d(\.\d)? [0-9]{3}/', $raw_headers[$i])) {
                    list(, $status) = explode(' ', $raw_headers[$i], 2);
                    $code = explode(' ', $status);
                    $code = (int) $code[0];
                    $this->statusText = $raw_headers[$i];
                    $this->statusCode = $code;
                }
            }
        }

        return $http_headers; // array(isset($raw_headers['0']) ? $raw_headers['0'] : '', $http_headers);
    }  
    
    /**
     * CURL Info
     * @return array
     */
    public function getInfo()
    {
        return $this->info;
    }
    
    /**
     * Request headers
     * @return array
     */
    public function getRequestHeaders()
    {
        return $this->request_headers;
    }
    
    /**
     * Response headers
     * @return array
     */
    public function getResponseHeaders()
    {
        return $this->response_headers;
    }
    
    /**
     * HTTP Status text
     * @return string
     */
    public function getStatusText()
    {
        return $this->statusText;
    }
    
    /**
     * HTTP Status code
     * @return int
     */
    public function getStatusCode()
    {
        return $this->statusCode;
    }
    
    /**
     * Error number
     * @return int 
     */
    public function getErrorNumber()
    {
        return $this->error_no;
    }
    
    /**
     * Error message
     * @return string
     */
    public function getErrorMessage()
    {
        return $this->error_msg;
    }    
    
    /**
     * Response cookies
     * @return array
     */
    public function getResponseCookies()
    {
        return $this->response_cookies;
    }
    /**
     * Response body
     * @return string
     */
    public function getBody()
    {
        return $this->body;
    }
    /**
     * Convert the response instance to an array.
     *
     * @return array
     */
    public function toArray() {
        $result = [
            
            'request_headers' => $this->request_headers,
            'response_cookies' => $this->response_cookies,
            'response_headers' => $this->response_headers,
            'body' => $this->body,
            'info' => $this->info,
            'status_code' => $this->statusCode,
            'status_text' => $this->statusText
        ];
        if ($this->error_msg)
            $result['error_msg'] = $this->error_msg;
        if ($this->error_no)
            $result['error_no'] = $this->error_no;
        return $result;
    }

    /**
     * Convert the response object to a JSON string.
     *
     * @return string
     */
    public function toJson() {
        return json_encode($this->toArray());
    }

    /**
     * Convert the object to its string representation by returning the body.
     *
     * @return string
     */
    public function __toString() {
        return $this->body;
    }

}

/**
 * Class for performing HTTP request via cURL.
 */
class cmsCurl {

    /**
     * CURL instance
     * @var resource
     */
    private $curl = NULL;
    /**
     * Timeout
     * @var int 
     */
    private $default_timeout = 20;
    /**
     * CURL Options
     * @var array 
     */
    private $options = [];
    /**
     * Enable/disable caching
     * @var int 
     */
    private $cache_seconds = -1;

    /**
     * Cookies
     * @var array 
     */
    private $cookies = [];
    
    /**
     * Cache
     * @var \SCHLIX\cmsFSCache 
     */
    private $cache;
    
    /**
     * Response type
     * @var string
     */
    protected $response_type = null;
    
    /**
     * Response headers
     * @var array 
     */
    protected $request_headers = null;
    /**
     * Constructor
     *
     * @param  $namespace
     * @throws \ErrorException
     */
    public function __construct($namespace='curl') {
        if (!extension_loaded('curl')) {
            throw new \ErrorException('cURL library is not loaded');
        }
        $this->cache = new cmsFSCache($namespace);
        $this->reset();
    }

    /**
     * Destruct
     */
    public function __destruct() {
        $this->close();
    }

    /**
     * Reset CURL
     */
    public function reset() 
    {
        $this->close();
        if ( is_resource($this->curl)) {
            curl_reset($this->curl);
        }          
        $this->curl = curl_init();
    }

    /**
     * Close CURL Resource
     */
    public function close() {
        if (is_resource($this->curl)) {
            curl_close($this->curl);
        }
    }

    /**
     * Set SSL Verify
     * @param bool $bool
     * @param string $custom_ssl_ca_path
     */
    public function setSSLVerify($bool, $custom_ssl_ca_path = '') {
        $this->setOption(CURLOPT_SSL_VERIFYPEER, $bool);
        if ($bool)
        {
            if ((strlen($custom_ssl_ca_path) > 1) && file_exists($custom_ssl_ca_path) && is_readable($custom_ssl_ca_path))                
                $this->setOption(CURLOPT_CAINFO, $custom_ssl_ca_path);
            else if (file_exists(SCHLIX_DEFAULT_CA_BUNDLE))
                $this->setOption(CURLOPT_CAINFO, SCHLIX_DEFAULT_CA_BUNDLE);
            else
                throw new Exception ('SSL CA file cannot be found or read');
        }
    }

    /**
     * Set Referrer
     *
     
     * @param  $referrer
     */
    public function setReferrer($referrer) {
        $this->setOption(CURLOPT_REFERER, $referrer);
    }

    /**
     * Set CURL option
     * @param mixed $option
     * @param mixed $value
     */
    public function setOption($option, $value) {
        $this->options[$option] = $value;
    }

    /**
     * Set options
     * @param array $options
     * @return boolean
     */
    public function setOptions($options) {
        foreach ($options as $option => $value) {
            $this->setOption($option, $value);
        }
    }

    /**
     * Set Basic Authentication
     *
     
     * @param  $username
     * @param  $password
     */
    public function setBasicAuthentication($username, $password = '')
    {
        $this->setOption(CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
        $this->setOption(CURLOPT_USERPWD, $username . ':' . $password);
    }

    /**
     * Set Digest Authentication
     *
     
     * @param  $username
     * @param  $password
     */
    public function setDigestAuthentication($username, $password = '')
    {
        $this->setOption(CURLOPT_HTTPAUTH, CURLAUTH_DIGEST);
        $this->setOption(CURLOPT_USERPWD, $username . ':' . $password);
    }
    
    
    /**
     * Set Timeout
     *
     
     * @param  $seconds
     */
    public function setTimeout($seconds)
    {
        $this->setOption(CURLOPT_TIMEOUT, $seconds);
    }

    
    /**
     * Set User Agent. Valid shortcut: chrome, firefox, msie
     *
     * @param  $user_agent
     */
    public function setUserAgent($user_agent)
    {
        switch ($user_agent)
        {
            case 'chrome':
                $ua = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/77.0.3835.0 Safari/537.36';
                break;
            case 'firefox':
                $ua = 'Mozilla/5.0 (Windows NT 6.1; WOW64; rv:68.0) Gecko/20100101 Firefox/68.0';
                break;
            case 'msie':
                $ua = 'Mozilla/5.0 (Windows NT 10.0; WOW64; Trident/7.0; Touch; rv:11.0) like Gecko';
                break;
            case 'safari':
                $ua = 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0 Safari/605.1.15';
                break;
            default:
                $ua = $user_agent;
        }
        $this->setOption(CURLOPT_USERAGENT, $ua);
    }
    
    /**
     * Set response result as cmsCurlResponse
     */
    public function setResponseAsObject() {
        $this->response_type = 'o';
    }

    /**
     * Set Cookie String
     *
     * @param string $key
     * @param string $value
     * @return bool
     */
    public function setCookie($key, $value)
    {
        $this->cookies[$key] = $value;
        
    }
    /**
     * Set Port
     *
     * @param  int $port
     */
    public function setPort($port)
    {
        $this->setOption(CURLOPT_PORT, (int) $port);
    }
            
    /**
     * 
     * @param mixed $range
     */
    public function setRange($range)
    {
        $this->setOption(CURLOPT_RANGE, $range);
    }
    
    /**
     * Set response result as raw data
     */
    public function setResponseAsRawData() {
        $this->response_type = NULL;
    }

    public function setBufferSize($size)
    {
        if ($size <= 1024)
            $size = 64000;
        $this->setOption(CURLOPT_BUFFERSIZE, $size);
    }
    /**
     * Reset to default options
     */
    private function setDefaultOptions() {
        $this->setOption(CURLOPT_AUTOREFERER, TRUE);
        $this->setOption(CURLOPT_HEADER, $this->response_type === 'o');
        $this->setOption(CURLOPT_RETURNTRANSFER, 1);
        $this->setBufferSize( 64000);
        $this->setOption(CURLOPT_FOLLOWLOCATION, true);
        $this->setTimeout($this->default_timeout);
        $this->setOption(CURLINFO_HEADER_OUT, true);
        $this->setUserAgent( 'SCHLIX/1.0 '. fserver_string('SERVER_NAME'));
    }
    
    /**
     * 
     * @param string $key
     * @param string $value
     */
    public function setHeader($key, $value)
    {
        $this->request_headers[$key] = $value;
    }

    /**
     * Enable/disable caching. To disable, set it to false or any value below 1
     * @param int $seconds
     */
    public function setCache($seconds)
    {
        if ($seconds <= 0)
            $seconds = FALSE;
        $this->cache_seconds = $seconds;
    }
    /**
     * 
     * @param type $method
     * @param type $url
     * @param string $data
     * @return \cmsCurlResponse
     */
    private function request($method, $url, $data = null) {

        switch (strtoupper($method)) {
            case 'HEAD':
                $this->setOption(CURLOPT_NOBODY, true);
                break;
            case 'GET':
                $this->setOption(CURLOPT_HTTPGET, true);
                break;
            case 'POST':
                $this->setOption(CURLOPT_POST, true);
                break;
            default:
                $this->setOption(CURLOPT_CUSTOMREQUEST, $method);
        }
        $this->setOption(CURLOPT_URL, $url);
        $this->setDefaultOptions();
        if (!empty($this->request_headers))
        {
            
            $headers = array();            
            foreach ($this->request_headers as $key => $value) {
                $headers[] = $key . ': ' . $value;
            }
            
            $this->setOption(CURLOPT_HTTPHEADER, $headers);
        }
        if (!empty($this->cookies))
        {
            $cookies = [];
            foreach ($this->cookies as $key => $value) {
                $cookies[] = $key . '=' . $value;
            }
            $str_cookie = implode(' ;', $cookies);
            $this->setOption(CURLOPT_COOKIE, $str_cookie);            
        }
        if ($data !== NULL)
        {            
            $this->setOption(CURLOPT_POSTFIELDS, $data) ;
        }
        
        if ( $this->cache_seconds > 0)
        {
            
            $key = sha1(var_export ($this->options, true));
            $cache = $this->cache->get($key);
            
            if (isset($cache))
            {
                return $this->returnResponse($cache);
            }
        }
        
        curl_setopt_array($this->curl, $this->options);
        
        $response = curl_exec($this->curl);
        if ( $this->cache_seconds > 0)
        {
            $key = sha1(var_export ($this->options, true));
            $this->cache->set($key, $response, 0, 0, $this->cache_seconds);            
        }
        return $this->returnResponse($response);
    }
    
    private function returnResponse($response)
    {
        if ( $this->response_type === 'o') {
            return new cmsCurlResponse($this->curl, $response);
        } else {
            $err = curl_error($this->curl);
            if ($err)
                throw new \Exception(sprintf(('CURL Error: %s'), $err));

            return $response;
        }
        
    }
    
    /**
     * Build POST data
     * @param mixed $data
     * @return string
     */
    private function buildPostData($data)
    {
        $vars = '';
        if (is_array($data))
        {
            $vars =  http_build_query($data, '', '&');
        } else $vars = $data;
        return $vars;
    }
    
    /**
     * HTTP GET
     * @param string $url
     * @param array $vars
     * @return mixed
     */
    public function get($url, $vars = []) {
        if (!empty($vars)) {
            $url .= (stripos($url, '?') !== false) ? '&' : '?';
            $url .= (is_string($vars)) ? $vars : http_build_query($vars, '', '&');
        }
        return $this->request('GET', $url);
    }
    /**
     * HTTP POST
     * @param string $url
     * @param array $data
     * @return cmsCurlResponse|array
     */
    public function post($url, $data = []) {
        //$vars = (is_string($data)) ? $data : http_build_query($data, '', '&');
        return $this->request('POST', $url, $this->buildPostData($data));
    }
    /**
     * HTTP PUT
     * @param string $url
     * @param array $data
     * @return mixed
     */
    public function put($url, $data = []) {
        //$vars = (is_string($data)) ? $data : http_build_query($data, '', '&');
        return $this->request('PUT', $url, $this->buildPostData($data));
    }
    /**
     * HTTP DELETE
     * @param string $url
     * @param array $data
     * @return mixed
     */
    public function delete($url, $data = []) {
        //$vars = (is_string($data)) ? $data : http_build_query($data, '', '&');
        return $this->request('DELETE', $url, $this->buildPostData($data));
    }
    /**
     * HTTP HEAD
     * @param string $url
     * @param array $data
     * @return mixed
     */
    public function head($url) {
        //$vars = (is_string($data)) ? $data : http_build_query($data, '', '&');
        return $this->request('HEAD', $url, $this->buildPostData($data));
    }

}

class cmsHooks
{
    
    public static function refreshAll()
    {
        global $SystemDB;
        
        $applications = new \App\Core_ApplicationManager();
        $apps = $applications->getAllItems();
        //$hooks = $SystemDB->getQueryResultArray("SELECT * FROM gk_hook_functions");
        $SystemDB->query("TRUNCATE TABLE gk_hook_classes");        
        if ($apps)
        foreach ($apps as $app)
        {
            $full_app_name = '\\App\\'.$app['title'];
            if (class_exists($full_app_name))
            {
                $full_admin_app_name = $full_app_name.'_Admin';
                $app_regular = new $full_app_name();
                $app_admin = new $full_admin_app_name();
                self::register($app_regular);
                self::register($app_admin);

                $sub_apps = $applications->getAllSubApplications($app['title']);
                if ($sub_apps)
                foreach ($sub_apps as $sub_app)
                {
                    $sub_app_full_app_name = '\\App\\'.$app['title'].'_'.$sub_app;
                    $sub_app_full_admin_app_name = $full_app_name.'_'.$sub_app.'_Admin';
                    if (class_exists($sub_app_full_app_name))
                    {
                        $refl_class = new \ReflectionClass($sub_app_full_app_name);
                        $constructor_param_count = $refl_class->getConstructor()->getNumberOfParameters();
                        
                        if ($constructor_param_count == 0)
                        {
                            $sub_app_regular = new $sub_app_full_app_name();
                            self::register($sub_app_regular);
                        }
                        
                        if (class_exists($sub_app_full_admin_app_name))
                        {
                            $refl_admin_class = new \ReflectionClass($sub_app_full_admin_app_name);
                            $constructor_admin_param_count = $refl_admin_class->getConstructor()->getNumberOfParameters();
                            
                            if ($constructor_admin_param_count == 0)
                            {
                                $sub_app_admin = new $sub_app_full_admin_app_name();
                                self::register($sub_app_admin);
                            }                            
                        }
                    }
                }
            }
        }

    }
    /**
     * Get all hooks with this function name
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $function_name
     */
    protected static function getAll($function_name)
    {
        global $SystemDB;
        
        
        $function_name = alpha_numeric_with_underscore(  $function_name );
        $prm = ['function_name' => $function_name];
        $result = $SystemDB->getQueryResultArray("SELECT id, class_name FROM gk_hook_classes INNER JOIN gk_hook_functions ON category_id = cid WHERE LOWER(function_name) = LOWER(:function_name) AND gk_hook_classes.status > 0 ORDER BY sort_order ASC, id ASC", $prm);
        if (!$result)
        {
            $record = $SystemDB->getQueryResultSingleRow("SELECT 1 FROM gk_hook_functions WHERE LOWER(function_name) = LOWER(:function_name)", $prm);
            if (!$record && !empty($function_name))
                $SystemDB->simpleInsertInto('gk_hook_functions', ['function_name' => $function_name, 'date_created' => get_current_datetime() ]);
            return null;
        }
        return $result;
    }
    
    /**
     * Execute all hooks for this function
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param string $function_name
     * @param object $object
     * @return array
     */
    public static function execute($function_name, $object = null)
    {
        global $SystemLog, $SystemDB;
        
        $arg_v = func_get_args();
        if (___c($arg_v) > 1)
            array_shift($arg_v);
        
        $all_hooks =  self::getAll($function_name);
        $result = null;
        if ($all_hooks)
        foreach ($all_hooks as $hook)
        {
            $app_name = '\\'.$hook['class_name'];
            if (class_exists($app_name))
            {
                $obj = new $app_name();                
                $real_method_name =  'hook_'.$function_name;
                if (method_exists($obj, $real_method_name))
                    $result = call_user_func_array([$obj, 'hook_'.$function_name], $arg_v);
                else 
                {
                    $SystemDB->query("DELETE FROM gk_hook_classes WHERE id = :id",['id' => $hook['id']]);
                    $SystemLog->record("Deleted hook {$function_name} for {$app_name} because it could not be found", 'hook', 'warning');

                }
            }
        }
        return $result;
    }
    
    /**
     * Execute all hooks for this function. Must specify return argument position to be replaced
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param string $function_name
     * @param object $object
     * @param int $object_param_arg_post;
     * @return array
     */
    public static function executeModifyReturn($return_arg_pos, $function_name, $object = null)
    {
        global $SystemLog, $SystemDB;
        
        $arg_v = func_get_args();
        if (___c($arg_v) > 1)
        {
            array_shift($arg_v);
            array_shift($arg_v);
        }
        
        $all_hooks =  self::getAll($function_name);
        
        $result = null;
        if ($all_hooks)
        foreach ($all_hooks as $hook)
        {
            $app_name = '\\'.$hook['class_name'];
            $obj = new $app_name();
            
            $real_method_name =  'hook_'.$function_name;
            if (method_exists($obj, $real_method_name))
            {                
                $result = call_user_func_array([$obj, 'hook_'.$function_name], $arg_v);
                $arg_v[$return_arg_pos-1] = $result;
            }
            else 
            {
                $SystemDB->query("DELETE FROM gk_hook_classes WHERE id = :id",['id' => $hook['id']]);
                $SystemLog->record("Deleted hook {$function_name} for {$app_name} because it could not be found", 'hook', 'warning');

            }
        }
        return $result;
    }
    
    
    /**
     * Returns a string output of hooks with this function name. All parameters
     * specified after $function_name will be passed to the function
     * @param string $function_name
     * @return string
     */
    public static function output($function_name)
    {

        $arg_v = func_get_args();
        $result = call_user_func_array('self::execute', $arg_v); // self::execute($function_name);
        
        $s = '';
        if (___c($result) > 0)            
        {
            foreach ($result as $r)
                $s.=$r;            
        }
        return $s;
    }
    
    /**
     * Register hook for a specified class
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param object $obj
     */
    public static function register($obj)
    {
        global $SystemDB;
        
        $all_methods = get_class_methods($obj);
        $hook_methods = [];
        if (___c($all_methods) > 0)
        foreach ($all_methods as $method)
        {
            if (str_starts_with($method, 'hook_'))
                $hook_methods[] = $method;
        }
        if (___c($hook_methods) > 0)
        foreach ($hook_methods as $m)
        {
            $len = strlen ($m);
            $sfn = $len > 5 ? substr($m, 5, $len - 5) : '';            
            if (!empty($sfn))
            {
                $recfn = $SystemDB->getQueryResultSingleRow("SELECT cid FROM gk_hook_functions WHERE function_name = :function_name", ['function_name' => strtolower($sfn)]);
                if ($recfn)
                {
                    
                    $class_name = get_class($obj);
                    $recclass = $SystemDB->getQueryResultSingleRow("SELECT 1 FROM gk_hook_classes WHERE category_id = :cid AND LOWER(class_name) = LOWER(:class_name)",['cid' => $recfn['cid'], 'class_name' => $class_name]);
                    if (!$recclass)
                    {
                        $priority = $obj->getHookPriority();
                        $SystemDB->simpleInsertInto('gk_hook_classes', ['category_id' => $recfn['cid'], 'sort_order'=> $priority, 'class_name' => $class_name, 'date_created' => get_current_datetime(), 'status' => 1]);
                    }
                }
            }
        }
    }
}

class cmsDBCache
{
    private static $cache_content = [];
    /**
     * Set DB cache
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $cache_key
     * @param mixed $content
     * @param int $hours
     * @param int $minutes
     * @param int $seconds
     */
    public static function set($cache_key, $content, $hours = 0, $minutes = 10, $seconds = 0)
    {
        global $SystemDB;
        
        $next_expiry_seconds = ($hours * 3600) + ($minutes * 60) + $seconds;
        $sanitized_expiry = sanitize_string(date("Y-m-d H:i:s", time() + $next_expiry_seconds));
        $sanitized_cache_key = sanitize_string($cache_key);
        if (is_array($content))
            $content = serialize($content);
        $sanitized_content = sanitize_string($content);
        $sql = "REPLACE INTO `gk_cache_items` (`cache_key`,`cache_content`,`date_expiry`) VALUES({$sanitized_cache_key}, {$sanitized_content}, {$sanitized_expiry})";
        $SystemDB->query($sql);

        
    }
    
    /**
     * Get DB cache
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $cache_key
     * @return mixed
     */
    public static function get($cache_key)
    {
        global $SystemDB;
        
        $now = sanitize_string(get_current_datetime());
        $sanitized_cache_key = sanitize_string($cache_key);
        $sql = "SELECT `cache_content` FROM `gk_cache_items` WHERE `cache_key` = {$sanitized_cache_key} AND date_expiry > {$now}";
        $result = $SystemDB->getQueryResultSingleRow($sql);
        if ($result)
        {
            $cache_content = $result['cache_content'];
            return (is_serialized($cache_content)) ? @unserialize($cache_content) : $cache_content;
        } else return NULL;
    }
    /**
     * CRON command - delete expired cache
     * @global SCHLIX\cmsLogger $SystemLog
     */
    public static function processRunPurgeExpiredCache() 
    {
        global $SystemDB;
        
        $now = sanitize_string(get_current_datetime());
        $sql = "DELETE FROM `gk_cache_items` WHERE `date_expiry` < {$now}";
        $SystemDB->query($sql);
    }    
}
/**
 * Just cache for current request
 */
class cmsContextCache
{
    private static $cache_content = [];
    
    public static function set($section, $cache_key, $content)
    {
        self::$cache_content[$section][$cache_key] = $content;
    }
    
    public static function purge($section, $cache_key)
    {
        if (self::$cache_content[$section][$cache_key])
        {
            unset(self::$cache_content[$section][$cache_key]);
        }        
    }
    public static function get($section, $cache_key)
    {
        /*if (empty($cache_key) || empty($section) || ___c(self::$cache_content) == 0 || ___c(self::$cache_content[$section]) == 0 || !array_key_exists($section,self::$cache_content) || !array_key_exists($cache_key,self::$cache_content[$section]))
            return NULL;*/
        if (empty($cache_key) || empty($section) || ___c(self::$cache_content) == 0)
            return NULL;
        if ((___c(self::$cache_content) == 0)  || !array_key_exists($section, self::$cache_content) || !array_key_exists($cache_key, self::$cache_content[$section]))
            return NULL;
        
        if (self::$cache_content[$section][$cache_key])
        {
            return self::$cache_content[$section][$cache_key];
        }
        return NULL;
    }
    
    public static function dump()
    {
        return self::$cache_content;
    }
}
/**
 * CMS Filesystem Cache
 */
class cmsFSCache
{
    private $dir;
    private $cache_dir;
    private $ext;
    
    public function __construct($dir) {
        $this->dir = $dir;
        $this->ext = 'cache';
        $this->cache_dir = SCHLIX_FS_CACHE_FOLDER.'/'.$dir;
        $result = create_directory_if_not_exists($this->cache_dir, true);
        if (!$result)
        {
                die('Cannot create '.$this->cache_dir);
        }
    }
    
    private function getCacheExpiryFileName($cache_key)
    {
        $cache_key = strtolower(str_replace('../','', $cache_key));
        $filename = $this->cache_dir.'/'.$cache_key.'.expiry';
        return $filename;
    }
    
    private function getCacheContentFileName($cache_key)
    {
        $cache_key = strtolower(str_replace('../','', $cache_key));
        $filename = $this->cache_dir.'/'.$cache_key.'.'.$this->ext;
        return $filename;
    }

    /**
     * Set cache content
     * @param string $cache_key
     * @param mixed $content
     * @param int $hours
     * @param int $minutes
     * @param int $seconds
     * @return bool
     */
    public function set($cache_key, $content, $hours = 1, $minutes = 0, $seconds = 0)
    {        
        $next_expiry_seconds = ($hours * 3600) + ($minutes * 60) + $seconds;
        $expiry_time = time() + $next_expiry_seconds;
        $fname = $this->getCacheContentFileName($cache_key);
        if (is_writable(dirname($fname)))
        {
            file_put_contents($this->getCacheContentFileName($cache_key), @serialize($content));
            return file_put_contents($this->getCacheExpiryFileName($cache_key), $expiry_time);            
        }
        return false;
    }
    
    /**
     * Get cache content
     * @param string $cache_key
     * @return boolean
     */
    public function get($cache_key)
    {
        $file_content = $this->getCacheContentFileName($cache_key);
        $file_expiry = $this->getCacheExpiryFileName($cache_key);
        if (file_exists($file_content) & file_exists($file_expiry))
        {
            $expiry = file_get_contents($file_expiry);
            if ($expiry > time())
            {                
                return @unserialize(file_get_contents($file_content));
            } else
            {
                @unlink($file_content);
                @unlink($file_expiry);
                return false;
            }
        }
        return false;
    }
    
    /**
     * CRON command - delete expired cache
     * @global SCHLIX\cmsLogger $SystemLog
     */
    public static function processRunPurgeExpiredCache() 
    {
        global $SystemLog;
        
        $count_deleted = 0;        
        $files_cannot_be_deleted = [];        
        $items = cmsDirectoryFilter::getRecursiveDirectoryIterator(SCHLIX_FS_CACHE_FOLDER, cmsDirectoryFilter::FILTER_FILE_ONLY, array('web.config','.htaccess','README.txt'));        
        foreach ($items as $item) {
            if ($item->isFile()) {
                
                $ext = $item->getExtension();
                if ($ext == 'expiry')
                {
                    $path =  $item->getPath();
                    $expiry_time = (int) file_get_contents($path.'/'.$item->getFileName());
                    if (time() - $expiry_time > 0)
                    {
                        $base = $path.'/'.$item->getBaseName(".{$ext}");
                        $cache_file = $base.'.cache';
                        $expiry_file = $base.'.expiry';
                        if (file_exists($cache_file))
                        {
                            if (!@unlink($cache_file))
                                $files_cannot_be_deleted[] = $cache_file;
                            else
                            {
                                $count_deleted++;
                                @unlink($expiry_file);
                            }
                        }
                    } 
                }
            }
        }

        $str = "{$count_deleted} expired cache file(s) have been deleted.";
        if (___c($files_cannot_be_deleted) > 0)
            $str.="Error while trying to delete:\n".implode("\n", $files_cannot_be_deleted);
        $SystemLog->record($str);
    }

}
/* * ******************************************************** */

/**
 * This class controls locking for process. 
 */
class cmsProcessMutex {

    //_____________________________________________________//
    private static function setLockStatus($process_name, $lockstatus) {
        global $SystemDB;

        $process_name = alpha_numeric_with_dash_underscore($process_name); // sanitize, just in case
        $lockstatus = ($lockstatus != 0) ? 1 : 0; // sanitize
        $sql = "INSERT INTO gk_mutexlock_items (process_name, status) VALUES ('{$process_name}', {$lockstatus}) ON DUPLICATE KEY UPDATE status={$lockstatus}";
        /* $locking = $lockstatus ? 'LOCK' : 'UNLOCK';
          echo $locking.' '.$process_name.br(); */
        $SystemDB->query($sql);
    }

    public static function refreshStalledEntries($maxlocktime_before_unlock = HARDCODE_MAX_MUTEX_LOCKTIME) {
        global $SystemDB;

        // minimum 1 minute
        $maxlocktime_before_unlock = $maxlocktime_before_unlock < 60 ? 60 : $maxlocktime_before_unlock;
        $sql = "update gk_mutexlock_items SET status = 0 WHERE TIME_TO_SEC(TIMEDIFF(NOW(), timestamp_last_touched)) > {$maxlocktime_before_unlock}";
        $SystemDB->query($sql);
    }

    //_____________________________________________________//
    public static function getLockStatus($process_name, $maxlocktime_before_unlock = HARDCODE_MAX_MUTEX_LOCKTIME) {
        global $SystemDB;

        self::refreshStalledEntries($maxlocktime_before_unlock);
        $sql = "SELECT status FROM gk_mutexlock_items WHERE process_name = " . sanitize_string($process_name);
        $item = $SystemDB->getQueryResultSingleRow($sql, false);
        if ($item) {
            /* if (($item['timestamp_last_touched'] > 0) && (time() - $item['timestamp_last_touched'] > $maxlocktime_before_unlock)) {
              echo "Unlocking ".$process_name.' exceeds time: '.time().' - '.$item['timestamp_last_touched'].br();
              ProcessMutex::Unlock($process_name);
              return false;
              }
              else
              return $item; */
            return $item['status'];
        } else
            return false;
    }

    //_____________________________________________________//
    public static function Lock($process_name) {
        self::setLockStatus($process_name, 1);
    }

    //_____________________________________________________//
    public static function Unlock($process_name) {
        self::setLockStatus($process_name, 0);
    }

}

/* * ******************************************************** */
/**
 * Class for directory iterator
 */
class cmsDirectoryFilter extends \FilterIterator {

    private $_ext;
    private $_options;
    private $_it;
    private $_forbidden_listing;

    const FILTER_NONE = 0;
    const FILTER_FILE_ONLY = 1;
    const FILTER_DIR_ONLY = 2;

    //_______________________________________________________________________________________________________________//

    public function __construct(\Iterator $it, $options, $forbidden_listing = []) {
        parent::__construct($it);
        $this->_it = $it;
        $this->_options = $options;
        $this->_forbidden_listing = (is_array($forbidden_listing)) ? $forbidden_listing : [];
    }

    //_______________________________________________________________________________________________________________//
    #[\ReturnTypeWillChange]
    public function accept() {
        $filename = $this->_it->getFileName();
        if (($filename[0] == '.') || (in_array($filename, $this->_forbidden_listing)))
            return false;
        if ($this->_options == cmsDirectoryFilter::FILTER_NONE)
            return true;
        if ($this->_options == cmsDirectoryFilter::FILTER_DIR_ONLY && $this->_it->isDir())
            return true;
        if ($this->_options == cmsDirectoryFilter::FILTER_FILE_ONLY && $this->_it->isFile())
            return true;
        return false;
    }

    //_______________________________________________________________________________________________________________//
    public static function getRecursiveDirectoryIterator($path, $filter, $forbidden_listing = []) {
        $realpath = str_replace('\\', '/', realpath($path)); // for security, prevent path traversal in case developer forgets
        if (strpos($realpath, SCHLIX_ROOT_PATH) === false) {
            'Path input is insecure';
            return false;
        }
        return new cmsDirectoryFilter(new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($realpath, \RecursiveDirectoryIterator::UNIX_PATHS), \RecursiveIteratorIterator::SELF_FIRST), $filter, $forbidden_listing);
    }
    //_______________________________________________________________________________________________________________//

    /**
     * 
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param string $path
     * @param \SCHLIX\cmsDirectoryFilter $filter
     * @param array $forbidden_listing
     * @return \SCHLIX\cmsDirectoryFilter
     */
    public static function getDirectoryIterator($path, $filter, $forbidden_listing = []) {
        global $SystemLog;
        $realpath = str_replace('\\', '/', realpath($path)); // for security, prevent path traversal in case developer forgets
        if (strpos($realpath, SCHLIX_ROOT_PATH) === FALSE) {

            $SystemLog->record("cmsDirectoryFilter - requested path input does not exist or is insecure: {$path}");
            //die('Insecure path or path does not exist');
            return false;
        }
        return new cmsDirectoryFilter(new \DirectoryIterator($realpath), $filter, $forbidden_listing);
    }

}


/**
 * Class for displaying HTML Page Header
 */
class cmsHTMLPageHeader {

    private $header = [];
    private $header_scripts = [];
    private $called = [];
     //_______________________________________________________________________________________________________________//
    /**
     * Adds a text to the header
     * @param string $str
     */
    public function add($str) {
        if (empty($str)) $str = '';
        $txt = trim($str);
        $key = md5($str);
        if (!empty($txt) )
        {
            if (str_starts_with($txt, "<script"))
                $this->header_scripts[$key] = $txt;
            else
                $this->header[$key]=$txt;
        }
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Adds a locally hosted CSS relative to the httpbase
     * @param string $src
     * @param string $media
     */
    public function META($name, $content) {
        $attr = [];
        $name = trim($name);
        $content = trim($content);
        if (strlen($name) > 0 && strlen($content) > 0)
        {
            $attr [$name] = $content;
            $metatext = \__HTML::OpenTag('meta', $attr, true);            
            $this->add($metatext);
        }
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Adds a locally hosted CSS relative to the httpbase
     * @param string $src
     * @param string $media
     */
    public function CSS($src, $media = '') {
        $attr = [];
        if ($media)
            $attr['media'] = $media;
        $csstext = \__HTML::CSS($src,$attr);
        
        $this->add($csstext);
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Adds CSS Text to the page header
     * @param string $str
     */
    public function CSS_TEXT($str) {
        $this->add(\__HTML::CSS($str));
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Adds CSS Text to the page header
     * @param string $src
     */
    public function CSS_EXTERNAL($src, $media = '') {
        $attr = [];
        if ($media)
            $attr['media'] = $media;
        $csstext = \__HTML::CSS_EXTERNAL($src,$attr);
        
        $this->add($csstext);
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Adds a locally hosted Javascript file relative to the URL of this site
     * @param string $str
     * @param bool $no_http_base
     */
    public function JAVASCRIPT($str, $no_http_base = false) {
        $base = $no_http_base ? '' : SCHLIX_SITE_HTTPBASE;
        $jstext = "<script src=\"" . $base . "{$str}\"></script>"; //JAVASCRIPT($str);
        $this->add($jstext);
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Adds a Javascript text to the page header
     * @param string $str
     */
    public function JAVASCRIPT_TEXT($str) {
        $this->add(\__HTML::JAVASCRIPT_TEXT($str));
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Adds an externally hosted Javascript file to the header
     * @param string $str
     */
    public function JAVASCRIPT_EXTERNAL($str) {
        $this->add(\__HTML::JAVASCRIPT_EXTERNAL($str));
    }

    /**
     * get JQuery 1.12
     * @param bool $return_as_string
     * @return bool|string
     */    
    public function JAVASCRIPT_JQUERY1($return_as_string = false) {
        $jquery_url = SCHLIX_SYSTEM_URL_PATH . '/js/jquery/jquery-1.12.4.min.js';
        $js_str = "<script src=\"" . $jquery_url . "\"></script>";
        if ($return_as_string)
            return $js_str;
        else {
            $this->JAVASCRIPT($jquery_url, true);
            return false;
        }
        
    }
    
    
    /**
     * get JQuery 2
     * @param bool $return_as_string
     * @return bool|string
     */
    public function JAVASCRIPT_JQUERY2($return_as_string = false) {
        $jquery_url = SCHLIX_SYSTEM_URL_PATH . '/js/jquery/jquery-2.2.4.min.js';
        $js_str = "<script src=\"" . $jquery_url . "\"></script>";
        if ($return_as_string)
            return $js_str;
        else {
            $this->JAVASCRIPT($jquery_url, true);
            return false;
        }
        
    }
    
    /**
     * Set Canonical URL of this page. If you set $absolute to true, it will not
     * add the HTTPS or HTTP URL
     * @param string $url
     * @param bool $absolute
     */
    public function setCanonicalURL($url, $absolute = false)
    {
        $url_host = SCHLIX_SITE_SSL_ENABLED ? SCHLIX_SITE_HTTPS_URL : SCHLIX_SITE_HTTP_URL;
        $full_canonical_url =  $absolute ? $url_host. remove_multiple_slashes('/' .$url) : $url;

        $this->add('<link rel="canonical" href="'.$full_canonical_url.'" />');
    }
    /**
     * get JQuery 3
     * @param bool $return_as_string
     * @return bool|string
     */    
    public function JAVASCRIPT_JQUERY3($return_as_string = false) {
        $jquery_url = SCHLIX_SYSTEM_URL_PATH . '/js/jquery/jquery-3.7.1.min.js';
        $js_str = "<script src=\"" . $jquery_url . "\"></script>";
        if ($return_as_string)
            return $js_str;
        else {
            $this->JAVASCRIPT($jquery_url, true);
            return false;
        }
        
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Adds SCHLIX UI Javascript to the page header. If return_as_string is specified 
     * then it will won't be added to the the header
     * @param bool $return_as_string
     * @param int $js_type
     * @return bool
     */
    public function JAVASCRIPT_SCHLIX_UI($return_as_string = false, $js_type = 0) {
        $called = array_key_exists('schlix_ui', $this->called) ? $this->called['schlix_ui'] : null;
        if ($called)
            return NULL;        
        if ($js_type == 0)
            $js_type = SCHLIX_JS_MODE;
        
        $schlix_ui_url = NULL;
        $file_found = false;

        switch ($js_type) {
            case JS_SCHLIX_GZIP:
            case JS_SCHLIX_MIN_ONLY:
            case JS_SCHLIX_DEBUG: //deprecated
                $schlix_ui_file = 'schlix-ui.minified.js';
                break;
        }

        $file_found = file_exists(SCHLIX_SYSTEM_PATH . '/js/schlix-ui/' . $schlix_ui_file);
        if (!$schlix_ui_url)
            $schlix_ui_url = SCHLIX_SYSTEM_URL_PATH . '/js/schlix-ui/' . $schlix_ui_file;

        $js_str = ($file_found) ? "<script src=\"" . $schlix_ui_url . "\"></script>" : self::JAVASCRIPT_TEXT('alert("Javascript file ' . $schlix_ui_file . ' not found");');
        $this->called['schlix_ui'] = true;
        
        if ($return_as_string)
            return $js_str;
        else {
            $this->JAVASCRIPT($schlix_ui_url, true);
            return false;
        }
    }

    /**
     * Adds SCHLIX init script to the page header. If return_as_string is specified 
     * then it will won't be added to the the header
     * @param bool $return_as_string
     * @return string|bool
     */
   public function JAVASCRIPT_SCHLIX_INIT($return_as_string = false) {
        $called = array_key_exists('schlix_cms_init', $this->called) ? $this->called['schlix_cms_init'] : null;
        if ($called)
            return NULL;
        
        $schlix_cms_init_url = SCHLIX_SYSTEM_URL_PATH.'/js/schlix-cms/initjs.php';
        $this->called['schlix_cms_init'] = true;
        
        $js_str = "<script src=\"" . $schlix_cms_init_url . "\"></script>";
        if ($return_as_string)
            return $js_str . "\n";
        else {
            $this->Add($js_str);
            return false;
        }

        
   }    
    //_________________________________________________________________________//
    /**
     * Adds SCHLIX CMS Javascript to the page header. If return_as_string is specified 
     * then it will won't be added to the the header
     * @param bool $return_as_string
     * @param int $js_type
     * @return string|bool
     */
   public function JAVASCRIPT_SCHLIX_CMS($return_as_string = false, $js_type = 0) {
        $called = array_key_exists('schlix_cms', $this->called) ? $this->called['schlix_cms'] : null;
        if ($called)
            return NULL;

        if ($js_type == 0)
            $js_type = SCHLIX_JS_MODE;
        $schlix_cms_url = '';
        $file_found = false;
        $js_str = "\n";
        switch ($js_type) {

            case JS_SCHLIX_GZIP:

            case JS_SCHLIX_MIN_ONLY:
                $schlix_cms_file = 'schlix-cms.minified.js';
                $file_found = file_exists(SCHLIX_SYSTEM_PATH . '/js/schlix-cms/' . $schlix_cms_file);
                $schlix_cms_url = SCHLIX_SYSTEM_URL_PATH . '/js/schlix-cms/' . $schlix_cms_file;
                break;
            case JS_SCHLIX_DEBUG:
                $schlix_cms_file = 'schlix-cms.debug.js';
                $f_iterator = new \GlobIterator(SCHLIX_SYSTEM_PATH . '/js/schlix-cms/t*.js', \FilesystemIterator::KEY_AS_FILENAME);
                $js_array = [];
                foreach ($f_iterator as $it) {
                    $js_array[] = $it->getFileName();
                }
                natsort($js_array);

                break;
        } 
        $schlix_cms_init_url = SCHLIX_SYSTEM_URL_PATH.'/js/schlix-cms/initjs.php';
        $this->called['schlix_cms_init'] = true;
        
        $js_str.="<script src=\"" . $schlix_cms_init_url . "\"></script>";
       if ($js_type != JS_SCHLIX_DEBUG) {
            $js_str.= $file_found ? 
                    "<script src=\"" . $schlix_cms_url .'?v='.SCHLIX_VERSION. "\"></script>" : 
                self::JAVASCRIPT_TEXT('alert("Javascript file ' . $schlix_cms_file . ' not found");');
        } else {
            if (___c($js_array) > 1) {
                foreach ($js_array as $js) {
                    $js_str.="<script src=\"" . SCHLIX_SYSTEM_URL_PATH . '/js/schlix-cms/' . $js.'?v='.SCHLIX_VERSION . "\"></script>\n";
                }
            } else {
                $js_str = self::JAVASCRIPT_TEXT('alert("SCHLIX CMS Debug files can not be found");');
            }
        }
        $this->called['schlix_cms'] = true;
        
        if ($return_as_string)
            return $js_str . "\n";
        else {
            $this->Add($js_str);
            return false;
        }
    }
    //_______________________________________________________________________________________________________________//
    /**
     * 
     * @return string
     */
    public function getAll() {

        $internal_msgs = fsession_array_any('schlix_internal_messages');
        if (___c($internal_msgs) > 0) {
            $this->JAVASCRIPT_SCHLIX_UI();
            $this->JAVASCRIPT_SCHLIX_CMS();
            $this->add(static::JAVASCRIPT_TEXT('_schlix_internal_messages = ' . json_encode($_SESSION['schlix_internal_messages']) . ';'));
            // This is done so that when there's a redirect, message box will still be displayed;
            if ($_SESSION['schlix_internal_redirect'] == 1) {
                unset($_SESSION['schlix_internal_redirect']);
            } else {
                unset($_SESSION['schlix_internal_messages']);
            }
        }
        if ($this->header || $this->header_scripts)
        {
            $result = "<!-- schlix header -->\n";
            if ($this->header)
                $result.= implode("\n",$this->header);            
            if ($this->header_scripts)
                $result.= implode("\n",$this->header_scripts);
            $result.= "\n<!-- end of schlix header -->\n";
         
        } else $result = '';
        return $result;        
    }
    
    public function getHeaderTagsNonScriptsOnly()
    {
        $result = '';
        if ($this->header )
        {
            $result = "<!-- schlix header ns -->\n";
            if ($this->header)
                $result.= implode("\n",$this->header);            
            $result.= "\n<!-- end of schlix header ns -->\n";
         
        } else $result = '';
        return $result;        
    }
    
    public function getHeaderTagsScriptsOnly()
    {
        if (array_key_exists('schlix_internal_messages', $_SESSION) && is_array($_SESSION['schlix_internal_messages']) && ___c($_SESSION['schlix_internal_messages']) > 0) {
            $this->JAVASCRIPT_SCHLIX_UI();
            $this->JAVASCRIPT_SCHLIX_CMS();
            $this->add(static::JAVASCRIPT_TEXT('_schlix_internal_messages = ' . json_encode($_SESSION['schlix_internal_messages']) . ';'));
            // This is done so that when there's a redirect, message box will still be displayed;
            if (fsession_int('schlix_internal_redirect') == 1) {
                unset($_SESSION['schlix_internal_redirect']);
            } else {
                unset($_SESSION['schlix_internal_messages']);
            }
        }
        if ($this->header_scripts)
        {
             $result = "<!-- schlix header s -->\n";             
            if ($this->header_scripts)
                $result.= implode("\n",$this->header_scripts);
            $result.= "\n<!-- end of schlix header s -->\n";
         
        } else $result = '';
        return $result;        
    }
    

//_______________________________________________________________________________________________________________//
}

/**
 * Class to display HTML output in the theme
 */
class cmsPageOutput {

    /**
     * Outputs HTML string
     */
    public static function ApplicationBody() {
        global $__html_output;
        echo $__html_output;
    }

    /**
     * Returns a string of HTML Header
     * @global \SCHLIX\cmsHTMLPageHeader $HTMLHeader
     * @return string
     */
    public static function HTMLHeader()
    {
        global $HTMLHeader, $SystemConfig;

        $basehref = SCHLIX_SITE_URL . remove_multiple_slashes(SCHLIX_SITE_HTTPBASE . '/');
        $s = '<meta name="generator" content="SCHLIX" />' . "\n";
        $s.= '<base href="' . $basehref . '" />' . "\n";
        $seo_hidden = $SystemConfig->get('system','int_search_engine_hidden');
        if ($seo_hidden)
        {
            $s.= '<meta name="robots" content="noindex, nofollow">' . "\n";
        }
        $s.= $HTMLHeader->getAll();        
        return $s;
    }


    /**
     * Returns a string of HTML Header
     * @global \SCHLIX\cmsHTMLPageHeader $HTMLHeader
     * @return string
     */
    public static function HTMLHeaderNonScripts()
    {
        global $HTMLHeader, $SystemConfig;

        $basehref = SCHLIX_SITE_URL . remove_multiple_slashes(SCHLIX_SITE_HTTPBASE . '/');
        $s = '<meta name="generator" content="SCHLIX" />' . "\n";
        $s.= '<base href="' . $basehref . '" />' . "\n";
        $s.= $HTMLHeader->getHeaderTagsNonScriptsOnly();        
        $seo_hidden = $SystemConfig->get('system','int_search_engine_hidden');
        if ($seo_hidden)
        {
            $s.= '<meta name="robots" content="noindex, nofollow">' . "\n";
        }        
        return $s;
    }
    
    /**
     * Returns a string of HTML Header
     * @global \SCHLIX\cmsHTMLPageHeader $HTMLHeader
     * @return string
     */
    public static function HTMLFooterScripts()
    {
        global $HTMLHeader;
        return $HTMLHeader->getHeaderTagsScriptsOnly();
    }
    
                        
    
    //_________________________________________________________________________//
    /**
     * Returns the currently set page title 
     * @global \SCHLIX\cmsApplication_Basic $Application
     * @return string
     */
    public static function HTMLPageTitle() {
        global $Application;
        return $Application->getPageTitle();
    }

    //_________________________________________________________________________//
    /**
     * Returns the meta description
     * @global \SCHLIX\cmsApplication_Basic $Application
     * @return string
     */
    public static function HTMLMetaDescription() {
        global $Application;
        return $Application->getPageMetaDescription();
    }

    //_________________________________________________________________________//
    /**
     * Returns the meta keyword
     * @global \SCHLIX\cmsApplication_Basic $Application
     * @return string
     */    
    public static function HTMLMetaKeywords() {
        global $Application;
        return $Application->getPageMetaKeywords();
    }

    //_________________________________________________________________________//
    /**
     * Returns the breadcrumb (usually set by the app)
     * @global \SCHLIX\cmsApplication_Basic $Application
     * @return string
     */    
    public static function BreadCrumbs() {
        global $Application;

        return $Application->displayBreadCrumbs();
    }

    //_________________________________________________________________________//
    /**
     * Returns the output of all blocks in the specified category
     * @global \App\Core_BlockManager $Blocks
     * @param string $position
     * @return string
     */
    public static function BlockCategory($position) {
        global $Blocks, $CurrentUser;
        
        /*$s = '<div style="border:1px solid orange;display:block;">';
        $s.='<div style="background:orange;color:white">'.___h($position).'</div>';
        $s.=$Blocks->getBlocksByCategoryName($position);
        $s.= '</div>';
        return  $s;
         * 
         */
         $prefix = '';
         $suffix = '';
         $show_block = fcookie_int('schlix_frontendedit_control_showblock'); 
         if ($show_block == 2)
         {
             if ($CurrentUser->authenticated())
             {
                $prefix = \__HTML::DIV_start(array('class'=>'schlix_frontendedit_control-blockcategory-container'));
                $prefix.= \__HTML::DIV_start(array('class'=>'schlix_frontendedit_control-blockcategory-title')).
                    \__HTML::I('',array('class'=>'fa fa-th-large')).' '.
                    $position.
                    \__HTML::DIV_end();
                $suffix = \__HTML::DIV_end();
             }
         }
        return $prefix.$Blocks->getBlocksByCategoryName($position).$suffix;
    }

    //_________________________________________________________________________//
    /**
     * Returns the number of blocks in a block category. Useful if you
     * want to check whether or not a certain tag to be dispalyed or not
     * @global \App\Core_BlockManager $Blocks
     * @param string $position
     * @return int
     */
    public static function BlockCountInCategory($position) {
        global $Blocks;

        return $Blocks->getBlockCountInCategory($position);
    }

    //_________________________________________________________________________//
    /**
     * Returns the output of a single block
     * @global \App\Core_BlockManager $Blocks
     * @param string $block_name
     * @return string
     */
    public static function displayBlock($block_name) {
        global $Blocks;

        return $Blocks->getSingleBlock($block_name);
    }

}



/**
 * Animated GIF Resizer utility.
 */
class AnimatedGIFResizer {

    private $gif_buffer;
    private $pointer = 0;
    private $gif_header;
    private $global_rgb;
    private $gl_mod;
    private $gl_mode;
    private $width;
    private $height;
    private $au = 0;
    private $nt = 0;
    private $correct_gif = false;
    private $frame_loop = 0;
    private $array_frames = [];
    private $gn_fld = [];
    private $dl_frmf = [];
    private $dl_frms = [];

    public function __construct($src, $optimize = FALSE) {
        
        if (!is_file($src) && !is_readable($src))
            return FALSE;
        $this->gif_buffer = file_get_contents($src);
        $this->gif_header = $this->getByte(13);
        if (substr($this->gif_header, 0, 6) != 'GIF87a' && substr($this->gif_header, 0, 6) != 'GIF89a') {
            return false;
        }        
        $this->correct_gif = true;
        
        $this->width = $this->rlInt($this->gif_header[6] . $this->gif_header[7]);
        $this->height = $this->rlInt($this->gif_header[8] . $this->gif_header[9]);
        
        if (($vt = ord($this->gif_header[10])) & 128 ? 1 : 0) {
            $this->global_rgb = $this->getByte(pow(2, ($vt & 7) + 1) * 3);
        }
        $buffer_add = '';
        if ($this->gif_buffer[$this->pointer] == "\x21") {
            while ($this->gif_buffer[$this->pointer + 1] != "\xF9" && $this->gif_buffer[$this->pointer] != "\x2C") 
            {
                switch ($this->gif_buffer[$this->pointer + 1]) {
                    case "\xFE":
                        $sum = 2;
                        while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                            $sum += $lc_i + 1;
                        }
                        $optimize ? $this->getByte($sum + 1) : $buffer_add .= $this->getByte($sum + 1);
                        break;
                    case "\xFF":
                        $sum = 14;
                        while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                            $sum += $lc_i + 1;
                        }
                        $buffer_add .= $this->getByte($sum + 1);
                        break;
                    case "\x01":
                        $sum = 15;
                        while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                            $sum += $lc_i + 1;
                        }
                        $optimize ? $this->getByte($sum + 1) : $buffer_add .= $this->getByte($sum + 1);
                }
            }
            $this->gl_mod = $buffer_add;
        }
        while ($this->gif_buffer[$this->pointer] != "\x3B" && $this->gif_buffer[$this->pointer + 1] != "\xFE" && $this->gif_buffer[$this->pointer + 1] != "\xFF" && $this->gif_buffer[$this->pointer + 1] != "\x01") {
            $lc_mod = $lc_palet = $header = $gr_mod = null;
            $pnts = [];
            $this->frame_loop++;
            while ($this->gif_buffer[$this->pointer] != "\x2C") {
                switch ($this->gif_buffer[$this->pointer + 1]) {
                    case "\xF9":
                        $this->gn_fld[] = $this->gif_buffer[$this->pointer + 3];
                        $this->dl_frmf[] = $this->gif_buffer[$this->pointer + 4];
                        $this->dl_frms[] = $this->gif_buffer[$this->pointer + 5];
                        $gr_mod = $buffer_add = $this->getByte(8);
                        break;
                    case "\xFE":
                        $sum = 2;
                        while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                            $sum += $lc_i + 1;
                        }
                        $optimize ? $this->getByte($sum + 1) : $buffer_add .= $this->getByte($sum + 1);
                        break;
                    case "\xFF":
                        $sum = 14;
                        while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                            $sum += $lc_i + 1;
                        }
                        if (substr($tmp_buffer = $this->getByte($sum + 1), 3, 8) == "NETSCAPE") 
                        {
                            if (!$this->nt) {
                                $this->nt = 1;
                                $this->gl_mod .= $tmp_buffer;
                            }
                        } else 
                        $buffer_add .= $tmp_buffer;

                        break;
                    case "\x01":
                        $sum = 15;
                        while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                            $sum += $lc_i + 1;
                        }
                        $optimize ? $this->getByte($sum + 1) : $buffer_add .= $this->getByte($sum + 1);
                }
            }
            $lc_mod = $buffer_add;
            $pnts[] = $this->msInt(1, 2);
            $pnts[] = $this->msInt(3, 2);
            $pnts[] = $this->msInt(5, 2);
            $pnts[] = $this->msInt(7, 2);
            $header = $this->getByte(10);
            if ((($pnts[0] + $pnts[2]) - $this->width) > 0) {
                $header[1] = $header[2] = $header[6] = "\x00";
                $header[5] = $this->asciiToChar($this->width);

                $pnts[0] = 0;
                $pnts[2] = $this->width;
            }
            if ((($pnts[1] + $pnts[3]) - $this->height) > 0) {
                $header[3] = $header[4] = $header[8] = "\x00";
                $header[7] = $this->asciiToChar($this->height);
                $pnts[1] = 0;
                $pnts[3] = $this->height;
            }
            if ((ord($this->gif_buffer[$this->pointer - 1]) & 128 ? 1 : 0)) {
                $lc_i = pow(2, (ord($this->gif_buffer[$this->pointer - 1]) & 7) + 1) * 3;
                $lc_palet = $this->getByte($lc_i);
            }
            $sum = 0;
            $this->pointer++;
            while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                $sum += $lc_i + 1;
            }
            $this->pointer--;
            $the_frame = [];
            $the_frame['lc_mod'] = $lc_mod;
            $the_frame['lc_palet'] = $lc_palet;
            $the_frame['lmage'] = $this->getByte($sum + 2);
            $the_frame['header'] = $header;
            $the_frame['x'] = $pnts[0];
            $the_frame['y'] = $pnts[1];
            $the_frame['frame_width'] = $pnts[2];
            $the_frame['frame_height'] = $pnts[3];
            $the_frame['gr_mod'] = $gr_mod;
            $the_frame['tr_frame'] = ord($gr_mod[3]) & 1 ? 1 : 0;
            $this->array_frames[] = $the_frame;
        }
        
        $buffer_add = "";
        while ($this->gif_buffer[$this->pointer] != "\x3B") {
            switch ($this->gif_buffer[$this->pointer + 1]) {
                case "\xFE":
                    $sum = 2;
                    while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                        $sum += $lc_i + 1;
                    }
                    $optimize ? $this->getByte($sum + 1) : $buffer_add .= $this->getByte($sum + 1);
                    $this->au = ($sum == 17);
                    break;
                case "\xFF":
                    $sum = 14;
                    while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                        $sum += $lc_i + 1;
                    }$buffer_add .= $this->getByte($sum + 1);
                    break;
                case "\x01":
                    $sum = 15;
                    while (($lc_i = ord($this->gif_buffer[$this->pointer + $sum])) != 0x00) {
                        $sum += $lc_i + 1;
                    }
                    $optimize ? $this->getByte($sum + 1) : $buffer_add .= $this->getByte($sum + 1);
            }
        }
        $this->gl_mode = $buffer_add;
        $this->gif_buffer = "";
    }

    private function getByte($n) {
        $b = substr($this->gif_buffer, $this->pointer, $n);
        $this->pointer += $n;
        return $b;
    }

    private function rlInt($hw) {        
        $z = isset($hw[1]) ? ord($hw[1]) << 8 : 0;
        $c = ord($hw[0]);
        $x = $z | $c;
        return $x;
    }

    private function msInt($g_f, $g_s) {
        return $this->rlInt(substr($this->gif_buffer, $this->pointer + $g_f, $g_s));
    }

    private function asciiToChar($t) {
        return chr($t & 0xFF) . chr(($t & 0xFF00) >> 8);
    }

    private function createImage($i) {
        return $this->gif_header . $this->global_rgb . $this->gl_mod . $this->array_frames[$i]['lc_mod'] . $this->array_frames[$i]['header'] . $this->array_frames[$i]['lc_palet'] . $this->array_frames[$i]['lmage'] . "\x3B";
    }

    /**
     * Resize frame, returns it as a string
     * @param string $buffer
     * @param int $frame_num
     * @param array $des
     * @return string
     */
    private function resizeFrame($buffer, $frame_num, $des) {
        $buf_n = round($this->array_frames[$frame_num]['frame_width'] * $des[0]);
        $n_width = $buf_n ? $buf_n : 1;
        $buf_n = round($this->array_frames[$frame_num]['frame_height'] * $des[1]);
        $n_height = $buf_n ? $buf_n : 1;
        $n_pos_x = round($this->array_frames[$frame_num]['x'] * $des[0]);
        $n_pos_y = round($this->array_frames[$frame_num]['y'] * $des[1]);
        $this->array_frames[$frame_num]['off_xy'] = $this->asciiToChar($n_pos_x) . $this->asciiToChar($n_pos_y);
        $str_img = @imagecreatefromstring($buffer);
        if ($this->frame_loop == 1 || $des[3]) {
            $img_s = @imagecreatetruecolor($n_width, $n_height);
        } else {
            $img_s = @imagecreate($n_width, $n_height);
        }if ($this->array_frames[$frame_num]['tr_frame']) {
            $in_trans = @imagecolortransparent($str_img);
            if ($in_trans >= 0 && $in_trans < @imagecolorstotal($img_s)) {
                $tr_clr = @imagecolorsforindex($str_img, $in_trans);
            }if ($this->frame_loop == 1 || $des[3]) {
                $n_trans = @imagecolorallocatealpha($img_s, 255, 255, 255, 127);
            } else {
                $n_trans = @imagecolorallocate($img_s, $tr_clr['red'], $tr_clr['green'], $tr_clr['blue']);
            }
            @imagecolortransparent($img_s, $n_trans);
            @imagefill($img_s, 0, 0, $n_trans);
        }
        @imagecopyresampled($img_s, $str_img, 0, 0, 0, 0, $n_width, $n_height, $this->array_frames[$frame_num]['frame_width'], $this->array_frames[$frame_num]['frame_height']);
        @ob_start();
        @imagegif($img_s);
        $t_img = ob_get_clean();
        @ob_end_clean();
        @imagedestroy($str_img);
        @imagedestroy($img_s);

        return $t_img;
    }

    private function encodeFrame($str_img, $gr_i) {
        $hd = $offset = 13 + pow(2, (ord($str_img[10]) & 7) + 1) * 3;
        $palet = "";
        $i_hd = 0;
        $m_off = 0;
        for ($i = 13; $i < $offset; $i++) {
            $palet .= $str_img[$i];
        }
        if ($this->array_frames[$gr_i]['tr_frame']) 
        {
            while ($str_img[$offset + $m_off] != "\xF9") $m_off++;
            $str_img[$offset + $m_off + 2] = $this->gn_fld[$gr_i];
            $str_img[$offset + $m_off + 3] = $this->dl_frmf[$gr_i];
            $str_img[$offset + $m_off + 4] = $this->dl_frms[$gr_i];
        }
        while ($str_img[$offset] != "\x2C") {
            $offset = $offset + $this->rlInt($str_img[$offset + 2]) + 4;
            $i_hd = $i_hd + $this->rlInt($str_img[$offset + 2]) + 8;
        }
        $str_img[$offset + 1] = $this->array_frames[$gr_i]['off_xy'][0];
        $str_img[$offset + 2] = $this->array_frames[$gr_i]['off_xy'][1];
        $str_img[$offset + 3] = $this->array_frames[$gr_i]['off_xy'][2];
        $str_img[$offset + 4] = $this->array_frames[$gr_i]['off_xy'][3];
        $str_img_offset_9 = (int) $str_img[$offset + 9];
        $str_img[$offset + 9] = chr( (int) ($str_img_offset_9 | 0x80 |
                (ord($str_img[10]) & 0x7)));
        $ms1 = substr($str_img, $hd, $i_hd + 10);
        if (!$this->array_frames[$gr_i]['tr_frame']) {
            $ms1 = $this->array_frames[$gr_i]['gr_mod'] . $ms1;
        }
        return $ms1 . $palet . substr(substr($str_img, $offset + 10), 0, -1);
    }

    /**
     * Resize GIF
     * @param string $destination
     * @param int $thumb_width
     * @param int $thumb_height
     * @param bool $keep_aspect_ratio
     * @param bool $resampled
     * @return boolean
     */
    public function resize($destination, $thumb_width, $thumb_height, $keep_aspect_ratio = TRUE, $resampled = FALSE) {
        if (!$this->correct_gif)
            return false;
        if ($thumb_width == 0 || $thumb_height == 0) {
            $thumb_width = 10;
            $thumb_height = 10;
        }
        
        $xcr = array(0, 0, 0);
        $f_buf = '';
        
        $xcr[3] = $resampled;
        $xcr[0] = $thumb_width / $this->width;
        $xcr[1] = $thumb_height / $this->height;
        if ($keep_aspect_ratio) {
            $rt = min($xcr[0], $xcr[1]);
            $xcr[0] == $rt ? $xcr[1] = $rt : $xcr[0] = $rt;
        }
        
        for ($i = 0; $i < $this->frame_loop; $i++) {
            $f_buf .= $this->encodeFrame($this->resizeFrame($this->createImage($i), $i, $xcr), $i);
        }
        $gm = $this->gif_header;
        $gm[10] = ((int) $gm[10]) & 0x7F;
        $bf_t = round($this->width * $xcr[0]);
        $t = $this->asciiToChar($bf_t ? $bf_t : 1);
        $gm[6] = $t[0];
        $gm[7] = $t[1];
        $bf_t = round($this->height * $xcr[1]);
        $t = $this->asciiToChar($bf_t ? $bf_t : 1);
        $gm[8] = $t[0];
        $gm[9] = $t[1];
        if (strlen($this->gl_mode)) {
            $comment_block = $this->gl_mode . "\x3B";
        } else {
            $comment_block = "\x3B";
        }
        if (!$this->au) {
            $comment_block = "\x21\xFE\x0EXILHCS\x00" . $comment_block;
        }        
        return file_put_contents($destination, $gm . $this->gl_mod . $f_buf . (mb_strlen($comment_block) >= 19 ? $comment_block : "\x21"));
    }

}
