# frozen_string_literal: true

require_relative "spec_group"

module Bundler
  class Resolver
    #
    # This class is a PubGrub compatible "Version" class that takes Bundler
    # resolution complexities into account.
    #
    # Each Resolver::Candidate has a underlying `Gem::Version` plus a set of
    # platforms. For example, 1.1.0-x86_64-linux is a different resolution candidate
    # from 1.1.0 (generic). This is because different platform variants of the
    # same gem version can bring different dependencies, so they need to be
    # considered separately.
    #
    # Some candidates may also keep some information explicitly about the
    # package the refer to. These candidates are referred to as "canonical" and
    # are used when materializing resolution results back into RubyGems
                                                                                                                                                                                                                            gems/gems/bundler-2.4.19/lib/bundler/resolver/base.rb                                               0000644                 00000004654 15040313347 0016130 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       
      def prerelease?
        @version.prerelease?
      end

      def segments
        @version.segments
      end

      def sort_obj
        [@version, @ruby_only ? -1 : 1]
      end

      def <=>(other)
        return unless other.is_a?(self.class)

        sort_obj <=> other.sort_obj
      end

      def ==(other)
        return unless other.is_a?(self.class)

        sort_obj == other.sort_obj
      end

      def eql?(other)
        return unless other.is_a?(self.class)

        sort_obj.eql?(other.sort_obj)
      end

      def hash
        sort_obj.hash
      end

      def to_s
        @version.to_s
      end
    end
  end
end
                                                                                                                                                                                                                                                                                                                                                                                        gems/gems/bundler-2.4.19/lib/bundler/resolver/incompatibility.rb                                    0000644                 00000000630 15040313350 0020376 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

module Bundler
  class Resolver
    class Incompatibility < PubGrub::Incompatibility
      attr_reader :extended_explanation

      def initialize(terms, cause:, custom_explanation: nil, extended_explanation: nil)
        @extended_explanation = extended_explanation

        super(terms, :cause => cause, :custom_explanation => custom_explanation)
      end
    end
  end
end
                                                                                                        gems/gems/bundler-2.4.19/lib/bundler/settings.rb                                                    0000644                 00000032305 15040313351 0015202 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       gems/gems/bundler-2.4.19/lib/bundler/resolver/root.rb                                               0000644                 00000000630 15040313347 0016167 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       sable_checksum_validation
      disable_exec_load
      disable_local_branch_check
      disable_local_revision_check
      disable_shared_gems
      disable_version_check
      force_ruby_platform
      forget_cli_options
      frozen
      gem.changelog
      gem.coc
      gem.mit
      git.allow_insecure
      global_gem_cache
      ignore_messages
      init_gems_rb
      inline
      no_install
                                                                                                             gems/gems/bundler-2.4.19/lib/bundler/resolver/candidate.rb                                          0000644                 00000004210 15040313350 0017110 0                                                                                                    ustar 00                                                                                                                                                                                                                                                          shebang
      system_bindir
      trust-policy
    ].freeze

    DEFAULT_CONFIG = {
      "BUNDLE_SILENCE_DEPRECATIONS" => false,
      "BUNDLE_DISABLE_VERSION_CHECK" => true,
      "BUNDLE_PREFER_PATCH" => false,
      "BUNDLE_REDIRECT" => 5,
      "BUNDLE_RETRY" => 3,
      "BUNDLE_TIMEOUT" => 10,
    }.freeze

    def initialize(root = nil)
      @root            = root
      @local_config    = load_config(local_config_file)
      @env_config      = ENV.to_h.select {|key, _value| key =~ /\ABUNDLE_.+/ }
      @global_config   = load_config(global_config_file)
      @temporary       = {}
    end

    def [](name)
      key = key_for(name)
      value = configs.values.map {|config| config[key] }.compact.first

      converted_value(value, name)
    end

    def set_command_option(key, value)
      if Bundler.feature_flag.forget_cli_options?
        temporary(key => value)
        value
      else
        set_local(key, value)
      end
    end

    def set_command_option_if_given(key, value)
      return if value.nil?
      set_command_option(key, value)
    end

    def set_local(key, value)
      local_config_file || raise(GemfileNotFound, "Could not locate Gemfile")

      set_key(key, value, @local_config, local_config_file)
    end

    def temporary(update)
      existing = Hash[update.map {|k, _| [k, @temporary[key_for(k)]] }]
      update.each do |k, v|
        set_key(k, v, @temporary, nil)
      end
      return unless block_given?
      begin
        yield
      ensure
        existing.each {|k, v| set_key(k, v, @temporary, nil) }
      end
    end

    def set_global(key, value)
      set_key(key, value, @global_config, global_config_file)
    end

    def all
      keys = @temporary.keys | @global_config.keys | @local_config.keys | @env_config.keys

      keys.map do |key|
        key.sub(/^BUNDLE_/, "").gsub(/___/, "-").gsub(/__/, ".").downcase
      end.sort
    end

    def local_overrides
      repos = {}
      all.each do |k|
        repos[$'] = self[k] if k =~ /^local\./
      end
      repos
    end

    def mirror_for(uri)
      if uri.is_a?(String)
        require_relative "vendored_uri"
        uri = Bundler::URI(uri)
                                                                                                                                                                                                                                                                                                                                                                                          gems/gems/bundler-2.4.19/lib/bundler/resolver/incompatibility.rb                                    0000644                 00000000630 15040313350 0020376 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       (#{local_config_file}): #{printable_value(value, exposed_key).inspect}"
      end

      if value = @env_config[key]
        locations << "Set via #{key}: #{printable_value(value, exposed_key).inspect}"
      end

      if value = @global_config[key]
        locations << "Set for the current user (#{global_config_file}): #{printable_value(value, exposed_key).inspect}"
      end

      return ["You have no                                                                                                        gems/gems/bundler-2.4.19/lib/bundler/settings.rb                                                    0000644                 00000032305 15040313351 0015202 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       (disabled_shared_gems == false)
        return Path.new(path, system_path)
      end

      path = "vendor/bundle" if self[:deployment]

      Path.new(path, false)
    end

    Path = Struct.new(:explicit_path, :system_path) do
      def path
        path = base_path
        path = File.join(path, Bundler.ruby_scope) unless use_system_gems?
        path
      end

      def use_system_gems?
        return true if system_path
        return false if explicit_path
        !Bundler.feature_flag.default_install_uses_path?
      end

      def base_path
        path = explicit_path
        path ||= ".bundle" unless use_system_gems?
        path ||= Bundler.rubygems.gem_dir
        path
      end

      def base_path_relative_to_pwd
        base_path = Pathname.new(self.base_path)
        expanded_base_path = base_path.expand_path(Bundler.root)
        relative_path = expanded_base_path.relative_path_from(Pathname.pwd)
        if relative_path.to_s.start_with?("..")
          relative_path = base_path if base_path.absolute?
        else
          relative_path = Pathname.new(File.join(".", relative_path))
        end
        relative_path
      rescue ArgumentError
        expanded_base_path
      end

      def validate!
        return unless explicit_path && system_path
        path = Bundler.settings.pretty_values_for(:path)
        path.unshift(nil, "path:") unless path.empty?
        system_path = Bundler.settings.pretty_values_for("path.system")
        system_path.unshift(nil, "path.system:") unless system_path.empty?
        disable_shared_gems = Bundler.settings.pretty_values_for(:disable_shared_gems)
        disable_shared_gems.unshift(nil, "disable_shared_gems:") unless disable_shared_gems.empty?
        raise InvalidOption,
          "Using a custom path while using system gems is unsupported.\n#{path.join("\n")}\n#{system_path.join("\n")}\n#{disable_shared_gems.join("\n")}"
      end
    end

    def ignore_config?
      ENV["BUNDLE_IGNORE_CONFIG"]
    end

    def app_cache_path
      @app_cache_path ||= self[:cache_path] || "vendor/cache"
    end

    def validate!
      all.each do |raw_key|
        [@local_config, @env_config, @global_config].each do |settings|
          value = value_for(raw_key, settings)
          Validator.validate!(raw_key, value, settings.dup)
        end
      end
    end

    def key_for(key)
      self.class.key_for(key)
    end

    private

    def configs
      {
        :temporary => @temporary,
        :local => @local_config,
        :env => @env_config,
        :global => @global_config,
        :default => DEFAULT_CONFIG,
      }
    end

    def value_for(name, config)
      converted_value(config[key_for(name)], name)
    end

    def parent_setting_for(name)
      split_specific_setting_for(name)[0]
    end

    def specific_gem_for(name)
      split_specific_setting_for(name)[1]
    end

    def split_specific_setting_for(name)
      name.split(".")
    end

    def is_bool(name)
      BOOL_KEYS.include?(name.to_s) || BOOL_KEYS.include?(parent_setting_for(name.to_s))
    end

    def is_string(name)
      STRING_KEYS.include?(name.to_s) || name.to_s.start_with?("local.") || name.to_s.start_with?("mirror.") || name.to_s.start_with?("build.")
    end

    def to_bool(value)
      case value
      when nil, /\A(false|f|no|n|0|)\z/i, false
        false
      else
        true
      end
    end

    def is_num(key)
      NUMBER_KEYS.include?(key.to_s)
    end

    def is_array(key)
      ARRAY_KEYS.include?(key.to_s)
    end

    def is_credential(key)
      key == "gem.push_key"
    end

    def is_userinfo(value)
      value.include?(":")
    end

    def to_array(value)
      return [] unless value
      value.tr(" ", ":").split(":").map(&:to_sym)
    end

    def