
    # At present it only checks whether it contains plugins.rb file
    #
    # @param [Pathname] plugin_path the path plugin is installed at
    # @raise [MalformattedPlugin] if plugins.rb file is not found
    def validate_plugin!(plugin_path)
      plugin_file = plugin_path.join(PLUGIN_FILE_NAME)
      raise MalformattedPlugin, "#{PLUGIN_FILE_NAME} was not found in the plugin." unless plugin_file.file?
    end

    # Validates and registers a plugin.
    #
    # @param [String] name the name of the plugin
    # @param [Specification] spec of installed plugin
    # @param [Boolean] optional_plugin, removed if there is conflict with any
    #                     other plugin (used for default source plugins)
    #
    # @raise [PluginInstallError] if validation or registration raises any error
    def save_plugin(name, spec, optional_plugin = false)
      validate_plugin! Pathname.new(spec.full_gem_path)
      installed = register_plugin(name, spec, optional_plugin)
      Bundler.ui.info "Installed plugin #{name}" if installed
    rescue PluginError => e
      raise PluginInstallError, "Failed to install plugin `#{spec.name}`, due to #{e.class} (#{e.message})"
    end

    # Runs the plugins.rb file in an isolated namespace, records the plugin
    # actions it registers for and then passes the data to index to be stored.
    #
    # @param [String] name the name of the plugin
    # @param [Specification] spec of installed plugin
    # @param [Boolean] optional_plugin, removed if there is conflict with any
    #                     other plugin (used for default source plugins)
    #
    # @raise [MalformattedPlugin] if plugins.rb raises any error
    def register_plugin(name, spec, optional_plugin = false)
      commands = @commands
      sources = @sources
      hooks = @hooks_by_event

      @commands = {}
      @sources = {}
      @hooks_by_event = Hash.new {|h, k| h[k] = [] }

      load_paths = spec.load_paths
      Bundler.rubygems.add_to_load_path(load_paths)
      path = Pathname.new spec.full_gem_path

      begin
        load path.join(PLUGIN_FILE_NAME), true
      rescue StandardError => e
        raise MalformattedPlugin, "#{e.class}: #{e.message}"
      end

      if optional_plugin && @sources.keys.any? {|s| source? s }
        Bundler.rm_rf(path)
        false
      else
        index.register_plugin(name, path.to_s, load_paths, @commands.keys,
          @sources.keys, @hooks_by_event.keys)
        true
      end
    ensure
      @commands = commands
      @sources = sources
      @hooks_by_event = hooks
    end

    # Executes the plugins.rb file
    #
    # @param [String] name of the plugin
    def load_plugin(name)
      return unless name && !name.empty?

      # Need to ensure before this that plugin root where the rest of gems
      # are installed to be on load path to support plugin deps. Currently not
      # done to avoid conflicts
      path = index.plugin_path(name)

      Bundler.rubygems.add_to_load_path(index.load_paths(name))

      load path.join(PLUGIN_FILE_NAME)
