 # instance and the value can be either a +String+ or +Array+ instance.
    def initialize(body = nil, status = 200, headers = {})
      @status = status.to_i

      unless headers.is_a?(Hash)
        warn "Providing non-hash headers to Rack::Response is deprecated and will be removed in Rack 3.1", uplevel: 1
      end

      @headers = Headers.new
      # Convert headers input to a plain hash with lowercase keys.
      headers.each do |k, v|
        @headers[k] = v
      end

      @writer = self.method(:append)

      @block = nil

      # Keep track of whether we have expanded the user supplied body.
      if body.nil?
        @body = []
        @buffered = true
        @length = 0
      elsif body.respond_to?(:to_str)
        @body = [body]
        @buffered = true
        @length = body.to_str.bytesize
      else
        @body = body
        @buffered = nil # undetermined as of yet.
        @length = 0
      end

      yield self if block_given?
    end

    def redirect(target, status = 302)
      self.status = status
      self.location = target
    end

    def chunked?
      CHUNKED == get_header(TRANSFER_ENCODING)
    end

    def no_entity_body?
      # The response body is an enumerable body and it is not allowed to have an entity body.
      @body.respond_to?(:each) && STATUS_WITH_NO_ENTITY_BODY[@status]
    end
    
    # Generate a response array consistent with the requirements of the SPEC.
    # @return [Array] a 3-tuple suitable of `[status, headers, body]`
    # which is suitable to be returned from the middleware `#call(env)` method.
    def finish(&block)
      if no_entity_body?
        delete_header CONTENT_TYPE
        delete_header CONTENT_LENGTH
        close
        return [@status, @headers, []]
      else
        if block_given?
          @block = block
          return [@status, @headers, self]
        else
          return [@status, @headers, @body]
        end
      end
    end

    alias to_a finish           # For *response

    def each(&callback)
      @body.each(&callback)
      @buffered = true

      if @block
        @writer = callback
        @block.call(self)
      end
    end

    # Append to body and update content-length.
    #
    # NOTE: Do not mix #write and direct #body access!
    #
    def write(chunk)
      buffered_body!

      @writer.call(chunk.to_s)
    end

    def close
      @body.close if @body.respond_to?(:close)
    end

    def empty?
      @block == nil && @body.empty?
    end

    def has_header?(key)
      raise ArgumentError unless key.is_a?(String)
      @headers.key?(key)
    end
    def get_header(key)
      raise ArgumentError unless key.is_a?(String)
      @headers[key]
    end
    def set_header(key, value)
      raise ArgumentError unless key.is_a?(String)
      @headers[key] = value
    end
    def delete_header(key)
      raise ArgumentError unless key.is_a?(String)
      @headers.delete key
    end

    alias :[] :get_header
    alias :[]= :set_header

    module Helpers
      def invalid?;             status < 100 || status >= 600;        end

      def informational?;       status >= 100 && status < 200;        end
      def successful?;          status >= 200 && status < 300;        end
      def redirection?;         status >= 300 && status < 400;        end
      def client_error?;        status >= 400 && status < 500;        end
      def server_error?;        status >= 500 && status < 600;        end

      def ok?;                  status == 200;                        end
      def created?;             status == 201;                        end
      def accepted?;            status == 202;                        end
      def no_content?;          status == 204;                        end
      def moved_permanently?;   status == 301;                        end
      def bad_request?;         status == 400;                        end
      def unauthorized?;        status == 401;                        end
      def forbidden?;           status == 403;                        end
      def not_found?;           status == 404;                        end
      def method_not_allowed?;  status == 405;                        end
      def not_acceptable?;      status == 406;                        end
      def request_timeout?;     status == 408;                        end
      def precondition_failed?; status == 412;                        end
      def unprocessable?;       status == 422;                        end

      def redirect?;            [301, 302, 303, 307, 308].include? status; end

      def include?(header)
        has_header?(header)
      end

      # Add a header that may have multiple values.
      #
      # Example:
      #   response.add_header 'vary', 'accept-encoding'
      #   response.add_header 'vary', 'cookie'
      #
      #   assert_equal 'accept-encoding,cookie', response.get_header('vary')
      #
      # http://www.w3.org/Protocols/rfc2616/rfc2616-sec4.html#sec4.2
      def add_header(key, value)
        raise ArgumentError unless key.is_a?(String)

        if value.nil?
          return get_header(key)
        end

        value = value.to_s

        if header = get_header(key)
          if header.is_a?(Array)
            header << value
          else
            set_header(key, [header, value])
          end
        else
          set_header(key, value)
        end
      end

      # Get the content type of the response.
      def content_type
        get_header CONTENT_TYPE
      end

      # Set the content type of the response.
      def content_type=(content_type)
        set_header CONTENT_TYPE, content_type
      end

      def media_type
        MediaType.type(content_type)
      end

      def media_type_params
        MediaType.params(content_type)
      end

      def content_length
        cl = get_header CONTENT_LENGTH
        cl ? cl.to_i : cl
      end

      def location
        get_header "location"
      end

      def location=(location)
        set_header "location", location
      end

      def set_cookie(key, value)
        add_header SET_COOKIE, Utils.set_cookie_header(key, value)
      end

      def delete_cookie(key, value = {})
        set_header(SET_COOKIE,
          Utils.delete_set_cookie_header!(
            get_header(SET_COOKIE), key, value
          )
        )
      end

      def set_cookie_header
        get_header SET_COOKIE
      end

      def set_cookie_header=(value)
        set_header SET_COOKIE, value
      end

      def cache_control
        get_header CACHE_CONTROL
      end

      def cache_control=(value)
        set_header CACHE_CONTROL, value
      end

      # Specifies that the content shouldn't be cached. Overrides `cache!` if already called.
      def do_not_cache!
        set_header CACHE_CONTROL, "no-cache, must-revalidate"
        set_header EXPIRES, Time.now.httpdate
      end

      # Specify that the content should be cached.
      # @param duration [Integer] The number of seconds until the cache expires.
      # @option directive [String] The cache control directive, one of "public", "private", "no-cache" or "no-store".
      def cache!(duration = 3600, directive: "public")
        unless headers[CACHE_CONTROL] =~ /no-cache/
          set_header CACHE_CONTROL, "#{directive}, max-age=#{duration}"
          set_header EXPIRES, (Time.now + duration).httpdate
        end
      end

      def etag
        get_header ETAG
      end

      def etag=(value)
        set_header ETAG, value
      end

    protected

      def buffered_body!
        if @buffered.nil?
          if @body.is_a?(Array)
            # The user supplied body was an array:
            @body = @body.compact
            @body.each do |part|
              @length += part.to_s.bytesize
            end
          elsif @body.respond_to?(:each)
            # Turn the user supplied body into a buffered array:
            body = @body
            @body = Array.new

            body.each do |part|
              @writer.call(part.to_s)
            end

            body.close if body.respond_to?(:close)

            @buffered = true
          else
            @buffered = false
          end
        end

        return @buffered
      end

      def append(chunk)
        @body << chunk

        unless chunked?
          @length += chunk.bytesize
          set_header(CONTENT_LENGTH, @length.to_s)
        end

        return chunk
      end
    end

    include Helpers

    class Raw
      include Helpers

      attr_reader :headers
      attr_accessor :status

      def initialize(status, headers)
        @status = status
        @headers = headers
      end

      def has_header?(key)
        headers.key?(key)
      end

      def get_header(key)
        headers[key]
      end

      def set_header(key, value)
        headers[key] = value
      end

      def delete_header(key)
        headers.delete(key)
      end
    end
  end
end
                                                                                                                                                                                                                                                                                           gems/gems/rack-3.0.8/lib/rack/mime.rb                                                               0000644                 00000101301 15040313411 0012744 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

module Rack
  module Mime
    # Returns String with mime type if found, otherwise use +fallback+.
    # +ext+ should be filename extension in the '.ext' format that
    #       File.extname(file) returns.
    # +fallback+ may be any object
    #
    # Also see the documentation for MIME_TYPES
    #
    # Usage:
    #     Rack::Mime.mime_type('.foo')
    #
    # This is a shortcut for:
    #     Rack::Mime::MIME_TYPES.fetch('.foo', 'application/octet-stream')

    def mime_type(ext, fallback = 'application/octet-stream')
      MIME_TYPES.fetch(ext.to_s.downcase, fallback)
    end
    module_function :mime_type

    # Returns true if the given value is a mime match for the given mime match
    # specification, false otherwise.
    #
    #    Rack::Mime.match?('text/html', 'text/*') => true
    #    Rack::Mime.match?('text/plain', '*') => true
    #    Rack::Mime.match?('text/html', 'application/json') => false

    def match?(value, matcher)
      v1, v2 = value.split('/', 2)
      m1, m2 = matcher.split('/', 2)

      (m1 == '*' || v1 == m1) && (m2.nil? || m2 == '*' || m2 == v2)
    end
    module_function :match?

    # List of most common mime-types, selected various sources
    # according to their usefulness in a webserving scope for Ruby
    # users.
    #
    # To amend this list with your local mime.types list you can use:
    #
    #     require 'webrick/httputils'
    #     list = WEBrick::HTTPUtils.load_mime_types('/etc/mime.types')
    #     Rack::Mime::MIME_TYPES.merge!(list)
    #
    # N.B. On Ubuntu the mime.types file does not include the leading period, so
    # users may need to modify the data before merging into the hash.

    MIME_TYPES = {
      ".123"       => "application/vnd.lotus-1-2-3",
      ".3dml"      => "text/vnd.in3d.3dml",
      ".3g2"       => "video/3gpp2",
      ".3gp"       => "video/3gpp",
      ".a"         => "application/octet-stream",
      ".acc"       => "application/vnd.americandynamics.acc",
      ".ace"       => "application/x-ace-compressed",
      ".acu"       => "application/vnd.acucobol",
      ".aep"       => "application/vnd.audiograph",
      ".afp"       => "application/vnd.ibm.modcap",
      ".ai"        => "application/postscript",
      ".aif"       => "audio/x-aiff",
      ".aiff"      => "audio/x-aiff",
      ".ami"       => "application/vnd.amiga.ami",
      ".apng"      => "image/apng",
      ".appcache"  => "text/cache-manifest",
      ".apr"       => "application/vnd.lotus-approach",
      ".asc"       => "application/pgp-signature",
      ".asf"       => "video/x-ms-asf",
      ".asm"       => "text/x-asm",
      ".aso"       => "application/vnd.accpac.simply.aso",
      ".asx"       => "video/x-ms-asf",
      ".atc"       => "application/vnd.acucorp",
      ".atom"      => "application/atom+xml",
      ".atomcat"   => "application/atomcat+xml",
      ".atomsvc"   => "application/atomsvc+xml",
      ".atx"       => "application/vnd.antix.game-component",
      ".au"        => "audio/basic",
      ".avi"       => "video/x-msvideo",
      ".avif"      => "image/avif",
      ".bat"       => "application/x-msdownload",
      ".bcpio"     => "application/x-bcpio",
      ".bdm"       => "application/vnd.syncml.dm+wbxml",
      ".bh2"       => "application/vnd.fujitsu.oasysprs",
      ".bin"       => "application/octet-stream",
      ".bmi"       => "application/vnd.bmi",
      ".bmp"       => "image/bmp",
      ".box"       => "application/vnd.previewsystems.box",
      ".btif"      => "image/prs.btif",
      ".bz"        => "application/x-bzip",
      ".bz2"       => "application/x-bzip2",
      ".c"         => "text/x-c",
      ".c4g"       => "application/vnd.clonk.c4group",
      ".cab"       => "application/vnd.ms-cab-compressed",
      ".cc"        => "text/x-c",
      ".ccxml"     => "application/ccxml+xml",
      ".cdbcmsg"   => "application/vnd.contact.cmsg",
      ".cdkey"     => "application/vnd.mediastation.cdkey",
      ".cdx"       => "chemical/x-cdx",
      ".cdxml"     => "application/vnd.chemdraw+xml",
      ".cdy"       => "application/vnd.cinderella",
      ".cer"       => "application/pkix-cert",
      ".cgm"       => "image/cgm",
      ".chat"      => "application/x-chat",
      ".chm"       => "application/vnd.ms-htmlhelp",
      ".chrt"      => "application/vnd.kde.kchart",
      ".cif"       => "chemical/x-cif",
      ".cii"       => "application/vnd.anser-web-certificate-issue-initiation",
      ".cil"       => "application/vnd.ms-artgalry",
      ".cla"       => "application/vnd.claymore",
      ".class"     => "application/octet-stream",
      ".clkk"      => "application/vnd.crick.clicker.keyboard",
      ".clkp"      => "application/vnd.crick.clicker.palette",
      ".clkt"      => "application/vnd.crick.clicker.template",
      ".clkw"      => "application/vnd.crick.clicker.wordbank",
      ".clkx"      => "application/vnd.crick.clicker",
      ".clp"       => "application/x-msclip",
      ".cmc"       => "application/vnd.cosmocaller",
      ".cmdf"      => "chemical/x-cmdf",
      ".cml"       => "chemical/x-cml",
      ".cmp"       => "application/vnd.yellowriver-custom-menu",
      ".cmx"       => "image/x-cmx",
      ".com"       => "application/x-msdownload",
      ".conf"      => "text/plain",
      ".cpio"      => "application/x-cpio",
      ".cpp"       => "text/x-c",
      ".cpt"       => "application/mac-compactpro",
      ".crd"       => "application/x-mscardfile",
      ".crl"       => "application/pkix-crl",
      ".crt"       => "application/x-x509-ca-cert",
      ".csh"       => "application/x-csh",
      ".csml"      => "chemical/x-csml",
      ".csp"       => "application/vnd.commonspace",
      ".css"       => "text/css",
      ".csv"       => "text/csv",
      ".curl"      => "application/vnd.curl",
      ".cww"       => "application/prs.cww",
      ".cxx"       => "text/x-c",
      ".daf"       => "application/vnd.mobius.daf",
      ".davmount"  => "application/davmount+xml",
      ".dcr"       => "application/x-director",
      ".dd2"       => "application/vnd.oma.dd2+xml",
      ".ddd"       => "application/vnd.fujixerox.ddd",
      ".deb"       => "application/x-debian-package",
      ".der"       => "application/x-x509-ca-cert",
      ".dfac"      => "application/vnd.dreamfactory",
      ".diff"      => "text/x-diff",
      ".dis"       => "application/vnd.mobius.dis",
      ".djv"       => "image/vnd.djvu",
      ".djvu"      => "image/vnd.djvu",
      ".dll"       => "application/x-msdownload",
      ".dmg"       => "application/octet-stream",
      ".dna"       => "application/vnd.dna",
      ".doc"       => "application/msword",
      ".docm"      => "application/vnd.ms-word.document.macroEnabled.12",
      ".docx"      => "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      ".dot"       => "application/msword",
      ".dotm"      => "application/vnd.ms-word.template.macroEnabled.12",
      ".dotx"      => "application/vnd.openxmlformats-officedocument.wordprocessingml.template",
      ".dp"        => "application/vnd.osgi.dp",
      ".dpg"       => "application/vnd.dpgraph",
      ".dsc"       => "text/prs.lines.tag",
      ".dtd"       => "application/xml-dtd",
      ".dts"       => "audio/vnd.dts",
      ".dtshd"     => "audio/vnd.dts.hd",
      ".dv"        => "video/x-dv",
      ".dvi"       => "application/x-dvi",
      ".dwf"       => "model/vnd.dwf",
      ".dwg"       => "image/vnd.dwg",
      ".dxf"       => "image/vnd.dxf",
      ".dxp"       => "application/vnd.spotfire.dxp",
      ".ear"       => "application/java-archive",
      ".ecelp4800" => "audio/vnd.nuera.ecelp4800",
      ".ecelp7470" => "audio/vnd.nuera.ecelp7470",
      ".ecelp9600" => "audio/vnd.nuera.ecelp9600",
      ".ecma"      => "application/ecmascript",
      ".edm"       => "application/vnd.novadigm.edm",
      ".edx"       => "application/vnd.novadigm.edx",
      ".efif"      => "application/vnd.picsel",
      ".ei6"       => "application/vnd.pg.osasli",
      ".eml"       => "message/rfc822",
      ".eol"       => "audio/vnd.digital-winds",
      ".eot"       => "application/vnd.ms-fontobject",
      ".eps"       => "application/postscript",
      ".es3"       => "application/vnd.eszigno3+xml",
      ".esf"       => "application/vnd.epson.esf",
      ".etx"       => "text/x-setext",
      ".exe"       => "application/x-msdownload",
      ".ext"       => "application/vnd.novadigm.ext",
      ".ez"        => "application/andrew-inset",
      ".ez2"       => "application/vnd.ezpix-album",
      ".ez3"       => "application/vnd.ezpix-package",
      ".f"         => "text/x-fortran",
      ".f77"       => "text/x-fortran",
      ".f90"       => "text/x-fortran",
      ".fbs"       => "image/vnd.fastbidsheet",
      ".fdf"       => "application/vnd.fdf",
      ".fe_launch" => "application/vnd.denovo.fcselayout-link",
      ".fg5"       => "application/vnd.fujitsu.oasysgp",
      ".fli"       => "video/x-fli",
      ".flif"      => "image/flif",
      ".flo"       => "application/vnd.micrografx.flo",
      ".flv"       => "video/x-flv",
      ".flw"       => "application/vnd.kde.kivio",
      ".flx"       => "text/vnd.fmi.flexstor",
      ".fly"       => "text/vnd.fly",
      ".fm"        => "application/vnd.framemaker",
      ".fnc"       => "application/vnd.frogans.fnc",
      ".for"       => "text/x-fortran",
      ".fpx"       => "image/vnd.fpx",
      ".fsc"       => "application/vnd.fsc.weblaunch",
      ".fst"       => "image/vnd.fst",
      ".ftc"       => "application/vnd.fluxtime.clip",
      ".fti"       => "application/vnd.anser-web-funds-transfer-initiation",
      ".fvt"       => "video/vnd.fvt",
      ".fzs"       => "application/vnd.fuzzysheet",
      ".g3"        => "image/g3fax",
      ".gac"       => "application/vnd.groove-account",
      ".gdl"       => "model/vnd.gdl",
      ".gem"       => "application/octet-stream",
      ".gemspec"   => "text/x-script.ruby",
      ".ghf"       => "application/vnd.groove-help",
      ".gif"       => "image/gif",
      ".gim"       => "application/vnd.groove-identity-message",
      ".gmx"       => "application/vnd.gmx",
      ".gph"       => "application/vnd.flographit",
      ".gqf"       => "application/vnd.grafeq",
      ".gram"      => "application/srgs",
      ".grv"       => "application/vnd.groove-injector",
      ".grxml"     => "application/srgs+xml",
      ".gtar"      => "application/x-gtar",
      ".gtm"       => "application/vnd.groove-tool-message",
      ".gtw"       => "model/vnd.gtw",
      ".gv"        => "text/vnd.graphviz",
      ".gz"        => "application/x-gzip",
      ".h"         => "text/x-c",
      ".h261"      => "video/h261",
      ".h263"      => "video/h263",
      ".h264"      => "video/h264",
      ".hbci"      => "application/vnd.hbci",
      ".hdf"       => "application/x-hdf",
      ".heic"      => "image/heic",
      ".heics"     => "image/heic-sequence",
      ".heif"      => "image/heif",
      ".heifs"     => "image/heif-sequence",
      ".hh"        => "text/x-c",
      ".hlp"       => "application/winhlp",
      ".hpgl"      => "application/vnd.hp-hpgl",
      ".hpid"      => "application/vnd.hp-hpid",
      ".hps"       => "application/vnd.hp-hps",
      ".hqx"       => "application/mac-binhex40",
      ".htc"       => "text/x-component",
      ".htke"      => "application/vnd.kenameaapp",
      ".htm"       => "text/html",
      ".html"      => "text/html",
      ".hvd"       => "application/vnd.yamaha.hv-dic",
      ".hvp"       => "application/vnd.yamaha.hv-voice",
      ".hvs"       => "application/vnd.yamaha.hv-script",
      ".icc"       => "application/vnd.iccprofile",
      ".ice"       => "x-conference/x-cooltalk",
      ".ico"       => "image/vnd.microsoft.icon",
      ".ics"       => "text/calendar",
      ".ief"       => "image/ief",
      ".ifb"       => "text/calendar",
      ".ifm"       => "application/vnd.shana.informed.formdata",
      ".igl"       => "application/vnd.igloader",
      ".igs"       => "model/iges",
      ".igx"       => "application/vnd.micrografx.igx",
      ".iif"       => "application/vnd.shana.informed.interchange",
      ".imp"       => "application/vnd.accpac.simply.imp",
      ".ims"       => "application/vnd.ms-ims",
      ".ipk"       => "application/vnd.shana.informed.package",
      ".irm"       => "application/vnd.ibm.rights-management",
      ".irp"       => "application/vnd.irepository.package+xml",
      ".iso"       => "application/octet-stream",
      ".itp"       => "application/vnd.shana.informed.formtemplate",
      ".ivp"       => "application/vnd.immervision-ivp",
      ".ivu"       => "application/vnd.immervision-ivu",
      ".jad"       => "text/vnd.sun.j2me.app-descriptor",
      ".jam"       => "application/vnd.jam",
      ".jar"       => "application/java-archive",
      ".java"      => "text/x-java-source",
      ".jisp"      => "application/vnd.jisp",
      ".jlt"       => "application/vnd.hp-jlyt",
      ".jnlp"      => "application/x-java-jnlp-file",
      ".joda"      => "application/vnd.joost.joda-archive",
      ".jp2"       => "image/jp2",
      ".jpeg"      => "image/jpeg",
      ".jpg"       => "image/jpeg",
      ".jpgv"      => "video/jpeg",
      ".jpm"       => "video/jpm",
      ".js"        => "application/javascript",
      ".json"      => "application/json",
      ".karbon"    => "application/vnd.kde.karbon",
      ".kfo"       => "application/vnd.kde.kformula",
      ".kia"       => "application/vnd.kidspiration",
      ".kml"       => "application/vnd.google-earth.kml+xml",
      ".kmz"       => "application/vnd.google-earth.kmz",
      ".kne"       => "application/vnd.kinar",
      ".kon"       => "application/vnd.kde.kontour",
      ".kpr"       => "application/vnd.kde.kpresenter",
      ".ksp"       => "application/vnd.kde.kspread",
      ".ktz"       => "application/vnd.kahootz",
      ".kwd"       => "application/vnd.kde.kword",
      ".latex"     => "application/x-latex",
      ".lbd"       => "application/vnd.llamagraphics.life-balance.desktop",
      ".lbe"       => "application/vnd.llamagraphics.life-balance.exchange+xml",
      ".les"       => "application/vnd.hhe.lesson-player",
      ".link66"    => "application/vnd.route66.link66+xml",
      ".log"       => "text/plain",
      ".lostxml"   => "application/lost+xml",
      ".lrm"       => "application/vnd.ms-lrm",
      ".ltf"       => "application/vnd.frogans.ltf",
      ".lvp"       => "audio/vnd.lucent.voice",
      ".lwp"       => "application/vnd.lotus-wordpro",
      ".m3u"       => "audio/x-mpegurl",
      ".m3u8"      => "application/x-mpegurl",
      ".m4a"       => "audio/mp4a-latm",
      ".m4v"       => "video/mp4",
      ".ma"        => "application/mathematica",
      ".mag"       => "application/vnd.ecowin.chart",
      ".man"       => "text/troff",
      ".manifest"  => "text/cache-manifest",
      ".mathml"    => "application/mathml+xml",
      ".mbk"       => "application/vnd.mobius.mbk",
      ".mbox"      => "application/mbox",
      ".mc1"       => "application/vnd.medcalcdata",
      ".mcd"       => "application/vnd.mcd",
      ".mdb"       => "application/x-msaccess",
      ".mdi"       => "image/vnd.ms-modi",
      ".mdoc"      => "text/troff",
      ".me"        => "text/troff",
      ".mfm"       => "application/vnd.mfmp",
      ".mgz"       => "application/vnd.proteus.magazine",
      ".mid"       => "audio/midi",
      ".midi"      => "audio/midi",
      ".mif"       => "application/vnd.mif",
      ".mime"      => "message/rfc822",
      ".mj2"       => "video/mj2",
      ".mlp"       => "application/vnd.dolby.mlp",
      ".mmd"       => "application/vnd.chipnuts.karaoke-mmd",
      ".mmf"       => "application/vnd.smaf",
      ".mml"       => "application/mathml+xml",
      ".mmr"       => "image/vnd.fujixerox.edmics-mmr",
      ".mng"       => "video/x-mng",
      ".mny"       => "application/x-msmoney",
      ".mov"       => "video/quicktime",
      ".movie"     => "video/x-sgi-movie",
      ".mp3"       => "audio/mpeg",
      ".mp4"       => "video/mp4",
      ".mp4a"      => "audio/mp4",
      ".mp4s"      => "application/mp4",
      ".mp4v"      => "video/mp4",
      ".mpc"       => "application/vnd.mophun.certificate",
      ".mpd"       => "application/dash+xml",
      ".mpeg"      => "video/mpeg",
      ".mpg"       => "video/mpeg",
      ".mpga"      => "audio/mpeg",
      ".mpkg"      => "application/vnd.apple.installer+xml",
      ".mpm"       => "application/vnd.blueice.multipass",
      ".mpn"       => "application/vnd.mophun.application",
      ".mpp"       => "application/vnd.ms-project",
      ".mpy"       => "application/vnd.ibm.minipay",
      ".mqy"       => "application/vnd.mobius.mqy",
      ".mrc"       => "application/marc",
      ".ms"        => "text/troff",
      ".mscml"     => "application/mediaservercontrol+xml",
      ".mseq"      => "application/vnd.mseq",
      ".msf"       => "application/vnd.epson.msf",
      ".msh"       => "model/mesh",
      ".msi"       => "application/x-msdownload",
      ".msl"       => "application/vnd.mobius.msl",
      ".msty"      => "application/vnd.muvee.style",
      ".mts"       => "model/vnd.mts",
      ".mus"       => "application/vnd.musician",
      ".mvb"       => "application/x-msmediaview",
      ".mwf"       => "application/vnd.mfer",
      ".mxf"       => "application/mxf",
      ".mxl"       => "application/vnd.recordare.musicxml",
      ".mxml"      => "application/xv+xml",
      ".mxs"       => "application/vnd.triscape.mxs",
      ".mxu"       => "video/vnd.mpegurl",
      ".n"         => "application/vnd.nokia.n-gage.symbian.install",
      ".nc"        => "application/x-netcdf",
      ".ngdat"     => "application/vnd.nokia.n-gage.data",
      ".nlu"       => "application/vnd.neurolanguage.nlu",
      ".nml"       => "application/vnd.enliven",
      ".nnd"       => "application/vnd.noblenet-directory",
      ".nns"       => "application/vnd.noblenet-sealer",
      ".nnw"       => "application/vnd.noblenet-web",
      ".npx"       => "image/vnd.net-fpx",
      ".nsf"       => "application/vnd.lotus-notes",
      ".oa2"       => "application/vnd.fujitsu.oasys2",
      ".oa3"       => "application/vnd.fujitsu.oasys3",
      ".oas"       => "application/vnd.fujitsu.oasys",
      ".obd"       => "application/x-msbinder",
      ".oda"       => "application/oda",
      ".odc"       => "application/vnd.oasis.opendocument.chart",
      ".odf"       => "application/vnd.oasis.opendocument.formula",
      ".odg"       => "application/vnd.oasis.opendocument.graphics",
      ".odi"       => "application/vnd.oasis.opendocument.image",
      ".odp"       => "application/vnd.oasis.opendocument.presentation",
      ".ods"       => "application/vnd.oasis.opendocument.spreadsheet",
      ".odt"       => "application/vnd.oasis.opendocument.text",
      ".oga"       => "audio/ogg",
      ".ogg"       => "application/ogg",
      ".ogv"       => "video/ogg",
      ".ogx"       => "application/ogg",
      ".org"       => "application/vnd.lotus-organizer",
      ".otc"       => "application/vnd.oasis.opendocument.chart-template",
      ".otf"       => "application/vnd.oasis.opendocument.formula-template",
      ".otg"       => "application/vnd.oasis.opendocument.graphics-template",
      ".oth"       => "application/vnd.oasis.opendocument.text-web",
      ".oti"       => "application/vnd.oasis.opendocument.image-template",
      ".otm"       => "application/vnd.oasis.opendocument.text-master",
      ".ots"       => "application/vnd.oasis.opendocument.spreadsheet-template",
      ".ott"       => "application/vnd.oasis.opendocument.text-template",
      ".oxt"       => "application/vnd.openofficeorg.extension",
      ".p"         => "text/x-pascal",
      ".p10"       => "application/pkcs10",
      ".p12"       => "application/x-pkcs12",
      ".p7b"       => "application/x-pkcs7-certificates",
      ".p7m"       => "application/pkcs7-mime",
      ".p7r"       => "application/x-pkcs7-certreqresp",
      ".p7s"       => "application/pkcs7-signature",
      ".pas"       => "text/x-pascal",
      ".pbd"       => "application/vnd.powerbuilder6",
      ".pbm"       => "image/x-portable-bitmap",
      ".pcl"       => "application/vnd.hp-pcl",
      ".pclxl"     => "application/vnd.hp-pclxl",
      ".pcx"       => "image/x-pcx",
      ".pdb"       => "chemical/x-pdb",
      ".pdf"       => "application/pdf",
      ".pem"       => "application/x-x509-ca-cert",
      ".pfr"       => "application/font-tdpfr",
      ".pgm"       => "image/x-portable-graymap",
      ".pgn"       => "application/x-chess-pgn",
      ".pgp"       => "application/pgp-encrypted",
      ".pic"       => "image/x-pict",
      ".pict"      => "image/pict",
      ".pkg"       => "application/octet-stream",
      ".pki"       => "application/pkixcmp",
      ".pkipath"   => "application/pkix-pkipath",
      ".pl"        => "text/x-script.perl",
      ".plb"       => "application/vnd.3gpp.pic-bw-large",
      ".plc"       => "application/vnd.mobius.plc",
      ".plf"       => "application/vnd.pocketlearn",
      ".pls"       => "application/pls+xml",
      ".pm"        => "text/x-script.perl-module",
      ".pml"       => "application/vnd.ctc-posml",
      ".png"       => "image/png",
      ".pnm"       => "image/x-portable-anymap",
      ".pntg"      => "image/x-macpaint",
      ".portpkg"   => "application/vnd.macports.portpkg",
      ".pot"       => "application/vnd.ms-powerpoint",
      ".potm"      => "application/vnd.ms-powerpoint.template.macroEnabled.12",
      ".potx"      => "application/vnd.openxmlformats-officedocument.presentationml.template",
      ".ppa"       => "application/vnd.ms-powerpoint",
      ".ppam"      => "application/vnd.ms-powerpoint.addin.macroEnabled.12",
      ".ppd"       => "application/vnd.cups-ppd",
      ".ppm"       => "image/x-portable-pixmap",
      ".pps"       => "application/vnd.ms-powerpoint",
      ".ppsm"      => "application/vnd.ms-powerpoint.slideshow.macroEnabled.12",
      ".ppsx"      => "application/vnd.openxmlformats-officedocument.presentationml.slideshow",
      ".ppt"       => "application/vnd.ms-powerpoint",
      ".pptm"      => "application/vnd.ms-powerpoint.presentation.macroEnabled.12",
      ".pptx"      => "application/vnd.openxmlformats-officedocument.presentationml.presentation",
      ".prc"       => "application/vnd.palm",
      ".pre"       => "application/vnd.lotus-freelance",
      ".prf"       => "application/pics-rules",
      ".ps"        => "application/postscript",
      ".psb"       => "application/vnd.3gpp.pic-bw-small",
      ".psd"       => "image/vnd.adobe.photoshop",
      ".ptid"      => "application/vnd.pvi.ptid1",
      ".pub"       => "application/x-mspublisher",
      ".pvb"       => "application/vnd.3gpp.pic-bw-var",
      ".pwn"       => "application/vnd.3m.post-it-notes",
      ".py"        => "text/x-script.python",
      ".pya"       => "audio/vnd.ms-playready.media.pya",
      ".pyv"       => "video/vnd.ms-playready.media.pyv",
      ".qam"       => "application/vnd.epson.quickanime",
      ".qbo"       => "application/vnd.intu.qbo",
      ".qfx"       => "application/vnd.intu.qfx",
      ".qps"       => "application/vnd.publishare-delta-tree",
      ".qt"        => "video/quicktime",
      ".qtif"      => "image/x-quicktime",
      ".qxd"       => "application/vnd.quark.quarkxpress",
      ".ra"        => "audio/x-pn-realaudio",
      ".rake"      => "text/x-script.ruby",
      ".ram"       => "audio/x-pn-realaudio",
      ".rar"       => "application/x-rar-compress