   raise LintError, "a header value must be a String or Array of Strings, but the value of '#{key}' is a #{value.class}"
          end
        end
      end

      def check_header_value(key, value)
        ## such that each String instance must not contain characters below 037.
        if value =~ /[\000-\037]/
          raise LintError, "invalid header value #{key}: #{value.inspect}"
        end
      end

      ##
      ## === The content-type
      ##
      def check_content_type(status, headers)
        headers.each { |key, value|
          ## There must not be a <tt>content-type</tt> header key when the +Status+ is 1xx,
          ## 204, or 304.
          if key == "content-type"
            if Rack::Utils::STATUS_WITH_NO_ENTITY_BODY.key? status.to_i
              raise LintError, "content-type header found in #{status} response, not allowed"
            end
            return
          end
        }
      end

      ##
      ## === The content-length
      ##
      def check_content_length(status, headers)
        headers.each { |key, value|
          if key == 'content-length'
            ## There must not be a <tt>content-length</tt> header key when the
            ## +Status+ is 1xx, 204, or 304.
            if Rack::Utils::STATUS_WITH_NO_ENTITY_BODY.key? status.to_i
              raise LintError, "content-length header found in #{status} response, not allowed"
            end
            @content_length = value
          end
        }
      end

      def verify_content_length(size)
        if @head_request
          unless size == 0
            raise LintError, "Response body was given for HEAD request, but should be empty"
          end
        elsif @content_length
          unless @content_length == size.to_s
            raise LintError, "content-length header was #{@content_length}, but should be #{size}"
          end
        end
      end

      ##
      ## === The Body
      ##
      ## The Body is typically an +Array+ of +String+ instances, an enumerable
      ## that yields +String+ instances, a +Proc+ instance, or a File-like
      ## object.
      ##
      ## The Body must respond to +each+ or +call+. It may optionally respond
      ## to +to_path+ or +to_ary+. A Body that responds to +each+ is considered
      ## to be an Enumerable Body. A Body that responds to +call+ is considered
      ## to be a Streaming Body.
      ##
      ## A Body that responds to both +each+ and +call+ must be treated as an
      ## Enumerable Body, not a Streaming Body. If it responds to +each+, you
      ## must call +each+ and not +call+. If the Body doesn't respond to
      ## +each+, then you can assume it responds to +call+.
      ##
      ## The Body must either be consumed or returned. The Body is consumed by
      ## optionally calling either +each+ or +call+.
      ## Then, if the Body responds to +close+, it must be called to release
      ## any resources associated with the generation of the body.
      ## In other words, +close+ must always be called at least once; typically
      ## after the web server has sent the response to the client, but also in
      ## cases where the Rack application makes internal/virtual requests and
      ## discards the response.
      ##
      def close
        ##
        ## After calling +close+, the Body is considered closed and should not
        ## be consumed again.
        @closed = true

        ## If the original Body is replaced by a new Body, the new Body must
        ## also consume the original Body by calling +close+ if possible.
        @body.close if @body.respond_to?(:close)

        index = @lint.index(self)
        unless @env['rack.lint'][0..index].all? {|lint| lint.instance_variable_get(:@closed)}
          raise LintError, "Body has not been closed"
        end
      end

      def verify_to_path
        ##
        ## If the Body responds to +to_path+, it must return a +String+
        ## path for the local file system whose contents are identical
        ## to that produced by calling +each+; this may be used by the
        ## server as an alternative, possibly more efficient way to
        ## transport the response. The +to_path+ method does not consume
        ## the body.
        if @body.respond_to?(:to_path)
          unless ::File.exist? @body.to_path
            raise LintError, "The file identified by body.to_path does not exist"
          end
        end
      end

      ##
      ## ==== Enumerable Body
      ##
      def each
        ## The Enumerable Body must respond to +each+.
        raise LintError, "Enumerable Body must respond to each" unless @body.respond_to?(:each)

        ## It must only be called once.
        raise LintError, "Response body must only be invoked once (#{@invoked})" unless @invoked.nil?

        ## It must not be called after being closed.
        raise LintError, "Response body is already closed" if @closed

        @invoked = :each

        @body.each do |chunk|
          ## and must only yield String values.
          unless chunk.kind_of? String
            raise LintError, "Body yielded non-string value #{chunk.inspect}"
          end

          ##
          ## The Body itself should not be an instance of String, as this will
          ## break in Ruby 1.9.
          ##
          ## Middleware must not call +each+ directly on the Body.
          ## Instead, middleware can return a new Body that calls +each+ on the
          ## original Body, yielding at least once per iteration.
          if @lint[0] == self
            @env['rack.lint.body_iteration'] += 1
          else
            if (@env['rack.lint.body_iteration'] -= 1) > 0
              raise LintError, "New body must yield at least once per iteration of old body"
            end
          end

          @size += chunk.bytesize
          yield chunk
        end

        verify_content_length(@size)

        verify_to_path
      end

      BODY_METHODS = {to_ary: true, each: true, call: true, to_path: true}

      def to_path
        @body.to_path
      end

      def respond_to?(name, *)
        if BODY_METHODS.key?(name)
          @body.respond_to?(name)
        else
          super
        end
      end

      ##
      ## If the Body responds to +to_ary+, it must return an +Array+ whose
      ## contents are identical to that produced by calling +each+.
      ## Middleware may call +to_ary+ directly on the Body and return a new
      ## Body in its place. In other words, middleware can only process the
      ## Body directly if it responds to +to_ary+. If the Body responds to both
      ## +to_ary+ and +close+, its implementation of +to_ary+ must call
      ## +close+.
      def to_ary
        @body.to_ary.tap do |content|
          unless content == @body.enum_for.to_a
            raise LintError, "#to_ary not identical to contents produced by calling #each"
          end
        end
      ensure
        close
      end

      ##
      ## ==== Streaming Body
      ##
      def call(stream)
        ## The Streaming Body must respond to +call+.
        raise LintError, "Streaming Body must respond to call" unless @body.respond_to?(:call)

        ## It must only be called once.
        raise LintError, "Response body must only be invoked once (#{@invoked})" unless @invoked.nil?

        ## It must not be called after being closed.
        raise LintError, "Response body is already closed" if @closed

        @invoked = :call

        ## It takes a +stream+ argument.
        ##
        ## The +stream+ argument must implement:
        ## <tt>read, write, <<, flush, close, close_read, close_write, closed?</tt>
        ##
        @body.call(StreamWrapper.new(stream))
      end

      class StreamWrapper
        extend Forwardable

        ## The semantics of these IO methods must be a best effort match to
        ## those of a normal Ruby IO or Socket object, using standard arguments
        ## and raising standard exceptions. Servers are encouraged to simply
        ## pass on real IO objects, although it is recognized that this approach
        ## is not directly compatible with HTTP/2.
        REQUIRED_METHODS = [
          :read, :write, :<<, :flush, :close,
          :close_read, :close_write, :closed?
        ]

        def_delegators :@stream, *REQUIRED_METHODS

        def initialize(stream)
          @stream = stream

          REQUIRED_METHODS.each do |method_name|
            raise LintError, "Stream must respond to #{method_name}" unless stream.respond_to?(method_name)
          end
        end
      end

      # :startdoc:
    end
 