# frozen_string_literal: true

require_relative 'constants'
require_relative 'utils'

module Rack
  warn "Rack::Chunked is deprecated and will be removed in Rack 3.1", uplevel: 1

  # Middleware that applies chunked transfer encoding to response bodies
  # when the response does not include a content-length header.
  #
  # This supports the trailer response header to allow the use of trailing
  # headers in the chunked encoding.  However, using this requires you manually
  # specify a response body that supports a +trailers+ method.  Example:
  #
  #   [200, { 'trailer' => 'expires'}, ["Hello", "World"]]
  #   # error raised
  #
  #   body = ["Hello", "World"]
  #   def body.trailers
  #     { 'expires' => Time.now.to_s }
  #   end
  #   [200, { 'trailer' => 'expires'}, body]
  #   # No exception raised
  class Chunked
    include Rack::Utils

    # A body wrapper that emits chunked responses.
    class Body
      TERM = "\r\n"
      TAIL = "0#{TERM}"

      # Store the response body to be chunked.
      def initialize(body)
        @body = body
      end

      # For each element yielded by the response body, yield
      # the element in chunked encoding.
      def each(&block)
        term = TERM
        @body.each do |chunk|
          size = chunk.bytesize
          next if size == 0

          yield [size.to_s(16), term, chunk.b, term].join
        end
        yield TAIL
        yield_trailers(&block)
        yield term
      end

      # Close the response body if the response body supports it.
      def close
        @body.close if @body.respond_to?(:close)
      end

      private

      # Do nothing as this class does not support trailer headers.
      def yield_trailers
      end
    end

    # A body wrapper that emits chunked responses and also supports
    # sending Trailer headers.  Note that the response body provided to
    # initialize must have a +trailers+ method that returns a hash
    # of trailer headers, and the rack response itself should have a
    # Trailer header listing the headers that the +trailers+ method
    # will return.
    class TrailerBody < Body
      private

      # Yield strings for each trailer header.
      def yield_trailers
        @body.trailers.each_pair do |k, v|
          yield "#{k}: #{v}\r\n"
        end
      end
    end

    def initialize(app)
      @app = app
    end

    # Whether the HTTP version supports chunked encoding (HTTP 1.1 does).
    def chunkable_version?(ver)
      case ver
      # pre-HTTP/1.0 (informally "HTTP/0.9") HTTP requests did not have
      # a version (nor response headers)
      when 'HTTP/1.0', nil, 'HTTP/0.9'
        false
      else
        true
      end
    end

    # If the rack app returns a response that should have a body,
    # but does not have content-length or transfer-encoding headers,
    # modify the response to use chunked transfer-encoding.
    def call(env)
      status, headers, body = response = @app.call(env)

      if chunkable_version?(env[SERVER_PROTOCOL]) &&
         !STATUS_WITH_NO_ENTITY_BODY.key?(status.to_i) &&
         !headers[CONTENT_LENGTH] &&
         !headers[TRANSFER_ENCODING]

        headers[TRANSFER_ENCODING] = 'chunked'
        if headers['trailer']
          response[2] = TrailerBody.new(body)
        else
          response[2] = Body.new(body)
        end
      end

      response
    end
  end
end
                                                                                                                                                                                                            gems/gems/rack-3.0.8/lib/rack/events.rb                                                             0000644                 00000011464 15040313406 0013337 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative 'body_proxy'
require_relative 'request'
require_relative 'response'

module Rack
  ### This middleware provides hooks to certain places in the request /
  # response lifecycle.  This is so that middleware that don't need to filter
  # the response data can safely leave it alone and not have to send messages
  # down the traditional "rack stack".
  #
  # The events are:
  #
  # * on_start(request, response)
  #
  #   This event is sent at the start of the request, before the next
  #   middleware in the chain is called.  This method is called with a request
  #   object, and a response object.  Right now, the response object is always
  #   nil, but in the future it may actually be a real response object.
  #
  # * on_commit(request, response)
  #
  #   The response has been committed.  The application has returned, but the
  #   response has not been sent to the webserver yet.  This method is always
  #   called with a request object and the response object.  The response
  #   object is constructed from the rack triple that the application returned.
  #   Changes may still be made to the response object at this point.
  #
  # * on_send(request, response)
  #
  #   The webserver has started iterating over the response body and presumably
  #   has started sending data over the wire. This method is always called with
  #   a request object and the response object.  The response object is
  #   constructed from the rack triple that the application returned.  Changes
  #   SHOULD NOT be made to the response object as the webserver has already
  #   started sending data.  Any mutations will likely result in an exception.
  #
  # * on_finish(request, response)
  #
  #   The webserver has closed the response, and all data has been written to
  #   the response socket.  The request and response object should both be
  #   read-only at this point.  The body MAY NOT be available on the response
  #   object as it may have been flushed to the socket.
  #
  # * on_error(request, response, error)
  #
  #   An exception has occurred in the application or an `on_commit` event.
  #   This method will get the request, the response (if available) and the
  #   exception that was raised.
  #
  # ## Order
  #
  # `on_start` is called on the handlers in the order that they were passed to
  # the constructor.  `on_commit`, on_send`, `on_finish`, and `on_error` are
  # called in the reverse order.  `on_finish` handlers are called inside an
  # `ensure` block, so they are guaranteed to be called even if something
  # raises an exception.  If something raises an exception in a `on_finish`
  # method, then nothing is guaranteed.

  class Events
    module Abstract
      def on_start(req, res)
      end

      def on_commit(req, res)
      end

      def on_send(req, res)
      end

      def on_finish(req, res)
      end

      def on_error(req, res, e)
      end
    end

    class EventedBodyProxy < Rack::BodyProxy # :nodoc:
      attr_reader :request, :response

      def initialize(body, request, response, handlers, &block)
        super(body, &block)
        @request  = request
        @response = response
        @handlers = handlers
      end

      def each
        @handlers.reverse_each { |handler| handler.on_send request, response }
        super
      end
    end

    class BufferedResponse < Rack::Response::Raw # :nodoc:
      attr_reader :body

      def initialize(status, headers, body)
        super(status, headers)
        @body = body
      end

      def to_a; [status, headers, body]; end
    end

    def initialize(app, handlers)
      @app      = app
      @handlers = handlers
    end

    def call(env)
      request = make_request env
      on_start request, nil

      begin
        status, headers, body = @app.call request.env
        response = make_response status, headers, body
        on_commit request, response
      rescue StandardError => e
        on_error request, response, e
        on_finish request, response
        raise
      end

      body = EventedBodyProxy.new(body, request, response, @handlers) do
        on_finish request, response
      end
      [response.status, response.headers, body]
    end

    private

    def on_error(request, response, e)
      @handlers.reverse_each { |handler| handler.on_error request, response, e }
    end

    def on_commit(request, response)
      @handlers.reverse_each { |handler| handler.on_commit request, response }
    end

    def on_start(request, response)
      @handlers.each { |handler| handler.on_start request, nil }
    end

    def on_finish(request, response)
      @handlers.reverse_each { |handler| handler.on_finish request, response }
    end

    def make_request(env)
      Rack::Request.new env
    end

    def make_response(status, headers, body)
      BufferedResponse.new status, headers, body
    end
  end
end
                                                                                                                                                                                                            gems/gems/rack-3.0.8/lib/rack/request.rb                                                            0000644                 00000061107 15040313406 0013522 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative 'constants'
require_relative 'utils'
require_relative 'media_type'

module Rack
  # Rack::Request provides a convenient interface to a Rack
  # environment.  It is stateless, the environment +env+ passed to the
  # constructor will be directly modified.
  #
  #   req = Rack::Request.new(env)
  #   req.post?
  #   req.params["data"]

  class Request
    class << self
      attr_accessor :ip_filter

      # The priority when checking forwarded headers. The default
      # is <tt>[:forwarded, :x_forwarded]</tt>, which means, check the
      # +Forwarded+ header first, followed by the appropriate
      # <tt>X-Forwarded-*</tt> header.  You can revert the priority by
      # reversing the priority, or remove checking of either
      # or both headers by removing elements from the array.
      #
      # This should be set as appropriate in your environment
      # based on what reverse proxies are in use.  If you are not
      # using reverse proxies, you should probably use an empty
      # array.
      attr_accessor :forwarded_priority

      # The priority when checking either the <tt>X-Forwarded-Proto</tt>
      # or <tt>X-Forwarded-Scheme</tt> header for the forwarded protocol.
      # The default is <tt>[:proto, :scheme]</tt>, to try the
      # <tt>X-Forwarded-Proto</tt> header before the
      # <tt>X-Forwarded-Scheme</tt> header.  Rack 2 had behavior
      # similar to <tt>[:scheme, :proto]</tt>.  You can remove either or
      # both of the entries in array to ignore that respective header.
      attr_accessor :x_forwarded_proto_priority
    end

    @forwarded_priority = [:forwarded, :x_forwarded]
    @x_forwarded_proto_priority = [:proto, :scheme]

    valid_ipv4_octet = /\.(25[0-5]|2[0-4][0-9]|[01]?[0-9]?[0-9])/

    trusted_proxies = Regexp.union(
      /\A127#{valid_ipv4_octet}{3}\z/,                          # localhost IPv4 range 127.x.x.x, per RFC-3330
      /\A::1\z/,                                                # localhost IPv6 ::1
      /\Af[cd][0-9a-f]{2}(?::[0-9a-f]{0,4}){0,7}\z/i,           # private IPv6 range fc00 .. fdff
      /\A10#{valid_ipv4_octet}{3}\z/,                           # private IPv4 range 10.x.x.x
      /\A172\.(1[6-9]|2[0-9]|3[01])#{valid_ipv4_octet}{2}\z/,   # private IPv4 range 172.16.0.0 .. 172.31.255.255
      /\A192\.168#{valid_ipv4_octet}{2}\z/,                     # private IPv4 range 192.168.x.x
      /\Alocalhost\z|\Aunix(\z|:)/i,                            # localhost hostname, and unix domain sockets
    )

    self.ip_filter = lambda { |ip| trusted_proxies.match?(ip) }

    ALLOWED_SCHEMES = %w(https http wss ws).freeze

    def initialize(env)
      @env = env
      @params = nil
    end

    def params
      @params ||= super
    end

    def update_param(k, v)
      super
      @params = nil
    end

    def delete_param(k)
      v = super
      @params = nil
      v
    end

    module Env
      # The environment of the request.
      attr_reader :env

      def initialize(env)
        @env = env
        # This module is included at least in `ActionDispatch::Request`
        # The call to `super()` allows additional mixed-in initializers are called
        super()
      end

      # Predicate method to test to see if `name` has been set as request
      # specific data
      def has_header?(name)
        @env.key? name
      end

      # Get a request specific value for `name`.
      def get_header(name)
        @env[name]
      end

      # If a block is given, it yields to the block if the value hasn't been set
      # on the request.
      def fetch_header(name, &block)
        @env.fetch(name, &block)
      end

      # Loops through each key / value pair in the request specific data.
      def each_header(&block)
        @env.each(&block)
      end

      # Set a request specific value for `name` to `v`
      def set_header(name, v)
        @env[name] = v
      end

      # Add a header that may have multiple values.
      #
      # Example:
      #   request.add_header 'Accept', 'image/png'
      #   request.add_header 'Accept', '*/*'
      #
      #   assert_equal 'image/png,*/*', request.get_header('Accept')
      #
      # http://www.w3.org/Protocols/rfc2616/rfc2616-sec4.html#sec4.2
      def add_header(key, v)
        if v.nil?
          get_header key
        elsif has_header? key
          set_header key, "#{get_header key},#{v}"
        else
          set_header key, v
        end
      end

      # Delete a request specific value for `name`.
      def delete_header(name)
        @env.delete name
      end

      def initialize_copy(other)
        @env = other.env.dup
      end
    end

    module Helpers
      # The set of form-data media-types. Requests that do not indicate
      # one of the media types present in this list will not be eligible
      # for form-data / param parsing.
      FORM_DATA_MEDIA_TYPES = [
        'application/x-www-form-urlencoded',
        'multipart/form-data'
      ]

      # The set of media-types. Requests that do not indicate
      # one of the media types present in this list will not be eligible
      # for param parsing like soap attachments or generic multiparts
      PARSEABLE_DATA_MEDIA_TYPES = [
        'multipart/related',
        'multipart/mixed'
      ]

      # Default ports depending on scheme. Used to decide whether or not
      # to include the port in a generated URI.
      DEFAULT_PORTS = { 'http' => 80, 'https' => 443, 'coffee' => 80 }

      # The address of the client which connected to the proxy.
      HTTP_X_FORWARDED_FOR = 'HTTP_X_FORWARDED_FOR'

      # The contents of the host/:authority header sent to the proxy.
      HTTP_X_FORWARDED_HOST = 'HTTP_X_FORWARDED_HOST'

      HTTP_FORWARDED          = 'HTTP_FORWARDED'

      # The value of the scheme sent to the proxy.
      HTTP_X_FORWARDED_SCHEME = 'HTTP_X_FORWARDED_SCHEME'

      # The protocol used to connect to the proxy.
      HTTP_X_FORWARDED_PROTO = 'HTTP_X_FORWARDED_PROTO'

      # The port used to connect to the proxy.
      HTTP_X_FORWARDED_PORT = 'HTTP_X_FORWARDED_PORT'

      # Another way for specifying https scheme was used.
      HTTP_X_FORWARDED_SSL = 'HTTP_X_FORWARDED_SSL'

      def body;            get_header(RACK_INPUT)                         end
      def script_name;     get_header(SCRIPT_NAME).to_s                   end
      def script_name=(s); set_header(SCRIPT_NAME, s.to_s)                end

      def path_info;       get_header(PATH_INFO).to_s                     end
      def path_info=(s);   set_header(PATH_INFO, s.to_s)                  end

      def request_method;  get_header(REQUEST_METHOD)                     end
      def query_string;    get_header(QUERY_STRING).to_s                  end
      def content_length;  get_header('CONTENT_LENGTH')                   end
      def logger;          get_header(RACK_LOGGER)                        end
      def user_agent;      get_header('HTTP_USER_AGENT')                  end

      # the referer of the client
      def referer;         get_header('HTTP_REFERER')                     end
      alias referrer referer

      def session
        fetch_header(RACK_SESSION) do |k|
          set_header RACK_SESSION, default_session
        end
      end

      def session_options
        fetch_header(RACK_SESSION_OPTIONS) do |k|
          set_header RACK_SESSION_OPTIONS, {}
        end
      end

      # Checks the HTTP request method (or verb) to see if it was of type DELETE
      def delete?;  request_method == DELETE  end

      # Checks the HTTP request method (or verb) to see if it was of type GET
      def get?;     request_method == GET     end

      # Checks the HTTP request method (or verb) to see if it was of type HEAD
      def head?;    request_method == HEAD    end

      # Checks the HTTP request method (or verb) to see if it was of type OPTIONS
      def options?; request_method == OPTIONS end

      # Checks the HTTP request method (or verb) to see if it was of type LINK
      def link?;    request_method == LINK    end

      # Checks the HTTP request method (or verb) to see if it was of type PATCH
      def patch?;   request_method == PATCH   end

      # Checks the HTTP request method (or verb) to see if it was of type POST
      def post?;    request_method == POST    end

      # Checks the HTTP request method (or verb) to see if it was of type PUT
      def put?;     request_method == PUT     end

      # Checks the HTTP request method (or verb) to see if it was of type TRACE
      def trace?;   request_method == TRACE   end

      # Checks the HTTP request method (or verb) to see if it was of type UNLINK
      def unlink?;  request_method == UNLINK  end

      def scheme
        if get_header(HTTPS) == 'on'
          'https'
        elsif get_header(HTTP_X_FORWARDED_SSL) == 'on'
          'https'
        elsif forwarded_scheme
          forwarded_scheme
        else
          get_header(RACK_URL_SCHEME)
        end
      end

      # The authority of the incoming request as defined by RFC3976.
      # https://tools.ietf.org/html/rfc3986#section-3.2
      #
      # In HTTP/1, this is the `host` header.
      # In HTTP/2, this is the `:authority` pseudo-header.
      def authority
        forwarded_authority || host_authority || server_authority
      end

      # The authority as defined by the `SERVER_NAME` and `SERVER_PORT`
      # variables.
      def server_authority
        host = self.server_name
        port = self.server_port

        if host
          if port
            "#{host}:#{port}"
          else
            host
          end
        end
      end

      def server_name
        get_header(SERVER_NAME)
      end

      def server_port
        get_header(SERVER_PORT)
      end

      def cookies
        hash = fetch_header(RACK_REQUEST_COOKIE_HASH) do |key|
          set_header(key, {})
        end

        string = get_header(HTTP_COOKIE)

        unless string == get_header(RACK_REQUEST_COOKIE_STRING)
          hash.replace Utils.parse_cookies_header(string)
          set_header(RACK_REQUEST_COOKIE_STRING, string)
        end

        hash
      end

      def content_type
        content_type = get_header('CONTENT_TYPE')
        content_type.nil? || content_type.empty? ? nil : content_type
      end

      def xhr?
        get_header("HTTP_X_REQUESTED_WITH") == "XMLHttpRequest"
      end

      # The `HTTP_HOST` header.
      def host_authority
        get_header(HTTP_HOST)
      end

      def host_with_port(authority = self.authority)
        host, _, port = split_authority(authority)

        if port == DEFAULT_PORTS[self.scheme]
          host
        else
          authority
        end
      end

      # Returns a formatted host, suitable for being used in a URI.
      def host
        split_authority(self.authority)[0]
      end

      # Returns an address suitable for being to resolve to an address.
      # In the case of a domain name or IPv4 address, the result is the same
      # as +host+. In the case of IPv6 or future address formats, the square
      # brackets are removed.
      def hostname
        split_authority(self.authority)[1]
      end

      def port
        if authority = self.authority
          _, _, port = split_authority(authority)
        end

        port || forwarded_port&.last || DEFAULT_PORTS[scheme] || server_port
      end

      def forwarded_for
        forwarded_priority.each do |type|
          case type
          when :forwarded
            if forwarded_for = get_http_forwarded(:for)
              return(forwarded_for.map! do |authority|
                split_authority(authority)[1]
              end)
            end
          when :x_forwarded
            if value = get_header(HTTP_X_FORWARDED_FOR)
              return(split_header(value).map do |authority|
                split_authority(wrap_ipv6(authority))[1]
              end)
            end
          end
        end

        nil
      end

      def forwarded_port
        forwarded_priority.each do |type|
          case type
          when :forwarded
            if forwarded = get_http_forwarded(:for)
              return(forwarded.map do |authority|
                split_authority(authority)[2]
              end.compact)
            end
          when :x_forwarded
            if value = get_header(HTTP_X_FORWARDED_PORT)
              return split_header(value).map(&:to_i)
            end
          end
        end

        nil
      end

      def forwarded_authority
        forwarded_priority.each do |type|
          case type
          when :forwarded
            if forwarded = get_http_forwarded(:host)
              return forwarded.last
            end
          when :x_forwarded
            if value = get_header(HTTP_X_FORWARDED_HOST)
              return wrap_ipv6(split_header(value).last)
            end
          end
        end

        nil
      end

      def ssl?
        scheme == 'https' || scheme == 'wss'
      end

      def ip
        remote_addresses = split_header(get_header('REMOTE_ADDR'))
        external_addresses = reject_trusted_ip_addresses(remote_addresses)

        unless external_addresses.empty?
          return external_addresses.last
        end

        if (forwarded_for = self.forwarded_for) && !forwarded_for.empty?
          # The forwarded for addresses are ordered: client, proxy1, proxy2.
          # So we reject all the trusted addresses (proxy*) and return the
          # last client. Or if we trust everyone, we just return the first
          # address.
          return reject_trusted_ip_addresses(forwarded_for).last || forwarded_for.first
        end

        # If all the addresses are trusted, and we aren't forwarded, just return
        # the first remote address, which represents the source of the request.
        remote_addresses.first
      end

      # The media type (type/subtype) portion of the CONTENT_TYPE header
      # without any media type parameters. e.g., when CONTENT_TYPE is
      # "text/plain;charset=utf-8", the media-type is "text/plain".
      #
      # For more information on the use of media types in HTTP, see:
      # http://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.7
      def media_type
        MediaType.type(content_type)
      end

      # The media type parameters provided in CONTENT_TYPE as a Hash, or
      # an empty Hash if no CONTENT_TYPE or media-type parameters were
      # provided.  e.g., when the CONTENT_TYPE is "text/plain;charset=utf-8",
      # this method responds with the following Hash:
      #   { 'charset' => 'utf-8' }
      def media_type_params
        MediaType.params(content_type)
      end

      # The character set of the request body if a "charset" media type
      # parameter was given, or nil if no "charset" was specified. Note
      # that, per RFC2616, text/* media types that specify no explicit
      # charset are to be considered ISO-8859-1.
      def content_charset
        media_type_params['charset']
      end

      # Determine whether the request body contains form-data by checking
      # the request content-type for one of the media-types:
      # "application/x-www-form-urlencoded" or "multipart/form-data". The
      # list of form-data media types can be modified through the
      # +FORM_DATA_MEDIA_TYPES+ array.
      #
      # A request body is also assumed to contain form-data when 