# frozen_string_literal: true

require_relative 'constants'
require_relative 'utils'

module Rack

  # Sets the content-type header on responses which don't have one.
  #
  # Builder Usage:
  #   use Rack::ContentType, "text/plain"
  #
  # When no content type argument is provided, "text/html" is the
  # default.
  class ContentType
    include Rack::Utils

    def initialize(app, content_type = "text/html")
      @app = app
      @content_type = content_type
    end

    def call(env)
      status, headers, _ = response = @app.call(env)

      unless STATUS_WITH_NO_ENTITY_BODY.key?(status.to_i)
        headers[CONTENT_TYPE] ||= @content_type
      end

      response
    end
  end
end
                                                                                                                                                                                                                                                                                                                                         gems/gems/rack-3.0.8/lib/rack/body_proxy.rb                                                         0000644                 00000002463 15040313411 0014224 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

module Rack
  # Proxy for response bodies allowing calling a block when
  # the response body is closed (after the response has been fully
  # sent to the client).
  class BodyProxy
    # Set the response body to wrap, and the block to call when the
    # response has been fully sent.
    def initialize(body, &block)
      @body = body
      @block = block
      @closed = false
    end

    # Return whether the wrapped body responds to the method.
    def respond_to_missing?(method_name, include_all = false)
      super or @body.respond_to?(method_name, include_all)
    end

    # If not already closed, close the wrapped body and
    # then call the block the proxy was initialized with.
    def close
      return if @closed
      @closed = true
      begin
        @body.close if @body.respond_to?(:close)
      ensure
        @block.call
      end
    end

    # Whether the proxy is closed.  The proxy starts as not closed,
    # and becomes closed on the first call to close.
    def closed?
      @closed
    end

    # Delegate missing methods to the wrapped body.
    def method_missing(method_name, *args, &block)
      @body.__send__(method_name, *args, &block)
    end
    # :nocov:
    ruby2_keywords(:method_missing) if respond_to?(:ruby2_keywords, true)
    # :nocov:
  end
end
                                                                                                                                                                                                             gems/gems/rack-3.0.8/lib/rack/config.rb                                                             0000644                 00000000632 15040313411 0013267 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

module Rack
  # Rack::Config modifies the environment using the block given during
  # initialization.
  #
  # Example:
  #     use Rack::Config do |env|
  #       env['my-key'] = 'some-value'
  #     end
  class Config
    def initialize(app, &block)
      @app = app
      @block = block
    end

    def call(env)
      @block.call(env)
      @app.call(env)
    end
  end
end
                                                                                                      gems/gems/rack-3.0.8/lib/rack/media_type.rb                                                         0000644                 00000002631 15040313411 0014143 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

module Rack
  # Rack::MediaType parse media type and parameters out of content_type string

  class MediaType
    SPLIT_PATTERN = %r{\s*[;,]\s*}

    class << self
      # The media type (type/subtype) portion of the CONTENT_TYPE header
      # without any media type parameters. e.g., when CONTENT_TYPE is
      # "text/plain;charset=utf-8", the media-type is "text/plain".
      #
      # For more information on the use of media types in HTTP, see:
      # http://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.7
      def type(content_type)
        return nil unless content_type
        content_type.split(SPLIT_PATTERN, 2).first.tap(&:downcase!)
      end

      # The media type parameters provided in CONTENT_TYPE as a Hash, or
      # an empty Hash if no CONTENT_TYPE or media-type parameters were
      # provided.  e.g., when the CONTENT_TYPE is "text/plain;charset=utf-8",
      # this method responds with the following Hash:
      #   { 'charset' => 'utf-8' }
      def params(content_type)
        return {} if content_type.nil?

        content_type.split(SPLIT_PATTERN)[1..-1].each_with_object({}) do |s, hsh|
          k, v = s.split('=', 2)

          hsh[k.tap(&:downcase!)] = strip_doublequotes(v)
        end
      end

      private

        def strip_doublequotes(str)
          (str.start_with?('"') && str.end_with?('"')) ? str[1..-2] : str
        end
    end
  end
end
                                                                                                       gems/gems/rack-3.0.8/lib/rack/lock.rb                                                               0000644                 00000001075 15040313411 0012754 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative 'body_proxy'

module Rack
  # Rack::Lock locks every request inside a mutex, so that every request
  # will effectively be executed synchronously.
  class Lock
    def initialize(app, mutex = Mutex.new)
      @app, @mutex = app, mutex
    end

    def call(env)
      @mutex.lock
      begin
        response = @app.call(env)
        returned = response << BodyProxy.new(response.pop) { unlock }
      ensure
        unlock unless returned
      end
    end

    private

    def unlock
      @mutex.unlock
    end
  end
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                   gems/gems/rack-3.0.8/lib/rack/content_length.rb                                                     0000644                 00000001446 15040313411 0015041 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative 'constants'
require_relative 'utils'

module Rack

  # Sets the content-length header on responses that do not specify
  # a content-length or transfer-encoding header.  Note that this
  # does not fix responses that have an invalid content-length
  # header specified.
  class ContentLength
    include Rack::Utils

    def initialize(app)
      @app = app
    end

    def call(env)
      status, headers, body = response = @app.call(env)

      if !STATUS_WITH_NO_ENTITY_BODY.key?(status.to_i) &&
         !headers[CONTENT_LENGTH] &&
         !headers[TRANSFER_ENCODING] &&
         body.respond_to?(:to_ary)

        response[2] = body = body.to_ary
        headers[CONTENT_LENGTH] = body.sum(&:bytesize).to_s
      end

      response
    end
  end
end
                                                                                                                                                                                                                          gems/gems/rack-3.0.8/lib/rack/response.rb                                                           0000644                 00000025345 15040313411 0013670 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require 'time'

require_relative 'constants'
require_relative 'utils'
require_relative 'media_type'
require_relative 'headers'

module Rack
  # Rack::Response provides a convenient interface to create a Rack
  # response.
  #
  # It allows setting of headers and cookies, and provides useful
  # defaults (an OK response with empty headers and body).
  #
  # You can use Response#write to iteratively generate your response,
  # but note that this is buffered by Rack::Response until you call
  # +finish+.  +finish+ however can take a block inside which calls to
  # +write+ are synchronous with the Rack response.
  #
  # Your application's +call+ should end returning Response#finish.
  class Response
    def self.[](status, headers, body)
      self.new(body, status, headers)
    end

    CHUNKED = 'chunked'
    STATUS_WITH_NO_ENTITY_BODY = Utils::STATUS_WITH_NO_ENTITY_BODY

    attr_accessor :length, :status, :body
    attr_reader :headers

    # Deprecated, use headers instead.
    def header
      warn 'Rack::Response#header is deprecated and will be removed in Rack 3.1', uplevel: 1

      headers
    end

    # Initialize the response object with the specified +body+, +status+
    # and +headers+.
    #
