# frozen_string_literal: true

require_relative 'constants'
require_relative 'utils'
require_relative 'body_proxy'

module Rack

  # Middleware that enables conditional GET using if-none-match and
  # if-modified-since. The application should set either or both of the
  # last-modified or etag response headers according to RFC 2616. When
  # either of the conditions is met, the response body is set to be zero
  # length and the response status is set to 304 Not Modified.
  #
  # Applications that defer response body generation until the body's each
  # message is received will avoid response body generation completely when
  # a conditional GET matches.
  #
  # Adapted from Michael Klishin's Merb implementation:
  # https://github.com/wycats/merb/blob/master/merb-core/lib/merb-core/rack/middleware/conditional_get.rb
  class ConditionalGet
    def initialize(app)
      @app = app
    end

    # Return empty 304 response if the response has not been
    # modified since the last request.
    def call(env)
      case env[REQUEST_METHOD]
      when "GET", "HEAD"
        status, headers, body = response = @app.call(env)

        if status == 200 && fresh?(env, headers)
          response[0] = 304
          headers.delete(CONTENT_TYPE)
          headers.delete(CONTENT_LENGTH)
          response[2] = Rack::BodyProxy.new([]) do
            body.close if body.respond_to?(:close)
          end
        end
        response
      else
        @app.call(env)
      end
    end

  private

    # Return whether the response has not been modified since the
    # last request.
    def fresh?(env, headers)
      # if-none-match has priority over if-modified-since per RFC 7232
      if none_match = env['HTTP_IF_NONE_MATCH']
        etag_matches?(none_match, headers)
      elsif (modified_since = env['HTTP_IF_MODIFIED_SINCE']) && (modified_since = to_rfc2822(modified_since))
        modified_since?(modified_since, headers)
      end
    end

    # Whether the etag response header matches the if-none-match request header.
    # If so, the request has not been modified.
    def etag_matches?(none_match, headers)
      headers[ETAG] == none_match
    end

    # Whether the last-modified response header matches the if-modified-since
    # request header.  If so, the request has not been modified.
    def modified_since?(modified_since, headers)
      last_modified = to_rfc2822(headers['last-modified']) and
        modified_since >= last_modified
    end

    # Return a Time object for the given string (which should be in RFC2822
    # format), or nil if the string cannot be parsed.
    def to_rfc2822(since)
      # shortest possible valid date is the obsolete: 1 Nov 97 09:55 A
      # anything shorter is invalid, this avoids exceptions for common cases
      # most common being the empty string
      if since && since.length >= 16
        # NOTE: there is no trivial way to write this in a non exception way
        #   _rfc2822 returns a hash but is not that usable
        Time.rfc2822(since) rescue nil
      end
    end
  end
end
           gems/gems/rack-3.0.8/lib/rack/constants.rb                                                          0000644                 00000004745 15040313406 0014053 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

module Rack
  # Request env keys
  HTTP_HOST         = 'HTTP_HOST'
  HTTP_PORT         = 'HTTP_PORT'
  HTTPS             = 'HTTPS'
  PATH_INFO         = 'PATH_INFO'
  REQUEST_METHOD    = 'REQUEST_METHOD'
  REQUEST_PATH      = 'REQUEST_PATH'
  SCRIPT_NAME       = 'SCRIPT_NAME'
  QUERY_STRING      = 'QUERY_STRING'
  SERVER_PROTOCOL   = 'SERVER_PROTOCOL'
  SERVER_NAME       = 'SERVER_NAME'
  SERVER_PORT       = 'SERVER_PORT'
  HTTP_COOKIE       = 'HTTP_COOKIE'

  # Response Header Keys
  CACHE_CONTROL     = 'cache-control'
  CONTENT_LENGTH    = 'content-length'
  CONTENT_TYPE      = 'content-type'
  ETAG              = 'etag'
  EXPIRES           = 'expires'
  SET_COOKIE        = 'set-cookie'
  TRANSFER_ENCODING = 'transfer-encoding'

  # HTTP method verbs
  GET     = 'GET'
  POST    = 'POST'
  PUT     = 'PUT'
  PATCH   = 'PATCH'
  DELETE  = 'DELETE'
  HEAD    = 'HEAD'
  OPTIONS = 'OPTIONS'
  LINK    = 'LINK'
  UNLINK  = 'UNLINK'
  TRACE   = 'TRACE'

  # Rack environment variables
  RACK_VERSION                        = 'rack.version'
  RACK_TEMPFILES                      = 'rack.tempfiles'
  RACK_ERRORS                         = 'rack.errors'
  RACK_LOGGER                         = 'rack.logger'
  RACK_INPUT                          = 'rack.input'
  RACK_SESSION                        = 'rack.session'
  RACK_SESSION_OPTIONS                = 'rack.session.options'
  RACK_SHOWSTATUS_DETAIL              = 'rack.showstatus.detail'
  RACK_URL_SCHEME                     = 'rack.url_scheme'
  RACK_HIJACK                         = 'rack.hijack'
  RACK_IS_HIJACK                      = 'rack.hijack?'
  RACK_RECURSIVE_INCLUDE              = 'rack.recursive.include'
  RACK_MULTIPART_BUFFER_SIZE          = 'rack.multipart.buffer_size'
  RACK_MULTIPART_TEMPFILE_FACTORY     = 'rack.multipart.tempfile_factory'
  RACK_RESPONSE_FINISHED              = 'rack.response_finished'
  RACK_REQUEST_FORM_INPUT             = 'rack.request.form_input'
  RACK_REQUEST_FORM_HASH              = 'rack.request.form_hash'
  RACK_REQUEST_FORM_VARS              = 'rack.request.form_vars'
  RACK_REQUEST_FORM_ERROR             = 'rack.request.form_error'
  RACK_REQUEST_COOKIE_HASH            = 'rack.request.cookie_hash'
  RACK_REQUEST_COOKIE_STRING          = 'rack.request.cookie_string'
  RACK_REQUEST_QUERY_HASH             = 'rack.request.query_hash'
  RACK_REQUEST_QUERY_STRING           = 'rack.request.query_string'
  RACK_METHODOVERRIDE_ORIGINAL_METHOD = 'rack.methodoverride.original_method'
end
                           gems/gems/rack-3.0.8/lib/rack/cascade.rb                                                            0000644                 00000004401 15040313406 0013407 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative 'constants'

module Rack
  # Rack::Cascade tries a request on several apps, and returns the
  # first response that is not 404 or 405 (or in a list of configured
  # status codes).  If all applications tried return one of the configured
  # status codes, return the last response.

  class Cascade
    # deprecated, no longer used
    NotFound = [404, { CONTENT_TYPE => "text/plain" }, []]

    # An array of applications to try in order.
    attr_reader :apps

    # Set the apps to send requests to, and what statuses result in
    # cascading.  Arguments:
    #
    # apps: An enumerable of rack applications.
    # cascade_for: The statuses to use cascading for.  If a response is received
    #              from an app, the next app is tried.
    def initialize(apps, cascade_for = [404, 405])
      @apps = []
      apps.each { |app| add app }

      @cascade_for = {}
      [*cascade_for].each { |status| @cascade_for[status] = true }
    end

    # Call each app in order.  If the responses uses a status that requires
    # cascading, try the next app.  If all responses require cascading,
    # return the response from the last app.
    def call(env)
      return [404, { CONTENT_TYPE => "text/plain" }, []] if @apps.empty?
      result = nil
      last_body = nil

      @apps.each do |app|
        # The SPEC says that the body must be closed after it has been iterated
        # by the server, or if it is replaced by a middleware action. Cascade
        # replaces the body each time a cascade happens. It is assumed that nil
        # does not respond to close, otherwise the previous application body
        # will be closed. The final application body will not be closed, as it
        # will be passed to the server as a result.
        last_body.close if last_body.respond_to? :close

        result = app.call(env)
        return result unless @cascade_for.include?(result[0].to_i)
        last_body = result[2]
      end

      result
    end

    # Append an app to the list of apps to cascade.  This app will
    # be tried last.
    def add(app)
      @apps << app
    end

    # Whether the given app is one of the apps to cascade to.
    def include?(app)
      @apps.include?(app)
    end

    alias_method :<<, :add
  end
end
                                                                                                                                                                                                                                                               gems/gems/rack-3.0.8/lib/rack/show_exceptions.rb                                                    0000644                 00000033360 15040313406 0015253 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require 'ostruct'
require 'erb'

require_relative 'constants'
require_relative 'utils'
require_relative 'request'

module Rack
  # Rack::ShowExceptions catches all exceptions raised from the app it
  # wraps.  It shows a useful backtrace with the sourcefile and
  # clickable context, the whole Rack environment and the request
  # data.
  #
  # Be careful when you use this on public-facing sites as it could
  # reveal information helpful to attackers.

  class ShowExceptions
    CONTEXT = 7

    def initialize(app)
      @app = app
    end

    def call(env)
      @app.call(env)
    rescue StandardError, LoadError, SyntaxError => e
      exception_string = dump_exception(e)

      env[RACK_ERRORS].puts(exception_string)
      env[RACK_ERRORS].flush

      if accepts_html?(env)
        content_type = "text/html"
        body = pretty(env, e)
      else
        content_type = "text/plain"
        body = exception_string
      end

      [
        500,
        {
          CONTENT_TYPE => content_type,
          CONTENT_LENGTH => body.bytesize.to_s,
        },
        [body],
      ]
    end

    def prefers_plaintext?(env)
      !accepts_html?(env)
    end

    def accepts_html?(env)
      Rack::Utils.best_q_match(env["HTTP_ACCEPT"], %w[text/html])
    end
    private :accepts_html?

    def dump_exception(exception)
      if exception.respond_to?(:detailed_message)
        message = exception.detailed_message(highlight: false)
      else
        message = exception.message
      end
      string = "#{exception.class}: #{message}\n".dup
      string << exception.backtrace.map { |l| "\t#{l}" }.join("\n")
      string
    end

    def pretty(env, exception)
      req = Rack::Request.new(env)

      # This double assignment is to prevent an "unused variable" warning.
      # Yes, it is dumb, but I don't like Ruby yelling at me.
      path = path = (req.script_name + req.path_info).squeeze("/")

      # This double assignment is to prevent an "unused variable" warning.
      # Yes, it is dumb, but I don't like Ruby yelling at me.
      frames = frames = exception.backtrace.map { |line|
        frame = OpenStruct.new
        if line =~ /(.*?):(\d+)(:in `(.*)')?/
          frame.filename = $1
          frame.lineno = $2.to_i
          frame.function = $4

          begin
            lineno = frame.lineno - 1
            lines = ::File.readlines(frame.filename)
            frame.pre_context_lineno = [lineno - CONTEXT, 0].max
            frame.pre_context = lines[frame.pre_context_lineno...lineno]
            frame.context_line = lines[lineno].chomp
            frame.post_context_lineno = [lineno + CONTEXT, lines.size].min
            frame.post_context = lines[lineno + 1..frame.post_context_lineno]
          rescue
          end

          frame
        else
          nil
        end
      }.compact

      template.result(binding)
    end

    def template
      TEMPLATE
    end

    def h(obj)                  # :nodoc:
      case obj
      when String
        Utils.escape_html(obj)
      else
        Utils.escape_html(obj.inspect)
      end
    end

    # :stopdoc:

    # adapted from Django <www.djangoproject.com>
    # Copyright (c) Django Software Foundation and individual contributors.
    # Used under the modified BSD license:
    # http://www.xfree86.org/3.3.6/COPYRIGHT2.html#5
    TEMPLATE = ERB.new(<<-'HTML'.gsub(/^      /, ''))
      <!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">
      <html lang="en">
      <head>
        <meta http-equiv="content-type" content="text/html; charset=utf-8" />
        <meta name="robots" content="NONE,NOARCHIVE" />
        <title><%=h exception.class %> at <%=h path %></title>
        <style type="text/css">
          html * { padding:0; margin:0; }
          body * { padding:10px 20px; }
          body * * { padding:0; }
          body { font:small sans-serif; }
          body>div { border-bottom:1px solid #ddd; }
          h1 { font-weight:normal; }
          h2 { margin-bottom:.8em; }
          h2 span { font-size:80%; color:#666; font-weight:normal; }
          h3 { margin:1em 0 .5em 0; }
          h4 { margin:0 0 .5em 0; font-weight: normal; }
          table {
              border:1px solid #ccc; border-collapse: collapse; background