# frozen_string_literal: true

# Released under the MIT License.
# Copyright, 2019-2022, by Samuel Williams.

module Rackup
  # The input stream is an IO-like object which contains the raw HTTP POST data. When applicable, its external encoding must be “ASCII-8BIT” and it must be opened in binary mode, for Ruby 1.9 compatibility. The input stream must respond to gets, each, read and rewind.
  class Stream
    def initialize(input = nil, output = Buffered.new)
      @input = input
      @output = output

      raise ArgumentError, "Non-writable output!" unless output.respond_to?(:write)

      # Will hold remaining data in `#read`.
      @buffer = nil
      @closed = false
    end

    attr :input
    attr :output

    # This provides a read-only interface for data, which is surprisingly tricky to implement correctly.
    module Reader
      # rack.hijack_io must respond to:
      # read, write, read_nonblock, write_nonblock, flush, close, close_read, close_write, closed?

      # read behaves like IO#read. Its signature is read([length, [buffer]]). If given, length must be a non-negative Integer (>= 0) or nil, and buffer must be a String and may not be nil. If length is given and not nil, then this method reads at most length bytes from the input stream. If length is not given or nil, then this method reads all data until EOF. When EOF is reached, this method returns nil if length is given and not nil, or “” if length is not given or is nil. If buffer is given, then the read data will be placed into buffer instead of a newly created String object.
      # @param length [Integer] the amount of data to read
      # @param buffer [String] the buffer which will receive the data
      # @return a buffer containing the data
      def read(length = nil, buffer = nil)
        return '' if length == 0

        buffer ||= String.new.force_encoding(Encoding::BINARY)

        # Take any previously buffered data and replace it into the given buffer.
        if @buffer
          buffer.replace(@buffer)
          @buffer = nil
        else
          buffer.clear
        end

        if length
          while buffer.bytesize < length and chunk = read_next
            buffer << chunk
          end

          # This ensures the subsequent `slice!` works correctly.
          buffer.force_encoding(Encoding::BINARY)

          # This will be at least one copy:
          @buffer = buffer.byteslice(length, buffer.bytesize)

          # This should be zero-copy:
          buffer.slice!(length, buffer.bytesize)

          if buffer.empty?
            return nil
          else
            return buffer
          end
        else
          while chunk = read_next
            buffer << chunk
          end

          return buffer
        end
      end

      # Read at most `length` bytes from the stream. Will avoid reading from the underlying stream if possible.
      def read_partial(length = nil)
        if @buffer
          buffer = @buffer
          @buffer = nil
        else
          buffer = read_next
        end

        if buffer and length
          if buffer.bytesize > length
            # This ensures the subsequent `slice!` works correctly.
            buffer.force_encoding(Encoding::BINARY)

            @buffer = buffer.byteslice(length, buffer.bytesize)
            buffer.slice!(length, buffer.bytesize)
          end
        end

        return buffer
      end

      def gets
        read_partial
      end

      def each
        while chunk = read_partial
          yield chunk
        end
      end

      def read_nonblock(length, buffer = nil)
        @buffer ||= read_next
        chunk = nil

        unless @buffer
          buffer&.clear
          return
        end

        if @buffer.bytesize > length
          chunk = @buffer.byteslice(0, length)
          @buffer = @buffer.byteslice(length, @buffer.bytesize)
        else
          chunk = @buffer
          @buffer = nil
        end

        if buffer
          buffer.replace(chunk)
        else
          buffer = chunk
        end

        return buffer
      end
    end

    include Reader

    def write(buffer)
      if @output
        @output.write(buffer)
        return buffer.bytesize
      else
        raise IOError, "Stream is not writable, output has been closed!"
      end
    end

    def write_nonblock(buffer)
      write(buffer)
    end

    def <<(buffer)
      write(buffer)
    end

    def flush
    end

    def close_read
      @input&.close
      @input = nil
    end

    # close must never be called on the input stream. huh?
    def close_write
      if @output.respond_to?(:close)
        @output&.close
      end

      @output = nil
    end

    # Close the input and output bodies.
    def close(error = nil)
      self.close_read
      self.close_write

      return nil
    ensure
      @closed = true
    end

    # Whether the stream has been closed.
    def closed?
      @closed
    end

    # Whether there are any output chunks remaining?
    def empty?
      @output.empty?
    end

    private

    def read_next
      if @input
        return @input.read
      else
        @input = nil
        raise IOError, "Stream is not readable, input has been closed!"
      end
    end
  end
end
                                                                                                                                                                                                                                                                                                                                                                                      gems/gems/rackup-2.1.0/lib/rackup/handler.rb                                                        0000644                 00000005473 15040313405 0014354 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

# Released under the MIT License.
# Copyright, 2022-2023, by Samuel Williams.

module Rackup
  # *Handlers* connect web servers with Rack.
  #
  # Rackup includes Handlers for WEBrick and CGI.
  #
  # Handlers usually are activated by calling <tt>MyHandler.run(myapp)</tt>.
  # A second optional hash can be passed to include server-specific
  # configuration.
  module Handler
    @handlers = {}

    # Register a named handler class.
    def self.register(name, klass)
      if klass.is_a?(String)
        warn "Calling Rackup::Handler.register with a string is deprecated, use the class/module itself.", uplevel: 1

        klass = self.const_get(klass, false)
      end

      name = name.to_sym

      @handlers[name] = klass
    end

    def self.[](name)
      name = name.to_sym

      begin
        @handlers[name] || self.const_get(name, false)
      rescue NameError
        # Ignore.
      end
    end

    def self.get(name)
      return nil unless name

      name = name.to_sym

      if server = self[name]
        return server
      end

      begin
        require_handler("rackup/handler", name)
      rescue LoadError
        require_handler("rack/handler", name)
      end

      return self[name]
    end

    RACK_HANDLER = 'RACK_HANDLER'
    RACKUP_HANDLER = 'RACKUP_HANDLER'

    SERVER_NAMES = %i(puma falcon webrick).freeze
    private_constant :SERVER_NAMES

    # Select first available Rack handler given an `Array` of server names.
    # Raises `LoadError` if no handler was found.
    #
    #   > pick ['puma', 'webrick']
    #   => Rackup::Handler::WEBrick
    def self.pick(server_names)
      server_names = Array(server_names)

      server_names.each do |server_name|
        begin
          server = self.get(server_name)
         return server if server
        rescue LoadError
          # Ignore.
        end
      end

      raise LoadError, "Couldn't find handler for: #{server_names.join(', ')}."
    end

    def self.default
      if rack_handler = ENV[RACKUP_HANDLER]
        self.get(rack_handler)
      elsif rack_handler = ENV[RACK_HANDLER]
        warn "RACK_HANDLER is deprecated, use RACKUP_HANDLER."
        self.get(rack_handler)
      else
        pick SERVER_NAMES
      end
    end

    # Transforms server-name constants to their canonical form as filenames,
    # then tries to require them but silences the LoadError if not found
    #
    # Naming convention:
    #
    #   Foo # => 'foo'
    #   FooBar # => 'foo_bar.rb'
    #   FooBAR # => 'foobar.rb'
    #   FOObar # => 'foobar.rb'
    #   FOOBAR # => 'foobar.rb'
    #   FooBarBaz # => 'foo_bar_baz.rb'
    def self.require_handler(prefix, const_name)
      file = const_name.to_s.gsub(/^[A-Z]+/) { |pre| pre.downcase }.
        gsub(/[A-Z]+[^A-Z]/, '_\&').downcase

      require(::File.join(prefix, file))
    end
  end
end
                                                                                                                                                                                                     gems/gems/rackup-2.1.0/lib/rackup/server.rb                                                         0000644                 00000032327 15040313405 0014243 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

# Released under the MIT License.
# Copyright, 2022-2023, by Samuel Williams.

require 'optparse'
require 'fileutils'

require 'rack/builder'
require 'rack/common_logger'
require 'rack/content_length'
require 'rack/show_exceptions'
require 'rack/lint'
require 'rack/tempfile_reaper'

require 'rack/version'

require_relative 'version'
require_relative 'handler'

module Rackup
  class Server
    class Options
      def parse!(args)
        options = {}
        opt_parser = OptionParser.new("", 24, '  ') do |opts|
          opts.banner = "Usage: rackup [ruby options] [rack options] [rackup config]"

          opts.separator ""
          opts.separator "Ruby options:"

          lineno = 1
          opts.on("-e", "--eval LINE", "evaluate a LINE of code") { |line|
            eval line, TOPLEVEL_BINDING, "-e", lineno
            lineno += 1
          }

          opts.on("-d", "--debug", "set debugging flags (set $DEBUG to true)") {
            options[:debug] = true
          }
          opts.on("-w", "--warn", "turn warnings on for your script") {
            options[:warn] = true
          }
          opts.on("-q", "--quiet", "turn off logging") {
            options[:quiet] = true
          }

          opts.on("-I", "--include PATH",
                  "specify $LOAD_PATH (may be used more than once)") { |path|
            (options[:include] ||= []).concat(path.split(":"))
          }

          opts.on("-r", "--require LIBRARY",
                  "require the library, before executing your script") { |library|
            (options[:require] ||= []) << library
          }

          opts.separator ""
          opts.separator "Rack options:"
          opts.on("-b", "--builder BUILDER_LINE", "evaluate a BUILDER_LINE of code as a builder script") { |line|
            options[:builder] = line
          }

          opts.on("-s", "--server SERVER", "serve using SERVER (thin/puma/webrick)") { |s|
            options[:server] = s
          }

          opts.on("-o", "--host HOST", "listen on HOST (default: localhost)") { |host|
            options[:Host] = host
          }

          opts.on("-p", "--port PORT", "use PORT (default: 9292)") { |port|
            options[:Port] = port
          }

          opts.on("-O", "--option NAME[=VALUE]", "pass VALUE to the server as option NAME. If no VALUE, sets it to true. Run '#{$0} -s SERVER -h' to get a list of options for SERVER") { |name|
            name, value = name.split('=', 2)
            value = true if value.nil?
            options[name.to_sym] = value
          }

          opts.on("-E", "--env ENVIRONMENT", "use ENVIRONMENT for defaults (default: development)") { |e|
            options[:environment] = e
          }

          opts.on("-D", "--daemonize", "run daemonized in the background") { |d|
            options[:daemonize] ||= true
          }

          opts.on("--daemonize-noclose", "run daemonized in the background without closing stdout/stderr") {
            options[:daemonize] = :noclose
          }

          opts.on("-P", "--pid FILE", "file to store PID") { |f|
            options[:pid] = ::File.expand_path(f)
          }

          opts.separator ""
          opts.separator "Profiling options:"

          opts.on("--heap HEAPFILE", "Build the application, then dump the heap to HEAPFILE") do |e|
            options[:heapfile] = e
          end

          opts.on("--profile PROFILE", "Dump CPU or Memory profile to PROFILE (defaults to a tempfile)") do |e|
            options[:profile_file] = e
          end

          opts.on("--profile-mode MODE", "Profile mode (cpu|wall|object)") do |e|
            unless %w[cpu wall object].include?(e)
              raise OptionParser::InvalidOption, "unknown profile mode: #{e}"
            end
        