 ##
  # Creates a new TopLevel for the file at +absolute_name+.  If documentation
  # is being generated outside the source dir +relative_name+ is relative to
  # the source directory.

  def initialize absolute_name, relative_name = absolute_name
    super()
    @name = nil
    @absolute_name = absolute_name
    @relative_name = relative_name
    @file_stat     = File.stat(absolute_name) rescue nil # HACK for testing
    @diagram       = nil
    @parser        = nil

    @classes_or_modules = []
  end

  def parser=(val)
    @parser = val
    @store.update_parser_of_file(absolute_name, val) if @store
    @parser
  end

  ##
  # An RDoc::TopLevel is equal to another with the same relative_name

  def == other
    self.class === other and @relative_name == other.relative_name
  end

  alias eql? ==

  ##
  # Adds +an_alias+ to +Object+ instead of +self+.

  def add_alias(an_alias)
    object_class.record_location self
    return an_alias unless @document_self
    object_class.add_alias an_alias
  end

  ##
  # Adds +constant+ to +Object+ instead of +self+.

  def add_constant constant
    object_class.record_location self
    return constant unless @document_self
    object_class.add_constant constant
  end

  ##
  # Adds +include+ to +Object+ instead of +self+.

  def add_include(include)
    object_class.record_location self
    return include unless @document_self
    object_class.add_include include
  end

  ##
  # Adds +method+ to +Object+ instead of +self+.

  def add_method(method)
    object_class.record_location self
    return method unless @document_self
    object_class.add_method method
  end

  ##
  # Adds class or module +mod+. Used in the building phase
  # by the Ruby parser.

  def add_to_classes_or_modules mod
    @classes_or_modules << mod
  end

  ##
  # Base name of this file

  def base_name
    File.basename @relative_name
  end

  alias name base_name

  ##
  # Only a TopLevel that contains text file) will be displayed.  See also
  # RDoc::CodeObject#display?

  def display?
    text? and super
  end

  ##
  # See RDoc::TopLevel::find_class_or_module
  #--
  # TODO Why do we search through all classes/modules found, not just the
  #       ones of this instance?

  def find_class_or_module name
    @store.find_class_or_module name
  end

  ##
  # Finds a class or module named +symbol+

  def find_local_symbol(symbol)
    find_class_or_module(symbol) || super
  end

  ##
  # Finds a module or class with +name+

  def find_module_named(name)
    find_class_or_module(name)
  end

  ##
  # Returns the relative name of this file

  def full_name
    @relative_name
  end

  ##
  # An RDoc::TopLevel has the same hash as another with the same
  # relative_name

  def hash
    @relative_name.hash
  end

  ##
  # URL for this with a +prefix+

  def http_url(prefix)
    path = [prefix, @relative_name.tr('.', '_')]

    File.join(*path.compact) + '.html                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   gems/gems/rdoc-6.5.1.1/lib/rdoc/stats/quiet.rb                                                      0000644                 00000001500 15040313414 0014463 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       n self
      oc
    end
  end

  ##
  # Base name of this file without the extension

  def page_name
    basename = File.basename @relative_name
    basename =~ /\.(rb|rdoc|txt|md)$/i

    $` || basename
  end

  ##
  # Path to this file for use with HTML generator output.

  def path
    http_url @store.rdoc.generator.file_dir
  end

  def pretty_print q # :nodoc:
    q.group 2, "[#{self.class}: ", "]" do
      q.text "base name: #{base_name.inspect}"
      q.breakable

      items = @modules.map { |n,m| m }
      ite                                                                                                                                                                                            gems/gems/rdoc-6.5.1.1/lib/rdoc/generator.rb                                                        0000644                 00000003513 15040313414 0014172 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                               ustar 00                                                                                                                                                                                                                                                       gems/gems/rdoc-6.5.1.1/lib/rdoc/stats/quiet.rb                                                      0000644                 00000001500 15040313414 0014463 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Stats printer that prints nothing

class RDoc::Stats::Quiet

  ##
  # Creates a new Quiet that will print nothing

  def initialize num_files
    @num_files = num_files
  end

  ##
  # Prints a message at the beginning of parsing

  def begin_adding(*) end

  ##
  # Prints when an alias is added

  def print_alias(*) end

  ##
  # Prints when an attribute is added

  def print_attribute(*) end

  ##
  # Prints when a class is added

  def print_class(*) end

              gems/gems/rdoc-6.5.1.1/lib/rdoc/stats/normal.rb                                                     0000644                 00000003060 15040313414 0014627 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       gems/gems/rdoc-6.5.1                                                                                                                                                                                     gems/gems/rdoc-6.5.1.1/lib/rdoc/include.rb                                                          0000644                 00000000254 15040313414 0013626 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       oc::CodeObject +co+ being undocumented

  def nodoc co
    " (undocumented)" unless co.documented?
  end

  def print_alias as # :nodoc:
    puts "    alias #{as.new_name}                                                                                                                                                                                                                                                                                                                                                     gems/gems/rdoc-6.5.1.1/lib/rdoc/extend.rb                                                           0000644                 00000000252 15040313414 0013470 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       gems/gems/rdoc-6.5.1.1/lib/rdoc/markdown.rb                                                         0000644                 00001412356 15040313414 0014040 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       e
      # will be truncated if necessary.
      size = IO.respond_to?(:console_size) ? IO.console_size : IO.console.winsize
      terminal_width = size[1].to_i.nonzero? || 80
      max_filename_size = (terminal_width - progress_bar.size) - 1

      if filename.size > max_filename_size then
        # Turn "some_long_filename.rb" to "...ong_filename.rb"
        filename = filename[(filename.size - max_filename_size) .. -1]
        filename[0..2] = "..."
      end

      # Clean the line with whitespaces so that leftover output from the
      # previous line doesn't show up.
      $stdout.print("\r\e[K") if @last_width && @last_width > 0
      @last_width = progress_bar.size + filename.size
      term = "\r"
    else
      term = "\n"
    end
    $stdout.print(progress_bar, filename, term)
    $stdout.flush
  end

  def done_adding # :nodoc:
    puts
  end

end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                gems/gems/rdoc-6.5.1.1/lib/rdoc/context/section.rb                                                  0000644                 00000011301 15040313414 0015326 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require 'cgi/util'

##
# A section of documentation like:
#
#   # :section: The title
#   # The body
#
# Sections can be referenced multiple times and will be collapsed into a
# single section.

class RDoc::Context::Section

  include RDoc::Text

  MARSHAL_VERSION = 0 # :nodoc:

  ##
  # Section comment

  attr_reader :comment

  ##
  # Section comments

  attr_reader :comments

  ##
  # Context this Section lives in

  attr_reader :parent

  ##
  # Section title

  attr_reader :title

  ##
  # Creates a new section with +title+ and +comment+

  def initialize parent, title, comment
    @parent = parent
    @title = title ? title.strip : title

    @comments = []

    add_comment comment
  end

  ##
  # Sections are equal when they have the same #title

  def == other
    self.class === other and @title == other.title
  end

  alias eql? ==

  ##
  # Adds +comment+ to this section

  def add_comment comment
    comment = extract_comment comment

    return if comment.empty?

    case comment
    when RDoc::Comment then
      @comments << comment
    when RDoc::Markup::Document then
      @comments.concat comment.parts
    when Array then
      @comments.concat comment
    else
      raise TypeError, "unknown comment type: #{comment.inspect}"
    end
  end

  ##
  # Anchor reference for linking to this section

  def aref
    title = @title || '[untitled]'

    CGI.escape(title).gsub('%', '-').sub(/^-/, '')
  end

  ##
  # Extracts the comment for this section from the original comment block.
  # If the first line contains :section:, strip it and use the rest.
  # Otherwise remove lines up to the line containing :section:, and look
  # for those lines again at the end and remove them. This lets us write
  #
  #   # :section: The title
  #   # The body

  def extract_comment comment
    case comment
    when Array then
      comment.map do |c|
        extract_comment c
      end
    when nil
      RDoc::Comment.new ''
    when RDoc::Comment then
      if comment.text =~ /^#[ \t]*:section:.*\n/ then
        start = $`
        rest = $'

        comment.text = if start.empty? then
                         rest
                       else
                         rest.sub(/#{start.chomp}\Z/, '')
                       end
      end

      comment
    when RDoc::Markup::Document then
      comment
    else
      raise TypeError, "unknown comment #{comment.inspect}"
    end
  end

  def inspect # :nodoc:
    "#<%s:0x%x %p>" % [self.class, object_id, title]
  end

  def hash # :nodoc:
    @title.hash
  end

  ##
  # The files comments in this section come from

  def in_files
    return [] if @comments.empty?

    case @comments
    when Array then
      @comments.map do |comment|
                                                                                                                                                                                                                                                                                                                                gems/gems/rdoc-6.5.1.1/lib/rdoc/code_objects.rb                                                     0000644                 00000000242 15040313414 0014623 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ng then
      super
    when Array then
      docs = @comments.map do |comment, location|
        doc = super comment
        doc.file = location if location
                                                                                                                                                                                                                                                                                                                                                                  gems/gems/rdoc-6.5.1.1/lib/rdoc/ghost_method.rb                                                     0000644                 00000000220 15040313414 0014660 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       omments.parts.delete_if do |document|
        document.file == comment.file.name
      end
    else
      raise RDoc::Error, "BUG: unknown comme                                                                                                                                                                                                                                                                                                                                                                                gems/gems/rdoc-6.5.1.1/lib/rdoc/any_method.rb                                                       0000644                 00000020305 15040313414 0014331 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
# This file was used to load all the RDoc::CodeObject subclasses at once.  Now
# autoload handles this.

require_relative '../rdoc'
                                                                                                                                                                                                                                                                                                                                                              gems/gems/rdoc-6.5.1.1/lib/rdoc/ghost_method.rb                                                     0000644                 00000000220 15040313414 0014660 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# GhostMethod represents a method referenced only by a comment

class RDoc::GhostMethod < RDoc::AnyMethod
end

                                                                                                                                                                                                                                                                                                                                                                                gems/gems/rdoc-6.5.1.1/lib/rdoc/any_method.rb                                                       0000644                 00000020305 15040313414 0014331 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# AnyMethod is the base class for objects representing methods

class RDoc::AnyMethod < RDoc::MethodAttr

  ##
  # 2::
  #   RDoc 4
  #   Added calls_super
  #   Added parent name and class
  #   Added section title
  # 3::
  #   RDoc 4.1
  #   Added is_alias_for

  MARSHAL_VERSION = 3 # :nodoc:

  ##
  # Don't rename \#initialize to \::new

  attr_accessor :dont_rename_initialize

  ##
  # The C function that implements this method (if it was defined in a C file)

  attr_accessor :c_function

  # The section title of the method (if defined in a C file via +:category:+)
  attr_accessor :section_title

  # Parameters for this method

  attr_accessor :params

  ##
  # If true this method uses +super+ to call a superclass version

  attr_accessor :calls_super

  include RDoc::TokenStream

  ##
  # Creates a new AnyMethod with a token stream +text+ and +name+

  def initialize text, name
    super

    @c_function = nil
    @dont_rename_initialize = false
    @token_stream = nil
    @calls_super = false
    @superclass_method = nil
  end

  ##
  # Adds +an_alias+ as an alias for this method in +context+.

  def add_alias an_alias, context = nil
    method = self.class.new an_alias.text, an_alias.new_name

    method.record_location an_alias.file
    method.singleton = self.singleton
    method.params = self.params
    method.visibility = self.visibility
    method.comment = an_alias.comment
    method.is_alias_for = self
    @aliases << method
    context.add_method method if context
    method
  end

  ##
  # Prefix for +aref+ is 'method'.

  def aref_prefix
    'method'
  end

  ##
  # The call_seq or the param_seq with method name, if there is no call_seq.
  #
  # Use this for displaying a method's argument lists.

  def arglists
    if @call_seq then
      @call_seq
    elsif @params then
      "#{name}#{param_seq}"
    end
  end

  ##
  # Different ways to call this method

  def call_seq
    unless call_seq = _call_seq
      call_seq = is_alias_for._call_seq if is_alias_for
    end

    return unless call_seq

    deduplicate_call_seq(call_seq)
  end

  ##
  # Sets the different ways you can call this method.  If an empty +call_seq+
  # is given nil is assumed.
  #
  # See also #param_seq

  def call_seq= call_seq
    return if call_seq.empty?

    @call_seq = call_seq
  end

  ##
  # Loads is_alias_for from the internal name.  Returns nil if the alias
  # cannot be found.

  def is_alias_for # :nodoc:
    case @is_alias_for
    when RDoc::MethodAttr then
      @is_alias_for
    when Array then
      return nil unless @store

      klass_name, singleton, method_name = @is_alias_for

      return nil unless klass = @store.find_class_or_module(klass_name)

      @is_alias_for = klass.find_method method_name, singleton
    end
  end

  ##
  # Dumps this AnyMethod for use by ri.  See also #marshal_load

  def marshal_dump
    aliases = @aliases.map do |a|
      [a.name, parse(a.comment)]
    end

    is_alias_for = [
      @is_alias_for.parent.full_name,
      @is_alias_for.singleton,
      @is_alias_for.name
    ] if @is_alias_for

    [ MARSHAL_VERSION,
      @name,
      full_name,
      @singleton,
      @visibility,
      parse(@comment),
      @call_seq,
      @block_params,
      aliases,
      @params,
      @file.relative_name,
      @calls_super,
      @parent.name,
      @parent.class,
      @section.title,
      is_alias_for,
    ]
  end

  ##
  # Loads this AnyMethod from +array+.  For a loaded AnyMethod the following
  # methods will return cached values:
  #
  # * #full_name
  # * #parent_name

  def marshal_load array
    initialize_visibility

    @dont_rename_initialize = nil
    @token_stream           = nil
    @aliases                = []
    @parent                 = nil
    @parent_name            = nil
    @parent_class           = nil
    @section                = nil
    @file                   = nil

    version        = array[0]
    @name          = array[1]
    @full_name     = array[2]
    @singleton     = array[3]
    @visibility    = array[4]
    @comment       = array[5]
    @call_seq      = array[6]
    @block_params  = array[7]
    #                      8 handled below
    @params        = array[9]
    #                      10 handled below
    @calls_super   = array[11]
    @parent_name   = array[12]
    @parent_title  = array[13]
    @section_title = array[14]
    @is_alias_for  = array[15]

    array[8].each do |new_name, comment|
      add_alias RDoc::Alias.new(nil, @name, new_name, comment, @singleton)
    end

    @parent_name ||= if @full_name =~ /#/ then
                       $`
                     else
                       name = @full_name.split('::')
                       name.pop
                       name.join '::'
                     end

    @file = RDoc::TopLevel.new array[10] if version > 0
  end

  ##
  # Method name
  #
  # If the method has no assigned name, it extracts it from #call_seq.

  def name
    return @name if @name

    @name =
      @call_seq[/^.*?\.(\w+)/, 1] ||
      @call_seq[/^.*?(\w+)/, 1] ||
      @call_seq if @call_seq
  end

  ##
  # A list of this method's method and yield parameters.  +call-seq+ params
  # are preferred over parsed method and block params.

  def param_list
    if @call_seq then
      params = @call_seq.split("\n").last
      params = params.sub(/.*?\((.*)\)/, '\1')
      params = params.sub(/(\{|do)\s*\|([^|]*)\|.*/, ',\2')
    elsif @params then
      params = @params.sub(/\((.*)\)/, '\1')

      params << ",#{@block_params}" if @block_params
    elsif @block_params then
      params = @block_params
    else
      return []
    end

    if @block_params then
      # If this method has explicit block parameters, remove any explicit
      # &block
      params = params.sub(/,?\s*&\w+/, '')
    else
      params = params.sub(/\&(\w+)/, '\1')
    end

    params = params.gsub(/\s+/, '').split(',').reject(&:empty?)

    params.map { |param| param.sub(/=.*/, '') }
  end

  ##
  # Pretty parameter list for this method.  If the method's parameters were
  # given by +call-seq+ it is preferred over the parsed values.

  def param_seq
    if @call_seq then
      params = @call_seq.split("\n").last
      params = params.sub(/[^( ]+/, '')
      params = params.sub(/(\|[^|]+\|)\s*\.\.\.\s*(end|\})/, '\                                                                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/comment.rb                                                          0000644                 00000013362 15040313414 0013651 0                                                                                                    ustar 00                                                                                                                                                                                                                                                           return unless @calls_super
    return @superclass_method if @superclass_method

    parent.each_ancestor do |ancestor|
      if method = ancestor.method_list.find { |m| m.name == @name } then
        @superclass_method = method
        break
      end
    end

    @superclass_method
  end

  protected

  ##
  # call_seq without deduplication and alias lookup.

  def _call_seq
    @call_seq if defined?(@call_seq) && @call_seq
  end

  private

  ##
  # call_seq with alias examples information removed, if this
  # method is an alias method.

  def deduplicate_call_seq(call_seq)
    return call_seq unless is_alias_for || !aliases.empty?

    method_name = self.name
    method_name = method_name[0, 1] if method_name =~ /\A\[/

    entries = call_seq.split "\n"

    ignore = aliases.map(&:name)
    if is_alias_for
      ignore << is_alias_for.name
      ignore.concat is_alias_for.aliases.map(&:name)
    end
    ignore.map! { |n| n =~ /\A\[/ ? /\[.*\]/ : n}
    ignore.delete(method_name)
    ignore = Regexp.union(ignore)

    matching = entries.reject do |entry|
      entry =~ /^\w*\.?#{ignore}[$\(\s]/ or
        entry =~ /\s#{ignore}\s/
    end

    matching.empty? ? nil : matching.join("\n")
  end
end
                                                                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/comment.rb                                                          0000644                 00000013362 15040313414 0013651 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# A comment holds the text comment for a RDoc::CodeObject and provides a
# unified way of cleaning it up and parsing it into an RDoc::Markup::Document.
#
# Each comment may have a different markup format set by #format=.  By default
# 'rdoc' is used.  The :markup: directive tells RDoc which format to use.
#
# See RDoc::Markup@Other+directives for instructions on adding an alternate
# format.

class RDoc::Comment

  include RDoc::Text

  ##
  # The format of this comment.  Defaults to RDoc::Markup

  attr_reader :format

  ##
  # The RDoc::TopLevel this comment was found in

  attr_accessor :location

  ##
  # Line where this Comment was written

  attr_accessor :line

  ##
  # For duck-typing when merging classes at load time

  alias file location # :nodoc:

  ##
  # The text for this comment

  attr_reader :text

  ##
  # Alias for text

  alias to_s text

  ##
  # Overrides the content returned by #parse.  Use when there is no #text
  # source for this comment

  attr_writer   :document

  ##
  # Creates a new comment with +text+ that is found in the RDoc::TopLevel
  # +location+.

  def initialize text = nil, location = nil, language = nil
    @location = location
    @text     = text.nil? ? nil : text.dup
    @language = language

    @document   = nil
    @format     = 'rdoc'
    @normalized = false
  end

  ##
  #--
  # TODO deep copy @document

  def initialize_copy copy # :nodoc:
    @text = copy.text.dup
  end

  def == other # :nodoc:
    self.class === other and
      other.text == @text and other.location == @location
  end

  ##
  # Look for a 'call-seq' in the comment to override the normal parameter
  # handling.  The :call-seq: is indented from the baseline.  All lines of the
  # same indentation level and prefix are consumed.
  #
  # For example, all of the following will be used as the :call-seq:
  #
  #   # :call-seq:
  #   #   ARGF.readlines(sep=$/)     -> array
  #   #   ARGF.readlines(limit)      -> array
  #   #   ARGF.readlines(sep, limit) -> array
  #   #
  #   #   ARGF.to_a(sep=$/)     -> array
  #   #   ARGF.to_a(limit)      -> array
  #   #   ARGF.to_a(sep, limit) -> array

  def extract_call_seq method
    # we must handle situations like the above followed by an unindented first
    # comment.  The difficulty is to make sure not to match lines starting
    # with ARGF at the same indent, but that are after the first description
    # paragraph.
    if @text =~ /^\s*:?call-seq:(.*?(?:\S).*?)^\s*$/m then
      all_start, all_stop = $~.offset(0)
      seq_start, seq_stop = $~.offset(1)

      # we get the following lines that start with the leading word at the
      # same indent, even if they have blank lines before
      if $1 =~ /(^\s*\n)+^(\s*\w+)/m then
        leading = $2 # ' *    ARGF' in the example above
        re = %r%
          \A(
             (^\s*\n)+
             (^#{Regexp.escape leading}.*?\n)+
            )+
          ^\s*$
        %xm

        if @text[seq_stop..-1] =~ re then
          all_stop = seq_stop + $~.offset(0).last
          seq_stop = seq_stop + $~.offset(1).last
        end
      end

      seq = @text[seq_start..seq_stop]
      seq.gsub!(/^\s*(\S|\n)/m, '\1')
      @text.slice! all_start...all_stop

      method.call_seq = seq.chomp

    else
      regexp = /^\s*:?call-seq:(.*?)(^\s*$|\z)/m
      if regexp =~ @text then
        @text = @text.sub(regexp, '')
        seq = $1
        seq.gsub!(/^\s*/, '')
        method.call_seq = seq
      end
    end

    method
  end

  ##
  # A comment is empty if its text String is empty.

  def empty?
    @text.empty?
  end

  ##
  # HACK dubious

  def encode! encoding
    # TODO: Remove this condition after Ruby 2.2 EOL
    if RUBY_VERSION < '2.3.0'
      @text = @text.force_encoding encoding
    else
      @text = String.                                                                                                                                                                                                                                                                              gems/gems/rdoc-6.5.1.1/lib/rdoc/single_class.rb                                                     0000644                 00000001000 15040313414 0014637 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       arsed document is
  # cached until the text is changed.

  def parse
    return @document if @document

    @document = super @text, @format
    @document.file = @location
    @document
  end

  ##
  # Removes private sections from this comment.  Private sections are flush to
  # the comment marker and start with <tt>--</tt> and end with <tt>++</tt>.
  # For C-style comments, a private marker may not start at the opening of the
  # comment.
  #
  #   /*
  #    *--
  #    * private
  #    *++
  #    * publicgems/gems/rdoc-6.5.1.1/lib/rdoc/markdown/entities.rb                                                0000644                 00000154041 15040313414 0015655 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       nt-only comment is not allowed' if
      @text.nil? and @document

    @document = nil
    @text = text.nil? ? nil : text.dup
  end

  ##
  # Returns true if this comment is in TomDoc format.

  def tomdoc?
    @format == 'tomdoc'
  end

end
                                                                                                                                                                                                                                                                              gems/gems/rdoc-6.5.1.1/lib/rdoc/single_class.rb                                                     0000644                 00000001000 15040313414 0014637 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# A singleton class

class RDoc::SingleClass < RDoc::ClassModule

  ##
  # Adds the superclass to the included modules.

  def ancestors
    superclass ? super + [superclass] : super
  end

  def aref_prefix # :nodoc:
    'sclass'
  end

  ##
  # The definition of this singleton class, <tt>class << MyClassName</tt>

  def definition
    "class << #{full_name}"
  end

  def pretty_print q # :nodoc:
    q.group 2, "[class << #{full_name}", "]" do
      next
    end
  end
end

gems/gems/rdoc-6.5.1.1/lib/rdoc/markdown/entities.rb                                                0000644                 00000154041 15040313414 0015655 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# HTML entity name map for RDoc::Markdown

RDoc::Markdown::HTML_ENTITIES = {
  "AElig" => [0x000C6],
  "AMP" => [0x00026],
  "Aacute" => [0x000C1],
  "Abreve" => [0x00102],
  "Acirc" => [0x000C2],
  "Acy" => [0x00410],
  "Afr" => [0x1D504],
  "Agrave" => [0x000C0],
  "Alpha" => [0x00391],
  "Amacr" => [0x00100],
  "And" => [0x02A53],
  "Aogon" => [0x00104],
  "Aopf" => [0x1D538],
  "ApplyFunction" => [0x02061],
  "Aring" => [0x000C5],
  "Ascr" => [0x1D49C],
  "Assign" => [0x02254],
  "Atilde" => [0x000C3],
  "Auml" => [0x000C4],
  "Backslash" => [0x02216],
  "Barv" => [0x02AE7],
  "Barwed" => [0x02306],
  "Bcy" => [0x00411],
  "Because" => [0x02235],
  "Bernoullis" => [0x0212C],
  "Beta" => [0x00392],
  "Bfr" => [0x1D505],
  "Bopf" => [0x1D539],
  "Breve" => [0x002D8],
  "Bscr" => [0x0212C],
  "Bumpeq" => [0x0224E],
  "CHcy" => [0x00427],
  "COPY" => [0x000A9],
  "Cacute" => [0x00106],
  "Cap" => [0x022D2],
  "CapitalDifferentialD" => [0x02145],
  "Cayleys" => [0x0212D],
  "Ccaron" => [0x0010C],
  "Ccedil" => [0x000C7],
  "Ccirc" => [0x00108],
  "Cconint" => [0x02230],
  "Cdot" => [0x0010A],
  "Cedilla" => [0x000B8],
  "CenterDot" => [0x000B7],
  "Cfr" => [0x0212D],
  "Chi" => [0x003A7],
  "CircleDot" => [0x02299],
  "CircleMinus" => [0x02296],
  "CirclePlus" => [0x02295],
  "CircleTimes" => [0x02297],
  "ClockwiseContourIntegral" => [0x02232],
  "CloseCurlyDoubleQuote" => [0x0201D],
  "CloseCurlyQuote" => [0x02019],
  "Colon" => [0x02237],
  "Colone" => [0x02A74],
  "Congruent" => [0x02261],
  "Conint" => [0x0222F],
  "ContourIntegral" => [0x0222E],
  "Copf" => [0x02102],
  "Coproduct" => [0x02210],
  "CounterClockwiseContourIntegral" => [0x02233],
  "Cross" => [0x02A2F],
  "Cscr" => [0x1D49E],
  "Cup" => [0x022D3],
  "CupCap" => [0x0224D],
  "DD" => [0x02145],
  "DDotrahd" => [0x02911],
  "DJcy" => [0x00402],
  "DScy" => [0x00405],
  "DZcy" => [0x0040F],
  "Dagger" => [0x02021],
  "Darr" => [0x021A1],
  "Dashv" => [0x02AE4],
  "Dcaron" => [0x0010E],
  "Dcy" => [0x00414],
  "Del" => [0x02207],
  "Delta" => [0x00394],
  "Dfr" => [0x1D507],
  "DiacriticalAcute" => [0x000B4],
  "DiacriticalDot" => [0x002D9],
  "DiacriticalDoubleAcute" => [0x002DD],
  "DiacriticalGrave" => [0x00060],
  "DiacriticalTilde" => [0x002DC],
  "Diamond" => [0x022C4],
  "DifferentialD" => [0x02146],
  "Dopf" => [0x1D53B],
  "Dot" => [0x000A8],
  "DotDot" => [0x020DC],
  "DotEqual" => [0x02250],
  "DoubleContourIntegral" => [0x0222F],
  "DoubleDot" => [0x000A8],
  "DoubleDownArrow" => [0x021D3],
  "DoubleLeftArrow" => [0x021D0],
  "DoubleLeftRightArrow" => [0x021D4],
  "DoubleLeftTee" => [0x02AE4],
  "DoubleLongLeftArrow" => [0x027F8],
  "DoubleLongLeftRightArrow" => [0x027FA],
  "DoubleLongRightArrow" => [0x027F9],
  "DoubleRightArrow" => [0x021D2],
  "DoubleRightTee" => [0x022A8],
  "DoubleUpArrow" => [0x021D1],
  "DoubleUpDownArrow" => [0x021D5],
  "DoubleVerticalBar" => [0x02225],
  "DownArrow" => [0x02193],
  "DownArrowBar" => [0x02913],
  "DownArrowUpArrow" => [0x021F5],
  "DownBreve" => [0x00311],
  "DownLeftRightVector" => [0x02950],
  "DownLeftTeeVector" => [0x0295E],
  "DownLeftVector" => [0x021BD],
  "DownLeftVectorBar" => [0x02956],
  "DownRightTeeVector" => [0x0295F],
  "DownRightVector" => [0x021C1],
  "DownRightVectorBar" => [0x02957],
  "DownTee" => [0x022A4],
  "DownTeeArrow" => [0x021A7],
  "Downarrow" => [0x021D3],
  "Dscr" => [0x1D49F],
  "Dstrok" => [0x00110],
  "ENG" => [0x0014A],
  "ETH" => [0x000D0],
  "Eacute" => [0x000C9],
  "Ecaron" => [0x0011A],
  "Ecirc" => [0x000CA],
  "Ecy" => [0x0042D],
  "Edot" => [0x00116],
  "Efr" => [0x1D508],
  "Egrave" => [0x000C8],
  "Element" => [0x02208],
  "Emacr" => [0x00112],
  "EmptySmallSquare" => [0x025FB],
  "EmptyVerySmallSquare" => [0x025AB],
  "Eogon" => [0x00118],
  "Eopf" => [0x1D53C],
  "Epsilon" => [0x00395],
  "Equal" => [0x02A75],
  "EqualTilde" => [0x02242],
  "Equilibrium" => [0x021CC],
  "Escr" => [0x02130],
  "Esim" => [0x02A73],
  "Eta" => [0x00397],
  "Euml" => [0x000CB],
  "Exists" => [0x02203],
  "ExponentialE" => [0x02147],
  "Fcy" => [0x00424],
  "Ffr" => [0x1D509],
  "FilledSmallSquare" => [0x025FC],
  "FilledVerySmallSquare" => [0x025AA],
  "Fopf" => [0x1D53D],
  "ForAll" => [0x02200],
  "Fouriertrf" => [0x02131],
  "Fscr" => [0x02131],
  "GJcy" => [0x00403],
  "GT" => [0x0003E],
  "Gamma" => [0x00393],
  "Gammad" => [0x003DC],
  "Gbreve" => [0x0011E],
  "Gcedil" => [0x00122],
  "Gcirc" => [0x0011C],
  "Gcy" => [0x00413],
  "Gdot" => [0x00120],
  "Gfr" => [0x1D50A],
  "Gg" => [0x022D9],
  "Gopf" => [0x1D53E],
  "GreaterEqual" => [0x02265],
  "GreaterEqualLess" => [0x022DB],
  "GreaterFullEqual" => [0x02267],
  "GreaterGreater" => [0x02AA2],
  "GreaterLess" => [0x02277],
  "GreaterSlantEqual" => [0x02A7E],
  "GreaterTilde" => [0x02273],
  "Gscr" => [0x1D4A2],
  "Gt" => [0x0226B],
  "HARDcy" => [0x0042A],
  "Hacek" => [0x002C7],
  "Hat" => [0x0005E],
  "Hcirc" => [0x00124],
  "Hfr" => [0x0210C],
  "HilbertSpace" => [0x0210B],
  "Hopf" => [0x0210D],
  "HorizontalLine" => [0x02500],
  "Hscr" => [0x0210B],
  "Hstrok" => [0x00126],
  "HumpDownHump" => [0x0224E],
  "HumpEqual" => [0x0224F],
  "IEcy" => [0x00415],
  "IJlig" => [0x00132],
  "IOcy" => [0x00401],
  "Iacute" => [0x000CD],
  "Icirc" => [0x000CE],
  "Icy" => [0x00418],
  "Idot" => [0x00130],
  "Ifr" => [0x02111],
  "Igrave" => [0x000CC],
  "Im" => [0x02111],
  "Imacr" => [0x0012A],
  "ImaginaryI" => [0x02148],
  "Implies" => [0x021D2],
  "Int" => [0x0222C],
  "Integral" => [0x0222B],
  "Intersection" => [0x022C2],
  "InvisibleComma" => [0x02063],
  "InvisibleTimes" => [0x02062],
  "Iogon" => [0x0012E],
  "Iopf" => [0x1D540],
  "Iota" => [0x00399],
  "Iscr" => [0x02110],
  "Itilde" => [0x00128],
  "Iukcy" => [0x00406],
  "Iuml" => [0x000CF],
  "Jcirc" => [0x00134],
  "Jcy" => [0x00419],
  "Jfr" => [0x1D50D],
  "Jopf" => [0x1D541],
  "Jscr" => [0x1D4A5],
  "Jsercy" => [0x00408],
  "Jukcy" => [0x00404],
  "KHcy" => [0x00425],
  "KJcy" => [0x0040C],
  "Kappa" => [0x0039A],
  "Kcedil" => [0x00136],
  "Kcy" => [0x0041A],
  "Kfr" => [0x1D50E],
  "Kopf" => [0x1D542],
  "Kscr" => [0x1D4A6],
  "LJcy" => [0x00409],
  "LT" => [0x0003C],
  "Lacute" => [0x00139],
  "Lambda" => [0x0039B],
  "Lang" => [0x027EA],
  "Laplacetrf" => [0x02112],
  "Larr" => [0x0219E],
  "Lcaron" => [0x0013D],
  "Lcedil" => [0x0013B],
  "Lcy" => [0x0041B],
  "LeftAngleBracket" => [0x027E8],
  "LeftArrow" => [0x02190],
  "LeftArrowBar" => [0x021E4],
  "LeftArrowRightArrow" => [0x021C6],
  "LeftCeiling" => [0x02308],
  "LeftDoubleBracket" => [0x027E6],
  "LeftDownTeeVector" => [0x02961],
  "LeftDownVector" => [0x021C3],
  "LeftDownVectorBar" => [0x02959],
  "LeftFloor" => [0x0230A],
  "LeftRightArrow" => [0x02194],
  "LeftRightVector" => [0x0294E],
  "LeftTee" => [0x022A3],
  "LeftTeeArrow" => [0x021A4],
  "LeftTeeVector" => [0x0295A],
  "LeftTriangle" => [0x022B2],
  "LeftTriangleBar" => [0x029CF],
  "LeftTriangleEqual" => [0x022B4],
  "LeftUpDownVector" => [0x02951],
  "LeftUpTeeVector" => [0x02960],
  "LeftUpVector" => [0x021BF],
  "LeftUpVectorBar" => [0x02958],
  "LeftVector" => [0x021BC],
  "LeftVectorBar" => [0x02952],
  "Leftarrow" => [0x021D0],
  "Leftrightarrow" => [0x021D4],
  "LessEqualGreater" => [0x022DA],
  "LessFullEqual" => [0x02266],
  "LessGreater" => [0x02276],
  "LessLess" => [0x02AA1],
  "LessSlantEqual" => [0x02A7D],
  "LessTilde" => [0x02272],
  "Lfr" => [0x1D50F],
  "Ll" => [0x022D8],
  "Lleftarrow" => [0x021DA],
  "Lmidot" => [0x0013F],
  "LongLeftArrow" => [0x027F5],
  "LongLeftRightArrow" => [0x027F7],
  "LongRightArrow" => [0x027F6],
  "Longleftarrow" => [0x027F8],
  "Longleftrightarrow" => [0x027FA],
  "Longrightarrow" => [0x027F9],
  "Lopf" => [0x1D543],
  "LowerLeftArrow" => [0x02199],
  "LowerRightArrow" => [0x02198],
  "Lscr" => [0x02112],
  "Lsh" => [0x021B0],
  "Lstrok" => [0x00141],
  "Lt" => [0x0226A],
  "Map" => [0x02905],
  "Mcy" => [0x0041C],
  "MediumSpace" => [0x0205F],
  "Mellintrf" => [0x02133],
  "Mfr" => [0x1D510],
  "MinusPlus" => [0x02213],
  "Mopf" => [0x1D544],
  "Mscr" => [0x02133],
  "Mu" => [0x0039C],
  "NJcy" => [0x0040A],
  "Nacute" => [0x00143],
  "Ncaron" => [0x00147],
  "Ncedil" => [0x00145],
  "Ncy" => [0x0041D],
  "NegativeMediumSpace" => [0x0200B],
  "NegativeThickSpace" => [0x0200B],
  "NegativeThinSpace" => [0x0200B],
  "NegativeVeryThinSpace" => [0x0200B],
  "NestedGreaterGreater" => [0x0226B],
  "NestedLessLess" => [0x0226A],
  "NewLine" => [0x0000A],
  "Nfr" => [0x1D511],
  "NoBreak" => [0x02060],
  "NonBreakingSpace" => [0x000A0],
  "Nopf" => [0x02115],
  "Not" => [0x02AEC],
  "NotCongruent" => [0x02262],
  "NotCupCap" => [0x0226D],
  "NotDoubleVerticalBar" => [0x02226],
  "NotElement" => [0x02209],
  "NotEqual" => [0x02260],
  "NotEqualTilde" => [0x02242, 0x00338],
  "NotExists" => [0x02204],
  "NotGreater" => [0x0226F],
  "NotGreaterEqual" => [0x02271],
  "NotGreaterFullEqual" => [0x02267, 0x00338],
  "NotGreaterGreater" => [0x0226B, 0x00338],
  "NotGreaterLess" => [0x02279],
  "NotGreaterSlantEqual" => [0x02A7E, 0x00338],
  "NotGreaterTilde" => [0x02275],
  "NotHumpDownHump" => [0x0224E, 0x00338],
  "NotHumpEqual" => [0x0224F, 0x00338],
  "NotLeftTriangle" => [0x022EA],
  "NotLeftTriangleBar" => [0x029CF, 0x00338],
  "NotLeftTriangleEqual" => [0x022EC],
  "NotLess" => [0x0226E],
  "NotLessEqual" => [0x02270],
  "NotLessGreater" => [0x02278],
  "NotLessLess" => [0x0226A, 0x00338],
  "NotLessSlantEqual" => [0x02A7D, 0x00338],
  "NotLessTilde" => [0x02274],
  "NotNestedGreaterGreater" => [0x02AA2, 0x00338],
  "NotNestedLessLess" => [0x02AA1, 0x00338],
  "NotPrecedes" => [0x02280],
  "NotPrecedesEqual" => [0x02AAF, 0x00338],
  "NotPrecedesSlantEqual" => [0x022E0],
  "NotReverseElement" => [0x0220C],
  "NotRightTriangle" => [0x022EB],
  "NotRightTriangleBar" => [0x029D0, 0x00338],
  "NotRightTriangleEqual" => [0x022ED],
  "NotSquareSubset" => [0x0228F, 0x00338],
  "NotSquareSubsetEqual" => [0x022E2],
  "NotSquareSuperset" => [0x02290, 0x00338],
  "NotSquareSupersetEqual" => [0x022E3],
  "NotSubset" => [0x02282, 0x020D2],
  "NotSubsetEqual" => [0x02288],
  "NotSucceeds" => [0x02281],
  "NotSucceedsEqual" => [0x02AB0, 0x00338],
  "NotSucceedsSlantEqual" => [0x022E1],
  "NotSucceedsTilde" => [0x0227F, 0x00338],
  "NotSuperset" => [0x02283, 0x020D2],
  "NotSupersetEqual" => [0x02289],
  "NotTilde" => [0x02241],
  "NotTildeEqual" => [0x02244],
  "NotTildeFullEqual" => [0x02247],
  "NotTildeTilde" => [0x02249],
  "NotVerticalBar" => [0x02224],
  "Nscr" => [0x1D4A9],
  "Ntilde" => [0x000D1],
  "Nu" => [0x0039D],
  "OElig" => [0x00152],
  "Oacute" => [0x000D3],
  "Ocirc" => [0x000D4],
  "Ocy" => [0x0041E],
  "Odblac" => [0x00150],
  "Ofr" => [0x1D512],
  "Ograve" => [0x000D2],
  "Omacr" => [0x0014C],
  "Omega" => [0x003A9],
  "Omicron" => [0x0039F],
  "Oopf" => [0x1D546],
  "OpenCurlyDoubleQuote" => [0x0201C],
  "OpenCurlyQuote" => [0x02018],
  "Or" => [0x02A54],
  "Oscr" => [0x1D4AA],
  "Oslash" => [0x000D8],
  "Otilde" => [0x000D5],
  "Otimes" => [0x02A37],
  "Ouml" => [0x000D6],
  "OverBar" => [0x0203E],
  "OverBrace" => [0x023DE],
  "OverBracket" => [0x023B4],
  "OverParenthesis" => [0x023DC],
  "PartialD" => [0x02202],
  "Pcy" => [0x0041F],
  "Pfr" => [0x1D513],
  "Phi" => [0x003A6],
  "Pi" => [0x003A0],
  "PlusMinus" => [0x000B1],
  "Poincareplane" => [0x0210C],
  "Popf" => [0x02119],
  "Pr" => [0x02ABB],
  "Precedes" => [0x0227A],
  "PrecedesEqual" => [0x02AAF],
  "PrecedesSlantEqual" => [0x0227C],
  "PrecedesTilde" => [0x0227E],
  "Prime" => [0x02033],
  "Product" => [0x0220F],
  "Proportion" => [0x02237],
  "Proportional" => [0x0221D],
  "Pscr" => [0x1D4AB],
  "Psi" => [0x003A8],
  "QUOT" => [0x00022],
  "Qfr" => [0x1D514],
  "Qopf" => [0x0211A],
  "Qscr" => [0x1D4AC],
  "RBarr" => [0x02910],
  "REG" => [0x000AE],
  "Racute" => [0x00154],
  "Rang" => [0x027EB],
  "Rarr" => [0x021A0],
  "Rarrtl" => [0x02916],
  "Rcaron" => [0x00158],
  "Rcedil" => [0x00156],
  "Rcy" => [0x00420],
  "Re" => [0x0211C],
  "ReverseElement" => [0x0220B],
  "ReverseEquilibrium" => [0x021CB],
  "ReverseUpEquilibrium" => [0x0296F],
  "Rfr" => [0x0211C],
  "Rho" => [0x003A1],
  "RightAngleBracket" => [0x027E9],
  "RightArrow" => [0x02192],
  "RightArrowBar" => [0x021E5],
  "RightArrowLeftArrow" => [0x021C4],
  "RightCeiling" => [0x02309],
  "RightDoubleBracket" => [0x027E7],
  "RightDownTeeVector" => [0x0295D],
  "RightDownVector" => [0x021C2],
  "RightDownVectorBar" => [0x02955],
  "RightFloor" => [0x0230B],
  "RightTee" => [0x022A2],
  "RightTeeArrow" => [0x021A6],
  "RightTeeVector" => [0x0295B],
  "RightTriangle" => [0x022B3],
  "RightTriangleBar" => [0x029D0],
  "RightTriangleEqual" => [0x022B5],
  "RightUpDownVector" => [0x0294F],
  "RightUpTeeVector" => [0x0295C],
  "RightUpVector" => [0x021BE],
  "RightUpVectorBar" => [0x02954],
  "RightVector" => [0x021C0],
  "RightVectorBar" => [0x02953],
  "Rightarrow" => [0x021D2],
  "Ropf" => [0x0211D],
  "RoundImplies" => [0x02970],
  "Rrightarrow" => [0x021DB],
  "Rscr" => [0x0211B],
  "Rsh" => [0x021B1],
  "RuleDelayed" => [0x029F4],
  "SHCHcy" => [0x00429],
  "SHcy" => [0x00428],
  "SOFTcy" => [0x0042C],
  "Sacute" => [0x0015A],
  "Sc" => [0x02ABC],
  "Scaron" => [0x00160],
  "Scedil" => [0x0015E],
  "Scirc" => [0x0015C],
  "Scy" => [0x00421],
  "Sfr" => [0x1D516],
  "ShortDownArrow" => [0x02193],
  "ShortLeftArrow" => [0x02190],
  "ShortRightArrow" => [0x02192],
  "ShortUpArrow" => [0x02191],
  "Sigma" => [0x003A3],
  "SmallCircle" => [0x02218],
  "Sopf" => [0x1D54A],
  "Sqrt" => [0x0221A],
  "Square" => [0x025A1],
  "SquareIntersection" => [0x02293],
  "SquareSubset" => [0x0228F],
  "SquareSubsetEqual" => [0x02291],
  "SquareSuperset" => [0x02290],
  "SquareSupersetEqual" => [0x02292],
  "SquareUnion" => [0x02294],
  "Sscr" => [0x1D4AE],
  "Star" => [0x022C6],
  "Sub" => [0x022D0],
  "Subset" => [0x022D0],
  "SubsetEqual" => [0x02286],
  "Succeeds" => [0x0227B],
  "SucceedsEqual" => [0x02AB0],
  "SucceedsSlantEqual" => [0x0227D],
  "SucceedsTilde" => [0x0227F],
  "SuchThat" => [0x0220B],
  "Sum" => [0x02211],
  "Sup" => [0x022D1],
  "Superset" => [0x02283],
  "SupersetEqual" => [0x02287],
  "Supset" => [0x022D1],
  "THORN" => [0x000DE],
  "TRADE" => [0x02122],
  "TSHcy" => [0x0040B],
  "TScy" => [0x00426],
  "Tab" => [0x00009],
  "Tau" => [0x003A4],
  "Tcaron" => [0x00164],
  "Tcedil" => [0x00162],
  "Tcy" => [0x00422],
  "Tfr" => [0x1D517],
  "Therefore" => [0x02234],
  "Theta" => [0x00398],
  "ThickSpace" => [0x0205F, 0x0200A],
  "ThinSpace" => [0x02009],
  "Tilde" => [0x0223C],
  "TildeEqual" => [0x02243],
  "TildeFullEqual" => [0x02245],
  "TildeTilde" => [0x02248],
  "Topf" => [0x1D54B],
  "TripleDot" => [0x020DB],
  "Tscr" => [0x1D4AF],
  "Tstrok" => [0x00166],
  "Uacute" => [0x000DA],
  "Uarr" => [0x0219F],
  "Uarrocir" => [0x02949],
  "Ubrcy" => [0x0040E],
  "Ubreve" => [0x0016C],
  "Ucirc" => [0x000DB],
  "Ucy" => [0x00423],
  "Udblac" => [0x00170],
  "Ufr" => [0x1D518],
  "Ugrave" => [0x000D9],
  "Umacr" => [0x0016A],
  "UnderBar" => [0x0005F],
  "UnderBrace" => [0x023DF],
  "UnderBracket" => [0x023B5],
  "UnderParenthesis" => [0x023DD],
  "Union" => [0x022C3],
  "UnionPlus" => [0x0228E],
  "Uogon" => [0x00172],
  "Uopf" => [0x1D54C],
  "UpArrow" => [0x02191],
  "UpArrowBar" => [0x02912],
  "UpArrowDownArrow" => [0x021C5],
  "UpDownArrow" => [0x02195],
  "UpEquilibrium" => [0x0296E],
  "UpTee" => [0x022A5],
  "UpTeeArrow" => [0x021A5],
  "Uparrow" => [0x021D1],
  "Updownarrow" => [0x021D5],
  "UpperLeftArrow" => [0x02196],
  "UpperRightArrow" => [0x02197],
  "Upsi" => [0x003D2],
  "Upsilon" => [0x003A5],
  "Uring" => [0x0016E],
  "Uscr" => [0x1D4B0],
  "Utilde" => [0x00168],
  "Uuml" => [0x000DC],
  "VDash" => [0x022AB],
  "Vbar" => [0x02AEB],
  "Vcy" => [0x00412],
  "Vdash" => [0x022A9],
  "Vdashl" => [0x02AE6],
  "Vee" => [0x022C1],
  "Verbar" => [0x02016],
  "Vert" => [0x02016],
  "VerticalBar" => [0x02223],
  "VerticalLine" => [0x0007C],
  "VerticalSeparator" => [0x02758],
  "VerticalTilde" => [0x02240],
  "VeryThinSpace" => [0x0200A],
  "Vfr" => [0x1D519],
  "Vopf" => [0x1D54D],
  "Vscr" => [0x1D4B1],
  "Vvdash" => [0x022AA],
  "Wcirc" => [0x00174],
  "Wedge" => [0x022C0],
  "Wfr" => [0x1D51A],
  "Wopf" => [0x1D54E],
  "Wscr" => [0x1D4B2],
  "Xfr" => [0x1D51B],
  "Xi" => [0x0039E],
  "Xopf" => [0x1D54F],
  "Xscr" => [0x1D4B3],
  "YAcy" => [0x0042F],
  "YIcy" => [0x00407],
  "YUcy" => [0x0042E],
  "Yacute" => [0x000DD],
  "Ycirc" => [0x00176],
  "Ycy" => [0x0042B],
  "Yfr" => [0x1D51C],
  "Yopf" => [0x1D550],
  "Yscr" => [0x1D4B4],
  "Yuml" => [0x00178],
  "ZHcy" => [0x00416],
  "Zacute" => [0x00179],
  "Zcaron" => [0x0017D],
  "Zcy" => [0x00417],
  "Zdot" => [0x0017B],
  "ZeroWidthSpace" => [0x0200B],
  "Zeta" => [0x00396],
  "Zfr" => [0x02128],
  "Zopf" => [0x02124],
  "Zscr" => [0x1D4B5],
  "aacute" => [0x000E1],
  "abreve" => [0x00103],
  "ac" => [0x0223E],
  "acE" => [0x0223E, 0x00333],
  "acd" => [0x0223F],
  "acirc" => [0x000E2],
  "acute" => [0x000B4],
  "acy" => [0x00430],
  "aelig" => [0x000E6],
  "af" => [0x02061],
  "afr" => [0x1D51E],
  "agrave" => [0x000E0],
  "alefsym" => [0x02135],
  "aleph" => [0x02135],
  "alpha" => [0x003B1],
  "amacr" => [0x00101],
  "amalg" => [0x02A3F],
  "amp" => [0x00026],
  "and" => [0x02227],
  "andand" => [0x02A55],
  "andd" => [0x02A5C],
  "andslope" => [0x02A58],
  "andv" => [0x02A5A],
  "ang" => [0x02220],
  "ange" => [0x029A4],
  "angle" => [0x02220],
  "angmsd" => [0x02221],
  "angmsdaa" => [0x029A8],
  "angmsdab" => [0x029A9],
  "angmsdac" => [0x029AA],
  "angmsdad" => [0x029AB],
  "angmsdae" => [0x029AC],
  "angmsdaf" => [0x029AD],
  "angmsdag" => [0x029AE],
  "angmsdah" => [0x029AF],
  "angrt" => [0x0221F],
  "angrtvb" => [0x022BE],
  "angrtvbd" => [0x0299D],
  "angsph" => [0x02222],
  "angst" => [0x000C5],
  "angzarr" => [0x0237C],
  "aogon" => [0x00105],
  "aopf" => [0x1D552],
  "ap" => [0x02248],
  "apE" => [0x02A70],
  "apacir" => [0x02A6F],
  "ape" => [0x0224A],
  "apid" => [0x0224B],
  "apos" => [0x00027],
  "approx" => [0x02248],
  "approxeq" => [0x0224A],
  "aring" => [0x000E5],
  "ascr" => [0x1D4B6],
  "ast" => [0x0002A],
  "asymp" => [0x02248],
  "asympeq" => [0x0224D],
  "atilde" => [0x000E3],
  "auml" => [0x000E4],
  "awconint" => [0x02233],
  "awint" => [0x02A11],
  "bNot" => [0x02AED],
  "backcong" => [0x0224C],
  "backepsilon" => [0x003F6],
  "backprime" => [0x02035],
  "backsim" => [0x0223D],
  "backsimeq" => [0x022CD],
  "barvee" => [0x022BD],
  "barwed" => [0x02305],
  "barwedge" => [0x02305],
  "bbrk" => [0x023B5],
  "bbrktbrk" => [0x023B6],
  "bcong" => [0x0224C],
  "bcy" => [0x00431],
  "bdquo" => [0x0201E],
  "becaus" => [0x02235],
  "because" => [0x02235],
  "bemptyv" => [0x029B0],
  "bepsi" => [0x003F6],
  "bernou" => [0x0212C],
  "beta" => [0x003B2],
  "beth" => [0x02136],
  "between" => [0x0226C],
  "bfr" => [0x1D51F],
  "bigcap" => [0x022C2],
  "bigcirc" => [0x025EF],
  "bigcup" => [0x022C3],
  "bigodot" => [0x02A00],
  "bigoplus" => [0x02A01],
  "bigotimes" => [0x02A02],
  "bigsqcup" => [0x02A06],
  "bigstar" => [0x02605],
  "bigtriangledown" => [0x025BD],
  "bigtriangleup" => [0x025B3],
  "biguplus" => [0x02A04],
  "bigvee" => [0x022C1],
  "bigwedge" => [0x022C0],
  "bkarow" => [0x0290D],
  "blacklozenge" => [0x029EB],
  "blacksquare" => [0x025AA],
  "blacktriangle" => [0x025B4],
  "blacktriangledown" => [0x025BE],
  "blacktriangleleft" => [0x025C2],
  "blacktriangleright" => [0x025B8],
  "blank" => [0x02423],
  "blk12" => [0x02592],
  "blk14" => [0x02591],
  "blk34" => [0x02593],
  "block" => [0x02588],
  "bne" => [0x0003D, 0x020E5],
  "bnequiv" => [0x02261, 0x020E5],
  "bnot" => [0x02310],
  "bopf" => [0x1D553],
  "bot" => [0x022A5],
  "bottom" => [0x022A5],
  "bowtie" => [0x022C8],
  "boxDL" => [0x02557],
  "boxDR" => [0x02554],
  "boxDl" => [0x02556],
  "boxDr" => [0x02553],
  "boxH" => [0x02550],
  "boxHD" => [0x02566],
  "boxHU" => [0x02569],
  "boxHd" => [0x02564],
  "boxHu" => [0x02567],
  "boxUL" => [0x0255D],
  "boxUR" => [0x0255A],
  "boxUl" => [0x0255C],
  "boxUr" => [0x02559],
  "boxV" => [0x02551],
  "boxVH" => [0x0256C],
  "boxVL" => [0x02563],
  "boxVR" => [0x02560],
  "boxVh" => [0x0256B],
  "boxVl" => [0x02562],
  "boxVr" => [0x0255F],
  "boxbox" => [0x029C9],
  "boxdL" => [0x02555],
  "boxdR" => [0x02552],
  "boxdl" => [0x02510],
  "boxdr" => [0x0250C],
  "boxh" => [0x02500],
  "boxhD" => [0x02565],
  "boxhU" => [0x02568],
  "boxhd" => [0x0252C],
  "boxhu" => [0x02534],
  "boxminus" => [0x0229F],
  "boxplus" => [0x0229E],
  "boxtimes" => [0x022A0],
  "boxuL" => [0x0255B],
  "boxuR" => [0x02558],
  "boxul" => [0x02518],
  "boxur" => [0x02514],
  "boxv" => [0x02502],
  "boxvH" => [0x0256A],
  "boxvL" => [0x02561],
  "boxvR" => [0x0255E],
  "boxvh" => [0x0253C],
  "boxvl" => [0x02524],
  "boxvr" => [0x0251C],
  "bprime" => [0x02035],
  "breve" => [0x002D8],
  "brvbar" => [0x000A6],
  "bscr" => [0x1D4B7],
  "bsemi" => [0x0204F],
  "bsim" => [0x0223D],
  "bsime" => [0x022CD],
  "bsol" => [0x0005C],
  "bsolb" => [0x029C5],
  "bsolhsub" => [0x027C8],
  "bull" => [0x02022],
  "bullet" => [0x02022],
  "bump" => [0x0224E],
  "bumpE" => [0x02AAE],
  "bumpe" => [0x0224F],
  "bumpeq" => [0x0224F],
  "cacute" => [0x00107],
  "cap" => [0x02229],
  "capand" => [0x02A44],
  "capbrcup" => [0x02A49],
  "capcap" => [0x02A4B],
  "capcup" => [0x02A47],
  "capdot" => [0x02A40],
  "caps" => [0x02229, 0x0FE00],
  "caret" => [0x02041],
  "caron" => [0x002C7],
  "ccaps" => [0x02A4D],
  "ccaron" => [0x0010D],
  "ccedil" => [0x000E7],
  "ccirc" => [0x00109],
  "ccups" => [0x02A4C],
  "ccupssm" => [0x02A50],
  "cdot" => [0x0010B],
  "cedil" => [0x000B8],
  "cemptyv" => [0x029B2],
  "cent" => [0x000A2],
  "centerdot" => [0x000B7],
  "cfr" => [0x1D520],
  "chcy" => [0x00447],
  "check" => [0x02713],
  "checkmark" => [0x02713],
  "chi" => [0x003C7],
  "cir" => [0x025CB],
  "cirE" => [0x029C3],
  "circ" => [0x002C6],
  "circeq" => [0x02257],
  "circlearrowleft" => [0x021BA],
  "circlearrowright" => [0x021BB],
  "circledR" => [0x000AE],
  "circledS" => [0x024C8],
  "circledast" => [0x0229B],
  "circledcirc" => [0x0229A],
  "circleddash" => [0x0229D],
  "cire" => [0x02257],
  "cirfnint" => [0x02A10],
  "cirmid" => [0x02AEF],
  "cirscir" => [0x029C2],
  "clubs" => [0x02663],
  "clubsuit" => [0x02663],
  "colon" => [0x0003A],
  "colone" => [0x02254],
  "coloneq" => [0x02254],
  "comma" => [0x0002C],
  "commat" => [0x00040],
  "comp" => [0x02201],
  "compfn" => [0x02218],
  "complement" => [0x02201],
  "complexes" => [0x02102],
  "cong" => [0x02245],
  "congdot" => [0x02A6D],
  "conint" => [0x0222E],
  "copf" => [0x1D554],
  "coprod" => [0x02210],
  "copy" => [0x000A9],
  "copysr" => [0x02117],
  "crarr" => [0x021B5],
  "cross" => [0x02717],
  "cscr" => [0x1D4B8],
  "csub" => [0x02ACF],
  "csube" => [0x02AD1],
  "csup" => [0x02AD0],
  "csupe" => [0x02AD2],
  "ctdot" => [0x022EF],
  "cudarrl" => [0x02938],
  "cudarrr" => [0x02935],
  "cuepr" => [0x022DE],
  "cuesc" => [0x022DF],
  "cularr" => [0x021B6],
  "cularrp" => [0x0293D],
  "cup" => [0x0222A],
  "cupbrcap" => [0x02A48],
  "cupcap" => [0x02A46],
  "cupcup" => [0x02A4A],
  "cupdot" => [0x0228D],
  "cupor" => [0x02A45],
  "cups" => [0x0222A, 0x0FE00],
  "curarr" => [0x021B7],
  "curarrm" => [0x0293C],
  "curlyeqprec" => [0x022DE],
  "curlyeqsucc" => [0x022DF],
  "curlyvee" => [0x022CE],
  "curlywedge" => [0x022CF],
  "curren" => [0x000A4],
  "curvearrowleft" => [0x021B6],
  "curvearrowright" => [0x021B7],
  "cuvee" => [0x022CE],
  "cuwed" => [0x022CF],
  "cwconint" => [0x02232],
  "cwint" => [0x02231],
  "cylcty" => [0x0232D],
  "dArr" => [0x021D3],
  "dHar" => [0x02965],
  "dagger" => [0x02020],
  "daleth" => [0x02138],
  "darr" => [0x02193],
  "dash" => [0x02010],
  "dashv" => [0x022A3],
  "dbkarow" => [0x0290F],
  "dblac" => [0x002DD],
  "dcaron" => [0x0010F],
  "dcy" => [0x00434],
  "dd" => [0x02146],
  "ddagger" => [0x02021],
  "ddarr" => [0x021CA],
  "ddotseq" => [0x02A77],
  "deg" => [0x000B0],
  "delta" => [0x003B4],
  "demptyv" => [0x029B1],
  "dfisht" => [0x0297F],
  "dfr" => [0x1D521],
  "dharl" => [0x021C3],
  "dharr" => [0x021C2],
  "diam" => [0x022C4],
  "diamond" => [0x022C4],
  "diamondsuit" => [0x02666],
  "diams" => [0x02666],
  "die" => [0x000A8],
  "digamma" => [0x003DD],
  "disin" => [0x022F2],
  "div" => [0x000F7],
  "divide" => [0x000F7],
  "divideontimes" => [0x022C7],
  "divonx" => [0x022C7],
  "djcy" => [0x00452],
  "dlcorn" => [0x0231E],
  "dlcrop" => [0x0230D],
  "dollar" => [0x00024],
  "dopf" => [0x1D555],
  "dot" => [0x002D9],
  "doteq" => [0x02250],
  "doteqdot" => [0x02251],
  "dotminus" => [0x02238],
  "dotplus" => [0x02214],
  "dotsquare" => [0x022A1],
  "doublebarwedge" => [0x02306],
  "downarrow" => [0x02193],
  "downdownarrows" => [0x021CA],
  "downharpoonleft" => [0x021C3],
  "downharpoonright" => [0x021C2],
  "drbkarow" => [0x02910],
  "drcorn" => [0x0231F],
  "drcrop" => [0x0230C],
  "dscr" => [0x1D4B9],
  "dscy" => [0x00455],
  "dsol" => [0x029F6],
  "dstrok" => [0x00111],
  "dtdot" => [0x022F1],
  "dtri" => [0x025BF],
  "dtrif" => [0x025BE],
  "duarr" => [0x021F5],
  "duhar" => [0x0296F],
  "dwangle" => [0x029A6],
  "dzcy" => [0x0045F],
  "dzigrarr" => [0x027FF],
  "eDDot" => [0x02A77],
  "eDot" => [0x02251],
  "eacute" => [0x000E9],
  "easter" => [0x02A6E],
  "ecaron" => [0x0011B],
  "ecir" => [0x02256],
  "ecirc" => [0x000EA],
  "ecolon" => [0x02255],
  "ecy" => [0x0044D],
  "edot" => [0x00117],
  "ee" => [0x02147],
  "efDot" => [0x02252],
  "efr" => [0x1D522],
  "eg" => [0x02A9A],
  "egrave" => [0x000E8],
  "egs" => [0x02A96],
  "egsdot" => [0x02A98],
  "el" => [0x02A99],
  "elinters" => [0x023E7],
  "ell" => [0x02113],
  "els" => [0x02A95],
  "elsdot" => [0x02A97],
  "emacr" => [0x00113],
  "empty" => [0x02205],
  "emptyset" => [0x02205],
  "emptyv" => [0x02205],
  "emsp" => [0x02003],
  "emsp13" => [0x02004],
  "emsp14" => [0x02005],
  "eng" => [0x0014B],
  "ensp" => [0x02002],
  "eogon" => [0x00119],
  "eopf" => [0x1D556],
  "epar" => [0x022D5],
  "eparsl" => [0x029E3],
  "eplus" => [0x02A71],
  "epsi" => [0x003B5],
  "epsilon" => [0x003B5],
  "epsiv" => [0x003F5],
  "eqcirc" => [0x02256],
  "eqcolon" => [0x02255],
  "eqsim" => [0x02242],
  "eqslantgtr" => [0x02A96],
  "eqslantless" => [0x02A95],
  "equals" => [0x0003D],
  "equest" => [0x0225F],
  "equiv" => [0x02261],
  "equivDD" => [0x02A78],
  "eqvparsl" => [0x029E5],
  "erDot" => [0x02253],
  "erarr" => [0x02971],
  "escr" => [0x0212F],
  "esdot" => [0x02250],
  "esim" => [0x02242],
  "eta" => [0x003B7],
  "eth" => [0x000F0],
  "euml" => [0x000EB],
  "euro" => [0x020AC],
  "excl" => [0x00021],
  "exist" => [0x02203],
  "expectation" => [0x02130],
  "exponentiale" => [0x02147],
  "fallingdotseq" => [0x02252],
  "fcy" => [0x00444],
  "female" => [0x02640],
  "ffilig" => [0x0FB03],
  "fflig" => [0x0FB00],
  "ffllig" => [0x0FB04],
  "ffr" => [0x1D523],
  "filig" => [0x0FB01],
  "fjlig" => [0x00066, 0x0006A],
  "flat" => [0x0266D],
  "fllig" => [0x0FB02],
  "fltns" => [0x025B1],
  "fnof" => [0x00192],
  "fopf" => [0x1D557],
  "forall" => [0x02200],
  "fork" => [0x022D4],
  "forkv" => [0x02AD9],
  "fpartint" => [0x02A0D],
  "frac12" => [0x000BD],
  "frac13" => [0x02153],
  "frac14" => [0x000BC],
  "frac15" => [0x02155],
  "frac16" => [0x02159],
  "frac18" => [0x0215B],
  "frac23" => [0x02154],
  "frac25" => [0x02156],
  "frac34" => [0x000BE],
  "frac35" => [0x02157],
  "frac38" => [0x0215C],
  "frac45" => [0x02158],
  "frac56" => [0x0215A],
  "frac58" => [0x0215D],
  "frac78" => [0x0215E],
  "frasl" => [0x02044],
  "frown" => [0x02322],
  "fscr" => [0x1D4BB],
  "gE" => [0x02267],
  "gEl" => [0x02A8C],
  "gacute" => [0x001F5],
  "gamma" => [0x003B3],
  "gammad" => [0x003DD],
  "gap" => [0x02A86],
  "gbreve" => [0x0011F],
  "gcirc" => [0x0011D],
  "gcy" => [0x00433],
  "gdot" => [0x00121],
  "ge" => [0x02265],
  "gel" => [0x022DB],
  "geq" => [0x02265],
  "geqq" => [0x02267],
  "geqslant" => [0x02A7E],
  "ges" => [0x02A7E],
  "gescc" => [0x02AA9],
  "gesdot" => [0x02A80],
  "gesdoto" => [0x02A82],
  "gesdotol" => [0x02A84],
  "gesl" => [0x022DB, 0x0FE00],
  "gesles" => [0x02A94],
  "gfr" => [0x1D524],
  "gg" => [0x0226B],
  "ggg" => [0x022D9],
  "gimel" => [0x02137],
  "gjcy" => [0x00453],
  "gl" => [0x02277],
  "glE" => [0x02A92],
  "gla" => [0x02AA5],
  "glj" => [0x02AA4],
  "gnE" => [0x02269],
  "gnap" => [0x02A8A],
  "gnapprox" => [0x02A8A],
  "gne" => [0x02A88],
  "gneq" => [0x02A88],
  "gneqq" => [0x02269],
  "gnsim" => [0x022E7],
  "gopf" => [0x1D558],
  "grave" => [0x00060],
  "gscr" => [0x0210A],
  "gsim" => [0x02273],
  "gsime" => [0x02A8E],
  "gsiml" => [0x02A90],
  "gt" => [0x0003E],
  "gtcc" => [0x02AA7],
  "gtcir" => [0x02A7A],
  "gtdot" => [0x022D7],
  "gtlPar" => [0x02995],
  "gtquest" => [0x02A7C],
  "gtrapprox" => [0x02A86],
  "gtrarr" => [0x02978],
  "gtrdot" => [0x022D7],
  "gtreqless" => [0x022DB],
  "gtreqqless" => [0x02A8C],
  "gtrless" => [0x02277],
  "gtrsim" => [0x02273],
  "gvertneqq" => [0x02269, 0x0FE00],
  "gvnE" => [0x02269, 0x0FE00],
  "hArr" => [0x021D4],
  "hairsp" => [0x0200A],
  "half" => [0x000BD],
  "hamilt" => [0x0210B],
  "hardcy" => [0x0044A],
  "harr" => [0x02194],
  "harrcir" => [0x02948],
  "harrw" => [0x021AD],
  "hbar" => [0x0210F],
  "hcirc" => [0x00125],
  "hearts" => [0x02665],
  "heartsuit" => [0x02665],
  "hellip" => [0x02026],
  "hercon" => [0x022B9],
  "hfr" => [0x1D525],
  "hksearow" => [0x02925],
  "hkswarow" => [0x02926],
  "hoarr" => [0x021FF],
  "homtht" => [0x0223B],
  "hookleftarrow" => [0x021A9],
  "hookrightarrow" => [0x021AA],
  "hopf" => [0x1D559],
  "horbar" => [0x02015],
  "hscr" => [0x1D4BD],
  "hslash" => [0x0210F],
  "hstrok" => [0x00127],
  "hybull" => [0x02043],
  "hyphen" => [0x02010],
  "iacute" => [0x000ED],
  "ic" => [0x02063],
  "icirc" => [0x000EE],
  "icy" => [0x00438],
  "iecy" => [0x00435],
  "iexcl" => [0x000A1],
  "iff" => [0x021D4],
  "ifr" => [0x1D526],
  "igrave" => [0x000EC],
  "ii" => [0x02148],
  "iiiint" => [0x02A0C],
  "iiint" => [0x0222D],
  "iinfin" => [0x029DC],
  "iiota" => [0x02129],
  "ijlig" => [0x00133],
  "imacr" => [0x0012B],
  "image" => [0x02111],
  "imagline" => [0x02110],
  "imagpart" => [0x02111],
  "imath" => [0x00131],
  "imof" => [0x022B7],
  "imped" => [0x001B5],
  "in" => [0x02208],
  "incare" => [0x02105],
  "infin" => [0x0221E],
  "infintie" => [0x029DD],
  "inodot" => [0x00131],
  "int" => [0x0222B],
  "intcal" => [0x022BA],
  "integers" => [0x02124],
  "intercal" => [0x022BA],
  "intlarhk" => [0x02A17],
  "intprod" => [0x02A3C],
  "iocy" => [0x00451],
  "iogon" => [0x0012F],
  "iopf" => [0x1D55A],
  "iota" => [0x003B9],
  "iprod" => [0x02A3C],
  "iquest" => [0x000BF],
  "iscr" => [0x1D4BE],
  "isin" => [0x02208],
  "isinE" => [0x022F9],
  "isindot" => [0x022F5],
  "isins" => [0x022F4],
  "isinsv" => [0x022F3],
  "isinv" => [0x02208],
  "it" => [0x02062],
  "itilde" => [0x00129],
  "iukcy" => [0x00456],
  "iuml" => [0x000EF],
  "jcirc" => [0x00135],
  "jcy" => [0x00439],
  "jfr" => [0x1D527],
  "jmath" => [0x00237],
  "jopf" => [0x1D55B],
  "jscr" => [0x1D4BF],
  "jsercy" => [0x00458],
  "jukcy" => [0x00454],
  "kappa" => [0x003BA],
  "kappav" => [0x003F0],
  "kcedil" => [0x00137],
  "kcy" => [0x0043A],
  "kfr" => [0x1D528],
  "kgreen" => [0x00138],
  "khcy" => [0x00445],
  "kjcy" => [0x0045C],
  "kopf" => [0x1D55C],
  "kscr" => [0x1D4C0],
  "lAarr" => [0x021DA],
  "lArr" => [0x021D0],
  "lAtail" => [0x0291B],
  "lBarr" => [0x0290E],
  "lE" => [0x02266],
  "lEg" => [0x02A8B],
  "lHar" => [0x02962],
  "lacute" => [0x0013A],
  "laemptyv" => [0x029B4],
  "lagran" => [0x02112],
  "lambda" => [0x003BB],
  "lang" => [0x027E8],
  "langd" => [0x02991],
  "langle" => [0x027E8],
  "lap" => [0x02A85],
  "laquo" => [0x000AB],
  "larr" => [0x02190],
  "larrb" => [0x021E4],
  "larrbfs" => [0x0291F],
  "larrfs" => [0x0291D],
  "larrhk" => [0x021A9],
  "larrlp" => [0x021AB],
  "larrpl" => [0x02939],
  "larrsim" => [0x02973],
  "larrtl" => [0x021A2],
  "lat" => [0x02AAB],
  "latail" => [0x02919],
  "late" => [0x02AAD],
  "lates" => [0x02AAD, 0x0FE00],
  "lbarr" => [0x0290C],
  "lbbrk" => [0x02772],
  "lbrace" => [0x0007B],
  "lbrack" => [0x0005B],
  "lbrke" => [0x0298B],
  "lbrksld" => [0x0298F],
  "lbrkslu" => [0x0298D],
  "lcaron" => [0x0013E],
  "lcedil" => [0x0013C],
  "lceil" => [0x02308],
  "lcub" => [0x0007B],
  "lcy" => [0x0043B],
  "ldca" => [0x02936],
  "ldquo" => [0x0201C],
  "ldquor" => [0x0201E],
  "ldrdhar" => [0x02967],
  "ldrushar" => [0x0294B],
  "ldsh" => [0x021B2],
  "le" => [0x02264],
  "leftarrow" => [0x02190],
  "leftarrowtail" => [0x021A2],
  "leftharpoondown" => [0x021BD],
  "leftharpoonup" => [0x021BC],
  "leftleftarrows" => [0x021C7],
  "leftrightarrow" => [0x02194],
  "leftrightarrows" => [0x021C6],
  "leftrightharpoons" => [0x021CB],
  "leftrightsquigarrow" => [0x021AD],
  "leftthreetimes" => [0x022CB],
  "leg" => [0x022DA],
  "leq" => [0x02264],
  "leqq" => [0x02266],
  "leqslant" => [0x02A7D],
  "les" => [0x02A7D],
  "lescc" => [0x02AA8],
  "lesdot" => [0x02A7F],
  "lesdoto" => [0x02A81],
  "lesdotor" => [0x02A83],
  "lesg" => [0x022DA, 0x0FE00],
  "lesges" => [0x02A93],
  "lessapprox" => [0x02A85],
  "lessdot" => [0x022D6],
  "lesseqgtr" => [0x022DA],
  "lesseqqgtr" => [0x02A8B],
  "lessgtr" => [0x02276],
  "lesssim" => [0x02272],
  "lfisht" => [0x0297C],
  "lfloor" => [0x0230A],
  "lfr" => [0x1D529],
  "lg" => [0x02276],
  "lgE" => [0x02A91],
  "lhard" => [0x021BD],
  "lharu" => [0x021BC],
  "lharul" => [0x0296A],
  "lhblk" => [0x02584],
  "ljcy" => [0x00459],
  "ll" => [0x0226A],
  "llarr" => [0x021C7],
  "llcorner" => [0x0231E],
  "llhard" => [0x0296B],
  "lltri" => [0x025FA],
  "lmidot" => [0x00140],
  "lmoust" => [0x023B0],
  "lmoustache" => [0x023B0],
  "lnE" => [0x02268],
  "lnap" => [0x02A89],
  "lnapprox" => [0x02A89],
  "lne" => [0x02A87],
  "lneq" => [0x02A87],
  "lneqq" => [0x02268],
  "lnsim" => [0x022E6],
  "loang" => [0x027EC],
  "loarr" => [0x021FD],
  "lobrk" => [0x027E6],
  "longleftarrow" => [0x027F5],
  "longleftrightarrow" => [0x027F7],
  "longmapsto" => [0x027FC],
  "longrightarrow" => [0x027F6],
  "looparrowleft" => [0x021AB],
  "looparrowright" => [0x021AC],
  "lopar" => [0x02985],
  "lopf" => [0x1D55D],
  "loplus" => [0x02A2D],
  "lotimes" => [0x02A34],
  "lowast" => [0x02217],
  "lowbar" => [0x0005F],
  "loz" => [0x025CA],
  "lozenge" => [0x025CA],
  "lozf" => [0x029EB],
  "lpar" => [0x00028],
  "lparlt" => [0x02993],
  "lrarr" => [0x021C6],
  "lrcorner" => [0x0231F],
  "lrhar" => [0x021CB],
  "lrhard" => [0x0296D],
  "lrm" => [0x0200E],
  "lrtri" => [0x022BF],
  "lsaquo" => [0x02039],
  "lscr" => [0x1D4C1],
  "lsh" => [0x021B0],
  "lsim" => [0x02272],
  "lsime" => [0x02A8D],
  "lsimg" => [0x02A8F],
  "lsqb" => [0x0005B],
  "lsquo" => [0x02018],
  "lsquor" => [0x0201A],
  "lstrok" => [0x00142],
  "lt" => [0x0003C],
  "ltcc" => [0x02AA6],
  "ltcir" => [0x02A79],
  "ltdot" => [0x022D6],
  "lthree" => [0x022CB],
  "ltimes" => [0x022C9],
  "ltlarr" => [0x02976],
  "ltquest" => [0x02A7B],
  "ltrPar" => [0x02996],
  "ltri" => [0x025C3],
  "ltrie" => [0x022B4],
  "ltrif" => [0x025C2],
  "lurdshar" => [0x0294A],
  "luruhar" => [0x02966],
  "lvertneqq" => [0x02268, 0x0FE00],
  "lvnE" => [0x02268, 0x0FE00],
  "mDDot" => [0x0223A],
  "macr" => [0x000AF],
  "male" => [0x02642],
  "malt" => [0x02720],
  "maltese" => [0x02720],
  "map" => [0x021A6],
  "mapsto" => [0x021A6],
  "mapstodown" => [0x021A7],
  "mapstoleft" => [0x021A4],
  "mapstoup" => [0x021A5],
  "marker" => [0x025AE],
  "mcomma" => [0x02A29],
  "mcy" => [0x0043C],
  "mdash" => [0x02014],
  "measuredangle" => [0x02221],
  "mfr" => [0x1D52A],
  "mho" => [0x02127],
  "micro" => [0x000B5],
  "mid" => [0x02223],
  "midast" => [0x0002A],
  "midcir" => [0x02AF0],
  "middot" => [0x000B7],
  "minus" => [0x02212],
  "minusb" => [0x0229F],
  "minusd" => [0x02238],
  "minusdu" => [0x02A2A],
  "mlcp" => [0x02ADB],
  "mldr" => [0x02026],
  "mnplus" => [0x02213],
  "models" => [0x022A7],
  "mopf" => [0x1D55E],
  "mp" => [0x02213],
  "mscr" => [0x1D4C2],
  "mstpos" => [0x0223E],
  "mu" => [0x003BC],
  "multimap" => [0x022B8],
  "mumap" => [0x022B8],
  "nGg" => [0x022D9, 0x00338],
  "nGt" => [0x0226B, 0x020D2],
  "nGtv" => [0x0226B, 0x00338],
  "nLeftarrow" => [0x021CD],
  "nLeftrightarrow" => [0x021CE],
  "nLl" => [0x022D8, 0x00338],
  "nLt" => [0x0226A, 0x020D2],
  "nLtv" => [0x0226A, 0x00338],
  "nRightarrow" => [0x021CF],
  "nVDash" => [0x022AF],
  "nVdash" => [0x022AE],
  "nabla" => [0x02207],
  "nacute" => [0x00144],
  "nang" => [0x02220, 0x020D2],
  "nap" => [0x02249],
  "napE" => [0x02A70, 0x00338],
  "napid" => [0x0224B, 0x00338],
  "napos" => [0x00149],
  "napprox" => [0x02249],
  "natur" => [0x0266E],
  "natural" => [0x0266E],
  "naturals" => [0x02115],
  "nbsp" => [0x000A0],
  "nbump" => [0x0224E, 0x00338],
  "nbumpe" => [0x0224F, 0x00338],
  "ncap" => [0x02A43],
  "ncaron" => [0x00148],
  "ncedil" => [0x00146],
  "ncong" => [0x02247],
  "ncongdot" => [0x02A6D, 0x00338],
  "ncup" => [0x02A42],
  "ncy" => [0x0043D],
  "ndash" => [0x02013],
  "ne" => [0x02260],
  "neArr" => [0x021D7],
  "nearhk" => [0x02924],
  "nearr" => [0x02197],
  "nearrow" => [0x02197],
  "nedot" => [0x02250, 0x00338],
  "nequiv" => [0x02262],
  "nesear" => [0x02928],
  "nesim" => [0x02242, 0x00338],
  "nexist" => [0x02204],
  "nexists" => [0x02204],
  "nfr" => [0x1D52B],
  "ngE" => [0x02267, 0x00338],
  "nge" => [0x02271],
  "ngeq" => [0x02271],
  "ngeqq" => [0x02267, 0x00338],
  "ngeqslant" => [0x02A7E, 0x00338],
  "nges" => [0x02A7E, 0x00338],
  "ngsim" => [0x02275],
  "ngt" => [0x0226F],
  "ngtr" => [0x0226F],
  "nhArr" => [0x021CE],
  "nharr" => [0x021AE],
  "nhpar" => [0x02AF2],
  "ni" => [0x0220B],
  "nis" => [0x022FC],
  "nisd" => [0x022FA],
  "niv" => [0x0220B],
  "njcy" => [0x0045A],
  "nlArr" => [0x021CD],
  "nlE" => [0x02266, 0x00338],
  "nlarr" => [0x0219A],
  "nldr" => [0x02025],
  "nle" => [0x02270],
  "nleftarrow" => [0x0219A],
  "nleftrightarrow" => [0x021AE],
  "nleq" => [0x02270],
  "nleqq" => [0x02266, 0x00338],
  "nleqslant" => [0x02A7D, 0x00338],
  "nles" => [0x02A7D, 0x00338],
  "nless" => [0x0226E],
  "nlsim" => [0x02274],
  "nlt" => [0x0226E],
  "nltri" => [0x022EA],
  "nltrie" => [0x022EC],
  "nmid" => [0x02224],
  "nopf" => [0x1D55F],
  "not" => [0x000AC],
  "notin" => [0x02209],
  "notinE" => [0x022F9, 0x00338],
  "notindot" => [0x022F5, 0x00338],
  "notinva" => [0x02209],
  "notinvb" => [0x022F7],
  "notinvc" => [0x022F6],
  "notni" => [0x0220C],
  "notniva" => [0x0220C],
  "notnivb" => [0x022FE],
  "notnivc" => [0x022FD],
  "npar" => [0x02226],
  "nparallel" => [0x02226],
  "nparsl" => [0x02AFD, 0x020E5],
  "npart" => [0x02202, 0x00338],
  "npolint" => [0x02A14],
  "npr" => [0x02280],
  "nprcue" => [0x022E0],
  "npre" => [0x02AAF, 0x00338],
  "nprec" => [0x02280],
  "npreceq" => [0x02AAF, 0x00338],
  "nrArr" => [0x021CF],
  "nrarr" => [0x0219B],
  "nrarrc" => [0x02933, 0x00338],
  "nrarrw" => [0x0219D, 0x00338],
  "nrightarrow" => [0x0219B],
  "nrtri" => [0x022EB],
  "nrtrie" => [0x022ED],
  "nsc" => [0x02281],
  "nsccue" => [0x022E1],
  "nsce" => [0x02AB0, 0x00338],
  "nscr" => [0x1D4C3],
  "nshortmid" => [0x02224],
  "nshortparallel" => [0x02226],
  "nsim" => [0x02241],
  "nsime" => [0x02244],
  "nsimeq" => [0x02244],
  "nsmid" => [0x02224],
  "nspar" => [0x02226],
  "nsqsube" => [0x022E2],
  "nsqsupe" => [0x022E3],
  "nsub" => [0x02284],
  "nsubE" => [0x02AC5, 0x00338],
  "nsube" => [0x02288],
  "nsubset" => [0x02282, 0x020D2],
  "nsubseteq" => [0x02288],
  "nsubseteqq" => [0x02AC5, 0x00338],
  "nsucc" => [0x02281],
  "nsucceq" => [0x02AB0, 0x00338],
  "nsup" => [0x02285],
  "nsupE" => [0x02AC6, 0x00338],
  "nsupe" => [0x02289],
  "nsupset" => [0x02283, 0x020D2],
  "nsupseteq" => [0x02289],
  "nsupseteqq" => [0x02AC6, 0x00338],
  "ntgl" => [0x02279],
  "ntilde" => [0x000F1],
  "ntlg" => [0x02278],
  "ntriangleleft" => [0x022EA],
  "ntrianglelefteq" => [0x022EC],
  "ntriangleright" => [0x022EB],
  "ntrianglerighteq" => [0x022ED],
  "nu" => [0x003BD],
  "num" => [0x00023],
  "numero" => [0x02116],
  "numsp" => [0x02007],
  "nvDash" => [0x022AD],
  "nvHarr" => [0x02904],
  "nvap" => [0x0224D, 0x020D2],
  "nvdash" => [0x022AC],
  "nvge" => [0x02265, 0x020D2],
  "nvgt" => [0x0003E, 0x020D2],
  "nvinfin" => [0x029DE],
  "nvlArr" => [0x02902],
  "nvle" => [0x02264, 0x020D2],
  "nvlt" => [0x0003C, 0x020D2],
  "nvltrie" => [0x022B4, 0x020D2],
  "nvrArr" => [0x02903],
  "nvrtrie" => [0x022B5, 0x020D2],
  "nvsim" => [0x0223C, 0x020D2],
  "nwArr" => [0x021D6],
  "nwarhk" => [0x02923],
  "nwarr" => [0x02196],
  "nwarrow" => [0x02196],
  "nwnear" => [0x02927],
  "oS" => [0x024C8],
  "oacute" => [0x000F3],
  "oast" => [0x0229B],
  "ocir" => [0x0229A],
  "ocirc" => [0x000F4],
  "ocy" => [0x0043E],
  "odash" => [0x0229D],
  "odblac" => [0x00151],
  "odiv" => [0x02A38],
  "odot" => [0x02299],
  "odsold" => [0x029BC],
  "oelig" => [0x00153],
  "ofcir" => [0x029BF],
  "ofr" => [0x1D52C],
  "ogon" => [0x002DB],
  "ograve" => [0x000F2],
  "ogt" => [0x029C1],
  "ohbar" => [0x029B5],
  "ohm" => [0x003A9],
  "oint" => [0x0222E],
  "olarr" => [0x021BA],
  "olcir" => [0x029BE],
  "olcross" => [0x029BB],
  "oline" => [0x0203E],
  "olt" => [0x029C0],
  "omacr" => [0x0014D],
  "omega" => [0x003C9],
  "omicron" => [0x003BF],
  "omid" => [0x029B6],
  "ominus" => [0x02296],
  "oopf" => [0x1D560],
  "opar" => [0x029B7],
  "operp" => [0x029B9],
  "oplus" => [0x02295],
  "or" => [0x02228],
  "orarr" => [0x021BB],
  "ord" => [0x02A5D],
  "order" => [0x02134],
  "orderof" => [0x02134],
  "ordf" => [0x000AA],
  "ordm" => [0x000BA],
  "origof" => [0x022B6],
  "oror" => [0x02A56],
  "orslope" => [0x02A57],
  "orv" => [0x02A5B],
  "oscr" => [0x02134],
  "oslash" => [0x000F8],
  "osol" => [0x02298],
  "otilde" => [0x000F5],
  "otimes" => [0x02297],
  "otimesas" => [0x02A36],
  "ouml" => [0x000F6],
  "ovbar" => [0x0233D],
  "par" => [0x02225],
  "para" => [0x000B6],
  "parallel" => [0x02225],
  "parsim" => [0x02AF3],
  "parsl" => [0x02AFD],
  "part" => [0x02202],
  "pcy" => [0x0043F],
  "percnt" => [0x00025],
  "period" => [0x0002E],
  "permil" => [0x02030],
  "perp" => [0x022A5],
  "pertenk" => [0x02031],
  "pfr" => [0x1D52D],
  "phi" => [0x003C6],
  "phiv" => [0x003D5],
  "phmmat" => [0x02133],
  "phone" => [0x0260E],
  "pi" => [0x003C0],
  "pitchfork" => [0x022D4],
  "piv" => [0x003D6],
  "planck" => [0x0210F],
  "planckh" => [0x0210E],
  "plankv" => [0x0210F],
  "plus" => [0x0002B],
  "plusacir" => [0x02A23],
  "plusb" => [0x0229E],
  "pluscir" => [0x02A22],
  "plusdo" => [0x02214],
  "plusdu" => [0x02A25],
  "pluse" => [0x02A72],
  "plusmn" => [0x000B1],
  "plussim" => [0x02A26],
  "plustwo" => [0x02A27],
  "pm" => [0x000B1],
  "pointint" => [0x02A15],
  "popf" => [0x1D561],
  "pound" => [0x000A3],
  "pr" => [0x0227A],
  "prE" => [0x02AB3],
  "prap" => [0x02AB7],
  "prcue" => [0x0227C],
  "pre" => [0x02AAF],
  "prec" => [0x0227A],
  "precapprox" => [0x02AB7],
  "preccurlyeq" => [0x0227C],
  "preceq" => [0x02AAF],
  "precnapprox" => [0x02AB9],
  "precneqq" => [0x02AB5],
  "precnsim" => [0x022E8],
  "precsim" => [0x0227E],
  "prime" => [0x02032],
  "primes" => [0x02119],
  "prnE" => [0x02AB5],
  "prnap" => [0x02AB9],
  "prnsim" => [0x022E8],
  "prod" => [0x0220F],
  "profalar" => [0x0232E],
  "profline" => [0x02312],
  "profsurf" => [0x02313],
  "prop" => [0x0221D],
  "propto" => [0x0221D],
  "prsim" => [0x0227E],
  "prurel" => [0x022B0],
  "pscr" => [0x1D4C5],
  "psi" => [0x003C8],
  "puncsp" => [0x02008],
  "qfr" => [0x1D52E],
  "qint" => [0x02A0C],
  "qopf" => [0x1D562],
  "qprime" => [0x02057],
  "qscr" => [0x1D4C6],
  "quaternions" => [0x0210D],
  "quatint" => [0x02A16],
  "quest" => [0x0003F],
  "questeq" => [0x0225F],
  "quot" => [0x00022],
  "rAarr" => [0x021DB],
  "rArr" => [0x021D2],
  "rAtail" => [0x0291C],
  "rBarr" => [0x0290F],
  "rHar" => [0x02964],
  "race" => [0x0223D, 0x00331],
  "racute" => [0x00155],
  "radic" => [0x0221A],
  "raemptyv" => [0x029B3],
  "rang" => [0x027E9],
  "rangd" => [0x02992],
  "range" => [0x029A5],
  "rangle" => [0x027E9],
  "raquo" => [0x000BB],
  "rarr" => [0x02192],
  "rarrap" => [0x02975],
  "rarrb" => [0x021E5],
  "rarrbfs" => [0x02920],
  "rarrc" => [0x02933],
  "rarrfs" => [0x0291E],
  "rarrhk" => [0x021AA],
  "rarrlp" => [0x021AC],
  "rarrpl" => [0x02945],
  "rarrsim" => [0x02974],
  "rarrtl" => [0x021A3],
  "rarrw" => [0x0219D],
  "ratail" => [0x0291A],
  "ratio" => [0x02236],
  "rationals" => [0x0211A],
  "rbarr" => [0x0290D],
  "rbbrk" => [0x02773],
  "rbrace" => [0x0007D],
  "rbrack" => [0x0005D],
  "rbrke" => [0x0298C],
  "rbrksld" => [0x0298E],
  "rbrkslu" => [0x02990],
  "rcaron" => [0x00159],
  "rcedil" => [0x00157],
  "rceil" => [0x02309],
  "rcub" => [0x0007D],
  "rcy" => [0x00440],
  "rdca" => [0x02937],
  "rdldhar" => [0x02969],
  "rdquo" => [0x0201D],
  "rdquor" => [0x0201D],
  "rdsh" => [0x021B3],
  "real" => [0x0211C],
  "realine" => [0x0211B],
  "realpart" => [0x0211C],
  "reals" => [0x0211D],
  "rect" => [0x025AD],
  "reg" => [0x000AE],
  "rfisht" => [0x0297D],
  "rfloor" => [0x0230B],
  "rfr" => [0x1D52F],
  "rhard" => [0x021C1],
  "rharu" => [0x021C0],
  "rharul" => [0x0296C],
  "rho" => [0x003C1],
  "rhov" => [0x003F1],
  "rightarrow" => [0x02192],
  "rightarrowtail" => [0x021A3],
  "rightharpoondown" => [0x021C1],
  "rightharpoonup" => [0x021C0],
  "rightleftarrows" => [0x021C4],
  "rightleftharpoons" => [0x021CC],
  "rightrightarrows" => [0x021C9],
  "rightsquigarrow" => [0x0219D],
  "rightthreetimes" => [0x022CC],
  "ring" => [0x002DA],
  "risingdotseq" => [0x02253],
  "rlarr" => [0x021C4],
  "rlhar" => [0x021CC],
  "rlm" => [0x0200F],
  "rmoust" => [0x023B1],
  "rmoustache" => [0x023B1],
  "rnmid" => [0x02AEE],
  "roang" => [0x027ED],
  "roarr" => [0x021FE],
  "robrk" => [0x027E7],
  "ropar" => [0x02986],
  "ropf" => [0x1D563],
  "roplus" => [0x02A2E],
  "rotimes" => [0x02A35],
  "rpar" => [0x00029],
  "rpargt" => [0x02994],
  "rppolint" => [0x02A12],
  "rrarr" => [0x021C9],
  "rsaquo" => [0x0203A],
  "rscr" => [0x1D4C7],
  "rsh" => [0x021B1],
  "rsqb" => [0x0005D],
  "rsquo" => [0x02019],
  "rsquor" => [0x02019],
  "rthree" => [0x022CC],
  "rtimes" => [0x022CA],
  "rtri" => [0x025B9],
  "rtrie" => [0x022B5],
  "rtrif" => [0x025B8],
  "rtriltri" => [0x029CE],
  "ruluhar" => [0x02968],
  "rx" => [0x0211E],
  "sacute" => [0x0015B],
  "sbquo" => [0x0201A],
  "sc" => [0x0227B],
  "scE" => [0x02AB4],
  "scap" => [0x02AB8],
  "scaron" => [0x00161],
  "sccue" => [0x0227D],
  "sce" => [0x02AB0],
  "scedil" => [0x0015F],
  "scirc" => [0x0015D],
  "scnE" => [0x02AB6],
  "scnap" => [0x02ABA],
  "scnsim" => [0x022E9],
  "scpolint" => [0x02A13],
  "scsim" => [0x0227F],
  "scy" => [0x00441],
  "sdot" => [0x022C5],
  "sdotb" => [0x022A1],
  "sdote" => [0x02A66],
  "seArr" => [0x021D8],
  "searhk" => [0x02925],
  "searr" => [0x02198],
  "searrow" => [0x02198],
  "sect" => [0x000A7],
  "semi" => [0x0003B],
  "seswar" => [0x02929],
  "setminus" => [0x02216],
  "setmn" => [0x02216],
  "sext" => [0x02736],
  "sfr" => [0x1D530],
  "sfrown" => [0x02322],
  "sharp" => [0x0266F],
  "shchcy" => [0x00449],
  "shcy" => [0x00448],
  "shortmid" => [0x02223],
  "shortparallel" => [0x02225],
  "shy" => [0x000AD],
  "sigma" => [0x003C3],
  "sigmaf" => [0x003C2],
  "sigmav" => [0x003C2],
  "sim" => [0x0223C],
  "simdot" => [0x02A6A],
  "sime" => [0x02243],
  "simeq" => [0x02243],
  "simg" => [0x02A9E],
  "simgE" => [0x02AA0],
  "siml" => [0x02A9D],
  "simlE" => [0x02A9F],
  "simne" => [0x02246],
  "simplus" => [0x02A24],
  "simrarr" => [0x02972],
  "slarr" => [0x02190],
  "smallsetminus" => [0x02216],
  "smashp" => [0x02A33],
  "smeparsl" => [0x029E4],
  "smid" => [0x02223],
  "smile" => [0x02323],
  "smt" => [0x02AAA],
  "smte" => [0x02AAC],
  "smtes" => [0x02AAC, 0x0FE00],
  "softcy" => [0x0044C],
  "sol" => [0x0002F],
  "solb" => [0x029C4],
  "solbar" => [0x0233F],
  "sopf" => [0x1D564],
  "spades" => [0x02660],
  "spadesuit" => [0x02660],
  "spar" => [0x02225],
  "sqcap" => [0x02293],
  "sqcaps" => [0x02293, 0x0FE00],
  "sqcup" => [0x02294],
  "sqcups" => [0x02294, 0x0FE00],
  "sqsub" => [0x0228F],
  "sqsube" => [0x02291],
  "sqsubset" => [0x0228F],
  "sqsubseteq" => [0x02291],
  "sqsup" => [0x02290],
  "sqsupe" => [0x02292],
  "sqsupset" => [0x02290],
  "sqsupseteq" => [0x02292],
  "squ" => [0x025A1],
  "square" => [0x025A1],
  "squarf" => [0x025AA],
  "squf" => [0x025AA],
  "srarr" => [0x02192],
  "sscr" => [0x1D4C8],
  "ssetmn" => [0x02216],
  "ssmile" => [0x02323],
  "sstarf" => [0x022C6],
  "star" => [0x02606],
  "starf" => [0x02605],
  "straightepsilon" => [0x003F5],
  "straightphi" => [0x003D5],
  "strns" => [0x000AF],
  "sub" => [0x02282],
  "subE" => [0x02AC5],
  "subdot" => [0x02ABD],
  "sube" => [0x02286],
  "subedot" => [0x02AC3],
  "submult" => [0x02AC1],
  "subnE" => [0x02ACB],
  "subne" => [0x0228A],
  "subplus" => [0x02ABF],
  "subrarr" => [0x02979],
  "subset" => [0x02282],
  "subseteq" => [0x02286],
  "subseteqq" => [0x02AC5],
  "subsetneq" => [0x0228A],
  "subsetneqq" => [0x02ACB],
  "subsim" => [0x02AC7],
  "subsub" => [0x02AD5],
  "subsup" => [0x02AD3],
  "succ" => [0x0227B],
  "succapprox" => [0x02AB8],
  "succcurlyeq" => [0x0227D],
  "succeq" => [0x02AB0],
  "succnapprox" => [0x02ABA],
  "succneqq" => [0x02AB6],
  "succnsim" => [0x022E9],
  "succsim" => [0x0227F],
  "sum" => [0x02211],
  "sung" => [0x0266A],
  "sup" => [0x02283],
  "sup1" => [0x000B9],
  "sup2" => [0x000B2],
  "sup3" => [0x000B3],
  "supE" => [0x02AC6],
  "supdot" => [0x02ABE],
  "supdsub" => [0x02AD8],
  "supe" => [0x02287],
  "supedot" => [0x02AC4],
  "suphsol" => [0x027C9],
  "suphsub" => [0x02AD7],
  "suplarr" => [0x0297B],
  "supmult" => [0x02AC2],
  "supnE" => [0x02ACC],
  "supne" => [0x0228B],
  "supplus" => [0x02AC0],
  "supset" => [0x02283],
  "supseteq" => [0x02287],
  "supseteqq" => [0x02AC6],
  "supsetneq" => [0x0228B],
  "supsetneqq" => [0x02ACC],
  "supsim" => [0x02AC8],
  "supsub" => [0x02AD4],
  "supsup" => [0x02AD6],
  "swArr" => [0x021D9],
  "swarhk" => [0x02926],
  "swarr" => [0x02199],
  "swarrow" => [0x02199],
  "swnwar" => [0x0292A],
  "szlig" => [0x000DF],
  "target" => [0x02316],
  "tau" => [0x003C4],
  "tbrk" => [0x023B4],
  "tcaron" => [0x00165],
  "tcedil" => [0x00163],
  "tcy" => [0x00442],
  "tdot" => [0x020DB],
  "telrec" => [0x02315],
  "tfr" => [0x1D531],
  "there4" => [0x02234],
  "therefore" => [0x02234],
  "theta" => [0x003B8],
  "thetasym" => [0x003D1],
  "thetav" => [0x003D1],
  "thickapprox" => [0x02248],
  "thicksim" => [0x0223C],
  "thinsp" => [0x02009],
  "thkap" => [0x02248],
  "thksim" => [0x0223C],
  "thorn" => [0x000FE],
  "tilde" => [0x002DC],
  "times" => [0x000D7],
  "timesb" => [0x022A0],
  "timesbar" => [0x02A31],
  "timesd" => [0x02A30],
  "tint" => [0x0222D],
  "toea" => [0x02928],
  "top" => [0x022A4],
  "topbot" => [0x02336],
  "topcir" => [0x02AF1],
  "topf" => [0x1D565],
  "topfork" => [0x02ADA],
  "tosa" => [0x02929],
  "tprime" => [0x02034],
  "trade" => [0x02122],
  "triangle" => [0x025B5],
  "triangledown" => [0x025BF],
  "triangleleft" => [0x025C3],
  "trianglelefteq" => [0x022B4],
  "triangleq" => [0x0225C],
  "triangleright" => [0x025B9],
  "trianglerighteq" => [0x022B5],
  "tridot" => [0x025EC],
  "trie" => [0x0225C],
  "triminus" => [0x02A3A],
  "triplus" => [0x02A39],
  "trisb" => [0x029CD],
  "tritime" => [0x02A3B],
  "trpezium" => [0x023E2],
  "tscr" => [0x1D4C9],
  "tscy" => [0x00446],
  "tshcy" => [0x0045B],
  "tstrok" => [0x00167],
  "twixt" => [0x0226C],
  "twoheadleftarrow" => [0x0219E],
  "twoheadrightarrow" => [0x021A0],
  "uArr" => [0x021D1],
  "uHar" => [0x02963],
  "uacute" => [0x000FA],
  "uarr" => [0x02191],
  "ubrcy" => [0x0045E],
  "ubreve" => [0x0016D],
  "ucirc" => [0x000FB],
  "ucy" => [0x00443],
  "udarr" => [0x021C5],
  "udblac" => [0x00171],
  "udhar" => [0x0296E],
  "ufisht" => [0x0297E],
  "ufr" => [0x1D532],
  "ugrave" => [0x000F9],
  "uharl" => [0x021BF],
  "uharr" => [0x021BE],
  "uhblk" => [0x02580],
  "ulcorn" => [0x0231C],
  "ulcorner" => [0x0231C],
  "ulcrop" => [0x0230F],
  "ultri" => [0x025F8],
  "umacr" => [0x0016B],
  "uml" => [0x000A8],
  "uogon" => [0x00173],
  "uopf" => [0x1D566],
  "uparrow" => [0x02191],
  "updownarrow" => [0x02195],
  "upharpoonleft" => [0x021BF],
  "upharpoonright" => [0x021BE],
  "uplus" => [0x0228E],
  "upsi" => [0x003C5],
  "upsih" => [0x003D2],
  "upsilon" => [0x003C5],
  "upuparrows" => [0x021C8],
  "urcorn" => [0x0231D],
  "urcorner" => [0x0231D],
  "urcrop" => [0x0230E],
  "uring" => [0x0016F],
  "urtri" => [0x025F9],
  "uscr" => [0x1D4CA],
  "utdot" => [0x022F0],
  "utilde" => [0x00169],
  "utri" => [0x025B5],
  "utrif" => [0x025B4],
  "uuarr" => [0x021C8],
  "uuml" => [0x000FC],
  "uwangle" => [0x029A7],
  "vArr" => [0x021D5],
  "vBar" => [0x02AE8],
  "vBarv" => [0x02AE9],
  "vDash" => [0x022A8],
  "vangrt" => [0x0299C],
  "varepsilon" => [0x003F5],
  "varkappa" => [0x003F0],
  "varnothing" => [0x02205],
  "varphi" => [0x003D5],
  "varpi" => [0x003D6],
  "varpropto" => [0x0221D],
  "varr" => [0x02195],
  "varrho" => [0x003F1],
  "varsigma" => [0x003C2],
  "varsubsetneq" => [0x0228A, 0x0FE00],
  "varsubsetneqq" => [0x02ACB, 0x0FE00],
  "varsupsetneq" => [0x0228B, 0x0FE00],
  "varsupsetneqq" => [0x02ACC, 0x0F                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               gems/gems/rdoc-6.5.1.1/lib/rdoc/markdown/literals.rb                                                0000644                 00000024367 15040313414 0015657 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       xdtri" => [0x025BD],
  "xfr" => [0x1D535],
  "xhArr" => [0x027FA],
  "xharr" => [0x027F7],
  "xi" => [0x003BE],
  "xlArr" => [0x027F8],
  "xlarr" => [0x027F5],
  "xmap" => [0x027FC],
  "xnis" => [0x022FB],
  "xodot" => [0x02A00],
  "xopf" => [0x1D569],
  "xoplus" => [0x02A01],
  "xotime" => [0x02A02],
  "xrArr" => [0x027F9],
  "xrarr" => [0x027F6],
  "xscr" => [0x1D4CD],
  "xsqcup" => [0x02A06],
  "xuplus" => [0x02A04],
  "xutri" => [0x025B3],
  "xvee" => [0x022C1],
  "xwedge" => [0x022C0],
  "yacute" => [0x000FD],
  "yacy" => [0x0044F],
  "ycirc" => [0x00177],
  "ycy" => [0x0044B],
  "yen" => [0x000A5],
  "yfr" => [0x1D536],
  "yicy" => [0x00457],
  "yopf" => [0x1D56A],
  "yscr" => [0x1D4CE],
  "yucy" => [0x0044E],
  "yuml" => [0x000FF],
  "zacute" => [0x0017A],
  "zcaron" => [0x0017E],
  "zcy" => [0x00437],
  "zdot" => [0x0017C],
  "zeetrf" => [0x02128],
  "zeta" => [0x003B6],
  "zfr" => [0x1D537],
  "zhcy" => [0x00436],
  "zigrarr" => [0x021DD],
  "zopf" => [0x1D56B],
  "zscr" => [0x1D4CF],
  "zwj" => [0x0200D],
  "zwnj" => [0x0200C],
}

                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               gems/gems/rdoc-6.5.1.1/lib/rdoc/markdown/literals.rb                                                0000644                 00000024367 15040313414 0015657 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # coding: UTF-8
# frozen_string_literal: true
# :markup: markdown

##
#--
# This set of literals is for Ruby 1.9 regular expressions and gives full
# unicode support.
#
# Unlike peg-markdown, this set of literals recognizes Unicode alphanumeric
# characters, newlines and spaces.
class RDoc::Markdown::Literals
  # :stopdoc:

    # This is distinct from setup_parser so that a standalone parser
    # can redefine #initialize and still have access to the proper
    # parser setup code.
    def initialize(str, debug=false)
      setup_parser(str, debug)
    end



    # Prepares for parsing +str+.  If you define a custom initialize you must
    # call this method before #parse
    def setup_parser(str, debug=false)
      set_string str, 0
      @memoizations = Hash.new { |h,k| h[k] = {} }
      @result = nil
      @failed_rule = nil
      @failing_rule_offset = -1
      @line_offsets = nil

      setup_foreign_grammar
    end

    attr_reader :string
    attr_reader :failing_rule_offset
    attr_accessor :result, :pos

    def current_column(target=pos)
      if string[target] == "\n" && (c = string.rindex("\n", target-1) || -1)
        return target - c
      elsif c = string.rindex("\n", target)
        return target - c
      end

      target + 1
    end

    def position_line_offsets
      unless @position_line_offsets
        @position_line_offsets = []
        total = 0
        string.each_line do |line|
          total += line.size
          @position_line_offsets << total
        end
      end
      @position_line_offsets
    end

    if [].respond_to? :bsearch_index
      def current_line(target=pos)
        if line = position_line_offsets.bsearch_index {|x| x > target }
          return line + 1
        end
        raise "Target position #{target} is outside of string"
      end
    else
      def current_line(target=pos)
        if line = position_line_offsets.index {|x| x > target }
          return line + 1
        end

        raise "Target position #{target} is outside of string"
      end
    end

    def current_character(target=pos)
      if target < 0 || target >= string.size
        raise "Target position #{target} is outside of string"
      end
      string[target, 1]
    end

    KpegPosInfo = Struct.new(:pos, :lno, :col, :line, :char)

    def current_pos_info(target=pos)
      l = current_line target
      c = current_column target
      ln = get_line(l-1)
      chr = string[target,1]
      KpegPosInfo.new(target, l, c, ln, chr)
    end

    def lines
      string.lines
    end

    def get_line(no)
      loff = position_line_offsets
      if no < 0
        raise "Line No is out of range: #{no} < 0"
      elsif no >= loff.size
        raise "Line No is out of range: #{no} >= #{loff.size}"
      end
      lend = loff[no]-1
      lstart = no > 0 ? loff[no-1] : 0
      string[lstart..lend]
    end



    def get_text(start)
      @string[start..@pos-1]
    end

    # Sets the string and current parsing position for the parser.
    def set_string string, pos
      @string = string
      @string_size = string ? string.size : 0
      @pos = pos
      @position_line_offsets = nil
    end

    def show_pos
      width = 10
      if @pos < width
        "#{@pos} (\"#{@string[0,@pos]}\" @ \"#{@string[@pos,width]}\")"
      else
        "#{@pos} (\"... #{@string[@pos - width, width]}\" @ \"#{@string[@pos,width]}\")"
      end
    end

    def failure_info
      l = current_line @failing_rule_offset
      c = current_column @failing_rule_offset

      if @failed_rule.kind_of? Symbol
        info = self.class::Rules[@failed_rule]
        "line #{l}, column #{c}: failed rule '#{info.name}' = '#{info.rendered}'"
      else
        "line #{l}, column #{c}: failed rule '#{@failed_rule}'"
      end
    end

    def failure_caret
      p = current_pos_info @failing_rule_offset
      "#{p.line.chomp}\n#{' ' * (p.col - 1)}^"
    end

    def failure_character
      current_character @failing_rule_offset
    end

    def failure_oneline
      p = current_pos_info @failing_rule_offset

      if @failed_rule.kind_of? Symbol
        info = self.class::Rules[@failed_rule]
        "@#{p.lno}:#{p.col} failed rule '#{info.name}', got '#{p.char}'"
      else
        "@#{p.lno}:#{p.col} failed rule '#{@failed_rule}', got '#{p.char}'"
      end
    end

    class ParseError < RuntimeError
    end

    def raise_error
      raise ParseError, failure_oneline
    end

    def show_error(io=STDOUT)
      error_pos = @failing_rule_offset
      p = current_pos_info(error_pos)

      io.puts "On line #{p.lno}, column #{p.col}:"

      if @failed_rule.kind_of? Symbol
        info = self.class::Rules[@failed_rule]
        io.puts "Failed to match '#{info.rendered}' (rule '#{info.name}')"
      else
        io.puts "Failed to match rule '#{@failed_rule}'"
      end

      io.puts "Got: #{p.char.inspect}"
      io.puts "=> #{p.line}"
      io.print(" " * (p.col + 2))
      io.puts "^"
    end

    def set_failed_rule(name)
      if @pos > @failing_rule_offset
        @failed_rule = name
        @failing_rule_offset = @pos
      end
    end

    attr_reader :failed_rule

    def match_string(str)
      len = str.size
      if @string[pos,len] == str
        @pos += len
        return str
      end

      return nil
    end

    def scan(reg)
      if m = reg.match(@string, @pos)
        @pos = m.end(0)
        return true
      end

      return nil
    end

    if "".respond_to? :ord
      def get_byte
        if @pos >= @string_size
          return nil
        end

        s = @string[@pos].ord
        @pos += 1
        s
      end
    else
      def get_byte
        if @pos >= @string_size
          return nil
        end

        s = @string[@pos]
        @pos += 1
        s
      end
    end

    def parse(rule=nil)
      # We invoke the rules indirectly via apply
      # instead of by just calling them as methods because
      # if the rules use left recursion, apply needs to
      # manage that.

      if !rule
        apply(:_root)
      else
        method = rule.gsub("-","_hyphen_")
        apply :"_#{method}"
      end
    end

    class MemoEntry
      def initialize(ans, pos)
        @ans = ans
        @pos = pos
        @result = nil
        @set = false
        @left_rec = false
      end

      attr_reader :ans, :pos, :result, :set
      attr_accessor :left_rec

      def move!(ans, pos, result)
        @ans = ans
        @pos = pos
        @result = result
        @set = true
        @left_rec = false
      end
    end

    def external_invoke(other, rule, *args)
      old_pos = @pos
      old_string = @string

      set_string other.string, other.pos

      begin
        if val = __send__(rule, *args)
          other.pos = @pos
          other.result = @result
        else
          other.set_failed_rule "#{self.class}##{rule}"
        end
        val
      ensure
        set_string old_string, old_pos
      end
    end

    def apply_with_args(rule, *args)
      @result = nil
      memo_key = [rule, args]
      if m = @memoizations[memo_key][@pos]
        @pos = m.pos
        if !m.set
          m.left_rec = true
          return nil
        end

        @result = m.result

        return m.ans
      else
        m = MemoEntry.new(nil, @pos)
        @memoizations[memo_key][@pos] = m
        start_pos = @pos

        ans = __send__ rule, *args

        lr = m.left_rec

        m.move! ans, @pos, @result

        # Don't bother trying to grow the left recursion
        # if it's failing straight away (thus there is no seed)
        if ans and lr
          return grow_lr(rule, args, start_pos, m)
        else
          return ans
        end
      end
    end

    def apply(rule)
      @result = nil
      if m = @memoizations[rule][@pos]
        @pos = m.pos
        if !m.set
          m.left_rec = true
          return nil
        end

        @result = m.result

        return m.ans
      else
        m = MemoEntry.new(nil, @pos)
        @memoizations[rule][@pos] = m
        start_pos = @pos

        ans = __send__ rule

        lr = m.left_rec

        m.move! ans, @pos, @result

        # Don't bother trying to grow the left recursion
        # if it's failing straight away (thus there is no seed)
        if ans and lr
          return grow_lr(rule, nil, start_pos, m)
        else
          return ans
        end
      end
    end

    def grow_lr(rule, args, start_pos, m)
      w                                                                                                                                                                                                                                                                         gems/gems/rdoc-6.5.1.1/lib/rdoc/alias.rb                                                            0000644                 00000004175 15040313414 0013302 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       meric unless _tmp
    return _tmp
  end

  # AlphanumericAscii = /[A-Za-z0-9]/
  def _AlphanumericAscii
    _tmp = scan(/\G(?-mix:[A-Za-z0-9])/)
    set_failed_rule :_AlphanumericAscii unless _tmp
    return _tmp
  end

  # BOM = "uFEFF"
  def _BOM
    _tmp = match_string("uFEFF")
    set_failed_rule :_BOM unless _tmp
    return _tmp
  end

  # Newline = /\n|\r\n?|\p{Zl}|\p{Zp}/
  def _Newline
    _tmp = scan(/\G(?-mix:\n|\r\n?|\p{Zl}|\p{Zp})/)
    set_failed_rule :_Newline unless _tmp
    return _tmp
  end

  # NonAlphanumeric = /\p{^Word}/
  def _NonAlphanumeric
    _tmp = scan(/\G(?-mix:\p{^Word})/)
    set_failed_rule :_NonAlphanumeric unless _tmp
    return _tmp
  end

  # Spacechar = /\t|\p{Zs}/
  def _Spacechar
    _tmp = scan(/\G(?-mix:\t|\p{Zs})/)
    set_failed_rule :_Spacechar unless _tmp
    return _tmp
  end

  Rules = {}
  Rules[:_Alphanumeric] = rule_info("Alphanumeric", "/\\p{Word}/")
  Rules[:_AlphanumericAscii] = rule_info("AlphanumericAscii", "/[A-Za-z0-9]/")
  Rules[:_BOM] = rule_info("BOM", "\"uFEFF\"")
  Rules[:_Newline] = rule_info("Newline", "/\\n|\\r\\n?|\\p{Zl}|\\p{Zp}/")
  Rules[:_NonAlphanumeric] = rule_info("NonAlphanumeric", "/\\p{^Word}/")
  Rules[:_Spacechar] = rule_info("Spacechar", "/\\t|\\p{Zs}/")
  # :startdoc:
end
                                                                                                                                                                                                                                                                         gems/gems/rdoc-6.5.1.1/lib/rdoc/alias.rb                                                            0000644                 00000004175 15040313414 0013302 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Represent an alias, which is an old_name/new_name pair associated with a
# particular cont                                                                                                                                                                                                                                                                                                                                                                                                   gems/gems/rdoc-6.5.1.1/lib/rdoc/meta_method.rb                                                      0000644                 00000000201 15040313414 0014461 0                                                                                                    ustar 00                                                                                                                                                                                                                                                        : 1, other.new_name]
  end

  ##
  # HTML fragment reference for this alias

  def aref
    type = singleton ? 'c' : 'i'
    "#a                                                                                                                                                                                                                                                                                                                                                                                               gems/gems/rdoc-6.5.1.1/lib/rdoc/normal_class.rb                                                     0000644                 00000004206 15040313414 0014661 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ame pretty_new_name

  def to_s # :nodoc:
    "alias: #{self.new_name} -> #{self.pretty_old_name} in: #{parent}"
  end

end

                                                                                                                                                                                                                                                                                                                                                                                                   gems/gems/rdoc-6.5.1.1/lib/rdoc/meta_method.rb                                                      0000644                 00000000201 15040313414 0014461 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# MetaMethod represents a meta-programmed method

class RDoc::MetaMethod < RDoc::AnyMethod
end

                                                                                                                                                                                                                                                                                                                                                                                               gems/gems/rdoc-6.5.1.1/lib/rdoc/normal_class.rb                                                     0000644                 00000004206 15040313414 0014661 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# A normal class, neither singleton nor anonymous

class RDoc::NormalClass < RDoc::ClassModule

  ##
                                                                                                                                                                                                                                                                                                                                                                                          gems/gems/rdoc-6.5.1.1/lib/rdoc/rubygems_hook.rb                                                    0000644                 00000012325 15040313414 0015062 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       erclass, @includes, @extends, @attributes, @method_list, @aliases
    ]
  end

  def to_s # :nodoc:
    display = "#{self.class.name} #{self.full_name}"
    if superclass
      display += ' < ' + (superclass.is_a?(String) ? superclass : superclass.full_name)
    end
    display += ' -> ' + is_alias_for.to_s if is_alias_for
    display
  end

  def pretty_print q # :nodoc:
    superclass = @superclass ? " < #{@superclass}" : nil

    q.group 2, "[class #{full_name}#{superclass}", "]" do
      q.breakable
      q.text "includes:"
      q.breakable
      q.seplist @includes do |inc| q.pp inc end

      q.breakable
      q.text "constants:"
      q.breakable
      q.seplist @constants do |const| q.pp const end

      q.breakable
      q.text "attributes:"
      q.breakable
      q.seplist @attributes do |attr| q.pp attr end

      q.breakable
      q.text "methods:"
      q.breakable
      q.seplist @method_list do |meth| q.pp meth end

      q.breakable
      q.text "aliases:"
      q.breakable
      q.seplist @aliases do |aliaz| q.pp aliaz end

      q.breakable
      q.text "comment:"
      q.breakable
      q.pp comment
    end
  end

end

                                                                                                                                                                                                                                                                                                                                                                                          gems/gems/rdoc-6.5.1.1/lib/rdoc/rubygems_hook.rb                                                    0000644                 00000012325 15040313414 0015062 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require 'rubygems/user_interaction'
require 'fileutils'
require_relative '../rdoc'

##
# Gem::RDoc provides methods to generate RDoc and ri data for installed gems
# upon gem installation.
#
# This file is automatically required by RubyGems 1.9 and newer.

class RDoc::RubygemsHook

  include Gem::UserInteraction
  extend  Gem::UserInteraction

  @rdoc_version = nil
  @specs = []

  ##
  # Force installation of documentation?

  attr_accessor :force

  ##
  # Generate rdoc?

  attr_accessor :generate_rdoc

  ##
  # Generate ri data?

  attr_accessor :generate_ri

  class << self

    ##
    # Loaded version of RDoc.  Set by ::load_rdoc

    attr_reader :rdoc_version

  end

  ##
  # Post installs hook that generates documentation for each specification in
  # +specs+

  def self.generation_hook installer, specs
    start = Time.now
    types = installer.document

    generate_rdoc = types.include? 'rdoc'
    generate_ri   = types.include? 'ri'

    specs.each do |spec|
      new(spec, generate_rdoc, generate_ri).generate
    end

    return unless generate_rdoc or generate_ri

    duration = (Time.now - start).to_i
    names    = specs.map(&:name).join ', '

    say "Done installing documentation for #{names} after #{duration} seconds"
  end

  ##
  # Loads the RDoc generator

  def self.load_rdoc
    return if @rdoc_version

    require_relative 'rdoc'

    @rdoc_version = Gem::Version.new ::RDoc::VERSION
  end

  ##
  # Creates a new documentation generator for +spec+.  RDoc and ri data
  # generation can be enabled or disabled through +generate_rdoc+ and
  # +generate_ri+ respectively.
  #
  # Only +generate_ri+ is enabled by default.

  def initialize spec, generate_rdoc = false, generate_ri = true
    @doc_dir   = spec.doc_dir
    @force     = false
    @rdoc      = nil
    @spec      = spec

    @generate_rdoc = generate_rdoc
    @generate_ri   = generate_ri

    @rdoc_dir = spec.doc_dir 'rdoc'
    @ri_dir   = spec.doc_dir 'ri'
  end

  ##
  # Removes legacy rdoc arguments from +args+
  #--
  # TODO move to RDoc::Options

  def delete_legacy_args args
    args.delete '--inline-source'
    args.delete '--promiscuous'
    args.delete '-p'
    args.delete '--one-file'
  end

  ##
  # Generates documentation using the named +generator+ ("darkfish" or "ri")
  # and following the given +options+.
  #
  # Documentation will be generated into +destination+

  def document generator, options, destination
    generator_name = generator

    options = options.dup
    options.exclude ||= [] # TODO maybe move to RDoc::Options#finish
    options.setup_generator generator
    options.op_dir = destination
    Dir.chdir @spec.full_gem_path do
      options.finish
    end

    generator = options.generator.new @rdoc.store, options

    @rdoc.options = options
    @rdoc.generator = generator

    say "Installing #{generator_name} documentation for #{@spec.full_name}"

    FileUtils.mkdir_p options.op_dir

    Dir.chdir options.op_dir do
      begin
        @rdoc.class.current = @rdoc
        @rdoc.generator.generate
      ensure
        @rdoc.class.current = nil
      end
    end
  end

  ##
  # Generates RDoc and ri data

  def generate
    return if @spec.default_gem?
    return unless @generate_ri or @gener                                                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/rdoc.rb                                                             0000644                 00000033272 15040313414 0013140 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       
    @rdoc.store = store

    say "Parsing documentation for #{@spec.full_name}"

    Dir.chdir @spec.full_gem_path do
      @rdoc.parse_files options.files
    end

    document 'ri',       options, @ri_dir if
      @generate_ri   and (@force or not File.exist? @ri_dir)

    document 'darkfish', options, @rdoc_dir if
      @generate_rdoc and (@force or not File.exist? @rdoc_dir)
  end

  ##
  # #new_rdoc creates a new RDoc instance.  This method is provided only to
  # make testing easier.

  def new_rdoc # :nodoc:
    ::RDoc::RDoc.new
  end

  ##
  # Is rdoc documentation installed?

  def rdoc_installed?
    File.exist? @rdoc_dir
  end

  ##
  # Removes generated RDoc and ri data

  def remove
    base_dir = @spec.base_dir

    raise Gem::FilePermissionError, base_dir unless File.writable? base_dir

    FileUtils.rm_rf @rdoc_dir
    FileUtils.rm_rf @ri_dir
  end

  ##
  # Is ri data installed?

  def ri_installed?
    File.exist? @ri_dir
  end

  ##
  # Prepares the spec for documentation generation

  def setup
    self.class.load_rdoc

    raise Gem::FilePermissionError, @doc_dir if
      File.exist?(@doc_dir) and not File.writable?(@doc_dir)

    FileUtils.mkdir_p @doc_dir unless File.exist? @doc_dir
  end

end
                                                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/rdoc.rb                                                             0000644                 00000033272 15040313414 0013140 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require_relative '../rdoc'

require 'find'
require 'fileutils'
require 'pathname'
require 'time'

##
# This is the driver for generating RDoc output.  It handles file parsing and
# generation of output.
#
# To use this class to generate RDoc output via the API, the recommended way
# is:
#
#   rdoc = RDoc::RDoc.new
#   options = RDoc::Options.load_options # returns an RDoc::Options instance
#   # set extra options
#   rdoc.document options
#
# You can also generate output like the +rdoc+ executable:
#
#   rdoc = RDoc::RDoc.new
#   rdoc.document argv
#
# Where +argv+ is an array of strings, each corresponding to an argument you'd
# give rdoc on the command line.  See <tt>rdoc --help</tt> for details.

class RDoc::RDoc

  @current = nil

  ##
  # This is the list of supported output generators

  GENERATORS = {}

  ##
  # List of directory names always skipped

  UNCONDITIONALLY_SKIPPED_DIRECTORIES = %w[CVS .svn .git].freeze

  ##
  # List of directory names skipped if test suites should be skipped

  TEST_SUITE_DIRECTORY_NAMES = %w[spec test].freeze


  ##
  # Generator instance used for creating output

  attr_accessor :generator

  ##
  # Hash of files and their last modified times.

  attr_reader :last_modified

  ##
  # RDoc options

  attr_accessor :options

  ##
  # Accessor for statistics.  Available after each call to parse_files

  attr_reader :stats

  ##
  # The current documentation store

  attr_reader :store

  ##
  # Add +klass+ that can generate output after parsing

  def self.add_generator(klass)
    name = klass.name.sub(/^RDoc::Generator::/, '').downcase
    GENERATORS[name] = klass
  end

  ##
  # Active RDoc::RDoc instance

  def self.current
    @current
  end

  ##
  # Sets the active RDoc::RDoc instance

  def self.current= rdoc
    @current = rdoc
  end

  ##
  # Creates a new RDoc::RDoc instance.  Call #document to parse files and
  # generate documentation.

  def initialize
    @current       = nil
    @generator     = nil
    @last_modified = {}
    @old_siginfo   = nil
    @options       = nil
    @stats         = nil
    @store         = nil
  end

  ##
  # Report an error message and exit

  def error(msg)
    raise RDoc::Error, msg
  end

  ##
  # Gathers a set of parseable files from the files and directories listed in
  # +files+.

  def gather_files files
    files = ["."] if files.empty?

    file_list = normalized_file_list files, true, @options.exclude

    file_list = remove_unparseable(file_list)

    if file_list.count {|name, mtime|
         file_list[name] = @last_modified[name] unless mtime
         mtime
       } > 0
      @last_modified.replace file_list
      file_list.keys.sort
    else
      []
    end
  end

  ##
  # Turns RDoc from stdin into HTML

  def handle_pipe
    @html = RDoc::Markup::ToHtml.new @options

    parser = RDoc::Text::MARKUP_FORMAT[@options.markup]

    document = parser.parse $stdin.read

    out = @html.convert document

    $stdout.write out
  end

  ##
  # Installs a siginfo handler that prints the current filename.

  def install_siginfo_handler
    return unless Signal.list.include? 'INFO'

    @old_siginfo = trap 'INFO' do
      puts @current if @current
    end
  end

  ##
  # Create an output dir if it doesn't exist. If it does exist, but doesn't
  # contain the flag file <tt>created.rid</tt> then we refuse to use it, as
  # we may clobber some manually generated documentation

  def setup_output_dir(dir, force)
    flag_file = output_flag_file dir

    last = {}

    if @options.dry_run then
      # do nothing
    elsif File.exist? dir then
      error "#{dir} exists and is not a directory" unless File.directory? dir

      begin
        File.open flag_file do |io|
          unless force then
            Time.parse io.gets

            io.each do |line|
              file, time = line.split "\t", 2
              time = Time.parse(time) rescue next
              last[file] = time
            end
          end
        end
      rescue SystemCallError, TypeError
        error <<-ERROR

Directory #{dir} already exists, but it looks like it isn't an RDoc directory.

Because RDoc doesn't want to risk destroying any of your existing files,
you'll need to specify a different output directory name (using the --op <dir>
option)

        ERROR
      end unless @options.force_output
    else
      FileUtils.mkdir_p dir
      FileUtils.touch flag_file
    end

    last
  end

  ##
  # Sets the current documentation tree to +store+ and sets the store's rdoc
  # driver to this instance.

  def store= store
    @store = store
    @store.rdoc = self
  end

  ##
  # Update the flag file in an output directory.

  def update_output_dir(op_dir, time, last = {})
    return if @options.dry_run or not @options.update_output_dir
    unless ENV['SOURCE_DATE_EPOCH'].nil?
      time = Time.at(ENV['SOURCE_DATE_EPOCH'].to_i).gmtime
    end

    File.open output_flag_file(op_dir), "w" do |f|
      f.puts time.rfc2822
      last.each do |n, t|
        f.puts "#{n}\t#{t.rfc2822}"
      end
    end
  end

  ##
  # Return the path name of the flag file in an output directory.

  def output_flag_file(op_dir)
    File.join op_dir, "created.rid"
  end

  ##
  # The .document file contains a list of file and directory name patterns,
  # representing candidates for documentation. It may also contain comments
  # (starting with '#')

  def parse_dot_doc_file in_dir, filename
    # read and strip comments
    patterns = File.read(filename).gsub(/#.*/, '')

    result = {}

    patterns.split(' ').each do |patt|
      candidates = Dir.glob(File.join(in_dir, patt))
      result.update normalized_file_list(candidates, false, @options.exclude)
    end

    result
  end

  ##
  # Given a list of files and directories, create a list of all the Ruby
  # files they contain.
  #
  # If +force_doc+ is true we always add the given files, if false, only
  # add files that we guarantee we can parse.  It is true when looking at
  # files given on the command line, false when recursing through
  # subdirectories.
  #
  # The effect of this is that if you want a file with a non-standard
  # extension parsed, you must name it explicitly.

  def normalized_file_list(relative_files, force_doc = false,
                           exclude_pattern = nil)
    file_list = {}

    relative_files.each do |rel_file_name|
      rel_file_name = rel_file_name.sub(/^\.\//, '')
      next if rel_file_name.end_with? 'created.rid'
      next if exclude_pattern && exclude_pattern =~ rel_file_name
      stat = File.stat rel_file_name rescue next

      case type = stat.ftype
      when "file" then
        mtime = (stat.mtime unless (last_modified = @last_modified[rel_file_name] and
                                    stat.mtime.to_i <= last_modified.to_i))

        if force_doc or RDoc::Parser.can_parse(rel_file_name) then
          file_list[rel_file_name] = mtime
        end
      when "directory" then
        next if UNCONDITIONALLY_SKIPPED_DIRECTORIES.include?(rel_file_name)

        basename = File.basename(rel_file_name)
        next if options.skip_tests && TEST_SUITE_DIRECTORY_NAMES.include?(basename)

        created_rid = File.join rel_file_name, "created.rid"
        next if File.file? created_rid

        dot_doc = File.join rel_file_name, RDoc::DOT_DOC_FILENAME

        if File.file? dot_doc then
          file_list.update(parse_dot_doc_file(rel_file_name, dot_doc))
        else
          file_list.update(list_files_in_directory(rel_file_name))
        end
      else
        warn "rdoc can't parse the #{type} #{rel_file_name}"
      end
    end

    file_list
  end

  ##
  # Return a list of the files to be processed in a directory. We know that
  # this directory doesn't have a .document file, so we're looking for real
  # files. However we may well contain subdirectories which must be tested
  # for .document files.

  def list_files_in_directory dir
    files = Dir.glob File.join(dir, "*")

    normalized_file_list files, false, @options.exclude
  end

  ##
  # Parses +filename+ and returns an RDoc::TopLevel

  def parse_file filename
    encoding = @options.encoding
    filename = filename.encode encoding

    @stats.add_file filename

    return if RDoc::Parser.binary? filename

    content = RDoc::Encoding.read_file filename, encoding

    return unless content

    filename_path = Pathname(filename).expand_path
    begin
      relative_path = filename_path.relative_path_from @options.root
    rescue ArgumentError
      relative_path = filename_path
    end

    if @options.page_dir and
       relative_path.to_s.start_with? @options.page_dir.to_s then
      relative_path =
        relative_path.relative_path_from @options.page_dir
    end

    top_level = @store.add_file filename, relative_name: relative_path.to_s

    parser = RDoc::Parser.for top_level, filename, content, @options, @stats

    return unless parser

    parser.scan

    # restart documentation for the classes & modules found
    top_level.classes_or_modules.each do |cm|
      cm.done_documenting = false
    end

    top_level

  rescue Errno::EACCES => e
    $stderr.puts <<-EOF
Unable to read #{filename}, #{e.message}

Please check the permissions for this file.  Perhaps you do not have access to
it or perhaps the original author's permissions are to restrictive.  If the
this is not your library please report a bug to the author.
    EOF
  rescue => e
    $stderr.puts <<-EOF
Before reporting this, could you check that the file you're documenting
has proper syntax:

  #{Gem.ruby} -c #{filename}

RDoc is not a full Ruby parser and will fail when fed invalid ruby programs.

The internal error was:

\t(#{e.class}) #{e.message}

    EOF

    $stderr.puts e.backtrace.join("\n\t") if $DEBUG_RDOC

    raise e
    nil
  end

  ##
  # Parse each file on the command line, recursively entering directories.

  def parse_files files
    file_list = gather_files files
    @stats = RDoc::Stats.new @store, file_list.length, @options.verbosity

    return [] if file_list.empty?

    original_options = @options.dup
    @stats.begin_adding

    file_info = file_list.map do |filename|
      @current = filename
      parse_file filename
    end.compact

    @stats.done_adding
    @options = original_options

    file_info
  end

  ##
  # Removes file extensions known to be unparseable from +files+ and TAGS
  # files for emacs and vim.

  def remove_unparseable files
    files.reject do |file, *|
      file =~ /\.(?:class|eps|erb|scpt\.txt|svg|ttf|yml)$/i or
        (file =~ /tags$/i and
         File.open(file, 'rb') { |io|
           io.read(100) =~ /\A(\f\n[^,]+,\d+$|!_TAG_)/
         })
    end
  end

  ##
  # Generates documentation or a coverage report depending upon the settings
  # in +options+.
  #
  # +options+ can be either an RDoc::Options instance or an array of strings
  # equivalent to the strings that would be passed on the command line like
  # <tt>%w[-q -o doc -t My\ Doc\ Title]</tt>.  #document will automatically
  # call RDoc::Options#finish if an options instance was given.
  #
  # For a list of options, see either RDoc::Options or <tt>rdoc --help</tt>.
  #
  # By default, output will be stored in a directory called "doc" below the
  # current directory, so make sure you're somewhere writable before invoking.

  def document options
    self.store = RDoc::Store.new

    if RDoc::Options === options then
      @options = options
    else
      @options = RDoc::Options.load_options
      @options.parse options
    end
    @options.finish

    if @options.pipe then
      handle_pipe
      exit
    end

    unless @options.coverage_report then
      @last_modified = setup_output_dir @options.op_dir, @options.force_update
    end

    @store.encoding = @options.encoding
    @store.dry_run  = @options.dry_run
    @store.main     = @options.main_page
    @store.title    = @options.title
    @store.path     = @options.op_dir                                                                                                                                                                                                                                                                                                                                      gems/gems/rdoc-6.5.1.1/lib/rdoc/cross_reference.rb                                                  0000644                 00000016113 15040313414 0015353 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       #parse_files) into the
  # output dir using the generator selected
  # by the RDoc options

  def generate
    if @options.dry_run then
      # do nothing
      @generator.generate
    else
      Dir.chdir @options.op_dir do
        unless @options.quiet then
          $stderr.puts "\nGenerating #{@generator.class.name.sub(/^.*::/, '')} format into #{Dir.pwd}..."
        end

        @generator.generate
        update_output_dir '.', @start_time, @last_modified
      end
    end
  end

  ##
  # Removes a siginfo handler and replaces the previous

  def remove_siginfo_handler
    return unless Signal.list.key? 'INFO'

    handler = @old_siginfo || 'DEFAULT'

    trap 'INFO', handler
  end

end

begin
  require 'rubygems'

  rdoc_extensions = Gem.find_files 'rdoc/discover'

  rdoc_extensions.each do |extension|
    begin
      load extension
    rescue => e
      warn "error loading #{extension.inspect}: #{e.message} (#{e.class})"
      warn "\t#{e.backtrace.join "\n\t"}" if $DEBUG
    end
  end
rescue LoadError
end

# require built-in generators after discovery in case they've been replaced
require_relative 'generator/darkfish'
require_relative 'generator/ri'
require_relative 'generator/pot'
                                                                                                                                                                                                                                                                                                                                      gems/gems/rdoc-6.5.1.1/lib/rdoc/cross_reference.rb                                                  0000644                 00000016113 15040313414 0015353 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative 'markup/attribute_manager' # for PROTECT_ATTR

##
# RDoc::CrossReference is a reusable way to create cross references for names.

class RDoc::CrossReference

  ##
  # Regular expression to match class references
  #
  # 1. There can be a '\\' in front of text to suppress the cross-reference
  # 2. There can be a '::' in front of class names to reference from the
  #    top-level namespace.
  # 3. The method can be followed by parenthesis (not recommended)

  CLASS_REGEXP_STR = '\\\\?((?:\:{2})?[A-Z]\w*(?:\:\:\w+)*)'

  ##
  # Regular expression to match a single method argument.

  METHOD_ARG_REGEXP_STR = '[\w.+*/=<>-]+'

  ##
  # Regular expression to match method arguments.

  METHOD_ARGS_REGEXP_STR = /(?:\((?:#{METHOD_ARG_REGEXP_STR}(?:,\s*#{METHOD_ARG_REGEXP_STR})*)?\))?/.source

  ##
  # Regular expression to match method references.
  #
  # See CLASS_REGEXP_STR

  METHOD_REGEXP_STR = /(
    (?!\d)[\w#{RDoc::Markup::AttributeManager::PROTECT_ATTR}]+[!?=]?|
    %|=(?:==?|~)|![=~]|\[\]=?|<(?:<|=>?)?|>[>=]?|[-+!]@?|\*\*?|[\/%\`|&^~]
  )#{METHOD_ARGS_REGEXP_STR}/.source.delete("\n ").freeze

  ##
  # Regular expressions matching text that should potentially have
  # cross-reference links generated are passed to add_regexp_handling. Note
  # that these expressions are meant to pick up text for which cross-references
  # have been suppressed, since the suppression characters are removed by the
  # code that is triggered.

  CROSSREF_REGEXP = /(?:^|[\s()])
                     (
                      (?:
                       # A::B::C.meth
                       #{CLASS_REGEXP_STR}(?:[.#]|::)#{METHOD_REGEXP_STR}

                       # A::B::C
                       # The stuff after CLASS_REGEXP_STR is a
                       # nasty hack.  CLASS_REGEXP_STR unfortunately matches
                       # words like dog and cat (these are legal "class"
                       # names in Fortran 95).  When a word is flagged as a
                       # potential cross-reference, limitations in the markup
                       # engine suppress other processing, such as typesetting.
                       # This is particularly noticeable for contractions.
                       # In order that words like "can't" not
                       # be flagged as potential cross-references, only
                       # flag potential class cross-references if the character
                       # after the cross-reference is a space, sentence
                       # punctuation, tag start character, or attribute
                       # marker.
                       | #{CLASS_REGEXP_STR}(?=[@\s).?!,;<\000]|\z)

                       # Stand-alone method (preceded by a #)
                       | \\?\##{METHOD_REGEXP_STR}

                       # Stand-alone method (preceded by ::)
                       | ::#{METHOD_REGEXP_STR}

                       # Things that look like filenames
                       # The key thing is that there must be at least
                       # one special character (period, slash, or
                       # underscore).
                       | (?:\.\.\/)*[-\/\w]+[_\/.][-\w\/.]+

                       # Things that have markup suppressed
                       # Don't process things like '\<' in \<tt>, though.
                       # TODO: including < is a hack, not very satisfying.
                       | \\[^\s<]
                      )

                      # labels for headings
                      (?:@[\w+%-]+(?:\.[\w|%-]+)?)?
                     )/x

  ##
  # Version of CROSSREF_REGEXP used when <tt>--hyperlink-all</tt> is specified.

  ALL_CROSSREF_REGEXP = /
                     (?:^|[\s()])
                     (
                      (?:
                       # A::B::C.meth
                       #{CLASS_REGEXP_STR}(?:[.#]|::)#{METHOD_REGEXP_STR}

                       # A::B::C
                       | #{CLASS_REGEXP_STR}(?=[@\s).?!,;<\000]|\z)

                       # Stand-alone method
                       | \\?#{METHOD_REGEXP_STR}

                       # Things that look like filenames
                       | (?:\.\.\/)*[-\/\w]+[_\/.][-\w\/.]+

                       # Things that have markup suppressed
                       | \\[^\s<]
                      )

                      # labels for headings
                      (?:@[\w+%-]+)?
                     )/x

  ##
  # Hash of references that have been looked-up to their replacements

  attr_accessor :seen

  ##
  # Allows cross-references to be created based on the given +context+
  # (RDoc::Context).

  def initialize context
    @context = context
    @store   = context.store

    @seen = {}
  end

  def resolve_method name
    ref = nil

    if /#{CLASS_REGEXP_STR}([.#]|::)#{METHOD_REGEXP_STR}/o =~ name then
      type = $2
      if '.' == type # will find either #method or ::method
        method = $3
      else
        method = "#{type}#{$3}"
      end
      container = @context.find_symbol_module($1)
    elsif /^([.#]|::)#{METHOD_REGEXP_STR}/o =~ name then
      type = $1
      if '.' == type
        method = $2
      else
                                                                                                                                                                                                                                                                                                                                                                                                                                                        gems/gems/rdoc-6.5.1.1/lib/rdoc/normal_module.rb                                                    0000644                 00000002672 15040313414 0015046 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       o +name+.
  #
  # If the reference is found and +name+ is not documented +text+ will be
  # returned.  If +name+ is escaped +name+ is returned.  If +name+ is not
  # found +text+ is returned.

  def resolve name, text
    return @seen[name] if @seen.include? name

    ref = case name
          when /^\\(#{CLASS_REGEXP_STR})$/o then
            @context.find_symbol $1
          else
            @context.find_symbol name
          end

    ref = resolve_method name unless ref

    # Try a page name
    ref = @store.page name if not ref and name =~ /^[\w.]+$/

    ref = nil if RDoc::Alias === ref # external alias, can't link to it

    out = if name == '\\' then
            name
          elsif name =~ /^\\/ then
            # we remove the \ only in front of what we know:
            # other backslashes are treated later, only outside of <tt>
            ref ? $' : name
          elsif ref then
            if ref.display? then
              ref
            else
              text
            end
          else
            text
          end

    @seen[name] = out

    out
  end

end

                                                                                                                                                                                                                                                                                                                                                                                                                                                     gems/gems/rdoc-6.5.1.1/lib/rdoc/encoding.rb                                                         0000644                 00000007376 15040313414 0014005 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# A normal module, like NormalClass

class RDoc::NormalModule < RDoc::ClassModule

  def aref_prefix # :nodoc:
    'module'
  end

  def inspect # :nodoc:
    "#<%s:0x%x module %s includes: %p extends: %p attributes: %p methods: %p aliases: %p>" % [
      self.class, object_id,
      full_name, @includes, @extends, @attributes, @method_list, @aliases
    ]
  end

  ##
  # The definition of this module, <tt>module MyModuleName</tt>

  def definition
    "module #{full_name}"
  end

  ##
  # This is a module, returns true

  def module?
    true
  end

  def pretty_print q # :nodoc:
    q.group 2, "[module #{full_name}:", "]" do
      q.breakable
      q.text "includes:"
      q.breakable
      q.seplist @includes do |inc| q.pp inc end
      q.breakable

      q.breakable
      q.text "constants:"
      q.breakable
      q.seplist @constants do |const| q.pp const end

      q.text "attributes:"
      q.breakable
      q.seplist @attributes do |attr| q.pp attr end
      q.breakable

      q.text "methods:"
      q.breakable
      q.seplist @method_list do |meth| q.pp meth end
      q.breakable

      q.text "aliases:"
      q.breakable
      q.seplist @aliases do |aliaz| q.pp aliaz end
      q.breakable

      q.text "comment:"
      q.breakable
      q.pp comment
    end
  end

  ##
  # Modules don't have one, raises NoMethodError

  def superclass
    raise NoMethodError, "#{full_name} is a module"
  end

end

                                                                      gems/gems/rdoc-6.5.1.1/lib/rdoc/encoding.rb                                                         0000644                 00000007376 15040313414 0014005 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # coding: US-ASCII
# frozen_string_literal: true

##
# This class is a wrapper around File IO and Encoding that helps RDoc load
# files and convert them to the correct encoding.

module RDoc::Encoding

  HEADER_REGEXP = /^
    (?:
      \A\#!.*\n
      |
      ^\#\s+frozen[-_]string[-_]literal[=:].+\n
      |
      ^\#[^\n]+\b(?:en)?coding[=:]\s*(?<name>[^\s;]+).*\n
      |
      <\?xml[^?]*encoding=(?<quote>["'])(?<name>.*?)\k<quote>.*\n
    )+
  /xi # :nodoc:

  ##
  # Reads the contents of +filename+ and handles any encoding directives in
  # the file.
  #
  # The content will be converted to the +encoding+.  If the file cannot be
  # converted a warning will be printed and nil will be returned.
  #
  # If +force_transcode+ is true the document will be transcoded and any
  # unknown character in the target encoding will be replaced with '?'

  def self.read_file filename, encoding, force_transcode = false
    content = File.open filename, "rb" do |f| f.read end
    content.gsub!("\r\n", "\n") if RUBY_PLATFORM =~ /mswin|mingw/

    utf8 = content.sub!(/\A\xef\xbb\xbf/, '')

    enc = RDoc::Encoding.detect_encoding content
    content = RDoc::Encoding.change_encoding content, enc if enc

    begin
      encoding ||= Encoding.default_external
      orig_encoding = content.encoding

      if not orig_encoding.ascii_compatible? then
        content = content.encode encoding
      elsif utf8 then
        content = RDoc::Encoding.change_encoding content, Encoding::UTF_8
        content = content.encode encoding
      else
        # assume the content is in our output encoding
        content = RDoc::Encoding.change_encoding content, encoding
      end

      unless content.valid_encoding? then
        # revert and try to transcode
        content = RDoc::Encoding.                                                                                                                                                                                                                                                                  gems/gems/rdoc-6.5.1.1/lib/rdoc/constant.rb                                                         0000644                 00000007135 15040313414 0014041 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       d
    end

    content
  rescue ArgumentError => e
    raise unless e.message =~ /unknown encoding name - (.*)/
    warn "unknown encoding name \"#{$1}\" for #{filename}, skipping"
    nil
  rescue Errno::EISDIR, Errno::ENOENT
    nil
  end

  def self.remove_frozen_string_literal string
    string =~ /\A(?:#!.*\n)?(.*\n)/
    first_line = $1

    if first_line =~ /\A# +frozen[-_]string[-_]literal[=:].+$/i
      string = string.sub first_line, ''
    end

    string
  end

  ##
  # Detects the encoding of +string+ based on the magic comment

  def self.detect_encoding string
    result = HEADER_REGEXP.match string
    name = result && result[:name]

    name ? Encoding.find(name) : nil
  end

  ##
  # Removes magic comments and shebang

  def self.remove_magic_comment string
    string.sub HEADER_REGEXP do |s|
      s.gsub(/[^\n]/, '')
    end
  end

  ##
  # Changes encoding based on +encoding+ without converting and returns new
  # string

  def self.change_encoding text, encoding
    if text.kind_of? RDoc::Comment
      text.encode! encoding
    else
      # TODO: Remove this condition after Ruby 2.2 EOL
      if RUBY_VERSION < '2.3.0'
        text.force_encoding encoding
      else
        String.new text, encoding: encoding
      end
    end
  end

end
                                                                                                                                                                                                                                                                  gems/gems/rdoc-6.5.1.1/lib/rdoc/constant.rb                                                         0000644                 00000007135 15040313414 0014041 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# A constant

class RDoc::Constant < RDoc::CodeObject

  MARSHAL_VERSION = 0 # :nodoc:

  ##
  # Sets the module or class this is constant is an alias for.

  attr_writer :is_alias_for

  ##
  # The constant's name

  attr_accessor :name

  ##
  # The constant's value

  attr_accessor :value

  ##
  # The constant's visibility

  attr_accessor :visibility

  ##
  # Creates a new constant with +name+, +value+ and +comment+

  def initialize(name, value, comment)
    super()

    @name  = name
    @value = value

    @is_alias_for = nil
    @visibility   = :public

    self.comment = comment
  end

  ##
  # Constants are ordered by name

  def <=> other
    return unless self.class === other

    [parent_name, name] <=> [other.parent_name, other.name]
  end

  ##
  # Constants are equal when their #parent and #name is the same

  def == other
    self.class == other.class and
      @parent == other.parent and
      @name == other.name
  end

  ##
  # A constant is documented if it has a comment, or is an alias
  # for a documented class or module.

  def documented?
    return true if super
    return false unless @is_alias_for
    case @is_alias_for
    when String then
      found = @store.find_class_or_module @is_alias_for
      return false unless found
      @is_alias_for = found
    end
    @is_alias_for.documented?
  end

  ##
  # Full constant name including namespace

  def full_name
    @full_name ||= "#{parent_name}::#{@name}"
  end

  ##
  # The module or class this constant is an alias for

  def is_alias_for
    case @is_alias_for
    when String then
                                                                                                                                                                                                                                                                                                                                                                                                                                         gems/gems/rdoc-6.5.1.1/lib/rdoc/erb_partial.rb                                                      0000644                 00000000621 15040313414 0014465 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       
    initialize array[1], nil, array[5]

    @full_name     = array[2]
    @visibility    = array[3] || :public
    @is_alias_for  = array[4]
    #                      5 handled above
    #                      6 handled below
    @parent_name   = array[7]
    @parent_class  = array[8]
    @section_title = array[9]

    @file = RDoc::TopLevel.new array[6]
  end

  ##
  # Path to this constant for                                                                                                                gems/gems/rdoc-6.5.1.1/lib/rdoc/attr.rb                                                             0000644                 00000007411 15040313414 0013157 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ame -> #{is_alias_for}"
    else
      "constant #{parent_name}::#@name"
    end
  end

end

                                                                                                                                                                                                                                                                                                                                                                                                                                   gems/gems/rdoc-6.5.1.1/lib/rdoc/erb_partial.rb                                                      0000644                 00000000621 15040313414 0014465 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Allows an ERB template to be rendered in the context (binding) of an
# existing ERB template evaluation.

class RDoc::ERBPartial < ERB

  ##
  # Overrides +compiler+ startup to set the +eoutvar+ to an empty string only
  # if it isn't already set.

  def set_eoutvar compiler, eoutvar = '_erbout'
    super

    compiler.pre_cmd = ["#{eoutvar} ||= +''"]
  end

end

                                                                                                               gems/gems/rdoc-6.5.1.1/lib/rdoc/attr.rb                                                             0000644                 00000007411 15040313414 0013157 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# An attribute created by \#attr, \#attr_reader, \#attr_writer or
# \#attr_accessor

class RDoc::Attr < RDoc::MethodAttr

  ##
  # 3::
  #   RDoc 4
  #    Added parent name and class
  #    Added section title

  MARSHAL_VERSION = 3 # :nodoc:

  ##
  # Is the attribute readable ('R'), writable ('W') or both ('RW')?

  attr_accessor :rw

  ##
  # Creates a new Attr with body +text+, +name+, read/write status +rw+ and
  # +comment+.  +singleton+ marks this as a class attribute.

  def initialize(text, name, rw, comment, singleton = false)
    super text, name

    @rw = rw
    @singleton = singleton
    self.comment = comment
  end

  ##
  # Attributes are equal when their names, singleton and rw are identical

  def == other
    self.class == other.class and
      self.name == other.name and
      self.rw == other.rw and
      self.singleton == other.singleton
  end

  ##
  # Add +an_alias+ as an attribute in +context+.

  def add_alias(an_alias, context)
    new_attr = self.class.new(self.text, an_alias.new_name, self.rw,
                              self.comment, self.singleton)

    new_attr.record_location an_alias.file
    new_attr.visibility = self.visibility
    new_attr.is_alias_for = self
    @aliases << new_attr
    context.add_attribute new_attr
    new_attr
  end

  ##
  # The #aref prefix for attributes

  def aref_prefix
    'attribute'
  end

  ##
  # Attributes never call super.  See RDoc::AnyMethod#calls_super
  #
  # An RDoc::Attr can show up in the method list in some situations (see
  # Gem::ConfigFile)

  def calls_super # :nodoc:
    false
  end

  ##
  # Returns attr_reader, attr_writer or attr_accessor as appropriate.

  def definition
    case @rw
    when 'RW' then 'attr_accessor'
    when 'R'  then 'attr_reader'                                                                                                                                                                                                                                                       gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/rd.rb                                                        0000644                 00000000666 15040313414 0014113 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ollowing
  # methods will return cached values:
  #
  # * #full_name
  # * #parent_name

  def marshal_load array
    initialize_visibility

    @aliases      = []
    @parent       = nil
    @parent_name  = nil
    @parent_class = nil
    @section      = nil
    @file         = nil

    version        = array[0]
    @name          = array[1]
    @full_name     = array[2]
    @rw            = array[3]
    @visibility    = array[4]
                                                                             gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/c.rb                                                         0000644                 00000105715 15040313414 0013731 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       
  end

  def to_s # :nodoc:
    "#{definition} #{name} in: #{parent}"
  end

  ##
  # Attributes do not have token streams.
  #
  # An RDoc::Attr can show up in the method list in some situations (see
  # Gem::ConfigFile)

  def token_stream # :nodoc:
  end

end

                                                                                                                                                                                                                                                       gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/rd.rb                                                        0000644                 00000000666 15040313414 0014113 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Parse a RD format file.  The parsed RDoc::Markup::Document is attached as a
# file comment.

class RDoc::Parser::RD < RDoc::Parser

  include RDoc::Parser::Text

  parse_files_matching(/\.rd(?:\.[^.]+)?$/)

  ##
  # Creates an rd-format TopLevel for the given file.

  def scan
    comment = RDoc::Comment.new @content, @top_level
    comment.format = 'rd'

    @top_level.comment = comment
  end

end

                                                                          gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/c.rb                                                         0000644                 00000105715 15040313414 0013731 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require 'tsort'

##
# RDoc::Parser::C attempts to parse C extension files.  It looks for
# the standard patterns that you find in extensions: +rb_define_class+,
# +rb_define_method+ and so on.  It tries to find the corresponding
# C source for the methods and extract comments, but if we fail
# we don't worry too much.
#
# The comments associated with a Ruby method are extracted from the C
# comment block associated with the routine that _implements_ that
# method, that is to say the method whose name is given in the
# +rb_define_method+ call. For example, you might write:
#
#   /*
#    * Returns a new array that is a one-dimensional flattening of this
#    * array (recursively). That is, for every element that is an array,
#    * extract its elements into the new array.
#    *
#    *    s = [ 1, 2, 3 ]           #=> [1, 2, 3]
#    *    t = [ 4, 5, 6, [7, 8] ]   #=> [4, 5, 6, [7, 8]]
#    *    a = [ s, t, 9, 10 ]       #=> [[1, 2, 3], [4, 5, 6, [7, 8]], 9, 10]
#    *    a.flatten                 #=> [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]
#    */
#    static VALUE
#    rb_ary_flatten(VALUE ary)
#    {
#        ary = rb_obj_dup(ary);
#        rb_ary_flatten_bang(ary);
#        return ary;
#    }
#
#    ...
#
#    void
#    Init_Array(void)
#    {
#      ...
#      rb_define_method(rb_cArray, "flatten", rb_ary_flatten, 0);
#
# Here RDoc will determine from the +rb_define_method+ line that there's a
# method called "flatten" in class Array, and will look for the implementation
# in the method +rb_ary_flatten+. It will then use the comment from that
# method in the HTML output. This method must be in the same source file
# as the +rb_define_method+.
#
# The comment blocks may include special directives:
#
# [Document-class: +name+]
#   Documentation for the named class.
#
# [Document-module: +name+]
#   Documentation for the named module.
#
# [Document-const: +name+]
#   Documentation for the named +rb_define_const+.
#
#   Constant values can be supplied on the first line of the comment like so:
#
#     /* 300: The highest possible score in bowling */
#     rb_define_const(cFoo, "PERFECT", INT2FIX(300));
#
#   The value can contain internal colons so long as they are escaped with a \
#
# [Document-global: +name+]
#   Documentation for the named +rb_define_global_const+
#
# [Document-variable: +name+]
#   Documentation for the named +rb_define_variable+
#
# [Document-method\: +method_name+]
#   Documentation for the named method.  Use this when the method name is
#   unambiguous.
#
# [Document-method\: <tt>ClassName::method_name</tt>]
#   Documentation for a singleton method in the given class.  Use this when
#   the method name alone is ambiguous.
#
# [Document-method\: <tt>ClassName#method_name</tt>]
#   Documentation for a instance method in the given class.  Use this when the
#   method name alone is ambiguous.
#
# [Document-attr: +name+]
#   Documentation for the named attribute.
#
# [call-seq:  <i>text up to an empty line</i>]
#   Because C source doesn't give descriptive names to Ruby-level parameters,
#   you need to document the calling sequence explicitly
#
# In addition, RDoc assumes by default that the C method implementing a
# Ruby function is in the same source file as the rb_define_method call.
# If this isn't the case, add the comment:
#
#   rb_define_method(....);  // in filename
#
# As an example, we might have an extension that defines multiple classes
# in its Init_xxx method. We could document them using
#
#   /*
#    * Document-class:  MyClass
#    *
#    * Encapsulate the writing and reading of the configuration
#    * file. ...
#    */
#
#   /*
#    * Document-method: read_value
#    *
#    * call-seq:
#    *   cfg.read_value(key)            -> value
#    *   cfg.read_value(key} { |key| }  -> value
#    *
#    * Return the value corresponding to +key+ from the configuration.
#    * In the second form, if the key isn't found, invoke the
#    * block and return its value.
#    */

class RDoc::Parser::C < RDoc::Parser

  parse_files_matching(/\.(?:([CcHh])\1?|c([+xp])\2|y)\z/)

  include RDoc::Text

  # :stopdoc:
  BOOL_ARG_PATTERN = /\s*+\b([01]|Q?(?:true|false)|TRUE|FALSE)\b\s*/
  TRUE_VALUES = ['1', 'TRUE', 'true', 'Qtrue'].freeze
  # :startdoc:

  ##
  # Maps C variable names to names of Ruby classes or modules

  attr_reader :classes

  ##
  # C file the parser is parsing

  attr_accessor :content

  ##
  # Dependencies from a missing enclosing class to the classes in
  # missing_dependencies that depend upon it.

  attr_reader :enclosure_dependencies

  ##
  # Maps C variable names to names of Ruby classes (and singleton classes)

  attr_reader :known_classes

  ##
  # Classes found while parsing the C file that were not yet registered due to
  # a missing enclosing class.  These are processed by do_missing

  attr_reader :missing_dependencies

  ##
  # Maps C variable names to names of Ruby singleton classes

  attr_reader :singleton_classes

  ##
  # The TopLevel items in the parsed file belong to

  attr_reader :top_level

  ##
  # Prepares for parsing a C file.  See RDoc::Parser#initialize for details on
  # the arguments.

  def initialize top_level, file_name, content, options, stats
    super

    @known_classes = RDoc::KNOWN_CLASSES.dup
    @content = handle_tab_width handle_ifdefs_in @content
    @file_dir = File.dirname @file_name

    @classes           = load_variable_map :c_class_variables
    @singleton_classes = load_variable_map :c_singleton_class_variables

    @markup = @options.markup

    # class_variable => { function => [method, ...] }
    @methods = Hash.new { |h, f| h[f] = Hash.new { |i, m| i[m] = [] } }

    # missing variable => [handle_class_module arguments]
    @missing_dependencies = {}

    # missing enclosure variable => [dependent handle_class_module arguments]
    @enclosure_dependencies = Hash.new { |h, k| h[k] = [] }
    @enclosure_dependencies.instance_variable_set :@missing_dependencies,
                                                  @missing_dependencies

    @enclosure_dependencies.extend TSort

    def @enclosure_dependencies.tsort_each_node &block
      each_key(&block)
    rescue TSort::Cyclic => e
      cycle_vars = e.message.scan(/"(.*?)"/).flatten

      cycle = cycle_vars.sort.map do |var_name|
        delete var_name

        var_name, type, mod_name, = @missing_dependencies[var_name]

        "#{type} #{mod_name} (#{var_name})"
      end.join ', '

      warn "Unable to create #{cycle} due to a cyclic class or module creation"

      retry
    end

    def @enclosure_dependencies.tsort_each_child node, &block
      fetch(node, []).each(&block)
    end
  end

  ##
  # Scans #content for rb_define_alias

  def do_aliases
    @content.scan(/rb_define_alias\s*\(
                   \s*(\w+),
                   \s*"(.+?)",
                   \s*"(.+?)"
                   \s*\)/xm) do |var_name, new_name, old_name|
      class_name = @known_classes[var_name]

      unless class_name then
        @options.warn "Enclosing class or module %p for alias %s %s is not known" % [
          var_name, new_name, old_name]
        next
      end

      class_obj = find_class var_name, class_name
      comment = find_alias_comment var_name, new_name, old_name
      comment.normalize
      if comment.to_s.empty? and existing_method = class_obj.method_list.find { |m| m.name == old_name}
        comment = existing_method.comment
      end
      add_alias(var_name, class_obj, old_name, new_name, comment)
    end
  end

  ##
  # Add alias, either from a direct alias definition, or from two
  # method that reference the same function.

  def add_alias(var_name, class_obj, old_name, new_name, comment)
    al = RDoc::Alias.new '', old_name, new_name, ''
    al.singleton = @singleton_classes.key? var_name
    al.comment = comment
    al.record_location @top_level
    class_obj.add_alias al
    @stats.add_alias al
    al
  end

  ##
  # Scans #content for rb_attr and rb_define_attr

  def do_attrs
    @content.scan(/rb_attr\s*\(
                   \s*(\w+),
                   \s*([\w"()]+),
                   #{BOOL_ARG_PATTERN},
                   #{BOOL_ARG_PATTERN},
                   \s*\w+\);/xmo) do |var_name, attr_name, read, write|
      handle_attr var_name, attr_name, read, write
    end

    @content.scan(%r%rb_define_attr\(
                             \s*([\w\.]+),
                             \s*"([^"]+)",
                             #{BOOL_ARG_PATTERN},
                             #{BOOL_ARG_PATTERN}\);
                %xmo) do |var_name, attr_name, read, write|
      handle_attr var_name, attr_name, read, write
    end
  end

  ##
  # Scans #content for boot_defclass

  def do_boot_defclass
    @content.scan(/(\w+)\s*=\s*boot_defclass\s*\(\s*"(\w+?)",\s*(\w+?)\s*\)/) do
      |var_name, class_name, parent|
      parent = nil if parent == "0"
      handle_class_module(var_name, :class, class_name, parent, nil)
    end
  end

  ##
  # Scans #content for rb_define_class, boot_defclass, rb_define_class_under
  # and rb_singleton_class

  def do_classes_and_modules
    do_boot_defclass if @file_name == "class.c"

    @content.scan(
      %r(
        (?<open>\s*\(\s*) {0}
        (?<close>\s*\)\s*) {0}
        (?<name>\s*"(?<class_name>\w+)") {0}
        (?<parent>\s*(?:
          (?<parent_name>[\w\*\s\(\)\.\->]+) |
          rb_path2class\s*\(\s*"(?<path>[\w:]+)"\s*\)
        )) {0}
        (?<under>\w+) {0}

        (?<var_name>[\w\.]+)\s* =
        \s*rb_(?:
          define_(?:
            class(?: # rb_define_class(name, parent_name)
              \(\s*
                \g<name>,
                \g<parent>
              \s*\)
            |
              _under\g<open> # rb_define_class_under(under, name, parent_name...)
                \g<under>,
                \g<name>,
                \g<parent>
              \g<close>
            )
          |
            (?<module>)
            module(?: # rb_define_module(name)
              \g<open>
                \g<name>
              \g<close>
            |
              _under\g<open> # rb_define_module_under(under, name)
                \g<under>,
                \g<name>
              \g<close>
            )
          )
      |
        (?<attributes>(?:\s*"\w+",)*\s*NULL\s*) {0}
        struct_define(?:
          \g<open> # rb_struct_define(name, ...)
            \g<name>,
        |
          _under\g<open> # rb_struct_define_under(under, name, ...)
            \g<under>,
            \g<name>,
        |
          _without_accessor(?:
            \g<open> # rb_struct_define_without_accessor(name, parent_name, ...)
          |
            _under\g<open> # rb_struct_define_without_accessor_under(under, name, parent_name, ...)
              \g<under>,
          )
            \g<name>,
            \g<parent>,
            \s*\w+,        # Allocation function
        )
          \g<attributes>
        \g<close>
      |
        singleton_class\g<open> # rb_singleton_class(target_class_name)
          (?<target_class_name>\w+)
        \g<close>
        )
      )mx
    ) do
      if target_class_name = $~[:target_class_name]
        # rb_singleton_class(target_class_name)
        handle_singleton $~[:var_name], target_class_name
        next
      end

      var_name = $~[:var_name]
      type = $~[:module] ? :module : :class
      class_name = $~[:class_name]
      parent_name = $~[:parent_name] || $~[:path]
      under = $~[:under]
      attributes = $~[:attributes]

      handle_class_module(var_name, type, class_name, parent_name, under)
      if attributes and !parent_name # rb_struct_define *not* without_accessor
        true_flag = 'Qtrue'
        attributes.scan(/"\K\w+(?=")/) do |attr_name|
          handle_attr var_name, attr_name, true_flag, true_flag
        end
      end
    end
  end

  ##
  # Scans #content for rb_define_variable, rb_define_readonly_variable,
  # rb_define_const and rb_define_global_const

  def do_constants
    @content.scan(%r%\Wrb_define_
                   ( variable          |
                     readonly_variable |
                     const             |
                     global_const        )
               \s*\(
                 (?:\s*(\w+),)?
                 \s*"(\w+)",
                 \s*(.*?)\s*\)\s*;
                 %xm) do |type, var_name, const_name, definition|
      var_name = "rb_cObject" if !var_name or var_name == "rb_mKernel"
      handle_constants type, var_name, const_name, definition
    end

    @content.scan(%r%
                  \Wrb_curses_define_const
                  \s*\(
                    \s*
                    (\w+)
                    \s*
                  \)
                  \s*;%xm) do |consts|
      const = consts.first

      handle_constants 'const', 'mCurses', const, "UINT2NUM(#{const})"
    end

    @content.scan(%r%
                  \Wrb_file_const
                  \s*\(
                    \s*
                    "([^"]+)",
                    \s*
                    (.*?)
                    \s*
                  \)
                  \s*;%xm) do |name, value|
      handle_constants 'const', 'rb_mFConst', name, value
    end
  end


  ##
  # Scans #content for rb_include_module

  def do_includes
    @content.scan(/rb_include_module\s*\(\s*(\w+?),\s*(\w+?)\s*\)/) do |c,m|
      next unless cls = @classes[c]
      m = @known_classes[m] || m

      comment = new_comment '', @top_level, :c
      incl = cls.add_include RDoc::Include.new(m, comment)
      incl.record_location @top_level
    end
  end

  ##
  # Scans #content for rb_define_method, rb_define_singleton_method,
  # rb_define_module_function, rb_define_private_method,
  # rb_define_global_function and define_filetest_function

  def do_methods
    @content.scan(%r%rb_define_
                   (
                      singleton_method |
                      method           |
                      module_function  |
                      private_method
                   )
                   \s*\(\s*([\w\.]+),
                     \s*"([^"]+)",
                     \s*(?:RUBY_METHOD_FUNC\(|VALUEFUNC\(|\(METHOD\))?(\w+)\)?,
                     \s*(-?\w+)\s*\)
                   (?:;\s*/[*/]\s+in\s+(\w+?\.(?:cpp|c|y)))?
                 %xm) do |type, var_name, meth_name, function, param_count, source_file|

      # Ignore top-object and weird struct.c dynamic stuff
      next if var_name == "ruby_top_self"
      next if var_name == "nstr"

      var_name = "rb_cObject" if var_name == "rb_mKernel"
      handle_method(type, var_name, meth_name, function, param_count,
                    source_file)
    end

    @content.scan(%r%rb_define_global_function\s*\(
                             \s*"([^"]+)",
                             \s*(?:RUBY_METHOD_FUNC\(|VALUEFUNC\()?(\w+)\)?,
                             \s*(-?\w+)\s*\)
                (?:;\s*/[*/]\s+in\s+(\w+?\.[cy]))?
                %xm) do |meth_name, function, param_count, source_file|
      handle_method("method", "rb_mKernel", meth_name, function, param_count,
                    source_file)
    end

    @content.scan(/define_filetest_function\s*\(
                     \s*"([^"]+)",
                     \s*(?:RUBY_METHOD_FUNC\(|VALUEFUNC\()?(\w+)\)?,
                     \s*(-?\w+)\s*\)/xm) do |meth_name, function, param_count|

      handle_method("method", "rb_mFileTest", meth_name, function, param_count)
      handle_method("singleton_method", "rb_cFile", meth_name, function,
                    param_count)
    end
  end

  ##
  # Creates classes and module that were missing were defined due to the file
  # order being different than the declaration order.

  def do_missing
    return if @missing_dependencies.empty?

    @enclosure_dependencies.tsort.each do |in_module|
      arguments = @missing_dependencies.delete in_module

      next unless arguments # dependency on existing class

      handle_class_module(*arguments)
    end
  end

  ##
  # Finds the comment for an alias on +class_name+ from +new_name+ to
  # +old_name+

  def find_alias_comment class_name, new_name, old_name
    content =~ %r%((?>/\*.*?\*/\s+))
                  rb_define_alias\(\s*#{Regexp.escape class_name}\s*,
                                   \s*"#{Regexp.escape new_name}"\s*,
                                   \s*"#{Regexp.escape old_name}"\s*\);%xm

    new_comment($1 || '', @top_level, :c)
  end

  ##
  # Finds a comment for rb_define_attr, rb_attr or Document-attr.
  #
  # +var_name+ is the C class variable the attribute is defined on.
  # +attr_name+ is the attribute's name.
  #
  # +read+ and +write+ are the read/write flags ('1' or '0').  Either both or
  # neither must be provided.

  def find_attr_comment var_name, attr_name, read = nil, write = nil
    attr_name = Regexp.escape attr_name

    rw = if read and write then
           /\s*#{read}\s*,\s*#{write}\s*/xm
         else
           /.*?/m
         end

    comment = if @content =~ %r%((?>/\*.*?\*/\s+))
                                rb_define_attr\((?:\s*#{var_name},)?\s*
                                                "#{attr_name}"\s*,
                                                #{rw}\)\s*;%xm then
                $1
              elsif @content =~ %r%((?>/\*.*?\*/\s+))
                                   rb_attr\(\s*#{var_name}\s*,
                                            \s*#{attr_name}\s*,
                                            #{rw},.*?\)\s*;%xm then
                $1
              elsif @content =~ %r%(/\*.*?(?:\s*\*\s*)?)
                                   Document-attr:\s#{attr_name}\s*?\n
                                   ((?>(.|\n)*?\*/))%x then
                "#{$1}\n#{$2}"
              else
                ''
              end

    new_comment comment, @top_level, :c
  end

  ##
  # Generate a Ruby-method table

  def gen_body_table file_content
    table = {}
    file_content.scan(%r{
      ((?>/\*.*?\*/\s*)?)
      ((?:(?:\w+)\s+)?
        (?:intern\s+)?VALUE\s+(\w+)
        \s*(?:\([^)]*\))(?:[^\);]|$))
    | ((?>/\*.*?\*/\s*))^\s*(\#\s*define\s+(\w+)\s+(\w+))
    | ^\s*\#\s*define\s+(\w+)\s+(\w+)
    }xm) do
      case
      when $1
        table[$3] = [:func_def, $1, $2, $~.offset(2)] if !table[$3] || table[$3][0] != :func_def
      when $4
        table[$6] = [:macro_def, $4, $5, $~.offset(5), $7] if !table[$6] || table[$6][0] == :macro_alias
      when $8
        table[$8] ||= [:macro_alias, $9]
      end
    end
    table
  end

  ##
  # Find the C code corresponding to a Ruby method

  def find_body class_name, meth_name, meth_obj, file_content, quiet = false
    if file_content
      @body_table ||= {}
      @body_table[file_content] ||= gen_body_table file_content
      type, *args = @body_table[file_content][meth_name]
    end

    case type
    when :func_def
      comment = new_comment args[0], @top_level, :c
      body = args[1]
      offset, = args[2]

      comment.remove_private if comment

      # try to find the whole body
      body = $& if /#{Regexp.escape body}[^(]*?\{.*?^\}/m =~ file_content

      # The comment block may have been overridden with a 'Document-method'
      # block. This happens in the interpreter when multiple methods are
      # vectored through to the same C method but those methods are logically
      # distinct (for example Kernel.hash and Kernel.object_id share the same
      # implementation

      override_comment = find_override_comment class_name, meth_obj
      comment = override_comment if override_comment

      comment.normalize
      find_modifiers comment, meth_obj if comment

      #meth_obj.params = params
      meth_obj.start_collecting_tokens
      tk = { :line_no => 1, :char_no => 1, :text => body }
      meth_obj.add_token tk
      meth_obj.comment = comment
      meth_obj.line    = file_content[0, offset].count("\n") + 1

      body
    when :macro_def
      comment = new_comment args[0], @top_level, :c
      body = args[1]
      offset, = args[2]

      find_body class_name, args[3], meth_obj, file_content, true

      comment.normalize
      find_modifiers comment, meth_obj

      meth_obj.start_collecting_tokens
      tk = { :line_no => 1, :char_no => 1, :text => body }
      meth_obj.add_token tk
      meth_obj.comment = comment
      meth_obj.line    = file_content[0, offset].count("\n") + 1

      body
    when :macro_alias
      # with no comment we hope the aliased definition has it and use it's
      # definition

      body = find_body(class_name, args[0], meth_obj, file_content, true)

      return body if body

      @options.warn "No definition for #{meth_name}"
      false
    else # No body, but might still have an override comment
      comment = find_override_comment class_name, meth_obj

      if comment then
        comment.normalize
        find_modifiers comment, meth_obj
        meth_obj.comment = comment

        ''
      else
        @options.warn "No definition for #{meth_name}"
        false
      end
    end
  end

  ##
  # Finds a RDoc::NormalClass or RDoc::NormalModule for +raw_name+

  def find_class(raw_name, name, base_name = nil)
    unless @classes[raw_name]
      if raw_name =~ /^rb_m/
        container = @top_level.add_module RDoc::NormalModule, name
      else
        container = @top_level.add_class RDoc::NormalClass, name
      end
      container.name = base_name if base_name

      container.record_location @top_level
      @classes[raw_name] = container
    end
    @classes[raw_name]
  end

  ##
  # Look for class or module documentation above Init_+class_name+(void),
  # in a Document-class +class_name+ (or module) comment or above an
  # rb_define_class (or module).  If a comment is supplied above a matching
  # Init_ and a rb_define_class the Init_ comment is used.
  #
  #   /*
  #    * This is a comment for Foo
  #    */
  #   Init_Foo(void) {
  #       VALUE cFoo = rb_define_class("Foo", rb_cObject);
  #   }
  #
  #   /*
  #    * Document-class: Foo
  #    * This is a comment for Foo
  #    */
  #   Init_foo(void) {
  #       VALUE cFoo = rb_define_class("Foo", rb_cObject);
  #   }
  #
  #   /*
  #    * This is a comment for Foo
  #    */
  #   VALUE cFoo = rb_define_class("Foo", rb_cObject);

  def find_class_comment class_name, class_mod
    comment = nil

    if @content =~ %r%
        ((?>/\*.*?\*/\s+))
        (static\s+)?
        void\s+
        Init(?:VM)?_(?i:#{class_name})\s*(?:_\(\s*)?\(\s*(?:void\s*)?\)%xm then
      comment = $1.sub(%r%Document-(?:class|module):\s+#{class_name}%, '')
    elsif @content =~ %r%Document-(?:class|module):\s+#{class_name}\s*?
                         (?:<\s+[:,\w]+)?\n((?>.*?\*/))%xm then
      comment = "/*\n#{$1}"
    elsif @content =~ %r%((?>/\*.*?\*/\s+))
                         ([\w\.\s]+\s* = \s+)?rb_define_(class|module)[\t (]*?"(#{class_name})"%xm then
      comment = $1
    elsif @content =~ %r%((?>/\*.*?\*/\s+))
                         ([\w\. \t]+ = \s+)?rb_define_(class|module)_under[\t\w, (]*?"(#{class_name.split('::').last})"%xm then
      comment = $1
    else
      comment = ''
    end

    comment = new_comment comment, @top_level, :c
    comment.normalize

    look_for_directives_in class_mod, comment

    class_mod.add_comment comment, @top_level
  end

  ##
  # Generate a const table

  def gen_const_table file_content
    table = {}
    @content.scan(%r{
      ((?>^\s*/\*.*?\*/\s+))
        rb_define_(\w+)\((?:\s*(?:\w+),)?\s*
                           "(\w+)"\s*,
                           .*?\)\s*;
    | Document-(?:const|global|variable):\s
        ((?:\w+::)*\w+)
        \s*?\n((?>.*?\*/))
    }mxi) do
      case
      when $1 then table[[$2, $3]] = $1
      when $4 then table[$4] = "/*\n" + $5
      end
    end
    table
  end

  ##
  # Finds a comment matching +type+ and +const_name+ either above the
  # comment or in the matching Document- section.

  def find_const_comment(type, const_name, class_name = nil)
    @const_table ||= {}
    @const_table[@content] ||= gen_const_table @content
    table = @const_table[@content]

    comment =
      table[[type, const_name]] ||
      (class_name && table[class_name + "::" + const_name]) ||
      table[const_name] ||
      ''

    new_comment comment, @top_level, :c
  end

  ##
  # Handles modifiers in +comment+ and updates +meth_obj+ as appropriate.

  def find_modifiers comment, meth_obj
    comment.normalize
    comment.extract_call_seq meth_obj

    look_for_directives_in meth_obj, comment
  end

  ##
  # Finds a <tt>Document-method</tt> override for +meth_obj+ on +class_name+

  def find_override_comment class_name, meth_obj
    name = Regexp.escape meth_obj.name
    prefix = Regexp.escape meth_obj.name_prefix

    comment = if @content =~ %r%Document-method:
                                \s+#{class_name}#{prefix}#{name}
                                \s*?\n((?>.*?\*/))%xm then
                "/*#{$1}"
              elsif @content =~ %r%Document-method:
                                   \s#{name}\s*?\n((?>.*?\*/))%xm then
                "/*#{$1}"
              end

    return unless comment

    new_comment comment, @top_level, :c
  end

  ##
  # Creates a new RDoc::Attr +attr_name+ on class +var_name+ that is either
  # +read+, +write+ or both

  def handle_attr(var_name, attr_name, read, write)
    rw = ''
    rw += 'R' if TRUE_VALUES.include?(read)
    rw += 'W' if TRUE_VALUES.include?(write)

    class_name = @known_classes[var_name]

    return unless class_name

    class_obj = find_class var_name, class_name

    return unless class_obj

    comment = find_attr_comment var_name, attr_name
    comment.normalize

    name = attr_name.gsub(/rb_intern(?:_const)?\("([^"]+)"\)/, '\1')

    attr = RDoc::Attr.new '', name, rw, comment

    attr.record_location @top_level
    class_obj.add_attribute attr
    @stats.add_attribute attr
  end

  ##
  # Creates a new RDoc::NormalClass or RDoc::NormalModule based on +type+
  # named +class_name+ in +parent+ which was assigned to the C +var_name+.

  def handle_class_module(var_name, type, class_name, parent, in_module)
    parent_name = @known_classes[parent] || parent

    if in_module then
      enclosure = @classes[in_module] || @store.find_c_enclosure(in_module)

      if enclosure.nil? and enclosure = @known_classes[in_module] then
        enc_type = /^rb_m/ =~ in_module ? :module : :class
        handle_class_module in_module, enc_type, enclosure, nil, nil
        enclosure = @classes[in_module]
      end

      unless enclosure then
        @enclosure_dependencies[in_module] << var_name
        @missing_dependencies[var_name] =
          [var_name, type, class_name, parent, in_module]

        return
      end
    else
      enclosure = @top_level
    end

    if type == :class then
      full_name = if RDoc::ClassModule === enclosure then
                    enclosure.full_name + "::#{class_name}"
                  else
                    class_name
                  end

      if @content =~ %r%Document-class:\s+#{full_name}\s*<\s+([:,\w]+)% then
        parent_name = $1
      end

      cm = enclosure.add_class RDoc::NormalClass, class_name, parent_name
    else
      cm = enclosure.add_module RDoc::NormalModule, class_name
    end

    cm.record_location enclosure.top_level

    find_class_comment cm.full_name, cm

    case cm
    when RDoc::NormalClass
      @stats.add_class cm
    when RDoc::NormalModule
      @stats.add_module cm
    end

    @classes[var_name] = cm
    @known_classes[var_name] = cm.full_name
    @store.add_c_enclosure var_name, cm
  end

  ##
  # Adds constants.  By providing some_value: at the start of the comment you
  # can override the C value of the comment to give a friendly definition.
  #
  #   /* 300: The perfect score in bowling */
  #   rb_define_const(cFoo, "PERFECT", INT2FIX(300));
  #
  # Will override <tt>INT2FIX(300)</tt> with the value +300+ in the output
  # RDoc.  Values may include quotes and escaped colons (\:).

  def handle_constants(type, var_name, const_name, definition)
    class_name = @known_classes[var_name]

    return unless class_name

    class_obj = find_class var_name, class_name, class_name[/::\K[^:]+\z/]

    unless class_obj then
      @options.warn 'Enclosing class or module %p is not known' % [const_name]
      return
    end

    comment = find_const_comment type, const_name, class_name
    comment.normalize

    # In the case of rb_define_const, the definition and comment are in
    # "/* definition: comment */" form.  The literal ':' and '\' characters
    # can be escaped with a backslash.
    if type.downcase == 'const' then
      no_match, new_definition, new_comment = comment.text.split(/(\A.*):/)

      if no_match and no_match.empty? then
        if new_definition.empty? then # Default to literal C definition
          new_definition = definition
        else
          new_definition = new_definition.gsub("\:", ":")
          new_definition = new_definition.gsub("\\", '\\')
        end

        new_definition.sub!(/\A(\s+)/, '')

        new_comment = "#{$1}#{new_comment.lstrip}"

        new_comment = self.new_comment(new_comment, @top_level, :c)

        con = RDoc::Constant.new const_name, new_definition, new_comment
      else
        con = RDoc::Constant.new const_name, definition, comment
      end
    else
      con = RDoc::Constant.new const_name, definition, comment
    end

    con.record_location @top_level
    @stats.add_constant con
    class_obj.add_constant con
  end

  ##
  # Removes #ifdefs that would otherwise confuse us

  def handle_ifdefs_in(body)
    body.gsub(/^#ifdef HAVE_PROTOTYPES.*?#else.*?\n(.*?)#endif.*?\n/m, '\1')
  end

  ##
  # Adds an RDoc::AnyMethod +meth_name+ defined on a class or module assigned
  # to +var_name+.  +type+ is the type of method definition function used.
  # +singleton_method+ and +module_function+ create a singleton method.

  def handle_method(type, var_name, meth_name, function, param_count,
                    source_file = nil)
    class_name = @known_classes[var_name]
    singleton  = @singleton_classes.key? var_name

    @methods[var_name][function] << meth_name

    return unless class_name

    class_obj = find_class var_name, class_name

    if existing_method = class_obj.method_list.find { |m| m.c_function == function }
      add_alias(var_name, class_obj, existing_method.name, meth_name, existing_method.comment)
    end

    if class_obj then
      if meth_name == 'initialize' then
        meth_name = 'new'
        singleton = true
        type = 'method' # force public
      end

      meth_obj = RDoc::AnyMethod.new '', meth_name
      meth_obj.c_function = function
      meth_obj.singleton =
        singleton || %w[singleton_method module_function].include?(type)

      p_count = Integer(param_count) rescue -1

      if source_file then
        file_name = File.join @file_dir, source_file

        if File.exist? file_name then
          file_content = File.read file_name
        else
          @options.warn "unknown source #{source_file} for #{meth_name} in #{@file_name}"
        end
      else
        file_content = @content
      end

      body = find_body class_name, function, meth_obj, file_content

      if body and meth_obj.document_self then
        meth_obj.params = if p_count < -1 then # -2 is Array
                            '(*args)'
                          elsif p_count == -1 then # argc, argv
                            rb_scan_args body
                          else
                            args = (1..p_count).map { |i| "p#{i}" }
                            "(#{args.join ', '})"
                          end


        meth_obj.record_location @top_level

        if meth_obj.section_title
          class_obj.temporary_section = class_obj.add_section(meth_obj.section_title)
        end
        class_obj.add_method meth_obj

        @stats.add_method meth_obj
        meth_obj.visibility = :private if 'private_method' == type
      end
    end
  end

  ##
  # Registers a singleton class +sclass_var+ as a singleton of +class_var+

  def handle_singleton sclass_var, class_var
    class_name = @known_classes[class_var]

    @known_classes[sclass_var]     = class_name
    @singleton_classes[sclass_var] = class_name
  end

  ##
  # Loads the variable map with the given +name+ from the RDoc::Store, if
  # present.

  def load_variable_map map_name
    return {} unless files = @store.cache[map_name]
    return {} unless name_map = files[@file_name]

    class_map = {}

    name_map.each do |variable, name|
      next unless mod = @store.find_class_or_module(name)

      class_map[variable] = if map_name == :c_class_variables then
                              mod
                            else
                              name
                            end
      @known_classes[variable] = name
    end

    class_map
  end

  ##
  # Look for directives in a normal comment block:
  #
  #   /*
  #    * :title: My Awesome Project
  #    */
  #
  # This method modifies the +comment+

  def look_for_directives_in context, comment
    @preprocess.handle comment, context do |directive, param|
      case directive
      when 'main' then
        @options.main_page = param
        ''
      when 'title' then
        @options.default_title = param if @options.respond_to? :default_title=
        ''
      end
    end

    comment
  end

  ##
  # Extracts parameters from the +method_body+ and returns a method
  # parameter string.  Follows 1.9.3dev's scan-arg-spec, see README.EXT

  def rb_scan_args method_body
    method_body =~ /rb_scan_args\((.*?)\)/m
    return '(*args)' unless $1

    $1.split(/,/)[2] =~ /"(.*?)"/ # format argument
    format = $1.split(//)

    lea                                                   gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/text.rb                                                      0000644                 00000000463 15040313414 0014465 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       .shift
    end

    if format.first == '&' then
      block = true
      format.shift
    end

    # if the format string is not empty there's a bug in the C code, ignore it

    args = []
    position = 1

    (1...(position + lead)).each do |index|
      args << "p#{index}"
    end

    position += lead
                                                                                                                                                                                                             gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/ripper_state_lex.rb                                          0000644                 00000037321 15040313414 0017055 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ases
  # from a C file and returns an RDoc::TopLevel for this file

  def scan
    remove_commented_out_lines

    do_classes_and_modules
    do_missing

    do_constants
    do_methods
    do_includes
    do_aliases
    do_attrs

    @store.add_c_variables self

    @top_level
  end

  def new_comment text = nil, location = nil, language = nil
    RDoc::Comment.new(text, location, language).tap do |comment|
      comment.format = @markup
    end
  end
end
                                                   gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/text.rb                                                      0000644                 00000000463 15040313414 0014465 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Indicates this parser is text and doesn't contain code constructs.
#
# Include this module in a RDoc::Parser subclass to make it show up as a file,
# not as part of a class or module.
#--
# This is not named File to avoid overriding ::File

module RDoc::Parser::Text
end

                                                                                                                                                                                                             gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/ripper_state_lex.rb                                          0000644                 00000037321 15040313414 0017055 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require 'ripper'

class RDoc::Parser::RipperStateLex
  # TODO: Remove this constants after Ruby 2.4 EOL
  RIPPER_HAS_LEX_STATE = Ripper::Filter.method_defined?(:state)

  Token = Struct.new(:line_no, :char_no, :kind, :text, :state)

  EXPR_NONE = 0
  EXPR_BEG = 1
  EXPR_END = 2
  EXPR_ENDARG = 4
  EXPR_ENDFN = 8
  EXPR_ARG = 16
  EXPR_CMDARG = 32
  EXPR_MID = 64
  EXPR_FNAME = 128
  EXPR_DOT = 256
  EXPR_CLASS = 512
  EXPR_LABEL = 1024
  EXPR_LABELED = 2048
  EXPR_FITEM = 4096
  EXPR_VALUE = EXPR_BEG
  EXPR_BEG_ANY  =  (EXPR_BEG | EXPR_MID | EXPR_CLASS)
  EXPR_ARG_ANY  =  (EXPR_ARG | EXPR_CMDARG)
  EXPR_END_ANY  =  (EXPR_END | EXPR_ENDARG | EXPR_ENDFN)

  class InnerStateLex < Ripper::Filter
    attr_accessor :lex_state

    def initialize(code)
      @lex_state = EXPR_BEG
      @in_fname = false
      @continue = false
      reset
      super(code)
    end

    def reset
      @command_start = false
      @cmd_state = @command_start
    end

    def on_nl(tok, data)
      case @lex_state
      when EXPR_FNAME, EXPR_DOT
        @continue = true
      else
        @continue = false
        @lex_state = EXPR_BEG unless (EXPR_LABEL & @lex_state) != 0
      end
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_ignored_nl(tok, data)
      case @lex_state
      when EXPR_FNAME, EXPR_DOT
        @continue = true
      else
        @continue = false
        @lex_state = EXPR_BEG unless (EXPR_LABEL & @lex_state) != 0
      end
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_op(tok, data)
      case tok
      when '&', '|', '!', '!=', '!~'
        case @lex_state
        when EXPR_FNAME, EXPR_DOT
          @lex_state = EXPR_ARG
        else
          @lex_state = EXPR_BEG
        end
      when '<<'
        # TODO next token?
        case @lex_state
        when EXPR_FNAME, EXPR_DOT
          @lex_state = EXPR_ARG
        else
          @lex_state = EXPR_BEG
        end
      when '?'
        @lex_state = EXPR_BEG
      when '&&', '||', '+=', '-=', '*=', '**=',
           '&=', '|=', '^=', '<<=', '>>=', '||=', '&&='
        @lex_state = EXPR_BEG
      when '::'
        case @lex_state
        when EXPR_ARG, EXPR_CMDARG
          @lex_state = EXPR_DOT
        when EXPR_FNAME, EXPR_DOT
          @lex_state = EXPR_ARG
        else
          @lex_state = EXPR_BEG
        end
      else
        case @lex_state
        when EXPR_FNAME, EXPR_DOT
          @lex_state = EXPR_ARG
        else
          @lex_state = EXPR_BEG
        end
      end
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_kw(tok, data)
      case tok
      when 'class'
        @lex_state = EXPR_CLASS
        @in_fname = true
      when 'def'
        @lex_state = EXPR_FNAME
        @continue = true
        @in_fname = true
      when 'if', 'unless', 'while', 'until'
        if ((EXPR_MID | EXPR_END | EXPR_ENDARG | EXPR_ENDFN | EXPR_ARG | EXPR_CMDARG) & @lex_state) != 0 # postfix if
          @lex_state = EXPR_BEG | EXPR_LABEL
        else
          @lex_state = EXPR_BEG
        end
      when 'begin', 'case', 'when'
        @lex_state = EXPR_BEG
      when 'return', 'break'
        @lex_state = EXPR_MID
      else
        if @lex_state == EXPR_FNAME
          @lex_state = EXPR_END
        else
          @lex_state = EXPR_END
        end
      end
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_tstring_beg(tok, data)
      @lex_state = EXPR_BEG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_tstring_end(tok, data)
      @lex_state = EXPR_END | EXPR_ENDARG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_CHAR(tok, data)
      @lex_state = EXPR_END
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_period(tok, data)
      @lex_state = EXPR_DOT
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_int(tok, data)
      @lex_state = EXPR_END | EXPR_ENDARG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_float(tok, data)
      @lex_state = EXPR_END | EXPR_ENDARG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_rational(tok, data)
      @lex_state = EXPR_END | EXPR_ENDARG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_imaginary(tok, data)
      @lex_state = EXPR_END | EXPR_ENDARG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_symbeg(tok, data)
      @lex_state = EXPR_FNAME
      @continue = true
      @in_fname = true
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    private def on_variables(event, tok, data)
      if @in_fname
        @lex_state = EXPR_ENDFN
        @in_fname = false
        @continue = false
      elsif @continue
        case @lex_state
        when EXPR_DOT
          @lex_state = EXPR_ARG
        else
          @lex_state = EXPR_ENDFN
          @continue = false
        end
      else
        @lex_state = EXPR_CMDARG
      end
      data << Token.new(lineno, column, event, tok, @lex_state)
    end

    def on_ident(tok, data)
      on_variables(__method__, tok, data)
    end

    def on_ivar(tok, data)
      @lex_state = EXPR_END
      on_variables(__method__, tok, data)
    end

    def on_cvar(tok, data)
      @lex_state = EXPR_END
      on_variables(__method__, tok, data)
    end

    def on_gvar(tok, data)
      @lex_state = EXPR_END
      on_variables(__method__, tok, data)
    end

    def on_backref(tok, data)
      @lex_state = EXPR_END
      on_variables(__method__, tok, data)
    end

    def on_lparen(tok, data)
      @lex_state = EXPR_LABEL | EXPR_BEG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_rparen(tok, data)
      @lex_state = EXPR_ENDFN
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_lbrace(tok, data)
      @lex_state = EXPR_LABEL | EXPR_BEG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_rbrace(tok, data)
      @lex_state = EXPR_ENDARG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_lbracket(tok, data)
      @lex_state = EXPR_LABEL | EXPR_BEG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_rbracket(tok, data)
      @lex_state = EXPR_ENDARG
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_const(tok, data)
      case @lex_state
      when EXPR_FNAME
        @lex_state = EXPR_ENDFN
      when EXPR_CLASS, EXPR_CMDARG, EXPR_MID
        @lex_state = EXPR_ARG
      else
        @lex_state = EXPR_CMDARG
      end
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_sp(tok, data)
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_comma(tok, data)
      @lex_state = EXPR_BEG | EXPR_LABEL if (EXPR_ARG_ANY & @lex_state) != 0
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_comment(tok, data)
      @lex_state = EXPR_BEG unless (EXPR_LABEL & @lex_state) != 0
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_ignored_sp(tok, data)
      @lex_state = EXPR_BEG unless (EXPR_LABEL & @lex_state) != 0
      data << Token.new(lineno, column, __method__, tok, @lex_state)
    end

    def on_heredoc_beg(tok, data)
      data << Token.new(lineno, column, __method__, tok, @lex_state)
      @lex_state = EXPR_END
      data
    end

    def on_heredoc_end(tok, data)
      data << Token.new(lineno, column, __method__, tok, @lex_state)
      @lex_state = EXPR_BEG
      data
    end

    def on_default(event, tok, data)
      reset
      data << Token.new(lineno, column, event, tok, @lex_state)
    end
  end unless RIPPER_HAS_LEX_STATE

  class InnerStateLex < Ripper::Filter
    def initialize(code)
      super(code)
    end

    def on_default(event, tok, data)
      data << Token.new(lineno, column, event, tok, state)
    end
  end if RIPPER_HAS_LEX_STATE

  def get_squashed_tk
    if @buf.empty?
      tk = @tokens.shift
    else
      tk = @buf.shift
    end
    return nil if tk.nil?
    case tk[:kind]
    when :on_symbeg then
      tk = get_symbol_tk(tk)
    when :on_tstring_beg then
      tk = get_string_tk(tk)
    when :on_backtick then
      if (tk[:state] & (EXPR_FNAME | EXPR_ENDFN)) != 0
        @inner_lex.lex_state = EXPR_ARG unless RIPPER_HAS_LEX_STATE
        tk[:kind] = :on_ident
        tk[:state] = Ripper::Lexer.const_defined?(:State) ? Ripper::Lexer::State.new(EXPR_ARG) : EXPR_ARG
      else
        tk = get_string_tk(tk)
      end
    when :on_regexp_beg then
      tk = get_regexp_tk(tk)
    when :on_embdoc_beg then
      tk = get_embdoc_tk(tk)
    when :on_heredoc_beg then
      @heredoc_queue << retrieve_heredoc_info(tk)
      @inner_lex.lex_state = EXPR_END unless RIPPER_HAS_LEX_STATE
    when :on_nl, :on_ignored_nl, :on_comment, :on_heredoc_end then
      if !@heredoc_queue.empty?
        get_heredoc_tk(*@heredoc_queue.shift)
      elsif tk[:text].nil? # :on_ignored_nl sometimes gives nil
        tk[:text] = ''
      end
    when :on_words_beg then
      tk = get_words_tk(tk)
    when :on_qwords_beg then
      tk = get_words_tk(tk)
    when :on_symbols_beg then
      tk = get_words_tk(tk)
    when :on_qsymbols_beg then
      tk = get_words_tk(tk)
    when :on_op then
      if '&.' == tk[:text]
        tk[:kind] = :on_period
      else
        tk = get_op_tk(tk)
      end
    end
    tk
  end

  private def get_symbol_tk(tk)
    is_symbol = true
    symbol_tk = Token.new(tk.line_no, tk.char_no, :on_symbol)
    if ":'" == tk[:text] or ':"' == tk[:text]
      tk1 = get_string_tk(tk)
      symbol_tk[:text] = tk1[:text]
      symbol_tk[:state] = tk1[:state]
    else
      case (tk1 = get_squashed_tk)[:kind]
      when :on_ident
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = tk1[:state]
      when :on_tstring_content
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = get_squashed_tk[:state] # skip :on_tstring_end
      when :on_tstring_end
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = tk1[:state]
      when :on_op
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = tk1[:state]
      when :on_ivar
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = tk1[:state]
      when :on_cvar
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = tk1[:state]
      when :on_gvar
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = tk1[:state]
      when :on_const
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = tk1[:state]
      when :on_kw
        symbol_tk[:text] = ":#{tk1[:text]}"
        symbol_tk[:state] = tk1[:state]
      else
        is_symbol = false
        tk = tk1
      end
    end
    if is_symbol
      tk = symbol_tk
    end
    tk
  end

  private def get_string_tk(tk)
    string = tk[:text]
    state = nil
    kind = :on_tstring
    loop do
      inner_str_tk = get_squashed_tk
      if inner_str_tk.nil?
        break
      elsif :on_tstring_end == inner_str_tk[:kind]
        string = string + inner_str_tk[:text]
        state = inner_str_tk[:state]
        break
      elsif :on_label_end == inner_str_tk[:kind]
        string = string + inner_str_tk[:text]
        state = inner_str_tk[:state]
        kind = :on_symbol
        break
      else
        string = string + inner_str_tk[:text]
        if :on_embexpr_beg == inner_str_tk[:kind] then
          kind = :on_dstring if :on_tstring == kind
        end
      end
    end
    Token.new(tk.line_no, tk.char_no, kind, string, state)
  end

  private def get_regexp_tk(tk)
    string = tk[:text]
    state = nil
    loop do
      inner_str_tk = get_squashed_tk
      if inner_str_tk.nil?
        break
      elsif :on_regexp_end == inner_str_tk[:kind]
        string = string + inner_str_tk[:text]
        state = inner_str_tk[:state]
        break
      else
        string = string + inner_str_tk[:text]
      end
    end
    Token.new(tk.line_no, tk.char_no, :on_regexp, string, state)
  end

  private def get_embdoc_tk(tk)
    string = tk[:text]
    until :on_embdoc_end == (embdoc_tk = get_squashed_tk)[:kind] do
      string = string + embdoc_tk[:text]
    end
    string = string + embdoc_tk[:text]
    Token.new(tk.line_no, tk.char_no, :on_embdoc, string, embdoc_tk.state)
  end

  private def get_heredoc_tk(heredoc_name, indent)
    string = ''
    start_tk = nil
    prev_tk = nil
    until heredoc_end?(heredoc_name, indent, tk = @tokens.shift) do
      start_tk = tk unless start_tk
      if (prev_tk.nil? or "\n" == prev_tk[:text][-1]) and 0 != tk[:char_no]
        string = string + (' ' * tk[:char_no])
      end
      string = string + tk[:text]
      prev_tk = tk
    end
    start_tk = tk unless start_tk
    prev_tk = tk unless prev_tk
    @buf.unshift tk # closing heredoc
    heredoc_tk = Token.new(start_tk.line_no, start_tk.char_no, :on_heredoc, string, prev_tk.state)
    @buf.unshift heredoc_tk
  end

  private def retrieve_heredoc_info(tk)
    name = tk[:text].gsub(/\A<<[-~]?(['"`]?)(.+)\1\z/, '\2')
    indent = tk[:text] =~ /\A<<[-~]/
    [name, indent]
  end

  private def heredoc_end?(name, indent, tk)
    result = false
    if :on_heredoc_end == tk[:kind] then
      tk_name = tk[:text].chomp
      tk_name.lstrip! if indent
      if name == tk_name
        result = true
      end
    end
    result
  end

  private def get_words_tk(tk)
    string = ''
    start_token = tk[:text]
    start_quote = tk[:text].rstrip[-1]
    line_no = tk[:line_no]
    char_no = tk[:char_no]
    state = tk[:state]
    end_quote =
      case start_quote
      when ?( then ?)
                                                                                                                                                                                                                                                                                                                   gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/simple.rb                                                    0000644                 00000002653 15040313414 0014775 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ` | ~]
    if redefinable_operators.include?(tk[:text]) and tk[:state] == EXPR_ARG then
      @inner_lex.lex_state = EXPR_ARG unless RIPPER_HAS_LEX_STATE
      tk[:state] = Ripper::Lexer.const_defined?(:State) ? Ripper::Lexer::State.new(EXPR_ARG) : EXPR_ARG
      tk[:kind] = :on_ident
    elsif tk[:text] =~ /^[-+]$/ then
      tk_ahead = get_squashed_tk
      case tk_ahead[:kind]
      when :on_int, :on_float, :on_rational, :on_imaginary then
        tk[:text] += tk_ahead[:text]
        tk[:kind] = tk_ahead[:kind]
        tk[:state] = tk_ahead[:state]
      when :on_heredoc_beg, :on_tstring, :on_dstring # frozen/non-frozen string literal
        tk[:text] += tk_ahead[:text]
        tk[:kind] = tk_ahead[:kind]
        tk[:state] = tk_ahead[:state]
      else
        @buf.unshift tk_ahead
      end
    end
    tk
  end

  def initialize(code)
    @buf = []
    @heredoc_queue = []
    @inner_lex = InnerStateLex.new(code)
    @tokens = @inner_lex.parse([])
  end

  def self.parse(code)
    lex = self.new(code)
    tokens = []
    begin
      while tk = lex.get_squashed_tk
        tokens.push tk
      end
    rescue StopIteration
    end
    tokens
  end

  def self.end?(token)
    (token[:state] & EXPR_END)
  end
end
                                                                                                                                                                                                                                                                                                               gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/ruby_tools.rb                                                0000644                 00000005206 15040313414 0015702 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Parse a non-source file. We basically take the whole thing as one big
# comment.

class RDoc::Parser::Simple < RDoc::Parser

  include RDoc::Parser::Text

  parse_files_matching(//)

  attr_reader :content # :nodoc:

  ##
  # Prepare to parse a plain file

  def initialize(top_level, file_name, content, options, stats)
    super

    preprocess = RDoc::Markup::PreProcess.new @file_name, @options.rdoc_include

    @content = preprocess.handle @content, @top_level
  end

  ##
  # Extract the file contents and attach them to the TopLevel as a comment

  def scan
    comment = remove_coding_comment @content
    comment = remove_private_comment comment

    comment = RDoc::Comment.new comment, @top_level

    @top_level.comment = comment
    @top_level
  end

  ##
  # Removes the encoding magic comment from +text+

  def remove_coding_comment text
    text.sub(/\A# .*coding[=:].*$/, '')
  end

  ##
  # Removes private comments.
  #
  # Unlike RDoc::Comment#remove_private this implementation only looks for two
  # dashes at the beginning of the line.  Three or more dashes are considered
  # to be a rule and ignored.

  def remove_private_comment comment
    # Workaround for gsub encoding for Ruby 1.9.2 and earlier
    empty = ''
    empty = RDoc::Encoding.change_encoding empty, comment.encoding

    comment = comment.gsub(%r%^--\n.*?^\+\+\n?%m, empty)
    comment.sub(%r%^--\n.*%m, empty)
  end

end
                                                                                     gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/ruby_tools.rb                                                0000644                 00000005206 15040313414 0015702 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Collection of methods for writing parsers

module RDoc::Parser::RubyTools

  ##
  # Adds a token listener +obj+, but you should probably use token_listener

  def add_token_listener(obj)
    @token_listeners ||= []
    @token_listeners << obj
  end

  ##
  # Fetches the next token from the scanner

  def get_tk
    tk = nil

    if @tokens.empty? then
      if @scanner_point >= @scanner.size
        return nil
      else
        tk = @scanner[@scanner_point]
        @scanner_point += 1
        @read.push tk[:text]
      end
    else
      @read.push @unget_read.shift
      tk = @tokens.shift
    end

                                                                                                                                                                                                                                                                                                                                                                                             gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/markdown.rb                                                  0000644                 00000000732 15040313414 0015322 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       
  end

  ##
  # Removes the token listener +obj+

  def remove_token_listener(obj)
    @token_listeners.delete(obj)
  end

  ##
  # Resets the tools

  def reset
    @read       = []
    @tokens     = []
    @unget_read = []
    @nest = 0
    @scanner_point = 0
  end

  ##
  # Skips whitespace tokens including newlines

  def skip_tkspace
    tokens = []

    while (tk = get_tk) and (:on_sp == tk[:kind] or :on_nl == tk[:kind] or :on_ignored_nl == tk[:kind]) do
      to                                      gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/ruby.rb                                                      0000644                 00000166156 15040313414 0014476 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       his token from any listeners
    @token_listeners.each do |obj|
      obj.pop_token
    end if @token_listeners

    nil
  end

end


                                                                                                                                                                                                                                                                                                                                                                                          gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/markdown.rb                                                  0000644                 00000000732 15040313414 0015322 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Parse a Markdown format file.  The parsed RDoc::Markup::Document is attached
# as a file comment.

class RDoc::Parser::Markdown < RDoc::Parser

  include RDoc::Parser::Text

  parse_files_matching(/\.(md|markdown)(?:\.[^.]+)?$/)

  ##
  # Creates an Markdown-format TopLevel for the given file.

  def scan
    comment = RDoc::Comment.new @content, @top_level
    comment.format = 'markdown'

    @top_level.comment = comment
  end

end


                                      gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/ruby.rb                                                      0000644                 00000166156 15040313414 0014476 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# This file contains stuff stolen outright from:
#
#   rtags.rb -
#   ruby-lex.rb - ruby lexcal analyzer
#   ruby-token.rb - ruby tokens
#       by Keiju ISHITSUKA (Nippon Rational Inc.)
#

##
# Extracts code elements from a source file returning a TopLevel object
# containing the constituent file elements.
#
# This file is based on rtags
#
# RubyParser understands how to document:
# * classes
# * modules
# * methods
# * constants
# * aliases
# * private, public, protected
# * private_class_function, public_class_function
# * private_constant, public_constant
# * module_function
# * attr, attr_reader, attr_writer, attr_accessor
# * extra accessors given on the command line
# * metaprogrammed methods
# * require
# * include
#
# == Method Arguments
#
#--
# NOTE: I don't think this works, needs tests, remove the paragraph following
# this block when known to work
#
# The parser extracts the arguments from the method definition.  You can
# override this with a custom argument definition using the :args: directive:
#
#   ##
#   # This method tries over and over until it is tired
#
#   def go_go_go(thing_to_try, tries = 10) # :args: thing_to_try
#     puts thing_to_try
#     go_go_go thing_to_try, tries - 1
#   end
#
# If you have a more-complex set of overrides you can use the :call-seq:
# directive:
#++
#
# The parser extracts the arguments from the method definition.  You can
# override this with a custom argument definition using the :call-seq:
# directive:
#
#   ##
#   # This method can be called with a range or an offset and length
#   #
#   # :call-seq:
#   #   my_method(Range)
#   #   my_method(offset, length)
#
#   def my_method(*args)
#   end
#
# The parser extracts +yield+ expressions from method bodies to gather the
# yielded argument names.  If your method manually calls a block instead of
# yielding or you want to override the discovered argument names use
# the :yields: directive:
#
#   ##
#   # My method is awesome
#
#   def my_method(&block) # :yields: happy, times
#     block.call 1, 2
#   end
#
# == Metaprogrammed Methods
#
# To pick up a metaprogrammed method, the parser looks for a comment starting
# with '##' before an identifier:
#
#   ##
#   # This is a meta-programmed method!
#
#   add_my_method :meta_method, :arg1, :arg2
#
# The parser looks at the token after the identifier to determine the name, in
# this example, :meta_method.  If a name cannot be found, a warning is printed
# and 'unknown is used.
#
# You can force the name of a method using the :method: directive:
#
#   ##
#   # :method: some_method!
#
# By default, meta-methods are instance methods.  To indicate that a method is
# a singleton method instead use the :singleton-method: directive:
#
#   ##
#   # :singleton-method:
#
# You can also use the :singleton-method: directive with a name:
#
#   ##
#   # :singleton-method: some_method!
#
# You can define arguments for metaprogrammed methods via either the
# :call-seq:, :arg: or :args: directives.
#
# Additionally you can mark a method as an attribute by
# using :attr:, :attr_reader:, :attr_writer: or :attr_accessor:.  Just like
# for :method:, the name is optional.
#
#   ##
#   # :attr_reader: my_attr_name
#
# == Hidden methods and attributes
#
# You can provide documentation for methods that don't appear using
# the :method:, :singleton-method: and :attr: directives:
#
#   ##
#   # :attr_writer: ghost_writer
#   # There is an attribute here, but you can't see it!
#
#   ##
#   # :method: ghost_method
#   # There is a method here, but you can't see it!
#
#   ##
#   # this is a comment for a regular method
#
#   def regular_method() end
#
# Note that by default, the :method: directive will be ignored if there is a
# standard rdocable item following it.

require 'ripper'
require_relative 'ripper_state_lex'

class RDoc::Parser::Ruby < RDoc::Parser

  parse_files_matching(/\.rbw?$/)

  include RDoc::TokenStream
  include RDoc::Parser::RubyTools

  ##
  # RDoc::NormalClass type

  NORMAL = "::"

  ##
  # RDoc::SingleClass type

  SINGLE = "<<"

  ##
  # Creates a new Ruby parser.

  def initialize(top_level, file_name, content, options, stats)
    super

    content = handle_tab_width(content)

    @size = 0
    @token_listeners = nil
    content = RDoc::Encoding.remove_magic_comment content
    @scanner = RDoc::Parser::RipperStateLex.parse(content)
    @content = content
    @scanner_point = 0
    @prev_seek = nil
    @markup = @options.markup
    @track_visibility = :nodoc != @options.visibility
    @encoding = @options.encoding

    reset
  end

  def tk_nl?(tk)
    :on_nl == tk[:kind] or :on_ignored_nl == tk[:kind]
  end

  ##
  # Retrieves the read token stream and replaces +pattern+ with +replacement+
  # using gsub.  If the result is only a ";" returns an empty string.

  def get_tkread_clean pattern, replacement # :nodoc:
    read = get_tkread.gsub(pattern, replacement).strip
    return '' if read == ';'
    read
  end

  ##
  # Extracts the visibility information for the visibility token +tk+
  # and +single+ class type identifier.
  #
  # Returns the visibility type (a string), the visibility (a symbol) and
  # +singleton+ if the methods following should be converted to singleton
  # methods.

  def get_visibility_information tk, single # :nodoc:
    vis_type  = tk[:text]
    singleton = single == SINGLE

    vis =
      case vis_type
      when 'private'   then :private
      when 'protected' then :protected
      when 'public'    then :public
      when 'private_class_method' then
        singleton = true
        :private
      when 'public_class_method' then
        singleton = true
        :public
      when 'module_function' then
        singleton = true
        :public
      else
        raise RDoc::Error, "Invalid visibility: #{tk.name}"
      end

    return vis_type, vis, singleton
  end

  ##
  # Look for the first comment in a file that isn't a shebang line.

  def collect_first_comment
    skip_tkspace
    comment = ''.dup
    comment = RDoc::Encoding.change_encoding comment, @encoding if @encoding
    first_line = true
    first_comment_tk_kind = nil
    line_no = nil

    tk = get_tk

    while tk && (:on_comment == tk[:kind] or :on_embdoc == tk[:kind])
      comment_body = retrieve_comment_body(tk)
      if first_line and comment_body =~ /\A#!/ then
        skip_tkspace
        tk = get_tk
      elsif first_line and comment_body =~ /\A#\s*-\*-/ then
        first_line = false
        skip_tkspace
        tk = get_tk
      else
        break if first_comment_tk_kind and not first_comment_tk_kind === tk[:kind]
        first_comment_tk_kind = tk[:kind]

        line_no = tk[:line_no] if first_line
        first_line = false
        comment << comment_body
        tk = get_tk

        if :on_nl === tk then
          skip_tkspace_without_nl
          tk = get_tk
        end
      end
    end

    unget_tk tk

    new_comment comment, line_no
  end

  ##
  # Consumes trailing whitespace from the token stream

  def consume_trailing_spaces # :nodoc:
    skip_tkspace_without_nl
  end

  ##
  # Creates a new attribute in +container+ with +name+.

  def create_attr container, single, name, rw, comment # :nodoc:
    att = RDoc::Attr.new get_tkread, name, rw, comment, single == SINGLE
    record_location att

    container.add_attribute att
    @stats.add_attribute att

    att
  end

  ##
  # Creates a module alias in +container+ at +rhs_name+ (or at the top-level
  # for "::") with the name from +constant+.

  def create_module_alias container, constant, rhs_name # :nodoc:
    mod = if rhs_name =~ /^::/ then
            @store.find_class_or_module rhs_name
          else
            container.find_module_named rhs_name
          end

    container.add_module_alias mod, rhs_name, constant, @top_level
  end

  ##
  # Aborts with +msg+

  def error(msg)
    msg = make_message msg

    abort msg
  end

  ##
  # Looks for a true or false token.

  def get_bool
    skip_tkspace
    tk = get_tk
    if :on_kw == tk[:kind] && 'true' == tk[:text]
      true
    elsif :on_kw == tk[:kind] && ('false' == tk[:text] || 'nil' == tk[:text])
      false
    else
      unget_tk tk
      true
    end
  end

  ##
  # Look for the name of a class of module (optionally with a leading :: or
  # with :: separated named) and return the ultimate name, the associated
  # container, and the given name (with the ::).

  def get_class_or_module container, ignore_constants = false
    skip_tkspace
    name_t = get_tk
    given_name = ''.dup

    # class ::A -> A is in the top level
    if :on_op == name_t[:kind] and '::' == name_t[:text] then # bug
      name_t = get_tk
      container = @top_level
      given_name << '::'
    end

    skip_tkspace_without_nl
    given_name << name_t[:text]

    is_self = name_t[:kind] == :on_op && name_t[:text] == '<<'
    new_modules = []
    while !is_self && (tk = peek_tk) and :on_op == tk[:kind] and '::' == tk[:text] do
      prev_container = container
      container = container.find_module_named name_t[:text]
      container ||=
        if ignore_constants then
          c = RDoc::NormalModule.new name_t[:text]
          c.store = @store
          new_modules << [prev_container, c]
          c
        else
          c = prev_container.add_module RDoc::NormalModule, name_t[:text]
          c.ignore unless prev_container.document_children
          @top_level.add_to_classes_or_modules c
          c
        end

      record_location container

      get_tk
      skip_tkspace
      if :on_lparen == peek_tk[:kind] # ProcObjectInConstant::()
        parse_method_or_yield_parameters
        break
      end
      name_t = get_tk
      unless :on_const == name_t[:kind] || :on_ident == name_t[:kind]
        raise RDoc::Error, "Invalid class or module definition: #{given_name}"
      end
      if prev_container == container and !ignore_constants
        given_name = name_t[:text]
      else
        given_name << '::' + name_t[:text]
      end
    end

    skip_tkspace_without_nl

    return [container, name_t, given_name, new_modules]
  end

  ##
  # Skip opening parentheses and yield the block.
  # Skip closing parentheses too when exists.

  def skip_parentheses(&block)
    left_tk = peek_tk

    if :on_lparen == left_tk[:kind]
      get_tk

      ret = skip_parentheses(&block)

      right_tk = peek_tk
      if :on_rparen == right_tk[:kind]
        get_tk
      end

      ret
    else
      yield
    end
  end

  ##
  # Return a superclass, which can be either a constant of an expression

  def get_class_specification
    tk = peek_tk
    if tk.nil?
      return ''
    elsif :on_kw == tk[:kind] && 'self' == tk[:text]
      return 'self'
    elsif :on_gvar == tk[:kind]
      return ''
    end

    res = get_constant

    skip_tkspace_without_nl

    get_tkread # empty out read buffer

    tk = get_tk
    return res unless tk

    case tk[:kind]
    when :on_nl, :on_comment, :on_embdoc, :on_semicolon then
      unget_tk(tk)
      return res
    end

    res += parse_call_parameters(tk)
    res
  end

  ##
  # Parse a constant, which might be qualified by one or more class or module
  # names

  def get_constant
    res = ""
    skip_tkspace_without_nl
    tk = get_tk

    while tk && ((:on_op == tk[:kind] && '::' == tk[:text]) || :on_const == tk[:kind]) do
      res += tk[:text]
      tk = get_tk
    end

    unget_tk(tk)
    res
  end

  ##
  # Get an included module that may be surrounded by parens

  def get_included_module_with_optional_parens
    skip_tkspace_without_nl
    get_tkread
    tk = get_tk
    end_token = get_end_token tk
    return '' unless end_token

    nest = 0
    continue = false
    only_constant = true

    while tk != nil do
      is_element_of_constant = false
      case tk[:kind]
      when :on_semicolon then
        break if nest == 0
      when :on_lbracket then
        nest += 1
      when :on_rbracket then
        nest -= 1
      when :on_lbrace then
        nest += 1
      when :on_rbrace then
        nest -= 1
        if nest <= 0
          # we might have a.each { |i| yield i }
          unget_tk(tk) if nest < 0
          break
        end
      when :on_lparen then
        nest += 1
      when end_token[:kind] then
        if end_token[:kind] == :on_rparen
          nest -= 1
          break if nest <= 0
        else
          break if nest <= 0
        end
      when :on_rparen then
        nest -= 1
      when :on_comment, :on_embdoc then
        @read.pop
        if :on_nl == end_token[:kind] and "\n" == tk[:text][-1] and
          (!continue or (tk[:state] & RDoc::Parser::RipperStateLex::EXPR_LABEL) != 0) then
          break if !continue and nest <= 0
        end
      when :on_comma then
        continue = true
      when :on_ident then
        continue = false if continue
      when :on_kw then
        case tk[:text]
        when 'def', 'do', 'case', 'for', 'begin', 'class', 'module'
          nest += 1
        when 'if', 'unless', 'while', 'until', 'rescue'
          # postfix if/unless/while/until/rescue must be EXPR_LABEL
          nest += 1 unless (tk[:state] & RDoc::Parser::RipperStateLex::EXPR_LABEL) != 0
        when 'end'
          nest -= 1
          break if nest == 0
        end
      when :on_const then
        is_element_of_constant = true
      when :on_op then
        is_element_of_constant = true if '::' == tk[:text]
      end
      only_constant = false unless is_element_of_constant
      tk = get_tk
    end

    if only_constant
      get_tkread_clean(/\s+/, ' ')
    else
      ''
    end
  end

  ##
  # Little hack going on here. In the statement:
  #
  #   f = 2*(1+yield)
  #
  # We see the RPAREN as the next token, so we need to exit early.  This still
  # won't catch all cases (such as "a = yield + 1"

  def get_end_token tk # :nodoc:
    case tk[:kind]
    when :on_lparen
      token = RDoc::Parser::RipperStateLex::Token.new
      token[:kind] = :on_rparen
      token[:text] = ')'
      token
    when :on_rparen
      nil
    else
      token = RDoc::Parser::RipperStateLex::Token.new
      token[:kind] = :on_nl
      token[:text] = "\n"
      token
    end
  end

  ##
  # Retrieves the method container for a singleton method.

  def get_method_container container, name_t # :nodoc:
    prev_container = container
    container = container.find_module_named(name_t[:text])

    unless container then
      constant = prev_container.constants.find do |const|
        const.name == name_t[:text]
      end

      if constant then
        parse_method_dummy prev_container
        return
      end
    end

    unless container then
      # TODO seems broken, should starting at Object in @store
      obj = name_t[:text].split("::").inject(Object) do |state, item|
        state.const_get(item)
      end rescue nil

      type = obj.class == Class ? RDoc::NormalClass : RDoc::NormalModule

      unless [Class, Module].include?(obj.class) then
        warn("Couldn't find #{name_t[:text]}. Assuming it's a module")
      end

      if type == RDoc::NormalClass then
        sclass = obj.superclass ? obj.superclass.name : nil
        container = prev_container.add_class type, name_t[:text], sclass
      else
        container = prev_container.add_module type, name_t[:text]
      end

      record_location container
    end

    container
  end

  ##
  # Extracts a name or symbol from the token stream.

  def get_symbol_or_name
    tk = get_tk
    case tk[:kind]
    when :on_symbol then
      text = tk[:text].sub(/^:/, '')

      next_tk = peek_tk
      if next_tk && :on_op == next_tk[:kind] && '=' == next_tk[:text] then
        get_tk
        text << '='
      end

      text
    when :on_ident, :on_const, :on_gvar, :on_cvar, :on_ivar, :on_op, :on_kw then
      tk[:text]
    when :on_tstring, :on_dstring then
      tk[:text][1..-2]
    else
      raise RDoc::Error, "Name or symbol expected (got #{tk})"
    end
  end

  ##
  # Marks containers between +container+ and +ancestor+ as ignored

  def suppress_parents container, ancestor # :nodoc:
    while container and container != ancestor do
      container.suppress unless container.documented?
      container = container.parent
    end
  end

  ##
  # Look for directives in a normal comment block:
  #
  #   # :stopdoc:
  #   # Don't display comment from this point forward
  #
  # This routine modifies its +comment+ parameter.

  def look_for_directives_in container, comment
    @preprocess.handle comment, container do |directive, param|
      case directive
      when 'method', 'singleton-method',
           'attr', 'attr_accessor', 'attr_reader', 'attr_writer' then
        false # handled elsewhere
      when 'section' then
        break unless container.kind_of?(RDoc::Context)
        container.set_current_section param, comment.dup
        comment.text = ''
        break
      end
    end

    comment.remove_private
  end

  ##
  # Adds useful info about the parser to +message+

  def make_message message
    prefix = "#{@file_name}:".dup

    tk = peek_tk
    prefix << "#{tk[:line_no]}:#{tk[:char_no]}:" if tk

    "#{prefix} #{message}"
  end

  ##
  # Creates a comment with the correct format

  def new_comment comment, line_no = nil
    c = RDoc::Comment.new comment, @top_level, :ruby
    c.line = line_no
    c.format = @markup
    c
  end

  ##
  # Creates an RDoc::Attr for the name following +tk+, setting the comment to
  # +comment+.

  def parse_attr(context, single, tk, comment)
    line_no = tk[:line_no]

    args = parse_symbol_arg 1
    if args.size > 0 then
      name = args[0]
      rw = "R"
      skip_tkspace_without_nl
      tk = get_tk

      if :on_comma == tk[:kind] then
        rw = "RW" if get_bool
      else
        unget_tk tk
      end

      att = create_attr context, single, name, rw, comment
      att.line   = line_no

      read_documentation_modifiers att, RDoc::ATTR_MODIFIERS
    else
      warn "'attr' ignored - looks like a variable"
    end
  end

  ##
  # Creates an RDoc::Attr for each attribute listed after +tk+, setting the
  # comment for each to +comment+.

  def parse_attr_accessor(context, single, tk, comment)
    line_no = tk[:line_no]

    args = parse_symbol_arg
    rw = "?"

    tmp = RDoc::CodeObject.new
    read_documentation_modifiers tmp, RDoc::ATTR_MODIFIERS
    # TODO In most other places we let the context keep track of document_self
    # and add found items appropriately but here we do not.  I'm not sure why.
    return if @track_visibility and not tmp.document_self

    case tk[:text]
    when "attr_reader"   then rw = "R"
    when "attr_writer"   then rw = "W"
    when "attr_accessor" then rw = "RW"
    else
      rw = '?'
    end

    for name in args
      att = create_attr context, single, name, rw, comment
      att.line   = line_no
    end
  end

  ##
  # Parses an +alias+ in +context+ with +comment+

  def parse_alias(context, single, tk, comment)
    line_no = tk[:line_no]

    skip_tkspace

    if :on_lparen === peek_tk[:kind] then
      get_tk
      skip_tkspace
    end

    new_name = get_symbol_or_name

    skip_tkspace
    if :on_comma === peek_tk[:kind] then
      get_tk
      skip_tkspace
    end

    begin
      old_name = get_symbol_or_name
    rescue RDoc::Error
      return
    end

    al = RDoc::Alias.new(get_tkread, old_name, new_name, comment,
                         single == SINGLE)
    record_location al
    al.line   = line_no

    read_documentation_modifiers al, RDoc::ATTR_MODIFIERS
    context.add_alias al
    @stats.add_alias al

    al
  end

  ##
  # Extracts call parameters from the token stream.

  def parse_call_parameters(tk)
    end_token = case tk[:kind]
                when :on_lparen
                  :on_rparen
                when :on_rparen
                  return ""
                else
                  :on_nl
                end
    nest = 0

    loop do
      break if tk.nil?
      case tk[:kind]
      when :on_semicolon
        break
      when :on_lparen
        nest += 1
      when end_token
        if end_token == :on_rparen
          nest -= 1
          break if RDoc::Parser::RipperStateLex.end?(tk) and nest <= 0
        else
          break if RDoc::Parser::RipperStateLex.end?(tk)
        end
      when :on_comment, :on_embdoc
        unget_tk(tk)
        break
      when :on_op
        if tk[:text] =~ /^(.{1,2})?=$/
          unget_tk(tk)
          break
        end
      end
      tk = get_tk
    end

    get_tkread_clean "\n", " "
  end

  ##
  # Parses a class in +context+ with +comment+

  def parse_class container, single, tk, comment
    line_no = tk[:line_no]

    declaration_context = container
    container, name_t, given_name, = get_class_or_module container

    if name_t[:kind] == :on_const
      cls = parse_class_regular container, declaration_context, single,
        name_t, given_name, comment
    elsif name_t[:kind] == :on_op && name_t[:text] == '<<'
      case name = skip_parentheses { get_class_specification }
      when 'self', container.name
        read_documentation_modifiers cls, RDoc::CLASS_MODIFIERS
        parse_statements container, SINGLE
        return # don't update line
      else
        cls = parse_class_singleton container, name, comment
      end
    else
      warn "Expected class name or '<<'. Got #{name_t[:kind]}: #{name_t[:text].inspect}"
      return
    end

    cls.line   = line_no

    # after end modifiers
    read_documentation_modifiers cls, RDoc::CLASS_MODIFIERS

    cls
  end

  ##
  # Parses and creates a regular class

  def parse_class_regular container, declaration_context, single, # :nodoc:
                          name_t, given_name, comment
    superclass = '::Object'

    if given_name =~ /^::/ then
      declaration_context = @top_level
      given_name = $'
    end

    tk = peek_tk
    if tk[:kind] == :on_op && tk[:text] == '<' then
      get_tk
      skip_tkspace
      superclass = get_class_specification
      superclass = '(unknown)' if superclass.empty?
    end

    cls_type = single == SINGLE ? RDoc::SingleClass : RDoc::NormalClass
    cls = declaration_context.add_class cls_type, given_name, superclass
    cls.ignore unless container.document_children

    read_documentation_modifiers cls, RDoc::CLASS_MODIFIERS
    record_location cls

    cls.add_comment comment, @top_level

    @top_level.add_to_classes_or_modules cls
    @stats.add_class cls

    suppress_parents container, declaration_context unless cls.document_self

    parse_statements cls

    cls
  end

  ##
  # Parses a singleton class in +container+ with the given +name+ and
  # +comment+.

  def parse_class_singleton container, name, comment # :nodoc:
    other = @store.find_class_named name

    unless other then
      if name =~ /^::/ then
        name = $'
        container = @top_level
      end

      other = container.add_module RDoc::NormalModule, name
      record_location other

      # class << $gvar
      other.ignore if name.empty?

      other.add_comment comment, @top_level
    end

    # notify :nodoc: all if not a constant-named class/module
    # (and remove any comment)
    unless name =~ /\A(::)?[A-Z]/ then
      other.document_self = nil
      other.document_children = false
      other.clear_comment
    end

    @top_level.add_to_classes_or_modules other
    @stats.add_class other

    read_documentation_modifiers other, RDoc::CLASS_MODIFIERS
    parse_statements(other, SINGLE)

    other
  end

  ##
  # Parses a constant in +context+ with +comment+.  If +ignore_constants+ is
  # true, no found constants will be added to RDoc.

  def parse_constant container, tk, comment, ignore_constants = false
    line_no = tk[:line_no]

    name = tk[:text]
    skip_tkspace_without_nl

    return unless name =~ /^\w+$/

    new_modules = []
    if :on_op == peek_tk[:kind] && '::' == peek_tk[:text] then
      unget_tk tk

      container, name_t, _, new_modules = get_class_or_module container, true

      name = name_t[:text]
    end

    is_array_or_hash = false
    if peek_tk && :on_lbracket == peek_tk[:kind]
      get_tk
      nest = 1
      while bracket_tk = get_tk
        case bracket_tk[:kind]
        when :on_lbracket
          nest += 1
        when :on_rbracket
          nest -= 1
          break if nest == 0
        end
      end
      skip_tkspace_without_nl
      is_array_or_hash = true
    end

    unless peek_tk && :on_op == peek_tk[:kind] && '=' == peek_tk[:text] then
      return false
    end
    get_tk

    unless ignore_constants
      new_modules.each do |prev_c, new_module|
        prev_c.add_module_by_normal_module new_module
        new_module.ignore unless prev_c.document_children
        @top_level.add_to_classes_or_modules new_module
      end
    end

    value = ''
    con = RDoc::Constant.new name, value, comment

    body = parse_constant_body container, con, is_array_or_hash

    return unless body

    con.value = body
    record_location con
    con.line   = line_no
    read_documentation_modifiers con, RDoc::CONSTANT_MODIFIERS

    return if is_array_or_hash

    @stats.add_constant con
    container.add_constant con

    true
  end

  def parse_constant_body container, constant, is_array_or_hash # :nodoc:
    nest     = 0
    rhs_name = ''.dup

    get_tkread

    tk = get_tk

    body = nil
    loop do
      break if tk.nil?
      if :on_semicolon == tk[:kind] then
        break if nest <= 0
      elsif [:on_tlambeg, :on_lparen, :on_lbrace, :on_lbracket].include?(tk[:kind]) then
        nest += 1
      elsif (:on_kw == tk[:kind] && 'def' == tk[:text]) then
        nest += 1
      elsif (:on_kw == tk[:kind] && %w{do if unless case begin}.include?(tk[:text])) then
        if (tk[:state] & RDoc::Parser::RipperStateLex::EXPR_LABEL) == 0
          nest += 1
        end
      elsif [:on_rparen, :on_rbrace, :on_rbracket].include?(tk[:kind]) ||
            (:on_kw == tk[:kind] && 'end' == tk[:text]) then
        nest -= 1
      elsif (:on_comment == tk[:kind] or :on_embdoc == tk[:kind]) then
        unget_tk tk
        if nest <= 0 and RDoc::Parser::RipperStateLex.end?(tk) then
          body = get_tkread_clean(/^[ \t]+/, '')
          read_documentation_modifiers constant, RDoc::CONSTANT_MODIFIERS
          break
        else
          read_documentation_modifiers constant, RDoc::CONSTANT_MODIFIERS
        end
      elsif :on_const == tk[:kind] then
        rhs_name << tk[:text]

        next_tk = peek_tk
        if nest <= 0 and (next_tk.nil? || :on_nl == next_tk[:kind]) then
          create_module_alias container, constant, rhs_name unless is_array_or_hash
          break
        end
      elsif :on_nl == tk[:kind] then
        if nest <= 0 and RDoc::Parser::RipperStateLex.end?(tk) then
          unget_tk tk
          break
        end
      elsif :on_op == tk[:kind] && '::' == tk[:text]
        rhs_name << '::'
      end
      tk = get_tk
    end

    body ? body : get_tkread_clean(/^[ \t]+/, '')
  end

  ##
  # Generates an RDoc::Method or RDoc::Attr from +comment+ by looking for
  # :method: or :attr: directives in +comment+.

  def parse_comment container, tk, comment
    return parse_comment_tomdoc container, tk, comment if @markup == 'tomdoc'
    column  = tk[:char_no]
    line_no = comment.line.nil? ? tk[:line_no] : comment.line

    comment.text = comment.text.sub(/(^# +:?)(singleton-)(method:)/, '\1\3')
    singleton = !!$~

    co =
      if (comment.text = comment.text.sub(/^# +:?method: *(\S*).*?\n/i, '')) && !!$~ then
        line_no += $`.count("\n")
        parse_comment_ghost container, comment.text, $1, column, line_no, comment
      elsif (comment.text = comment.text.sub(/# +:?(attr(_reader|_writer|_accessor)?): *(\S*).*?\n/i, '')) && !!$~ then
        parse_comment_attr container, $1, $3, comment
      end

    if co then
      co.singleton = singleton
      co.line      = line_no
    end

    true
  end

  ##
  # Parse a comment that is describing an attribute in +container+ with the
  # given +name+ and +comment+.

  def parse_comment_attr container, type, name, comment # :nodoc:
    return if name.empty?

    rw = case type
         when 'attr_reader' then 'R'
         when 'attr_writer' then 'W'
         else 'RW'
         end

    create_attr container, NORMAL, name, rw, comment
  end

  def parse_comment_ghost container, text, name, column, line_no, # :nodoc:
                          comment
    name = nil if name.empty?

    meth = RDoc::GhostMethod.new get_tkread, name
    record_location meth

    meth.start_collecting_tokens
    indent = RDoc::Parser::RipperStateLex::Token.new(1, 1, :on_sp, ' ' * column)
    position_comment = RDoc::Parser::RipperStateLex::Token.new(line_no, 1, :on_comment)
    position_comment[:text] = "# File #{@top_level.relative_name}, line #{line_no}"
    newline = RDoc::Parser::RipperStateLex::Token.new(0, 0, :on_nl, "\n")
    meth.add_tokens [position_comment, newline, indent]

    meth.params =
      if text.sub!(/^#\s+:?args?:\s*(.*?)\s*$/i, '') then
        $1
      else
        ''
      end

    comment.normalize
    comment.extract_call_seq meth

    return unless meth.name

    container.add_method meth

    meth.comment = comment

    @stats.add_method meth

    meth
  end

  ##
  # Creates an RDoc::Method on +container+ from +comment+ if there is a
  # Signature section in the comment

  def parse_comment_tomdoc container, tk, comment
    return unless signature = RDoc::TomDoc.signature(comment)
    column  = tk[:char_no]
    line_no = tk[:line_no]

    name, = signature.split %r%[ \(]%, 2

    meth = RDoc::GhostMethod.new get_tkread, name
    record_location meth
    meth.line      = line_no

    meth.start_collecting_tokens
    indent = RDoc::Parser::RipperStateLex::Token.new(1, 1, :on_sp, ' ' * column)
    position_comment = RDoc::Parser::RipperStateLex::Token.new(line_no, 1, :on_comment)
    position_comment[:text] = "# File #{@top_level.relative_name}, line #{line_no}"
    newline = RDoc::Parser::RipperStateLex::Token.new(0, 0, :on_nl, "\n")
    meth.add_tokens [position_comment, newline, indent]

    meth.call_seq = signature

    comment.normalize

    return unless meth.name

    container.add_method meth

    meth.comment = comment

    @stats.add_method meth
  end

  ##
  # Parses an +include+ or +extend+, indicated by the +klass+ and adds it to
  # +container+ # with +comment+

  def parse_extend_or_include klass, container, comment # :nodoc:
    loop do
      skip_tkspace_comment

      name = get_included_module_with_optional_parens

      unless name.empty? then
        obj = container.add klass, name, comment
        record_location obj
      end

      return if peek_tk.nil? || :on_comma != peek_tk[:kind]

      get_tk
    end
  end

  ##
  # Parses an +included+ with a block feature of ActiveSupport::Concern.

  def parse_included_with_activesupport_concern container, comment # :nodoc:
    skip_tkspace_without_nl
    tk = get_tk
    unless tk[:kind] == :on_lbracket || (tk[:kind] == :on_kw && tk[:text] == 'do')
      unget_tk tk
      return nil # should be a block
    end

    parse_statements container

    container
  end

  ##
  # Parses identifiers that can create new methods or change visibility.
  #
  # Returns true if the comment was not consumed.

  def parse_identifier container, single, tk, comment # :nodoc:
    case tk[:text]
    when 'private', 'protected', 'public', 'private_class_method',
         'public_class_method', 'module_function' then
      parse_visibility container, single, tk
      return true
    when 'private_constant', 'public_constant'
      parse_constant_visibility container, single, tk
      return true
    when 'attr' then
      parse_attr container, single, tk, comment
    when /^attr_(reader|writer|accessor)$/ then
      parse_attr_accessor container, single, tk, comment
    when 'alias_method' then
      parse_alias container, single, tk, comment
    when 'require', 'include' then
      # ignore
    else
      if comment.text =~ /\A#\#$/ then
        case comment.text
        when /^# +:?attr(_reader|_writer|_accessor)?:/ then
          parse_meta_attr container, single, tk, comment
        else
          method = parse_meta_method container, single, tk, comment
          method.params = container.params if
            container.params
          method.block_params = container.block_params if
            container.block_params
        end
      end
    end

    false
  end

  ##
  # Parses a meta-programmed attribute and creates an RDoc::Attr.
  #
  # To create foo and bar attributes on class C with comment "My attributes":
  #
  #   class C
  #
  #     ##
  #     # :attr:
  #     #
  #     # My attributes
  #
  #     my_attr :foo, :bar
  #
  #   end
  #
  # To create a foo attribute on class C with comment "My attribute":
  #
  #   class C
  #
  #     ##
  #     # :attr: foo
  #     #
  #     # My attribute
  #
  #     my_attr :foo, :bar
  #
  #   end

  def parse_meta_attr(context, single, tk, comment)
    args = parse_symbol_arg
    rw = "?"

    # If nodoc is given, don't document any of them

    tmp = RDoc::CodeObject.new
    read_documentation_modifiers tmp, RDoc::ATTR_MODIFIERS

    regexp = /^# +:?(attr(_reader|_writer|_accessor)?): *(\S*).*?\n/i
    if regexp =~ comment.text then
      comment.text = comment.text.sub(regexp, '')
      rw = case $1
           when 'attr_reader' then 'R'
           when 'attr_writer' then 'W'
           else 'RW'
           end
      name = $3 unless $3.empty?
    end

    if name then
      att = create_attr context, single, name, rw, comment
    else
      args.each do |attr_name|
        att = create_attr context, single, attr_name, rw, comment
      end
    end

    att
  end

  ##
  # Parses a meta-programmed method

  def parse_meta_method(container, single, tk, comment)
    column  = tk[:char_no]
    line_no = tk[:line_no]

    start_collecting_tokens
    add_token tk
    add_token_listener self

    skip_tkspace_without_nl

    comment.text = comment.text.sub(/(^# +:?)(singleton-)(method:)/, '\1\3')
    singleton = !!$~

    name = parse_meta_method_name comment, tk

    return unless name

    meth = RDoc::MetaMethod.new get_tkread, name
    record_location meth
    meth.line   = line_no
    meth.singleton = singleton

    remove_token_listener self

    meth.start_collecting_tokens
    indent = RDoc::Parser::RipperStateLex::Token.new(1, 1, :on_sp, ' ' * column)
    position_comment = RDoc::Parser::RipperStateLex::Token.new(line_no, 1, :on_comment)
    position_comment[:text] = "# File #{@top_level.relative_name}, line #{line_no}"
    newline = RDoc::Parser::RipperStateLex::Token.new(0, 0, :on_nl, "\n")
    meth.add_tokens [position_comment, newline, indent]
    meth.add_tokens @token_stream

    parse_meta_method_params container, single, meth, tk, comment

    meth.comment = comment

    @stats.add_method meth

    meth
  end

  ##
  # Parses the name of a metaprogrammed method.  +comment+ is used to
  # determine the name while +tk+ is used in an error message if the name
  # cannot be determined.

  def parse_meta_method_name comment, tk # :nodoc:
    if comment.text.sub!(/^# +:?method: *(\S*).*?\n/i, '') then
      return $1 unless $1.empty?
    end

    name_t = get_tk

    if :on_symbol == name_t[:kind] then
      name_t[:text][1..-1]
    elsif :on_tstring == name_t[:kind] then
      name_t[:text][1..-2]
    elsif :on_op == name_t[:kind] && '=' == name_t[:text] then # ignore
      remove_token_listener self

      nil
    else
      warn "unknown name token #{name_t.inspect} for meta-method '#{tk[:text]}'"
      'unknown'
    end
  end

  ##
  # Parses the parameters and block for a meta-programmed method.

  def parse_meta_method_params container, single, meth, tk, comment # :nodoc:
    token_listener meth do
      meth.params = ''

      look_for_directives_in meth, comment
      comment.normalize
      comment.extract_call_seq meth

      container.add_method meth

      last_tk = tk

      while tk = get_tk do
        if :on_semicolon == tk[:kind] then
          break
        elsif :on_nl == tk[:kind] then
          break unless last_tk and :on_comma == last_tk[:kind]
        elsif :on_sp == tk[:kind] then
          # expression continues
        elsif :on_kw == tk[:kind] && 'do' == tk[:text] then
          parse_statements container, single, meth
          break
        else
          last_tk = tk
        end
      end
    end
  end

  ##
  # Parses a normal method defined by +def+

  def parse_method(container, single, tk, comment)
    singleton = nil
    added_container = false
    name = nil
    column  = tk[:char_no]
    line_no = tk[:line_no]

    start_collecting_tokens
    add_token tk

    token_listener self do
      prev_container = container
      name, container, singleton = parse_method_name container
      added_container = container != prev_container
    end

    return unless name

    meth = RDoc::AnyMethod.new get_tkread, name
    look_for_directives_in meth, comment
    meth.singleton = single == SINGLE ? true : singleton

    record_location meth
    meth.line   = line_no

    meth.start_collecting_tokens
    indent = RDoc::Parser::RipperStateLex::Token.new(1, 1, :on_sp, ' ' * column)
    token = RDoc::Parser::RipperStateLex::Token.new(line_no, 1, :on_comment)
    token[:text] = "# File #{@top_level.relative_name}, line #{line_no}"
    newline = RDoc::Parser::RipperStateLex::Token.new(0, 0, :on_nl, "\n")
    meth.add_tokens [token, newline, indent]
    meth.add_tokens @token_stream

    parse_method_params_and_body container, single, meth, added_container

    comment.normalize
    comment.extract_call_seq meth

    meth.comment = comment

    # after end modifiers
    read_documentation_modifiers meth, RDoc::METHOD_MODIFIERS

    @stats.add_method meth
  end

  ##
  # Parses the parameters and body of +meth+

  def parse_method_params_and_body container, single, meth, added_container
    token_listener meth do
      parse_method_parameters meth

      if meth.document_self or not @track_visibility then
        container.add_method meth
      elsif added_container then
        container.document_self = false
      end

      # Having now read the method parameters and documentation modifiers, we
      # now know whether we have to rename #initialize to ::new

      if meth.name == "initialize" && !meth.singleton then
        if meth.dont_rename_initialize then
          meth.visibility = :protected
        else
          meth.singleton = true
          meth.name = "new"
          meth.visibility = :public
        end
      end

      parse_statements container, single, meth
    end
  end

  ##
  # Parses a method that needs to be ignored.

  def parse_method_dummy container
    dummy = RDoc::Context.new
    dummy.parent = container
    dummy.store  = container.store
    skip_method dummy
  end

  ##
  # Parses the name of a method in +container+.
  #
  # Returns the method name, the container it is in (for def Foo.name) and if
  # it is a singleton or regular method.

  def parse_method_name container # :nodoc:
    skip_tkspace
    name_t = get_tk
    back_tk = skip_tkspace_without_nl
    singleton = false

    dot = get_tk
    if dot[:kind] == :on_period || (dot[:kind] == :on_op && dot[:text] == '::') then
      singleton = true

      name, container = parse_method_name_singleton container, name_t
    else
      unget_tk dot
      back_tk.reverse_each do |token|
        unget_tk token
      end

      name = parse_method_name_regular container, name_t
    end

    return name, container, singleton
  end

  ##
  # For the given +container+ and initial name token +name_t+ the method name
  # is parsed from the token stream for a regular method.

  def parse_method_name_regular container, name_t # :nodoc:
    if :on_op == name_t[:kind] && (%w{* & [] []= <<}.include?(name_t[:text])) then
      name_t[:text]
    else
      unless [:on_kw, :on_const, :on_ident].include?(name_t[:kind]) then
        warn "expected method name token, . or ::, got #{name_t.inspect}"
        skip_method container
        return
      end
      name_t[:text]
    end
  end

  ##
  # For the given +container+ and initial name token +name_t+ the method name
  # and the new +container+ (if necessary) are parsed from the token stream
  # for a singleton method.

  def parse_method_name_singleton container, name_t # :nodoc:
    skip_tkspace
    name_t2 = get_tk

    if (:on_kw == name_t[:kind] && 'self' == name_t[:text]) || (:on_op == name_t[:kind] && '%' == name_t[:text]) then
      # NOTE: work around '[' being consumed early
      if :on_lbracket == name_t2[:kind]
        get_tk
        name = '[]'
      else
        name = name_t2[:text]
      end
    elsif :on_const == name_t[:kind] then
      name = name_t2[:text]

      container = get_method_container container, name_t

      return unless container

      name
    elsif :on_ident == name_t[:kind] || :on_ivar == name_t[:kind] || :on_gvar == name_t[:kind] then
      parse_method_dummy container

      name = nil
    elsif (:on_kw == name_t[:kind]) && ('true' == name_t[:text] || 'false' == name_t[:text] || 'nil' == name_t[:text]) then
      klass_name = "#{name_t[:text].capitalize}Class"
      container = @store.find_class_named klass_name
      container ||= @top_level.add_class RDoc::NormalClass, klass_name

      name = name_t2[:text]
    else
      warn "unexpected method name token #{name_t.inspect}"
      # break
      skip_method container

      name = nil
    end

    return name, container
  end

  ##
  # Extracts +yield+ parameters from +method+

  def parse_method_or_yield_parameters(method = nil,
                                       modifiers = RDoc::METHOD_MODIFIERS)
    skip_tkspace_without_nl
    tk = get_tk
    end_token = get_end_token tk
    return '' unless end_token

    nest = 0
    continue = false

    while tk != nil do
      case tk[:kind]
      when :on_semicolon then
        break if nest == 0
      when :on_lbracket then
        nest += 1
      when :on_rbracket then
        nest -= 1
      when :on_lbrace then
        nest += 1
      when :on_rbrace then
        nest -= 1
        if nest <= 0
          # we might have a.each { |i| yield i }
          unget_tk(tk) if nest < 0
          break
        end
      when :on_lparen then
        nest += 1
      when end_token[:kind] then
        if end_token[:kind] == :on_rparen
          nest -= 1
          break if nest <= 0
        else
          break
        end
      when :on_rparen then
        nest -= 1
      when :on_comment, :on_embdoc then
        @read.pop
        if :on_nl == end_token[:kind] and "\n" == tk[:text][-1] and
          (!continue or (tk[:state] & RDoc::Parser::RipperStateLex::EXPR_LABEL) != 0) then
          if method && method.block_params.nil? then
            unget_tk tk
            read_documentation_modifiers method, modifiers
          end
          break if !continue and nest <= 0
        end
      when :on_comma then
        continue = true
      when :on_ident then
        continue = false if continue
      end
      tk = get_tk
    end

    get_tkread_clean(/\s+/, ' ')
  end

  ##
  # Capture the method's parameters. Along the way, look for a comment
  # containing:
  #
  #    # yields: ....
  #
  # and add this as the block_params for the method

  def parse_method_parameters method
    res = parse_method_or_yield_parameters method

    res = "(#{res})" unless res =~ /\A\(/
    method.params = res unless method.params

    return if  method.block_params

    skip_tkspace_without_nl
    read_documentation_modifiers method, RDoc::METHOD_MODIFIERS
  end

  ##
  # Parses an RDoc::NormalModule in +container+ with +comment+

  def parse_module container, single, tk, comment
    container, name_t, = get_class_or_module container

    name = name_t[:text]

    mod = container.add_module RDoc::NormalModule, name
    mod.ignore unless container.document_children
    record_location mod

    read_documentation_modifiers mod, RDoc::CLASS_MODIFIERS
    mod.add_comment comment, @top_level
    parse_statements mod

    # after end modifiers
    read_documentation_modifiers mod, RDoc::CLASS_MODIFIERS

    @stats.add_module mod
  end

  ##
  # Parses an RDoc::Require in +context+ containing +comment+

  def parse_require(context, comment)
    skip_tkspace_comment
    tk = get_tk

    if :on_lparen == tk[:kind] then
      skip_tkspace_comment
      tk = get_tk
    end

    name = tk[:text][1..-2] if :on_tstring == tk[:kind]

    if name then
      @top_level.add_require RDoc::Require.new(name, comment)
    else
      unget_tk tk
    end
  end

  ##
  # Parses a rescue

  def parse_rescue
    skip_tkspace_without_nl

    while tk = get_tk
      case tk[:kind]
      when :on_nl, :on_semicolon, :on_comment then
        break
      when :on_comma then
        skip_tkspace_without_nl

        get_tk if :on_nl == peek_tk[:kind]
      end

      skip_tkspace_without_nl
    end
  end

  ##
  # Retrieve comment body without =begin/=end

  def retrieve_comment_body(tk)
    if :on_embdoc == tk[:kind]
      tk[:text].gsub(/\A=begin.*\n/, '').gsub(/=end\n?\z/, '')
    else
      tk[:text]
    end
  end

  ##
  # The core of the Ruby parser.

  def parse_statements(container, single = NORMAL, current_method = nil,
                       comment = new_comment(''))
    raise 'no' unless RDoc::Comment === comment
    comment = RDoc::Encoding.change_encoding comment, @encoding if @encoding

    nest = 1
    save_visibility = container.visibility

    non_comment_seen = true

    while tk = get_tk do
      keep_comment = false
      try_parse_comment = false

      non_comment_seen = true unless (:on_comment == tk[:kind] or :on_embdoc == tk[:kind])

      case tk[:kind]
      when :on_nl, :on_ignored_nl, :on_comment, :on_embdoc then
        if :on_nl == tk[:kind] or :on_ignored_nl == tk[:kind]
          skip_tkspace
          tk = get_tk
        else
          past_tokens = @read.size > 1 ? @read[0..-2] : []
          nl_position = 0
          past_tokens.reverse.each_with_index do |read_tk, i|
            if read_tk =~ /^\n$/ then
              nl_position = (past_tokens.size - 1) - i
              break
            elsif read_tk =~ /^#.*\n$/ then
              nl_position = ((past_tokens.size - 1) - i) + 1
              break
            end
          end
          comment_only_line = past_tokens[nl_position..-1].all?{ |c| c =~ /^\s+$/ }
          unless comment_only_line then
            tk = get_tk
          end
        end

        if tk and (:on_comment == tk[:kind] or :on_embdoc == tk[:kind]) then
          if non_comment_seen then
            # Look for RDoc in a comment about to be thrown away
            non_comment_seen = parse_comment container, tk, comment unless
              comment.empty?

            comment = ''
            comment = RDoc::Encoding.change_encoding comment, @encoding if @encoding
          end

          line_no = nil
          while tk and (:on_comment == tk[:kind] or :on_embdoc == tk[:kind]) do
            comment_body = retrieve_comment_body(tk)
            line_no = tk[:line_no] if comment.empty?
            comment += comment_body
            comment << "\n" unless comment_body =~ /\n\z/

            if comment_body.size > 1 && comment_body =~ /\n\z/ then
              skip_tkspace_without_nl # leading spaces
            end
            tk = get_tk
          end

          comment = new_comment comment, line_no

          unless comment.empty? then
            look_for_directives_in container, comment

            if container.done_documenting then
              throw :eof if RDoc::TopLevel === container
              container.ongoing_visibility = save_visibility
            end
          end

          keep_comment = true
        else
          non_comment_seen = true
        end

        unget_tk tk
        keep_comment = true
        container.current_line_visibility = nil

      when :on_kw then
        case tk[:text]
        when 'class' then
          parse_class container, single, tk, comment

        when 'module' then
          parse_module container, single, tk, comment

        when 'def' then
          parse_method container, single, tk, comment

        when 'alias' then
          parse_alias container, single, tk, comment unless current_method

        when 'yield' then
          if current_method.nil? then
            warn "Warning: yield outside of method" if container.document_self
          else
            parse_yield container, single, tk, current_method
          end

        when 'until', 'while' then
          if (tk[:state] & RDoc::Parser::RipperStateLex::EXPR_LABEL) == 0
            nest += 1
            skip_optional_do_after_expression
          end

        # Until and While can have a 'do', which shouldn't increase the nesting.
        # We can't solve the general case, but we can handle most occurrences by
        # ignoring a do at the end of a line.

        # 'for' is trickier
        when 'for' then
          nest += 1
          skip_for_variable
          skip_optional_do_after_expression

        when 'case', 'do', 'if', 'unless', 'begin' then
          if (tk[:state] & RDoc::Parser::RipperStateLex::EXPR_LABEL) == 0
            nest += 1
          end

        when 'super' then
          current_method.calls_super = true if current_method

        when 'rescue' then
          parse_rescue

        when 'end' then
          nest -= 1
          if nest == 0 then
            container.ongoing_visibility = save_visibility

            parse_comment container, tk, comment unless comment.empty?

            return
          end
        end

      when :on_const then
        unless parse_constant container, tk, comment, current_method then
          try_parse_comment = true
        end

      when :on_ident then
        if nest == 1 and current_method.nil? then
          keep_comment = parse_identifier container, single, tk, comment
        end

        case tk[:text]
        when "require" then
          parse_require container, comment
        when "include" then
          parse_extend_or_include RDoc::Include, container, comment
        when "extend" then
          parse_extend_or_include RDoc::Extend, container, comment
        when "included" then
          parse_included_with_activesupport_concern container, comment
        end

      else
        try_parse_comment = nest == 1
      end

      if try_parse_comment then
        non_comment_seen = parse_comment container, tk, comment unless
          comment.empty?

        keep_comment = false
      end

      unless keep_comment then
        comment = new_comment ''
        comment = RDoc::Encoding.change_encoding comment, @encoding if @encoding
        container.params = nil
        container.block_params = nil
      end

      consume_trailing_spaces
    end

    container.params = nil
    container.block_params = nil
  end

  ##
  # Parse up to +no+ symbol arguments

  def parse_symbol_arg(no = nil)
    skip_tkspace_comment

    tk = get_tk
    if tk[:kind] == :on_lparen
      parse_symbol_arg_paren no
    else
      parse_symbol_arg_space no, tk
    end
  end

  ##
  # Parses up to +no+ symbol arguments surrounded by () and places them in
  # +args+.

  def parse_symbol_arg_paren no # :nodoc:
    args = []

    loop do
      skip_tkspace_comment
      if tk1 = parse_symbol_in_arg
        args.push tk1
        break if no and args.size >= no
      end

      skip_tkspace_comment
      case (tk2 = get_tk)[:kind]
      when :on_rparen
        break
      when :on_comma
      else
        warn("unexpected token: '#{tk2.inspect}'") if $DEBUG_RDOC
        break
      end
    end

    args
  end

  ##
  # Parses up to +no+ symbol arguments separated by spaces and places them in
  # +args+.

  def parse_symbol_arg_space no, tk # :nodoc:
    args = []

    unget_tk tk
    if tk = parse_symbol_in_arg
      args.push tk
      return args if no and args.size >= no
    end

    loop do
      skip_tkspace_without_nl

      tk1 = get_tk
      if tk1.nil? || :on_comma != tk1[:kind] then
        unget_tk tk1
        break
      end

      skip_tkspace_comment
      if tk = parse_symbol_in_arg
        args.push tk
        break if no and args.size >= no
      end
    end

    args
  end

  ##
  # Returns symbol text from the next token

  def parse_symbol_in_arg
    tk = get_tk
    if :on_symbol == tk[:kind] then
      tk[:text].sub(/^:/, '')
    elsif :on_tstring == tk[:kind] then
      tk[:text][1..-2]
    elsif :on_dstring == tk[:kind] or :on_ident == tk[:kind] then
      nil # ignore
    else
      warn("Expected symbol or string, got #{tk.inspect}") if $DEBUG_RDOC
      nil
    end
  end

  ##
  # Parses statements in the top-level +container+

  def parse_top_level_statements container
    comment = collect_first_comment

    look_for_directives_in container, comment

    throw :eof if container.done_documenting

    @markup = comment.format

    # HACK move if to RDoc::Context#comment=
    container.comment = comment if container.document_self unless comment.empty?

    parse_statements container, NORMAL, nil, comment
  end

  ##
  # Determines the visibility in +container+ from +tk+

  def parse_visibility(container, single, tk)
    vis_type, vis, singleton = get_visibility_information tk, single

    skip_tkspace_comment false

    ptk = peek_tk
    # Ryan Davis suggested the extension to ignore modifiers, because he
    # often writes
    #
    #   protected unless $TESTING
    #
    if [:on_nl, :on_semicolon].include?(ptk[:kind]) || (:on_kw == ptk[:kind] && (['if', 'unless'].include?(ptk[:text]))) then
      container.ongoing_visibility = vis
    elsif :on_kw == ptk[:kind] && 'def' == ptk[:text]
      container.current_line_visibility = vis
    else
      update_visibility container, vis_type, vis, singleton
    end
  end

  ##
  # Parses a Module#private_constant or Module#public_constant call from +tk+.

  def parse_constant_visibility(container, single, tk)
    args = parse_symbol_arg
    case tk[:text]
    when 'private_constant'
      vis = :private
    when 'public_constant'
      vis = :public
    else
      raise RDoc::Error, 'Unreachable'
    end
    container.set_constant_visibility_for args, vis
  end

  ##
  # Determines the block parameter for +context+

  def parse_yield(context, single, tk, method)
    return if method.block_params

    get_tkread
    method.block_params = parse_method_or_yield_parameters
  end

  ##
  # Directives are modifier comments that can appear after class, module, or
  # method names. For example:
  #
  #   def fred # :yields: a, b
  #
  # or:
  #
  #   class MyClass # :nodoc:
  #
  # We return the directive name and any parameters as a two element array if
  # the name is in +allowed+.  A directive can be found anywhere up to the end
  # of the current line.

  def read_directive allowed
    tokens = []

    while tk = get_tk do
      tokens << tk

      if :on_nl == tk[:kind] or (:on_kw == tk[:kind] && 'def' == tk[:text]) then
        return
      elsif :on_comment == tk[:kind] or :on_embdoc == tk[:kind] then
        return unless tk[:text] =~ /\s*:?([\w-]+):\s*(.*)/

        directive = $1.downcase

        return [directive, $2] if allowed.include? directive

        return
      end
    end
  ensure
    unless tokens.length == 1 and (:on_comment == tokens.first[:kind] or :on_embdoc == tokens.first[:kind]) then
      tokens.reverse_each do |token|
        unget_tk token
      end
    end
  end

  ##
  # Handles directives following the definition for +context+ (any
  # RDoc::CodeObject) if the directives are +allowed+ at this point.
  #
  # See also RDoc::Markup::PreProcess#handle_directive

  def read_documentation_modifiers context, allowed
    skip_tkspace_without_nl
    directive, value = read_directive allowed

    return unless directive

    @preprocess.handle_directive '', directive, value, context do |dir, param|
      if %w[notnew not_new not-new].include? dir then
        context.dont_rename_initialize = true

        true
      end
    end
  end

  ##
  # Records the location of this +container+ in the file for this parser and
  # adds it to the list of classes and modules in the file.

  def record_location container # :nodoc:
    case container
    when RDoc::ClassModule then
      @top_level.add_to_classes_or_modules container
    end

    container.record_location @top_level
  end

  ##
  # Scans this Ruby file for Ruby constructs

  def scan
    reset

    catch :eof do
      begin
        parse_top_level_statements @top_level

      rescue StandardError => e
        if @content.include?('<%') and @content.include?('%>') then
          # Maybe, this is ERB.
          $stderr.puts "\033[2KRDoc detects ERB file. Skips it for compatibility:"
          $stderr.puts @file_name
          return
        end

        if @scanner_point >= @scanner.size
          now_line_no = @scanner[@scanner.size - 1][:line_no]
        else
          now_line_no = peek_tk[:line_no]
        end
        first_tk_index = @scanner.find_index { |tk| tk[:line_no] == now_line_no }
        last_tk_index = @scanner.find_index { |tk| tk[:line_no] == now_line_no + 1 }
        last_tk_index = last_tk_index ? last_tk_index - 1 : @scanner.size - 1
        code = @scanner[first_tk_index..last_tk_index].map{ |t| t[:text] }.join

        $stderr.puts <<-EOF

#{self.class} failure around line #{now_line_no} of
#{@file_name}

        EOF

        unless code.empty? then
          $stderr.puts code
          $stderr.puts
        end

        raise e
      end
    end

    @top_level
  end

  ##
  # while, until, and for have an optional do

  def skip_optional_do_after_expression
    skip_tkspace_without_nl
    tk = get_tk

    b_nest = 0
    nest = 0

    loop do
      break unless tk
      case tk[:kind]
      when :on_semicolon, :on_nl, :on_ignored_nl then
        break if b_nest.zero?
      when :on_lparen then
        nest += 1
      when :on_rparen then
        nest -= 1
      when :on_kw then
        case tk[:text]
        when 'begin'
          b_nest += 1
        when 'end'
          b_nest -= 1
        when 'do'
          break if nest.zero?
        end
      when :on_comment, :on_embdoc then
        if b_nest.zero? and "\n" == tk[:text][-1] then
          break
        end
      end
      tk = get_tk
    end

    skip_tkspace_without_nl

    get_tk if peek_                                                                                                                                                                                                                                                                                                                                                                                                                  gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/changelog.rb                                                 0000644                 00000020473 15040313414 0015433 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       _type, vis, singleton # :nodoc:
    new_methods = []

    case vis_type
    when 'module_function' then
      args = parse_symbol_arg
      container.set_visibility_for args, :private, false

      container.methods_matching args do |m|
        s_m = m.dup
        record_location s_m
        s_m.singleton = true
        new_methods << s_m
      end
    when 'public_class_method', 'private_class_method' then
      args = parse_symbol_arg

      container.methods_matching args, true do |m|
        if m.parent != container then
          m = m.dup
          record_location m
          new_methods << m
        end

        m.visibility = vis
      end
    else
      args = parse_symbol_arg
      container.set_visibility_for args, vis, singleton
    end

    new_methods.each do |method|
      case method
      when RDoc::AnyMethod then
        container.add_method method
      when RDoc::Attr then
        container.add_attribute method
      end
      method.visibility = vis
    end
  end

  ##
  # Prints +message+ to +$stderr+ unless we're being quiet

  def warn message
    @options.warn make_message message
  end

end
                                                                                                                                                                                                                                                                                                                                                                                                                  gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/changelog.rb                                                 0000644                 00000020473 15040313414 0015433 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

##
# A ChangeLog file parser.
#
# This parser converts a ChangeLog into an RDoc::Markup::Document.  When
# viewed as HTML a ChangeLog page will have an entry for each day's entries in
# the sidebar table of contents.
#
# This parser is meant to parse the MRI ChangeLog, but can be used to parse any
# {GNU style Change
# Log}[http://www.gnu.org/prep/standards/html_node/Style-of-Change-Logs.html].

class RDoc::Parser::ChangeLog < RDoc::Parser

  include RDoc::Parser::Text

  parse_files_matching(/(\/|\\|\A)ChangeLog[^\/\\]*\z/)

  ##
  # Attaches the +continuation+ of the previous line to the +entry_body+.
  #
  # Continued function listings are joined together as a single entry.
  # Continued descriptions are joined to make a single paragraph.

  def continue_entry_body entry_body, continuation
    return unless last = entry_body.last

    if last =~ /\)\s*\z/ and continuation =~ /\A\(/ then
      last.sub!(/\)\s*\z/, ',')
      continuation = continuation.sub(/\A\(/, '')
    end

    if last =~ /\s\z/ then
      last << continuation
    else
      last << ' ' + continuation
    end
  end

  ##
  # Creates an RDoc::Markup::Document given the +groups+ of ChangeLog entries.

  def create_document groups
    doc = RDoc::Markup::Document.new
    doc.omit_headings_below = 2
    doc.file = @top_level

    doc << RDoc::Markup::Heading.new(1, File.basename(@file_name))
    doc << RDoc::Markup::BlankLine.new

    groups.sort_by do |day,| day end.reverse_each do |day, entries|
      doc << RDoc::Markup::Heading.new(2, day.dup)
      doc << RDoc::Markup::BlankLine.new

      doc.concat create_entries entries
    end

    doc
  end

  ##
  # Returns a list of ChangeLog entries an RDoc::Markup nodes for the given
  # +entries+.

  def create_entries entries
    out = []

    entries.each do |entry, items|
      out << RDoc::Markup::Heading.new(3, entry)
      out << RDoc::Markup::BlankLine.new

      out << create_items(items)
    end

    out
  end

  ##
  # Returns an RDoc::Markup::List containing the given +items+ in the
  # ChangeLog

  def create_items items
    list = RDoc::Markup::List.new :NOTE

    items.each do |item|
      item =~ /\A(.*?(?:\([^)]+\))?):\s*/

      title = $1
      body = $'

      paragraph = RDoc::Markup::Paragraph.new body
      list_item = RDoc::Markup::ListItem.new title, paragraph
      list << list_item
    end

    list
  end

  ##
  # Groups +entries+ by date.

  def group_entries entries
    @time_cache ||= {}
    entries.group_by do |title, _|
      begin
        time = @time_cache[title]
        (time || parse_date(title)).strftime '%Y-%m-%d'
      rescue NoMethodError, ArgumentError
        time, = title.split '  ', 2
        parse_date(time).strftime '%Y-%m-%d'
      end
    end
  end

  ##
  # Parse date in ISO-8601, RFC-2822, or default of Git

  def parse_date(date)
    case date
    when /\A\s*(\d+)-(\d+)-(\d+)(?:[ T](\d+):(\d+):(\d+) *([-+]\d\d):?(\d\d))?\b/
      Time.new($1, $2, $3, $4, $5, $6, ("#{$7}:#{$8}" if $7))
    when /\A\s*\w{3}, +(\d+) (\w{3}) (\d+) (\d+):(\d+):(\d+) *(?:([-+]\d\d):?(\d\d))\b/
      Time.new($3, $2, $1, $4, $5, $6, ("#{$7}:#{$8}" if $7))
    when /\A\s*\w{3} (\w{3}) +(\d+) (\d+) (\d+):(\d+):(\d+) *(?:([-+]\d\d):?(\d\d))\b/
      Time.new($3, $1, $2, $4, $5, $6, ("#{$7}:#{$8}" if $7))
    when /\A\s*\w{3} (\w{3}) +(\d+) (\d+):(\d+):(\d+) (\d+)\b/
      Time.new($6, $1, $2, $3, $4, $5)
    else
      raise ArgumentError, "bad date: #{date}"
    end
  end

  ##
  # Parses the entries in the ChangeLog.
  #
  # Returns an Array of each ChangeLog entry in order of parsing.
  #
  # A ChangeLog entry is an Array containing the ChangeLog title (date and
  # committer) and an Array of ChangeLog items (file and function changed with
  # description).
  #
  # An example result would be:
  #
  #    [ 'Tue Dec  4 08:33:46 2012  Eric Hodel  <drbrain@segment7.net>',
  #      [ 'README.EXT:  Converted to RDoc format',
  #        'README.EXT.ja:  ditto']]

  def parse_entries
    @time_cache ||= {}

    if /\A((?:.*\n){,3})commit\s/ =~ @content
      class << self; prepend Git; end
      parse_info($1)
      return parse_entries
    end

    entries = []
    entry_name = nil
    entry_body = []

    @content.each_line do |line|
      case line
      when /^\s*$/ then
        next
      when /^\w.*/ then
        entries << [entry_name, entry_body] if entry_name

        entry_name = $&

        begin
          time = parse_date entry_name
          @time_cache[entry_name] = time
        rescue ArgumentError
          entry_name = nil
        end

        entry_body = []
      when /^(\t| {8})?\*\s*(.*)/ then # "\t* file.c (func): ..."
        entry_body << $2.dup
      when /^(\t| {8})?\s*(\(.*)/ then # "\t(func): ..."
        entry = $2

        if entry_body.last =~ /:/ then
          entry_body << entry.dup
        else
          continue_entry_body entry_body, entry
        end
      when /^(\t| {8})?\s*(.*)/ then
        continue_entry_body entry_body, $2
      end
    end

    entries << [entry_name, entry_body] if entry_name

    entries.reject! do |(entry,_)|
      entry == nil
    end

    entries
  end

  ##
  # Converts the ChangeLog into an RDoc::Markup::Document

  def scan
    @time_cache = {}

    entries = parse_entries
    grouped_entries = group_entries entries

    doc = create_document grouped_entries

    @top_level.comment = doc

    @top_level
  end

  module Git
    def parse_info(info)
      /^\s*base-url\s*=\s*(.*\S)/ =~ info
      @base_url = $1
    end

    def parse_entries
      entries = []

      @content.scan(/^commit\s+(\h{20})\h*\n((?:.+\n)*)\n((?: {4}.*\n+)*)/) do
        entry_name, header, entry_body = $1, $2, $3.gsub(/^ {4}/, '')
        # header = header.scan(/^ *(\S+?): +(.*)/).to_h
        # date = header["CommitDate"] || header["Date"]
        date = header[/^ *(?:Author)?Date: +(.*)/, 1]
        author = header[/^ *Author: +(.*)/, 1]
        begin
          time = parse_date(header[/^ *CommitDate: +(.*)/, 1] || date)
          @time_cache[entry_name] = time
          author.sub!(/\s*<(.*)>/, '')
          email = $1
          entries << [entry_name, [author, email, date, entry_body]]
        rescue ArgumentError
        end
      end

      entries
    end

    def create_entries entries
      # git log entries have no strictly itemized style like the old
      # style, just assume Markdown.
      entries.map do |commit, e                                                                                                                                                                                                     gems/gems/rdoc-6.5.1.1/lib/rdoc/servlet.rb                                                          0000644                 00000030365 15040313415 0013676 0                                                                                                    ustar 00                                                                                                                                                                                                                                                        def level
        HEADING_LEVEL
      end

      def aref
        "label-#{commit}"
      end

      def label context = nil
        aref
      end

      def text
        case base
        when nil
          "#{date}"
        when /%s/
          "{#{date}}[#{base % commit}]"
        else
          "{#{date}}[#{base}#{commit}]"
        end + " {#{author}}[mailto:#{email}]"
      end

      def accept visitor
        visitor.accept_heading self
        begin
          if visitor.respond_to?(:code_object=)
            code_object = visitor.code_object
            visitor.code_object = self
          end
          contents.each do |body|
            body.accept visitor
          end
        ensure
          if visitor.respond_to?(:code_object)
            visitor.code_object = code_object
          end
        end
      end

      def pretty_print q # :nodoc:
        q.group(2, '[log_entry: ', ']') do
          q.text commit
          q.text ','
          q.breakable
          q.group(2, '[date: ', ']') { q.text date }
          q.text ','
          q.breakable
          q.group(2, '[author: ', ']') { q.text author }
          q.text ','
          q.breakable
          q.group(2, '[email: ', ']') { q.text email }
          q.text ','
          q.breakable
          q.pp contents
        end
      end
    end
  end
end

                                                                                                                                                                                                     gems/gems/rdoc-6.5.1.1/lib/rdoc/servlet.rb                                                          0000644                 00000030365 15040313415 0013676 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require_relative '../rdoc'
require 'erb'
require 'time'
require 'json'

begin
  require 'webrick'
rescue LoadError
  abort "webrick is not found. You may need to `gem install webrick` to install webrick."
end

##
# This is a WEBrick servlet that allows you to browse ri documentation.
#
# You can show documentation through either `ri --server` or, with RubyGems
# 2.0 or newer, `gem server`.  For ri, the server runs on port 8214 by
# default.  For RubyGems the server runs on port 8808 by default.
#
# You can use this servlet in your own project by mounting it on a WEBrick
# server:
#
#   require 'webrick'
#
#   server = WEBrick::HTTPServer.new Port: 8000
#
#   server.mount '/', RDoc::Servlet
#
# If you want to mount the servlet some other place than the root, provide the
# base path when mounting:
#
#   server.mount '/rdoc', RDoc::Servlet, '/rdoc'

class RDoc::Servlet < WEBrick::HTTPServlet::AbstractServlet

  @server_stores = Hash.new { |hash, server| hash[server] = {} }
  @cache         = Hash.new { |hash, store|  hash[store]  = {} }

  ##
  # Maps an asset type to its path on the filesystem

  attr_reader :asset_dirs

  ##
  # An RDoc::Options instance used for rendering options

  attr_reader :options

  ##
  # Creates an instance of this servlet that shares cached data between
  # requests.

  def self.get_instance server, *options # :nodoc:
    stores = @server_stores[server]

    new server, stores, @cache, *options
  end

  ##
  # Creates a new WEBrick servlet.
  #
  # Use +mount_path+ when mounting the servlet somewhere other than /.
  #
  # Use +extra_doc_dirs+ for additional documentation directories.
  #
  # +server+ is provided automatically by WEBrick when mounting.  +stores+ and
  # +cache+ are provided automatically by the servlet.

  def initialize server, stores, cache, mount_path = nil, extra_doc_dirs = []
    super server

    @cache      = cache
    @mount_path = mount_path
    @extra_doc_dirs = extra_doc_dirs
    @stores     = stores

    @options = RDoc::Options.new
    @options.op_dir = '.'

    darkfish_dir = nil

    # HACK dup
    $LOAD_PATH.each do |path|
      darkfish_dir = File.join path, 'rdoc/generator/template/darkfish/'
      next unless File.directory? darkfish_dir
      @options.template_dir = darkfish_dir
      break
    end

    @asset_dirs = {
      :darkfish   => darkfish_dir,
      :json_index =>
        File.expand_path('../generator/template/json_index/', __FILE__),
    }
  end

  ##
  # Serves the asset at the path in +req+ for +generator_name+ via +res+.

  def asset generator_name, req, res
    asset_dir = @asset_dirs[generator_name]

    asset_path = File.join asset_dir, req.path

    if_modified_since req, res, asset_path

    res.body = File.read asset_path

    res.content_type = case req.path
                       when /\.css\z/ then 'text/css'
                       when /\.js\z/  then 'application/javascript'
                       else                'application/octet-stream'
                       end
  end

  ##
  # GET request entry point.  Fills in +res+ for the path, etc. in +req+.

  def do_GET req, res
    req.path.sub!(/\A#{Regexp.escape @mount_path}/, '') if @mount_path

    case req.path
    when '/' then
      root req, res
    when '/js/darkfish.js', '/js/jquery.js', '/js/search.js',
         %r%^/css/%, %r%^/images/%, %r%^/fonts/% then
      asset :darkfish, req, res
    when '/js/navigation.js', '/js/searcher.js' then
      asset :json_index, req, res
    when '/js/search_index.js' then
      root_search req, res
    else
      show_documentation req, res
    end
  rescue WEBrick::HTTPStatus::NotFound => e
    generator = generator_for RDoc::Store.new

    not_found generator, req, res, e.message
  rescue WEBrick::HTTPStatus::Status
    raise
  rescue => e
    error e, req, res
  end

  ##
  # Fills in +res+ with the class, module or page for +req+ from +store+.
  #
  # +path+ is relative to the mount_path and is used to determine the class,
  # module or page name (/RDoc/Servlet.html becomes RDoc::Servlet).
  # +generator+ is used to create the page.

  def documentation_page store, generator, path, req, res
    text_name = path.chomp '.html'
    name = text_name.gsub '/', '::'

    if klass = store.find_class_or_module(name) then
      res.body = generator.generate_class klass
    elsif page = store.find_text_page(name.sub(/_([^_]*)\z/, '.\1')) then
      res.body = generator.generate_page page
    elsif page = store.find_text_page(text_name.sub(/_([^_]*)\z/, '.\1')) then
      res.body = generator.generate_page page
    else
      not_found generator, req, res
    end
  end

  ##
  # Creates the JSON search index on +res+ for the given +store+.  +generator+
  # must respond to \#json_index to build.  +req+ is ignored.

  def documentation_search store, generator, req, res
    json_index = @cache[store].fetch :json_index do
      @cache[store][:json_index] =
        JSON.dump generator.json_index.build_index
    end

    res.content_type = 'application/javascript'
    res.body = "var search_data = #{json_index}"
  end

  ##
  # Returns the RDoc::Store and path relative to +mount_path+ for
  # documentation at +path+.

  def documentation_source path
    _, source_name, path = path.split '/', 3

    store = @stores[source_name]
    return store, path if store

    store = store_for source_name

    store.load_all

    @stores[source_name] = store

    return store, path
  end

  ##
  # Generates an error page for the +exception+ while handling +req+ on +res+.

  def error exception, req, res
    backtrace = exception.backtrace.join "\n"

    res.content_type = 'text/html'
    res.status = 500
    res.body = <<-BODY
<!DOCTYPE html>
<html>
<head>
<meta content="text/html; charset=UTF-8" http-equiv="Content-Type">

<title>Error - #{ERB::Util.html_escape exception.class}</title>

<link type="text/css" media="screen" href="#{@mount_path}/css/rdoc.css" rel="stylesheet">
</head>
<body>
<h1>Error</h1>

<p>While processing <code>#{ERB::Util.html_escape req.request_uri}</code> the
RDoc (#{ERB::Util.html_escape RDoc::VERSION}) server has encountered a
<code>#{ERB::Util.html_escape exception.class}</code>
exception:

<pre>#{ERB::Util.html_escape exception.message}</pre>

<p>Please report this to the
<a href="https://github.com/ruby/rdoc/issues">RDoc issues tracker</a>.  Please
include the RDoc version, the URI above and exception class, message and
backtrace.  If you're viewing a gem's documentation, include the gem name and
version.  If you're viewing Ruby's documentation, include the version of ruby.

<p>Backtrace:

<pre>#{ERB::Util.html_escape backtrace}</pre>

</body>
</html>
    BODY
  end

  ##
  # Instantiates a Darkfish generator for +store+

  def generator_for store
    generator = RDoc::Generator::Darkfish.new store, @options
    generator.file_output = false
    generator.asset_rel_path = '..'

    rdoc = RDoc::RDoc.new
    rdoc.store     = store
    rdoc.generator = generator
    rdoc.options   = @options

    @options.main_page = store.main
    @options.title     = store.title

    generator
  end

  ##
  # Handles the If-Modified-Since HTTP header on +req+ for +path+.  If the
  # file has not been modified a Not Modified response is returned.  If the
  # file has been modified a Last-Modified header is added to +res+.

  def if_modified_since req, res, path = nil
    last_modified = File.stat(path).mtime if path

    res['last-modified'] = last_modified.httpdate

    return unless ims = req['if-modified-since']

    ims = Time.parse ims

    unless ims < last_modified then
      res.body = ''
      raise WEBrick::HTTPStatus::NotModified
    end
  end

  ##
  # Returns an Array of installed documentation.
  #
  # Each entry contains the documentation name (gem name, 'Ruby
  # Documentation', etc.), the path relative to the mount point, whether the
  # documentation exists, the type of documentation (See RDoc::RI::Paths#each)
  # and the filesystem to the RDoc::Store for the documentation.

  def installed_docs
    extra_counter = 0
    ri_paths.map do |path, type|
      store = RDoc::Store.new path, type
      exists = File.exist? store.cache_path

      case type
      when :gem then
        gem_path = path[%r%/([^/]*)/ri$%, 1]
        [gem_path, "#{gem_path}/", exists, type, path]
      when :system then
        ['Ruby Documentation', 'ruby/', exists, type, path]
      when :site then
        ['Site Documentation', 'site/', exists, type, path]
      when :home then
        ['Home Documentation', 'home/', exists, type, path]
      when :extra then
        extra_counter += 1
        store.load_cache if exists
        title = store.title || "Extra Documentation"
        [title, "extra-#{extra_counter}/", exists, type, path]
      end
    end
  end

  ##
  # Returns a 404 page built by +generator+ for +req+ on +res+.

  def not_found generator, req, res, message = nil
    message ||= "The page <kbd>#{ERB::Util.h req.path}</kbd> was not found"
    res.body = generator.generate_servlet_not_found message
    res.status = 404
  end

  ##
  # Enumerates the ri paths.  See RDoc::RI::Paths#each

  def ri_paths &block
    RDoc::RI::Paths.each true, true, true, :all, *@extra_doc_dirs, &block #TODO: pass extra_dirs
  end

  ##
  # Generates the root page on +res+.  +req+ is ignored.

  def root req, res
    generator = RDoc::Generator::Darkfish.new nil, @options

    res.body = generator.generate_servlet_root installed_docs

    res.content_type = 'text/html'
  end

  ##
  # Generates a search index for the root page on +res+.  +req+ is ignored.

  def root_search req, res
    search_index = []
    info         = []

    installed_docs.map do |name, href, exists, type, path|
      next unless exists

      search_index << name

      case type
      when :gem
        gemspec = path.gsub(%r%/doc/([^/]*?)/ri$%,
                            '/specifications/\1.gemspec')

        spec = Gem::Specification.load gemspec

        path    = spec.full_name
        comment = spec.summary
      when :system then
        path    = 'ruby'
        comment = 'Documentation for the Ruby standard library'
      when :site then
        path    = 'site'
        comment = 'Documentation for non-gem libraries'
      when :home then
        path    = 'home'
        comment = 'Documentation from your home directory'
      when :extra
        comment = name
      end

      info << [name, '', path, '', comment]
    end

                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/ri.rb                                                               0000644                 00000000566 15040313415 0012624 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       earch store, generator, req, res
    else
      documentation_page store, generator, path, req, res
    end
  ensure
    res.content_type ||= 'text/html'
  end

  ##
  # Returns an RDoc::Store for the given +source_name+ ('ruby' or a gem name).

  def store_for source_name
    case source_name
    when 'home' then
      RDoc::Store.new RDoc::RI::Paths.home_dir, :home
                                                                                                                                              gems/gems/rdoc-6.5.1.1/lib/rdoc/rd/inline_parser.rb                                                 0000644                 00000076646 15040313415 0015465 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       pe

      return store if File.exist? store.cache_path

      raise WEBrick::HTTPStatus::NotFound,
            "Could not find documentation for \"#{ERB::Util.html_escape(source_name)}\". Please run `gem rdoc --ri gem_name`"

    end
  end

end
                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/ri.rb                                                               0000644                 00000000566 15040313415 0012624 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require_relative '../rdoc'

##
# Namespace for the ri command line tool's implementation.
#
# See <tt>ri --help</tt> for details.

module RDoc::RI

  ##
  # Base RI error class

  class Error < RDoc::Error; end

  autoload :Driver, "#{__dir__}/ri/driver"
  autoload :Paths,  "#{__dir__}/ri/paths"
  autoload :Store,  "#{__dir__}/ri/store"

end
                                                                                                                                          gems/gems/rdoc-6.5.1.1/lib/rdoc/rd/inline_parser.rb                                                 0000644                 00000076646 15040313415 0015465 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#
# DO NOT MODIFY!!!!
# This file is automatically generated by Racc 1.6.0
# from Racc grammar file "".
#

require 'racc/parser.rb'

require 'strscan'

class RDoc::RD

##
# RD format parser for inline markup such as emphasis, links, footnotes, etc.

class InlineParser < Racc::Parser


# :stopdoc:

EM_OPEN = '((*'
EM_OPEN_RE = /\A#{Regexp.quote(EM_OPEN)}/
EM_CLOSE = '*))'
EM_CLOSE_RE = /\A#{Regexp.quote(EM_CLOSE)}/
CODE_OPEN = '(({'
CODE_OPEN_RE = /\A#{Regexp.quote(CODE_OPEN)}/
CODE_CLOSE = '}))'
CODE_CLOSE_RE = /\A#{Regexp.quote(CODE_CLOSE)}/
VAR_OPEN = '((|'
VAR_OPEN_RE = /\A#{Regexp.quote(VAR_OPEN)}/
VAR_CLOSE = '|))'
VAR_CLOSE_RE = /\A#{Regexp.quote(VAR_CLOSE)}/
KBD_OPEN = '((%'
KBD_OPEN_RE = /\A#{Regexp.quote(KBD_OPEN)}/
KBD_CLOSE = '%))'
KBD_CLOSE_RE = /\A#{Regexp.quote(KBD_CLOSE)}/
INDEX_OPEN = '((:'
INDEX_OPEN_RE = /\A#{Regexp.quote(INDEX_OPEN)}/
INDEX_CLOSE = ':))'
INDEX_CLOSE_RE = /\A#{Regexp.quote(INDEX_CLOSE)}/
REF_OPEN = '((<'
REF_OPEN_RE = /\A#{Regexp.quote(REF_OPEN)}/
REF_CLOSE = '>))'
REF_CLOSE_RE = /\A#{Regexp.quote(REF_CLOSE)}/
FOOTNOTE_OPEN = '((-'
FOOTNOTE_OPEN_RE = /\A#{Regexp.quote(FOOTNOTE_OPEN)}/
FOOTNOTE_CLOSE = '-))'
FOOTNOTE_CLOSE_RE = /\A#{Regexp.quote(FOOTNOTE_CLOSE)}/
VERB_OPEN = "(('"
VERB_OPEN_RE = /\A#{Regexp.quote(VERB_OPEN)}/
VERB_CLOSE = "'))"
VERB_CLOSE_RE = /\A#{Regexp.quote(VERB_CLOSE)}/

BAR = "|"
BAR_RE = /\A#{Regexp.quote(BAR)}/
QUOTE = '"'
QUOTE_RE = /\A#{Regexp.quote(QUOTE)}/
SLASH = "/"
SLASH_RE = /\A#{Regexp.quote(SLASH)}/
BACK_SLASH = "\\"
BACK_SLASH_RE = /\A#{Regexp.quote(BACK_SLASH)}/
URL = "URL:"
URL_RE = /\A#{Regexp.quote(URL)}/

other_re_mode = Regexp::EXTENDED
other_re_mode |= Regexp::MULTILINE

OTHER_RE = Regexp.new(
  "\\A.+?(?=#{Regexp.quote(EM_OPEN)}|#{Regexp.quote(EM_CLOSE)}|
              #{Regexp.quote(CODE_OPEN)}|#{Regexp.quote(CODE_CLOSE)}|
              #{Regexp.quote(VAR_OPEN)}|#{Regexp.quote(VAR_CLOSE)}|
              #{Regexp.quote(KBD_OPEN)}|#{Regexp.quote(KBD_CLOSE)}|
              #{Regexp.quote(INDEX_OPEN)}|#{Regexp.quote(INDEX_CLOSE)}|
              #{Regexp.quote(REF_OPEN)}|#{Regexp.quote(REF_CLOSE)}|
            #{Regexp.quote(FOOTNOTE_OPEN)}|#{Regexp.quote(FOOTNOTE_CLOSE)}|
              #{Regexp.quote(VERB_OPEN)}|#{Regexp.quote(VERB_CLOSE)}|
              #{Regexp.quote(BAR)}|
              #{Regexp.quote(QUOTE)}|
              #{Regexp.quote(SLASH)}|
              #{Regexp.quote(BACK_SLASH)}|
              #{Regexp.quote(URL)})", other_re_mode)

# :startdoc:

##
# Creates a new parser for inline markup in the rd format.  The +block_parser+
# is used to for footnotes and labels in the inline text.

def initialize block_parser
  @block_parser = block_parser
end

##
# Parses the +inline+ text from RD format into RDoc format.

def parse inline
  @inline = inline
  @src = StringScanner.new inline
  @pre = "".dup
  @yydebug = true
  do_parse.to_s
end

##
# Returns the next token from the inline text

def next_token
  return [false, false] if @src.eos?
#  p @src.rest if @yydebug
  if ret = @src.scan(EM_OPEN_RE)
    @pre << ret
    [:EM_OPEN, ret]
  elsif ret = @src.scan(EM_CLOSE_RE)
    @pre << ret
    [:EM_CLOSE, ret]
  elsif ret = @src.scan(CODE_OPEN_RE)
    @pre << ret
    [:CODE_OPEN, ret]
  elsif ret = @src.scan(CODE_CLOSE_RE)
    @pre << ret
    [:CODE_CLOSE, ret]
  elsif ret = @src.scan(VAR_OPEN_RE)
    @pre << ret
    [:VAR_OPEN, ret]
  elsif ret = @src.scan(VAR_CLOSE_RE)
    @pre << ret
    [:VAR_CLOSE, ret]
  elsif ret = @src.scan(KBD_OPEN_RE)
    @pre << ret
    [:KBD_OPEN, ret]
  elsif ret = @src.scan(KBD_CLOSE_RE)
    @pre << ret
    [:KBD_CLOSE, ret]
  elsif ret = @src.scan(INDEX_OPEN_RE)
    @pre << ret
    [:INDEX_OPEN, ret]
  elsif ret = @src.scan(INDEX_CLOSE_RE)
    @pre << ret
    [:INDEX_CLOSE, ret]
  elsif ret = @src.scan(REF_OPEN_RE)
    @pre << ret
    [:REF_OPEN, ret]
  elsif ret = @src.scan(REF_CLOSE_RE)
    @pre << ret
    [:REF_CLOSE, ret]
  elsif ret = @src.scan(FOOTNOTE_OPEN_RE)
    @pre << ret
    [:FOOTNOTE_OPEN, ret]
  elsif ret = @src.scan(FOOTNOTE_CLOSE_RE)
    @pre << ret
    [:FOOTNOTE_CLOSE, ret]
  elsif ret = @src.scan(VERB_OPEN_RE)
    @pre << ret
    [:VERB_OPEN, ret]
  elsif ret = @src.scan(VERB_CLOSE_RE)
    @pre << ret
    [:VERB_CLOSE, ret]
  elsif ret = @src.scan(BAR_RE)
    @pre << ret
    [:BAR, ret]
  elsif ret = @src.scan(QUOTE_RE)
    @pre << ret
    [:QUOTE, ret]
  elsif ret = @src.scan(SLASH_RE)
    @pre << ret
    [:SLASH, ret]
  elsif ret = @src.scan(BACK_SLASH_RE)
    @pre << ret
    [:BACK_SLASH, ret]
  elsif ret = @src.scan(URL_RE)
    @pre << ret
    [:URL, ret]
  elsif ret = @src.scan(OTHER_RE)
    @pre << ret
    [:OTHER, ret]
  else
    ret = @src.rest
    @pre << ret
    @src.terminate
    [:OTHER, ret]
  end
end

##
# Raises a ParseError when invalid formatting is found

def on_error(et, ev, values)
  lines_of_rest = @src.rest.lines.to_a.length
  prev_words = prev_words_on_error(ev)
  at = 4 + prev_words.length

  message = <<-MSG
RD syntax error: line #{@block_parser.line_index - lines_of_rest}:
...#{prev_words} #{(ev||'')} #{next_words_on_error()} ...
  MSG

  message << " " * at + "^" * (ev ? ev.length : 0) + "\n"
  raise ParseError, message
end

##
# Returns words before the error

def prev_words_on_error(ev)
  pre = @pre
  if ev and /#{Regexp.quote(ev)}$/ =~ pre
    pre = $`
  end
  last_line(pre)
end

##
# Returns the last line of +src+

def last_line(src)
  if n = src.rindex("\n")
    src[(n+1) .. -1]
  else
    src
  end
end
private :last_line

##
# Returns words following an error

def next_words_on_error
  if n = @src.rest.index("\n")
    @src.rest[0 .. (n-1)]
  else
    @src.rest
  end
end

##
# Creates a new RDoc::RD::Inline for the +rdoc+ markup and the raw +reference+

def inline rdoc, reference = rdoc
  RDoc::RD::Inline.new rdoc, reference
end

# :stopdoc:
##### State transition tables begin ###

racc_action_table = [
   104,   103,   102,   100,   101,    99,   115,   116,   117,    29,
   105,   106,   107,   108,   109,   110,   111,   112,   113,   114,
    84,   118,   119,    63,    64,    65,    61,    81,    62,    76,
    78,    79,    85,    66,    67,    68,    69,    70,    71,    72,
    73,    74,    75,    77,    80,   149,    63,    64,    65,   153,
    81,    62,    76,    78,    79,    86,    66,    67,    68,    69,
    70,    71,    72,    73,    74,    75,    77,    80,   152,   104,
   103,   102,   100,   101,    99,   115,   116,   117,    87,   105,
   106,   107,   108,   109,   110,   111,   112,   113,   114,    88,
   118,   119,   104,   103,   102,   100,   101,    99,   115,   116,
   117,    89,   105,   106,   107,   108,   109,   110,   111,   112,
   113,   114,    96,   118,   119,   104,   103,   102,   100,   101,
    99,   115,   116,   117,   124,   105,   106,   107,   108,   109,
   110,   111,   112,   113,   114,   137,   118,   119,    22,    23,
    24,    25,    26,    21,    18,    19,   176,   177,    13,   148,
    14,   154,    15,   137,    16,   161,    17,   164,   173,    20,
    22,    23,    24,    25,    26,    21,    18,    19,   175,   177,
    13,   nil,    14,   nil,    15,   nil,    16,   nil,    17,   nil,
   nil,    20,    22,    23,    24,    25,    26,    21,    18,    19,
   nil,   nil,    13,   nil,    14,   nil,    15,   nil,    16,   nil,
    17,   nil,   nil,    20,    22,    23,    24,    25,    26,    21,
    18,    19,   nil,   nil,    13,   nil,    14,   nil,    15,   nil,
    16,   nil,    17,   nil,   nil,    20,    22,    23,    24,    25,
    26,    21,    18,    19,   nil,   nil,    13,   nil,    14,   nil,
    15,   nil,    16,   nil,    17,   nil,   nil,    20,    22,    23,
    24,    25,    26,    21,    18,    19,   nil,   nil,    13,   nil,
    14,   nil,    15,   nil,    16,   nil,    17,   nil,   nil,    20,
    22,    23,    24,    25,    26,    21,    18,    19,   nil,   nil,
    13,   nil,    14,   nil,    15,   nil,    16,   nil,    17,    42,
   nil,    20,    54,    38,    53,    55,    56,    57,   nil,    13,
   nil,    14,   nil,    15,   nil,    16,   nil,    17,   nil,   nil,
    20,    22,    23,    24,    25,    26,    21,    18,    19,   nil,
   nil,    13,   nil,    14,   nil,    15,   nil,    16,   nil,    17,
   nil,   nil,    20,    63,    64,    65,    61,    81,    62,    76,
    78,    79,   nil,    66,    67,    68,    69,    70,    71,    72,
    73,    74,    75,    77,    80,   122,   nil,   nil,    54,   nil,
    53,    55,    56,    57,   nil,    13,   nil,    14,   nil,    15,
   nil,    16,   nil,    17,   145,   nil,    20,    54,   133,    53,
    55,    56,    57,   nil,    13,   nil,    14,   nil,    15,   nil,
    16,   nil,    17,   145,   nil,    20,    54,   133,    53,    55,
    56,    57,   nil,    13,   nil,    14,   nil,    15,   nil,    16,
   nil,    17,   145,   nil,    20,    54,   133,    53,    55,    56,
    57,   nil,    13,   nil,    14,   nil,    15,   nil,    16,   nil,
    17,   145,   nil,    20,    54,   133,    53,    55,    56,    57,
   nil,    13,   nil,    14,   nil,    15,   nil,    16,   nil,    17,
   nil,   nil,    20,   135,   136,    54,   133,    53,    55,    56,
    57,   nil,    13,   nil,    14,   nil,    15,   nil,    16,   nil,
    17,   nil,   nil,    20,   135,   136,    54,   133,    53,    55,
    56,    57,   nil,    13,   nil,    14,   nil,    15,   nil,    16,
   nil,    17,   nil,   nil,    20,   135,   136,    54,   133,    53,
    55,    56,    57,   nil,    13,   nil,    14,   nil,    15,   nil,
    16,   nil,    17,    95,   nil,    20,    54,    91,    53,    55,
    56,    57,   145,   nil,   nil,    54,   133,    53,    55,    56,
    57,   158,   nil,   nil,    54,   nil,    53,    55,    56,    57,
   165,   135,   136,    54,   133,    53,    55,    56,    57,   145,
   nil,   nil,    54,   133,    53,    55,    56,    57,   172,   135,
   136,    54,   133,    53,    55,    56,    57,   174,   135,   136,
    54,   133,    53,    55,    56,    57,   178,   135,   136,    54,
   133,    53,    55,    56,    57,   135,   136,    54,   133,    53,
    55,    56,    57,   135,   136,    54,   133,    53,    55,    56,
    57,   135,   136,    54,   133,    53,    55,    56,    57,    22,
    23,    24,    25,    26,    21 ]

racc_action_check = [
    38,    38,    38,    38,    38,    38,    38,    38,    38,     1,
    38,    38,    38,    38,    38,    38,    38,    38,    38,    38,
    29,    38,    38,    59,    59,    59,    59,    59,    59,    59,
    59,    59,    31,    59,    59,    59,    59,    59,    59,    59,
    59,    59,    59,    59,    59,    59,    61,    61,    61,    61,
    61,    61,    61,    61,    61,    32,    61,    61,    61,    61,
    61,    61,    61,    61,    61,    61,    61,    61,    61,    91,
    91,    91,    91,    91,    91,    91,    91,    91,    33,    91,
    91,    91,    91,    91,    91,    91,    91,    91,    91,    34,
    91,    91,    97,    97,    97,    97,    97,    97,    97,    97,
    97,    35,    97,    97,    97,    97,    97,    97,    97,    97,
    97,    97,    37,    97,    97,   155,   155,   155,   155,   155,
   155,   155,   155,   155,    41,   155,   155,   155,   155,   155,
   155,   155,   155,   155,   155,    43,   155,   155,     0,     0,
     0,     0,     0,     0,     0,     0,   165,   165,     0,    58,
     0,    90,     0,    94,     0,   100,     0,   125,   162,     0,
     2,     2,     2,     2,     2,     2,     2,     2,   164,   172,
     2,   nil,     2,   nil,     2,   nil,     2,   nil,     2,   nil,
   nil,     2,    13,    13,    13,    13,    13,    13,    13,    13,
   nil,   nil,    13,   nil,    13,   nil,    13,   nil,    13,   nil,
    13,   nil,   nil,    13,    14,    14,    14,    14,    14,    14,
    14,    14,   nil,   nil,    14,   nil,    14,   nil,    14,   nil,
    14,   nil,    14,   nil,   nil,    14,    15,    15,    15,    15,
    15,    15,    15,    15,   nil,   nil,    15,   nil,    15,   nil,
    15,   nil,    15,   nil,    15,   nil,   nil,    15,    16,    16,
    16,    16,    16,    16,    16,    16,   nil,   nil,    16,   nil,
    16,   nil,    16,   nil,    16,   nil,    16,   nil,   nil,    16,
    17,    17,    17,    17,    17,    17,    17,    17,   nil,   nil,
    17,   nil,    17,   nil,    17,   nil,    17,   nil,    17,    18,
   nil,    17,    18,    18,    18,    18,    18,    18,   nil,    18,
   nil,    18,   nil,    18,   nil,    18,   nil,    18,   nil,   nil,
    18,    19,    19,    19,    19,    19,    19,    19,    19,   nil,
   nil,    19,   nil,    19,   nil,    19,   nil,    19,   nil,    19,
   nil,   nil,    19,    20,    20,    20,    20,    20,    20,    20,
    20,    20,   nil,    20,    20,    20,    20,    20,    20,    20,
    20,    20,    20,    20,    20,    39,   nil,   nil,    39,   nil,
    39,    39,    39,    39,   nil,    39,   nil,    39,   nil,    39,
   nil,    39,   nil,    39,    44,   nil,    39,    44,    44,    44,
    44,    44,    44,   nil,    44,   nil,    44,   nil,    44,   nil,
    44,   nil,    44,    45,   nil,    44,    45,    45,    45,    45,
    45,    45,   nil,    45,   nil,    45,   nil,    45,   nil,    45,
   nil,    45,   138,   nil,    45,   138,   138,   138,   138,   138,
   138,   nil,   138,   nil,   138,   nil,   138,   nil,   138,   nil,
   138,   146,   nil,   138,   146,   146,   146,   146,   146,   146,
   nil,   146,   nil,   146,   nil,   146,   nil,   146,   nil,   146,
   nil,   nil,   146,    42,    42,    42,    42,    42,    42,    42,
    42,   nil,    42,   nil,    42,   nil,    42,   nil,    42,   nil,
    42,   nil,   nil,    42,   122,   122,   122,   122,   122,   122,
   122,   122,   nil,   122,   nil,   122,   nil,   122,   nil,   122,
   nil,   122,   nil,   nil,   122,   127,   127,   127,   127,   127,
   127,   127,   127,   nil,   127,   nil,   127,   nil,   127,   nil,
   127,   nil,   127,    36,   nil,   127,    36,    36,    36,    36,
    36,    36,    52,   nil,   nil,    52,    52,    52,    52,    52,
    52,    92,   nil,   nil,    92,   nil,    92,    92,    92,    92,
   126,   126,   126,   126,   126,   126,   126,   126,   126,   142,
   nil,   nil,   142,   142,   142,   142,   142,   142,   159,   159,
   159,   159,   159,   159,   159,   159,   159,   163,   163,   163,
   163,   163,   163,   163,   163,   163,   171,   171,   171,   171,
   171,   171,   171,   171,   171,    95,    95,    95,    95,    95,
    95,    95,    95,   158,   158,   158,   158,   158,   158,   158,
   158,   168,   168,   168,   168,   168,   168,   168,   168,    27,
    27,    27,    27,    27,    27 ]

racc_action_pointer = [
   135,     9,   157,   nil,   nil,   nil,   nil,   nil,   nil,   nil,
   nil,   nil,   nil,   179,   201,   223,   245,   267,   286,   308,
   330,   nil,   nil,   nil,   nil,   nil,   nil,   606,   nil,    20,
   nil,    18,    39,    60,    69,    79,   510,    89,    -3,   352,
   nil,   120,   449,   130,   371,   390,   nil,   nil,   nil,   nil,
   nil,   nil,   519,   nil,   nil,   nil,   nil,   nil,   138,    20,
   nil,    43,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,
   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,
   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,
   128,    66,   528,   nil,   148,   581,   nil,    89,   nil,   nil,
   149,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,
   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,
   nil,   nil,   470,   nil,   nil,   154,   537,   491,   nil,   nil,
   nil,   nil,   nil,   nil,   nil,   nil,   nil,   nil,   409,   nil,
   nil,   nil,   546,   nil,   nil,   nil,   428,   nil,   nil,   nil,
   nil,   nil,   nil,   nil,   nil,   112,   nil,   nil,   589,   555,
   nil,   nil,   155,   564,   164,   142,   nil,   nil,   597,   nil,
   nil,   573,   164,   nil,   nil,   nil,   nil,   nil,   nil ]

racc_action_default = [
  -138,  -138,    -1,    -3,    -4,    -5,    -6,    -7,    -8,    -9,
   -10,   -11,   -12,  -138,  -138,  -138,  -138,  -138,  -138,  -138,
  -138,  -103,  -104,  -105,  -106,  -107,  -108,  -111,  -110,  -138,
    -2,  -138,  -138,  -138,  -138,  -138,  -138,  -138,  -138,   -27,
   -26,   -35,  -138,   -58,   -41,   -40,   -47,   -48,   -49,   -50,
   -51,   -52,   -63,   -66,   -67,   -68,   -69,   -70,  -138,  -138,
  -112,  -138,  -116,  -117,  -118,  -119,  -120,  -121,  -122,  -123,
  -124,  -125,  -126,  -127,  -128,  -129,  -130,  -131,  -132,  -133,
  -134,  -135,  -137,  -109,   179,   -13,   -14,   -15,   -16,   -17,
  -138,  -138,   -23,   -22,   -33,  -138,   -19,   -24,   -79,   -80,
  -138,   -82,   -83,   -84,   -85,   -86,   -87,   -88,   -89,   -90,
   -91,   -92,   -93,   -94,   -95,   -96,   -97,   -98,   -99,  -100,
   -25,   -35,  -138,   -58,   -28,  -138,   -59,   -42,   -46,   -55,
   -56,   -65,   -71,   -72,   -75,   -76,   -77,   -31,   -38,   -44,
   -53,   -54,   -57,   -61,   -73,   -74,   -39,   -62,  -101,  -102,
  -136,  -113,  -114,  -115,   -18,   -20,   -21,   -33,  -138,  -138,
   -78,   -81,  -138,   -59,   -36,   -37,   -64,   -45,   -59,   -43,
   -60,  -138,   -34,   -36,   -37,   -29,   -30,   -32,   -34 ]

racc_goto_table = [
   126,    44,   125,    43,   144,   144,   160,    93,    97,    52,
   166,    82,   144,    40,    41,    39,   138,   146,   169,    30,
    36,    94,    44,     1,   123,   129,   169,    52,    90,    37,
    52,   167,   147,    92,   120,   121,    31,    32,    33,    34,
    35,   170,    58,   166,    59,    83,   170,   166,   151,   nil,
   150,   nil,   166,   159,     4,   166,     4,   nil,   nil,   nil,
   nil,   155,   nil,   156,   160,   nil,   nil,     4,     4,     4,
     4,     4,   nil,     4,     5,   nil,     5,   157,   nil,   nil,
   163,   nil,   162,    52,   nil,   168,   nil,     5,     5,     5,
     5,     5,   nil,     5,   nil,   nil,   nil,   nil,   144,   nil,
   nil,   nil,   144,   nil,   nil,   129,   144,   144,   nil,     6,
   129,     6,   nil,   nil,   nil,   nil,   171,     7,   nil,     7,
   nil,   nil,     6,     6,     6,     6,     6,     8,     6,     8,
     7,     7,     7,     7,     7,    11,     7,    11,   nil,   nil,
     8,     8,     8,     8,     8,   nil,     8,   nil,    11,    11,
    11,    11,    11,   nil,    11 ]

racc_goto_check = [
    22,    24,    21,    23,    36,    36,    37,    18,    16,    34,
    35,    41,    36,    19,    20,    17,    25,    25,    28,     3,
    13,    23,    24,     1,    23,    24,    28,    34,    14,    15,
    34,    29,    32,    17,    19,    20,     1,     1,     1,     1,
     1,    33,     1,    35,    38,    39,    33,    35,    42,   nil,
    41,   nil,    35,    22,     4,    35,     4,   nil,   nil,   nil,
   nil,    16,   nil,    18,    37,   nil,   nil,     4,     4,     4,
     4,     4,   nil,     4,     5,   nil,     5,    23,   nil,   nil,
    22,   nil,    21,    34,   nil,    22,   nil,     5,     5,     5,
     5,     5,   nil,     5,   nil,   nil,   nil,   nil,    36,   nil,
   nil,   nil,    36,   nil,   nil,    24,    36,    36,   nil,     6,
    24,     6,   nil,   nil,   nil,   nil,    22,     7,   nil,     7,
   nil,   nil,     6,     6,     6,     6,     6,     8,     6,     8,
     7,     7,     7,     7,     7,    11,     7,    11,   nil,   nil,
     8,     8,     8,     8,     8,   nil,     8,   nil,    11,    11,
    11,    11,    11,   nil,    11 ]

racc_goto_pointer = [
   nil,    23,   nil,    17,    54,    74,   109,   117,   127,   nil,
   nil,   135,   nil,     2,    -8,    11,   -30,    -3,   -29,    -5,
    -4,   -40,   -42,   -15,   -17,   -28,   nil,   nil,  -120,   -96,
   nil,   nil,   -20,  -101,    -9,  -116,   -40,   -91,    24,    18,
   nil,    -9,   -13 ]

racc_goto_default = [
   nil,   nil,     2,     3,    46,    47,    48,    49,    50,     9,
    10,    51,    12,   nil,   nil,   nil,   nil,   nil,   nil,   nil,
   nil,   nil,   nil,   nil,   140,   nil,    45,   127,   139,   128,
   141,   130,   142,   143,   132,   131,   134,    98,   nil,    28,
    27,   nil,    60 ]

racc_reduce_table = [
  0, 0, :racc_error,
  1, 27, :_reduce_none,
  2, 28, :_reduce_2,
  1, 28, :_reduce_3,
  1, 29, :_reduce_none,
  1, 29, :_reduce_none,
  1, 29, :_reduce_none,
  1, 29, :_reduce_none,
  1, 29, :_reduce_none,
  1, 29, :_reduce_none,
  1, 29, :_reduce_none,
  1, 29, :_reduce_none,
  1, 29, :_reduce_none,
  3, 30, :_reduce_13,
  3, 31, :_reduce_14,
  3, 32, :_reduce_15,
  3, 33, :_reduce_16,
  3, 34, :_reduce_17,
  4, 35, :_reduce_18,
  3, 35, :_reduce_19,
  2, 40, :_reduce_20,
  2, 40, :_reduce_21,
  1, 40, :_reduce_22,
  1, 40, :_reduce_23,
  2, 41, :_reduce_24,
  2, 41, :_reduce_25,
  1, 41, :_reduce_26,
  1, 41, :_reduce_27,
  2, 39, :_reduce_none,
  4, 39, :_reduce_29,
  4, 39, :_reduce_30,
  2, 43, :_reduce_31,
  4, 43, :_reduce_32,
  1, 44, :_reduce_33,
  3, 44, :_reduce_34,
  1, 45, :_reduce_none,
  3, 45, :_reduce_36,
  3, 45, :_reduce_37,
  2, 46, :_reduce_38,
  2, 46, :_reduce_39,
  1, 46, :_reduce_40,
  1, 46, :_reduce_41,
  1, 47, :_reduce_none,
  2, 51, :_reduce_43,
  1, 51, :_reduce_44,
  2, 53, :_reduce_45,
  1, 53, :_reduce_46,
  1, 50, :_reduce_none,
  1, 50, :_reduce_none,
  1, 50, :_reduce_none,
  1, 50, :_reduce_none,
  1, 50, :_reduce_none,
  1, 50, :_reduce_none,
  1, 54, :_reduce_none,
  1, 54, :_reduce_none,
  1, 55, :_reduce_none,
  1, 55, :_reduce_none,
  1, 56, :_reduce_57,
  1, 52, :_reduce_58,
  1, 57, :_reduce_59,
  2, 58, :_reduce_60,
  1, 58, :_reduce_none,
  2, 49, :_reduce_62,
  1, 49, :_reduce_none,
  2, 48, :_reduce_64,
  1, 48, :_reduce_none,
  1, 60, :_reduce_none,
  1, 60, :_reduce_none,
  1, 60, :_reduce_none,
  1, 60, :_reduce_none,
  1, 60, :_reduce_none,
  1, 62, :_reduce_none,
  1, 62, :_reduce_none,
  1, 59, :_reduce_none,
  1, 59, :_reduce_none,
  1, 61, :_reduce_none,
  1, 61, :_reduce_none,
  1, 61, :_reduce_none,
  2, 42, :_reduce_78,
  1, 42, :_reduce_none,
  1, 63, :_reduce_none,
  2, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  1, 63, :_reduce_none,
  3, 36, :_reduce_101,
  3, 37, :_reduce_102,
  1, 65, :_reduce_none,
  1, 65, :_reduce_none,
  1, 65, :_reduce_none,
  1, 65, :_reduce_none,
  1, 65, :_reduce_none,
  1, 65, :_reduce_none,
  2, 66, :_reduce_109,
  1, 66, :_reduce_none,
  1, 38, :_reduce_111,
  1, 67, :_reduce_none,
  2, 67, :_reduce_113,
  2, 67, :_reduce_114,
  2, 67, :_reduce_115,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  1, 68, :_reduce_none,
  2, 64, :_reduce_136,
  1, 64, :_reduce_none ]

racc_reduce_n = 138

racc_shift_n = 179

racc_token_table = {
  false => 0,
  :error => 1,
  :EX_LOW => 2,
  :QUOTE => 3,
  :BAR => 4,
  :SLASH => 5,
  :BACK_SLASH => 6,
  :URL => 7,
  :OTHER => 8,
  :REF_OPEN => 9,
  :FOOTNOTE_OPEN => 10,
  :FOOTNOTE_CLOSE => 11,
  :EX_HIGH => 12,
  :EM_OPEN => 13,
  :EM_CLOSE => 14,
  :CODE_OPEN => 15,
  :CODE_CLOSE => 16,
  :VAR_OPEN => 17,
  :VAR_CLOSE => 18,
  :KBD_OPEN => 19,
  :KBD_CLOSE => 20,
  :INDEX_OPEN => 21,
  :INDEX_CLOSE => 22,
  :REF_CLOSE => 23,
  :VERB_OPEN => 24,
  :VERB_CLOSE => 25 }

racc_nt_base = 26

racc_use_result_var = true

Racc_arg = [
  racc_action_table,
  racc_action_check,
  racc_action_default,
  racc_action_pointer,
  racc_goto_table,
  racc_goto_check,
  racc_goto_default,
  racc_goto_pointer,
  racc_nt_base,
  racc_reduce_table,
  racc_token_table,
  racc_shift_n,
  racc_reduce_n,
  racc_use_result_var ]

Racc_token_to_s_table = [
  "$end",
  "error",
  "EX_LOW",
  "QUOTE",
  "BAR",
  "SLASH",
  "BACK_SLASH",
  "URL",
  "OTHER",
  "REF_OPEN",
  "FOOTNOTE_OPEN",
  "FOOTNOTE_CLOSE",
  "EX_HIGH",
  "EM_OPEN",
  "EM_CLOSE",
  "CODE_OPEN",
  "CODE_CLOSE",
  "VAR_OPEN",
  "VAR_CLOSE",
  "KBD_OPEN",
  "KBD_CLOSE",
  "INDEX_OPEN",
  "INDEX_CLOSE",
  "REF_CLOSE",
  "VERB_OPEN",
  "VERB_CLOSE",
  "$start",
  "content",
  "elements",
  "element",
  "emphasis",
  "code",
  "var",
  "keyboard",
  "index",
  "reference",
  "footnote",
  "verb",
  "normal_str_ele",
  "substitute",
  "ref_label",
  "ref_label2",
  "ref_url_strings",
  "filename",
  "element_label",
  "element_label2",
  "ref_subst_content",
  "ref_subst_content_q",
  "ref_subst_strings_q",
  "ref_subst_strings_first",
  "ref_subst_ele2",
  "ref_subst_eles",
  "ref_subst_str_ele_first",
  "ref_subst_eles_q",
  "ref_subst_ele",
  "ref_subst_ele_q",
  "ref_subst_str_ele",
  "ref_subst_str_ele_q",
  "ref_subst_strings",
  "ref_subst_string3",
  "ref_subst_string",
  "ref_subst_string_q",
  "ref_subst_string2",
  "ref_url_string",
  "verb_strings",
  "normal_string",
  "normal_strings",
  "verb_string",
  "verb_normal_string" ]

Racc_debug_parser = false

##### State transition tables end #####

# reduce 0 omitted

# reduce 1 omitted

def _reduce_2(val, _values, result)
 result.append val[1]
    result
end

def _reduce_3(val, _values, result)
 result = val[0]
    result
end

# reduce 4 omitted

# reduce 5 omitted

# reduce 6 omitted

# reduce 7 omitted

# reduce 8 omitted

# reduce 9 omitted

# reduce 10 omitted

# reduce 11 omitted

# reduce 12 omitted

def _reduce_13(val, _values, result)
      content = val[1]
      result = inline "<em>#{content}</em>", content

    result
end

def _reduce_14(val, _values, result)
      content = val[1]
      result = inline "<code>#{content}</code>", content

    result
end

def _reduce_15(val, _values, result)
      content = val[1]
      result = inline "+#{content}+", content

    result
end

def _reduce_16(val, _values, result)
      content = val[1]
      result = inline "<tt>#{content}</tt>", content

    result
end

def _reduce_17(val, _values, result)
      label = val[1]
      @block_parser.add_label label.reference
      result = "<span id=\"label-#{label}\">#{label}</span>"

    result
end

def _reduce_18(val, _values, result)
      result = "{#{val[1]}}[#{val[2].join}]"

    result
end

def _reduce_19(val, _values, result)
      scheme, inline = val[1]

      result = "{#{inline}}[#{scheme}#{inline.reference}]"

    result
end

def _reduce_20(val, _values, result)
      result = [nil, inline(val[1])]

    result
end

def _reduce_21(val, _values, result)
      result = [
        'rdoc-label:',
        inline("#{val[0].reference}/#{val[1].reference}")
      ]

    result
end

def _reduce_22(val, _values, result)
      result = ['rdoc-label:', val[0].reference]

    result
end

def _reduce_23(val, _values, result)
      result = ['rdoc-label:', "#{val[0].reference}/"]

    result
end

def _reduce_24(val, _values, result)
      result = [nil, inline(val[1])]

    result
end

def _reduce_25(val, _values, result)
      result = [
        'rdoc-label:',
        inline("#{val[0].reference}/#{val[1].reference}")
      ]

    result
end

def _reduce_26(val, _values, result)
      result = ['rdoc-label:', val[0]]

    result
end

def _reduce_27(val, _values, result)
      ref = val[0].reference
      result = ['rdoc-label:', inline(ref, "#{ref}/")]

    result
end

# reduce 28 omitted

def _reduce_29(val, _values, result)
 result = val[1]
    result
end

def _reduce_30(val, _values, result)
 result = val[1]
    result
end

def _reduce_31(val, _values, result)
      result = inline val[0]

    result
end

def _reduce_32(val, _values, result)
      result = inline "\"#{val[1]}\""

    result
end

def _reduce_33(val, _values, result)
      result = inline val[0]

    result
end

def _reduce_34(val, _values, result)
      result = inline "\"#{val[1]}\""

    result
end

# reduce 35 omitted

def _reduce_36(val, _values, result)
 result = val[1]
    result
end

def _reduce_37(val, _values, result)
 result = inline val[1]
    result
end

def _reduce_38(val, _values, result)
      result = val[0].append val[1]

    result
end

def _reduce_39(val, _values, result)
      result = val[0].append val[1]

    result
end

def _reduce_40(val, _values, result)
      result = val[0]

    result
end

def _reduce_41(val, _values, result)
      result = inline val[0]

    result
end

# reduce 42 omitted

def _reduce_43(val, _values, result)
      result = val[0].append val[1]

    result
end

def _reduce_44(val, _values, result)
      result = inline val[0]

    result
end

def _reduce_45(val, _values, result)
      result = val[0].append val[1]

    result
end

def _reduce_46(val, _values, result)
      result = val[0]

    result
end

# reduce 47 omitted

# reduce 48 omitted

# reduce 49 omitted

# reduce 50 omitted

# reduce 51 omitted

# reduce 52 omitted

# reduce 53 omitted

# reduce 54 omitted

# reduce 55 omitted

# reduce 56 omitted

def _reduce_57(val, _values, result)
      result = val[0]

    result
end

def _reduce_58(val, _values, result)
      result = inline val[0]

    result
end

def _reduce_59(val, _values, result)
      result = inline val[0]

    result
end

def _reduce_60(val, _values, result)
 result << val[1]
    result
end

# reduce 61 omitted

def _reduce_62(val, _values, result)
      result << val[1]

    result
end

# reduce 63 omitted

def _reduce_64(val, _values, result)
      result << val[1]

    result
end

# reduce 65 omitted

# reduce 66 omitted

# reduce 67 omitted

# reduce 68 omitted

# reduce 69 omitted

# reduce 70 omitted

# reduce 71 omitted

# reduce 72 omitted

# reduce 73 omitted

# reduce 74 omitted

# re                                                                                          gems/gems/rdoc-6.5.1.1/lib/rdoc/rd/inline.rb                                                        0000644                 00000002613 15040313415 0014070 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ce_101(val, _values, result)
      index = @block_parser.add_footnote val[1].rdoc
      result = "{*#{index}}[rdoc-label:foottext-#{index}:footmark-#{index}]"

    result
end

def _reduce_102(val, _values, result)
      result = inline "<tt>#{val[1]}</tt>", val[1]

    result
end

# reduce 103 omitted

# reduce 104 omitted

# reduce 105 omitted

# reduce 106 omitted

# reduce 107 omitted

# reduce 108 omitted

def _reduce_109(val, _values, result)
 result << val[1]
    result
end

# reduce 110 omitted

def _reduce_111(val, _values, result)
      result = inline val[0]

    result
end

# reduce 112 omitted

def _reduce_113(val, _values, result)
 result = val[1]
    result
end

def _reduce_114(val, _values, result)
 result = val[1]
    result
end

def _reduce_115(val, _values, result)
 result = val[1]
    result
end

# reduce 116 omitted

# reduce 117 omitted

# reduce 118 omitted

# reduce 119 omitted

# reduce 120 omitted

# reduce 121 omitted

# reduce 122 omitted

# reduce 123 omitted

# reduce 124 omitted

# reduce 125 omitted

# reduce 126 omitted

# reduce 127 omitted

# reduce 128 omitted

# reduce 129 omitted

# reduce 130 omitted

# reduce 131 omitted

# reduce 132 omitted

# reduce 133 omitted

# reduce 134 omitted

# reduce 135 omitted

def _reduce_136(val, _values, result)
 result << val[1]
    result
end

# reduce 137 omitted

def _reduce_none(val, _values, result)
  val[0]
end

end                                                                                                                       gems/gems/rdoc-6.5.1.1/lib/rdoc/rd/block_parser.rb                                                  0000644                 00000054447 15040313415 0015274 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# Inline keeps track of markup and labels to create proper links.

class RDoc::RD::Inline

  ##
  # The text of the reference

  attr_reader :reference

  ##
  # The markup of this reference in RDoc format

  attr_reader :rdoc

  ##
  # Creates a new Inline for +rdoc+ and +reference+.
  #
  # +rdoc+ may be another Inline or a String.  If +reference+ is not given it
  # will use the text from +rdoc+.

  def self.new rdoc, reference = rdoc
    if self === rdoc and reference.equal? rdoc then
      rdoc
    else
      super
    end
  end

  ##
  # Initializes the Inline with +rdoc+ and +inline+

  def initialize rdoc, reference # :not-new:
    @reference = reference.equal?(rdoc) ? reference.dup : reference

    # unpack
    @reference = @reference.reference if self.class === @reference
    @rdoc      = rdoc
  end

  def == other # :nodoc:
    self.class === other and
      @reference == other.reference and @rdoc == other.rdoc
  end

  ##
  # Appends +more+ to this inline.  +more+ may be a String or another Inline.

  def append more
    case more
    when String then
      @reference += more
      @rdoc      += more
    when RDoc::RD::Inline then
      @reference += more.reference
      @rdoc      += more.rdoc
    else
      raise "unknown thingy #{more}"
    end

    self
  end

  def inspect # :nodoc:
    "(inline: #{self})"
  end

  alias to_s rdoc # :nodoc:

end

                                                                                                                     gems/gems/rdoc-6.5.1.1/lib/rdoc/rd/block_parser.rb                                                  0000644                 00000054447 15040313415 0015274 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#
# DO NOT MODIFY!!!!
# This file is automatically generated by Racc 1.6.0
# from Racc grammar file "".
#

require 'racc/parser.rb'

class RDoc::RD

##
# RD format parser for headings, paragraphs, lists, verbatim sections that
# exist as blocks.

class BlockParser < Racc::Parser


# :stopdoc:

MARK_TO_LEVEL = {
  '='    => 1,
  '=='   => 2,
  '==='  => 3,
  '====' => 4,
  '+'    => 5,
  '++'   => 6,
}

# :startdoc:

##
# Footnotes for this document

attr_reader :footnotes

##
# Labels for items in this document

attr_reader :labels

##
# Path to find included files in

attr_accessor :include_path

##
# Creates a new RDoc::RD::BlockParser.  Use #parse to parse an rd-format
# document.

def initialize
  @inline_parser = RDoc::RD::InlineParser.new self
  @include_path = []

  # for testing
  @footnotes = []
  @labels    = {}
end

##
# Parses +src+ and returns an RDoc::Markup::Document.

def parse src
  @src = src
  @src.push false

  @footnotes = []
  @labels    = {}

  # @i: index(line no.) of src
  @i = 0

  # stack for current indentation
  @indent_stack = []

  # how indented.
  @current_indent = @indent_stack.join("")

  # RDoc::RD::BlockParser for tmp src
  @subparser = nil

  # which part is in now
  @in_part = nil
  @part_content = []

  @in_verbatim = false

  @yydebug = true

  document = do_parse

  unless @footnotes.empty? then
    blankline = document.parts.pop

    document.parts << RDoc::Markup::Rule.new(1)
    document.parts.concat @footnotes

    document.parts.push blankline
  end

  document
end

##
# Returns the next token from the document

def next_token # :nodoc:
  # preprocessing
  # if it is not in RD part
  # => method
  while @in_part != "rd"
    line = @src[@i]
    @i += 1 # next line

    case line
    # src end
    when false
      return [false, false]
    # RD part begin
    when /^=begin\s*(?:\bRD\b.*)?\s*$/
      if @in_part # if in non-RD part
        @part_content.push(line)
      else
        @in_part = "rd"
        return [:WHITELINE, "=begin\n"] # <= for textblockand
      end
    # non-RD part begin
    when /^=begin\s+(\w+)/
      part = $1
=begin # not imported to RDoc
      if @in_part # if in non-RD part
        @part_content.push(line)
      else
        @in_part = part if @tree.filter[part] # if filter exists
#  p "BEGIN_PART: #{@in_part}" # DEBUG
      end
=end
      @in_part = part
    # non-RD part end
    when /^=end(?:$|[\s\0\C-d\C-z])/
      if @in_part # if in non-RD part
=begin # not imported to RDoc
#  p "END_PART: #{@in_part}" # DEBUG
        # make Part-in object
        part = RDoc::RD::Part.new(@part_content.join(""), @tree, "r")
        @part_content.clear
        # call filter, part_out is output(Part object)
        part_out = @tree.filter[@in_part].call(part)

        if @tree.filter[@in_part].mode == :rd # if output is RD formatted
          subtree = parse_subtree(part_out.to_a)
        else # if output is target formatted
          basename = Tempfile.create(["rdtmp", ".#{@in_part}"], @tree.tmp_dir) do |tmpfile|
            tmpfile.print(part_out)
            File.basename(tmpfile.path)
          end
          subtree = parse_subtree(["=begin\n", "<<< #{basename}\n", "=end\n"])
        end
        @in_part = nil
        return [:SUBTREE, subtree]
=end
      end
    else
=begin # not imported to RDoc
      if @in_part # if in non-RD part
        @part_content.push(line)
      end
=end
    end
  end

  @current_indent = @indent_stack.join("")
  line = @src[@i]
  case line
  when false
    if_current_indent_equal("") do
      [false, false]
    end
  when /^=end/
    if_current_indent_equal("") do
      @in_part = nil
      [:WHITELINE, "=end"] # MUST CHANGE??
    end
  when /^\s*$/
    @i += 1 # next line
    return [:WHITELINE, ':WHITELINE']
  when /^\#/  # comment line
    @i += 1 # next line
    self.next_token()
  when /^(={1,4})(?!=)\s*(?=\S)/, /^(\+{1,2})(?!\+)\s*(?=\S)/
    rest = $'                    # '
    rest.strip!
    mark = $1
    if_current_indent_equal("") do
      return [:HEADLINE, [MARK_TO_LEVEL[mark], rest]]
    end
  when /^<<<\s*(\S+)/
    file = $1
    if_current_indent_equal("") do
      suffix = file[-3 .. -1]
      if suffix == ".rd" or suffix == ".rb"
        subtree = parse_subtree(get_included(file))
        [:SUBTREE, subtree]
      else
        [:INCLUDE, file]
      end
    end
  when /^(\s*)\*(\s*)/
    rest = $'                   # '
    newIndent = $2
    if_current_indent_equal($1) do
      if @in_verbatim
        [:STRINGLINE, line]
      else
        @indent_stack.push("\s" + newIndent)
        [:ITEMLISTLINE, rest]
      end
    end
  when /^(\s*)(\(\d+\))(\s*)/
    rest = $'                     # '
    mark = $2
    newIndent = $3
    if_current_indent_equal($1) do
      if @in_verbatim
        [:STRINGLINE, line]
      else
        @indent_stack.push("\s" * mark.size + newIndent)
        [:ENUMLISTLINE, rest]
      end
    end
  when /^(\s*):(\s*)/
    rest = $'                    # '
    newIndent = $2
    if_current_indent_equal($1) do
      if @in_verbatim
        [:STRINGLINE, line]
      else
        @indent_stack.push("\s#{$2}")
        [:DESCLISTLINE, rest]
      end
    end
  when /^(\s*)---(?!-|\s*$)/
    indent = $1
    rest = $'
    /\s*/ === rest
    term = $'
    new_indent = $&
    if_current_indent_equal(indent) do
      if @in_verbatim
        [:STRINGLINE, line]
      else
        @indent_stack.push("\s\s\s" + new_indent)
        [:METHODLISTLINE, term]
      end
    end
  when /^(\s*)/
    if_current_indent_equal($1) do
      [:STRINGLINE, line]
    end
  else
    raise "[BUG] parsing error may occurred."
  end
end

##
# Yields to the given block if +indent+ matches the current indent, otherwise
# an indentation token is processed.

def if_current_indent_equal(indent)
  indent = indent.sub(/\t/, "\s" * 8)
  if @current_indent == indent
    @i += 1 # next line
    yield
  elsif indent.index(@current_indent) == 0
    @indent_stack.push(indent[@current_indent.size .. -1])
    [:INDENT, ":INDENT"]
  else
    @indent_stack.pop
    [:DEDENT, ":DEDENT"]
  end
end
private :if_current_indent_equal

##
# Cuts off excess whitespace in +src+

def cut_off(src)
  ret = []
  whiteline_buf = []

  line = src.shift
  /^\s*/ =~ line

  indent = Regexp.quote($&)
  ret.push($')

  while line = src.shift
    if /^(\s*)$/ =~ line
      whiteline_buf.push(line)
    elsif /^#{indent}/ =~ line
      unless whiteline_buf.empty?
        ret.concat(whiteline_buf)
        whiteline_buf.clear
      end
      ret.push($')
    else
      raise "[BUG]: probably Parser Error while cutting off.\n"
    end
  end
  ret
end
private :cut_off

def set_term_to_element(parent, term)
#  parent.set_term_under_document_struct(term, @tree.document_struct)
  parent.set_term_without_document_struct(term)
end
private :set_term_to_element

##
# Raises a ParseError when invalid formatting is found

def on_error(et, ev, _values)
  prv, cur, nxt = format_line_num(@i, @i+1, @i+2)

  raise ParseError, <<Msg

RD syntax error: line #{@i+1}:
  #{prv}  |#{@src[@i-1].chomp}
  #{cur}=>|#{@src[@i].chomp}
  #{nxt}  |#{@src[@i+1].chomp}

Msg
end

##
# Current line number

def line_index
  @i
end

##
# Parses subtree +src+

def parse_subtree src
  @subparser ||= RDoc::RD::BlockParser.new

  @subparser.parse src
end
private :parse_subtree

##
# Retrieves the content for +file+ from the include_path

def get_included(file)
  included = []

  @include_path.each do |dir|
    file_name = File.join dir, file

    if File.exist? file_name then
      included = IO.readlines file_name
      break
    end
  end

  included
end
private :get_included

##
# Formats line numbers +line_numbers+ prettily

def format_line_num(*line_numbers)
  width = line_numbers.collect{|i| i.to_s.length }.max
  line_numbers.collect{|i| sprintf("%#{width}d", i) }
end
private :format_line_num

##
# Retrieves the content of +values+ as a single String

def content values
 values.map { |value| value.content }.join
end

##
# Creates a paragraph for +value+

def paragraph value
  content = cut_off(value).join(' ').rstrip
  contents = @inline_parser.parse content

  RDoc::Markup::Paragraph.new(*contents)
end

##
# Adds footnote +content+ to the document

def add_footnote content
  index = @footnotes.length / 2 + 1

  footmark_link = "{^#{index}}[rdoc-label:footmark-#{index}:foottext-#{index}]"

  @footnotes << RDoc::Markup::Paragraph.new(footmark_link, ' ', *content)
  @footnotes << RDoc::Markup::BlankLine.new

  index
end

##
# Adds label +label+ to the document

def add_label label
  @labels[label] = true

  label
end

# :stopdoc:

##### State transition tables begin ###

racc_action_table = [
    34,    35,    30,    33,    40,    34,    35,    30,    33,    40,
    65,    34,    35,    30,    33,    14,    73,    36,    38,    34,
    15,    88,    34,    35,    30,    33,    14,     9,    10,    11,
    12,    15,    34,    35,    30,    33,    14,     9,    10,    11,
    12,    15,    34,    35,    30,    33,    35,    47,    30,    54,
    33,    15,    34,    35,    30,    33,    54,    47,    14,    14,
    59,    15,    34,    35,    30,    33,    14,    73,    67,    76,
    77,    15,    34,    35,    30,    33,    14,    73,    54,    81,
    38,    15,    34,    35,    30,    33,    14,    73,    38,    40,
    83,    15,    34,    35,    30,    33,    14,    73,   nil,   nil,
   nil,    15,    34,    35,    30,    33,    14,    73,   nil,   nil,
   nil,    15,    34,    35,    30,    33,    14,    73,   nil,   nil,
   nil,    15,    34,    35,    30,    33,    14,    73,   nil,   nil,
   nil,    15,    34,    35,    30,    33,    14,    73,   nil,   nil,
   nil,    15,    34,    35,    30,    33,    14,    73,    61,    63,
   nil,    15,    14,    62,    60,    61,    63,    79,    61,    63,
    62,    87,   nil,    62,    34,    35,    30,    33 ]

racc_action_check = [
    41,    41,    41,    41,    41,    15,    15,    15,    15,    15,
    41,    86,    86,    86,    86,    86,    86,     1,    13,    22,
    86,    86,     0,     0,     0,     0,     0,     0,     0,     0,
     0,     0,     2,     2,     2,     2,     2,     2,     2,     2,
     2,     2,    24,    24,    24,    24,    25,    24,    28,    30,
    31,    24,    27,    27,    27,    27,    33,    27,    34,    35,
    36,    27,    45,    45,    45,    45,    45,    45,    44,    49,
    51,    45,    46,    46,    46,    46,    46,    46,    54,    56,
    57,    46,    47,    47,    47,    47,    47,    47,    58,    62,
    66,    47,    68,    68,    68,    68,    68,    68,   nil,   nil,
   nil,    68,    74,    74,    74,    74,    74,    74,   nil,   nil,
   nil,    74,    75,    75,    75,    75,    75,    75,   nil,   nil,
   nil,    75,    78,    78,    78,    78,    78,    78,   nil,   nil,
   nil,    78,    79,    79,    79,    79,    79,    79,   nil,   nil,
   nil,    79,    85,    85,    85,    85,    85,    85,    39,    39,
   nil,    85,    52,    39,    39,    82,    82,    52,    64,    64,
    82,    82,   nil,    64,    20,    20,    20,    20 ]

racc_action_pointer = [
    19,    17,    29,   nil,   nil,   nil,   nil,   nil,   nil,   nil,
   nil,   nil,   nil,    11,   nil,     2,   nil,   nil,   nil,   nil,
   161,   nil,    16,   nil,    39,    42,   nil,    49,    43,   nil,
    41,    44,   nil,    48,    51,    52,    60,   nil,   nil,   141,
   nil,    -3,   nil,   nil,    55,    59,    69,    79,   nil,    56,
   nil,    57,   145,   nil,    70,   nil,    66,    73,    81,   nil,
   nil,   nil,    82,   nil,   151,   nil,    77,   nil,    89,   nil,
   nil,   nil,   nil,   nil,    99,   109,   nil,   nil,   119,   129,
   nil,   nil,   148,   nil,   nil,   139,     8,   nil,   nil ]

racc_action_default = [
    -2,   -73,    -1,    -4,    -5,    -6,    -7,    -8,    -9,   -10,
   -11,   -12,   -13,   -14,   -16,   -73,   -23,   -24,   -25,   -26,
   -27,   -31,   -32,   -34,   -72,   -36,   -38,   -72,   -40,   -42,
   -59,   -44,   -46,   -59,   -63,   -65,   -73,    -3,   -15,   -73,
   -22,   -73,   -30,   -33,   -73,   -69,   -70,   -71,   -37,   -73,
   -41,   -73,   -51,   -58,   -61,   -45,   -73,   -62,   -64,    89,
   -17,   -19,   -73,   -21,   -18,   -28,   -73,   -35,   -66,   -53,
   -54,   -55,   -56,   -57,   -67,   -68,   -39,   -43,   -49,   -73,
   -60,   -47,   -73,   -29,   -52,   -48,   -73,   -20,   -50 ]

racc_goto_table = [
     4,    39,     4,    68,    74,    75,     5,     6,     5,     6,
    44,    42,    51,    49,     3,    56,    37,    57,    58,     1,
     2,    66,    84,    41,    43,    48,    50,    64,    84,    84,
    45,    46,    42,    45,    46,    55,    85,    86,    80,    84,
    84,   nil,   nil,   nil,   nil,   nil,   nil,   nil,    82,   nil,
   nil,   nil,    78 ]

racc_goto_check = [
     4,    10,     4,    31,    31,    31,     5,     6,     5,     6,
    21,    12,    27,    21,     3,    27,     3,     9,     9,     1,
     2,    11,    32,    17,    19,    23,    26,    10,    32,    32,
     5,     6,    12,     5,     6,    29,    31,    31,    33,    32,
    32,   nil,   nil,   nil,   nil,   nil,   nil,   nil,    10,   nil,
   nil,   nil,     4 ]

racc_goto_pointer = [
   nil,    19,    20,    14,     0,     6,     7,   nil,   nil,   -17,
   -14,   -20,    -9,   nil,   nil,   nil,   nil,     8,   nil,     2,
   nil,   -14,   nil,     0,   nil,   nil,    -2,   -18,   nil,     4,
   nil,   -42,   -46,   -16 ]

racc_goto_default = [
   nil,   nil,   nil,   nil,    70,    71,    72,     7,     8,    13,
   nil,   nil,    21,    16,    17,    18,    19,    20,    22,    23,
    24,   nil,    25,    26,    27,    28,    29,   nil,    31,    32,
    52,   nil,    69,    53 ]

racc_reduce_table = [
  0, 0, :racc_error,
  1, 15, :_reduce_1,
  0, 15, :_reduce_2,
  2, 16, :_reduce_3,
  1, 16, :_reduce_4,
  1, 17, :_reduce_5,
  1, 17, :_reduce_6,
  1, 17, :_reduce_none,
  1, 17, :_reduce_8,
  1, 17, :_reduce_9,
  1, 17, :_reduce_10,
  1, 17, :_reduce_11,
  1, 21, :_reduce_12,
  1, 22, :_reduce_13,
  1, 18, :_reduce_14,
  2, 23, :_reduce_15,
  1, 23, :_reduce_16,
  3, 19, :_reduce_17,
  1, 25, :_reduce_18,
  2, 24, :_reduce_19,
  4, 24, :_reduce_20,
  2, 24, :_reduce_21,
  1, 24, :_reduce_22,
  1, 26, :_reduce_none,
  1, 26, :_reduce_none,
  1, 26, :_reduce_none,
  1, 26, :_reduce_none,
  1, 20, :_reduce_27,
  3, 20, :_reduce_28,
  4, 20, :_reduce_29,
  2, 31, :_reduce_30,
  1, 31, :_reduce_31,
  1, 27, :_reduce_32,
  2, 32, :_reduce_33,
  1, 32, :_reduce_34,
  3, 33, :_reduce_35,
  1, 28, :_reduce_36,
  2, 36, :_reduce_37,
  1, 36, :_reduce_38,
  3, 37, :_reduce_39,
  1, 29, :_reduce_40,
  2, 39, :_reduce_41,
  1, 39, :_reduce_42,
  3, 40, :_reduce_43,
  1, 30, :_reduce_44,
  2, 42, :_reduce_45,
  1, 42, :_reduce_46,
  3, 43, :_reduce_47,
  3, 41, :_reduce_48,
  2, 41, :_reduce_49,
  4, 41, :_reduce_50,
  1, 41, :_reduce_51,
  2, 45, :_reduce_52,
  1, 45, :_reduce_none,
  1, 46, :_reduce_54,
  1, 46, :_reduce_55,
  1, 46, :_reduce_none,
  1, 46, :_reduce_57,
  1, 44, :_reduce_none,
  0, 44, :_reduce_none,
  2, 47, :_reduce_none,
  1, 47, :_reduce_none,
  2, 34, :_reduce_62,
  1, 34, :_reduce_63,
  2, 38, :_reduce_64,
  1, 38, :_reduce_65,
  2, 35, :_reduce_66,
  2, 35, :_reduce_67,
  2, 35, :_reduce_68,
  1, 35, :_reduce_69,
  1, 35, :_reduce_none,
  1, 35, :_reduce_71,
  0, 35, :_reduce_72 ]

racc_reduce_n = 73

racc_shift_n = 89

racc_token_table = {
  false => 0,
  :error => 1,
  :DUMMY => 2,
  :ITEMLISTLINE => 3,
  :ENUMLISTLINE => 4,
  :DESCLISTLINE => 5,
  :METHODLISTLINE => 6,
  :STRINGLINE => 7,
  :WHITELINE => 8,
  :SUBTREE => 9,
  :HEADLINE => 10,
  :INCLUDE => 11,
  :INDENT => 12,
  :DEDENT => 13 }

racc_nt_base = 14

racc_use_result_var = true

Racc_arg = [
  racc_action_table,
  racc_action_check,
  racc_action_default,
  racc_action_pointer,
  racc_goto_table,
  racc_goto_check,
  racc_goto_default,
  racc_goto_pointer,
  racc_nt_base,
  racc_reduce_table,
  racc_token_table,
  racc_shift_n,
  racc_reduce_n,
  racc_use_result_var ]

Racc_token_to_s_table = [
  "$end",
  "error",
  "DUMMY",
  "ITEMLISTLINE",
  "ENUMLISTLINE",
  "DESCLISTLINE",
  "METHODLISTLINE",
  "STRINGLINE",
  "WHITELINE",
  "SUBTREE",
  "HEADLINE",
  "INCLUDE",
  "INDENT",
  "DEDENT",
  "$start",
  "document",
  "blocks",
  "block",
  "textblock",
  "verbatim",
  "lists",
  "headline",
  "include",
  "textblockcontent",
  "verbatimcontent",
  "verbatim_after_lists",
  "list",
  "itemlist",
  "enumlist",
  "desclist",
  "methodlist",
  "lists2",
  "itemlistitems",
  "itemlistitem",
  "first_textblock_in_itemlist",
  "other_blocks_in_list",
  "enumlistitems",
  "enumlistitem",
  "first_textblock_in_enumlist",
  "desclistitems",
  "desclistitem",
  "description_part",
  "methodlistitems",
  "methodlistitem",
  "whitelines",
  "blocks_in_list",
  "block_in_list",
  "whitelines2" ]

Racc_debug_parser = false

##### State transition tables end #####

# reduce 0 omitted

def _reduce_1(val, _values, result)
 result = RDoc::Markup::Document.new(*val[0])
    result
end

def _reduce_2(val, _values, result)
 raise ParseError, "file empty"
    result
end

def _reduce_3(val, _values, result)
 result = val[0].concat val[1]
    result
end

def _reduce_4(val, _values, result)
 result = val[0]
    result
end

def _reduce_5(val, _values, result)
 result = val
    result
end

def _reduce_6(val, _values, result)
 result = val
    result
end

# reduce 7 omitted

def _reduce_8(val, _values, result)
 result = val
    result
end

def _reduce_9(val, _values, result)
 result = val
    result
end

def _reduce_10(val, _values, result)
 result = [RDoc::Markup::BlankLine.new]
    result
end

def _reduce_11(val, _values, result)
 result = val[0].parts
    result
end

def _reduce_12(val, _values, result)
      # val[0] is like [level, title]
      title = @inline_parser.parse(val[0][1])
      result = RDoc::Markup::Heading.new(val[0][0], title)

    result
end

def _reduce_13(val, _values, result)
      result = RDoc::Markup::Include.new val[0], @include_path

    result
end

def _reduce_14(val, _values, result)
      # val[0] is Array of String
      result = paragraph val[0]

    result
end

def _reduce_15(val, _values, result)
 result << val[1].rstrip
    result
end

def _reduce_16(val, _values, result)
 result = [val[0].rstrip]
    result
end

def _reduce_17(val, _values, result)
      # val[1] is Array of String
      content = cut_off val[1]
      result = RDoc::Markup::Verbatim.new(*content)

      # imform to lexer.
      @in_verbatim = false

    result
end

def _reduce_18(val, _values, result)
      # val[0] is Array of String
      content = cut_off val[0]
      result = RDoc::Markup::Verbatim.new(*content)

      # imform to lexer.
      @in_verbatim = false

    result
end

def _reduce_19(val, _values, result)
      result << val[1]

    result
end

def _reduce_20(val, _values, result)
      result.concat val[2]

    result
end

def _reduce_21(val, _values, result)
      result << "\n"

    result
end

def _reduce_22(val, _values, result)
      result = val
      # inform to lexer.
      @in_verbatim = true

    result
end

# reduce 23 omitted

# reduce 24 omitted

# reduce 25 omitted

# reduce 26 omitted

def _reduce_27(val, _values, result)
      result = val[0]

    result
end

def _reduce_28(val, _values, result)
      result = val[1]

    result
end

def _reduce_29(val, _values, result)
      result = val[1].push(val[2])

    result
end

def _reduce_30(val, _values, result)
 result = val[0] << val[1]
    result
end

def _reduce_31(val, _values, result)
 result = [val[0]]
    result
end

def _reduce_32(val, _values, result)
      result = RDoc::Markup::List.new :BULLET, *val[0]

    result
end

def _reduce_33(val, _values, result)
 result.push(val[1])
    result
end

def _reduce_34(val, _values, result)
 result = val
    result
end

def _reduce_35(val, _values, result)
      result = RDoc::Markup::ListItem.new nil, val[0], *val[1]

    result
end

def _reduce_36(val, _values, result)
      result = RDoc::Markup::List.new :NUMBER, *val[0]

    result
end

def _reduce_37(val, _values, result)
 result.push(val[1])
    result
end

def _reduce_38(val, _values, result)
 result = val
    result
end

def _reduce_39(val, _values, result)
      result = RDoc::Markup::ListItem.new nil, val[0], *val[1]

    result
end

def _reduce_40(val, _values, result)
      result = RDoc::Markup::List.new :NOTE, *val[0]

    result
end

def _reduce_41(val, _values, result)
 result.push(val[1])
    result
end

def _reduce_42(val, _values, result)
 result = val
    result
end

def _reduce_43(val, _values, result)
      term = @inline_parser.parse val[0].strip

      result = RDoc::Markup::ListItem.new term, *v                                                                                                                                                                                                                         gems/gems/rdoc-6.5.1.1/lib/rdoc/known_classes.rb                                                    0000644                 00000005276 15040313415 0015066 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       educe_52(val, _values, result)
 result.concat val[1]
    result
end

# reduce 53 omitted

def _reduce_54(val, _values, result)
 result = val
    result
end

def _reduce_55(val, _values, result)
 result = val
    result
end

# reduce 56 omitted

def _reduce_57(val, _values, result)
 result = []
    result
end

# reduce 58 omitted

# reduce 59 omitted

# reduce 60 omitted

# reduce 61 omitted

def _reduce_62(val, _values, result)
      result = paragraph [val[0]].concat(val[1])

    result
end

def _reduce_63(val, _values, result)
      result = paragraph [val[0]]

    result
end

def _reduce_64(val, _values, result)
      result = paragraph [val[0]].concat(val[1])

    result
end

def _reduce_65(val, _values, result)
      result = paragraph [val[0]]

    result
end

def _reduce_66(val, _values, result)
      result = [val[0]].concat(val[1])

    result
end

def _reduce_67(val, _values, result)
 result.concat val[1]
    result
end

def _reduce_68(val, _values, result)
 result = val[1]
    result
end

def _reduce_69(val, _values, result)
 result = val
    result
end

# reduce 70 omitted

def _reduce_71(val, _values, result)
 result = []
    result
end

def _reduce_72(val, _values, result)
 result = []
    result
end

def _reduce_none(val, _values, result)
  val[0]
end

end   # class BlockParser

end
                                                                                                                                                                                                                         gems/gems/rdoc-6.5.1.1/lib/rdoc/known_classes.rb                                                    0000644                 00000005276 15040313415 0015066 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
module RDoc

  ##
  # Ruby's built-in classes, modules and exceptions

  KNOWN_CLASSES = {
    "rb_cArray"            => "Array",
    "rb_cBasicObject"      => "BasicObject",
    "rb_cBignum"           => "Bignum",
    "rb_cClass"            => "Class",
    "rb_cData"             => "Data",
    "rb_cDir"              => "Dir",
    "rb_cEncoding"         => "Encoding",
    "rb_cFalseClass"       => "FalseClass",
    "rb_cFile"             => "File",
    "rb_cFixnum"           => "Fixnum",
    "rb_cFloat"            => "Float",
    "rb_cHash"             => "Hash",
    "rb_cIO"               => "IO",
    "rb_cInteger"          => "Integer",
    "rb_cModule"                                                                                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/ri/task.rb                                                          0000644                 00000002475 15040313415 0013567 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       Error",
    "rb_eIOError"          => "IOError",
    "rb_eIndexError"       => "IndexError",
    "rb_eInterrupt"        => "Interrupt",
    "rb_eLoadError"        => "LoadError",
    "rb_eNameError"        => "NameError",
    "rb_eNoMemError"       => "NoMemoryError",
    "rb_eNotImpError"      => "NotImplementedError",
    "rb_eRangeError"       => "RangeError",
    "rb_eRuntimeError"     => "RuntimeError",
    "rb_eScriptError"      => "ScriptError",
    "rb_eSecurityError"    => "SecurityError",
    "rb_eSignal"           => "SignalException",
    "rb_eStandardError"    => "StandardError",
    "rb_eSyntaxError"      => "SyntaxError",
    "rb_eSystemCallError"  => "SystemCallError",
    "rb_eSystemExit"       => "SystemExit",
    "rb_eTypeError"        => "TypeError",
    "rb_eZeroDivError"     => "ZeroDivisionError",

    "rb_mComparable"       => "Comparable",
    "rb_mEnumerable"       => "Enumerable",
    "rb_mErrno"            => "Errno",
    "rb_mFConst"           => "File::Constants",
    "rb_mFileTest"         => "FileTest",
    "rb_mGC"               => "GC",
    "rb_mKernel"           => "Kernel",
    "rb_mMath"             => "Math",
    "rb_mProcess"          => "Process"
  }

end
                                                                                                                                                                                                                                                                                                                                  gems/gems/rdoc-6.5.1.1/lib/rdoc/ri/store.rb                                                         0000644                 00000000125 15040313415 0013747 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
begin
  gem 'rdoc'
rescue Gem::LoadError
end unless def                                                                                                                                                                                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/ri/formatter.rb                                                     0000644                 00000000162 15040313415 0014617 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       
  def clobber_task_description # :nodoc:
    "Remove RI data files"
  end

  ##
  # Sets default task values

  d                                                                                                                                                                                                                                                                                                                                                                                                              gems/gems/rdoc-6.5.1.1/lib/rdoc/ri/paths.rb                                                         0000644                 00000010536 15040313415 0013741 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
module RDoc::RI

  Store = RDoc::Store # :nodoc:

end

                                                                                                                                                                                                                                                                                                                                                                                                                                           gems/gems/rdoc-6.5.1.1/lib/rdoc/ri/formatter.rb                                                     0000644                 00000000162 15040313415 0014617 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
##
# For RubyGems backwards compatibility

module RDoc::RI::Formatter # :nodoc:
end
                                                                                                                                                                                                                                                                                                                                                                                                              gems/gems/rdoc-6.5.1.1/lib/rdoc/ri/paths.rb                                                         0000644                 00000010536 15040313415 0013741 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require_relative '../rdoc'

##
# The directories where ri data lives.  Paths can be enumerated via ::each, or
# queried individually via ::system_dir, ::site_dir, ::home_dir and ::gem_dir.

module RDoc::RI::Paths

  #:stopdoc:
  require 'rbconfig'

  version = RbConfig::CONFIG['ruby_version_dir_name'] || RbConfig::CONFIG['ruby_version']

  BASE    = File.join RbConfig::CONFIG['ridir'], version

  HOMEDIR = RDoc.home
  #:startdoc:

  ##
  # Iterates over each selected path yielding the directory and type.
  #
  # Yielded types:
  # :system:: Where Ruby's ri data is stored.  Yielded when +system+ is
  #           true
  # :site:: Where ri for installed libraries are stored.  Yielded when
  #         +site+ is true.  Normally no ri data is stored here.
  # :home:: ~/.rdoc.  Yielded when +home+ is true.
  # :gem:: ri data for an installed gem.  Yielded when +gems+ is true.
  # :extra:: ri data directory from the command line.  Yielded for each
  #          entry in +extra_dirs+

  def self.each system = true, site = true, home = true, gems = :latest, *extra_dirs # :yields: directory, type
    return enum_for __method__, system, site, home, gems, *extra_dirs unless
      block_given?

    extra_dirs.each do |dir|
      yield dir, :extra
    end

    yield system_dir,  :system if system
    yield site_dir,    :site   if site
    yield home_dir,    :home   if home and HOMEDIR

    gemdirs(gems).each do |dir|
      yield dir, :gem
    end if gems

    nil
  end

  ##
  # The ri directory for the gem with +gem_name+.

  def self.gem_dir name, version
    req = Gem::Requirement.new "= #{version}"

    spec = Gem::Specification.find_by_name name, req

    File.join spec.doc_dir, 'ri'
  end

  ##
  # The latest installed gems' ri directories.  +filter+ can be :all or
  # :latest.
  #
  # A +filter+ :all includes all versions of gems and includes gems without
  # ri documentation.

  def self.gemdirs filter = :latest
    ri_paths = {}

    all = Gem::Specification.map do |spec|
      [File.join(spec.doc_dir, 'ri'), spec.name, spec.version]
    end

    if filter == :all then
      gemdirs = []

      all.group_by do |_, name, _|
        name
      end.sort_by do |group, _|
        group
      end.map do |group, items|
        items.sort_by do |_, _, version|
          version
        end.reverse_each do |dir,|
          gemdirs << dir
                                                                                                                                                                      gems/gems/rdoc-6.5.1.1/lib/rdoc/ri/driver.rb                                                        0000644                 00000107400 15040313415 0014112 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       e selected documentation directories
  # as an Array.
  #
  # See also ::each

  def self.path(system = true, site = true, home = true, gems = :latest, *extra_dirs)
    path = raw_path system, site, home, gems, *extra_dirs

    path.select { |directory| File.directory? directory }
  end

  ##
  # Returns selected documentation directories including nonexistent
  # directories.
  #
  # See also ::each

  def self.raw_path(system, site, home, gems, *extra_dirs)
    path = []

    each(system, site, home, gems, *extra_dirs) do |dir, type|
      path << dir
    end

    path.compact
  end

  ##
  # The location of ri data installed into the site dir.
  #
  # Historically this was available for documentation installed by Ruby
  # libraries predating RubyGems.  It is unlikely to contain any content for
  # modern Ruby installations.

  def self.site_dir
    File.join BASE, 'site'
  end

  ##
  # The location of the built-in ri data.
  #
  # This data is built automatically when `make` is run when Ruby is
  # installed.  If you did not install Ruby by hand you may need to install
  # the documentation yourself.  Please consult the documentation for your
  # package manager or Ruby installer for details.  You can also use the
  # rdoc-data gem to install system ri data for common versions of Ruby.

  def self.system_dir
    File.join BASE, 'system'
  end

end
                                                                                                                                                                  gems/gems/rdoc-6.5.1.1/lib/rdoc/ri/driver.rb                                                        0000644                 00000107400 15040313415 0014112 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require 'optparse'

require_relative '../../rdoc'

require_relative 'formatter' # For RubyGems backwards compatibility
# TODO: Fix weird documentation with `require_relative`

##
# The RI driver implements the command-line ri tool.
#
# The driver supports:
# * loading RI data from:
#   * Ruby's standard library
#   * RubyGems
#   * ~/.rdoc
#   * A user-supplied directory
# * Paging output (uses RI_PAGER environment variable, PAGER environment
#   variable or the less, more and pager programs)
# * Interactive mode with tab-completion
# * Abbreviated names (ri Zl shows Zlib documentation)
# * Colorized output
# * Merging output from multiple RI data sources

class RDoc::RI::Driver

  ##
  # Base Driver error class

  class Error < RDoc::RI::Error; end

  ##
  # Raised when a name isn't found in the ri data stores

  class NotFoundError < Error

    def initialize(klass, suggestions = nil) # :nodoc:
      @klass = klass
      @suggestions = suggestions
    end

    ##
    # Name that wasn't found

    def name
      @klass
    end

    def message # :nodoc:
      str = "Nothing known about #{@klass}"
      if @suggestions and !@suggestions.empty?
        str += "\nDid you mean?  #{@suggestions.join("\n               ")}"
      end
      str
    end
  end

  ##
  # Show all method documentation following a class or module

  attr_accessor :show_all

  ##
  # An RDoc::RI::Store for each entry in the RI path

  attr_accessor :stores

  ##
  # Controls the user of the pager vs $stdout

  attr_accessor :use_stdout

  ##
  # Default options for ri

  def self.default_options
    options = {}
    options[:interactive] = false
    options[:profile]     = false
    options[:show_all]    = false
    options[:use_stdout]  = !$stdout.tty?
    options[:width]       = 72

    # By default all standard paths are used.
    options[:use_system]     = true
    options[:use_site]       = true
    options[:use_home]       = true
    options[:use_gems]       = true
    options[:extra_doc_dirs] = []

    return options
  end

  ##
  # Dump +data_path+ using pp

  def self.dump data_path
    require 'pp'

    File.open data_path, 'rb' do |io|
      pp Marshal.load(io.read)
    end
  end

  ##
  # Parses +argv+ and returns a Hash of options

  def self.process_args argv
    options = default_options

    opts = OptionParser.new do |opt|
      opt.accept File do |file,|
        File.readable?(file) and not File.directory?(file) and file
      end

      opt.program_name = File.basename $0
      opt.version = RDoc::VERSION
      opt.release = nil
      opt.summary_indent = ' ' * 4

      opt.banner = <<-EOT
Usage: #{opt.program_name} [options] [name ...]

Where name can be:

  Class | Module | Module::Class

  Class::method | Class#method | Class.method | method

  gem_name: | gem_name:README | gem_name:History

  ruby: | ruby:NEWS | ruby:globals

All class names may be abbreviated to their minimum unambiguous form.
If a name is ambiguous, all valid options will be listed.

A '.' matches either class or instance methods, while #method
matches only instance and ::method matches only class methods.

README and other files may be displayed by prefixing them with the gem name
they're contained in.  If the gem name is followed by a ':' all files in the
gem will be shown.  The file name extension may be omitted where it is
unambiguous.

'ruby' can be used as a pseudo gem name to display files from the Ruby
core documentation. Use 'ruby:' by itself to get a list of all available
core documentation files.

For example:

    #{opt.program_name} Fil
    #{opt.program_name} File
    #{opt.program_name} File.new
    #{opt.program_name} zip
    #{opt.program_name} rdoc:README
    #{opt.program_name} ruby:comments

Note that shell quoting or escaping may be required for method names
containing punctuation:

    #{opt.program_name} 'Array.[]'
    #{opt.program_name} compact\\!

To see the default directories #{opt.program_name} will search, run:

    #{opt.program_name} --list-doc-dirs

Specifying the --system, --site, --home, --gems, or --doc-dir options
will limit ri to searching only the specified directories.

ri options may be set in the RI environment variable.

The ri pager can be set with the RI_PAGER environment variable
or the PAGER environment variable.
      EOT

      opt.separator nil
      opt.separator "Options:"

      opt.separator nil

      opt.on("--[no-]interactive", "-i",
             "In interactive mode you can repeatedly",
             "look up methods with autocomplete.") do |interactive|
        options[:interactive] = interactive
      end

      opt.separator nil

      opt.on("--[no-]all", "-a",
             "Show all documentation for a class or",
             "module.") do |show_all|
        options[:show_all] = show_all
      end

      opt.separator nil

      opt.on("--[no-]list", "-l",
             "List classes ri knows about.") do |list|
        options[:list] = list
      end

      opt.separator nil

      opt.on("--[no-]pager",
             "Send output to a pager,",
             "rather than directly to stdout.") do |use_pager|
        options[:use_stdout] = !use_pager
      end

      opt.separator nil

      opt.on("-T",
             "Synonym for --no-pager.") do
        options[:use_stdout] = true
      end

      opt.separator nil

      opt.on("--width=WIDTH", "-w", OptionParser::DecimalInteger,
             "Set the width of the output.") do |width|
        options[:width] = width
      end

      opt.separator nil

      opt.on("--server[=PORT]", Integer,
             "Run RDoc server on the given port.",
             "The default port is 8214.") do |port|
        options[:server] = port || 8214
      end

      opt.separator nil

      formatters = RDoc::Markup.constants.grep(/^To[A-Z][a-z]+$/).sort
      formatters = formatters.sort.map do |formatter|
        formatter.to_s.sub('To', '').downcase
      end
      formatters -= %w[html label test] # remove useless output formats

      opt.on("--format=NAME", "-f",
             "Use the selected formatter.  The default",
             "formatter is bs for paged output and ansi",
             "otherwise.  Valid formatters are:",
             "#{formatters.join(', ')}.", formatters) do |value|
        options[:formatter] = RDoc::Markup.const_get "To#{value.capitalize}"
      end

      opt.separator nil

      opt.on("--help", "-h",
             "Show help and exit.") do
        puts opts
        exit
      end

      opt.separator nil

      opt.on("--version", "-v",
             "Output version information and exit.") do
        puts "#{opts.program_name} #{opts.version}"
        exit
      end

      opt.separator nil
      opt.separator "Data source options:"
      opt.separator nil

      opt.on("--[no-]list-doc-dirs",
             "List the directories from which ri will",
             "source documentation on stdout and exit.") do |list_doc_dirs|
        options[:list_doc_dirs] = list_doc_dirs
      end

      opt.separator nil

      opt.on("--doc-dir=DIRNAME", "-d", Array,
             "List of directories from which to source",
             "documentation in addition to the standard",
             "directories.  May be repeated.") do |value|
        value.each do |dir|
          unless File.directory? dir then
            raise OptionParser::InvalidArgument, "#{dir} is not a directory"
          end

          options[:extra_doc_dirs] << File.expand_path(dir)
        end
      end

      opt.separator nil

      opt.on("--no-standard-docs",
             "Do not include documentation from",
             "the Ruby standard library, site_lib,",
             "installed gems, or ~/.rdoc.",
             "Use with --doc-dir.") do
        options[:use_system] = false
        options[:use_site] = false
        options[:use_gems] = false
        options[:use_home] = false
      end

      opt.separator nil

      opt.on("--[no-]system",
             "Include documentation from Ruby's",
             "standard library.  Defaults to true.") do |value|
        options[:use_system] = value
      end

      opt.separator nil

      opt.on("--[no-]site",
             "Include documentation from libraries",
             "installed in site_lib.",
             "Defaults to true.") do |value|
        options[:use_site] = value
      end

      opt.separator nil

      opt.on("--[no-]gems",
             "Include documentation from RubyGems.",
             "Defaults to true.") do |value|
        options[:use_gems] = value
      end

      opt.separator nil

      opt.on("--[no-]home",
             "Include documentation stored in ~/.rdoc.",
             "Defaults to true.") do |value|
        options[:use_home] = value
      end

      opt.separator nil
      opt.separator "Debug options:"
      opt.separator nil

      opt.on("--[no-]profile",
             "Run with the ruby profiler.") do |value|
        options[:profile] = value
      end

      opt.separator nil

      opt.on("--dump=CACHE", File,
             "Dump data from an ri cache or data file.") do |value|
        options[:dump_path] = value
      end
    end

    argv = ENV['RI'].to_s.split(' ').concat argv

    opts.parse! argv

    options[:names] = argv

    options[:use_stdout] ||= !$stdout.tty?
    options[:use_stdout] ||= options[:interactive]
    options[:width] ||= 72

    options

  rescue OptionParser::InvalidArgument, OptionParser::InvalidOption => e
    puts opts
    puts
    puts e
    exit 1
  end

  ##
  # Runs the ri command line executable using +argv+

  def self.run argv = ARGV
    options = process_args argv

    if options[:dump_path] then
      dump options[:dump_path]
      return
    end

    ri = new options
    ri.run
  end

  ##
  # Creates a new driver using +initial_options+ from ::process_args

  def initialize initial_options = {}
    @paging = false
    @classes = nil

    options = self.class.default_options.update(initial_options)

    @formatter_klass = options[:formatter]

    require 'profile' if options[:profile]

    @names = options[:names]
    @list = options[:list]

    @doc_dirs = []
    @stores   = []

    RDoc::RI::Paths.each(options[:use_system], options[:use_site],
                         options[:use_home], options[:use_gems],
                         *options[:extra_doc_dirs]) do |path, type|
      @doc_dirs << path

      store = RDoc::RI::Store.new path, type
      store.load_cache
      @stores << store
    end

    @list_doc_dirs = options[:list_doc_dirs]

    @interactive = options[:interactive]
    @server      = options[:server]
    @use_stdout  = options[:use_stdout]
    @show_all    = options[:show_all]
    @width       = options[:width]
  end

  ##
  # Adds paths for undocumented classes +also_in+ to +out+

  def add_also_in out, also_in
    return if also_in.empty?

    out << RDoc::Markup::Rule.new(1)
    out << RDoc::Markup::Paragraph.new("Also found in:")

    paths = RDoc::Markup::Verbatim.new
    also_in.each do |store|
      paths.parts.push store.friendly_path, "\n"
    end
    out << paths
  end

  ##
  # Adds a class header to +out+ for class +name+ which is described in
  # +classes+.

  def add_class out, name, classes
    heading = if classes.all? { |klass| klass.module? } then
                name
              else
                superclass = classes.map do |klass|
                  klass.superclass unless klass.module?
                end.compact.shift || 'Object'

                superclass = superclass.full_name unless String === superclass

                "#{name} < #{superclass}"
              end

    out << RDoc::Markup::Heading.new(1, heading)
    out << RDoc::Markup::BlankLine.new
  end

  ##
  # Adds "(from ...)" to +out+ for +store+

  def add_from out, store
    out << RDoc::Markup::Paragraph.new("(from #{store.friendly_path})")
  end

  ##
  # Adds +extends+ to +out+

  def add_extends out, extends
    add_extension_modules out, 'Extended by', extends
  end

  ##
  # Adds a list of +extensions+ to this module of the given +type+ to +out+.
  # add_includes and add_extends call this, so you should use those directly.

  def add_extension_modules out, type, extensions
    return if extensions.empty?

    out << RDoc::Markup::Rule.new(1)
    out << RDoc::Markup::Heading.new(1, "#{type}:")

    extensions.each do |modules, store|
      if modules.length == 1 then
        add_extension_modules_single out, store, modules.first
      else
        add_extension_modules_multiple out, store, modules
      end
    end
  end

  ##
  # Renders multiple included +modules+ from +store+ to +out+.

  def add_extension_modules_multiple out, store, modules # :nodoc:
    out << RDoc::Markup::Paragraph.new("(from #{store.friendly_path})")

    wout, with = modules.partition { |incl| incl.comment.empty? }

    out << RDoc::Markup::BlankLine.new unless with.empty?

    with.each do |incl|
      out << RDoc::Markup::Paragraph.new(incl.name)
      out << RDoc::Markup::BlankLine.new
      out << incl.comment
    end

    unless wout.empty? then
      verb = RDoc::Markup::Verbatim.new

      wout.each do |incl|
        verb.push incl.name, "\n"
      end

      out << verb
    end
  end

  ##
  # Adds a single extension module +include+ from +store+ to +out+

  def add_extension_modules_single out, store, include # :nodoc:
    name = include.name
    path = store.friendly_path
    out << RDoc::Markup::Paragraph.new("#{name} (from #{path})")

    if include.comment then
      out << RDoc::Markup::BlankLine.new
      out << include.comment
    end
  end

  ##
  # Adds +includes+ to +out+

  def add_includes out, includes
    add_extension_modules out, 'Includes', includes
  end

  ##
  # Looks up the method +name+ and adds it to +out+

  def add_method out, name
    filtered   = lookup_method name

    method_out = method_document name, filtered

    out.concat method_out.parts
  end

  ##
  # Adds documentation for all methods in +klass+ to +out+

  def add_method_documentation out, klass
    klass.method_list.each do |method|
      begin
        add_method out, method.full_name
      rescue NotFoundError
        next
      end
    end
  end

  ##
  # Adds a list of +methods+ to +out+ with a heading of +name+

  def add_method_list out, methods, name
    return if methods.empty?

    out << RDoc::Markup::Heading.new(1, "#{name}:")
    out << RDoc::Markup::BlankLine.new

    if @use_stdout and !@interactive then
      out.concat methods.map { |method|
        RDoc::Markup::Verbatim.new method
      }
    else
      out << RDoc::Markup::IndentedParagraph.new(2, methods.join(', '))
    end

    out << RDoc::Markup::BlankLine.new
  end

  ##
  # Returns ancestor classes of +klass+

  def ancestors_of klass
    ancestors = []

    unexamined = [klass]
    seen = []

    loop do
      break if unexamined.empty?
      current = unexamined.shift
      seen << current

      stores = classes[current]

      next unless stores and not stores.empty?

      klasses = stores.flat_map do |store|
        store.ancestors[current] || []
      end.uniq

      klasses = klasses - seen

      ancestors.concat klasses
      unexamined.concat klasses
    end

    ancestors.reverse
  end

  ##
  # For RubyGems backwards compatibility

  def class_cache # :nodoc:
  end

  ##
  # Builds a RDoc::Markup::Document from +found+, +klasess+ and +includes+

  def class_document name, found, klasses, includes, extends
    also_in = []

    out = RDoc::Markup::Document.new

    add_class out, name, klasses

    add_includes out, includes
    add_extends  out, extends

    found.each do |store, klass|
      render_class out, store, klass, also_in
    end

    add_also_in out, also_in

    out
  end

  ##
  # Adds the class +comment+ to +out+.

  def class_document_comment out, comment # :nodoc:
    unless comment.empty? then
      out << RDoc::Markup::Rule.new(1)

      if comment.merged? then
        parts = comment.parts
        parts = parts.zip [RDoc::Markup::BlankLine.new] * parts.length
        parts.flatten!
        parts.pop

        out.concat parts
      else
        out << comment
      end
    end
  end

  ##
  # Adds the constants from +klass+ to the Document +out+.

  def class_document_constants out, klass # :nodoc:
    return if klass.constants.empty?

    out << RDoc::Markup::Heading.new(1, "Constants:")
    out << RDoc::Markup::BlankLine.new
    list = RDoc::Markup::List.new :NOTE

    constants = klass.constants.sort_by { |constant| constant.name }

    list.items.concat constants.map { |constant|
      parts = constant.comment.parts if constant.comment
      parts << RDoc::Markup::Paragraph.new('[not documented]') if
        parts.empty?

      RDoc::Markup::ListItem.new(constant.name, *parts)
    }

    out << list
    out << RDoc::Markup::BlankLine.new
  end

  ##
  # Hash mapping a known class or module to the stores it can be loaded from

  def classes
    return @classes if @classes

    @classes = {}

    @stores.each do |store|
      store.cache[:modules].each do |mod|
        # using default block causes searched-for modules to be added
        @classes[mod] ||= []
        @classes[mod] << store
      end
    end

    @classes
  end

  ##
  # Returns the stores wherein +name+ is found along with the classes,
  # extends and includes that match it

  def classes_and_includes_and_extends_for name
    klasses = []
    extends = []
    includes = []

    found = @stores.map do |store|
      begin
        klass = store.load_class name
        klasses  << klass
        extends  << [klass.extends,  store] if klass.extends
        includes << [klass.includes, store] if klass.includes
        [store, klass]
      rescue RDoc::Store::MissingFileError
      end
    end.compact

    extends.reject!  do |modules,| modules.empty? end
    includes.reject! do |modules,| modules.empty? end

    [found, klasses, includes, extends]
  end

  ##
  # Completes +name+ based on the caches.  For Readline

  def complete name
    completions = []

    klass, selector, method = parse_name name

    complete_klass  name, klass, selector, method, completions
    complete_method name, klass, selector,         completions

    completions.sort.uniq
  end

  def complete_klass name, klass, selector, method, completions # :nodoc:
    klasses = classes.keys

    # may need to include Foo when given Foo::
    klass_name = method ? name : klass

    if name !~ /#|\./ then
      completions.replace klasses.grep(/^#{Regexp.escape klass_name}[^:]*$/)
      completions.concat klasses.grep(/^#{Regexp.escape name}[^:]*$/) if
        name =~ /::$/

      completions << klass if classes.key? klass # to complete a method name
    elsif selector then
      completions << klass if classes.key? klass
    elsif classes.key? klass_name then
      completions << klass_name
    end
  end

  def complete_method name, klass, selector, completions # :nodoc:
    if completions.include? klass and name =~ /#|\.|::/ then
      methods = list_methods_matching name

      if not methods.empty? then
        # remove Foo if given Foo:: and a method was found
        completions.delete klass
      elsif selector then
        # replace Foo with Foo:: as given
        completions.delete klass
        completions << "#{klass}#{selector}"
      end

      completions.concat methods
    end
  end

  ##
  # Converts +document+ to text and writes it to the pager

  def display document
    page do |io|
      f = formatter(io)
      f.width = @width if @width and f.respond_to?(:width)
      text = document.accept f

      io.write text
    end
  end

  ##
  # Outputs formatted RI data for class +name+.  Groups undocumented classes

  def display_class name
    return if name =~ /#|\./

    found, klasses, includes, extends =
      classes_and_includes_and_extends_for name

    return if found.empty?

    out = class_document name, found, klasses, includes, extends

    display out
  end

  ##
  # Outputs formatted RI data for method +name+

  def display_method name
    out = RDoc::Markup::Document.new

    add_method out, name

    display out
  end

  ##
  # Outputs formatted RI data for the class or method +name+.
  #
  # Returns true if +name+ was found, false if it was not an alternative could
  # be guessed, raises an error if +name+ couldn't be guessed.

  def display_name name
    if name =~ /\w:(\w|$)/ then
      display_page name
      return true
    end

    return true if display_class name

    display_method name if name =~ /::|#|\./

    true
  rescue NotFoundError
    matches = list_methods_matching name if name =~ /::|#|\./
    matches = classes.keys.grep(/^#{Regexp.escape name}/) if matches.empty?

    raise if matches.empty?

    page do |io|
      io.puts "#{name} not found, maybe you meant:"
      io.puts
      io.puts matches.sort.join("\n")
    end

    false
  end

  ##
  # Displays each name in +name+

  def display_names names
    names.each do |name|
      name = expand_name name

      display_name name
    end
  end

  ##
  # Outputs formatted RI data for page +name+.

  def display_page name
    store_name, page_name = name.split ':', 2

    store = @stores.find { |s| s.source == store_name }

    return display_page_list store if page_name.empty?

    pages = store.cache[:pages]

    unless pages.include? page_name then
      found_names = pages.select do |n|
        n =~ /#{Regexp.escape page_name}\.[^.]+$/
      end

      if found_names.length.zero? then
        return display_page_list store, pages
      elsif found_names.length > 1 then
        return display_page_list store, found_names, page_name
      end

      page_name = found_names.first
    end

    page = store.load_page page_name

    display page.comment
  end

  ##
  # Outputs a formatted RI page list for the pages in +store+.

  def display_page_list store, pages = store.cache[:pages], search = nil
    out = RDoc::Markup::Document.new

    title = if search then
              "#{search} pages"
            else
              'Pages'
            end

    out << RDoc::Markup::Heading.new(1, "#{title} in #{store.friendly_path}")
    out << RDoc::Markup::BlankLine.new

    list = RDoc::Markup::List.new(:BULLET)

    pages.each do |page|
      list << RDoc::Markup::Paragraph.new(page)
    end

    out << list

    display out
  end

  def check_did_you_mean # :nodoc:
    if defined? DidYouMean::SpellChecker
      true
    else
      begin
        require 'did_you_mean'
        if defined? DidYouMean::SpellChecker
          true
        else
          false
        end
      rescue LoadError
        false
      end
    end
  end

  ##
  # Expands abbreviated klass +klass+ into a fully-qualified class.  "Zl::Da"
  # will be expanded to Zlib::DataError.

  def expand_class klass
    class_names = classes.keys
    ary = class_names.grep(Regexp.new("\\A#{klass.gsub(/(?=::|\z)/, '[^:]*')}\\z"))
    if ary.length != 1 && ary.first != klass
      if check_did_you_mean
        suggestions = DidYouMean::SpellChecker.new(dictionary: class_names).correct(klass)
        raise NotFoundError.new(klass, suggestions)
      else
        raise NotFoundError, klass
      end
    end
    ary.first
  end

  ##
  # Expands the class portion of +name+ into a fully-qualified class.  See
  # #expand_class.

  def expand_name name
    klass, selector, method = parse_name name

    return [selector, method].join if klass.empty?

    case selec