e

    # Number of columns on the terminal
    attr_accessor :terminal_columns

    # List of the top level task names (task names from the command line).
    attr_reader :top_level_tasks

    # Override the detected TTY output state (mostly for testing)
    attr_writer :tty_output

    DEFAULT_RAKEFILES = [
      "rakefile",
      "Rakefile",
      "rakefile.rb",
      "Rakefile.rb"
    ].freeze

    # Initialize a Rake::Application object.
    def initialize
      super
      @name = "rake"
      @rakefiles = DEFAULT_RAKEFILES.dup
      @rakefile = nil
      @pending_imports = []
      @imported = []
      @loaders = {}
      @default_loader = Rake::DefaultLoader.new
      @original_dir = Dir.pwd
      @top_level_tasks = []
      add_loader("rb", DefaultLoader.new)
      add_loader("rf", DefaultLoader.new)
      add_loader("rake", DefaultLoader.new)
      @tty_output = STDOUT.tty?
      @terminal_columns = ENV["RAKE_COLUMNS"].to_i

      set_default_options
    end

    # Run the Rake application.  The run method performs the following
    # three steps:
    #
    # * Initialize the command line options (+init+).
    # * Define the tasks (+load_rakefile+).
    # * Run the top level tasks (+top_level+).
    #
    # If you wish to build a custom rake command, you should call
    # +init+ on your application.  Then define any tasks.  Finally,
    # call +top_level+ to run your top level tasks.
    def run(argv = ARGV)
      standard_exception_handling do
        init "rake", argv
        load_rakefile
        top_level
      end
    end

    # Initialize the command line parameters and app name.
    def init(app_name="rake", argv = ARGV)
      standard_exception_handling do
        @name = app_name
        begin
          args = handle_options argv
        rescue ArgumentError
          # Backward compatibility for capistrano
          args = handle_options
        end
        collect_command_line_tasks(args)
      end
    end

    # Find the rakefile and then load it and any pending imports.
    def load_rakefile
      standard_exception_handling do
        raw_load_rakefile
      end
    end

    # Run the top level tasks of a Rake application.
    def top_level
      run_with_threads do
        if options.show_tasks
          display_tasks_and_comments
        elsif options.show_prereqs
          display_prerequisites
        else
          top_level_tasks.each { |task_name| invoke_task(task_name) }
        end
      end
    end

    # Run the given block with the thread startup and shutdown.
    def run_with_threads
      thread_pool.gather_history if options.job_stats == :history

      yield

      thread_pool.join
      if options.job_stats
        stats = thread_pool.statistics
        puts "Maximum active threads: #{stats[:max_active_threads]} + main"
        puts "Total threads in play:  #{stats[:total_threads_in_play]} + main"
      end
      ThreadHistoryDisplay.new(thread_pool.history).show if
        options.job_stats == :history
    end

    # Add a loader to handle imported files ending in the extension
    # +ext+.
    def add_loader(ext, loader)
      ext = ".#{ext}" unless ext =~ /^\./
      @loaders[ext] = loader
    end

    # Application options from the command line
    def options
      @options ||= OpenStruct.new
    end

    # Return the thread pool used for multithreaded processing.
    def thread_pool             # :nodoc:
      @thread_pool ||= ThreadPool.new(options.thread_pool_size || Rake.suggested_thread_count-1)
    end

    # internal ----------------------------------------------------------------

    # Invokes a task with arguments that are extracted from +task_string+
    def invoke_task(task_string) # :nodoc:
      name, args = parse_task_string(task_string)
      t = self[name]
      t.invoke(*args)
    end

    def parse_task_string(string) # :nodoc:
      /^([^\[]+)(?:\[(.*)\])$/ =~ string.to_s

      name           = $1
      remaining_args = $2

      return string, [] unless name
      return name,   [] if     remaining_args.empty?

      args = []

      begin
        /\s*((?:[^\\,]|\\.)*?)\s*(?:,\s*(.*))?$/ =~ remaining_args

        remaining_args = $2
        args << $1.gsub(/\\(.)/, '\1')
      end while remaining_args

      return name, args
    end

    # Provide standard exception handling for the given block.
    def standard_exception_handling # :nodoc:
      yield
    rescue SystemExit
      # Exit silently with current status
      raise
    rescue OptionParser::InvalidOption => ex
      $stderr.puts ex.message
      exit(false)
    rescue Exception => ex
      # Exit with error message
      display_error_message(ex)
      exit_because_of_exception(ex)
    end

    # Exit the program because of an unhandled exception.
    # (may be overridden by subclasses)
    def exit_because_of_exception(ex) # :nodoc:
      exit(false)
    end

    # Display the error message that caused the exception.
    def display_error_message(ex) # :nodoc:
      trace "#{name} aborted!"
      display_exception_details(ex)
      trace "Tasks: #{ex.chain}" if has_chain?(ex)
      trace "(See full trace by running task with --trace)" unless
         options.backtrace
    end

    def display_exception_details(ex) # :nodoc:
      display_exception_details_seen << ex

      display_exception_message_details(ex)
      display_exception_backtrace(ex)
      display_cause_details(ex.cause) if has_cause?(ex)
    end

    def display_cause_details(ex) # :nodoc:
      return if display_exception_details_seen.include? ex

      trace "\nCaused by:"
      display_exception_details(ex)
    end

    def display_exception_details_seen # :nodoc:
      Thread.current[:rake_display_exception_details_seen] ||= []
    end

    def has_cause?(ex) # :nodoc:
      ex.respond_to?(:cause) && ex.cause
    end

    def display_exception_message_details(ex) # :nodoc:
      if ex.instance_of?(RuntimeError)
        trace ex.message
      else
        trace "#{ex.class.name}: #{ex.message}"
      end
    end

    def display_exception_backtrace(ex) # :nodoc:
      if options.backtrace
        trace ex.backtrace.join("\n")
      else
        trace Backtrace.collapse(ex.backtrace).join("\n")
      end
    end

    # Warn about deprecated usage.
    #
    # Example:
    #    Rake.application.deprecate("import", "Rake.import", caller.first)
    #
    def deprecate(old_usage, new_usage, call_site) # :nodoc:
      unless options.ignore_deprecate
        $stderr.puts "WARNING: '#{old_usage}' is deprecated.  " +
          "Please use '#{new_usage}' instead.\n" +
          "    at #{call_site}"
      end
    end

    # Does the exception have a task invocation chain?
    def has_chain?(exception) # :nodoc:
      exception.respond_to?(:chain) && exception.chain
    end
    private :has_chain?

    # True if one of the files in RAKEFILES is in the current directory.
    # If a match is found, it is copied into @rakefile.
    def have_rakefile # :nodoc:
      @rakefiles.each do |fn|
        if File.exist?(fn)
          others = FileList.glob(fn, File::FNM_CASEFOLD)
          return others.size == 1 ? others.first : fn
        elsif fn == ""
          return fn
        end
      end
      return nil
    end

    # True if we are outputting to TTY, false otherwise
    def tty_output? # :nodoc:
      @tty_output
    end

    # We will truncate output if we are outputting to a TTY or if we've been
    # given an explicit column width to honor
    def truncate_output? # :nodoc:
      tty_output? || @terminal_columns.nonzero?
    end

    # Display the tasks and comments.
    def display_tasks_and_comments # :nodoc:
      displayable_tasks = tasks.select { |t|
        (options.show_all_tasks || t.comment) &&
          t.name =~ options.show_task_pattern
      }
      case options.show_tasks
      when :tasks
        width = displayable_tasks.map { |t| t.name_with_args.length }.max || 10
        if truncate_output?
          max_column = terminal_width - name.size - width - 7
        else
          max_column = nil
        end

        displayable_tasks.each do |t|
          printf("#{name} %-#{width}s                                                     gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/text.rb                                                      0000644                 00000000463 15040313414 0014465 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       
      else
        fail "Unknown show task mode: '#{options.show_tasks}'"
      end
    end

    def terminal_width # :nodoc:
      if @terminal_columns.nonzero?
        result = @terminal_columns
      else
        result = unix? ? dynamic_width : 80
      end
      (result < 10) ? 80 : result
    rescue                                                                                                                                                                                                             gems/gems/rdoc-6.5.1.1/lib/rdoc/parser/ripper_state_lex.rb                                          0000644                 00000037321 15040313414 0017055 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ] || "") + "..."
      end
    end

    # Display the tasks and prerequisites
    def display_prerequisites # :nodoc:
      tasks.each do |t|
        puts "#{name} #{t.name}"
        t.prerequisites.each { |pre| puts "    #{pre}" }
      end
    end

    def trace(*strings) # :nodoc:
      options.trace_output ||= $stderr
      trace_on(options.trace_output, *strings)
    end

    def sort_options(options) # :nodoc:
      options.sort_by { |opt|
        opt.select { |o| o.is_a?(String) && o =~ /^-/ }.map(&:downcase).sort.reverse
      }
    end
    private :sort_options

    # A list of all the standard options used in rake, suitable for
    # passing to OptionParser.
    def standard_rake_options # :nodoc:
      sort_options(
        [
          ["--all", "-A",
            "Show all tasks, even uncommented ones (in combination with -T or -D)",
            lambda { |value|
              options.show_all_tasks = value
            }
          ],
          ["--backtrace=[OUT]",
            "Enable full backtrace.  OUT can be stderr (default) or stdout.",
            lambda { |value|
              options.backtrace = true
              select_trace_output(options, "backtrace", value)
            }
          ],
          ["--build-all", "-B",
           "Build all prerequisites, including those which are up-to-date.",
           lambda { |value|
             options.build_all = true
           }
          ],
          ["--comments",
            "Show commented tasks only",
            lambda { |value|
              options.show_all_tasks = !value
            }
          ],
          ["--describe", "-D [PATTERN]",
            "Describe the tasks (matching optional PATTERN), then exit.",
            lambda { |value|
              select_tasks_to_show(options, :describe, value)
            }
          ],
          ["--directory", "-C [DIRECTORY]",
            "Change to DIRECTORY before doing anything.",
            lambda { |value|
              Dir.chdir value
              @original_dir = Dir.pwd
            }
          ],
          ["--dry-run", "-n",
            "Do a dry run without executing actions.",
            lambda { |value|
              Rake.verbose(true)
              Rake.nowrite(true)
              options.dryrun = true
              options.trace = true
            }
          ],
          ["--execute", "-e CODE",
            "Execute some Ruby code and exit.",
            lambda { |value|
              eval(value)
              exit
            }
          ],
          ["--execute-print", "-p CODE",
            "Execute some Ruby code, print the result, then exit.",
            lambda { |value|
              puts eval(value)
              exit
            }
          ],
          ["--execute-continue",  "-E CODE",
            "Execute some Ruby code, " +
            "then continue with normal task processing.",
            lambda { |value| eval(value) }
          ],
          ["--jobs",  "-j [NUMBER]",
            "Specifies the maximum number of tasks to execute in parallel. " +
            "(default is number of CPU cores + 4)",
            lambda { |value|
              if value.nil? || value == ""
                value = Float::INFINITY
              elsif value =~ /^\d+$/
                value = value.to_i
              else
                value = Rake.suggested_thread_count
              end
              value = 1 if value < 1
              options.thread_pool_size = value - 1
            }
          ],
          ["--job-stats [LEVEL]",
            "Display job statistics. " +
            "LEVEL=history displays a complete job list",
            lambda { |value|
              if value =~ /^history/i
                options.job_stats = :history
              else
                options.job_stats = true
              end
            }
          ],
          ["--libdir", "-I LIBDIR",
            "Include LIBDIR in the search path for required modules.",
            lambda { |value| $:.push(value) }
          ],
          ["--multitask", "-m",
            "Treat all tasks as multitasks.",
            lambda { |value| options.always_multitask = true }
          ],
          ["--no-search", "--nosearch",
            "-N", "Do not search parent directories for the Rakefile.",
            lambda { |value| options.nosearch = true }
          ],
          ["--prereqs", "-P",
            "Display the tasks and dependencies, then exit.",
            lambda { |value| options.show_prereqs = true }
          ],
          ["--quiet", "-q",
            "Do not log messages to standard output.",
            lambda { |value| Rake.verbose(false) }
          ],
          ["--rakefile", "-f [FILENAME]",
            "Use FILENAME as the rakefile to search for.",
            lambda { |value|
              value ||= ""
              @rakefiles.clear
              @rakefiles << value
            }
          ],
          ["--rakelibdir", "--rakelib", "-R RAKELIBDIR",
            "Auto-import any .rake files in RAKELIBDIR. " +
            "(default is 'rakelib')",
            lambda { |value|
              options.rakelib = value.split(File::PATH_SEPARATOR)
            }
          ],
          ["--require", "-r MODULE",
            "Require MODULE before executing rakefile.",
            lambda { |value|
              begin
                require value
              rescue LoadError => ex
                begin
                  rake_require value
                rescue LoadError
                  raise ex
                end
              end
            }
          ],
          ["--rules",
            "Trace the rules resolution.",
            lambda { |value| options.trace_rules = true }
          ],
          ["--silent", "-s",
            "Like --quiet, but also suppresses the " +
            "'in directory' announcement.",
            lambda { |value|
              Rake.verbose(false)
              options.silent = true
            }
          ],
          ["--suppress-backtrace PATTERN",
            "Suppress backtrace lines matching regexp PATTERN. " +
            "Ignored if --trace is on.",
            lambda { |value|
              options.suppress_backtrace_pattern = Regexp.new(value)
            }
          ],
          ["--system",  "-g",
            "Using system wide (global) rakefiles " +
            "(usually '~/.rake/*.ra