                                                        ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

module SyntaxSuggest
  # Converts a SyntaxError message to a path
  #
  # Handles the case where the filename has a colon in it
  # such as on a windows file system: https://github.com/ruby/syntax_suggest/issues/111
  #
  # Example:
  #
  #    message = "/tmp/scratch:2:in `require_relative': /private/tmp/bad.rb:1: syntax error, unexpected `end' (SyntaxError)"
  #    puts PathnameFromMessage.new(message).call.name
  #    # => "/tmp/scratch.rb"
  #
  class PathnameFromMessage
    EVAL_RE = /^\(eval\):\d+/
    STREAMING_RE = /^-:\d+/
    attr_reader :name

    def initialize(message, io: $stderr)
      @line = message.lines.first
      @parts = @line.split(":")
      @guess = []
      @name = nil
      @io = io
    end

    def call
      if skip_missing_file_name?
        if ENV["SYNTAX_SUGGEST_DEBUG"]
          @io.puts "SyntaxSuggest: Could not find filename from #{@line.inspect}"
        end
      else
        until stop?
          @guess << @parts.shift
          @name = Pathname(@guess.join(":"))
        end

        if @parts.empty?
          @io.puts "SyntaxSuggest: Could not find filename from #{@line.inspect}"
          @name = nil
        end
      end

      self
    end

    def stop?
      return true if @parts.empty?
      return false if @guess.empty?

      @name&.exist?
                                                                                                                                                                                                                                                                                                                                                                                                                ruby/drb/unix.rb                                                                                    0000644                 00000005433 15040313424 0007605 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ptures everything within the same indentation block.
  #
  #   def dog
  #     puts "bow"
  #     puts "wow"
  #   end
  #
  # block = BlockExpand.new(code_lines: code_lines)
  #   .call(CodeBlock.new(lines: code_lines[1]))
  #
  # puts block.to_s
  # # => puts "bow"
  #      puts "wow"
  #
  #
  # Once a code block has captured everything at a given indentation level
  # then it will expand to capture surrounding indentation.
  #
  # block = BlockExpand.new(code_lines: code_lines)
  #   .call(block)
  #
  # block.to_s
  # # => def dog
  #        puts "bow"
  #        puts "wow"
  #      end
  #
  class BlockExpand
    def initialize(code_lines:)
      @code_lines = code_lines
    end

    # Main interface. Expand current indentation, before
    # expanding to a lower indentation
    def call(block)
      if (next_block = expand_neighbors(block))
        next_block
      else
        expand_indent(block)
      end
    end

    # Expands code to the next lowest indentation
    #
    # For example:
    #
    #   1 def dog
    #   2   print "dog"
    #   3 end
    #
    # If a block starts on line 2 then it has captured all it's "neighbors" (code at
    # the same indentation or higher). To continue expanding, this block must capture
    # lines one and three which are at a different indentation level.
    #
    # This method allows fully expanded blocks to decrease their indentation level (so
    # they can expand to capture more code up and down). It does this conservatively
    # as there's no undo (currently).
    def expand_indent(block)
      now = AroundBlockScan.new(code_lines: @code_lines, block: block)
        .force_add_hidden
        .stop_after_kw
        .scan_adjacent_indent

      now.lookahead_balance_one_line

      now.code_block
    end

    # A neighbor is code that is at or above the current indent line.
    #
    # First we build a block with all neighbors. If we can't go further
    # then we decrease the indentation threshold and expand via indentation
    # i.e. `expand_indent`
    #
    # Handles two general cases.
    #
    # ## Case #1: Check code inside of methods/classes/etc.
    #
    # It's important to note, that not everything in a given indentation level can be parsed
    # as valid code even if it's part of valid code. For example:
    #
    #   1 hash = {
    #   2   name: "richard",
    #   3   dog: "cinco",
    #   4 }
    #
    # In this case lines 2 and 3 will be neighbors, but they're invalid until `expand_indent`
    # is called on them.
    #
    # When we are adding code within a method or class (at the same indentation level),
    # use the empty lines to denote the programmer intended logical chunks.
    # Stop and check each one. For example:
    #                                                                                                                                                                                                                                                                                                                                          ruby/drb/observer.rb                                                                                0000644                 00000001237 15040313424 0010447 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       es 4, 5, and 6 are in a block when it tries to expand neighbors
    # it will expand up. If it stops after line 2 or 3 it may cause problems since there's a
    # valid kw/end pair, but the block will be checked without it.
    #
    # We try to resolve this edge case with `lookahead_balance_one_line` below.
    def expand_neighbors(block)
      now = AroundBlockScan.new(code_lines: @code_lines, block: block)

      # Initial scan
      now
        .force_add_hidden
        .stop_after_kw
        .scan_neighbors_not_empty

      # Slurp up empties
      now
        .scan_while { |lin