ons of StandardError and its descendants.
  # - <tt>noop: true</tt> - does not move files.
  # - <tt>secure: true</tt> - removes +src+ securely;
  #   see details at FileUtils.remove_entry_secure.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.mv('src0', 'dest0', noop: true, verbose: true)
  #     FileUtils.mv(['src1.txt', 'src1'], 'dest1', noop: true, verbose: true)
  #
  #   Output:
  #
  #     mv src0 dest0
  #     mv src1.txt src1 dest1
  #
  # FileUtils.move is an alias for FileUtils.mv.
  #
  def mv(src, dest, force: nil, noop: nil, verbose: nil, secure: nil)
    fu_output_message "mv#{force ? ' -f' : ''} #{[src,dest].flatten.join ' '}" if verbose
    return if noop
    fu_each_src_dest(src, dest) do |s, d|
      destent = Entry_.new(d, nil, true)
      begin
        if destent.exist?
          if destent.directory?
            raise Errno::EEXIST, d
          end
        end
        begin
          File.rename s, d
        rescue Errno::EXDEV,
               Errno::EPERM # move from unencrypted to encrypted dir (ext4)
          copy_entry s, d, true
          if secure
            remove_entry_secure s, force
          else
            remove_entry s, force
          end
        end
      rescue SystemCallError
        raise unless force
      end
    end
  end
  module_function :mv

  alias move mv
  module_function :move

  # Removes entries at the paths in the given +list+
  # (a single path or an array of paths)
  # returns +list+, if it is an array, <tt>[list]</tt> otherwise.
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # With no keyword arguments, removes files at the paths given in +list+:
  #
  #   FileUtils.touch(['src0.txt', 'src0.dat'])
  #   FileUtils.rm(['src0.dat', 'src0.txt']) # => ["src0.dat", "src0.txt"]
  #
  # Keyword arguments:
  #
  # - <tt>force: true</tt> - ignores raised exceptions of StandardError
  #   and its descendants.
  # - <tt>noop: true</tt> - does not remove files; returns +nil+.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.rm(['src0.dat', 'src0.txt'], noop: true, verbose: true)
  #
  #   Output:
  #
  #     rm src0.dat src0.txt
  #
  # FileUtils.remove is an alias for FileUtils.rm.
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def rm(list, force: nil, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message "rm#{force ? ' -f' : ''} #{list.join ' '}" if verbose
    return if noop

    list.each do |path|
      remove_file path, force
    end
  end
  module_function :rm

  alias remove rm
  module_function :remove

  # Equivalent to:
  #
  #   FileUtils.rm(list, force: true, **kwargs)
  #
  # Argument +list+ (a single path or an array of paths)
  # should be {interpretable as paths}[rdoc-ref