h no shell and no shell expansion:
  #
  #   Open3.capture2('doesnt_exist') # Raises Errno::ENOENT
  #
  # If one or more +args+ is given, each is an argument or option
  # to be passed to the executable:
  #
  #   Open3.capture2('echo', 'C #')
  #   # => ["C #\n", #<Process::Status: pid 2326267 exit 0>]
  #   Open3.capture2('echo', 'hello', 'world')
  #   # => ["hello world\n", #<Process::Status: pid 2326299 exit 0>]
  #
  def capture2(*cmd)
    if Hash === cmd.last
      opts = cmd.pop.dup
    else
      opts = {}
    end

    stdin_data = opts.delete(:stdin_data)
    binmode = opts.delete(:binmode)

    popen2(*cmd, opts) {|i, o, t|
      if binmode
        i.binmode
        o.binmode
      end
      out_reader = Thread.new { o.read }
      if stdin_data
        begin
          if stdin_data.respond_to? :readpartial
            IO.copy_stream(stdin_data, i)
          else
            i.write stdin_data
          end
        rescue Errno::EPIPE
        end
      end
      i.close
      [out_reader.value, t.value]
    }
  end
  module_function :capture2

  # :call-seq:
  #   Open3.capture2e([env, ] command_line, options = {}) -> [stdout_and_stderr_s, status]
  #   Open3.capture2e([env, ] exe_path, *args, options = {}) -> [stdout_and_stderr_s, status]
  #
  # Basically a wrapper for Open3.popen3 that:
  #
  # - Creates a child process, by calling Open3.popen3 with the given arguments
  #   (except for certain entries in hash +options+; see below).
  # - Returns as string +stdout_and_stderr_s+ the merged standard output
  #   and standard error of the child process.
  # - Returns as +status+ a <tt>Process::Status</tt> object
  #   that represents the exit status of the child process.
  #
  # Returns the array <tt>[stdout_and_stderr_s, status]</tt>:
  #
  #   stdout_and_stderr_s, status = Open3.capture2e('echo "Foo"')
  #   # => ["Foo\n", #<Process::Status: pid 2371692 exit 0>]
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc@Command+Injection].
  #
  # Unlike Process.spawn, this method waits for the child process to exit
  # before returning, so the caller need not do so.
  #
  # If the first argument is a hash, it becomes leading argument +env+
  # in the call to Open3.popen3;
  # see {Execution Environment}[rdoc-ref:Process@Execution+Environment].
  #
  # If the last argument is a hash, it becomes trailing argument +options+
  # in the call to Open3.popen3;
  # see {Execution Options}[rdoc-ref:Process@Execution+Options].
  #
  # The hash +options+ is given;
  # two options have local effect in method Open3.capture2e:
  #
  # - If entry <tt>options[:stdin_data]</tt> exists, the entry is removed
  #   and its string value is sent to the command's standard input:
  #
  #     Open3.capture2e('tee', stdin_data: 'Foo')
  #     # => ["Foo", #<Process::Status: pid 2371732 exit 0>]
  #
  # - If entry <tt>options[:binmode]</tt> exists, the entry is removed and
  #   the internal streams are set to binary mode.
  #
  # The single required argument is one of the following:
  #
  # - +command_line+ if it is a string,
  #   and if it begins with a shell reserved word or special built-in,
  #   or if it contains one or more metacharacters.
  # - +exe_path+ otherwise.
  #
  # <b>Argument +command_line+</b>
  #
  # \String argument +command_line+ is a command line to be passed to a shell;
  # it must begin with a shell reserved word, begin with a special built-in,
  # or contain meta characters:
  #
  #   Open3.capture2e('if true; then echo "Foo"; fi') # Shell reserved word.
  #   # => ["Foo\n", #<Process::Status: pid 2371740 exit 0>]
  #   Open3.capture2e('echo')                         # Built-in.
  #   # => ["\n", #<Process::Status: pid 2371774 exit 0>]
  #   Open3.capture2e('date > date.tmp')              # Contains meta character.
  #   # => ["", #<Process::Status: pid 2371812 exit 0>]
  #
  # The command line may also contain arguments and options for the command:
  #
  #   Open3.capture2e('echo "Foo"')
  #   # => ["Foo\n", #<Process::Status: pid 2326183 exit 0>]
  #
  # <b>Argument +exe_path+</b>
  #
  # Argument +exe_path+ is one of the following:
  #
  # - The string path to an executable to be called.
  # - A 2-element array containing the path to an executable
  #   and the string to be used as the name of the executing process.
  #
  # Example:
  #
  #   Open3.capture2e('/usr/bin/date')
  #   # => ["Sat Sep 30 09:01:46 AM CDT 2023\n", #<Process::Status: pid 2371820 exit 0>]
  #
  # Ruby invokes the executable directly, with no shell and no shell expansion:
  #
  #   Open3.capture2e('doesnt_exist') # Raises Errno::ENOENT
  #
  # If one or more +args+ is given, each is an argument or option
  # to be passed to the executable:
  #
  #   Open3.capture2e('echo', 'C #')
  #   # => ["C #\n", #<Process::Status: pid 2371856 exit 0>]
  #   Open3.capture2e('echo', 'hello', 'world')
  #   # => ["hello world\n", #<Process::Status: pid 2371894 exit 0>]
  #
  def capture2e(*cmd)
    if Hash === cmd.last
      opts = cmd.pop.dup
    else
      opts = {}
    end

    stdin_data = opts.delete(:stdin_data)
    binmode = opts.delete(:binmode)

    popen2e(*cmd, opts) {|i, oe, t|
      if binmode
        i.binmode
        oe.binmode
      end
      outerr_reader = Thread.new { oe.read }
      if stdin_data
        begin
          if stdin_data.respond_to? :readpartial
            IO.copy_stream(stdin_data, i)
          else
            i.write stdin_data
          end
        rescue Errno::EPIPE
        end
      end
      i.close
      [outerr_reader.value, t.value]
    }
  end
  module_function :capture2e

  # :call-seq:
  #   Open3.pipeline_rw([env, ] *cmds, options = {}) -> [first_stdin, last_stdout, wait_threads]
  #
  # Basically a wrapper for
  # {Process.spawn}[rdoc-ref:Process.spawn]
  # that:
  #
  # - Creates a child process for each of the given +cmds+
  #   by calling Process.spawn.
  # - Pipes the +stdout+ from each child to the +stdin+ of the next child,
  #   or, for the first child, from the caller's +stdin+,
  #   or, for the last child, to the caller's +stdout+.
  #
  # The method does not wait for child processes to exit,
  # so the caller must do so.
  #
  # With no block given, returns a 3-element array containing:
  #
  # - The +stdin+ stream of the first child process.
  # - The +stdout+ stream of the last child process.
  # - An array of the wait threads for all of the child processes.
  #
  # Example:
  #
  #   first_stdin, last_stdout, wait_threads = Open3.pipeline_rw('sort', 'cat -n')
  #   # => [#<IO:fd 20>, #<IO:fd 21>, [#<Process::Waiter:0x000055e8de29ab40 sleep>, #<Process::Waiter:0x000055e8de29a690 sleep>]]
  #   first_stdin.puts("foo\nbar\nbaz")
  #   first_stdin.close # Send EOF to sort.
  #   puts last_stdout.read
  #   wait_threads.each do |wait_thread|
  #     wait_thread.join
  #   end
  #
  # Output:
  #
  #   1	bar
  #   2	baz
  #   3	foo
  #
  # With a block given, calls the block with the +stdin+ stream of the first child,
  # the +stdout+ stream  of the last child,
  # and an array of the wait processes:
  #
  #   Open3.pipeline_rw('sort', 'cat -n') do |first_stdin, last_stdout, wait_threads|
  #     first_stdin.puts "foo\nbar\nbaz"
  #     first_stdin.close # send EOF to sort.
  #     puts last_stdout.read
  #     wait_threads.each do |wait_thread|
  #       wait_thread.join
  #     end
  #   end
  #
  # Output:
  #
  #   1	bar
  #   2	baz
  #   3	foo
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc@Command+Injection].
  #
  # If the first argument is a hash, it becomes leading argument +env+
  # in each call to Process.spawn;
  # see {Execution Environment}[rdoc-ref:Process@Execution+Environment].
  #
  # If the last argument is a hash, it becomes trailing argument +options+
  # in each call to Process.spawn;
  # see {Execution Options}[rdoc-ref:Process@Execution+Options].
  #
  # Each remaining argument in +cmds+ is one of:
  #
  # - A +command_line+: a string that begins with a shell reserved word
  #   or special built-in, or contains one or more metacharacters.
  # - An +exe_path+: the string path to an executable to be called.
  # - An array containing a +command_line+ or an +exe_path+,
  #   along with zero or more string arguments for the command.
  #
  # See {Argument command_line or exe_path}[rdoc-ref:Process@Argument+command_line+or+exe_path].
  #
  def pipeline_rw(*cmds, &block)
    if Hash === cmds.last
      opts = cmds.pop.dup
    else
      opts = {}
    end

    in_r, in_w = IO.pipe
    opts[:in] = in_r
    in_w.sync = true

    out_r, out_w = IO.pipe
    opts[:out] = out_w

    pipeline_run(cmds, opts, [in_r, out_w], [in_w, out_r], &block)
  end
  module_function :pipeline_rw

  # :call-seq:
  #   Open3.pipeline_r([env, ] *cmds, options = {}) -> [last_stdout, wait_threads]
  #
  # Basically a wrapper for
  # {Process.spawn}[rdoc-ref:Process.spawn]
  # that:
  #
  # - Creates a child process for each of the given +cmds+
  #   by calling Process.spawn.
  # - Pipes the +stdout+ from each child to the +stdin+ of the next child,
  #   or, for the last child, to the caller's +stdout+.
  #
  # The method does not wait for child processes to exit,
  # so the caller must do so.
  #
  # With no block given, returns a 2-element array containing:
  #
  # - The +stdout+ stream of the last child process.
  # - An array of the wait threads for all of the child processes.
  #
  # Example:
  #
  #   last_stdout, wait_threads = Open3.pipeline_r('ls', 'grep R')
  #   # => [#<IO:fd 5>, [#<Process::Waiter:0x000055e8de2f9898 dead>, #<Process::Waiter:0x000055e8de2f94b0 sleep>]]
  #   puts last_stdout.read
  #   wait_threads.each do |wait_thread|
  #     wait_thread.join
  #   end
  #
  # Output:
  #
  #   Rakefile
  #   README.md
  #
  # With a block given, calls the block with the +stdout+ stream
  # of the last child process,
  # and an array of the wait processes:
  #
  #   Open3.pipeline_r('ls', 'grep R') do |last_stdout, wait_threads|
  #     puts last_stdout.read
  #     wait_threads.each do |wait_thread|
  #       wait_thread.join
  #     end
  #   end
  #
  # Output:
  #
  #   Rakefile
  #   README.md
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc@Command+Injection].
  #
  # If the first argument is a hash, it becomes leading argument +env+
  # in each call to Process.spawn;
  # see {Execution Environment}[rdoc-ref:Process@Execution+Environment].
  #
  # If the last argument is a hash, it becomes trailing argument +options+
  # in each call to Process.spawn;
  # see {Execution Options}[rdoc-ref:Process@Execution+Options].
  #
  # Each remaining argument in +cmds+ is one of:
  #
  # - A +command_line+: a string that begins with a shell reserved word
  #   or special built-in, or contains one or more metacharacters.
  # - An +exe_path+: the string path to an executable to be called.
  # - An array containing a +command_line+ or an +exe_path+,
  #   along with zero or more string arguments for the command.
  #
  # See {Argument command_line or exe_path}[rdoc-ref:Process@Argument+command_line+or+exe_path].
  #
  def pipeline_r(*cmds, &block)
    if Hash === cmds.last
      opts = cmds.pop.dup
    else
      opts = {}
    end

    out_r, out_w = IO.pipe
    opts[:out] = out_w

    pipeline_run(cmds, opts, [out_w], [out_r], &block)
  end
  module_function :pipeline_r


  # :call-seq:
  #   Open3.pipeline_w([env, ] *cmds, options = {}) -> [first_stdin, wait_threads]
  #
  # Basically a wrapper for
  # {Process.spawn}[rdoc-ref:Process.spawn]
  # that:
  #
  # - Creates a child process for each of the given +cmds+
  #   by calling Process.spawn.
  # - Pipes the +stdout+ from each child to the +stdin+ of the next child,
  #   or, for the first child, pipes the caller's +stdout+ to the child's +stdin+.
  #
  # The method does not wait for child processes to exit,
  # so the caller must do so.
  #
  # With no block given, returns a 2-element array containing:
  #
  # - The +stdin+ stream of the first child process.
  # - An array of the wait threads for all of the child processes.
  #
  # Example:
  #
  #   first_stdin, wait_threads = Open3.pipeline_w('sort', 'cat -n')
  #   # => [#<IO:fd 7>, [#<Process::Waiter:0x000055e8de928278 run>, #<Process::Waiter:0x000055e8de923e80 run>]]
  #   first_stdin.puts("foo\nbar\nbaz")
  #   first_stdin.close # Send EOF to sort.
  #   wait_threads.each do |wait_thread|
  #     wait_thread.join
  #   end
  #
  # Output:
  #
  #   1	bar
  #   2	baz
  #   3	foo
  #
  # With a block given, calls the block with the +stdin+ stream
  # of the first child process,
  # and an array of the wait processes:
  #
  #   Open3.pipeline_w('sort', 'cat -n') do |first_stdin, wait_threads|
  #     first_stdin.puts("foo\nbar\nbaz")
  #     first_stdin.close # Send EOF to sort.
  #     wait_threads.each do |wait_thread|
  #       wait_thread.join
  #     end
  #   end
  #
  # Output:
  #
  #   1	bar
  #   2	baz
  #   3	foo
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc@Command+Injection].
  #
  # If the first argument is a hash, it becomes leading argument +env+
  # in each call to Process.spawn;
  # see {Execution Environment}[rdoc-ref:Process@Execution+Environment].
  #
  # If the last argument is a hash, it becomes trailing argument +options+
  # in each call to Process.spawn;
  # see {Execution Options}[rdoc-ref:Process@Execution+Options].
  #
  # Each remaining argument in +cmds+ is one of:
  #
  # - A +command_line+: a string that begins with a shell reserved word
  #   or special built-in, or contains one or more metacharacters.
  # - An +exe_path+: the string path to an executable to be called.
  # - An array containing a +command_line+ or an +exe_path+,
  #   along with zero or more string arguments for the command.
  #
  # See {Argument command_line or exe_path}[rdoc-ref:Process@Argument+command_line+or+exe_path].
  #
  def pipeline_w(*cmds, &block)
    if Hash === cmds.last
      opts = cmds.pop.dup
    else
      opts = {}
    end

    in_r, in_w = IO.pipe
    opts[:in] = in_r
    in_w.sync = true

    pipeline_run(cmds, opts, [in_r], [in_w], &block)
  end
  module_function :pipeline_w

  # :call-seq:
  #   Open3.pipeline_start([env, ] *cmds, options = {}) -> [wait_threads]
  #
  # Basically a wrapper for
  # {Process.spawn}[rdoc-ref:Process.spawn]
  # that:
  #
  # - Creates a child process for each of the given +cmds+
  #   by calling Process.spawn.
  # - Does not wait for child processes to exit.
  #
  # With no block given, returns an array of the wait threads
  # for all of the child processes.
  #
  # Example:
  #
  #   wait_threads = Open3.pipeline_start('ls', 'grep R')
  #   # => [#<Process::Waiter:0x000055e8de9d2bb0 run>, #<Process::Waiter:0x000055e8de9d2890 run>]
  #   wait_threads.each do |wait_thread|
  #     wait_thread.join
  #   end
  #
  # Output:
  #
  #   Rakefile
  #   README.md
  #
  # With a block given, calls the block with an array of the wait processes:
  #
  #   Open3.pipeline_start('ls', 'grep R') do |wait_threads|
  #     wait_threads.each do |wait_thread|
  #       wait_thread.join
  #     end
  #   end
  #
  # Output:
  #
  #   Rakefile
  #   README.md
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc@Command+Injection].
  #
  # If the first argument is a hash, it becomes leading argument +env+
  # in each call to Process.spawn;
  # see {Execution Environment}[rdoc-ref:Process@Execution+Environment].
  #
  # If the last argument is a hash, it becomes trailing argument +options+
  # in each call to Process.spawn;
  # see {Execution Options}[rdoc-ref:Process@Execution+Options].
  #
  # Each remaining argument in +cmds+ is one of:
  #
  # - A +command_line+: a string that begins with a shell reserved word
  #   or special built-in, or contains one or more metacharacters.
  # - An +exe_path+: the string path to an executable to be called.
  # - An array containing a +command_line+ or an +exe_path+,
  #   along with zero or more string arguments for the command.
  #
  # See {Argument command_line or exe_path}[rdoc-ref:Process@Argument+command_line+or+exe_path].
  #
  def pipeline_start(*cmds, &block)
    if Hash === cmds.last
      opts = cmds.pop.dup
    else
      opts = {}
    end

    if block
      pipeline_run(cmds, opts, [], [], &block)
    else
      ts, = pipeline_run(cmds, opts, [], [])
      ts
    end
  end
  module_function :pipeline_start

  # :call-seq:
  #   Open3.pipeline([env, ] *cmds, options = {}) -> array_of_statuses
  #
  # Basically a wrapper for
  # {Process.spawn}[rdoc-ref:Process.spawn]
  # that:
  #
  # - Creates a child process for each of the given +cmds+
  #   by calling Process.spawn.
  # - Pipes the +stdout+ from each child to the +stdin+ of the next child,
  #   or, for the last child, to the caller's +stdout+.
  # - Waits for the child processes to exit.
  # - Returns an array of Process::Status objects (one for each child).
  #
  # Example:
  #
  #   wait_threads = Open3.pipeline('ls', 'grep R')
  #   # => [#<Process::Status: pid 2139200 exit 0>, #<Process::Status: pid 2139202 exit 0>]
  #
  # Output:
  #
  #   Rakefile
  #   README.md
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc@Command+Injection].
  #
  # If the first argument is a hash, it becomes leading argument +env+
  # in each call to Process.spawn;
  # see {Execution Environment}[rdoc-ref:Process@Execution+Environment].
  #
  # If the last argument is a hash, it becomes trailing argument +options+
  # in each call to Process.spawn'
  # see {Execution Options}[rdoc-ref:Process@Execution+Options].
  #
  # Each remaining argument in +cmds+ is one of:
  #
  # - A +command_line+: a string that begins with a shell reserved word
  #   or special built-in, or contains one or more metacharacters.
  # - An +exe_path+: the string path to an executable to be called.
  # - An array containing a +command_line+ or an +exe_path+,
  #   along with zero or more string arguments for the command.
  #
  # See {Argument command_line or exe_path}[rdoc-ref:Process@Argument+command_line+or+exe_path].
  #
  def pipeline(*cmds)
    if Hash === cmds.last
      opts = cmds.pop.dup
    else
      opts = {}
    end

    pipeline_run(cmds, opts, [], []) {|ts|
      ts.map(&:value)
    }
  end
  module_function :pipeline

  def pipeline_run(cmds, pipeline_opts, child_io, parent_io) # :nodoc:
    if cmds.empty?
      raise ArgumentError, "no commands"
    end

    opts_base = pipeline_opts.dup
    opts_base.delete :in
    opts_base.delete :out

    wait_thrs = []
    r = nil
    cmds.each_with_index {|cmd, i|
      cmd_opts = opts_base.dup
      if String === cmd
        cmd = [cmd]
      else
        cmd_opts.update cmd.pop if Hash === cmd.last
      end
      if i == 0
        if !cmd_opts.include?(:in)
          if pipeline_opts.include?(:in)
            cmd_opts[:in] = pipeline_opts[:in]
          end
        end
      else
        cmd_opts[:in] = r
      end
      if i != cmds.length - 1
        r2, w2 = IO.pipe
        cmd_opts[:out] = w2
      else
        if !cmd_opts.include?(:out)
          if pipeline_opts.include?(:out)
            cmd_opts[:out] = pipeline_opts[:out]
          end
        end
      end
      pid = spawn(*cmd, cmd_opts)
      wait_thrs << Process.detach(pid)
      r&.close
      w2&.close
      r = r2
    }
    result = parent_io + [wait_thrs]
    child_io.each(&:close)
    if defined? yield
      begin
        return yield(*result)
      ensure
        parent_io.each(&:close)
        wait_thrs.each(&:join)
      end
    end
    result
  end
  module_function :pipeline_run
  class << self
    private :pipeline_run
  end

end

# JRuby uses different popen logic on Windows, require it here to reuse wrapper methods above.
require 'open3/jruby_windows' if RUBY_ENGINE == 'jruby' && JRuby::Util::ON_WINDOWS
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     ruby/did_you_mean.rb                                                                                0000644                 00000011006 15040313424 0010500 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       require_relative "did_you_mean/version"
require_relative "did_you_mean/core_ext/name_error"

require_relative "did_you_mean/spell_checker"
require_relative 'did_you_mean/spell_checkers/name_error_checkers'
require_relative 'did_you_mean/spell_checkers/method_name_checker'
require_relative 'did_you_mean/spell_checkers/key_error_checker'
require_relative 'did_you_mean/spell_checkers/null_checker'
require_relative 'did_you_mean/spell_checkers/require_path_checker'
require_relative 'did_you_mean/spell_checkers/pattern_key_name_checker'
require_relative 'did_you_mean/formatter'
require_relative 'did_you_mean/tree_spell_checker'

# The +DidYouMean+ gem adds functionality to suggest possible method/class
# names upon errors such as +NameError+ and +NoMethodError+. In Ruby 2.3 or
# later, it is automatically activated during startup.
#
# @example
#
#   methosd
#   # => NameError: undefined local variable or method `methosd' for main:Object
#   #   Did you mean?  methods
#   #                  method
#
#   OBject
#   # => NameError: uninitialized constant OBject
#   #    Did you mean?  Object
#
#   @full_name = "Yuki Nishijima"
#   first_name, last_name = full_name.split(" ")
#   # => NameError: undefined local variable or method `full_name' for main:Object
#   #    Did you mean?  @full_name
#
#   @@full_name = "Yuki Nishijima"
#   @@full_anme
#   # => NameError: uninitialized class variable @@full_anme in Object
#   #    Did you mean?  @@full_name
#
#   full_name = "Yuki Nishijima"
#   full_name.starts_with?("Y")
#   # => NoMethodError: undefined method `starts_with?' for "Yuki Nishijima":String
#   #    Did you mean?  start_with?
#
#   hash = {foo: 1, bar: 2, baz: 3}
#   hash.fetch(:fooo)
#   # => KeyError: key not found: :fooo
#   #    Did you mean?  :foo
#
#
# == Disabling +did_you_mean+
#
# Occasionally, you may want to disable the +did_you_mean+ gem for e.g.
# debugging issues in the error object itself. You can disable it entirely by
# specifying +--disable-did_you_mean+ option to the +ruby+ command:
#
#   $ ruby --disable-did_you_mean -e "1.zeor?"
#   -e:1:in `<main>': undefined method `zeor?' for 1:Integer (NameError)
#
# When you do not have direct access to the +ruby+ command (e.g.
# +rails console+, +irb+), you could applyoptions using the +RUBYOPT+
# environment variable:
#
#   $ RUBYOPT='--disable-did_you_mean' irb
#   irb:0> 1.zeor?
#   # => NoMethodError (undefined method `zeor?' for 1:Integer)
#
#
# == Getting the original error message
#
# Sometimes, you do not want to disable the gem entirely, but need to get the
# original error message without suggestions (e.g. testing). In this case, you
# could use the +#original_message+ method on the error object:
#
#   no_method_error = begin
#                       1.zeor?
#                     rescue NoMethodError => error
#                       error
#                     end
#
#   no_method_error.message
#   # => NoMethodError (undefined method `zeor?' for 1:Integer)
#   #    Did you mean?  zero?
#
#   no_method_error.original_message
#   # => NoMethodError (undefined method `zeor?' for 1:Integer)
#
module DidYouMean
  # Map of error types and spell checker objects.
  @spell_checkers = Hash.new(NullChecker)

  # Returns a sharable hash map of error types and spell checker objects.
  def self.spell_checkers
    @spell_checkers
  end

  # Adds +DidYouMean+ functionality to an error using a given spell checker
  def self.correct_error(error_class, spell_checker)
    if defined?(Ractor)
      new_mapping = { **@spell_checkers, error_class.to_s => spell_checker }
      new_mapping.default = NullChecker

      @spell_checkers = Ractor.make_shareable(new_mapping)
    else
      spell_checkers[error_class.to_s] = spell_checker
    end

    error_class.prepend(Correctable) if error_class.is_a?(Class) && !(error_class < Correctable)
  end

  correct_error NameError, NameErrorCheckers
  correct_error KeyError, KeyErrorChecker
  correct_error NoMethodError, MethodNameChecker
  correct_error LoadError, RequirePathChecker if RUBY_VERSION >= '2.8.0'
  correct_error NoMatchingPatternKeyError, PatternKeyNameChecker if defined?(::NoMatchingPatternKeyError)

  # Returns the currently set formatter. By default, it is set to +DidYouMean::Formatter+.
  def self.formatter
    if defined?(Ractor)
      Ractor.current[:__did_you_mean_formatter__] || Formatter
    else
      Formatter
    end
  end

  # Updates the primary formatter used to format the suggestions.
  def self.formatter=(formatter)
    if defined?(Ractor)
      Ractor.current[:__did_you_mean_formatter__] = formatter
    end
  end
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          ruby/drb/version.rb                                                                                 0000644                 00000000043 15040313424 0010277 0                                                                                                    ustar 00                                                                                                                                                                                                                                                        mfile.print("install-so: ")
    if                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ruby/drb/timeridconv.rb                                                                             0000644                 00000004245 15040313424 0011145 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ("do-install-rb:\n")
    mfile.print("do-install-rb-default:\n")
    for sfx, i in [["-default", [["lib/**/*.rb", "$(RUBYLIBDIR)", "lib"]]], ["", $INSTALLFILES]]
      files = install_files(mfile, i, nil, srcprefix) or next
      for dir, *files in files
        unless dirs.include?(dir)
          dirs << dir
          mfile.print "pre-install-rb#{sfx}: #{timestamp_file(dir, target_prefix)}\n"
        end
        for f in files
          dest = "#{dir}/#{File.basename(f)}"
          mfile.print("do-install-rb#{sfx}: #{dest}\n")
          mfile.print("#{dest}: #{f} #{timestamp_file(dir, target_prefix)}\n")
          mfile.print("\t$(Q) $(#{$extout ? 'COPY' : 'INSTALL_DATA'}) #{f} $(@D)\n")
          if defined?($installed_list) and !$extout
            mfile.print("\t@echo #{dest}>>$(INSTALLED_LIST)\n")
          end
          if $extout
            mfile.print("clean-rb#{sfx}::\n")
            mfile.print("\t-$(Q)$(RM) #{fseprepl[dest]}\n")
          end
        end
      end
      mfile.print "pre-install-rb#{sfx}:\n"
      if files.empty?
        mfile.print("\t@$(NULLCMD)\n")
      else
        q = "$(MAKE) -q do-install-rb#{sfx}"
        if $nmake
          mfile.print "!if \"$(Q)\" == \"@\"\n\t@#{q} || \\\n!endif\n\t"
        else
          mfile.print "\t$(Q1:0=@#{q} || )"
        end
        mfile.print "$(ECHO1:0=echo) installing#{sfx.sub(/^-/, " ")} #{target} libraries\n"
      end
      if $extout
        dirs.uniq!
        unless dirs.empty?
          mfile.print("clean-rb#{sfx}::\n")
          for dir in dirs.sort_by {|d| -d.count('/')}
            stamp = timestamp_file(dir, target_prefix)
            mfile.print("\t-$(Q)$(RM) #{fseprepl[stamp]}\n")
            mfile.print("\t-$(Q)$(RMDIRS) #{fseprepl[dir]}#{$ignore_error}\n")
          end
        end
      end
    end
    if target and !dirs.include?(sodir)
      mfile.print "$(TARGET_SO_DIR_TIMESTAMP):\n\t$(Q) $(MAKEDIRS) $(@D) #{sodir}\n\t$(Q) $(TOUCH) $@\n"
    end
    dirs.each do |d|
      t = timestamp_file(d, target_prefix)
      mfile.print "#{t}:\n\t$(Q) $(MAKEDIRS) $(@D) #{d}\n\t$(Q) $(TOUCH) $@\n"
    end

    mfile.print <<-SITEINSTALL

site-install: site-install-so site-install-rb
site-install-so:                                                                                                                                                                                                                                                                                                                                                           ruby/drb/eq.rb                                                                                      0000644                 00000000423 15040313424 0007221 0                                                                                                    ustar 00                                                                                                                                                                                                                                                        mfile.printf(rule, e, $ASMEXT)
        mfile.print(asm_command)
      end
    end

    mfile.print "$(TARGET_SO): "
    mfile.print "$(DEFFILE) " if makedef
    mfile.print "$(OBJS) Makefile"
    mfile.print " $(TARGET_SO_DIR_TIMESTAMP)" if $extout
    mfile.print "\n"
                                                                                                                                                                                                                                                 ruby/drb/invokemethod.rb                                                                            0000644                 00000001411 15040313424 0011306 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       file.print "$(DEFFILE): #{origdef}\n"
      mfile.print "\t$(ECHO) generating $(@#{rsep})\n"
      mfile.print "\t$(Q) #{makedef} > $@\n\n"
    end

    depend = File.join(srcdir, "depend")
    if File.exist?(depend)
      mfile.print("###\n", *depend_rules(File.read(depend)))
    else
      mfile.print "$(OBJS): $(HDRS) $(ruby_headers)\n"
    end

    $makefile_created = true
  ensure
    mfile.close if mfile
  end

  # :stopdoc:

  def init_mkmf(config = CONFIG, rbconfig = RbConfig::CONFIG)
    $makefile_created = false
    $arg_config = []
    $enable_shared = config['ENABLE_SHARED'] == 'yes'
    $defs = []
    $extconf_h = nil
    $config_dirs = {}

    if $warnflags = CONFIG['warnflags'] and CONFIG['GCC'] == 'yes'
      # turn warnings into errors only for bundl                                                                                                                                                                                                                                                       ruby/drb/extservm.rb                                                                                0000644                 00000003467 15040313424 0010504 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       xflags", arg_config("CXXFLAGS", config["CXXFLAGS"]))||'').dup
    $ARCH_FLAG = with_config("arch_flag", arg_config("ARCH_FLAG", config["ARCH_FLAG"])).dup
    $CPPFLAGS = with_config("cppflags", arg_config("CPPFLAGS", config["CPPFLAGS"])).dup
    $LDFLAGS = with_config("ldflags", arg_config("LDFLAGS", config["LDFLAGS"])).dup
    $INCFLAGS = "-I$(arch_hdrdir)"
    $INCFLAGS << " -I$(hdrdir)/ruby/backward" unless $extmk
    $INCFLAGS << " -I$(hdrdir) -I$(srcdir)"
    $DLDFLAGS = with_config("dldflags", arg_config("DLDFLAGS", config["DLDFLAGS"])).dup
    config_string("ADDITIONAL_DLDFLAGS") {|flags| $DLDFLAGS << " " << flags} unless $extmk
    $LIBEXT = config['LIBEXT'].dup
    $OBJEXT = config["OBJEXT"].dup
    $EXEEXT = config["EXEEXT"].dup
    $ASMEXT = config_string('ASMEXT', &:dup) || 'S'
    $LIBS = "#{config['LIBS']} #{config['DLDLIBS']}"
    $LIBRUBYARG = ""
    $LIBRUBYARG_STATIC = config['LIBRUBYARG_STATIC']
    $LIBRUBYARG_SHARED = config['LIBRUBYARG_SHARED']
    $DEFLIBPATH = [$extmk ? "$(topdir)" : "$(#{config["libdirname"] || "libdir"})"]
    $DEFLIBPATH.unshift(".")
    $LIBPATH = []
    $INSTALLFILES = []
    $NONINSTALLFILES = [/~\z/, /\A#.*#\z/, /\A\.#/, /\.bak\z/i, /\.orig\z/, /\.rej\z/, /\.l[ao]\z/, /\.o\z/]
    $VPATH = %w[$(srcdir) $(arch_hdrdir)/ruby $(hdrdir)/ruby]

    $objs = nil
    $srcs = nil
    $headers = []
    $libs = ""
    if $enable_shared or RbConfig.expand(config["LIBRUBY"].dup) != RbConfig.expand(config["LIBRUBY_A"].dup)
      $LIBRUBYARG = config['LIBRUBYARG']
    end

    $LOCAL_LIBS = ""

    $cleanfiles = config_string('CLEANFILES') {|s| Shellwords.shellwords(s)} || []
    $cleanfiles << "mkmf.log"
    $distcleanfiles = config_string('DISTCLEANFILES') {|s| Shellwords.shellwords(s)} || []
    $distcleandirs = config_string('DISTCLEANDIRS') {|s| Shellwords.shellwords(s)} || []

                                                                                                                                                                                                          ruby/drb/gw.rb                                                                                      0000644                 00000006005 15040313424 0007233 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       \n" + FailedMessage + opts.join
    end
  end

  private

  def _libdir_basename
    @libdir_basename ||= config_string("libdir") {|name| name[/\A\$\(exec_prefix\)\/(.*)/, 1]} || "lib"
  end

  def MAIN_DOES_NOTHING(*refs)
    src = MAIN_DOES_NOTHING
    unless refs.empty?
      src = src.sub(/\{/) do
        $& +
          "\n  if (argc > 1000000) {\n" +
          refs.map {|n|"    int (* volatile #{n}p)(void)=(int (*)(void))&#{n};\n"}.join("") +
          refs.map {|n|"    printf(\"%d\", (*#{n}p)());\n"}.join("") +
          "  }\n"
      end
    end
    src
  end

  extend self
  init_mkmf

  $make = with_config("make-prog", ENV["MAKE"] || "make")
  make, = Shellwords.shellwords($make)
  $nmake = nil
  case
  when $mswin
    $nmake = ?m if /nmake/i =~ make
  end
  $ignore_error = $nmake ? '' : ' 2> /dev/null || true'

  RbConfig::CONFIG["srcdir"] = CONFIG["srcdir"] =
    $srcdir = arg_config("--srcdir", File.dirname($0))
  $configure_args["--topsrcdir"] ||= $srcdir
  if $curdir = arg_config("--curdir")
    RbConfig.expand(curdir = $curdir.dup)
  else
    curdir = $curdir = "."
  end
  unless File.expand_path(RbConfig::CONFIG["topdir"]) == File.expand_path(curdir)
    CONFIG["topdir"] = $curdir
    RbConfig::CONFIG["topdir"] = curdir
  end
  $configure_args["--topdir"] ||= $curdir
  $ruby = arg_config("--ruby", File.join(RbConfig::CONFIG["bindir"], CONFIG["ruby_install_name"]))

  RbConfig.expand(CONFIG["RUBY_SO_NAME"])

  # :startdoc:

  split = Shellwords.method(:shellwords).to_proc

  EXPORT_PREFIX = config_string('EXPORT_PREFIX') {|s| s.strip}

  hdr = ['#include "ruby.h"' "\n"]
  config_string('COMMON_MACROS') do |s|
    Shellwords.shellwords(s).each do |w|
      w, v = w.split(/=/, 2)
      hdr << "#ifndef #{w}"
      hdr << "#define #{[w, v].compact.join(" ")}"
      hdr << "#endif /* #{w} */"
    end
  end
  config_string('COMMON_HEADERS') do |s|
    Shellwords.shellwords(s).each {|w| hdr << "#include <#{w}>"}
  end

  ##
  # Common headers for Ruby C extensions

  COMMON_HEADERS = hdr.join("\n")

  ##
  # Common libraries for Ruby C extensions

  COMMON_LIBS = config_string('COMMON_LIBS', &split) || []

  ##
  # make compile rules

  COMPILE_RULES = config_string('COMPILE_RULES', &split) || %w[.%s.%s:]
  RULE_SUBST = config_string('RULE_SUBST')

  ##
  # Command which will compile C files in the generated Makefile

  COMPILE_C = config_string('COMPILE_C') || '$(CC) $(INCFLAGS) $(CPPFLAGS) $(CFLAGS) $(COUTFLAG)$@ -c $(CSRCFLAG)$<'

  ##
  # Command which will compile C++ files in the generated Makefile

  COMPILE_CXX = config_string('COMPILE_CXX') || '$(CXX) $(INCFLAGS) $(CPPFLAGS) $(CXXFLAGS) $(COUTFLAG)$@ -c $(CSRCFLAG)$<'

  ##
  # Command which will translate C files to assembler sources in the generated Makefile

  ASSEMBLE_C = config_string('ASSEMBLE_C') || COMPILE_C.sub(/(?<=\s)-c(?=\s)/, '-S')

  ##
  # Command which will translate C++ files to assembler sources in the generated Makefile

  ASSEMBLE_CXX = config_string('ASSEMBLE_CXX') || COMPILE_CXX.sub(/(?<=\s)-c(?=\s)/, '-S')

  ##
  # Command which wi                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           ruby/drb/acl.rb                                                                                     0000644                 00000011161 15040313424 0007354 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       onfig_string('UNIVERSAL_INTS') {|s| Shellwords.shellwords(s)} ||
    %w[int short long long\ long]

  sep = config_string('BUILD_FILE_SEPARATOR') {|s| ":/=#{s}" if s != "/"} || ""

  ##
  # Makefile rules that                                                                                                      ruby/uri.rb                                                                                         0000644                 00000006073 15040313425 0006654 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       

realclean: distclean
"

  @lang = Hash.new(self)

  def self.[](name)
    @lang.fetch(name)
  end

  def self.[]=(name, mod)
    @lang[name] = mod
  end

  self["C++"] = Module.new do
    include MakeMakefile
    extend self

    CONFTEST_CXX = "#{CONFTEST}.#{config_string('CXX_EXT') || CXX_EXT[0]}"

    TRY_LINK_CXX = config_string('TRY_LINK_CXX') ||
                   ((cmd = TRY_LINK.gsub(/\$\(C(?:C|(FLAGS))\)/, '$(CXX\1)')) != TRY_LINK && cmd) ||
                   "$(CXX) #{OUTFLAG}#{CONFTEST}#{$EXEEXT} $(INCFLAGS) $(CPPFLAGS) " \
                   "$(CXXFLAGS) $(src) $(LIBPATH) $(LDFLAGS) $(ARCH_FLAG) $(LOCAL_LIBS) $(LIBS)"

    def have_devel?
      unless defined? @have_devel
        @have_devel = true
        @have_devel = try_link(MAIN_DOES_NOTHING)
      end
      @have_devel
    end

    def conftest_source
      CONFTEST_CXX
    end

    def cc_command(opt="")
      conf = cc_config(opt)
      RbConfig::expand("$(CXX) #$INCFLAGS #$CPPFLAGS #$CXXFLAGS #$ARCH_FLAG #{opt} -c #{CONFTEST_CXX}",
                       conf)
    end

    def link_command(ldflags, *opts)
      conf = link_config(ldflags, *opts)
      RbConfig::expand(TRY_LINK_CXX.dup, conf)
    end
  end
end

# MakeMakefile::Global = #
m = Module.new {
  include(MakeMakefile)
  private(*MakeMakefile.public_instance_methods(false))
}
include m

if not $extmk and /\A(extconf|makefile).rb\z/ =~ File.basename($0)
  END {mkmf_failed($0)}
end
                                                                                                     ruby/uri.rb                                                                                         0000644                 00000006073 15040313425 0006654 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
# URI is a module providing classes to handle Uniform Resource Identifiers
# (RFC2396[http://tools.ietf.org/html/rfc2396]).
#
# == Features
#
# * Uniform way of handling URIs.
# * Flexibility to introduce custom URI schemes.
# * Flexibility to have an alternate URI::Parser (or just different patterns
#   and regexp's).
#
# == Basic example
#
#   require 'uri'
#
#   uri = URI("http://foo.com/posts?id=30&limit=5#time=1305298413")
#   #=> #<URI::HTTP http://foo.com/posts?id=30&limit=5#time=1305298413>
#
#   uri.scheme    #=> "http"
#   uri.host      #=> "foo.com"
#   uri.path      #=> "/posts"
#   uri.query     #=> "id=30&limit=5"
#   uri.fragment  #=> "time=1305298413"
#
#   uri.to_s      #=> "http://foo.com/posts?id=30&limit=5#time=1305298413"
#
# == Adding custom URIs
#
#   module URI
#     class RSYNC < Generic
#       DEFAULT_PORT = 873
#     end
#     register_scheme 'RSYNC', RSYNC
#   end
#   #=> URI::RSYNC
#
#   URI.scheme_list
#   #=> {"FILE"=>URI::File, "FTP"=>URI::FTP, "HTTP"=>URI::HTTP,
#   #    "HTTPS"=>URI::HTTPS, "LDAP"=>URI                                                                                                                                                                                                                                                                                                                                                                                                                                                                     ruby/securerandom.rb                                                                                0000644                 00000004162 15040313425 0010541 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                                                                                                                                                                                ruby/drb/weakidconv.rb                                                                              0000644                 00000002175 15040313424 0010754 0                                                                                                    ustar 00                                                                                                                                                                                                                                                        Ruby.
#

module URI
end

require_relative 'uri/version'
require_relative 'uri/common'
require_relative 'uri/generic'
require_relative 'uri/file'
require_relative 'uri/ftp'
require_relative 'uri/http'
require_relative 'uri/https'
require_relative 'uri/ldap'
require_relative 'uri/ldaps'
require_relative 'uri/mailto'
require_relative 'uri/ws'
require_relative 'uri/wss'
                                                                                                                                                                                                                                                                                                                                                                                                                                                                     ruby/securerandom.rb                                                                                0000644                 00000004162 15040313425 0010541 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                                                                                                                                                                                ruby/drb/ssl.rb                                                                                     0000644                 00000027022 15040313424 0007421 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                               ustar 00                                                                                                                                                                                                                                                       
      @pid = 0 unless defined?(@pid)
      pid = $$
      unless @pid == pid
        now = Process.clock_gettime(Process::CLOCK_REALTIME, :nanosecond)
        OpenSSL::Random.random_add([now, @pid, pid].join(""), 0.0)
        seed = Random.urandom(16)
        if (seed)
          OpenSSL::Random.random_add(seed, 16)
        end
        @pid = pid
      end
      return OpenSSL::Random.random_bytes(n)
    end

    def gen_random_urandom(n)
      ret = Random.urandom(n)
      unless ret
        raise NotImplementedError, "No random device"
      end
      unless ret.length == n
        raise NotImplementedError, "Unexpected partial read from random device: only #{ret.length} for #{n} bytes"
      end
      ret
    end

    begin
      # Check if Random.urandom is available
      Random.urandom(1)
      alias gen_random gen_random_urandom
    rescue RuntimeError
      begin
        require 'openssl'
      rescue NoMethodError
        raise NotImplementedError, "No random device"
      else
        alias gen_random gen_random_openssl
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          ruby/abbrev.rb                                                                                      0000644                 00000006711 15040313425 0007315 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
=begin
= Info
  'OpenSSL for Ruby 2' project
  Copyright (C) 2002  Michal Rokos <m.rokos@sh.cvut.cz>
  All rights reserved.

= Licence
  This program is licensed under the same licence as Ruby.
  (See the file 'LICENCE'.)
=end

require 'openssl.so'

require_relative 'openssl/bn'
require_relative 'openssl/pkey'
require_relative 'openssl/cipher'
require_relative 'openssl/digest'
require_relative 'openssl/hmac'
require_relative 'openssl/x509'
require_relative 'openssl/ssl'
require_relative 'openssl/pkcs5'
require_relative 'openssl/version'

module OpenSSL
  # call-seq:
  #   OpenSSL.secure_compare(string, string) -> boolean
  #
  # Constant time memory comparison. Inputs are hashed using SHA-256 to mask
  # the length of the secret. Returns +true+ if the strings are identical,
  # +false+ otherwise.
  def self.secure_compare(a, b)
    hashed_a = OpenSSL::Digest.digest('SHA256', a)
    hashed_b = OpenSSL::Digest.digest('SHA256', b)
    OpenSSL.fixed_length_secure_compare(hashed_a, hashed_b) && a == b
  end
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    ruby/abbrev.rb                                                                                      0000644                 00000006711 15040313425 0007315 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#--
# Copyright (c) 2001,2003 Akinori MUSHA <knu@iDaemons.org>
#
# All rights reserved.  You can redistribute and/or modify it under
# the same terms as Ruby.
#
# $Idaemons: /home/cvs/rb/abbrev.rb,v 1.2 2001/05/30 09:37:45 knu Exp $
# $RoughId: abbrev.rb,v 1.4 2003/10/14 19:45:42 knu Exp $
# $Id$
#++

##
# Calculates the set of unambiguous abbreviations for a given set of strings.
#
#   require 'abbrev'
#   require 'pp'
#
#   pp Abbrev.abbrev(['ruby'])
#   #=>  {"ruby"=>"ruby", "rub"=>"ruby", "ru"=>"ruby", "r"=>"ruby"}
#
#   pp Abbrev.abbrev(%w{ ruby rules })
#
# _Generates:_
#   { "ruby"  =>  "ruby",
#     "rub"   =>  "ruby",
#     "rules" =>  "rules",
#     "rule"  =>  "rules",
#     "rul"   =>  "rules" }
#
# It also provides an array core extension, Array#abbrev.
#
#   pp %w{ summer winter }.abbrev
#
# _Generates:_
#   { "summer"  => "summer",
#     "summe"   => "summer",
#     "summ"    => "summer",
#     "sum"     => "summer",
#     "su"      => "summer",
#     "s"       => "summer",
#     "winter"  => "winter",
#     "winte"   => "winter",
#     "wint"    => "winter",
#     "win"     => "winter",
#     "wi"      => "winter",
#     "w"       => "winter" }

module Abbrev

  # Given a set of strings, calculate the set of unambiguous abbreviations for
  # those strings, and return a hash where the keys are all the possible
  # abbreviations and the values are the full strings.
  #
  # Thus, given +words+ is "car" and "cone",                                                        ruby/shellwords.rb                                                                                  0000644                 00000016133 15040313425 0010241 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       "}
  #
  #   Abbrev.abbrev(%w{car box cone}, 'ca')
  #   #=> {"car"=>"car", "ca"=>"car"}
  def abbrev(words, pattern = nil)
    table = {}
    seen = Hash.new(0)

    if pattern.is_a?(String)
      pattern = /\A#{Regexp.quote(pattern)}/  # regard as a prefix
    end

    words.each do |word|
      next if word.empty?
      word.size.downto(1) { |len|
        abbrev = word[0...len]

        next if pattern && pattern !~ abbrev

        case seen[abbrev] += 1
        when 1
          table[abbrev] = word
        when 2
          table.delete(abbrev)
        else
          break
        end
      }
    end

    words.each do |word|
      next if pattern && pattern !~ word

      table[word] = word
    end

    table
  end

  module_function :abbrev
end

class Array
  # Calculates the set of unambiguous abbreviations for the strings in +self+.
  #
  #   require 'abbrev'
  #   %w{ car cone }.abbrev
  #   #=> {"car"=>"car", "ca"=>"car", "cone"=>"cone", "con"=>"cone", "co"=>"cone"}
  #
  # The optional +pattern+ parameter is a pattern or a string. Only input
  # strings that match the pattern or start with the string are included in the
  # output hash.
  #
  #   %w{ fast boat day }.abbrev(/^.a/)
  #   #=> {"fast"=>"fast", "fas"=>"fast", "fa"=>"fast", "day"=>"day", "da"=>"day"}
  #
  #   Abbrev.abbrev(%w{car box cone}, "ca")
  #   #=> {"car"=>"car", "ca"=>"car"}
  #
  # See also Abbrev.abbrev
  def abbrev(pattern = nil)
    Abbrev::abbrev(self, pattern)
  end
end
                                                       ruby/shellwords.rb                                                                                  0000644                 00000016133 15040313425 0010241 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen-string-literal: true
##
# == Manipulates strings like the UNIX Bourne shell
#
# This module manipulates strings according to the word parsing rules
# of the UNIX Bourne shell.
#
# The shellwords() function was originally a port of shellwords.pl,
# but modified to conform to the Shell & Utilities volume of the IEEE
# Std 1003.1-2008, 2016 Edition [1].
#
# === Usage
#
# You can use Shellwords to parse a string into a Bourne shell friendly Array.
#
#   require 'shellwords'
#
#   argv = Shellwords.split('three blind "mice"')
#   argv #=> ["three", "blind", "mice"]
#
# Once you've required Shellwords, you can use the #split alias
# String#shellsplit.
#
#   argv = "see how they run".shellsplit
#   argv #=> ["see", "how", "they", "run"]
#
# They treat quotes as special characters, so an unmatched quote will
# cause an ArgumentError.
#
#   argv = "they all ran after the farmer's wife".shellsplit
#        #=> ArgumentError: Unmatched quote: ...
#
# Shellwords also provides methods that do the opposite.
# Shellwords.escape, or its alias, String#shellescape, escapes
# shell metacharacters in a string for use in a command line.
#
#   filename = "special's.txt"
#
#   system("cat -- #{filename.shellescape}")
#   # runs "cat -- special\\'s.txt"
#
# Note the '--'.  Without it, cat(1) will treat the following argument
# as a command line option if it starts with '-'.  It is guaranteed
# that Shellwords.escape converts a string to a form that a Bourne
# shell will parse back to the original string, but it is the
# programmer's responsibility to make sure that passing an arbitrary
# argument to a command does no harm.
#
# Shellwords also comes with a core extension for Array, Array#shelljoin.
#
#   dir = "Funny GIFs"
#   argv = %W[ls -lta -- #{dir}]
#   system(argv.shelljoin + " | less")
#   # runs "ls -lta -- Funny\\ GIFs | less"
#
# You can use this method to build a complete command line out of an
# array of arguments.
#
# === Authors
# * Wakou Aoyama
# * Akinori MUSHA <knu@iDaemons.org>
#
# === Contact
# * Akinori MUSHA <knu@iDaemons.org> (current maintainer)
#
# === Resources
#
# 1: {IEEE Std 1003.1-2008, 2016 Edition, the Shell & Utilities volume}[http://pubs.opengroup.org/onlinepubs/9699919799/utilities/contents.html]

module Shellwords
  # Splits a string into an array of tokens in the same way the UNIX
  # Bourne shell does