 popen2(*cmd, &block)
    if Hash === cmd.last
      opts = cmd.pop.dup
    else
      opts = {}
    end

    in_r, in_w = IO.pipe
    opts[:in] = in_r
    in_w.sync = true

    out_r, out_w = IO.pipe
    opts[:out] = out_w

    popen_run(cmd, opts, [in_r, out_w], [in_w, out_r], &block)
  end
  module_function :popen2

  # :call-seq:
  #   Open3.popen2e([env, ] command_line, options = {}) -> [stdin, stdout_and_stderr, wait_thread]
  #   Open3.popen2e([env, ] exe_path, *args, options = {}) -> [stdin, stdout_and_stderr, wait_thread]
  #   Open3.popen2e([env, ] command_line, options = {}) {|stdin, stdout_and_stderr, wait_thread| ... } -> object
  #   Open3.popen2e([env, ] exe_path, *args, options = {}) {|stdin, stdout_and_stderr, wait_thread| ... } -> object
  #
  # Basically a wrapper for
  # {Process.spawn}[rdoc-ref:Process.spawn]
  # that:
  #
  # - Creates a child process, by calling Process.spawn with the given arguments.
  # - Creates streams +stdin+, +stdout_and_stderr+,
  #   which are the standard input and the merge of the standard output
  #   and standard error streams in the child process.
  # - Creates thread +wait_thread+ that waits for the child process to exit;
  #   the thread has method +pid+, which returns the process ID
  #   of the child process.
  #
  # With no block given, returns the array
  # <tt>[stdin, stdout_and_stderr, wait_thread]</tt>.
  # The caller should close each of the two returned streams.
  #
  #   stdin, stdout_and_stderr, wait_thread = Open3.popen2e('echo')
  #   # => [#<IO:fd 6>, #<IO:fd 7>, #<Process::Waiter:0x00007f7577da4398 run>]
  #   stdin.close
  #   stdout_and_stderr.close
  #   wait_thread.pid   # => 2274600
  #   wait_thread.value # => #<Process::Status: pid 2274600 exit 0>
  #
  # With a block given, calls the block with the three variables
  # (two streams and the wait thread)
  # and returns the block's return value.
  # The caller need not close the streams:
  #
  #   Open3.popen2e('echo') do |stdin, stdout_and_stderr, wait_thread|
  #     p stdin
  #     p stdout_and_stderr
  #     p wait_thread
  #     p wait_thread.pid
  #     p wait_thread.value
  #   end
  #
  # Output:
  #
  #   #<IO:fd 6>
  #   #<IO:fd 7>
  #   #<Process::Waiter:0x00007f75777578c8 sleep>
  #   2274763
  #   #<Process::Status: pid 2274763 exit 0>
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc@Command+Injection].
  #
  # Unlike Process.spawn, this method waits for the child process to exit
  # before returning, so the caller need not do so.
  #
  # If the first argument is a hash, it becomes leading argument +env+
  # in the call to Process.spawn;
  # see {Execution Environment}[rdoc-ref:Process@Execution+Environment].
  #
  # If the last argument is a hash, it becomes trailing argument +options+
  # in the call to Process.spawn;
  # see {Execution Options}[rdoc-ref:Process@Execution+Options].
  #
  # The single required argument is one of the following:
  #
  # - +command_line+ if it is a string,
  #   and if it begins with a shell reserved word or special built-in,
  #   or if it contains one or more metacharacters.
  # - +exe_path+ otherwise.
  #
  # <b>Argument +command_line+</b>
  #
  # \String argument +command_line+ is a command line to be passed to a shell;
  # it must begin with a shell reserved word, begin with a special built-in,
  # or contain meta characters:
  #
  #   Open3.popen2e('if true; then echo "Foo"; fi') {|*args| p args } # Shell reserved word.
  #   Open3.popen2e('echo') {|*args| p args }                         # Built-in.
  #   Open3.popen2e('date > date.tmp') {|*args| p args }              # Contains meta character.
  #
  # Output (similar for each call above):
  #
  #   # => [#<IO:(closed)>, #<IO:(closed)>, #<Process::Waiter:0x00007f7577d8a1f0 dead>]
  #
  # The command line may also contain arguments and options for the command:
  #
  #   Open3.popen2e('echo "Foo"') { |i, o_and_e, t| o_and_e.gets }
  #   "Foo\n"
  #
  # <b>Argument +exe_path+</b>
  #
  # Argument +exe_path+ is one of the following:
  #
  # - The string path to an executable to be called.
  # - A 2-element array containing the path to an executable
  #   and the string to be used as the name of the executing process.
  #
  # Example:
  #
  #   Open3.popen2e('/usr/bin/date') { |i, o_and_e, t| o_and_e.gets }
  #   # => "Thu Sep 28 01:58:45 PM CDT 2023\n"
  #
  # Ruby invokes the executable directly, with no shell and no shell expansion:
  #
  #   Open3.popen2e('doesnt_exist') { |i, o_and_e, t| o_and_e.gets } # Raises Errno::ENOENT
  #
  # If one or more +args+ is given, each is an argument or option
  # to be passed to the executable:
  #
  #   Open3.popen2e('echo', 'C #') { |i, o_and_e, t| o_and_e.gets }
  #   # => "C #\n"
  #   Open3.popen2e('echo', 'hello', 'world') { |i, o_and_e, t| o_and_e.gets }
  #   # => "hello world\n"
  #
  # Related:
  #
  # - Open3.popen2: Makes the standard input and standard output streams
  #   of the child process available as separate streams,
  #   with no access to the standard error stream.
  # - Open3.popen3: Makes the standard input, standard output,
  #   and standard error streams
  #   of the child process available as separate streams.
  #
  def popen2e(*cmd, &block)
    if Hash === cmd.last
      opts = cmd.pop.dup
    else
      opts = {}
    end

    in_r, in_w = IO.pipe
    opts[:in] = in_r
    in_w.sync = true

    out_r, out_w = IO.pipe
    opts[[:out, :err]] = out_w

    popen_run(cmd, opts, [in_r, out_w], [in_w, out_r], &block)
  ensure
    if block
      in_r.close
      in_w.close
      out_r.close
      out_w.close
    end
  end
  module_function :popen2e

  def popen_run(cmd, opts, child_io, parent_io) # :nodoc:
    pid = spawn(*cmd, opts)
    wait_thr = Process.detach(pid)
    child_io.each(&:close)
    result = [*parent_io, wait_thr]
    if defined? yield
      begin
        return yield(*result)
      ensure
        parent_io.each(&:close)
        wait_thr.join
      end
    end
    result
  end
  module_function :popen_run
  class << self
    private :popen_run
  end

  # :call-seq:
  #   Open3.capture3([env, ] command_line, options = {}) -> [stdout_s, stderr_s, status]
  #   Open3.capture3([env, ] exe_path, *args, options = {}) -> [stdout_s, stderr_s, status]
  #
  # Basically a wrapper for Open3.popen3 that:
  #
  # - Creates a child process, by calling Open3.popen3 with the given arguments
  #   (except for certain entries in hash +options+; see below).
  # - Returns as strings +stdout_s+ and +stderr_s+ the standard output
  #   and standard error of the child process.
  # - Returns as +status+ a <tt>Process::Status</tt> object
  #   that represents the exit status of the child process.
  #
  # Returns the array <tt>[stdout_s, stderr_s, status]</tt>:
  #
  #   stdout_s, stderr_s, status = Open3.capture3('echo "Foo"')
  #   # => ["Foo\n", "", #<Process::Status: pid 2281954 exit 0>]
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc@Command+Injection].
  #
  # Unlike Process.spawn, this method waits for the child process to exit
  # before returning, so the caller need not do so.
  #
  # If the first argument is a hash, it becomes leading argument +env+
  # in the call to Open3.popen3;
  # see {Execution Environment}[rdoc-ref:Process@Execution+Environment].
  #
  # If the last argument is a hash, it becomes trailing argument +options+
  # in the call to Open3.popen3;
  # see {Execution Options}[rdoc-ref:Process@Execution+Options].
  #
  # The hash +options+ is given;
  # two options have local effect in method Open3.capture3:
  #
  # - If entry <tt>options[:stdin_data]</tt> exists, the entry is removed
  #   and its string value is sent to the command's standard input:
  #
  #     Open3.capture3('tee', stdin_data: 'Foo')
  #     # => ["Foo", "", #<Process::Status: pid 2319575 exit 0>]
  #
  # - If entry <tt>options[:binmode]</tt> exists, the entry is removed and
  #   the internal streams are set to binary mode.
  #
  # The single required argument is one of the following:
  #
  # - +command_line+ if it is a string,
  #   and if it begins with a shell reserved word or special built-in,
  #   or if it contains one or more metacharacters.
  # - +exe_path+ otherwise.
  #
  # <b>Argument +command_line+</b>
  #
  # \String argument +command_line+ is a command line to be passed to a shell;
  # it must begin with a shell reserved word, begin with a special built-in,
  # or contain meta characters:
  #
  #   Open3.capture3('if true; then echo "Foo"; fi') # Shell reserved word.
  #   # => ["Foo\n", "", #<Process::Status: pid 2282025 exit 0>]
  #   Open3.capture3('echo')                         # Built-in.
  #   # => ["\n", "", #<Process::Status: pid 2282092 exit 0>]
  #   Open3.capture3('date > date.tmp')              # Contains meta character.
  #   # => ["", "", #<Process::Status: pid 2282110 exit 0>]
  #
  # The command line may also contain arguments and options for the command:
  #
  #   Open3.capture3('echo "Foo"')
  #   # => ["Foo\n", "", #<Process::Status: pid 2282092 exit 0>]
  #
  # <b>Argument +exe_path+</b>
  #
  # Argument +exe_path+ is one of the following:
  #
  # - The string path to an executable to be called.
  # - A 2-element array containing the path to an executable
  #   and the string to be used as the name of the executing process.
  #
  # Example:
  #
  #   Open3.capture3('/usr/bin/date')
  #   # => ["Thu Sep 28 05:03:51 PM CDT 2023\n", "", #<Process::Status: pid 2282300 exit 0>]
  #
  # Ruby invokes the executable directly, with no shell and no shell expansion:
  #
  #   Open3.capture3('doesnt_exist') # Raises Errno::ENOENT
  #
  # If one or more +args+ is given, each is an argument or option
  # to be passed to the executable:
  #
  #   Open3.capture3('echo', 'C #')
  #   # => ["C #\n", "", #<Process::Status: pid 2282368 exit 0>]
  #   Open3.capture3('echo', 'hello', 'world')
  #   # => ["hello world\n", "", #<Process::Status: pid 2282372 exit 0>]
  #
  def capture3(*cmd)
    if Hash === cmd.last
      opts = cmd.pop.dup
    else
      opts = {}
    end

    stdin_data = opts.delete(:stdin_data) || ''
    binmode = opts.delete(:binmode)

    popen3(*cmd, opts) {|i, o, e, t|
      if binmode
        i.binmode
        o.binmode
        e.binmode
      end
      out_reader = Thread.new { o.read }
      err_reader = Thread.new { e.read }
      begin
        if stdin_data.respond_to? :readpartial
          IO.copy_stream(stdin_data, i)
        else
          i.write stdin_data
        end
      rescue Errno::EPIPE
      end
      i.close
      [out_reader.value, err_reader.value, t.value]
    }
  end
  module_function :capture3

  # :call-seq:
  #   Open3.capture2([env, ] command_line, options = {}) -> [stdout_s, status]
  #   Open3.capture2([env, ] exe_path, *args, options = {}) -> [stdout_s, status]
  #
  # Basically a wrapper for Open3.popen3 that:
  #
  # - Creates a child process, by calling Open3.popen3 with the given arguments
  #   (except for certain entries in hash +options+; see below).
  # - Returns as string +stdout_s+ the standard output of the child process.
  # - Returns as +status+ a <tt>Process::Status</tt> object
  #   that represents the exit status of the child process.
  #
  # Returns the array <tt>[stdout_s, status]</tt>:
  #
  #   stdout_s, status = Open3.capture2('echo "Foo"')
  #   # => ["Foo\n", #<Process::Status: pid 2326047 exit 0>]
  #
  # Like Process.spawn, this method has potential security vulnerabilities
  # if called with untrusted input;
  # see {Command Injection}[rdoc-ref:command_injection.rdoc