 can get a new (and populated) store by calling thus:
#
#   example_store do |store|
#     # Example code using store goes here.
#   end
#
# All we really need to know about +example_store+
# is that it yields a fresh store with a known population of entries;
# its implementation:
#
#   require 'pstore'
#   require 'tempfile'
#   # Yield a pristine store for use in examples.
#   def example_store
#     # Create the store in a temporary file.
#     Tempfile.create do |file|
#       store = PStore.new(file)
#       # Populate the store.
#       store.transaction do
#         store[:foo] = 0
#         store[:bar] = 1
#         store[:baz] = 2
#       end
#       yield store
#     end
#   end
#
# == The Store
#
# The contents of the store are maintained in a file whose path is specified
# when the store is created (see PStore.new).
# The objects are stored and retrieved using
# module Marshal, which means that certain objects cannot be added to the store;
# see {Marshal::dump}[rdoc-ref:Marshal.dump].
#
# == Entries
#
# A store may have any number of entries.
# Each entry has a key and a value, just as in a hash:
#
# - Key: as in a hash, the key can be (almost) any object;
#   see {Hash Keys}[rdoc-ref:Hash@Hash+Keys].
#   You may find it convenient to keep it simple by using only
#   symbols or strings as keys.
# - Value: the value may be any object that can be marshalled by \Marshal
#   (see {Marshal::dump}[rdoc-ref:Marshal.dump])
#   and in fact may be a collection
#   (e.g., an array, a hash, a set, a range, etc).
#   That collection may in turn contain nested objects,
#   including collections, to any depth;
#   those objects must also be \Marshal-able.
#   See {Hierarchical Values}[rdoc-ref:PStore@Hierarchical+Values].
#
# == Transactions
#
# === The Transaction Block
#
# The block given with a call to method #transaction#
# contains a _transaction_,
# which consists of calls to \PStore methods that
# read from or write to the store
# (that is, all \PStore methods except #transaction itself,
# #path, and Pstore.new):
#
#   example_store do |store|
#     store.transaction do
#       store.keys # => [:foo, :bar, :baz]
#       store[:bat] = 3
#       store.keys # => [:foo, :bar, :baz, :bat]
#     end
#   end
#
# Execution of the transaction is deferred until the block exits,
# and is executed _atomically_ (all-or-nothing):
# either all transaction calls are executed, or none are.
# This maintains the integrity of the store.
#
# Other code in the block (including even calls to #path and PStore.new)
# is executed immediately, not deferred.
#
# The transaction block:
#
# - May not contain a nested call to #transaction.
# - Is the only context where methods that read from or write to
#   the store are allowed.
#
# As seen above, changes in a transaction are made automatically
# when the block exits.
# The block may be exited early by calling method #commit or #abort.
#
# - Method #commit triggers the update to the store and exits the block:
#
#     example_store do |store|
#       store.transaction do
#         store.keys # => [:foo, :bar, :baz]
#         store[:bat] = 3
#         store.commit
#         fail 'Cannot get here'
#       end
#       store.transaction do
#         # Update was completed.
#         store.keys # => [:foo, :bar, :baz, :bat]
#       end
#     end
#
# - Method #abort discards the update to the store and exits the block:
#
#     example_store do |store|
#       store.transaction do
#         store.keys # => [:foo, :bar, :baz]
#         store[:bat] = 3
#         store.abort
#         fail 'Cannot get here'
#       end
#       store.transaction do
#         # Update was not completed.
#         store.keys # => [:foo, :bar, :baz]
#       end
#     end
#
# === Read-Only Transactions
#
# By default, a transaction allows both reading from and writing to
# the store:
#
#   store.transaction do
#     # Read-write transaction.
#     # Any code except a call to #transaction is allowed here.
#   end
#
# If argument +read_only+ is passed as +true+,
# only reading is allowed:
#
#   store.transaction(true) do
#     # Read-only transaction:
#     # Calls to #transaction, #[]=, and #delete are not allowed here.
#   end
#
# == Hierarchical Values
#
# The value for an entry may be a simple object (as seen above).
# It may also be a hierarchy of objects nested to any depth:
#
#   deep_store = PStore.new('deep.store')
#   deep_store.transaction do
#     array_of_hashes = [{}, {}, {}]
#     deep_store[:array_of_hashes] = array_of_hashes
#     deep_store[:array_of_hashes] # => [{}, {}, {}]
#     hash_of_arrays = {foo: [], bar: [], baz: []}
#     deep_store[:hash_of_arrays] = hash_of_arrays
#     deep_store[:hash_of_arrays]  # => {:foo=>[], :bar=>[], :baz=>[]}
#     deep_store[:hash_of_arrays][:foo].push(:bat)
#     deep_store[:hash_of_arrays]  # => {:foo=>[:bat], :bar=>[], :baz=>[]}
#   end
#
# And recall that you can use
# {dig methods}[rdoc-ref:dig_methods.rdoc]
# in a returned hierarchy of objects.
#
# == Working with the Store
#
# === Creating a Store
#
# Use method PStore.new to create a store.
# The new store creates or opens its containing file:
#
#   store = PStore.new('t.store')
#
# === Modifying the Store
#
# Use method #[]= to update or create an entry:
#
#   example_store do |store|
#     store.transaction do
#       store[:foo] = 1 # Update.
#       store[:bam] = 1 # Create.
#     end
#   end
#
# Use method #delete to remove an entry:
#
#   example_store do |store|
#     store.transaction do
#       store.delete(:foo)
#       store[:foo] # => nil
#     end
#   end
#
# === Retrieving Values
#
# Use method #fetch (allows default) or #[] (defaults to +nil+)
# to retrieve an entry:
#
#   example_store do |store|
#     store.transaction do
#       store[:foo]             # => 0
#       store[:nope]            # => nil
#       store.fetch(:baz)       # => 2
#       store.fetch(:nope, nil) # => nil
#       store.fetch(:nope)      # Raises exception.
#     end
#   end
#
# === Querying the Store
#
# Use method #key? to determine whether a given key exists:
#
#   example_store do |store|
#     store.transaction do
#       store.key?(:foo) # => true
#     end
#   end
#
# Use method #keys to retrieve keys:
#
#   example_store do |store|
#     store.transaction do
#       store.keys # => [:foo, :bar, :baz]
#     end
#   end
#
# Use method #path to retrieve the path to the store's underlying file;
# this method may be called from outside a transaction block:
#
#   store = PStore.new('t.store')
#   store.path # => "t.store"
#
# == Transaction Safety
#
# For transaction safety, see:
#
# - Optional argument +thread_safe+ at method PStore.new.
# - Attribute #ultra_safe.
#
# Needless to say, if you're storing valuable data with \PStore, then you should
# backup the \PStore file from time to time.
#
# == An Example Store
#
#  require "pstore"
#
#  # A mock wiki object.
#  class WikiPage
#
#    attr_reader :page_name
#
#    def initialize(page_name, author, contents)
#      @page_name = page_name
#      @revisions = Array.new
#      add_revision(author, contents)
#    end
#
#    def add_revision(author, contents)
#      @revisions << {created: Time.now,
#                     author: author,
#                     contents: contents}
#    end
#
#    def wiki_page_references
#      [@page_name] + @revisions.last[:contents].scan(/\b(?:[A-Z]+[a-z]+){2,}/)
#    end
#
#  end
#
#  # Create a new wiki page.
#  home_page = WikiPage.new("HomePage", "James Edward Gray II",
#                           "A page about the JoysOfDocumentation..." )
#
#  wiki = PStore.new("wiki_pages.pstore")
#  # Update page data and the index together, or not at all.
#  wiki.transaction do
#    # Store page.
#    wiki[home_page.page_name] = home_page
#    # Create page index.
#    wiki[:wiki_index] ||= Array.new
#    # Update wiki index.
#    wiki[:wiki_index].push(*home_page.wiki_page_references)
#  end
#
#  # Read wiki data, setting argument read_only to true.
#  wiki.transaction(true) do
#    wiki.keys.each do |key|
#      puts key
#      puts wiki[key]
#    end
#  end
#
class PStore
  VERSION = "0.1.4"

  RDWR_ACCESS = {mode: IO::RDWR | IO::CREAT | IO::BINARY, encoding: Encoding::ASCII_8BIT}.freeze
  RD_ACCESS = {mode: IO::RDONLY | IO::BINARY, encoding: Encoding::ASCII_8BIT}.freeze
  WR_ACCESS = {mode: IO::WRONLY | IO::CREAT | IO::TRUNC | IO::BINARY, encoding: Encoding::ASCII_8BIT}.freeze

  # The error type thrown by all PStore methods.
  class Error < StandardError
  end

  # Whether \PStore should do its best to prevent file corruptions,
  # even when an unlikely error (such as memory-error or filesystem error) occurs:
  #
  # - +true+: changes are posted by creating a temporary file,
  #   writing the updated data to it, then renaming the file to the given #path.
  #   File integrity is maintained.
  #   Note: has effect only if the filesystem has atomic file rename
  #   (as do POSIX platforms Linux, MacOS, FreeBSD and others).
  #
  # - +false+ (the default): changes are posted by rewinding the open file
  #   and writing the updated data.
  #   File integrity is maintained if the filesystem raises
  #   no unexpected I/O error;
  #   if such an error occurs during a write to the store,
  #   the file may become corrupted.
  #
  attr_accessor :ultra_safe

  # Returns a new \PStore object.
  #
  # Argument +file+ is the path to the file in which objects are to be stored;
  # if the file exists, it should be one that was written by \PStore.
  #
  #   path = 't.store'
  #   store = PStore.new(path)
  #
  # A \PStore object is
  # {reentrant}[https://en.wikipedia.org/wiki/Reentrancy_(computing)].
  # If argument +thread_safe+ is given as +true+,
  # the object is also thread-safe (at the cost of a small performance penalty):
  #
  #   store = PStore.new(path, true)
  #
  def initialize(file, thread_safe = false)
    dir = File::dirname(file)
    unless File::directory? dir
      raise PStore::Error, format("directory %s does not exist", dir)
    end
    if File::exist? file and not File::readable? file
      raise PStore::Error, format("file %s not readable", file)
    end
    @filename = file
    @abort = false
    @ultra_safe = false
    @thread_safe = thread_safe
    @lock = Thread::Mutex.new
  end

  # Raises PStore::Error if the calling code is not in a PStore#transaction.
  def in_transaction
    raise PStore::Error, "not in transaction" unless @lock.locked?
  end
  #
  # Raises PStore::Error if the calling code is not in a PStore#transaction or
  # if the code is in a read-only PStore#transaction.
  #
  def in_transaction_wr
    in_transaction
    raise PStore::Error, "in read-only transaction" if @rdonly
  end
  private :in_transaction, :in_transaction_wr

  # Returns the value for the given +key+ if the key exists.
  # +nil+ otherwise;
  # if not +nil+, the returned value is an object or a hierarchy of objects:
  #
  #   example_store do |store|
  #     store.transaction do
  #       store[:foo]  # => 0
  #       store[:nope] # => nil
  #     end
  #   end
  #
  # Returns +nil+ if there is no such key.
  #
  # See also {Hierarchical Values}[rdoc-ref:PStore@Hierarchical+Values].
  #
  # Raises an exception if called outside a transaction block.
  def [](key)
    in_transaction
    @table[key]
  end

  # Like #[], except that it accepts a default value for the store.
  # If the +key+ does not exist:
  #
  # - Raises an exception if +default+ is +PStore::Error+.
  # - Returns the value of +default+ otherwise:
  #
  #     example_store do |store|
  #       store.transaction do
  #         store.fetch(:nope, nil) # => nil
  #         store.fetch(:nope)      # Raises an exception.
  #       end
  #     end
  #
  # Raises an exception if called outside a transaction block.
  def fetch(key, default=PStore::Error)
    in_transaction
    unless @table.key? key
      if default == PStore::Error
        raise PStore::Error, format("undefined key '%s'", key)
      else
        return default
      end
    end
    @table[key]
  end

  # Creates or replaces the value for the given +key+:
  #
  #   example_store do |store|
  #     temp.transaction do
  #       temp[:bat] = 3
  #     end
  #   end
  #
  # See also {Hierarchical Values}[rdoc-ref:PStore@Hierarchical+Values].
  #
  # Raises an exception if called outside a transaction block.
  def []=(key, value)
    in_transaction_wr
    @table[key] = value
  end

  # Removes and returns the value at +key+ if it exists:
  #
  #   example_store do |store|
  #     store.transaction do
  #       store[:bat] = 3
  #       store.delete(:bat)
  #     end
  #   end
  #
  # Returns +nil+ if there is no such key.
  #
  # Raises an exception if called outside a transaction block.
  def delete(key)
    in_transaction_wr
    @table.delete key
  end

  # Returns an array of the existing keys:
  #
  #   example_store do |store|
  #     store.transaction do
  #       store.keys # => [:foo, :bar, :baz]
  #     end
  #   end
  #
  # Raises an exception if called outside a transaction block.
  def keys
    in_transaction
    @table.keys
  end
  alias roots keys

  # Returns +true+ if +key+ exists, +false+ otherwise:
  #
  #   example_store do |store|
  #     store.transaction do
  #       store.key?(:foo) # => true
  #     end
  #   end
  #
  # Raises an exception if called outside a transaction block.
  def key?(key)
    in_transaction
    @table.key? key
  end
  alias root? key?

  # Returns the string file path used to create the store:
  #
  #   store.path # => "flat.store"
  #
  def path
    @filename
  end

  # Exits the current transaction block, committing any changes
  # specified in the
  # {transaction block}[rdoc-ref:PStore@The+Transaction+Block].
  #
  # Raises an exception if called outside a transaction block.
  def commit
    in_transaction
    @abort = false
    throw :pstore_abort_transaction
  end

  # Exits the current transaction block, discarding any changes
  # specified in the
  # {transaction block}[rdoc-ref:PStore@The+Transaction+Block].
  #
  # Raises an exception if called outside a transaction block.
  def abort
    in_transaction
    @abort = true
    throw :pstore_abort_transaction
  end

  # Opens a transaction block for the store.
  # See {Transactions}[rdoc-ref:PStore@Transactions].
  #
  # With argument +read_only+ as +false+, the block may both read from
  # and write to the store.
  #
  # With argument +read_only+ as +true+, the block may not include calls
  # to #transaction, #[]=, or #delete.
  #
  # Raises an exception if called within a transaction block.
  def transaction(read_only = false)  # :yields:  pstore
    value = nil
    if !@thread_safe
      raise PStore::Error, "nested transaction" unless @lock.try_lock
    else
      begin
        @lock.lock
      rescue ThreadError
        raise PStore::Error, "nested transaction"
      end
    end
    begin
      @rdonly = read_only
      @abort = false
      file = open_and_lock_file(@filename, read_only)
      if file
        begin
          @table, checksum, original_data_size = load_data(file, read_only)

          catch(:pstore_abort_transaction) do
            value = yield(self)
          end

          if !@abort && !read_only
            save_data(checksum, original_data_size, file)
          end
        ensure
          file.close
        end
      else
        # This can only occur if read_only == true.
        @table = {}
        catch(:pstore_abort_transaction) do
          value = yield(self)
        end
      end
    ensure
      @lock.unlock
    end
    value
  end

  private
  # Constant for relieving Ruby's garbage collector.
  CHECKSUM_ALGO = %w[SHA512 SHA384 SHA256 SHA1 RMD160 MD5].each do |algo|
    begin
      break Digest(algo)
    rescue LoadError
    end
  end
  EMPTY_STRING = ""
  EMPTY_MARSHAL_DATA = Marshal.dump({})
  EMPTY_MARSHAL_CHECKSUM = CHECKSUM_ALGO.digest(EMPTY_MARSHAL_DATA)

  #
  # Open the specified filename (either in read-only mode or in
  # read-write mode) and lock it for reading or writing.
  #
  # The opened File object will be returned. If _read_only_ is true,
  # and the file does not exist, then nil will be returned.
  #
  # All exceptions are propagated.
  #
  def open_and_lock_file(filename, read_only)
    if read_only
      begin
        file = File.new(filename, **RD_ACCESS)
        begin
          file.flock(File::LOCK_SH)
          return file
        rescue
          file.close
          raise
        end
      rescue Errno::ENOENT
        return nil
      end
    else
      file = File.new(filename, **RDWR_ACCESS)
      file.flock(File::LOCK_EX)
      return file
    end
  end

  # Load the given PStore file.
  # If +read_only+ is true, the unmarshalled Hash will be returned.
  # If +read_only+ is false, a 3-tuple will be returned: the unmarshalled
  # Hash, a checksum of the data, and the size of the data.
  def load_data(file, read_only)
    if read_only
      begin
        table = load(file)
        raise Error, "PStore file seems to be corrupted." unless table.is_a?(Hash)
      rescue EOFError
        # This seems to be a newly-created file.
        table = {}
      end
      table
    else
      data = file.read
      if data.empty?
        # This seems to be a newly-created file.
        table = {}
        checksum = empty_marshal_checksum
        size = empty_marshal_data.bytesize
      else
        table = load(data)
        checksum = CHECKSUM_ALGO.digest(data)
        size = data.bytesize
        raise Error, "PStore file seems to be corrupted." unless table.is_a?(Hash)
      end
      data.replace(EMPTY_STRING)
      [table, checksum, size]
    end
  end

  def on_windows?
    is_windows = RUBY_PLATFORM =~ /mswin|mingw|bccwin|wince/
    self.class.__send__(:define_method, :on_windows?) do
      is_windows
    end
    is_windows
  end

  def save_data(original_checksum, original_file_size, file)
    new_data = dump(@table)

    if new_data.bytesize != original_file_size || CHECKSUM_ALGO.digest(new_data) != original_checksum
      if @ultra_safe && !on_windows?
        # Windows doesn't support atomic file renames.
        save_data_with_atomic_file_rename_strategy(new_data, file)
      else
        save_data_with_fast_strategy(new_data, file)
      end
    end

    new_data.replace(EMPTY_STRING)
  end

  def save_data_with_atomic_file_rename_strategy(data, file)
    temp_filename = "#{@filename}.tmp.#{Process.pid}.#{rand 1000000}"
    temp_file = File.new(temp_filename, **WR_ACCESS)
    begin
      temp_file.flock(File::LOCK_EX)
      temp_file.write(data)
      temp_file.flush
      File.rename(temp_filename, @filename)
    rescue
      File.unlink(temp_file) rescue nil
      raise
    ensure
      temp_file.close
    end
  end

  def save_data_with_fast_strategy(data, file)
    file.rewind
    file.write(data)
    file.truncate(data.bytesize)
  end


  # This method is just a wrapped around Marshal.dump
  # to allow subclass overriding used in YAML::Store.
  def dump(table)  # :nodoc:
    Marshal::dump(table)
  end

  # This method is just a wrapped around Marshal.load.
  # to allow subclass overriding used in YAML::Store.
  def load(content)  # :nodoc:
    Marshal::load(content)
  end

  def empty_marshal_data
    EMPTY_MARSHAL_DATA
  end
  def empty_marshal_checksum
    EMPTY_MARSHAL_CHECKSUM
  end
end
                                                                                                                                           ruby/net/protocol.rb                                                                                0000644                 00000026772 15040313424 0010513 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#
# = net/protocol.rb
#
#--
# Copyright (c) 1999-2004 Yukihiro Matsumoto
# Copyright (c) 1999-2004 Minero Aoki
#
# written and maintained by Minero Aoki <aamine@loveruby.net>
#
# This program is free software. You can re-distribute and/or
# modify this program under the same terms as Ruby itself,
# Ruby Distribute License or GNU General Public License.
#
# $Id$
#++
#
# WARNING: This file is going to remove.
# Do not rely on the implementation written in this file.
#

require 'socket'
require 'timeout'
require 'io/wait'

module Net # :nodoc:

  class Protocol   #:nodoc: internal use only
    VERSION = "0.2.2"

    private
    def Protocol.protocol_param(name, val)
      module_eval(<<-End, __FILE__, __LINE__ + 1)
        def #{name}
          #{val}
        end
      End
    end

    def ssl_socket_connect(s, timeout)
      if timeout
        while true
          raise Net::OpenTimeout if timeout <= 0
          start = Process.clock_gettime Process::CLOCK_MONOTONIC
          # to_io is required because SSLSocket doesn't have wait_readable yet
          case s.connect_nonblock(exception: false)
          when :wait_readable; s.to_io.wait_readable(timeout)
          when :wait_writable; s.to_io.wait_writable(timeout)
          else; break
          end
          timeout -= Process.clock_gettime(Process::CLOCK_MONOTONIC) - start
        end
      else
        s.connect
      end
    end
  end


  class ProtocolError          < StandardError; end
  class ProtoSyntaxError       < ProtocolError; end
  class ProtoFatalError        < ProtocolError; end
  class ProtoUnknownError      < ProtocolError; end
  class ProtoServerError       < ProtocolError; end
  class ProtoAuthError         < ProtocolError; end
  class ProtoCommandError      < ProtocolError; end
  class ProtoRetriableError    < ProtocolError; end
  ProtocRetryError = ProtoRetriableError

  ##
  # OpenTimeout, a subclass of Timeout::Error, is raised if a connection cannot
  # be created within the open_timeout.

  class OpenTimeout            < Timeout::Error; end

  ##
  # ReadTimeout, a subclass of Timeout::Error, is raised if a chunk of the
  # response cannot be read within the read_timeout.

  class ReadTimeout < Timeout::Error
    def initialize(io = nil)
      @io = io
    end
    attr_reader :io

    def message
      msg = super
      if @io
        msg = "#{msg} with #{@io.inspect}"
      end
      msg
    end
  end

  ##
  # WriteTimeout, a subclass of Timeout::Error, is raised if a chunk of the
  # response cannot be written within the write_timeout.  Not raised on Windows.

  class WriteTimeout < Timeout::Error
    def initialize(io = nil)
      @io = io
    end
    attr_reader :io

    def message
      msg = super
      if @io
        msg = "#{msg} with #{@io.inspect}"
      end
      msg
    end
  end


  class BufferedIO   #:nodoc: internal use only
    def initialize(io, read_timeout: 60, write_timeout: 60, continue_timeout: nil, debug_output: nil)
      @io = io
      @read_timeout = read_timeout
      @write_timeout = write_timeout
      @continue_timeout = continue_timeout
      @debug_output = debug_output
      @rbuf = ''.b
      @rbuf_empty = true
      @rbuf_offset = 0
    end

    attr_reader :io
    attr_accessor :read_timeout
    attr_accessor :write_timeout
    attr_accessor :continue_timeout
    attr_accessor :debug_output

    def inspect
      "#<#{self.class} io=#{@io}>"
    end

    def eof?
      @io.eof?
    end

    def closed?
      @io.closed?
    end

    def close
      @io.close
    end

    #
    # Read
    #

    public

    def read(len, dest = ''.b, ignore_eof = false)
      LOG "reading #{len} bytes..."
      read_bytes = 0
      begin
        while read_bytes + rbuf_size < len
          if s = rbuf_consume_all
            read_bytes += s.bytesize
            dest << s
          end
          rbuf_fill
        end
        s = rbuf_consume(len - read_bytes)
        read_bytes += s.bytesize
        dest << s
      rescue EOFError
        raise unless ignore_eof
      end
      LOG "read #{read_bytes} bytes"
      dest
    end

    def read_all(dest = ''.b)
      LOG 'reading all...'
      read_bytes = 0
      begin
        while true
          if s = rbuf_consume_all
            read_bytes += s.bytesize
            dest << s
          end
          rbuf_fill
        end
      rescue EOFError
        ;
      end
      LOG "read #{read_bytes} bytes"
      dest
    end

    def readuntil(terminator, ignore_eof = false)
      offset = @rbuf_offset
      begin
        until idx = @rbuf.index(terminator, offset)
          offset = @rbuf.bytesize
          rbuf_fill
        end
        return rbuf_consume(idx + terminator.bytesize - @rbuf_offset)
      rescue EOFError
        raise unless ignore_eof
        return rbuf_consume
      end
    end

    def readline
      readuntil("\n").chop
    end

    private

    BUFSIZE = 1024 * 16

    def rbuf_fill
      tmp = @rbuf_empty ? @rbuf : nil
      case rv = @io.read_nonblock(BUFSIZE, tmp, exception: false)
      when String
        @rbuf_empty = false
        if rv.equal?(tmp)
          @rbuf_offset = 0
        else
          @rbuf << rv
          rv.clear
        end
        return
      when :wait_readable
        (io = @io.to_io).wait_readable(@read_timeout) or raise Net::ReadTimeout.new(io)
        # continue looping
      when :wait_writable
        # OpenSSL::Buffering#read_nonblock may fail with IO::WaitWritable.
        # http://www.openssl.org/support/faq.html#PROG10
        (io = @io.to_io).wait_writable(@read_timeout) or raise Net::ReadTimeout.new(io)
        # continue looping
      when nil
        raise EOFError, 'end of file reached'
      end while true
    end

    def rbuf_flush
      if @rbuf_empty
        @rbuf.clear
        @rbuf_offset = 0
      end
      nil
    end

    def rbuf_size
      @rbuf.bytesize - @rbuf_offset
    end

    def rbuf_consume_all
      rbuf_consume if rbuf_size > 0
    end

    def rbuf_consume(len = nil)
      if @rbuf_offset == 0 && (len.nil? || len == @rbuf.bytesize)
        s = @rbuf
        @rbuf = ''.b
        @rbuf_offset = 0
        @rbuf_empty = true
      elsif len.nil?
        s = @rbuf.byteslice(@rbuf_offset..-1)
        @rbuf = ''.b
        @rbuf_offset = 0
        @rbuf_empty = true
      else
        s = @rbuf.byteslice(@rbuf_offset, len)
        @rbuf_offset += len
        @rbuf_empty = @rbuf_offset == @rbuf.bytesize
        rbuf_flush
      end

      @debug_output << %Q[-> #{s.dump}\n] if @debug_output
      s
    end

    #
    # Write
    #

    public

    def write(*strs)
      writing {
        write0(*strs)
      }
    end

    alias << write

    def writeline(str)
      writing {
        write0 str + "\r\n"
      }
    end

    private

    def writing
      @written_bytes = 0
      @debug_output << '<- ' if @debug_output
      yield
      @debug_output << "\n" if @debug_output
      bytes = @written_bytes
      @written_bytes = nil
      bytes
    end

    def write0(*strs)
      @debug_output << strs.map(&:dump).join if @debug_output
      orig_written_bytes = @written_bytes
      strs.each_with_index do |str, i|
        need_retry = true
        case len = @io.write_nonblock(str, exception: false)
        when Integer
          @written_bytes += len
          len -= str.bytesize
          if len == 0
            if strs.size == i+1
              return @written_bytes - orig_written_bytes
            else
              need_retry = false
              # next string
            end
          elsif len < 0
            str = str.byteslice(len, -len)
          else # len > 0
            need_retry = false
            # next string
          end
          # continue looping
        when :wait_writable
          (io = @io.to_io).wait_writable(@write_timeout) or raise Net::WriteTimeout.new(io)
          # continue looping
        end while need_retry
      end
    end

    #
    # Logging
    #

    private

    def LOG_off
      @save_debug_out = @debug_output
      @debug_output = nil
    end

    def LOG_on
      @debug_output = @save_debug_out
    end

    def LOG(msg)
      return unless @debug_output
      @debug_output << msg + "\n"
    end
  end


  class InternetMessageIO < BufferedIO   #:nodoc: internal use only
    def initialize(*, **)
      super
      @wbuf = nil
    end

    #
    # Read
    #

    def each_message_chunk
      LOG 'reading message...'
      LOG_off()
      read_bytes = 0
      while (line = readuntil("\r\n")) != ".\r\n"
        read_bytes += line.size
        yield line.delete_prefix('.')
      end
      LOG_on()
      LOG "read message (#{read_bytes} bytes)"
    end

    # *library private* (cannot handle 'break')
    def each_list_item
      while (str = readuntil("\r\n")) != ".\r\n"
        yield str.chop
      end
    end

    def write_message_0(src)
      prev = @written_bytes
      each_crlf_line(src) do |line|
        write0 dot_stuff(line)
      end
      @written_bytes - prev
    end

    #
    # Write
    #

    def write_message(src)
      LOG "writing message from #{src.class}"
      LOG_off()
      len = writing {
        using_each_crlf_line {
          write_message_0 src
        }
      }
      LOG_on()
      LOG "wrote #{len} bytes"
      len
    end

    def write_message_by_block(&block)
      LOG 'writing message from block'
      LOG_off()
      len = writing {
        using_each_crlf_line {
          begin
            block.call(WriteAdapter.new(self.method(:write_message_0)))
          rescue LocalJumpError
            # allow `break' from writer block
          end
        }
      }
      LOG_on()
      LOG "wrote #{len} bytes"
      len
    end

    private

    def dot_stuff(s)
      s.sub(/\A\./, '..')
    end

    def using_each_crlf_line
      @wbuf = ''.b
      yield
      if not @wbuf.empty?   # unterminated last line
        write0 dot_stuff(@wbuf.chomp) + "\r\n"
      elsif @written_bytes == 0   # empty src
        write0 "\r\n"
      end
      write0 ".\r\n"
      @wbuf = nil
    end

    def each_crlf_line(src)
      buffer_filling(@wbuf, src) do
        while line = @wbuf.slice!(/\A[^\r\n]*(?:\n|\r(?:\n|(?!\z)))/)
          yield line.chomp("\n") + "\r\n"
        end
      end
    end

    def buffer_filling(buf, src)
      case src
      when String    # for speeding up.
        0.step(src.size - 1, 1024) do |i|
          buf << src[i, 1024]
          yield
        end
      when File    # for speeding up.
        while s = src.read(1024)
          buf << s
          yield
        end
      else    # generic reader
        src.each do |str|
          buf << str
          yield if buf.size > 1024
        end
        yield unless buf.empty?
      end
    end
  end


  #
  # The writer adapter class
  #
  class WriteAdapter
    def initialize(writer)
      @writer = writer
    end

    def inspect
      "#<#{self.class} writer=#{@writer.inspect}>"
    end

    def write(str)
      @writer.call(str)
    end

    alias print write

    def <<(str)
      write str
      self
    end

    def puts(str = '')
      write str.chomp("\n") + "\n"
    end

    def printf(*args)
      write sprintf(*args)
    end
  end


  class ReadAdapter   #:nodoc: internal use only
    def initialize(block)
      @block = block
    end

    def inspect
      "#<#{self.class}>"
    end

    def <<(str)
      call_block(str, &@block) if @block
    end

    private

    # This method is needed because @block must be called by yield,
    # not Proc#call.  You can see difference when using `break' in
    # the block.
    def call_block(str)
      yield str
    end
  end


  module NetPrivate   #:nodoc: obsolete
    Socket = ::Net::InternetMessageIO
  end

end   # module Net
      ruby/net/https.rb                                                                                   0000644                 00000001023 15040313424 0007772 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
=begin

= net/https -- SSL/TLS enhancement for Net::HTTP.

  This file has been merged with net/http.  There is no longer any need to
  require 'net/https' to use HTTPS.

  See Net::HTTP for details on how to make HTTPS connections.

== Info
  'OpenSSL for Ruby 2' project
  Copyright (C) 2001 GOTOU Yuuzou <gotoyuzo@notwork.org>
  All rights reserved.

== Licence
  This program is licensed under the same licence as Ruby.
  (See the file 'LICENCE'.)

=end

require_relative 'http'
require 'openssl'
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ruby/net/http.rb                                                                                    0000644                 00000253071 15040313424 0007623 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#
# = net/http.rb
#
# Copyright (c) 1999-2007 Yukihiro Matsumoto
# Copyright (c) 1999-2007 Minero Aoki
# Copyright (c) 2001 GOTOU Yuuzou
#
# Written and maintained by Minero Aoki <aamine@loveruby.net>.
# HTTPS support added by GOTOU Yuuzou <gotoyuzo@notwork.org>.
#
# This file is derived from "http-access.rb".
#
# Documented by Minero Aoki; converted to RDoc by William Webber.
#
# This program is free software. You can re-distribute and/or
# modify this program under the same terms of ruby itself ---
# Ruby Distribution License or GNU General Public License.
#
# See Net::HTTP for an overview and examples.
#

require 'net/protocol'
require 'uri'
require 'resolv'
autoload :OpenSSL, 'openssl'

module Net   #:nodoc:

  # :stopdoc:
  class HTTPBadResponse < StandardError; end
  class HTTPHeaderSyntaxError < StandardError; end
  # :startdoc:

  # \Class \Net::HTTP provides a rich library that implements the client
  # in a client-server model that uses the \HTTP request-response protocol.
  # For information about \HTTP, see:
  #
  # - {Hypertext Transfer Protocol}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol].
  # - {Technical overview}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Technical_overview].
  #
  # == About the Examples
  #
  # :include: doc/net-http/examples.rdoc
  #
  # == Strategies
  #
  # - If you will make only a few GET requests,
  #   consider using {OpenURI}[rdoc-ref:OpenURI].
  # - If you will make only a few requests of all kinds,
  #   consider using the various singleton convenience methods in this class.
  #   Each of the following methods automatically starts and finishes
  #   a {session}[rdoc-ref:Net::HTTP@Sessions] that sends a single request:
  #
  #     # Return string response body.
  #     Net::HTTP.get(hostname, path)
  #     Net::HTTP.get(uri)
  #
  #     # Write string response body to $stdout.
  #     Net::HTTP.get_print(hostname, path)
  #     Net::HTTP.get_print(uri)
  #
  #     # Return response as Net::HTTPResponse object.
  #     Net::HTTP.get_response(hostname, path)
  #     Net::HTTP.get_response(uri)
  #     data = '{"title": "foo", "body": "bar", "userId": 1}'
  #     Net::HTTP.post(uri, data)
  #     params = {title: 'foo', body: 'bar', userId: 1}
  #     Net::HTTP.post_form(uri, params)
  #     data = '{"title": "foo", "body": "bar", "userId": 1}'
  #     Net::HTTP.put(uri, data)
  #
  # - If performance is important, consider using sessions, which lower request overhead.
  #   This {session}[rdoc-ref:Net::HTTP@Sessions] has multiple requests for
  #   {HTTP methods}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Request_methods]
  #   and {WebDAV methods}[https://en.wikipedia.org/wiki/WebDAV#Implementation]:
  #
  #     Net::HTTP.start(hostname) do |http|
  #       # Session started automatically before block execution.
  #       http.get(path)
  #       http.head(path)
  #       body = 'Some text'
  #       http.post(path, body)  # Can also have a block.
  #       http.put(path, body)
  #       http.delete(path)
  #       http.options(path)
  #       http.trace(path)
  #       http.patch(path, body) # Can also have a block.
  #       http.copy(path)
  #       http.lock(path, body)
  #       http.mkcol(path, body)
  #       http.move(path)
  #       http.propfind(path, body)
  #       http.proppatch(path, body)
  #       http.unlock(path, body)
  #       # Session finished automatically at block exit.
  #     end
  #
  # The methods cited above are convenience methods that, via their few arguments,
  # allow minimal control over the requests.
  # For greater control, consider using {request objects}[rdoc-ref:Net::HTTPRequest].
  #
  # == URIs
  #
  # On the internet, a URI
  # ({Universal Resource Identifier}[https://en.wikipedia.org/wiki/Uniform_Resource_Identifier])
  # is a string that identifies a particular resource.
  # It consists of some or all of: scheme, hostname, path, query, and fragment;
  # see {URI syntax}[https://en.wikipedia.org/wiki/Uniform_Resource_Identifier#Syntax].
  #
  # A Ruby {URI::Generic}[rdoc-ref:URI::Generic] object
  # represents an internet URI.
  # It provides, among others, methods
  # +scheme+, +hostname+, +path+, +query+, and +fragment+.
  #
  # === Schemes
  #
  # An internet \URI has
  # a {scheme}[https://en.wikipedia.org/wiki/List_of_URI_schemes].
  #
  # The two schemes supported in \Net::HTTP are <tt>'https'</tt> and <tt>'http'</tt>:
  #
  #   uri.scheme                       # => "https"
  #   URI('http://example.com').scheme # => "http"
  #
  # === Hostnames
  #
  # A hostname identifies a server (host) to which requests may be sent:
  #
  #   hostname = uri.hostname # => "jsonplaceholder.typicode.com"
  #   Net::HTTP.start(hostname) do |http|
  #     # Some HTTP stuff.
  #   end
  #
  # === Paths
  #
  # A host-specific path identifies a resource on the host:
  #
  #   _uri = uri.dup
  #   _uri.path = '/todos/1'
  #   hostname = _uri.hostname
  #   path = _uri.path
  #   Net::HTTP.get(hostname, path)
  #
  # === Queries
  #
  # A host-specific query adds name/value pairs to the URI:
  #
  #   _uri = uri.dup
  #   params = {userId: 1, completed: false}
  #   _uri.query = URI.encode_www_form(params)
  #   _uri # => #<URI::HTTPS https://jsonplaceholder.typicode.com?userId=1&completed=false>
  #   Net::HTTP.get(_uri)
  #
  # === Fragments
  #
  # A {URI fragment}[https://en.wikipedia.org/wiki/URI_fragment] has no effect
  # in \Net::HTTP;
  # the same data is returned, regardless of whether a fragment is included.
  #
  # == Request Headers
  #
  # Request headers may be used to pass additional information to the host,
  # similar to arguments passed in a method call;
  # each header is a name/value pair.
  #
  # Each of the \Net::HTTP methods that sends a request to the host
  # has optional argument +headers+,
  # where the headers are expressed as a hash of field-name/value pairs:
  #
  #   headers = {Accept: 'application/json', Connection: 'Keep-Alive'}
  #   Net::HTTP.get(uri, headers)
  #
  # See lists of both standard request fields and common request fields at
  # {Request Fields}[https://en.wikipedia.org/wiki/List_of_HTTP_header_fields#Request_fields].
  # A host may also accept other custom fields.
  #
  # == \HTTP Sessions
  #
  # A _session_ is a connection between a server (host) and a client that:
  #
  # - Is begun by instance method Net::HTTP#start.
  # - May contain any number of requests.
  # - Is ended by instance method Net::HTTP#finish.
  #
  # See example sessions at {Strategies}[rdoc-ref:Net::HTTP@Strategies].
  #
  # === Session Using \Net::HTTP.start
  #
  # If you have many requests to make to a single host (and port),
  # consider using singleton method Net::HTTP.start with a block;
  # the method handles the session automatically by:
  #
  # - Calling #start before block execution.
  # - Executing the block.
  # - Calling #finish after block execution.
  #
  # In the block, you can use these instance methods,
  # each of which that sends a single request:
  #
  # - {HTTP methods}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Request_methods]:
  #
  #   - #get, #request_get: GET.
  #   - #head, #request_head: HEAD.
  #   - #post, #request_post: POST.
  #   - #delete: DELETE.
  #   - #options: OPTIONS.
  #   - #trace: TRACE.
  #   - #patch: PATCH.
  #
  # - {WebDAV methods}[https://en.wikipedia.org/wiki/WebDAV#Implementation]:
  #
  #   - #copy: COPY.
  #   - #lock: LOCK.
  #   - #mkcol: MKCOL.
  #   - #move: MOVE.
  #   - #propfind: PROPFIND.
  #   - #proppatch: PROPPATCH.
  #   - #unlock: UNLOCK.
  #
  # === Session Using \Net::HTTP.start and \Net::HTTP.finish
  #
  # You can manage a session manually using methods #start and #finish:
  #
  #   http = Net::HTTP.new(hostname)
  #   http.start
  #   http.get('/todos/1')
  #   http.get('/todos/2')
  #   http.delete('/posts/1')
  #   http.finish # Needed to free resources.
  #
  # === Single-Request Session
  #
  # Certain convenience methods automatically handle a session by:
  #
  # - Creating an \HTTP object
  # - Starting a session.
  # - Sending a single request.
  # - Finishing the session.
  # - Destroying the object.
  #
  # Such methods that send GET requests:
  #
  # - ::get: Returns the string response body.
  # - ::get_print: Writes the string response body to $stdout.
  # - ::get_response: Returns a Net::HTTPResponse object.
  #
  # Such methods that send POST requests:
  #
  # - ::post: Posts data to the host.
  # - ::post_form: Posts form data to the host.
  #
  # == \HTTP Requests and Responses
  #
  # Many of the methods above are convenience methods,
  # each of which sends a request and returns a string
  # without directly using \Net::HTTPRequest and \Net::HTTPResponse objects.
  #
  # You can, however, directly create a request object, send the request,
  # and retrieve the response object; see:
  #
  # - Net::HTTPRequest.
  # - Net::HTTPResponse.
  #
  # == Following Redirection
  #
  # Each returned response is an instance of a subclass of Net::HTTPResponse.
  # See the {response class hierarchy}[rdoc-ref:Net::HTTPResponse@Response+Subclasses].
  #
  # In particular, class Net::HTTPRedirection is the parent
  # of all redirection classes.
  # This allows you to craft a case statement to handle redirections properly:
  #
  #   def fetch(uri, limit = 10)
  #     # You should choose a better exception.
  #     raise ArgumentError, 'Too many HTTP redirects' if limit == 0
  #
  #     res = Net::HTTP.get_response(URI(uri))
  #     case res
  #     when Net::HTTPSuccess     # Any success class.
  #       res
  #     when Net::HTTPRedirection # Any redirection class.
  #       location = res['Location']
  #       warn "Redirected to #{location}"
  #       fetch(location, limit - 1)
  #     else                      # Any other class.
  #       res.value
  #     end
  #   end
  #
  #   fetch(uri)
  #
  # == Basic Authentication
  #
  # Basic authentication is performed according to
  # {RFC2617}[http://www.ietf.org/rfc/rfc2617.txt]:
  #
  #   req = Net::HTTP::Get.new(uri)
  #   req.basic_auth('user', 'pass')
  #   res = Net::HTTP.start(hostname) do |http|
  #     http.request(req)
  #   end
  #
  # == Streaming Response Bodies
  #
  # By default \Net::HTTP reads an entire response into memory.  If you are
  # handling large files or wish to implement a progress bar you can instead
  # stream the body directly to an IO.
  #
  #   Net::HTTP.start(hostname) do |http|
  #     req = Net::HTTP::Get.new(uri)
  #     http.request(req) do |res|
  #       open('t.tmp', 'w') do |f|
  #         res.read_body do |chunk|
  #           f.write chunk
  #         end
  #       end
  #     end
  #   end
  #
  # == HTTPS
  #
  # HTTPS is enabled for an \HTTP connection by Net::HTTP#use_ssl=:
  #
  #   Net::HTTP.start(hostname, :use_ssl => true) do |http|
  #     req = Net::HTTP::Get.new(uri)
  #     res = http.request(req)
  #   end
  #
  # Or if you simply want to make a GET request, you may pass in a URI
  # object that has an \HTTPS URL. \Net::HTTP automatically turns on TLS
  # verification if the URI object has a 'https' URI scheme:
  #
  #   uri # => #<URI::HTTPS https://jsonplaceholder.typicode.com/>
  #   Net::HTTP.get(uri)
  #
  # == Proxy Server
  #
  # An \HTTP object can have
  # a {proxy server}[https://en.wikipedia.org/wiki/Proxy_server].
  #
  # You can create an \HTTP object with a proxy server
  # using method Net::HTTP.new or method Net::HTTP.start.
  #
  # The proxy may be defined either by argument +p_addr+
  # or by environment variable <tt>'http_proxy'</tt>.
  #
  # === Proxy Using Argument +p_addr+ as a \String
  #
  # When argument +p_addr+ is a string hostname,
  # the returned +http+ has the given host as its proxy:
  #
  #   http = Net::HTTP.new(hostname, nil, 'proxy.example')
  #   http.proxy?          # => true
  #   http.proxy_from_env? # => false
  #   http.proxy_address   # => "proxy.example"
  #   # These use default values.
  #   http.proxy_port      # => 80
  #   http.proxy_user      # => nil
  #   http.proxy_pass      # => nil
  #
  # The port, username, and password for the proxy may also be given:
  #
  #   http = Net::HTTP.new(hostname, nil, 'proxy.example', 8000, 'pname', 'ppass')
  #   # => #<Net::HTTP jsonplaceholder.typicode.com:80 open=false>
  #   http.proxy?          # => true
  #   http.proxy_from_env? # => false
  #   http.proxy_address   # => "proxy.example"
  #   http.proxy_port      # => 8000
  #   http.proxy_user      # => "pname"
  #   http.proxy_pass      # => "ppass"
  #
  # === Proxy Using '<tt>ENV['http_proxy']</tt>'
  #
  # When environment variable <tt>'http_proxy'</tt>
  # is set to a \URI string,
  # the returned +http+ will have the server at that URI as its proxy;
  # note that the \URI string must have a protocol
  # such as <tt>'http'</tt> or <tt>'https'</tt>:
  #
  #   ENV['http_proxy'] = 'http://example.com'
  #   http = Net::HTTP.new(hostname)
  #   http.proxy?          # => true
  #   http.proxy_from_env? # => true
  #   http.proxy_address   # => "example.com"
  #   # These use default values.
  #   http.proxy_port      # => 80
  #   http.proxy_user      # => nil
  #   http.proxy_pass      # => nil
  #
  # The \URI string may include proxy username, password, and port number:
  #
  #   ENV['http_proxy'] = 'http://pname:ppass@example.com:8000'
  #   http = Net::HTTP.new(hostname)
  #   http.proxy?          # => true
  #   http.proxy_from_env? # => true
  #   http.proxy_address   # => "example.com"
  #   http.proxy_port      # => 8000
  #   http.proxy_user      # => "pname"
  #   http.proxy_pass      # => "ppass"
  #
  # === Filtering Proxies
  #
  # With method Net::HTTP.new (but not Net::HTTP.start),
  # you can use argument +p_no_proxy+ to filter proxies:
  #
  # - Reject a certain address:
  #
  #     http = Net::HTTP.new('example.com', nil, 'proxy.example', 8000, 'pname', 'ppass', 'proxy.example')
  #     http.proxy_address # => nil
  #
  # - Reject certain domains or subdomains:
  #
  #     http = Net::HTTP.new('example.com', nil, 'my.proxy.example', 8000, 'pname', 'ppass', 'proxy.example')
  #     http.proxy_address # => nil
  #
  # - Reject certain addresses and port combinations:
  #
  #     http = Net::HTTP.new('example.com', nil, 'proxy.example', 8000, 'pname', 'ppass', 'proxy.example:1234')
  #     http.proxy_address # => "proxy.example"
  #
  #     http = Net::HTTP.new('example.com', nil, 'proxy.example', 8000, 'pname', 'ppass', 'proxy.example:8000')
  #     http.proxy_address # => nil
  #
  # - Reject a list of the types above delimited using a comma:
  #
  #     http = Net::HTTP.new('example.com', nil, 'proxy.example', 8000, 'pname', 'ppass', 'my.proxy,proxy.example:8000')
  #     http.proxy_address # => nil
  #
  #     http = Net::HTTP.new('example.com', nil, 'my.proxy', 8000, 'pname', 'ppass', 'my.proxy,proxy.example:8000')
  #     http.proxy_address # => nil
  #
  # == Compression and Decompression
  #
  # \Net::HTTP does not compress the body of a request before sending.
  #
  # By default, \Net::HTTP adds header <tt>'Accept-Encoding'</tt>
  # to a new {request object}[rdoc-ref:Net::HTTPRequest]:
  #
  #   Net::HTTP::Get.new(uri)['Accept-Encoding']
  #   # => "gzip;q=1.0,deflate;q=0.6,identity;q=0.3"
  #
  # This requests the server to zip-encode the response body if there is one;
  # the server is not required to do so.
  #
  # \Net::HTTP does not automatically decompress a response body
  # if the response has header <tt>'Content-Range'</tt>.
  #
  # Otherwise decompression (or not) depends on the value of header
  # {Content-Encoding}[https://en.wikipedia.org/wiki/List_of_HTTP_header_fields#content-encoding-response-header]:
  #
  # - <tt>'deflate'</tt>, <tt>'gzip'</tt>, or <tt>'x-gzip'</tt>:
  #   decompresses the body and deletes the header.
  # - <tt>'none'</tt> or <tt>'identity'</tt>:
  #   does not decompress the body, but deletes the header.
  # - Any other value:
  #   leaves the body and header unchanged.
  #
  # == What's Here
  #
  # First, what's elsewhere. Class Net::HTTP:
  #
  # - Inherits from {class Object}[rdoc-ref:Object@What-27s+Here].
  #
  # This is a categorized summary of methods and attributes.
  #
  # === \Net::HTTP Objects
  #
  # - {::new}[rdoc-ref:Net::HTTP.new]:
  #   Creates a new instance.
  # - {#inspect}[rdoc-ref:Net::HTTP#inspect]:
  #   Returns a string representation of +self+.
  #
  # === Sessions
  #
  # - {::start}[rdoc-ref:Net::HTTP.start]:
  #   Begins a new session in a new \Net::HTTP object.
  # - {#started?}[rdoc-ref:Net::HTTP#started?]:
  #   Returns whether in a session.
  # - {#finish}[rdoc-ref:Net::HTTP#finish]:
  #   Ends an active session.
  # - {#start}[rdoc-ref:Net::HTTP#start]:
  #   Begins a new session in an existing \Net::HTTP object (+self+).
  #
  # === Connections
  #
  # - {:continue_timeout}[rdoc-ref:Net::HTTP#continue_timeout]:
  #   Returns the continue timeout.
  # - {#continue_timeout=}[rdoc-ref:Net::HTTP#continue_timeout=]:
  #   Sets the continue timeout seconds.
  # - {:keep_alive_timeout}[rdoc-ref:Net::HTTP#keep_alive_timeout]:
  #   Returns the keep-alive timeout.
  # - {:keep_alive_timeout=}[rdoc-ref:Net::HTTP#keep_alive_timeout=]:
  #   Sets the keep-alive timeout.
  # - {:max_retries}[rdoc-ref:Net::HTTP#max_retries]:
  #   Returns the maximum retries.
  # - {#max_retries=}[rdoc-ref:Net::HTTP#max_retries=]:
  #   Sets the maximum retries.
  # - {:open_timeout}[rdoc-ref:Net::HTTP#open_timeout]:
  #   Returns the open timeout.
  # - {:open_timeout=}[rdoc-ref:Net::HTTP#open_timeout=]:
  #   Sets the open timeout.
  # - {:read_timeout}[rdoc-ref:Net::HTTP#read_timeout]:
  #   Returns the open timeout.
  # - {:read_timeout=}[rdoc-ref:Net::HTTP#read_timeout=]:
  #   Sets the read timeout.
  # - {:ssl_timeout}[rdoc-ref:Net::HTTP#ssl_timeout]:
  #   Returns the ssl timeout.
  # - {:ssl_timeout=}[rdoc-ref:Net::HTTP#ssl_timeout=]:
  #   Sets the ssl timeout.
  # - {:write_timeout}[rdoc-ref:Net::HTTP#write_timeout]:
  #   Returns the write timeout.
  # - {write_timeout=}[rdoc-ref:Net::HTTP#write_timeout=]:
  #   Sets the write timeout.
  #
  # === Requests
  #
  # - {::get}[rdoc-ref:Net::HTTP.get]:
  #   Sends a GET request and returns the string response body.
  # - {::get_print}[rdoc-ref:Net::HTTP.get_print]:
  #   Sends a GET request and write the string response body to $stdout.
  # - {::get_response}[rdoc-ref:Net::HTTP.get_response]:
  #   Sends a GET request and returns a response object.
  # - {::post_form}[rdoc-ref:Net::HTTP.post_form]:
  #   Sends a POST request with form data and returns a response object.
  # - {::post}[rdoc-ref:Net::HTTP.post]:
  #   Sends a POST request with data and returns a response object.
  # - {::put}[rdoc-ref:Net::HTTP.put]:
  #   Sends a PUT request with data and returns a response object.
  # - {#copy}[rdoc-ref:Net::HTTP#copy]:
  #   Sends a COPY request and returns a response object.
  # - {#delete}[rdoc-ref:Net::HTTP#delete]:
  #   Sends a DELETE request and returns a response object.
  # - {#get}[rdoc-ref:Net::HTTP#get]:
  #   Sends a GET request and returns a response object.
  # - {#head}[rdoc-ref:Net::HTTP#head]:
  #   Sends a HEAD request and returns a response object.
  # - {#lock}[rdoc-ref:Net::HTTP#lock]:
  #   Sends a LOCK request and returns a response object.
  # - {#mkcol}[rdoc-ref:Net::HTTP#mkcol]:
  #   Sends a MKCOL request and returns a response object.
  # - {#move}[rdoc-ref:Net::HTTP#move]:
  #   Sends a MOVE request and returns a response object.
  # - {#options}[rdoc-ref:Net::HTTP#options]:
  #   Sends a OPTIONS request and returns a response object.
  # - {#patch}[rdoc-ref:Net::HTTP#patch]:
  #   Sends a PATCH request and returns a response object.
  # - {#post}[rdoc-ref:Net::HTTP#post]:
  #   Sends a POST request and returns a response object.
  # - {#propfind}[rdoc-ref:Net::HTTP#propfind]:
  #   Sends a PROPFIND request and returns a response object.
  # - {#proppatch}[rdoc-ref:Net::HTTP#proppatch]:
  #   Sends a PROPPATCH request and returns a response object.
  # - {#put}[rdoc-ref:Net::HTTP#put]:
  #   Sends a PUT request and returns a response object.
  # - {#request}[rdoc-ref:Net::HTTP#request]:
  #   Sends a request and returns a response object.
  # - {#request_get}[rdoc-ref:Net::HTTP#request_get]:
  #   Sends a GET request and forms a response object;
  #   if a block given, calls the block with the object,
  #   otherwise returns the object.
  # - {#request_head}[rdoc-ref:Net::HTTP#request_head]:
  #   Sends a HEAD request and forms a response object;
  #   if a block given, calls the block with the object,
  #   otherwise returns the object.
  # - {#request_post}[rdoc-ref:Net::HTTP#request_post]:
  #   Sends a POST request and forms a response object;
  #   if a block given, calls the block with the object,
  #   otherwise returns the object.
  # - {#send_request}[rdoc-ref:Net::HTTP#send_request]:
  #   Sends a request and returns a response object.
  # - {#trace}[rdoc-ref:Net::HTTP#trace]:
  #   Sends a TRACE request and returns a response object.
  # - {#unlock}[rdoc-ref:Net::HTTP#unlock]:
  #   Sends an UNLOCK request and returns a response object.
  #
  # === Responses
  #
  # - {:close_on_empty_response}[rdoc-ref:Net::HTTP#close_on_empty_response]:
  #   Returns whether to close connection on empty response.
  # - {:close_on_empty_response=}[rdoc-ref:Net::HTTP#close_on_empty_response=]:
  #   Sets whether to close connection on empty response.
  # - {:ignore_eof}[rdoc-ref:Net::HTTP#ignore_eof]:
  #   Returns whether to ignore end-of-file when reading a response body
  #   with <tt>Content-Length</tt> headers.
  # - {:ignore_eof=}[rdoc-ref:Net::HTTP#ignore_eof=]:
  #   Sets whether to ignore end-of-file when reading a response body
  #   with <tt>Content-Length</tt> headers.
  # - {:response_body_encoding}[rdoc-ref:Net::HTTP#response_body_encoding]:
  #   Returns the encoding to use for the response body.
  # - {#response_body_encoding=}[rdoc-ref:Net::HTTP#response_body_encoding=]:
  #   Sets the response body encoding.
  #
  # === Proxies
  #
  # - {:proxy_address}[rdoc-ref:Net::HTTP#proxy_address]:
  #   Returns the proxy address.
  # - {:proxy_address=}[rdoc-ref:Net::HTTP#proxy_address=]:
  #   Sets the proxy address.
  # - {::proxy_class?}[rdoc-ref:Net::HTTP.proxy_class?]:
  #   Returns whether +self+ is a proxy class.
  # - {#proxy?}[rdoc-ref:Net::HTTP#proxy?]:
  #   Returns whether +self+ has a proxy.
  # - {#proxy_address}[rdoc-ref:Net::HTTP#proxy_address]:
  #   Returns the proxy address.
  # - {#proxy_from_env?}[rdoc-ref:Net::HTTP#proxy_from_env?]:
  #   Returns whether the proxy is taken from an environment variable.
  # - {:proxy_from_env=}[rdoc-ref:Net::HTTP#proxy_from_env=]:
  #   Sets whether the proxy is to be taken from an environment variable.
  # - {:proxy_pass}[rdoc-ref:Net::HTTP#proxy_pass]:
  #   Returns the proxy password.
  # - {:proxy_pass=}[rdoc-ref:Net::HTTP#proxy_pass=]:
  #   Sets the proxy password.
  # - {:proxy_port}[rdoc-ref:Net::HTTP#proxy_port]:
  #   Returns the proxy port.
  # - {:proxy_port=}[rdoc-ref:Net::HTTP#proxy_port=]:
  #   Sets the proxy port.
  # - {#proxy_user}[rdoc-ref:Net::HTTP#proxy_user]:
  #   Returns the proxy user name.
  # - {:proxy_user=}[rdoc-ref:Net::HTTP#proxy_user=]:
  #   Sets the proxy user.
  #
  # === Security
  #
  # - {:ca_file}[rdoc-ref:Net::HTTP#ca_file]:
  #   Returns the path to a CA certification file.
  # - {:ca_file=}[rdoc-ref:Net::HTTP#ca_file=]:
  #   Sets the path to a CA certification file.
  # - {:ca_path}[rdoc-ref:Net::HTTP#ca_path]:
  #   Returns the path of to CA directory containing certification files.
  # - {:ca_path=}[rdoc-ref:Net::HTTP#ca_path=]:
  #   Sets the path of to CA directory containing certification files.
  # - {:cert}[rdoc-ref:Net::HTTP#cert]:
  #   Returns the OpenSSL::X509::Certificate object to be used for client certification.
  # - {:cert=}[rdoc-ref:Net::HTTP#cert=]:
  #   Sets the OpenSSL::X509::Certificate object to be used for client certification.
  # - {:cert_store}[rdoc-ref:Net::HTTP#cert_store]:
  #   Returns the X509::Store to be used for verifying peer certificate.
  # - {:cert_store=}[rdoc-ref:Net::HTTP#cert_store=]:
  #   Sets the X509::Store to be used for verifying peer certificate.
  # - {:ciphers}[rdoc-ref:Net::HTTP#ciphers]:
  #   Returns the available SSL ciphers.
  # - {:ciphers=}[rdoc-ref:Net::HTTP#ciphers=]:
  #   Sets the available SSL ciphers.
  # - {:extra_chain_cert}[rdoc-ref:Net::HTTP#extra_chain_cert]:
  #   Returns the extra X509 certificates to be added to the certificate chain.
  # - {:extra_chain_cert=}[rdoc-ref:Net::HTTP#extra_chain_cert=]:
  #   Sets the extra X509 certificates to be added to the certificate chain.
  # - {:key}[rdoc-ref:Net::HTTP#key]:
  #   Returns the OpenSSL::PKey::RSA or OpenSSL::PKey::DSA object.
  # - {:key=}[rdoc-ref:Net::HTTP#key=]:
  #   Sets the OpenSSL::PKey::RSA or OpenSSL::PKey::DSA object.
  # - {:max_version}[rdoc-ref:Net::HTTP#max_version]:
  #   Returns the maximum SSL version.
  # - {:max_version=}[rdoc-ref:Net::HTTP#max_version=]:
  #   Sets the maximum SSL version.
  # - {:min_version}[rdoc-ref:Net::HTTP#min_version]:
  #   Returns the minimum SSL version.
  # - {:min_version=}[rdoc-ref:Net::HTTP#min_version=]:
  #   Sets the minimum SSL version.
  # - {#peer_cert}[rdoc-ref:Net::HTTP#peer_cert]:
  #   Returns the X509 certificate chain for the session's socket peer.
  # - {:ssl_version}[rdoc-ref:Net::HTTP#ssl_version]:
  #   Returns the SSL version.
  # - {:ssl_version=}[rdoc-ref:Net::HTTP#ssl_version=]:
  #   Sets the SSL version.
  # - {#use_ssl=}[rdoc-ref:Net::HTTP#use_ssl=]:
  #   Sets whether a new session is to use Transport Layer Security.
  # - {#use_ssl?}[rdoc-ref:Net::HTTP#use_ssl?]:
  #   Returns whether +self+ uses SSL.
  # - {:verify_callback}[rdoc-ref:Net::HTTP#verify_callback]:
  #   Returns the callback for the server certification verification.
  # - {:verify_callback=}[rdoc-ref:Net::HTTP#verify_callback=]:
  #   Sets the callback for the server certification verification.
  # - {:verify_depth}[rdoc-ref:Net::HTTP#verify_depth]:
  #   Returns the maximum depth for the certificate chain verification.
  # - {:verify_depth=}[rdoc-ref:Net::HTTP#verify_depth=]:
  #   Sets the maximum depth for the certificate chain verification.
  # - {:verify_hostname}[rdoc-ref:Net::HTTP#verify_hostname]:
  #   Returns the flags for server the certification verification at the beginning of the SSL/TLS session.
  # - {:verify_hostname=}[rdoc-ref:Net::HTTP#verify_hostname=]:
  #   Sets he flags for server the certification verification at the beginning of the SSL/TLS session.
  # - {:verify_mode}[rdoc-ref:Net::HTTP#verify_mode]:
  #   Returns the flags for server the certification verification at the beginning of the SSL/TLS session.
  # - {:verify_mode=}[rdoc-ref:Net::HTTP#verify_mode=]:
  #   Sets the flags for server the certification verification at the beginning of the SSL/TLS session.
  #
  # === Addresses and Ports
  #
  # - {:address}[rdoc-ref:Net::HTTP#address]:
  #   Returns the string host name or host IP.
  # - {::default_port}[rdoc-ref:Net::HTTP.default_port]:
  #   Returns integer 80, the default port to use for HTTP requests.
  # - {::http_default_port}[rdoc-ref:Net::HTTP.http_default_port]:
  #   Returns integer 80, the default port to use for HTTP requests.
  # - {::https_default_port}[rdoc-ref:Net::HTTP.https_default_port]:
  #   Returns integer 443, the default port to use for HTTPS requests.
  # - {#ipaddr}[rdoc-ref:Net::HTTP#ipaddr]:
  #   Returns the IP address for the connection.
  # - {#ipaddr=}[rdoc-ref:Net::HTTP#ipaddr=]:
  #   Sets the IP address for the connection.
  # - {:local_host}[rdoc-ref:Net::HTTP#local_host]:
  #   Returns the string local host used to establish the connection.
  # - {:local_host=}[rdoc-ref:Net::HTTP#local_host=]:
  #   Sets the string local host used to establish the connection.
  # - {:local_port}[rdoc-ref:Net::HTTP#local_port]:
  #   Returns the integer local port used to establish the connection.
  # - {:local_port=}[rdoc-ref:Net::HTTP#local_port=]:
  #   Sets the integer local port used to establish the connection.
  # - {:port}[rdoc-ref:Net::HTTP#port]:
  #   Returns the integer port number.
  #
  # === \HTTP Version
  #
  # - {::version_1_2?}[rdoc-ref:Net::HTTP.version_1_2?]
  #   (aliased as {::version_1_2}[rdoc-ref:Net::HTTP.version_1_2]):
  #   Returns true; retained for compatibility.
  #
  # === Debugging
  #
  # - {#set_debug_output}[rdoc-ref:Net::HTTP#set_debug_output]:
  #   Sets the output stream for debugging.
  #
  class HTTP < Protocol

    # :stopdoc:
    VERSION = "0.6.0"
    HTTPVersion = '1.1'
    begin
      require 'zlib'
      HAVE_ZLIB=true
    rescue LoadError
      HAVE_ZLIB=false
    end
    # :startdoc:

    # Returns +true+; retained for compatibility.
    def HTTP.version_1_2
      true
    end

    # Returns +true+; retained for compatibility.
    def HTTP.version_1_2?
      true
    end

    # Returns +false+; retained for compatibility.
    def HTTP.version_1_1?  #:nodoc:
      false
    end

    class << HTTP
      alias is_version_1_1? version_1_1?   #:nodoc:
      alias is_version_1_2? version_1_2?   #:nodoc:
    end

    # :call-seq:
    #   Net::HTTP.get_print(hostname, path, port = 80) -> nil
    #   Net::HTTP:get_print(uri, headers = {}, port = uri.port) -> nil
    #
    # Like Net::HTTP.get, but writes the returned body to $stdout;
    # returns +nil+.
    def HTTP.get_print(uri_or_host, path_or_headers = nil, port = nil)
      get_response(uri_or_host, path_or_headers, port) {|res|
        res.read_body do |chunk|
          $stdout.print chunk
        end
      }
      nil
    end

    # :call-seq:
    #   Net::HTTP.get(hostname, path, port = 80) -> body
    #   Net::HTTP:get(uri, headers = {}, port = uri.port) -> body
    #
    # Sends a GET request and returns the \HTTP response body as a string.
    #
    # With string arguments +hostname+ and +path+:
    #
    #   hostname = 'jsonplaceholder.typicode.com'
    #   path = '/todos/1'
    #   puts Net::HTTP.get(hostname, path)
    #
    # Output:
    #
    #   {
    #     "userId": 1,
    #     "id": 1,
    #     "title": "delectus aut autem",
    #     "completed": false
    #   }
    #
    # With URI object +uri+ and optional hash argument +headers+:
    #
    #   uri = URI('https://jsonplaceholder.typicode.com/todos/1')
    #   headers = {'Content-type' => 'application/json; charset=UTF-8'}
    #   Net::HTTP.get(uri, headers)
    #
    # Related:
    #
    # - Net::HTTP::Get: request class for \HTTP method +GET+.
    # - Net::HTTP#get: convenience method for \HTTP method +GET+.
    #
    def HTTP.get(uri_or_host, path_or_headers = nil, port = nil)
      get_response(uri_or_host, path_or_headers, port).body
    end

    # :call-seq:
    #   Net::HTTP.get_response(hostname, path, port = 80) -> http_response
    #   Net::HTTP:get_response(uri, headers = {}, port = uri.port) -> http_response
    #
    # Like Net::HTTP.get, but returns a Net::HTTPResponse object
    # instead of the body string.
    def HTTP.get_response(uri_or_host, path_or_headers = nil, port = nil, &block)
      if path_or_headers && !path_or_headers.is_a?(Hash)
        host = uri_or_host
        path = path_or_headers
        new(host, port || HTTP.default_port).start {|http|
          return http.request_get(path, &block)
        }
      else
        uri = uri_or_host
        headers = path_or_headers
        start(uri.hostname, uri.port,
              :use_ssl => uri.scheme == 'https') {|http|
          return http.request_get(uri, headers, &block)
        }
      end
    end

    # Posts data to a host; returns a Net::HTTPResponse object.
    #
    # Argument +url+ must be a URL;
    # argument +data+ must be a string:
    #
    #   _uri = uri.dup
    #   _uri.path = '/posts'
    #   data = '{"title": "foo", "body": "bar", "userId": 1}'
    #   headers = {'content-type': 'application/json'}
    #   res = Net::HTTP.post(_uri, data, headers) # => #<Net::HTTPCreated 201 Created readbody=true>
    #   puts res.body
    #
    # Output:
    #
    #   {
    #     "title": "foo",
    #     "body": "bar",
    #     "userId": 1,
    #     "id": 101
    #   }
    #
    # Related:
    #
    # - Net::HTTP::Post: request class for \HTTP method +POST+.
    # - Net::HTTP#post: convenience method for \HTTP method +POST+.
    #
    def HTTP.post(url, data, header = nil)
      start(url.hostname, url.port,
            :use_ssl => url.scheme == 'https' ) {|http|
        http.post(url, data, header)
      }
    end

    # Posts data to a host; returns a Net::HTTPResponse object.
    #
    # Argument +url+ must be a URI;
    # argument +data+ must be a hash:
    #
    #   _uri = uri.dup
    #   _uri.path = '/posts'
    #   data = {title: 'foo', body: 'bar', userId: 1}
    #   res = Net::HTTP.post_form(_uri, data) # => #<Net::HTTPCreated 201 Created readbody=true>
    #   puts res.body
    #
    # Output:
    #
    #   {
    #     "title": "foo",
    #     "body": "bar",
    #     "userId": "1",
    #     "id": 101
    #   }
    #
    def HTTP.post_form(url, params)
      req = Post.new(url)
      req.form_data = params
      req.basic_auth url.user, url.password if url.user
      start(url.hostname, url.port,
            :use_ssl => url.scheme == 'https' ) {|http|
        http.request(req)
      }
    end

    # Sends a PUT request to the server; returns a Net::HTTPResponse object.
    #
    # Argument +url+ must be a URL;
    # argument +data+ must be a string:
    #
    #   _uri = uri.dup
    #   _uri.path = '/posts'
    #   data = '{"title": "foo", "body": "bar", "userId": 1}'
    #   headers = {'content-type': 'application/json'}
    #   res = Net::HTTP.put(_uri, data, headers) # => #<Net::HTTPCreated 201 Created readbody=true>
    #   puts res.body
    #
    # Output:
    #
    #   {
    #     "title": "foo",
    #     "body": "bar",
    #     "userId": 1,
    #     "id": 101
    #   }
    #
    # Related:
    #
    # - Net::HTTP::Put: request class for \HTTP method +PUT+.
    # - Net::HTTP#put: convenience method for \HTTP method +PUT+.
    #
    def HTTP.put(url, data, header = nil)
      start(url.hostname, url.port,
            :use_ssl => url.scheme == 'https' ) {|http|
        http.put(url, data, header)
      }
    end

    #
    # \HTTP session management
    #

    # Returns integer +80+, the default port to use for \HTTP requests:
    #
    #   Net::HTTP.default_port # => 80
    #
    def HTTP.default_port
      http_default_port()
    end

    # Returns integer +80+, the default port to use for \HTTP requests:
    #
    #   Net::HTTP.http_default_port # => 80
    #
    def HTTP.http_default_port
      80
    end

    # Returns integer +443+, the default port to use for HTTPS requests:
    #
    #   Net::HTTP.https_default_port # => 443
    #
    def HTTP.https_default_port
      443
    end

    def HTTP.socket_type   #:nodoc: obsolete
      BufferedIO
    end

    # :call-seq:
    #   HTTP.start(address, port = nil, p_addr = :ENV, p_port = nil, p_user = nil, p_pass = nil, opts) -> http
    #   HTTP.start(address, port = nil, p_addr = :ENV, p_port = nil, p_user = nil, p_pass = nil, opts) {|http| ... } -> object
    #
    # Creates a new \Net::HTTP object, +http+, via \Net::HTTP.new:
    #
    # - For arguments +address+ and +port+, see Net::HTTP.new.
    # - For proxy-defining arguments +p_addr+ through +p_pass+,
    #   see {Proxy Server}[rdoc-ref:Net::HTTP@Proxy+Server].
    # - For argument +opts+, see below.
    #
    # With no block given:
    #
    # - Calls <tt>http.start</tt> with no block (see #start),
    #   which opens a TCP connection and \HTTP session.
    # - Returns +http+.
    # - The caller should call #finish to close the session:
    #
    #     http = Net::HTTP.start(hostname)
    #     http.started? # => true
    #     http.finish
    #     http.started? # => false
    #
    # With a block given:
    #
    # - Calls <tt>http.start</tt> with the block (see #start), which:
    #
    #   - Opens a TCP connection and \HTTP session.
    #   - Calls the block,
    #     which may make any number of requests to the host.
    #   - Closes the \HTTP session and TCP connection on block exit.
    #   - Returns the block's value +object+.
    #
    # - Returns +object+.
    #
    # Example:
    #
    #   hostname = 'jsonplaceholder.typicode.com'
    #   Net::HTTP.start(hostname) do |http|
    #     puts http.get('/todos/1').body
    #     puts http.get('/todos/2').body
    #   end
    #
    # Output:
    #
    #   {
    #     "userId": 1,
    #     "id": 1,
    #     "title": "delectus aut autem",
    #     "completed": false
    #   }
    #   {
    #     "userId": 1,
    #     "id": 2,
    #     "title": "quis ut nam facilis et officia qui",
    #     "completed": false
    #   }
    #
    # If the last argument given is a hash, it is the +opts+ hash,
    # where each key is a method or accessor to be called,
    # and its value is the value to be set.
    #
    # The keys may include:
    #
    # - #ca_file
    # - #ca_path
    # - #cert
    # - #cert_store
    # - #ciphers
    # - #close_on_empty_response
    # - +ipaddr+ (calls #ipaddr=)
    # - #keep_alive_timeout
    # - #key
    # - #open_timeout
    # - #read_timeout
    # - #ssl_timeout
    # - #ssl_version
    # - +use_ssl+ (calls #use_ssl=)
    # - #verify_callback
    # - #verify_depth
    # - #verify_mode
    # - #write_timeout
    #
    # Note: If +port+ is +nil+ and <tt>opts[:use_ssl]</tt> is a truthy value,
    # the value passed to +new+ is Net::HTTP.https_default_port, not +port+.
    #
    def HTTP.start(address, *arg, &block) # :yield: +http+
      arg.pop if opt = Hash.try_convert(arg[-1])
      port, p_addr, p_port, p_user, p_pass = *arg
      p_addr = :ENV if arg.size < 2
      port = https_default_port if !port && opt && opt[:use_ssl]
      http = new(address, port, p_addr, p_port, p_user, p_pass)
      http.ipaddr = opt[:ipaddr] if opt && opt[:ipaddr]

      if opt
        if opt[:use_ssl]
          opt = {verify_mode: OpenSSL::SSL::VERIFY_PEER}.update(opt)
        end
        http.methods.grep(/\A(\w+)=\z/) do |meth|
          key = $1.to_sym
          opt.key?(key) or next
          http.__send__(meth, opt[key])
        end
      end

      http.start(&block)
    end

    class << HTTP
      alias newobj new # :nodoc:
    end

    # Returns a new \Net::HTTP object +http+
    # (but does not open a TCP connection or \HTTP session).
    #
    # With only string argument +address+ given
    # (and <tt>ENV['http_proxy']</tt> undefined or +nil+),
    # the returned +http+:
    #
    # - Has the given address.
    # - Has the default port number, Net::HTTP.default_port (80).
    # - Has no proxy.
    #
    # Example:
    #
    #   http = Net::HTTP.new(hostname)
    #   # => #<Net::HTTP jsonplaceholder.typicode.com:80 open=false>
    #   http.address # => "jsonplaceholder.typicode.com"
    #   http.port    # => 80
    #   http.proxy?  # => false
    #
    # With integer argument +port+ also given,
    # the returned +http+ has the given port:
    #
    #   http = Net::HTTP.new(hostname, 8000)
    #   # => #<Net::HTTP jsonplaceholder.typicode.com:8000 open=false>
    #   http.port # => 8000
    #
    # For proxy-defining arguments +p_addr+ through +p_no_proxy+,
    # see {Proxy Server}[rdoc-ref:Net::HTTP@Proxy+Server].
    #
    def HTTP.new(address, port = nil, p_addr = :ENV, p_port = nil, p_user = nil, p_pass = nil, p_no_proxy = nil, p_use_ssl = nil)
      http = super address, port

      if proxy_class? then # from Net::HTTP::Proxy()
        http.proxy_from_env = @proxy_from_env
        http.proxy_address  = @proxy_address
        http.proxy_port     = @proxy_port
        http.proxy_user     = @proxy_user
        http.proxy_pass     = @proxy_pass
        http.proxy_use_ssl  = @proxy_use_ssl
      elsif p_addr == :ENV then
        http.proxy_from_env = true
      else
        if p_addr && p_no_proxy && !URI::Generic.use_proxy?(address, address, port, p_no_proxy)
          p_addr = nil
          p_port = nil
        end
        http.proxy_address = p_addr
        http.proxy_port    = p_port || default_port
        http.proxy_user    = p_user
        http.proxy_pass    = p_pass
        http.proxy_use_ssl = p_use_ssl
      end

      http
    end

    class << HTTP
      # Allows to set the default configuration that will be used
      # when creating a new connection.
      #
      # Example:
      #
      #   Net::HTTP.default_configuration = {
      #     read_timeout: 1,
      #     write_timeout: 1
      #   }
      #   http = Net::HTTP.new(hostname)
      #   http.open_timeout   # => 60
      #   http.read_timeout   # => 1
      #   http.write_timeout  # => 1
      #
      attr_accessor :default_configuration
    end

    # Creates a new \Net::HTTP object for the specified server address,
    # without opening the TCP connection or initializing the \HTTP session.
    # The +address+ should be a DNS hostname or IP address.
    def initialize(address, port = nil) # :nodoc:
      defaults = {
        keep_alive_timeout: 2,
        close_on_empty_response: false,
        open_timeout: 60,
        read_timeout: 60,
        write_timeout: 60,
        continue_timeout: nil,
        max_retries: 1,
        debug_output: nil,
        response_body_encoding: false,
        ignore_eof: true
      }
      options = defaults.merge(self.class.default_configuration || {})

      @address = address
      @port    = (port || HTTP.default_port)
      @ipaddr = nil
      @local_host = nil
      @local_port = nil
      @curr_http_version = HTTPVersion
      @keep_alive_timeout = options[:keep_alive_timeout]
      @last_communicated = nil
      @close_on_empty_response = options[:close_on_empty_response]
      @socket  = nil
      @started = false
      @open_timeout = options[:open_timeout]
      @read_timeout = options[:read_timeout]
      @write_timeout = options[:write_timeout]
      @continue_timeout = options[:continue_timeout]
      @max_retries = options[:max_retries]
      @debug_output = options[:debug_output]
      @response_body_encoding = options[:response_body_encoding]
      @ignore_eof = options[:ignore_eof]

      @proxy_from_env = false
      @proxy_uri      = nil
      @proxy_address  = nil
      @proxy_port     = nil
      @proxy_user     = nil
      @proxy_pass     = nil
      @proxy_use_ssl  = nil

      @use_ssl = false
      @ssl_context = nil
      @ssl_session = nil
      @sspi_enabled = false
      SSL_IVNAMES.each do |ivname|
        instance_variable_set ivname, nil
      end
    end

    # Returns a string representation of +self+:
    #
    #   Net::HTTP.new(hostname).inspect
    #   # => "#<Net::HTTP jsonplaceholder.typicode.com:80 open=false>"
    #
    def inspect
      "#<#{self.class} #{@address}:#{@port} open=#{started?}>"
    end

    # *WARNING* This method opens a serious security hole.
    # Never use this method in production code.
    #
    # Sets the output stream for debugging:
    #
    #   http = Net::HTTP.new(hostname)
    #   File.open('t.tmp', 'w') do |file|
    #     http.set_debug_output(file)
    #     http.start
    #     http.get('/nosuch/1')
    #     http.finish
    #   end
    #   puts File.read('t.tmp')
    #
    # Output:
    #
    #   opening connection to jsonplaceholder.typicode.com:80...
    #   opened
    #   <- "GET /nosuch/1 HTTP/1.1\r\nAccept-Encoding: gzip;q=1.0,deflate;q=0.6,identity;q=0.3\r\nAccept: */*\r\nUser-Agent: Ruby\r\nHost: jsonplaceholder.typicode.com\r\n\r\n"
    #   -> "HTTP/1.1 404 Not Found\r\n"
    #   -> "Date: Mon, 12 Dec 2022 21:14:11 GMT\r\n"
    #   -> "Content-Type: application/json; charset=utf-8\r\n"
    #   -> "Content-Length: 2\r\n"
    #   -> "Connection: keep-alive\r\n"
    #   -> "X-Powered-By: Express\r\n"
    #   -> "X-Ratelimit-Limit: 1000\r\n"
    #   -> "X-Ratelimit-Remaining: 999\r\n"
    #   -> "X-Ratelimit-Reset: 1670879660\r\n"
    #   -> "Vary: Origin, Accept-Encoding\r\n"
    #   -> "Access-Control-Allow-Credentials: true\r\n"
    #   -> "Cache-Control: max-age=43200\r\n"
    #   -> "Pragma: no-cache\r\n"
    #   -> "Expires: -1\r\n"
    #   -> "X-Content-Type-Options: nosniff\r\n"
    #   -> "Etag: W/\"2-vyGp6PvFo4RvsFtPoIWeCReyIC8\"\r\n"
    #   -> "Via: 1.1 vegur\r\n"
    #   -> "CF-Cache-Status: MISS\r\n"
    #   -> "Server-Timing: cf-q-config;dur=1.3000000762986e-05\r\n"
    #   -> "Report-To: {\"endpoints\":[{\"url\":\"https:\\/\\/a.nel.cloudflare.com\\/report\\/v3?s=yOr40jo%2BwS1KHzhTlVpl54beJ5Wx2FcG4gGV0XVrh3X9OlR5q4drUn2dkt5DGO4GDcE%2BVXT7CNgJvGs%2BZleIyMu8CLieFiDIvOviOY3EhHg94m0ZNZgrEdpKD0S85S507l1vsEwEHkoTm%2Ff19SiO\"}],\"group\":\"cf-nel\",\"max_age\":604800}\r\n"
    #   -> "NEL: {\"success_fraction\":0,\"report_to\":\"cf-nel\",\"max_age\":604800}\r\n"
    #   -> "Server: cloudflare\r\n"
    #   -> "CF-RAY: 778977dc484ce591-DFW\r\n"
    #   -> "alt-svc: h3=\":443\"; ma=86400, h3-29=\":443\"; ma=86400\r\n"
    #   -> "\r\n"
    #   reading 2 bytes...
    #   -> "{}"
    #   read 2 bytes
    #   Conn keep-alive
    #
    def set_debug_output(output)
      warn 'Net::HTTP#set_debug_output called after HTTP started', uplevel: 1 if started?
      @debug_output = output
    end

    # Returns the string host name or host IP given as argument +address+ in ::new.
    attr_reader :address

    # Returns the integer port number given as argument +port+ in ::new.
    attr_reader :port

    # Sets or returns the string local host used to establish the connection;
    # initially +nil+.
    attr_accessor :local_host

    # Sets or returns the integer local port used to establish the connection;
    # initially +nil+.
    attr_accessor :local_port

    # Returns the encoding to use for the response body;
    # see #response_body_encoding=.
    attr_reader :response_body_encoding

    # Sets the encoding to be used for the response body;
    # returns the encoding.
    #
    # The given +value+ may be:
    #
    # - An Encoding object.
    # - The name of an encoding.
    # - An alias for an encoding name.
    #
    # See {Encoding}[rdoc-ref:Encoding].
    #
    # Examples:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.response_body_encoding = Encoding::US_ASCII # => #<Encoding:US-ASCII>
    #   http.response_body_encoding = 'US-ASCII'         # => "US-ASCII"
    #   http.response_body_encoding = 'ASCII'            # => "ASCII"
    #
    def response_body_encoding=(value)
      value = Encoding.find(value) if value.is_a?(String)
      @response_body_encoding = value
    end

    # Sets whether to determine the proxy from environment variable
    # '<tt>ENV['http_proxy']</tt>';
    # see {Proxy Using ENV['http_proxy']}[rdoc-ref:Net::HTTP@Proxy+Using+-27ENV-5B-27http_proxy-27-5D-27].
    attr_writer :proxy_from_env

    # Sets the proxy address;
    # see {Proxy Server}[rdoc-ref:Net::HTTP@Proxy+Server].
    attr_writer :proxy_address

    # Sets the proxy port;
    # see {Proxy Server}[rdoc-ref:Net::HTTP@Proxy+Server].
    attr_writer :proxy_port

    # Sets the proxy user;
    # see {Proxy Server}[rdoc-ref:Net::HTTP@Proxy+Server].
    attr_writer :proxy_user

    # Sets the proxy password;
    # see {Proxy Server}[rdoc-ref:Net::HTTP@Proxy+Server].
    attr_writer :proxy_pass
    attr_writer :proxy_use_ssl

    # Returns the IP address for the connection.
    #
    # If the session has not been started,
    # returns the value set by #ipaddr=,
    # or +nil+ if it has not been set:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.ipaddr # => nil
    #   http.ipaddr = '172.67.155.76'
    #   http.ipaddr # => "172.67.155.76"
    #
    # If the session has been started,
    # returns the IP address from the socket:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.start
    #   http.ipaddr # => "172.67.155.76"
    #   http.finish
    #
    def ipaddr
      started? ?  @socket.io.peeraddr[3] : @ipaddr
    end

    # Sets the IP address for the connection:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.ipaddr # => nil
    #   http.ipaddr = '172.67.155.76'
    #   http.ipaddr # => "172.67.155.76"
    #
    # The IP address may not be set if the session has been started.
    def ipaddr=(addr)
      raise IOError, "ipaddr value changed, but session already started" if started?
      @ipaddr = addr
    end

    # Sets or returns the numeric (\Integer or \Float) number of seconds
    # to wait for a connection to open;
    # initially 60.
    # If the connection is not made in the given interval,
    # an exception is raised.
    attr_accessor :open_timeout

    # Returns the numeric (\Integer or \Float) number of seconds
    # to wait for one block to be read (via one read(2) call);
    # see #read_timeout=.
    attr_reader :read_timeout

    # Returns the numeric (\Integer or \Float) number of seconds
    # to wait for one block to be written (via one write(2) call);
    # see #write_timeout=.
    attr_reader :write_timeout

    # Sets the maximum number of times to retry an idempotent request in case of
    # \Net::ReadTimeout, IOError, EOFError, Errno::ECONNRESET,
    # Errno::ECONNABORTED, Errno::EPIPE, OpenSSL::SSL::SSLError,
    # Timeout::Error.
    # The initial value is 1.
    #
    # Argument +retries+ must be a non-negative numeric value:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.max_retries = 2   # => 2
    #   http.max_retries       # => 2
    #
    def max_retries=(retries)
      retries = retries.to_int
      if retries < 0
        raise ArgumentError, 'max_retries should be non-negative integer number'
      end
      @max_retries = retries
    end

    # Returns the maximum number of times to retry an idempotent request;
    # see #max_retries=.
    attr_reader :max_retries

    # Sets the read timeout, in seconds, for +self+ to integer +sec+;
    # the initial value is 60.
    #
    # Argument +sec+ must be a non-negative numeric value:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.read_timeout # => 60
    #   http.get('/todos/1') # => #<Net::HTTPOK 200 OK readbody=true>
    #   http.read_timeout = 0
    #   http.get('/todos/1') # Raises Net::ReadTimeout.
    #
    def read_timeout=(sec)
      @socket.read_timeout = sec if @socket
      @read_timeout = sec
    end

    # Sets the write timeout, in seconds, for +self+ to integer +sec+;
    # the initial value is 60.
    #
    # Argument +sec+ must be a non-negative numeric value:
    #
    #   _uri = uri.dup
    #   _uri.path = '/posts'
    #   body = 'bar' * 200000
    #   data = <<EOF
    #   {"title": "foo", "body": "#{body}", "userId": "1"}
    #   EOF
    #   headers = {'content-type': 'application/json'}
    #   http = Net::HTTP.new(hostname)
    #   http.write_timeout # => 60
    #   http.post(_uri.path, data, headers)
    #   # => #<Net::HTTPCreated 201 Created readbody=true>
    #   http.write_timeout = 0
    #   http.post(_uri.path, data, headers) # Raises Net::WriteTimeout.
    #
    def write_timeout=(sec)
      @socke