-header].
# - {Request header Host}[https://en.wikipedia.org/wiki/List_of_HTTP_header_fields#host-request-header].
#
# You can add headers or override default headers:
#
#   #   res = Net::HTTP::Get.new(uri, {'foo' => '0', 'bar' => '1'})
#
# This class (and therefore its subclasses) also includes (indirectly)
# module Net::HTTPHeader, which gives access to its
# {methods for setting headers}[rdoc-ref:Net::HTTPHeader@Setters].
#
# == Request Subclasses
#
# Subclasses for HTTP requests:
#
# - Net::HTTP::Get
# - Net::HTTP::Head
# - Net::HTTP::Post
# - Net::HTTP::Put
# - Net::HTTP::Delete
# - Net::HTTP::Options
# - Net::HTTP::Trace
# - Net::HTTP::Patch
#
# Subclasses for WebDAV requests:
#
# - Net::HTTP::Propfind
# - Net::HTTP::Proppatch
# - Net::HTTP::Mkcol
# - Net::HTTP::Copy
# - Net::HTTP::Move
# - Net::HTTP::Lock
# - Net::HTTP::Unlock
#
class Net::HTTPRequest < Net::HTTPGenericRequest
  # Creates an HTTP request object for +path+.
  #
  # +initheader+ are the default headers to use.  Net::HTTP adds
  # Accept-Encoding to enable compression of the response body unless
  # Accept-Encoding or Range are supplied in +initheader+.

  def initialize(path, initheader = nil)
    super self.class::METHOD,
          self.class::REQUEST_HAS_BODY,
          self.class::RESPONSE_HAS_BODY,
          path, initheader
  end
end
                                                                                                                                                                                                           ruby/net/http/backward.rb                                                                           0000644                 00000002065 15040313424 0011374 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       e(
                    case env_table['REQUEST_METHOD']
                    when "GET", "HEAD"
                      if defined?(MOD_RUBY)
                        Apache::request.args or ""
                      else
                        env_table['QUERY_STRING'] or ""
                      end
                    when "POST"
                      stdinput.binmode if defined? stdinput.binmode
                      stdinput.read(Integer(env_table['CONTENT_LENGTH'])) or ''
                    else
                      read_from_cmdline
                    end.dup.force_encoding(@accept_charset)
                  )
        unless Encoding.find(@accept_charset) == Encoding::ASCII_8BIT
          @params.each do |key,values|
            values.each do |value|
              unless value.valid_encoding?
                if @accept_charset_error_block
                  @accept_charset_error_block.call(key,value)
                else
                  raise InvalidEncoding,"Accept-Charset encoding error"
                end
              end
            end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     ruby/net/http/proxy_delta.rb                                                                        0000644                 00000000417 15040313424 0012147 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
module Net::HTTP::ProxyDelta   #:nodoc: internal use only
  private

  def conn_address
    proxy_address()
  end

  def conn_port
    proxy_port()
  end

  def edit_path(path)
    use_ssl? ? path : "http://#{addr_port()}#{path}"
  end
end

                                                                                                                                                                                                                                                 ruby/net/http/generic_request.rb                                                                    0000644                 00000030160 15040313424 0012777 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#
# \HTTPGenericRequest is the parent of the Net::HTTPRequest class.
#
# Do not use this directly; instead, use a subclass of Net::HTTPRequest.
#
# == About the Examples
#
# :include: doc/net-http/examples.rdoc
#
class Net::HTTPGenericRequest

  include Net::HTTPHeader

  def initialize(m, reqbody, resbody, uri_or_path, initheader = nil) # :nodoc:
    @method = m
    @request_has_body = reqbody
    @response_has_body = resbody

    if URI === uri_or_path then
      raise ArgumentError, "not an HTTP URI" unless URI::HTTP === uri_or_path
      hostname = uri_or_path.hostname
      raise ArgumentError, "no host component for URI" unless (hostname && hostname.length > 0)
      @uri = uri_or_path.dup
      host = @uri.hostname.dup
      host << ":" << @uri.port.to_s if @uri.port != @uri.default_port
      @path = uri_or_path.request_uri
      raise ArgumentError, "no HTTP request path given" unless @path
    else
      @uri = nil
      host = nil
      raise ArgumentError, "no HTTP request path given" unless uri_or_path
      raise ArgumentError, "HTTP request path is empty" if uri_or_path.empty?
      @path = uri_or_path.dup
    end

    @decode_content = false

    if Net::HTTP::HAVE_ZLIB then
      if !initheader ||
         !initheader.keys.any? { |k|
           %w[accept-encoding range].include? k.downcase
         } then
        @decode_content = true if @response_has_body
        initheader = initheader ? initheader.dup : {}
        initheader["accept-encoding"] =
          "gzip;q=1.0,deflate;q=0.6,identity;q=0.3"
      end
    end

    initialize_http_header initheader
    self['Accept'] ||= '*/*'
    self['User-Agent'] ||= 'Ruby'
    self['Host'] ||= host if host
    @body = nil
    @body_stream = nil
    @body_data = nil
  end

  # Returns the string method name for the request:
  #
  #   Net::HTTP::Get.new(uri).method  # => "GET"
  #   Net::HTTP::Post.new(uri).method # => "POST"
  #
  attr_reader :method

  # Returns the string path for the request:
  #
  #   Net::HTTP::Get.new(uri).path # => "/"
  #   Net::HTTP::Post.new('example.com').path # => "example.com"
  #
  attr_reader :path

  # Returns the URI object for the request, or +nil+ if none:
  #
  #   Net::HTTP::Get.new(uri).uri
  #   # => #<URI::HTTPS https://jsonplaceholder.typicode.com/>
  #   Net::HTTP::Get.new('example.com').uri # => nil
  #
  attr_reader :uri

  # Returns +false+ if the request's header <tt>'Accept-Encoding'</tt>
  # has been set manually or deleted
  # (indicating that the user intends to handle encoding in the response),
  # +true+ otherwise:
  #
  #   req = Net::HTTP::Get.new(uri) # => #<Net::HTTP::Get GET>
  #   req['Accept-Encoding']        # => "gzip;q=1.0,deflate;q=0.6,identity;q=0.3"
  #   req.decode_content            # => true
  #   req['Accept-Encoding'] = 'foo'
  #   req.decode_content            # => false
  #   req.delete('Accept-Encoding')
  #   req.decode_content            # => false
  #
  attr_reader :decode_content

  # Returns a string representation of the request:
  #
  #   Net::HTTP::Post.new(uri).inspect # => "#<Net::HTTP::Post POST>"
  #
  def inspect
    "\#<#{self.class} #{@method}>"
  end

  ##
  # Don't automatically decode response content-encoding if the user indicates
  # they want to handle it.

  def []=(key, val) # :nodoc:
    @decode_content = false if key.downcase == 'accept-encoding'

    super key, val
  end

  # Returns whether the request may have a body:
  #
  #   Net::HTTP::Post.new(uri).request_body_permitted? # => true
  #   Net::HTTP::Get.new(uri).request_body_permitted?  # => false
  #
  def request_body_permitted?
    @request_has_body
  end

  # Returns whether the response may have a body:
  #
  #   Net::HTTP::Post.new(uri).response_body_permitted? # => true
  #   Net::HTTP::Head.new(uri).response_body_permitted? # => false
  #
  def response_body_permitted?
    @response_has_body
  end

  def body_exist? # :nodoc:
    warn "Net::HTTPRequest#body_exist? is obsolete; use response_body_permitted?", uplevel: 1 if $VERBOSE
    response_body_permitted?
  end

  # Returns the string body for the request, or +nil+ if there is none:
  #
  #   req = Net::HTTP::Post.new(uri)
  #   req.body # => nil
  #   req.body = '{"title": "foo","body": "bar","userId": 1}'
  #   req.body # => "{\"title\": \"foo\",\"body\": \"bar\",\"userId\": 1}"
  #
  attr_reader :body

  # Sets the body for the request:
  #
  #   req = Net::HTTP::Post.new(uri)
  #   req.body # => nil
  #   req.body = '{"title": "foo","body": "bar","userId": 1}'
  #   req.body # => "{\"title\": \"foo\",\"body\": \"bar\",\"userId\": 1}"
  #
  def body=(str)
    @body = str
    @body_stream = nil
    @body_data = nil
    str
  end

  # Returns the body stream object for the request, or +nil+ if there is none:
  #
  #   req = Net::HTTP::Post.new(uri)          # => #<Net::HTTP::Post POST>
  #   req.body_stream                         # => nil
  #   require 'stringio'
  #   req.body_stream = StringIO.new('xyzzy') # => #<StringIO:0x0000027d1e5affa8>
  #   req.body_stream                         # => #<StringIO:0x0000027d1e5affa8>
  #
  attr_reader :body_stream

  # Sets the body stream for the request:
  #
  #   req = Net::HTTP::Post.new(uri)          # => #<Net::HTTP::Post POST>
  #   req.body_stream                         # => nil
  #   require 'stringio'
  #   req.body_stream = StringIO.new('xyzzy') # => #<StringIO:0x0000027d1e5affa8>
  #   req.body_stream                         # => #<StringIO:0x0000027d1e5affa8>
  #
  def body_stream=(input)
    @body = nil
    @body_stream = input
    @body_data = nil
    input
  end

  def set_body_internal(str)   #:nodoc: internal use only
    raise ArgumentError, "both of body argument and HTTPRequest#body set" if str and (@body or @body_stream)
    self.body = str if str
    if @body.nil? && @body_stream.nil? && @body_data.nil? && request_body_permitted?
      self.body = ''
    end
  end

  #
  # write
  #

  def exec(sock, ver, path)   #:nodoc: internal use only
    if @body
      send_request_with_body sock, ver, path, @body
    elsif @body_stream
      send_request_with_body_stream sock, ver, path, @body_stream
    elsif @body_data
      send_request_with_body_data sock, ver, path, @body_data
    else
      write_header sock, ver, path
    end
  end

  def update_uri(addr, port, ssl) # :nodoc: internal use only
    # reflect the connection and @path to @uri
    return unless @uri

    if ssl
      scheme = 'https'
      klass = URI::HTTPS
    else
      scheme = 'http'
      klass = URI::HTTP
    end

    if host = self['host']
      host.sub!(/:.*/m, '')
    elsif host = @uri.host
    else
     host = addr
    end
    # convert the class of the URI
    if @uri.is_a?(klass)
      @uri.host = host
      @uri.port = port
    else
      @uri = klass.new(
        scheme, @uri.userinfo,
        host, port, nil,
        @uri.path, nil, @uri.query, nil)
    end
  end

  private

  class Chunker #:nodoc:
    def initialize(sock)
      @sock = sock
      @prev = nil
    end

    def write(buf)
      # avoid memcpy() of buf, buf can huge and eat memory bandwidth
      rv = buf.bytesize
      @sock.write("#{rv.to_s(16)}\r\n", buf, "\r\n")
      rv
    end

    def finish
      @sock.write("0\r\n\r\n")
    end
  end

  def send_request_with_body(sock, ver, path, body)
    self.content_length = body.bytesize
    delete 'Transfer-Encoding'
    supply_default_content_type
    write_header sock, ver, path
    wait_for_continue sock, ver if sock.continue_timeout
    sock.write body
  end

  def send_request_with_body_stream(sock, ver, path, f)
    unless content_length() or chunked?
      raise ArgumentError,
          "Content-Length not given and Transfer-Encoding is not `chunked'"
    end
    supply_default_content_type
    write_header sock, ver, path
    wait_for_continue sock, ver if sock.continue_timeout
    if chunked?
      chunker = Chunker.new(sock)
      IO.copy_stream(f, chunker)
      chunker.finish
    else
      IO.copy_stream(f, sock)
    end
  end

  def send_request_with_body_data(sock, ver, path, params)
    if /\Amultipart\/form-data\z/i !~ self.content_type
      self.content_type = 'application/x-www-form-urlencoded'
      return send_request_with_body(sock, ver, path, URI.encode_www_form(params))
    end

    opt = @form_option.dup
    require 'securerandom' unless defined?(SecureRandom)
    opt[:boundary] ||= SecureRandom.urlsafe_base64(40)
    self.set_content_type(self.content_type, boundary: opt[:boundary])
    if chunked?
      write_header sock, ver, path
      encode_multipart_form_data(sock, params, opt)
    else
      require 'tempfile'
      file = Tempfile.new('multipart')
      file.binmode
      encode_multipart_form_data(file, params, opt)
      file.rewind
      self.content_length = file.size
      write_header sock, ver, path
      IO.copy_stream(file, sock)
      file.close(true)
    end
  end

  def encode_multipart_form_data(out, params, opt)
    charset = opt[:charset]
    boundary = opt[:boundary]
    require 'securerandom' unless defined?(SecureRandom)
    boundary ||= SecureRandom.urlsafe_base64(40)
    chunked_p = chunked?

    buf = +''
    params.each do |key, value, h={}|
      key = quote_string(key, charset)
      filename =
        h.key?(:filename) ? h[:filename] :
        value.respond_to?(:to_path) ? File.basename(value.to_path) :
        nil

      buf << "--#{boundary}\r\n"
      if filename
        filename = quote_string(filename, charset)
        type = h[:content_type] || 'application/octet-stream'
        buf << "Content-Disposition: form-data; " \
          "name=\"#{key}\"; filename=\"#{filename}\"\r\n" \
          "Content-Type: #{type}\r\n\r\n"
        if !out.respond_to?(:write) || !value.respond_to?(:read)
          # if +out+ is not an IO or +value+ is not an IO
          buf << (value.respond_to?(:read) ? value.read : value)
        elsif value.respond_to?(:size) && chunked_p
          # if +out+ is an IO and +value+ is a File, use IO.copy_stream
          flush_buffer(out, buf, chunked_p)
          out << "%x\r\n" % value.size if chunked_p
          IO.copy_stream(value, out)
          out << "\r\n" if chunked_p
        else
          # +out+ is an IO, and +value+ is not a File but an IO
          flush_buffer(out, buf, chunked_p)
          1 while flush_buffer(out, value.read(4096), chunked_p)
        end
      else
        # non-file field:
        #   HTML5 says, "The parts of the generated multipart/form-data
        #   resource that correspond to non-file fields must not have a
        #   Content-Type header specified."
        buf << "Content-Disposition: form-data; name=\"#{key}\"\r\n\r\n"
        buf << (value.respond_to?(:read) ? value.read : value)
      end
      buf << "\r\n"
    end
    buf << "--#{boundary}--\r\n"
    flush_buffer(out, buf, chunked_p)
    out << "0\r\n\r\n" if chunked_p
  end

  def quote_string(str, charset)
    str = str.encode(charset, fallback:->(c){'&#%d;'%c.encode("UTF-8").ord}) if charset
    str.gsub(/[\\"]/, '\\\\\&')
  end

  def flush_buffer(out, buf, chunked_p)
    return unless buf
    out << "%x\r\n"%buf.bytesize if chunked_p
    out << buf
    out << "\r\n" if chunked_p
    buf.clear
  end

  def supply_default_content_type
    return if content_type()
    warn 'net/http: Content-Type did not set; using application/x-www-form-urlencoded', uplevel: 1 if $VERBOSE
    set_content_type 'application/x-www-form-urlencoded'
  end

  ##
  # Waits up to the continue timeout for a response from the server provided
  # we're speaking HTTP 1.1 and are expecting a 100-continue response.

  def wait_for_continue(sock, ver)
    if ver >= '1.1' and @header['expect'] and
        @header['expect'].include?('100-continue')
      if sock.io.to_io.wait_readable(sock.continue_timeout)
        res = Net::HTTPResponse.read_new(sock)
        unless res.kind_of?(Net::HTTPContinue)
          res.decode_content = @decode_content
          throw :response, res
        end
      end
    end
  end

  def write_header(sock, ver, path)
    reqline = "#{@method} #{path} HTTP/#{ver}"
    if /[\r\n]/ =~ reqline
      raise ArgumentError, "A Request-Line must not contain CR or LF"
    end
    buf = +''
    buf << reqline << "\r\n"
    each_capitalized do |k,v|
      buf << "#{k}: #{v}\r\n"
    end
    buf << "\r\n"
    sock.write buf
  end

end

                                                                                                                                                                                                                                                                                                                                                                                                                ruby/net/http/requests.rb                                                                           0000644                 00000031760 15040313424 0011475 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

# HTTP/1.1 methods --- RFC2616

# \Class for representing
# {HTTP method GET}[https://en.wikipedia.org/w/index.php?title=Hypertext_Transfer_Protocol#GET_method]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Get.new(uri) # => #<Net::HTTP::Get GET>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Properties:
#
# - Request body: optional.
# - Response body: yes.
# - {Safe}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods]: yes.
# - {Idempotent}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Idempotent_methods]: yes.
# - {Cacheable}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Cacheable_methods]: yes.
#
# Related:
#
# - Net::HTTP.get: sends +GET+ request, returns response body.
# - Net::HTTP#get: sends +GET+ request, returns response object.
#
class Net::HTTP::Get < Net::HTTPRequest
  METHOD = 'GET'
  REQUEST_HAS_BODY  = false
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {HTTP method HEAD}[https://en.wikipedia.org/w/index.php?title=Hypertext_Transfer_Protocol#HEAD_method]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Head.new(uri) # => #<Net::HTTP::Head HEAD>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Properties:
#
# - Request body: optional.
# - Response body: no.
# - {Safe}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods]: yes.
# - {Idempotent}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Idempotent_methods]: yes.
# - {Cacheable}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Cacheable_methods]: yes.
#
# Related:
#
# - Net::HTTP#head: sends +HEAD+ request, returns response object.
#
class Net::HTTP::Head < Net::HTTPRequest
  METHOD = 'HEAD'
  REQUEST_HAS_BODY = false
  RESPONSE_HAS_BODY = false
end

# \Class for representing
# {HTTP method POST}[https://en.wikipedia.org/w/index.php?title=Hypertext_Transfer_Protocol#POST_method]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   uri.path = '/posts'
#   req = Net::HTTP::Post.new(uri) # => #<Net::HTTP::Post POST>
#   req.body = '{"title": "foo","body": "bar","userId": 1}'
#   req.content_type = 'application/json'
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Properties:
#
# - Request body: yes.
# - Response body: yes.
# - {Safe}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods]: no.
# - {Idempotent}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Idempotent_methods]: no.
# - {Cacheable}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Cacheable_methods]: yes.
#
# Related:
#
# - Net::HTTP.post: sends +POST+ request, returns response object.
# - Net::HTTP#post: sends +POST+ request, returns response object.
#
class Net::HTTP::Post < Net::HTTPRequest
  METHOD = 'POST'
  REQUEST_HAS_BODY = true
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {HTTP method PUT}[https://en.wikipedia.org/w/index.php?title=Hypertext_Transfer_Protocol#PUT_method]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   uri.path = '/posts'
#   req = Net::HTTP::Put.new(uri) # => #<Net::HTTP::Put PUT>
#   req.body = '{"title": "foo","body": "bar","userId": 1}'
#   req.content_type = 'application/json'
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Properties:
#
# - Request body: yes.
# - Response body: yes.
# - {Safe}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods]: no.
# - {Idempotent}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Idempotent_methods]: yes.
# - {Cacheable}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Cacheable_methods]: no.
#
# Related:
#
# - Net::HTTP.put: sends +PUT+ request, returns response object.
# - Net::HTTP#put: sends +PUT+ request, returns response object.
#
class Net::HTTP::Put < Net::HTTPRequest
  METHOD = 'PUT'
  REQUEST_HAS_BODY = true
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {HTTP method DELETE}[https://en.wikipedia.org/w/index.php?title=Hypertext_Transfer_Protocol#DELETE_method]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   uri.path = '/posts/1'
#   req = Net::HTTP::Delete.new(uri) # => #<Net::HTTP::Delete DELETE>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Properties:
#
# - Request body: optional.
# - Response body: yes.
# - {Safe}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods]: no.
# - {Idempotent}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Idempotent_methods]: yes.
# - {Cacheable}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Cacheable_methods]: no.
#
# Related:
#
# - Net::HTTP#delete: sends +DELETE+ request, returns response object.
#
class Net::HTTP::Delete < Net::HTTPRequest
  METHOD = 'DELETE'
  REQUEST_HAS_BODY = false
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {HTTP method OPTIONS}[https://en.wikipedia.org/w/index.php?title=Hypertext_Transfer_Protocol#OPTIONS_method]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Options.new(uri) # => #<Net::HTTP::Options OPTIONS>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Properties:
#
# - Request body: optional.
# - Response body: yes.
# - {Safe}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods]: yes.
# - {Idempotent}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Idempotent_methods]: yes.
# - {Cacheable}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Cacheable_methods]: no.
#
# Related:
#
# - Net::HTTP#options: sends +OPTIONS+ request, returns response object.
#
class Net::HTTP::Options < Net::HTTPRequest
  METHOD = 'OPTIONS'
  REQUEST_HAS_BODY = false
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {HTTP method TRACE}[https://en.wikipedia.org/w/index.php?title=Hypertext_Transfer_Protocol#TRACE_method]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Trace.new(uri) # => #<Net::HTTP::Trace TRACE>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Properties:
#
# - Request body: no.
# - Response body: yes.
# - {Safe}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods]: yes.
# - {Idempotent}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Idempotent_methods]: yes.
# - {Cacheable}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Cacheable_methods]: no.
#
# Related:
#
# - Net::HTTP#trace: sends +TRACE+ request, returns response object.
#
class Net::HTTP::Trace < Net::HTTPRequest
  METHOD = 'TRACE'
  REQUEST_HAS_BODY = false
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {HTTP method PATCH}[https://en.wikipedia.org/w/index.php?title=Hypertext_Transfer_Protocol#PATCH_method]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   uri.path = '/posts'
#   req = Net::HTTP::Patch.new(uri) # => #<Net::HTTP::Patch PATCH>
#   req.body = '{"title": "foo","body": "bar","userId": 1}'
#   req.content_type = 'application/json'
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Properties:
#
# - Request body: yes.
# - Response body: yes.
# - {Safe}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods]: no.
# - {Idempotent}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Idempotent_methods]: no.
# - {Cacheable}[https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Cacheable_methods]: no.
#
# Related:
#
# - Net::HTTP#patch: sends +PATCH+ request, returns response object.
#
class Net::HTTP::Patch < Net::HTTPRequest
  METHOD = 'PATCH'
  REQUEST_HAS_BODY = true
  RESPONSE_HAS_BODY = true
end

#
# WebDAV methods --- RFC2518
#

# \Class for representing
# {WebDAV method PROPFIND}[http://www.webdav.org/specs/rfc4918.html#METHOD_PROPFIND]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Propfind.new(uri) # => #<Net::HTTP::Propfind PROPFIND>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Related:
#
# - Net::HTTP#propfind: sends +PROPFIND+ request, returns response object.
#
class Net::HTTP::Propfind < Net::HTTPRequest
  METHOD = 'PROPFIND'
  REQUEST_HAS_BODY = true
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {WebDAV method PROPPATCH}[http://www.webdav.org/specs/rfc4918.html#METHOD_PROPPATCH]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Proppatch.new(uri) # => #<Net::HTTP::Proppatch PROPPATCH>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Related:
#
# - Net::HTTP#proppatch: sends +PROPPATCH+ request, returns response object.
#
class Net::HTTP::Proppatch < Net::HTTPRequest
  METHOD = 'PROPPATCH'
  REQUEST_HAS_BODY = true
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {WebDAV method MKCOL}[http://www.webdav.org/specs/rfc4918.html#METHOD_MKCOL]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Mkcol.new(uri) # => #<Net::HTTP::Mkcol MKCOL>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Related:
#
# - Net::HTTP#mkcol: sends +MKCOL+ request, returns response object.
#
class Net::HTTP::Mkcol < Net::HTTPRequest
  METHOD = 'MKCOL'
  REQUEST_HAS_BODY = true
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {WebDAV method COPY}[http://www.webdav.org/specs/rfc4918.html#METHOD_COPY]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Copy.new(uri) # => #<Net::HTTP::Copy COPY>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Related:
#
# - Net::HTTP#copy: sends +COPY+ request, returns response object.
#
class Net::HTTP::Copy < Net::HTTPRequest
  METHOD = 'COPY'
  REQUEST_HAS_BODY = false
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {WebDAV method MOVE}[http://www.webdav.org/specs/rfc4918.html#METHOD_MOVE]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Move.new(uri) # => #<Net::HTTP::Move MOVE>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Related:
#
# - Net::HTTP#move: sends +MOVE+ request, returns response object.
#
class Net::HTTP::Move < Net::HTTPRequest
  METHOD = 'MOVE'
  REQUEST_HAS_BODY = false
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {WebDAV method LOCK}[http://www.webdav.org/specs/rfc4918.html#METHOD_LOCK]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Lock.new(uri) # => #<Net::HTTP::Lock LOCK>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Related:
#
# - Net::HTTP#lock: sends +LOCK+ request, returns response object.
#
class Net::HTTP::Lock < Net::HTTPRequest
  METHOD = 'LOCK'
  REQUEST_HAS_BODY = true
  RESPONSE_HAS_BODY = true
end

# \Class for representing
# {WebDAV method UNLOCK}[http://www.webdav.org/specs/rfc4918.html#METHOD_UNLOCK]:
#
#   require 'net/http'
#   uri = URI('http://example.com')
#   hostname = uri.hostname # => "example.com"
#   req = Net::HTTP::Unlock.new(uri) # => #<Net::HTTP::Unlock UNLOCK>
#   res = Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end
#
# See {Request Headers}[rdoc-ref:Net::HTTPRequest@Request+Headers].
#
# Related:
#
# - Net::HTTP#unlock: sends +UNLOCK+ request, returns response object.
#
class Net::HTTP::Unlock < Net::HTTPRequest
  METHOD = 'UNLOCK'
  REQUEST_HAS_BODY = true
  RESPONSE_HAS_BODY = true
end

                ruby/net/http/status.rb                                                                             0000644                 00000004373 15040313424 0011145 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative '../http'

if $0 == __FILE__
  require 'open-uri'
  File.foreach(__FILE__) do |line|
    puts line
    break if line.start_with?('end')
  end
  puts
  puts "Net::HTTP::STATUS_CODES = {"
  url = "https://www.iana.org/assignments/http-status-codes/http-status-codes-1.csv"
  URI(url).read.each_line do |line|
    code, mes, = line.split(',')
    next if ['(Unused)', 'Unassigned', 'Description'].include?(mes)
    puts "  #{code} => '#{mes}',"
  end
  puts "} # :nodoc