equest(Put.new(path, initheader), data)
    end

    # Sends a PROPPATCH request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Proppatch object
    # created from string +path+, string +body+, and initial headers hash +initheader+.
    #
    #   data = '{"userId": 1, "id": 1, "title": "delectus aut autem", "completed": false}'
    #   http = Net::HTTP.new(hostname)
    #   http.proppatch('/todos/1', data)
    #
    def proppatch(path, body, initheader = nil)
      request(Proppatch.new(path, initheader), body)
    end

    # Sends a LOCK request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Lock object
    # created from string +path+, string +body+, and initial headers hash +initheader+.
    #
    #   data = '{"userId": 1, "id": 1, "title": "delectus aut autem", "completed": false}'
    #   http = Net::HTTP.new(hostname)
    #   http.lock('/todos/1', data)
    #
    def lock(path, body, initheader = nil)
      request(Lock.new(path, initheader), body)
    end

    # Sends an UNLOCK request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Unlock object
    # created from string +path+, string +body+, and initial headers hash +initheader+.
    #
    #   data = '{"userId": 1, "id": 1, "title": "delectus aut autem", "completed": false}'
    #   http = Net::HTTP.new(hostname)
    #   http.unlock('/todos/1', data)
    #
    def unlock(path, body, initheader = nil)
      request(Unlock.new(path, initheader), body)
    end

    # Sends an Options request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Options object
    # created from string +path+ and initial headers hash +initheader+.
    #
    #   http = Net::HTTP.new(hostname)
    #   http.options('/')
    #
    def options(path, initheader = nil)
      request(Options.new(path, initheader))
    end

    # Sends a PROPFIND request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Propfind object
    # created from string +path+, string +body+, and initial headers hash +initheader+.
    #
    #   data = '{"userId": 1, "id": 1, "title": "delectus aut autem", "completed": false}'
    #   http = Net::HTTP.new(hostname)
    #   http.propfind('/todos/1', data)
    #
    def propfind(path, body = nil, initheader = {'Depth' => '0'})
      request(Propfind.new(path, initheader), body)
    end

    # Sends a DELETE request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Delete object
    # created from string +path+ and initial headers hash +initheader+.
    #
    #   http = Net::HTTP.new(hostname)
    #   http.delete('/todos/1')
    #
    def delete(path, initheader = {'Depth' => 'Infinity'})
      request(Delete.new(path, initheader))
    end

    # Sends a MOVE request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Move object
    # created from string +path+ and initial headers hash +initheader+.
    #
    #   http = Net::HTTP.new(hostname)
    #   http.move('/todos/1')
    #
    def move(path, initheader = nil)
      request(Move.new(path, initheader))
    end

    # Sends a COPY request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Copy object
    # created from string +path+ and initial headers hash +initheader+.
    #
    #   http = Net::HTTP.new(hostname)
    #   http.copy('/todos/1')
    #
    def copy(path, initheader = nil)
      request(Copy.new(path, initheader))
    end

    # Sends a MKCOL request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Mkcol object
    # created from string +path+, string +body+, and initial headers hash +initheader+.
    #
    #   data = '{"userId": 1, "id": 1, "title": "delectus aut autem", "completed": false}'
    #   http.mkcol('/todos/1', data)
    #   http = Net::HTTP.new(hostname)
    #
    def mkcol(path, body = nil, initheader = nil)
      request(Mkcol.new(path, initheader), body)
    end

    # Sends a TRACE request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Trace object
    # created from string +path+ and initial headers hash +initheader+.
    #
    #   http = Net::HTTP.new(hostname)
    #   http.trace('/todos/1')
    #
    def trace(path, initheader = nil)
      request(Trace.new(path, initheader))
    end

    # Sends a GET request to the server;
    # forms the response into a Net::HTTPResponse object.
    #
    # The request is based on the Net::HTTP::Get object
    # created from string +path+ and initial headers hash +initheader+.
    #
    # With no block given, returns the response object:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.request_get('/todos') # => #<Net::HTTPOK 200 OK readbody=true>
    #
    # With a block given, calls the block with the response object
    # and returns the response object:
    #
    #   http.request_get('/todos') do |res|
    #     p res
    #   end # => #<Net::HTTPOK 200 OK readbody=true>
    #
    # Output:
    #
    #   #<Net::HTTPOK 200 OK readbody=false>
    #
    def request_get(path, initheader = nil, &block) # :yield: +response+
      request(Get.new(path, initheader), &block)
    end

    # Sends a HEAD request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Head object
    # created from string +path+ and initial headers hash +initheader+.
    #
    #   http = Net::HTTP.new(hostname)
    #   http.head('/todos/1') # => #<Net::HTTPOK 200 OK readbody=true>
    #
    def request_head(path, initheader = nil, &block)
      request(Head.new(path, initheader), &block)
    end

    # Sends a POST request to the server;
    # forms the response into a Net::HTTPResponse object.
    #
    # The request is based on the Net::HTTP::Post object
    # created from string +path+, string +data+, and initial headers hash +initheader+.
    #
    # With no block given, returns the response object:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.post('/todos', 'xyzzy')
    #   # => #<Net::HTTPCreated 201 Created readbody=true>
    #
    # With a block given, calls the block with the response body
    # and returns the response object:
    #
    #   http.post('/todos', 'xyzzy') do |res|
    #     p res
    #   end # => #<Net::HTTPCreated 201 Created readbody=true>
    #
    # Output:
    #
    #   "{\n  \"xyzzy\": \"\",\n  \"id\": 201\n}"
    #
    def request_post(path, data, initheader = nil, &block) # :yield: +response+
      request Post.new(path, initheader), data, &block
    end

    # Sends a PUT request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTP::Put object
    # created from string +path+, string +data+, and initial headers hash +initheader+.
    #
    #   http = Net::HTTP.new(hostname)
    #   http.put('/todos/1', 'xyzzy')
    #   # => #<Net::HTTPOK 200 OK readbody=true>
    #
    def request_put(path, data, initheader = nil, &block)   #:nodoc:
      request Put.new(path, initheader), data, &block
    end

    alias get2   request_get    #:nodoc: obsolete
    alias head2  request_head   #:nodoc: obsolete
    alias post2  request_post   #:nodoc: obsolete
    alias put2   request_put    #:nodoc: obsolete

    # Sends an \HTTP request to the server;
    # returns an instance of a subclass of Net::HTTPResponse.
    #
    # The request is based on the Net::HTTPRequest object
    # created from string +path+, string +data+, and initial headers hash +header+.
    # That object is an instance of the
    # {subclass of Net::HTTPRequest}[rdoc-ref:Net::HTTPRequest@Request+Subclasses],
    # that corresponds to the given uppercase string +name+,
    # which must be
    # an {HTTP request method}[https://en.wikipedia.org/wiki/HTTP#Request_methods]
    # or a {WebDAV request method}[https://en.wikipedia.org/wiki/WebDAV#Implementation].
    #
    # Examples:
    #
    #   http = Net::HTTP.new(hostname)
    #   http.send_request('GET', '/todos/1')
    #   # => #<Net::HTTPOK 200 OK readbody=true>
    #   http.send_request('POST', '/todos', 'xyzzy')
    #   # => #<Net::HTTPCreated 201 Created readbody=true>
    #
    def send_request(name, path, data = nil, header = nil)
      has_response_body = name != 'HEAD'
      r = HTTPGenericRequest.new(name,(data ? true : false),has_response_body,path,header)
      request r, data
    end

    # Sends the given request +req+ to the server;
    # forms the response into a Net::HTTPResponse object.
    #
    # The given +req+ must be an instance of a
    # {subclass of Net::HTTPRequest}[rdoc-ref:Net::HTTPRequest@Request+Subclasses].
    # Argument +body+ should be given only if needed for the request.
    #
    # With no block given, returns the response object:
    #
    #   http = Net::HTTP.new(hostname)
    #
    #   req = Net::HTTP::Get.new('/todos/1')
    #   http.request(req)
    #   # => #<Net::HTTPOK 200 OK readbody=true>
    #
    #   req = Net::HTTP::Post.new('/todos')
    #   http.request(req, 'xyzzy')
    #   # => #<Net::HTTPCreated 201 Created readbody=true>
    #
    # With a block given, calls the block with the response and returns the response:
    #
    #   req = Net::HTTP::Get.new('/todos/1')
    #   http.request(req) do |res|
    #     p res
    #   end # => #<Net::HTTPOK 200 OK readbody=true>
    #
    # Output:
    #
    #   #<Net::HTTPOK 200 OK readbody=false>
    #
    def request(req, body = nil, &block)  # :yield: +response+
      unless started?
        start {
          req['connection'] ||= 'close'
          return request(req, body, &block)
        }
      end
      if proxy_user()
        req.proxy_basic_auth proxy_user(), proxy_pass() unless use_ssl?
      end
      req.set_body_internal body
      res = transport_request(req, &block)
      if sspi_auth?(res)
        sspi_auth(req)
        res = transport_request(req, &block)
      end
      res
    end

    private

    # Executes a request which uses a representation
    # and returns its body.
    def send_entity(path, data, initheader, dest, type, &block)
      res = nil
      request(type.new(path, initheader), data) {|r|
        r.read_body dest, &block
        res = r
      }
      res
    end

    IDEMPOTENT_METHODS_ = %w/GET HEAD PUT DELETE OPTIONS TRACE/ # :nodoc:

    def transport_request(req)
      count = 0
      begin
        begin_transport req
        res = catch(:response) {
          begin
            req.exec @socket, @curr_http_version, edit_path(req.path)
          rescue Errno::EPIPE
            # Failure when writing full request, but we can probably
            # still read the received response.
          end

          begin
            res = HTTPResponse.read_new(@socket)
            res.decode_content = req.decode_content
            res.body_encoding = @response_body_encoding
            res.ignore_eof = @ignore_eof
          end while res.kind_of?(HTTPInformation)

          res.uri = req.uri

          res
        }
        res.reading_body(@socket, req.response_body_permitted?) {
          if block_given?
            count = max_retries # Don't restart in the middle of a download
            yield res
          end
        }
      rescue Net::OpenTimeout
        raise
      rescue Net::ReadTimeout, IOError, EOFError,
             Errno::ECONNRESET, Errno::ECONNABORTED, Errno::EPIPE, Errno::ETIMEDOUT,
             # avoid a dependency on OpenSSL
             defined?(OpenSSL::SSL) ? OpenSSL::SSL::SSLError : IOError,
             Timeout::Error => exception
        if count < max_retries && IDEMPOTENT_METHODS_.include?(req.method)
          count += 1
          @socket.close if @socket
          debug "Conn close because of error #{exception}, and retry"
          retry
        end
        debug "Conn close because of error #{exception}"
        @socket.close if @socket
        raise
      end

      end_transport req, res
      res
    rescue => exception
      debug "Conn close because of error #{exception}"
      @socket.close if @socket
      raise exception
    end

    def begin_transport(req)
      if @socket.closed?
        connect
      elsif @last_communicated
        if @last_communicated + @keep_alive_timeout < Process.clock_gettime(Process::CLOCK_MONOTONIC)
          debug 'Conn close because of keep_alive_timeout'
          @socket.close
          connect
        elsif @socket.io.to_io.wait_readable(0) && @socket.eof?
          debug "Conn close because of EOF"
          @socket.close
          connect
        end
      end

      if not req.respo