t support.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/415].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-415-unsupported-media-type].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#415].
  #
  class HTTPUnsupportedMediaType < HTTPClientError
    HAS_BODY = true
  end

  # Response class for <tt>Range Not Satisfiable</tt> responses (status code 416).
  #
  # The request entity has a media type which the server or resource does not support.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/416].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-416-range-not-satisfiable].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#416].
  #
  class HTTPRangeNotSatisfiable < HTTPClientError
    HAS_BODY = true
  end
  HTTPRequestedRangeNotSatisfiable = HTTPRangeNotSatisfiable

  # Response class for <tt>Expectation Failed</tt> responses (status code 417).
  #
  # The server cannot meet the requirements of the Expect request-header field.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/417].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-417-expectation-failed].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#417].
  #
  class HTTPExpectationFailed < HTTPClientError
    HAS_BODY = true
  end

  # 418 I'm a teapot - RFC 2324; a joke RFC
  # See https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#418.

  # 420 Enhance Your Calm - Twitter

  # Response class for <tt>Misdirected Request</tt> responses (status code 421).
  #
  # The request was directed at a server that is not able to produce a response.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-421-misdirected-request].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#421].
  #
  class HTTPMisdirectedRequest < HTTPClientError
    HAS_BODY = true
  end

  # Response class for <tt>Unprocessable Entity</tt> responses (status code 422).
  #
  # The request was well-formed but had semantic errors.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/422].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-422-unprocessable-content].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#422].
  #
  class HTTPUnprocessableEntity < HTTPClientError
    HAS_BODY = true
  end

  # Response class for <tt>Locked (WebDAV)</tt> responses (status code 423).
  #
  # The requested resource is locked.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {RFC 4918}[https://www.rfc-editor.org/rfc/rfc4918#section-11.3].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#423].
  #
  class HTTPLocked < HTTPClientError
    HAS_BODY = true
  end

  # Response class for <tt>Failed Dependency (WebDAV)</tt> responses (status code 424).
  #
  # The request failed because it depended on another request and that request failed.
  # See {424 Failed Dependency (WebDAV)}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#424].
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {RFC 4918}[https://www.rfc-editor.org/rfc/rfc4918#section-11.4].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#424].
  #
  class HTTPFailedDependency < HTTPClientError
    HAS_BODY = true
  end

  # 425 Too Early
  # https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#425.

  # Response class for <tt>Upgrade Required</tt> responses (status code 426).
  #
  # The client should switch to the protocol given in the Upgrade header field.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/426].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-426-upgrade-required].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#426].
  #
  class HTTPUpgradeRequired < HTTPClientError
    HAS_BODY = true
  end

  # Response class for <tt>Precondition Required</tt> responses (status code 428).
  #
  # The origin server requires the request to be conditional.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/428].
  # - {RFC 6585}[https://www.rfc-editor.org/rfc/rfc6585#section-3].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#428].
  #
  class HTTPPreconditionRequired < HTTPClientError
    HAS_BODY = true
  end

  # Response class for <tt>Too Many Requests</tt> responses (status code 429).
  #
  # The user has sent too many requests in a given amount of time.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/429].
  # - {RFC 6585}[https://www.rfc-editor.org/rfc/rfc6585#section-4].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#429].
  #
  class HTTPTooManyRequests < HTTPClientError
    HAS_BODY = true
  end

  # Response class for <tt>Request Header Fields Too Large</tt> responses (status code 431).
  #
  # An individual header field is too large,
  # or all the header fields collectively, are too large.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/431].
  # - {RFC 6585}[https://www.rfc-editor.org/rfc/rfc6585#section-5].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#431].
  #
  class HTTPRequestHeaderFieldsTooLarge < HTTPClientError
    HAS_BODY = true
  end

  # Response class for <tt>Unavailable For Legal Reasons</tt> responses (status code 451).
  #
  # A server operator has received a legal demand to deny access to a resource or to a set of resources
  # that includes the requested resource.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/451].
  # - {RFC 7725}[https://www.rfc-editor.org/rfc/rfc7725.html#section-3].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#451].
  #
  class HTTPUnavailableForLegalReasons < HTTPClientError
    HAS_BODY = true
  end
  # 444 No Response - Nginx
  # 449 Retry With - Microsoft
  # 450 Blocked by Windows Parental Controls - Microsoft
  # 499 Client Closed Request - Nginx

  # Response class for <tt>Internal Server Error</tt> responses (status code 500).
  #
  # An unexpected condition was encountered and no more specific message is suitable.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/500].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-500-internal-server-error].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#500].
  #
  class HTTPInternalServerError < HTTPServerError
    HAS_BODY = true
  end

  # Response class for <tt>Not Implemented</tt> responses (status code 501).
  #
  # The server either does not recognize the request method,
  # or it lacks the ability to fulfil the request.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/501].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-501-not-implemented].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#501].
  #
  class HTTPNotImplemented < HTTPServerError
    HAS_BODY = true
  end

  # Response class for <tt>Bad Gateway</tt> responses (status code 502).
  #
  # The server was acting as a gateway or proxy
  # and received an invalid response from the upstream server.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/502].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-502-bad-gateway].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#502].
  #
  class HTTPBadGateway < HTTPServerError
    HAS_BODY = true
  end

  # Response class for <tt>Service Unavailable</tt> responses (status code 503).
  #
  # The server cannot handle the request
  # (because it is overloaded or down for maintenance).
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/503].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-503-service-unavailable].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#503].
  #
  class HTTPServiceUnavailable < HTTPServerError
    HAS_BODY = true
  end

  # Response class for <tt>Gateway Timeout</tt> responses (status code 504).
  #
  # The server was acting as a gateway or proxy
  # and did not receive a timely response from the upstream server.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/504].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-504-gateway-timeout].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#504].
  #
  class HTTPGatewayTimeout < HTTPServerError
    HAS_BODY = true
  end
  HTTPGatewayTimeOut = HTTPGatewayTimeout

  # Response class for <tt>HTTP Version Not Supported</tt> responses (status code 505).
  #
  # The server does not support the HTTP version used in the request.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/505].
  # - {RFC 9110}[https://www.rfc-editor.org/rfc/rfc9110.html#name-505-http-version-not-suppor].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#505].
  #
  class HTTPVersionNotSupported < HTTPServerError
    HAS_BODY = true
  end

  # Response class for <tt>Variant Also Negotiates</tt> responses (status code 506).
  #
  # Transparent content negotiation for the request results in a circular reference.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/506].
  # - {RFC 2295}[https://www.rfc-editor.org/rfc/rfc2295#section-8.1].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#506].
  #
  class HTTPVariantAlsoNegotiates < HTTPServerError
    HAS_BODY = true
  end

  # Response class for <tt>Insufficient Storage (WebDAV)</tt> responses (status code 507).
  #
  # The server is unable to store the representation needed to complete the request.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/507].
  # - {RFC 4918}[https://www.rfc-editor.org/rfc/rfc4918#section-11.5].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#507].
  #
  class HTTPInsufficientStorage < HTTPServerError
    HAS_BODY = true
  end

  # Response class for <tt>Loop Detected (WebDAV)</tt> responses (status code 508).
  #
  # The server detected an infinite loop while processing the request.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/508].
  # - {RFC 5942}[https://www.rfc-editor.org/rfc/rfc5842.html#section-7.2].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#508].
  #
  class HTTPLoopDetected < HTTPServerError
    HAS_BODY = true
  end
  # 509 Bandwidth Limit Exceeded - Apache bw/limited extension

  # Response class for <tt>Not Extended</tt> responses (status code 510).
  #
  # Further extensions to the request are required for the server to fulfill it.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/510].
  # - {RFC 2774}[https://www.rfc-editor.org/rfc/rfc2774.html#section-7].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#510].
  #
  class HTTPNotExtended < HTTPServerError
    HAS_BODY = true
  end

  # Response class for <tt>Network Authentication Required</tt> responses (status code 511).
  #
  # The client needs to authenticate to gain network access.
  #
  # :include: doc/net-http/included_getters.rdoc
  #
  # References:
  #
  # - {Mozilla}[https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/511].
  # - {RFC 6585}[https://www.rfc-editor.org/rfc/rfc6585#section-6].
  # - {Wikipedia}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes#511].
  #
  class HTTPNetworkAuthenticationRequired < HTTPServerError
    HAS_BODY = true
  end

end

class Net::HTTPResponse
  CODE_CLASS_TO_OBJ = {
    '1' => Net::HTTPInformation,
    '2' => Net::HTTPSuccess,
    '3' => Net::HTTPRedirection,
    '4' => Net::HTTPClientError,
    '5' => Net::HTTPServerError
  }
  CODE_TO_OBJ = {
    '100' => Net::HTTPContinue,
    '101' => Net::HTTPSwitchProtocol,
    '102' => Net::HTTPProcessing,
    '103' => Net::HTTPEarlyHints,

    '200' => Net::HTTPOK,
    '201' => Net::HTTPCreated,
    '202' => Net::HTTPAccepted,
    '203' => Net::HTTPNonAuthoritativeInformation,
    '204' => Net::HTTPNoContent,
    '205' => Net::HTTPResetContent,
    '206' => Net::HTTPPartialContent,
    '207' => Net::HTTPMultiStatus,
    '208' => Net::HTTPAlreadyReported,
    '226' => Net::HTTPIMUsed,

    '300' => Net::HTTPMultipleChoices,
    '301' => Net::HTTPMovedPermanently,
    '302' => Net::HTTPFound,
    '303' => Net::HTTPSeeOther,
    '304' => Net::HTTPNotModified,
    '305' => Net::HTTPUseProxy,
    '307' => Net::HTTPTemporaryRedirect,
    '308' => Net::HTTPPermanentRedirect,

    '400' => Net::HTTPBadRequest,
    '401' => Net::HTTPUnauthorized,
    '402' => Net::HTTPPaymentRequired,
    '403' => Net::HTTPForbidden,
    '404' => Net::HTTPNotFound,
    '405' => Net::HTTPMethodNotAllowed,
    '406' => Net::HTTPNotAcceptable,
    '407' => Net::HTTPProxyAuthenticationRequired,
    '408' => Net::HTTPRequestTimeout,
    '409' => Net::HTTPConflict,
    '410' => Net::HTTPGone,
    '411' => Net::HTTPLengthRequired,
    '412' => Net::HTTPPreconditionFailed,
    '413' => Net::HTTPPayloadTooLarge,
    '414' => Net::HTTPURITooLong,
    '415' => Net::HTTPUnsupportedMediaType,
    '416' => Net::HTTPRangeNotSatisfiable,
    '417' => Net::HTTPExpectationFailed,
    '421' => Net::HTTPMisdirectedRequest,
    '422' => Net::HTTPUnprocessableEntity,
    '423' => Net::HTTPLocked,
    '424' => Net::HTTPFailedDependency,
    '426' => Net::HTTPUpgradeRequired,
    '428' => Net::HTTPPreconditionRequired,
    '429' => Net::HTTPTooManyRequests,
    '431' => Net::HTTPRequestHeaderFieldsTooLarge,
    '451' => Net::HTTPUnavailableForLegalReasons,

    '500' => Net::HTTPInternalServerError,
    '501' => Net::HTTPNotImplemented,
    '502' => Net::HTTPBadGateway,
    '503' => Net::HTTPServiceUnavailable,
    '504' => Net::HTTPGatewayTimeout,
    '505' => Net::HTTPVersionNotSupported,
    '506' => Net::HTTPVariantAlsoNegotiates,
    '507' => Net::HTTPInsufficientStorage,
    '508' => Net::HTTPLoopDetected,
    '510' => Net::HTTPNotExtended,
    '511' => Net::HTTPNetworkAuthenticationRequired,
  }
end
   ruby/net/http/response.rb                                                                           0000644                 00000046173 15040313424 0011464 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

# This class is the base class for \Net::HTTP response classes.
#
# == About the Examples
#
# :include: doc/net-http/examples.rdoc
#
# == Returned Responses
#
# \Method Net::HTTP.get_response returns
# an instance of one of the subclasses of \Net::HTTPResponse:
#
#   Net::HTTP.get_response(uri)
#   # => #<Net::HTTPOK 200 OK readbody=true>
#   Net::HTTP.get_response(hostname, '/nosuch')
#   # => #<Net::HTTPNotFound 404 Not Found readbody=true>
#
# As does method Net::HTTP#request:
#
#   req = Net::HTTP::Get.new(uri)
#   Net::HTTP.start(hostname) do |http|
#     http.request(req)
#   end # => #<Net::HTTPOK 200 OK readbody=true>
#
# \Class \Net::HTTPResponse includes module Net::HTTPHeader,
# which provides access to response header values via (among others):
#
# - \Hash-like method <tt>[]</tt>.
# - Specific reader methods, such as +content_type+.
#
# Examples:
#
#   res = Net::HTTP.get_response(uri) # => #<Net::HTTPOK 200 OK readbody=true>
#   res['Content-Type']               # => "text/html; charset=UTF-8"
#   res.content_type                  # => "text/html"
#
# == Response Subclasses
#
# \Class \Net::HTTPResponse has a subclass for each
# {HTTP status code}[https://en.wikipedia.org/wiki/List_of_HTTP_status_codes].
# You can look up the response class for a given code:
#
#   Net::HTTPResponse::CODE_TO_OBJ['200'] # => Net::HTTPOK
#   Net::HTTPResponse::CODE_TO_OBJ['400'] # => Net::HTTPBadRequest
#   Net::HTTPResponse::CODE_TO_OBJ['404'] # => Net::HTTPNotFound
#
# And you can retrieve the status code for a response object:
#
#   Net::HTTP.get_response(uri).code                 # => "200"
#   Net::HTTP.get_response(hostname, '/nosuch').code # => "404"
#
# The response subclasses (indentation shows class hierarchy):
#
# - Net::HTTPUnknownResponse (for unhandled \HTTP extensions).
#
# - Net::HTTPInformation:
#
#   - Net::HTTPContinue (100)
#   - Net::HTTPSwitchProtocol (101)
#   - Net::HTTPProcessing (102)
#   - Net::HTTPEarlyHints (103)
#
# - Net::HTTPSuccess:
#
#   - Net::HTTPOK (200)
#   - Net::HTTPCreated (201)
#   - Net::HTTPAccepted (202)
#   - Net::HTTPNonAuthoritativeInformation (203)
#   - Net::HTTPNoContent (204)
#   - Net::HTTPResetContent (205)
#   - Net::HTTPPartialContent (206)
#   - Net::HTTPMultiStatus (207)
#   - Net::HTTPAlreadyReported (208)
#   - Net::HTTPIMUsed (226)
#
# - Net::HTTPRedirection:
#
#   - Net::HTTPMultipleChoices (300)
#   - Net::HTTPMovedPermanently (301)
#   - Net::HTTPFound (302)
#   - Net::HTTPSeeOther (303)
#   - Net::HTTPNotModified (304)
#   - Net::HTTPUseProxy (305)
#   - Net::HTTPTemporaryRedirect (307)
#   - Net::HTTPPermanentRedirect (308)
#
# - Net::HTTPClientError:
#
#   - Net::HTTPBadRequest (400)
#   - Net::HTTPUnauthorized (401)
#   - Net::HTTPPaymentRequired (402)
#   - Net::HTTPForbidden (403)
#   - Net::HTTPNotFound (404)
#   - Net::HTTPMethodNotAllowed (405)
#   - Net::HTTPNotAcceptable (406)
#   - Net::HTTPProxyAuthenticationRequired (407)
#   - Net::HTTPRequestTimeOut (408)
#   - Net::HTTPConflict (409)
#   - Net::HTTPGone (410)
#   - Net::HTTPLengthRequired (411)
#   - Net::HTTPPreconditionFailed (412)
#   - Net::HTTPRequestEntityTooLarge (413)
#   - Net::HTTPRequestURITooLong (414)
#   - Net::HTTPUnsupportedMediaType (415)
#   - Net::HTTPRequestedRangeNotSatisfiable (416)
#   - Net::HTTPExpectationFailed (417)
#   - Net::HTTPMisdirectedRequest (421)
#   - Net::HTTPUnprocessableEntity (422)
#   - Net::HTTPLocked (423)
#   - Net::HTTPFailedDependency (424)
#   - Net::HTTPUpgradeRequired (426)
#   - Net::HTTPPreconditionRequired (428)
#   - Net::HTTPTooManyRequests (429)
#   - Net::HTTPRequestHeaderFieldsTooLarge (431)
#   - Net::HTTPUnavailableForLegalReasons (451)
#
# - Net::HTTPServerError:
#
#   - Net::HTTPInternalServerError (500)
#   - Net::HTTPNotImplemented (501)
#   - Net::HTTPBadGateway (502)
#   - Net::HTTPServiceUnavailable (503)
#   - Net::HTTPGatewayTimeOut (504)
#   - Net::HTTPVersionNotSupported (505)
#   - Net::HTTPVariantAlsoNegotiates (506)
#   - Net::HTTPInsufficientStorage (507)
#   - Net::HTTPLoopDetected (508)
#   - Net::HTTPNotExtended (510)
#   - Net::HTTPNetworkAuthenticationRequired (511)
#
# There is also the Net::HTTPBadResponse exception which is raised when
# there is a protocol error.
#
class Net::HTTPResponse
  class << self
    # true if the response has a body.
    def body_permitted?
      self::HAS_BODY
    end

    def exception_type   # :nodoc: internal use only
      self::EXCEPTION_TYPE
    end

    def read_new(sock)   #:nodoc: internal use only
      httpv, code, msg = read_status_line(sock)
      res = response_class(code).new(httpv, code, msg)
      each_response_header(sock) do |k,v|
        res.add_field k, v
      end
      res
    end

    private

    def read_status_line(sock)
      str = sock.readline
      m = /\AHTTP(?:\/(\d+\.\d+))?\s+(\d\d\d)(?:\s+(.*))?\z/in.match(str) or
        raise Net::HTTPBadResponse, "wrong status line: #{str.dump}"
      m.captures
    end

    def response_class(code)
      CODE_TO_OBJ[code] or
      CODE_CLASS_TO_OBJ[code[0,1]] or
      Net::HTTPUnknownResponse
    end

    def each_response_header(sock)
      key = value = nil
      while true
        line = sock.readuntil("\n", true).sub(/\s+\z/, '')
        break if line.empty?
        if line[0] == ?\s or line[0] == ?\t and value
          value << ' ' unless value.empty?
          value << line.strip
        else
          yield key, value if key
          key, value = line.strip.split(/\s*:\s*/, 2)
          raise Net::HTTPBadResponse, 'wrong header line format' if value.nil?
        end
      end
      yield key, value if key
    end
  end

  # next is to fix bug in RDoc, where the private inside class << self
  # spills out.
  public

  include Net::HTTPHeader

  def initialize(httpv, code, msg)   #:nodoc: internal use only
    @http_version = httpv
    @code         = code
    @message      = msg
    initialize_http_header nil
    @body = nil
    @read = false
    @uri  = nil
    @decode_content = false
    @body_encoding = false
    @ignore_eof = true
  end

  # The HTTP version supported by the server.
  attr_reader :http_version

  # The HTTP result code string. For example, '302'.  You can also
  # determine th