 See {Custom Header Converters}[#class-CSV-label-Custom+Header+Converters].
  HeaderConverters = {
    downcase: lambda { |h| h.encode(ConverterEncoding).downcase },
    symbol:   lambda { |h|
      h.encode(ConverterEncoding).downcase.gsub(/[^\s\w]+/, "").strip.
                                           gsub(/\s+/, "_").to_sym
    },
    symbol_raw: lambda { |h| h.encode(ConverterEncoding).to_sym }
  }

  # Default values for method options.
  DEFAULT_OPTIONS = {
    # For both parsing and generating.
    col_sep:            ",",
    row_sep:            :auto,
    quote_char:         '"',
    # For parsing.
    field_size_limit:   nil,
    max_field_size:     nil,
    converters:         nil,
    unconverted_fields: nil,
    headers:            false,
    return_headers:     false,
    header_converters:  nil,
    skip_blanks:        false,
    skip_lines:         nil,
    liberal_parsing:    false,
    nil_value:          nil,
    empty_value:        "",
    strip:              false,
    # For generating.
    write_headers:      nil,
    quote_empty:        true,
    force_quotes:       false,
    write_converters:   nil,
    write_nil_value:    nil,
    write_empty_value:  "",
  }.freeze

  class << self
    # :call-seq:
    #   instance(string, **options)
    #   instance(io = $stdout, **options)
    #   instance(string, **options) {|csv| ... }
    #   instance(io = $stdout, **options) {|csv| ... }
    #
    # Creates or retrieves cached \CSV objects.
    # For arguments and options, see CSV.new.
    #
    # This API is not Ractor-safe.
    #
    # ---
    #
    # With no block given, returns a \CSV object.
    #
    # The first call to +instance+ creates and caches a \CSV object:
    #   s0 = 's0'
    #   csv0 = CSV.instance(s0)
    #   csv0.class # => CSV
    #
    # Subsequent calls to +instance+ with that _same_ +string+ or +io+
    # retrieve that same cached object:
    #   csv1 = CSV.instance(s0)
    #   csv1.class # => CSV
    #   csv1.equal?(csv0) # => true # Same CSV object
    #
    # A subsequent call to +instance+ with a _different_ +string+ or +io+
    # creates and caches a _different_ \CSV object.
    #   s1 = 's1'
    #   csv2 = CSV.instance(s1)
    #   csv2.equal?(csv0) # => false # Different CSV object
    #
    # All the cached objects remains available:
    #   csv3 = CSV.instance(s0)
    #   csv3.equal?(csv0) # true # Same CSV object
    #   csv4 = CSV.instance(s1)
    #   csv4.equal?(csv2) # true # Same CSV object
    #
    # ---
    #
    # When a block is given, calls the block with the created or retrieved
    # \CSV object; returns the block's return value:
    #   CSV.instance(s0) {|csv| :foo } # => :foo
    def instance(data = $stdout, **options)
      # create a _signature_ for this method call, data object and options
      sig = [data.object_id] +
            options.values_at(*DEFAULT_OPTIONS.keys)

      # fetch or create the instance for this signature
      @@instances ||= Hash.new
      instance = (@@instances[sig] ||= new(data, **options))

      if block_given?
        yield instance  # run block, if given, returning result
      else
        instance        # or return the instance
      end
    end

    # :call-seq:
    #   filter(in_string_or_io, **options) {|row| ... } -> array_of_arrays or csv_table
    #   filter(in_string_or_io, out_string_or_io, **options) {|row| ... } -> array_of_arrays or csv_table
    #   filter(**options) {|row| ... } -> array_of_arrays or csv_table
    #
    # - Parses \CSV from a source (\String, \IO stream, or ARGF).
    # - Calls the given block with each parsed row:
    #   - Without headers, each row is an \Array.
    #   - With headers, each row is a CSV::Row.
    # - Generates \CSV to an output (\String, \IO stream, or STDOUT).
    # - Returns the parsed source:
    #   - Without headers, an \Array of \Arrays.
    #   - With headers, a CSV::Table.
    #
    # When +in_string_or_io+ is given, but not +out_string_or_io+,
    # parses from the given +in_string_or_io+
    # and generates to STDOUT.
    #
    # \String input without headers:
    #
    #   in_string = "foo,0\nbar,1\nbaz,2"
    #   CSV.filter(in_string) do |row|
    #     row[0].upcase!
    #     row[1] = - row[1].to_i
    #   end # => [["FOO", 0], ["BAR", -1], ["BAZ", -2]]
    #
    # Output (to STDOUT):
    #
    #   FOO,0
    #   BAR,-1
    #   BAZ,-2
    #
    # \String input with headers:
    #
    #   in_string = "Name,Value\nfoo,0\nbar,1\nbaz,2"
    #   CSV.filter(in_string, headers: true) do |row|
    #     row[0].upcase!
    #     row[1] = - row[1].to_i
    #   end # => #<CSV::Table mode:col_or_row row_count:4>
    #
    # Output (to STDOUT):
    #
    #   Name,Value
    #   FOO,0
    #   BAR,-1
    #   BAZ,-2
    #
    # \IO stream input without headers:
    #
    #   File.write('t.csv', "foo,0\nbar,1\nbaz,2")
    #   File.open('t.csv') do |in_io|
    #     CSV.filter(in_io) do |row|
    #       row[0].upcase!
    #       row[1] = - row[1].to_i
    #     end
    #   end # => [["FOO", 0], ["BAR", -1], ["BAZ", -2]]
    #
    # Output (to STDOUT):
    #
    #   FOO,0
    #   BAR,-1
    #   BAZ,-2
    #
    # \IO stream input with headers:
    #
    #   File.write('t.csv', "Name,Value\nfoo,0\nbar,1\nbaz,2")
    #   File.open('t.csv') do |in_io|
    #     CSV.filter(in_io, headers: true) do |row|
    #       row[0].upcase!
    #       row[1] = - row[1].to_i
    #     end
    #   end # => #<CSV::Table mode:col_or_row row_count:4>
    #
    # Output (to STDOUT):
    #
    #   Name,Value
    #   FOO,0
    #   BAR,-1
    #   BAZ,-2
    #
    # When both +in_string_or_io+ and +out_string_or_io+ are given,
    # parses from +in_string_or_io+ and generates to +out_string_or_io+.
    #
    # \String output without headers:
    #
    #   in_string = "foo,0\nbar,1\nbaz,2"
    #   out_string = ''
    #   CSV.filter(in_string, out_string) do |row|
    #     row[0].upcase!
    #     row[1] = - row[1].to_i
    #   end # => [["FOO", 0], ["BAR", -1], ["BAZ", -2]]
    #   out_string # => "FOO,0\nBAR,-1\nBAZ,-2\n"
    #
    # \String output with headers:
    #
    #   in_string = "Name,Value\nfoo,0\nbar,1\nbaz,2"
    #   out_string = ''
    #   CSV.filter(in_string, out_string, headers: true) do |row|
    #     row[0].upcase!
    #     row[1] = - row[1].to_i
    #   end # => #<CSV::Table mode:col_or_row row_count:4>
    #   out_string # => "Name,Value\nFOO,0\nBAR,-1\nBAZ,-2\n"
    #
    # \IO stream output without headers:
    #
    #   in_string = "foo,0\nbar,1\nbaz,2"
    #   File.open('t.csv', 'w') do |out_io|
    #     CSV.filter(in_string, out_io) do |row|
    #       row[0].upcase!
    #       row[1] = - row[1].to_i
    #     end
    #   end # => [["FOO", 0], ["BAR", -1], ["BAZ", -2]]
    #   File.read('t.csv') # => "FOO,0\nBAR,-1\nBAZ,-2\n"
    #
    # \IO stream output with headers:
    #
    #   in_string = "Name,Value\nfoo,0\nbar,1\nbaz,2"
    #   File.open('t.csv', 'w') do |out_io|
    #     CSV.filter(in_string, out_io, headers: true) do |row|
    #       row[0].upcase!
    #       row[1] = - row[1].to_i
    #     end
    #   end # => #<CSV::Table mode:col_or_row row_count:4>
    #   File.read('t.csv') # => "Name,Value\nFOO,0\nBAR,-1\nBAZ,-2\n"
    #
    # When neither +in_string_or_io+ nor +out_string_or_io+ given,
    # parses from {ARGF}[rdoc-ref:ARGF]
    # and generates to STDOUT.
    #
    # Without headers:
    #
    #   # Put Ruby code into a file.
    #   ruby = <<-EOT
    #     require 'csv'
    #     CSV.filter do |row|
    #       row[0].upcase!
    #       row[1] = - row[1].to_i
    #     end
    #   EOT
    #   File.write('t.rb', ruby)
    #   # Put some CSV into a file.
    #   File.write('t.csv', "foo,0\nbar,1\nbaz,2")
    #   # Run the Ruby code with CSV filename as argument.
    #   system(Gem.ruby, "t.rb", "t.csv")
    #
    # Output (to STDOUT):
    #
    #   FOO,0
    #   BAR,-1
    #   BAZ,-2
    #
    # With headers:
    #
    #   # Put Ruby code into a file.
    #   ruby = <<-EOT
    #     require 'csv'
    #     CSV.filter(headers: true) do |row|
    #       row[0].upcase!
    #       row[1] = - row[1].to_i
    #     end
    #   EOT
    #   File.write('t.rb', ruby)
    #   # Put some CSV into a file.
    #   File.write('t.csv', "Name,Value\nfoo,0\nbar,1\nbaz,2")
    #   # Run the Ruby code with CSV filename as argument.
    #   system(Gem.ruby, "t.rb", "t.csv")
    #
    # Output (to STDOUT):
    #
    #   Name,Value
    #   FOO,0
    #   BAR,-1
    #   BAZ,-2
    #
    # Arguments:
    #
    # * Argument +in_string_or_io+ must be a \String or an \IO stream.
    # * Argument +out_string_or_io+ must be a \String or an \IO stream.
    # * Arguments <tt>**options</tt> must be keyword options.
    #   See {Options for Parsing}[#class-CSV-label-Options+for+Parsing].
    def filter(input=nil, output=nil, **options)
      # parse options for input, output, or both
      in_options, out_options = Hash.new, {row_sep: InputRecordSeparator.value}
      options.each do |key, value|
        case key
        when /\Ain(?:put)?_(.+)\Z/
          in_options[$1.to_sym] = value
        when /\Aout(?:put)?_(.+)\Z/
          out_options[$1.to_sym] = value
        else
          in_options[key]  = value
          out_options[key] = value
        end
      end

      # build input and output wrappers
      input  = new(input  || ARGF, **in_options)
      output = new(output || $stdout, **out_options)

      # process headers
      need_manual_header_output =
        (in_options[:headers] and
         out_options[:headers] == true and
         out_options[:write_headers])
      if need_manual_header_output
        first_row = input.shift
        if first_row
          if first_row.is_a?(Row)
            headers = first_row.headers
            yield headers
            output << headers
          end
          yield first_row
          output << first_row
        end
      end

      # read, yield, write
      input.each do |row|
        yield row
        output << row
      end
    end

    #
    # :call-seq:
    #   foreach(path_or_io, mode='r', **options) {|row| ... )
    #   foreach(path_or_io, mode='r', **options) -> new_enumerator
    #
    # Calls the block with each row read from source +path_or_io+.
    #
    # \Path input without headers:
    #
    #   string = "foo,0\nbar,1\nbaz,2\n"
    #   in_path = 't.csv'
    #   File.write(in_path, string)
    #   CSV.foreach(in_path) {|row| p row }
    #
    # Output:
    #
    #   ["foo", "0"]
    #   ["bar", "1"]
    #   ["baz", "2"]
    #
    # \Path input with headers:
    #
    #   string = "Name,Value\nfoo,0\nbar,1\nbaz,2\n"
    #   in_path = 't.csv'
    #   File.write(in_path, string)
    #   CSV.foreach(in_path, headers: true) {|row| p row }
    #
    # Output:
    #
    #   <CSV::Row "Name":"foo" "Value":"0">
    #   <CSV::Row "Name":"bar" "Value":"1">
    #   <CSV::Row "Name":"baz" "Value":"2">
    #
    # \IO stream input without headers:
    #
    #   string = "foo,0\nbar,1\nbaz,2\n"
    #   path = 't.csv'
    #   File.write(path, string)
    #   File.open('t.csv') do |in_io|
    #     CSV.foreach(in_io) {|row| p row }
    #   end
    #
    # Output:
    #
    #   ["foo", "0"]
    #   ["bar", "1"]
    #   ["baz", "2"]
    #
    # \IO stream input with headers:
    #
    #   string = "Name,Value\nfoo,0\nbar,1\nbaz,2\n"
    #   path = 't.csv'
    #   File.write(path, string)
    #   File.open('t.csv') do |in_io|
    #     CSV.foreach(in_io, headers: true) {|row| p row }
    #   end
    #
    # Output:
    #
    #   <CSV::Row "Name":"foo" "Value":"0">
    #   <CSV::Row "Name":"bar" "Value":"1">
    #   <CSV::Row "Name":"baz" "Value":"2">
    #
    # With no block given, returns an \Enumerator:
    #
    #   string = "foo,0\nbar,1\nbaz,2\n"
    #   path = 't.csv'
    #   File.write(path, string)
    #   CSV.foreach(path) # => #<Enumerator: CSV:foreach("t.csv", "r")>
    #
    # Arguments:
    # * Argument +path_or_io+ must be a file path or an \IO stream.
    # * Argument +mode+, if given, must be a \File mode
    #   See {Open Mode}[https://ruby-doc.org/core/IO.html#method-c-new-label-Open+Mode].
    # * Arguments <tt>**options</tt> must be keyword options.
    #   See {Options for Parsing}[#class-CSV-label-Options+for+Parsing].
    # * This method optionally accepts an additional <tt>:encoding</tt> option
    #   that you can use to specify the Encoding of the data read from +path+ or +io+.
    #   You must provide this unless your data is in the encoding
    #   given by <tt>Encoding::default_external</tt>.
    #   Parsing will use this to determine how to parse the data.
    #   You may provide a second Encoding to
    #   have the data transcoded as it is read. For example,
    #     encoding: 'UTF-32BE:UTF-8'
    #   would read +UTF-32BE+ data from the file
    #   but transcode it to +UTF-8+ before parsing.
    def foreach(path, mode="r", **options, &block)
      return to_enum(__method__, path, mode, **options) unless block_given?
      open(path, mode, **options) do |csv|
        csv.each(&block)
      end
    end

    #
    # :call-seq:
    #   generate(csv_string, **options) {|csv| ... }
    #   generate(**options) {|csv| ... }
    #
    # * Argument +csv_string+, if given, must be a \String object;
    #   defaults to a new empty \String.
    # * Arguments +options+, if given, should be generating options.
    #   See {Options for Generating}[#class-CSV-label-Options+for+Generating].
    #
    # ---
    #
    # Creates a new \CSV object via <tt>CSV.new(csv_string, **options)</tt>;
    # calls the block with the \CSV object, which the block may modify;
    # returns the \String generated from the \CSV object.
    #
    # Note that a passed \String *is* modified by this method.
    # Pass <tt>csv_string</tt>.dup if the \String must be preserved.
    #
    # This method has one additional option: <tt>:encoding</tt>,
    # which sets the base Encoding for the output if no no +str+ is specified.
    # CSV needs this hint if you plan to output non-ASCII compatible data.
    #
    # ---
    #
    # Add lines:
    #   input_string = "foo,0\nbar,1\nbaz,2\n"
    #   output_string = CSV.generate(input_string) do |csv|
    #     csv << ['bat', 3]
    #     csv << ['bam', 4]
    #   end
    #   output_string # => "foo,0\nbar,1\nbaz,2\nbat,3\nbam,4\n"
    #   input_string # => "foo,0\nbar,1\nbaz,2\nbat,3\nbam,4\n"
    #   output_string.equal?(input_string) # => true # Same string, modified
    #
    # Add lines into new string, preserving old string:
    #   input_string = "foo,0\nbar,1\nbaz,2\n"
    #   output_string = CSV.generate(input_string.dup) do |csv|
    #     csv << ['bat', 3]
    #     csv << ['bam', 4]
    #   end
    #   output_string # => "foo,0\nbar,1\nbaz,2\nbat,3\nbam,4\n"
    #   input_string # => "foo,0\nbar,1\nbaz,2\n"
    #   output_string.equal?(input_string) # => false # Different strings
    #
    # Create lines from nothing:
    #   output_string = CSV.generate do |csv|
    #     csv << ['foo', 0]
    #     csv << ['bar', 1]
    #     csv << ['baz', 2]
    #   end
    #   output_string # => "foo,0\nbar,1\nbaz,2\n"
    #
    # ---
    #
    # Raises an exception if +csv_string+ is not a \String object:
    #   # Raises TypeError (no implicit conversion of Integer into String)
    #   CSV.generate(0)
    #
    def generate(str=nil, **options)
      encoding = options[:encoding]
      # add a default empty String, if none was given
      if str
        str = StringIO.new(str)
        str.seek(0, IO::SEEK_END)
        str.set_encoding(encoding) if encoding
      else
        str = +""
        str.force_encoding(encoding) if encoding
      end
      csv = new(str, **options) # wrap
      yield csv         # yield for appending
      csv.string        # return final String
    end

    # :call-seq:
    #   CSV.generate_line(ary)
    #   CSV.generate_line(ary, **options)
    #
    # Returns the \String created by generating \CSV from +ary+
    # using the specified +options+.
    #
    # Argument +ary+ must be an \Array.
    #
    # Special options:
    # * Option <tt>:row_sep</tt> defaults to <tt>"\n"> on Ruby 3.0 or later
    #   and <tt>$INPUT_RECORD_SEPARATOR</tt> (<tt>$/</tt>) otherwise.:
    #     $INPUT_RECORD_SEPARATOR # => "\n"
    # * This method accepts an additional option, <tt>:encoding</tt>, which sets the base
    #   Encoding for the output. This method will try to guess your Encoding from
    #   the first non-+nil+ field in +row+, if possible, but you may need to use
    #   this parameter as a backup plan.
    #
    # For other +options+,
    # see {Options for Generating}[#class-CSV-label-Options+for+Generating].
    #
    # ---
    #
    # Returns the \String generated from an \Array:
    #   CSV.generate_line(['foo', '0']) # => "foo,0\n"
    #
    # ---
    #
    # Raises an exception if +ary+ is not an \Array:
    #   # Raises NoMethodError (undefined method `find' for :foo:Symbol)
    #   CSV.generate_line(:foo)
    #
    def generate_line(row, **options)
      options = {row_sep: InputRecordSeparator.value}.merge(options)
      str = +""
      if options[:encoding]
        str.force_encoding(options[:encoding])
      else
        fallback_encoding = nil
        output_encoding = nil
        row.each do |field|
          next unless field.is_a?(String)
          fallback_encoding ||= field.encoding
          next if field.ascii_only?
          output_encoding = field.encoding
          break
        end
        output_encoding ||= fallback_encoding
        if output_encoding
          str.force_encoding(output_encoding)
        end
      end
      (new(str, **options) << row).string
    end

    # :call-seq:
    #   CSV.generate_lines(rows)
    #   CSV.generate_lines(rows, **options)
    #
    # Returns the \String created by generating \CSV from
    # using the specified +options+.
    #
    # Argument +rows+ must be an \Array of row. Row is \Array of \St